/*++
This file contains a 'Sample Driver' and is licensed as such
under the terms of your license agreement with Intel or your
vendor.  This file may be modified by the user, subject to  
the additional terms of the license agreement               
--*/
/*++

Copyright (c)  2009-2017 Intel Corporation. All rights reserved
This software and associated documentation (if any) is furnished
under a license and may only be used or copied in accordance
with the terms of the license. Except as permitted by such
license, no part of this software or documentation may be
reproduced, stored in a retrieval system, or transmitted in any
form or by any means without the express written consent of
Intel Corporation.

Module Name:

FwUpdLclApp.c

Abstract:

FW Update Local Sample Code demonstrating usage of FW Update Library

--*/


#include "efi.h"
#include "efilib.h"
#include "Fwu_Common.h"
#include "Common.h"
#include "me_status.h"
#include "FWUpdateLib.h"
#include "cse_basic_types.h"
#include "typedef.h"



#define CMD_LINE_STATUS_UPDATE_1        0
#define CMD_LINE_STATUS_UPDATE_2        1
#define CMD_LINE_STATUS_UPDATE_3        2
#define CMD_LINE_STATUS_UPDATE_4        3

#define HECI1_CSE_GS1_PHASE_FWUPDATE                  7


// Info Strings 
#define ID_INFO_DOT     L"."
#define ID_INFO_2       L"\nTrying to receive update status..."
#define ID_INFO_3       L"Communication Mode: MEI\n"
#define ID_INFO_16      L"Maximum Retry limit reached. Please reboot to try again\n"
#define ID_INFO_1       L"Do not exit the process or power off the machine before the firmware update process ends.\n"

// Error Strings 
#define ID_ERROR_2      L"\nError %d: Firmware update not initiated due to invalid hostname specified\n"
#define ID_ERROR_3      L"\nError %d: Firmware update tool failed due to insufficient memory\n"
#define ID_ERROR_4      L"\nError %d: Update operation timed-out; cannot determine if the operation succeeded\n"
#define ID_ERROR_5      L"\nError %d: Cannot receive the current version from the firmware after update\n"
#define ID_ERROR_6      L"\nError %d: Update finished but version mismatch after the update\n"
#define ID_ERROR_7      L"\nError %d: Firmware update not initiated due to an invalid FW image \n"
#define ID_ERROR_8      L"\nError %d: Firmware update not initiated due to integrity failure or invalid FW image"
#define ID_ERROR_9      L"\nError %d: Firmware update operation not initiated due to a SKU mismatch\n"
#define ID_ERROR_10     L"\nError %d: Firmware update not initiated due to version mismatch\n"
#define ID_ERROR_11     L"\nError %d: Firmware update failed due to authentication failure\n"
#define ID_ERROR_12     L"\nError %d: Firmware update failed due to insufficient memory\n"
#define ID_ERROR_13     L"\nError %d: Firmware update iAMT communication failed, Failed to find certificate '%s' in certificate store\n"
#define ID_ERROR_14     L"\nError %d: Firmware update iAMT communication failed, Failed to set HTTP certificate options (%d): %s\n"
#define ID_ERROR_15     L"\nError %d: Firmware update iAMT communication failed, Failed to find certificate names\n"
#define ID_ERROR_16     L"\nError %d: Firmware update iAMT communication failed, Failed to open system certificate store (%d): %s\n"
#define ID_ERROR_17     L"\nError %d: Firmware update tool failed to connect iAMT through LMS, due to a HTTP operation failure,\nPlease verify the inputs (host, user, password, certificate, work mode etc).\n"

#define ID_ERROR_18     L"\nError %d: Invalid usage\n"
#define ID_ERROR_18B    L"\nError %d: Invalid usage, -allowsv switch required to update the same version firmware\n"
#define ID_ERROR_19     L"\nError %d: Failed to receive last update status from the firmware\n"
#define ID_ERROR_20     L"\nError %d: Firmware Update operation not initiated because a firmware update is already in progress\n"
#define ID_ERROR_21     L"\nError %d: Invalid UUID provided with the OEMID switch\n"
#define ID_ERROR_22     L"\nError %d: OEM ID verification failed.\n"
#define ID_ERROR_24     L"\nError %d: Firmware update tool failed to get the firmware parameters\n"

#define ID_ERROR_25     L"\nError %d: Firmware update failed due to an internal error\n"
#define ID_ERROR_25A    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns SAL notification error, Please try after ME-reset or re-flashing the ME image.\n"
#define ID_ERROR_25B    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns Audit policy error, Please try after ME-reset or re-flashing the ME image.\n"
#define ID_ERROR_25C    L"\nError %d: Firmware update failed due to an internal error\nFirmware failed to create fault tolerant partition, Please try after ME-reset or re-flashing the ME image.\n"
#define ID_ERROR_25D    L"\nError %d: Firmware update failed due to an internal error\nFirmware failed  on image version history check.\n"
#define ID_ERROR_25E    L"\nError %d: Firmware update failed due to an internal error\nFirmware failed with write file issue.\n"
#define ID_ERROR_25F    L"\nError %d: Firmware update failed due to an internal error\nFirmware failed with read file issue.\n"
#define ID_ERROR_25G    L"\nError %d: Firmware update failed due to an internal error\nFirmware failed with delete file issue.\n"
#define ID_ERROR_25H    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns invalid flash code partition.\n"
#define ID_ERROR_25I    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns invalid flash NFT partition.\n"
#define ID_ERROR_25J    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns invalid image length.\n"
#define ID_ERROR_25K    L"\nError %d: Firmware update failed due to an internal error\nFirmware returns invalid GLUT.\n"
#define ID_ERROR_25L    L"\nError %d: Firmware update failed due to an internal error\nFirmware Update client is not ready to perform an update.\n"

#define ID_ERROR_26     L"\nError %d: Unsupported Operating System\n"
#define ID_ERROR_27     L"\nError %d: No Firmware update is happening\n"
#define ID_ERROR_28     L"\nError %d: Unknown or Unsupported Platform\n"
#define ID_ERROR_29     L"\nError %d: Firmware update cannot be initiated because Local Firmware update is disabled\n"
#define ID_ERROR_30     L"\nError %d: Firmware update cannot be initiated because Secure FW update is disabled\n"
#define ID_ERROR_31     L"\nError %d: Firmware update not initiated due to an invalid FW image header\n"
#define ID_ERROR_32     L"\nError %d: Firmware update not initiated due to file [%s] open or read failure\n"
#define ID_ERROR_33     L"\nError %d: Firmware update cannot be initiated because the OEM ID provided is incorrect\n"
#define ID_ERROR_34     L"\nError %d: Firmware update not initiated due to file open or read failure\n"
#define ID_ERROR_35     L"\nError %d: Firmware update iAMT communication failed, HTTP request failed: Certificate rejected\n"
#define ID_ERROR_36     L"\nError %d: This version of the Intel (R) FW Update Tool is not compatible with the current platform.\n"
#define ID_ERROR_37     L"\nError %d: Intel (R) ME Interface : Cannot locate ME device driver\n"
#define ID_ERROR_38     L"\nError %d: Platform did not respond to update request.\n"
#define ID_ERROR_39     L"\nError %d: Intel (R) ME Interface : Unsupported message type\n"
#define ID_ERROR_40     L"\nError %d: Firmware update not initiated due to invalid image length\n"
#define ID_ERROR_41     L"\nError %d: Firmware update not initiated due to an unavailable global buffer\n"
#define ID_ERROR_44     L"\nError %d: Firmware update not initiated due to invalid firmware parameters\n"
#define ID_ERROR_46     L"\nError %d: Firmware update iAMT communication failed, WSMAN not supported\n"
#define ID_ERROR_51     L"\nError %d: PLEASE REBOOT YOUR SYSTEM. Firmware update cannot be initiated without a reboot.\n"
#define ID_ERROR_52     L"\nError %d: An internal error to the AMT device has occurred.\n"
#define ID_ERROR_53     L"\nError %d: AMT Status is not ready.\n"
#define ID_ERROR_54     L"\nError %d: Invalid AMT Mode.\n"
#define ID_ERROR_55     L"\nError %d: Encountered error writing to file.\n"
#define ID_ERROR_56     L"\nError %d: Display FW Version failed.\n"
#define ID_ERROR_57     L"\nError %d: The image provided is not supported by the platform.\n"
#define ID_ERROR_57A    L"\nError %d: This platform does not allow downgrade/upgrade with the file provided.\n"
#define ID_ERROR_58     L"\nError %d: Internal Error.\n"
#define ID_ERROR_59     L"\nError %d: Update downgrade vetoed.\n"
#define ID_ERROR_60     L"\nError %d: Firmware write file failure.\n"
#define ID_ERROR_61     L"\nError %d: Firmware read file failure.\n"
#define ID_ERROR_62     L"\nError %d: Firmware delete file failure.\n"
#define ID_ERROR_63     L"\nError %d: Partition layout NOT compatible.\n"
#define ID_ERROR_64     L"\nError %d: Downgrade NOT allowed, data mismatched.\n"
#define ID_ERROR_65     L"\nError %d: Password did not match.\n"
#define ID_ERROR_66     L"\nError %d: Password exceeded maximum number of retries.\n"
#define ID_ERROR_67     L"\nError %d: Password Not provided when required.\n"
#define ID_ERROR_68     L"\nError %d: Polling for FW Update Failed.\n"
#define ID_ERROR_69     L"\nError %d: FW Update Failed.\n"
#define ID_ERROR_70     L"\nError %d: Intel (R) ME Interface : Cannot locate ME device driver; unable to determine if the operation succeeded.\n"
#define ID_ERROR_71     L"\nError %d: Intel (R) ME Interface : Lost communication with platform during update; unable to determine if the update completed.\n"
#define ID_ERROR_72     L"\nError %d: Unable to read FW version from file. Please verify the update image used.\n"
#define ID_ERROR_73     L"\nError %d: Firmware is in recovery mode. Please update only using the previously attempted version.\n"

// Warning Messages 
#define ID_WARN_0       "\nWarning: Do not exit the process or power off the machine before the firmware update process ends.\n"

//
// Specify the OEM ID to be passed to the ME
// The bit order must match what was created with FITC.
//
// For example:
// FITC Setting: 00000000-0001-0000-0000-00000000001
// MEINFO.EXE returns: 00000000-0001-0000-0000-00000000001
// 
// mOemId would be: 
//  _UUID                      mOemId = {0x00000000, 0x0001, 0x0000, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01};
//
_UUID                      mOemId = {0x00000000, 0x0000, 0x0000, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

//Defines
// Last Reset Types
#define MFT_PART_INFO_EXT_UPDATE_ACTION_NONE         0
#define MFT_PART_INFO_EXT_UPDATE_ACTION_HOST_RESET   1
#define MFT_PART_INFO_EXT_UPDATE_ACTION_CSE_RESET    2
#define MFT_PART_INFO_EXT_UPDATE_ACTION_GLOBAL_RESET 3

// Get Interface
#define FW_UPDATE_DISABLED 0
#define FW_UPDATE_ENABLED 1
#define FW_UPDATE_PASSWORD_PROTECTED 2

//
//  MSFT compiler uses this variable to inform itself whether floating
//  point support is loaded.
//
int _fltused;

unsigned int      index = 0;
signed int        timer30s = 0;

FlashVersion  mVersion;

//
// Print a message to the desired output device
// Customize for platform
//
void DisplayConsoleMessage (IN CHAR16 *String)
{
    Print(String);
}

// This fucntion prints the progress bar line during hte fw update process. 
static UINT32 ProgressDot(void)
{
    // new code
    INT32 i = 0;
    char string[3][80] = {"                       Do not Interrupt", 
        "                       Do not Interrupt", 
        "                       Do not Interrupt"};

    for (i = 0; i < 3; i++)
    {

        Print (L"Progress Dot %s \r", string[i]);
        timer30s++; 
    }
    return 0;
} 

void DisplaySendStatus(float BytesSent, float BytestobeSent)
{
    float  Value = BytesSent/BytestobeSent * 100;

    UINT32  pValue = (UINT32)Value;

    if (pValue != 100)
    {
        Print (L"Sending the update image to FW for verification:  [ %d%% ]\r",pValue);
    }else 
    {
        Print (L"Sending the update image to FW for verification:  [ COMPLETE ] \n");
    }
}

void DisplayTextForReturnErrorCode(UINT32 Code)
{
    switch(Code) {
    case FWU_NO_MEMORY:
        Print (ID_ERROR_3, FWU_NO_MEMORY);
        break;
    case FWU_UPDATE_TIMEOUT:
        Print (ID_ERROR_4, FWU_UPDATE_TIMEOUT);
        break;
    case FWU_IMG_HEADER:
        Print (ID_ERROR_7, FWU_IMG_HEADER);
        break;
    case FWU_SGN_MISMATCH:
        Print (ID_ERROR_8, FWU_SGN_MISMATCH);
        break;
    case FWU_SKU_MISMATCH: 
        Print (ID_ERROR_9, FWU_SKU_MISMATCH);
        break;
    case FWU_VER_MISMATCH:
        Print (ID_ERROR_10, FWU_VER_MISMATCH);
        break;
    case FWU_USAGE_ERROR:
        Print (ID_ERROR_18, FWU_USAGE_ERROR);
        break;
    case FWU_USAGE_ERROR_B:
        Print (ID_ERROR_18B, FWU_USAGE_ERROR_B);
        break;
    case FWU_LAST_STATUS:
        Print (ID_ERROR_19, FWU_LAST_STATUS);
        break;
    case FWU_GET_VER_ERR:
        Print (ID_ERROR_24 , FWU_GET_VER_ERR);
        break;
    case FWU_AUDIT_POLICY_FAILURE:
        Print (ID_ERROR_25C, FWU_GENERAL);
        break;
    case FWU_ERROR_CREATING_FT:
        Print (ID_ERROR_25C, FWU_GENERAL);
        break;
    case FWU_SAL_NOTIFICATION_ERROR:
        Print (ID_ERROR_25A, FWU_GENERAL);
        break;
    case FWU_NO_UPDATE:
        Print (ID_ERROR_27, FWU_NO_UPDATE);
        break;
    case FWU_LOCAL_DIS:
        Print (ID_ERROR_29, FWU_LOCAL_DIS);
        break;
    case FWU_VERIFY_OEM_ID_ERR:
        Print (ID_ERROR_22 , FWU_VERIFY_OEM_ID_ERR);
        break;
    case FWU_INVALID_OEM_ID:
        Print (ID_ERROR_33, FWU_INVALID_OEM_ID);
        break;
    case FWU_FILE_OPEN:
        Print (ID_ERROR_34, FWU_FILE_OPEN);
        break;
    case FWU_IME_SMALL_BUFF:
        Print (ID_ERROR_36, FWU_IME_SMALL_BUFF);
        break;
    case FWU_IME_NO_DEVICE:
        Print (ID_ERROR_37, FWU_IME_NO_DEVICE);
        break;
    case FWU_IME_NOT_READY:
        Print (ID_ERROR_38, FWU_IME_NOT_READY);
        break;
    case FWU_IME_UN_SUP_MESS:
        Print (ID_ERROR_39, FWU_IME_UN_SUP_MESS);
        break;
    case FWU_INVALID_IMG_LENGTH: 
        Print (ID_ERROR_40, FWU_INVALID_IMG_LENGTH);
        break;
    case FWU_GLBL_BUFF_UNAVAILABLE:
        Print (ID_ERROR_41, FWU_GLBL_BUFF_UNAVAILABLE);
        break;
        //case STATUS_OUT_OF_MEMORY:
        //	Print (ID_ERROR_12, FWU_NO_MEMORY);
        //	break;
    case FWU_INVALID_FW_PARAMS:
        Print (ID_ERROR_44, FWU_INVALID_FW_PARAMS);
        break;
    case FWU_FILE_WRITE:
        Print (ID_ERROR_55, FWU_FILE_WRITE);
        break;
    case FWU_DISPLAY_FW_VERSION:
        Print (ID_ERROR_56, FWU_DISPLAY_FW_VERSION);
        break;
    case FWU_IMAGE_UNDER_VCN:
        Print(ID_ERROR_57, FWU_IMAGE_UNDER_VCN);
        break;
    case FWU_IMAGE_VER_HIST_CHK_FAIL:
        Print (ID_ERROR_58, FWU_IMAGE_VER_HIST_CHK_FAIL);
        break;
    case FWU_DOWNGRADE_VETOED:
        Print (ID_ERROR_59, FWU_DOWNGRADE_VETOED);
        break;
    case FWU_FW_WRITE_FILE_FAIL:
        Print (ID_ERROR_60, FWU_FW_WRITE_FILE_FAIL);
        break;
    case FWU_FW_READ_FILE_FAIL:
        Print (ID_ERROR_61, FWU_FW_READ_FILE_FAIL);
        break;
    case 29:
        Print (ID_ERROR_62, FWU_FW_DELETE_FILE_FAIL);
        break;
    case FWU_PARTITION_LAYOUT_NOT_COMP:
        Print (ID_ERROR_63, FWU_PARTITION_LAYOUT_NOT_COMP);
        break;
    case FWU_DOWNGRADE_NOT_ALLOWED_DATA_MISMATCH:
        Print (L"\nDowngrade is NOT permitted due to mismatched data format version.\n");
        Print (ID_ERROR_64, FWU_DOWNGRADE_NOT_ALLOWED_DATA_MISMATCH);
        break;
    case FWU_UPDATE_PASSWORD_NOT_MATCHED:
        Print (ID_ERROR_65, FWU_UPDATE_PASSWORD_NOT_MATCHED);
        break;
    case FWU_UPDATE_PASSWORD_EXCEED_MAXIMUM_RETRY:
        Print (ID_ERROR_66, FWU_UPDATE_PASSWORD_EXCEED_MAXIMUM_RETRY);
        Print (ID_INFO_16);
        break;
    case FWU_PID_NOT_EXPECTED:
        Print (ID_ERROR_72, FWU_PID_NOT_EXPECTED);
        break;
    case FWU_FILE_ALREADY_EXISTS:
        Print (L"Error %d: File already exists\n", FWU_FILE_ALREADY_EXISTS);
        break;
    case FWU_FILE_INVALID:
        Print (L"Error %d: Invalid File\n", FWU_FILE_INVALID);
        break;
    case FWU_SAVE_RESTORE_POINT_ERROR:
        Print (L"\nError %d: Restore Point Image Failure\n", FWU_SAVE_RESTORE_POINT_ERROR);
        break;
    case FWU_GET_BLIST_ERROR:
        Print (L"\nError %d: Get Black List Failure\n" , FWU_GET_BLIST_ERROR);
        break;
    case FWU_GET_PATTRIB_ERROR:
        Print (L"\nError %d: Get Partition Attribute Failure\n" , FWU_GET_PATTRIB_ERROR);
        break;
    case FWU_GET_UPD_INFO_STATUS:
        Print (L"\nError %d: Get Update Info Status Failure\n" , FWU_GET_UPD_INFO_STATUS);
        break;
    case FWU_BUFFER_COPY_FAILED:
        Print (L"\nError %d:Buffer Copy Failure\n" , FWU_BUFFER_COPY_FAILED);
        break;
    default:
        Print (ID_ERROR_25, FWU_GENERAL);
        break;
    }
}

/*++

Routine Description:

Validate command line is present and remove current working directory if present

Arguments:

ImageHandle - handle for application
CommandLine - buffer of command line arguments passed during Load call

Returns:

GC_TODO: add return values

--*/


BOOLEAN
    ParseLoadOptions (
    EFI_HANDLE  ImageHandle,
    OUT CHAR16  **CommandLine
    )
{
    EFI_LOADED_IMAGE   *Image;
    EFI_STATUS         Status;

    //
    // Set defaults.
    //
    *CommandLine = NULL;

    Status = BS->HandleProtocol (
        ImageHandle,
        &LoadedImageProtocol,
        (VOID *) &Image
        );

    if (!EFI_ERROR (Status)) {
        CHAR16  *CmdLine;

        //
        // Make sure it is power of 2
        //
        UINT32  CmdSize;

        CmdLine = Image->LoadOptions;
        CmdSize = Image->LoadOptionsSize &~1;

        if (CmdLine && CmdSize) {
            //
            // Set command line pointer for caller
            //
            *CommandLine = CmdLine;

            //
            // See if current working directory was passed, if so skip it
            // and move to the first argument
            //
            while((*CmdLine != 0) && CmdSize) {
                CmdLine++;
                CmdSize -= sizeof (CHAR16);
            }

            return TRUE;
        }
    }

    return FALSE;
}

#define WHITESPACE( p ) (InQuote == FALSE && (*p == L' ' || *p == L'\t'))

EFI_STATUS
    ParseCommandLine (
    IN  EFI_HANDLE  ImageHandle,
    IN  CHAR16      *FileName,
    OUT BOOLEAN     *bAllowSV,
    OUT BOOLEAN     *bUsePassword,
    OUT BOOLEAN     *bPid,
    OUT BOOLEAN     *bF,
    OUT BOOLEAN     *bPdt,
    OUT BOOLEAN     *bIshVer
    )
    /*++

    Routine Description:

    Parse the command line

    Arguments:

    ImageHandle - handle for application
    CommandLine - buffer of command line arguments passed during Load call

    Returns:

    GC_TODO: add return values

    --*/
{
    BOOLEAN                   InQuote = FALSE;
    UINT32                    i;
    CHAR16                    *CommandLine;

    *bAllowSV = FALSE;
    *bUsePassword = FALSE;
    *bPid = FALSE;
    *bF   = FALSE;
    *bPdt = FALSE;
    *bIshVer = FALSE;

    if (ParseLoadOptions (ImageHandle, &CommandLine)) {
        //
        // Skip the 1st argument which should be us.
        //

        while (*CommandLine != L' ' && *CommandLine != 0)
        {
            CommandLine++;
        }

        //
        // Get to the beginning of the next argument.
        //
        while (WHITESPACE (CommandLine ))
        {
            CommandLine++;
        }

        if (*CommandLine == 0)
        {
            Print (L"Usage: FwUpdLclApp.efi /p /s /i /d /g <updateimage>\n");
            return EFI_INVALID_PARAMETER;
        }

        FileName[0] = '\0';

        while (*CommandLine)
        {
            //
            // Skip leading white space
            //
            if (WHITESPACE (CommandLine))
            {
                CommandLine += 1;
                continue;
            }

            if ((*CommandLine == '/') || (*CommandLine == '-'))
            {
                CommandLine += 1; // Next char
                //
                // Check char
                //
                switch (*CommandLine)
                {
                case 's':
                    *bAllowSV = TRUE;
                    break;

                case 'p':
                    *bUsePassword = TRUE;
                    break;

                case 'i':
                    *bPid = TRUE;
                    break;

                case 'f':
                    *bF = TRUE;
                    break;

                case 'd':
                    *bPdt = TRUE;
                    break;
                  
                case 'g':
                    *bIshVer = TRUE;
                    break;
                  
                default:
                    Print (L"Usage: FwUpdLclApp.efi /p /s /i /d /g <updateimage>\r\n");
                    return EFI_INVALID_PARAMETER;
                }
                CommandLine += 1; // Next char
            } else 
            {
                // Copy filename
                i = 0;
                while (*CommandLine != 0)
                {
                    FileName[i] = *CommandLine;
                    CommandLine++;
                    i++;
                }

                FileName[i] = '\0';
                break; 
            }
        }

        if (FileName[0] == '\0')
        {
            Print (L"Usage: FwUpdLclApp.efi /p /s /i /d /g <updateimage>\r\n");
            return EFI_INVALID_PARAMETER;
        }
    } else 
    {
        Print (L"Usage: FwUpdLclApp.efi /p /s /i /d /g <updateimage>\r\n");
        return EFI_INVALID_PARAMETER;
    }

    return EFI_SUCCESS;
}

EFI_STATUS
    GetUpdateImage (
    IN  EFI_HANDLE       ImageHandle,
    IN  CHAR16          *ImageName,
    OUT UINTN           *ImageLength,
    IN  UINT8           **ImageBuffer
    )
{
    EFI_STATUS                      Status;
    EFI_FILE_IO_INTERFACE           *Vol;
    EFI_FILE_INFO                   *FileInfo;
    EFI_FILE_HANDLE                 FileHandle;
    EFI_FILE_HANDLE                 RootFs;
    EFI_FILE_HANDLE                 CurDir;
    //UINTN                           ScratchBufferSize;
    unsigned long                            ScratchBufferSize;
    EFI_GUID                        FileInfoGuid = EFI_FILE_INFO_ID;
    EFI_LOADED_IMAGE                *LoadedImage;
    //
    // Open the ME FW Update Image
    //
    //Print(L"\n entered Getupdate Image Func ");
    Status = BS->HandleProtocol (
        ImageHandle,
        &LoadedImageProtocol,
        (VOID *) &LoadedImage
        );

    Status = BS->HandleProtocol (
        LoadedImage->DeviceHandle,
        &FileSystemProtocol,
        (VOID *) &Vol
        );

    Status  = Vol->OpenVolume (Vol, &RootFs);
    CurDir  = RootFs;
    //Print(L"\n entered CurDir open ");
    Status = CurDir->Open (
        CurDir,
        &FileHandle,
        ImageName,
        EFI_FILE_MODE_READ,
        0
        );

    if (EFI_ERROR (Status)) {
        DEBUG ((D_ERROR, "Unable to Open File - %r\n", Status));
        //Print (L"Cannot open update file!\r\n");
        return Status;
    }

    //
    // Figure out how big the file is
    //
    //ScratchBufferSize  = SIZE_OF_EFI_FILE_INFO + 1024;
    ScratchBufferSize  = 1024;
    //Print(L"\n file info scratchbufsize");
    FileInfo = (EFI_FILE_INFO *)AllocatePool (ScratchBufferSize);
    Status = FileHandle->GetInfo (
        FileHandle,
        &FileInfoGuid,
        &ScratchBufferSize,
        FileInfo
        );
    if (EFI_ERROR (Status)) {
        DEBUG ((D_ERROR, "Unable to GetInfo Buffer - %r\n", Status));
        //Print(L"Unable to GetInfo Buffer - %r\n", Status);
        FileHandle->Close (FileHandle);
        FreePool (FileInfo);
        return Status;
    }

    *ImageLength = (UINTN)FileInfo->FileSize;
    //Print("\n Image Length: %d",&ImageLength);
    FreePool (FileInfo);

    //
    // Read the update image into the ImageBuffer
    //
    //Print(L"Enter read update image into buffer ");
    *ImageBuffer = (UINT8 *)AllocatePool (*ImageLength);
    if (!ImageBuffer) {
        DEBUG ((D_ERROR, "Unable to allocate File Buffer - %r\n", Status));
        //Print (L"FwUpdLcl: Cannot allocate memory for Update Image!\r\n");
        Status = EFI_OUT_OF_RESOURCES;
        FileHandle->Close (FileHandle);
        return Status;
    }
    //Print(L"\n enter filehandle->Read");
    Status = FileHandle->Read (
        FileHandle,
        ImageLength,
        (VOID *)*ImageBuffer
        );
    if (EFI_ERROR (Status)) {
        DEBUG ((D_ERROR, "Unable to read file - %r\n", Status));
        //Print (L"FwUpdLcl: Cannot read Update Image into memory!\r\n");
        FileHandle->Close (FileHandle);
        FreePool (ImageBuffer);
        return Status;
    } 

    FileHandle->Close (FileHandle);
    //Print(L"\n Close \n ");

    return Status;
}

char uppercase(char s){
	
	if( s >=  'a' && s <= 'z')
	return (s = s +'A' - 'a');
		else
	return s;
	
}

INT32 stringcompare(const char *s1, const char *s2)
{
    if(NULL == s1 || NULL == s2)
    {
        return 1;
    }

    while (uppercase(*s1) == uppercase(*s2))
    {
        if (*s1 == 0)
            return 0;
        s1++;
        s2++;
    }
    return uppercase(*(unsigned const char *)s1) - uppercase(*(unsigned const char *)(s2));
}

EFI_DRIVER_ENTRY_POINT (InitializeFwUpdLclApplication)

    EFI_STATUS
    InitializeFwUpdLclApplication (
    IN EFI_HANDLE           ImageHandle,
    IN EFI_SYSTEM_TABLE     *SystemTable
    )
{
    EFI_STATUS        Status;
    CHAR16            ImageName[256];
    UINTN	            ImageLength = 0;
    UINT8             *ImageBuffer = NULL;
    BOOLEAN           bAllowSV;
    BOOLEAN           bUsePassword;
    BOOLEAN           bPid;
    BOOLEAN           bF;
    BOOLEAN           bPdt;
    BOOLEAN           bIshVer;
    //CHAR              Password[9];
    char              *Password = NULL;
    UINT32            FWUpdateStatus;
    DWORD             loops = 500;
    BOOLEAN           done = FALSE;
    UINT32            lastStatus = 0;
    UINT32            platCheck = 0;
    FWVersion         fwVersion;
    INT32             platCheckReturn = 0; 
    UINT32            CheckPolicyStatus = 0;
    UPDATE_TYPE       Upd_Type;
    VersionLib        ver;
    UINT32            index = 0;
    UINT32            status;
    UINT32            UpdateStatus = 0;
    UINT32            TotalStages = 0;
    UINT32            PercentWritten = 0;  
    CHAR8             symbol;
    UINT32            lastResetType;
    UPDATE_FLAGS_LIB  update_flags;
    UINT16            interfaces;
    int               timer30s = 0;
    unsigned int       indexMod;
    int                percentage0s = 0;
    int                percentdiff = 0;
    UINT32             ComparePartID = 0;
    UINT32             hexValueInstId = 0;
    IPU_UPDATED_INFO   IpuUpdatedInfo;
    UINT32             PartId = 0;
    UINT32             fwuError;
    FWU_GET_IPU_PT_ATTRB_MSG_REPLY FwuGetIpuAttrbMsgInfo;
    bool               found = false;
    UINT32             i, j = 0;
    UINT16 major = 0;
    UINT16 minor = 0;
    UINT16 hotfix = 0;
    UINT16 build = 0;
    UINT32 itr = 0;
	SKU_TYPE FwType;
    PCH_SKU sku;
	UINT32 continueLoop;
	int status2;
	//
    // Zero out the update flag structure
    //
    ZeroMem(&update_flags, sizeof(UPDATE_FLAGS_LIB));
    ZeroMem((char*)&IpuUpdatedInfo,sizeof(IPU_UPDATED_INFO));

    //
    // Initialize the EFI Toolkit Library. Set BS, RT, &ST globals
    //  BS = Boot Services RT = RunTime Services
    //  ST = System Table
    //
    InitializeLib (ImageHandle, SystemTable);


    Print (L"\n Intel (R) Firmware Update Utility Sample Application \n");
    Print (L"\n Intel (R) Firmware Update Utility Version: %d.%d.%d.", VER_MAJOR, VER_MINOR, VER_HOTFIX);
    Print (L"%d\n", VER_BUILD);

    Print (L"\n");

    Print (ID_INFO_1);

	
    //
    // Determine the command line arguments
    //
    Status = ParseCommandLine (ImageHandle, ImageName, &bAllowSV, &bUsePassword, &bPid, &bF, &bPdt, &bIshVer);
    if (EFI_ERROR (Status)) 
    {
        DEBUG ((D_ERROR, "Unable to process command line - %r\n", Status));
        return Status;
    }

    //
    // Display ISH FW Version with '/g' option
    //
    if (bIshVer){
        Status = GetPartVersion(FPT_PARTITION_NAME_ISHC, &major, &minor, &hotfix, &build);

        if (EFI_ERROR (Status)) 
        {
            DEBUG ((D_ERROR, "GetPartVersion Error %r\n", Status));
            return Status;
        }

        Print(L"ISH FW Version: %d.%d.%d.%d \n\n", major, minor, hotfix, build);
    }

    //
    // Load image into memory buffer
    //
    Print (L"\n Loading image into memory : ...\n");

    Status = GetUpdateImage (ImageHandle, ImageName, &ImageLength, &ImageBuffer);
    if (EFI_ERROR (Status)) {
        Print(L" %r ",Status);
        return Status;
    }

	if (bPdt)
	{
		Print(L"Sending Image for Executing PDT Update. \n");

		Status = HeciPdt((char *)ImageBuffer, (unsigned int)ImageLength);
		if (EFI_ERROR(Status)) {
			Print(L"Send Failed. \n");
		}
		else {
			Print(L"Send Succeeded. \n");
		}
	}
	else
	{
		//
		// Get the current status of the ME FWUpdate Client - verifies if the client is 
		// installed
		//

		if (GetLastStatus(&lastStatus))
		{
			Print (ID_ERROR_19, FWU_LAST_STATUS);
			return EFI_SUCCESS;
		}
		//
		// Is there a pending reset?
		//
		
		if (GetLastUpdateResetType (&lastResetType)) 
		{
			Print (ID_ERROR_19, FWU_LAST_STATUS);
			return EFI_SUCCESS;
		}
		if (STATUS_UPDATE_HOST_RESET_REQUIRED == lastStatus)
		{
			Print (ID_ERROR_51, FWU_REBOOT_NEEDED);
			return EFI_SUCCESS;
		}

		if (IsUpdateStatusPending (lastStatus))
		{
			Print (ID_ERROR_20, FWU_UPD_PROCESS);
			return EFI_SUCCESS;
		}

		switch (lastResetType) 
		{
		case MFT_PART_INFO_EXT_UPDATE_ACTION_HOST_RESET:

		case MFT_PART_INFO_EXT_UPDATE_ACTION_GLOBAL_RESET:
			Print (ID_ERROR_51, FWU_REBOOT_NEEDED);
			return EFI_SUCCESS;
			break;
		default:

			break;
		}


		Print (ID_INFO_3);

		//
		// Is update supported? 
		//
		if (GetInterfaces (&interfaces))
		{
			Print (ID_ERROR_19, FWU_LAST_STATUS);
			return EFI_SUCCESS;
		}

		switch (interfaces) 
		{
		case FW_UPDATE_DISABLED:	
			Print (L"Local FWUpdate is Disabled\n"); 
			return EFI_SUCCESS;
		case FW_UPDATE_PASSWORD_PROTECTED:
			Print (L"Local FWUpdate is Password Protected\n"); 
			break;
		case FW_UPDATE_ENABLED:
			break;
		default:
			break;
		}

		status2 = GetFwType((UINT32*)&FwType);
		//Print(L"Status: %d \n",status2);
	if (status2){
        Print(L"error");
	}
    else if(SKU_5_MB == FwType) {
		Print(L"corporate\n"); 
	}
    else if(SKU_1_5_MB == FwType){
		Print(L"consumer\n");
	}
    else if(SKU_RESERVED == FwType){
        Print(L"Reserved\n");
	}
    else{
        Print(L"invalid\n");
	}

	status2 = GetPchSKU((UINT32*)&sku);
	//Print(L"Status: %d \n",status2);
	
    if(status2){
        Print(L"error\n");
	}
    else if(PCH_SKU_LP == sku){
		Print(L"LP \n"); 
	}
	else if(PCH_SKU_H == sku){
		Print(L"H \n");
	}
    else{
        Print(L"invalid\n");
	}

		Print (L"\n Checking Firmware Parameters ... \n \n");
		CheckPolicyStatus = CheckPolicyBuffer((char *)ImageBuffer, (INT32)ImageLength, (INT32)bAllowSV, &Upd_Type, &ver);

		switch (Upd_Type) 
		{
		case DOWNGRADE_SUCCESS:

		case SAMEVERSION_SUCCESS:

		case UPGRADE_SUCCESS:
			break;

		case DOWNGRADE_FAILURE:
			Print (L"FW Update downgrade not allowed\n");
			return EFI_SUCCESS;
			break;
		case SAMEVERSION_FAILURE:
			Print (L"FW Update same version not allowed, specify /s on command line\n");
			return EFI_SUCCESS;
			break;
		default:
			break;
		}
    
		
		if(bPid)
		{
			Print(L"\n Executing ISH Partial FWUpdate");

			ComparePartID = FPT_PARTITION_NAME_ISHC; 
			//Print(L"\n compareID: 0x%x",ComparePartID);

			//Get Partition Attribute from Firmware
			if (FWU_ERROR_SUCCESS != (fwuError = GetExtendedIpuPartitionAttributes(&FwuGetIpuAttrbMsgInfo, FWU_IPU_UPDATE_OPERATION)))
			{
				DisplayTextForReturnErrorCode(fwuError);
				return fwuError;
			}

			PartId = 0;
		   //Loop through expected partitions from FW to find partition requested
			for(j=0;j<FwuGetIpuAttrbMsgInfo.NumOfPartition;j++)
			{
				  if(ComparePartID == FwuGetIpuAttrbMsgInfo.PtAttribute[j].PtNameId)
				  {
						PartId = FwuGetIpuAttrbMsgInfo.PtAttribute[j].PtNameId;
						found = true;
						break;
				  }
			}

			if(!found)
			{
			   DisplayTextForReturnErrorCode(FWU_PID_NOT_EXPECTED);
			   //Print(L"ParID: 0x%x\tInstId: 0x%x \n",ComparePartID,hexValueInstId);
				return FWU_PID_NOT_EXPECTED;
			}
			Print(L"%s", ID_WARN_0);
		  ////Actual Partial FW update
		  //
		  // Password hack for testing - replace with OEM version if password required
		  // 
			if (!bUsePassword) 
			{
				ZeroMem (Password, sizeof (Password));
			}
			if (bUsePassword)
			{
				 FWUpdateStatus = FwUpdatePartialBuffer ((char *)ImageBuffer, (unsigned int)ImageLength,PartId,0,&IpuUpdatedInfo,"P@ssw0rd",FWU_ENV_MANUFACTURING, mOemId, update_flags, &DisplaySendStatus);
			} 
			else
			{ 
				 FWUpdateStatus = FwUpdatePartialBuffer ((char *)ImageBuffer, (unsigned int)ImageLength,PartId,0,&IpuUpdatedInfo,Password, FWU_ENV_MANUFACTURING, mOemId, update_flags, &DisplaySendStatus);
			}

			if (FWU_ERROR_SUCCESS != FWUpdateStatus)
			{
				 DisplayTextForReturnErrorCode(FWUpdateStatus);
				if (ImageBuffer)
				{
					FreePool (ImageBuffer);
				}
				return EFI_SUCCESS;
			}

			if (ImageBuffer) 
			{
				 FreePool (ImageBuffer);
			}
		}else{
				//
				// Password hack for testing - replace with OEM version if password required
				// 
				Print(L"\n");
				Print(L"%s \n", ID_WARN_0);
				/*if (!bUsePassword) 
				{
					  ZeroMem (Password, sizeof (Password));
				}*/

				//if (bUsePassword)
				//{
					//  FWUpdateStatus = FwUpdateFullBuffer ((char *)ImageBuffer, (unsigned int)ImageLength, "P@ssw0rd", 0, FWU_ENV_MANUFACTURING, mOemId, update_flags, &DisplaySendStatus);
				//} 
				//else
				//{
					  FWUpdateStatus = FwUpdateFullBuffer ((char *)ImageBuffer, (unsigned int)ImageLength, Password, 0, FWU_ENV_MANUFACTURING, mOemId, update_flags, &DisplaySendStatus);
				//}

				if (FWU_ERROR_SUCCESS != FWUpdateStatus)
				{
					  DisplayTextForReturnErrorCode(FWUpdateStatus);
						if (ImageBuffer)
						{
							 FreePool (ImageBuffer);
						}
						return EFI_SUCCESS;
						//return FWUpdateStatus;
				}

				if (ImageBuffer) 
				{
					FreePool (ImageBuffer);
				}
		}

		// 
		// Image downloaded to FW Update Client
		// Now query the status of the update being applied
		//
		//Print(L"\n FW Update:  [  0 %% ]\r");

		//index  =  0;
		//
		// Loop through Polling for Fw Update Stages
		//
		Print(L"\n FW Update:  [  0 %% ]\r");

		index  =  0;
		//
		// Loop through Polling for Fw Update Stages
		//
		ProgressDot();
		do {
			//We mod4 the index to determine which ascii animation frame to display for this iteration.
			indexMod = (++index % 4);
			//symbol = (++index % 2 == 0)?'|': '-';
			switch(indexMod)
				//loop through (|) (/) (-) (\) (|) (/) ...
			{
			case CMD_LINE_STATUS_UPDATE_1: symbol =  '|';  break;
			case CMD_LINE_STATUS_UPDATE_2: symbol =  '/';  break;
			case CMD_LINE_STATUS_UPDATE_3: symbol =  '-';  break;
			case CMD_LINE_STATUS_UPDATE_4: symbol =  '\\'; break;
			}
			Status = FWUpdate_QueryStatus_Get_Response(&UpdateStatus, &TotalStages, &PercentWritten, &lastStatus, &lastResetType);
			if(PercentWritten > 100)
			{
				break;
			}
			if (Status == FWU_ERROR_SUCCESS)
			{
				Print (L"FW Update:  [ %d%% (%c)]\r" ,PercentWritten, symbol);
			} else if (lastStatus != STATUS_UPDATE_NOT_READY)
			{
				Print (L"\n");
				break; //break out of the loop
			}

			BS->Stall(100000); // Wait 1 sec before polling again
			if(timer30s >= 30) 
			{
				percentdiff = PercentWritten - percentage0s;
				if(percentdiff < 1)
				{
					//TODO: Add timeout when add cmdline option
					//Status = FWU_UPDATE_TIMEOUT;
				} else 
				{
					percentage0s = PercentWritten;
					timer30s = 0;
				}
			} else 
			{
				timer30s++;
			}
		} while ((PercentWritten != 100) && (Status == FWU_ERROR_SUCCESS)); 
	if (UpdateStatus != HECI1_CSE_GS1_PHASE_FWUPDATE)// we are NOT in FW update phase
			{
				if (lastStatus == STATUS_UPDATE_NOT_READY)// waiting for HECI message -> continue to wait
					continueLoop = 1;
				else 
					continueLoop = 0;
			}
			else //we are in FW update phase
			{
				if (PercentWritten == 100) //FWupdate proccess completed successfully -> exit
					continueLoop = 0;
				else 
				{
					if (lastStatus == STATUS_UPDATE_SUCCESS) //FWupdate proccess in progress (0-99%) -> continue loop
							continueLoop = 1;
					else // we got error from FW update (progress) registers e.g. 127 (0x7F) and not a value between 0-99%
					{
						if (lastStatus == STATUS_UPDATE_NOT_READY) // waiting for HECI message -> continue to wait
							continueLoop = 1;
						else
							continueLoop = 0;
					}
				}
			}
		} while (continueLoop);
		
		switch (Status)
		{
		case FWU_NO_MEMORY:

		case FWU_IME_NO_DEVICE:
			Print (ID_ERROR_68, FWU_UPDATE_POLLING_FAILED);
			return EFI_SUCCESS;
		case FWU_IME_NOT_READY:
			DisplayTextForReturnErrorCode(status);
			return EFI_SUCCESS;
		case FWU_ERROR_FW:
			Print (ID_ERROR_69, FWU_ERROR_FW, UpdateStatus);
			return EFI_SUCCESS;
		default:
			break;
		}

		switch (lastStatus) 
		{
		case STATUS_SUCCESS:
			if(PercentWritten == 99)
			{
				Print("L\nFW Update:  [  100 %% ]\r");
			}

			switch (lastResetType) 
			{
			case MFT_PART_INFO_EXT_UPDATE_ACTION_NONE:

			case MFT_PART_INFO_EXT_UPDATE_ACTION_CSE_RESET:
				Print (L"\nFW Update is completed successfully.\n");
				break;
			case MFT_PART_INFO_EXT_UPDATE_ACTION_HOST_RESET:

			case MFT_PART_INFO_EXT_UPDATE_ACTION_GLOBAL_RESET:
				Print (L"\nFW Update is complete and a reboot will run the new FW.\n");
				break;
			default:
				Print (L"\nFW Update is complete and a reboot will run the new FW.\n");
				break;
			}
			fwuError = FWU_ERROR_SUCCESS;
			break;
		case STATUS_UPDATE_IMAGE_INVALID:
			DisplayTextForReturnErrorCode(FWU_IMG_HEADER);
			break;
		case STATUS_UPDATE_INTEGRITY_FAILURE:
			DisplayTextForReturnErrorCode(FWU_SGN_MISMATCH);
			break;
		case STATUS_UPDATE_SKU_MISMATCH:
			DisplayTextForReturnErrorCode(FWU_SKU_MISMATCH);
			break;
		case STATUS_UPDATE_FW_VERSION_MISMATCH:
			DisplayTextForReturnErrorCode(FWU_VER_MISMATCH);
			break;
		case STATUS_UPDATE_GENERAL_FAILURE:
			DisplayTextForReturnErrorCode(FWU_GENERAL);
			break;
		case STATUS_UPDATE_OUT_OF_RESOURCES:
			DisplayTextForReturnErrorCode(FWU_NO_MEMORY);
			break;
		case STATUS_UPDATE_AUDIT_POLICY_FAILURE:
			DisplayTextForReturnErrorCode(FWU_AUDIT_POLICY_FAILURE);
			break;
		case STATUS_UPDATE_ERROR_CREATING_FT:
			DisplayTextForReturnErrorCode(FWU_ERROR_CREATING_FT);
			break;
		case STATUS_UPDATE_SAL_NOTIFICATION_ERROR:
			DisplayTextForReturnErrorCode(FWU_SAL_NOTIFICATION_ERROR);
			break;
		case STATUS_INVALID_OEM_ID:
			DisplayTextForReturnErrorCode(FWU_INVALID_OEM_ID);
			break;
		case STATUS_DOWNGRADE_NOT_ALLOWED_VCN_RESTRICTION:
			DisplayTextForReturnErrorCode(FWU_IMAGE_UNDER_VCN);
			break; 
		case STATUS_DOWNGRADE_NOT_ALLOWED_SVN_RESTRICTION: 
			Print("FW downgrade is not allowed due to SVN restriction.\n");
			break;
		case STATUS_UPDATE_IMAGE_BLACKLISTED:
			Print("FW update/downgrade is not allowed to the supplied FW image.\n");
			break;
		default:
			DEBUG ((D_ERROR, "lastStatus = %d\n",lastStatus));
			DisplayTextForReturnErrorCode(FWU_GENERAL);
			break;
		}
		return EFI_SUCCESS;
	
}
