import json
import logging
import re

from utils.utils import (
    bytes_to_int,
    int_to_bytes,
    hex_string_to_int,
    extract_bits_from_byte,
    bitwise_expression_calc
)

logger = logging.getLogger(__name__)


class BaseItem(object):
    """Base class for all items
    """

    def __init__(self, data=None, **kwargs):
        """
        Arguments:
            data {bytes} -- Item bytes from CSE FW image
            kwargs {dict} -- Describes the data_bytes
        """
        members = {}
        members.update(kwargs)
        self.offset, self.size, self.value = self.set_item_data(data, members)

    def set_item_data(self, data, data_desc):
        """[summary]

        Arguments:
            data {bytes} -- data as bytes
            data_desc {dict} -- dict that describes the data

        Returns:
            [tuple] -- offset, size and bytes
        """
        int_offset = hex_string_to_int(data_desc.get('offset'))
        if isinstance(data_desc.get('size'), dict):
            from cse_image_items.DataItem import DataItem
            int_size = DataItem(data[int_offset:], **data_desc['size']).int_value()
        else:
            int_size = hex_string_to_int(data_desc.get('size'))
        item_data = data[int_offset:int_offset + int_size]
        if data_desc.get('bits'):
            item_data = self.extract_bits(item_data, data_desc['bits'])
        if data_desc.get('bitwiseOperation'):
            temp = bitwise_expression_calc(bytes_to_int(item_data), data_desc['bitwiseOperation'])
            item_data = int_to_bytes(temp)
        return int_offset, int_size, item_data

    def extract_bits(self, data, bits):
        r = re.compile(r'\d{1,}:\d{1,}')
        if not bool(r.match(bits)):
            start = bits
            end = int(start) + 1
        else:
            end, start = bits.split(':')
        return extract_bits_from_byte(data, int(start), int(end))

    def bytes_value(self, byteorder='little'):
        """Getter for item bytes

        Keyword Arguments:
            byteorder {str} -- byteorder (default: {'little'})

        Returns:
            bytes -- item bytes
        """
        if not (byteorder == 'little' or byteorder == 'big'):
            logger.exception('byteorder [%s] must be little or big' % (byteorder))
            raise KeyError('byteorder [%s] must be little or big' % (byteorder))
        return self.value if byteorder == 'little' else self.value[::-1]

    def int_value(self):
        """Getter for item bytes as integer

        Returns:
            int -- item bytes as integer
        """
        return bytes_to_int(self.bytes_value())

    def hex_value(self):
        """Getter for item bytes as hex string

        Returns:
            str -- item bytes as hex string
        """
        return hex(self.int_value())

    def bool_value(self):
        """Getter for item bytes as boolean

        Returns:
            bool -- item bytes as boolean
        """
        return self.int_value() != 0

    def str_value(self):
        """Getter for item bytes as string

        Returns:
            str -- item bytes as string
        """
        return self.bytes_value().decode('unicode_escape').rstrip('\x00')
