import json
import logging
from jsonschema import validate
from cse_image_items.BaseItem import BaseItem
from cse_image_items.DataItem import DataItem
from utils.utils import bytes_to_int, hex_string_to_int


logger = logging.getLogger(__name__)
EXTENSION_JSON = 'layout/extension.json'
CLIENT_SYSTEM_INFO_EXTENSION_JSON = 'layout/client_system_info_extension.json'
SKU_TYPES_JSON = 'layout/data/sku_types.json'
EXTENSION_SCHEMA_JSON = 'layout/JSON schemas/extension.json'
CLIENT_SYSTEM_INFO_EXTENSION_SCHEMA_JSON = 'layout/JSON schemas/client_system_info_extension.json'
SKU_TYPES_SCHEMA_JSON = 'layout/data/JSON schemas/sku_types.json'
CLIENT_SYSTEM_INFO_EXTENSION_ID = '0xc'


class ManifestExtensionFactory(object):
    @staticmethod
    def create(value, offset):
        logger.info('START ManifestExtensionFactory.create FUNCTION')

        extension_type = hex(bytes_to_int(value[offset:offset + 4]))
        if extension_type == CLIENT_SYSTEM_INFO_EXTENSION_ID:
            return ClientSystemInfoExtensionExtension(data=value, offset=hex(offset))
        return ManifestExtension(data=value, offset=hex(offset))


class ManifestExtension(BaseItem):
    def __init__(self, data, **kwargs):
        logger.info('START ManifestExtension.__init__ FUNCTION')

        try:
            with open(EXTENSION_JSON) as extension_json:
                extension = json.load(extension_json)
            with open(EXTENSION_SCHEMA_JSON) as extension_schema_json:
                extension_schema = json.load(extension_schema_json)
            validate(instance=extension, schema=extension_schema)
        except Exception:
            logger.exception('Exception occurred while reading manifest extension JSON file')
            raise Exception('Exception occurred while reading manifest extension JSON file')
        extension['offset'] = kwargs['offset']
        super(ManifestExtension, self).__init__(data, **extension)


class ClientSystemInfoExtensionExtension(BaseItem):
    def __init__(self, data, **kwargs):
        logger.info('START ClientSystemInfoExtensionExtension.__init__ FUNCTION')

        try:
            with open(CLIENT_SYSTEM_INFO_EXTENSION_JSON) as client_system_info_extension_json:
                client_system_info_extension = json.load(client_system_info_extension_json)
            with open(CLIENT_SYSTEM_INFO_EXTENSION_SCHEMA_JSON) as client_system_info_extension_schema_json:
                client_system_info_extension_schema = json.load(
                    client_system_info_extension_schema_json)
            validate(instance=client_system_info_extension,
                     schema=client_system_info_extension_schema)
        except Exception:
            logger.exception(
                'Exception occurred while reading client system info manifest extension JSON file')
            raise Exception(
                'Exception occurred while reading client system info manifest extension JSON file')
        try:
            with open(SKU_TYPES_JSON) as sku_types_json:
                sku_types = json.load(sku_types_json)
            with open(SKU_TYPES_SCHEMA_JSON) as sku_types_schema_json:
                sku_types_schema = json.load(sku_types_schema_json)
            validate(instance=sku_types, schema=sku_types_schema)
        except Exception:
            logger.info('Exception occurred while reading sku types JSON file')
            raise Exception('Exception occurred while reading sku types JSON file')

        client_system_info_extension['offset'] = kwargs['offset']
        super(ClientSystemInfoExtensionExtension, self).__init__(
            data, **client_system_info_extension)
        sku_type = DataItem(
            self.value, **client_system_info_extension['FW_SKU_attributes']['firmware_sku'])
        self.sku_type = sku_types[sku_type.hex_value()]
        self.is_LP = DataItem(
            self.value, **client_system_info_extension['FW_SKU_Capabilities']['is_LP'])
        self.is_H = DataItem(
            self.value, **client_system_info_extension['FW_SKU_Capabilities']['is_H'])
