import struct
import math
import logging


INT_LEN = 4
BYTE_SIZE = 8

logger = logging.getLogger(__name__)
OPERATIONS = ['>>', '<<', '|', '&', '^']


def hex_string_to_int(hex_string):
    """Convets hex string to int

    Arguments:
            hex_string {str} -- hex string (e.g. "0xff")

    Returns:
        {int} --- int value of hex_str
    """
    try:
        return int(hex_string, 16)
    except Exception:
        logger.exception('hex_string [%s] must be hex string' % (hex_string))
        raise Exception('hex_string [%s] must be hex string' % (hex_string))


def bytes_to_int(data, byteorder='little'):
    """Convets bytes to int

    Arguments:
        data {bytes} -- bytes item (e.g. b'\x00')

    Keyword Arguments:
        byteorder {str} -- byteorder: little/big (default: {'little'})

    Returns:
        {int} -- int value of data
    """
    if not (byteorder == 'little' or byteorder == 'big'):
        logger.exception('byteorder [%s] must be little or big' % (byteorder))
        raise KeyError('byteorder [%s] must be little or big' % (byteorder))
    try:
        bytes_length = int(len(data))
        if bytes_length % INT_LEN != 0 or bytes_length == 0:
            if byteorder == 'little':
                data = data + (INT_LEN - bytes_length) * b'\x00'
            else:
                data = (INT_LEN - bytes_length) * b'\x00' + data
        int_length = int(len(data) / INT_LEN)
        if byteorder == 'big':
            s = struct.Struct('>' + int_length * 'I')
        else:
            s = struct.Struct('<' + int_length * 'I')
        return s.unpack(data)[0]
    except Exception:
        logger.exception('data must be bytes')
        raise Exception('data must be bytes')


def int_to_bytes(integer, byteorder='little'):
    """Convets int to bytes

    Arguments:
        integer {int} -- int to convert

    Keyword Arguments:
        byteorder {str} -- byteorder: little/big (default: {'little'})

    Returns:
        {bytes} -- bytes value of integer
    """

    if not (byteorder == 'little' or byteorder == 'big'):
        logger.exception('byteorder [%s] must be little or big' % (byteorder))
        raise KeyError('byteorder [%s] must be little or big' % (byteorder))
    try:
        length = ((integer.bit_length() + 7) // 8) // 4 or 1  # adjust to full byte
        if byteorder == 'big':
            s = struct.Struct('>' + length * 'I')
        else:
            s = struct.Struct('<' + length * 'I')
        return s.pack(integer)
    except Exception:
        logger.exception('integer [%s] must be integer' % (integer))
        raise Exception('integer [%s] must be integer' % (integer))


def extract_bits_from_int(integer, start, end):
    """Extract bytes from specified range 

    Arguments:
        integer {int} -- value to extract from
        start {int} -- start bit
        end {int} -- end bit

    Returns:
        {int} -- extracted bits as int
    """
    if not (all(isinstance(_, int) for _ in [start, end]) and start < end and start >= 0 and end <= 16 * 8):
        logger.exception('start [%s] and end [%s] range is not valid' % (start, end))
        raise KeyError('start [%s] and end [%s] range is not valid' % (start, end))
    try:
        return (integer >> start) & ((1 << end - start) - 1)
    except Exception:
        logger.exception('integer [%s] must be integer' % (integer))
        raise Exception('integer [%s] must be integer' % (integer))


def extract_bits_from_byte(data, start, end):
    """Extract bytes from specified range 

    Arguments:
        data {bytes} -- value to extract from
        start {int} -- start bit
        end {int} -- end bit

    Returns:
        {bytes} -- extracted bits as bytes
    """
    if not (all(isinstance(_, int) for _ in [start, end]) and start < end and start >= 0 and end <= len(data) * BYTE_SIZE):
        logger.exception('start [%s] and end [%s] range is not valid' % (start, end))
        raise KeyError('start [%s] and end [%s] range is not valid len[%s]' %
                       (start, end, len(data)))
    data = bytes_to_int(data)
    extracted = extract_bits_from_int(data, start, end)
    return int_to_bytes(extracted)


def get_bitwise_result(integer, bitwise_operation, integer_operation):
    if bitwise_operation not in OPERATIONS:
        logger.error('invalid bitwise_operation [%s]' % (bitwise_operation))
        raise KeyError('invalid bitwise_operation [%s]' % (bitwise_operation))
    return {'>>': integer >> integer_operation,
            '<<': integer << integer_operation,
            '|': integer | integer_operation,
            '&': integer & integer_operation,
            '^': integer ^ integer_operation}.get(bitwise_operation)


def extract_bitwise_operation(integer, expression_list):
    if len(expression_list) == 2:
        return get_bitwise_result(integer, expression_list[0], hex_string_to_int(expression_list[1]))
    elif len(expression_list) > 2:
        result = get_bitwise_result(
            integer, expression_list[0], hex_string_to_int(expression_list[1]))
        return extract_bitwise_operation(result, expression_list[2:])
    else:
        logger.error('invalid expression %s' % (expression_list))
        raise KeyError('invalid expression %s' % (expression_list))


def bitwise_expression_calc(integer, expression):
    expression_list = expression.split(' ')
    expression_list = [x for x in expression_list if x != '']
    return extract_bitwise_operation(integer, expression_list)
