/** @file
  Debug for Dump Memory Map.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 2016 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

//
// Undefined DEBUG flag to enable DEBUG MACRO
//

#undef MDEPKG_NDEBUG

#include <PiDxe.h>
#include <Library/DebugLib.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Library/MemoryAllocationLib.h>

/**
  This function is for debug purpose to dump memory map in S4 resume.
  1. This function call must be called by last ExitBootServices callback
     to capture all potential memory map changes even allocated in callback phase.
     User has to decide where to call it.
  2. The funtion call could be placed in any driver with ExitBootServices callback
     to minimized the FV size changed as smaller as possible.
  3. The driver with the function call should enable the debug function.
     Update following text in dsc file to enable serial debug message for the driver in release mode.
     $(DRIVER_PATH)/Driver.inf {
       <LibraryClasses>
         DebugLib|MdePkg/Library/BaseDebugLibSerialPort/BaseDebugLibSerialPort.inf
         SerialPortLib|MdeModulePkg/Library/BaseSerialPortLib16550/BaseSerialPortLib16550.inf
         PlatformHookLib|$(PLATFORM_PACKAGE)/Library/BasePlatformHookLib/BasePlatformHookLib.inf
       <PcdsFixedAtBuild>
         gEfiMdePkgTokenSpaceGuid.PcdDebugPropertyMask|0x2F
        gEfiMdePkgTokenSpaceGuid.PcdDebugPrintErrorLevel|0x80080046
     }

  @param  None

  @retval None
**/
VOID
DumpMemMap (
  VOID
  )
{
  EFI_MEMORY_DESCRIPTOR           *MemoryMap;
  EFI_MEMORY_DESCRIPTOR           *MemDescriptorPtr;
  UINTN                           MemoryMapSize;
  UINTN                           MemoryMapDescriptorSize;

  EFI_STATUS                      Status;
  UINTN                           Index;
  UINTN                           MapKey;
  UINT32                          MemoryMapDescriptorVersion;
  EFI_MEMORY_DESCRIPTOR           TmpMemoryMap;

  MemoryMapSize = 1;
  Status = gBS->GetMemoryMap (&MemoryMapSize, &TmpMemoryMap, &MapKey, &MemoryMapDescriptorSize, &MemoryMapDescriptorVersion);
  ASSERT (Status == EFI_BUFFER_TOO_SMALL);

  //
  // Enlarge space here, because we will allocate pool now.
  //
  MemoryMapSize += EFI_PAGE_SIZE;
  MemoryMap = (EFI_MEMORY_DESCRIPTOR *) AllocateZeroPool (MemoryMapSize);
  ASSERT(MemoryMap != NULL);

  //
  // Get System MemoryMap
  //
  Status = gBS->GetMemoryMap (&MemoryMapSize, MemoryMap, &MapKey, &MemoryMapDescriptorSize, &MemoryMapDescriptorVersion);
  ASSERT_EFI_ERROR (Status);

  MemDescriptorPtr = MemoryMap;

  DEBUG ((DEBUG_INFO, "Type  Base        Pages\n"));
  DEBUG ((DEBUG_INFO, "----------------------------\n"));
  for (Index = 0; Index < (MemoryMapSize/MemoryMapDescriptorSize); Index++) {
    DEBUG ((DEBUG_INFO, "0x%02x  0x%08x  0x%08x\n", MemDescriptorPtr->Type, MemDescriptorPtr->PhysicalStart, MemDescriptorPtr->NumberOfPages));

    MemDescriptorPtr = (EFI_MEMORY_DESCRIPTOR *)((UINT8 *) MemDescriptorPtr + MemoryMapDescriptorSize);
  }

  FreePool (MemoryMap);
}
