/** @file
  Implementation of PeiBoardConfigLib.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2015 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <PiPei.h>
#include <Library/DebugLib.h>
#include <Library/PcdLib.h>
#include <Library/PostCodeLib.h>
#include <Library/TimerLib.h>
#include <Library/EcMiscLib.h>
#include <Library/PeiDxeBoardIdsLib.h>
#include <Library/GpioLib.h>

#include <PlatformBoardType.h>
#include <PlatformBoardId.h>
#include <PlatformBoardConfig.h>
#include <OemSetup.h>
#include <SetupVariable.h>
#include <Ppi/ReadOnlyVariable2.h>
#include <Library/PeiServicesLib.h>
#include <Library/SiliconInitLib.h>
#include <Library/PmcLib.h>
#include <Guid/BootStateCapsule.h>
#include <PlatformDpInPcdConfig.h>

#define STALL_TIME                          1000000    // 1,000,000 microseconds = 1 second

/**
  Returns the BoardId ID of the platform from the EC for Mobile.

  @param[in]  BoardId           BoardId ID as determined through the EC.

  @retval     EFI_SUCCESS       The function completed successfully.
  @retval     EFI_DEVICE_ERROR  EC fails to respond.
**/
EFI_STATUS
EFIAPI
InternalGetBoardId (
  OUT UINT16          *BoardId
  )
{
  EFI_STATUS    Status;
  BOARD_ID_INFO EcBoardInfo;

  Status = EFI_SUCCESS;

  //
  // Return BoardIdUnknown1 in case of error.
  //
  *BoardId = BoardIdUnknown1;

  EcBoardInfo.Raw = 0;
  Status = GetBoardInfo ((UINT16 *)&EcBoardInfo);
  DEBUG ((DEBUG_INFO, "Raw Board ID from EC is 0x%x\n", EcBoardInfo.Raw));

  if (Status == EFI_SUCCESS) {
    *BoardId = (UINT16)EcBoardInfo.TglRvpFields.BoardId;
    *BoardId &= 0x03F;
    PcdSet16S (PcdBoardId, (UINT16) (EcBoardInfo.TglRvpFields.BoardId));
    PcdSet16S (PcdBoardBomId, (UINT16) (EcBoardInfo.TglRvpFields.BomId));
    PcdSet16S (PcdBoardRev, (UINT16) (EcBoardInfo.TglRvpFields.FabId));
    PcdSetBoolS (PcdSpdPresent, (BOOLEAN) (EcBoardInfo.TglRvpFields.SpdPresent));
    DEBUG ((DEBUG_INFO, "Fields.BoardId from EC is 0x%x\n", PcdGet16 (PcdBoardId)));
    DEBUG ((DEBUG_INFO, "Fields.BomId from EC is 0x%x\n", PcdGet16 (PcdBoardBomId)));
    DEBUG ((DEBUG_INFO, "Fields.FabId from EC is 0x%x\n", PcdGet16 (PcdBoardRev)));
    DEBUG ((DEBUG_INFO, "Fields.SpdPresent from EC = %x\n", PcdGetBool (PcdSpdPresent)));
  } else {
    DEBUG ((DEBUG_ERROR, "Failed to get Board ID from EC\n"));
  }

  return Status;
}

VOID
EFIAPI
InternalUpdateRvpBoardConfig (
  IN OUT UINT16         BoardId
  )
{
  UINTN         Size;
  UINT8         BoardType;
  UINT8         PlatformType;
  UINT8         PlatformFlavor;

  if (BoardId < mSizeOfmBoardIndex) {
    Size = StrSize (mBoardIdIndex[BoardId]);
    PcdSetPtrS (PcdBoardName, &Size, mBoardIdIndex[BoardId]);
  }

  //
  // Update Board Type/Platform Type/Platform Flavor
  //
  switch (BoardId) {
  //
  // @ToDo TGL Board Porting
  //
    case BoardIdTglUDdr4:
    case BoardIdTglUDdr4Erb:
    case BoardIdTglULp4Aep:
    case BoardIdTglULp4Gcs:
    case BoardIdTglULp4Dg1Aep:
    case BoardIdTglULp4Type4:
    case BoardIdTglULp4Exs:
    case BoardIdTglYLp4Type4:
    case BoardIdTglYLp4Type4Crb:
    case BoardIdTglYExsLp4Type4:
    case BoardIdTglYLp5Type4:
      BoardType = BoardTypeRvp;
      PlatformType = TypeUltUlx;
      PlatformFlavor = FlavorMobile;
      if(PcdSet64S(PcdAcpiDefaultOemTableId, ACPI_OEM_TABLE_ID_TGL_ULT) != EFI_SUCCESS)
      {
         DEBUG ((DEBUG_INFO, "Set PcdAcpiDefaultOemTableId error!!!\n"));
      }
      break;
    case BoardIdTglHDdr4SODimm:
      BoardType = BoardTypeRvp;
      PlatformType = TypeTrad;
      PlatformFlavor = FlavorMobile;
      if(PcdSet64S(PcdAcpiDefaultOemTableId, ACPI_OEM_TABLE_ID_TGL_ULT) != EFI_SUCCESS)
      {
         DEBUG ((DEBUG_INFO, "Set PcdAcpiDefaultOemTableId error!!!\n"));
      }
      break;

    default:
      BoardType = BoardTypeRvp;
      PlatformType = TypeUnknown;
      PlatformFlavor = FlavorUnknown;
      break;
  }

  PcdSet8S (PcdBoardType, BoardType);
  PcdSet8S (PcdPlatformType, PlatformType);
  PcdSet8S (PcdPlatformFlavor, PlatformFlavor);

  DEBUG ((DEBUG_INFO, "PcdAcpiDefaultOemTableId is 0x%llX\n", PcdGet64 (PcdAcpiDefaultOemTableId)));
}

/**
  This function call checks if valid Board information is available in the stored region
  on the previous boot and restores them.

  @retval    TRUE    Valid Board information is available and restored successfully.
  @retval    FALSE   No valid Board information found.
**/
BOOLEAN
IsValidBoardInformationStored (
  VOID
  )
{
  EFI_STATUS                        Status;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI  *VariablePpi;
  UINTN                             Size;
  SETUP_DATA                        SetupData;

  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **)&VariablePpi
             );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_INFO, "IsValidBoardInformationStored : Read Only Variable PPI is not found.\n"));
    return FALSE;
  }

  //
  // 1. Check if Board information variables are available in NV storage.
  //    @todo decouple Board information variables from SETUP_DATA.
  Size = sizeof (SETUP_DATA);
  Status = VariablePpi->GetVariable (
                          VariablePpi,
                          L"Setup",
                          &gSetupVariableGuid,
                          NULL,
                          &Size,
                          &SetupData
                          );
  if (!EFI_ERROR (Status) && (SetupData.BoardId != BoardIdUnknown1) && (SetupData.BoardId != 0)) {
    DEBUG ((DEBUG_INFO, "IsValidBoardInformationStored : Found Board information variables in NV storage.\n"));
  } else {
    DEBUG ((DEBUG_INFO, "IsValidBoardInformationStored : Board information variables are not stored in NV storage.\n"));
    return FALSE;
  }

  //
  // 2. Check if this is NOT the first boot after BIOS/EC update by Capsule update.
  //
#if FixedPcdGetBool(PcdCapsuleEnable) == 1
  {
    BOOLEAN  BootStateAfterCapsule;

    Size = sizeof (BOOLEAN);
    BootStateAfterCapsule = TRUE;
    Status = VariablePpi->GetVariable (
                            VariablePpi,
                            BOOT_STATE_AFTER_CAPSULE_VARIABLE_NAME,
                            &gBootStateAfterCapsuleGuid,
                            NULL,
                            &Size,
                            &BootStateAfterCapsule
                            );
    if (EFI_ERROR (Status) || BootStateAfterCapsule) {
      DEBUG ((DEBUG_INFO, "IsValidBoardInformationStored : BootStateAfterCapsule variable is not present or TRUE.\n"));
      DEBUG ((DEBUG_INFO, "This is the very first boot after BIOS flash or Capsule update for EC/BIOS.\n"));
      return FALSE;
    }
  }
#endif

  //
  // 3. Check if this is NOT the first boot after RTC clear.
  //    Technically Board information variables are not impacted by RTC clear, but check for fail safe to force to read EC.
  if (!PmcIsRtcBatteryGood ()) {
    DEBUG ((DEBUG_INFO, "IsValidBoardInformationStored : This is the very first boot after RTC clear.\n"));
    return FALSE;
  }

  /*
    Note : The check logic #1-#3 do not capture the following scenario. Recommendation is to do RTC clear to support those.
     - EC is updated by flash programming tool.
     - HW rework is performed on the board for BomId, FabId or BoardID update.
  */

  DEBUG ((DEBUG_INFO, "IsValidBoardInformationStored : Valid Board information is available.\n"));

  //
  // Restore the valid Board information to PCDs
  //
  DEBUG ((DEBUG_INFO, "IsValidBoardInformationStored : Restoring Board information.\n"));
  PcdSetBoolS(PcdEcPresent, SetupData.EcPresent);
  PcdSet8S(PcdEcEspiFlashSharingMode, SetupData.EcEspiFlashSharingMode);
  PcdSet8S(PcdEcMajorRevision, SetupData.EcMajorRevision);
  PcdSet8S(PcdEcMinorRevision, SetupData.EcMinorRevision);

  DEBUG((DEBUG_INFO, "  +==============================================+\n"));
  DEBUG((DEBUG_INFO, "  | EC Major Revision: %02X  EC Minor Revision: %02X |\n", SetupData.EcMajorRevision, SetupData.EcMinorRevision));
  DEBUG((DEBUG_INFO, "  +==============================================+\n"));

  PcdSet16S(PcdBoardId, SetupData.BoardId);
  PcdSet16S(PcdBoardBomId, SetupData.BoardBomId);
  PcdSet16S(PcdBoardRev, SetupData.FabId);
  PcdSetBoolS(PcdSpdPresent, SetupData.SpdPresent);
  PcdSet8S(PcdPlatformGeneration, SetupData.PlatformGeneration);

  DEBUG((DEBUG_INFO, "  BoardId is 0x%x\n", PcdGet16(PcdBoardId)));
  DEBUG((DEBUG_INFO, "  BomId is 0x%x\n", PcdGet16(PcdBoardBomId)));
  DEBUG((DEBUG_INFO, "  FabId is 0x%x\n", PcdGet16(PcdBoardRev)));
  DEBUG((DEBUG_INFO, "  SpdPresent = %x\n", PcdGetBool(PcdSpdPresent)));

  return TRUE;
}

/**
  Procedure to detect current board HW configuration.

**/
VOID
GetBoardConfig (
  VOID
  )
{
  UINT16          BoardId;
  RETURN_STATUS   Status;
  UINT8           DataBuffer[4];
  UINT8           Retry;
  UINT8           EcData;

  //
  // Get Platform Info and fill the PCD
  //

  BoardId   = BoardIdUnknown1;
  PcdSet8S (PcdBoardType, BoardTypeMax);
  PcdSetBoolS (PcdEcPresent, FALSE);

  EarlyLpcIoDecode();
  Status = WakeUpEc ();
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Wake Up Ec Fail. Status = %r \n", Status));
  }

  if (!IsValidBoardInformationStored ()) {
    //
    // Detect EC Revision
    //
    DEBUG((DEBUG_INFO, "Reading EC to set Board information to PCDs \n"));
    Retry = 5;
    do {
      Status = DetectEcSupport ((UINT8 *)DataBuffer);
      Retry--;
      if ((RETURN_ERROR (Status)) && (Retry != 0)) {
        MicroSecondDelay (STALL_TIME);
      }
    } while ((RETURN_ERROR (Status)) && (Retry != 0));
    if (RETURN_ERROR (Status)) {
      PcdSetBoolS (PcdEcPresent, FALSE);
    } else {
      // Byte3[4:3] 0 = G3
      //            1 = SAF
      //            2 = MAF
      PcdSet8S (PcdEcEspiFlashSharingMode,(DataBuffer[3]>>3)&0x3);

      PcdSetBoolS (PcdEcPresent, TRUE);
      Status = DetectEcRevision ((UINT8 *)DataBuffer);
      PcdSet8S (PcdEcMajorRevision, DataBuffer[0]);
      PcdSet8S (PcdEcMinorRevision, DataBuffer[1]);

      DEBUG ((DEBUG_INFO, "+==============================================+\n"));
      DEBUG ((DEBUG_INFO, "| EC Major Revision: %02X  EC Minor Revision: %02X |\n", DataBuffer[0], DataBuffer[1]));
      DEBUG ((DEBUG_INFO, "+==============================================+\n"));
    }
    //
    // get RVP board ID
    //
    if (PcdGetBool (PcdEcPresent)) {
      // get board ID from EC
      DEBUG ((DEBUG_INFO, "Reading Board ID from EC ...\n"));
      do {
        Status = InternalGetBoardId (&BoardId);
      } while (Status != EFI_SUCCESS);
      DEBUG ((DEBUG_INFO, "Got Board ID from EC: 0x%x  \n", BoardId));
    } else {
      // @todo: get board ID on an EC less board. hard code BoardIdUnknown1 for temp use
      PcdSet16S(PcdBoardId, BoardIdUnknown1);
    }
  }

  BoardId = PcdGet16 (PcdBoardId);
  //
  // update RVP board config
  //
  InternalUpdateRvpBoardConfig (BoardId);

  //
  // check dock status if support
  //
  if ((PcdGetBool (PcdEcPresent)) && (PcdGet8 (PcdPlatformFlavor) == FlavorMobile)) {
    Status = GetPcieDockStatus (&EcData);
    if ((Status == EFI_SUCCESS) && (EcData & EC_B_DOCK_STATUS_ATTACH)) {
      PcdSetBoolS (PcdDockAttached, TRUE);
    }
  }

  DEBUG ((DEBUG_INFO, "Platform Information:\nPlatformType: %x\nBoardName: %s\n", PcdGet8 (PcdPlatformType), PcdGetPtr (PcdBoardName)));
  DEBUG ((DEBUG_INFO, "PlatformFlavor: %x\nBoardID: 0x%x\n", PcdGet8 (PcdPlatformFlavor), BoardId));
  DEBUG ((DEBUG_INFO, "BoardRev: %x\nBoardBomId: %x\nBoardType: %x\n", PcdGet16 (PcdBoardRev), PcdGet16 (PcdBoardBomId), PcdGet8 (PcdBoardType)));
  DEBUG ((DEBUG_INFO, "PlatformGeneration: %x\nEcPresent: %d\nSpdPresent: %d\n", PcdGet8 (PcdPlatformGeneration), PcdGetBool (PcdEcPresent), PcdGetBool (PcdSpdPresent)));

  //
  // Halt the system if BoardID unknown
  //
  if (BoardId == BoardIdUnknown1) {
    UINT32 code = 0;

    switch (PcdGet8 (PcdBoardType)) {
      case BoardTypeRvp:
        code = 0xBD10;
        break;
      case BoardTypeSv:
        code = 0xBD20;
        break;
      default:
        code = 0xBD30;
        break;
    }
    DEBUG ((DEBUG_ERROR, "Platform BoardID unknown, set PostCode=0x%x and halt the system !\n", code));

    PostCode (code);
    CpuDeadLoop ();
  }

}

/**
  Count the number of GPIO settings in the Table.

  @param[in]  GpioTable   The pointer of GPIO config table
  @param[out] GpioCount   The number of GPIO config entries
**/
VOID
GetGpioTableSize (
  GPIO_INIT_CONFIG   *GpioTable,
  OUT UINT16         *GpioCount
  )
{
  *GpioCount = 0;
  if(GpioTable != NULL) {
    while (GpioTable[*GpioCount].GpioPad != 0 && *GpioCount < MAX_GPIO_PINS) {
      DEBUG ((DEBUG_INFO, "GpioTable[%d]->GpioPad = %x \n", *GpioCount, GpioTable[*GpioCount].GpioPad));
      (*GpioCount) ++;
    }
  } else {
    DEBUG ((DEBUG_INFO, "GpioTable is NULL\n"));
  }
  DEBUG ((DEBUG_INFO, "GetGpioTableSize() GpioCount = %d\n", *GpioCount));
}

/**
Configure GPIO Table setting to PcdBoardGpioTablePreMem && PcdBoardGpioTable

@param[in]  GpioTable   The pointer of GPIO config table
@param[in]  IsPostMem   Is call from PostMem/PreMem
                        True - PostMem, False - PreMem
**/
VOID
ConfigureGpioTabletoPCD(
  IN GPIO_INIT_CONFIG   *GpioTable,
  IN BOOLEAN            IsPostMem
  )
{
  UINT16        GpioCount = 0;
  UINTN         Size = 0;
  EFI_STATUS    Status = EFI_SUCCESS;
  BOOLEAN       DisableVpdGpioTable = FALSE;

  DEBUG ((DEBUG_INFO, "ConfigureGpioTabletoPCD() Start\n"));

  DisableVpdGpioTable = (BOOLEAN) PcdGetBool (PcdDisableVpdGpioTable);
  DEBUG((DEBUG_INFO, "PcdDisableVpdGpioTable = %d\n", DisableVpdGpioTable));
  if (!DisableVpdGpioTable) {
    if (GpioTable != NULL) {
      GetGpioTableSize (GpioTable, &GpioCount);
      if (GpioCount) {
        if (IsPostMem) {    // Post Mem GPIO Configuration
          if (GpioCount >= MAX_GPIO_PINS) {
            DEBUG ((DEBUG_ERROR, "GpioTable entries exceeds limit, Configure only MAX_GPIO_PINS Pins.\n"));
            GpioCount = MAX_GPIO_PINS;
          }
          DEBUG ((DEBUG_INFO, "GpioTable Count = %d\n", GpioCount));
          Size = (UINTN) (GpioCount * sizeof (GPIO_INIT_CONFIG));
          Status = PcdSetPtrS (PcdBoardGpioTable, &Size, GpioTable);
        } else {    // Pre Mem GPIO Configuration
          if (GpioCount >= MAX_PRE_MEM_GPIO_PINS) {
            DEBUG ((DEBUG_ERROR, "PreMem GpioTable entries exceeds limit, Configure only MAX_PRE_MEM_GPIO_PINS Pins.\n"));
            GpioCount = MAX_PRE_MEM_GPIO_PINS;
          }
          DEBUG ((DEBUG_INFO, "GpioTable Count = %d\n", GpioCount));
          Size = (UINTN) (GpioCount * sizeof (GPIO_INIT_CONFIG));
          Status = PcdSetPtrS (PcdBoardGpioTablePreMem, &Size, GpioTable);
        }
        ASSERT_EFI_ERROR (Status);
      } else {
        DEBUG ((DEBUG_INFO, "GpioTable is Empty\n"));
      }
    } else {
      DEBUG ((DEBUG_INFO, "GpioTable is NULL\n"));
    }
  } else {
    DEBUG ((DEBUG_INFO, "PcdDisableVpdGpioTable is TRUE, GPIO Tables will be updated by PCT PEIM \n"));
  }

  DEBUG ((DEBUG_INFO, "ConfigureGpioTabletoPCD() End\n"));
}

/**
  Configures GPIO

  @param[in]  GpioTable       Point to Platform Gpio table
  @param[in]  GpioTableCount  Number of Gpio table entries
**/
STATIC
VOID
ConfigureGpio (
  IN GPIO_INIT_CONFIG                 *GpioTable,
  IN UINT16                           GpioTableCount
  )
{
  EFI_STATUS          Status;

  DEBUG ((DEBUG_INFO, "ConfigureGpio() Start\n"));

  Status = GpioConfigurePads (GpioTableCount, GpioTable);
  ASSERT_EFI_ERROR (Status);

  DEBUG ((DEBUG_INFO, "ConfigureGpio() End\n"));
}

/**
  Configure GPIO Before Memory is initialized.

  @param[in]  GpioTable  Pointer to Gpio table
**/
VOID
GpioInit (
  IN GPIO_INIT_CONFIG *GpioTable
  )
{
  UINT16             GpioCount;

  if (GpioTable != 0) {
    GpioCount = 0;
    GetGpioTableSize (GpioTable, &GpioCount);
    if (GpioCount != 0) {
      ConfigureGpio ((VOID *) GpioTable, (UINTN) GpioCount);
    }
  }
}

/**
  Configure GPIO group GPE tier.

**/
VOID
GpioGroupTierInit (
  VOID
  )
{
  DEBUG ((DEBUG_INFO, "GpioGroupTierInit Start\n"));

  if (PcdGet32 (PcdGpioGroupToGpeDw0)) {
    GpioSetGroupToGpeDwX (PcdGet32 (PcdGpioGroupToGpeDw0),
                          PcdGet32 (PcdGpioGroupToGpeDw1),
                          PcdGet32 (PcdGpioGroupToGpeDw2));
  }
  DEBUG ((DEBUG_INFO, "GpioGroupTierInit End\n"));
}

/**
  Configure Dpin External Port Map PCD

**/
VOID
DpInExtPortMapPcdInit (
  VOID
  )
{
  UINT32                Data32;
  UINT8                 i;
  UINT32                Size;
  DPIN_PORT_MAP_DATA    DpInExtPortMapPcdData;
  DPIN_GPIO_DATA        *DpInGpioTable;
  DPIN_GPIO_DATA_ARRAY  *DpInGpioDataArray;
  DPIN_PORT_MAP_DATA    *DpInExtPortMapPcdDataPtr;
  EFI_STATUS            Status;

// Initialize the Local Variable
  Data32                = 0;
  i                     = 0;
  Size                  = 0;
  DpInGpioTable         = NULL;
  DpInGpioDataArray     = NULL;
  DpInExtPortMapPcdDataPtr                 = NULL;
  DpInExtPortMapPcdData.NumberOfDpInPort   = 0;
  DpInExtPortMapPcdData.DpInPortConnectMap = 0;
  DpInExtPortMapPcdDataPtr                 = PcdGetPtr (PcdBoardDpInPortMapData);

  if (DpInExtPortMapPcdDataPtr->SkipPortDetectionViaGpio == TRUE) {
    // External Gfx Dp-In Port Connect Map either Hardcoded or wiil be detected through someother method
    return;
  }

  DpInGpioDataArray     = PcdGetPtr (PcdBoardDpInGpioTable);
  if (DpInGpioDataArray != 0) {
    DpInGpioTable       = &(DpInGpioDataArray->DpInGpioConfig[0]);
  }

  DEBUG ((DEBUG_INFO, "DpInExtPortMapPcdInit Start\n"));

  if (DpInGpioTable != 0) {
    //
    // Check if Valid GPIO Number is passed.
    //
    if ((DpInGpioDataArray->NumberOfDpInPort != 0) &&
        (DpInGpioDataArray->NumberOfDpInPort <= PcdGet8 (PcdMaxDpInExtPortSupported))) {
    // Copy DpIn External Port number to "PcdBoardDpInPortMapData" from Board Data
    DpInExtPortMapPcdData.NumberOfDpInPort   = (UINT8) DpInGpioDataArray->NumberOfDpInPort;
    //
    // Check for all GPIO PAD status for Dp_in Port connect detection
    //
      for (i=0; (i < DpInExtPortMapPcdData.NumberOfDpInPort)\
        && (i < PcdGet8 (PcdMaxDpInExtPortSupported)); i++) {
        if ((DpInGpioTable[i].DpInSkipGpioChecking != 0)\
                || (DpInGpioTable[i].DpInGpioPad != 0)) { // Check if all valid GPIO PAD is passed
          //
          // Check GPIO Label Status
          //
          Status = GpioGetInputValue (DpInGpioTable[i].DpInGpioPad, &Data32);
          if (Status == EFI_SUCCESS) {
            if ((Data32 & BIT0) == (UINT32)((DpInGpioTable[i].DpInPresenceDetectLabel) & BIT0)) {

              //
              // If DpIn port connection is detected then save in Map.
              //
              DpInExtPortMapPcdData.DpInPortConnectMap |= (BIT0 << DpInGpioTable[i].DpInPortNumber);
              DEBUG ((DEBUG_INFO, "DpInExtPortMapPcdInit  Ext Gfx DpIn Port[%d] Connect Detected\n", i));
            }
          }
        }
      }
    }
  }

  Size = sizeof (DPIN_PORT_MAP_DATA);
  PcdSetPtrS (PcdBoardDpInPortMapData, &Size, &DpInExtPortMapPcdData);
  DEBUG ((DEBUG_INFO, "DpInExtPortMapPcdInit End\n"));
}
