/** @file

@copyright
  INTEL CONFIDENTIAL
  Copyright 2017 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <PiPei.h>
#include <Library/BaseMemoryLib.h>
#include <Library/BiosIdLib.h>
#include <Library/PcdLib.h>
#include <Library/DebugLib.h>
#include <Library/PciLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/PeiLib.h>
#include <Guid/MemoryOverwriteControl.h>
#include <Library/MmioInitLib.h>
#include <PlatformBoardConfig.h>
#include <Library/SiliconInitLib.h>
#include <Library/PchCycleDecodingLib.h>
#include <Register/PmcRegs.h>
#include <Library/PmcLib.h>
#include <Library/PeiBootModeLib.h>
#include <Ppi/ReadOnlyVariable2.h>
#include <Library/PeiServicesLib.h>
#include <Library/WakeupEventLib.h>
#include <Library/GpioLib.h>
#include <Library/BoardConfigLib.h>
#include <Library/TimerLib.h>
#include <Library/CnviLib.h>
#include <TigerLakeUBoardConfigPatchTable.h>
#include <PlatformBoardId.h>
#include <SioRegs.h>
#include <Library/IoLib.h>
#include <Pins/GpioPinsVer2Lp.h>
#include <Library/PchInfoLib.h>
#include <Library/CpuPlatformLib.h>


#define SIO_RUNTIME_REG_BASE_ADDRESS      0x0680

/**
  Tigerlake U boards configuration init function for PEI pre-memory phase.

  @retval EFI_SUCCESS             The function completed successfully.
**/
EFI_STATUS
EFIAPI
TglUInitPreMem (
  VOID
  )
{
  EFI_STATUS                        Status;
  UINTN                             VariableSize;
  VOID                              *MemorySavedData;
  UINT8                             MorControl;
  VOID                              *MorControlPtr;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI   *VariableServices;

  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &VariableServices
             );
  ASSERT_EFI_ERROR (Status);
  //
  // Initialize S3 Data variable (S3DataPtr). It may be used for warm and fast boot paths.
  //
  VariableSize = 0;
  MemorySavedData = NULL;
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"MemoryConfig",
                               &gFspNonVolatileStorageHobGuid,
                               NULL,
                               &VariableSize,
                               MemorySavedData
                               );
  if (Status == EFI_BUFFER_TOO_SMALL) {
    //
    // Set the DISB bit
    // after memory Data is saved to NVRAM.
    //
    PmcSetDramInitScratchpad ();
  }

  //
  // MOR
  //
  MorControl = 0;
  MorControlPtr = &MorControl;
  VariableSize = sizeof (MorControl);
  Status = PeiGetVariable (
             MEMORY_OVERWRITE_REQUEST_VARIABLE_NAME,
             &gEfiMemoryOverwriteControlDataGuid,
             &MorControlPtr,
             &VariableSize
             );
  DEBUG ((DEBUG_INFO, "MorControl - 0x%x (%r)\n", MorControl, Status));
  if (MOR_CLEAR_MEMORY_VALUE (MorControl)) {
    PcdSet8S (PcdCleanMemory, MorControl & MOR_CLEAR_MEMORY_BIT_MASK);
  }

  PcdSet32S (PcdStackBase, PcdGet32 (PcdTemporaryRamBase) + PcdGet32 (PcdTemporaryRamSize) - (PcdGet32 (PcdFspTemporaryRamSize) + PcdGet32 (PcdFspReservedBufferSize)));
  PcdSet32S (PcdStackSize, PcdGet32 (PcdFspTemporaryRamSize));

  PcdSet8S (PcdCpuRatio, 0x0);
  PcdSet8S (PcdBiosGuard, 0x0);

  return EFI_SUCCESS;
}

/**
  Updates the wakeupType.
**/
VOID
TglUWakeUpTypeUpdate (
  VOID
  )
{
  UINT8   WakeupType;
  //
  // Updates the wakeupType which will be used to update the same in Smbios table 01
  //
  GetWakeupEvent (&WakeupType);
  PcdSet8S (PcdWakeupType, WakeupType);
}

VOID
TglUMrcConfigInit (
  VOID
  );

VOID
TglUSaMiscConfigInit (
  VOID
  );

VOID
TglUSaGpioConfigInit (
  VOID
  );

VOID
TglUSaDisplayConfigInit (
  VOID
  );

VOID
TglUSaUsbConfigInit (
  VOID
  );

EFI_STATUS
TglURootPortClkInfoInit (
  VOID
  );

EFI_STATUS
TglUUsbConfigInit (
  VOID
  );

VOID
TglUGpioTablePreMemInit(
  VOID
  );

VOID
TglUGpioGroupTierInit (
  VOID
  );

/**
  HSIO init function for PEI pre-memory phase.
**/
VOID
TglUHsioInit (
  VOID
  )
{
}

/**
  Configure Super IO
**/
STATIC
VOID
TglUSioInit (
  VOID
  )
{
  //
  // Program and Enable Default Super IO Configuration Port Addresses and range
  //
  PchLpcGenIoRangeSet (PcdGet16 (PcdLpcSioConfigDefaultPort) & (~0xF), 0x10);

  //
  // Enable LPC decode for KCS and mailbox SIO for iBMC communication
  //
  if (PcdGet8 (PcdPlatformFlavor) == FlavorUpServer) {
    PchLpcGenIoRangeSet (BMC_KCS_BASE_ADDRESS, 0x10);
    PchLpcGenIoRangeSet (PILOTIII_MAILBOX_BASE_ADDRESS, 0x10);
  } else {
  //
  // 128 Byte Boundary and SIO Runtime Register Range is 0x0 to 0xF;
  //
    PchLpcGenIoRangeSet (SIO_RUNTIME_REG_BASE_ADDRESS  & (~0x7F), 0x10);
  }

  //
  // We should not depend on SerialPortLib to initialize KBC for legacy USB
  // So initialize KBC for legacy USB driver explicitly here.
  // After we find how to enable mobile KBC, we will add enabling code for mobile then.
  //
  if ((PcdGet8 (PcdPlatformFlavor) == FlavorDesktop) ||
      (PcdGet8 (PcdPlatformFlavor) == FlavorWorkstation)) {
    //
    // Enable KBC for National PC8374 SIO
    //
    if (PcdGetBool (PcdPc8374SioKbcPresent)) {
      IoWrite8 (PcdGet16 (PcdDesktopLpcSioIndexDefaultPort), 0x07);
      IoWrite8 (PcdGet16 (PcdDesktopLpcSioDataDefaultPort), 0x06);
      IoWrite8 (PcdGet16 (PcdDesktopLpcSioIndexDefaultPort), 0x30);
      IoWrite8 (PcdGet16 (PcdDesktopLpcSioDataDefaultPort), 0x01);
    }
  }

  return;
}

static EFI_PEI_PPI_DESCRIPTOR mSetupVariablesReadyPpi = {
  (EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gSetupVariablesReadyPpiGuid,
  NULL
};

/**
  Notifies the gPatchConfigurationDataPreMemPpiGuid has been Installed

  @param[in] PeiServices          General purpose services available to every PEIM.
  @param[in] NotifyDescriptor     The notification structure this PEIM registered on install.
  @param[in] Ppi                  The memory discovered PPI.  Not used.

  @retval EFI_SUCCESS             The function completed successfully.
**/
EFI_STATUS
EFIAPI
TglUBoardPatchConfigurationDataPreMemCallback (
  IN EFI_PEI_SERVICES           **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR  *NotifyDescriptor,
  IN VOID                       *Ppi
  )
{
  UINTN           ConfigPatchStructSize;
  UINT16          BoardId;
  CPU_STEPPING    CpuSteppingId;
  CPU_GENERATION  CpuGeneration;
  CPU_FAMILY      CpuFamily;


  BoardId         = PcdGet16(PcdBoardId);
  // Reading CPU information for stepping speicfic patch action
  CpuSteppingId   = GetCpuStepping ();
  CpuGeneration   = GetCpuGeneration ();
  CpuFamily       = GetCpuFamily();


  switch (BoardId) {
    case BoardIdTglUDdr4:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeUDdr4RvpConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
      PatchSetupConfigurationDataPreMem (mTigerlakeUDdr4RvpConfigPatchStruct, ConfigPatchStructSize);
      break;
    case BoardIdTglULp4Type4:
    case BoardIdTglULp4Aep:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeULp4RvpConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
      PatchSetupConfigurationDataPreMem (mTigerlakeULp4RvpConfigPatchStruct, ConfigPatchStructSize);
      break;
    case BoardIdTglULp4Dg1Aep:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeULp4Dg1ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
      PatchSetupConfigurationDataPreMem (mTigerlakeULp4Dg1ConfigPatchStruct, ConfigPatchStructSize);
      break;
    case BoardIdTglULp4Gcs:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeULp4GcsRvp, CONFIG_PATCH_STRUCTURE);
      PatchSetupConfigurationDataPreMem (mTigerlakeULp4GcsRvp, ConfigPatchStructSize);
      break;
    case BoardIdTglULp5Type4:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerLakeULp5ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
      PatchSetupConfigurationDataPreMem (mTigerLakeULp5ConfigPatchStruct, ConfigPatchStructSize);
      break;
    default:
      break;
  }

  if (IsTglPch () && IsPchLp () && (PchStepping () == PCH_Z0)) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglPchZ0ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    PatchSetupConfigurationDataPreMem (mTglPchZ0ConfigPatchStruct, ConfigPatchStructSize);
  }

  if (IsTglPch () && IsPchLp () && (PchStepping () < PCH_B0)) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglPchA0ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    PatchSetupConfigurationDataPreMem (mTglPchA0ConfigPatchStruct, ConfigPatchStructSize);
  }

  // Patch for Ax specific SaSetupConfig updates
  if (CpuGeneration == EnumTglCpu && CpuFamily == EnumCpuTglUltUlx && CpuSteppingId == EnumTglA0) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglAxSaSetupConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    PatchSetupConfigurationDataPreMem (mTglAxSaSetupConfigPatchStruct, ConfigPatchStructSize);
  }

  // SAGV to be disabled for Lp4 on Ax stepping
  if ((CpuGeneration == EnumTglCpu && CpuFamily == EnumCpuTglUltUlx && CpuSteppingId == EnumTglA0)\
       && (BoardId != BoardIdTglUDdr4) && (BoardId != BoardIdTglUDdr4Sv)) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglAxMrcSetupConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    PatchSetupConfigurationDataPreMem (mTglAxMrcSetupConfigPatchStruct, ConfigPatchStructSize);
  }

/**
  WA to enable BT audio offload for integrated bluetooth TGL LP A0/Z0
**/
  if (CnviIsPresent () && (IsTglPch () && IsPchLp () && ((PchStepping () == PCH_Z0) || (PchStepping () == PCH_A0)))) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglUA0ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    PatchSetupConfigurationDataPreMem (mTglUA0ConfigPatchStruct, ConfigPatchStructSize);
  }

#if FixedPcdGet8(PcdEmbeddedEnable) == 0x1
  ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeEmbeddedConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
  PatchSetupConfigurationDataPreMem (mTigerlakeEmbeddedConfigPatchStruct, ConfigPatchStructSize);
#endif

  PeiServicesInstallPpi (&mSetupVariablesReadyPpi);

  return RETURN_SUCCESS;
}

/**
  Board Misc init function for PEI pre-memory phase.
**/
VOID
TglUBoardMiscInitPreMem (
  VOID
  )
{
  PCD64_BLOB PcdData;
  UINT16    BoardId;
  BoardId = PcdGet16(PcdBoardId);

  //
  // RecoveryMode GPIO
  //
  PcdData.Blob = 0;
  PcdData.BoardGpioConfig.Type = BoardGpioTypeNotSupported;
  PcdSet64S (PcdRecoveryModeGpio, PcdData.Blob);

  //
  // OddPower Init
  //
  PcdSetBoolS (PcdOddPowerInitEnable, FALSE);

  //
  // Pc8374SioKbc Present
  //
  PcdSetBoolS (PcdPc8374SioKbcPresent, FALSE);

  //
  // Smbus Alert function Init.
  //
  PcdSetBoolS (PcdSmbusAlertEnable, FALSE);

  //
  // Configure WWAN Full Card Power Off and reset pins
  //
  switch (BoardId) {
    case BoardIdTglUDdr4:
    case BoardIdTglULp4Type4:
    case BoardIdTglULp4Exs:
    case BoardIdTglULp5Type4:
      PcdSet32S (PcdWwanFullCardPowerOffGpio, GPIO_VER2_LP_GPP_C11);
      PcdSet32S (PcdWwanBbrstGpio, GPIO_VER2_LP_GPP_C10);
      PcdSet32S (PcdWwanPerstGpio, GPIO_VER2_LP_GPP_B17);
      PcdSet32S (PcdWwanWakeGpio, GPIO_VER2_LP_GPP_C9);
      PcdSetBoolS (PcdWwanPerstGpioPolarity, 0);
      PcdSetBoolS (PcdWwanFullCardPowerOffGpioPolarity, PIN_GPIO_ACTIVE_HIGH);
      PcdSetBoolS (PcdWwanBbrstGpioPolarity, PIN_GPIO_ACTIVE_LOW);
      PcdSet8S (PcdWwanSourceClock, 2);
      PcdSet8S (PcdWwanRootPortNumber, 0x04);
      break;
    case BoardIdTglULp4Aep:
    case BoardIdTglULp4Dg1Aep:
    case BoardIdTglULp4Gcs:
      PcdSet32S (PcdWwanFullCardPowerOffGpio, GPIO_VER2_LP_GPP_C11);
      PcdSet32S (PcdWwanBbrstGpio, GPIO_VER2_LP_GPP_C10);
      PcdSet32S (PcdWwanPerstGpio, GPIO_VER2_LP_GPP_B17);
      PcdSet32S (PcdWwanWakeGpio, GPIO_VER2_LP_GPP_C9);
      PcdSetBoolS (PcdWwanPerstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);
      PcdSetBoolS (PcdWwanFullCardPowerOffGpioPolarity, PIN_GPIO_ACTIVE_HIGH);
      PcdSetBoolS (PcdWwanBbrstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);
      PcdSet8S (PcdWwanSourceClock, 2);
      break;
  }

  //EC FailSafe Cpu Temp and Fan Speed Setting
  PcdSet8S (PcdEcFailSafeActionCpuTemp, 85);
  PcdSet8S (PcdEcFailSafeActionFanPwm, 65);
}

/**
  A hook for board-specific initialization prior to memory initialization.

  @retval EFI_SUCCESS   The board initialization was successful.
**/
EFI_STATUS
EFIAPI
TglUBoardInitBeforeMemoryInit (
  VOID
  )
{
  EFI_STATUS        Status;

  DEBUG ((DEBUG_INFO, "TglUBoardInitBeforeMemoryInit\n"));

  GetBiosId (NULL);

  TglUInitPreMem ();

  TglUWakeUpTypeUpdate ();

  TglUSioInit ();

  ///
  /// Do basic PCH init
  ///
  SiliconInit ();

  TglUGpioGroupTierInit ();
  TglUGpioTablePreMemInit ();

  TglUHsioInit ();
  TglUMrcConfigInit ();
  TglUSaGpioConfigInit ();
  TglUSaMiscConfigInit ();
  Status = TglURootPortClkInfoInit ();
  Status = TglUUsbConfigInit ();
  TglUBoardMiscInitPreMem ();
  TglUSaDisplayConfigInit ();
  TglUSaUsbConfigInit ();

  // Configure GPIO Before Memory is not ready
  GpioInit (PcdGetPtr (PcdBoardGpioTablePreMem));

  // Configure GPIO group GPE tier
  GpioGroupTierInit ();

  ASSERT(Status == EFI_SUCCESS);
  return EFI_SUCCESS;
}

/**
  A hook for board-specific initialization after memory initialization.

  @retval EFI_SUCCESS   The board initialization was successful.
**/
EFI_STATUS
EFIAPI
TglUBoardInitAfterMemoryInit (
  VOID
  )
{
  DEBUG ((DEBUG_INFO, "TglUBoardInitAfterMemoryInit\n"));

  return MmioInit ();
}

/**
  This board service initializes board-specific debug devices.

  @retval EFI_SUCCESS   Board-specific debug initialization was successful.
**/
EFI_STATUS
EFIAPI
TglUBoardDebugInit (
  VOID
  )
{
  DEBUG ((DEBUG_INFO, "TglUBoardDebugInit\n"));

  return EFI_SUCCESS;
}

/**
  This board service detects the boot mode.

  @retval EFI_BOOT_MODE The boot mode.
**/
EFI_BOOT_MODE
EFIAPI
TglUBoardBootModeDetect (
  VOID
  )
{
  EFI_BOOT_MODE                             BootMode;

  DEBUG ((DEBUG_INFO, "TglUBoardBootModeDetect\n"));
  BootMode = DetectBootMode ();
  DEBUG ((DEBUG_INFO, "BootMode: 0x%02X\n", BootMode));

  return BootMode;
}
