/** @file
  Platform FV address library instance to support TopSwap based fault tolerance case.

 @copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification
**/

#include <PiDxe.h>

#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/PcdLib.h>
#include <Library/SpiAccessLib.h>
#include <Library/BaseMemoryLib.h>
#define GET_OCCUPIED_SIZE(ActualSize, Alignment) \
  ((ActualSize) + (((Alignment) - ((ActualSize) & ((Alignment) - 1))) & ((Alignment) - 1)))

/**
  Get Firmware Binary FV base address

  @return FvBase associated with FvBinary FV.
**/
UINT32
GetFvBinaryBase (
  VOID
  )
{
  UINT32 FvBase;
  //if (SpiIsTopSwapEnabled ()) {
  //  FvBase = FixedPcdGet32 (PcdFtFlashFirmwareBinariesFvBase);
  //} else {
    FvBase = FixedPcdGet32 (PcdFlashFvFirmwareBinariesBase);
  //}
  DEBUG ((DEBUG_INFO, "GetFvBinaryBase - :0x%x  ......\n", FvBase));
  return FvBase;
}

/**
  Search and identify the physical address of a
  file module inside the FW_BINARIES_FV_SIGNED FV

  @param[in]       FvHeader    Pointer to the firmware volume header to be checked.
  @param[in]       GuidPtr     Pointer to the Guid of the binary to find with in the FV.
  @param[in][out]  ModulePtr   Pointer to the specific binary within the FV.
  @param[in][out]  ModuleSize  Size of the specific binary.

  @retval  EFI_SUCCESS  If address has been found
  @retval  Others       If address has not been found
**/
EFI_STATUS
FindModuleInFlash2 (
  IN EFI_FIRMWARE_VOLUME_HEADER *FvHeader,
  IN EFI_GUID                   *GuidPtr,
  IN OUT UINT32                 *ModulePtr,
  IN OUT UINT32                 *ModuleSize
  )
{
  EFI_FFS_FILE_HEADER        *FfsHeader;
  EFI_COMMON_SECTION_HEADER  *SectionHeader;
  UINT32                     ModuleAddr;
  UINT32                     FileLength;
  UINT32                     FileOccupiedSize;
  UINT32                     SectionSize;
  EFI_GUID                   InvalidGuid = {0xffffffff, 0xffff, 0xffff, {0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff}};

  //
  // the module is encapsulated in the FDF as such:
  // - FW_BINARIES_FV_SIGNED Firmware Volume
  //  - PAD FFS
  //   - BOOT Guard ACM FFS
  //    - PAD FFS
  //     - Signed FFS
  //      - RAW Section
  //       - GUIDED Section
  //        - RAW Section
  //         - RAW Section
  //          - Unsigned FW_BINARIES_FV Firmware Volume
  //            - FFS as defined in the FW_BINARIES_FV
  //
  FfsHeader     = NULL;
  SectionHeader = NULL;
  ModuleAddr    = 0;

  //
  // FvHeader
  //
  DEBUG ((DEBUG_VERBOSE, "FvHeader 0x%X:\n", FvHeader, FvHeader));
  DEBUG ((DEBUG_VERBOSE, " FileSystemGuid = 0x%g\n", &FvHeader->FileSystemGuid));
  DEBUG ((DEBUG_VERBOSE, " FvLength       = 0x%X\n", FvHeader->FvLength));
  DEBUG ((DEBUG_VERBOSE, " FvHeader Size  = 0x%X\n", FvHeader->HeaderLength));

  //
  // PAD FFS = FvHeader + FvHeader->HeaderLength
  //
  FfsHeader = (EFI_FFS_FILE_HEADER *)((UINT8 *) FvHeader + FvHeader->HeaderLength);
  ModuleAddr = (UINT32) (UINT8 *) FfsHeader;


  if (IS_FFS_FILE2 (FfsHeader)) {
    FileLength = FFS_FILE2_SIZE (FfsHeader);
    ASSERT (FileLength > 0x00FFFFFF);
  } else {
    FileLength = FFS_FILE_SIZE (FfsHeader);
  }
  //
  // FileLength is adjusted to FileOccupiedSize as it is 8 byte aligned.
  //
  FileOccupiedSize = (UINT32) GET_OCCUPIED_SIZE (FileLength, 8);
  DEBUG ((DEBUG_VERBOSE, "FfsHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " Name = 0x%g\n", &FfsHeader->Name));
  DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", FileLength));
  DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", FfsHeader->Type));
  DEBUG ((DEBUG_VERBOSE, " Occupied Size         = 0x%X\n", FileOccupiedSize));

  //
  // BOOT Guard ACM FFS = FfsHeader + FfsHeader->Size
  //
  FfsHeader = (EFI_FFS_FILE_HEADER *)(((UINT32) (UINT8 *)FfsHeader) + FileOccupiedSize);
  ModuleAddr = (UINT32) (UINT8 *) FfsHeader;
  if (IS_FFS_FILE2 (FfsHeader)) {
    FileLength = FFS_FILE2_SIZE (FfsHeader);
    ASSERT (FileLength > 0x00FFFFFF);
  } else {
    FileLength = FFS_FILE_SIZE (FfsHeader);
  }
  //
  // FileLength is adjusted to FileOccupiedSize as it is 8 byte aligned.
  //
  FileOccupiedSize = (UINT32) GET_OCCUPIED_SIZE (FileLength, 8);
  DEBUG ((DEBUG_VERBOSE, "FfsHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " Name = 0x%g\n", &FfsHeader->Name));
  DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", FileLength));
  DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", FfsHeader->Type));
  DEBUG ((DEBUG_VERBOSE, " Occupied Size         = 0x%X\n", FileOccupiedSize));

  //
  // PAD FFS = FfsHeader + FfsHeader->Size
  //
  FfsHeader = (EFI_FFS_FILE_HEADER *)(((UINT32) (UINT8 *)FfsHeader) + FileOccupiedSize);
  ModuleAddr = (UINT32) (UINT8 *) FfsHeader;
  if (IS_FFS_FILE2 (FfsHeader)) {
    FileLength = FFS_FILE2_SIZE (FfsHeader);
    ASSERT (FileLength > 0x00FFFFFF);
  } else {
    FileLength = FFS_FILE_SIZE (FfsHeader);
  }
  //
  // FileLength is adjusted to FileOccupiedSize as it is 8 byte aligned.
  //
  FileOccupiedSize = (UINT32) GET_OCCUPIED_SIZE (FileLength, 8);
  DEBUG ((DEBUG_VERBOSE, "FfsHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " Name = 0x%g\n", &FfsHeader->Name));
  DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", FileLength));
  DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", FfsHeader->Type));
  DEBUG ((DEBUG_VERBOSE, " Occupied Size         = 0x%X\n", FileOccupiedSize));

  //
  // Signed FFS = FfsHeader + FfsHeader->Size
  //
  FfsHeader = (EFI_FFS_FILE_HEADER *)(((UINT32) (UINT8 *)FfsHeader) + FileOccupiedSize);
  ModuleAddr = (UINT32) (UINT8 *) FfsHeader;
  if (IS_FFS_FILE2 (FfsHeader)) {
    FileLength = FFS_FILE2_SIZE (FfsHeader);
    ASSERT (FileLength > 0x00FFFFFF);
  } else {
    FileLength = FFS_FILE_SIZE (FfsHeader);
  }
  //
  // FileLength is adjusted to FileOccupiedSize as it is 8 byte aligned.
  //
  FileOccupiedSize = (UINT32) GET_OCCUPIED_SIZE (FileLength, 8);
  DEBUG ((DEBUG_VERBOSE, "FfsHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " Name = 0x%g\n", &FfsHeader->Name));
  DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", FileLength));
  DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", FfsHeader->Type));
  DEBUG ((DEBUG_VERBOSE, " Occupied Size         = 0x%X\n", FileOccupiedSize));

  //
  // RAW Section = FfsHeader + sizeof(FfsHeader)
  //
  if (IS_FFS_FILE2 (FfsHeader)) {
    FileLength = sizeof(EFI_FFS_FILE_HEADER2);
  } else {
    FileLength = sizeof(EFI_FFS_FILE_HEADER);
  }

  SectionHeader = (EFI_COMMON_SECTION_HEADER *)(((UINT32) (UINT8 *)FfsHeader) + FileLength);
  ModuleAddr = (UINT32) (UINT8 *) SectionHeader;
  if (IS_SECTION2(SectionHeader)) {
    SectionSize = SECTION2_SIZE(SectionHeader);
  } else {
    SectionSize = SECTION_SIZE(SectionHeader);
  }
  DEBUG ((DEBUG_VERBOSE, "RAW SectionHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", SectionSize));
  DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", SectionHeader->Type));

  //
  // GUIDED Section = SectionHeader + SectionHeader->Size
  //
  SectionHeader = (EFI_COMMON_SECTION_HEADER *)(((UINT32) (UINT8 *)SectionHeader) + SectionSize);
  ModuleAddr = (UINT32) (UINT8 *) SectionHeader;
  DEBUG ((DEBUG_VERBOSE, "GUIDED SectionHeader 0x%X:\n", ModuleAddr));
  if (IS_SECTION2(SectionHeader)) {
    DEBUG ((DEBUG_VERBOSE, " Guid      = 0x%g\n", &((EFI_GUID_DEFINED_SECTION2 *)SectionHeader)->SectionDefinitionGuid));
    DEBUG ((DEBUG_VERBOSE, " DataOfset = 0x%X\n", ((EFI_GUID_DEFINED_SECTION2 *)SectionHeader)->DataOffset));
  } else {
    DEBUG ((DEBUG_VERBOSE, " Guid      = 0x%g\n", &((EFI_GUID_DEFINED_SECTION *)SectionHeader)->SectionDefinitionGuid));
    DEBUG ((DEBUG_VERBOSE, " DataOfset = 0x%X\n", ((EFI_GUID_DEFINED_SECTION *)SectionHeader)->DataOffset));
  }
  DEBUG ((DEBUG_VERBOSE, " Size      = 0x%X\n", SectionSize));
  DEBUG ((DEBUG_VERBOSE, " Type      = 0x%X\n", SectionHeader->Type));

  //
  // RAW Section = SectionHeader + SectionHeader->DataOffset
  //
  if (IS_SECTION2(SectionHeader)) {
    SectionHeader = (EFI_COMMON_SECTION_HEADER *)(((UINT32) (UINT8 *)SectionHeader) + ((EFI_GUID_DEFINED_SECTION2 *)SectionHeader)->DataOffset);
    SectionSize   = SECTION2_SIZE(SectionHeader);
  } else {
    SectionHeader = (EFI_COMMON_SECTION_HEADER *)(((UINT32) (UINT8 *)SectionHeader) + ((EFI_GUID_DEFINED_SECTION *)SectionHeader)->DataOffset);
    SectionSize   = SECTION_SIZE(SectionHeader);
  }
  ModuleAddr = (UINT32) (UINT8 *) SectionHeader;
  DEBUG ((DEBUG_VERBOSE, "RAW SectionHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", SectionSize));
  DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", SectionHeader->Type));

  //
  // RAW Section = SectionHeader + SectionHeader->Size
  //
  SectionHeader = (EFI_COMMON_SECTION_HEADER *)(((UINT32) (UINT8 *)SectionHeader) + SectionSize);
  ModuleAddr = (UINT32) (UINT8 *) SectionHeader;
  if (IS_SECTION2(SectionHeader)) {
    SectionSize   = SECTION2_SIZE(SectionHeader);
  } else {
    SectionSize   = SECTION_SIZE(SectionHeader);
  }
  DEBUG ((DEBUG_VERBOSE, "RAW SectionHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", SectionSize));
  DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", SectionHeader->Type));

  //
  // Unsigned FW_BINARIES_FV = SectionHeader + sizeof(EFI_COMMON_SECTION_HEADER)
  //
  FvHeader = (EFI_FIRMWARE_VOLUME_HEADER *)(((UINT32) (UINT8 *)SectionHeader) + sizeof(EFI_COMMON_SECTION_HEADER));
  ModuleAddr = (UINT32) (UINT8 *) FvHeader;
  DEBUG ((DEBUG_VERBOSE, "Unsigned FvHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " FileSystemGuid = 0x%g\n", &FvHeader->FileSystemGuid));
  DEBUG ((DEBUG_VERBOSE, " FvLength       = 0x%X\n", FvHeader->FvLength));
  DEBUG ((DEBUG_VERBOSE, " FvHeader Size  = 0x%X\n", FvHeader->HeaderLength));
  DEBUG ((DEBUG_VERBOSE, " Max limit  = 0x%X\n", (((UINT32) (UINT8 *) FvHeader) + FvHeader->FvLength)));
  DEBUG ((DEBUG_VERBOSE, "GUID to FIND = 0x%g\n", GuidPtr));

  //
  // 1st FFS = FvHeader + FvHeader->HeaderLength
  //
  FfsHeader = (EFI_FFS_FILE_HEADER *)((UINT8 *) FvHeader + FvHeader->HeaderLength);
  ModuleAddr = (UINT32) (UINT8 *) FfsHeader;
  if (IS_FFS_FILE2 (FfsHeader)) {
    FileLength = FFS_FILE2_SIZE (FfsHeader);
    ASSERT (FileLength > 0x00FFFFFF);
  } else {
    FileLength = FFS_FILE_SIZE (FfsHeader);
  }
  //
  // FileLength is adjusted to FileOccupiedSize as it is 8 byte aligned.
  //
  FileOccupiedSize = (UINT32) GET_OCCUPIED_SIZE (FileLength, 8);
  DEBUG ((DEBUG_VERBOSE, "FfsHeader 0x%X:\n", ModuleAddr));
  DEBUG ((DEBUG_VERBOSE, " Name = 0x%g\n", &FfsHeader->Name));
  DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", FileLength));
  DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", FfsHeader->Type));
  DEBUG ((DEBUG_VERBOSE, " Occupied Size         = 0x%X\n", FileOccupiedSize));

  //
  // Traverse through the FFS of the FW_BINARIES_FV
  //
  while(ModuleAddr < (((UINT32) (UINT8 *) FvHeader) + FvHeader->FvLength)) {
    //
    // check for matching FFS guid
    //
    if(CompareGuid (GuidPtr, &FfsHeader->Name)) {
      *ModulePtr = ModuleAddr + sizeof(EFI_FFS_FILE_HEADER);
      *ModuleSize = FileLength - sizeof(EFI_FFS_FILE_HEADER);
      DEBUG (( DEBUG_INFO, "Module found at = 0x%X, Module Size = 0x%X\n", *ModulePtr, *ModuleSize));
      return EFI_SUCCESS;
    } else if (CompareGuid (&InvalidGuid, &FfsHeader->Name)) {
      DEBUG (( DEBUG_INFO, "Module not found\n"));
      return EFI_NOT_FOUND;
    }

    //
    // Next FFS = FfsHeader + FfsHeader->Size
    //
    FfsHeader = (EFI_FFS_FILE_HEADER *)(((UINT32) (UINT8 *)FfsHeader) + FileOccupiedSize);
    ModuleAddr = (UINT32) (UINT8 *) FfsHeader;
    if (IS_FFS_FILE2 (FfsHeader)) {
      FileLength = FFS_FILE2_SIZE (FfsHeader);
      ASSERT (FileLength > 0x00FFFFFF);
    } else {
      FileLength = FFS_FILE_SIZE (FfsHeader);
    }
    //
    // FileLength is adjusted to FileOccupiedSize as it is 8 byte aligned.
    //
    FileOccupiedSize = (UINT32) GET_OCCUPIED_SIZE (FileLength, 8);
    DEBUG ((DEBUG_VERBOSE, "FfsHeader 0x%X:\n", ModuleAddr));
    DEBUG ((DEBUG_VERBOSE, " Name = 0x%g\n", &FfsHeader->Name));
    DEBUG ((DEBUG_VERBOSE, " Size = 0x%X\n", FileLength));
    DEBUG ((DEBUG_VERBOSE, " Type = 0x%X\n", FfsHeader->Type));
    DEBUG ((DEBUG_VERBOSE, " Occupied Size         = 0x%X\n", FileOccupiedSize));
  }

  DEBUG (( DEBUG_INFO, "Module not found\n"));
  return EFI_NOT_FOUND;
}

/**
  Get Microcode FV base address

  @return FvBase associated with Microcode FV.
**/
UINT32
GetFvMicrocodeBase (
  VOID
  )
{
  UINT32 FvBase;
  //if (SpiIsTopSwapEnabled ()) {
  //  FvBase = FixedPcdGet32 (PcdFtFlashMicrocodeFvBase);
  //} else {
    FvBase = FixedPcdGet32 (PcdFlashMicrocodeFvBase);
  //}
  DEBUG ((DEBUG_INFO, "GetFvMicrocodeBase - :0x%x  ......\n", FvBase));
  return FvBase;
}

/**
  Get Fsp-S base address

  @return FvBase associated with Fsp-S FV.
**/
UINT32
GetFvFspsBase (
  VOID
  )
{
  UINT32 FvBase;
  //if (SpiIsTopSwapEnabled ()) {
  //  FvBase = FixedPcdGet32 (PcdFtFlashFvFspsBase);
  //} else {
    FvBase = FixedPcdGet32 (PcdFlashFvFspSBase);
  //}
  DEBUG ((DEBUG_INFO, "GetFvFspsBase - :0x%x.....\n", FvBase));
  return FvBase;
}
