/** @file
  Implements the interface for TbtNvmDrvRetimerThruHr class.
  This class is in charge of providing the way to access the retimer
  through TBT integrated HR.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:

**/
#include <Library/UefiBootServicesTableLib.h>

#include "TbtNvmDrvRetimerThruHr.h"

#define IECS_CMD_ADDR 8
#define IECS_DATA_ADDR 9

#define TBT_LC_SW_FW_MAILBOX_IN_PA 0x139
#define TBT_LC_SW_FW_MAILBOX_IN_PB 0x239
#define TBT_LC_SW_FW_MAILBOX_IN_CMD BIT(0)
#define TBT_LC_SW_FW_MAILBOX_IN_CMD_WA BIT(31)       // TODO TEMP WA until CP fix.
#define TBT_LC_SW_FW_MAILBOX_DATA_IN0_PA 0x197
#define TBT_LC_SW_FW_MAILBOX_DATA_IN0_PB 0x297
#define TBT_LC_SW_FW_MAILBOX_DATA_IN1_PA 0x198
#define TBT_LC_SW_FW_MAILBOX_DATA_IN1_PB 0x298

#define TBT_LC_SW_FW_MAILBOX_DATA_OUT0_PA 0x19b
#define TBT_LC_SW_FW_MAILBOX_DATA_OUT0_PB 0x29b

#define TBT_LC_CMD_SUCCESS 0x0

#define TBT_IECS_CMD_LSUP             0x5055534c
#define TBT_IECS_CMD_USUP             0x50555355
#define TBT_IECS_CMD_ENUM             0x4d554e45
#define TBT_IECS_CMD_LSEN             0x4e45534c


#define TBT_TOTAL_ENUM_ACCESSES 4
#define TBT_WAIT_TIME_BETWEEN_ENUM_ACCESSES   10000   // time in us, 10ms
#define TBT_WAIT_TIME_BEFORE_NEXT_IECS_ACCESS 100     // time in us
#define TBT_WAIT_TIME_BEFORE_NEXT_MSG_OUT_ACCESS 100  // time in us
#define TBT_TOTAL_WAIT_TIME_UNTIL_TIMEOUT 200000      // 200ms in us

#define TBT_TOTAL_ACCESSES_WHILE_WAIT_FOR_IECS  \
 TBT_TOTAL_WAIT_TIME_UNTIL_TIMEOUT / TBT_WAIT_TIME_BEFORE_NEXT_IECS_ACCESS
#define TBT_TOTAL_ACCESSES_WHILE_WAIT_FOR_MSG_OUT \
 TBT_TOTAL_WAIT_TIME_UNTIL_TIMEOUT / TBT_WAIT_TIME_BEFORE_NEXT_MSG_OUT_ACCESS

#define TBT_TOTAL_RETRIES_ON_MSG_OUT 10

#define TBT_MSG_OUT_CMD_VALID       BIT(31)
#define TBT_MSG_OUT_ATCT1_LT0       BIT(30)
#define TBT_MSG_OUT_WR_RD_OFFSET    23
#define TBT_MSG_OUT_TIMEOUT         BIT(25)
#define TBT_MSG_OUT_INVALID         BIT(26)
#define TBT_MSG_OUT_LENGTH_OFFSET   16
#define TBT_MSG_OUT_REG_ADDR_OFFSET 8
#define TBT_MSG_OUT_START_CT        BIT(6)
#define TBT_MSG_OUT_RETIMER_INDEX_OFFSET 1
#define TBT_MSG_OUT_CMD             BIT(0)

#define TBT_MSG_OUT_DEFAULT_VALS TBT_MSG_OUT_CMD_VALID | \
                                 TBT_MSG_OUT_ATCT1_LT0 | \
                                 TBT_MSG_OUT_START_CT  | \
                                 TBT_MSG_OUT_CMD

#define TBT_MSG_OUT_PA_TAR_ADDR 0x176
#define TBT_MSG_OUT_PB_TAR_ADDR 0x276
#define TBT_MSG_OUT_PA_WRDATA_0_TAR_ADDR 0x177
#define TBT_MSG_OUT_PB_WRDATA_0_TAR_ADDR 0x277
#define TBT_MSG_OUT_PA_RDDATA_0_TAR_ADDR 0x187
#define TBT_MSG_OUT_PB_RDDATA_0_TAR_ADDR 0x287

#define TBT_MSG_OUT_RDATA_LENGTH 1    // in DWs

#define TBT_IECS_MAILBOX_LEGTH_IN_DW 16

/// A struct to store the required fields during the operation
typedef struct {
  DN_PORT           DnFacingPort;
  RETIMER_INDEX     RetimerIndex;
  TBT_HR            *Hr;
  TARGET_DEVICE     *Comm;
} RETIMER_THRU_HR;

/**
  Sends IECS command and waits for completion.

  @param[in] DevCom Pointer to the device interface
  @param[in] Cmd IECS CMD to send

  @retval TBT_STATUS_SUCCESS Command was successefully send
  @retval TBT_STATUS_NON_RECOVERABLE_ERROR A device error has accured.
  @retval TBT_STATUS_RETRY LC reported error, the command might be retried

**/
TBT_STATUS
TbtNvmDrvSendCmd (
  IN TARGET_DEVICE *DevCom,
  UINT32           Cmd
  );

/**
  Wait for command to be completed
  Operation:
    Poll on MSG_OUT_CMD register to see when local LC completes the sending of a command to retimer

  @param[in] Hr                Pointer to the HR structure
  @param[in] MsgOutCmdOffset   The MSG_OUT_CMD address in target space, different for each CIO port group

  @retval status:
    TBT_STATUS_NON_RECOVERABLE_ERROR - need to terminate the execution
    TBT_STATUS_RETRY - the caller might retry
    TBT_STATUS_SUCCESS - indicate operation success

**/
STATIC
TBT_STATUS
WaitForMsgOutTxDone (
  IN TBT_HR    *Hr,
  IN UINT16    MsgOutCmdOffset,
  IN UINT8     DbgData
  )
{
  UINT32 data;
  TBT_STATUS status;
  UINT32 accessCnt = 0;
  do {
    // TODO improve it, using timer.
    gBS->Stall(TBT_WAIT_TIME_BEFORE_NEXT_MSG_OUT_ACCESS);
    status = Hr->ReadCioDevReg(Hr, MsgOutCmdOffset, &data);
    if (TBT_STATUS_ERR(status)) {
       DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::WaitForMsgOutTxDone: ERROR! Reading register MSG_OUT_CMD is failed. \
        Status is %d. d=%dExiting...\n", status, DbgData));
      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
    accessCnt++;
  } while ((data & TBT_MSG_OUT_CMD_VALID) != 0 && accessCnt < TBT_TOTAL_ACCESSES_WHILE_WAIT_FOR_MSG_OUT);
  if ((data & TBT_MSG_OUT_CMD_VALID) != 0) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::WaitForMsgOutTxDone: ERROR! Local LC seems to be stuck. d=%d Exiting...\n", DbgData));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  if ((data & TBT_MSG_OUT_TIMEOUT) != 0) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::WaitForMsgOutTxDone: IECS transaction was timeouted d=%d\n", DbgData));
  }
  else if ((data & TBT_MSG_OUT_INVALID) != 0) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::WaitForMsgOutTxDone: IECS transaction was invalid d=%d\n", DbgData));
  }
  else {
    return TBT_STATUS_SUCCESS;
  }
  return TBT_STATUS_RETRY;

}



/**
  Write Retimer's IECS register
  Operation:
    Write the data to write to MSG_OUT_WDATA corresponding to the port.
    Write the command to the MSG_OUT_CMD corresponding to the port.
    Wait for command to complete, return the status.

  @param[in] This      Pointer to the generic device interface - TARGET_DEVICE
  @param[in] RegNum    Retimer's IECS register number to write to
  @param[in] Data      Data to write
  @param[in] Length    How many DWs to write

  @retval status:
    TBT_STATUS_NON_RECOVERABLE_ERROR - need to terminate the execution
    TBT_STATUS_SUCCESS - indicate operation succes
**/
STATIC
TBT_STATUS
WriteIecs (
  IN TARGET_DEVICE    *This,
  IN UINT8            RegNum,
  IN UINT32           *Data,
  IN UINT8            Length
  )
{
  ASSERT(Length <= TBT_IECS_MAILBOX_LEGTH_IN_DW && Length > 0);
  ASSERT(This != NULL);
  if (Length > TBT_IECS_MAILBOX_LEGTH_IN_DW || Length == 0 || This == NULL) {
    return TBT_STATUS_INVALID_PARAM;
  }

  DEBUG ((DEBUG_VERBOSE, "RetimerThruHr::WriteIecs: IECS reg num - 0x%x, Length - 0x%x\n", RegNum, Length));

  RETIMER_THRU_HR *retimer = (RETIMER_THRU_HR *)This->Impl;
  TBT_STATUS status;
  UINT16 msgOutWDataOffset = (retimer->DnFacingPort == PORT_PA) ? TBT_MSG_OUT_PA_WRDATA_0_TAR_ADDR :
    TBT_MSG_OUT_PB_WRDATA_0_TAR_ADDR;
  UINT16 msgOutCmdOffset = (retimer->DnFacingPort == PORT_PA) ? TBT_MSG_OUT_PA_TAR_ADDR : TBT_MSG_OUT_PB_TAR_ADDR;
  UINT32 msgOutCmd = TBT_MSG_OUT_DEFAULT_VALS | BIT(TBT_MSG_OUT_WR_RD_OFFSET) |
    (Length * 4) << TBT_MSG_OUT_LENGTH_OFFSET | (RegNum << TBT_MSG_OUT_REG_ADDR_OFFSET) |
    (retimer->RetimerIndex << TBT_MSG_OUT_RETIMER_INDEX_OFFSET);

  // Write data
  status = retimer->Hr->WriteCioDevReg(retimer->Hr, msgOutWDataOffset, Length, Data);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHrWriteIecs: ERROR when writing the data, status - %d. Exiting...\n", status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }

  for (UINTN retryCnt = 0; retryCnt < TBT_TOTAL_RETRIES_ON_MSG_OUT; retryCnt++) {
    // Write command
    status = retimer->Hr->WriteCioDevReg(retimer->Hr, msgOutCmdOffset, 1, &msgOutCmd);
    if (TBT_STATUS_ERR(status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::WriteIecs: ERROR when writing the command(retryCnt=%d, Off=%x, Cmd=%x), status - %d. Exiting...\n", retryCnt, msgOutCmdOffset, msgOutCmd, status));
      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
    status = WaitForMsgOutTxDone(retimer->Hr, msgOutCmdOffset, 0);
    if (status == TBT_STATUS_NON_RECOVERABLE_ERROR || status == TBT_STATUS_SUCCESS) {
      return status;
    }
    DEBUG ((DEBUG_ERROR, "RetimerThruHr::WriteIecs: retryCnt=%d, Off=%x, Cmd=%x\n", retryCnt, msgOutCmdOffset, msgOutCmd));
  }

  return TBT_STATUS_NON_RECOVERABLE_ERROR;
}


/**
  Read Retimer's IECS register
  Operation:
    Write the command to the MSG_OUT_CMD corresponding to the port.
    Wait for command to complete
    Read MSG_OUT_RDDATA and assign it to buffer pointed by Data
    Return status

  @param[in]  This     Pointer to the generic device interface - TARGET_DEVICE
  @param[in]  RegNum   Retimer's IECS register number to read from
  @param[out] Data     Pointer to a buffer where the read data will be assigned

  @retval status:
    TBT_STATUS_NON_RECOVERABLE_ERROR - need to terminate the execution
    TBT_STATUS_SUCCESS - indicate operation succes

  Limitation: read suppors only one DW
**/
STATIC
TBT_STATUS
ReadIecs (
  IN TARGET_DEVICE  *This,
  IN UINT8          RegNum,   // TODO: RegNum type should be enum
  OUT UINT32        *Data
  )
{
  ASSERT(This != NULL);
  ASSERT(Data != NULL);

  if ((This == NULL) || (Data == NULL)) {
    return TBT_STATUS_INVALID_PARAM;
  }

  DEBUG ((DEBUG_VERBOSE, "RetimerThruHr::ReadIecs: IECS reg num - 0x%x\n", RegNum));
  RETIMER_THRU_HR *retimer = (RETIMER_THRU_HR *)This->Impl;
  TBT_STATUS status;
  UINT32 rdata;
  UINT16 msgOutCmdOffset = (retimer->DnFacingPort == PORT_PA) ? TBT_MSG_OUT_PA_TAR_ADDR : TBT_MSG_OUT_PB_TAR_ADDR;
  UINT16 msgOutRdataOffset = (retimer->DnFacingPort == PORT_PA) ? TBT_MSG_OUT_PA_RDDATA_0_TAR_ADDR :
    TBT_MSG_OUT_PB_RDDATA_0_TAR_ADDR;
  UINT32 msgOutCmd = TBT_MSG_OUT_DEFAULT_VALS |
    (TBT_MSG_OUT_RDATA_LENGTH * 4) << TBT_MSG_OUT_LENGTH_OFFSET |  // Read length is one DW
    (RegNum << TBT_MSG_OUT_REG_ADDR_OFFSET) | (retimer->RetimerIndex << TBT_MSG_OUT_RETIMER_INDEX_OFFSET);

  for (UINTN retryCnt = 0; retryCnt < TBT_TOTAL_RETRIES_ON_MSG_OUT; retryCnt++) {
    // Write command
    status = retimer->Hr->WriteCioDevReg(retimer->Hr, msgOutCmdOffset, 1, &msgOutCmd);
    if (TBT_STATUS_ERR(status)) {
      DEBUG ((DEBUG_ERROR, "RetimerThruHrReadIecs: ERROR when writing the command(retryCnt=%d, Off=%x, Cmd=%x), status - %d. Exiting...\n", retryCnt, msgOutCmdOffset, msgOutCmd, status));
      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
    status = WaitForMsgOutTxDone(retimer->Hr, msgOutCmdOffset, 1);
    if (status == TBT_STATUS_NON_RECOVERABLE_ERROR) {
      return status;
    }
    else if (status == TBT_STATUS_SUCCESS) {
      break;
    }
    DEBUG ((DEBUG_ERROR, "RetimerThruHr::ReadIecs: retryCnt=%d, Off=%x, Cmd=%x\n", retryCnt, msgOutCmdOffset, msgOutCmd));
  }
  if (TBT_STATUS_ERR(status)) {
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  // Read the data
  status = retimer->Hr->ReadCioDevReg(retimer->Hr, msgOutRdataOffset, &rdata);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "RetimerThruHr::ReadIecs: ERROR when reading from IECS rdata, status - %d. Exiting...\n", status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  *Data = rdata;
  return TBT_STATUS_SUCCESS;

}

/**
  Send command to the local LC
  Operation:
    If there is data to write it writes it to the LC_SW_FW_MAILBOX_DATA_IN1 register.
    Writes the command to the LC_SW_FW_MAILBOX_DATA_IN0
    Issues command indication to LC in TBT_LC_SW_FW_MAILBOX_IN
    Wait for command to be cleared in LC_SW_FW_MAILBOX_DATA_IN0
    Check the status and return


  @param Retimer pointer to the current implementation of TARGET_DEVICE interface
  @param Cmd The command to local LC
  @param Data pointer to a buffer where the write data

  @retval status:
    TBT_STATUS_NON_RECOVERABLE_ERROR - need to terminate the execution
    TBT_STATUS_SUCCESS - indicate operation succes

**/
STATIC
TBT_STATUS
SendCommandToLocalLc (
  RETIMER_THRU_HR *Retimer,
  UINT32 Cmd,
  UINT32 *Data OPTIONAL
  )
{
  UINT32 cmdResp = 0x0;
  UINT32 cmdTrig;
  UINT16 dataIn0TarOffset = (Retimer->DnFacingPort == PORT_PA) ? TBT_LC_SW_FW_MAILBOX_DATA_IN0_PA :
    TBT_LC_SW_FW_MAILBOX_DATA_IN0_PB;
  UINT16 dataIn1TarOffset = dataIn0TarOffset + 1;
  UINT16 cmdTarOffset = (Retimer->DnFacingPort == PORT_PA) ? TBT_LC_SW_FW_MAILBOX_IN_PA :
    TBT_LC_SW_FW_MAILBOX_IN_PB;
  TBT_STATUS status;
  UINT32 accessCnt = 0;

  DEBUG ((DEBUG_INFO, "\nSendCommandToLocalLc::Sending Cmd - 0x%x to port %s of the HR\n",
    Cmd, Retimer->DnFacingPort == PORT_PA ? "A" : "B"));

  if (Data != NULL) {
    status = Retimer->Hr->WriteCioDevReg(Retimer->Hr, dataIn1TarOffset, 1, Data);
    if (TBT_STATUS_ERR(status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::SendCommandToLocalLc: ERROR when writing 0x%x data to local LC, status - %d. \
        Exiting...\n", Data, status));
      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
  }

  status = Retimer->Hr->WriteCioDevReg(Retimer->Hr, dataIn0TarOffset, 1, &Cmd);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::SendCommandToLocalLc: ERROR when writing 0x%x cmd to local LC, status - %d.\
      Exiting...\n", Cmd, status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  // Read modify write the LC command
  status = Retimer->Hr->ReadCioDevReg(Retimer->Hr, cmdTarOffset, &cmdTrig);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::SendCommandToLocalLc: ERROR when reading cmd from local LC, status - %d. \
      Exiting...\n", status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  cmdTrig |= (TBT_LC_SW_FW_MAILBOX_IN_CMD | TBT_LC_SW_FW_MAILBOX_IN_CMD_WA);

  status = Retimer->Hr->WriteCioDevReg(Retimer->Hr, cmdTarOffset, 1, &cmdTrig);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::SendCommandToLocalLc: ERROR when writing 0x%x cmd to local LC, status - %d. \
    Exiting...\n", Cmd, status));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }

  // TODO improve this with timer
  do {
    gBS->Stall(TBT_WAIT_TIME_BEFORE_NEXT_IECS_ACCESS);
    status = Retimer->Hr->ReadCioDevReg(Retimer->Hr, dataIn0TarOffset, &cmdResp);
    if (TBT_STATUS_ERR(status)) {
      DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::SendCommandToLocalLc: ERROR when waiting for command 0x%x to complete, \
        status - %d. Exiting...\n", Cmd, status));
      return TBT_STATUS_NON_RECOVERABLE_ERROR;
    }
    accessCnt++;
  } while (cmdResp == Cmd && accessCnt < TBT_TOTAL_ACCESSES_WHILE_WAIT_FOR_IECS);

  if (cmdResp == Cmd) {  // Timeouted
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::SendCommandToLocalLc: Local LC Couldn't perform 0x%x command - timeouted. \
      Exiting...\n", Cmd));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  else if (cmdResp != TBT_LC_CMD_SUCCESS) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::SendCommandToLocalLc: Local LC reported error while performing 0x%x command - got error: 0x%x. \
      Exiting...\n", Cmd, cmdResp));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  return TBT_STATUS_SUCCESS;
}

/**
  Send enumeration command to local LC
  Need to repeate it for a number of times.
  Write to SW_FW_MAILBOX_IN0 corresponding to the port number the enum command.
  Then trigger cmd valid in SW_FW_MAILBOX_IN[0]

  Poll on the SW_FW_MAILBOX_IN0 register:
    if !ENUM -> Success
    if timeout exit
  Read SW_FW_MAILBOX_OUT0 reg
    if 0x1 error -> exit
    if 0x2 success
    else error -> exit

  @param Retimer pointer to the current implementation of TARGET_DEVICE interface

  @retval status:
    TBT_STATUS_NON_RECOVERABLE_ERROR - need to terminate the execution
    TBT_STATUS_SUCCESS - indicate operation succes

**/
STATIC
TBT_STATUS
SendEnumCmd (
  RETIMER_THRU_HR *Retimer
  )
{
  TBT_STATUS status;
  UINTN i;

  DEBUG ((DEBUG_INFO, "\nTbtNvmDrvRetimerThruHr::Sending ENUM to port %s of the HR\n", Retimer->DnFacingPort == PORT_PA ? "A" : "B"));

  for (i = 0; i < TBT_TOTAL_ENUM_ACCESSES; i++) {
    status = SendCommandToLocalLc(Retimer, TBT_IECS_CMD_ENUM, NULL);
    gBS->Stall(TBT_WAIT_TIME_BETWEEN_ENUM_ACCESSES);
  }
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::Enum wasn't sent successefully, got error...\n"));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }

  DEBUG ((DEBUG_INFO, "TbtNvmDrvRetimerThruHr::Enum was send successefully!\n"));

  return TBT_STATUS_SUCCESS;
}


/**
  Indicate to the device about the flow state

  @param[in] This The pointer to the interface, where this function is defined.
  @param[in] State Indicate to the device about flow state, as it might be required to perform some steps.

**/
STATIC
VOID
StateFromDriver (
  IN TARGET_DEVICE    *This,
  IN DRIVER_STATE     State
  )
{
  if (State == AFTER_AUTH) {
    // Need to send ENUM again
    SendEnumCmd((RETIMER_THRU_HR *)This->Impl);
  }
}

/**
  Send LSUP disable command to the target retimer
  TODO: add description

**/
STATIC
TBT_STATUS
SendLsupCmdDis (
  IN RETIMER_THRU_HR *Retimer
  )
{
  TBT_STATUS status;

  DEBUG ((DEBUG_INFO, "\nTbtNvmDrvRetimerThruHr::Sending LSUP to port %s of the HR, retimer index - %d.\
    The required operation is disable\n",
    Retimer->DnFacingPort == PORT_PA ? L"PA" : L"PB", (UINT32)Retimer->RetimerIndex));

  status = TbtNvmDrvSendCmd(Retimer->Comm, TBT_IECS_CMD_USUP);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "USUP send is failed!\n"));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }

  DEBUG ((DEBUG_INFO, "TbtNvmDrvRetimerThruHr::USUP was send successefully!\n"));

  return TBT_STATUS_SUCCESS;
}

/**
  Send LSUP enable command to the target retimer
  TODO: add description

**/
STATIC
TBT_STATUS
SendLsupCmdEn (
  IN RETIMER_THRU_HR *Retimer
  )
{
  TBT_STATUS status;

  DEBUG ((DEBUG_INFO, "\nTbtNvmDrvRetimerThruHr::Sending LSUP to port %s of the HR, retimer index - %d. \
         The required operation is enable\n",
    Retimer->DnFacingPort == PORT_PA ? "PA" : "PB", (UINT32)Retimer->RetimerIndex));

  status = TbtNvmDrvSendCmd(Retimer->Comm, TBT_IECS_CMD_LSUP);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::LSUP command send had failed\n"));
    return status;
  }

  DEBUG ((DEBUG_INFO, "TbtNvmDrvRetimerThruHr::LSUP was send successefully!\n"));
  return TBT_STATUS_SUCCESS;
}

/**
  Destroy the given connection
  This module destructor.
**/
STATIC
VOID
Dtor (
  IN TARGET_DEVICE *This
  )
{
  RETIMER_THRU_HR *retimer = (RETIMER_THRU_HR *)This->Impl;
  TBT_STATUS status;
  UINT32 data;

  DEBUG ((DEBUG_INFO, "TbtNvmDrvRetimerThruHr::Dtor was called.\n"));

  status = SendLsupCmdDis(retimer);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::Dtor Failed to send LSUP to retimer\n."));
  }
  data = 0x1;
  status = SendCommandToLocalLc(retimer, TBT_IECS_CMD_LSEN, &data);  // Enable LS back
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHr::Dtor Failed to enable LS in HR\n."));
  }

  // Destroy all resources
  retimer->Hr->Dtor(retimer->Hr);
  TbtNvmDrvDeAllocateMem(retimer);
  TbtNvmDrvDeAllocateMem(This);
}

/**
  Constructs the Retimer thru HR module.
  Initializes all the internal data structures and initialize the required HW.

  @param[in] PcieBdf        The BDF of the HR connected to the target retimer.
  @param[in] DnFacingPort   The downstream port, connected to the target retimer.
  @param[in] RetimerIndex   The index of the retimer in the path from HR to the target retimer.
  @param[in] ForcePwrFunc   HR Force power function, applied if supplied.

  @retval A pointer to the device interface.

**/
TARGET_DEVICE*
TbtNvmDrvRetimerThruHrCtor (
  IN PCIE_BDF       PcieBdf,
  IN DN_PORT        DnFacingPort,
  IN RETIMER_INDEX  RetimerIndex,
  IN FORCE_PWR_HR   ForcePwrFunc OPTIONAL
  )
{
  EFI_STATUS        status;
  RETIMER_THRU_HR   *pRetimer;
  TARGET_DEVICE     *pCommunication;
  TBT_HR            *pHr;
  UINT32            data;

  // Create all the resources
  pRetimer = TbtNvmDrvAllocateMem(sizeof(RETIMER_THRU_HR));
  if (pRetimer == NULL) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHrCtor: Failed to allocate memory for RETIMER_THRU_HR\n"));
    return NULL;
  }
  pCommunication = TbtNvmDrvAllocateMem(sizeof(TARGET_DEVICE));
  if (pCommunication == NULL) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHrCtor: Failed to allocate memory for DEVICE_COMMUNICATION_INTERFACE\n"));
    goto free_retimer;
  }
  pHr = TbtNvmDrvHrCtor(PcieBdf, ForcePwrFunc);
  if (pHr == NULL) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHrCtor: Failed to create DMA\n"));
    goto free_communication;
  }

  // Assign communication interface implementation
  pCommunication->Impl            = pRetimer;
  pCommunication->WriteIecsReg    = WriteIecs;
  pCommunication->ReadIecsReg     = ReadIecs;
  pCommunication->Destroy         = Dtor;
  pCommunication->IndicateState   = StateFromDriver;

  pRetimer->Comm = pCommunication;
  pRetimer->DnFacingPort = DnFacingPort;
  pRetimer->RetimerIndex = RetimerIndex;
  pRetimer->Hr = pHr;

  // Send LS disable, to prevent LC to bring up the link
  data = 0;
  status = SendCommandToLocalLc(pRetimer, TBT_IECS_CMD_LSEN, &data);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHrCtor: The retimer could not perform LS disable, status %d. Exiting...\n", status));
    goto free_hr;
  }
  // Enumerate retimers on a path to the target retimer
  status = SendEnumCmd(pRetimer);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHrCtor: The retimer could not enumerate, status %d. Exiting...\n", status));
    goto free_hr;
  }

  status = SendLsupCmdEn(pRetimer);
  if (TBT_STATUS_ERR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruHrCtor: The retimer could not perform LSUP, status %d. Exiting...\n", status));
    goto free_hr;
  }

  return pCommunication;

free_hr:
  pHr->Dtor(pHr);
free_communication:
  TbtNvmDrvDeAllocateMem(pCommunication);
free_retimer:
  TbtNvmDrvDeAllocateMem(pRetimer);
  return NULL;
}
