/** @file
  This file serves as the driver entry point. It initializes the driver with specific devices.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:

**/

#include <Library/UefiBootServicesTableLib.h>

#include "TcI2cmDrvLoader.h"
#include "TcI2cmDrvUtils.h"
#include "TcI2cmDrvDriver.h"
#include "TcI2cmDrvRetimerThruI2C.h"

static EFI_GUID gTbtRetimerDeviceGuid =
  {0xC8219E94, 0x61EF, 0x4C0E, {0x8E, 0x27, 0x74, 0xFD, 0x64, 0x7D, 0x77, 0X0E}};

///
/// Forward declaration for the _EFI_TCSS_RETIMER_PROTOCOL.
///
typedef struct _EFI_TCSS_RETIMER_PROTOCOL EFI_TCSS_RETIMER_PROTOCOL;
typedef struct _EFI_TCSS_RETIMER_PROTOCOL_CAPABILITY EFI_TCSS_RETIMER_PROTOCOL_CAPABILITY;


// {B9AC2BD1-D450-4BEC-BD5B-192601119B84}
#define EFI_TCSS_RETIMER_PROTOCOL_GUID \
  { \
    0xB9AC2BD1, 0xD450, 0x4BEC, { 0xBD, 0x5B, 0x19, 0x26, 0x1, 0x11, 0x9B, 0x84 } \
  }
static EFI_GUID gEfiTcssRetimerProtocolGuid = EFI_TCSS_RETIMER_PROTOCOL_GUID;

#pragma pack (push,1)
//
// Interface structure for the Tcss retimer Protocol
//
typedef
EFI_STATUS
(EFIAPI *EFI_FW_UPDATE_MODE_ENTRY)(
  IN EFI_TCSS_RETIMER_PROTOCOL  *This,
  IN EFI_GUID                   RetimerGuid ///< GUID from ESRT ACPI Table
);

typedef
EFI_STATUS
(EFIAPI *EFI_FW_UPDATE_MODE_EXIT)(
  IN EFI_TCSS_RETIMER_PROTOCOL  *This,
  IN EFI_GUID                   RetimerGuid ///< GUID from ESRT ACPI Table
);

struct _EFI_TCSS_RETIMER_PROTOCOL_CAPABILITY {
  UINT8   RetimerType;
  UINT8   RetimerDeviceCount;
  UINT16  RetimerDeviceIndexBitMap;
  UINT8   Reserved[12];
};

typedef
EFI_STATUS
(EFIAPI *EFI_TCSS_RETIMER_INFO)(
  IN EFI_TCSS_RETIMER_PROTOCOL              *This,
  IN EFI_GUID                               RetimerGuid, ///< GUID from ESRT ACPI Table
  OUT EFI_TCSS_RETIMER_PROTOCOL_CAPABILITY  *RetimerInfo
);

struct _EFI_TCSS_RETIMER_PROTOCOL {
  EFI_TCSS_RETIMER_INFO     GetRetimerInfo;   ///< Get TCSS Retimer Device info
  EFI_FW_UPDATE_MODE_ENTRY  Drive;            ///< Drive TCSS retimer device into FW Update mode.
  EFI_FW_UPDATE_MODE_EXIT   Restore;          ///< Restore TCSS retimer device into original mode.
};

static EFI_HANDLE gTBTRetimerHandler = NULL;
static EFI_TCSS_RETIMER_PROTOCOL  *gTBTRetimer = NULL;


/**
  Drives TBT mode thru BIOS->EFI_TCSS_RETIMER_PROTOCOL->Drive.

  @retval  EFI_SUCCESS  The entry point is executed successfully.
  @retval  other        Some error occurs when executing this entry point.
**/
EFI_STATUS
DriveTcssRetimerMode ()
{
  EFI_STATUS  Status;
  UINTN       NumHandles;
  EFI_HANDLE  *HandleBuffer;
  UINTN       Index;

  gTBTRetimer          = NULL;
  gTBTRetimerHandler  = NULL;

  Status = gBS->LocateHandleBuffer (
                  ByProtocol,
                  &gEfiTcssRetimerProtocolGuid,
                  NULL,
                  &NumHandles,
                  &HandleBuffer
                  );
  if (EFI_ERROR(Status)) {
    DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: Cannot LocateHandleBuffer(gEfiTcssRetimerProtocolGuid), status=%r\n", Status));
    return Status;
  }
  DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: NumHandles = %x\n", NumHandles));

  for (Index = 0; Index < NumHandles; ++Index) {
    DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: gBS->HandleProtocol(Index = %x, %p, %g)\n", Index, HandleBuffer[Index], &gEfiTcssRetimerProtocolGuid));
    Status = gBS->HandleProtocol (HandleBuffer[Index], &gEfiTcssRetimerProtocolGuid, (VOID **) &gTBTRetimer);
    if (EFI_ERROR(Status)) {
      DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: gBS->HandleProtocol ret = %s\n", Status));
      continue;
  }

    // Save the handle
    gTBTRetimerHandler = HandleBuffer[Index];
    DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: gTBTRetimerHandler found = %x:%x\n", gTBTRetimerHandler, gTBTRetimer));
    break;
  }

  if(Status == EFI_SUCCESS && gTBTRetimerHandler != NULL && gTBTRetimer != NULL) {
    DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: gTBTRetimer->Drive(%g)\n", &gTbtRetimerDeviceGuid));
    Status = gTBTRetimer->Drive(gTBTRetimer, gTbtRetimerDeviceGuid);

    if (Status == EFI_SUCCESS) {
      DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: gTBTRetimer->Drive call succeed, waiting 2 sec\n"));
    gBS->Stall(1000 * 1000 * 2);
  } else {
      DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: gTBTRetimer->Drive call failed: %r\n", Status));
  }
  } else {
    DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: Cannot find gTBTRetimer protocol (%r:%x:%x)\n", Status, gTBTRetimerHandler, gTBTRetimer));
  gTBTRetimerHandler = NULL;
  gTBTRetimer = NULL;
  }
  DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode: gBS->FreePool\n"));
  (gBS->FreePool)(HandleBuffer);
  DEBUG ((DEBUG_ERROR, "DriveTcssRetimerMode all done: %r\n", Status));
  return Status;
}

/**
  Restores original TBT mode thru BIOS->EFI_TCSS_RETIMER_PROTOCOL->Restore.

  @retval  EFI_SUCCESS  The entry point is executed successfully.
  @retval  other        Some error occurs when executing this entry point.
**/
void
RestoreTcssRetimerMode ()
{
  DEBUG ((DEBUG_ERROR, "RestoreTcssRetimerMode, (%x:%x)\n", gTBTRetimerHandler, gTBTRetimer));

  if(gTBTRetimerHandler == NULL || gTBTRetimer == NULL) {
    DEBUG ((DEBUG_ERROR, "RestoreTcssRetimerMode: Nothing to restore\n"));
  } else {
    EFI_STATUS Status = gTBTRetimer->Restore(gTBTRetimer, gTbtRetimerDeviceGuid);

    if (Status == EFI_SUCCESS) {
      DEBUG ((DEBUG_ERROR, "RestoreTcssRetimerMode: gTBTRetimer->Restore call succeed, waiting 2 sec\n"));
      gBS->Stall(1000 * 1000 * 2);
    } else {
      DEBUG ((DEBUG_ERROR, "RestoreTcssRetimerMode: gTBTRetimer->Restore call failed: %r\n", Status));
    }
  }

  gTBTRetimer         = NULL;
  gTBTRetimerHandler  = NULL;
}

/**
  The user Entry Point for Application. The user code starts with this function
  as the real entry point for the application.

  @param[in]  ImageHandle  The firmware allocated handle for the EFI image.
  @param[in]  SystemTable  A pointer to the EFI System Table.

  @retval  EFI_SUCCESS  The entry point is executed successfully.
  @retval  other        Some error occurs when executing this entry point.
**/
EFI_STATUS
EFIAPI
TbtNvmDrvEntryPoint (
  IN EFI_HANDLE        ImageHandle,
  IN EFI_SYSTEM_TABLE  *SystemTable
  )
{
  /* This defines how many devices to update. Currently we support only one device per driver */
  EFI_STATUS     status;
  TARGET_DEVICE  *device;
  EFI_INPUT_KEY  Key;
  UINTN          EventIndex;

  gST->ConOut->ClearScreen(gST->ConOut);
  gST->ConOut->SetAttribute(gST->ConOut, EFI_YELLOW);
  gST->ConOut->EnableCursor(gST->ConOut, FALSE);
  DEBUG ((DEBUG_ERROR, "Starting NVM update on the retimer with the GUID = %g\n", &TBT_RETIMER_GUID));
  DEBUG ((DEBUG_ERROR, "TBT_RETIMER_SLAVE_ADDR = 0x%02x, I2C_BUS_NUMBER = 0x%02x\n", TBT_RETIMER_SLAVE_ADDR, I2C_BUS_NUMBER));

  DEBUG ((DEBUG_ERROR, "Press any key\n"));
  gBS->WaitForEvent (1, &gST->ConIn->WaitForKey, &EventIndex);
  gST->ConIn->ReadKeyStroke (gST->ConIn, &Key);

  //---------------------------------------------------------------------------
  // Force TBT mode
  //

  DEBUG ((DEBUG_ERROR, "Forcing TBT mode\n"));
  status = DriveTcssRetimerMode();

  if (EFI_ERROR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtDrvEntryPoint: DriveTcssRetimerMode failed, status=%r\n", status));
    return status;
  }

  //---------------------------------------------------------------------------
  // Repeat the next lines per device to be updated */
  //
  device = TypeCNvmDrvRetimerThruI2cCtor (TBT_RETIMER_SLAVE_ADDR, I2C_BUS_NUMBER);
  if (device == NULL) {
    status = EFI_DEVICE_ERROR;
    goto free_methods;
  }

  return EFI_SUCCESS;

free_methods:
  if (device != NULL) {
    device->Destroy(device);
  }
  TcssNvmDrvDeAllocateMem(device);
  return status;
}
