/** @file
  Implements the interface for TbtNvmDrvRetimerThruI2C class.
  This class is in charge of providing the way to access the retimer through I2C.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:

**/

#include <Bus/I2c/I2cDxe/I2cDxe.h>
#include <Library/PciSegmentLib.h>
#include <IndustryStandard/Pci.h>

#include <Library/I2cAccessLib.h>
#include <Register/SerialIoRegs.h>
#include <Library/SerialIoAccessLib.h>

#include "TcI2cmDrvRetimerThruI2C.h"

void
RestoreTcssRetimerMode ();

#ifndef WAIT_1_SECOND
#define WAIT_1_SECOND            1600000000 //1.6 * 10^9
#endif

#define IECS_DATA_ADDR 9

/// The TBT retimer accepts SMBus formatted data,
/// this struct helps to transmit SMBus protocol over I2C
typedef struct _SMBUS_TO_I2C
{
  UINT8 Command;
  UINT8 ByteCount;
  UINT8 Data[64];
} SMBUS_TO_I2C;

typedef struct _I2C_LIB_PROTOCOL I2C_LIB_PROTOCOL;

/**
  Function to write IECS register in the target device.

  @param[in]  This    The pointer to the interface, where this function is defined.
  @param[in]  RegNum  IECS register number.
  @param[in]  Data    Pointer to 32b data to write.
  @param[in]  Length  How many DWs to write.
**/
typedef
EFI_STATUS
(EFIAPI *I2C_LIB_WRITE)(
  IN I2C_LIB_PROTOCOL   *This,
  IN UINT8            RegNum,
  IN UINT32           *Data,
  IN UINT8            Length
);

/**
  Function to read IECS register from the target device.

  @param[in]   This    The pointer to the interface, where this function is defined.
  @param[in]   RegNum  IECS register number.
  @param[out]  Data    Pointer to the buffer.
**/
typedef
EFI_STATUS
(EFIAPI *I2C_LIB_READ)(
  IN I2C_LIB_PROTOCOL    *This,
  IN UINT8            RegNum,
  OUT UINT32          *Data
);

/// The interface to access the target device IECS registers using I2cAccessLib.
struct _I2C_LIB_PROTOCOL {
  UINTN              MmioBase;
  UINT8              SlaveAddress;
  I2C_LIB_WRITE      WriteIecsReg;
  I2C_LIB_READ       ReadIecsReg;
};

/// Delays Read/Write access call to I2C device
UINTN I2CAccessDelay = 40*1000;

/**
  Function to write IECS register in the target device using I2CAccessLib.

  @param[in]  This    The pointer to the I2C_LIB_PROTOCOL, where this function is defined.
  @param[in]  RegNum  IECS register number.
  @param[in]  Data    Pointer to 32b data to write.
  @param[in]  Length  How many Bytes to write.
**/
EFI_STATUS
I2C_Lib_Write(
  IN I2C_LIB_PROTOCOL *This,
  IN UINT8            RegNum,
  IN UINT32           *Data,
  IN UINT8            Length
  )
{
  EFI_STATUS    Status = EFI_UNSUPPORTED;
  SMBUS_TO_I2C smbus_2_i2c;
  //DEBUG ((DEBUG_ERROR, "I2C_Lib_Write(RegNum=0x%X, Length=%X)\n", RegNum, Length));

  gBS->Stall(I2CAccessDelay);

  if(Length > 64 || Data == NULL) {
    DEBUG ((DEBUG_ERROR, "I2C_Lib_Write: Length(%d) > 64 | Data = %x\n", Length, Data));
    return EFI_INVALID_PARAMETER;
  }

  smbus_2_i2c.Command = RegNum;
  smbus_2_i2c.ByteCount = Length;

// To read FW version as 1 DW it is required to add 0x0f000000 to the offset
  if(RegNum == IECS_DATA_ADDR && Length == 4) {
    UINT32 Date_9 = *Data | 0x0f000000;
  gBS->CopyMem(smbus_2_i2c.Data, &Date_9, Length);
    DEBUG ((DEBUG_ERROR, "I2C_Lib_Write: Data[0] = %X\n", Date_9));
  }
  else {
    gBS->CopyMem(smbus_2_i2c.Data, Data, Length);
    DEBUG ((DEBUG_ERROR, "I2C_Lib_Write: Data[0] = %X\n", *Data));
  }
  Status = I2cWriteRead (This->MmioBase, This->SlaveAddress, smbus_2_i2c.ByteCount + 2, &smbus_2_i2c.Command, 0, NULL, WAIT_1_SECOND);
  DEBUG ((DEBUG_ERROR, "I2C_Lib_Write: I2cWriteRead = %r\n", Status));
  return Status;
}

/**
  Function to read IECS register from the target device using I2CAccessLib.

  @param[in]   This    The pointer to the I2C_LIB_PROTOCOL, where this function is defined.
  @param[in]   RegNum  IECS register number.
  @param[out]  Data    Pointer to the buffer.
**/
EFI_STATUS
I2C_Lib_Read(
  IN I2C_LIB_PROTOCOL  *This,
  IN UINT8             RegNum,
  OUT UINT32           *Data
  )
{
  EFI_STATUS    Status = EFI_UNSUPPORTED;
  SMBUS_TO_I2C smbus_2_i2c;
  //DEBUG ((DEBUG_ERROR, "I2C_Lib_Read(RegNum=0x%X)\n", RegNum));

  gBS->Stall(I2CAccessDelay);

  if(Data == NULL) {
    DEBUG ((DEBUG_ERROR, "I2C_Lib_Read: Data = %x\n", Data));
    return EFI_INVALID_PARAMETER;
  }

  smbus_2_i2c.Command = RegNum;
  smbus_2_i2c.ByteCount = 0;

  Status = I2cWriteRead (This->MmioBase, This->SlaveAddress, 1, &smbus_2_i2c.Command, 5, &smbus_2_i2c.ByteCount, WAIT_1_SECOND/2); // Wait 500 ms
  if(EFI_SUCCESS == Status) {
    gBS->CopyMem(Data, smbus_2_i2c.Data, 4);
  }
  else {
    DEBUG ((DEBUG_ERROR, "I2C_Lib_Read: I2cWriteRead = %r\n", Status));
  }
  DEBUG ((DEBUG_ERROR, "I2C_Lib_Read: I2cWriteRead = %r, RegSize = %x, Data=0x%08x\n", Status, smbus_2_i2c.ByteCount, Data));
  return Status;
}

I2C_LIB_PROTOCOL I2C_Lib_Protocol =
{
  0,
  0,
  I2C_Lib_Write,
  I2C_Lib_Read
};

#define TBT_IECS_RDDATA_LEGTH_IN_DW 16

typedef struct {
  TARGET_DEVICE             *Comm;
  I2C_LIB_PROTOCOL     *I2CLibProto;
} RETIMER_THRU_I2C;

/**
  Function to write IECS register in the target device.

  @param[in]  This    The pointer to the interface, where this function is defined.
  @param[in]  RegNum  IECS register number.
  @param[in]  Data    Pointer to 32b data to write.
  @param[in]  Length  How many DWs to write.
**/
TBT_STATUS
TbtNvmDrvRetimerThruI2CWriteIecs(
  IN TARGET_DEVICE    *This,
  IN UINT8            RegNum,
  IN UINT32           *Data,
  IN UINT8            Length
  )
{
  //DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CWriteIecs(RegNum=0x%X, Length=0x%X)\n", RegNum, Length));
  ASSERT(Length <= TBT_IECS_RDDATA_LEGTH_IN_DW && Length > 0);
  ASSERT(This != NULL);
  if (This == NULL || Length > TBT_IECS_RDDATA_LEGTH_IN_DW || Length == 0) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CWriteIecs=This=%x || Length(%d) > TBT_IECS_RDDATA_LEGTH_IN_DW(%d) || Length == 0\n",
    This, Length, TBT_IECS_RDDATA_LEGTH_IN_DW));
    return TBT_STATUS_INVALID_PARAM;
  }
  DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CWriteIecs: IECS reg num - 0x%x, Length - 0x%x\n", RegNum, Length));
  RETIMER_THRU_I2C *retimer = (RETIMER_THRU_I2C *)This->Impl;
  EFI_STATUS status = EFI_SUCCESS;
  UINT8 lengthInByte = (UINT8) Length << 2;

  status = retimer->I2CLibProto->WriteIecsReg(
                                              retimer->I2CLibProto,
                                              RegNum,
                                              Data,
                                          lengthInByte
                                             );

  if (EFI_ERROR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CWriteIecs=%r\n", status));
    DEBUG ((DEBUG_ERROR, "Got an error while writing(SlaveAddress=0x%x, RegNum=0x%x, length=0x%x, first DW is %x\n",
    retimer->I2CLibProto->SlaveAddress, RegNum, lengthInByte, *Data));
  }

  if (status==EFI_SUCCESS) return TBT_STATUS_SUCCESS;

  return TBT_STATUS_NON_RECOVERABLE_ERROR;
}

/**
  Function to read IECS register from the target device.

  @param[in]   This    The pointer to the interface, where this function is defined.
  @param[in]   RegNum  IECS register number.
  @param[out]  Data    Pointer to the buffer.
**/
TBT_STATUS
TbtNvmDrvRetimerThruI2CReadIecs (
  IN TARGET_DEVICE    *This,
  IN UINT8            RegNum,
  OUT UINT32          *Data
  )
{
  ASSERT(This != NULL);
  ASSERT(Data != NULL);
  //DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CReadIecs(RegNum=0x%X)\n", RegNum));

  if (This == NULL || Data == NULL) {
    return TBT_STATUS_INVALID_PARAM;
  }
  DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CReadIecs: IECS reg num - 0x%x\n", RegNum));
  RETIMER_THRU_I2C *retimer = (RETIMER_THRU_I2C *)This->Impl;
  EFI_STATUS status = EFI_SUCCESS;

  status = retimer->I2CLibProto->ReadIecsReg(
    retimer->I2CLibProto,
    RegNum,
    Data
  );

  if (EFI_ERROR(status)) {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CReadIecs=%r: Got an error while reading from reg %x, len 4\n", status, RegNum));
    return TBT_STATUS_NON_RECOVERABLE_ERROR;
  }
  else {
    DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CReadIecs: Data=0x%08x\n", *Data));
  }

  return TBT_STATUS_SUCCESS;
}

/**
  Indicate to the device about the flow state.

  @param[in]  This   The pointer to the interface, where this function is defined.
  @param[in]  State  Indicate to the device about flow state, as it might be required to perform some steps.
**/
VOID
TbtNvmDrvRetimerThruI2CStateFromDriver (
  IN TARGET_DEVICE   *This,
  IN DRIVER_STATE State
  )
{
  // nothing to be done yet
}

/**
  Function to disconnect the interface and release the implementation.

  @param[in]  This  The pointer to the interface, where this function is defined.
**/
VOID
TbtNvmDrvRetimerThruI2CDestroy (
  IN TARGET_DEVICE   *This
  )
{
  RETIMER_THRU_I2C *retimer = (RETIMER_THRU_I2C *)This->Impl;
  DEBUG ((DEBUG_ERROR, "TbtNvmDrvRetimerThruI2CDestroy was called.\n"));

  TcssNvmDrvDeAllocateMem(retimer);
  TcssNvmDrvDeAllocateMem(This);

  RestoreTcssRetimerMode();
}

/**
  Constructs the Retimer thru I2C.

  @param[in]  SlaveAddress  The slave address of target Retimer device.
  @param[in]  I2CBusNumber  The bus number of I2C host controller that connects to Retimer device.
**/
TARGET_DEVICE*
TypeCNvmDrvRetimerThruI2cCtor (
  IN UINT8 SlaveAddress,
  IN UINT8 I2CBusNumber
  )
{
  TARGET_DEVICE             *pCommunication;
  RETIMER_THRU_I2C          *pRetimer;
  UINT64                    MmioBase;
  UINT64                    PciRegisterAddress;
  UINT8                     PciPmeCtrlSts;
  UINT8                     PciCommand;
  UINT32                    MemResetState;

  DEBUG ((DEBUG_ERROR, "TypeCNvmDrvRetimerThruI2cCtor()\n"));

  // Create all the resources
  pRetimer = TcssNvmDrvAllocateMem(sizeof(RETIMER_THRU_I2C));
  if (pRetimer == NULL) {
    DEBUG ((DEBUG_ERROR, "TypeCNvmDrvRetimerThruI2cCtor: Failed to allocate memory for RETIMER_THRU_I2C\n"));
    return NULL;
  }
  pCommunication = TcssNvmDrvAllocateMem(sizeof(TARGET_DEVICE));
  if (pCommunication == NULL) {
    DEBUG ((DEBUG_ERROR, "TypeCNvmDrvRetimerThruI2cCtor: Failed to allocate memory for DEVICE_COMMUNICATION_INTERFACE\n"));
    goto free_retimer;
  }
  DEBUG ((DEBUG_ERROR, "CntrlNum = 0x%x, SlaveAddress = 0x%x\n", I2CBusNumber, SlaveAddress));
  PciRegisterAddress  = GetSerialIoI2cPciCfg ((UINT8)(I2CBusNumber));
  DEBUG ((DEBUG_ERROR, "PciRegisterAddress = 0x%08x\n", PciRegisterAddress));
  MmioBase = GetSerialIoBar (PciRegisterAddress);
  DEBUG ((DEBUG_ERROR, "MmioBase = 0x%08x\n", MmioBase));

  //
  // Enable the PCI memory resource if it's disabled.
  //
  PciCommand    = PciSegmentRead8 (PciRegisterAddress + PCI_COMMAND_OFFSET);
  PciPmeCtrlSts = PciSegmentRead8 (PciRegisterAddress + R_SERIAL_IO_CFG_PME_CTRL_STS);
  MemResetState = MmioRead32 (MmioBase + R_SERIAL_IO_MEM_PPR_RESETS);

  if ((PciCommand & EFI_PCI_COMMAND_MEMORY_SPACE) == 0) {
    PciSegmentOr8 (PciRegisterAddress + PCI_COMMAND_OFFSET, EFI_PCI_COMMAND_MEMORY_SPACE);
  }

  if ((PciPmeCtrlSts & (B_SERIAL_IO_CFG_PME_CTRL_STS_PWR_ST)) == (B_SERIAL_IO_CFG_PME_CTRL_STS_PWR_ST)) {
    PciSegmentAnd8 (PciRegisterAddress + R_SERIAL_IO_CFG_PME_CTRL_STS, (UINT8) (~B_SERIAL_IO_CFG_PME_CTRL_STS_PWR_ST));
  }

  MemResetState = MmioRead32 ((UINTN)MmioBase + R_SERIAL_IO_MEM_PPR_RESETS);
  DEBUG ((DEBUG_ERROR, "  MemResetState = 0x%08X\n", MemResetState));
  if ((MemResetState & (0x7)) == 0) { // BIT0 | BIT1 | BIT2
    MmioOr32 ((UINTN)MmioBase + R_SERIAL_IO_MEM_PPR_RESETS,
      B_SERIAL_IO_MEM_PPR_RESETS_FUNC | B_SERIAL_IO_MEM_PPR_RESETS_APB | B_SERIAL_IO_MEM_PPR_RESETS_IDMA);
  }

  pCommunication->Impl          = pRetimer;
  pCommunication->WriteIecsReg  = TbtNvmDrvRetimerThruI2CWriteIecs;
  pCommunication->ReadIecsReg   = TbtNvmDrvRetimerThruI2CReadIecs;
  pCommunication->Destroy       = TbtNvmDrvRetimerThruI2CDestroy;
  pCommunication->IndicateState = TbtNvmDrvRetimerThruI2CStateFromDriver;

  pRetimer->Comm            = pCommunication;
  pRetimer->I2CLibProto      = &I2C_Lib_Protocol;
  pRetimer->I2CLibProto->MmioBase = MmioBase;
  pRetimer->I2CLibProto->SlaveAddress = SlaveAddress;
  return pCommunication;

free_retimer:
  TcssNvmDrvDeAllocateMem(pRetimer);
  return NULL;
}
