#
# This script auto gens uCode Capsule
# uCode Capsule is a FMP capsule with a XDR format payload
# XDR format payload carries
#       1. uCode FV
#       2. Full range BgupScript
#       3. uCode Version FFS
#       4. uCode Arrary
#
import os
import sys
import argparse
import subprocess
import glob
import shutil
import struct
import xdrlib
import re

gIntelMicrocodeFirmwareVolumeGuid   = '9BC06401-13CE-4A6A-B2CD-31D874793B70'
gIntelMicrocodeVersionFfsFileGuid   = 'E0562501-B41B-4566-AC0F-7EA8EE817F20'
gIntelMicrocodeArrayFfsFileGuid     = '197DB236-F856-4924-90F8-CDF12FB875F3'
gFmpDevicePlatformuCodeGuid         = '69585D92-B50A-4AD7-B265-2EB1AE066574'
# gIntelFmpCapsuleGuid                = '96D4FDCD-1502-424D-9D4C-9B12D2DCAE5C'

#
# Globals for help information
#
__prog__        = sys.argv[0]
__copyright__   = 'Copyright (c) 2019, Intel Corporation. All rights reserved.'
__description__ = 'Generate Microcode capsules.\n'

def RemoveTempDirect(Target):
    if os.path.exists(Target):
        shutil.rmtree(Target)

def CreateTempDirect(Target):
    RemoveTempDirect(Target)
    os.makedirs(Target)

def GenMicrocodeArrayRawData(uCodeFileList, Output):
    Buffer = b''
    for File in uCodeFileList:
        with open(File, "rb") as fd:
          Buffer = Buffer + fd.read()

    with open(Output, "wb") as fd:
        fd.write(Buffer)

def ProcessCommand(Command):
    Process = subprocess.Popen(Command, stdin=subprocess.PIPE, stdout=subprocess.PIPE, stderr=subprocess.PIPE, shell=True)
    ProcessOutput = Process.communicate()
    if Process.returncode != 0:
        print (Command)
        print (ProcessOutput[1].decode())
        assert (False)


def GenMicrocodeFv(Fdf, Dsc, Output):
    GenFvCommand = '''
        call {GenFds}
        -f {fdf}
        -p {dsc}
        -i {Section}
        -o {OutputDir}
        '''
    Command = GenFvCommand.format (
                GenFds = "GenFds.bat",
                fdf = Fdf,
                dsc = Dsc,
                Section = "CapsulePayloaduCode",
                OutputDir = Output
                )
    ProcessCommand(' '.join(Command.splitlines()).strip())

def GenBgupSignedData(BgupDirectory, MicrocodeFv):
    GenSignedDataCommand = '''
        call {BuildBGUP_SPI}
        -d {MicrocodeFv}
        -p {Bgsl}
        -v {BIOS_SVN}
        '''
    Command = GenSignedDataCommand.format (
                BuildBGUP_SPI = os.path.join(BgupDirectory, "BuildBGUP_SPI.bat"),
                MicrocodeFv = MicrocodeFv,
                Bgsl = os.path.join(BgupDirectory, "script_BuildBGUP_template_TopSwapUpdate.bgsl"),
                BIOS_SVN = 2422
                )
    ProcessCommand(' '.join(Command.splitlines()).strip())

def GetRealFilePath(Path):
    if len(Path.split()) > 1:
        FilePath = Path.split()[-1]
    else:
        FilePath = Path
    index = 0
    while True:
        index1 = FilePath.find("$(", index)
        if index1 == -1:
            break
        index2 = FilePath.find(")",index1)
        FilePath = FilePath.replace(FilePath[index1:index2+1], os.environ[FilePath[index1+2:index2]])
        index = index2 + 1
    return FilePath

def GetMicrocodeListFromFdf(File):
    with open(File, "r") as fd:
        data = fd.read()

    result = re.findall(r"FILE *RAW *= *197DB236-F856-4924-90F8-CDF12FB875F3", data)
    #
    # assumption: just one fv for microcode
    #
    assert(len(result) == 1)

    #
    # Find FILE RAW section according 197DB236-F856-4924-90F8-CDF12FB875F3 GUID
    #
    index1 = data.find(result[0])
    index2 = data.find("}", index1)
    count = data[index1:index2].count("{") - 1
    while count > 0:
        index2 = data.find("}", index2 + 1)
    index2 = index2 + 1

    PatchList = []
    for line in data[index1:index2].split("\n"):
        line = line.strip("\n").strip()

        if line.find("#") != -1:
            index1 = line.find("#")
            index2 = len(line)
            line = line.replace(line[index1:index2],"")

        if len(line) == 0:
            continue

        line = line.strip("\n").strip()
        if line[-4:] not in [".pdb", ".mcb", ".bin"]:
            continue

        PatchList.append(os.path.join(os.environ["WORKSPACE_PLATFORM"], GetRealFilePath(line)))
    return PatchList

if __name__ == "__main__":
    def Validate32BitInteger (Argument):
        try:
            Value = int (Argument, 0)
        except:
            Message = '{Argument} is not a valid integer value.'.format (Argument = Argument)
            raise argparse.ArgumentTypeError (Message)
        if Value < 0:
            Message = '{Argument} is a negative value.'.format (Argument = Argument)
            raise argparse.ArgumentTypeError (Message)
        if Value > 0xffffffff:
            Message = '{Argument} is larger than 32-bits.'.format (Argument = Argument)
            raise argparse.ArgumentTypeError (Message)
        return Value

    def ValidateSlotSize (Argument):
        try:
            Value = int (Argument, 0)
        except:
            Message = '{Argument} is not a valid integer value.'.format (Argument = Argument)
            raise argparse.ArgumentTypeError (Message)
        if Value % (4096) != 0:
            Message = '{Argument} must be a multiple of 4KB'.format (Argument = Argument)
            raise argparse.ArgumentTypeError (Message)
        return Value

    #
    # Create command line argument parser object
    #
    parser = argparse.ArgumentParser (
                        prog = __prog__,
                        description = __description__ + __copyright__,
                        )

    #
    # For MicrocodeVersion.ffs
    #
    parser.add_argument ("--fw-version", dest = 'FwVersion', type = Validate32BitInteger,
                         help = 'The 32-bit version of the binary payload (e.g. 0x11223344 or 5678).')
    parser.add_argument ("--lsv", dest = 'LowestSupportedVersion', type = Validate32BitInteger,
                         help = 'The 32-bit lowest supported version of the binary payload (e.g. 0x11223344 or 5678).')
    parser.add_argument ("--fw-version-string", dest = 'VersionString',
                         help = 'The version string of the binary payload (e.g. "Version 0.1.2.3").')

    #
    # Support Slot,Bgup,Full mode
    #
    parser.add_argument ("-m", "--mode", dest = 'Mode', choices = ["ucodebgup", "ucodefull", "ucodeslot"], required = True,
                         help = 'Capsule mode')

    args, remaining = parser.parse_known_args ()

    FwVersion = os.environ["FW_VERSION"]
    Lsv = os.environ["LSV"]
    FwVersionString = os.environ["FW_VERSION_STRING"]
    SlotSize = int(os.environ["SLOT_SIZE"], 16)

    if args.FwVersion != None:
        FwVersion = str(hex(args.FwVersion))
    if args.LowestSupportedVersion != None:
        Lsv = str(hex(args.LowestSupportedVersion))
    if args.VersionString != None:
        FwVersionString = str(args.VersionString)

    #
    # Init folder and path
    #
    CWD                         = os.getcwd()
    PLATFORM_BOARD_PACKAGE      = os.path.join(os.environ["WORKSPACE_PLATFORM"], os.environ["PLATFORM_BOARD_PACKAGE"])
    TEMP_FOLDER                 = os.path.join(CWD, "Temp")
    SCRIPT_FOLDER               = os.path.join(CWD, "PyScript")
    OUTOUT_FOLDER               = os.path.join(CWD, "Output")
    WINDOWS_CAP_FOLDER          = os.path.join(CWD, "WindowsCapsule")
    WINDOWS_CAP_OUTPUT_FOLDER   = os.path.join(OUTOUT_FOLDER, "WindowsCapsule")

    MICROCODE_VERSION_FFS       = os.path.join(PLATFORM_BOARD_PACKAGE, "Binaries", "Microcode", "MicrocodeVersion.data")
    MICROCODE_ARRAY_FFS         = os.path.join(TEMP_FOLDER, "MicrocodeArray.data")

    MICROCODE_FV_FDF            = os.path.join(CWD, "MicrocodeFv", "MicrocodeFv.fdf")
    MICROCODE_FV_DSC            = os.path.join(CWD, "MicrocodeFv", "MicrocodeFv.dsc")

    SCRIPT_PY_GENXDR            = os.path.join(SCRIPT_FOLDER, "GenXdr.py")
    SCRIPT_PY_GENBGUP           = os.path.join(SCRIPT_FOLDER, "GenBgup.py")

    CreateTempDirect (TEMP_FOLDER)

    #
    # Call microcode_padding.py
    #
    os.chdir(PLATFORM_BOARD_PACKAGE)
    os.system("py -3 microcode_padding.py" + " " + FwVersion + " " + Lsv + " " + FwVersionString + " " + str(hex(SlotSize)) + " " + MICROCODE_FV_FDF)
    os.chdir(CWD)

    #
    # Genearte MicrocodeFv
    #
    GenMicrocodeFv (MICROCODE_FV_FDF, MICROCODE_FV_DSC, TEMP_FOLDER)
    # while os.path.exists(TEMP_MICROCODE_VERSION_FFS) != 1 :
    #     print("MicrocodeVersion.data is not ready\n")
    #     if os.path.exists(TEMP_MICROCODE_VERSION_FFS):
    #         break

    #
    # Insert XDR structure
    #
    if args.Mode.lower() == "ucodefull":
        GenXdr = '''
            py -3 {GenXdr}
            -i {MicrocodeFv}
            -o {Output}
            '''
        Command = GenXdr.format (
                    GenXdr = SCRIPT_PY_GENXDR,
                    MicrocodeFv = os.path.join(TEMP_FOLDER, "FV\CAPSULEPAYLOADUCODE.Fv"),
                    Output = os.path.join(TEMP_FOLDER, "XdrMicrocodeFv.data")
                    )
        ProcessCommand(' '.join(Command.splitlines()).strip())

        CapInput  = os.path.join(TEMP_FOLDER, "XdrMicrocodeFv.data")
        CapOutput = os.path.join(OUTOUT_FOLDER, "uCodeFull.cap")

    elif args.Mode.lower() == "ucodebgup":
        GenerateBgup = '''
                py -3 {GenBgup}
                -o {Output}
                --bios-svn {BiosSvn}
                --microcode-fv {MicrocodeFv}
            '''
        Command = GenerateBgup.format (
                    GenBgup = SCRIPT_PY_GENBGUP,
                    Output = TEMP_FOLDER,
                    BiosSvn = 2422,
                    MicrocodeFv = os.path.join(TEMP_FOLDER, "FV\CAPSULEPAYLOADUCODE.Fv")
                    )
        ProcessCommand(' '.join(Command.splitlines()).strip())

        GenXdr = '''
            py -3 {GenXdr}
            -i {MicrocodeFv}
            -i {BgupContent}
            -o {Output}
            '''
        Command = GenXdr.format (
                    GenXdr = SCRIPT_PY_GENXDR,
                    MicrocodeFv = os.path.join(TEMP_FOLDER, "FV\CAPSULEPAYLOADUCODE.Fv"),
                    BgupContent = os.path.join(TEMP_FOLDER, "BgupContent.data"),
                    Output = os.path.join(TEMP_FOLDER, "XdrMicrocodeFvXdrBgup.data")
                    )
        ProcessCommand(' '.join(Command.splitlines()).strip())

        CapInput  = os.path.join(TEMP_FOLDER, "XdrMicrocodeFvXdrBgup.data")
        CapOutput = os.path.join(OUTOUT_FOLDER, "uCodeBgup.cap")

    elif args.Mode.lower() == "ucodeslot":
        GenMicrocodeArrayRawData (GetMicrocodeListFromFdf (MICROCODE_FV_FDF), MICROCODE_ARRAY_FFS)
        GenXdr = '''
            py -3 {GenXdr}
            -i {SlotVersionFfs}
            -i {SlotuCodeArray}
            -o {Output}
            '''
        Command = GenXdr.format (
                    GenXdr = SCRIPT_PY_GENXDR,
                    SlotVersionFfs = MICROCODE_VERSION_FFS,
                    SlotuCodeArray = MICROCODE_ARRAY_FFS,
                    Output = os.path.join(TEMP_FOLDER, "XdrSlotVersionXdrSlotuCodeArray.data")
                    )
        ProcessCommand(' '.join(Command.splitlines()).strip())

        buffer = struct.pack('II', 0, 0)
        with open(os.path.join(TEMP_FOLDER, "XdrSlotVersionXdrSlotuCodeArray.data"), "rb") as File:
            buffer += File.read()

        with open(os.path.join(TEMP_FOLDER, "XdrSlotVersionXdrSlotuCodeArray.data"), "wb") as File:
            File.write(buffer)

        CapInput  = os.path.join(TEMP_FOLDER, "XdrSlotVersionXdrSlotuCodeArray.data")
        CapOutput = os.path.join(OUTOUT_FOLDER, "uCodeSlot.cap")

    #
    # Create output folder
    #
    if not os.path.isdir(OUTOUT_FOLDER):
        CreateTempDirect(OUTOUT_FOLDER)

    #
    # Generate capsule file
    #
    GenerateCapsule = '''
        call {GenerateCapsule}
        --encode
        -v
        --guid {FMP_CAPSULE_GUID}
        --fw-version {FMP_CAPSULE_VERSION}
        --lsv {FMP_CAPSULE_LSV}
        --capflag PersistAcrossReset
        --capflag InitiateReset
        --capoemflag {FMP_CAPSULE_OEM_FLAG}
        --signing-tool-path {OpenSSL}
        --signer-private-cert {PrivateCert}
        --other-public-cert {PublicCert}
        --trusted-public-cert {TrustedPublicCert}
        -o {Output}
        {Input}
        '''
    Command = GenerateCapsule.format (
                GenerateCapsule = "GenerateCapsule.bat",
                FMP_CAPSULE_GUID = gFmpDevicePlatformuCodeGuid,
                FMP_CAPSULE_VERSION = FwVersion,
                FMP_CAPSULE_LSV = Lsv,
                FMP_CAPSULE_OEM_FLAG = 0x0001,
                OpenSSL = os.path.join(os.environ["WORKSPACE"], "Intel\TigerLakePlatSamplePkg\Tools\OpenSSL"),
                PrivateCert = os.path.join(os.environ["WORKSPACE"], "edk2\BaseTools\Source\Python\Pkcs7Sign\TestCert.pem"),
                PublicCert = os.path.join(os.environ["WORKSPACE"], "edk2\BaseTools\Source\Python\Pkcs7Sign\TestSub.pub.pem"),
                TrustedPublicCert = os.path.join(os.environ["WORKSPACE"], "edk2\BaseTools\Source\Python\Pkcs7Sign\TestRoot.pub.pem"),
                Output = CapOutput,
                Input = CapInput
                )
    ProcessCommand(' '.join(Command.splitlines()).strip())

    # #
    # # Gen Windows Capsule file
    # #
    # shutil.copy (CapOutput, WINDOWS_CAP_FOLDER)
    # os.chdir(WINDOWS_CAP_FOLDER)
    # GenerateWindowsCapsule = '''
    #     py -3 {CreateWindowsCapsule}
    #     windowscapsule
    #     "{FMP_CAPSULE_STRING}"
    #     "{FMP_CAPSULE_GUID}"
    #     "{FMP_CAPSULE_FILE}"
    #     "{FMP_CAPSULE_VERSION}"
    #     "Intel"
    #     "Intel"
    #     "uCode"
    #     "SAMPLE_DEVELOPMENT.pfx"
    #     '''
    # Command = GenerateWindowsCapsule.format (
    #             CreateWindowsCapsule = os.path.join(WINDOWS_CAP_FOLDER, "CreateWindowsCapsule.py"),
    #             FMP_CAPSULE_STRING = args.VersionString,
    #             FMP_CAPSULE_GUID = gIntelFmpCapsuleGuid,
    #             FMP_CAPSULE_FILE = CapOutput,
    #             FMP_CAPSULE_VERSION = hex(args.FwVersion,),
    #             )
    # ProcessCommand(' '.join(Command.splitlines()).strip())
    # os.chdir(CWD)

    # #
    # # Copy windows capsule file to output folder
    # #
    # CreateTempDirect(WINDOWS_CAP_OUTPUT_FOLDER)
    # for file in os.listdir(WINDOWS_CAP_FOLDER):
    #     if os.path.splitext(file)[1] in [".cap", ".cat", ".inf"]:
    #         shutil.move(os.path.join(WINDOWS_CAP_FOLDER, file), WINDOWS_CAP_OUTPUT_FOLDER)
