/** @file
  This is part of the implementation of DG1 OpRegion initialization.
  Software SCI interface between system BIOS, ASL code, and Graphics drivers.
  The code in this file will load the driver and initialize the interface

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

  @par Specification Reference:
  - OpRegion / Software SCI SPEC
**/
#include <IndustryStandard/Acpi.h>
#include <Protocol/AcpiSystemDescriptionTable.h>
#include <Library/BaseLib.h>
#include <Library/UefiLib.h>
#include <Library/IoLib.h>
#include <Library/DebugLib.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Library/UefiRuntimeServicesTableLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/TimerLib.h>
#include <Protocol/AcpiTable.h>
#include <Protocol/PciIo.h>
#include <Library/PciSegmentLib.h>
#include <Library/CpuPlatformLib.h>
#include <PcieRegs.h>
#include <Register/SaRegsHostBridge.h>
#include "Dg1OpRegion.h"
#include "HybridGraphicsInfo.h"
#include <Protocol/PlatformNvsArea.h>
#include <SetupVariable.h>

GLOBAL_REMOVE_IF_UNREFERENCED DG1_OPREGION_PROTOCOL  Dg1OpRegion;

/**
  Calculate the OpRegion size

  @param[in] LocalRomImage        A pointer to ROM image.
  @param[in] ConfigDataOffset     Offset to configuration data from base of ROM image.

  @retval    ConfigDataSize       Size of configuration data in bytes

**/
UINT64
GetConfigDataSize (
  IN VBIOS_OPTION_ROM_HEADER  *LocalRomImage,
  IN UINT32                    ConfigDataOffset
)
{
  UINT64 ConfigDataSize;

  ASSERT (LocalRomImage != NULL);
  ConfigDataSize = 0;

  ///
  ///  Offset 0x02 from Rom image base has 2 byte offset to Rom image size.
  ///  The Rom size is 512 times the value at the above offset.
  ///
  ConfigDataSize = (((UINT8 *) LocalRomImage + ROM_SIZE_OFFSET)[0] + (((UINT8 *) LocalRomImage + ROM_SIZE_OFFSET)[1] << 8)) * 512;
  DEBUG ((DEBUG_INFO, "Rom size Offset [0]:0x%x [1]:0x%x \n", ((UINT8 *) LocalRomImage + ROM_SIZE_OFFSET)[0], ((UINT8 *) LocalRomImage + ROM_SIZE_OFFSET)[1]));

  ///
  /// Need to exclude the bytes till the config data offset as it contains
  /// Rom / Pcir header.
  ///
  ConfigDataSize -= ConfigDataOffset;
  DEBUG ((DEBUG_INFO, "GetConfigDataSize calculated OpRegion Size:0x%x \n", ConfigDataSize));

  if (ConfigDataSize < sizeof (DG1_OPREGION_STRUCTURE)) {
    DEBUG ((DEBUG_WARN, "GetConfigDataSize  calculated OpRegion Size less than 8KB, treating as error \n"));
    ASSERT (FALSE);
  }

  return ConfigDataSize;
}

/**
  Check if the device id is DG1

  @param[in] UINT16      DeviceId

  @retval    BOOLEAN     TRUE if DG1 DeviceId , FALSE otherwise.

**/
BOOLEAN
IsDg1Device (
  IN UINT16  DeviceId
)
{
  if (DG1_DEVICE_ID == DeviceId) {
    return TRUE;
  }

  return FALSE;
}

/**
  This function gets registered as a callback to Update DG1 Graphics OpRegion after PCI enumeration for VRAMSR support

  @param[in] Event     - A pointer to the Event that triggered the callback.
  @param[in] Context   - A pointer to private data registered with the callback function.

**/
VOID
EFIAPI
UpdateDg1OpRegionEndOfDxe (
  IN EFI_EVENT    Event,
  IN VOID         *Context
  )
{
  EFI_STATUS                     Status;
  UINTN                          HandleCount;
  PCI_TYPE00                     Pci;
  UINTN                          Index;
  UINTN                          Segment;
  UINTN                          Bus;
  UINTN                          Device;
  UINTN                          Function;
  EFI_HANDLE                     *HandleBuffer;
  EFI_PCI_IO_PROTOCOL            *PciIo;
  VBIOS_PCIR_STRUCTURE           *PcirBlockPtr;
  VBIOS_OPTION_ROM_HEADER        *LocalRomImage;
  UINT8                          *ConfigDataStartAddress;
  UINT32                         ConfigDataOffset;
  PLATFORM_NVS_AREA_PROTOCOL     *PlatformNvsAreaProtocol;
  UINT64                         Dg1BaseAddress;
  UINT64                         ConfigDataSize;
  UINTN                          VariableSize;
  SETUP_DATA                     SetupVariables;

  ConfigDataStartAddress        = NULL;
  Bus                           = 0;
  Device                        = 0;
  Function                      = 0;
  ///
  ///  By default assume the size to be allocated is equal to DG1 OpRegion size.
  ///  which is will be 8K bytes. This is based on assumption that VBT will be
  ///  present in MB4 and it size will be < 6KB
  ///
  ConfigDataSize = sizeof (DG1_OPREGION_STRUCTURE);
  DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe\n"));

  ///
  ///  Get the SA policy and the required Dg1 config block.
  ///
  VariableSize = sizeof (SETUP_DATA);
  ZeroMem (&SetupVariables, VariableSize);
  Status = gRT->GetVariable (
                  L"Setup",
                  &gSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &SetupVariables
                  );
  ASSERT_EFI_ERROR (Status);

  ///
  ///  Locate the SA Global NVS Protocol.
  ///
  Status = gBS->LocateProtocol (&gPlatformNvsAreaProtocolGuid, NULL, (VOID **) &PlatformNvsAreaProtocol);
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe - Not able to locate NVS protocol 0x%x \n", Status));
    return;
  }

  ///
  ///  Get all PCI IO protocols handles
  ///
  Status = gBS->LocateHandleBuffer (
                  ByProtocol,
                  &gEfiPciIoProtocolGuid,
                  NULL,
                  &HandleCount,
                  &HandleBuffer
                  );

  if (EFI_ERROR (Status)) {
     DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe -Not able to get PCI IO handles 0x%x \n", Status));
     return;
  }

  for (Index = 0; Index < HandleCount; Index++) {
    ///
    ///  Get the PCI IO Protocol Interface corresponding to each handle
    ///
    PciIo = NULL;

    Status = gBS->HandleProtocol (
                    HandleBuffer[Index],
                    &gEfiPciIoProtocolGuid,
                    (VOID **) &PciIo
                    );

    if (EFI_ERROR (Status)) {
      ///
      /// Not able to locate the pci io protocol, skip to the next device,
      ///
      continue;
    }

    Status = PciIo->Pci.Read (
                              PciIo,
                              EfiPciIoWidthUint32,
                              0,
                              sizeof (Pci) / sizeof (UINT32),
                              &Pci
                              );

    if (EFI_ERROR (Status) || (NULL == PciIo->RomImage)) {
      ///
      ///  If this PCI device doesn't have a ROM image, skip to the next device,
      ///
      continue;
    }

    LocalRomImage = PciIo->RomImage;
    PcirBlockPtr = (VBIOS_PCIR_STRUCTURE *) ((UINTN) LocalRomImage + LocalRomImage->PcirOffset);
    if ((OPTION_ROM_SIGNATURE != LocalRomImage->Signature) || (NULL == PcirBlockPtr)) {
      ///
      ///  If this PCI device doesn't have a valid Option rom image or pcir block,
      ///  skip to the next device,
      ///
      continue;
    }

    ///
    /// If DG1 device is found then get the OpRegion start address, OpRegion offset and size
    ///
    if (IsDg1Device(Pci.Hdr.DeviceId)) {
      ///
      ///  Offset 0x1A from OptionRom header has 2 byte offset to config data.
      ///
      ConfigDataOffset = ((UINT8 *) LocalRomImage + CONFIG_DATA_OFFSET)[0] + (((UINT8 *) LocalRomImage + CONFIG_DATA_OFFSET)[1] << 8);
      ConfigDataStartAddress = (UINT8 *) LocalRomImage + ConfigDataOffset;
      DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe - OpRegion Address:0x%x OpRegion Offset:0x%x\n", ConfigDataStartAddress, ConfigDataOffset));

      ConfigDataSize = GetConfigDataSize(LocalRomImage, ConfigDataOffset);
      break;
    }
  }

  ///
  ///  Allocate an ACPI NVS memory buffer as the DG1 NVS OpRegion.
  ///
  Status  = (gBS->AllocatePool) (EfiACPIMemoryNVS, ConfigDataSize, (VOID **) &Dg1OpRegion.OpRegion);
  ASSERT_EFI_ERROR (Status);
  DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe allocated Nvs OpRegion 0x%x \n", (UINT32) (UINTN) (Dg1OpRegion.OpRegion)));

  SetMem (Dg1OpRegion.OpRegion, ConfigDataSize, 0);
  PlatformNvsAreaProtocol->Area->Dg1OpRegionAddress = (UINT32) (UINTN) (Dg1OpRegion.OpRegion);

  if (NULL == ConfigDataStartAddress) {
    ///
    ///  DG1 device was not found or valid OpRegon not found, no need to proceed further
    ///
    DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe - DG1 device not found \n"));
    return;
  }

  ///
  ///  DG1 device found, Copy the OpRegion from the DG1 PCI
  ///  device to the NVS area
  ///
  CopyMem(Dg1OpRegion.OpRegion, ConfigDataStartAddress, ConfigDataSize);

  ///
  /// Find the b.d.f of the DG1 device
  ///
  Status = PciIo->GetLocation (
                      PciIo,
                      &Segment,
                      &Bus,
                      &Device,
                      &Function
                      );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe - Not able to locate the b.d.f of DG1 device 0x%x \n", Status));
    return;
  }

  DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe - DG1 device b.d.f 0x%x : 0x%x : 0x%x \n", Bus, Device, Function));

  Dg1BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, Bus, Device, Function, 0);

  if (SetupVariables.Dg1PlatformSupport == 1) {
    //
    // Set PCON BIT9 & BIT10 if External Gfx adapter is in MB-down config
    //
    Dg1OpRegion.OpRegion->Header.PCON |= (BIT9 | BIT10);
  }

  // For DG1 MBD RVP, BIT 12 & BIT 11 of PCON variable in DG1 Op-region are required to be set to support VRAM Self Refresh feature
  // By default enabling VRAM SR, oem can override this
  Dg1OpRegion.OpRegion->Header.PCON |= (BIT12 | BIT11);

  DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe - PCON 0x%x Nvs OpRegion 0x%x \n", Dg1OpRegion.OpRegion->Header.PCON, PlatformNvsAreaProtocol->Area->Dg1OpRegionAddress));
  ///
  ///  Set ASLS Register to the OpRegion physical memory address..
  ///
  PciSegmentWrite32 (Dg1BaseAddress + R_SA_DG1_ASLS_OFFSET, (UINT32) (UINTN) (Dg1OpRegion.OpRegion));
  DEBUG ((DEBUG_INFO, "UpdateDg1OpRegionEndOfDxe - Value at R_SA_DG1_ASLS_OFFSET 0x%x \n", (PciSegmentRead32 (Dg1BaseAddress + R_SA_DG1_ASLS_OFFSET))));

}

/**
 DG1 Op-Region Initialization for VRAM SR support

  @param[in] ImageHandle             Handle for the image of this driver

  @retval EFI_SUCCESS             The function completed successfully
**/
EFI_STATUS
EFIAPI
DxeDg1OpregionInit (
  IN EFI_HANDLE        ImageHandle,
  IN EFI_SYSTEM_TABLE  *SystemTable
  )
{
  EFI_STATUS                Status;
  EFI_EVENT                 EndOfDxeEvent;

  EndOfDxeEvent             = NULL;
  //
  // Register an end of DXE event for setting Dg1 OpRegion
  //
  Status = gBS->CreateEventEx (
                  EVT_NOTIFY_SIGNAL,
                  TPL_CALLBACK,
                  UpdateDg1OpRegionEndOfDxe,
                  NULL,
                  &gEfiEndOfDxeEventGroupGuid,
                  &EndOfDxeEvent
                  );

  ASSERT_EFI_ERROR (Status);

  return Status;
}
