/** @file
  Non Tbt I2C Retimer Slave Device Library Functions.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
  - UEFI Specification 2.6, Section 22.3

**/

#include <Library/DebugLib.h>
#include <Protocol/RetimerCapsuleUpdate.h>
#include <Library/TimerLib.h>

#pragma pack (push,1)
//
// Tcss Private Data structure for the Non TBT I2C Retimer Device Mapping
//
typedef struct {
  GPIO_PAD  ForcePowerGpioPad;
  BOOLEAN   ForcePowerGpioLevel;
  UINT8     GpioToI2cControllerIndexBitMapping;
  UINT16    GpioToNonTbtRetimerIndexBitMapping;
  UINT8     Reserved[8];
}FORCE_GPIO_TO_I2C_RETIMER_MAPPING;
#pragma pack (pop)

static  FORCE_GPIO_TO_I2C_RETIMER_MAPPING  gForcePowerGpioToNonTbtRetimerMappingBuffer [MAX_TBT_RETIMER_DEVICE];

extern BOOLEAN  mRetimerCapsuleUpdateNotSupported;
extern BOOLEAN  mRetimerCapsuleUpdateRunning;
extern UINT32   gTotalCountOfI2cRetimerForcePowerGpio;

/**
  Get Non TBT I2c Retimer Device Data from Tcss Retimer Platform Data Pcd.

  @param[in]   RetimerPlatformDataTable  Pointer to Retimer Platform Specific Data table.
  @param[in]   RetimerDataTableSize      Size of Retimer Platform Specific Data table.
  @param[out]  TcssRetimerDeviceInfo     Pointer to buffer where output will be stored.

  @retval  The function returns total count of Non TBT I2C Retimer Force Power GPIO.
**/
UINT32
GetNonTbtI2cRetimerDeviceData (
  OUT TCSS_RETIMER_PROTOCOL_CAPABILITY  *TcssRetimerDeviceInfo,
  IN  RETIMER_PLATFORM_DATA             *RetimerPlatformDataTable,
  IN  UINT32                            RetimerDataTableSize
  )
{
  UINT8    Index;
  UINT8    BufferIndex;
  UINT32   Count;
  BOOLEAN  GpioPadMatchFound;

  DEBUG ((DEBUG_INFO, "\nGetNonTbtI2cRetimerDeviceData: Start\n"));
  if ((gTotalCountOfI2cRetimerForcePowerGpio !=0) && (gTotalCountOfI2cRetimerForcePowerGpio < MAX_TBT_RETIMER_DEVICE)) {
    DEBUG ((DEBUG_INFO, "\nAlready Initialized previously with Value of TotalCountOfI2cRetimerForcePowerGpio = %d\n", gTotalCountOfI2cRetimerForcePowerGpio));
    return gTotalCountOfI2cRetimerForcePowerGpio;
  }

  // Initialize Data Default Value
  Count                       = 0;
  GpioPadMatchFound           = FALSE;

  if ((RetimerPlatformDataTable == NULL) || (RetimerDataTableSize == 0)) {
    DEBUG ((DEBUG_ERROR, "\nRetimer Platform Data Table Not Available.\n"));
    return 0;
  }

  // Initialize gForcePowerGpioToNonTbtRetimerMappingBuffer
  DEBUG ((DEBUG_INFO, "  Initialize gForcePowerGpioToNonTbtRetimerMappingBuffer\n"));
  for (BufferIndex = 0; BufferIndex < MAX_TBT_RETIMER_DEVICE; BufferIndex++) {
    gForcePowerGpioToNonTbtRetimerMappingBuffer[BufferIndex].ForcePowerGpioPad                  = 0;
    gForcePowerGpioToNonTbtRetimerMappingBuffer[BufferIndex].ForcePowerGpioLevel                = 0;
    gForcePowerGpioToNonTbtRetimerMappingBuffer[BufferIndex].GpioToI2cControllerIndexBitMapping = 0;
    gForcePowerGpioToNonTbtRetimerMappingBuffer[BufferIndex].GpioToNonTbtRetimerIndexBitMapping = 0;
  }

  for (Index = 0; Index < RetimerDataTableSize; Index++, RetimerPlatformDataTable++) {
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->RetimerType = %X\n",   RetimerPlatformDataTable->RetimerType));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerType               = %X\n",   RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerType));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDevicePresent      = %X\n",   RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDevicePresent));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex        = %02X\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.PdControllerIndex         = %02X\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.PdControllerIndex));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.I2cControllerIndex        = %02X\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.I2cControllerIndex));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.ForcePowerGpioLevel       = %X\n",   RetimerPlatformDataTable->NonTbtI2cRetimerData.ForcePowerGpioLevel));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.ForcePowerGpioPad         = %X\n",   RetimerPlatformDataTable->NonTbtI2cRetimerData.ForcePowerGpioPad));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerBusConfig          = %04X\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerBusConfig));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerHardwareConfig     = %02X\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerHardwareConfig));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceUid          = %02X\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceUid));
    DEBUG ((DEBUG_ERROR, "  RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceSlaveAddress = %08X\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceSlaveAddress));

    if (RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDevicePresent != TRUE) {
      continue;
    }

    if (RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex >= MAX_TBT_RETIMER_DEVICE) {
      // Invalid Non Tbt I2c Retimer Device Index
      DEBUG ((DEBUG_ERROR, "\nINVALID Retimer Device Index = %d found.\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex));
      continue;
    }

    if (Count >= MAX_TBT_RETIMER_DEVICE) {
      // Array Boundry Condition Reached
      DEBUG ((DEBUG_ERROR, "\nBuffer Array Boundry Condition Reached with I2C Index = %d.\n Retimer Index =%d\n Invalid Buffer Index = %d", \
             RetimerPlatformDataTable->NonTbtI2cRetimerData.I2cControllerIndex, RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex, Count));
      break;
    }

    // Count Non Tbt I2c Retimer Device Number
    if ( (RetimerPlatformDataTable->RetimerType == NonTbtI2cRetimerType) \
          && ((TcssRetimerDeviceInfo->RetimerDeviceIndexBitMap & ( BIT0 << RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex)) == 0) \
      ) {
      // Fill Non-Tbt I2c Retimer Data only for NonTbtI2cRetimerType
      TcssRetimerDeviceInfo->RetimerDeviceIndexBitMap |= ( BIT0 << RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex);
      TcssRetimerDeviceInfo->RetimerDeviceCount++;
    }

    if (RetimerPlatformDataTable->RetimerType != NonTbtI2cRetimerType) {
      // Only Count Force Power GPIO PAD Number Data for NonTbtI2cRetimerType
      continue;
    }

    // Travel through Force Power GPIO PAD Index Buffer to check if already added
    GpioPadMatchFound = FALSE; // Initialize this variable before every loop start
    for (BufferIndex = 0; BufferIndex < Count; BufferIndex++) {
      if (gForcePowerGpioToNonTbtRetimerMappingBuffer[BufferIndex].ForcePowerGpioPad == RetimerPlatformDataTable->NonTbtI2cRetimerData.ForcePowerGpioPad) {  // Dulicate Entry Found
        GpioPadMatchFound = TRUE;
        gForcePowerGpioToNonTbtRetimerMappingBuffer[BufferIndex].GpioToNonTbtRetimerIndexBitMapping |= ( BIT0 << RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex);
        gForcePowerGpioToNonTbtRetimerMappingBuffer[BufferIndex].GpioToI2cControllerIndexBitMapping |= ( BIT0 << RetimerPlatformDataTable->NonTbtI2cRetimerData.I2cControllerIndex);
        break;
      }
    }
    if (GpioPadMatchFound == FALSE) { // Reached to end and no Force Power GPIO PAD Index Entry matched
      gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].ForcePowerGpioPad                  = RetimerPlatformDataTable->NonTbtI2cRetimerData.ForcePowerGpioPad; // Add Entry
      gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].ForcePowerGpioLevel                = RetimerPlatformDataTable->NonTbtI2cRetimerData.ForcePowerGpioLevel; // Add Entry
      DEBUG ((DEBUG_INFO, "\nNew Force Power GPIO PAD = %X found.\n", RetimerPlatformDataTable->NonTbtI2cRetimerData.ForcePowerGpioPad));
      gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].GpioToNonTbtRetimerIndexBitMapping = ( BIT0 << RetimerPlatformDataTable->NonTbtI2cRetimerData.RetimerDeviceIndex); // 1st Entry of BIT MAPPING
      gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].GpioToI2cControllerIndexBitMapping = ( BIT0 << RetimerPlatformDataTable->NonTbtI2cRetimerData.I2cControllerIndex);
      Count++;
    }
  }
  DEBUG ((DEBUG_INFO, "\nI2c Retimer Force Power GPIO Final Count = %d\n", Count));
  DEBUG ((DEBUG_INFO, "\nNon Tbt I2c Retimer Device Final Count = %d\n Retimer Device BIT Map = %X\n",\
                                TcssRetimerDeviceInfo->RetimerDeviceCount, TcssRetimerDeviceInfo->RetimerDeviceIndexBitMap));
  DEBUG ((DEBUG_INFO, "\nGetNonTbtI2cRetimerDeviceData: End.\n"));
  return Count;
}

/**
  Set I2c Retimer Slave Device Mode By Enabling Force Power GPIO.

  @param[in]  I2cRetimerDeviceMode        Mode of I2c Retimer Device to SET.
  @param[in]  I2cRetimerSlaveDeviceIndex  Number or Index of I2c Retimer Slave Device.

  @retval  EFI_SUCCESS            Command success.
  @retval  EFI_INVALID_PARAMETER  Command usage error.
**/
EFI_STATUS
SetI2cRetimerSlaveDeviceMode (
  IN  UINT8  I2cRetimerDeviceMode,
  IN  UINT8  I2cRetimerSlaveDeviceIndex
  )
{
  EFI_STATUS  Status;
  UINT32      Count;

  Status = EFI_INVALID_PARAMETER; // By Default assume Force Power GPIO Not Found. So Invalid Parameter

  if (I2cRetimerSlaveDeviceIndex != ALL_I2C_RETIMER_SLAVE_DEVICE) {
    DEBUG ((DEBUG_INFO, "\nSetI2cRetimerDeviceMode of Non Tbt I2c Retimer Device Number = %x", I2cRetimerSlaveDeviceIndex));
  } else {
    DEBUG ((DEBUG_INFO, "\nSetI2cRetimerDeviceMode of ALL Non Tbt I2c Retimer Device"));
  }
  DEBUG ((DEBUG_INFO, " to Retimer FW Update Mode = %s\n", ((I2cRetimerDeviceMode == RetimerFirmWareUpdateEnableMode) ? L"RetimerFirmWareUpdateEnableMode" : L"RetimerFirmWareUpdateDisableMode" )));

  if ((I2cRetimerSlaveDeviceIndex >= MAX_TBT_RETIMER_DEVICE) && (I2cRetimerSlaveDeviceIndex != ALL_I2C_RETIMER_SLAVE_DEVICE)) {
    DEBUG ((DEBUG_INFO, "\nSetI2cRetimerDeviceMode Returning as Invalid Non Tbt I2c Retimer Device Number\n"));
    return EFI_INVALID_PARAMETER;
  }

  if ((I2cRetimerDeviceMode < RetimerFirmWareUpdateDisableMode) || (I2cRetimerDeviceMode >= InvalidRetimerFirmWareUpdateMode)) {
    DEBUG ((DEBUG_INFO, "\nSetI2cRetimerDeviceMode Returning as Invalid Non Tbt I2c Retimer Device Mode\n"));
    return EFI_INVALID_PARAMETER;
  }
  //
  // Travers through ForcePowerGpioToNonTbtRetimerMappingBuffer to find matching GPIO
  // PAD number for I2cRetimerSlaveDeviceIndex
  //
  for (Count = 0; Count < gTotalCountOfI2cRetimerForcePowerGpio; Count++) {
    // Send EC Command to Set Non Tbt I2c Retimer Device Mode
    if ( (I2cRetimerSlaveDeviceIndex == ALL_I2C_RETIMER_SLAVE_DEVICE) ||\
        ((gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].GpioToNonTbtRetimerIndexBitMapping & (BIT0 << I2cRetimerSlaveDeviceIndex)) != 0)
      ) {
      // Force Power GPIO Match Found. Now Enable the Force Power GPIO for Non Tbt I2c Retimer Slave Device
      if (I2cRetimerDeviceMode == RetimerFirmWareUpdateEnableMode) {
        Status = GpioSetOutputValue(gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].ForcePowerGpioPad, gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].ForcePowerGpioLevel);
      } else if (I2cRetimerDeviceMode == RetimerFirmWareUpdateDisableMode) {
        Status = GpioSetOutputValue(gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].ForcePowerGpioPad,(!gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].ForcePowerGpioLevel));
      }
    }
  }

  // Wait (EC_SET_PD_MODE_WAIT_TIME_IN_FACTOR_OF_50_MS * 50)ms time before exiting
  for (Count = 0; Count < EC_SET_PD_MODE_WAIT_TIME_IN_FACTOR_OF_50_MS; Count++) {
    MicroSecondDelay( 50 * 1000); // 50ms Delay
  }

  DEBUG ((DEBUG_INFO, "\nSetI2cRetimerDeviceMode Returning Status =%r\n", Status));
  return Status;
}

/**
  Get I2c Retimer Slave Device Mode.

  @param[in]   I2cRetimerSlaveDeviceIndex  Number or Index of I2c Retimer Slave Device.
  @param[out]  I2cRetimerDeviceModeBuffer  Pointer to store retimer device mode.

  @retval  EFI_SUCCESS            Command success.
  @retval  EFI_INVALID_PARAMETER  Command usage error.
**/
EFI_STATUS
GetI2cRetimerSlaveDeviceMode (
  OUT  UINT8  *I2cRetimerDeviceModeBuffer,
  IN   UINT8  I2cRetimerSlaveDeviceIndex
  )
{
  UINT8       TempData8;
  EFI_STATUS  Status;
  UINT32      Count;
  UINT32      TempGpioLabel;

  if (I2cRetimerDeviceModeBuffer == NULL) {
    DEBUG ((DEBUG_INFO, "\nGetI2cRetimerDeviceMode Returning as Null Buffer Passed\n"));
    return EFI_INVALID_PARAMETER;
  }

  TempGpioLabel = 0XFF;
  *I2cRetimerDeviceModeBuffer = InvalidRetimerFirmWareUpdateMode;
  Status = EFI_SUCCESS;

  if (I2cRetimerSlaveDeviceIndex != ALL_I2C_RETIMER_SLAVE_DEVICE) {
    DEBUG ((DEBUG_INFO, "\nGetI2cRetimerDeviceMode of Non Tbt I2c Retimer Device Number = %x", I2cRetimerSlaveDeviceIndex));
  } else {
    DEBUG ((DEBUG_INFO, "\nGetI2cRetimerDeviceMode of ALL Non Tbt I2c Retimer Device"));
  }
  DEBUG ((DEBUG_INFO, " with MaxI2cRetimerForcePowerGpioCount =%x\n", gTotalCountOfI2cRetimerForcePowerGpio));

  if ((I2cRetimerSlaveDeviceIndex >= MAX_TBT_RETIMER_DEVICE) && (I2cRetimerSlaveDeviceIndex != ALL_I2C_RETIMER_SLAVE_DEVICE)) {
    DEBUG ((DEBUG_INFO, "\nGetI2cRetimerDeviceMode Returning as Invalid Non Tbt I2c Retimer Device Number\n"));
    return EFI_INVALID_PARAMETER;
  }

  //
  // Travers through ForcePowerGpioToNonTbtRetimerMappingBuffer to find matching GPIO
  // PAD number for I2cRetimerSlaveDeviceIndex
  //
  for (Count = 0; ((Count < MAX_TBT_RETIMER_DEVICE) && (Count < gTotalCountOfI2cRetimerForcePowerGpio)); Count++) {
    // Send EC Command to Set Non Tbt I2c Retimer Device Mode
    if ( (I2cRetimerSlaveDeviceIndex != ALL_I2C_RETIMER_SLAVE_DEVICE) &&\
        ((gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].GpioToNonTbtRetimerIndexBitMapping & (BIT0 << I2cRetimerSlaveDeviceIndex)) == 0)
      ) {
      continue; // I2c Retimer GPIO DATA not Matched
    }
    // Force Power GPIO Match Found. Now Read the Force Power GPIO for Non Tbt I2c Retimer Slave Device
    TempGpioLabel = 0xFF;
    TempData8     = InvalidRetimerFirmWareUpdateMode;
    Status = GpioGetOutputValue (gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].ForcePowerGpioPad, &TempGpioLabel);
    if (EFI_ERROR (Status) || (TempGpioLabel==0xFF)) {
      Status = EFI_DEVICE_ERROR;
      break; // Some Error Occured
    }

    // Check Force Power GPIO in Which Mode
    if (gForcePowerGpioToNonTbtRetimerMappingBuffer[Count].ForcePowerGpioLevel == (BOOLEAN)TempGpioLabel) { // GPIO In FW Update Mode
      TempData8 = RetimerFirmWareUpdateEnableMode;
    } else {
      // GPIO in Normal Mode
      TempData8 = RetimerFirmWareUpdateDisableMode;
    }
    if (*I2cRetimerDeviceModeBuffer == InvalidRetimerFirmWareUpdateMode){
      // 1st Entry
      *I2cRetimerDeviceModeBuffer = (UINT8)TempData8;
    } else {
      // Check with previous GPIO Data.
      if ((*I2cRetimerDeviceModeBuffer != (UINT8)TempData8) || (*I2cRetimerDeviceModeBuffer >= InvalidRetimerFirmWareUpdateMode)) {
        Status = EFI_DEVICE_ERROR;
        break;
      }
    }
  }

  if (I2cRetimerSlaveDeviceIndex != ALL_I2C_RETIMER_SLAVE_DEVICE) {
    DEBUG ((DEBUG_INFO, "\nGetI2cRetimerDeviceMode of Non Tbt I2c Retimer Device Number = %x", I2cRetimerSlaveDeviceIndex));
  } else {
    DEBUG ((DEBUG_INFO, "\nGetI2cRetimerDeviceMode of ALL Non Tbt I2c Retimer Device"));
  }
  DEBUG ((DEBUG_INFO, " Current Retimer FW Update Mode = %s With return Status = %r\n", ((*I2cRetimerDeviceModeBuffer == RetimerFirmWareUpdateEnableMode) ? L"RetimerFirmWareUpdateEnableMode" : L"RetimerFirmWareUpdateDisableMode" ), Status));
  return Status;
}

/**
  Change Non Tbt I2C Retimer Fw Update Mode.

  @param[in]  I2cRetimerDeviceMode        Mode of I2c Retimer Device to SET.
  @param[in]  I2cRetimerSlaveDeviceIndex  Number or Index of I2c Retimer Slave Device.

  @retval  EFI_SUCCESS            Successfully Mode is Changed to Retimer FW Update Mode.
  @retval  EFI_INVALID_PARAMETER  Invalid GUID from ESRT Table is Passed.
  @retval  EFI_UNSUPPORTED        This driver does not support.
  @retval  EFI_DEVICE_ERROR       This driver cannot be started due to device Error.
  @retval  EFI_ALREADY_STARTED    This driver has been started. This is applicable for PD FwUpdate Mode Entry.
  @retval  EFI_NOT_STARTED        This driver has not been started. This is applicable for PD FwUpdate Mode Exit.
  @retval  EFI_TIMEOUT            Mode Change Command timeout Happen.
**/
EFI_STATUS
NonTbtI2cRetimerFwUpdateModeChange (
  IN  UINT8  I2cRetimerDeviceMode,
  IN  UINT8  I2cRetimerSlaveDeviceIndex
  )
{
  EFI_STATUS  Status;
  UINT8       DeviceMode;
  CHAR16      *String = NULL;

  String = ((I2cRetimerDeviceMode == RetimerFirmWareUpdateEnableMode) ? L"Enable" : L"Disable" );

  // Check if I2c Retimer Slave Device available in board or not.
  if (gTotalCountOfI2cRetimerForcePowerGpio == 0) {
    Status = EFI_UNSUPPORTED;
    DEBUG ((DEBUG_ERROR, "\nNonTbtI2cRetimerFwUpdateMode.%S: Retimer Capsule Update is not supported as no I2c Retimer Slave Device available. Returning with Status = %r\n", String, Status));
    return Status;
  }

  // Check the I2c Retimer Slave Device Mode
  // Initialize DeviceMode with Invalid Data before getting I2c Retimer Slave Device Mode
  DeviceMode = INVALID_DEVICE_MODE;
  Status = GetI2cRetimerSlaveDeviceMode (&DeviceMode, I2cRetimerSlaveDeviceIndex);
  // Check if any ERROR status on Get Mode
  if (EFI_ERROR(Status)) {
    DEBUG ((DEBUG_ERROR, "\nNonTbtI2cRetimerFwUpdateMode.%S: I2c Retimer Slave Device GET MODE Not Responding. Returning with Status = %r\n", String, Status));
    return Status;
  }
  // Check if I2c Retimer Slave Device is in or not in FW Update mode already.
  if (DeviceMode == I2cRetimerDeviceMode) {
    if (I2cRetimerDeviceMode == RetimerFirmWareUpdateEnableMode) {
      DEBUG ((DEBUG_INFO, "\nNonTbtI2cRetimerFwUpdateMode.Drive: I2c Retimer Slave Device is already in FW Update Mode\n"));
      Status = EFI_ALREADY_STARTED;
      // Now I2c Retimer Slave Device is In FW Update Mode.
      mRetimerCapsuleUpdateRunning = TRUE;
    }
    if (I2cRetimerDeviceMode == RetimerFirmWareUpdateDisableMode) {
      DEBUG ((DEBUG_INFO, "\nNonTbtI2cRetimerFwUpdateMode.Restore: I2c Retimer Slave Device is already exited FW Update Mode.\n"));
      Status = EFI_NOT_STARTED;
      // Now I2c Retimer Slave Device already EXITED From FW Update Mode.
      mRetimerCapsuleUpdateRunning = FALSE;
    }
    return Status;
  }
  // Send command to I2c Retimer Slave Device to either enter or exit FW Update Mode
  Status = SetI2cRetimerSlaveDeviceMode (I2cRetimerDeviceMode, I2cRetimerSlaveDeviceIndex);
  if (EFI_ERROR(Status)) {
    DEBUG ((DEBUG_ERROR, "\nNonTbtI2cRetimerFwUpdateMode.%S: I2c Retimer Slave Device SET MODE Not Responding. Returning with Status = %r\n", String, Status));
    return Status;
  }
  // Initialize DeviceMode with Invalid Data before getting I2c Retimer Slave Device Mode
  DeviceMode = INVALID_DEVICE_MODE;
  Status = GetI2cRetimerSlaveDeviceMode (&DeviceMode, I2cRetimerSlaveDeviceIndex);
  // Check if any ERROR status on Get Mode
  if (EFI_ERROR(Status)) {
    DEBUG ((DEBUG_ERROR, "\nNonTbtI2cRetimerFwUpdateMode.%S: I2c Retimer Slave Device GET MODE Not Responding. Returning with Status = %r\n", String, Status));
    return Status;
  }
  // Check if any ERROR status on Set Mode
  if (DeviceMode != I2cRetimerDeviceMode) {
    Status = EFI_TIMEOUT;
    DEBUG ((DEBUG_ERROR, "\nNonTbtI2cRetimerFwUpdateMode.%S: I2c Retimer Slave Device SET MODE is not Completed. Returning with Status = %r\n", String, Status));
    return Status;
  }

  if (I2cRetimerDeviceMode == RetimerFirmWareUpdateEnableMode) {
    // Now I2c Retimer Slave Device Succesfully Enter Into FW Update Mode.
    mRetimerCapsuleUpdateRunning = TRUE;
    DEBUG ((DEBUG_INFO, "\nNonTbtI2cRetimerFwUpdateMode.Drive: "));
    DEBUG ((DEBUG_INFO, "I2C Retimer Slave Device Enter Into FW Update Mode with Status = %r\n", Status));
  }
  if (I2cRetimerDeviceMode == RetimerFirmWareUpdateDisableMode) {
    // Now I2c Retimer Slave Device Succesfully EXIT From FW Update Mode.
    mRetimerCapsuleUpdateRunning = FALSE;
    DEBUG ((DEBUG_INFO, "\nNonTbtI2cRetimerFwUpdateMode.Restore: "));
    DEBUG ((DEBUG_INFO, "I2C Retimer Slave Device Exit from FW Update Mode with Status = %r\n", Status));
  }

  return EFI_SUCCESS;
}
