/** @file
Do Platform Stage System Agent initialization.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2015 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include "PeiSaPolicyUpdate.h"
#include <Library/SiPolicyLib.h>
#include <Protocol/GraphicsOutput.h>
#include <IndustryStandard/Bmp.h>
#include <Guid/MemoryTypeInformation.h>
#include <Library/HobLib.h>
#include <CpuPcieHob.h>
#include <Platform.h>
#include <Pi/PiFirmwareFile.h>
#include <Library/BaseMemoryLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/PeiSaPolicyLib.h>
#include <TcssPeiConfig.h>
#include <TcssPeiPreMemConfig.h>
#include <VmdPeiConfig.h>
#include <CpuPcieConfig.h>
#include <Library/CpuPcieInfoFruLib.h>
#if FixedPcdGet8(PcdFspModeSelection) == 1
#include <FspsUpd.h>
#endif
#include <TelemetryPeiConfig.h>
#include <PolicyUpdateMacro.h>
#include <Library/BmpSupportLib.h>
#include <Library/PeiGetFvInfoLib.h>

//@todo Port this table for CPU PCIE
GLOBAL_REMOVE_IF_UNREFERENCED CPU_PCIE_DEVICE_OVERRIDE mCpuPcieDeviceTable[] = {
  { 0 }
};

/**
  UpdatePeiSaPolicy performs SA PEI Policy initialization

  @retval EFI_SUCCESS              The policy is installed and initialized.
**/
EFI_STATUS
EFIAPI
UpdatePeiSaPolicy (
  VOID
  )
{
  EFI_GUID                        BmpImageGuid;
  EFI_STATUS                      Status;
  EFI_STATUS                      Status2;
  SETUP_DATA                      SetupData;
  EFI_GUID                        FileGuid;
  SA_SETUP                        SaSetup;
  UINTN                           VarSize;
  VOID                            *Buffer;
  UINT32                          Size;
  UINT8                           Index;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices;
  EFI_BOOT_MODE                   BootMode;
  PCH_SETUP                       PchSetup;
  UINTN                           MaxPciePorts;
  VOID                            *VmdVariablePtr;
#if FixedPcdGet8(PcdFspModeSelection) == 1
  VOID                            *MemBuffer;
  VOID                            *FspsUpd;
  BMP_IMAGE_HEADER                *BmpHeader;
  UINT64                          BltBufferSize;
#else
  SI_POLICY_PPI                   *SiPolicyPpi;
  GRAPHICS_PEI_CONFIG             *GtConfig;
  GNA_CONFIG                      *GnaConfig;
  HOST_BRIDGE_PEI_CONFIG          *HostBridgePeiConfig;
  VMD_PEI_CONFIG                  *VmdPeiConfig;
#ifndef CPU_CFL
  TCSS_PEI_CONFIG                 *TcssPeiConfig;
  TCSS_PEI_PREMEM_CONFIG          *TcssPeiPreMemConfig;
#endif
  SI_PREMEM_POLICY_PPI            *SiPreMemPolicyPpi;
  CPU_PCIE_CONFIG                 *CpuPcieRpConfig;
    TELEMETRY_PEI_CONFIG            *TelemetryPeiConfig;

#endif
  CPU_PCIE_HOB                    *CpuPcieHob;
  EFI_PEI_PPI_DESCRIPTOR          *ReadyForGopConfigPpiDesc;
  VOID                            *VbtPtr;
  EFI_GRAPHICS_OUTPUT_BLT_PIXEL    *Blt;
  UINTN                            BltSize;
  UINTN                            Height;
  UINTN                            Width;

  DEBUG ((DEBUG_INFO, "Update PeiSaPolicyUpdate Pos-Mem Start\n"));

  Size   = 0;
  Buffer = NULL;
  Blt    = NULL;
#if FixedPcdGet8(PcdFspModeSelection) == 1
  FspsUpd = NULL;
  MemBuffer = NULL;
  BmpHeader = NULL;
  BltBufferSize = 0;
#else
  GtConfig              = NULL;
  SiPolicyPpi           = NULL;
  GnaConfig             = NULL;
  VmdPeiConfig          = NULL;
  HostBridgePeiConfig   = NULL;
#ifndef CPU_CFL
  TcssPeiConfig         = NULL;
  TcssPeiPreMemConfig   = NULL;
#endif
  SiPreMemPolicyPpi     = NULL;
  CpuPcieRpConfig       = NULL;
  TelemetryPeiConfig    = NULL;

#endif

  CpuPcieHob = NULL;
  CpuPcieHob = (CPU_PCIE_HOB *)GetFirstGuidHob(&gCpuPcieHobGuid);
  if (CpuPcieHob == NULL) {
    DEBUG((DEBUG_ERROR, "CpuPcieHob not found\n"));
    ASSERT(CpuPcieHob == NULL);
  }

#if FixedPcdGet8(PcdFspModeSelection) == 1
  FspsUpd = (FSPS_UPD *) PcdGet32 (PcdFspsUpdDataAddress);
  ASSERT (FspsUpd != NULL);
#else
  Status = PeiServicesLocatePpi (&gSiPreMemPolicyPpiGuid, 0, NULL, (VOID **) &SiPreMemPolicyPpi);
  ASSERT_EFI_ERROR (Status);

  Status = PeiServicesLocatePpi (&gSiPolicyPpiGuid, 0, NULL, (VOID **) &SiPolicyPpi);
  ASSERT_EFI_ERROR (Status);

#ifndef CPU_CFL
  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gTcssPeiPreMemConfigGuid, (VOID *) &TcssPeiPreMemConfig);
  ASSERT_EFI_ERROR(Status);
#endif

  Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gGraphicsPeiConfigGuid, (VOID *) &GtConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gGnaConfigGuid, (VOID *) &GnaConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gHostBridgePeiConfigGuid, (VOID *) &HostBridgePeiConfig);
  ASSERT_EFI_ERROR (Status);

#ifndef CPU_CFL
  Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gTcssPeiConfigGuid, (VOID *) &TcssPeiConfig);
  ASSERT_EFI_ERROR(Status);
#endif

  Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gCpuPcieRpConfigGuid, (VOID *) &CpuPcieRpConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gVmdPeiConfigGuid, (VOID *) &VmdPeiConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock((VOID *)SiPolicyPpi, &gTelemetryPeiConfigGuid, (VOID *)&TelemetryPeiConfig);
  ASSERT_EFI_ERROR(Status);

#endif

  Status = PeiServicesGetBootMode (&BootMode);
  ASSERT_EFI_ERROR (Status);

  //
  // Locate system configuration variable
  //
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid, // GUID
             0,                                // INSTANCE
             NULL,                             // EFI_PEI_PPI_DESCRIPTOR
             (VOID **) &VariableServices       // PPI
             );
  ASSERT_EFI_ERROR (Status);

  VarSize = sizeof (PCH_SETUP);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"PchSetup",
                               &gPchSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &PchSetup
                               );
  ASSERT_EFI_ERROR (Status);
#ifndef CPU_CFL
  if (!EFI_ERROR (Status)) {
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.PchUsbOverCurrentEnable, TcssPeiConfig->UsbConfig.OverCurrentEnable, PchSetup.PchUsbOverCurrentEnable);
  }
#endif

  VarSize = sizeof (SETUP_DATA);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               PLATFORM_SETUP_VARIABLE_NAME,
                               &gSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &SetupData
                               );

  VarSize = sizeof (SA_SETUP);
  Status2 = VariableServices->GetVariable(
                                VariableServices,
                                L"SaSetup",
                                &gSaSetupVariableGuid,
                                NULL,
                                &VarSize,
                                &SaSetup
                                );
  ASSERT_EFI_ERROR(Status2);

  if (IsSimicsEnvironment()) {
    SaSetup.PeiGraphicsPeimInit = 0;
  }


  if (SaSetup.VbtSelect == VBT_SELECT_MIPI) {
    CopyMem (&BmpImageGuid, PcdGetPtr(PcdVbtMipiGuid), sizeof(BmpImageGuid));
  } else if (SaSetup.VbtSelect == VBT_SELECT_EDP_HDMI) {
    CopyMem (&BmpImageGuid, PcdGetPtr (PcdVbtHdmiGuid), sizeof (BmpImageGuid));
  } else {
    CopyMem (&BmpImageGuid, PcdGetPtr(PcdIntelGraphicsVbtFileGuid), sizeof(BmpImageGuid));
  }

  if (!EFI_ERROR (Status)) {
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.PavpEnable,             GtConfig->PavpEnable,          SaSetup.PavpEnable);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CdClock,                GtConfig->CdClock,             SaSetup.CdClock);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.PeiGraphicsPeimInit,    GtConfig->PeiGraphicsPeimInit, SaSetup.PeiGraphicsPeimInit);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.GtFreqMax,              GtConfig->GtFreqMax,           SaSetup.GtFreqMax);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.DisableTurboGt,         GtConfig->DisableTurboGt,      SaSetup.DisableTurboGt);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CdynmaxClampEnable,     GtConfig->CdynmaxClampEnable,  SaSetup.CdynmaxClampEnable);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.RC1pFreqEnable,         GtConfig->RC1pFreqEnable,      SaSetup.RC1pFreqEnable);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.SkipCdClockInit,        GtConfig->SkipCdClockInit,     SaSetup.SkipCdClockInit);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.RenderStandby,          GtConfig->RenderStandby,       SaSetup.EnableRenderStandby);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.PmSupport,              GtConfig->PmSupport,           SaSetup.PmSupport);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.SkipFspGop,             GtConfig->SkipFspGop,          0x0);

    if (SaSetup.PeiGraphicsPeimInit == 1) {
      Buffer = NULL;

      CopyMem(&FileGuid, &BmpImageGuid, sizeof(FileGuid));
      PeiGetSectionFromFv(FileGuid, &Buffer, &Size);
      if (Buffer == NULL) {
        DEBUG((DEBUG_ERROR, "Could not locate VBT\n"));
      }

      if (BootMode == BOOT_ON_S3_RESUME) {
#if FixedPcdGet8(PcdFspModeSelection) == 1
        UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.GraphicsConfigPtr, GtConfig->GraphicsConfigPtr, 0);
#else
        UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.GraphicsConfigPtr, GtConfig->GraphicsConfigPtr, NULL);
#endif
      } else {
#if FixedPcdGet8(PcdFspModeSelection) == 1
        ((FSPS_UPD *) FspsUpd)->FspsConfig.GraphicsConfigPtr = (UINT32) Buffer;
        DEBUG ((DEBUG_INFO, "Vbt Pointer from PeiGetSectionFromFv is 0x%x\n", ((FSPS_UPD *) FspsUpd)->FspsConfig.GraphicsConfigPtr));
#else
        GtConfig->GraphicsConfigPtr = Buffer;
        DEBUG ((DEBUG_INFO, "Vbt Pointer from PeiGetSectionFromFv is 0x%x\n", GtConfig->GraphicsConfigPtr));
#endif
      }
      DEBUG ((DEBUG_INFO, "Vbt Size from PeiGetSectionFromFv is 0x%x\n", Size));
      GET_POLICY ((VOID *) ((FSPS_UPD *) FspsUpd)->FspsConfig.GraphicsConfigPtr, GtConfig->GraphicsConfigPtr, VbtPtr);

      Buffer = NULL;
      PeiGetSectionFromFv (gTianoLogoGuid, &Buffer, &Size);
      if (Buffer == NULL) {
        DEBUG ((DEBUG_WARN, "Could not locate Logo\n"));
      }
#if FixedPcdGet8(PcdFspModeSelection) == 1
      ((FSPS_UPD *) FspsUpd)->FspsConfig.LogoPtr  = (UINT32) Buffer;
      ((FSPS_UPD *) FspsUpd)->FspsConfig.LogoSize = Size;
      DEBUG ((DEBUG_INFO, "LogoPtr from PeiGetSectionFromFv is 0x%x\n", ((FSPS_UPD *) FspsUpd)->FspsConfig.LogoPtr));
      DEBUG ((DEBUG_INFO, "LogoSize from PeiGetSectionFromFv is 0x%x\n", ((FSPS_UPD *) FspsUpd)->FspsConfig.LogoSize));
#else
      GtConfig->LogoPtr  = Buffer;
      GtConfig->LogoSize = Size;
      DEBUG ((DEBUG_INFO, "LogoPtr from PeiGetSectionFromFv is 0x%x\n", GtConfig->LogoPtr));
      DEBUG ((DEBUG_INFO, "LogoSize from PeiGetSectionFromFv is 0x%x\n", GtConfig->LogoSize));
#endif

      //
      // Install ReadyForGopConfig PPI to trigger PEI phase GopConfig callback.
      //
      ReadyForGopConfigPpiDesc = (EFI_PEI_PPI_DESCRIPTOR *) AllocateZeroPool (sizeof (EFI_PEI_PPI_DESCRIPTOR));
      if (ReadyForGopConfigPpiDesc == NULL) {
        ASSERT (FALSE);
        return EFI_OUT_OF_RESOURCES;
      }
      ReadyForGopConfigPpiDesc->Flags = EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST;
      ReadyForGopConfigPpiDesc->Guid  = &gReadyForGopConfigPpiGuid;
      ReadyForGopConfigPpiDesc->Ppi   = VbtPtr;
      Status = PeiServicesInstallPpi (ReadyForGopConfigPpiDesc);
    }

    Status = TranslateBmpToGopBlt (
              Buffer,
              Size,
              &Blt,
              &BltSize,
              &Height,
              &Width
              );

    if (Status == EFI_BUFFER_TOO_SMALL) {
      Blt = NULL;
      Status = TranslateBmpToGopBlt (
                Buffer,
                Size,
                &Blt,
                &BltSize,
                &Height,
                &Width
                );
      if (EFI_ERROR (Status)) {
        DEBUG ((DEBUG_ERROR, "TranslateBmpToGopBlt, Status = %r\n", Status));
        ASSERT_EFI_ERROR (Status);
        return Status;
      }
    }

    //
    // Initialize Blt, BltSize, LogoPixel Height and Width
    //
#if FixedPcdGet8(PcdFspModeSelection) == 1
    ((FSPS_UPD *) FspsUpd)->FspsConfig.BltBufferAddress = (UINT32) Blt;
#else
    GtConfig->BltBufferAddress = Blt;
#endif
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.BltBufferSize,   GtConfig->BltBufferSize,  BltSize);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.LogoPixelHeight, GtConfig->LogoPixelHeight, Height);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.LogoPixelWidth,  GtConfig->LogoPixelWidth,  Width);
    //
    // Initialize GNA Configuration
    //
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.GnaEnable, GnaConfig->GnaEnable,   SaSetup.GnaEnable);
    //
    // Initialize Misc SA Configuration
    //
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.EdramTestMode, HostBridgePeiConfig->EdramTestMode, SaSetup.EdramTestMode);
    //
    // Intel(R) Dynamic Tuning Technology might need SA thermal device to be enabled.
    //
    if ((SetupData.EnableDptf == 1) && (SetupData.EnableSaDevice == 1)) {
      SaSetup.SaDevice4 = 1;
    }
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.Device4Enable, HostBridgePeiConfig->Device4Enable, SaSetup.SaDevice4);
  }

  //
  // VMD related settings from setup variable
  //
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdEnable,        VmdPeiConfig->VmdEnable,      SaSetup.VmdEnable);
  for ( Index = 0; Index < VMD_MAX_DEVICES; ++Index) {
   UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdPort[Index],      VmdPeiConfig->VmdPortEnable[Index].RpEnable, SaSetup.VmdPort[Index]);
   //Update dev and Fuc
   UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdPortDev[Index],   VmdPeiConfig->VmdPortEnable[Index].RpDevice, SaSetup.VmdPortDev[Index]);
   UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdPortFunc[Index],  VmdPeiConfig->VmdPortEnable[Index].RpFunction, SaSetup.VmdPortFunc[Index]);
  }
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdCfgBarSize,  VmdPeiConfig->VmdCfgBarSize,  SaSetup.VmdCfgBarSize);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdCfgBarAttr,  VmdPeiConfig->VmdCfgBarAttr,  SaSetup.VmdCfgBarAttr);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdMemBarSize1, VmdPeiConfig->VmdMemBarSize1, SaSetup.VmdMemBarSize1);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdMemBar1Attr, VmdPeiConfig->VmdMemBar1Attr, SaSetup.VmdMemBar1Attr);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdMemBarSize2, VmdPeiConfig->VmdMemBarSize2, SaSetup.VmdMemBarSize2);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VmdMemBar2Attr, VmdPeiConfig->VmdMemBar2Attr, SaSetup.VmdMemBar2Attr);

  // Read VmdVariable
  VarSize = 0;
  VmdVariablePtr = NULL;
  Status = VariableServices->GetVariable (
                              VariableServices,
                              EFI_VMD_OS_VARIABLE_NAME,
                              &gEfiVmdFeatureVariableGuid,
                              NULL,
                              &VarSize,
                              VmdVariablePtr
                              );
  if (Status == EFI_BUFFER_TOO_SMALL) {
    VmdVariablePtr = AllocateZeroPool (VarSize);
    DEBUG ((DEBUG_VERBOSE, "VMD VARIABLE_BUFFER_TOO_SMALL\n"));
    ASSERT (VmdVariablePtr != NULL);

    Status = VariableServices->GetVariable (
                                  VariableServices,
                                  EFI_VMD_OS_VARIABLE_NAME,
                                  &gEfiVmdFeatureVariableGuid,
                                  NULL,
                                  &VarSize,
                                  VmdVariablePtr
                                  );
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_INFO, "Vmd OS Variable not found Status is %r\n", Status));
    }
  }
#if FixedPcdGet8(PcdFspModeSelection) == 1
        ((FSPS_UPD *) FspsUpd)->FspsConfig.VmdVariablePtr = (UINT32) VmdVariablePtr;
        DEBUG ((DEBUG_INFO, "VmdVariablePtr from PeiGetSectionFromFv is 0x%x\n", ((FSPS_UPD *) FspsUpd)->FspsConfig.VmdVariablePtr));
#else
        VmdPeiConfig->VmdVariablePtr = VmdVariablePtr;
        DEBUG ((DEBUG_INFO, "VmdVariablePtr from PeiGetSectionFromFv is 0x%x\n", VmdPeiConfig->VmdVariablePtr));
#endif

  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuCrashLogEnable,  TelemetryPeiConfig->CpuCrashLogEnable, SetupData.EnableCrashLog);
  //
  // BIOS-IOM Interaction policy update
  //
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.UsbOverride,   TcssPeiConfig->IomConfig.IomInterface.UsbOverride,  SaSetup.UsbOverride);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.VccSt,         TcssPeiConfig->IomConfig.IomInterface.VccSt,        SaSetup.TcssVccstStatus);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.D3HotEnable,   TcssPeiConfig->IomConfig.IomInterface.D3HotEnable,  SaSetup.D3HotEnable);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.D3ColdEnable,  TcssPeiConfig->IomConfig.IomInterface.D3ColdEnable, SaSetup.D3ColdEnable);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.TcCstateLimit, TcssPeiConfig->IomConfig.TcStateLimit,              SaSetup.TcStateLimit);

  //
  // Itbt PCI Root Port Policy Initialization
  //
  for (Index = 0; Index < MAX_ITBT_PCIE_PORT; Index++) {
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.PtmEnabled[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].PtmEnabled, SaSetup.PtmEnabled[Index]);
    ///
    /// LTR Settings
    ///
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpLtrEnable[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].LtrEnable, SaSetup.SaPcieItbtLtrEnable[Index]);
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpSnoopLatencyOverrideMode[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].SnoopLatencyOverrideMode, SaSetup.SaPcieItbtSnoopLatencyOverrideMode[Index]);
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpSnoopLatencyOverrideMultiplier[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].SnoopLatencyOverrideMultiplier, SaSetup.SaPcieItbtSnoopLatencyOverrideMultiplier[Index]);
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpSnoopLatencyOverrideValue[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].SnoopLatencyOverrideValue, SaSetup.SaPcieItbtSnoopLatencyOverrideValue[Index]);
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpNonSnoopLatencyOverrideMode[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].NonSnoopLatencyOverrideMode, SaSetup.SaPcieItbtNonSnoopLatencyOverrideMode[Index]);
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpNonSnoopLatencyOverrideMultiplier[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].NonSnoopLatencyOverrideMultiplier, SaSetup.SaPcieItbtNonSnoopLatencyOverrideMultiplier[Index]);
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpNonSnoopLatencyOverrideValue[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].NonSnoopLatencyOverrideValue, SaSetup.SaPcieItbtNonSnoopLatencyOverrideValue[Index]);
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpForceLtrOverride[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].ForceLtrOverride, SaSetup.SaPcieItbtForceLtrOverride[Index]);
    UPDATE_POLICY(((FSPS_UPD *)FspsUpd)->FspsConfig.SaPcieItbtRpLtrConfigLock[Index], TcssPeiConfig->PciePolicy.PciePortPolicy[Index].LtrConfigLock, SaSetup.SaPcieItbtLtrConfigLock[Index]);
  }
  //
  // BIOS-PMC Interaction policy update
  //
  if ((SetupData.UsbcBiosTcssHandshake == 1) && (PcdGetBool (PcdBoardPmcPdEnable))) {
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.PmcPdEnable, TcssPeiConfig->IomConfig.PmcInterface.PmcPdEnable, 1);
  } else {
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.PmcPdEnable, TcssPeiConfig->IomConfig.PmcInterface.PmcPdEnable, 0);
  }
  //
  // PCI express config
  //
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieFiaProgramming,                            CpuPcieRpConfig->FiaProgramming,                                              SaSetup.PcieFiaProgramming);
  if (SaSetup.PcieComplianceTestMode == 1) {
    DEBUG ((DEBUG_INFO , "Compliance Test Mode is enabled!!! Disabling PCI Express PowerGating\n"));
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPciePowerGating,                             CpuPcieRpConfig->PowerGating,                                                 0);
  } else {
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPciePowerGating,                             CpuPcieRpConfig->PowerGating,                                                 SaSetup.PcieRootPortPowerGating);
  }
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieClockGating,                               CpuPcieRpConfig->ClockGating,                                                 SaSetup.PcieRootPortClockGating);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieSetSecuredRegisterLock,                    CpuPcieRpConfig->SetSecuredRegisterLock,                                      SaSetup.PcieSetSecuredRegisterLock);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieComplianceTestMode,                        CpuPcieRpConfig->PcieCommonConfig.ComplianceTestMode,                         SaSetup.PcieComplianceTestMode);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpFunctionSwap,                            CpuPcieRpConfig->PcieCommonConfig.RpFunctionSwap,                             SaSetup.RpFunctionSwap);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieDeviceOverrideTablePtr,                    CpuPcieRpConfig->PcieDeviceOverrideTablePtr,                                  (UINT32)mCpuPcieDeviceTable);
  UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieSlotSelection,                             CpuPcieRpConfig->SlotSelection,                                               SaSetup.PcieSlotSelection);
  MaxPciePorts = GetMaxCpuPciePortNum ();
  for (Index = 0; Index < MaxPciePorts; Index++) {
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpMaxPayload[Index],                     CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.MaxPayload,               CpuPcieMaxPayload256);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpPhysicalSlotNumber[Index],             CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.PhysicalSlotNumber,       (UINT8) Index);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpAspm[Index],                           CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.Aspm,                     SaSetup.PcieRootPortAspm[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpL1Substates[Index],                    CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.L1Substates,              SaSetup.PcieRootPortL1SubStates[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpGen3EqPh3Method[Index],                CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.Gen3EqPh3Method,          SaSetup.PcieRootPortEqPh3Method[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpAcsEnabled[Index],                     CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.AcsEnabled,               SaSetup.PcieRootPortACS[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpPtmEnabled[Index],                     CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.PtmEnabled ,              SaSetup.PcieRootPortPTM[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpDpcEnabled[Index],                     CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.DpcEnabled ,              SaSetup.PcieRootPortDPC[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpDpcExtensionsEnabled[Index],           CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.RpDpcExtensionsEnabled ,  SaSetup.PcieRootPortEDPC[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpVcEnabled[Index],                      CpuPcieRpConfig->RootPort[Index].VcEnabled ,                                  SaSetup.PcieRootPortVC[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpMultiVcEnabled[Index],                 CpuPcieRpConfig->RootPort[Index].MultiVcEnabled ,                             SaSetup.PcieRootPortMultiVc[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpPeerToPeerMode[Index],                 CpuPcieRpConfig->RootPort[Index].PeerToPeer,                                  SaSetup.CpuPcieRootPortPeerToPeerMode[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieFomsCp[Index],                           CpuPcieRpConfig->RootPort[Index].FomsCp,                                      SaSetup.CpuPcieFomsCp[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpSlotImplemented[Index],                CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.SlotImplemented,          SaSetup.PcieRootPortSI[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpPmSci[Index],                          CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.PmSci,                    SaSetup.PcieRootPortPMCE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpHotPlug[Index],                        CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.HotPlug,                  SaSetup.PcieRootPortHPE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpAdvancedErrorReporting[Index],         CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.AdvancedErrorReporting,         SaSetup.PcieRootPortAER[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpUnsupportedRequestReport[Index],       CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.UnsupportedRequestReport,       SaSetup.PcieRootPortURE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpFatalErrorReport[Index],               CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.FatalErrorReport,               SaSetup.PcieRootPortFEE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpNoFatalErrorReport[Index],             CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.NoFatalErrorReport,             SaSetup.PcieRootPortNFE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpCorrectableErrorReport[Index],         CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.CorrectableErrorReport,         SaSetup.PcieRootPortCEE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpSystemErrorOnFatalError[Index],        CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.SystemErrorOnFatalError,        SaSetup.PcieRootPortSFE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpSystemErrorOnNonFatalError[Index],     CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.SystemErrorOnNonFatalError,     SaSetup.PcieRootPortSNE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpSystemErrorOnCorrectableError[Index],  CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.SystemErrorOnCorrectableError,  SaSetup.PcieRootPortSCE[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpTransmitterHalfSwing[Index],           CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.TransmitterHalfSwing,           SaSetup.PcieRootPortTHS[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpGen3Uptp[Index],                   CpuPcieRpConfig->RootPort[Index].Gen3Uptp,                                    SaSetup.PcieRootPortGen3Uptp[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpGen3Dptp[Index],                   CpuPcieRpConfig->RootPort[Index].Gen3Dptp,                                    SaSetup.PcieRootPortGen3Dptp[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpGen4Uptp[Index],                   CpuPcieRpConfig->RootPort[Index].Gen4Uptp,                                    SaSetup.PcieRootPortGen4Uptp[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpGen4Dptp[Index],                   CpuPcieRpConfig->RootPort[Index].Gen4Dptp,                                    SaSetup.PcieRootPortGen4Dptp[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpDetectTimeoutMs[Index],                CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.DetectTimeoutMs,          SaSetup.PcieDetectTimeoutMs[Index]);
  }

  for (Index = 0; Index < SA_PEG_MAX_LANE; ++Index) {
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieEqPh3LaneParamCm[Index],                 CpuPcieRpConfig->EqPh3LaneParam[Index].Cm, SaSetup.PcieLaneCm[Index]);
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieEqPh3LaneParamCp[Index],                 CpuPcieRpConfig->EqPh3LaneParam[Index].Cp, SaSetup.PcieLaneCp[Index]);
  }

  for (Index = 0; Index < GetMaxCpuPciePortNum (); Index++) {
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpLtrConfigLock[Index],                    CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.PcieRpLtrConfig.LtrConfigLock, SaSetup.CpuPcieLtrConfigLock[Index]);
#if FixedPcdGet8(PcdFspModeSelection) == 1
      ((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpLtrMaxSnoopLatency[Index]   = 0x1003;
      ((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpLtrMaxNoSnoopLatency[Index] = 0x1003;
#endif
    UPDATE_POLICY (((FSPS_UPD *) FspsUpd)->FspsConfig.CpuPcieRpLtrEnable[Index],                      CpuPcieRpConfig->RootPort[Index].PcieRpCommonConfig.LtrEnable, SaSetup.CpuPcieLtrEnable[Index]);
  }
  return EFI_SUCCESS;
}
