/** @file
  Pci Hotplug Driver : This file will perform specific PCI-EXPRESS
  Devics resource configuration.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2006 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

//
// Statements that include other files
//
#include "PciHotPlug.h"
#include "PciHotPlugHelpers.h"

#include <Protocol/ITbtPolicy.h>

#include <Library/PchPcieRpLib.h>
#include <Library/TbtCommonLib.h>
#include <Register/SaRegsHostBridge.h>
#include <PchPcieRpInfo.h>
#include <Library/ItbtPcieRpLib.h>
#include <Library/TcssInfoLib.h>
#include <PchBdfAssignment.h>

#define ITBT_CONTROLLER                   0x80
#define INVALID_RP_CONTROLLER_TYPE        0xFF

#define PCIE_NUM  (24)

#define PEG_NUM   (4)

#define PADDING_BUS (1)
#define PADDING_NONPREFETCH_MEM (1)
#define PADDING_PREFETCH_MEM (1)
#define PADDING_IO (1)
#define PADDING_NUM (PADDING_BUS + PADDING_NONPREFETCH_MEM + PADDING_PREFETCH_MEM + PADDING_IO)

GLOBAL_REMOVE_IF_UNREFERENCED EFI_HPC_LOCATION          mPcieLocation[PCIE_NUM + PEG_NUM + MAX_ITBT_PCIE_PORT];
GLOBAL_REMOVE_IF_UNREFERENCED P2PB_RESOURCE_RECORD      mP2pbResourceRecord[MAX_P2PB_OF_TBT_DAISY_CHAIN];
GLOBAL_REMOVE_IF_UNREFERENCED UINT8                     mP2pbResourceRecordCount = 0;
// Record the number of HotPlug capability Bridge on the same Bus.
GLOBAL_REMOVE_IF_UNREFERENCED NUMBER_OF_HOTPLUG_BRIDGE  mNumOfHotPlugBridge[MAX_P2PB_OF_TBT_DAISY_CHAIN];

GLOBAL_REMOVE_IF_UNREFERENCED UINTN   mHpcCount = 0;
GLOBAL_REMOVE_IF_UNREFERENCED UINT32  mRemainderBus = 0;
GLOBAL_REMOVE_IF_UNREFERENCED UINTN   gItbtSegment = 0;

GLOBAL_REMOVE_IF_UNREFERENCED PCIE_HOT_PLUG_DEVICE_PATH mHotplugPcieDevicePathTemplate = {
  ACPI,
  PCI(0xFF, 0xFF), // Dummy Device no & Function no
  END
};

/**
  Entry point for the driver.

  This routine reads the PlatformType GPI on FWH and produces a protocol
  to be consumed by the chipset driver to effect those settings.

  @param[in]  ImageHandle    An image handle.
  @param[in]  SystemTable    A pointer to the system table.

  @retval     EFI_SUCCESS.
**/
EFI_STATUS
EFIAPI
PciHotPlug (
  IN EFI_HANDLE                   ImageHandle,
  IN EFI_SYSTEM_TABLE             *SystemTable
  )
{
  EFI_STATUS                       Status;
  PCI_HOT_PLUG_INSTANCE            *PciHotPlug;
  SA_POLICY_PROTOCOL               *SaPolicy;
  UINTN                            Index;
  UINTN                            RpDev;
  UINTN                            RpFunc;
  PCIE_HOT_PLUG_DEVICE_PATH        *HotplugPcieDevicePath;
  UINTN                            VariableSize;
  PCH_SETUP                        PchSetup;
  PCIE_DXE_CONFIG                  *PcieDxeConfig;
  ITBT_INFO_HOB                    *ITbtInfoHob;
  UINTN                            RpBus;
  UINTN                            RpSegment;
  UINT8                            PegDeviceNumber [SA_PEG_MAX_FUN] = {1, 1, 1, 6};
  UINT8                            PegFunctionNumber [SA_PEG_MAX_FUN] = {0, 1, 2, 0};

  RpSegment = 0;
  RpDev = 0;
  RpFunc = 0;

  //
  // Get TBT INFO HOB
  //
  ITbtInfoHob = NULL;
  ITbtInfoHob = (ITBT_INFO_HOB *) GetFirstGuidHob (&gITbtInfoHobGuid);
  if (ITbtInfoHob == NULL) {
    DEBUG ((DEBUG_INFO, "ITbtInfoHob not found\n"));
  }
  ZeroMem (mP2pbResourceRecord, sizeof (mP2pbResourceRecord));
  ZeroMem (mNumOfHotPlugBridge, sizeof (mNumOfHotPlugBridge));

  VariableSize = sizeof (PCH_SETUP);
  Status = gRT->GetVariable (
                  L"PchSetup",
                  &gPchSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &PchSetup
                  );
  ASSERT_EFI_ERROR (Status);

  //
  // PCH Rootports Hotplug device path creation
  //
  for (Index = 0; Index < PCIE_NUM; Index++) {
    if (PchSetup.PcieRootPortHPE[Index]) { // Check the Rootport no's hotplug is set
      Status = GetPchPcieRpDevFun (Index, &RpDev, &RpFunc); // Get the actual device/function no corresponding to the Rootport no provided
      ASSERT_EFI_ERROR (Status);

      HotplugPcieDevicePath = NULL;
      HotplugPcieDevicePath = AllocatePool (sizeof (PCIE_HOT_PLUG_DEVICE_PATH));
      ASSERT (HotplugPcieDevicePath != NULL);
      if (HotplugPcieDevicePath == NULL) {
        return EFI_OUT_OF_RESOURCES;
      }
      CopyMem (HotplugPcieDevicePath, &mHotplugPcieDevicePathTemplate, sizeof (PCIE_HOT_PLUG_DEVICE_PATH));
      HotplugPcieDevicePath->PciRootPortNode.Device = (UINT8) RpDev; // Update real Device no
      HotplugPcieDevicePath->PciRootPortNode.Function = (UINT8) RpFunc; // Update real Function no

      mPcieLocation[mHpcCount].HpcDevicePath = (EFI_DEVICE_PATH_PROTOCOL *)HotplugPcieDevicePath;
      mPcieLocation[mHpcCount].HpbDevicePath = (EFI_DEVICE_PATH_PROTOCOL *)HotplugPcieDevicePath;
      mHpcCount++;

      DEBUG ((DEBUG_INFO, "(%02d) PciHotPlug (PCH RP#) : Bus 0x00, Device 0x%x, Function 0x%x is added to the Hotplug Device Path list \n", mHpcCount, RpDev, RpFunc));
    }
  }

  Status = gBS->LocateProtocol (
                  &gSaPolicyProtocolGuid,
                  NULL,
                  (VOID**) &SaPolicy
                  );
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *)SaPolicy, &gPcieDxeConfigGuid, (VOID *)&PcieDxeConfig);
  ASSERT_EFI_ERROR (Status);

  //
  // PEG Rootports Hotplug device path creation
  //
  for (Index = 0; Index < (SA_PEG_MAX_FUN); Index++) {
    if (PcieDxeConfig->PegRootPortHPE[Index]) { // Check the Rootport no's hotplug is set
      RpDev  = PegDeviceNumber[Index];
      RpFunc = PegFunctionNumber[Index];

      HotplugPcieDevicePath = NULL;
      HotplugPcieDevicePath = AllocatePool (sizeof (PCIE_HOT_PLUG_DEVICE_PATH));
      ASSERT (HotplugPcieDevicePath != NULL);
      if (HotplugPcieDevicePath == NULL) {
        return EFI_OUT_OF_RESOURCES;
      }
      CopyMem (HotplugPcieDevicePath, &mHotplugPcieDevicePathTemplate, sizeof (PCIE_HOT_PLUG_DEVICE_PATH));
      HotplugPcieDevicePath->PciRootPortNode.Device = (UINT8) RpDev; // Update real Device no
      HotplugPcieDevicePath->PciRootPortNode.Function = (UINT8) RpFunc; // Update real Function no

      mPcieLocation[mHpcCount].HpcDevicePath = (EFI_DEVICE_PATH_PROTOCOL *)HotplugPcieDevicePath;
      mPcieLocation[mHpcCount].HpbDevicePath = (EFI_DEVICE_PATH_PROTOCOL *)HotplugPcieDevicePath;
      mHpcCount++;

      DEBUG ((DEBUG_INFO, "(%02d) PciHotPlug (PEG RP#) : Bus 0x00, Device 0x%x, Function 0x%x is added to the Hotplug Device Path list \n", mHpcCount, RpDev, RpFunc));
    }
  }
  for (Index = 0; Index < MAX_ITBT_PCIE_PORT; Index++) {
    if((ITbtInfoHob != NULL) && (ITbtInfoHob->ITbtRootPortConfig[Index].ITbtPcieRootPortEn)) {
      Status = GetItbtPcieRpInfo((UINTN) Index, &RpSegment, &RpBus, &RpDev, &RpFunc);
      ASSERT_EFI_ERROR (Status);
      gItbtSegment = RpSegment;
      HotplugPcieDevicePath = NULL;
      HotplugPcieDevicePath = AllocatePool (sizeof (PCIE_HOT_PLUG_DEVICE_PATH));
      ASSERT (HotplugPcieDevicePath != NULL);
      if (HotplugPcieDevicePath == NULL) {
        return EFI_OUT_OF_RESOURCES;
      }
      CopyMem (HotplugPcieDevicePath, &mHotplugPcieDevicePathTemplate, sizeof (PCIE_HOT_PLUG_DEVICE_PATH));
      HotplugPcieDevicePath->PciRootPortNode.Device = (UINT8) RpDev; // Update real Device no
      HotplugPcieDevicePath->PciRootPortNode.Function = (UINT8) RpFunc; // Update real Function no

      mPcieLocation[mHpcCount].HpcDevicePath = (EFI_DEVICE_PATH_PROTOCOL *)HotplugPcieDevicePath;
      mPcieLocation[mHpcCount].HpbDevicePath = (EFI_DEVICE_PATH_PROTOCOL *)HotplugPcieDevicePath;
      mHpcCount++;

      DEBUG ((DEBUG_INFO, "(%02d) CPU PCIe HotPlug (ITBT RP#) : Bus 0x00, Device 0x%x, Function 0x%x is added to the Hotplug Device Path list \n", mHpcCount, RpDev, RpFunc));
    }
  }
  PciHotPlug = AllocatePool (sizeof (PCI_HOT_PLUG_INSTANCE));
  ASSERT (PciHotPlug != NULL);
  if (PciHotPlug == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  //
  // Initialize driver private data.
  //
  ZeroMem (PciHotPlug, sizeof (PCI_HOT_PLUG_INSTANCE));

  PciHotPlug->Signature                               = PCI_HOT_PLUG_DRIVER_PRIVATE_SIGNATURE;
  PciHotPlug->HotPlugInitProtocol.GetRootHpcList      = GetRootHpcList;
  PciHotPlug->HotPlugInitProtocol.InitializeRootHpc   = InitializeRootHpc;
  PciHotPlug->HotPlugInitProtocol.GetResourcePadding  = GetResourcePadding;

  Status = gBS->InstallProtocolInterface (
                  &PciHotPlug->Handle,
                  &gEfiPciHotPlugInitProtocolGuid,
                  EFI_NATIVE_INTERFACE,
                  &PciHotPlug->HotPlugInitProtocol
                  );
  ASSERT_EFI_ERROR (Status);

  return EFI_SUCCESS;
}

/**
  This procedure returns a list of Root Hot Plug controllers that require
  initialization during boot process

  @param[in]  This      The pointer to the instance of the EFI_PCI_HOT_PLUG_INIT protocol.
  @param[out] HpcCount  The number of Root HPCs returned.
  @param[out] HpcList   The list of Root HPCs. HpcCount defines the number of elements in this list.

  @retval EFI_SUCCESS.
**/
EFI_STATUS
EFIAPI
GetRootHpcList (
  IN EFI_PCI_HOT_PLUG_INIT_PROTOCOL    *This,
  OUT UINTN                            *HpcCount,
  OUT EFI_HPC_LOCATION                 **HpcList
  )
{
  *HpcCount = mHpcCount;
  *HpcList  = mPcieLocation;

  return EFI_SUCCESS;
}

/**
  This procedure Initializes one Root Hot Plug Controller
  This process may casue initialization of its subordinate buses

  @param[in]  This            The pointer to the instance of the EFI_PCI_HOT_PLUG_INIT protocol.
  @param[in]  HpcDevicePath   The Device Path to the HPC that is being initialized.
  @param[in]  HpcPciAddress   The address of the Hot Plug Controller function on the PCI bus.
  @param[in]  Event           The event that should be signaled when the Hot Plug Controller initialization is complete. Set to NULL if the caller wants to wait until the entire initialization process is complete. The event must be of the type EFI_EVT_SIGNAL.
  @param[out] HpcState        The state of the Hot Plug Controller hardware. The type EFI_Hpc_STATE is defined in section 3.1.

  @retval   EFI_SUCCESS.
**/
EFI_STATUS
EFIAPI
InitializeRootHpc (
  IN  EFI_PCI_HOT_PLUG_INIT_PROTOCOL      *This,
  IN  EFI_DEVICE_PATH_PROTOCOL            *HpcDevicePath,
  IN  UINT64                              HpcPciAddress,
  IN  EFI_EVENT                           Event, OPTIONAL
  OUT EFI_HPC_STATE                       *HpcState
  )
{
  if (Event) {
    gBS->SignalEvent (Event);
  }

  *HpcState = EFI_HPC_STATE_INITIALIZED;

  return EFI_SUCCESS;
}

/**
  Selects the proper TBT Root port to assign resources
  based on the user input value

  @param[in]  SetupData          Pointer to Setup data

  @retval     TbtSelectorChosen  Rootport number.
**/
UINT8
GetRootporttoSetResourcesforTbt (
  IN SETUP_DATA *SetupData,
  IN UINTN    RpIndex,
  IN UINT8    TbtType
  )
{
  UINT8                    TbtSelectorChosen;
  UINT8                    Index;
  UINT8                    Rootportselected;

  TbtSelectorChosen = 0xFF;
  Rootportselected = 0;

  //
  // Get TBT INFO HOB
  //
  Rootportselected = 0;
  if (TbtType == ITBT_CONTROLLER) {
    for (Index = 0; Index < MAX_ITBT_PCIE_PORT; Index++) {
      //
      // Check if Controller is set to AUTO and Enabled by default.
      //
      if (SetupData->ITbtRootPort[Index] == 0x01) {
        //
        // Get the Exact Root port number corresponding to the Controller Selected
        //
        Rootportselected = Index;
      }
      if (Rootportselected == (UINT8) RpIndex) {
        TbtSelectorChosen = Index;
        break;
      }
    } // for loop
  }

  DEBUG ((DEBUG_INFO, "TbtSelectorChosen = %x\n", TbtSelectorChosen));

  return (TbtSelectorChosen);
}

/**
  Update P2P Bridge Resource Record

  @param[in] *HpcDevicePath       The Device Path to the HPC that is being initialized.
  @param[in] HpcPciAddress        The address of the Hot Plug Controller function on the PCI bus.
  @param[in] RsvdExtraBusNum      Reserved Bus resource Number.
  @param[in] RsvdPcieMegaMem      Reserved Alignment of MEM resource.
  @param[in] PcieMemAddrRngMax    Alignment of MEM resource.
  @param[in] RsvdPciePMegaMem     Reserved Pcie Kilo Io Number.
  @param[in] PciePMemAddrRngMax   Alignment of PMEM resource.
  @param[in] RsvdPcieKiloIo       Reserved Pcie Kilo Io Number.

  @retval    EFI_SUCCESS.
**/
EFI_STATUS
UpdateP2pbResourceRecord (
  IN     EFI_DEVICE_PATH_PROTOCOL        *HpcDevicePath,
  IN     UINT64                          HpcPciAddress,
  IN     UINT8                           RsvdExtraBusNum,
  IN     UINT16                          RsvdPcieMegaMem,
  IN     UINT8                           PcieMemAddrRngMax,
  IN     UINT16                          RsvdPciePMegaMem,
  IN     UINT8                           PciePMemAddrRngMax,
  IN     UINT8                           RsvdPcieKiloIo
  )
{
  UINT16 Index;

  for (Index = 0; Index < MAX_P2PB_OF_TBT_DAISY_CHAIN; Index++) {
    if (mP2pbResourceRecord[Index].HpcPciAddress == 0) {
      continue;
    }
    if (mP2pbResourceRecord[Index].HpcPciAddress == HpcPciAddress) {
      //
      // Record already existed, so ignore it.
      //
      return EFI_ABORTED;
    }
  }
  //
  // Record the current HPC's resource assignment
  //
  mP2pbResourceRecord[mP2pbResourceRecordCount].HpcDevicePath = HpcDevicePath;
  mP2pbResourceRecord[mP2pbResourceRecordCount].HpcPciAddress = HpcPciAddress;
  mP2pbResourceRecord[mP2pbResourceRecordCount].RsvdExtraBusNum = RsvdExtraBusNum;
  mP2pbResourceRecord[mP2pbResourceRecordCount].RsvdPcieMegaMem = RsvdPcieMegaMem;
  mP2pbResourceRecord[mP2pbResourceRecordCount].PcieMemAddrRngMax = PcieMemAddrRngMax;
  mP2pbResourceRecord[mP2pbResourceRecordCount].RsvdPciePMegaMem = RsvdPciePMegaMem;
  mP2pbResourceRecord[mP2pbResourceRecordCount].PciePMemAddrRngMax = PciePMemAddrRngMax;
  mP2pbResourceRecord[mP2pbResourceRecordCount].RsvdPcieKiloIo = RsvdPcieKiloIo;
  DEBUG ((DEBUG_INFO, "mP2pbResourceRecord[%x].HpcDevicePath = %s \n", mP2pbResourceRecordCount, ConvertDevicePathToText (mP2pbResourceRecord[mP2pbResourceRecordCount].HpcDevicePath, TRUE, TRUE)));
  DEBUG ((DEBUG_INFO, "mP2pbResourceRecord[%x].HpcPciAddress = %lx \n", mP2pbResourceRecordCount, mP2pbResourceRecord[mP2pbResourceRecordCount].HpcPciAddress));
  DEBUG ((DEBUG_INFO, "mP2pbResourceRecord[%x].RsvdExtraBusNum = %x \n", mP2pbResourceRecordCount, mP2pbResourceRecord[mP2pbResourceRecordCount].RsvdExtraBusNum));
  DEBUG ((DEBUG_INFO, "mP2pbResourceRecord[%x].RsvdPcieMegaMem = %x \n", mP2pbResourceRecordCount, mP2pbResourceRecord[mP2pbResourceRecordCount].RsvdPcieMegaMem));
  DEBUG ((DEBUG_INFO, "mP2pbResourceRecord[%x].PcieMemAddrRngMax = %x \n", mP2pbResourceRecordCount, mP2pbResourceRecord[mP2pbResourceRecordCount].PcieMemAddrRngMax));
  DEBUG ((DEBUG_INFO, "mP2pbResourceRecord[%x].RsvdPciePMegaMem = %x \n", mP2pbResourceRecordCount, mP2pbResourceRecord[mP2pbResourceRecordCount].RsvdPciePMegaMem));
  DEBUG ((DEBUG_INFO, "mP2pbResourceRecord[%x].PciePMemAddrRngMax = %x \n", mP2pbResourceRecordCount, mP2pbResourceRecord[mP2pbResourceRecordCount].PciePMemAddrRngMax));
  DEBUG ((DEBUG_INFO, "mP2pbResourceRecord[%x].RsvdPcieKiloIo = %x \n", mP2pbResourceRecordCount, mP2pbResourceRecord[mP2pbResourceRecordCount].RsvdPcieKiloIo));

  return EFI_SUCCESS;
}

/**
  Handle the Resource Padding For RootPort

  @param[in] *HpcDevicePath       The Device Path to the HPC that is being initialized.
  @param[in] HpcPciAddress        The address of the Hot Plug Controller function on the PCI bus.
  @param[in] RsvdExtraBusNum      Reserved Bus resource Number.
  @param[in] RsvdPcieMegaMem      Reserved Alignment of MEM resource.
  @param[in] PcieMemAddrRngMax    Alignment of MEM resource.
  @param[in] RsvdPciePMegaMem     Reserved Pcie Kilo Io Number.
  @param[in] PciePMemAddrRngMax   Alignment of PMEM resource.
  @param[in] RsvdPcieKiloIo       Reserved Pcie Kilo Io Number.

**/
VOID
HandleResourcePaddingForRootPort (
  IN     EFI_DEVICE_PATH_PROTOCOL        *HpcDevicePath,
  IN     UINT64                          HpcPciAddress,
  OUT    UINT8                           *RsvdExtraBusNum,
  OUT    UINT16                          *RsvdPcieMegaMem,
  OUT    UINT8                           *PcieMemAddrRngMax,
  OUT    UINT16                          *RsvdPciePMegaMem,
  OUT    UINT8                           *PciePMemAddrRngMax,
  OUT    UINT8                           *RsvdPcieKiloIo
  )
{
  EFI_STATUS                        Status;
  EFI_STATUS                        Status2;
  EFI_STATUS                        Status3;
  SA_SETUP                          SaSetup;
  PCH_SETUP                         PchSetup;
  SETUP_DATA                        SetupData;
  UINTN                             VariableSize;
  UINTN                             RpIndex = 0;
  UINTN                             RpBus;
  UINTN                             RpDev;
  UINTN                             RpFunc;
  UINT8                             TbtController;
  UINT8                             TbtType = INVALID_RP_CONTROLLER_TYPE;
  ITBT_POLICY_PROTOCOL              *ITbtPolicy;
  DXE_ITBT_CONFIG                   *DxeITbtConfig;

  Status = gBS->LocateProtocol (&gITbtPolicyProtocolGuid, NULL, (VOID **) &ITbtPolicy);
  Status = GetConfigBlock ((VOID *) ITbtPolicy, &gDxeITbtConfigGuid, (VOID *) &DxeITbtConfig);

  //
  // Check if PCIe Root HPC Controller need to reserve bus for docking downstream P2P hotplug
  //
  VariableSize = sizeof(SETUP_DATA);
  Status = gRT->GetVariable(
                  L"Setup",
                  &gSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &SetupData
                  );

  VariableSize = sizeof(SA_SETUP);
  Status2 = gRT->GetVariable(
                  L"SaSetup",
                  &gSaSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &SaSetup
                  );

  VariableSize = sizeof (PCH_SETUP);
  Status3 = gRT->GetVariable (
                  L"PchSetup",
                  &gPchSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &PchSetup
                  );

  RpBus  = (UINTN) ((HpcPciAddress >> 24) & 0xFF);
  RpDev  = (UINTN) ((HpcPciAddress >> 16) & 0xFF);
  RpFunc = (UINTN) ((HpcPciAddress >> 8) & 0xFF);

  // Get the actual Rootport no corresponding to the device/function no provided
  if (RpDev == GetITbtPcieDevNumber ()) {
    GetItbtPciePortIndex (gItbtSegment, RpBus, RpDev, RpFunc, &RpIndex);
    TbtType = ITBT_CONTROLLER;
    DEBUG ((DEBUG_INFO, "GetResourcePadding : ITBT Rootport no %02d Bus 0x%x, Device 0x%x, Function 0x%x \n", RpIndex, RpBus, RpDev, RpFunc));
  } else {
    // Non-Rootport
    DEBUG ((DEBUG_INFO, "GetResourcePadding : Non-Rootport no %02d Bus 0x%x, Device 0x%x, Function 0x%x \n", RpIndex, RpBus, RpDev, RpFunc));
  }

  if (TbtType == INVALID_RP_CONTROLLER_TYPE) {
    // Do not assign any IO Resource for this kind of Invalid RP.
    RsvdPcieKiloIo  = 0;
  }

  if (SetupData.IntegratedTbtSupport == 1) {
    if (!EFI_ERROR(Status)) {
      TbtController = GetRootporttoSetResourcesforTbt (&SetupData, RpIndex, TbtType);
      if ((SetupData.IntegratedTbtSupport == 1) && (TbtType == ITBT_CONTROLLER)) {
        if (TbtController < MAX_ITBT_PCIE_PORT) {
          if (DxeITbtConfig == NULL) {
            DEBUG ((DEBUG_ERROR, "DxeITbtConfig is NULL \n"));
            return;
          }
          *RsvdExtraBusNum    = DxeITbtConfig->ITbtResourceConfig[TbtController].TbtPcieExtraBusRsvd;
          *RsvdPcieMegaMem    = DxeITbtConfig->ITbtResourceConfig[TbtController].TbtPcieMemRsvd;
          *PcieMemAddrRngMax  = DxeITbtConfig->ITbtResourceConfig[TbtController].TbtPcieMemAddrRngMax;
          *RsvdPciePMegaMem   = DxeITbtConfig->ITbtResourceConfig[TbtController].TbtPciePMemRsvd;
          *PciePMemAddrRngMax = DxeITbtConfig->ITbtResourceConfig[TbtController].TbtPciePMemAddrRngMax;
          *RsvdPcieKiloIo     = 0;
        }
      }
    }
  } else {
    if ((RpBus == SA_PEG_BUS_NUM) && (RpDev == SA_PEG_DEV_NUM)) {
      if (!EFI_ERROR(Status2)) {
        // PEG
        *RsvdExtraBusNum = SaSetup.PegExtraBusRsvd[RpIndex];
        *RsvdPcieMegaMem = SaSetup.PegMemRsvd[RpIndex];
        *RsvdPcieKiloIo  = SaSetup.PegIoRsvd[RpIndex];
      }
    } else if((RpBus == 0) && ((RpDev == PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_1) || (RpDev == PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_2) || (RpDev == PCI_DEVICE_NUMBER_PCH_PCIE_ROOT_PORT_3))){
      if (!EFI_ERROR(Status3)) {
        // PCH
        *RsvdExtraBusNum = PchSetup.PcieExtraBusRsvd[RpIndex];
        *RsvdPcieMegaMem = PchSetup.PcieMemRsvd[RpIndex];
        *RsvdPcieKiloIo  = PchSetup.PcieIoRsvd[RpIndex];
      }
    }
  }
}

/**
  Handle the Resource Padding For Non-RootPort

  @param[in] *HpcDevicePath       The Device Path to the HPC that is being initialized.
  @param[in] HpcPciAddress        The address of the Hot Plug Controller function on the PCI bus.
  @param[in] RsvdExtraBusNum      Reserved Bus resource Number.
  @param[in] RsvdPcieMegaMem      Reserved Alignment of MEM resource.
  @param[in] PcieMemAddrRngMax    Alignment of MEM resource.
  @param[in] RsvdPciePMegaMem     Reserved Pcie Kilo Io Number.
  @param[in] PciePMemAddrRngMax   Alignment of PMEM resource.
  @param[in] RsvdPcieKiloIo       Reserved Pcie Kilo Io Number.

**/
VOID
HandleResourcePaddingForNonRootPort (
  IN  EFI_DEVICE_PATH_PROTOCOL        *HpcDevicePath,
  IN  UINT64                          HpcPciAddress,
  OUT UINT8                           *RsvdExtraBusNum,
  OUT UINT16                          *RsvdPcieMegaMem,
  OUT UINT8                           *PcieMemAddrRngMax,
  OUT UINT16                          *RsvdPciePMegaMem,
  OUT UINT8                           *PciePMemAddrRngMax,
  OUT UINT8                           *RsvdPcieKiloIo
  )
{
  UINT8       RpSeg;
  UINTN       RpBus;
  UINTN       RpDev;
  UINTN       RpFunc;
  UINT64      ParentPciAddress;
  EFI_STATUS  Status;
  UINT8       Index;
  UINT8       ConsumedBusNumber;

  RpSeg  = (UINT8) ((HpcPciAddress >> 32) & 0x01);
  RpBus  = (UINTN) ((HpcPciAddress >> 24) & 0xFF);
  RpDev  = (UINTN) ((HpcPciAddress >> 16) & 0xFF);
  RpFunc = (UINTN) ((HpcPciAddress >> 8) & 0xFF);

  // Non-Rootport
  DEBUG ((DEBUG_INFO, "GetResourcePadding : Non-Rootport Seg 0x%x, Bus 0x%x, Device 0x%x, Function 0x%x \n", RpSeg, RpBus, RpDev, RpFunc));

  *RsvdPcieKiloIo  = 0;
  ConsumedBusNumber = 0;

  ParentPciAddress = GetParentPciAddress (RpSeg, RpBus);

  GetParentResourceSize (ParentPciAddress, RsvdExtraBusNum, RsvdPcieMegaMem, RsvdPciePMegaMem, RsvdPcieKiloIo);
  DEBUG ((DEBUG_INFO, "Parent's resource as below \n"));
  DEBUG ((DEBUG_INFO, "BusNum = %x \n", *RsvdExtraBusNum));
  DEBUG ((DEBUG_INFO, "PcieMegaMem = %x \n", *RsvdPcieMegaMem));
  DEBUG ((DEBUG_INFO, "PciePMegaMem = %x \n", *RsvdPciePMegaMem));
  DEBUG ((DEBUG_INFO, "PcieKiloIo = %x \n", *RsvdPcieKiloIo));

  CalculateBusConsumptionBeforeAssign (ParentPciAddress, HpcPciAddress, &ConsumedBusNumber);

  Index = 0;
  Status = GetCurrentBusP2pbHotPlug(RpSeg, (UINT8)RpBus, &Index);
  if (Status == EFI_SUCCESS) {
    // RsvdExtraBusNum = (Reserved Bus - UpStream Bus - DownStream Bus - Tatal Non-HotPlug Bus) / Total HotPlug P2pb
    if (mNumOfHotPlugBridge[Index].NumOfP2pbHotPlug != 0) {
      *RsvdExtraBusNum = (UINT8) DivU64x32Remainder ((UINT64)(*RsvdExtraBusNum - ConsumedBusNumber - mNumOfHotPlugBridge[Index].NumOfP2pbNonHotPlug), (UINT32)mNumOfHotPlugBridge[Index].NumOfP2pbHotPlug, &mRemainderBus);
    }

    // If mRemainderBus not equal zero, check this device is the last one or not.
    if (mRemainderBus != 0) {
      DEBUG((DEBUG_INFO, "Remainder Bus = %x in the Seg:%x Bus:%x layer\n", mRemainderBus, RpSeg, RpBus));
      if (IsLastP2pbDevice(HpcPciAddress)) {
        DEBUG((DEBUG_INFO, "This device is the last P2pb in the same layer, so we need to add Remainder Bus\n"));
        *RsvdExtraBusNum = (UINT8)(*RsvdExtraBusNum + mRemainderBus);
      }
    }
    *RsvdPcieMegaMem = (*RsvdPcieMegaMem - mNumOfHotPlugBridge[Index].NumOfP2pbNonHotPlug)/mNumOfHotPlugBridge[Index].NumOfP2pbHotPlug;
    *RsvdPciePMegaMem = (*RsvdPciePMegaMem)/mNumOfHotPlugBridge[Index].NumOfP2pbHotPlug;
    *RsvdPcieKiloIo = (*RsvdPcieKiloIo)/mNumOfHotPlugBridge[Index].NumOfP2pbHotPlug;
    DEBUG ((DEBUG_INFO, "Current resource as below \n"));
    DEBUG ((DEBUG_INFO, "BusNum = %x \n", *RsvdExtraBusNum));
    DEBUG ((DEBUG_INFO, "PcieMegaMem = %x \n", *RsvdPcieMegaMem));
    DEBUG ((DEBUG_INFO, "PciePMegaMem = %x \n", *RsvdPciePMegaMem));
    DEBUG ((DEBUG_INFO, "PcieKiloIo = %x \n", *RsvdPcieKiloIo));
  } else {
    DEBUG ((DEBUG_INFO, "GetCurrentBusP2pbHotPlug Status =%r\n", Status));
  }
}

/**
  Returns the resource padding required by the PCI bus that is controlled by the specified Hot Plug Controller.

  @param[in]  This           The pointer to the instance of the EFI_PCI_HOT_PLUG_INIT protocol. initialized.
  @param[in]  HpcDevicePath  The Device Path to the Hot Plug Controller.
  @param[in]  HpcPciAddress  The address of the Hot Plug Controller function on the PCI bus.
  @param[out] HpcState       The state of the Hot Plug Controller hardware. The type EFI_HPC_STATE is defined in section 3.1.
  @param[out] Padding        This is the amount of resource padding required by the PCI bus under the control of the specified Hpc. Since the caller does not know the size of this buffer, this buffer is allocated by the callee and freed by the caller.
  @param[out] Attribute      Describes how padding is accounted for.

  @retval     EFI_SUCCESS.
**/
EFI_STATUS
EFIAPI
GetResourcePadding (
  IN  EFI_PCI_HOT_PLUG_INIT_PROTOCOL  *This,
  IN  EFI_DEVICE_PATH_PROTOCOL        *HpcDevicePath,
  IN  UINT64                          HpcPciAddress,
  OUT EFI_HPC_STATE                   *HpcState,
  OUT VOID                            **Padding,
  OUT EFI_HPC_PADDING_ATTRIBUTES      *Attributes
  )
{
  EFI_ACPI_ADDRESS_SPACE_DESCRIPTOR *PaddingResource;
  EFI_STATUS                        Status;
  UINT8                             RsvdExtraBusNum = 0;
  UINT16                            RsvdPcieMegaMem = 10;
  UINT8                             PcieMemAddrRngMax = 1;
  UINT16                            RsvdPciePMegaMem = 10;
  UINT8                             PciePMemAddrRngMax = 1;
  UINT8                             RsvdPcieKiloIo = 4;
  UINTN                             RpBus;
  CHAR16                            *DevicePathStr;
  UINTN                             VariableSize;
  SA_SETUP                          SaSetup;

  PaddingResource = AllocatePool (PADDING_NUM * sizeof (EFI_ACPI_ADDRESS_SPACE_DESCRIPTOR) + sizeof (EFI_ACPI_END_TAG_DESCRIPTOR));
  ASSERT (PaddingResource != NULL);
  if (PaddingResource == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  *Padding = (VOID *) PaddingResource;

  // Record the segment number in Bit[32] of HpcPciAddress.
  DevicePathStr = ConvertDevicePathToText (HpcDevicePath, TRUE, TRUE);
  if (DevicePathStr != NULL) {
    if (StrnCmp (DevicePathStr, PCI_ROOT_0_STRING, StrLen (PCI_ROOT_0_STRING)) == 0) {
      DEBUG ((DEBUG_INFO, "Segment0 \n"));
      HpcPciAddress &= ~BIT32;
    } else if (StrnCmp (DevicePathStr, PCI_ROOT_1_STRING, StrLen (PCI_ROOT_1_STRING)) == 0) {
      DEBUG ((DEBUG_INFO, "Segment1 \n"));
      HpcPciAddress |= BIT32;
    }
  } else {
      return EFI_OUT_OF_RESOURCES;
  }

  RpBus  = (UINTN) ((HpcPciAddress >> 24) & 0xFF);

  if (RpBus == 0x00) {
    //
    // Handle the GetResourcePadding for RootPort
    //
    HandleResourcePaddingForRootPort (
      HpcDevicePath,
      HpcPciAddress,
      &RsvdExtraBusNum,
      &RsvdPcieMegaMem,
      &PcieMemAddrRngMax,
      &RsvdPciePMegaMem,
      &PciePMemAddrRngMax,
      &RsvdPcieKiloIo
    );
  } else {
    //
    // Handle the GetResourcePadding for Non-RootPort
    //
    HandleResourcePaddingForNonRootPort (
      HpcDevicePath,
      HpcPciAddress,
      &RsvdExtraBusNum,
      &RsvdPcieMegaMem,
      &PcieMemAddrRngMax,
      &RsvdPciePMegaMem,
      &PciePMemAddrRngMax,
      &RsvdPcieKiloIo
    );
  }

  Status = UpdateP2pbResourceRecord (
    HpcDevicePath,
    HpcPciAddress,
    RsvdExtraBusNum,
    RsvdPcieMegaMem,
    PcieMemAddrRngMax,
    RsvdPciePMegaMem,
    PciePMemAddrRngMax,
    RsvdPcieKiloIo
    );
  if (Status == EFI_SUCCESS) {
    mP2pbResourceRecordCount ++;
  }

  //
  // Padding for bus
  //
  ZeroMem (PaddingResource, PADDING_NUM * sizeof (EFI_ACPI_ADDRESS_SPACE_DESCRIPTOR) + sizeof (EFI_ACPI_END_TAG_DESCRIPTOR));
  *Attributes                   = EfiPaddingPciBus;

  PaddingResource->Desc         = 0x8A;
  PaddingResource->Len          = 0x2B;
  PaddingResource->ResType      = ACPI_ADDRESS_SPACE_TYPE_BUS;
  PaddingResource->GenFlag      = 0x0;
  PaddingResource->SpecificFlag = 0;
  PaddingResource->AddrRangeMin = 0;
  PaddingResource->AddrRangeMax = 0;
  PaddingResource->AddrLen      = RsvdExtraBusNum - 1; // Subordinate Bus Number = Secondary Bus Number + Reserved Bus Number, but Secondary Bus Number already consumed 1 bus number itself.

  //
  // Padding for non-prefetchable memory
  //
  PaddingResource++;
  PaddingResource->Desc                 = 0x8A;
  PaddingResource->Len                  = 0x2B;
  PaddingResource->ResType              = ACPI_ADDRESS_SPACE_TYPE_MEM;
  PaddingResource->GenFlag              = 0x0;

  //
  // Check Enable Above 4GB MMIO or not
  //
  VariableSize = sizeof(SA_SETUP);
  Status = gRT->GetVariable (
                  L"SaSetup",
                  &gSaSetupVariableGuid,
                  NULL,
                  &VariableSize,
                  &SaSetup
                  );
  if (!EFI_ERROR(Status)) {
    if (SaSetup.EnableAbove4GBMmio == 1) {
      PaddingResource->AddrSpaceGranularity = 64;
    } else {
      PaddingResource->AddrSpaceGranularity = 32;
    }
  }
  PaddingResource->SpecificFlag         = 0;
  //
  // Pad non-prefetchable
  //
  PaddingResource->AddrRangeMin = 0;
  PaddingResource->AddrLen      = RsvdPcieMegaMem * 0x100000;
  PaddingResource->AddrRangeMax = (1 << PcieMemAddrRngMax) - 1;

  //
  // Padding for prefetchable memory
  //
  PaddingResource++;
  PaddingResource->Desc                 = 0x8A;
  PaddingResource->Len                  = 0x2B;
  PaddingResource->ResType              = ACPI_ADDRESS_SPACE_TYPE_MEM;
  PaddingResource->GenFlag              = 0x0;
  if (!EFI_ERROR(Status) && (SaSetup.EnableAbove4GBMmio == 1)) {
    PaddingResource->AddrSpaceGranularity = 64;
  } else {
    PaddingResource->AddrSpaceGranularity = 32;
  }
  PaddingResource->SpecificFlag         = 06;
  //
  // Padding for prefetchable memory
  //
  PaddingResource->AddrRangeMin = 0;
  PaddingResource->AddrLen      = (UINT64) RsvdPciePMegaMem * 0x100000;
  //
  // Pad 16 MB of MEM
  //
  PaddingResource->AddrRangeMax = (1 << PciePMemAddrRngMax) - 1;

  //
  // Alignment
  //
  // Padding for I/O
  //
  PaddingResource++;
  PaddingResource->Desc         = 0x8A;
  PaddingResource->Len          = 0x2B;
  PaddingResource->ResType      = ACPI_ADDRESS_SPACE_TYPE_IO;
  PaddingResource->GenFlag      = 0x0;
  PaddingResource->SpecificFlag = 0;
  PaddingResource->AddrRangeMin = 0;
  PaddingResource->AddrLen      = RsvdPcieKiloIo * 0x400;
  //
  // Pad 4K of IO
  //
  PaddingResource->AddrRangeMax = 1;
  //
  // Alignment
  //
  // Terminate the entries.
  //
  PaddingResource++;
  ((EFI_ACPI_END_TAG_DESCRIPTOR *) PaddingResource)->Desc     = ACPI_END_TAG_DESCRIPTOR;
  ((EFI_ACPI_END_TAG_DESCRIPTOR *) PaddingResource)->Checksum = 0x0;

  *HpcState = EFI_HPC_STATE_INITIALIZED | EFI_HPC_STATE_ENABLED;

  return EFI_SUCCESS;
}
