/** @file
  This is the driver that locates the TRACEHUB_DATA_HOB, if it
  exists, and saves the data to NVRAM.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/
#include "SaveTraceHubConfig.h"
#include <TraceHubDataHob.h>

//
// Variable to save TraceHub Data
//
GLOBAL_REMOVE_IF_UNREFERENCED CHAR16    mTraceHubConfigVariable[] = L"TraceHubData";

/**
  This is the standard EFI driver point that detects whether there is a
  TRACEHUB_DATA_HOB and, if so, saves its data to NVRAM.

  @param[in] ImageHandle    Handle for the image of this driver
  @param[in] SystemTable    Pointer to the EFI System Table

  @retval    EFI_UNSUPPORTED
**/
EFI_STATUS
EFIAPI
SaveTraceHubConfigEntryPoint (
  IN EFI_HANDLE         ImageHandle,
  IN EFI_SYSTEM_TABLE   *SystemTable
  )
{
  EFI_STATUS                          Status;
  EDKII_VARIABLE_LOCK_PROTOCOL        *VariableLock;
  TRACEHUB_DATA_HOB                   *TraceHubDataHob;
  UINT32                              BaseAddr;
  UINTN                               VariableSize;

  BaseAddr = 0;
  VariableSize = sizeof (BaseAddr);

  DEBUG ((DEBUG_INFO, "SaveTraceHubConfigEntryPoint\n"));
  //
  // Search for the TraceHub Data HOB.  If it is not present, then
  // there's nothing we can do. It may not exist on the update path.
  //
  TraceHubDataHob = NULL;
  TraceHubDataHob = (TRACEHUB_DATA_HOB *) GetFirstGuidHob (&gTraceHubDataHobGuid);
  if (TraceHubDataHob == NULL) {
    DEBUG ((DEBUG_ERROR, "Cannot get TraceHubDataHob\n"));
    return EFI_UNSUPPORTED;
  }
  BaseAddr = TraceHubDataHob->TraceHubMemBase;

  //
  // Save TraceHub Configuration Data to variable from HOB
  //

  Status = gRT->GetVariable (
                  mTraceHubConfigVariable,
                  &gTraceHubDataHobGuid,
                  NULL,
                  &VariableSize,
                  &BaseAddr
                  );

  if (EFI_ERROR (Status)) {
    // Create Variable first time
    Status = gRT->SetVariable (
                    mTraceHubConfigVariable,
                    &gTraceHubConfigVariableGuid,
                    (EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS),
                    VariableSize,
                    &BaseAddr
                    );
    ASSERT_EFI_ERROR (Status);
  } else {
    if (TraceHubDataHob->TraceHubMemBase > 0) {
      if (BaseAddr != TraceHubDataHob->TraceHubMemBase) {
        // Update variable
        Status = gRT->SetVariable (
                        mTraceHubConfigVariable,
                        &gTraceHubConfigVariableGuid,
                        (EFI_VARIABLE_NON_VOLATILE | EFI_VARIABLE_BOOTSERVICE_ACCESS),
                        VariableSize,
                        &BaseAddr
                        );
        ASSERT_EFI_ERROR (Status);
      }
    } else {
      // Delete variable
      Status = gRT->SetVariable (
                      mTraceHubConfigVariable,
                      &gTraceHubConfigVariableGuid,
                      0,
                      0,
                      NULL
                      );
      ASSERT_EFI_ERROR (Status);
    }
 }

  DEBUG ((DEBUG_INFO, "TraceHubMemBase is 0x%x\n", TraceHubDataHob->TraceHubMemBase));

  //
  // Mark TraceHubData to read-only if the Variable Lock protocol exists
  //
  Status = gBS->LocateProtocol (&gEdkiiVariableLockProtocolGuid, NULL, (VOID **)&VariableLock);
  if (!EFI_ERROR (Status)) {
    Status = VariableLock->RequestToLock (VariableLock, L"TraceHubData", &gTraceHubConfigVariableGuid);
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "Fail to Lock TraceHubData\n"));
    }
  }
  //
  // This driver does not produce any protocol services, so always unload it.
  //
  return EFI_UNSUPPORTED;
}
