/** @file
  Source code file for the Platform Init Advanced DXE module

@copyright
  INTEL CONFIDENTIAL
  Copyright 2014 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/
#include "PlatformInitAdvancedDxe.h"
#include <CpuRegs.h>
#include <Guid/EventGroup.h>
#include <Library/BootGuardLib.h>
#include <Protocol/CpuInfo.h>
#include <Protocol/PlatformNvsArea.h>
#include <Protocol/VariableLock.h>
#include <Library/CpuPlatformLib.h>
#include <Library/SmbiosProcessorLib.h>
#include <Guid/TpmInstance.h>
#include <Library/Tpm2CommandLib.h>
#include <Library/DxeSmbiosFirmwareVersionInfoLib.h>
#include <Library/PostCodeLib.h>
#include <PlatformPostCode.h>
#include <Register/Cpuid.h>
#include <Register/PmcRegs.h>
#include <Protocol/Tcg2Protocol.h>
#include <Library/TpmMeasurementLib.h>
#include <Library/UefiLib.h>
#include <Protocol/ResetNotification.h>
#include <Register/CommonMsr.h>
#include <Register/TglMsr.h>
#if FixedPcdGetBool(PcdFspWrapperEnable) == 1
#include <Library/MtrrLib.h>
#include <Library/SiMtrrLib.h>
#include <Protocol/MpService.h>
#endif


GLOBAL_REMOVE_IF_UNREFERENCED EFI_EVENT                         EndOfDxeEvent;
GLOBAL_REMOVE_IF_UNREFERENCED PLATFORM_NVS_AREA_PROTOCOL        *mPlatformNvsAreaProtocol;
GLOBAL_REMOVE_IF_UNREFERENCED PLATFORM_NVS_AREA                 *mPlatformNvsAreaPtr;

#define TBT_SECURITY_EVENT_STRING      "DMA Protection Disabled"
#define TBT_SECURITY_EVENT_STRING_LEN  (sizeof(TBT_SECURITY_EVENT_STRING) - 1)

VOID
EFIAPI
PlatformResetNotificationCallback (
  IN EFI_RESET_TYPE           ResetType,
  IN EFI_STATUS               ResetStatus,
  IN UINTN                    DataSize,
  IN VOID                     *ResetData OPTIONAL
  );

/**
  This function handles PlatformInit task at the end of DXE

  @param[in]  Event     The Event this notify function registered to.
  @param[in]  Context   Pointer to the context data registered to the
                        Event.
**/
VOID
EFIAPI
EndofDxeCallback (
  IN EFI_EVENT Event,
  IN VOID      *Context
  )
{
  MSR_ANC_SACM_INFO_REGISTER    AncSacmInfoMsr;
  static BOOLEAN                S3DataSaved = FALSE;
  UINT16                        ABase;
  UINT32                        SmiEn;
  UINT16                        Pm1Sts;

  gBS->CloseEvent (Event);

  if (S3DataSaved) {
    return ;
  }

  ABase = PmcGetAcpiBase ();
  SmiEn = IoRead32 (ABase + R_ACPI_IO_SMI_EN);
  //
  // Disabled Legacy USB Logic that generates SMI, during S3 Resume.
  //
  SmiEn &= ~(B_ACPI_IO_SMI_EN_LEGACY_USB | B_ACPI_IO_SMI_EN_LEGACY_USB2 | B_ACPI_IO_SMI_EN_PERIODIC | B_ACPI_IO_SMI_EN_SWSMI_TMR);

  S3BootScriptSaveIoWrite (
    S3BootScriptWidthUint32,
    (UINTN) (ABase + R_ACPI_IO_SMI_EN),
    1,
    &SmiEn
    );
  //
  // Clear bust master status bit on S3 resume
  //
  Pm1Sts = B_ACPI_IO_PM1_STS_BM;

  S3BootScriptSaveIoWrite (
    S3BootScriptWidthUint16,
    (UINTN) (ABase + R_ACPI_IO_PM1_STS),
    1,
    &Pm1Sts
    );

  S3DataSaved = TRUE;


  ///
  /// Verify if Boot Guard is supported
  ///
  AncSacmInfoMsr.Uint64 = AsmReadMsr64 (MSR_ANC_SACM_INFO);
  if (IsBootGuardSupported ()) {
    ///
    /// Identify if Revocation is requested by Boot Guard ACM
    ///
    if (AncSacmInfoMsr.Bits.SacmData & BIT7) {
      BootGuardOemRevocationHook ();
    }
  }

}

/**
  Registers callback for.PlatformInitEndOfDxe

**/
VOID
RegisterEndOfDxeCallbacks (
  VOID
  )
{
  EFI_STATUS                     Status;

  ///
  /// Performing PlatformInitEndOfDxe after the gEfiEndOfDxeEventGroup is signaled.
  ///
  Status = gBS->CreateEventEx (
                  EVT_NOTIFY_SIGNAL,
                  TPL_CALLBACK,
                  EndofDxeCallback,
                  NULL,
                  &gEfiEndOfDxeEventGroupGuid,
                  &EndOfDxeEvent
                  );
  ASSERT_EFI_ERROR (Status);

}

#if FixedPcdGetBool(PcdFspWrapperEnable) == 1
/**
  A minimal wrapper function that allows MtrrSetAllMtrrs () to be passed to
  EFI_MP_SERVICES_PROTOCOL.StartupAllAPs () as Procedure.

  @param[in] Buffer  Pointer to an MTRR_SETTINGS object, to be passed to
                     MtrrSetAllMtrrs ().
**/
VOID
EFIAPI
SetMtrrsFromBuffer (
  IN VOID *Buffer
  )
{
  MtrrSetAllMtrrs (Buffer);
}
#endif

/**
  This function handles PlatformInitDxe task at the ready to boot

  @param[in]  Event     The Event this notify function registered to.
  @param[in]  Context   Pointer to the context data registered to the Event.
**/
VOID
EFIAPI
PlatformDxeReadyToBootCallback (
  IN EFI_EVENT Event,
  IN VOID      *Context
  )
{
#if FixedPcdGetBool(PcdFspWrapperEnable) == 1
  EFI_STATUS                Status;
  MTRR_SETTINGS             MtrrSetting;
  EFI_MP_SERVICES_PROTOCOL  *MpService;

  DEBUG ((DEBUG_INFO, "PlatformDxeReadyToBootCallback start\n"));

  Status = MtrrTransfer2DefaultWB (&MtrrSetting);
  if (EFI_ERROR (Status)) {
    return;
  }

  //
  // Synchronize the update with all APs
  //
  Status = gBS->LocateProtocol (
                  &gEfiMpServiceProtocolGuid,
                  NULL,
                  (VOID **)&MpService
                  );
  if (!EFI_ERROR (Status)) {
    MtrrGetAllMtrrs (&MtrrSetting);
    MpService->StartupAllAPs (
                 MpService,          // This
                 SetMtrrsFromBuffer, // Procedure
                 FALSE,              // SingleThread
                 NULL,               // WaitEvent
                 0,                  // TimeoutInMicrosecsond
                 &MtrrSetting,       // ProcedureArgument
                 NULL                // FailedCpuList
                 );
  }

  DEBUG ((DEBUG_INFO, "PlatformDxeReadyToBootCallback end\n"));
#endif
  gBS->CloseEvent (Event);

  return;
}

/**
  Registers callback for PlatformDxeReadyToBoot

**/
VOID
RegisterReadyToBootCallback (
  VOID
  )
{
  EFI_STATUS                     Status;
  EFI_EVENT                      Event;

  Status = EfiCreateEventReadyToBootEx (
             TPL_CALLBACK,
             PlatformDxeReadyToBootCallback,
             NULL,
             &Event
             );

  ASSERT_EFI_ERROR (Status);
}

/**
  Call back function for reset notification.

  @param[in] ResetType            UEFI defined reset type.
  @param[in] ResetStatus          The status code for the reset.
  @param[in] DataSize             The size of ResetData in bytes.
  @param[in] ResetData            Optional element used to introduce a platform specific reset.
                                  The exact type of the reset is defined by the EFI_GUID that follows
                                  the Null-terminated Unicode string.
**/
VOID
EFIAPI
PlatformResetNotificationCallback (
  IN EFI_RESET_TYPE           ResetType,
  IN EFI_STATUS               ResetStatus,
  IN UINTN                    DataSize,
  IN VOID                     *ResetData OPTIONAL
  )
{
  EFI_STATUS                  Status;

  DEBUG ((DEBUG_INFO, "PlatformResetNotificationCallback\n"));

  Status = Tpm2Shutdown (TPM_SU_CLEAR);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Tpm2Shutdown (TPM_SU_CLEAR) failed: %r\n", Status));
  }
}


/**
  Hook to reset notification protocol to properly reset function with TPM.
  @param[in]  Event     Event whose notification function is being invoked
  @param[in]  Context   Pointer to the notification function's context
**/
VOID
EFIAPI
OnResetNotifyInstall (
  IN EFI_EVENT                      Event,
  IN VOID                           *Context
  )
{

  EFI_STATUS  Status;
  EFI_RESET_NOTIFICATION_PROTOCOL   *ResetNotify;

  Status = gBS->LocateProtocol (&gEfiResetNotificationProtocolGuid, NULL, (VOID **) &ResetNotify);
  if (!EFI_ERROR (Status)) {
    ResetNotify->RegisterResetNotify (ResetNotify, PlatformResetNotificationCallback);
    if(Event) gBS->CloseEvent (Event);
  }
}

/**
  Security EndOfDxe CallBack Function
  If the firmware/BIOS has an option to enable and disable DMA protections via a VT-d switch in BIOS options, then the shipping configuration must be with VT-d protection enabled.
  On every boot where VT-d/DMA protection is disabled, or will be disabled, or configured to a lower security state, and a platform has a TPM enabled, then the platform SHALL
  extend an EV_EFI_ACTION event into PCR[7] before enabling external DMA
  The event string SHALL be "DMA Protection Disabled". The platform firmware MUST log this measurement in the event log using the string "DMA Protection Disabled" for the Event Data.
  Measure and log launch of TBT Security, and extend the measurement result into a specific PCR.
  Extend an EV_EFI_ACTION event into PCR[7] before enabling external DMA. The event string SHALL be "DMA Protection Disabled". The platform firmware MUST log this measurement
  in the event log using the string "DMA Protection Disabled" for the Event Data.

  @param[in] Event     - A pointer to the Event that triggered the callback.
  @param[in] Context   - A pointer to private data registered with the callback function.
**/
VOID
EFIAPI
ExtendPCR7CallBack (
  IN EFI_EVENT          Event,
  IN VOID               *Context
  )
{
  UINTN                 Status;
  UINT64                HashDataLen;

  DEBUG ((DEBUG_INFO, "ExtendPCR7CallBack START\n"));

  //
  // When VT-d/DMA protection is disabled and a platform has a TPM enabled,
  // the platform SHALL extend an EV_EFI_ACTION event into PCR[7]
  //
  HashDataLen = TBT_SECURITY_EVENT_STRING_LEN;

  Status = TpmMeasureAndLogData (
             7,
             EV_EFI_ACTION,
             TBT_SECURITY_EVENT_STRING,
             (UINT32) HashDataLen,
             TBT_SECURITY_EVENT_STRING,
             HashDataLen
             );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "TpmMeasureAndLogData Status: %d\n", Status));
  } else {
    DEBUG ((DEBUG_INFO, "TpmMeasureAndLogData Successfully\n"));
  }

  DEBUG ((DEBUG_INFO, "ExtendPCR7CallBack END\n"));
}

/**
  Register an End of DXE event for extended a TPM log to PCR[7] when vtd is diable
  This feature is introduced by TBT Security requirment
**/
VOID
RegisterExtendPCR7CallBack (
  VOID
  )
{
  EFI_STATUS                Status;
  UINTN                     DataSize;
  SA_SETUP                  *SaSetup;

  Status                    = EFI_SUCCESS;
  SaSetup                   = NULL;
  DataSize                  = 0;


  DataSize = sizeof (SA_SETUP);
  SaSetup = AllocateZeroPool (DataSize);
  if (SaSetup == NULL) {
    DEBUG ((DEBUG_ERROR, "Failed to allocate SaSetup size\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

  Status = gRT->GetVariable (
                  L"SaSetup",
                  &gSaSetupVariableGuid,
                  NULL,
                  &DataSize,
                  SaSetup
                  );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Get SaSetup Status: %d\n", Status));
    goto Exit;
  }

  if (SaSetup->EnableVtd == FALSE) {
    //
    // Register an End of DXE event for extended a TPM log to PCR[7]
    //
    DEBUG ((DEBUG_ERROR, "Rgister an End of DXE event for extended a TPM log to PCR[7] when Vtd is disable\n"));
    Status = gBS->CreateEventEx (
                    EVT_NOTIFY_SIGNAL,
                    TPL_CALLBACK,
                    ExtendPCR7CallBack,
                    NULL,
                    &gEfiEndOfDxeEventGroupGuid,
                    &EndOfDxeEvent
                    );
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "Failed to Register an End of DXE event for extended a TPM log to PCR[7], Status: %d\n", Status));
      gBS->CloseEvent (EndOfDxeEvent);
      goto Exit;
    }
  }

Exit:
  if (SaSetup != NULL) {
    FreePool (SaSetup);
  }

}

/**
  Allocate MemoryType below 4G memory address.

  @param[in] MemoryType    The type of memory to allocate
  @param[in] Size               Size of memory to allocate.
  @param[OUT] Buffer         Allocated address for output.

  @retval EFI_SUCCESS       Memory successfully allocated.
  @retval Other                  Other errors occur.
**/
EFI_STATUS
AllocateMemoryBelow4G(
  IN   EFI_MEMORY_TYPE MemoryType,
  IN   UINTN           Size,
  OUT  VOID           **Buffer
)
{
  UINTN                 Pages;
  EFI_PHYSICAL_ADDRESS  Address;
  EFI_STATUS            Status;

  Pages = EFI_SIZE_TO_PAGES(Size);
  Address = 0xffffffff;

  Status = (gBS->AllocatePages) (
                  AllocateMaxAddress,
                  MemoryType,
                  Pages,
                  &Address
                  );
  DEBUG((DEBUG_INFO, "AllocateMemoryBelow4G: %r\n", Status));
  *Buffer = (VOID *)(UINTN)Address;

  return Status;
};

/**
  Read And load the LPM register in NVS area
**/
EFI_STATUS
ReadLowPowerModeRequirementReg(
  VOID
  )
{
  EFI_STATUS              Status;
  UINT32                  *LpmData;
  UINT32                  LpmDataLen;

  Status = gBS->LocateProtocol (
                &gPlatformNvsAreaProtocolGuid,
                NULL,
                (VOID **)&mPlatformNvsAreaProtocol
                );
  DEBUG ((DEBUG_ERROR, "mPlatformNvsAreaProtocol: %r\n", Status));
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return Status;
  }

  mPlatformNvsAreaPtr = mPlatformNvsAreaProtocol->Area;

  LpmDataLen = PMC_LPM_REQ_DATA_LEN; //LPM register counter is 48, so length is 192 byte

  // Allocate memory in ACPI NVS
  //
  Status = AllocateMemoryBelow4G (EfiACPIMemoryNVS, LpmDataLen, (VOID **)&LpmData);
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return Status;
  }

  ZeroMem (LpmData, LpmDataLen);

  //
  // Read out LPM requiremetn register data
  //
  Status = PmcReadLpmReqData (LpmData, LpmDataLen);
  if (EFI_ERROR (Status)) {
    return Status;
  }

  mPlatformNvsAreaPtr->LpmReqRegAddr = (UINT32) (UINTN) LpmData;

  return Status;
}

/**
  Entry point for the driver.

  @param[in]  ImageHandle  Image Handle.
  @param[in]  SystemTable  EFI System Table.

  @retval     EFI_SUCCESS  Function has completed successfully.
  @retval     Others       All other error conditions encountered result in an ASSERT.
**/
EFI_STATUS
EFIAPI
PlatformInitAdvancedDxeEntryPoint (
  IN EFI_HANDLE               ImageHandle,
  IN EFI_SYSTEM_TABLE         *SystemTable
  )
{
  EFI_STATUS                  Status;
  SETUP_DATA                  SetupData;
  UINTN                       VariableSize;
  UINT32                      VariableAttr;
  UINT8                       *SecureBootState;
  VOID                        *Registration;

  PostCode (PLATFORM_INIT_DXE_ENTRY);
  Status = EFI_SUCCESS;

#if FixedPcdGetBool (PcdSinitAcmBinEnable) == 1
  Status = TxtSinitAcmLoad ();
  if (Status == EFI_SUCCESS) {
    DEBUG ((DEBUG_INFO, "TXTDXE: Found SINIT ACM In FV\n"));
  } else {
      DEBUG ((DEBUG_INFO, "TXTDXE: Error finding SINIT binary in FVs\n"));
  }
#endif

  Status = InstallSmbiosFviOemType (); // todo: remove it once Bios Guard convert the HOB generation.
  ASSERT_EFI_ERROR (Status);

  SecureBootState = NULL;
  GetVariable2 (L"SecureBootEnable", &gEfiSecureBootEnableDisableGuid, (VOID**)&SecureBootState, NULL);

  //
  // Set CsmControl Setup variable reflecting Secure boot status. Whenever Secure boot is enabled, CSM must be turned off.
  //
  VariableSize = sizeof (SETUP_DATA);
  Status = gRT->GetVariable (
                  L"Setup",
                  &gSetupVariableGuid,
                  &VariableAttr,
                  &VariableSize,
                  &SetupData
                  );
  ASSERT_EFI_ERROR (Status);

  SetupData.OpRomPost  = OPROM_EFI;

  if ((SecureBootState != NULL) && (*SecureBootState)) {
    if (!EFI_ERROR (Status)) {
      SetupData.OpRomPost  = OPROM_EFI;
    }
  }

  Status = gRT->SetVariable (
                  L"Setup",
                  &gSetupVariableGuid,
                  VariableAttr,
                  VariableSize,
                  &SetupData
                  );
  ASSERT_EFI_ERROR (Status);

  if (SecureBootState != NULL) {
    FreePool (SecureBootState);
  }


  //
  // Initialize System Agent platform settings
  //
  SaPlatformInitDxe ();

  //
  // Update DIMM funtionalities for desktop and server boards
  //
  UpdateDimmPopulation ();

  ReadLowPowerModeRequirementReg ();

  RegisterEndOfDxeCallbacks ();
  RegisterReadyToBootCallback ();

  //
  // Add Smbios type 4 and type 7 table.
  //
  AddSmbiosProcessorAndCacheTables ();

  //
  // Check if TPM2.0 exist.
  //
  if (CompareGuid (PcdGetPtr (PcdTpmInstanceGuid), &gEfiTpmDeviceInstanceNoneGuid)) {
  } else if (CompareGuid (PcdGetPtr (PcdTpmInstanceGuid), &gEfiTpmDeviceInstanceTpm12Guid)) {
  } else {
    //
    // Register Pch reset callback to shutdown TPM
    //
    EfiCreateProtocolNotifyEvent (
      &gEfiResetNotificationProtocolGuid,
      TPL_CALLBACK,
      OnResetNotifyInstall,
      NULL,
      &Registration
      );

  }

  //
  // Inform OS by TPM PCR7 when Vtd is disable
  //
  RegisterExtendPCR7CallBack ();

  PostCode (PLATFORM_INIT_DXE_EXIT);

  return Status;
}

#if FixedPcdGetBool(PcdSinitAcmBinEnable) == 1
/**

  This function looks for SINIT ACM in FVs and updates TXT HOB
  with SINIT ACM Base and Size.

  @retval EFI_SUCCESS     - SINIT ACM found and copied.
  @retval EFI_NOT_FOUND   - If TxtInfoHob is not found
  @retval EFI_UNSUPPORTED - TXT Device memory not available.

**/
EFI_STATUS
TxtSinitAcmLoad (
  VOID
  )
{
  EFI_STATUS                    Status;
  TXT_INFO_HOB                  *HobList;
  TXT_INFO_DATA                 *TxtInfoData;
  EFI_HANDLE                    *HandleBuffer;
  UINTN                         NumberOfHandles;
  EFI_FIRMWARE_VOLUME2_PROTOCOL *FwVol;
  UINTN                         Size;
  EFI_FV_FILETYPE               FileType;
  UINT32                        FvStatus;
  EFI_FV_FILE_ATTRIBUTES        Attributes;
  UINT64                        TxtHeapMemoryBase;
  UINT64                        TxtSinitMemoryBase;
  UINT64                        *Ptr64;
  EFI_PHYSICAL_ADDRESS          TopAddr;
  BIOS_OS_DATA_REGION           *BiosOsDataRegion;

  HandleBuffer = NULL;
  NumberOfHandles = 0;
  FwVol = NULL;
  FileType = 0;
  Attributes = 0;
  BiosOsDataRegion = NULL;

  ///
  /// Find TXT HOB
  ///
  HobList = (TXT_INFO_HOB *) GetFirstGuidHob (&gTxtInfoHobGuid);
  if (HobList == NULL) {
    return EFI_NOT_FOUND;
  }

  TxtInfoData = &HobList->Data;
  ///
  /// Check TXT mode
  ///
  if (TxtInfoData->TxtMode == 0) {
    return EFI_UNSUPPORTED;
  }

  if ((TxtInfoData == 0) ||
      (TxtInfoData->TxtDprMemoryBase == 0) ||
      (TxtInfoData->TxtDprMemorySize == 0) ||
      (TxtInfoData->TxtHeapMemorySize == 0) ||
      (TxtInfoData->SinitMemorySize == 0)
      ) {
    return EFI_UNSUPPORTED;
  } else {
    ///
    /// Use address passed from PEI
    ///
    TopAddr             = TxtInfoData->TxtDprMemoryBase + TxtInfoData->TxtDprMemorySize;

    TxtHeapMemoryBase   = (UINT64) (TopAddr - TxtInfoData->TxtHeapMemorySize);

    TxtSinitMemoryBase  = TxtHeapMemoryBase - TxtInfoData->SinitMemorySize;
  }
  ///
  /// Start looking for SINIT ACM in FVs
  ///
  DEBUG ((DEBUG_INFO, "TXTDXE: Looking for SINIT ACM in FVs\n"));

  Status = gBS->LocateHandleBuffer (
    ByProtocol,
    &gEfiFirmwareVolume2ProtocolGuid,
    NULL,
    &NumberOfHandles,
    &HandleBuffer
    );
  ASSERT_EFI_ERROR (Status);
  ///
  /// Looking for FV with SinitAC binary
  ///
  for (UINTN i = 0; i < NumberOfHandles; i++) {
    ///
    /// Get the protocol on this handle
    ///
    Status = gBS->HandleProtocol (
      HandleBuffer[i],
      &gEfiFirmwareVolume2ProtocolGuid,
      &FwVol
      );
    ASSERT_EFI_ERROR (Status);
    ///
    /// See if it has the SinitACM file
    ///
    Size = 0;
    FvStatus = 0;
    Status = FwVol->ReadFile (
      FwVol,
      &gSinitModuleGuid,
      NULL,
      &Size,
      &FileType,
      &Attributes,
      &FvStatus
      );
    ///
    /// If the binary was located, then break
    ///
    if (Status == EFI_SUCCESS) {
      DEBUG ((DEBUG_INFO, "TXTDXE: Found SINIT ACM In FV\n"));
      break;
    }
  }
  FreePool (HandleBuffer);
  ///
  /// Sanity check
  ///
  if ( !(FwVol == NULL || Status != EFI_SUCCESS || Size >= TxtInfoData->SinitMemorySize ) ) {
    ///
    /// Read Sinit ACM from the storage file.
    ///
    Ptr64 = (UINT64 *)TxtSinitMemoryBase;
    Status = FwVol->ReadFile (
      FwVol,
      &gSinitModuleGuid,
      &Ptr64,
      &Size,
      &FileType,
      &Attributes,
      &FvStatus
    );
    ASSERT_EFI_ERROR (Status);
    TxtInfoData->SinitAcmSize = (UINT64) Size;

    Ptr64 = (UINT64 *) TxtHeapMemoryBase;
    ///
    /// BiosOsDataSize plus size of data size field itself
    ///
    BiosOsDataRegion                = (BIOS_OS_DATA_REGION *) (Ptr64 + 1);
    BiosOsDataRegion->BiosSinitSize = (UINT32) TxtInfoData->SinitAcmSize;
    DEBUG ((DEBUG_INFO, "TXTDXE: Copy SINIT ACM Done\n"));
  }
  return Status;
}
#endif
