/** @file
  Source code file for Platform Init PEI module

@copyright
  INTEL CONFIDENTIAL
  Copyright 2014 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <Register/Msr.h>
#include <CpuRegs.h>
#include <SetupVariable.h>
#include <Library/IoLib.h>
#include <Library/HobLib.h>
#include <Library/DebugLib.h>
#include <Library/PchInfoLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/PmcLib.h>
#include <Library/GpioLib.h>
#include <Library/GpioNativeLib.h>
#include <Library/PeiPolicyInitLib.h>
#include <Ppi/EndOfPeiPhase.h>
#include <Library/MtrrLib.h>
#include <Ppi/ReadOnlyVariable2.h>
#include <Guid/SmramMemoryReserve.h>
#include <Library/ConfigBlockLib.h>
#include <Ppi/SiPolicy.h>
#include <PchPolicyCommon.h>
#include <Library/SiPolicyLib.h>

#include <Guid/FirmwareFileSystem2.h>
#include <Protocol/FirmwareVolumeBlock.h>
#include <Guid/SystemNvDataGuid.h>

#include <Library/CnviLib.h>
#include <Library/EcLib.h>
#include <Library/EcMiscLib.h>
#include <Library/HdaLib.h>
#include <Library/PostCodeLib.h>
#include <PlatformPostCode.h>
#include <Library/EcTcssLib.h>
#include <Register/ArchMsr.h>
#include <Register/TglMsr.h>
#if FixedPcdGetBool(PcdITbtEnable) == 1
#include <TcssDataHob.h>
#endif
#include <Library/ItbtPcieRpLib.h>
#include <Ppi/Spi.h>
#include <Register/MeRegs.h>
#include <Register/HeciRegs.h>

#if FixedPcdGet8(PcdFspModeSelection) == 1
#include <FspmUpd.h>
#include <FspsUpd.h>
#include <Library/FspCommonLib.h>
#endif
#include <Library/MtrrLib.h>
#include <Library/PeiTbtTaskDispatchLib.h>
#include <Register/PchRegs.h>
#include <Register/PmcRegs.h>
#include <Register/TelemetryRegs.h>
#include <Library/TelemetryLib.h>
#include <Guid/CrashLog.h>
#include <AttemptUsbFirst.h>
#include <PlatformBoardId.h>
#include <Core/Pei/PeiMain.h>
#include <Library/PchPciBdfLib.h>
#include <Ppi/GraphicsPlatformPolicyPpi.h>
#if FixedPcdGetBool(PcdFspWrapperEnable) == 1
/**
  Initializes the valid bits mask and valid address mask for MTRRs.

  This function initializes the valid bits mask and valid address mask for MTRRs.

  @param[out]  MtrrValidBitsMask     The mask for the valid bit of the MTRR
  @param[out]  MtrrValidAddressMask  The valid address mask for the MTRR

**/
VOID
MtrrLibInitializeMtrrMask (
  OUT UINT64 *MtrrValidBitsMask,
  OUT UINT64 *MtrrValidAddressMask
  );

/**
  Convert variable MTRRs to a RAW MTRR_MEMORY_RANGE array.
  One MTRR_MEMORY_RANGE element is created for each MTRR setting.
  The routine doesn't remove the overlap or combine the near-by region.

  @param[in]   VariableSettings      The variable MTRR values to shadow
  @param[in]   VariableMtrrCount     The number of variable MTRRs
  @param[in]   MtrrValidBitsMask     The mask for the valid bit of the MTRR
  @param[in]   MtrrValidAddressMask  The valid address mask for MTRR
  @param[out]  VariableMtrr          The array to shadow variable MTRRs content

  @return      Number of MTRRs which has been used.

**/
UINT32
MtrrLibGetRawVariableRanges (
  IN  MTRR_VARIABLE_SETTINGS  *VariableSettings,
  IN  UINTN                   VariableMtrrCount,
  IN  UINT64                  MtrrValidBitsMask,
  IN  UINT64                  MtrrValidAddressMask,
  OUT MTRR_MEMORY_RANGE       *VariableMtrr
  );

/**
  Apply the variable MTRR settings to memory range array.

  @param[in]      VariableMtrr      The variable MTRR array.
  @param[in]      VariableMtrrCount The count of variable MTRRs.
  @param[in, out] Ranges            Return the memory range array with new MTRR settings applied.
  @param[in]      RangeCapacity     The capacity of memory range array.
  @param[in, out] RangeCount        Return the count of memory range.

  @retval RETURN_SUCCESS            The memory range array is returned successfully.
  @retval RETURN_OUT_OF_RESOURCES   The count of memory ranges exceeds capacity.
**/
RETURN_STATUS
MtrrLibApplyVariableMtrrs (
  IN     CONST MTRR_MEMORY_RANGE *VariableMtrr,
  IN     UINT32                  VariableMtrrCount,
  IN OUT MTRR_MEMORY_RANGE       *Ranges,
  IN     UINTN                   RangeCapacity,
  IN OUT UINTN                   *RangeCount
  );

/**
  This function attempts to set the attributes into MTRR setting buffer.

  @param[in, out] MtrrSetting   - A buffer holding all MTRRs content.
  @param[in]      Ranges        - Array holding memory type settings.
  @param[in]      RangeCount    - Memory range count in the array.

  @retval Count of used variable Mtrrs
**/
EFI_STATUS
EFIAPI
EOPSetMemoryAttributesInMtrrSettings (
  IN OUT MTRR_SETTINGS      *MtrrSetting,
  IN     MTRR_MEMORY_RANGE  *Ranges,
  IN     UINTN              RangeCount
  )
{
  EFI_STATUS        Status;
  UINTN             Index;
  UINTN             HighIndex;
  UINT64            TopHighMemory;

  Status = EFI_NOT_FOUND;

  for (Index = 0, HighIndex = 0xFF; Index < RangeCount; Index++) {
    //
    // Set Mtrr variables from 1M.
    //
    if (Ranges[Index].BaseAddress < 0x100000) {
      Ranges[Index].Length -= 0x100000;
      Ranges[Index].BaseAddress = 0x100000;
    }
    if ((Ranges[Index].BaseAddress >= SIZE_4GB) && (Ranges[Index].Type == CacheWriteBack)) {
      HighIndex = Index;                       // Set above 4G attributes at the latest step.
    } else {
      Status = MtrrSetMemoryAttributeInMtrrSettings (
                 MtrrSetting,
                 Ranges[Index].BaseAddress,
                 Ranges[Index].Length,
                 Ranges[Index].Type
                 );
      ASSERT_EFI_ERROR (Status);
    }
  }
  if (HighIndex != 0xFF) {
    TopHighMemory = Ranges[HighIndex].BaseAddress + Ranges[HighIndex].Length;
    //
    // Try to cover memory as mmuch as we can.
    // In later phase boot loader code can re-configure MTRR to exclude flash region and get back above 4GB coverage.
    //
    do {
      Status = MtrrSetMemoryAttributeInMtrrSettings (
                 MtrrSetting,
                 SIZE_4GB,
                 TopHighMemory - SIZE_4GB,
                 CacheWriteBack
                 );
      if (TopHighMemory > SIZE_4GB) {
        TopHighMemory = RShiftU64 (TopHighMemory, 1);
      }
    } while ((EFI_SUCCESS != Status) && (TopHighMemory > SIZE_4GB));
  }

  return Status;
}
#endif

EFI_STATUS
EFIAPI
PlatformInitAdvancedEndOfPei (
  IN CONST EFI_PEI_SERVICES     **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR  *NotifyDescriptor,
  IN VOID                       *Ppi
  );

static EFI_PEI_NOTIFY_DESCRIPTOR  mEndOfPeiNotifyList = {
  (EFI_PEI_PPI_DESCRIPTOR_NOTIFY_CALLBACK | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gEfiEndOfPeiSignalPpiGuid,
  (EFI_PEIM_NOTIFY_ENTRY_POINT) PlatformInitAdvancedEndOfPei
};

EFI_STATUS
EFIAPI
PeiGraphicsPlatformInit (
  IN CONST EFI_PEI_SERVICES     **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR  *NotifyDescriptor,
  IN VOID                       *Ppi
  );

EFI_STATUS
EFIAPI
GetPeiPlatformLidStatus (
  OUT LID_STATUS  *CurrentLidStatus
  );

static EFI_PEI_NOTIFY_DESCRIPTOR  mPeiGraphicsPlatformNotifyList = {
  (EFI_PEI_PPI_DESCRIPTOR_NOTIFY_CALLBACK | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gPeiGraphicsPlatformPpiGuid,
  (EFI_PEIM_NOTIFY_ENTRY_POINT) PeiGraphicsPlatformInit
};

/**
  This function will pass data to Pei graphics PEIM.

  @param[in]  PeiServices  Pointer to PEI Services Table.
  @param[in]  NotifyDesc   Pointer to the descriptor for the Notification event that
                           caused this function to execute.
  @param[in]  Ppi          Pointer to the PPI data associated with this function.

  @retval     EFI_SUCCESS  The function completes successfully
  @retval     others
**/
EFI_STATUS
EFIAPI
PeiGraphicsPlatformInit (
  IN CONST EFI_PEI_SERVICES     **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR  *NotifyDescriptor,
  IN VOID                       *Ppi
  )
{
  EFI_STATUS                         Status;
  PEI_GRAPHICS_PLATFORM_POLICY_PPI   *GfxPlatformPolicyPpi;

  DEBUG ((DEBUG_INFO, "PeiGraphicsPlatformInit: Begin \n"));

  Status = EFI_SUCCESS;
  ///
  /// Locate GfxPlatformPolicyPpi
  ///
  Status = PeiServicesLocatePpi (&gPeiGraphicsPlatformPpiGuid, 0, NULL, (VOID *) &GfxPlatformPolicyPpi);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_WARN, "Failed to locate Ppi GfxPlatformPolicyPpi.\n"));
    return Status;
  }

  GfxPlatformPolicyPpi->Revision = PEI_GRAPHICS_PLATFORM_POLICY_REVISION;
  GfxPlatformPolicyPpi->GetPlatformLidStatus = (PEI_GET_PLATFORM_LID_STATUS) GetPeiPlatformLidStatus;

  DEBUG ((DEBUG_INFO, "PeiGraphicsPlatformInit: End \n"));

  return Status;
}


/**
  This function will return Lid Status in PEI phase.

  @param[out] CurrentLidStatus

  @retval     EFI_SUCCESS
  @retval     EFI_UNSUPPORTED
**/

EFI_STATUS
EFIAPI
GetPeiPlatformLidStatus (
  OUT LID_STATUS  *CurrentLidStatus
  )
{
  EFI_STATUS              Status;
  UINT8                   PortDataOut;
  UINT8                   DataBuffer[1];

  DEBUG ((DEBUG_INFO, "LidStatus Entry\n"));
  //
  // If the platform does not support a lid, the function must return EFI_UNSUPPORTED
  //
  if (PcdGet8 (PcdPlatformType) == TypeTrad && PcdGet8 (PcdPlatformFlavor) == FlavorDesktop) {
    DEBUG ((DEBUG_INFO, "Returning Lid status as unsupported to GOP for DT/AIO board\n"));
    return EFI_UNSUPPORTED;
  }
  if (PcdGetBool (PcdEcPresent)) {
    DataBuffer[0] = EC_D_LID_STATE;
    Status = ReadEcRam (DataBuffer);
    if (Status == EFI_SUCCESS) {
      PortDataOut = DataBuffer[0];
      if ((PortDataOut & EC_B_LID_STATUS_OPEN) == EC_B_LID_STATUS_OPEN) {
        *CurrentLidStatus = LidOpen;
      } else {
        *CurrentLidStatus = LidClosed;
      }
      DEBUG ((DEBUG_INFO, "LidStatus Exit\n"));
      return EFI_SUCCESS;
    }
  }

  DEBUG ((DEBUG_INFO, "LidStatus UnSupported\n"));
  return EFI_UNSUPPORTED;
}

/**
  Update MTRR setting and set write back as default memory attribute.

  @retval  EFI_SUCCESS  The function completes successfully.
  @retval  Others       Some error occurs.
**/
EFI_STATUS
EFIAPI
SetCacheMtrrForTraceHubAfterEndOfPei (
  VOID
  )
{
  EFI_STATUS                            Status;
  EFI_PEI_HOB_POINTERS                  Hob;
  UINTN                                 Index;
  UINT64                                SmramSize;
  UINT64                                SmramBase;
  EFI_SMRAM_HOB_DESCRIPTOR_BLOCK        *SmramHobDescriptorBlock;
  MTRR_SETTINGS                         MtrrSetting;
#if FixedPcdGetBool(PcdFspWrapperEnable) == 1
  UINTN                                 RangeCount;
  UINT32                                VariableMtrrCount;
  UINT64                                MtrrValidBitsMask;
  UINT64                                MtrrValidAddressMask;
  MTRR_MEMORY_RANGE                     RawVariableRanges[MTRR_NUMBER_OF_VARIABLE_MTRR];
  MTRR_MEMORY_RANGE                     Ranges[MTRR_NUMBER_OF_VARIABLE_MTRR];
  MTRR_SETTINGS                         UCMtrrSetting;
#else
  UINT64                                MemoryBase;
  UINT64                                MemoryLength;
  UINT64                                ExtraMemoryLength;
  UINT64                                Power2Length;
  EFI_BOOT_MODE                         BootMode;
  UINT32                                TraceHubMemSize;
  SI_CONFIG                             *SiConfig;
  SI_POLICY_PPI                         *SiPolicy;
  SI_PREMEM_POLICY_PPI                  *SiPreMemPolicyPpi;
  SI_PREMEM_CONFIG                      *SiPreMemConfig;
  SA_MISC_PEI_PREMEM_CONFIG             *MiscPeiPreMemConfig;
  PCH_TRACE_HUB_PREMEM_CONFIG           *PchTraceHubConfig;
  CPU_TRACE_HUB_PREMEM_CONFIG           *CpuTraceHubConfig;

  DEBUG ((DEBUG_INFO, "SetCacheMtrrForTraceHubAfterEndOfPei Entry\n"));

  ExtraMemoryLength = 0;
  SiConfig = NULL;
  SiPolicy = NULL;
  SiPreMemPolicyPpi = NULL;
  PchTraceHubConfig = NULL;
  MiscPeiPreMemConfig = NULL;
  CpuTraceHubConfig = NULL;

  Status = PeiServicesGetBootMode (&BootMode);
  ASSERT_EFI_ERROR (Status);

  if (BootMode == BOOT_ON_S3_RESUME) {
    return EFI_SUCCESS;
  }
  //
  // Clear the CAR Settings
  //
  ZeroMem (&MtrrSetting, sizeof (MTRR_SETTINGS));

  Status = PeiServicesLocatePpi (&gSiPreMemPolicyPpiGuid, 0, NULL, (VOID **) &SiPreMemPolicyPpi);
  ASSERT_EFI_ERROR (Status);

  Status = PeiServicesLocatePpi (&gSiPolicyPpiGuid, 0, NULL, (VOID **) &SiPolicy);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gSiConfigGuid, (VOID *) &SiConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gSiPreMemConfigGuid, (VOID *) &SiPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gPchTraceHubPreMemConfigGuid, (VOID *) &PchTraceHubConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuTraceHubPreMemConfigGuid, (VOID *) &CpuTraceHubConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gSaMiscPeiPreMemConfigGuid, (VOID *) &MiscPeiPreMemConfig);
  ASSERT_EFI_ERROR (Status);


  //
  // Default Cachable attribute will be set to WB to support large memory size/hot plug memory
  //
  MtrrSetting.MtrrDefType &= ~((UINT64)(0xFF));
  MtrrSetting.MtrrDefType |= (UINT64) CacheWriteBack;

  //
  // Set fixed cache for memory range below 1MB
  //
  Status = MtrrSetMemoryAttributeInMtrrSettings (
             &MtrrSetting,
             0x0,
             0xA0000,
             CacheWriteBack
             );

  ASSERT_EFI_ERROR (Status);

  Status = MtrrSetMemoryAttributeInMtrrSettings (
             &MtrrSetting,
             0xA0000,
             0x20000,
             CacheUncacheable
             );
  ASSERT_EFI_ERROR (Status);

  Status = MtrrSetMemoryAttributeInMtrrSettings (
             &MtrrSetting,
             0xC0000,
             0x40000,
             CacheWriteProtected
             );
  ASSERT_EFI_ERROR ( Status);
#endif

  //
  // PI SMM IPL can't set SMRAM to WB because at that time CPU ARCH protocol is not available.
  // Set cacheability of SMRAM to WB here to improve SMRAM initialization performance.
  //
  SmramSize = 0;
  SmramBase = 0;
  Status = PeiServicesGetHobList ((VOID **) &Hob.Raw);
  while (!END_OF_HOB_LIST (Hob)) {
    if (Hob.Header->HobType == EFI_HOB_TYPE_GUID_EXTENSION) {
      if (CompareGuid (&Hob.Guid->Name, &gEfiSmmSmramMemoryGuid)) {
        SmramHobDescriptorBlock = (EFI_SMRAM_HOB_DESCRIPTOR_BLOCK *) (Hob.Guid + 1);
        for (Index = 0; Index < SmramHobDescriptorBlock->NumberOfSmmReservedRegions; Index++) {
          if (SmramHobDescriptorBlock->Descriptor[Index].PhysicalStart > 0x100000) {
            SmramSize += SmramHobDescriptorBlock->Descriptor[Index].PhysicalSize;
            if (SmramBase == 0 || SmramBase > SmramHobDescriptorBlock->Descriptor[Index].CpuStart) {
              SmramBase = SmramHobDescriptorBlock->Descriptor[Index].CpuStart;
            }
          }
        }
        break;
      }
    }
    Hob.Raw = GET_NEXT_HOB (Hob);
  }

#if FixedPcdGetBool(PcdFspWrapperEnable) == 1
  MtrrGetAllMtrrs (&MtrrSetting);
  VariableMtrrCount = GetVariableMtrrCount ();
  MtrrLibInitializeMtrrMask (&MtrrValidBitsMask, &MtrrValidAddressMask);

  Ranges[0].BaseAddress = 0;
  Ranges[0].Length      = MtrrValidBitsMask + 1;
  Ranges[0].Type        = (MTRR_MEMORY_CACHE_TYPE)(MtrrSetting.MtrrDefType & 0x07); //[Bits 2:0] Default Memory Type.
  RangeCount = 1;

  MtrrLibGetRawVariableRanges (
    &MtrrSetting.Variables, VariableMtrrCount,
    MtrrValidBitsMask, MtrrValidAddressMask, RawVariableRanges
    );

  MtrrLibApplyVariableMtrrs (
    RawVariableRanges, VariableMtrrCount,
    Ranges, ARRAY_SIZE (Ranges), &RangeCount
    );

  //
  // Set SMRAM as CacheWriteBack for performance.
  //
  Ranges[RangeCount].BaseAddress = SmramBase;
  Ranges[RangeCount].Length      = SmramSize;
  Ranges[RangeCount].Type        = CacheWriteBack;
  RangeCount++;

  ZeroMem (&UCMtrrSetting, sizeof (MTRR_SETTINGS));
  UCMtrrSetting.MtrrDefType = MtrrSetting.MtrrDefType;
  Status = EOPSetMemoryAttributesInMtrrSettings (&UCMtrrSetting, Ranges, RangeCount);

  CopyMem (&MtrrSetting.Variables, &UCMtrrSetting.Variables, sizeof (MTRR_VARIABLE_SETTINGS));
  MtrrSetting.MtrrDefType = UCMtrrSetting.MtrrDefType;
#else
  //
  // Set non system memory as UC
  //
  MemoryBase = 0x100000000;

  MemoryLength = MemoryBase - (SmramBase + SmramSize);
  //
  // Check if both the lowest bit and left of the lowest bit of MemoryLength are set
  // AND the size of lowest position is less than 32MB
  // then add the size of lowest position to MemoryLength for reducing the MTRR usage.
  //
  DEBUG ((DEBUG_INFO, "MemoryBase = %lx\n", MemoryBase));
  DEBUG ((DEBUG_INFO, "MemoryLength = %x\n", MemoryLength));
  if ((LShiftU64 (1, (UINTN) LowBitSet64 (MemoryLength))) <= SIZE_32MB) {
    if ((MemoryLength & (LShiftU64 (1, (UINTN) LowBitSet64 (MemoryLength) + 1))) != 0) {
      ExtraMemoryLength = (LShiftU64 (1, (UINTN) LowBitSet64 (MemoryLength)));
      MemoryLength += ExtraMemoryLength;
      DEBUG ((DEBUG_INFO, "Adjusted MemoryLength = %x\n", MemoryLength));

      BuildResourceDescriptorHob (
        EFI_RESOURCE_MEMORY_RESERVED,
        EFI_RESOURCE_ATTRIBUTE_PRESENT | EFI_RESOURCE_ATTRIBUTE_INITIALIZED | EFI_RESOURCE_ATTRIBUTE_UNCACHEABLE,
        MemoryBase - MemoryLength,
        ExtraMemoryLength
      );
    }
  }

  while (MemoryLength != 0) {
    Power2Length = GetPowerOfTwo64 (MemoryLength);
    MemoryBase -= Power2Length;
    Status = MtrrSetMemoryAttributeInMtrrSettings (
               &MtrrSetting,
               MemoryBase,
               Power2Length,
               CacheUncacheable
               );
    ASSERT_EFI_ERROR (Status);
    MemoryLength -= Power2Length;
  }

  //
  // Get Trace Hub memory base and size
  //
  TraceHubMemSize = TraceHubCalculateTotalBufferSize (
                      (UINT8) SiPreMemConfig->PlatformDebugConsent,
                      (UINT8) CpuTraceHubConfig->TraceHub.EnableMode,
                      (UINT8) CpuTraceHubConfig->TraceHub.MemReg0Size,
                      (UINT8) CpuTraceHubConfig->TraceHub.MemReg1Size,
                      (UINT8) PchTraceHubConfig->TraceHub.EnableMode,
                      (UINT8) PchTraceHubConfig->TraceHub.MemReg0Size,
                      (UINT8) PchTraceHubConfig->TraceHub.MemReg1Size
                      );

  if (TraceHubMemSize > 0) {
    DEBUG ((DEBUG_INFO, "PlatformInit Set MTRR UC for Trace hub\n"));
    DEBUG ((DEBUG_INFO, "Trace Hub Mem Size = 0x%08x\n", TraceHubMemSize));
    DEBUG ((DEBUG_INFO, "Trace Hub Mem Base = 0x%08x\n", SiConfig->TraceHubMemBase));

    Status = MtrrSetMemoryAttributeInMtrrSettings (
               &MtrrSetting,
               SiConfig->TraceHubMemBase,
               TraceHubMemSize,
               CacheUncacheable
               );
    ASSERT_EFI_ERROR (Status);
  }

  //
  // Update MTRR setting from MTRR buffer
  //
  MtrrSetAllMtrrs (&MtrrSetting);
#endif
  return Status;
}

/**
  This function sends IOM Ready Notify message to EC

  @retval  EFI_SUCCESS   The function completed successfully
  @retval  EFI_NOT_READY IomReady bit is not set
  @retval  others        Some error occurred
**/
EFI_STATUS
IOMReadyNotifyToEC (
  VOID
)
{
  EFI_STATUS                       Status;
#if FixedPcdGetBool(PcdITbtEnable) == 1
  SA_SETUP                         SaSetup;
  SETUP_DATA                       SystemConfiguration;
  UINTN                            VariableSize;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI  *VariableServices;
  UINT8                            EcDataBuffer;
  UINT8                            ReplayMsgBitMask;
  UINT8                            VccstBitMask;
  EFI_BOOT_MODE                    BootMode;
  TCSS_DATA_HOB                    *TcssHob;

  TcssHob = NULL;
  VccstBitMask = 0;
  ///
  /// Locate HOB for TCSS Data and read IOM ready state
  ///
  TcssHob = (TCSS_DATA_HOB *) GetFirstGuidHob (&gTcssHobGuid);
  if (TcssHob == NULL) {
    return EFI_NOT_READY;
  }
  if (!TcssHob->TcssData.IOMReady) {
     DEBUG ((DEBUG_ERROR, "IOMReadyNotifyToEC: IOMReady bit not set in TcssHob.\n"));
    return EFI_NOT_READY;
  }

  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,  // GUID
             0,                                 // INSTANCE
             NULL,                              // EFI_PEI_PPI_DESCRIPTOR
             (VOID **) &VariableServices        // PPI
             );
  if (EFI_ERROR(Status)) {
    DEBUG ((DEBUG_ERROR, "IOMReadyNotifyToEC: Failed to find PEI Read Variable support. Status = %r\n", Status));
  }
  else {
    VariableSize = sizeof (SETUP_DATA);
    Status = VariableServices->GetVariable (
                                 VariableServices,
                                 PLATFORM_SETUP_VARIABLE_NAME,
                                 &gSetupVariableGuid,
                                 NULL,
                                 &VariableSize,
                                 &SystemConfiguration
                                 );
    ASSERT_EFI_ERROR (Status);
    //
    // Send IOMReadyNotify only if EC Handshake is enabled
    //
    if ((!EFI_ERROR(Status)) && (SystemConfiguration.UsbcBiosTcssHandshake == 1) && (!PcdGetBool(PcdBoardPmcPdEnable))) {
      PeiServicesGetBootMode (&BootMode);
      VariableSize = sizeof (SA_SETUP);
      Status = VariableServices->GetVariable (
                                   VariableServices,
                                   L"SaSetup",
                                   &gSetupVariableGuid,
                                   NULL,
                                   &VariableSize,
                                   &SaSetup
                                   );
      // @Todo: add other condition (boot mode) checks here.

      EcDataBuffer = 0x00;
      ReplayMsgBitMask = EC_TCSS_BITMASK_REPLAY_MESSAGES;

      if (!EFI_ERROR (Status)) {
        VccstBitMask = SaSetup.TcssVccstStatus;
        //
        // S4 and S5 flow
        //
        if ((BootMode == BOOT_ON_S4_RESUME) || (BootMode == BOOT_ON_S5_RESUME)) {
          if (SaSetup.TcssVccstStatus == 1) {
            //
            // Clear ReplayMsgBitMask after checking VccstStatus policy.
            //
            ReplayMsgBitMask = 0x00;
          }
        } else if (BootMode == BOOT_ON_S3_RESUME) {
          //
          // For S3 resume, clear replay bit always as Platform EC doesn't support replay in S3
          //
          ReplayMsgBitMask = 0x00;
        }
      }
      EcDataBuffer = ReplayMsgBitMask | VccstBitMask;
      Status = UsbcIomReadyNotify(&EcDataBuffer);
      // @Todo: Add error condition check if Status is not success
    }
  }
#else
  Status = EFI_SUCCESS;
#endif

  return Status;
}

/**
  Checks if Premium PMIC present (VendorID == 1Fh)

  @retval  TRUE  if present
  @retval  FALSE it discrete/other PMIC
**/
BOOLEAN
IsPremiumPmicPresent (
  VOID
  )
{
  UINT8                           PmicVendorID;

  PmicVendorID = 0;
  //Send KSC Command to detect vendor ID of PMIC is present on the system(Applicable only for ULT/ULX Platforms)
  DetectPmicVendorID (&PmicVendorID);
  DEBUG((DEBUG_INFO, "Vendor ID of the Pmic Present on the system is: %x\n", PmicVendorID));

  if (PmicVendorID == 0x1F) {
    return TRUE;
  }

  return FALSE;
}

/**
Pmic Programming to Enable Voltage Margining
**/
VOID
PremiumPmicEnableSlpS0Voltage (
  VOID
  )
{
  EFI_STATUS                      Status;
  UINT8                           EcDataV085ACNT;
  UINT64                          HdaPciBase;
  BOOLEAN                         PremiumPmicPresent;

  HdaPciBase = HdaPciCfgBase ();
  PremiumPmicPresent = IsPremiumPmicPresent();
  if ((PciSegmentRead16(HdaPciBase + PCI_VENDOR_ID_OFFSET) == 0xFFFF) && (PremiumPmicPresent == TRUE)) { // If HDA Device is not detected & Premium PMIC is present
    DEBUG((DEBUG_INFO, "Enable VM in case Premium PMIC is Detected and HDA Disabled\n"));
    EcDataV085ACNT = 0x7A; //Enable Voltage Margining in case HDA is disabled
    Status = SetSlpS0Voltage(EcDataV085ACNT);
    ASSERT_EFI_ERROR(Status);
  }
}

/**
  Configure PciHostBridge related PCDs
**/
VOID
ConfigurePciHostBridgePcds (
  VOID
  )
{
  EFI_STATUS                       Status;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI  *VariableServices;
  EFI_PHYSICAL_ADDRESS             PciBaseAddress;
  UINT32                           Tolud;
  UINT64                           Length;
  UINT64                           McD0BaseAddress;
  UINTN                            ResMemLimit1;
  UINTN                            SaSetupSize;
  SA_SETUP                         SaSetup;
#if FixedPcdGetBool(PcdITbtEnable) == 1
  TCSS_DATA_HOB                    *TcssHob;

  TcssHob                   = NULL;
#endif
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,  // GUID
             0,                                 // INSTANCE
             NULL,                              // EFI_PEI_PPI_DESCRIPTOR
             (VOID **) &VariableServices        // PPI
             );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "ConfigurePciHostBridgePcds: PeiServicesLocatePpi failed\n"));
    return;
  }
#if FixedPcdGetBool(PcdITbtEnable) == 1
  //
  // Get Tcss Data HOB
  //
  TcssHob = (TCSS_DATA_HOB *) GetFirstGuidHob (&gTcssHobGuid);
  if (TcssHob != NULL) {
    if (TcssHob->TcssData.PcieMultipleSegmentEnabled) {
      PcdSet8 (PcdPciSegmentCount, 2);
    }
  }
#endif
  //
  // Allocate 56 KB of I/O space [0x2000..0xFFFF]
  //
  DEBUG ((DEBUG_INFO, " Assign IO resource for PCI_ROOT_BRIDGE from 0x%X to 0x%X\n", PcdGet16 (PcdPciReservedIobase) ,PcdGet16 (PcdPciReservedIoLimit)));

  //
  // Read memory map registers
  //
  McD0BaseAddress        = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, 0);
  Tolud                  = PciSegmentRead32 (McD0BaseAddress + R_SA_TOLUD) & B_SA_TOLUD_TOLUD_MASK;
  PciBaseAddress         = Tolud;

  ResMemLimit1 = PcdGet32 (PcdPciReservedMemLimit);
  if (ResMemLimit1 == 0) {
    ResMemLimit1 = (UINTN) PcdGet64 (PcdPciExpressBaseAddress);
  }

  Length = ResMemLimit1 - PciBaseAddress;

  if (Length != 0) {
    PcdSet32 (PcdPciReservedMemBase, (UINT32) PciBaseAddress);
    PcdSet32 (PcdPciReservedMemLimit, (UINT32) (PciBaseAddress + Length - 1));
    DEBUG ((DEBUG_INFO, " Assign Memory Resource for PCI_ROOT_BRIDGE from 0x%X", PcdGet32 (PcdPciReservedMemBase)));
    DEBUG ((DEBUG_INFO, " to 0x%X\n", PcdGet32 (PcdPciReservedMemLimit)));
  }

  //
  // Check Enable Above 4GB MMIO or not
  //
  SaSetupSize = sizeof (SA_SETUP);
  Status = VariableServices->GetVariable (
                                VariableServices,
                                L"SaSetup",
                                &gSaSetupVariableGuid,
                                NULL,
                                &SaSetupSize,
                                &SaSetup
                                );
  if (!EFI_ERROR(Status)) {
    if (SaSetup.EnableAbove4GBMmio == 1 || SaSetup.ApertureSize == 15) {
      //
      // Provide 256GB available above 4GB MMIO resource
      // limited to use single variable MTRR to cover this above 4GB MMIO region.
      //
      PcdSet64 (PcdPciReservedMemAbove4GBBase, BASE_256GB);
      PcdSet64 (PcdPciReservedMemAbove4GBLimit, BASE_256GB + SIZE_256GB - 1);
      if (PcdGet64 (PcdPciReservedMemAbove4GBBase) < PcdGet64 (PcdPciReservedMemAbove4GBLimit)) {
        DEBUG ((DEBUG_INFO, " PCI space that above 4GB MMIO is from 0x%lX", PcdGet64 (PcdPciReservedMemAbove4GBBase)));
        DEBUG ((DEBUG_INFO, " to 0x%lX\n", PcdGet64 (PcdPciReservedMemAbove4GBLimit)));
      }
    }
  }
}

/**
  This function handles PlatformInit task at the end of PEI

  @param[in]  PeiServices  Pointer to PEI Services Table.
  @param[in]  NotifyDesc   Pointer to the descriptor for the Notification event that
                           caused this function to execute.
  @param[in]  Ppi          Pointer to the PPI data associated with this function.

  @retval     EFI_SUCCESS  The function completes successfully
  @retval     others
**/
EFI_STATUS
EFIAPI
PlatformInitAdvancedEndOfPei (
  IN CONST EFI_PEI_SERVICES     **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR  *NotifyDescriptor,
  IN VOID                       *Ppi
  )
{
  EFI_STATUS     Status;
  Status = SetCacheMtrrForTraceHubAfterEndOfPei ();
  ASSERT_EFI_ERROR (Status);

  //
  // Send IOM Ready Notify message to EC
  //
  Status = IOMReadyNotifyToEC ();

  //
  // Function to enable VM in case HD Audio is Disabled
  //
  PremiumPmicEnableSlpS0Voltage ();

  //
  // Configure PciHostBridge related PCDs before DXE phase
  //
  ConfigurePciHostBridgePcds ();

  return Status;
}

//@todo Review this functionality and if it is required for SKL SDS
/**
  Create the HOB for hotkey status for 'Attempt USB First' feature

  @retval  EFI_SUCCESS  HOB Creating successful.
  @retval  Others       HOB Creating failed.
**/
EFI_STATUS
CreateAttemptUsbFirstHotkeyInfoHob (
  VOID
  )
{
  EFI_STATUS                     Status;
  ATTEMPT_USB_FIRST_HOTKEY_INFO  AttemptUsbFirstHotkeyInfo;

  Status = EFI_SUCCESS;

  ZeroMem (
    &AttemptUsbFirstHotkeyInfo,
    sizeof (AttemptUsbFirstHotkeyInfo)
    );

  AttemptUsbFirstHotkeyInfo.RevisonId = 0;
  AttemptUsbFirstHotkeyInfo.HotkeyTriggered = FALSE;

  ///
  /// Build HOB for Attempt USB First feature
  ///
  BuildGuidDataHob (
    &gAttemptUsbFirstHotkeyInfoHobGuid,
    &(AttemptUsbFirstHotkeyInfo),
    sizeof (ATTEMPT_USB_FIRST_HOTKEY_INFO)
    );

  return Status;
}

/**
  Get address of PMC crash log record from descriptor table

  @param[in]  DiscoveryBuffer    PMC IPC discover buffer

  @retval     UINT32             An address of PMC crash log record.
**/
UINT32
GetPmcCrashLogDescriptTblAddr (
  IN PMC_IPC_DISCOVERY_BUF       *DiscoveryBuffer
  )
{
  return PcdGet32 (PcdSiliconInitTempMemBaseAddr) + DiscoveryBuffer->Bits64.DesTableOffset;
}

/**
  Get how many PMC crash log records.

  @param[in]  DiscoveryBuffer    PMC IPC discover buffer

  @retval     UINT32             Amounts of PMC crash log.
**/
UINT32
GetPmcCrashLogAmount (
  IN PMC_IPC_DISCOVERY_BUF       *DiscoveryBuffer
  )
{
  UINT32                         NumberOfRegionsAddress;

  switch (DiscoveryBuffer->Bits64.Mech) {
    case CRASHLOG_MECH_LEGACY:
      return 1;
    case CRASHLOG_MECH_DESCRIPTOR_TABLE:
      NumberOfRegionsAddress = GetPmcCrashLogDescriptTblAddr (DiscoveryBuffer);
      return MmioRead32 (NumberOfRegionsAddress);
    default:
      return 0;
  }
}

/**
  Get offset and log size from PMC crash records.

  @param[in]  DiscoveryBuffer    PMC IPC discover buffer
  @param[in]  Offset             PMC records of region offset

  @retval     UINT32             PMC crash record
**/
UINT32
GetPmcCrashLogRecord (
  IN PMC_IPC_DISCOVERY_BUF       *DiscoveryBuffer,
  IN UINT32                      Offset
  )
{
  PMC_CRASHLOG_RECORDS     Record;
  UINT32                   PmcCrashRecordAddr;

  switch (DiscoveryBuffer->Bits64.Mech) {
    case CRASHLOG_MECH_LEGACY:
      //
      // Legacy Mechanism uses Baseoffset as PmcCrashLogAddr.
      // The BaseOffset is returned from PMC discovery buffer.
      //
      Record.Info.Offset = DiscoveryBuffer->Bits.BaseOffset;
      Record.Info.Size = (DiscoveryBuffer->Bits.Size != 0) ? DiscoveryBuffer->Bits.Size : 0x300;
      break;
    case CRASHLOG_MECH_DESCRIPTOR_TABLE:
      //   Descritpor Table Mechanism
      //---------------------------------
      // Number of records  |   4 bytes
      //  1 record offset   |   2 bytes
      //  1 record size     |   2 bytes
      //  2 record offset   |   2 bytes
      //  2 record size     |   2 bytes
      //---------------------------------
      PmcCrashRecordAddr = GetPmcCrashLogDescriptTblAddr (DiscoveryBuffer) + (4 * Offset);
      Record.Uint32 = MmioRead32 (PmcCrashRecordAddr);
      break;
    default:
      return 0;
  }

  //
  // PMC crash log size is number of DWORDs.
  //
  Record.Info.Size *= sizeof (UINT32);
  return Record.Uint32;
}

/**
  Consider the PMC crash log data is valid or invalid.

  @param[in]  PmcCrashLogAddr    Crash log data of memory address

  @retval     TRUE               Data is valid
  @retval     FALSE              Data is invalid
**/
BOOLEAN
GetPmcValidRecord (
  IN UINT32                      PmcCrashLogAddr
  )
{
  CRASHLOG_VERSION *PmcCrashLogVersion;

  PmcCrashLogVersion = (CRASHLOG_VERSION *) PmcCrashLogAddr;

  if (PmcCrashLogVersion->Uint32 == 0) {
    return FALSE;
  }

  if (PmcCrashLogVersion->Bits.Consumed == 1) {
    return FALSE;
  }

  return TRUE;
}

/**
  Get PMC crash log supported or not.

  @param[in]  DiscoveryBuffer    PMC IPC discover buffer

  @retval     TRUE               PMC crash log support
  @retval     FALSE              PMC crash log not support
**/
BOOLEAN
GetPmcCrashLogSupport (
  IN PMC_IPC_DISCOVERY_BUF       *DiscoveryBuffer
  )
{

  if (DiscoveryBuffer->Bits.Avail != 1) {
    return FALSE;
  }

  switch (DiscoveryBuffer->Bits64.Mech) {
    case CRASHLOG_MECH_LEGACY:
      if (DiscoveryBuffer->Bits.Dis == 1) {
        return FALSE;
      }
      break;
    case CRASHLOG_MECH_DESCRIPTOR_TABLE:
      if (DiscoveryBuffer->Bits64.CrashDisSts) {
        return FALSE;
      }
      break;
    default:
      return FALSE;
  }

  return TRUE;
}

/**
  Get re-arm supported or not.

  @param[in]  DiscoveryBuffer       PMC IPC discover buffer

  @retval     TRUE                  PMC Re-arm command support
  @retval     FALSE                 PMC Re-arm command not support
**/
BOOLEAN
GetPmcReArmSupport (
  IN PMC_IPC_DISCOVERY_BUF          *DiscoveryBuffer
  )
{
  if (DiscoveryBuffer->Bits64.Mech == CRASHLOG_MECH_DESCRIPTOR_TABLE) {
    return (BOOLEAN)(DiscoveryBuffer->Bits64.ReArm);
  }

  return FALSE;
}

/*
  Get pmc crash log clear enable or not.

  @param[in]  DiscoveryBuffer       PMC IPC discover buffer

  @retval     TRUE                  PMC crash log clear
  @retval     FALSE                 PMC crash log do not clear
*/
BOOLEAN
GetPmcClearSupport (
  IN PMC_IPC_DISCOVERY_BUF          *DiscoveryBuffer
  )
{
  EFI_STATUS                        Status;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI   *VariableServices;
  SETUP_DATA                        SetupData;
  UINTN                             VariableSize;

  if (DiscoveryBuffer->Bits64.Mech == CRASHLOG_MECH_DESCRIPTOR_TABLE) {
    if (DiscoveryBuffer->Bits64.Clr == 0) {
      return FALSE;
    }
  }

  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,  // GUID
             0,                                 // INSTANCE
             NULL,                              // EFI_PEI_PPI_DESCRIPTOR
             (VOID **) &VariableServices        // PPI
             );
  ASSERT_EFI_ERROR (Status);

  VariableSize = sizeof (SETUP_DATA);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               PLATFORM_SETUP_VARIABLE_NAME,
                               &gSetupVariableGuid,
                               NULL,
                               &VariableSize,
                               &SetupData
                               );

  return (BOOLEAN)(SetupData.CrashLogClearEnable);
}

/**
  Collect PCH CrashLog data from Pmc SSRAM and store in HOB .

**/
VOID
CrashLogCollectDataFromPmcSSRAM (
  VOID
  )
{
  EFI_STATUS                  Status;
  PMC_IPC_DISCOVERY_BUF       DiscoveryBuffer;
  UINT64                      PmcSsramBaseAddress;
  UINT32                      PmcSsramBar0;
  PMC_CRASHLOG_LINK           *PmcCrashLog;
  PMC_CRASHLOG_LINK           *PmcCrashLogList;
  PMC_CRASHLOG_LINK           *TempPmcCrashLog;
  CRASHLOG_HOB                *PmcCrashLogHob;
  UINT32                      Index;
  UINT32                      NumberOfRegions = 0;
  UINT32                      ValidRecords;
  UINT32                      PmcCrashLogAddr;
  UINT32                      *Destination = NULL;
  UINT32                      CopiedSize = 0;

  DEBUG ((DEBUG_INFO, "CrashLogCollectDataFromPmcSSRAM - start\n"));

  //
  // Check for the availability of CrashLog feature
  //
  ZeroMem (&DiscoveryBuffer, sizeof (PMC_IPC_DISCOVERY_BUF));

  Status = PmcCrashLogDiscovery (&DiscoveryBuffer);

  if (EFI_ERROR (Status) || (GetPmcCrashLogSupport (&DiscoveryBuffer) == FALSE)) {
    DEBUG ((DEBUG_INFO, "PCH CrashLog feature is not supported\n"));
    return;
  }

  //
  // Start to access PMC SSRAM MMIO
  //
  PmcSsramBaseAddress = PmcSsramPciCfgBase ();

  if (PciSegmentRead16 (PmcSsramBaseAddress + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
    DEBUG ((DEBUG_ERROR, "PMC SSRAM PCI device is disabled\n"));
    return;
  }

  //
  // Program BAR 0 and enable command register memory space decoding
  //
  PmcSsramBar0 = PcdGet32 (PcdSiliconInitTempMemBaseAddr);
  PciSegmentWrite32 (PmcSsramBaseAddress + PCI_BASE_ADDRESSREG_OFFSET, PmcSsramBar0);
  PciSegmentOr16 (PmcSsramBaseAddress + PCI_COMMAND_OFFSET, (UINT16) (EFI_PCI_COMMAND_MEMORY_SPACE));

  NumberOfRegions = GetPmcCrashLogAmount (&DiscoveryBuffer);
  PmcCrashLogList = NULL;

  for (Index = 1, ValidRecords = 0; Index <= NumberOfRegions; Index++) {
    PmcCrashLog = AllocateZeroPool (sizeof (PMC_CRASHLOG_LINK));
    if (PmcCrashLog == NULL) {
      DEBUG ((DEBUG_ERROR, "PmcCrashLogHob : buffer allocation failure\n"));
      ASSERT (FALSE);
      return;
    } else {
      PmcCrashLog->Record.Uint32 = GetPmcCrashLogRecord (&DiscoveryBuffer, Index);
      DEBUG ((DEBUG_INFO, "%d.Crashlog record  = 0x%08x\n", Index, PmcCrashLog->Record.Uint32));
      //
      // Check all records' data are valid or invalid, if not, skip the record.
      //
      PmcCrashLogAddr = PmcSsramBar0 + PmcCrashLog->Record.Info.Offset;
      DEBUG ((DEBUG_INFO, "%d.MmioRead32 (0x%08x) = 0x%08x\n", Index, PmcCrashLogAddr, MmioRead32 (PmcCrashLogAddr)));
      if (GetPmcValidRecord (PmcCrashLogAddr) == FALSE) {
        //
        // Invalid record. Skip this record and check the next.
        //
        FreePool (PmcCrashLog);
        continue;
      }

      //
      // Valid record, prepare memory space for this address of record.
      //
      Status = PeiServicesAllocatePages (
              EfiBootServicesData,
              EFI_SIZE_TO_PAGES (PmcCrashLog->Record.Info.Size),
              &(PmcCrashLog->AllocateAddress)
              );

      if (EFI_ERROR (Status)) {
        DEBUG ((DEBUG_ERROR, "Memory out of resource\n"));
        goto Exit;
      }

      ValidRecords++;
      PmcCrashLog->Next = NULL;
      TempPmcCrashLog = PmcCrashLogList;

      //
      // Link every valid records to PmcCrashLogList
      //
      if (PmcCrashLogList == NULL) {
        PmcCrashLogList = PmcCrashLog;
      } else {
        while (TempPmcCrashLog->Next != NULL) {
          TempPmcCrashLog = TempPmcCrashLog->Next;
        }
        TempPmcCrashLog->Next = PmcCrashLog;
      }
    }
  }

  //
  // If none valid records, exit.
  //
  if (ValidRecords == 0) {
    goto Exit;
  }

  //
  // Build of location for Crash log in Hob
  //
  PmcCrashLogHob = AllocateZeroPool (sizeof (CRASHLOG_HOB) * ValidRecords);
  if (PmcCrashLogHob == NULL) {
    DEBUG ((DEBUG_ERROR, "PmcCrashLogHob : buffer allocation failure\n"));
    ASSERT (FALSE);
    return;
  } else {
    PmcCrashLogHob = (CRASHLOG_HOB *) BuildGuidHob (
                                        &gPmcCrashLogDataBufferHobGuid,
                                        sizeof (CRASHLOG_HOB) * ValidRecords
                                        );
    DEBUG ((DEBUG_INFO, "CrashLogDataBuffer = 0x%08x\n", PmcCrashLogHob));

    for (Index = 0, TempPmcCrashLog = PmcCrashLogList; TempPmcCrashLog != NULL; Index++) {
      //
      // Fill allocate memory address and record size to Hob.
      //
      PmcCrashLogHob[Index].AllocateAddress = TempPmcCrashLog->AllocateAddress;
      PmcCrashLogHob[Index].Size = TempPmcCrashLog->Record.Info.Size;
      DEBUG ((DEBUG_INFO, "%d.CrashLogHob->AllocateAddress = 0x%08x\n", Index, PmcCrashLogHob[Index].AllocateAddress));
      DEBUG ((DEBUG_INFO, "%d.CrashLogHob->Size = 0x%x\n", Index, PmcCrashLogHob[Index].Size));

      //
      // Initial pointers for copy crash log data.
      //
      Destination = (UINT32 *)(UINTN) TempPmcCrashLog->AllocateAddress;
      PmcCrashLogAddr = PmcSsramBar0 + TempPmcCrashLog->Record.Info.Offset;
      CopiedSize = 0;

      //
      // Copy CrashLog data from SSRAM to allocated memory buffer
      //
      while (CopiedSize < TempPmcCrashLog->Record.Info.Size) {
        *Destination = MmioRead32 (PmcCrashLogAddr); // Byte access is not allowed to PMC SSRAM, hence copying DW by DW
        Destination++;
        PmcCrashLogAddr += 4;
        CopiedSize += 4;
      }

      //
      // Point to next PmcCrashLog
      //
      TempPmcCrashLog = TempPmcCrashLog->Next;
    }
  }

  if (GetPmcReArmSupport (&DiscoveryBuffer) == TRUE) {
    //
    // Trigger re-arm command.
    //
    Status = PmcCrashLogReArm ();
    DEBUG ((DEBUG_INFO, "Re-arm Status = %r\n", Status));
  }

  if (GetPmcClearSupport (&DiscoveryBuffer) == TRUE) {
    //
    // Clear the SSRAM region after copying the error log
    //
    Status = PmcCrashLogClear ();
    DEBUG ((DEBUG_INFO, "Clear CrashLog Status = %r\n", Status));
  }

Exit:
  //
  // Disable PMC SSRAM MMIO
  //
  PciSegmentAnd16 (PmcSsramBaseAddress + PCI_COMMAND_OFFSET, (UINT16) ~(EFI_PCI_COMMAND_MEMORY_SPACE | EFI_PCI_COMMAND_IO_SPACE));
  PciSegmentWrite32 (PmcSsramBaseAddress + PCI_BASE_ADDRESSREG_OFFSET, 0);
  DEBUG ((DEBUG_INFO, "CrashLogCollectDataFromPmcSSRAM - end\n"));

  return;
}

/**
  Callback to consume DefaultPolicyInit PPI to install and update policy.

  @param[in]  PeiServices  Pointer to PEI Services Table.
  @param[in]  NotifyDesc   Pointer to the descriptor for the Notification event that
                           caused this function to execute.
  @param[in]  Ppi          Pointer to the PPI data associated with this function.

  @retval     EFI_SUCCESS   The function completes successfully
  @retval     EFI_NOT_FOUND Either variable service or required variable not found.
**/
EFI_STATUS
EFIAPI
SiDefaultPolicyInitPpiNotifyCallback (
  IN CONST EFI_PEI_SERVICES     **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR  *NotifyDescriptor,
  IN VOID                       *Ppi
  )
{
  EFI_STATUS                        Status;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI   *VariableServices;
  SETUP_DATA                        SystemConfiguration;
  UINTN                             VariableSize;
  UINT8                             FwConfig;

  VariableServices = NULL;
  Status = PeiServicesLocatePpi (
              &gEfiPeiReadOnlyVariable2PpiGuid,
              0,
              NULL,
              (VOID **) &VariableServices
              );
  if (VariableServices != NULL) {
    VariableSize = sizeof (SETUP_DATA);
    Status = VariableServices->GetVariable (
                                 VariableServices,
                                 L"Setup",
                                 &gSetupVariableGuid,
                                 NULL,
                                 &VariableSize,
                                 &SystemConfiguration
                                 );
    if (!EFI_ERROR (Status)) {
      FwConfig = SystemConfiguration.FirmwareConfiguration;
      PeiPolicyInit (PeiServices, FwConfig);
      return EFI_SUCCESS;
    }
  }
  return EFI_NOT_FOUND;
}

static EFI_PEI_NOTIFY_DESCRIPTOR mSiDefaultPolicyInitNotifyList = {
  (EFI_PEI_PPI_DESCRIPTOR_NOTIFY_CALLBACK | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gSiDefaultPolicyInitPpiGuid,
  (EFI_PEIM_NOTIFY_ENTRY_POINT) SiDefaultPolicyInitPpiNotifyCallback
};

/**
  Collect CPU CrashLog data from Telemetry SRAM and store in HOB .

**/
VOID
CpuCrashLogCollectDataFromTelemetrySRAM (
  VOID
  )
{
  EFI_STATUS                  Status;
  TEL_CRASHLOG_DEVSC_CAP      CrashLogDevscCap;
  UINT64                      TelemetryPciDeviceBaseAddr;
  UINT32                      TempBarAddr;
  EFI_PHYSICAL_ADDRESS        MainLogAllocateAddress;
  EFI_PHYSICAL_ADDRESS        TelemetryAllocateAddress;
  EFI_PHYSICAL_ADDRESS        TraceAllocateAddress;
  UINT32                      MainLogSize = 0;
  UINT32                      TelemetrySize = 0;
  UINT32                      TraceSize = 0;
  UINT32                      CpuCrashLogAddr;
  CPU_CRASHLOG_STRUCT         CpuCrashLogStruct;
  CPU_CRASHLOG_HOB            CpuCrashLogHob;
  UINT32                      *Destination = NULL;
  UINT32                      CopiedSize = 0;

  DEBUG ((DEBUG_INFO, "CpuCrashLogCollectDataFromTelemetrySRAM - start\n"));

  //
  // Check for the availability of CPU CrashLog feature
  //
  ZeroMem (&CrashLogDevscCap, sizeof (TEL_CRASHLOG_DEVSC_CAP));

  Status = GetCpuCrashLogCapability (&CrashLogDevscCap);

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_INFO, "CPU CrashLog feature is not supported\n"));
    return;
  }

  //
  // Start to access Telemetry MMIO
  //
  TelemetryPciDeviceBaseAddr = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, TEL_BUS_NUM, TEL_DEV_NUM, TEL_FUN_NUM, 0);
  DEBUG ((DEBUG_INFO, "TelemetryPciDeviceBaseAddr = 0x%X\n", TelemetryPciDeviceBaseAddr));

  //
  // Program BAR address and enable command register memory space decoding
  //
  TempBarAddr = PcdGet32 (PcdSiliconInitTempMemBaseAddr);
  DEBUG ((DEBUG_INFO, "TempBarAddr = 0x%X\n", TempBarAddr));
  if (CrashLogDevscCap.DiscoveryData.Fields.TBir == V_TEL_DVSEC_TBIR_BAR0) {
    PciSegmentWrite32 (TelemetryPciDeviceBaseAddr + R_TEL_CFG_BAR0, TempBarAddr);
  } else if (CrashLogDevscCap.DiscoveryData.Fields.TBir == V_TEL_DVSEC_TBIR_BAR1) {
    PciSegmentWrite32 (TelemetryPciDeviceBaseAddr + R_TEL_CFG_BAR1, TempBarAddr);
  }
  PciSegmentOr16 (TelemetryPciDeviceBaseAddr + PCI_COMMAND_OFFSET, (UINT16) (EFI_PCI_COMMAND_MEMORY_SPACE));

  ZeroMem (&CpuCrashLogStruct, sizeof (CPU_CRASHLOG_STRUCT));

  Status = CpuCrashLogDiscovery (&CpuCrashLogStruct);

  MainLogSize = CpuCrashLogStruct.MainBuffer.Fields.DataBuffSize * 4;
  TelemetrySize = CpuCrashLogStruct.TelemetryBuffer.Fields.DataBuffSize * 4;
  TraceSize = CpuCrashLogStruct.TraceBuffer.Fields.DataBuffSize * 4;

  DEBUG ((DEBUG_INFO, "Use MainLog Size as 0x%X \n", MainLogSize));
  DEBUG ((DEBUG_INFO, "Use Telemetry Size as 0x%X \n", TelemetrySize));
  DEBUG ((DEBUG_INFO, "Use Trace Size as 0x%X \n", TraceSize));

  if (MainLogSize == 0) {
    DEBUG ((DEBUG_INFO, "MainLog is not present \n"));
    goto Exit;
  }

  if (TelemetrySize == 0) {
    DEBUG ((DEBUG_INFO, "TelemetryLog is not present \n"));
  }

  if (TraceSize == 0) {
    DEBUG ((DEBUG_INFO, "TraceLog is not present \n"));
  }

  //
  // Allocate memory buffer for MainLog data
  //
  Status = PeiServicesAllocatePages (
             EfiBootServicesData,
             EFI_SIZE_TO_PAGES (MainLogSize),
             &MainLogAllocateAddress
             );
  DEBUG ((DEBUG_INFO, "MainLogAllocateAddress : 0x%X \n", MainLogAllocateAddress));

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Memory out of resource\n"));
    goto Exit;
  }

  //
  // Allocate memory buffer for Telemetry data
  //
  Status = PeiServicesAllocatePages (
             EfiBootServicesData,
             EFI_SIZE_TO_PAGES (TelemetrySize),
             &TelemetryAllocateAddress
             );
  DEBUG ((DEBUG_INFO, "TelemetryAllocateAddress : 0x%X \n", TelemetryAllocateAddress));

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Memory out of resource\n"));
    goto Exit;
  }

  //
  // Allocate memory buffer for Trace data
  //
  Status = PeiServicesAllocatePages (
             EfiBootServicesData,
             EFI_SIZE_TO_PAGES (TraceSize),
             &TraceAllocateAddress
             );
  DEBUG ((DEBUG_INFO, "TraceAllocateAddress : 0x%X \n", TraceAllocateAddress));

  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Memory out of resource\n"));
    goto Exit;
  }


  CpuCrashLogHob.Main.AllocateAddress = (UINTN) MainLogAllocateAddress;
  CpuCrashLogHob.Main.Size = MainLogSize;

  CpuCrashLogHob.Telemetry.AllocateAddress = (UINTN) TelemetryAllocateAddress;
  CpuCrashLogHob.Telemetry.Size = TelemetrySize;

  CpuCrashLogHob.Trace.AllocateAddress = (UINTN) TraceAllocateAddress;
  CpuCrashLogHob.Trace.Size = TraceSize;

  //
  // Build of location for Crash log in Hob
  //
  BuildGuidDataHob (
    &gCpuCrashLogDataBufferHobGuid,
    &CpuCrashLogHob,
    sizeof (CPU_CRASHLOG_HOB)
    );
  DEBUG ((DEBUG_INFO, "CpuCrashLogHob = 0x%X, sizeof (CPU_CRASHLOG_HOB) : 0x%X \n", CpuCrashLogHob, sizeof (CPU_CRASHLOG_HOB)));

  ZeroMem ((VOID *) (UINTN) MainLogAllocateAddress, EFI_SIZE_TO_PAGES (MainLogSize) * EFI_PAGE_SIZE);
  ZeroMem ((VOID *) (UINTN) TelemetryAllocateAddress, EFI_SIZE_TO_PAGES (TelemetrySize) * EFI_PAGE_SIZE);
  ZeroMem ((VOID *) (UINTN) TraceAllocateAddress, EFI_SIZE_TO_PAGES (TraceSize) * EFI_PAGE_SIZE);

  //
  // copy Main pointer crashlog data
  //
//  CopyMem ((UINT8 *)(UINTN) MainLogAllocateAddress, &(TelemetryBaseAddress + CpuCrashLogStruct.MainBuffer.Fields.DataBuffAddress), (CpuCrashLogStruct.MainBuffer.Fields.DataBuffSize) * 4);
  CpuCrashLogAddr = (UINT32) TempBarAddr + CpuCrashLogStruct.MainBuffer.Fields.DataBuffAddress;
  Destination =  (UINT32 *)(UINTN) MainLogAllocateAddress;
  while (CopiedSize < MainLogSize) {
    *Destination = MmioRead32 (CpuCrashLogAddr); // Byte access is not allowed to PMC SSRAM, hence copying DW by DW
    if (CopiedSize < CRASHLOG_SIZE_DEBUG_PURPOSE) { // Dumping only few bytes to help debug
      DEBUG ((DEBUG_INFO, "Main CrashData = 0x%x\n", *Destination));
    }
    Destination++;
    CpuCrashLogAddr += 4;
    CopiedSize   += 4;
  }
  CpuCrashLogAddr = (UINT32) TempBarAddr + CpuCrashLogStruct.MainBuffer.Fields.DataBuffAddress;
  DEBUG ((DEBUG_INFO, "TempBarAddr = 0x%x, CpuCrashLogStruct.MainBuffer.Fields.DataBuffAddress = 0x%x,  MainLogCrashLogAddr = 0x%x\n", TempBarAddr, CpuCrashLogStruct.MainBuffer.Fields.DataBuffAddress, CpuCrashLogAddr));

  //
  // copy Telemetry pointer crashlog data
  //
 // CopyMem ((UINT8 *)(UINTN) TelemetryAllocateAddress, &(TelemetryBaseAddress + CpuCrashLogStruct.TelemetryBuffer.Fields.DataBuffAddress), (CpuCrashLogStruct.TelemetryBuffer.Fields.DataBuffSize) * 4);
 CpuCrashLogAddr = (UINT32) TempBarAddr + CpuCrashLogStruct.TelemetryBuffer.Fields.DataBuffAddress;
 DEBUG ((DEBUG_INFO, "TempBarAddr = 0x%x, TelemetryBuffer.Fields.DataBuffAddress = 0x%x,  TelemetryCrashLogAddr = 0x%x\n", TempBarAddr, CpuCrashLogStruct.TelemetryBuffer.Fields.DataBuffAddress, CpuCrashLogAddr));
 Destination =  (UINT32 *)(UINTN) TelemetryAllocateAddress;
 CopiedSize = 0;
  while (CopiedSize < TelemetrySize) {
    *Destination = MmioRead32 (CpuCrashLogAddr); // Byte access is not allowed to PMC SSRAM, hence copying DW by DW
    if (CopiedSize < CRASHLOG_SIZE_DEBUG_PURPOSE) { // Dumping only few bytes to help debug
      DEBUG ((DEBUG_INFO, "Telemetry CrashData = 0x%x\n", *Destination));
    }
    Destination++;
    CpuCrashLogAddr += 4;
    CopiedSize   += 4;
  }
  //
  // copy Trace pointer crashlog data
  //
//  CopyMem ((UINT8 *)(UINTN) TraceAllocateAddress, &(TelemetryBaseAddress + CpuCrashLogStruct.TraceBuffer.Fields.DataBuffAddress), (CpuCrashLogStruct.TraceBuffer.Fields.DataBuffSize) * 4);
  CpuCrashLogAddr = (UINT32) TempBarAddr + CpuCrashLogStruct.TraceBuffer.Fields.DataBuffAddress;
  DEBUG ((DEBUG_INFO, "TempBarAddr = 0x%x, TraceBuffer.Fields.DataBuffAddress = 0x%x,  TraceCrashLogAddr = 0x%x\n", TempBarAddr, CpuCrashLogStruct.TraceBuffer.Fields.DataBuffAddress, CpuCrashLogAddr));
  Destination =  (UINT32 *)(UINTN) TraceAllocateAddress;
  CopiedSize = 0;
  while (CopiedSize < TraceSize) {
    *Destination = MmioRead32 (CpuCrashLogAddr); // Byte access is not allowed to PMC SSRAM, hence copying DW by DW
    if (CopiedSize < CRASHLOG_SIZE_DEBUG_PURPOSE) { // Dumping only few bytes to help debug
      DEBUG ((DEBUG_INFO, "Trace CrashData = 0x%x\n", *Destination));
    }
    Destination++;
    CpuCrashLogAddr += 4;
    CopiedSize   += 4;
  }
  //
  // Clear the Telemetry SRAM region after copying the error log
  //
  CpuCrashLogClear ();

Exit:
  //
  // Disable Telemetry SRAM MMIO
  //
  PciSegmentAnd16 (TelemetryPciDeviceBaseAddr + PCI_COMMAND_OFFSET, (UINT16) ~(EFI_PCI_COMMAND_MEMORY_SPACE | EFI_PCI_COMMAND_IO_SPACE));
  if (CrashLogDevscCap.DiscoveryData.Fields.TBir == V_TEL_DVSEC_TBIR_BAR0) {
    PciSegmentWrite32 (TelemetryPciDeviceBaseAddr + R_TEL_CFG_BAR0, 0);
  } else if (CrashLogDevscCap.DiscoveryData.Fields.TBir == V_TEL_DVSEC_TBIR_BAR1) {
    PciSegmentWrite32 (TelemetryPciDeviceBaseAddr + R_TEL_CFG_BAR1, 0);
  }
  DEBUG ((DEBUG_INFO, "CpuCrashLogCollectDataFromTelemetrySRAM - end\n"));

  return;
}

/**
  Checks whether the platform is a sample part; or in manufacturing mode or debug mode.

  @retval  TRUE    Platform is a sample part; or in manufacturing mode or debug mode.
  @retval  FALSE   Platfomr is not a sample part; or in manufacturing mode or debug mode.
**/
BOOLEAN
PlatformDebugStateEnabled (
  VOID
  )
{
  HECI_FW_STS6_REGISTER             MeFirmwareStatus;
  MSR_IA32_FEATURE_CONTROL_REGISTER Ia32FeatureControlMsr;
  TGL_MSR_PLATFORM_INFO_REGISTER    TglPlatformInfoMsr;
  MSR_DEBUG_INTERFACE_REGISTER      DebugInterfaceMsr;

  //
  // Check for Sample part
  //
  TglPlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  if (TglPlatformInfoMsr.Bits.SamplePart) {
    return TRUE;
  }

  Ia32FeatureControlMsr.Uint64 = AsmReadMsr64 (MSR_IA32_FEATURE_CONTROL);
  if (Ia32FeatureControlMsr.Bits.Lock == 0) {
    return TRUE;
  }

  //
  // Check for Manufacturing Mode
  //
  MeFirmwareStatus.ul = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (ME_SEGMENT, ME_BUS, ME_DEVICE_NUMBER, HECI_FUNCTION_NUMBER, R_ME_HFS_6));
  if (MeFirmwareStatus.r.FpfSocConfigLock == 0) {
    return TRUE;
  }

  //
  // Check for Debug mode
  //
  DebugInterfaceMsr.Uint64 = AsmReadMsr64 (MSR_IA32_DEBUG_INTERFACE);
  if (DebugInterfaceMsr.Bits.Enable) {
    return TRUE;
  }

  return FALSE;
}

/**
  Configures GPIO

  @param[in]  GpioTable       Point to Platform Gpio table
  @param[in]  GpioTableCount  Number of Gpio table entries

**/
VOID
ConfigureGpio (
  IN GPIO_INIT_CONFIG                 *GpioDefinition,
  IN UINT16                           GpioTableCount
  )
{
  EFI_STATUS          Status;

  DEBUG ((DEBUG_INFO, "ConfigureGpio() Start\n"));

  Status = GpioConfigurePads (GpioTableCount, GpioDefinition);
  ASSERT_EFI_ERROR (Status);

  DEBUG ((DEBUG_INFO, "ConfigureGpio() End\n"));
}

/**
  Configure GPIO pads for TouchPanels use
**/
VOID
TouchPanelGpioInit (
  VOID
  )
{
  EFI_STATUS                      Status;
  PCH_SETUP                       PchSetup;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices;
  UINTN                           VarSize;
  UINT16                          BoardId;

  DEBUG ((DEBUG_INFO, "TouchPanelGpioInit Start\n"));

  //
  // Locate Setup variables
  //
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &VariableServices
             );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "TouchPanelGpioInit: PeiServicesLocatePpi failed\n"));
    return;
  }

  VarSize = sizeof (PCH_SETUP);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"PchSetup",
                               &gPchSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &PchSetup
                               );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "TouchPanelGpioInit: GetVariable (PchSetup) failed\n"));
    return;
  }
  BoardId = PcdGet16(PcdBoardId);

  switch (BoardId) {
    case BoardIdTglUDdr4:
    case BoardIdTglYLp4Type4:
    case BoardIdTglYLp5Type4:
    case BoardIdTglULp4Exs:
    case BoardIdTglULp4Aep:
    case BoardIdTglULp4Dg1Aep:
    case BoardIdTglULp4Gcs:
    default:
      //
      // Verify if THC0 or THC1 panels are enabled before changing GPIO configuration
      //
      if (PchSetup.ThcPort0Assignment == ThcAssignmentNone) {
        DEBUG ((DEBUG_INFO, "THC0 Disabled. Configuring GPIO Touch Panel 1 set for other controller use\n"));
        ConfigureGpio ((VOID *) (UINTN) PcdGet32 (PcdBoardGpioTableTouchPanel1), (UINTN) PcdGet16 (PcdBoardGpioTableTouchPanel1Size));
      }

      if (PchSetup.ThcPort1Assignment == ThcAssignmentNone) {
        DEBUG ((DEBUG_INFO, "THC1 Disabled. Configuring GPIO Touch Panel 2 set for other controller use\n"));
        ConfigureGpio ((VOID *) (UINTN) PcdGet32 (PcdBoardGpioTableTouchPanel2), (UINTN) PcdGet16 (PcdBoardGpioTableTouchPanel2Size));
      }
      break;
  }
  DEBUG ((DEBUG_INFO, "TouchPanelGpioInit End\n"));
}


/**
  Configure GPIO pads for CVF use
**/
VOID
CvfGpioInit (
  VOID
  )
{
  EFI_STATUS                      Status;
  SETUP_DATA                      SetupData;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices;
  UINTN                           VarSize;
  UINT16                          BoardId;

  DEBUG ((DEBUG_INFO, "CvfGpioInit Start\n"));

  //
  // Locate Setup variables
  //
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &VariableServices
             );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "CvfGpioInit: PeiServicesLocatePpi failed\n"));
    return;
  }

  VarSize = sizeof (SETUP_DATA);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"Setup",
                               &gSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &SetupData
                               );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "CvfGpioInit: GetVariable (Setup) failed\n"));
    return;
  }
  BoardId = PcdGet16(PcdBoardId);

  switch (BoardId) {
    case BoardIdTglUDdr4:
    case BoardIdTglYLp4Type4:
    case BoardIdTglYLp5Type4:
    case BoardIdTglULp4Exs:
    case BoardIdTglULp4Aep:
    case BoardIdTglULp4Gcs:
    default:
      //
      // Check if CVF Support is enabled in setup.
      //
      if (SetupData.MipiCam_CvfSupport == 1) {
        DEBUG ((DEBUG_INFO, "Configure CVF gpio\n"));
        ConfigureGpio ((VOID *) (UINTN) PcdGet32 (PcdBoardGpioTableCvf), (UINTN) PcdGet16 (PcdBoardGpioTableCvfSize));
      }
      break;
  }
  DEBUG ((DEBUG_INFO, "CvfGpioInit End\n"));
}

VOID
ConnectivityGpioInit (
  VOID
  )
{
  EFI_STATUS                      Status;
  PCH_SETUP                       PchSetup;
  SETUP_DATA                      SetupData;
  UINTN                           VarSize;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices;

  //
  // Locate Setup variables
  //
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &VariableServices
             );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "ConnectivityGpioInit: PeiServicesLocatePpi failed\n"));
    return;
  }

  VarSize = sizeof (PCH_SETUP);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"PchSetup",
                               &gPchSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &PchSetup
                               );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "ConnectivityGpioInit: GetVariable (PchSetup) failed\n"));
    return;
  }

  VarSize = sizeof (SETUP_DATA);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"Setup",
                               &gSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &SetupData
                               );
  if (EFI_ERROR(Status)) {
    DEBUG((DEBUG_ERROR, "ConnectivityGpioInit: GetVariable (SetupData) failed\n"));
    return;
  }
  if (!CnviIsPresent () || (PchSetup.CnviMode == CnviModeDisabled)) {
    //
    // Discrete BT Module Selection as Disabled, Over USB or UART
    //
    PcdSet8S (PcdDiscreteBtModule, SetupData.DiscreteBtModule);
  }

}

VOID
MipiCamConfigureGpio (
  IN GPIO_CONFIG *GpioConfig
  )
{
  ZeroMem(GpioConfig, sizeof(GPIO_CONFIG));
  GpioConfig->PadMode = GpioPadModeGpio;
  GpioConfig->HostSoftPadOwn = GpioHostOwnGpio;
  GpioConfig->Direction = GpioDirOut;
  GpioConfig->OutputState = GpioOutDefault;
  GpioConfig->InterruptConfig = GpioIntDis;
  GpioConfig->PowerConfig = GpioPlatformReset;
  GpioConfig->ElectricalConfig = GpioTermNone;
}

VOID
MipiCamGpioInit (
  VOID
  )
{
  PCH_SETUP                       PchSetup;
  SETUP_DATA                      SetupData;
  UINTN                           VarSize;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices;
  EFI_STATUS                      Status;
  GPIO_CONFIG                     GpioConfig;
  GPIO_PAD                        GpioPad;
  UINT8                           GpioPin;

  //
  // Locate Setup variables
  //
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &VariableServices
             );
  if (EFI_ERROR(Status)) {
    DEBUG((DEBUG_ERROR, "MipiCamGpioInit: PeiServicesLocatePpi failed\n"));
    return;
  }

  VarSize = sizeof (PCH_SETUP);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"PchSetup",
                               &gPchSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &PchSetup
                               );
  if (EFI_ERROR(Status)) {
    DEBUG((DEBUG_ERROR, "MipiCamGpioInit: GetVariable (PchSetup) failed\n"));
    return;
  }

  VarSize = sizeof (SETUP_DATA);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"Setup",
                               &gSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &SetupData
                               );
  if (EFI_ERROR(Status)) {
    DEBUG((DEBUG_ERROR, "MipiCamGpioInit: GetVariable (SetupData) failed\n"));
    return;
  }

  MipiCamConfigureGpio (&GpioConfig);
  if ((SetupData.MipiCam_ControlLogic0) && (SetupData.MipiCam_ControlLogic0_Type == (UINT8)1) && (SetupData.MipiCam_ControlLogic0_GpioPinsEnabled)) {
    DEBUG((DEBUG_INFO, "MipiCamGpioInit ControlLogic0\n"));
    for (GpioPin = 0; GpioPin < SetupData.MipiCam_ControlLogic0_GpioPinsEnabled; GpioPin ++) {
      GpioPad = GpioGetGpioPadFromGroupIndexAndPadNumber (
                                   (UINT32)SetupData.MipiCam_ControlLogic0_GpioGroupNumber[GpioPin],
                                   (UINT32)SetupData.MipiCam_ControlLogic0_GpioGroupPadNumber[GpioPin]);
      DEBUG((DEBUG_INFO, "MipiCamGpioInit GpioPad 0x%x\n", GpioPad));
      GpioSetPadConfig (GpioPad, &GpioConfig);
    }
  }
  if ((SetupData.MipiCam_ControlLogic1) && (SetupData.MipiCam_ControlLogic1_Type == (UINT8)1) && (SetupData.MipiCam_ControlLogic1_GpioPinsEnabled)) {
    DEBUG((DEBUG_INFO, "MipiCamGpioInit ControlLogic1\n"));
    for (GpioPin = 0; GpioPin < SetupData.MipiCam_ControlLogic1_GpioPinsEnabled; GpioPin ++) {
      GpioPad = GpioGetGpioPadFromGroupIndexAndPadNumber (
                                   (UINT32)SetupData.MipiCam_ControlLogic1_GpioGroupNumber[GpioPin],
                                   (UINT32)SetupData.MipiCam_ControlLogic1_GpioGroupPadNumber[GpioPin]);
      DEBUG((DEBUG_INFO, "MipiCamGpioInit GpioPad 0x%x\n", GpioPad));
      GpioSetPadConfig (GpioPad, &GpioConfig);
    }
  }
  if ((SetupData.MipiCam_ControlLogic2) && (SetupData.MipiCam_ControlLogic2_Type == (UINT8)1) && (SetupData.MipiCam_ControlLogic2_GpioPinsEnabled)) {
    DEBUG((DEBUG_INFO, "MipiCamGpioInit ControlLogic2\n"));
    for (GpioPin = 0; GpioPin < SetupData.MipiCam_ControlLogic2_GpioPinsEnabled; GpioPin ++) {
      GpioPad = GpioGetGpioPadFromGroupIndexAndPadNumber (
                                   (UINT32)SetupData.MipiCam_ControlLogic2_GpioGroupNumber[GpioPin],
                                   (UINT32)SetupData.MipiCam_ControlLogic2_GpioGroupPadNumber[GpioPin]);
      DEBUG((DEBUG_INFO, "MipiCamGpioInit GpioPad 0x%x\n", GpioPad));
      GpioSetPadConfig (GpioPad, &GpioConfig);
    }
  }
  if ((SetupData.MipiCam_ControlLogic3) && (SetupData.MipiCam_ControlLogic3_Type == (UINT8)1) && (SetupData.MipiCam_ControlLogic3_GpioPinsEnabled)) {
    DEBUG((DEBUG_INFO, "MipiCamGpioInit ControlLogic3\n"));
    for (GpioPin = 0; GpioPin < SetupData.MipiCam_ControlLogic3_GpioPinsEnabled; GpioPin ++) {
      GpioPad = GpioGetGpioPadFromGroupIndexAndPadNumber (
                                   (UINT32)SetupData.MipiCam_ControlLogic3_GpioGroupNumber[GpioPin],
                                   (UINT32)SetupData.MipiCam_ControlLogic3_GpioGroupPadNumber[GpioPin]);
      DEBUG((DEBUG_INFO, "MipiCamGpioInit GpioPad 0x%x\n", GpioPad));
      GpioSetPadConfig (GpioPad, &GpioConfig);
    }
  }
  return;
}


VOID
TsnDeviceGpioInit (
  VOID
  )
{
#if FixedPcdGet8(PcdEmbeddedEnable) == 0x1
  EFI_STATUS                      Status;
  PCH_SETUP                       PchSetup;
  UINTN                           VarSize;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices;

  DEBUG ((DEBUG_INFO, "TsnDeviceGpioInit() Start\n"));

  //
  // Locate Setup variables
  //
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &VariableServices
             );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "TsnGpioInit: PeiServicesLocatePpi failed\n"));
    return;
  }

  VarSize = sizeof (PCH_SETUP);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               L"PchSetup",
                               &gPchSetupVariableGuid,
                               NULL,
                               &VarSize,
                               &PchSetup
                               );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "TsnGpioInit: GetVariable (PchSetup) failed\n"));
    return;
  }

  if (PchSetup.PchTsnEnable == 1) {
    DEBUG ((DEBUG_INFO, "TSN Device GpioInit\n"));
    ConfigureGpio ((VOID *) (UINTN) PcdGet32 (PcdBoardGpioTableTsnDevice), (UINTN) PcdGet16 (PcdBoardGpioTableTsnDeviceSize));
  } else {
    //
    // Disable Tsn Pcs if TsnEnable feature not available
    //
    PchSetup.TsnPcsEnabled = 0;
  }
  DEBUG ((DEBUG_INFO, "TsnDeviceGpioInit() End\n"));
#endif
}

/**
  Configure advanced GPIO

**/
VOID
GpioInitAdvanced (
  VOID
  )
{

  TouchPanelGpioInit ();

  // Configure Connectivity options
  ConnectivityGpioInit();
  if (PcdGetBool (PcdMipiCamGpioEnable)) {
    MipiCamGpioInit();
    CvfGpioInit();
  }

  TsnDeviceGpioInit ();

  //
  // Lock pads after initializing platform GPIO.
  // Pads which were requested to be unlocked during configuration
  // will not be locked.
  //
  GpioLockPads ();

  return;
}
/**
  Platform Init PEI module entry point

  @param[in]  FileHandle           Not used.
  @param[in]  PeiServices          General purpose services available to every PEIM.

  @retval     EFI_SUCCESS          The function completes successfully
  @retval     EFI_OUT_OF_RESOURCES Insufficient resources to create database
**/
EFI_STATUS
EFIAPI
PlatformInitAdvancedPostMemEntryPoint (
  IN       EFI_PEI_FILE_HANDLE  FileHandle,
  IN CONST EFI_PEI_SERVICES     **PeiServices
  )
{
  EFI_STATUS                       Status;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI  *VariableServices;
  SETUP_DATA                       SetupData;
  UINTN                            VariableSize;
  UINT8                            FwConfig;
  UINT8                            CrashLogCollect;
#if FixedPcdGet8(PcdFspModeSelection) == 1
  VOID                             *FspmUpd;
#else
  SI_PREMEM_POLICY_PPI             *SiPreMemPolicyPpi;
  CPU_SECURITY_PREMEM_CONFIG       *CpuSecurityPreMemConfig;
#endif
  PEI_CORE_INSTANCE                *PrivateData;
  UINTN                            CurrentFv;
  PEI_CORE_FV_HANDLE               *CoreFvHandle;
  VOID                             *HobData;
  PostCode (PLATFORM_INIT_POSTMEM_ENTRY);

#if FixedPcdGet8(PcdFspModeSelection) == 1
  FspmUpd = NULL;
#else
  SiPreMemPolicyPpi = NULL;
  CpuSecurityPreMemConfig = NULL;
#endif

#if FixedPcdGet8(PcdFspModeSelection) == 1
  FspmUpd = (FSPM_UPD *) GetFspMemoryInitUpdDataPointer ();
  ASSERT (FspmUpd != NULL);
#else
  Status = PeiServicesLocatePpi (&gSiPreMemPolicyPpiGuid, 0, NULL, (VOID **) &SiPreMemPolicyPpi);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuSecurityPreMemConfigGuid, (VOID *) &CpuSecurityPreMemConfig);
  ASSERT_EFI_ERROR (Status);
#endif

  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,  // GUID
             0,                                 // INSTANCE
             NULL,                              // EFI_PEI_PPI_DESCRIPTOR
             (VOID **) &VariableServices        // PPI
             );
  ASSERT_EFI_ERROR (Status);

  VariableSize = sizeof (SETUP_DATA);
  Status = VariableServices->GetVariable (
                               VariableServices,
                               PLATFORM_SETUP_VARIABLE_NAME,
                               &gSetupVariableGuid,
                               NULL,
                               &VariableSize,
                               &SetupData
                               );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Fail to get System Configuration and set the configuration to production mode!\n"));
    FwConfig = 0;
    CrashLogCollect = 1;
  } else {
    FwConfig = SetupData.FirmwareConfiguration;
    CrashLogCollect = SetupData.EnableCrashLog;
  }
  //
  // BIOS shall extract Crash records from PMC SSRAM after memory init
  // and before enabling the Energy Reporting feature
  //
  if (CrashLogCollect == 1) {
    CrashLogCollectDataFromPmcSSRAM ();
    CpuCrashLogCollectDataFromTelemetrySRAM ();
  }
  GpioInitAdvanced();
  //
  // Build a HOB to show current FV location for SA policy update code to consume.
  //
  PrivateData = PEI_CORE_INSTANCE_FROM_PS_THIS(PeiServices);
  CurrentFv = PrivateData->CurrentPeimFvCount;
  CoreFvHandle = &(PrivateData->Fv[CurrentFv]);

  HobData = BuildGuidHob (
             &gPlatformInitFvLocationGuid,
             sizeof (VOID *)
             );
  ASSERT (HobData != NULL);
  CopyMem (HobData, (VOID *) &CoreFvHandle, sizeof (VOID *));

  //
  // Initialize Intel PEI Pos-Mem Platform Policy //move to board /change name
  //
#if (FixedPcdGet8(PcdFspModeSelection) == 1)
  //
  // FSP API mode call the function directly.
  //
  PeiPolicyInit (PeiServices, FwConfig);
#else
  //
  // FSP Dispatch mode or non-FSP build will depend on DefaultPolicyInit PPI.
  //
  Status = PeiServicesNotifyPpi (&mSiDefaultPolicyInitNotifyList);
  ASSERT_EFI_ERROR (Status);
#endif

  //
  // Create USB Boot First hotkey information HOB
  //
  CreateAttemptUsbFirstHotkeyInfoHob ();


  //
  // Check Platform debug state
  //
  if (PlatformDebugStateEnabled ()) {
    //
    // Extend platform state into TPM PCR[7]
    //
    PcdSetBoolS (PcdFirmwareDebuggerInitialized, TRUE);
  }

  //
  // Notify mPeiGraphicsPlatformNotifyList
  //
  DEBUG ((DEBUG_INFO, "Notify mPeiGraphicsPlatformNotifyList \n"));
  Status = PeiServicesNotifyPpi (&mPeiGraphicsPlatformNotifyList);

  //
  // Performing PlatformInitEndOfPei after EndOfPei PPI produced
  //
  Status = PeiServicesNotifyPpi (&mEndOfPeiNotifyList);
  PostCode (PLATFORM_INIT_POSTMEM_EXIT);

  return Status;
}
