/** @file
  The file contains SA related setup options

@copyright
  INTEL CONFIDENTIAL
  Copyright 2010 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <SetupPrivate.h>
#include "OemSetup.h"
#include "SaSetup.h"
#include "PlatformBoardId.h"
#include "CpuRegs.h"
#include "DimmInfo.h"
#include <Library/UefiBootServicesTableLib.h>
#include <Library/UefiRuntimeServicesTableLib.h>
#include <Library/CpuPlatformLib.h>
#include <Protocol/SaPolicy.h>
#include <Protocol/SmbusHc.h>
#include <Protocol/MemInfo.h>
#include <Protocol/GopPolicy.h>
#if FixedPcdGetBool(PcdITbtEnable) == 1
#include "TcssDataHob.h"
#endif
#include <IpuDataHob.h>
#include <Library/CpuPcieInfoFruLib.h>
#include <Library/CpuPcieInitCommon.h>
#include <CpuPcieHob.h>
#include <Library/CpuRegbarAccessLib.h>
#include <VmdInfoHob.h>
#include <Register/SataRegs.h>
#include <Library/SataLib.h>
#include <Library/GraphicsInfoLib.h>
#include <Library/VmdInfoLib.h>
#include <Library/PchPciBdfLib.h>
#include <Register/SaPcieDmiRegs.h>

typedef union {
  struct {
    UINT32  Low;
    UINT32  High;
  } Data32;
  UINT64 Data;
} UINT64_STRUCT;

static EFI_HII_HANDLE     gHiiHandle;


VOID
UpdatePegInfo (
  EFI_HII_HANDLE HiiHandle,
  UINT16         Class
  );

VOID
UpdateDmiInfo (
  EFI_HII_HANDLE HiiHandle,
  UINT16         Class
  );

EFI_STATUS
EFIAPI
UpdateVmdInfo (
  EFI_HII_HANDLE HiiHandle
  );
#define NB_MIN(a, b)  (((a) < (b)) ? (a) : (b))
#define MCDECS_CR_MRC_REVISION_TGL_UY             (0x00005034)
#define MCDECS_CR_MRC_REVISION_TGL_H              (0x0000d834)

EFI_STRING_ID  DimmSizeString[SLOT_NUM] = {
  STRING_TOKEN (STR_DIMM_SIZE_MC0_CH0_SLT0_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC0_CH0_SLT1_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC0_CH1_SLT0_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC0_CH1_SLT1_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC0_CH2_SLT0_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC0_CH2_SLT1_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC0_CH3_SLT0_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC0_CH3_SLT1_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC1_CH0_SLT0_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC1_CH0_SLT1_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC1_CH1_SLT0_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC1_CH1_SLT1_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC1_CH2_SLT0_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC1_CH2_SLT1_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC1_CH3_SLT0_VALUE),
  STRING_TOKEN (STR_DIMM_SIZE_MC1_CH3_SLT1_VALUE)
};

EFI_STRING_ID  RankInDimmString[SLOT_NUM] = {
  STRING_TOKEN (STR_DIMM_MC0_CH0_SLT0_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH0_SLT1_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH1_SLT0_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH1_SLT1_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH2_SLT0_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH2_SLT1_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH3_SLT0_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH3_SLT1_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH0_SLT0_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH0_SLT1_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH1_SLT0_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH1_SLT1_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH2_SLT0_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH2_SLT1_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH3_SLT0_RANK_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH3_SLT1_RANK_VALUE)
};

EFI_STRING_ID  DimmMfgString[SLOT_NUM] = {
  STRING_TOKEN (STR_DIMM_MC0_CH0_SLT0_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH0_SLT1_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH1_SLT0_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH1_SLT1_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH2_SLT0_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH2_SLT1_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH3_SLT0_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC0_CH3_SLT1_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH0_SLT0_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH0_SLT1_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH1_SLT0_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH1_SLT1_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH2_SLT0_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH2_SLT1_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH3_SLT0_MFG_VALUE),
  STRING_TOKEN (STR_DIMM_MC1_CH3_SLT1_MFG_VALUE)
};

EFI_STRING_ID  DimmStatusString[SLOT_NUM] = {
  STRING_TOKEN (STR_DIMM_MC0_CH0_SLT0_STATUS),
  STRING_TOKEN (STR_DIMM_MC0_CH0_SLT1_STATUS),
  STRING_TOKEN (STR_DIMM_MC0_CH1_SLT0_STATUS),
  STRING_TOKEN (STR_DIMM_MC0_CH1_SLT1_STATUS),
  STRING_TOKEN (STR_DIMM_MC0_CH2_SLT0_STATUS),
  STRING_TOKEN (STR_DIMM_MC0_CH2_SLT1_STATUS),
  STRING_TOKEN (STR_DIMM_MC0_CH3_SLT0_STATUS),
  STRING_TOKEN (STR_DIMM_MC0_CH3_SLT1_STATUS),
  STRING_TOKEN (STR_DIMM_MC1_CH0_SLT0_STATUS),
  STRING_TOKEN (STR_DIMM_MC1_CH0_SLT1_STATUS),
  STRING_TOKEN (STR_DIMM_MC1_CH1_SLT0_STATUS),
  STRING_TOKEN (STR_DIMM_MC1_CH1_SLT1_STATUS),
  STRING_TOKEN (STR_DIMM_MC1_CH2_SLT0_STATUS),
  STRING_TOKEN (STR_DIMM_MC1_CH2_SLT1_STATUS),
  STRING_TOKEN (STR_DIMM_MC1_CH3_SLT0_STATUS),
  STRING_TOKEN (STR_DIMM_MC1_CH3_SLT1_STATUS)
};

EFI_STRING_ID  PegInfo[4] = {
  STRING_TOKEN (STR_PEG0_INFO_VALUE),
  STRING_TOKEN (STR_PEG1_INFO_VALUE),
  STRING_TOKEN (STR_PEG2_INFO_VALUE),
  STRING_TOKEN (STR_PEG3_INFO_VALUE)
};

VOID
InitSaStrings (
  EFI_HII_HANDLE HiiHandle,
  UINT16         Class
  )
{
  UINT8                           MemoryType;
  UINT8                           Profile;
  UINT64                          MemorySize;
  UINT16                          DdrFrequency;
  UINT16                          VendorId;
  EFI_STATUS                      Status;
  UINTN                           SaSetupVariableSize;
  UINTN                           SetupVolVariableSize;
  UINT32                          Value32;
  UINTN                           Slot;
  UINT8                           w;
  UINT8                           x;
  UINT8                           y;
  UINT8                           z;
  UINT8                           Rp0;
  UINT8                           Rpn;
  UINT16                          Data16;
  UINT8                           NodeIndex;
  UINT8                           ChannelIndex;
  UINT8                           DimmIndex;
  UINT8                           MfgIdIndex;
  UINT8                           SkipStringUpdate;

  MEM_INFO_PROTOCOL               *MemInfoProtocol;
  MEMORY_INFO_DATA                *MemInfoData;
  MEMORY_TIMING                   *Timing;
  SA_POLICY_PROTOCOL              *SaPolicy;
  CHAR8                           *DimmStatusStr;
  UINT32                          SaVariableAttributes;
  UINT32                          SetupVolAttributes;
  GRAPHICS_DXE_CONFIG             *GraphicsDxeConfig;
  UINT8                           CpuSku;
  UINT32                          EdramSize;
  UINT64                          IgdPciD2F0RegBase;
  UINT8                           IndexOfMemTypeList;
  IPU_DATA_HOB                    *IpuDataHob;
  UINT32                          DssmClockSel;
  UINT64                          GttMmAdr;
  UINT64_STRUCT                   Data64;
#if FixedPcdGetBool(PcdITbtEnable) == 1
  TCSS_DATA_HOB         *TcssHob;
#endif


  if ((Class != MAIN_FORM_SET_CLASS) && (Class != ADVANCED_FORM_SET_CLASS)) {
    return;
  }

  DEBUG ((DEBUG_INFO, "<InitSaStrings>"));

  CpuSku = GetCpuSku();
  Status = gBS->LocateProtocol (&gSaPolicyProtocolGuid, NULL, (VOID **) &SaPolicy);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *)SaPolicy, &gGraphicsDxeConfigGuid, (VOID *)&GraphicsDxeConfig);
  ASSERT_EFI_ERROR (Status);

  SaSetupVariableSize = sizeof (SA_SETUP);
  Status = gRT->GetVariable (
                  L"SaSetup",
                  &gSaSetupVariableGuid,
                  &SaVariableAttributes,
                  &SaSetupVariableSize,
                  &mSaSetup
                  );
  DEBUG((DEBUG_INFO, "GetVariable 'SaSetup' Status = %r", Status));

  SetupVolVariableSize = sizeof (SETUP_VOLATILE_DATA);
  Status = gRT->GetVariable (
                  L"SetupVolatileData",
                  &gSetupVariableGuid,
                  &SetupVolAttributes,
                  &SetupVolVariableSize,
                  &mSetupVolatileData
                  );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return;
  }


  //
  // Get the Memory Info HOB Protocol if it exists.
  //
  Status = gBS->LocateProtocol (&gMemInfoProtocolGuid, NULL, (VOID **) &MemInfoProtocol);
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return;
  }

  MemInfoData = &MemInfoProtocol->MemInfoData;

  //
  // Force the DiSM policy to the Max allowed DiSM size in GB, for the DDR space available
  //
  if ((MemInfoData->memSize == MEM_4GB) && (mSaSetup.DismSize >= 1)) {
    mSaSetup.DismSize = 1;
  } else if ((MemInfoData->memSize == MEM_8GB) && (mSaSetup.DismSize >= 3)) {
    mSaSetup.DismSize = 3;
  } else if ((MemInfoData->memSize == MEM_12GB) && (mSaSetup.DismSize >= 5)) {
    mSaSetup.DismSize = 5;
  }

  // For 2LM Sku, DSM is still required but only needs to be 4MB
  //
  if (mSaSetup.DismSize != 0) {
    mSaSetup.IgdDvmt50PreAlloc = 240; // 4 MB
  }

  MemoryType    = MemInfoData->DdrType;
  DdrFrequency  = MemInfoData->ddrFreq;
  Profile       = MemInfoData->Profile;
  Timing        = &MemInfoData->Timing[Profile];

  mSaSetup.XmpProfileEnable = MemInfoData->XmpProfileEnable;
  Status = gRT->SetVariable (
                  L"SaSetup",
                  &gSaSetupVariableGuid,
                  SaVariableAttributes,
                  SaSetupVariableSize,
                  &mSaSetup
                  );

  //
  // Update the memory size string for main page and chipset page
  //
  if ((Class == MAIN_FORM_SET_CLASS) || (Class == ADVANCED_FORM_SET_CLASS)) {
    MemorySize = MemInfoData->memSize;
    InitString (HiiHandle, STRING_TOKEN (STR_MEMORY_SIZE_VALUE), L"%5ld MB", MemorySize);
    InitString (HiiHandle, STRING_TOKEN (STR_MEMORY_FREQ_VALUE), L"%5ld MT/s", DdrFrequency);

    //
    // MCDECS_CR_MRC_REVISION - MRC version
    // According to MrcCrOffsetProjAdj(), TGL H has offset with the TGL U/Y
    //
    if ((CpuSku == EnumCpuUlt) || (CpuSku == EnumCpuUlx)) {
      Value32 = MmioRead32 ((UINTN) PcdGet64 (PcdMchBaseAddress) + MCDECS_CR_MRC_REVISION_TGL_UY);
    } else {
      Value32 = MmioRead32 ((UINTN) PcdGet64 (PcdMchBaseAddress) + MCDECS_CR_MRC_REVISION_TGL_H);
    }

    x       = (UINT8) ((Value32 & 0xFF000000) >> 24);
    y       = (UINT8) ((Value32 & 0xFF0000) >> 16);
    z       = (UINT8) ((Value32 & 0xFF00) >> 8);
    w       = (UINT8) (Value32 & 0xFF);

    InitString (HiiHandle, STRING_TOKEN (STR_MRC_REV_VALUE), L"%d.%d.%d.%d", x, y, z, w);
  }

  IgdPciD2F0RegBase = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, GetIgdBusNumber(), GetIgdDevNumber(), GetIgdFuncNumber(), 0);
  Data64.Data32.High = (PciSegmentRead32 (IgdPciD2F0RegBase + GetGttMmAdrOffset() + 4)) & 0xFFFFFFFF;
  Data64.Data32.Low  = (PciSegmentRead32 (IgdPciD2F0RegBase + GetGttMmAdrOffset())) & 0xFFFFFFF0;
  GttMmAdr = (UINT64)Data64.Data;

  if (PciSegmentRead16 (IgdPciD2F0RegBase + 0x2) != 0xFFFF) {
    InitString (
      HiiHandle,
      STRING_TOKEN (STR_CHIP_IGFX_GOP_REV_VALUE),
      L"%s",
      &GraphicsDxeConfig->GopVersion
    );
  }

  //
  // GT frequency setting
  //
  if ((PciSegmentRead16 (IgdPciD2F0RegBase + 0x2) != 0xFFFF)) {
    Value32 = MmioRead32 ((UINTN) PcdGet64 (PcdMchBaseAddress) + 0x5998);
    Rpn = (UINT8) ((Value32 & 0xFF0000) >> 16);
    Rp0 = (UINT8) (Value32 & 0xFF);
    InitString (HiiHandle, STRING_TOKEN (STR_GT_FREQ_HELP), L"Maximum GT frequency limited by the user. Choose between %dMHz (RPN) and %dMHz (RP0). Value beyond the range will be clipped to min/max supported by SKU", Rpn*50, Rp0*50);
  }
    //
    // Populate CdClockSelector variable here.
    //
    if (!IgfxCmdRegEnabled()) {
      ///
      /// Enable Bus Master and Memory access on 0:2:0
      ///
      PciSegmentOr16 (IgdPciD2F0RegBase + PCI_COMMAND_OFFSET, (BIT2 | BIT1));
    }
    DssmClockSel = GetIgdDssmRefClockFreqValue (GttMmAdr);
    DEBUG ((DEBUG_INFO, "DssmClockSel = %d\n ",DssmClockSel));
    switch (DssmClockSel) {
      case 0: // 24Mhz
        mSetupVolatileData.CdClockSelector = 2;
        break;
      case 1: // 19.2MHz
        mSetupVolatileData.CdClockSelector = 0;
        break;
      case 2: // 38.4MHz
        mSetupVolatileData.CdClockSelector = 1;
        break;
      case 3: // 25Mhz
        mSetupVolatileData.CdClockSelector = 3;
        break;
      default:
        mSetupVolatileData.CdClockSelector = 2;
        break;
    }
    DEBUG ((DEBUG_INFO, "mSaSetupVolatileDataetup.CdClockSelector = %d\n",mSetupVolatileData.CdClockSelector));

    Status = gRT->SetVariable (
                L"SetupVolatileData",
                &gSetupVariableGuid,
                SetupVolAttributes,
                SetupVolVariableSize,
                &mSetupVolatileData
                );

  if (Class == MAIN_FORM_SET_CLASS) {
    gHiiHandle = HiiHandle;
    Data16 = PciSegmentRead16 (IgdPciD2F0RegBase + 0x2);
    if (Data16 != 0xFFFF) {
      InitString (gHiiHandle, STRING_TOKEN (STR_PROCESSOR_GT_VALUE), L"0x%X", Data16);
    }

    //
    // EDRAM Size display
    //
    if (IsEdramEnable ()) {
      EdramSize = 0;
      if ((CpuSku == EnumCpuUlt) || (CpuSku == EnumCpuUlx)) {
        EdramSize = 64;
#ifdef CPU_CFL
        if ((Data16 != 0xFFFF) && (Data16 == V_SA_PCI_DEV_2_GT3_CFL_ULT_2_ID)) {
          EdramSize = 128;
        }
#endif
      } else if ((CpuSku == EnumCpuTrad) || (CpuSku == EnumCpuHalo)) {
        EdramSize = 128;
#ifdef CPU_CFL
        if ((Data16 != 0xFFFF) && ((Data16 == V_SA_PCI_DEV_2_GT4_SDT_ID) || (Data16 == V_SA_PCI_DEV_2_GT4_KDT_ID))) {
          //
          // Exception for GT4e LGA sku due to LGA packaging limitation
          //
          EdramSize = 64;
        }
#endif
      }
      InitString (
        HiiHandle,
        STRING_TOKEN (STR_EDRAM_SIZE_VALUE),
        L"%d MB",
        EdramSize
      );
    }

  }

  if (Class == ADVANCED_FORM_SET_CLASS) {
    //
    //Find the memory type in the List
    //
    for (IndexOfMemTypeList = 0; IndexOfMemTypeList < MemoryTypeListSize; IndexOfMemTypeList++) {
      if (MemoryType == MemoryTypeList[IndexOfMemTypeList].DdrTypeNum) {
        break;
      }
    }

    SetupVolVariableSize = sizeof (SETUP_VOLATILE_DATA);
    Status = gRT->GetVariable (
                    L"SetupVolatileData",
                    &gSetupVariableGuid,
                    &SetupVolAttributes,
                    &SetupVolVariableSize,
                    &mSetupVolatileData
                    );
    ASSERT_EFI_ERROR (Status);
    if (!EFI_ERROR (Status)) {
      Slot = 0;
      SkipStringUpdate = FALSE;
      mSetupVolatileData.MemoryType = MemoryType;
      DEBUG((EFI_D_INFO, "MemoryType : %X\n", MemoryType));
      for (NodeIndex = 0; NodeIndex < NODE_NUM; NodeIndex++) {
        for (ChannelIndex = 0; ChannelIndex < CH_NUM; ChannelIndex++) {
          //
          // Skip the strings init for DDR4 and DDR5 when Channel is equal or above one
          //
          if (((ChannelIndex > 0) && (mSetupVolatileData.MemoryType == MRC_DDR_TYPE_DDR4)) || \
             ((ChannelIndex > 1) && (mSetupVolatileData.MemoryType == MRC_DDR_TYPE_DDR5))) {
            SkipStringUpdate = TRUE;
          }
          for (DimmIndex = 0; DimmIndex < DIMM_NUM; DimmIndex++,Slot++) {
            //
            // Skip the stings init for DDR5, LPDDR5 and LPDRR4 when Dimm is above zero
            //
            if ((DimmIndex > 0) && (mSetupVolatileData.MemoryType != MRC_DDR_TYPE_DDR4)) {
              SkipStringUpdate = TRUE;
            }
            if (!SkipStringUpdate) {
              InitString (
                HiiHandle,
                DimmSizeString[Slot],
                L"%4ld MB (%a)",
                MemInfoData->dimmSize[Slot],
                (mSetupVolatileData.MemoryType < MRC_DDR_TYPE_UNKNOWN) ? MemoryTypeStr[mSetupVolatileData.MemoryType] : "Unknown"
                );
              if (MemInfoData->DimmStatus[Slot] < DIMM_NOT_PRESENT) {
                mSetupVolatileData.DimmPresent[Slot] = 1;
                InitString (
                  HiiHandle,
                  RankInDimmString[Slot],
                  L"%1d",
                  MemInfoData->RankInDimm[Slot]
                  );
                DimmStatusStr = (MemInfoData->DimmStatus[Slot] == DIMM_DISABLED) ? "Populated & Disabled" : "Populated & Enabled";
                /**
                  Get the Memory Module Vendor JEDEC ID
                  Byte 117-118 for DDR3/LPDDR3 and byte 320-321 for DDR4
                  It's from first byte of SPD buffer.
                **/
                VendorId = *(UINT16 *) (UINTN) (MemInfoData->DimmsSpdData[Slot]);
                VendorId &= ~(BIT7);  // Clear the parity bit
                for (MfgIdIndex = 0; MfgIdIndex < MemoryModuleManufacturerListSize; MfgIdIndex++) {
                  if (VendorId == MemoryModuleManufacturerList[MfgIdIndex].MfgId) {
                    InitString (
                      HiiHandle,
                      DimmMfgString[Slot],
                      L"%a",
                      MemoryModuleManufacturerList[MfgIdIndex].String
                     );
                    break;
                  }
                }
              } else {
                mSetupVolatileData.DimmPresent[Slot] = 0;
                DimmStatusStr = "Not Populated / Disabled";
              }

              InitString (
                HiiHandle,
                DimmStatusString[Slot],
                L"%a",
                DimmStatusStr
                );
            } else {
              SkipStringUpdate = FALSE;
            }
          }
        }
      }
      //
      // SA PCIe code version
      //
      Value32 = MmioRead32 ((UINTN) PcdGet64 (PcdDmiBaseAddress) + R_SA_DMIBAR_SCRATCHPAD1_OFFSET);
      w       = (UINT8) ((Value32 & 0xFF000000) >> 24);
      x       = (UINT8) ((Value32 & 0x00FF0000) >> 16);
      y       = (UINT8) ((Value32 & 0x0000FF00) >> 8);
      z       = (UINT8)  (Value32 & 0x000000FF);

      InitString (
        HiiHandle,
        STRING_TOKEN (STR_CPU_PCIE_CODE_VERSION_VALUE),
        L"%d.%d.%d.%d",
        w,
        x,
        y,
        z
        );
      //
      // VT-d status report
      //
      InitString (
        HiiHandle,
        STRING_TOKEN (STR_SA_VTD_VALUE),
        mSetupVolatileData.VTdAvailable ? L"Supported" : L"Unsupported"
        );

      //
      // IPU BIOS Setup option display
      //
      IpuDataHob = (IPU_DATA_HOB *) GetFirstGuidHob (&gIpuDataHobGuid);
      if (IpuDataHob != NULL) {
        mSetupVolatileData.IpuSupport = 1;
      } else {
        mSetupVolatileData.IpuSupport = 0;
      }

      Status = gRT->SetVariable (
                      L"SetupVolatileData",
                      &gSetupVariableGuid,
                      SetupVolAttributes,
                      SetupVolVariableSize,
                      &mSetupVolatileData
                      );
      ASSERT_EFI_ERROR (Status);
    } // if SetupVolatileData found
    //
    // Update the tCL, tRCD, tRP and tRAS string with data obtained from MemInfo protocol
    //
    InitString (
      HiiHandle,
      STRING_TOKEN (STR_MEMORY_VDD_VALUE),
      L"%d",
      MemInfoData->VddVoltage[Profile]
      );

    InitString (
      HiiHandle,
      STRING_TOKEN (STR_MEMORY_TIMINGS_VALUE),
      L"%d-%d-%d-%d",
      Timing->tCL,
      Timing->tRCDtRP,
      Timing->tRCDtRP,
      Timing->tRAS
      );


  } // if ADVANCED_FORM_SET_CLASS

  //
  // Locate Platform Info Protocol.
  //
  if (PcdGet8 (PcdPlatformType) == TypeTrad) {
    UpdatePegInfo (HiiHandle, Class);
    UpdateDmiInfo (HiiHandle, Class);
  }


//
//  IOM, TBT firmware version and TBT IMR status
//
#if FixedPcdGetBool(PcdITbtEnable) == 1
  TcssHob = (TCSS_DATA_HOB *) GetFirstGuidHob (&gTcssHobGuid);
  if (TcssHob != NULL) {
    InitString (
      HiiHandle,
      STRING_TOKEN (STR_TCSS_TBT_STATUS_STRING),
      L"TBT FW IMR Status: %08X",
      TcssHob->TcssData.TbtImrStatus.RegValue
      );

  if (TcssHob->TcssData.TbtImrStatus.Bits.Done) {
      InitString (
        HiiHandle,
        STRING_TOKEN (STR_TCSS_TBT_VERSION_STRING),
        L"TBT FW version: %04d",
        TcssHob->TcssData.TbtImrStatus.Bits.TbtFwVersion
        );
    }

    if (TcssHob->TcssData.IOMReady) {
      InitString (
        HiiHandle,
        STRING_TOKEN (STR_TCSS_IOM_VERSION_STRING),
        L"IOM FW version: %08X",
        TcssHob->TcssData.IomFwVersion
        );

      InitString (
        HiiHandle,
        STRING_TOKEN (STR_TCSS_IOM_ENGR_VERSION_STRING),
        L"IOM FW version: %08X",
        TcssHob->TcssData.IomFwEngrVersion
        );

      InitString (
        HiiHandle,
        STRING_TOKEN (STR_TC_CSTATE_STATUS),
        L"Deepest TC state: %04X",
        TcssHob->TcssData.DeepstTcState
        );
    }
  }

  // update VMD info if VMD is supported.
  if (IsVmdSupported()){
    UpdateVmdInfo(HiiHandle);
  }
#endif

}

VOID
UpdatePegInfo (
  EFI_HII_HANDLE HiiHandle,
  UINT16         Class
  )
{
  UINT16              Data16;
  UINT8               Index;
  UINT8               PegDeviceNumber;
  UINT8               PegFunctionNumber;
  UINTN               SetupVolVariableSize;
  UINT32              SetupVolAttributes;
  EFI_STATUS          Status;

  Data16 = 0;

  if (Class == ADVANCED_FORM_SET_CLASS) {

    SetupVolVariableSize = sizeof (SETUP_VOLATILE_DATA);
    Status = gRT->GetVariable (
                    L"SetupVolatileData",
                    &gSetupVariableGuid,
                    &SetupVolAttributes,
                    &SetupVolVariableSize,
                    &mSetupVolatileData
                    );
    ASSERT_EFI_ERROR (Status);

    for (Index = 0; Index < 4; Index++) {

      if (Index == 3) { // PEG 3 is at 0.6.0
        PegDeviceNumber   = SA_PEG3_DEV_NUM;
        PegFunctionNumber = SA_PEG3_FUN_NUM;
      } else {
        PegDeviceNumber   = SA_PEG0_DEV_NUM;
        PegFunctionNumber = Index;
      }

      //
      // Initialize PegPresent = 0 (Not Present)
      //
      mSetupVolatileData.PegPresent[Index] = 0;

      //
      // Check for DID VID to check Root Port is present
      //
      Data16 = PciSegmentRead16 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, PegDeviceNumber, PegFunctionNumber, 0x0));

      if (Data16 != 0xFFFF) {
        //
        // Read Slot Status register
        //
        Data16 = PciSegmentRead16 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, PegDeviceNumber, PegFunctionNumber, R_SA_PEG_SLOTSTS_OFFSET));

        //
        // Check for Presence Detect State SlotStatus 0xba[6]
        //
        if (Data16 & 0x40) {
          //
          // Read LinkStatus 0xb2[3-0]-Current Link Speed, [9-4]-Negotiated Link Width.
          //
          Data16 = PciSegmentRead16 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, PegDeviceNumber, PegFunctionNumber, R_SA_PEG_LSTS_OFFSET));

          InitString (
            HiiHandle,
            PegInfo[Index],
            L"x%d  Gen%1d",
            ((Data16 & 0x3f0) >> 4),
            (Data16 & 0xf)
            );

          //
          // Initialize PegPresent = 1 (Present)
          //
          mSetupVolatileData.PegPresent[Index] = 1;

        }
      }
    }
    //
    // for loop
    //
    Status = gRT->SetVariable (
                    L"SetupVolatileData",
                    &gSetupVariableGuid,
                    SetupVolAttributes,
                    SetupVolVariableSize,
                    &mSetupVolatileData
                    );
    ASSERT_EFI_ERROR (Status);

  }
  //
  // ADVANCED_FORM_SET_CLASS
  //
}

VOID
UpdateDmiInfo (
  EFI_HII_HANDLE HiiHandle,
  UINT16         Class
  )
{
  UINT16  Data16;
  UINT64  DmiBar;

  //
  // Get DMIBAR
  //
  PciSegmentReadBuffer (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_DMIBAR), sizeof (DmiBar), &DmiBar);
  DmiBar &= ~((UINT64) BIT0);

  //
  // LSTS 9:4 - DMI Link Negotiated Width, 3:0 - Max Link Speed, Gen2/Gen1 Infomation
  //
  Data16 = *(UINT16 *) (UINTN) (DmiBar + 0x8a);

  InitString (
    HiiHandle,
    STRING_TOKEN (STR_DMI_INFO_VALUE),
    L"X%d  Gen%1d",
    ((Data16 >> 4) & 0xf),
    (Data16 & 0xf)
    );
}

EFI_STATUS
EFIAPI
TcssPcieRootPortCallback (
  IN CONST EFI_HII_CONFIG_ACCESS_PROTOCOL *This,
  IN EFI_BROWSER_ACTION                   Action,
  IN EFI_QUESTION_ID                      KeyValue,
  IN UINT8                                Type,
  IN EFI_IFR_TYPE_VALUE                   *Value,
  OUT EFI_BROWSER_ACTION_REQUEST          *ActionRequest
  )
{
#if FixedPcdGetBool(PcdITbtEnable) == 1
  SA_SETUP    *SaSetup;
  SETUP_DATA  *SetupData;
  UINTN       VarSize;
  EFI_STATUS  Status;
  EFI_STRING  RequestString;
  UINT8       TcssRootPort0;
  UINT8       TcssRootPort1;
  UINT8       TcssRootPort2;
  UINT8       TcssRootPort3;

  if (Action != EFI_BROWSER_ACTION_CHANGING && Action != EFI_BROWSER_ACTION_CHANGED) {
    return EFI_UNSUPPORTED;
  }

  TcssRootPort0 = 0;
  TcssRootPort1 = 0;
  TcssRootPort2 = 0;
  TcssRootPort3 = 0;

  DEBUG ((EFI_D_INFO, "TcssPcieRootPortCallback()\n"));

  Status = EFI_SUCCESS;
  VarSize = sizeof (SA_SETUP);
  SaSetup = AllocatePool (VarSize);
  ASSERT (SaSetup != NULL);
  if (SaSetup == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  if (!HiiGetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  if (!EFI_ERROR (Status)) {
    TcssRootPort0 = SaSetup->TcssItbtPcie0En;
    TcssRootPort1 = SaSetup->TcssItbtPcie1En;
    TcssRootPort2 = SaSetup->TcssItbtPcie2En;
    TcssRootPort3 = SaSetup->TcssItbtPcie3En;

    RequestString = NULL;

    if (RequestString != NULL) {
      if (!HiiSetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup, RequestString)) {
        Status = EFI_NOT_FOUND;
      }
      ASSERT_EFI_ERROR (Status);
      FreePool (RequestString);
    }
  }
  FreePool (SaSetup);

  VarSize = sizeof (SETUP_DATA);
  SetupData = AllocatePool (VarSize);
  ASSERT (SetupData != NULL);
  if (SetupData == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  if (!HiiGetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  if (!EFI_ERROR (Status)) {
    SetupData->ITbtRootPort[0] = TcssRootPort0;
    SetupData->ITbtRootPort[1] = TcssRootPort1;
    SetupData->ITbtRootPort[2] = TcssRootPort2;
    SetupData->ITbtRootPort[3] = TcssRootPort3;

    RequestString = NULL;
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[0]), sizeof(SetupData->ITbtRootPort[0]));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[1]), sizeof(SetupData->ITbtRootPort[1]));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[2]), sizeof(SetupData->ITbtRootPort[2]));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[3]), sizeof(SetupData->ITbtRootPort[3]));

    if (RequestString != NULL) {
      if (!HiiSetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData, RequestString)) {
        Status = EFI_NOT_FOUND;
      }
      ASSERT_EFI_ERROR (Status);
      FreePool (RequestString);
    }
  }
  FreePool (SetupData);

  return Status;
#else
  return EFI_SUCCESS;
#endif
}

EFI_STATUS
EFIAPI
TcssXhciCallback (
  IN CONST EFI_HII_CONFIG_ACCESS_PROTOCOL *This,
  IN EFI_BROWSER_ACTION                   Action,
  IN EFI_QUESTION_ID                      KeyValue,
  IN UINT8                                Type,
  IN EFI_IFR_TYPE_VALUE                   *Value,
  OUT EFI_BROWSER_ACTION_REQUEST          *ActionRequest
  )
{
  SA_SETUP    *SaSetup;
  SETUP_DATA  *SetupData;
  UINTN       VarSize;
  EFI_STATUS  Status;
  EFI_STRING  RequestString;

  DEBUG ((EFI_D_INFO, "TcssXhciCallback()\n"));

  if (Action != EFI_BROWSER_ACTION_CHANGING && Action != EFI_BROWSER_ACTION_CHANGED) {
    return EFI_UNSUPPORTED;
  }

  Status = EFI_SUCCESS;
  VarSize = sizeof (SA_SETUP);
  SaSetup = AllocatePool (VarSize);
  ASSERT (SaSetup != NULL);
  if (SaSetup == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  if (!HiiGetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  if (!EFI_ERROR (Status)) {
    if (Value->u8 == 0) {
      SaSetup->TcssItbtPcie0En            = 0;
      SaSetup->TcssItbtPcie1En            = 0;
      SaSetup->TcssItbtPcie2En            = 0;
      SaSetup->TcssItbtPcie3En            = 0;
      SaSetup->TcssDma0En                 = 0;
      SaSetup->TcssDma1En                 = 0;
    } else {
      SaSetup->TcssItbtPcie0En            = 1;
      SaSetup->TcssItbtPcie1En            = 1;
      SaSetup->TcssItbtPcie2En            = 1;
      SaSetup->TcssItbtPcie3En            = 1;
      SaSetup->TcssDma0En                 = 1;
      SaSetup->TcssDma1En                 = 1;
      if (PcdGet8(PcdITbtRootPortNumber) == 2) {
        SaSetup->TcssItbtPcie2En            = 0;
        SaSetup->TcssItbtPcie3En            = 0;
        SaSetup->TcssDma1En                 = 0;
      }
      if (PcdGet8(PcdITbtRootPortNumber) == 3) {
        SaSetup->TcssItbtPcie2En            = 1;
        SaSetup->TcssItbtPcie3En            = 0;
        SaSetup->TcssDma1En                 = 1;
      }
    }
    RequestString = NULL;
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, TcssItbtPcie0En),            sizeof (SaSetup->TcssItbtPcie0En));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, TcssItbtPcie1En),            sizeof (SaSetup->TcssItbtPcie1En));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, TcssItbtPcie2En),            sizeof (SaSetup->TcssItbtPcie2En));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, TcssItbtPcie3En),            sizeof (SaSetup->TcssItbtPcie3En));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, TcssXdciEn),                 sizeof (SaSetup->TcssXdciEn));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, TcssDma0En),                 sizeof (SaSetup->TcssDma0En));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, TcssDma1En),                 sizeof (SaSetup->TcssDma1En));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, TcssVccstStatus),            sizeof (SaSetup->TcssVccstStatus));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, UsbOverride),                sizeof (SaSetup->UsbOverride));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, D3ColdEnable),               sizeof (SaSetup->D3ColdEnable));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, D3HotEnable),                sizeof (SaSetup->D3HotEnable));

    if (RequestString != NULL) {
      if (!HiiSetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup, RequestString)) {
        Status = EFI_NOT_FOUND;
      }
      ASSERT_EFI_ERROR (Status);
      FreePool (RequestString);
    }
  }
  FreePool (SaSetup);

  VarSize = sizeof (SETUP_DATA);
  SetupData = AllocatePool (VarSize);
  ASSERT (SetupData != NULL);
  if (SetupData == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }

  if (!HiiGetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  if (!EFI_ERROR (Status)) {
    if (Value->u8 == 0) {
      SetupData->UsbcBiosTcssHandshake = 0;
      SetupData->ITbtRootPort[0] = 0;
      SetupData->ITbtRootPort[1] = 0;
      SetupData->ITbtRootPort[2] = 0;
      SetupData->ITbtRootPort[3] = 0;
    } else {
      SetupData->UsbcBiosTcssHandshake = 1;
      SetupData->ITbtRootPort[0] = 1;
      SetupData->ITbtRootPort[1] = 1;
      SetupData->ITbtRootPort[2] = 1;
      SetupData->ITbtRootPort[3] = 1;
      if (PcdGet8(PcdITbtRootPortNumber) == 2) {
        SetupData->ITbtRootPort[2] = 0;
        SetupData->ITbtRootPort[3] = 0;
      }
      if (PcdGet8(PcdITbtRootPortNumber) == 3) {
        SetupData->ITbtRootPort[3] = 0;
      }
    }
    RequestString = NULL;
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, UsbcBiosTcssHandshake), sizeof (SetupData->UsbcBiosTcssHandshake));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[0]), sizeof (SetupData->ITbtRootPort[0]));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[1]), sizeof (SetupData->ITbtRootPort[1]));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[2]), sizeof (SetupData->ITbtRootPort[2]));
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[3]), sizeof (SetupData->ITbtRootPort[3]));

    if (RequestString != NULL) {
      if (!HiiSetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData, RequestString)) {
        Status = EFI_NOT_FOUND;
      }
      ASSERT_EFI_ERROR (Status);
      FreePool (RequestString);
    }
  }
  FreePool (SetupData);

  return Status;
}
EFI_STATUS
EFIAPI
UpdateVmdInfo (
  EFI_HII_HANDLE HiiHandle
  )
{
  EFI_STRING_ID   StrRef;
  VMD_INFO_HOB    *VmdInfoHob;
  UINT8           Index;

  DEBUG ((EFI_D_INFO, "UpdateVmdSetupInfo()\n"));

  VmdInfoHob = (VMD_INFO_HOB *) GetFirstGuidHob (&gVmdInfoHobGuid);
  if (VmdInfoHob == NULL) {
    DEBUG ((EFI_D_INFO, "Vmd Info Hob not found\n"));
    return EFI_NOT_FOUND;
  }

  for ( Index = 0; Index < VMD_MAX_DEVICES; ++Index ) {
    switch(Index) {
      case 0:
        StrRef = STRING_TOKEN (STR_RP1_BDF_VALUE);
        break;
      case 1:
        StrRef = STRING_TOKEN (STR_RP2_BDF_VALUE);
        break;
      case 2:
        StrRef = STRING_TOKEN (STR_RP3_BDF_VALUE);
        break;
      case 3:
        StrRef = STRING_TOKEN (STR_RP4_BDF_VALUE);
        break;
      case 4:
        StrRef = STRING_TOKEN (STR_RP5_BDF_VALUE);
        break;
      case 5:
        StrRef = STRING_TOKEN (STR_RP6_BDF_VALUE);
        break;
      case 6:
        StrRef = STRING_TOKEN (STR_RP7_BDF_VALUE);
        break;
      case 7:
        StrRef = STRING_TOKEN (STR_RP8_BDF_VALUE);
        break;
      case 8:
        StrRef = STRING_TOKEN (STR_RP9_BDF_VALUE);
        break;
      case 9:
        StrRef = STRING_TOKEN (STR_RP10_BDF_VALUE);
        break;
      case 10:
        StrRef = STRING_TOKEN (STR_RP11_BDF_VALUE);
        break;
      case 11:
        StrRef = STRING_TOKEN (STR_RP12_BDF_VALUE);
        break;
      case 12:
        StrRef = STRING_TOKEN (STR_RP13_BDF_VALUE);
        break;
      case 13:
        StrRef = STRING_TOKEN (STR_RP14_BDF_VALUE);
        break;
      case 14:
        StrRef = STRING_TOKEN (STR_RP15_BDF_VALUE);
        break;
      case 15:
        StrRef = STRING_TOKEN (STR_RP16_BDF_VALUE);
        break;
      case 16:
        StrRef = STRING_TOKEN (STR_RP17_BDF_VALUE);
        break;
      case 17:
        StrRef = STRING_TOKEN (STR_RP18_BDF_VALUE);
        break;
      case 18:
        StrRef = STRING_TOKEN (STR_RP19_BDF_VALUE);
        break;
      case 19:
        StrRef = STRING_TOKEN (STR_RP20_BDF_VALUE);
        break;
      case 20:
        StrRef = STRING_TOKEN (STR_RP21_BDF_VALUE);
        break;
      case 21:
        StrRef = STRING_TOKEN (STR_RP22_BDF_VALUE);
        break;
      case 22:
        StrRef = STRING_TOKEN (STR_RP23_BDF_VALUE);
        break;
      case 23:
        StrRef = STRING_TOKEN (STR_RP24_BDF_VALUE);
        break;
      case 24:
        StrRef = STRING_TOKEN (STR_RP25_BDF_VALUE);
        break;
      case 25:
        StrRef = STRING_TOKEN (STR_RP26_BDF_VALUE);
        break;
      case 26:
        StrRef = STRING_TOKEN (STR_RP27_BDF_VALUE);
        break;
      case 27:
        StrRef = STRING_TOKEN (STR_RP28_BDF_VALUE);
        break;
      case 28:
        StrRef = STRING_TOKEN (STR_RP29_BDF_VALUE);
        break;
      case 29:
        StrRef = STRING_TOKEN (STR_RP30_BDF_VALUE);
        break;
      case 30:
        StrRef = STRING_TOKEN (STR_RP31_BDF_VALUE);
        break;
      default:
        StrRef = STRING_TOKEN (STR_RP1_BDF_VALUE);
    }
    if (VmdInfoHob->VmdPortInfo.PortInfo[Index].RpDev == SataDevNumber (0)) {
      InitString (
        HiiHandle,
        StrRef,
        L"%s", L"SATA Controller"
        );
    } else {
    InitString (
      HiiHandle,
      StrRef,
      L"%d/%d/%d",
      0,
      VmdInfoHob->VmdPortInfo.PortInfo[Index].RpDev,
      VmdInfoHob->VmdPortInfo.PortInfo[Index].RpFunc
      );
    }
  }
  return EFI_SUCCESS;
}
