/** @file
  Thunderbolt(TM) Setup Rountines.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2010 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/HiiLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/DebugLib.h>
#include "Base.h"
#include <Library/PciSegmentLib.h>
#include <SetupPrivate.h>
#include <TcssInfo.h>
#include <PcieRegs.h>
#include "PlatformBoardId.h"
#include <Library/TbtCommonLib.h>
#include <Guid/HobList.h>
#include <Library/TimerLib.h>
#include <Library/PcdLib.h>

#define LTSSM_FSM_RESTORE 0xC74
#define TBT_5S_TIMEOUT                    50

GLOBAL_REMOVE_IF_UNREFERENCED BOOLEAN         Entry = FALSE;

EFI_STATUS
EFIAPI
TbtFormCallBackFunction (
  IN CONST EFI_HII_CONFIG_ACCESS_PROTOCOL *This,
  IN EFI_BROWSER_ACTION                   Action,
  IN EFI_QUESTION_ID                      KeyValue,
  IN UINT8                                Type,
  IN EFI_IFR_TYPE_VALUE                   *Value,
  OUT EFI_BROWSER_ACTION_REQUEST          *ActionRequest
  )
{
  SA_SETUP                    *SaSetup;
  PCH_SETUP                   *PchSetup;
  SETUP_DATA                  *SetupData;
  UINTN                       VarSize;
  EFI_STATUS                  Status;
  volatile EFI_IFR_TYPE_VALUE lValue;
  BOOLEAN                     SaveOnExit;
  EFI_STRING                  RequestString;
  EFI_STRING                  SaRequestString;
  EFI_STRING                  PchRequestString;
  UINT8                       Rootportselected;
  UINT8                       Index;
  UINTN                       RpDev;
  UINTN                       RpFunc;

  Status                  = EFI_SUCCESS;
  SaveOnExit              = FALSE;
  RequestString           = NULL;
  SaRequestString         = NULL;
  PchRequestString        = NULL;
  Rootportselected        = 0;

  DEBUG ((DEBUG_INFO, "TbtFormCallBackFunction \n"));

  if (Action != EFI_BROWSER_ACTION_CHANGING && Action != EFI_BROWSER_ACTION_CHANGED)
    return EFI_UNSUPPORTED;

  if (Action == EFI_BROWSER_ACTION_CHANGING)
    return EFI_SUCCESS;

  if ((Value == NULL) || (ActionRequest == NULL))
    return EFI_INVALID_PARAMETER;

  VarSize = sizeof (SETUP_DATA);
  SetupData = AllocatePool (VarSize);
  ASSERT (SetupData != NULL);
  if (SetupData == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  if (!HiiGetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  VarSize = sizeof (SA_SETUP);
  SaSetup = AllocatePool (VarSize);
  ASSERT (SaSetup != NULL);
  if (SaSetup == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  if (!HiiGetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  VarSize = sizeof (PCH_SETUP);
  PchSetup = AllocatePool (VarSize);
  ASSERT (PchSetup != NULL);
  if (PchSetup == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  if (!HiiGetBrowserData (&gPchSetupVariableGuid, L"PchSetup", VarSize, (UINT8 *) PchSetup)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  IoWrite16(0x80, (UINT16)KeyValue);

  lValue = *Value;

  DEBUG ((DEBUG_INFO, "Thunderbolt(TM) call back actions begin \n"));

  switch (KeyValue) {
   case KEY_ITBT_SUPPORT:
     DEBUG ((DEBUG_INFO, "KEY_ITBT_SUPPORT CallBack \n"));
      if (SetupData->IntegratedTbtSupport == 1) {
        SetupData->TbtVtdBaseSecurity = 1;
        SetupData->ITbtRtd3 = 1;
      } else {
        for (Index = 0; Index < MAX_ITBT_PCIE_PORT; Index++) {
          SetupData->ITbtRootPort[Index] = 0;
          RequestString   = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRootPort[Index]), sizeof (SetupData->ITbtRootPort[Index]));
        }
        SetupData->TbtVtdBaseSecurity = 0;
        SetupData->ITbtRtd3 = 0;
      }
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, TbtVtdBaseSecurity), sizeof (SetupData->TbtVtdBaseSecurity));
      RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, ITbtRtd3), sizeof (SetupData->ITbtRtd3));
   break;
   case KEY_TBT_BOOT_ON:
   case KEY_TBT_USB_ON:
    if ((SetupData->TbtBootOn == 1) || (SetupData->TbtUsbOn == 1)) {
      SetupData->TbtSkipPciOprom = 0;
    }
    else {
      SetupData->TbtSkipPciOprom = 1;
    }
    RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, TbtSkipPciOprom), sizeof (SetupData->TbtSkipPciOprom));
    break;

    case TBT_SECURITY_LEVEL_KEY:
      DEBUG ((EFI_D_INFO, "TBT_SECURITY_LEVEL_KEY CallBack \n"));
      if (SetupData->IntegratedTbtSupport == 1) {
        for (Index = 0; Index < MAX_ITBT_PCIE_PORT; Index = Index + 2) {
          if ((SetupData->ITbtRootPort[Index] == 0x01) ||
              (SetupData->ITbtRootPort[Index + 1] == 0x01)) {

            Status = GetITbtDmaDevFun((Index/2), 0, 0, &RpDev, &RpFunc);
            if (EFI_ERROR (Status)) {
              DEBUG((DEBUG_INFO, "ITBT DMA input invalid \n"));
              continue;
            }
            DEBUG ((DEBUG_INFO, "ITBT DMA = 0x%x Device = 0x%x Function = 0x%x \n", Index/2,RpDev,RpFunc));
            if (TbtSetSecLevel (lValue.u8, FixedPcdGet8 (PcdITbtToPcieRegister), FixedPcdGet8 (PcdPcieToITbtRegister), 0, (UINT8)RpDev, (UINT8)RpFunc, PCIE2TBT_SET_SECURITY_LEVEL, TBT_5S_TIMEOUT) && (lValue.u8 == TbtGetSecLevel (FixedPcdGet8 (PcdITbtToPcieRegister), FixedPcdGet8 (PcdPcieToITbtRegister), 0, (UINT8)RpDev, (UINT8) RpFunc, PCIE2TBT_GET_SECURITY_LEVEL, TBT_5S_TIMEOUT))) {
              SetupData->SecurityMode = lValue.u8;
              SaveOnExit = TRUE;
              //If selected security level is SL3 (Display Port and USB), BIOS should disable "Boot from  Thunderbolt" and "Boot from USB/USB at boot"
              if (SetupData->SecurityMode == 3) {
                SetupData->TbtBootOn = 0;
                SetupData->TbtSkipPciOprom = 1;
                RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, TbtBootOn), sizeof (SetupData->TbtBootOn));
                RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SETUP_DATA, TbtSkipPciOprom), sizeof (SetupData->TbtSkipPciOprom));
              }
            } else {
              SetupData->SecurityMode = 0;
            }
          }
        }
      }
    break;

    default:
      return EFI_SUCCESS;
  }

  DEBUG ((DEBUG_INFO, "Thunderbolt(TM) call back - HiiSetBrowserData begin \n"));
  if (RequestString != NULL) {
    VarSize = sizeof (SETUP_DATA);
    if (!HiiSetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData, RequestString)) {
      Status = EFI_NOT_FOUND;
    }
    ASSERT_EFI_ERROR (Status);
    FreePool (RequestString);
  }
  if (SaRequestString != NULL) {
    VarSize = sizeof (SA_SETUP);
    if (!HiiSetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup, SaRequestString)) {
      Status = EFI_NOT_FOUND;
    }
    ASSERT_EFI_ERROR (Status);
    FreePool (SaRequestString);
  }
  if (PchRequestString != NULL) {
    VarSize = sizeof (PCH_SETUP);
    if (!HiiSetBrowserData (&gPchSetupVariableGuid, L"PchSetup", VarSize, (UINT8 *) PchSetup, PchRequestString)) {
      Status = EFI_NOT_FOUND;
    }
    ASSERT_EFI_ERROR (Status);
    FreePool (PchRequestString);
  }
  DEBUG ((DEBUG_INFO, "Thunderbolt(TM) call back - HiiSetBrowserData end \n"));
  FreePool (SaSetup);
  FreePool (PchSetup);
  FreePool (SetupData);

  *ActionRequest = EFI_BROWSER_ACTION_REQUEST_NONE;
  if (SaveOnExit)
    *ActionRequest = EFI_BROWSER_ACTION_REQUEST_SUBMIT;

  DEBUG ((DEBUG_INFO, "Thunderbolt(TM) call back actions end \n"));

  return EFI_SUCCESS;
}

EFI_STATUS
EFIAPI
TbtOsSelectorFormCallBackFunction (
  IN CONST EFI_HII_CONFIG_ACCESS_PROTOCOL *This,
  IN EFI_BROWSER_ACTION                   Action,
  IN EFI_QUESTION_ID                      KeyValue,
  IN UINT8                                Type,
  IN EFI_IFR_TYPE_VALUE                   *Value,
  OUT EFI_BROWSER_ACTION_REQUEST          *ActionRequest
  )
{
  SA_SETUP                   *SaSetup;
  PCH_SETUP                  *PchSetup;
  SETUP_DATA                 *SetupData;
  EFI_STRING                  RequestString;
  EFI_STRING                  SaRequestString;
  EFI_STRING                  PchRequestString;
  UINTN                       VarSize;
  EFI_STATUS                  Status;
  BOOLEAN                     SaveOnExit;
  UINT32                      Timeout;
  volatile EFI_IFR_TYPE_VALUE lValue;

  Timeout = 50;
  SaveOnExit = FALSE;
  Status = EFI_SUCCESS;
  RequestString = NULL;
  SaRequestString = NULL;
  PchRequestString = NULL;

  DEBUG ((DEBUG_INFO, "TbtOsSelectorFormCallBackFunction \n"));
  if ((Value == NULL) || (ActionRequest == NULL)) {
    return EFI_INVALID_PARAMETER;
  }

  VarSize = sizeof (SETUP_DATA);
  SetupData = AllocatePool (VarSize);
  ASSERT (SetupData != NULL);
  if (SetupData == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  if (!HiiGetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  VarSize = sizeof (SA_SETUP);
  SaSetup = AllocatePool (VarSize);
  ASSERT (SaSetup != NULL);
  if (SaSetup == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  if (!HiiGetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  VarSize = sizeof (PCH_SETUP);
  PchSetup = AllocatePool (VarSize);
  ASSERT (PchSetup != NULL);
  if (PchSetup == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  if (!HiiGetBrowserData (&gPchSetupVariableGuid, L"PchSetup", VarSize, (UINT8 *) PchSetup)) {
    Status = EFI_NOT_FOUND;
  }
  ASSERT_EFI_ERROR (Status);

  lValue = *Value;

  if (Action != EFI_BROWSER_ACTION_CHANGING && Action != EFI_BROWSER_ACTION_CHANGED)
    return EFI_UNSUPPORTED;

  if (Action == EFI_BROWSER_ACTION_CHANGING)
    return EFI_SUCCESS;

  if ((Value == NULL) || (ActionRequest == NULL))
    return EFI_INVALID_PARAMETER;

  switch (KeyValue) {
    case KEY_TBT_OS_SELECTOR:
      DEBUG ((DEBUG_INFO, "KEY_TBT_OS_SELECTOR CallBack \n"));
      break;
    default:
      return EFI_SUCCESS;
  }

  if (RequestString != NULL) {
    VarSize = sizeof (SETUP_DATA);
    if (!HiiSetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData, RequestString)) {
      Status = EFI_NOT_FOUND;
    }
    ASSERT_EFI_ERROR (Status);
    FreePool (RequestString);
  }
  if (SaRequestString != NULL) {
    VarSize = sizeof (SA_SETUP);
    if (!HiiSetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup, SaRequestString)) {
      Status = EFI_NOT_FOUND;
    }
    ASSERT_EFI_ERROR (Status);
    FreePool (SaRequestString);
  }
  if (PchRequestString != NULL) {
    VarSize = sizeof (PCH_SETUP);
    if (!HiiSetBrowserData (&gPchSetupVariableGuid, L"PchSetup", VarSize, (UINT8 *) PchSetup, PchRequestString)) {
      Status = EFI_NOT_FOUND;
    }
    ASSERT_EFI_ERROR (Status);
    FreePool (PchRequestString);
  }
  DEBUG ((DEBUG_INFO, "Thunderbolt(TM) call back - HiiSetBrowserData end \n"));
  FreePool (SaSetup);
  FreePool (PchSetup);
  FreePool (SetupData);

  *ActionRequest = EFI_BROWSER_ACTION_REQUEST_NONE;
  if (SaveOnExit)
    *ActionRequest = EFI_BROWSER_ACTION_REQUEST_SUBMIT;

  DEBUG ((DEBUG_INFO, "Thunderbolt(TM) call back actions end \n"));

  return EFI_SUCCESS;
}

/**
  Get uncommitted Vtd  Browser status from BIOS setup menu to check Vtd availability

  @retval EFI_SUCCESS           Vt-d is available.
  @retval EFI_UNSUPPORTED       Vt-d is not available.
  @retval EFI_OUT_OF_RESOURCES  Do not have enough resources to get buffer.
**/
EFI_STATUS
GetVtdStatusfromBrowser (
  VOID
  )
{
  SETUP_VOLATILE_DATA         *SetupVolatileData;
  UINTN                       VarSize;
  EFI_STATUS                  Status;

  //
  // Initialization
  //
  SetupVolatileData           = NULL;
  Status                      = EFI_SUCCESS;

  //
  // Allocate memory for SetupVolatileData
  //
  VarSize = sizeof (SETUP_VOLATILE_DATA);
  SetupVolatileData = AllocateZeroPool (VarSize);

  if (SetupVolatileData == NULL) {
    DEBUG ((DEBUG_ERROR, "Allocate SetupVolatileData size fail\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

  //
  // Retrive uncommitted SetupVolatileData data from Browser
  //
  if (FALSE == HiiGetBrowserData (&gSetupVariableGuid, L"SetupVolatileData", VarSize, (UINT8 *) SetupVolatileData)) {
    DEBUG ((DEBUG_ERROR, "Not able to retrive SetupVolatileData\n"));
    Status = EFI_INVALID_PARAMETER;
    goto Exit;
  }

  //
  // Return VTd Support
  //
  if (SetupVolatileData->VTdAvailable == TRUE) {
    Status = EFI_SUCCESS;
  } else {
    Status = EFI_UNSUPPORTED;
  }

Exit:
  if (SetupVolatileData != NULL) {
    FreePool (SetupVolatileData);
  }
  return Status;
}


/**
  Get uncommitted Vtd TBT Base Security Browser value from BIOS setup menu

  @retval EFI_SUCCESS           TBT Base Security enable.
  @retval EFI_OUT_OF_RESOURCES  Do not have enough resources to get buffer.
  @retval EFI_UNSUPPORTED       TBT Base Security diable.
**/
EFI_STATUS
BaseSecurityGetBrowserValue (
  VOID
  )
{
  SETUP_DATA                  *SetupData;
  UINTN                       VarSize;
  EFI_STATUS                  Status;

  //
  // Initialization
  //
  SetupData = NULL;
  Status = EFI_SUCCESS;

  //
  // Allocate memory for SetupData
  //
  VarSize = sizeof (SETUP_DATA);
  SetupData = AllocateZeroPool (VarSize);
  if (SetupData == NULL) {
    DEBUG ((DEBUG_ERROR, "Failed to allocate SetupData size\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

  //
  // Retrive uncommitted SetupData data from Browser
  //
  if (FALSE == HiiGetBrowserData (&gSetupVariableGuid, L"Setup", VarSize, (UINT8 *) SetupData)) {
    DEBUG ((DEBUG_ERROR, "Failed to retrive SetupData\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

  if (SetupData->TbtVtdBaseSecurity == FALSE) {
    Status = EFI_UNSUPPORTED;
  } else {
    Status = EFI_SUCCESS;
  }

Exit:
  if (SetupData != NULL) {
    FreePool (SetupData);
  }
  return Status;
}

/**
  Setup VTD Setup option value
  @param[in]  BOOLEAN Config

  @retval EFI_SUCCESS           Successfully disable Legacy USB.
  @retval EFI_OUT_OF_RESOURCES  Do not have enough resources to get buffer.
  @retval EFI_INVALID_PARAMETER Failed to disable Legacy USB.
  @retval EFI_UNSUPPORTED       Vt-d is not available.
**/
EFI_STATUS
SetVtdFromBrowser (
  IN BOOLEAN Config
  )
{
  SA_SETUP                    *SaSetup;
  UINTN                       VarSize;
  EFI_STATUS                  Status;
  EFI_STRING                  RequestString;

  //
  // Initialization
  //
  SaSetup = NULL;
  Status = EFI_SUCCESS;
  RequestString = NULL;

  if (GetVtdStatusfromBrowser() == EFI_UNSUPPORTED) {
    DEBUG ((DEBUG_ERROR, "Vtd Support is unavailable\n"));
    Status = EFI_UNSUPPORTED;
    goto Exit;
  }

  //
  // Allocate memory for SaSetup
  //
  VarSize = sizeof (SA_SETUP);
  SaSetup = AllocateZeroPool (VarSize);
  if (SaSetup == NULL) {
    DEBUG ((DEBUG_ERROR, "Failed to allocate SaSetup size\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

  //
  // Retrive uncommitted SaSetup data from Browser
  //
  if (FALSE == HiiGetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup)) {
    DEBUG ((DEBUG_ERROR, "Failed to retrive SaSetup\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

  //
  // Config Vtd Browser setup option
  //
  if (Config == TRUE) {
    SaSetup->EnableVtd = TRUE;
  } else {
    SaSetup->EnableVtd = FALSE;
  }

  RequestString = HiiConstructRequestString (RequestString, OFFSET_OF (SA_SETUP, EnableVtd), sizeof (SaSetup->EnableVtd));
  if (RequestString != NULL) {
    //
    // Pass changed uncommitted data back to Form Browser
    //
    VarSize = sizeof (SA_SETUP);
    if (FALSE == HiiSetBrowserData (&gSaSetupVariableGuid, L"SaSetup", VarSize, (UINT8 *) SaSetup, RequestString)) {
      DEBUG ((DEBUG_INFO, "Failed to config Vtd\n"));
      Status = EFI_INVALID_PARAMETER;
      goto Exit;
    }

    DEBUG ((DEBUG_INFO, "Successfully config Vtd\n"));
  }

Exit:
  if (RequestString != NULL) {
    FreePool (RequestString);
  }
  if (SaSetup != NULL) {
    FreePool (SaSetup);
  }

  return Status;
}



/**
  TBTSecurityCallBackFunction
  @param[in] This             Points to the EFI_HII_CONFIG_ACCESS_PROTOCOL.
  @param[in] Action           Specifies the type of action taken by the browser.
  @param[in] KeyValue         A unique value which is sent to the original exporting driver
                              so that it can identify the type of data to expect of data to expect.
  @param[in] Type             The type of value for the question.
  @param[in] Value            A pointer to the data being sent to the original exporting driver.

  @retval EFI_SUCCESS         The callback successfully handled the action
  @retval EFI_UNSUPPORTED     The specified action is not supported by the callback
  @retval EFI_INVALID_PARAMETER Failed to disable Legacy USB.
**/
EFI_STATUS
EFIAPI
TBTSecurityCallBackFunction (
  IN CONST EFI_HII_CONFIG_ACCESS_PROTOCOL *This,
  IN EFI_BROWSER_ACTION                   Action,
  IN EFI_QUESTION_ID                      KeyValue,
  IN UINT8                                Type,
  IN EFI_IFR_TYPE_VALUE                   *Value,
  OUT EFI_BROWSER_ACTION_REQUEST          *ActionRequest
  )
{
  SETUP_VOLATILE_DATA         *SetupVolatileData;
  UINTN                       VarSize;
  EFI_STATUS                  Status;
  UINT32                      SetupVolAttributes;

  //
  // Initialization
  //
  VarSize                     = 0;
  Status                      = EFI_SUCCESS;
  SetupVolatileData           = NULL;
  SetupVolAttributes          = 0;

  if (Action != EFI_BROWSER_ACTION_CHANGING && Action != EFI_BROWSER_ACTION_CHANGED) {
    return EFI_UNSUPPORTED;
  }

  if (Action == EFI_BROWSER_ACTION_CHANGING) {
    return EFI_SUCCESS;
  }

  if ((Value == NULL) || (ActionRequest == NULL)) {
    return EFI_INVALID_PARAMETER;
  }

  DEBUG ((DEBUG_INFO, "[TBT] TBTSecurityCallBackFunction START\n"));
  IoWrite16 (0x80, (UINT16)KeyValue);
  //
  // Allocate memory for SetupVolatileData
  //
  VarSize = sizeof (SETUP_VOLATILE_DATA);
  SetupVolatileData = AllocateZeroPool (VarSize);

  if (SetupVolatileData == NULL) {
    DEBUG ((DEBUG_ERROR, "Allocate SetupVolatileData size fail\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

  //
  // Retrive SetupVolatileData data from memory
  //
  Status = gRT->GetVariable (
                    L"SetupVolatileData",
                    &gSetupVariableGuid,
                    &SetupVolAttributes,
                    &VarSize,
                    SetupVolatileData
                    );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Not able to retrive SetupVolatileData from Nvram\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

  //
  // Retrive uncommitted BaseSecurity data from Browser
  //
  if (BaseSecurityGetBrowserValue() == EFI_SUCCESS) {
    DEBUG ((DEBUG_INFO, "User enable TbtVtdBaseSecurity. Set and grayout options\n"));
    SetupVolatileData->TbtVtdBaseSecurityGrayOut = TRUE;
    //
    // Enable VTd from browser
    //
    SetVtdFromBrowser (TRUE);

  } else {
    DEBUG ((DEBUG_INFO, "User disable TbtVtdBaseSecurity\n"));
    SetupVolatileData->TbtVtdBaseSecurityGrayOut = FALSE;
  }

  //
  // Save SetupVolatileData data back to memory
  //
  Status = gRT->SetVariable (
                    L"SetupVolatileData",
                    &gSetupVariableGuid,
                    SetupVolAttributes,
                    VarSize,
                    SetupVolatileData
                    );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "Not able to save SetupVolatileData\n"));
    Status = EFI_OUT_OF_RESOURCES;
    goto Exit;
  }

Exit:
  if (SetupVolatileData != NULL) {
    FreePool (SetupVolatileData);
  }

  DEBUG ((DEBUG_INFO, "[TBT] TBTSecurityCallBackFunction END\n"));
  return Status;
}
