/** @file
  CPU Platform Lib implementation.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2012 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include "CpuPlatformLibrary.h"
#include <Library/PciSegmentLib.h>
#include <Register/SaRegsHostBridge.h>
#include <Register/IgdRegs.h>
#include <Library/CpuMailboxLib.h>
#include <Library/CpuInfoFruLib.h>
#include <Register/CommonMsr.h>
#include <Register/ArchMsr.h>
#include <Library/MsrFruLib.h>
#include <CpuGenInfoFruLib.h>

#define SKIP_MICROCODE_CHECKSUM_CHECK 1

/**
  Return CPU Family ID

  @retval CPU_FAMILY              CPU Family ID
**/
CPU_FAMILY
EFIAPI
GetCpuFamily (
  VOID
  )
{
  CPUID_VERSION_INFO_EAX  Eax;
  ///
  /// Read the CPUID information
  ///
  AsmCpuid (CPUID_VERSION_INFO, &Eax.Uint32, NULL, NULL, NULL);
  return (Eax.Uint32 & CPUID_FULL_FAMILY_MODEL);
}

/**
  Return Cpu stepping type

  @retval UINT8                   Cpu stepping type
**/
CPU_STEPPING
EFIAPI
GetCpuStepping (
  VOID
  )
{
  CPUID_VERSION_INFO_EAX  Eax;
  ///
  /// Read the CPUID information
  ///
  AsmCpuid (CPUID_VERSION_INFO, &Eax.Uint32, NULL, NULL, NULL);
  return ((CPU_STEPPING) (Eax.Uint32 & CPUID_FULL_STEPPING));
}

/**
  Return CPU Sku

  @retval UINT8              CPU Sku
**/
UINT8
EFIAPI
GetCpuSku (
  VOID
  )
{
  UINT16                  CpuDid;
  UINT32                  CpuFamilyModel;
  CPUID_VERSION_INFO_EAX  Eax;

  ///
  /// Read the CPUID & DID information
  ///
  AsmCpuid (CPUID_VERSION_INFO, &Eax.Uint32, NULL, NULL, NULL);
  CpuFamilyModel = Eax.Uint32 & CPUID_FULL_FAMILY_MODEL;
  CpuDid = PciSegmentRead16 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_MC_DEVICE_ID));

  return GetCpuSkuInfo (CpuFamilyModel, CpuDid);

}

/**
  Check on the processor if SMX (Safer Mode Extensions) is supported.

  @retval  TRUE        CPU supports SMX.
  @retval  FALSE       CPU do not support SMX.
**/
BOOLEAN
IsSmxSupported (
  VOID
  )
{
  CPUID_VERSION_INFO_ECX   Ecx;

  ///
  /// Read CPUID Version Information returned in ECX for CPUID leaf.
  ///
  AsmCpuid (CPUID_VERSION_INFO, NULL, NULL, &Ecx.Uint32, NULL);
  if (Ecx.Bits.SMX == 1) {
    return TRUE;
  }

  return FALSE;
}

/**
  Return CPU Identifier used to identify various CPU types

  @retval CPU_IDENTIFIER           CPU Identifier
**/
UINT32
GetCpuIdentifier (
  )
{
  CPU_SKU                               CpuSku;
  CPU_STEPPING                          CpuStepping;

  ///
  /// Initialize local variables
  ///
  CpuSku    = GetCpuSku ();
  CpuStepping = GetCpuStepping ();
  return (UINT32) GetCpuSkuIdentifier (CpuSku, CpuStepping, 0);
}

/**
  Return CPU Identifier used to identify various CPU types
  @param[in] SelectedCtdpLevel    - Ctdp Level Selected in BIOS

  @retval CPU_IDENTIFIER           CPU Identifier
**/
UINT32
GetCpuIdentifierWithCtdp (
  IN UINT16               SelectedCtdpLevel
  )
{
  CPU_SKU                               CpuSku;
  CPU_STEPPING                          CpuStepping;
  ///
  /// Initialize local variables
  ///
  CpuSku    = GetCpuSku ();
  CpuStepping = GetCpuStepping ();
  return (UINT32) GetCpuSkuIdentifier (CpuSku, CpuStepping, SelectedCtdpLevel);
}

/**
  Returns the processor microcode revision of the processor installed in the system.

  @retval Processor Microcode Revision
**/
UINT32
GetCpuUcodeRevision (
  VOID
  )
{
  MSR_IA32_BIOS_SIGN_ID_REGISTER  BiosSignMsr;

  AsmWriteMsr64 (MSR_IA32_BIOS_SIGN_ID, 0);
  AsmCpuid (CPUID_VERSION_INFO, NULL, NULL, NULL, NULL);
  BiosSignMsr.Uint64 = AsmReadMsr64 (MSR_IA32_BIOS_SIGN_ID);

  return BiosSignMsr.Bits.MicrocodeUpdateSignature;
}

/**
  This function checks the MCU revision to decide if BIOS needs to load
  microcode.

  @param[in] MicrocodePointer - Microcode in memory
  @param[in] Revision         - Current CPU microcode revision

  @retval EFI_SUCCESS - BIOS needs to load microcode
  @retval EFI_ABORTED - Don't need to update microcode
**/
EFI_STATUS
CheckMcuRevision (
  IN CPU_MICROCODE_HEADER *MicrocodePointer,
  IN UINT32               Revision
  )
{
  EFI_STATUS Status;
  Status = EFI_ABORTED;

  if ((MicrocodePointer->UpdateRevision & 0x80000000) ||
      (MicrocodePointer->UpdateRevision >= Revision) ||
      (Revision == 0)) {
    Status = EFI_SUCCESS;
  }

  return Status;
}

/**
  Check if this microcode is correct one for processor

  @param[in] Cpuid               - processor CPUID
  @param[in] MicrocodeEntryPoint - entry point of microcode
  @param[in] Revision            - revision of microcode

  @retval CorrectMicrocode if this microcode is correct
**/
BOOLEAN
CheckMicrocode (
  IN UINT32               Cpuid,
  IN CPU_MICROCODE_HEADER *MicrocodeEntryPoint,
  IN UINT32               *Revision
  )
{
  UINT32                              CheckSum32;
  UINT8                               ExtendedIndex;
  MSR_IA32_PLATFORM_ID_REGISTER       Msr;
  UINT32                              ExtendedTableLength;
  UINT32                              ExtendedTableCount;
  BOOLEAN                             CorrectMicrocode;
  CPU_MICROCODE_EXTENDED_TABLE        *ExtendedTable;
  CPU_MICROCODE_EXTENDED_TABLE_HEADER *ExtendedTableHeader;

  CheckSum32          = 0;
  ExtendedTableLength = 0;
  CorrectMicrocode    = FALSE;

  if (MicrocodeEntryPoint == NULL) {
    return FALSE;
  }

  Msr.Uint64 = AsmReadMsr64 (MSR_IA32_PLATFORM_ID);

  ///
  /// Check if the microcode is for the Cpu and the version is newer
  /// and the update can be processed on the platform
  ///
  if ((MicrocodeEntryPoint->HeaderVersion == 0x00000001) &&
      !EFI_ERROR (CheckMcuRevision (MicrocodeEntryPoint, *Revision))
      ) {
    if ((MicrocodeEntryPoint->ProcessorId == Cpuid) && (MicrocodeEntryPoint->ProcessorFlags & (1 << (UINT8) Msr.Bits.PlatformId))) {
    #if SKIP_MICROCODE_CHECKSUM_CHECK
      CheckSum32 = 0;
    #else
      if (MicrocodeEntryPoint->DataSize == 0) {
        CheckSum32 = CalculateSum32 ((UINT32 *) MicrocodeEntryPoint, 2048);
      } else {
        CheckSum32 = CalculateSum32 (
                   (UINT32 *) MicrocodeEntryPoint,
                   MicrocodeEntryPoint->DataSize + sizeof (CPU_MICROCODE_HEADER)
                   );
      }
    #endif
      if (CheckSum32 == 0) {
        CorrectMicrocode = TRUE;
      }
    } else if ((MicrocodeEntryPoint->DataSize != 0)) {
      ///
      /// Check the  Extended Signature if the entended signature exist
      /// Only the data size != 0 the extended signature may exist
      ///
      ExtendedTableLength = MicrocodeEntryPoint->TotalSize - (MicrocodeEntryPoint->DataSize + sizeof (CPU_MICROCODE_HEADER));
      if (ExtendedTableLength != 0) {
        ///
        /// Extended Table exist, check if the CPU in support list
        ///
        ExtendedTableHeader = (CPU_MICROCODE_EXTENDED_TABLE_HEADER *) ((UINT8 *) (MicrocodeEntryPoint) + MicrocodeEntryPoint->DataSize + 48);
        ///
        /// Calulate Extended Checksum
        ///
        if ((ExtendedTableLength % 4) == 0) {
          #if SKIP_MICROCODE_CHECKSUM_CHECK
            CheckSum32 = 0;
          #else
            CheckSum32 = CalculateSum32 ((UINT32 *) ExtendedTableHeader, ExtendedTableLength);
          #endif
          if (CheckSum32 == 0) {
            ///
            /// Checksum correct
            ///
            ExtendedTableCount  = ExtendedTableHeader->ExtendedSignatureCount;
            ExtendedTable       = (CPU_MICROCODE_EXTENDED_TABLE *) (ExtendedTableHeader + 1);
            for (ExtendedIndex = 0; ExtendedIndex < ExtendedTableCount; ExtendedIndex++) {
              ///
              /// Verify Header
              ///
              if ((ExtendedTable->ProcessorSignature == Cpuid) && (ExtendedTable->ProcessorFlag & (1 << (UINT8) Msr.Bits.PlatformId))) {
                #if SKIP_MICROCODE_CHECKSUM_CHECK
                  CheckSum32 = 0;
                #else
                  CheckSum32 = CalculateSum32 (
                             (UINT32 *) ExtendedTable,
                             sizeof (CPU_MICROCODE_EXTENDED_TABLE)
                             );
                #endif
                if (CheckSum32 == 0) {
                  ///
                  /// Find one
                  ///
                  CorrectMicrocode = TRUE;
                  break;
                }
              }

              ExtendedTable++;
            }
          }
        }
      }
    }
  }

  return CorrectMicrocode;
}

/**
  This function is to program Trace Hub ACPI base address to processor's MSR TRACE_HUB_STH_ACPIBAR_BASE.

  @param[in]  TraceHubAcpiBase - Base address of Trace Hub ACPI Base address
**/
VOID
EFIAPI
CpuWriteTraceHubAcpiBase (
  IN UINT64  TraceHubAcpiBase
  )
{
  MSR_NPK_STH_ACPIBAR_BASE_REGISTER BaseAddressMask;
  MSR_NPK_STH_ACPIBAR_BASE_REGISTER MsrTraceHubAcpiBase;

  //
  // Make up Trace Hub ACPI Base address mask.
  //
  BaseAddressMask.Uint64 = MAX_UINT64;
  BaseAddressMask.Bits.Lock = 0;
  BaseAddressMask.Bits.Rsvd1 = 0;

  //
  // Check the pass in Trace Hub ACPI base if 256KB alignment.
  //
  if ((TraceHubAcpiBase & ~BaseAddressMask.Uint64) != 0) {
    ASSERT ((TraceHubAcpiBase & ~BaseAddressMask.Uint64) == 0);
    return;
  }

  ///
  /// Set MSR TRACE_HUB_STH_ACPIBAR_BASE[0] LOCK bit for the AET packets to be directed to NPK MMIO.
  ///
  MsrTraceHubAcpiBase.Uint64 = TraceHubAcpiBase;
  MsrTraceHubAcpiBase.Bits.Lock = 1;
  AsmWriteMsr64 (MSR_NPK_STH_ACPIBAR_BASE, MsrTraceHubAcpiBase.Uint64);

  return;
}

/**
  Check on the processor if SGX is supported.

  @dot
    digraph G {
      subgraph cluster_c0 {
        node [shape = box];
          b1[label="Read CPUID(EAX=7,ECX=0):EBX[2] \nto check SGX feature" fontsize=12 style=filled color=lightblue];
          b2[label="Return TRUE" fontsize=12 style=filled color=lightblue];
          b3[label="Return FALSE" fontsize=12 style=filled color=lightblue];

        node [shape = ellipse];
          e1[label="Start" fontsize=12 style=filled color=lightblue];
          e2[label="End" fontsize=12 style=filled color=lightblue];

        node [shape = diamond,style=filled,color=lightblue];
          d1[label="Are SGX feature supported and \nPRMRR configuration enabled" fontsize=12];

        label = "IsSgxSupported Flow"; fontsize=15; fontcolor=black; color=lightblue;
        e1 -> b1
        b1 -> d1
        d1 -> b2 [label="Yes" fontsize=9]
        d1 -> b3 [label="No" fontsize=9]
        b2 -> e2
        b3 -> e2

      }
    }
  @enddot

  @retval TRUE  if SGX supported
  @retval FALSE if SGX is not supported
**/
BOOLEAN
IsSgxSupported (
  VOID
  )
{
  CPUID_STRUCTURED_EXTENDED_FEATURE_FLAGS_EBX  Ebx;
  MSR_MTRRCAP_REGISTER                         Msr;

  //
  // Processor support SGX feature by reading CPUID.(EAX=7,ECX=0):EBX[2]
  //
  AsmCpuidEx (CPUID_STRUCTURED_EXTENDED_FEATURE_FLAGS, 0, NULL, &Ebx.Uint32,NULL,NULL);

  ///
  /// SGX feature is supported only with CPUID.(EAX=7,ECX=0):EBX[2]=1
  /// PRMRR configuration enabled, MSR IA32_MTRRCAP (FEh) [12] == 1
  ///
  Msr.Uint64 = AsmReadMsr64 (MSR_IA32_MTRRCAP);
  if ((Ebx.Bits.SGX) && (Msr.Bits.Prmrr)) {
    return TRUE;
  }
  return FALSE;
}

/**
  Detect if C6DRAM supported or not by reading it from PCODE mailbox.

  @retval TRUE - Supported
  @retval FALSE - Not Supported
**/
BOOLEAN
IsC6dramSupported (
  VOID
  )
{
  UINT32       MailboxStatus;
  UINT32       MailboxData;
  EFI_STATUS   Status;

  ///
  /// For C6DRAM, PCODE mailbox returns fuse_c6dram_en in bit 1.
  ///
  Status = MailboxRead (MAILBOX_TYPE_PCODE, MAILBOX_BIOS_CMD_READ_C6DRAM_CONFIG, &MailboxData , &MailboxStatus);
  if (Status != EFI_SUCCESS || MailboxStatus != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR, "Mailbox read command failed unexpectedly, C6DRAM is not supported. MailboxStatus = %x , Mailbox command return status %r\n", MailboxStatus, Status));
    return FALSE;
  }

  return (MailboxData & B_MAILBOX_C6DRAM_SUPPORTED) == B_MAILBOX_C6DRAM_SUPPORTED;
}

/**
  Get processor generation

  @retval CPU_GENERATION  Returns the executing thread's processor generation.
**/
CPU_GENERATION
GetCpuGeneration (
  VOID
  )
{
  CPU_FAMILY                 CpuFamilyModel;
  CPUID_VERSION_INFO_EAX     Eax;

  ///
  /// Read the CPUID information
  ///
  AsmCpuid (CPUID_VERSION_INFO, &Eax.Uint32, NULL, NULL, NULL);
  CpuFamilyModel = (CPU_FAMILY) (Eax.Uint32 & CPUID_FULL_FAMILY_MODEL);


  return GetCpuSkuGeneration (CpuFamilyModel);
}

/**
  Check if this is non-core processor - HT AP thread

  @retval TRUE if this is HT AP thread
  @retval FALSE if this is core thread
**/
BOOLEAN
IsSecondaryThread (
  VOID
  )
{
  CPUID_VERSION_INFO_EDX      CpuidVersionInfoEdx;
  CPUID_EXTENDED_TOPOLOGY_EAX CpuIdExtendedTopologyEax;
  UINT32                      ApicId;
  UINT32                      MaskShift;
  UINT32                      Mask;

  AsmCpuid (CPUID_VERSION_INFO, NULL, NULL, NULL, &CpuidVersionInfoEdx.Uint32);
  if (CpuidVersionInfoEdx.Bits.HTT == 0) {
    return FALSE;
  }

  AsmCpuidEx (
    CPUID_EXTENDED_TOPOLOGY,
    0,  // Sub-leaf 0
    &CpuIdExtendedTopologyEax.Uint32,
    NULL,
    NULL,
    &ApicId
    );

  MaskShift = CpuIdExtendedTopologyEax.Bits.ApicIdShift;
  Mask = ~(0xffffffff << MaskShift);

  return (ApicId & Mask) > 0;
}

/**
  This function returns Number of enabled cores in the package.

  @retval Number of enabled cores in the package.
**/

UINT16
GetEnabledCoreCount (
  VOID
  )
{
  MSR_CORE_THREAD_COUNT_REGISTER   MsrValue;

  MsrValue.Uint64 = AsmReadMsr64 (MSR_CORE_THREAD_COUNT);

  return (UINT16) MsrValue.Bits.Corecount;
}

/**
  This function returns Number of enabled Threads in the package.

  @retval Number of enabled threads in the package.
**/
UINT16
GetEnabledThreadCount (
  VOID
  )
{
  MSR_CORE_THREAD_COUNT_REGISTER   MsrValue;

  MsrValue.Uint64 = AsmReadMsr64 (MSR_CORE_THREAD_COUNT);

  return (UINT16) MsrValue.Bits.Threadcount;
}

/**
  This function checks for Whiskey Lake CPU presence.

  @retval TRUE  - WHL CPU Stepping W0 or V0 detected
  @retval FALSE - no WHL CPU detected
**/
BOOLEAN
IsWhlCpu (
  VOID
  )
{
  CPU_FAMILY   CpuFamily;
  CpuFamily   = GetCpuFamily ();

  switch (CpuFamily) {
    default:
      return FALSE;
  }
}

/**
  This function checks for Rocket Lake CPU presence.

  @retval TRUE  - RKL CPU detected
  @retval FALSE - no RKL CPU detected
**/
BOOLEAN
IsRklCpu (
  VOID
  )
{
  CPU_FAMILY   CpuFamily;
  CpuFamily   = GetCpuFamily ();

  switch (CpuFamily) {
    case EnumCpuRklUltUlx:
    case EnumCpuRklDtHalo:
      return TRUE;
    default:
      return FALSE;
  }
}

/**
  This function checks for CML CPU presence.

  @retval TRUE  - CML CPU detected
  @retval FALSE - no CML CPU detected
**/
BOOLEAN
IsCmlCpu (
  VOID
  )
{
  CPU_FAMILY   CpuFamily;
  CpuFamily   = GetCpuFamily ();

  switch (CpuFamily) {
    case EnumCpuCmlDtHalo:
      return TRUE;
    default:
      return FALSE;
  }
}

/**
  This function checks whether uCode loaded from FIT.

  @retval TRUE  - uCode loaded from FIT successful
  @retval FALSE - Failed on FIT to load uCode
**/
BOOLEAN
IsValiduCodeEntry (
  VOID
  )
{
  MSR_REGISTER TempMsr;
  UINT8        FitEntryType;
  UINT8        FitErrorCode;

  TempMsr.Qword = AsmReadMsr64 (MSR_FIT_BIOS_ERROR);
  FitEntryType = (UINT8) ((TempMsr.Dwords.Low & B_MSR_FIT_BIOS_ERROR_ENTRY_TYPE_MASK) >> N_MSR_FIT_BIOS_ERROR_ENTRY_TYPE_OFFSET);
  FitErrorCode = (UINT8) (TempMsr.Dwords.Low & B_MSR_FIT_BIOS_ERROR_ERROR_CODE_MASK);

  if (FitErrorCode != FIT_SUCCESSFUL) {
    switch (FitEntryType) {
      case FIT_HEADER_ENTRY:
        DEBUG ((DEBUG_ERROR, "uCode Load status: Unexpected Fit header incorrect\n"));
        break;
      case FIT_MICROCODE_UPDATE_ENTRY:
        DEBUG ((DEBUG_ERROR, "uCode Load status: Unexpected microcode update error\n"));
        break;
      case FIT_STARTUP_ACM_ENTRY:
        DEBUG ((DEBUG_ERROR, "uCode Load status: Unexpected startup ACM error\n"));
        break;
      case FIT_GENERAL_CHECKS:
        DEBUG ((DEBUG_ERROR, "uCode Load status: general checks error\n"));
        break;
      default:
        DEBUG ((DEBUG_ERROR, "uCode Load status: Unknown error\n"));
        break;
    }
    return FALSE;
  } else {
     TempMsr.Qword = AsmReadMsr64 (MSR_BIOS_SIGN_ID);
     DEBUG ((DEBUG_INFO, "Valid FIT uCode Found: 0x%08X\n", (UINT32) (TempMsr.Dwords.High)));
     return TRUE;  // FIT_SUCCESSFUL
  }
}

/**
  Return if CPU supports PFAT

  @retval TRUE             If CPU Supports
  @retval FALSE            If CPU doesn't Supports
**/
BOOLEAN
IsPfatEnabled (
  VOID
  )
{
  return MsrIsPfatEnabled ();
}

/**
  Is BIOS GUARD enabled.

  @retval TRUE   BIOS GUARD is supported and enabled.
  @retval FALSE  BIOS GUARD is disabled.
**/
BOOLEAN
IsBiosGuardEnabled (
  VOID
  )
{
#if FixedPcdGetBool(PcdBiosGuardEnable) == 1
  MSR_PLAT_FRMW_PROT_CTRL_REGISTER  ProtCtrlMsr;
  if (MsrIsPfatEnabled ()) {
    ProtCtrlMsr.Uint64 = AsmReadMsr64 (MSR_PLAT_FRMW_PROT_CTRL);
    if(ProtCtrlMsr.Bits.PfatEnable) {
      return TRUE;
    }
  }
#endif
  return FALSE;
}

/**
  Determine if CPU supports Intel Turbo Boost Max Technology 3.0 (ITBM).

  @retval TRUE   ITBM is supported and enabled.
  @retval FALSE  ITBM is disabled.
**/
BOOLEAN
IsItbmSupported (
  VOID
  )
{
  return (GetItbmSupportedStatus ());
}

/**
  Determine if CPU supports Overclocking by reading the number of bins in MSR FLEX_RATIO (194h)

  @retval TRUE   OC is supported and enabled.
  @retval FALSE  OC is disabled.
**/
BOOLEAN
IsOcSupported (
  VOID
  )
{
  MSR_FLEX_RATIO_REGISTER   FlexRatioMsr;
  FlexRatioMsr.Uint64 = AsmReadMsr64 (MSR_FLEX_RATIO);
  return (FlexRatioMsr.Bits.OcBins != 0);
}

/**
  Determine if CPU supports Programmable Core Ratio Limit for the Turbo mode.

  @retval TRUE   Core Ratio Limit for the Turbo mode is supported and enabled.
  @retval FALSE  Core Ratio Limit for the Turbo mode is disabled.
**/
BOOLEAN
IsCoreRatioLimitSupported (
  VOID
  )
{
  return MsrIsTurboRatioLimitProgrammable ();
}

/**
  Determine if CPU supports Programmable TDC/TDP Limit for the Turbo mode.

  @retval TRUE   TDC/TDP Limit is supported and enabled.
  @retval FALSE  TDC/TDP Limit is disabled.
**/
BOOLEAN
IsXETdcTdpLimitSupported (
  VOID
  )
{
  return MsrIsTdpLimitProgrammable ();
}

/**
  Determine if CPU supports Turbo mode.

  @retval TRUE   Turbo mode is supported and enabled.
  @retval FALSE  Turbo mode is disabled.
**/
BOOLEAN
IsTurboModeSupported (
  VOID
  )
{
  CPUID_THERMAL_POWER_MANAGEMENT_EAX   PowerEax;
  MSR_MISC_ENABLES_REGISTER            MiscMsr;

  AsmCpuid (
    CPUID_THERMAL_POWER_MANAGEMENT,
    &PowerEax.Uint32,
    NULL,
    NULL,
    NULL
    );

  MiscMsr.Uint64 = AsmReadMsr64 (MSR_IA32_MISC_ENABLE);
  return ((BOOLEAN) (PowerEax.Bits.TurboBoostTechnology) || (MiscMsr.Bits.TurboModeDisable != 0));
}

/**
  Determine if CPU supports PPIN (Protected Processor Inventory Number)

  @retval TRUE   PPIN feature is available.
  @retval FALSE  PPIN feature is not available.
**/
BOOLEAN
IsPpinFeatureAvailable (
  VOID
  )
{
  return MsrIsPpinCap ();
}

/**
  Determine if CPU supports Hardware P-States.

  @retval TRUE   Hardware P-States is supported and enabled.
  @retval FALSE  Hardware P-States is disabled.
**/
BOOLEAN
IsHwpSupported (
  VOID
  )
{
  MSR_MISC_PWR_MGMT_REGISTER  PwrMgmtMsr;
  PwrMgmtMsr.Uint64 = AsmReadMsr64 (MSR_MISC_PWR_MGMT);
  return (BOOLEAN)PwrMgmtMsr.Bits.EnableHwp;
}

/**
  Determine if CPU supports Mwait.

  @retval TRUE   Mwait is supported and enabled.
  @retval FALSE  Mwait is disabled.
**/
BOOLEAN
IsTimedMwaitSupported (
  VOID
  )
{
  return  MsrIsTimedMwaitSupported ();
}

/**
  Determine if CPU supports LPM.

  @retval TRUE   LPM is supported and enabled.
  @retval FALSE  LPM is disabled.
**/
BOOLEAN
IsLpmSupported (
  VOID
  )
{
  return MsrIsCpuSupportsLpm ();
}

/**
  Determine if CPU supports ConfigTdp.

  @retval TRUE   ConfigTdp is supported and enabled.
  @retval FALSE  ConfigTdp is disabled.
**/
BOOLEAN
IsConfigTdpSupported (
  VOID
  )
{
  UINT8 CtdpLevels;

  CtdpLevels = MsrGetConfigTdpLevels ();

  if ((CtdpLevels == 1) || (CtdpLevels == 2)) {
    return TRUE;
  }
  return FALSE;
}

/**
  Determine if CPU supports Turbo mode.

  @retval TRUE   Efficiency Turbo mode is supported and enabled.
  @retval FALSE  Efficiency Turbo mode is disabled.
**/
BOOLEAN
IsEnergyEfficientTurboSupported (
  VOID
  )
{
  CPUID_THERMAL_POWER_MANAGEMENT_ECX PowerEcx;
  AsmCpuid (
    CPUID_THERMAL_POWER_MANAGEMENT,
    NULL,
    NULL,
    &PowerEcx.Uint32,
    NULL
    );

  return (BOOLEAN) (PowerEcx.Bits.PerformanceEnergyBias);
}

/**
  Determine if CPU supports Hyper-Threading

  @retval TRUE   Hyper-Threading is supported and enabled.
  @retval FALSE  Hyper-Threading is disabled.
**/
BOOLEAN
IsHyperThreadingSupported (
  VOID
  )
{
  CPUID_EXTENDED_TOPOLOGY_EBX  Ebx;

  AsmCpuidEx (
    CPUID_EXTENDED_TOPOLOGY,
    0,
    NULL,
    &Ebx.Uint32,
    NULL,
    NULL
    );

  return ((Ebx.Bits.LogicalProcessors) > 1);
}

/**
  Determines what Ctdp levels are supported on the silicon.

  @param[in] CtdpDownSupported   - Pointer to CtdpDownSupported
  @param[in] CtdpUpSupported     - Pointer to CtdpUpSupported

  @retval VOID
**/
VOID GetConfigTdpLevelsSupported (
  IN UINT8     *CtdpDownSupported,
  IN UINT8     *CtdpUpSupported
  )
{
  UINT8                              CtdpLevels;
  UINT8                              CtdpLevel0Ratio;
  UINT8                              CtdpTempRatio;
  MSR_CONFIG_TDP_NOMINAL_REGISTER    TdpNominal;
  MSR_CONFIG_TDP_LEVEL2_REGISTER     TdpLevel2;
  MSR_CONFIG_TDP_LEVEL1_REGISTER     TdpLevel1;

  *CtdpDownSupported = FALSE;
  *CtdpUpSupported = FALSE;

  CtdpLevels = MsrGetConfigTdpLevels ();

  //
  // Check the known cases of 0 and 2 additional cTDP levels supported first
  //
  if (CtdpLevels == 0) {
    //
    // No additional cTdp Levels are supported
    //
    *CtdpDownSupported = FALSE;
    *CtdpUpSupported = FALSE;
    return;
  } else if (CtdpLevels == 2) {
    //
    // All additional cTDP levels are supported
    //
    *CtdpDownSupported = TRUE;
    *CtdpUpSupported = TRUE;
    return;
  }

  //
  // Now we read the cTDP ratios from each level
  //
  TdpNominal.Uint64 = AsmReadMsr64 (MSR_CONFIG_TDP_NOMINAL);
  CtdpLevel0Ratio = (UINT8) (TdpNominal.Bits.TdpRatio);

  TdpLevel1.Uint64 = AsmReadMsr64 (MSR_CONFIG_TDP_LEVEL1);
  CtdpTempRatio = (UINT8) TdpLevel1.Bits.TdpRatio;

  //
  // If there is no ratio value in LVL1 MSR, check LVL2 MSR
  //
  if (CtdpTempRatio == 0) {
    TdpLevel2.Uint64 = AsmReadMsr64 (MSR_CONFIG_TDP_LEVEL2);
    CtdpTempRatio = (UINT8) TdpLevel2.Bits.TdpRatio;

  }

  //
  //  Check if additional cTDP level is Up or Down
  //
  if (CtdpTempRatio > CtdpLevel0Ratio) {
    *CtdpUpSupported = TRUE;
  } else if (CtdpTempRatio <= CtdpLevel0Ratio) {
    *CtdpDownSupported = TRUE;
  }

  DEBUG ((DEBUG_INFO, "CTDP: CtdpDownSupported = %d, CtdpUpSupported = %d\n", CtdpDownSupported, CtdpUpSupported));

  return;
}

/**
  Determines what Ctdp levels are supported on the silicon.

  @param[in] MwaitEcx     - Pointer to MwaitEcx (CPUID_MONITOR_MWAIT_ECX)
  @param[in] MwaitEdx     - Pointer to MwaitEdx (CPUID_MONITOR_MWAIT_EDX)

  @retval VOID
**/
VOID GetSubCStateSupported (
  IN UINT32     *MwaitEcx,
  IN UINT32     *MwaitEdx
  )
{
  AsmCpuid (
    CPUID_MONITOR_MWAIT,
    NULL,
    NULL,
    MwaitEcx,
    MwaitEdx
    );
  return;
}

/**
  Determine Number of cores

  @retval Number of cores
**/
UINT8
GetCpuNumberofCores (
  VOID
  )
{
  UINT32 Threads;
  CPUID_EXTENDED_TOPOLOGY_EBX  Ebx;

  AsmCpuidEx (
    CPUID_EXTENDED_TOPOLOGY,
    0,
    NULL,
    &Ebx.Uint32,
    NULL,
    NULL
    );

  Threads = Ebx.Bits.LogicalProcessors;

  AsmCpuidEx (
    CPUID_EXTENDED_TOPOLOGY,
    1,
    NULL,
    &Ebx.Uint32,
    NULL,
    NULL
    );

  return (UINT8)(Ebx.Uint32/Threads);
}

/**
  Detect Supported CPU Features

  @param[InOut] CpuEcx   Pointer to CpuEcx (CPUID_VERSION_INFO_ECX).
**/
VOID
GetSupportedCpuFeatures (
  IN UINT32     *RegEcx
  )
{
  AsmCpuid (
    CPUID_VERSION_INFO,
    NULL,
    NULL,
    RegEcx,
    NULL
    );
  return;
}

/**
  Detect if Processor Trace supported or not

  @retval TRUE   IntelProcessorTrace is supported and enabled.
  @retval FALSE  IntelProcessorTrace is disabled.
**/
BOOLEAN
IsIntelProcessorTraceSupported (
  VOID
  )
{
  CPUID_STRUCTURED_EXTENDED_FEATURE_FLAGS_EBX Ebx;

  AsmCpuidEx (
    CPUID_STRUCTURED_EXTENDED_FEATURE_FLAGS,
    0,
    NULL,
    &Ebx.Uint32,
    NULL,
    NULL
    );

  return (BOOLEAN) (Ebx.Bits.IntelProcessorTrace);
}

/**
  Returns Generation string of the respective CPU

  @retval      Character pointer of Generation string
**/
CONST CHAR8*
GetGenerationString (
  VOID
  )
{
  UINT32               CpuFamilyId;
  CpuFamilyId = GetCpuFamily ();

  return GetFruGenerationString (CpuFamilyId);
}

/**
  Returns Revision Table

  @retval      Character pointer of Revision Table String
**/
CONST CHAR8*
GetRevisionTable (
  VOID
  )
{
  CPUID_VERSION_INFO_EAX  Eax;

  AsmCpuid (CPUID_VERSION_INFO, &Eax.Uint32, NULL, NULL, NULL);

  return GetRevisionTableString (Eax.Uint32 & CPUID_FULL_FAMILY_MODEL_STEPPING);
}

/**
  Returns Sku String

  @retval      Character pointer of Sku String
**/
CONST CHAR8*
GetSkuString (
  VOID
  )
{
  CPU_SKU                            CpuSku;
  CONST CHAR8                        *SkuString = NULL;

  CpuSku = GetCpuSku ();
  switch (CpuSku) {
    case EnumCpuUlt:
      SkuString = "ULT";
      break;

    case EnumCpuUlx:
      SkuString = "ULX";
      break;

    case EnumCpuTrad:
      SkuString = "DT";
      break;

    case EnumCpuHalo:
      SkuString = "Halo";
      break;

    default:
      SkuString = NULL;
      break;
  }
  return SkuString;
}

/**
  Determine if CPU supports Telemetry.

  @retval TRUE   if CPU Supports Telemetry.
  @retval FALSE  if CPU doesn't supports Telemetry.
**/
BOOLEAN
IsTelemetrySupported (
  VOID
  )
{
  return IsFruSupportedTelemetry ();
}

/**
  Returns PPAM version details to support DGR with Nifty Rock feature based on SKU.

  @retval 0    DGR with Nifty Rock feature is not supported, No need to load PPAM.
  @retval 11   DGR with Nifty Rock feature is supported with PPAM Version 1.1.
**/
UINT8
NiftyRockSupportLevel (
  VOID
  )
{
  //
  // DGR supports only on the CPUs where
  // Platform Info MSR bit59 (SMM_SUPOVR_STATE_LOCK_ENABLE) is set and
  // SMX is supported.
  //
  if (! MsrIsSmmSupovrStateLockSupported ()) {
    DEBUG ((DEBUG_INFO, "NiftyRock is not supported due to SMM_SUPOVR_STATE_LOCK MSR is not supported.\n"));
    return NO_NR_PPAM_SUPPORT;
  }

  if (! IsSmxSupported ()) {
    DEBUG ((DEBUG_INFO, "Nifty Rock is not supported due to SMX/TXT is not supported.\n"));
    return NO_NR_PPAM_SUPPORT;
  }

  return NR_PPAM_11_SUPPORT;
}

/**
  This function return max and min bus ratio.

  @param[out]  MaxBusRatio
  @param[out]  MinBusRatio
**/
VOID
GetBusRatio (
  OUT UINT8 *MaxBusRatio, OPTIONAL
  OUT UINT8 *MinBusRatio OPTIONAL
  )
{
  MsrGetBusRatio (MaxBusRatio, MinBusRatio);
  return;
}

/**
  This function return max Efficiency Ratio.

  @retval Max efficiency ratio
**/
UINT8
GetMaxEfficiencyRatio (
  VOID
  )
{
  return MsrGetMaxEfficiencyRatio ();
}

/**
  This function return max Non-Turbo Ratio.

  @retval Max Non-Turbo Ratio
**/
UINT8
GetMaxNonTurboRatio (
  VOID
  )
{
  return MsrGetMaxNonTurboRatio ();
}

/**
  This function return the supported Config TDP Levels.

  @retval number of config TDP levels
**/
UINT8
GetConfigTdpLevels (
  VOID
  )
{
  return MsrGetConfigTdpLevels ();
}

/**
  This function return the supported Prmmr Size

  @retval  Supported Prmrr Size
**/
UINT32
GetMaxSupportedPrmrrSize (
  VOID
  )
{
  return MsrGetSupportedPrmrrSize ();
}

/**
  This function returns the supported Physical Address Size

  @retval supported Physical Address Size.
**/
UINT8
GetMaxPhysicalAddressSize (
  VOID
  )
{
  return GetMaxPhysicalAddressSizeFru ();
}

/**
  Return if Edram Enable

  @retval TRUE             If Edram Enable
  @retval FALSE            If Edram Disable
**/
BOOLEAN
IsEdramEnable (
  VOID
  )
{
  return MsrIsEdramEnable ();
}
