/** @file
  CpuPowerOnConfig PEI Library

@copyright
  INTEL CONFIDENTIAL
  Copyright 2014 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/

#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/CpuPowerOnConfigLib.h>
#include <Library/DebugLib.h>
#include <Library/CpuPlatformLib.h>
#include <Library/CpuInfoFruLib.h>
#include <Ppi/MasterBootMode.h>
#include <Register/Cpuid.h>
#include <Register/Msr.h>
#include <CpuAccess.h>
#include <Library/ConfigBlockLib.h>
#include <Library/PostCodeLib.h>
#include <Library/PmcPrivateLib.h>
#include <Ppi/Spi.h>
#include <Register/CommonMsr.h>

/**
  Check if VT is fused and disabled by Setup Option so reset is needed.

  @param[in] VmxEnable - VmxEnable CPU policy.

  @retval NO_RESET          - No reset is needed.
  @retval COLD_RESET        - Update requires a cold reset.
**/
CPU_RESET_TYPE
CheckVmxIfNeedReset (
  IN UINT8         VmxEnable
  )
{
  MSR_IA32_FEATURE_CONTROL_REGISTER Msr;
  CPUID_VERSION_INFO_ECX            Ecx;

  AsmCpuid (CPUID_VERSION_INFO, &Ecx.Uint32, NULL, NULL, NULL);

  ///
  /// Check if VMX is supported
  ///
  if (Ecx.Bits.VMX == 0) {
    return NO_RESET;
  }

  Msr.Uint64 = AsmReadMsr64 (MSR_IA32_FEATURE_CONTROL);

  ///
  /// Need to reset only if locked and VMX state has to be changed.
  ///
  if ((Msr.Bits.Lock == 1) && (Msr.Bits.EnableVmxOutsideSmx != VmxEnable)) {
    return COLD_RESET;
  }

  return NO_RESET;
}

/**
  Get 32-bit CPU strap data.

  @param[out] CpuStrapData  CPU strap value

  @retval TRUE   Fetched data is valid
  @retval FALSE  Fetched data is not valid
**/
STATIC
BOOLEAN
GetCpuStrapData (
  OUT UINT32  *CpuStrapData
  )
{
  EFI_STATUS   Status;
  PCH_SPI_PPI  *SpiPpi;
  UINT8        TempBuffer[4];

  if (PmcIsSetStrapMsgInterfaceEnabled ()) {
    *CpuStrapData = PmcGetStrapMsgData32 ();
  } else {
    Status = PeiServicesLocatePpi (
               &gPchSpiPpiGuid,
               0,
               NULL,
               (VOID**) &SpiPpi
               );
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "Failed to locate SPI PPI\n"));
      return FALSE;
    }

    Status = SpiPpi->ReadCpuSoftStrap (SpiPpi, 0, sizeof (UINT32), TempBuffer);
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "Failed to read image descriptor\n"));
      return FALSE;
    }
    *CpuStrapData = TempBuffer[0] + (TempBuffer[1] << 8) + (TempBuffer[2] << 16) + (TempBuffer[3] << 24);
  }

  return TRUE;
}

/**
  Set Cpu Strap and Set Early Power On Configuration setting for feature change.

  @param[in] SiPreMemPolicyPpi    - Address of the SI_PREMEM_POLICY_PPI.
**/
VOID
SetCpuStrapAndEarlyPowerOnConfig (
  IN SI_PREMEM_POLICY_PPI       *SiPreMemPolicyPpi
  )
{
  EFI_STATUS                    Status;
  CPU_RESET_TYPE                ResetType;
  UINT32                        CpuStrapSetData;
  EFI_BOOT_MODE                 BootMode;
  CPU_CONFIG_LIB_PREMEM_CONFIG  *CpuConfigLibPreMemConfig;
  MSR_FLEX_RATIO_REGISTER       MsrFlexRatio;

  ResetType         = NO_RESET;
  CpuStrapSetData   = 0;

  DEBUG((DEBUG_INFO, "SetCpuStrapAndEarlyPowerOnConfig Start \n"));
  PostCode (0xC0A);

  Status = PeiServicesGetBootMode (&BootMode);
  DEBUG ((DEBUG_INFO, "SetCpuStrapAndEarlyPowerOnConfig: BootMode = %X\n", BootMode));
  if ((Status == EFI_SUCCESS) && (BootMode == BOOT_ON_S3_RESUME)) {
    PmcLockSetStrapMsgInterface ();
    return;
  }
  DEBUG ((DEBUG_INFO, "Set CPU Soft Straps Entry Point\n"));

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuConfigLibPreMemConfigGuid, (VOID *) &CpuConfigLibPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  ///
  /// Get Cpu Strap Set Data
  ///
  if (GetCpuStrapData (&CpuStrapSetData)) {
    DEBUG ((DEBUG_INFO, "Current CPU Strap Data = 0x%04X\n", CpuStrapSetData));
    ResetType |= CpuHtEnableDisable (&CpuStrapSetData, (UINT8) CpuConfigLibPreMemConfig->HyperThreading);
    ResetType |= CpuSetActiveCores (&CpuStrapSetData, (UINT8) CpuConfigLibPreMemConfig->ActiveCoreCount1, (UINT8) CpuConfigLibPreMemConfig->ActiveSmallCoreCount);
    ResetType |= CpuBistEnableDisable (&CpuStrapSetData, (UINT8) CpuConfigLibPreMemConfig->BistOnReset);

    ///
    /// Perform Flex Ratio if processor is fused to perform Flex Ratio
    ///
    MsrFlexRatio.Uint64 = AsmReadMsr64 (MSR_FLEX_RATIO);
    if (MsrFlexRatio.Bits.Enable == 1) {
      ResetType |= CpuProgramFlexMultiplier (&CpuStrapSetData, (UINT8) CpuConfigLibPreMemConfig->CpuRatio);
    }
    ResetType |= CpuBootRatioEnableDisable (&CpuStrapSetData, (UINT8) CpuConfigLibPreMemConfig->BootFrequency);
    ResetType |= CpuJtagPowerGateEnableDisable (&CpuStrapSetData, (UINT8) CpuConfigLibPreMemConfig->JtagC10PowerGateDisable);

    if (ResetType != NO_RESET) {
      DEBUG ((DEBUG_INFO, "New CPU Strap Data = 0x%04X\n", CpuStrapSetData));
      PmcSetStrapMsgData32 (CpuStrapSetData);
    }
  }
  PmcLockSetStrapMsgInterface ();

  Status = CpuEarlyEpocSet (CpuConfigLibPreMemConfig);
  if (!EFI_ERROR (Status)) {
    ResetType = COLD_RESET;
  }

  ResetType |= CheckVmxIfNeedReset ((UINT8) CpuConfigLibPreMemConfig->VmxEnable);

  if (ResetType != NO_RESET) {
    DEBUG ((DEBUG_INFO, "Cpu straps changed, performing reset - ResetType = %x\n", ResetType));
    PostCode (0xC13);
    PerformWarmOrColdReset (ResetType);
  }
  DEBUG((DEBUG_INFO, "SetCpuStrapAndEarlyPowerOnConfig Done \n"));
  PostCode (0xC14);
}

