/** @file
  Processor Power Management initialization code. This code determines current
  user configuration and modifies and loads ASL as well as initializing chipset
  and processor features to enable the proper power management.

  Acronyms:
  - PPM:  Processor Power Management
  - TM:   Thermal Monitor
  - IST:  Intel(R) Speedstep technology
  - HT:   Hyper-Threading Technology
  - ITBM: Intel(R) Turbo Boost Max Technology 3.0

@copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include "PowerMgmtInit.h"
#include <Protocol/CpuInfo.h>
#include <Protocol/MpService.h>
#include <Register/CommonMsr.h>


//
// Global variables
//

GLOBAL_REMOVE_IF_UNREFERENCED CPU_NVS_AREA_PROTOCOL *gCpuNvsAreaProtocol = NULL;  ///< CPU GlobalNvs Protocol
#if FixedPcdGetBool(PcdBiosGuardEnable) == 1
GLOBAL_REMOVE_IF_UNREFERENCED BIOSGUARD_NVS_AREA_PROTOCOL *gBiosGuardNvsAreaProtocol = NULL;  ///< BIOS Guard GlobalNvs Protocol
#endif

//
// FVID Table Information
// Default FVID table
// One header field plus states
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16     mNumberOfStates            = 0;
GLOBAL_REMOVE_IF_UNREFERENCED FVID_TABLE *mFvidPointer              = NULL;

//
// Globals to support updating ACPI Tables
//
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_TABLE_PROTOCOL     *mAcpiTable                = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mCpu0IstTable             = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mApIstTable               = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mCpu0CstTable             = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mApCstTable               = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mCpuSsdtTable             = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mCpu0TstTable             = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mApTstTable               = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mCpu0HwpTable             = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mApHwpTable               = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mItbmTable                = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mItbmSciTable             = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mCpu0PsdTable             = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_ACPI_DESCRIPTION_HEADER *mApPsdTable               = NULL;

GLOBAL_REMOVE_IF_UNREFERENCED CPU_INIT_DATA_HOB           *mCpuInitDataHob            = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED CPU_CONFIG_DATA             *mCpuConfigData             = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED EFI_MP_SERVICES_PROTOCOL    *mMpService                 = NULL;

typedef struct _MAILBOX_READ {
  UINT32                   *MailboxDataPtr;
  UINT32                   *MailboxStatus;
} MAILBOX_READ;

/**
  This function to be used with StartUpAllAps for retrieving
  the Max Core Frequency of all Cores.

  @param[in] VOID *Buffer
**/
VOID
MailboxReadFavoredCore (
  IN OUT VOID *Buffer
  )
{
  EFI_STATUS   Status;
  MAILBOX_READ *MailboxReadParameters;
  UINTN        CoreNumber;
  UINTN        ApNumber;
  UINT32       FuseCoreMax;
  UINT32       MailboxStatus;

  MailboxReadParameters = (MAILBOX_READ *) Buffer;
  ApNumber = 0;
  CoreNumber = 0;

  if (IsSecondaryThread ()) {
    return;
  }
  Status = mMpService->WhoAmI (mMpService, &ApNumber);
  ASSERT_EFI_ERROR (Status);

  if (ApNumber != 0) {
    CoreNumber = (ApNumber / 2);
  }

  MailboxRead (MAILBOX_TYPE_OC, MAILBOX_READ_PER_CORE_RATIO_LIMITS_CMD, &FuseCoreMax, &MailboxStatus);
  MailboxReadParameters->MailboxDataPtr[CoreNumber] = FuseCoreMax;
  MailboxReadParameters->MailboxStatus[CoreNumber] = MailboxStatus;
}

/**
  Initialize the power management support.
  This function will do boot time configuration:
    Detect HW capabilities and SW configuration
    Initialize HW and software state (primarily MSR and ACPI tables)

  @param[in] ImageHandle  Pointer to the loaded image protocol for this driver
  @param[in] SystemTable  Pointer to the EFI System Table

  @retval EFI_SUCCESS     The driver installes/initialized correctly.
  @retval Driver will ASSERT in debug builds on error.  PPM functionality is considered critical for mobile systems.
**/
EFI_STATUS
EFIAPI
InitPowerManagement (
  IN EFI_HANDLE       ImageHandle,
  IN EFI_SYSTEM_TABLE *SystemTable
  )
{
  EFI_STATUS                       Status;
  VOID                             *Hob;

  ///
  /// Get CPU Init Data Hob
  ///
  Hob = GetFirstGuidHob (&gCpuInitDataHobGuid);
  if (Hob == NULL) {
    DEBUG ((DEBUG_ERROR, "CPU Data HOB not available\n"));
    return EFI_NOT_FOUND;
  }
  mCpuInitDataHob   = (CPU_INIT_DATA_HOB *) ((UINTN) Hob + sizeof (EFI_HOB_GUID_TYPE));
  mCpuConfigData    = (CPU_CONFIG_DATA *) (UINTN) mCpuInitDataHob->CpuConfigData;
  ///
  /// Initialize the Global pointer for VID Table
  ///
  mFvidPointer         = (FVID_TABLE *)(UINTN)mCpuInitDataHob->FvidTable;

  ///
  /// Initialize Power management Global variables
  ///
  InitPowerManagementGlobalVariables ();

  /// Initialize CPU Power management code (Patch and install CPU ACPI tables, save S3 boot script info)
  ///
  Status = InitPpmDxe ();
  ASSERT_EFI_ERROR (Status);

  return EFI_SUCCESS;
}

/**
  Initializes the platform power management global variable.
  This must be called prior to any of the functions being used.
**/
VOID
InitPowerManagementGlobalVariables (
  VOID
  )
{
  EFI_STATUS                      Status;
  UINT8                           Index;
  MAILBOX_READ                    MailboxReadParameters;
  UINTN                           NumberOfCores;
  UINTN                           LogProcNum;
  UINTN                           BspIndex;
  MSR_CORE_THREAD_COUNT_REGISTER  MsrCoreThreadCount;
  UINT8                           LowestMaxPerf;
  UINT8                           EnableItbm;
#if FixedPcdGetBool(PcdBiosGuardEnable) == 1
  BIOSGUARD_HOB                   *BiosGuardHobPtr;
#endif
  ///
  /// Locate CPU Nvs Protocol.
  ///
  Status = gBS->LocateProtocol (
                  &gCpuNvsAreaProtocolGuid,
                  NULL,
                  (VOID **) &gCpuNvsAreaProtocol
                  );
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_ERROR, "CPU GloableNvs protocol not found"));
  }
  ASSERT_EFI_ERROR (Status);
  if (gCpuNvsAreaProtocol != NULL) {
    gCpuNvsAreaProtocol->Area->Cpuid = GetCpuFamily () | GetCpuStepping ();
  }
  //
  // Intel Turbo Boost Technology support, locate MP services
  //
  if (IsItbmSupported ()) {
    //
    // Locate the MP services protocol
    // Find the MP Protocol. This is an MP platform, so MP protocol must be there.
    //
    Status = gBS->LocateProtocol (&gEfiMpServiceProtocolGuid, NULL, (VOID **) &mMpService);
    ASSERT_EFI_ERROR (Status);
    if (EFI_ERROR (Status)) {
      return;
    }

    MsrCoreThreadCount.Uint64 = AsmReadMsr64 (MSR_CORE_THREAD_COUNT);

    NumberOfCores = (UINTN) MsrCoreThreadCount.Bits.Corecount;
    LogProcNum = (UINTN) MsrCoreThreadCount.Bits.Threadcount;

    MailboxReadParameters.MailboxDataPtr = AllocateZeroPool (sizeof (UINT32) * NumberOfCores);
    MailboxReadParameters.MailboxStatus  = AllocateZeroPool (sizeof (EFI_STATUS) * NumberOfCores);

    if ((MailboxReadParameters.MailboxDataPtr == NULL) || (MailboxReadParameters.MailboxStatus == NULL)) {
      return;
    }

    Status = mMpService->WhoAmI (mMpService, &BspIndex);
    //
    //  Read Favored Core Information from each core index
    //
    for (Index = 0; Index < LogProcNum; Index++) {
      if (Index == BspIndex){
        MailboxReadFavoredCore ((VOID*) &MailboxReadParameters);
      } else {
        mMpService->StartupThisAP (
                      mMpService,
                      (EFI_AP_PROCEDURE) MailboxReadFavoredCore,
                      Index,
                      NULL,
                      0,
                      (VOID*) &MailboxReadParameters,
                      NULL
                      );
      }
    }

    ///
    /// Find the lowest of core max ratio from OCMB 0x1C[15:8] FusedP0 max ratio
    ///
    LowestMaxPerf = 0xFF;
    for (Index = 0; Index < NumberOfCores; Index++) {
      if (LowestMaxPerf > (UINT8) ((MailboxReadParameters.MailboxDataPtr [Index] >> 8) & 0xFF)) {
        LowestMaxPerf = (UINT8) ((MailboxReadParameters.MailboxDataPtr [Index] >> 8) & 0xFF);
      }
    }
    DEBUG((DEBUG_INFO, "Itbm: Highest performance of slowest core LowestMaxPerf = 0x%x \n", LowestMaxPerf));

    ///
    /// We check OCMB 0x1C[15:8] FusedP0 max ratio to identify ITBMT supported
    /// non-ITBMT: All cores are the same value
    /// ITBMT: Difference value among all cores
    ///
    EnableItbm = 0;
    if (mCpuConfigData->EnableItbmPeriodicSmi) {
      for (Index = 0; Index < NumberOfCores; Index++) {
        if (LowestMaxPerf < (UINT8) ((MailboxReadParameters.MailboxDataPtr [Index] >> 8) & 0xFF)) {
          EnableItbm = 1;
          break;
        }
      }
    }
    DEBUG((DEBUG_INFO, "Itbm: Itbm Supported = 0x%x \n", EnableItbm));

    //
    // Update NVS with ITBMT support and Highest Performance of the slowest core.
    //
    if (gCpuNvsAreaProtocol != NULL) {
      gCpuNvsAreaProtocol->Area->EnableItbm    = EnableItbm;
      gCpuNvsAreaProtocol->Area->LowestMaxPerf = LowestMaxPerf;
    }
  }

#if FixedPcdGetBool(PcdBiosGuardEnable) == 1
  ///
  /// Get BIOS Guard Config Hob
  ///
  BiosGuardHobPtr = GetFirstGuidHob (&gBiosGuardHobGuid);
  if (BiosGuardHobPtr != NULL) {
    ///
    /// Locate BIOS Guard Nvs Protocol.
    ///
    Status = gBS->LocateProtocol (
                    &gBiosGuardNvsAreaProtocolGuid,
                    NULL,
                    (VOID **) &gBiosGuardNvsAreaProtocol
                    );
    if (EFI_ERROR (Status)) {
      DEBUG ((DEBUG_ERROR, "BIOS Guard Global Nvs protocol not found"));
    }
  }else {
    DEBUG ((DEBUG_ERROR, "BIOS Guard Hob not found, BIOS Guard Global Nvs protocol not initialized"));
  }
#endif

  ///
  /// Locate ACPI table protocol
  ///
  Status = gBS->LocateProtocol (&gEfiAcpiTableProtocolGuid, NULL, (VOID **) &mAcpiTable);
  ASSERT_EFI_ERROR (Status);
  return;
}

/**
  Initialize the processor power management based on hardware capabilities
  and user configuration settings.

  @retval EFI_SUCCESS - on success
  @retval Appropiate failure code on error
**/
EFI_STATUS
InitPpmDxe (
  VOID
  )
{
  EFI_STATUS   Status;

  Status      = EFI_SUCCESS;

  ///
  /// Initialize PAIR Configuration
  ///
  InitPpmIrmConfiguration ();

  Status = InitCpuAcpiTable ();
  if (EFI_ERROR (Status)) {
    return Status;
  }

  ///
  /// Initialise Miscellaneous features
  ///
  InitMiscFeatures ();

  return Status;
}
