/** @file
Intel Turbo Boost Max Technology 3.0

@copyright
INTEL CONFIDENTIAL
Copyright 2020 Intel Corporation.

The source code contained or described herein and all documents related to the
source code ("Material") are owned by Intel Corporation or its suppliers or
licensors. Title to the Material remains with Intel Corporation or its suppliers
and licensors. The Material may contain trade secrets and proprietary and
confidential information of Intel Corporation and its suppliers and licensors,
and is protected by worldwide copyright and trade secret laws and treaty
provisions. No part of the Material may be used, copied, reproduced, modified,
published, uploaded, posted, transmitted, distributed, or disclosed in any way
without Intel's prior express written permission.

No license under any patent, copyright, trade secret or other intellectual
property right is granted to or conferred upon you by disclosure or delivery
of the Materials, either expressly, by implication, inducement, estoppel or
otherwise. Any license under such intellectual property rights must be
express and approved by Intel in writing.

Unless otherwise agreed by Intel in writing, you may not remove or alter
this notice or any other notice embedded in Materials by Intel or
Intel's suppliers or licensors in any way.

This file contains an 'Intel Peripheral Driver' and is uniquely identified as
"Intel Reference Module" and is licensed for Intel CPUs and chipsets under
the terms of your license agreement with Intel or your vendor. This file may
be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <CpuAccess.h>
#include "PowerMgmtSmm.h"
#include <PowerMgmtNvsStruct.h>
#include "CpuDataStruct.h"
#include <Protocol/SmmIoTrapDispatch2.h>
#include <Library/IoLib.h>
#include <Library/SynchronizationLib.h>
#include <Library/CpuMailboxLib.h>
#include <Protocol/SmmCpu.h>
#include <Protocol/PchSmmIoTrapControl.h>

#define ITBMT_IOTRAP 0x820

GLOBAL_REMOVE_IF_UNREFERENCED EFI_SMM_PERIODIC_TIMER_DISPATCH2_PROTOCOL  *mEfiPeriodicSmmDispatch2Protocol = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED PCH_SMM_PERIODIC_TIMER_CONTROL_PROTOCOL    *mSmmPeriodicTimerControlProtocol = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED UINT8                                      *mHighestPerfPerCore2 = NULL; ///< Pointer to Buffer to store highest performance of cores.
GLOBAL_REMOVE_IF_UNREFERENCED EFI_SMM_CPU_PROTOCOL                       *mSmmCpuProtocol = NULL;
GLOBAL_REMOVE_IF_UNREFERENCED PCH_SMM_IO_TRAP_CONTROL_PROTOCOL           *mPchSmmIoTrapControl = NULL;  ///< To pause and resume IO trap
GLOBAL_REMOVE_IF_UNREFERENCED EFI_HANDLE                                 mSmmIoTrapHandle; ///< Pause and resume via EFI_HANDLE

UINT8      *mCurrentHighestPerfPerCore = NULL; ///< Pointer to Buffer to store current highest performance of cores.
UINT8      *mOldHighestPerfPerCore     = NULL; ///< Pointer to Buffer to store current highest performance of cores.
BOOLEAN    mCurrentCpuPerfChanged      = FALSE; ///< Flag to indicate that the highest performance has changed.
UINT8      *mFuse0HighestPerfPerCore   = NULL; ///< Pointer to Buffer to store Fuse0 highest performance of cores.
UINT8      *mIsCpuGenIoTrap            = NULL; // Identify if the CPU generate IO Trap

///
/// Spin lock used to serialize the multi threads read of OCMB.
///
extern SPIN_LOCK                                  mHwpCapabilitiesMsrSpinLock;
extern CPU_NVS_AREA                               *mCpuNvsAreaPtr;
extern EFI_SMM_PERIODIC_TIMER_REGISTER_CONTEXT    mPeriodicSmmContext;
extern EFI_HANDLE                                 mPeriodicSmmHandle;

/**
  Checks for changes to max performance ratio from OCMB 0x1C.

  @param[in] Buffer    Pointer to Buffer to indicate target CPU number.
**/
VOID
EFIAPI
CompareMaxCoreRatio (
  IN VOID *Buffer
  )
{
  EFI_STATUS  Status;
  UINTN       Index;
  UINT32      LibStatus;
  UINT32      MailboxData;

  Index = *(UINTN *) Buffer; ///< Cast to enhance readability.

  Status = MailboxRead (MAILBOX_TYPE_OC, MAILBOX_READ_PER_CORE_RATIO_LIMITS_CMD, &MailboxData, &LibStatus);
  if ((Status == EFI_SUCCESS) && (LibStatus == PCODE_MAILBOX_CC_SUCCESS)) {
    mHighestPerfPerCore2 [Index] = (UINT8) (MailboxData);
  }

  if (((mHighestPerfPerCore2 [Index] > 0) && (mHighestPerfPerCore2 [Index] != 0xFF)) &&
      (mHighestPerfPerCore2 [Index] != mOldHighestPerfPerCore [Index])
     ) {
    mCurrentCpuPerfChanged = TRUE; ///< Highest Performance has changed.
    mOldHighestPerfPerCore [Index] = mHighestPerfPerCore2 [Index]; // Save for next comparison
  }

  ///
  /// Release the SpinLock once execution is done
  ///
  ReleaseSpinLock (&mHwpCapabilitiesMsrSpinLock);

  return;
}

/**
  Overclocking systems can change the maximum cpu ratio during OS runtime.
  If the maximum cpu ratio changes, we must notify the OS of the change so that ITBM driver
  will operate using the new maximum ratio. Without the notification, changes to the max ratio are ignored
  and overclocking is broken. A periodic SMM handler (every 8seconds) is implemented to check for
  these changes in IA32_HWP_Capabilities. The OS will indicate the capability to re-read frequency
  by setting _OSC Bit [12] =1. In this case, the BIOS will trigger the SW SMI which will resume this periodic SMM.

  @param[in]     DispatchHandle  The unique handle assigned to this handler by SmiHandlerRegister().
  @param[in]     Context         Points to an optional handler context which was specified when the
                                 handler was registered.
  @param[in,out] CommBuffer      A pointer to a collection of data in memory that will
                                 be conveyed from a non-SMM environment into an SMM environment.
  @param[in,out] CommBufferSize  The size of the CommBuffer.

  @retval EFI_SUCCESS            Callback function executed.
**/
EFI_STATUS
EFIAPI
ItbmPeriodicSmmIoTrapCallback (
  IN EFI_HANDLE   DispatchHandle,
  IN CONST VOID   *DispatchContext      OPTIONAL,
  IN OUT VOID     *CommBuffer           OPTIONAL,
  IN OUT UINTN    *CommBufferSize       OPTIONAL
  )
{
  UINTN CpuIndex;
  EFI_STATUS Status;

  mCurrentCpuPerfChanged = FALSE;
  CpuIndex = gSmst->CurrentlyExecutingCpu;
  Status = EFI_SUCCESS;


  ///
  /// Execute on BSP. The BSP will release the spin lock when it is done executing CheckHwpCapabilities()
  ///
  AcquireSpinLock (&mHwpCapabilitiesMsrSpinLock);
  CompareMaxCoreRatio (&CpuIndex);

  for (CpuIndex = 0; CpuIndex < gSmst->NumberOfCpus; CpuIndex++) { ///< Execute on APs
     if (CpuIndex != gSmst->CurrentlyExecutingCpu) {
       ///
       /// Acquire spin lock. The AP will release the spin lock when it done executing CheckHwpCapabilties.
       ///
       AcquireSpinLock (&mHwpCapabilitiesMsrSpinLock);

       Status = gSmst->SmmStartupThisAp (
                         CompareMaxCoreRatio,
                         CpuIndex,
                         &CpuIndex
                         );
       ASSERT_EFI_ERROR (Status);

       ///
       /// Wait for the AP to release the spin lock.
       ///
       while (!AcquireSpinLockOrFail (&mHwpCapabilitiesMsrSpinLock)) {
         CpuPause ();
       }

       ///
       /// Release the spin lock.
       ///
       ReleaseSpinLock (&mHwpCapabilitiesMsrSpinLock);
     }
  }

  ///
  /// Check for change in max cpu performance ratio of any one of the cores.
  ///
  if (mCurrentCpuPerfChanged == TRUE) {
    ///
    /// Generate a _GPE _L62 SCI to ACPI OS.
    ///
    if (PmcIsSciEnabled ()) { ///< Check if SCI is enabled.
      ///
      /// Enable SCI assertion by setting SW GPE
      /// This will generate a SW SCI event in PCH
      ///
      PmcTriggerSwGpe ();
      mCpuNvsAreaPtr->ItbmInterruptStatus = 1; ///< Indicate interrupt for ITBM
    }
  }

  return EFI_SUCCESS;
}

/**
  Software SMI callback for resuming the periodic SMM in OC capable systems
  when the OS supports the notifications, platform wide _OSC bit 12 is set.

  @param[in]      DispatchHandle              The unique handle assigned to this handler by SmiHandlerRegister().
  @param[in]      Context                     Points to an optional handler context which was specified when the
                                              handler was registered.

  @retval EFI_SUCCESS                         The interrupt was handled successfully.
  @retval EFI_INVALID_PARAMETER               The DispatchHandle was not valid.
**/
EFI_STATUS
EFIAPI
ItbmResumePeriodicSmmIoTrapCallback (
   IN EFI_HANDLE                  DispatchHandle,
   IN EFI_SMM_SW_REGISTER_CONTEXT *DispatchContext
   )
{
  EFI_STATUS                                 Status;

  Status = mSmmPeriodicTimerControlProtocol->Resume (
                                                  mSmmPeriodicTimerControlProtocol,
                                                  mPeriodicSmmHandle
                                                  );

  return Status;
}

/**
  Reads OCMB 0x1C for current max performance ratio.

  @param[in] Buffer    Pointer to Buffer to indicate target CPU number.
**/
VOID
EFIAPI
GetCurrentMaxCoreRatio (
  IN VOID *Buffer
  )
{
  EFI_STATUS  Status;
  UINTN       Index;
  UINT32      LibStatus;
  UINT32      MailboxData;

  Index = *(UINTN *) Buffer; ///< Cast to enhance readability.

  Status = MailboxRead (MAILBOX_TYPE_OC, MAILBOX_READ_PER_CORE_RATIO_LIMITS_CMD, &MailboxData, &LibStatus);
  if ((Status == EFI_SUCCESS) && (LibStatus == PCODE_MAILBOX_CC_SUCCESS)) {
    mCurrentHighestPerfPerCore [Index] = (UINT8) (MailboxData);
    mFuse0HighestPerfPerCore [Index] = (UINT8) (MailboxData >> 8);
  }

  ///
  /// Release the SpinLock once execution is done
  ///
  ReleaseSpinLock (&mHwpCapabilitiesMsrSpinLock);

  return;
}

/**
  This catches IO trap SMI generated by OSPM to get current max core ratio

  @param[in]      DispatchHandle      Not used
  @param[in]      CallbackContext     Not used
  @param[in out]  CommBuffer          Not used
  @param[in out]  CommBufferSize      Not used
**/
VOID
EFIAPI
ItbmIoTrapCallback (
  IN EFI_HANDLE                             DispatchHandle,
  IN CONST VOID                             *CallbackContext,
  IN OUT VOID                               *CommBuffer,
  IN OUT UINTN                              *CommBufferSize
  )
{
  EFI_STATUS                 Status;
  EFI_SMM_SAVE_STATE_IO_INFO SmiIoInfo;
  UINTN                      CpuIndex;
  UINTN                      CurrentCpuIndex;
  UINTN                      CpuIndexIoTrap;

  Status = mPchSmmIoTrapControl->Pause (
                                   mPchSmmIoTrapControl,
                                   mSmmIoTrapHandle
                                   );
  ASSERT_EFI_ERROR (Status);

  CpuIndexIoTrap = 0;
  CurrentCpuIndex = gSmst->CurrentlyExecutingCpu;
  for (CpuIndex = 0; CpuIndex < gSmst->NumberOfCpus; CpuIndex++) {
    mIsCpuGenIoTrap [CpuIndex] = 0;
    Status = mSmmCpuProtocol->ReadSaveState (
                                mSmmCpuProtocol,
                                sizeof (EFI_SMM_SAVE_STATE_IO_INFO),
                                EFI_SMM_SAVE_STATE_REGISTER_IO,
                                CpuIndex,
                                &SmiIoInfo
                                );
    //
    // If this is not the SMI source, skip it.
    //
    if (EFI_ERROR (Status)) {
      continue;
    }
    //
    // If the IO address is not "BYTE" "READ" to "Port 0x820", skip it.
    //
    if ((SmiIoInfo.IoPort == ITBMT_IOTRAP) &&
        (SmiIoInfo.IoType == EFI_MM_SAVE_STATE_IO_TYPE_INPUT) &&
        (SmiIoInfo.IoWidth == EFI_SMM_SAVE_STATE_IO_WIDTH_UINT8)) {
      //
      // The 0-based index of the CPU which generated the IO Trap.
      //
      mIsCpuGenIoTrap [CpuIndex] = 1;
    }
  }

  for (CpuIndexIoTrap = 0; CpuIndexIoTrap < gSmst->NumberOfCpus; CpuIndexIoTrap++) {
    if ((CurrentCpuIndex == CpuIndexIoTrap) && (mIsCpuGenIoTrap [CpuIndexIoTrap] == 1)) {
      ///
      /// Execute on BSP. The BSP will release the spin lock when it is done executing
      ///
      AcquireSpinLock (&mHwpCapabilitiesMsrSpinLock);
      GetCurrentMaxCoreRatio (&CpuIndexIoTrap);
    } else {
      if (mIsCpuGenIoTrap [CpuIndexIoTrap] == 1) {
        ///
        /// Acquire spin lock. The AP will release the spin lock when it done executing.
        ///
        AcquireSpinLock (&mHwpCapabilitiesMsrSpinLock);
        Status = gSmst->SmmStartupThisAp (
                          GetCurrentMaxCoreRatio,
                          CpuIndexIoTrap,
                          &CpuIndexIoTrap
                          );
        ASSERT_EFI_ERROR (Status);
        ///
        /// Wait for the AP to release the spin lock.
        ///
        while (!AcquireSpinLockOrFail (&mHwpCapabilitiesMsrSpinLock)) {
          CpuPause ();
        }

        ///
        /// Release the spin lock.
        ///
        ReleaseSpinLock (&mHwpCapabilitiesMsrSpinLock);
      }
    }
  }


  for (CpuIndexIoTrap = 0; CpuIndexIoTrap < gSmst->NumberOfCpus; CpuIndexIoTrap++) {
    if (mIsCpuGenIoTrap [CpuIndexIoTrap] == 1) {
      if (mCurrentHighestPerfPerCore [CpuIndexIoTrap] == 0 ||
          mCurrentHighestPerfPerCore [CpuIndexIoTrap] == 0xFF) {
        mCurrentHighestPerfPerCore [CpuIndexIoTrap] = mFuse0HighestPerfPerCore [CpuIndexIoTrap];
      }
      mOldHighestPerfPerCore [CpuIndexIoTrap] = mCurrentHighestPerfPerCore [CpuIndexIoTrap];
      Status = mSmmCpuProtocol->WriteSaveState (
                                  mSmmCpuProtocol,
                                  sizeof (UINT8),
                                  EFI_SMM_SAVE_STATE_REGISTER_RAX,
                                  CpuIndexIoTrap,
                                  &mCurrentHighestPerfPerCore [CpuIndexIoTrap]
                                  );
      ASSERT_EFI_ERROR (Status);
    }
  }

  Status = mPchSmmIoTrapControl->Resume (
                                   mPchSmmIoTrapControl,
                                   mSmmIoTrapHandle
                                   );
  ASSERT_EFI_ERROR (Status);
  return;
}

/**
  Initialize all SMI handlers required for Intel Turbo Boost Max Technology 3.0 to function.
  Initialize the Periodic SMM to check for changes in IA32_HWP_Capabilities.
  This is paused after registration.This handler is required for systems that are
  OC capable and for Operating Systems that support updated frequency notification.
  Platform _OSC bit 12 is set to indicate this support in the OS.

  @retval EFI_SUCCESS     The driver installes/initialized correctly.
  @retval EFI_NOT_FOUND   CPU Data HOB not available.
**/
EFI_STATUS
EFIAPI
InitPowerMgmtItbmSmmIoTrap (
  VOID
  )
{
  CPU_NVS_AREA_PROTOCOL                      *CpuNvsAreaProtocol;
  EFI_HANDLE                                 SwHandle;
  EFI_SMM_SW_DISPATCH2_PROTOCOL              *SwDispatch;
  EFI_SMM_SW_REGISTER_CONTEXT                SwContext;
  EFI_STATUS                                 Status;
  UINTN                                      NumberOfProcessors;
  EFI_SMM_IO_TRAP_DISPATCH2_PROTOCOL         *SmmIoTrap;
  EFI_SMM_IO_TRAP_REGISTER_CONTEXT           SmmIoTrapContext;

  ///
  /// Locate Cpu Nvs area
  ///
  Status = gBS->LocateProtocol (&gCpuNvsAreaProtocolGuid, NULL, (VOID **) &CpuNvsAreaProtocol);
  ASSERT_EFI_ERROR (Status);
  mCpuNvsAreaPtr = CpuNvsAreaProtocol->Area;
  ///
  /// Allocate global structure to contain the highest performance of individual cores.
  ///
  NumberOfProcessors = gSmst->NumberOfCpus;
  Status = gSmst->SmmAllocatePool (
                    EfiRuntimeServicesData,
                    NumberOfProcessors * sizeof (UINT8),
                    (VOID **) &mHighestPerfPerCore2
                    );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return (Status);
  }
  Status = gSmst->SmmAllocatePool (
                    EfiRuntimeServicesData,
                    NumberOfProcessors * sizeof (UINT8),
                    (VOID **) &mCurrentHighestPerfPerCore
                    );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return (Status);
  }
  Status = gSmst->SmmAllocatePool (
                    EfiRuntimeServicesData,
                    NumberOfProcessors * sizeof (UINT8),
                    (VOID **) &mOldHighestPerfPerCore
                    );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return (Status);
  }
  Status = gSmst->SmmAllocatePool (
                    EfiRuntimeServicesData,
                    NumberOfProcessors * sizeof (UINT8),
                    (VOID **) &mFuse0HighestPerfPerCore
                    );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return (Status);
  }
  Status = gSmst->SmmAllocatePool (
                    EfiRuntimeServicesData,
                    NumberOfProcessors * sizeof (UINT8),
                    (VOID **) &mIsCpuGenIoTrap
                    );
  ASSERT_EFI_ERROR (Status);
  if (EFI_ERROR (Status)) {
    return (Status);
  }

  //
  // Locate PI SMM CPU protocol
  //
  Status = gSmst->SmmLocateProtocol (
                    &gEfiSmmCpuProtocolGuid,
                    NULL,
                    (VOID **) &mSmmCpuProtocol
                    );
  ASSERT_EFI_ERROR (Status);

  //
  // Locate the protocol for pause and resume
  //
  Status = gSmst->SmmLocateProtocol (
                    &gPchSmmIoTrapControlGuid,
                    NULL,
                    (VOID **) &mPchSmmIoTrapControl
                    );
  ASSERT_EFI_ERROR (Status);

  ///
  /// Initialize the spin lock used to serialize the MSR read in BSP and all APs
  ///
  InitializeSpinLock (&mHwpCapabilitiesMsrSpinLock);

  ///
  /// Register handler for periodic SMM polling (8sec) to check for changes in IA32_HWP_Capabilities.
  Status = gSmst->SmmLocateProtocol (
                    &gEfiSmmPeriodicTimerDispatch2ProtocolGuid,
                    NULL,
                    (VOID **) &mEfiPeriodicSmmDispatch2Protocol
                    );
  ASSERT_EFI_ERROR (Status);

  mPeriodicSmmContext.Period = 1;
  mPeriodicSmmContext.SmiTickInterval = TIME_8s;
  Status = mEfiPeriodicSmmDispatch2Protocol->Register (
                                            mEfiPeriodicSmmDispatch2Protocol,
                                            ItbmPeriodicSmmIoTrapCallback,
                                            &mPeriodicSmmContext,
                                            &mPeriodicSmmHandle
                                            );

  ///
  /// Pause the periodic timer because of responsiveness impact. This timer will be resumed
  /// through SW SMI only if the OS supports ITBM notifications. This support is indicated through
  /// platform _OSC Bit [12] = 1.
  Status = gSmst->SmmLocateProtocol (
                    &gPchSmmPeriodicTimerControlGuid,
                    NULL,
                    (VOID **) &mSmmPeriodicTimerControlProtocol
                    );
  ASSERT_EFI_ERROR (Status);

  Status = mSmmPeriodicTimerControlProtocol->Pause (
                                                  mSmmPeriodicTimerControlProtocol,
                                                  mPeriodicSmmHandle
                                                  );
  ASSERT_EFI_ERROR (Status);

  ///
  /// Register SW SMI handler to resume the periodic timer if the OS supports it. Indicated by
  /// _OSC Bit [12] = 1.
  Status = gSmst->SmmLocateProtocol (&gEfiSmmSwDispatch2ProtocolGuid, NULL, (VOID**) &SwDispatch);
  ASSERT_EFI_ERROR (Status);
  SwContext.SwSmiInputValue = (UINTN) PcdGet8 (PcdItbmSmi);
  Status = SwDispatch->Register (
                         SwDispatch,
                         (EFI_SMM_HANDLER_ENTRY_POINT2) ItbmResumePeriodicSmmIoTrapCallback,
                         &SwContext,
                         &SwHandle
                         );
  ASSERT_EFI_ERROR (Status);

  ///
  /// Locate the PCH Trap dispatch protocol
  ///
  Status = gSmst->SmmLocateProtocol (&gEfiSmmIoTrapDispatch2ProtocolGuid, NULL, (VOID **) &SmmIoTrap);
  ASSERT_EFI_ERROR (Status);

  SmmIoTrapContext.Type         = ReadTrap;
  SmmIoTrapContext.Length       = 1;
  SmmIoTrapContext.Address      = ITBMT_IOTRAP;
  Status = SmmIoTrap->Register (
                        SmmIoTrap,
                        (EFI_SMM_HANDLER_ENTRY_POINT2) ItbmIoTrapCallback,
                        &SmmIoTrapContext,
                        &mSmmIoTrapHandle
                        );
  ASSERT_EFI_ERROR (Status);
  return Status;
}
