/** @file
  CPU PCIe information library.

  All function in this library is available for PEI, DXE, and SMM,
  But do not support UEFI RUNTIME environment call.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Uefi/UefiBaseType.h>
#include <Library/CpuPlatformLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseLib.h>
#include <Library/PciSegmentLib.h>
#include <Register/CpuPcieRegs.h>
#include <Library/CpuPcieInfoFruLib.h>
#include <Library/CpuPcieInitCommon.h>
#include <CpuPcieInfo.h>
#include <Library/CpuPcieExpressHelpersLib.h>
#include <PcieRegs.h>
#include <CpuSbInfo.h>

/**
  PCIe controller configuration strings.
**/
GLOBAL_REMOVE_IF_UNREFERENCED CONST CHAR8* mCpuPcieControllerConfigName[] = {
  "1x16",
  "1x8-2x4",
  "2x8",
  "1x4"
};

/**
  PCIe controller VWM Mapping.
**/
GLOBAL_REMOVE_IF_UNREFERENCED VWM_MAPPING TglHVwmMapping[] = {
    {0xBB, 0x2B, 0x27, 0x26},
    {0xBB, 0x2A, 0x25, 0x24},
    {0xBB, 0x29, 0x23, 0x22},
    {0xBB, 0x28, 0x21, 0x20}
};

GLOBAL_REMOVE_IF_UNREFERENCED VWM_MAPPING TglUVwmMapping[] = {
    {0xBB, 0x2B, 0x27, 0x26}, // PEG060
};

/**
  Get VWM Mapping for the Platform

  @retval Pointer to VWM mapping structure
**/
VWM_MAPPING
*GetVwmMapping (
    IN     UINT32        RpIndex
  )
{
  switch (GetCpuFamily()) {
    case EnumCpuTglUltUlx:
      return &TglUVwmMapping[RpIndex];
    case EnumCpuTglDtHalo:
      return &TglHVwmMapping[RpIndex];
    default:
      return &TglUVwmMapping[RpIndex];
  }
}

/**
  Returns the PCIe controller configuration (1x16, 1x8-2x4, 2x8, 1x4)

  @retval PCIe controller configuration
**/
UINT8
GetCpuPcieControllerConfig (
  VOID
  )
{
  UINT32                     Data32;
  UINT8                      Config;
  CONST CHAR8*               ControllerName;

  if (PegPciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS(SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG_DEV_NUM, SA_PEG1_FUN_NUM, PCI_VENDOR_ID_OFFSET)) != 0xFFFF) {
    Data32 = PegPciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS(SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG_DEV_NUM, SA_PEG1_FUN_NUM, R_PCIE_STRPFUSECFG));
  } else {
    Data32 = PegPciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS(SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG_DEV_NUM, SA_PEG0_FUN_NUM, R_PCIE_STRPFUSECFG));
  }
  Config = (UINT8)((Data32 & B_PCIE_STRPFUSECFG_RPC_MASK) >> B_PCIE_STRPFUSECFG_RPC_OFFSET);
  DEBUG((DEBUG_INFO, "Controller Config Data32 = %x\n", Data32));
  DEBUG((DEBUG_INFO, "Controller Config = %x\n", Config));

  switch (Config) {
    case CPU_PCIE_1x16 :
      ControllerName = mCpuPcieControllerConfigName[0];
      break;
    case CPU_PCIE_1x8_2x4 :
      ControllerName = mCpuPcieControllerConfigName[1];
      break;
    case CPU_PCIE_2x8 :
     ControllerName = mCpuPcieControllerConfigName[2];
     break;
    case CPU_PCIE_1x4 :
      ControllerName = mCpuPcieControllerConfigName[3];
     break;
    default:
      ControllerName = mCpuPcieControllerConfigName[3];
      break;
  }
  DEBUG((DEBUG_INFO, "CPU PCIe Furcation is %a\n", ControllerName));
  return Config;
}

/**
  Get Maximum CPU Pcie Root Port Number

  @retval Maximum CPU Pcie Root Port Number
**/
UINT8
GetMaxCpuPciePortNum (
  VOID
  )
{
  switch (GetCpuFamily()) {
    case EnumCpuTglUltUlx:
      return CPU_PCIE_ULT_ULX_MAX_ROOT_PORT;
    case EnumCpuTglDtHalo:
      return CPU_PCIE_DT_HALO_MAX_ROOT_PORT;
    default:
      return CPU_PCIE_ULT_ULX_MAX_ROOT_PORT;
  }
}

/**
  Get max link width.

  @param[in] RpBase    Root Port base address
  @retval Max link width
**/
UINT8
CpuPcieGetMaxLinkWidth (
  UINT64  RpBase
  )
{
  UINT8  LinkWidth;
  DEBUG ((DEBUG_INFO, "CpuPcieGetMaxLinkWidth Start!\n"));
  if (PegPciSegmentRead32 (RpBase + PCI_VENDOR_ID_OFFSET) == 0xFFFF) {
    LinkWidth = 0;
    DEBUG((DEBUG_INFO, "VendoreId = %x so assign LinkWidth = %d\n", PegPciSegmentRead32 (RpBase + PCI_VENDOR_ID_OFFSET), LinkWidth));
    return LinkWidth;
  }
  LinkWidth = (UINT8) ((PegPciSegmentRead32 (RpBase + R_PCIE_LCAP) & B_PCIE_LCAP_MLW) >> B_PCIE_LCAP_MLW_OFFSET);
  if (GetCpuFamily() == CPUID_FULL_FAMILY_MODEL_TIGERLAKE_DT_HALO) { // For TGL H/S, Max link width cannot exceed 16
    //
    //
    if (LinkWidth > 16) {
      DEBUG ((DEBUG_INFO, "Link width reported was = %x for Rpbase = %x which is invalid, so BIOS made it 0\n", LinkWidth, RpBase));
      LinkWidth = 0;
    }
  } else { // For TGL U/Y, limit the link width to 4
    if (LinkWidth > 4) {
      DEBUG ((DEBUG_INFO, "LinkWidth reported was = %x for Rpbase = %x which is invalid, so BIOS made it 0\n", LinkWidth, RpBase));
      LinkWidth = 0;
    }
  }
  DEBUG ((DEBUG_INFO, "LinkWidth = %x\n", LinkWidth));
  DEBUG ((DEBUG_INFO, "CpuPcieGetMaxLinkWidth End!\n"));
  return LinkWidth;
}

/**
  Get Maximum CPU Pcie Controller Number

  @retval Maximum CPU Pcie Controller Number
**/
UINT8
GetMaxCpuPcieControllerNum (
  VOID
  )
{
  switch (GetCpuFamily()) {
    case EnumCpuTglUltUlx:
      return CPU_PCIE_ULT_ULX_MAX_CONTROLLER;
    case EnumCpuTglDtHalo:
      return CPU_PCIE_DT_HALO_MAX_CONTROLLER;
    default:
      return CPU_PCIE_ULT_ULX_MAX_CONTROLLER;
  }
}

/**
  Get CPU Pcie Controller Number from Root Port Number

  @retval CPU Pcie Controller Number
**/
UINT8
GetCpuPcieControllerNum (
  IN     UINT32        RpIndex
  )
{
  switch (RpIndex) {
    case 0:
      return CPU_PCIE_CONTROLLER0;
    case 1:
      return CPU_PCIE_CONTROLLER1;
    case 2:
    case 3:
      return CPU_PCIE_CONTROLLER2;
    default:
      return CPU_PCIE_CONTROLLER0;
  }
}

/**
  Check if a given Root Port is Multi VC

  @param[in] RpIndex (0-based)

  @return  TRUE if it is Multi VC
**/
BOOLEAN
IsRpMultiVc (
  IN  UINT32                   RpIndex
  )
{
  //
  // Note : PEG060 will be treated as controller 0 for all SKUs to maintain uniform coding and avoid CpuFamily check in all functions
  //
  if (RpIndex == 0) {
    return TRUE;
  }
  return FALSE;
}

/**
Read PCD register and return true based on port configuration and disable bits

@param[in] RpIndex (0-based)

@return  TRUE if root port is enabled
**/
BOOLEAN
CpuPcieReadPcdReg (
  IN UINT32  RpIndex
)
{
  UINT32       SpxPcd;
  UINT32       MaxPciePortNum;
  MaxPciePortNum = GetMaxCpuPciePortNum ();
  for (RpIndex = 0; RpIndex < MaxPciePortNum; RpIndex++) {
    CpuPcieRpSbiRead32 (RpIndex, R_SA_SPX_PCR_PCD, &SpxPcd);
    if ((SpxPcd & (B_SA_SPX_PCR_PCD_P0D << RpIndex)) != 0) {
      return TRUE;
    }
  }
  return FALSE;
}

/**
  Checks if PEG port is present

  @retval TRUE     PEG is presented
  @retval FALSE    PEG is not presented
**/
BOOLEAN
IsPegPresent (
  VOID
  )
{
  UINT64  PegBaseAddress;

  PegBaseAddress  = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG_DEV_NUM, 0, 0);
  if (PegPciSegmentRead16 (PegBaseAddress) != 0xFFFF) {
    return TRUE;
  }
  return FALSE;
}

/**
  Get CPU Pcie Root Port Device and Function Number by Root Port physical Number

  @param[in]  RpNumber              Root port physical number. (0-based)
  @param[out] RpDev                 Return corresponding root port device number.
  @param[out] RpFun                 Return corresponding root port function number.

  @retval     EFI_SUCCESS           Root port device and function is retrieved
  @retval     EFI_INVALID_PARAMETER RpNumber is invalid
**/
EFI_STATUS
EFIAPI
GetCpuPcieRpDevFun (
  IN  UINTN   RpNumber,
  OUT UINTN   *RpDev,
  OUT UINTN   *RpFun
  )
{
  if (RpNumber > GetMaxCpuPciePortNum ()) {
    DEBUG ((DEBUG_ERROR, "GetCpuPcieRpDevFun invalid RpNumber %x", RpNumber));
    ASSERT (FALSE);
    return EFI_INVALID_PARAMETER;
  }
  if (GetCpuFamily() == CPUID_FULL_FAMILY_MODEL_TIGERLAKE_ULT_ULX) { // For TGL - U/Y only one CPU PCIE Root port is present
    *RpDev = 6;
    *RpFun = 0;
    return EFI_SUCCESS;
  }

  switch(RpNumber){
    case 0:
      *RpDev = 6;
      *RpFun = 0;
      break;
    case 1:
      *RpDev = 1;
      *RpFun = 0;
      break;
    case 2:
      *RpDev = 1;
      *RpFun = 1;
      break;
    case 3:
      *RpDev = 1;
      *RpFun = 2;
      break;
    default:
      *RpDev = 6;
      *RpFun = 0;
      break;
  }
  return EFI_SUCCESS;
}

/**
  Get Root Port physical Number by CPU Pcie Root Port base address

  @param[in]  RpBase                Root Port pci segment base address.

  @return     Corresponding physical Root Port index (0-based)
**/
UINT32
EFIAPI
GetCpuPcieRpNumber (
  IN  UINT64  RpBase
  )
{
  //@TODO : This API needs to be tested to make sure it works well for CPU PCIE ports

  return (PegPciSegmentRead32 (RpBase + R_PCIE_LCAP) >> B_PCIE_LCAP_PN_OFFSET) -1;
}

/**
  Gets pci segment base address of PCIe root port.

  @param RpIndex    Root Port Index (0 based)

  @return PCIe port base address.
**/
UINT64
CpuPcieBase (
  IN  UINT32   RpIndex
  )
{
  UINTN   RpDevice;
  UINTN   RpFunction;
  GetCpuPcieRpDevFun (RpIndex, &RpDevice, &RpFunction);
  return PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, (UINT32) RpDevice, (UINT32) RpFunction, 0);
}

/**
  Determines whether CPU is supported Pciex16.

  @return TRUE if CPU is supported, FALSE otherwise
**/
BOOLEAN
IsCpuPciex16Supported (
  VOID
  )
{
  if (GetCpuFamily () == CPUID_FULL_FAMILY_MODEL_TIGERLAKE_DT_HALO) {
    return TRUE;
  } else {
    return FALSE;
  }
}

/**
  Get CPU Pcie SIP version

  @param[in] RpIndex    Root Port Index (0 based)
  @retval CPU SIP version
**/
UINT8
EFIAPI
GetCpuPcieSipInfo (
  IN     UINT32        RpIndex
  )
{
  return PCIE_SIP16;
}

/**
  Determines whether DEKEL work around required for FW download

  @return TRUE if DEKEL work around required, FALSE otherwise
**/
BOOLEAN
IsDekelWaRequired (
  VOID
  )
{
  // @todo: Need to verify on TGL B stepping
  if ((GetCpuFamily () == EnumCpuTglUltUlx) && (GetCpuStepping () < EnumTglB0)) {
    return TRUE;
  } else {
    return FALSE;
  }
}

/**
  This function returns PID according to PCIe controller index

  @param[in] ControllerIndex     PCIe controller index

  @retval CPU_SB_DEVICE_PID    Returns PID for SBI Access
**/
CPU_SB_DEVICE_PID
GetCpuPcieControllerSbiPid (
  IN  UINT32  ControllerIndex
  )
{
//
// Note : PEG060 will be treated as controller 0 for all SKUs to maintain uniform coding and avoid CpuFamily check in all functions
//
  switch(ControllerIndex){
    case 0:
      return CPU_SB_PID_PEG60;
    case 1:
      return CPU_SB_PID_PEG10;
    case 2:
      return CPU_SB_PID_PEG11_12;
    default:
      ASSERT(FALSE);
      return CPU_SB_PID_PEG60;
  }
}

/**
  This function returns PID according to Root Port Number

  @param[in] RpIndex     Root Port Index (0-based)

  @retval CPU_SB_DEVICE_PID    Returns PID for SBI Access
**/
CPU_SB_DEVICE_PID
GetCpuPcieRpSbiPid (
  IN  UINTN  RpIndex
  )
{
//
// Note : PEG060 will be treated as controller 0 for all SKUs to maintain uniform coding and avoid CpuFamily check in all functions
//
  switch(RpIndex){
    case 0:
      return CPU_SB_PID_PEG60;
    case 1:
      return CPU_SB_PID_PEG10;
    case 2:
    case 3:
      return CPU_SB_PID_PEG11_12;
    default:
      ASSERT(FALSE);
      return CPU_SB_PID_PEG60;
  }
}

/**
  Get max CPU PCIE lanes.

  @retval Max lanes
**/
UINT8
CpuPcieGetMaxLanes (
  )
{
  CPU_FAMILY  CpuFamily;
  CpuFamily = GetCpuFamily ();

  if (CpuFamily == CPUID_FULL_FAMILY_MODEL_TIGERLAKE_ULT_ULX) {
    return  MAX_CPU_PCIE_LANES_ULT_ULX;
  } else if (CpuFamily == CPUID_FULL_FAMILY_MODEL_TIGERLAKE_DT_HALO) {
    return  MAX_CPU_PCIE_LANES_DT_HALO;
  }
  return 0;
}
