/** @file
  This file contains functions that initializes Dekel

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/PeiCpuPcieDekelInitLib.h>
#include <Library/CpuPcieInfoFruLib.h>
#include <CpuSbInfo.h>
#include <Library/DebugLib.h>
#include <Library/PreSiliconEnvDetectLib.h>
#include <Library/CpuPlatformLib.h>

/**
  Read DEKEL FW download status

  @param[in]  CpuSbDevicePid   CPU SB Device Port ID

  @retval     BOOLEAN          Fw Download Completed TRUE or FALSE
**/

BOOLEAN
EFIAPI
DekelFwStatus (
  IN  CPU_SB_DEVICE_PID  CpuSbDevicePid
)
{
  ///
  ///
  if ((CpuRegbarRead32(CpuSbDevicePid, R_DEKEL_FW_STATUS) & (UINT32)BIT15) != BIT15) {
    if (!IsSimicsEnvironment()) {
      switch (CpuSbDevicePid) {
        case CPU_SB_PID_DEKEL_IO4:
          DEBUG ((DEBUG_ERROR, "ERROR: DEKEL FW download failed for x4 controller (PEG 060) PHY !!!\n"));
          ASSERT(FALSE);
          break;
        case CPU_SB_PID_DEKEL_IO16:
          DEBUG ((DEBUG_ERROR, "ERROR: DEKEL FW download failed for x16 controller PHY !!!\n"));
          ASSERT(FALSE);
          break;
        default:
          DEBUG ((DEBUG_ERROR, "ERROR: DEKEL FW download failed!!!\n"));
          break;
      }
    }
    return FALSE;
  }
  return TRUE;
}

/**
  Initialize the Dekel in PEI

  @param[in]  CpuSbDevicePid   CPU SB Device Port ID
**/

VOID
EFIAPI
DekelInit (
  IN  CPU_SB_DEVICE_PID  CpuSbDevicePid
)
{
  UINT8           Index;
  UINT16          NumberOfLanes;
  CPU_FAMILY      CpuFamily;
  CPU_STEPPING    CpuSteppingId;

  NumberOfLanes = 0;
  CpuFamily     = GetCpuFamily ();
  CpuSteppingId = GetCpuStepping ();

  if (CpuSbDevicePid == CPU_SB_PID_DEKEL_IO4) {
    NumberOfLanes = CPU_PCIE_x4_MAX_LANES;
  } else if (CpuSbDevicePid == CPU_SB_PID_DEKEL_IO16) {
    NumberOfLanes = CPU_PCIE_x16_MAX_LANES;
  }

  if (CpuFamily == EnumCpuTglUltUlx && CpuSteppingId == EnumTglA0) {
    for(Index = 0; Index < NumberOfLanes; Index++) {
      //
      //  Program data lane
      //
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_DL_SUSWELL_PWRGATE_CNTRL_OFFSET, 0x1002165);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_NOMEMPTY_UNDFLW_THRESOLD_OFFSET, 0x832C);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_REDO_RX_CALIB_EQ_TRAIN_OFFSET, 0x3A00407);
      //
      // PCS glue registers
      //
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_PCS_PMD_RX_TMRS_OFFSET, 0xCA625);
      //
      // PMD SUSWEL registers
      //
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_SQUELCH_CTRL_OFFSET, 0x11E0);
      //
      // PMD ANA CTRL registers
      //
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_PMD_LANE_ANA_CTRL_OFFSET, 0x6523);

      //
      // PMD ANA CSR registers
      //
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_PMD_LANE_ANA_CSR_VGA_CONFIG_OFFSET, 0x63C1);
    }
    //
    // Program Comon Lane with Data value
    //
    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CMN_LANE_OFFSET, 0x200B400);
    ///
    ///
    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CL_SUS_CLK_CRWELL_CNTRL_OFFSET, 0xD6C0);
    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_PER_COMP_DISABLE_OFFSET, 0x80018532);
    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CFG_VNEG_SEL_OFFSET, 0x220A4AE);

    ///
    ///
    for (Index = 0; Index < NumberOfLanes; Index++) {
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_PMD_LANE_ANA_CSR_VGA_CONFIG_OFFSET, 0x63C1);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_NOMEMPTY_UNDFLW_THRESOLD_OFFSET, 0x832C);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_REDO_RX_CALIB_EQ_TRAIN_OFFSET, 0x3A00407);
    }

    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CL_SUS_CLK_CRWELL_CNTRL_OFFSET, 0xD6C0);

    for (Index = 0; Index < NumberOfLanes; Index++) {
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_SQUELCH_CTRL_OFFSET, 0x11E0);
    }

    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CFG_VNEG_SEL_OFFSET, 0x220A4AE);
    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_PER_COMP_DISABLE_OFFSET, 0x18532);

    for (Index = 0; Index < NumberOfLanes; Index++) {
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_CALIB_SKIP_VECTOR_PERIODIC0, 0xFFEFE7FE);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_PMD_LANE_ANA_CTRL_OFFSET, 0x6523);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_DL_SUSWELL_PWRGATE_CNTRL_OFFSET, 0x1002165);
    }

    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CMN_LANE_OFFSET, 0x200B400);

    for (Index = 0; Index < NumberOfLanes; Index++) {
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_PCS_PMD_RX_TMRS_OFFSET, 0xCA625);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_PCS_PMD_FW, 0x300);
    }
  } else {
    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CL_SUS_CLK_CRWELL_CNTRL_OFFSET, 0xD6C0);
    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CMN_LANE_OFFSET, 0x200B400);
    CpuRegbarWrite32 (CpuSbDevicePid, R_DEKEL_CMN_LANE2_OFFSET, 0x200B400);

    for (Index = 0; Index < NumberOfLanes; Index++) {
      //
      //  Program data lane
      //
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_REDO_RX_CALIB_EQ_TRAIN_OFFSET, 0x3A00407);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_CFG13, 0x2E00017C);
      CpuRegbarWrite32 (CpuSbDevicePid, Index * LANE_MULTIPLIER + R_DEKEL_CDR_SAVE_RESTORE0, 0x276CE3FB);
    }
  }
}

/**
  Read DEKEL Firmware Version

  @param[in]  CpuSbDevicePid   CPU SB Device Port ID

  @retval     UINT32         Dekel Firmware Version
**/

UINT32
EFIAPI
GetDekelFwVersion (
  IN  CPU_SB_DEVICE_PID  CpuSbDevicePid
)
{
  UINT32       DekelFwVersion;

  DekelFwVersion = CpuRegbarRead32(CpuSbDevicePid, R_DEKEL_FW_VERSION_OFFSET);
  DEBUG ((DEBUG_INFO, "DekelFwVersion %x\n", DekelFwVersion));
  return DekelFwVersion;
}
