/** @file
  This file contains the Cpu Information for specific generation.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/
#include <CpuGenInfoFruLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseLib.h>
#include <Library/CpuInfoFruLib.h>
#include <CpuPowerMgmt.h>
#include <Library/CpuPlatformLib.h>
#include <Register/Cpuid.h>
#include <Register/TglMsr.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/CpuMailboxLib.h>
#include <Register/CommonMsr.h>
#include <Register/IgdRegs.h>
#include <IndustryStandard/SmBios.h>
#include <IndustryStandard/Pci22.h>

#define CTDP_NOMINAL_LEVEL 0x00

STATIC CONST CHAR8 mCpuFamilyString[] = "TigerLake";

typedef struct {
  UINT32  CPUID;
  CHAR8   String[16];
} CPU_REV;

///
/// Structure for CPU Strap settings
///
typedef struct {
  UINT32 HtDisabled              :  1; ///< Intel HT Technology Disable.
  UINT32 NumberOfActiveCores     :  4; ///< Number of Active Cores.
  UINT32 Bist                    :  1; ///< Built In Self Test (BIST) initiation.
  UINT32 FlexRatio               :  6; ///< Flex Ratio.
  UINT32 BootRatio               :  1; ///< Processor boot ratio; When set allows the processor to power up in maximum non-turbo ratio from the following boot.
  UINT32 JtagC10PowerGateDisable :  1; ///< JTAG Power gate disable.
  UINT32 Reserved2               : 18;
} CPU_STRAP_SET;

GLOBAL_REMOVE_IF_UNREFERENCED CONST CPU_REV  mProcessorRevisionTable[] = {
  {EnumCpuTglUltUlx    + EnumTglA0,    "A0"},
  {EnumCpuTglUltUlx    + EnumTglB0,    "B0"},
  {EnumCpuTglDtHalo    + EnumTglP0,    "P0"},
  {EnumCpuTglDtHalo    + EnumTglQ0,    "Q0"},
  {EnumCpuTglDtHalo    + EnumTglA0,    "A0"}
};

///
/// PowerLimits Override table for all SKUs. Non-cTDP parts would have '0' data for TDP level information.
///
GLOBAL_REMOVE_IF_UNREFERENCED PPM_OVERRIDE_TABLE mPowerLimitsOverrideTable[] = {
///
/// CpuIdentifier                 TDP      MSR PL1   MSR PL2   TdpUp   TdpUp    TdpNominal   TdpNominal  TdpDown   TdpDown    MSR
///                                         PL1      PL2       PL1     PL2      PL1          PL2         PL1       PL2        PL4
  {EnumTglY9WattES1CpuId,          900,     900,     4000,     0,      4000,    0,           4000,       0,        4000,      8300}, ///  9W TGL-Y ES1
  {EnumTglY9Watt42fCpuId,          900,     900,     4000,     0,      4000,    0,           4000,       0,        4000,      8300}, ///  9W TGL-Y 4+2
  {EnumTglU28WattES1CTdpCpuId,    1500,    1500,     6000,     0,      6000,    0,           6000,       0,        6000,     10500}, /// 15W TGL-U ES1 cTDP
  {EnumTglU28Watt42f15WCTdpCpuId, 1500,    1500,     6000,     0,      6000,    0,           6000,       0,        6000,     10500}, /// 15W TGL-U 4+2 cTDP
  {EnumTglU28Watt42f12WCTdpCpuId, 1250,    1250,     6000,     0,      6000,    0,           6000,       0,        6000,     10500}, /// 12.5W TGL-U 4+2 cTDP
  {EnumTglU28Watt42f10WCTdpCpuId, 1000,    1000,     6000,     0,      6000,    0,           6000,       0,        6000,     10500}, /// 10W TGL-U 4+2 cTDP
  {EnumTglU28Watt42fCpuId,        2800,    2800,     6400,     0,      6400,    0,           6400,       0,        6400,     12100}, /// 28W TGL-U 4+2
  {EnumTglU28WattES1CpuId,        2800,    2800,     6400,     0,      6400,    0,           6400,       0,        6400,     12100}, /// 28W TGL-U ES1
  {EnumTglU15Watt22CpuId,         1500,    1500,     3800,     0,      3800,    0,           3800,       0,        3800,      7100}, /// 15W TGL-U 2+2
  {EnumTglH45Watt81CpuId,         4500,    4500,    11800,     0,     11800,    0,          11800,       0,       11800,     20400}, /// 45W TGL-H 8+1
  {EnumTglH45Watt81CTdpCpuId,     6500,    6500,    14500,     0,     14500,    0,          14500,       0,       14500,     24000}  /// 65W TGL-H 8+1
};

/**
  Return if Elixir is present or not

  @retval TRUE  if Elixir is present
  @retval FALSE if Elixir is absent

**/
BOOLEAN
IsElixirPresent (
  VOID
  )
{
  UINT32       MailboxCommand;
  UINT32       MailboxData;
  UINT32       MailboxStatus;
  EFI_STATUS   Status;
  UINT8        VendorId;

  //
  // Version info (including VENDOR_ID) comes from VccIn Domain
  //
  MailboxCommand = SVID_GET_REGISTER + (ELIXIR_VENDOR_ID_MAILBOX_ADDRESS << SVID_REGISTER_ADDR_OFFSET) + (ELIXIR_VCC_IN_VR_ID << SVID_REGISTER_VR_ID_OFFSET);
  Status = MailboxRead (MAILBOX_TYPE_PCODE, MailboxCommand, &MailboxData, &MailboxStatus);
  if (EFI_ERROR (Status) || (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS)) {
    DEBUG ((DEBUG_ERROR, "Mailbox Command reading Elixir Springs Vendor ID failed. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
    return FALSE;
  }

  VendorId = (UINT8) MailboxData;
  return (VendorId == ELIXIR_VENDOR_ID_INTEL);
}

/**
  This function checks for HWP advanced features like EnableFastMsrHwpReq, EnableHwpAutoEppGrouping, EnableHwpAutoPerCorePstate, EnablePerCorePState.

  @retval TRUE  - Enable Per Core P State OS control mode.
  @retval FALSE - Disable Per Core P State OS control mode.
**/
BOOLEAN
IsHwpAdvancedFeatureSupport (
  VOID
  )
{
  return TRUE;
}

/**
  Return CPU Sku

  @param[in]  UINT32             CpuFamilyModel
  @param[in]  UINT16             CpuDid

  @retval     UINT8              CPU Sku
**/
UINT8
GetCpuSkuInfo (
  IN UINT32 CpuFamilyModel,
  IN UINT16 CpuDid
  )
{
  UINT8              CpuType;
  BOOLEAN            SkuFound;

  SkuFound  = TRUE;
  CpuType   = EnumCpuUnknown;

  switch (CpuFamilyModel) {
    case CPUID_FULL_FAMILY_MODEL_TIGERLAKE_ULT_ULX:
      switch (CpuDid) {
        case V_SA_DEVICE_ID_MB_ULT_1:    // ULT (4+2)
        case V_SA_DEVICE_ID_MB_ULT_2:    // ULT (2+2)
          CpuType = EnumCpuUlt;
          break;
        case V_SA_DEVICE_ID_MB_ULX_1:    // ULX (4+2)
        case V_SA_DEVICE_ID_MB_ULX_2:    // ULX (2+2)
          CpuType = EnumCpuUlx;
          break;
        default:
          SkuFound = FALSE;
          break;
      }
    break;
    case CPUID_FULL_FAMILY_MODEL_TIGERLAKE_DT_HALO:
      switch (CpuDid) {
        case V_SA_DEVICE_ID_HALO_1:      // Halo (6+1)
        case V_SA_DEVICE_ID_HALO_2:      // Halo (8+2)
        case V_SA_DEVICE_ID_HALO_3:      // Halo (8+1)
        case V_SA_DEVICE_ID_HALO_4:      // Halo (6+1)
        case V_SA_DEVICE_ID_HALO_5:      // Halo (4+1)
          CpuType = EnumCpuHalo;
          break;
        case V_SA_DEVICE_ID_DT_1:        // Desktop (6+1)
        case V_SA_DEVICE_ID_DT_2:        // Desktop (4+1)
        case V_SA_DEVICE_ID_DT_3:        // Reserved
          CpuType = EnumCpuTrad;
          break;
        default:
          SkuFound = FALSE;
          break;
      }
    break;
  }

  //
  // Simics doesn't simulate correct Device ID
  //
  if (IsSimicsEnvironment ()) {
    SkuFound = TRUE;
    switch (CpuFamilyModel) {
      case CPUID_FULL_FAMILY_MODEL_TIGERLAKE_ULT_ULX:
        CpuType = EnumCpuUlt;
        DEBUG ((DEBUG_WARN, "TGL Simics DID W/A for Unsupported CPU SKU: treating as ULT!\n"));
        break;
      case CPUID_FULL_FAMILY_MODEL_TIGERLAKE_DT_HALO:
        CpuType = EnumCpuHalo;
        DEBUG ((DEBUG_WARN, "TGL Simics DID W/A for Unsupported CPU SKU: treating as HALO!\n"));
        break;
    }
    DEBUG ((DEBUG_WARN, "Simics DID W/A for Unsupported CPU SKU, Device ID: 0x%02X, CPUID: 0x%08X!\n", CpuDid, CpuFamilyModel));
  }
  if (!SkuFound) {
    DEBUG ((DEBUG_ERROR, "Unsupported CPU SKU, Device ID: 0x%02X, CPUID: 0x%08X!\n", CpuDid, CpuFamilyModel));
    ASSERT (FALSE);
  }

  return CpuType;
}

/**
  Get processor generation

  @param[in]  CPU_FAMILY         CpuFamilyModel

  @retval     CPU_GENERATION     Returns the executing thread's processor generation.
**/
CPU_GENERATION
GetCpuSkuGeneration (
  CPU_FAMILY         CpuFamilyModel
  )
{
  CPU_GENERATION     CpuGeneration;

  switch (CpuFamilyModel) {
    case EnumCpuTglUltUlx:
    case EnumCpuTglDtHalo:
      CpuGeneration = EnumTglCpu;
      break;

    default:
      CpuGeneration = EnumCpuUnknownGeneration;
      ASSERT (FALSE);
      break;
  }

  return CpuGeneration;
}

/**
  Returns Generation string of the respective CPU

  @param[in]   CpuFamilyId

  @retval      Character pointer of Generation string
**/
CONST CHAR8*
GetFruGenerationString (
  IN   UINT32   CpuFamilyId
  )
{
  switch (CpuFamilyId) {
    case EnumCpuTglUltUlx:
    case EnumCpuTglDtHalo:
      return mCpuFamilyString;

    default:
      return NULL;
  }
}

/**
  Returns Revision Table string

  @param[in]   CpuId

  @retval      Character pointer of Revision Table string
**/
CONST CHAR8*
GetRevisionTableString (
  UINT32                CpuId
  )
{
  UINTN                 Index = 0;
  UINTN                 Count;

  Count = ARRAY_SIZE (mProcessorRevisionTable);

  for (Index = 0; Index < Count; Index++) {
    if (CpuId == mProcessorRevisionTable[Index].CPUID) {
      return mProcessorRevisionTable[Index].String;
    }
  }
  return NULL;
}

/**
  Determine if CPU supports Telemetry.

  @retval TRUE   if CPU Supports Telemetry.
  @retval FALSE  if CPU doesn't supports Telemetry.
**/
BOOLEAN
IsFruSupportedTelemetry (
  VOID
  )
{
  return TRUE;
}

/**
  This function returns if CPU support SyncFeatures

  @retval TRUE             SyncFeature is supported
  @retval FALSE            SyncFeature is not Supported
**/
BOOLEAN
IsSmmSyncFeatureSupported (
  VOID
  )
{
  return TRUE;
}

/**
  This function returns Number of CBO0 Bank Index.

  @retval Number of CBO0 Bank Index.
**/
UINT8
GetCbo0BankIndex (
  VOID
  )
{
  return 8;
}

/**
  Detect if Hetero Core is supported.

  @retval TRUE - Processor Support HeteroCore
  @retval FALSE - Processor doesnt support HeteroCore
**/
BOOLEAN
IsHeteroCoreSupported (
  VOID
  )
{
  return FALSE;
}

/**
  Detect the type of core.

  @retval the core type which is running
**/
UINT8
DetectCoreType (
  VOID
  )
{
  return 0;
}

/**
  Returns power limits Table

  @param[in]      NoOfOverrides

  @retval         override table pointer of power limits Table
**/
PPM_OVERRIDE_TABLE*
GetFruPowerLimits (
  IN UINTN                   *NoOfOverrides
  )
{
  *NoOfOverrides = ARRAY_SIZE (mPowerLimitsOverrideTable);
  return mPowerLimitsOverrideTable;
}

/**
  Return CPU Identifier used to identify various CPU types
  @param[in]  CPU_SKU                       CpuSku
  @param[in]  CPU_STEPPING                  CpuStepping
  @param[in]  SelectedCtdpLevel             Ctdp Level Selected in BIOS

  @retval CPU_IDENTIFIER           CPU Identifier
**/
CPU_IDENTIFIER
EFIAPI
GetCpuSkuIdentifier (
  IN  CPU_SKU         CpuSku,
  IN  CPU_STEPPING    CpuStepping,
  IN  UINT16          SelectedCtdpLevel
  )
{
  CPU_IDENTIFIER                        CpuIdentifier;
  UINT16                                PackageTdp;
  UINT16                                PackageTdpWatt;
  UINT16                                TempPackageTdp;
  UINT8                                 ProcessorPowerUnit;
  UINT16                                CtdpTempRatio1;
  UINT16                                CtdpTempRatio2;
  UINT16                                ConfigTdpWatt1;
  UINT16                                ConfigTdpWatt2;
  MSR_PACKAGE_POWER_SKU_REGISTER        PackagePowerSkuMsr;
  MSR_PACKAGE_POWER_SKU_UNIT_REGISTER   PackagePowerSkuUnitMsr;
  MSR_CONFIG_TDP_LEVEL1_REGISTER        CtdpMsr1;
  MSR_CONFIG_TDP_LEVEL2_REGISTER        CtdpMsr2;
  CPU_FAMILY                            CpuFamily;
  UINT16                                CpuDid;
  UINT16                                GtDid;

  CpuIdentifier = EnumUnknownCpuId;
  CpuDid = PciSegmentRead16 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_MC_DEVICE_ID));
  CpuFamily = GetCpuFamily ();

  GtDid = PciSegmentRead16 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, PCI_DEVICE_ID_OFFSET));

  ///
  /// Find Package TDP value in 1/100 Watt units
  ///

  PackagePowerSkuMsr.Uint64     = AsmReadMsr64 (MSR_PACKAGE_POWER_SKU);
  PackagePowerSkuUnitMsr.Uint64 = AsmReadMsr64 (MSR_PACKAGE_POWER_SKU_UNIT);

  ProcessorPowerUnit           = (UINT8) (PackagePowerSkuUnitMsr.Bits.PwrUnit);
  if (ProcessorPowerUnit == 0) {
    ProcessorPowerUnit = 1;
  } else {
    ProcessorPowerUnit = (UINT8) LShiftU64 (2, (ProcessorPowerUnit - 1));
    if (IsSimicsEnvironment () && ProcessorPowerUnit == 0) {
      ProcessorPowerUnit = 1;
    }
  }
  TempPackageTdp = (UINT16) PackagePowerSkuMsr.Bits.PkgTdp;
  PackageTdpWatt = (UINT16) DivU64x32 (TempPackageTdp, ProcessorPowerUnit);

  PackageTdp = (PackageTdpWatt * 100);
  if ((TempPackageTdp % ProcessorPowerUnit) !=0) {
    PackageTdp += ((TempPackageTdp % ProcessorPowerUnit) * 100) / ProcessorPowerUnit;
  }

  DEBUG ((DEBUG_INFO, "GetCpuIdentifier () - CpuSku     = 0x%X\n", CpuSku));
  DEBUG ((DEBUG_INFO, "                   - PackageTdp  = %d\n", PackageTdp));
  DEBUG ((DEBUG_INFO, "                   - CpuStepping = %d\n", CpuStepping));

  if (SelectedCtdpLevel != CTDP_NOMINAL_LEVEL) {
    CtdpMsr1.Uint64               = AsmReadMsr64 (MSR_CONFIG_TDP_LEVEL1);
    CtdpMsr2.Uint64               = AsmReadMsr64 (MSR_CONFIG_TDP_LEVEL2);
    CtdpTempRatio1 = (UINT16) (CtdpMsr1.Bits.PkgTdp);
    ConfigTdpWatt1 = (UINT16) DivU64x32 (CtdpTempRatio1, ProcessorPowerUnit);
    ConfigTdpWatt1 = (ConfigTdpWatt1 * 100);
    CtdpTempRatio2 = (UINT16) (CtdpMsr2.Bits.PkgTdp);
    ConfigTdpWatt2 = (UINT16) DivU64x32 (CtdpTempRatio2, ProcessorPowerUnit);
    ConfigTdpWatt2 = (ConfigTdpWatt2 * 100);
    DEBUG ((DEBUG_INFO, "                 - ConfigTdpWatt_level1 = %d\n", ConfigTdpWatt1));
    DEBUG ((DEBUG_INFO, "                 - ConfigTdpWatt_level2 = %d\n", ConfigTdpWatt2));
  }

  ///
  /// Logic to determine the CPU Identifier
  ///
  switch(CpuSku) {
    case EnumCpuUlx:
      if (CpuFamily == EnumCpuTglUltUlx) {
        if ((CpuDid == V_SA_DEVICE_ID_MB_ULX_1) || (CpuDid == V_SA_DEVICE_ID_MB_ULX_2)) {
          ///
          ///  TGL-Y 4+2f 9W
          ///
          DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-Y 4+2f 9W\n"));
          CpuIdentifier = EnumTglY9Watt42fCpuId;
          if (CpuStepping < EnumTglB0) {
            //
            // TGL-Y ES1 9W
            //
            DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-Y ES1 9W\n"));
            CpuIdentifier = EnumTglY9WattES1CpuId;
          }
        }
      }
      break;

    case EnumCpuUlt:
      if (CpuFamily == EnumCpuTglUltUlx) {
        if ((CpuDid == V_SA_DEVICE_ID_MB_ULT_1) || (CpuDid == V_SA_DEVICE_ID_MB_ULT_2)) {
          if (PackageTdp == CPU_TDP_28_WATTS) {
            if ((SelectedCtdpLevel != CTDP_NOMINAL_LEVEL) && (ConfigTdpWatt2 == CPU_TDP_15_WATTS)){
              ///
              ///  TGL-U 4+2f 15W cTDP
              ///
              DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-U 4+2 28W cTDP to 15W\n"));
              CpuIdentifier = EnumTglU28Watt42f15WCTdpCpuId;
            } else if ((SelectedCtdpLevel != CTDP_NOMINAL_LEVEL) && (ConfigTdpWatt2 == CPU_TDP_12_WATTS)){
              //
              ///  TGL-U 4+2f 12.5W cTDP
              ///
              DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-U 4+2 28W cTDP to 12.5W\n"));
              CpuIdentifier = EnumTglU28Watt42f12WCTdpCpuId;
            } else if ((SelectedCtdpLevel != CTDP_NOMINAL_LEVEL) && (ConfigTdpWatt2 == CPU_TDP_10_WATTS)){
              //
              ///  TGL-U 4+2f 10W cTDP
              ///
              DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-U 4+2 28W cTDP to 10W\n"));
              CpuIdentifier = EnumTglU28Watt42f10WCTdpCpuId;
            } else {
              ///
              ///  TGL U 4+2f 28W
              ///
              DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-U 4+2 28W\n"));
              CpuIdentifier = EnumTglU28Watt42fCpuId;
              if (CpuStepping < EnumTglB0) {
                //
                // TGL-U ES1 28W
                //
                DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-U ES1 28W\n"));
                CpuIdentifier = EnumTglU28WattES1CpuId;
              }
            }
          }
          if (PackageTdp == CPU_TDP_15_WATTS) {
            if ((CpuDid == V_SA_DEVICE_ID_MB_ULT_2) && (GtDid == V_SA_PCI_DEV_2_GT2_TGL_ULT_1_ID)) {
              //
              // TGL-U 2+2 15W
              //
              DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-U 2+2 15W\n"));
              CpuIdentifier = EnumTglU15Watt22CpuId;
            }
          }
        }
      }
      break;

    case EnumCpuHalo:
      if (CpuFamily == EnumCpuTglDtHalo) {
        if ((CpuDid == V_SA_DEVICE_ID_HALO_2) || (CpuDid == V_SA_DEVICE_ID_HALO_3)) {
          if (PackageTdp == CPU_TDP_45_WATTS) {
            if ((SelectedCtdpLevel != CTDP_NOMINAL_LEVEL) && (ConfigTdpWatt1 == CPU_TDP_65_WATTS)){
              ///
              ///  TGL-H 8+1 65W
              ///
              DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-H 8+1 45W cTDP up to 65W\n"));
              CpuIdentifier = EnumTglH45Watt81CTdpCpuId;
            } else {
              ///
              ///  TGL-H 8+1 45W
              ///
              DEBUG ((DEBUG_INFO, "CPU Identifier = TGL-H 8+1 45W\n"));
              CpuIdentifier = EnumTglH45Watt81CpuId;
            }
          }
        }
      }
      break;

    case EnumCpuTrad:
      break;

    default:
      CpuIdentifier = EnumUnknownCpuId;
      break;
    }

  return CpuIdentifier;
}

/**
  Compare the hyper threading setup option against the CPU strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData           - The current strap setting data.
  @param[in] HyperThreading            - hyper threading setting from BIOS Setup option.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuHtEnableDisable (
  IN VOID              *CpuStrapSetData,
  IN UINT8             HyperThreading
  )
{
  CPU_STRAP_SET      *CpuStrapSet;
  CPU_RESET_TYPE     ResetType;
  UINT16             TotalThreadsPerCore;
  EFI_CPUID_REGISTER Cpuid0B = { 0, 0, 0, 0 };

  ResetType   = NO_RESET;
  CpuStrapSet = (CPU_STRAP_SET *) CpuStrapSetData;

  ///
  /// Read CPUID(0xB) with ECX=0 for number of logical processors per Core.
  /// This value does NOT change based on Intel HT Technology disable and core disables.
  ///
  Cpuid0B.RegEcx = 0;
  AsmCpuidEx (CPUID_EXTENDED_TOPOLOGY, 0, NULL, &Cpuid0B.RegEbx, NULL, NULL);
  TotalThreadsPerCore = (UINT16) Cpuid0B.RegEbx;

  if (TotalThreadsPerCore > 1) {
    ///
    /// HT is supported
    ///
    DEBUG ((DEBUG_INFO, "HT is supported\n"));
    ///
    /// Check if the configuration of HT matches the BIOS Setup option.
    ///
    if (CpuStrapSet->HtDisabled == HyperThreading) {
      ///
      /// HT configuration doesn't match BIOS Setup option; update the Strap Set Data and Issue a Warm reset
      ///
      DEBUG ((DEBUG_INFO, "HT configuration doesn't match the setup value\n"));
      CpuStrapSet->HtDisabled = !(HyperThreading);
      ResetType |= WARM_RESET;
    }
  } else {
    ///
    /// HT is not supported by default fusing.
    ///
    DEBUG ((DEBUG_WARN, "HT is NOT supported\n"));
    CpuStrapSet->HtDisabled = 1;
  }

  return ResetType;
}

/**
  Compare the number of active cores setup option against the CPU strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData      - The current strap setting.
  @param[in] ActiveCoreCount      - Active big core count.
  @param[in] ActiveSmallCoreCount - Active small core count.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
  @retval COLD_RESET        - Update requires a cold reset.
**/
CPU_RESET_TYPE
CpuSetActiveCores (
  IN VOID              *CpuStrapSetData,
  IN UINT8             ActiveCoreCount,
  IN UINT8             ActiveSmallCoreCount
  )
{
  CPU_STRAP_SET                   *CpuStrapSet;
  CPU_RESET_TYPE                  ResetType;
  MSR_CORE_THREAD_COUNT_REGISTER  MsrCoreThreadCount;
  UINT32                          NumberOfActiveCores;
  UINT32                          NumberOfActiveThreads;

  ResetType   = NO_RESET;
  CpuStrapSet = (CPU_STRAP_SET *) CpuStrapSetData;

  /* CORE_THREAD_COUNT - msr 0x35
     Symbol      Name        MSB:LSB  Description
     CoreCount   CoreCount   31:16    The Core Count reflects the enabled cores based
                                      on the above thread count and the value of threads_
                                      available (to determine if HT is on) at reset time.

     ThreadCount ThreadCount 15:0     The Thread Count reflects the enabled threads based
                                      on the factory-configured thread count and the value
                                      of the PCH Soft Reset Data register for Client processors
                                      at reset time.

     Read MSR for Active Core and Thread Count.
  */
  MsrCoreThreadCount.Uint64 = AsmReadMsr64 (MSR_CORE_THREAD_COUNT);
  NumberOfActiveCores       = MsrCoreThreadCount.Bits.Corecount;
  NumberOfActiveThreads     = MsrCoreThreadCount.Bits.Threadcount;

  DEBUG ((DEBUG_INFO, "Number of Active Cores / Threads = 0x%x / 0x%x\n", NumberOfActiveCores, NumberOfActiveThreads));

  ///
  /// Check if the configuration of "Active Core" matches the BIOS Setup option.
  ///
  if (CpuStrapSet->NumberOfActiveCores != ActiveCoreCount) {
    DEBUG ((
      DEBUG_INFO,
      "NumberOfActiveCores doesn't match the value from Setup = %x / %x\n",
      CpuStrapSet->NumberOfActiveCores,
      ActiveCoreCount
      ));
    CpuStrapSet->NumberOfActiveCores = ActiveCoreCount;
    ResetType |= COLD_RESET;
  }

  return ResetType;
}

/**
  Compare the BIST setup option against the CPU strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData    - The current strap setting.
  @param[in] BistOnReset        - BistOnReset CPU Test Config Policy.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuBistEnableDisable (
  IN VOID              *CpuStrapSetData,
  IN UINT8             BistOnReset
  )
{
  CPU_STRAP_SET     *CpuStrapSet;

  CpuStrapSet = (CPU_STRAP_SET *) CpuStrapSetData;

  if (CpuStrapSet->Bist == BistOnReset) {
    return NO_RESET;
  } else {
    CpuStrapSet->Bist = BistOnReset;
    DEBUG ((DEBUG_INFO, "BIST configuration doesn't match the setup value\n"));
    return WARM_RESET;
  }
}

/**
  Compare the flex multiplier setup options against the CPU strap settings
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData    - The current strap setting.
  @param[in] CpuRatio           - CpuRatio CPU policy.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuProgramFlexMultiplier (
  IN VOID                        *CpuStrapSetData,
  IN UINT8                       CpuRatio
  )
{
  CPU_STRAP_SET             *CpuStrapSet;
  MSR_FLEX_RATIO_REGISTER   MsrFlexRatio;
  CPU_RESET_TYPE            ResetType;

  ResetType     = NO_RESET;
  CpuStrapSet   = (CPU_STRAP_SET *) CpuStrapSetData;

  DEBUG ((DEBUG_INFO, "Ratio from Policy is 0x%X\n", CpuRatio));

  ///
  /// Read and save current Flex Ratio data; disregard enable bit (MSR 194h [15:8])
  ///
  MsrFlexRatio.Uint64 = AsmReadMsr64 (MSR_FLEX_RATIO);
  DEBUG ((DEBUG_INFO, "Current Flex Ratio from MSR is 0x%X\n", MsrFlexRatio.Bits.FlexRatio));
  DEBUG ((DEBUG_INFO, "Current Flex Ratio from CPU Strap: 0x%X\n", CpuStrapSet->FlexRatio));
  ///
  /// Check and set Flex Ratio to requested ratio if possible
  ///
  if (CpuRatio == CpuStrapSet->FlexRatio) {
    ///
    /// Do nothing, ratio is already set to requested value and enabled
    ///
    DEBUG ((DEBUG_INFO, "No need to set Flex Ratio.\n"));
  } else {
    ///
    /// Set Flex Ratio to user selected value
    ///
    MsrFlexRatio.Bits.FlexRatio = CpuRatio;
    MsrFlexRatio.Bits.Enable = 1;
    AsmWriteMsr64 (MSR_FLEX_RATIO, MsrFlexRatio.Uint64);

    ///
    /// Set Soft Reset Data for Flex Ratio
    ///
    CpuStrapSet->FlexRatio = CpuRatio;

    ///
    /// Set RESET flag
    ///
    DEBUG ((DEBUG_INFO, "Setting Flex Ratio to 0x%X\n", CpuStrapSet->FlexRatio));
    ResetType = WARM_RESET;
  }

  return ResetType;
}

/**
  Compare the boot frequency setup option against the boot ratio strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSetData    - The current strap setting.
  @param[in] BootFrequency      - BootFrequency CPU policy.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuBootRatioEnableDisable (
  IN VOID                *CpuStrapSetData,
  IN UINT8               BootFrequency
  )
{
  CPU_STRAP_SET       *CpuStrapSet;
  CPU_RESET_TYPE      ResetType;

  ResetType   = NO_RESET;
  CpuStrapSet = (CPU_STRAP_SET *) CpuStrapSetData;

  ///
  /// Check if the configuration of BootRatio from Bit12 of strap setting is not aligned with the BootFrequency setup option.
  ///
  if (((CpuStrapSet->BootRatio == 1) && (BootFrequency == 0)) ||
      ((CpuStrapSet->BootRatio == 0) && (BootFrequency > 0))) {
    DEBUG ((
      DEBUG_INFO,
      "Boot Ratio strap setting of %x does not match the BootFrequency policy %x\n",
      CpuStrapSet->BootRatio,
      BootFrequency
      ));
    if (BootFrequency > 0) {
      CpuStrapSet->BootRatio = 1;
    } else {
      CpuStrapSet->BootRatio = 0;
    }
    ResetType |= WARM_RESET;
  }

  return ResetType;
}

/**
  Compare the JTAG power gate setup option against the CPU strap setting
  and in case of mismatch request a reset.

  @param[in] CpuStrapSet    - The current strap setting.
  @param[in] JtagC10PowerGateDisable - JtagC10PowerGateDisable CPU policy.

  @retval NO_RESET          - No reset is needed.
  @retval WARM_RESET        - Update requires a warm reset.
**/
CPU_RESET_TYPE
CpuJtagPowerGateEnableDisable (
  IN VOID                    *CpuStrapSetData,
  IN UINT8                   JtagC10PowerGateDisable
  )
{
  CPU_STRAP_SET       *CpuStrapSet;
  CPU_RESET_TYPE      ResetType;

  ResetType   = NO_RESET;
  CpuStrapSet = (CPU_STRAP_SET *) CpuStrapSetData;

  ///
  /// Check if the configuration of "JtagC10PowerGateDisable" from Bit13 of strap setting matches the BIOS Setup option.
  ///
  if (CpuStrapSet->JtagC10PowerGateDisable != JtagC10PowerGateDisable) {
    DEBUG ((
      DEBUG_INFO,
      "JtagC10PowerGateDisable strap setting doesn't match the value from Setup = %x / %x\n",
      CpuStrapSet->JtagC10PowerGateDisable,
      JtagC10PowerGateDisable
      ));
    CpuStrapSet->JtagC10PowerGateDisable = JtagC10PowerGateDisable;
    ResetType |= WARM_RESET;
  }

  return ResetType;
}

/**
  This function is to set the CPU Epoc on early phase.

  @param[in]  CpuConfigData

  @retval EFI_SUCCESS            Early CPU Epoc has been set successfully.
  @retval EFI_UNSUPPORTED        Early CPU Epoc seting is not supported.
  @retval EFI_INVALID_PARAMETER  Early Cpu Epoc returned invalid data.
**/
EFI_STATUS
CpuEarlyEpocSet (
  IN VOID *CpuConfigData
  )
{
  return EFI_UNSUPPORTED;
}

/**
  This function sends PECI related mailbox commands early in boot if needed

  @param[in] CpuConfigLibPreMemConfig Pointer to the cpu config lib premem config block instance

  @retval EFI_SUCCESS              - Function successfully executed.
  @retval Other                    - Error occurred during mailbox commands.
**/
EFI_STATUS
EFIAPI
CpuPeciMailboxCommands (
  IN CPU_CONFIG_LIB_PREMEM_CONFIG *CpuConfigLibPreMemConfig
  )
{
  return EFI_SUCCESS;
}

/**
  Programs Processor Upgrade for type 4 SMBIOS Processor Info HOB.

  @retval Returns Processor Upgrade value for type 4 SMBIOS Processor Info HOB.
**/
UINT8
SmbiosProcessorInfoHobType4 (
  VOID
  )
{
  return ProcessorUpgradeOther;
}

/**
  Get CPU default C state latency value.

  @param[out] CstateLatencyControl1Irtl  Interrupt Response Time Limit of LatencyContol1
  @param[out] CstateLatencyControl2Irtl  Interrupt Response Time Limit of LatencyContol2
  @param[out] CstateLatencyControl3Irtl  Interrupt Response Time Limit of LatencyContol3
  @param[out] CstateLatencyControl4Irtl  Interrupt Response Time Limit of LatencyContol4
  @param[out] CstateLatencyControl5Irtl  Interrupt Response Time Limit of LatencyContol5
**/
VOID
CpuGetCstateLatencyDefault (
  OUT UINT16 *CstateLatencyControl1Irtl,
  OUT UINT16 *CstateLatencyControl2Irtl,
  OUT UINT16 *CstateLatencyControl3Irtl,
  OUT UINT16 *CstateLatencyControl4Irtl,
  OUT UINT16 *CstateLatencyControl5Irtl
  )
{
  // default is zero for auto IRTL support.
}

/**
  Determine if CPU supports Intel Turbo Boost Max Technology 3.0 (ITBM).

  @retval Bit is set if ITBM is supported
**/
BOOLEAN
GetItbmSupportedStatus (
  VOID
  )
{
  CPUID_THERMAL_POWER_MANAGEMENT_EAX PowerEax;

  AsmCpuidEx (
    CPUID_THERMAL_POWER_MANAGEMENT,
    0,
    &PowerEax.Uint32,
    NULL,
    NULL,
    NULL
    );
  return (BOOLEAN) (PowerEax.Bits.TurboBoostMaxTechnology30 != 0);
}

/**
  Return Total Memory Encryption (TME) default policy setting

  @retval Total Memory Encryption (TME) default policy setting
**/
UINT32
TmeDefaultSetting (
  VOID
  )
{
  return 1;
}

/**
  Disable or enable CPU Chashlog dump feature by MSR.

  @param[in] BOOLEAN      Policy for CrashLog
**/
VOID
CpuCrashLogMsrEnable (
  IN  BOOLEAN                 CrashLogEnable
  )
{
  MSR_CRASHLOG_CONTROL_REGISTER    CurrentValue;
  MSR_CRASHLOG_CONTROL_REGISTER    ExpectedValue;

  CurrentValue.Uint64 = AsmReadMsr64 (MSR_CRASHLOG_CONTROL);

  ExpectedValue.Uint64 = CurrentValue.Uint64;

  if (CrashLogEnable) {
    ExpectedValue.Bits.Cddis = 0;
  } else {
    ExpectedValue.Bits.Cddis = 1;
  }

  if (CurrentValue.Uint64 != ExpectedValue.Uint64) {
    //
    // MSR value does not match policy.
    // Write MSR with the expected value.
    //
    AsmWriteMsr64 (MSR_CRASHLOG_CONTROL, ExpectedValue.Uint64);
  }
}

/**
  This function is used to Patch SmmSupovrStateLock.

  @retval This corresponds to bit 2 of MSR_SMM_SUPOVR_STATE_LOCK_REGISTER. When set, prevents WRMSR to IA32_SMM_MONITOR_CTL (aka MSEG) MSR.

**/
BOOLEAN
SmmSupovrStateLockPatch (
  VOID
  )
{
  return 1;
}

/**
  This function returns the supported Physical Address Size

  @retval returns the supported Physical Address Size.
**/
UINT8
GetMaxPhysicalAddressSizeFru (
  VOID
  )
{
  EFI_CPUID_REGISTER Cpuid;
  AsmCpuid (CPUID_VIR_PHY_ADDRESS_SIZE, &Cpuid.RegEax, NULL, NULL, NULL);
  return (UINT8) Cpuid.RegEax;
}

/**
  Initialize the Core CLOS for LLC Cache QOS mask.

**/
VOID
InitializeCoreClos (
  VOID
  )
{
  return;
}

/**
  This function return the Heuristics Policy Data for C6DRAM.

  @retval Return the Heuristics Policy Data for C6DRAM.
**/
UINT8
GetC6DramHeuristicsPolicyData (
  VOID
  )
{
  //
  // Set BIT0 (Forced ON) and BIT1 (Reserved).
  //
  return 3;
}
