/** @file
  This file contains MSR access for specific generation.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/

#include <Library/DebugLib.h>
#include <Library/MsrFruLib.h>
#include <Library/BaseLib.h>
#include <Register/TglMsr.h>
#include <CpuRegs.h>
#include <Register/Cpuid.h>

/**
  This function get MSR_FIT_BIOS_ERROR.

  @param[out]  FitEntryType
  @param[out]  FitErrorCode
**/
VOID
MsrGetFitBiosError (
  OUT UINT8 *FitEntryType,
  OUT UINT8 *FitErrorCode
  )
{
  TGL_MSR_FIT_BIOS_ERROR_REGISTER FitData;
  FitData.Uint64 = AsmReadMsr64 (TGL_MSR_FIT_BIOS_ERROR);
  *FitEntryType = (UINT8) FitData.Bits.EntryType;
  *FitErrorCode = (UINT8) FitData.Bits.ErrorCode;
}

/**
  This function returns if programable TJ offset enable

  @retval TRUE if programable TJ offset is supported
          FALSE if programable TJ offset is not supported
**/
BOOLEAN
MsrIsPrgTjOffsetEn (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return PlatformInfoMsr.Bits.PrgTjOffsetEn != 0;
}

/**
  This function return max and min bus ratio.

  @param[out]  MaxBusRatio  Maximum Non-Turbo Ratio
  @param[out]  MinBusRatio  Max efficiency ratio
**/
VOID
MsrGetBusRatio (
  OUT UINT8 *MaxBusRatio, OPTIONAL
  OUT UINT8 *MinBusRatio  OPTIONAL
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;

  if (MaxBusRatio != NULL || MinBusRatio != NULL) {
    PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
    if (MaxBusRatio != NULL) {
      *MaxBusRatio = (UINT8) PlatformInfoMsr.Bits.MaxNonTurboLimRatio;
    }
    if (MinBusRatio != NULL) {
      *MinBusRatio = (UINT8) PlatformInfoMsr.Bits.MaxEfficiencyRatio;
    }
  }
}

/**
  This function return max Efficiency Ratio.

  @retval Max efficiency ratio
**/
UINT8
MsrGetMaxEfficiencyRatio (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return (UINT8) PlatformInfoMsr.Bits.MaxEfficiencyRatio;
}

/**
  This function return max Non-Turbo Ratio.

  @retval Max Non-Turbo Ratio
**/
UINT8
MsrGetMaxNonTurboRatio (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return (UINT8) PlatformInfoMsr.Bits.MaxNonTurboLimRatio;
}

/**
  This function return the supported Config TDP Levels.

  @retval number of config TDP levels
**/
UINT8
MsrGetConfigTdpLevels (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return (UINT8) PlatformInfoMsr.Bits.ConfigTdpLevels;
}

/**
  This function returns if PPIN feature present

  @retval TRUE if PPIN feature present
          FALSE if PPIN feature not present
**/
BOOLEAN
MsrIsPpinCap (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return PlatformInfoMsr.Bits.PpinCap != 0;
}

/**
  This function returns if MCU 2nd patch load success

  @retval TRUE if MCU 2nd patch success
          FALSE if MCU 2nd patch not success
**/
BOOLEAN
MsrIsMcu2ndPatchSuccess (
  VOID
  )
{
  //
  // The status bit of 2nd patch is not supported in MSR,
  // so always returns TRUE here.
  //
  return TRUE;
}

/**
  This function returns if Turbo RatioLimit is Programmable or not

  @retval TRUE if Turbo RatioLimit is Programmable
          FALSE if Turbo RatioLimit is Not Programmable
**/
BOOLEAN
MsrIsTurboRatioLimitProgrammable (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return PlatformInfoMsr.Bits.PrgTurboRatioEn != 0;
}

/**
  This function returns if TDP Limit is Programmable or not

  @retval TRUE if TDP Limit is Programmable
          FALSE if TDP Limit is Not Programmable
**/
BOOLEAN
MsrIsTdpLimitProgrammable (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return PlatformInfoMsr.Bits.PrgTdpLimEn != 0;
}

/**
  This function returns if Timed Mwait is Supported or Not

  @retval TRUE if Timed Mwait is Supported
          FALSE if Timed Mwait is Not Supported
**/
BOOLEAN
MsrIsTimedMwaitSupported (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return PlatformInfoMsr.Bits.TimedMwaitEnable != 0;
}

/**
  Return if CPU supports PFAT

  @retval TRUE             If CPU Supports
  @retval FALSE            If CPU doesn't Supports
**/
BOOLEAN
MsrIsPfatEnabled (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER  PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return (BOOLEAN) PlatformInfoMsr.Bits.PfatEnable;
}

/**
  Return if CPU Supports LPM

  @retval TRUE             If Supports
  @retval FALSE            If not supports
**/
BOOLEAN
MsrIsCpuSupportsLpm (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER  PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return (BOOLEAN) PlatformInfoMsr.Bits.LpmSupport;
}

/**
  This function is to get platform PowerLimit.

  @param[out]  PowerLimit1
  @param[out]  PowerLimit1En
  @param[out]  PowerLimit1Time
  @param[out]  PowerLimit2
  @param[out]  PowerLimit2En
  @param[out]  Lock
**/
VOID
MsrGetPlatformPowerLimit (
  OUT UINT32   *PowerLimit1,
  OUT BOOLEAN  *PowerLimit1En,
  OUT UINT32   *PowerLimit1Time,
  OUT UINT32   *PowerLimit2,
  OUT BOOLEAN  *PowerLimit2En,
  OUT BOOLEAN  *Lock
  )
{
  TGL_MSR_PLATFORM_POWER_LIMIT_REGISTER PlatformPowerLimitMsr;

  PlatformPowerLimitMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_POWER_LIMIT);

  if (PowerLimit1 != NULL) {
    *PowerLimit1 = PlatformPowerLimitMsr.Bits.PowerLimit1;
  }

  if (PowerLimit1En != NULL) {
    *PowerLimit1En = (BOOLEAN) PlatformPowerLimitMsr.Bits.PowerLimit1En;
  }

  if (PowerLimit1Time != NULL) {
    *PowerLimit1Time = PlatformPowerLimitMsr.Bits.PowerLimit1Time;
  }

  if (PowerLimit2 != NULL) {
    *PowerLimit2 = PlatformPowerLimitMsr.Bits.PowerLimit2;
  }

  if (PowerLimit2En != NULL) {
    *PowerLimit2En = (BOOLEAN) PlatformPowerLimitMsr.Bits.PowerLimit2En;
  }

  if (Lock != NULL) {
    *Lock = (BOOLEAN) PlatformPowerLimitMsr.Bits.Lock;
  }
}

/**
  This function is to set platform PowerLimit. All input parameters
  must be provided to override the existing platform PowerLimit value.

  @param[in]  PowerLimit1
  @param[in]  PowerLimit1En
  @param[in]  PowerLimit1Time
  @param[in]  PowerLimit2
  @param[in]  PowerLimit2En
  @param[in]  Lock
**/
VOID
MsrSetPlatformPowerLimit (
  IN UINT32   PowerLimit1,
  IN BOOLEAN  PowerLimit1En,
  IN UINT32   PowerLimit1Time,
  IN UINT32   PowerLimit2,
  IN BOOLEAN  PowerLimit2En,
  IN BOOLEAN  Lock
  )
{
  TGL_MSR_PLATFORM_POWER_LIMIT_REGISTER PlatformPowerLimitMsr;

  PlatformPowerLimitMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_POWER_LIMIT);

  PlatformPowerLimitMsr.Bits.PowerLimit1 = PowerLimit1;
  PlatformPowerLimitMsr.Bits.PowerLimit1En = (UINT32) PowerLimit1En;
  //
  // Always set to allows processor to go below the OS requested P States.
  //
  PlatformPowerLimitMsr.Bits.CriticalPowerClamp1 = 1;
  PlatformPowerLimitMsr.Bits.PowerLimit1Time = PowerLimit1Time;
  PlatformPowerLimitMsr.Bits.PowerLimit2 = PowerLimit2;
  PlatformPowerLimitMsr.Bits.PowerLimit2En = (UINT32) PowerLimit2En;
  PlatformPowerLimitMsr.Bits.Lock = (UINT32) Lock;

  AsmWriteMsr64 (TGL_MSR_PLATFORM_POWER_LIMIT, PlatformPowerLimitMsr.Uint64);
}

/**
  Return if CPU Supports SMM_SUPOVR_STATE_LOCK

  @retval TRUE             If CPU supports
  @retval FALSE            If CPU doesn't support
**/
BOOLEAN
MsrIsSmmSupovrStateLockSupported (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER  PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return (BOOLEAN) PlatformInfoMsr.Bits.SmmSupovrStateLockEnable;
}

/**
  This function return the supported Prmmr Size

  @retval  Supported Prmrr Size
**/
UINT32
MsrGetSupportedPrmrrSize (
  VOID
  )
{
  TGL_MSR_PRMRR_VALID_CONFIG_REGISTER  PrmrrValidConfigMsr;
  PrmrrValidConfigMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PRMRR_VALID_CONFIG);
  return (UINT32) PrmrrValidConfigMsr.Uint64;
}

/**
  This function returns bit per logical processor indicating whether
  the logical processor is in the middle of long flow and hence will
  delay servicing of SMI.

  @retval Bit per logical processor

**/
UINT64
MsrGetSmmDelayed (
  VOID
  )
{
  TGL_MSR_SMM_DELAYED_REGISTER  SmmDelayed;
  SmmDelayed.Uint64 = AsmReadMsr64 (TGL_MSR_SMM_DELAYED);
  return SmmDelayed.Bits.LogProc;
}

/**
  This function returns bit per logical processor indicating whether
  the logical processor is in state where SMIs are blocked.

  @retval Bit per logical processor

**/
UINT64
MsrGetSmmBlocked (
  VOID
  )
{
  TGL_MSR_SMM_BLOCKED_REGISTER  SmmBlocked;
  SmmBlocked.Uint64 = AsmReadMsr64 (TGL_MSR_SMM_BLOCKED);
  return SmmBlocked.Bits.LogProc;
}

/**
  Return if Edram Enable

  @retval TRUE             If Edram Enable
  @retval FALSE            If Edram Disable
**/
BOOLEAN
MsrIsEdramEnable(
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER  PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return PlatformInfoMsr.Bits.EdramEnable != 0;
}

/**
  Sets WRC COS of specific policy

  @param[in] UINT8   Wrc COS Policy to set ways
  @param[in] UINT32  Cos Ways Mask

  @retval EFI_SUCCESS           set WRC COS successful
  @retval EFI_INVALID_PARAMETER invalid COS policy
**/
EFI_STATUS
MsrSetWrcCos (
  IN UINT8     CosPolicy,
  IN UINT32    CosWaysMask
  )
{
  TGL_MSR_WRC_COS_WAY_MASK_0_0_REGISTER WrcCosMsr;
  UINT32                                MsrAddress;

  switch (CosPolicy) {
    case 0:
      MsrAddress = TGL_MSR_WRC_COS_WAY_MASK_0_0;
      break;
    case 1:
      MsrAddress = TGL_MSR_WRC_COS_WAY_MASK_1_1;
      break;
    case 2:
      MsrAddress = TGL_MSR_WRC_COS_WAY_MASK_2_2;
      break;
    case 3:
      MsrAddress = TGL_MSR_WRC_COS_WAY_MASK_3_3;
      break;
    default:
      return EFI_INVALID_PARAMETER;
  }

  WrcCosMsr.Uint64 = AsmReadMsr64 (MsrAddress);
  WrcCosMsr.Bits.CosWaysMask = CosWaysMask;

  AsmWriteMsr64 (MsrAddress, WrcCosMsr.Uint64);

  return EFI_SUCCESS;
}

/**
  This function is to set MemoryControlSplitLock Disable
**/
VOID
MsrDisableMemoryControlSplitLock (
  VOID
  )
{
  TGL_MSR_MEMORY_CONTROL_REGISTER  MsrMemoryControl;
  MsrMemoryControl.Uint64 = AsmReadMsr64 (TGL_MSR_MEMORY_CONTROL);
  MsrMemoryControl.Bits.SplitLockDisable = 1;
  AsmWriteMsr64 (TGL_MSR_MEMORY_CONTROL, MsrMemoryControl.Uint64);
}

/**
  Return if the processor is a preproduction sample

  @retval TRUE          If the processor is a preproduction sample
  @retval FALSE         If the part is intended for production
**/
BOOLEAN
MsrIsSamplePart (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER    PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return PlatformInfoMsr.Bits.SamplePart != 0;
}

/**
  Return if FME_ACTIVATE MSR can be setup and
  the FZM functionality can be enabled

  @retval TRUE          If FME_ACTIVATE MSR can be setup and
                        the FZM functionality can be enabled
  @retval FALSE         If FME_ACTIVATE MSR can not be setup and
                        the FZM functionality can not be enabled
**/
BOOLEAN
MsrIsSxp2lmEnabled (
  VOID
  )
{
  TGL_MSR_PLATFORM_INFO_REGISTER    PlatformInfoMsr;
  PlatformInfoMsr.Uint64 = AsmReadMsr64 (TGL_MSR_PLATFORM_INFO);
  return PlatformInfoMsr.Bits.Sxp2lmEnable != 0;
}

/**
  This function is to Enable RAR Timer
**/
VOID
MsrEnableRarTimer (
  VOID
  )
{
  TGL_MSR_RAR_TIMER_CONFIG_REGISTER   RarTimerMsr;
  CPUID_VERSION_INFO_EAX              Eax;
  ///
  /// Read the CPUID information
  ///
  AsmCpuid (CPUID_VERSION_INFO, &Eax.Uint32, NULL, NULL, NULL);
  if (Eax.Bits.SteppingId >= EnumTglB0) {
    RarTimerMsr.Uint64 = AsmReadMsr64 (TGL_MSR_RAR_TIMER_CONFIG);
    //
    // Set RAR Timer to 10 micro Sec
    //
    RarTimerMsr.Bits.Threshold = 0x4;
    AsmWriteMsr64 (TGL_MSR_RAR_TIMER_CONFIG, RarTimerMsr.Uint64);
  }
  return;
}
