/** @file
  Fru file for 2LM feature initialization.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/PeiTwoLmInitFruLib.h>
#include <Library/PmcPrivateLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/PeiServicesTablePointerLib.h>
#include <IndustryStandard/Pci22.h>
#include <Include/PcieRegs.h>
#include <Register/SaRegsHostBridge.h>
#include <Library/TimerLib.h>
#include <Library/PciExpressHelpersLib.h>
#include <Library/PcieHelperLib.h>
#include <CpuRegs.h>
#include <CpuSbInfo.h>
#include <Library/IoLib.h>
#include <Library/CpuPlatformLib.h>
#include <Register/TwoLmRegs.h>
#include <Register/CpuPcieRegs.h>
#include <Library/CpuPcieInitCommon.h>

/**
  This function initializes Astro Near Memory Controller.

  @param[in]    TwoLmInfo    2LM HOB which will have necessary values to initialize Astro.

  @retval       EFI_SUCCESS      Successfully initialized Astro Near Memory Controller.
**/
EFI_STATUS
EFIAPI
AstroInit (
  IN TWOLM_INFO    *TwoLmInfo
)
{
  UINTN             Edrambar;
  UINT32            Data32;
  UINT32            Data32Or;
  UINT32            Data32And;
  UINT64            Data64Or;
  UINT64            Data64And;
  UINT16            MpcapRemainder;
  UINT16            MpcapDivident;
  EFI_BOOT_MODE     BootMode;
  EFI_STATUS        Status;
  UINT8             Index;


  DEBUG ((DEBUG_INFO, "2LM: Astro Initialization Start\n"));
  Edrambar = (UINTN) GetEdramBarValue();
  if(!Edrambar) {
    ASSERT (FALSE);
    return EFI_UNSUPPORTED;
  }

  Status = PeiServicesGetBootMode (&BootMode);
  ASSERT_EFI_ERROR (Status);

  if (TwoLmInfo->CurrentMemBootMode == MEM_BOOT_MODE_2LM ) {
    //
    // Programming Tag_RT_Init Register
    //
    if ((BootMode == BOOT_ON_S3_RESUME) || (BootMode == BOOT_ON_FLASH_UPDATE)) {
      Data32Or = (1 << N_TC_INIT_STYLE_OFFSET) | (0 << N_TAG_STORE_CLEAR_OFFSET) | (1 << N_RT_RESTORE_OFFSET);
    } else {
      Data32Or = (1 << N_TC_INIT_STYLE_OFFSET) | (1 << N_TAG_STORE_CLEAR_OFFSET) | (0 << N_RT_RESTORE_OFFSET);
    }
    Data32And = ~(UINT32) (B_TC_INIT_STYLE_MASK | B_TAG_STORE_CLEAR_MASK | B_RT_RESTORE_MASK);
    MmioAndThenOr32 (Edrambar + ASTRO_OFFSET + R_ASTRO_TAG_RT_INIT, Data32And, Data32Or);
    DEBUG ((DEBUG_INFO, "Astro Tag_Rt_Init = %x\n", MmioRead32 (Edrambar + ASTRO_OFFSET + R_ASTRO_TAG_RT_INIT)));

    Data64And = ~(UINT64) (B_SXP_SIZE_MASK | B_PMEM_EN | B_PMEM_SIZE_MASK);
    //
    // SXP_SIZE field is in GigaByte alignment.
    //
    Data64Or = ((((TwoLmInfo->FarMemSize) + (TwoLmInfo->PMemSize))/1024) << B_SXP_SIZE_OFFSET); // size in GB
    if (TwoLmInfo->PMemSize != 0) {
      Data64Or = Data64Or | B_PMEM_EN;
      Data64Or = Data64Or | LShiftU64(((TwoLmInfo->FarMemSize)/1024), B_PMEM_SIZE_OFFSET); // size in GB
    }
    MmioAndThenOr64 (Edrambar + ASTRO_OFFSET + R_ASTRO_SXP_CONFIG_1, Data64And, Data64Or);
    MmioAndThenOr64 (Edrambar + ASTRO_OFFSET + R_ASTRO_SXP_CONFIG_2, Data64And, Data64Or);

    Data32And = (UINT32) (~B_DDR_SIZE_MASK);
    if ( TwoLmInfo->NearMemSize <= TWOLM_NM_SIZE_UPPER_LIMIT) {
      Data32Or = (((TwoLmInfo->NearMemSize)/1024) << B_DDR_SIZE_OFFSET); // size in GB
    } else { // NM > 32GB
      Data32Or = ((TWOLM_NM_SIZE_UPPER_LIMIT/1024) << B_DDR_SIZE_OFFSET); // size in GB
    }
    MmioAndThenOr32 (Edrambar + ASTRO_OFFSET + R_ASTRO_DDR_CONFIG, Data32And, Data32Or);
  } else { //Reset default reg values to 0 (currently default is non zero )
    MmioWrite64 (Edrambar + ASTRO_OFFSET + R_ASTRO_SXP_CONFIG_1, 0x0);
    MmioWrite64 (Edrambar + ASTRO_OFFSET + R_ASTRO_SXP_CONFIG_2, 0x0);
    MmioWrite32 (Edrambar + ASTRO_OFFSET + R_ASTRO_DDR_CONFIG, 0x0);
  }

  // If PMem is enabled then set the MPCAP field
  // a) BIOS reads "Random Write" field Byte 235 of FIS Identify command
  // b) Multiply Random write value with 7 (it has to be 8 seconds. Using 7 second to give some buffer)
  // c) write the result to MPCAP field in ASTRO
  //    mpcap[15:0] Limit for Modified persistent capacity in GB
  //    Note that the bottom nibble is below the hexadecimal point, i.e. the
  //    value is 0xNNN.N. For example, if the value is 0x00A8, it means 10.5 GB.
  // d) Enable "Persistent-aware scrub" in scrub_type field.
  // e) tpersist[23:16] This is the period in which the scrubber should complete an iteration.
  //    BIOS to program value of 5min to this field. This value may need further tuning post Si.

  if (TwoLmInfo->PMemSize != 0) {
    UINT32 Mpcap = TwoLmInfo->RandomWriteSpeed * 100 * 7; //in MB
    MpcapRemainder = (UINT16)(Mpcap % 1024); // find the remainder
    MpcapRemainder = MpcapRemainder / 100 ; // taking MSB nibble of remainder
    MpcapRemainder = (MpcapRemainder * 16) / 10 ; // taking the hex equivalent for the below hexadecimal point nibble
    MpcapDivident = (UINT16)(Mpcap / 1024);
    Data32 = (MpcapRemainder << B_MPCAP_SIZE_FRC_OFFSET) | (MpcapDivident << B_MPCAP_SIZE_INT_OFFSET) |
              (1 << B_SCRUB_TYPE_SIZE_OFFSET) | (5 << B_TPERSIST_SIZE_OFFSET);
    MmioWrite32 (Edrambar + ASTRO_OFFSET + R_ASTRO_SCRUBBER_PCODE_CTL, Data32);
  }

  // Temporary BZM range registers programming to avoid tracking r/w to address zero.
  if ((BootMode == BOOT_ON_S3_RESUME) || (BootMode == BOOT_ON_FLASH_UPDATE)) {
  } else {
    for (Index = 0; Index < 8; ++Index) {
      MmioWrite64 (Edrambar + ASTRO_OFFSET + R_ASTRO_BZM_START_ADDR0 + (Index * 16), (UINT64) (~0));
      MmioWrite64 (Edrambar + ASTRO_OFFSET + R_ASTRO_BZM_END_ADDR0   + (Index * 16), (UINT64) (~0));
      DEBUG ((DEBUG_VERBOSE, "2LM: BZM Register Start offset is 0x%x \t value is 0X %0lx\n",(R_ASTRO_BZM_START_ADDR0 + (Index * 16)), MmioRead64 (0xFED83800 + R_ASTRO_BZM_START_ADDR0 + (Index*16))));
      DEBUG ((DEBUG_VERBOSE, "2LM: BZM Register End offset is 0x%x \t value is 0X %0lx\n",(R_ASTRO_BZM_END_ADDR0 + (Index * 16)), MmioRead64 (0xFED83800 + R_ASTRO_BZM_END_ADDR0 + (Index*16)) ));
    }
    DEBUG ((DEBUG_VERBOSE, "2LM: BZM Status register is 0x%x\n", MmioRead64 (Edrambar + ASTRO_OFFSET + R_ASTRO_BZM_WRITE_STATUS)));
  }
  return EFI_SUCCESS;
}
