/** @file
  This file updates voltage regulator overrides and  programs Fivr Rfi settings.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019-2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/
#include <Library/DebugLib.h>
#include <Library/PeiServicesLib.h>
#include <Library/CpuMailboxLib.h>
#include <Library/ConfigBlockLib.h>
#include <Library/CpuPlatformLib.h>
#include <CpuAccess.h>
#include <Library/IoLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Ppi/SiPolicy.h>
#include <Register/Cpuid.h>
#include <Library/PeiPmcPrivateLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/VoltageRegulatorDomains.h>
#include <Library/VoltageRegulatorCommands.h>
#include <Library/PeiVrLib.h>

//
// ENABLE_RFI_MITIGATION can only be used prior to BIOS_CPL,
// using it afterwards returns MAILBOX_BIOS_CC_COMMAND_LOCKED.
//
#define MAILBOX_BIOS_CMD_ENABLE_RFI_MITIGATION                         0x44
#define MAILBOX_BIOS_CMD_ENABLE_RFI_MITIGATION_IS_ENABLED              1
#define MAILBOX_BIOS_CMD_ENABLE_RFI_MITIGATION_IS_DISABLED             0

GLOBAL_REMOVE_IF_UNREFERENCED CPU_VR_OVERRIDE_TABLE mCpuVrOverrideTable[] = {
///
/// Cpu Identifier              Icc Current Limit    Tdc Current Limit    AC Loadlines     DC Loadlines
///                                   1/4A               1/8A                1/100mOhm       1/100mOhm
  { EnumTglY9WattES1CpuId,            42*4,              27*8,               540,            200},
  { EnumTglY9Watt42fCpuId,            42*4,              27*8,               540,            200},
  { EnumTglU28WattES1CTdpCpuId,       55*4,              36*8,               440,            200},
  { EnumTglU28Watt42f15WCTdpCpuId,    55*4,              36*8,               440,            200},
  { EnumTglU28Watt42f12WCTdpCpuId,    55*4,              36*8,               440,            200},
  { EnumTglU28Watt42f10WCTdpCpuId,    55*4,              36*8,               440,            200},
  { EnumTglU28Watt42fCpuId,           65*4,              43*8,               440,            200},
  { EnumTglU28WattES1CpuId,           65*4,              43*8,               440,            200},
  { EnumTglU15Watt22CpuId,            31*4,              20*8,               440,            200},
  { EnumTglH45Watt81CpuId,           110*4,              87*8,               200,            170},
  { EnumTglH45Watt81CTdpCpuId,       136*4,             102*8,               200,            170}
};


/**
  Updates the Vr Config block with Intel default override values if needed

  @param[IN] CpuPowerMgmtVrConfig    CPU Power Management VR Config Block
  @param[in]  SelectedCtdpLevel      Ctdp Level Selected in BIOS
**/
VOID
UpdateVrOverrides (
  IN CPU_POWER_MGMT_VR_CONFIG *CpuPowerMgmtVrConfig,
  IN UINT16                   SelectedCtdpLevel
  )
{
  EFI_STATUS                  Status;
  UINTN                       TableIndex;
  UINTN                       NoOfOverrides;
  CPU_VR_OVERRIDE_TABLE       *VrOverrideTable;
  CPU_IDENTIFIER              CpuIdentifier;
  SI_PREMEM_POLICY_PPI        *SiPreMemPolicyPpi;
  OVERCLOCKING_PREMEM_CONFIG  *OverClockingConfig;

  VrOverrideTable = mCpuVrOverrideTable;
  NoOfOverrides = 0;
  OverClockingConfig = NULL;

  Status = PeiServicesLocatePpi (
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **) &SiPreMemPolicyPpi
             );
  if ((Status == EFI_SUCCESS) && (SiPreMemPolicyPpi != NULL)) {
    Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gOverclockingPreMemConfigGuid, (VOID *) &OverClockingConfig);
    if (!EFI_ERROR (Status)) {
      ///
      /// When overclocking is enabled, we don't update the VR settings so that OC systems
      /// rely solely on config block input values.
      ///
      if (OverClockingConfig->OcSupport == 1) {
        DEBUG ((DEBUG_INFO, "VR: Overclocking is enabled. Bypassing VR overrides.\n"));
        return;
      }
    }
  }

  ///
  /// Get CpuIdentifier to identify which set of VR values we need to override
  ///
  DEBUG ((DEBUG_INFO, "VR: SelectedCtdpLevel = %X\n", SelectedCtdpLevel));
  CpuIdentifier = GetCpuIdentifierWithCtdp (SelectedCtdpLevel);
  DEBUG ((DEBUG_INFO, "VR: Cpu Identifier = %X\n", CpuIdentifier));
  if (CpuIdentifier == EnumUnknownCpuId) {
    DEBUG ((DEBUG_ERROR, "VR: Unknown Cpu Identifier, bypassing VR overrides.\n"));
    return;
  }

  ///
  /// Update the VR config block with the VR override table data..
  ///
  NoOfOverrides = (sizeof (mCpuVrOverrideTable)) / (sizeof (CPU_VR_OVERRIDE_TABLE));

  for (TableIndex = 0; TableIndex < NoOfOverrides; TableIndex++, VrOverrideTable++) {
    ///
    /// If Cpu Identifier matches, then update with overrides values
    ///
    if (VrOverrideTable->CpuIdentifier == CpuIdentifier) {
      ///
      /// ICC Max Overrides
      ///
      if (VrOverrideTable->VccInIccMax != 0) {
        CpuPowerMgmtVrConfig->IccMax[CPU_VR_DOMAIN_VCCIN] = VrOverrideTable->VccInIccMax;
      }

      ///
      /// VR TDC Overrides
      ///
      if (VrOverrideTable->VccInTdclimit != 0) {
        CpuPowerMgmtVrConfig->TdcCurrentLimit[CPU_VR_DOMAIN_VCCIN] = VrOverrideTable->VccInTdclimit;
        CpuPowerMgmtVrConfig->TdcEnable[CPU_VR_DOMAIN_VCCIN] = 1;
      }

      ///
      /// AC/DC Loadlines
      ///
      if (VrOverrideTable->VccInAcLoadLine != 0) {
        CpuPowerMgmtVrConfig->AcLoadline[CPU_VR_DOMAIN_VCCIN] = VrOverrideTable->VccInAcLoadLine;
      }
      if (VrOverrideTable->VccInDcLoadLine != 0) {
        CpuPowerMgmtVrConfig->DcLoadline[CPU_VR_DOMAIN_VCCIN] = VrOverrideTable->VccInDcLoadLine;
      }

      ///
      /// Exit loop when the correct CPU overrides have been applied.
      ///
      return;
    }
  }
  ///
  /// No matching CpuIdentifier found in the table
  ///
  DEBUG ((DEBUG_INFO, "No matching CpuIdentifier found in the VrOverride Table.\n"));
}

/**
  Enable RFI mitigation based on policy.

  @param[in] RfiMitigation      1: Enable
                                0: Disable

**/
VOID
EnableRfiMitigation (
  IN UINT8 RfiMitigation
  )
{
  UINT8         CpuSku;
  EFI_STATUS    Status;
  UINT32        MailboxStatus;

  CpuSku = GetCpuSku ();
  if ((CpuSku == EnumCpuUlt) || (CpuSku == EnumCpuUlx)) {
    if (RfiMitigation == 1) {
      Status = MailboxWrite (
                 MAILBOX_TYPE_PCODE,
                 MAILBOX_BIOS_CMD_ENABLE_RFI_MITIGATION,
                 MAILBOX_BIOS_CMD_ENABLE_RFI_MITIGATION_IS_ENABLED,
                 &MailboxStatus
                 );
      if (EFI_ERROR (Status) || (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS)) {
        DEBUG ((DEBUG_ERROR, "MAILBOX_BIOS_CMD_ENABLE_RFI_MITIGATION failed. EFI_STATUS = %r, Mailbox Status = 0x%x\n", Status, MailboxStatus));
      }
    }
  }
}

/**
  Programs the FIVR RFI settings based on XTAL clock frequency.

  @param[IN] CpuPowerMgmtVrConfig  - CPU Power Management VR Config Block
**/
VOID
UpdateFivrRfiSettings (
  IN CPU_POWER_MGMT_VR_CONFIG *CpuPowerMgmtVrConfig
  )
{
  UINTN                       MchBar;
  UINT32                      RfiData;
  UINT32                      RegEcx;
  UINT16                      RfiValue;
  UINT16                      DoubledValue;
  UINT8                       RfiLowBits;
  UINT8                       RfiHighByte;
  UINT8                       LowOffset;
  UINT8                       HighOffset;
  UINT32                      RfiMask;
  UINT32                      SpreadSpectrumData;
  UINT8                       EncodedSpectrum;
  //
  //for 38MHz clock frequency, CPU RFI frequency is 11bits and PCH RFI frequency is 10bits
  //
  UINT32                      RfiDataPch;
  UINT8                       LowOffsetPch;
  UINT32                      RfiMaskPch;

  EnableRfiMitigation ((UINT8) CpuPowerMgmtVrConfig->RfiMitigation);

  MchBar = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_MCHBAR)) & ~BIT0;
  SpreadSpectrumData = 0;
  if (CpuPowerMgmtVrConfig->FivrSpreadSpectrum == 0) {
    //
    // Disable FIVR Spread Spectrum
    //
    SpreadSpectrumData = SpreadSpectrumData & ~B_SPREAD_SPECTRUM_ENABLE;
    PmcConfigureFivrEmiControl0 (SpreadSpectrumData);
    DEBUG ((DEBUG_INFO, "Disable FIVR Spread Spectrum.\n"));
  } else if (CpuPowerMgmtVrConfig->FivrSpreadSpectrum != 0) {
    //
    // Sets clock spectrum spread percentage:
    // 0x00=0.2% , 0x3F=10%
    // 1 LSB = 0.1% increase in spread (for settings 0x01 thru 0x1C)
    // 1 LSB = 0.2% increase in spread (for settings 0x1E thru 0x3F)
    // The bit[7:6] are not used.
    //
    EncodedSpectrum = 0;
    if (CpuPowerMgmtVrConfig->FivrSpreadSpectrum > 2) {
      EncodedSpectrum = CpuPowerMgmtVrConfig->FivrSpreadSpectrum - 2;
      if (EncodedSpectrum > V_SPREAD_SPECTRUM_POINT_ONE_ENCODING_MAX) {
        EncodedSpectrum = EncodedSpectrum - V_SPREAD_SPECTRUM_POINT_ONE_ENCODING_MAX;
        EncodedSpectrum = EncodedSpectrum / 2;
        EncodedSpectrum = EncodedSpectrum + V_SPREAD_SPECTRUM_POINT_ONE_ENCODING_MAX;
        if (EncodedSpectrum > V_FIVR_SPREAD_SPECTRUM_MAX) {
          EncodedSpectrum = V_FIVR_SPREAD_SPECTRUM_MAX;
        }
      }
    }

    //
    // Write the Spread Spectrum value.
    //
    SpreadSpectrumData = MmioRead32 (MchBar + R_EMI_CONTROL_0_0_0_MCHBAR_PCU);
    SpreadSpectrumData = (SpreadSpectrumData & ~B_SPREAD_SPECTRUM_MASK);
    SpreadSpectrumData = SpreadSpectrumData | EncodedSpectrum;
    SpreadSpectrumData = SpreadSpectrumData | B_SPREAD_SPECTRUM_ENABLE;
    MmioWrite32 ((MchBar + R_EMI_CONTROL_0_0_0_MCHBAR_PCU), SpreadSpectrumData);

    //
    // Update PCH FIVR RFI setting based on RFIControl0/EMIControl0
    // settings on CPU side
    // @todo Clear reserved bits
    //
    PmcConfigureFivrEmiControl0 (SpreadSpectrumData);
  }

  if (CpuPowerMgmtVrConfig->FivrRfiFrequency != 0) {
    ///
    /// Determine XTAL clock frequency
    ///
    AsmCpuid (CPUID_TIME_STAMP_COUNTER, NULL, NULL, &RegEcx, NULL);

     if (RegEcx == CLOCK_FREQUENCY_38MHz) {
      RfiData = MmioRead32 (MchBar + R_RFI_CONTROL2_0_0_0_MCHBAR_PCU);
      DEBUG ((DEBUG_INFO, "Reading RFI Frequency, R_RFI_CONTROL2_0_0_0_MCHBAR_PCU = %X\n", RfiData));
      DEBUG ((DEBUG_INFO, "CpuPowerMgmtVrConfig->FivrRfiFrequency = %X\n", CpuPowerMgmtVrConfig->FivrRfiFrequency));
      //
      // RfiValue = Config->RfiFreq * 12800 / 38400.
      // RfiValue = Config->RfiFreq * 1 / 3.
      // Round to nearest by doubling before dividing by 3, then add 1 to the result and divide by 2.
      //
      DoubledValue = (CpuPowerMgmtVrConfig->FivrRfiFrequency * 2) / 3;
      RfiValue = (DoubledValue + 1) / 2;
      //
      // Set offsets and mask based on clock.
      //
      LowOffset = N_RFI_FREQ_LO_PF3_OFFSET;
      HighOffset = N_RFI_FREQ_HI_PF3_OFFSET;
      RfiMask = B_RFI_FREQ_PF3_MASK;
      if (GetCpuStepping () < EnumTglB0) {
        DEBUG ((DEBUG_INFO, "10bit RFI freq PCH support for 38MHz for stepping lower than B0\n"));
        LowOffsetPch = N_RFI_FREQ_LO_PF3_OFFSET2;
        RfiMaskPch = B_RFI_FREQ_PF3_MASK2;
      }
    } else {
      RfiData = MmioRead32 (MchBar + R_RFI_CONTROL_0_0_0_MCHBAR_PCU);
      ///
      /// RfiValue = Policy->RfiFreq * 100KHz * (128/ClockInKhz).
      ///
      if (RegEcx == CLOCK_FREQUENCY_19MHz) {
        //
        // RfiValue = Config->RfiFreq * 12800 / 19200.
        // RfiValue = Config->RfiFreq * 2 / 3.
        // Round to nearest by doubling before dividing by 3, then add 1 to the result and divide by 2.
        //
        DoubledValue = (CpuPowerMgmtVrConfig->FivrRfiFrequency * 4) / 3;
        RfiValue = (DoubledValue + 1) / 2;
        //
        // Set offsets and mask based on clock.
        //
        LowOffset = N_RFI_FREQ_LO_PF1_OFFSET;
        HighOffset = N_RFI_FREQ_HI_PF1_OFFSET;
        RfiMask = B_RFI_FREQ_PF1_MASK;
      } else if (RegEcx == CLOCK_FREQUENCY_24MHz) {
        //
        // RfiValue = Config->RfiFreq * 12800 / 24000.
        // RfiValue = Config->RfiFreq * 8 / 15.
        // Round to nearest by doubling before dividing by 15, then add 1 to the result and divide by 2.
        //
        DoubledValue = (CpuPowerMgmtVrConfig->FivrRfiFrequency * 16) / 15;
        RfiValue = (DoubledValue + 1) / 2;
        //
        // Set offsets and mask based on clock.
        //
        LowOffset = N_RFI_FREQ_LO_PF2_OFFSET;
        HighOffset = N_RFI_FREQ_HI_PF2_OFFSET;
        RfiMask = B_RFI_FREQ_PF2_MASK;
      } else {
        DEBUG ((DEBUG_ERROR, "Unexpected Clock Frequency! RegEcx: %d\n", RegEcx));
        return;
      }
    }
    if (RfiValue > V_MAX_RFI_VALUE) {
      RfiValue = V_MAX_RFI_VALUE;
    }
    if (RegEcx == CLOCK_FREQUENCY_38MHz) {
      RfiDataPch = RfiData;
    }

    if (RegEcx == CLOCK_FREQUENCY_38MHz) {
      RfiData = (RfiData & ~RfiMask);
      RfiLowBits = (RfiValue & (BIT2 | BIT1 | BIT0));
      RfiHighByte = (UINT8) (RfiValue >> 3);
      RfiData = RfiData | (UINT32) (RfiLowBits << LowOffset);
      RfiData = RfiData | (UINT32) (RfiHighByte << HighOffset);
      DEBUG ((DEBUG_INFO, "Writing RFI Frequency, R_RFI_CONTROL2_0_0_0_MCHBAR_PCU = %X\n", RfiData));
      MmioWrite32 ((MchBar + R_RFI_CONTROL2_0_0_0_MCHBAR_PCU), RfiData);
      //
      //Calculating the new RFI frequency for PCH with 10bits
      //
      if (GetCpuStepping () < EnumTglB0) {
        RfiData = (RfiDataPch & ~RfiMaskPch);
        RfiLowBits = (RfiValue & (BIT1 | BIT0));
        RfiHighByte = (UINT8) (RfiValue >> 2);
        RfiData = RfiData | (UINT32) (RfiLowBits << LowOffsetPch);
        RfiData = RfiData | (UINT32) (RfiHighByte << HighOffset);
        DEBUG ((DEBUG_INFO, "Writing RFI Frequency to PCH = %X\n", RfiData));
      }
    } else {
      RfiData = (RfiData & ~RfiMask);
      RfiLowBits = (RfiValue & (BIT1 | BIT0));
      RfiHighByte = (UINT8) (RfiValue >> 2);
      RfiData = RfiData | (UINT32) (RfiLowBits << LowOffset);
      RfiData = RfiData | (UINT32) (RfiHighByte << HighOffset);
      DEBUG ((DEBUG_INFO, "Writing RFI Frequency, R_RFI_CONTROL_0_0_0_MCHBAR_PCU = %X\n", RfiData));
      MmioWrite32 ((MchBar + R_RFI_CONTROL_0_0_0_MCHBAR_PCU), RfiData);
    }

    //
    // Update PCH FIVR RFI setting based on RFIControl0/EMIControl0
    // settings on CPU side
    // @todo Clear reserved bits
    //
    PmcConfigureFivrRfiControl4 (RfiData);
  }
}


/**
  Get Vr topologies which contain generation specific Vr address and bit which represents Svid Enabled/Disabled. Transferred as parameter from Fru to IP block.

  @param[IN,OUT] - VrDomainTopology - It takes VR_DOMAIN_TOPOLOGY as parameter and returns same which contains VR address and Svid Enable/Disabled.
**/
VOID
GetCpuVrTopology (
  IN OUT VR_DOMAIN_TOPOLOGY *VrDomainTopology
  )
{
  EFI_STATUS                  Status;
  UINT32                      MailboxStatus;
  UINT32                      MailboxType;
  VR_TOPOLOGY_DATA            VrTopology;
  UINTN                       VrIndex;

  ///
  /// CPU VR MSR mailbox
  ///
  MailboxType = MAILBOX_TYPE_VR_MSR;
  VrTopology.VrTopology = 0;

  ///
  /// Get CPU VR topology
  ///
  DEBUG ((DEBUG_INFO, "(MAILBOX) Mailbox Read Command = READ_VR_STRAP_CONFIG_CMD\n"));
  Status = MailboxRead (MailboxType, READ_VR_STRAP_CONFIG_CMD, (UINT32*)&VrTopology.VrTopology, &MailboxStatus);
  if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
    DEBUG ((DEBUG_ERROR, "VR: Error Reading VR topology. EFI_STATUS = %r, Mailbox Status = %X\n", Status, MailboxStatus));
  }

  ///
  /// Print VR Topology data
  ///
  DEBUG ((DEBUG_INFO, "VR: VrTopology data = 0x%x\n", VrTopology.VrTopology));
  DEBUG ((DEBUG_INFO, "    VR Type 0 = SVID, VR Type 1 = non-SVID\n"));
  DEBUG ((DEBUG_INFO, "    VCCIN VR Address  = 0x%x\n", VrTopology.Fields.FivrInVrAddress));
  DEBUG ((DEBUG_INFO, "    VCCIN VR Type     = 0x%x\n", VrTopology.Fields.FivrInVrType));

  ///
  /// Only VCCIN VR Exist
  ///
  VrIndex = CPU_VR_DOMAIN_VCCIN;
  VrDomainTopology->VrAddress[VrIndex] = (UINT8) VrTopology.Fields.FivrInVrAddress;
  VrDomainTopology->SvidEnabled[VrIndex] = (UINT8)(~VrTopology.Fields.FivrInVrType) & BIT0;
  VrDomainTopology->MinVrIndex = 0;
  VrDomainTopology->MaxVrIndex = 1;
}

/**
  Returns the maximum number of voltage regulator domains.

  @retval Maximum Number of VR Domains
**/
UINT16
GetMaxNumVrsFru (
  VOID
  )
{
  return 1;
}

/**
  This function uses mailbox communication to control settled time of Richtek VR which is enabled for
  VccSA to meet IMVP8 VR settled time.

  @param[IN] - VrIndex - It represent VR domains like SA,IA,RING,GT.
**/
VOID
VrSettledTimerRt (
  IN UINTN VrIndex
  )
{
  // This is needed only for Richtek VCCSA.
  return;
}
