/** @file
  CPU Dmi PEI Initialization library

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/PeiServicesLib.h>
#include <CpuRegs.h>
#include <Library/CpuDmiInfoLib.h>
#include <Library/CpuPlatformLib.h>
#include <Library/PchPcrLib.h>
#include <Library/PeiCpuDmiInitLib.h>
#include <Library/PeiCpuPcieVgaInitLib.h>
#include <Library/PeiPchDmiLib.h>
#include <ConfigBlock/SiConfig.h>
#include <Register/PchDmiRegs.h>
#include <Register/PchDmi15Regs.h>
#include <Register/SaPcieDmiRegs.h>
#include <Register/PchPcrRegs.h>
#include <CpuPcieInfo.h>
#include <HostBridgeConfig.h>
#include <Library/CpuMailboxLib.h>
typedef union {
  struct {
    UINT32  Low;
    UINT32  High;
  } Data32;
  UINT64 Data;
} UINT64_STRUCT;
#define LANE_STEP     0x10
#define BUNDLE_STEP   0x20

typedef struct {
  UINT32 RegOffset;
  UINT32 Value;
} CREDIT_CONTROL_INIT_ENTRY;

///
/// Controller Type
///
typedef enum {
  DMI_PORT = 0x0,
  PEG_PORT = 0x1,
} CONTROLLER_TYPE;
///
/// Functions
///

/**
  Reads, modifies and writes to PCODE mail box as per the input.

  @param[in]  CrOffset                    - Config Register Offset
  @param[in]  Data32And                   - Data to Clear
  @param[in]  Data32Or                    - Data to Set
**/
EFI_STATUS
PcodeMailboxReadThenWrite(
  IN  UINT32           CrOffset,
  IN  UINT32           Data32And,
  IN  UINT32           Data32Or,
  IN  CONTROLLER_TYPE  ControllerType
  )
{
  EFI_STATUS  Status;
  UINT32      LibStatus;
  UINT32      Data32;
  UINT32      TargetBlock;

  TargetBlock = 0;
  //
  // Target IO block 0 for DMI or 1 for PCI (Bit 24 in Mailbox Command)
  //
  if (ControllerType == PEG_PORT) {
    TargetBlock = 0x1 << N_MAILBOX_TARGET_IO_BLOCK_OFFSET;
  } else {
    TargetBlock = 0x0 << N_MAILBOX_TARGET_IO_BLOCK_OFFSET;
  }
  MailboxRead (MAILBOX_TYPE_PCODE, (READ_PEG_CRIO_CR | TargetBlock | CrOffset),&Data32,&LibStatus);
  if(LibStatus != PCODE_MAILBOX_CC_SUCCESS){
    DEBUG ((DEBUG_INFO, "PEG_CRIO_CR() - MailboxRead Error = %r\n", LibStatus));
  }
  Data32 &= Data32And;
  Data32 |= Data32Or;
  Status = MailboxWrite (MAILBOX_TYPE_PCODE, (WRITE_PEG_CRIO_CR | TargetBlock | CrOffset),Data32,&LibStatus);
  if((Status != EFI_SUCCESS)||(LibStatus != PCODE_MAILBOX_CC_SUCCESS)){
    DEBUG ((DEBUG_INFO, "PEG_CRIO_CR() - MailboxWrite Error = %r\n", LibStatus));
  }
  return Status;
}


/**
  Computes the Pre-Cursor, Cursor, and Post-Cursor from a preset

  @param[in]  Preset                      - Preset to compute coefficients for
  @param[in]  FullSwing                   - The full swing of the transmitter
  @param[out] PreCursor                   - Computed Pre-Cursor
  @param[out] Cursor                      - Computed Cursor
  @param[out] PostCursor                  - Computed Post-Cursor
**/
VOID
GetCoefficientsFromPreset (
  IN  UINT8                             Preset,
  IN  UINT8                             FullSwing,
  OUT UINT8                             *PreCursor,
  OUT UINT8                             *Cursor,
  OUT UINT8                             *PostCursor
)
{
  INT32   PreCursorMilli;
  INT32   PostCursorMilli;

  PreCursorMilli = 0;
  PostCursorMilli = 0;

  ///
  /// Get starting values from Table 4-16 of the PCIe Base Spec v3.0
  ///
  switch (Preset) {
  case  0:
    PreCursorMilli = 0;
    PostCursorMilli = -250;
    break;

  case  1:
    PreCursorMilli = 0;
    PostCursorMilli = -167;
    break;

  case  2:
    PreCursorMilli = 0;
    PostCursorMilli = -200;
    break;

  case  3:
    PreCursorMilli = 0;
    PostCursorMilli = -125;
    break;

  case  4:
    PreCursorMilli = 0;
    PostCursorMilli = 0;
    break;

  case  5:
    PreCursorMilli = -100;
    PostCursorMilli = 0;
    break;

  case  6:
    PreCursorMilli = -125;
    PostCursorMilli = 0;
    break;

  case  7:
    PreCursorMilli = -100;
    PostCursorMilli = -200;
    break;

  case  8:
    PreCursorMilli = -125;
    PostCursorMilli = -125;
    break;

  case  9:
    PreCursorMilli = -166;
    PostCursorMilli = 0;
    break;

  case 10:  ///< P10 is unsupported
  default:
    PreCursorMilli = -100;
    PostCursorMilli = -200;
    DEBUG((DEBUG_WARN, "GetCoefficientsFromPreset(): Unsupported Preset Requested: P%d. Using P7.\n", Preset));
    break;
  }

  ///
  /// Convert to absolute values
  ///
  if (PreCursorMilli < 0) {
    PreCursorMilli *= -1;
  }
  if (PostCursorMilli < 0) {
    PostCursorMilli *= -1;
  }

  ///
  /// Apply FullSwing
  ///
  PreCursorMilli *= FullSwing;
  PostCursorMilli *= FullSwing;

  ///
  /// Convert to integers
  ///
  *PreCursor = ((PreCursorMilli % 1000) >= 500) ? (UINT8)((PreCursorMilli / 1000) + 1) : (UINT8)(PreCursorMilli / 1000);
  *PostCursor = ((PostCursorMilli % 1000) >= 500) ? (UINT8)((PostCursorMilli / 1000) + 1) : (UINT8)(PostCursorMilli / 1000);
  *Cursor = FullSwing - (*PreCursor) - (*PostCursor);

  return;
}

/**
  Initialize DMI Tc/Vc mapping through SA-PCH.

  @retval EFI_SUCCESS
**/
EFI_STATUS
CpuDmiTcVcInit (
  VOID
  )
{
  EFI_STATUS                  Status;
  UINT64                      McD0BaseAddress;
  UINT64_STRUCT               DmiBar;
  PCH_DMI_TC_VC_MAP           PchDmiTcVcMap;

#if FixedPcdGetBool(PcdCpuPcieEnable) == 1
#endif

  McD0BaseAddress    = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, 0);
  DmiBar.Data32.High = PciSegmentRead32 (McD0BaseAddress + R_SA_DMIBAR + 4);
  DmiBar.Data32.Low  =  PciSegmentRead32 (McD0BaseAddress + R_SA_DMIBAR);
  DmiBar.Data       &= (UINT64) ~BIT0;

  ///
  /// SA OPI Initialization
  ///
  if (!IsPchLinkDmi ()) {
    MmioOr8 ((UINTN) (DmiBar.Data + 0xA78), BIT1);
  }

  ///
  /// Get default PchDmiTcVcMap
  ///
  ZeroMem (&PchDmiTcVcMap, sizeof (PCH_DMI_TC_VC_MAP));
  PchDmiTcVcMapInit (&PchDmiTcVcMap);

  ///
  /// Program NB TC/VC mapping
  ///
  SaSetDmiTcVcMapping (&PchDmiTcVcMap, DmiBar.Data);

  ///
  /// Call PchDmiTcVcProgPoll
  ///
  Status = PchDmiTcVcProgPoll (&PchDmiTcVcMap);
  ASSERT_EFI_ERROR (Status);

  ///
  /// Poll NB negotiation completion
  ///
  SaPollDmiVcStatus (&PchDmiTcVcMap, DmiBar.Data);
  ASSERT_EFI_ERROR (Status);

  return EFI_SUCCESS;

}

/**
  Map SA DMI TCs to VC

  @param[in] PchDmiTcVcMap        - Instance of PCH_DMI_TC_VC_MAP
  @param[in] DmiBar               - DMIBAR address

  @retval EFI_SUCCESS             -  Succeed.
  @retval EFI_INVALID_PARAMETER   -  Wrong phase parameter passed in.
**/
EFI_STATUS
SaSetDmiTcVcMapping (
  IN    PCH_DMI_TC_VC_MAP   *PchDmiTcVcMap,
  IN    UINT64              DmiBar
  )
{
  UINT32  Data32And;
  UINT32  Data32Or;
  UINT8   Index;
  UINT16  Register;
  UINT8   VcId;
  UINT8   VcMap[PchDmiVcTypeMax];

  ZeroMem (VcMap, PchDmiVcTypeMax);
  ///
  /// Set the TC/VC mappings
  ///
  for (Index = 0; Index < PchDmiTcTypeMax; Index++) {
    VcMap[PchDmiTcVcMap->DmiTc[Index].Vc] |= (BIT0 << PchDmiTcVcMap->DmiTc[Index].TcId);
  }
  ///
  /// System BIOS must perform the following steps for VC0 configuration.
  ///   Program the TCs/VC0 map by setting DMIBAR offset 014h [7:1] = '0111 101b'.
  ///
  /// Virtual Channel for ME (VCm) Configuration
  /// This is configured by ConfigMemMe
  ///
  /// Step1. Assign Virtual Channel ID 7 to VCm:
  ///    Programming the DMIVCMRCTL DMI Port Register DMIBAR Offset 038h[26:24] = '111b'.
  ///
  /// Step2. Enable VCm:
  ///    Programming the DMIVMPRCTL DMI Port Register DMIBAR Offset 038h[31] = '1b'.
  ///
  /// Step3. Enable VCm by programming the DMIVCMRCTL DMI Port Register DMIBAR Offset 038h[31] = '1b'.
  ///
  for (Index = 0; Index < PchDmiVcTypeMax; Index++) {
    if (PchDmiTcVcMap->DmiVc[Index].Enable == TRUE) {
      ///
      /// Map TCs to VC, Set the VC ID, Enable VC
      ///
      VcId = PchDmiTcVcMap->DmiVc[Index].VcId;

      Data32And = (UINT32) (~(V_SA_DMIBAR_DMIVCCTL_ID | B_SA_DMIBAR_DMIVCCTL_TVM_MASK));
      Data32Or = VcId << N_SA_DMIBAR_DMIVCCTL_ID;
      Data32Or |= (UINT32) VcMap[Index];
      Data32Or |= N_SA_DMIBAR_DMIVCCTL_EN;

      switch (Index) {
        case PchDmiVcTypeVc0:
          Register = R_SA_DMIBAR_DMIVC0RCTL_OFFSET;
          break;

        case PchDmiVcTypeVc1:
          Register = R_SA_DMIBAR_DMIVC1RCTL_OFFSET;
          break;

        case PchDmiVcTypeVcm:
          Register = R_SA_DMIBAR_DMIVCMRCTL_OFFSET;
          break;

        default:
          return EFI_INVALID_PARAMETER;
      }

      MmioAndThenOr32 ((UINTN) (DmiBar + Register), Data32And, Data32Or);
    }
  }
  return EFI_SUCCESS;
}

/**
  Poll SA DMI negotiation completion

  @param[in] PchDmiTcVcMap        - Instance of PCH_DMI_TC_VC_MAP
  @param[in] DmiBar               - DMIBAR address

  @retval EFI_SUCCESS             -  Succeed.
  @retval EFI_INVALID_PARAMETER   -  Wrong phase parameter passed in.
**/
EFI_STATUS
SaPollDmiVcStatus (
  IN    PCH_DMI_TC_VC_MAP   *PchDmiTcVcMap,
  IN    UINT64              DmiBar
  )
{
  UINT8   Index;
  UINT16  Register;

  ///
  /// 6.2.3.2 - Step 4, Poll until VC1 has been negotiated
  ///    Read the DMIVC1RSTS DMI Port Register Offset 026h until [1]==0
  ///
  /// 6.2.3.4 - Step4. Poll the VCm Negotiation Pending bit until it reads 0:
  ///    Read the DMIVCMRSTS DMI Port Register Offset 03Eh until [1]==0
  ///
  for (Index = 0; Index < PchDmiVcTypeMax; Index++) {
    if (PchDmiTcVcMap->DmiVc[Index].Enable == TRUE) {
      switch (Index) {
        case PchDmiVcTypeVc0:
          Register = R_SA_DMIBAR_DMIVC0RSTS_OFFSET;
          break;

        case PchDmiVcTypeVc1:
          Register = R_SA_DMIBAR_DMIVC1RSTS_OFFSET;
          break;

        case PchDmiVcTypeVcm:
          Register = R_SA_DMIBAR_DMIVCMRSTS_OFFSET;
          break;

        default:
          return EFI_INVALID_PARAMETER;
      }

      ///
      /// Wait for negotiation to complete
      ///
      while ((MmioRead16 ((UINTN) (DmiBar + Register)) & B_SA_DMIBAR_DMISTS_NP) != 0);
    }
  }

  return EFI_SUCCESS;
}

/**
  Programs static equalization phase 1 settings for DMI

  @param[in]  DmiBar                      - Current value for DMIBAR
  @param[in]  DmiGen3RootPortPreset       - RP Presets array, length must be == SA_DMI_MAX_LANE
  @param[in]  DmiGen3EndPointPreset       - EP Presets array, length must be == SA_DMI_MAX_LANE
  @param[in]  DmiGen3EndPointHint         - EP Hints array, length must be == SA_DMI_MAX_LANE
**/
VOID
EFIAPI
DmiProgramGen3EqPhase1 (
  IN  UINT64                            DmiBar,
  IN  UINT8                             *DmiGen3RootPortPresets,
  IN  UINT8                             *DmiGen3EndPointPresets,
  IN  UINT8                             *DmiGen3EndPointHints
  )
{
  UINT32        OrData;
  UINT8         BundleIndex;
  UINT8         LaneIndex;

  DEBUG ((DEBUG_INFO, "Programming DMI Gen3 Eq Phase1\n"));

  ///
  /// Do bounds checking on input
  ///
  for (LaneIndex = 0; LaneIndex < GetMaxDmiLanes(); LaneIndex++) {
    if (DmiGen3RootPortPresets[LaneIndex] > 9) {
      DmiGen3RootPortPresets[LaneIndex] = 4;
    }
    if (DmiGen3EndPointPresets[LaneIndex] > 9) {
      DmiGen3EndPointPresets[LaneIndex] = 7;
    }
    if (DmiGen3EndPointHints[LaneIndex] > 6) {
      DmiGen3EndPointHints[LaneIndex] = 2;
    }
  }


  for (BundleIndex = 0; BundleIndex < GetMaxDmiBundles(); BundleIndex++) {
    ///
    /// Compute data to program
    ///
    OrData  = (UINT32) (DmiGen3EndPointPresets[BundleIndex << 1] << 8);
    OrData |= (UINT32) (DmiGen3EndPointHints[BundleIndex << 1] << 12);
    OrData |= (UINT32) (DmiGen3EndPointPresets[ (BundleIndex << 1) + 1] << 24);
    OrData |= (UINT32) (DmiGen3EndPointHints[ (BundleIndex << 1) + 1] << 28);
    OrData |= (UINT32) (DmiGen3RootPortPresets[BundleIndex << 1]);
    OrData |= (UINT32) (DmiGen3RootPortPresets[ (BundleIndex << 1) + 1] << 16);

    ///
    /// Program Eq Settings
    ///
    MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_EQCTL0_1_OFFSET + (BundleIndex << 2)), 0x80F080F0, OrData);
  }
}

/**
  Invert the lane if LaneReversal bit is set based on DMI X4 width

  @param[in]  Lane                - Lane to potentially invert
  @param[in]  LaneReversal        - LaneReversal bit

  @retval     Lane number compensated for LaneReversal
**/
UINT8
DmiReverseLane (
  IN  UINT8 Lane,
  IN  UINT8 LaneReversal
  )
{
  if (LaneReversal != 0) {
    return (GetMaxDmiLanes() - Lane - 1);
  } else {
    return Lane;
  }
}

/**
  Get Full Swing value for EndPoint Transmitter

  @param[in]  DmiBar                      - Current value for DMIBAR
  @param[in]  Lane                        - Physical Lane Number
  @param[out] FullSwing                   - Full Swing value
**/
VOID
EFIAPI
DmiGetLinkPartnerFullSwing (
  IN  UINTN                             DmiBar,
  IN  UINT8                             Lane,
  OUT UINT8                             *FullSwing
  )
{
  UINT32  Data32;

  if (Lane > (GetMaxDmiLanes() - 1)) {
    ASSERT (Lane <= (GetMaxDmiLanes() - 1));
    Lane = 0;
  }

  Data32 = BIT25 | BIT23 | (Lane << 19) | BIT18;
  MmioWrite32 (DmiBar + R_SA_PEG_EQPH3_OFFSET, Data32);
  Data32 = MmioRead32 (DmiBar + R_SA_PEG_EQPH3_OFFSET);
  MmioWrite32 (DmiBar + R_SA_PEG_EQPH3_OFFSET, 0);

  *FullSwing = (Data32 >> 6) & 0x3F;

  return;
}

/**
  Sets the Phase 3 Hijack Equalization Coefficients

  @param[in]  DmiBar                      - Current value for DMIBAR
  @param[in]  Lane                        - Physical Lane Number
  @param[in]  PreCursor                   - Computed Pre-Cursor
  @param[in]  Cursor                      - Computed Cursor
  @param[in]  PostCursor                  - Computed Post-Cursor
**/
VOID
EFIAPI
DmiSetPartnerTxCoefficients (
  IN  UINTN                             DmiBar,
  IN  UINT8                             Lane,
  IN  UINT8                             *PreCursor,
  IN  UINT8                             *Cursor,
  IN  UINT8                             *PostCursor
  )
{
  UINT32  Data32;

  if (Lane > GetMaxDmiLanes()) {
    ASSERT (Lane <= GetMaxDmiLanes());
    return;
  }
  if ((*Cursor) > 63) {
    ASSERT ((*Cursor) <= 63);
    return;
  }
  if ((*PreCursor) > 63) {
    ASSERT ((*PreCursor) <= 63);
    return;
  }
  if ((*PostCursor) > 63) {
    ASSERT ((*PostCursor) <= 63);
    return;
  }

  Data32 = (Lane << 19) | BIT18 | (*Cursor << 12) | (*PreCursor << 6) | (*PostCursor);
  MmioWrite32 (DmiBar + R_SA_PEG_EQPH3_OFFSET, Data32);
  MmioWrite32 (DmiBar + R_SA_PEG_EQPH3_OFFSET, 0);

  return;
}

/**
  Wait until link is up.

  @param[in]  DmiBar                      - Current value for DMIBAR
  @param[in]  MaxDmiSpeed                 - Max DMI Link Speed

  @retval EFI_SUCCESS     - Completed successfully before timeout
  @retval EFI_TIMEOUT     - Timed out
**/
EFI_STATUS
EFIAPI
DmiWaitForL0 (
  IN  UINTN                             DmiBar,
  IN  UINT8                             MaxDmiSpeed
  )
{
  UINT32        Index;
  EFI_STATUS    Status;
  BOOLEAN       CheckEq;
  BOOLEAN       CompletedEq;
  UINT32        EqStatus;
  UINT32        LinkStatus;

  Status         = EFI_TIMEOUT;
  CheckEq        = (MaxDmiSpeed >= 0x3) ? TRUE : FALSE;
  CompletedEq    = FALSE;
  Index          = 0;

  ///
  /// If endpoint's LCAP.MLS (Spec section 7.8.6) indicated Gen3 capability, first wait for equalization to complete.
  /// Check equalization status LSTS2.EC (Spec section 7.8.20) until Gen3 equalization successfully completed.
  ///
  if (CheckEq) {
    for (; Index < 100; Index++) {
      EqStatus = MmioRead16 (DmiBar + R_SA_DMIBAR_LSTS2_OFFSET);
      EqStatus = (EqStatus >> 1) & 0x1;
      if (EqStatus == 0x1) {
        CompletedEq = TRUE;
        break;
      }
      MicroSecondDelay (STALL_ONE_MILLI_SECOND);
    }
  }

  ///
  /// Check for L0 status.
  /// Continue up to 100 msec of combined delay.
  /// Skip if equalization was needed but didn't successfully complete.
  ///
  if ((CheckEq && CompletedEq) || !CheckEq) {
    for (; Index < 100; Index++) {
      LinkStatus = MmioRead32 (DmiBar + R_SA_DMIBAR_DMISTS_OFFSET);
      LinkStatus = (LinkStatus >> 16) & 0xF;
      if (LinkStatus == 0x7) {
        Status = EFI_SUCCESS;
        break;
      }
      MicroSecondDelay (STALL_ONE_MILLI_SECOND);
    }
  }

  return Status;
}

/**
  Performs any additional equalization programming that needs to be done after
  initial link training and endpoint detection

  @param[in]  DmiBar                      - Current value for DMIBAR
  @param[in]  PciePeiPreMemConfig         - PCIe Configuration Block
  @param[in]  MaxDmiSpeed                 - Max DMI Link Speed
**/
VOID
EFIAPI
DmiPostDetectionEqProgramming (
  IN  UINTN                             DmiBar,
  IN  PCIE_PEI_PREMEM_CONFIG            *PciePeiPreMemConfig,
  IN  UINT8                             MaxDmiSpeed
  )
{
  BOOLEAN   Ph3Hijack;
  UINT8     LaneReversal;
  UINT8     FullSwing;
  UINT8     PreCursor;
  UINT8     Cursor;
  UINT8     PostCursor;
  UINT8     Lane;

  LaneReversal = (MmioRead32 (DmiBar + R_SA_DMIBAR_DMITST_OFFSET) >> 20) & 0x1;
  Ph3Hijack    = FALSE;
  FullSwing    = 0;

  ///
  ///
  if (PciePeiPreMemConfig->DmiGen3EqPh3Method == PH3_METHOD_STATIC ||
      PciePeiPreMemConfig->DmiGen3EqPh3Method == PH3_METHOD_SWEQ) {
    Ph3Hijack = TRUE;
    MmioOr32 (DmiBar + R_SA_PEG_EQCFG_OFFSET, BIT1);
  }

  ///
  /// If any Gen3 device, setup equalization values and retrain link
  ///
  if ((MaxDmiSpeed >= DMI_GEN3) && Ph3Hijack) {
    ///
    /// Program presets based upon endpoint fullswing value
    ///
    for (Lane = 0; Lane < SA_DMI_MAX_LANE; Lane++) {
      if (Lane == 0) {
        DmiGetLinkPartnerFullSwing (DmiBar, Lane, &FullSwing);
      }
      GetCoefficientsFromPreset (
        PciePeiPreMemConfig->DmiGen3EndPointPreset[DmiReverseLane (Lane, LaneReversal) ],
        FullSwing,
        &PreCursor,
        &Cursor,
        &PostCursor
        );
      DmiSetPartnerTxCoefficients (DmiBar, Lane, &PreCursor, &Cursor, &PostCursor);
    }

    ///
    /// Clear phase2 bypass
    ///
    MmioAnd32 (DmiBar + R_SA_PEG_EQCFG_OFFSET, (UINT32) ~(BIT15));

    ///
    /// Redo EQ
    ///
    MmioOr32 (DmiBar + R_SA_PEG_LCTL3_OFFSET, BIT0);    ///< DOEQ
  ///
  /// Retrain Link
  ///
    DmiLinkTrain (DmiBar);
    DmiWaitForL0 (DmiBar, MaxDmiSpeed);
  }
}

/**
 This function programs Equalization Phase 2/3 Bypass

  @param[in]  DmiBar                      - Current value for DMIBAR
  @param[in]  PciePeiPreMemConfig         - PCIe Configuration Block
**/
VOID
EFIAPI
DmiEqPh2Ph3BypassProgramming (
  IN  UINTN                             DmiBar,
  IN  PCIE_PEI_PREMEM_CONFIG            *PciePeiPreMemConfig
  )
{
  ///
  /// After last equalization, set PH3 bypass if needed
  ///
  if (PciePeiPreMemConfig->DmiGen3EqPh3Method == PH3_METHOD_SWEQ ||
      PciePeiPreMemConfig->DmiGen3EqPh3Method == PH3_METHOD_DISABLED) {
    MmioOr32 (DmiBar + R_SA_PEG_EQCFG_OFFSET, BIT14);
  }
  ///
  /// Set Ph2 Bypass if enabled by SA policy
  ///
  if (PciePeiPreMemConfig->DmiGen3EqPh2Enable == 0) {
    MmioOr32 (DmiBar + R_SA_PEG_EQCFG_OFFSET, BIT15);
  } else {
    MmioAnd32 (DmiBar + R_SA_PEG_EQCFG_OFFSET, (UINT32) ~(BIT15));
  }
}

/**
  Configure DMI Max Speed

  @param[in]  PciePeiPreMemConfig         - PCIe Configuration Block
  @param[in]  DmiBar                      - Current value for DMIBAR
**/
VOID
DmiConfigureMaxSpeed (
  IN  PCIE_PEI_PREMEM_CONFIG            *PciePeiPreMemConfig,
  IN  UINT64                            DmiBar
  )
{
  UINT16                       LinkStatus;
  UINT8                        DmiLcap;
  UINT8                        MaxDmiSpeed;
  UINT8                        ActualDmiSpeed;
  UINT32                       Data32And;
  UINT32                       Data32Or;
  ///
  /// Check to make sure Ph2 and Ph3 options are compatible
  ///
  if ((PciePeiPreMemConfig->DmiGen3EqPh2Enable == 0)                &&
      ((PciePeiPreMemConfig->DmiGen3EqPh3Method == PH3_METHOD_AUTO) ||
       (PciePeiPreMemConfig->DmiGen3EqPh3Method == PH3_METHOD_HWEQ))) {
    ///
    /// If Ph2 is disabled we can't enable Ph3
    ///
    PciePeiPreMemConfig->DmiGen3EqPh3Method = PH3_METHOD_DISABLED;
  }
  if (PciePeiPreMemConfig->DmiGen3EqPh3Method == PH3_METHOD_SWEQ) {
    DEBUG ((DEBUG_WARN, "SW EQ not supported on DMI, using Static EQ\n"));
    PciePeiPreMemConfig->DmiGen3EqPh3Method = PH3_METHOD_STATIC;
  }

  DmiLcap = PchPcrRead8 (PID_DMI, R_PCH_DMI_PCR_LCAP);
  if (((DmiLcap & (BIT0 | BIT1 | BIT2 | BIT3)) == 0x1) ||
      (PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_MC_CAPID0_A_OFFSET)) & BIT22)) {
    DEBUG ((DEBUG_WARN, "DMI Gen2 is fused off, staying at Gen1!\n"));
    MaxDmiSpeed = DMI_GEN1;
  }
  else if (((DmiLcap & (BIT0 | BIT1 | BIT2 | BIT3)) <= 0x2) ||
           (PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, R_SA_MC_CAPID0_B)) & BIT15)) {
    DEBUG ((DEBUG_WARN, "DMI Gen3 is fused off, staying at Gen2!\n"));
    MaxDmiSpeed = DMI_GEN2;
  }
  else {
    MaxDmiSpeed = DMI_GEN3;
  }

  if ((PciePeiPreMemConfig->DmiMaxLinkSpeed != DMI_AUTO) &&
      (PciePeiPreMemConfig->DmiMaxLinkSpeed < MaxDmiSpeed)) {
    DEBUG ((DEBUG_INFO, "Limiting DMI speed to Gen%d based on SA Policy\n", PciePeiPreMemConfig->DmiMaxLinkSpeed));
    MaxDmiSpeed = (UINT8) PciePeiPreMemConfig->DmiMaxLinkSpeed;
  }

  ///
  /// Read/Write LCAP to make LCAP read only.
  ///
  MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_DMIBAR_LCAP_OFFSET), (UINT32) 0xFFFFFFFF, (UINT32) 0);

  ///
  /// If we are going to Gen3, then perform Gen3 Equalization
  ///
  if (MaxDmiSpeed >= DMI_GEN3) {
    if (PciePeiPreMemConfig->DmiGen3ProgramStaticEq != 0) {
      DEBUG ((DEBUG_INFO, "Program DMI Gen3 Static Equalization...\n"));
      DmiProgramGen3EqPhase1 (
        DmiBar,
        &(PciePeiPreMemConfig->DmiGen3RootPortPreset[0]),
        &(PciePeiPreMemConfig->DmiGen3EndPointPreset[0]),
        &(PciePeiPreMemConfig->DmiGen3EndPointHint[0])
        );
    }
  }

  DEBUG ((DEBUG_INFO, "Programming DMI speed to Gen%d\n", MaxDmiSpeed));

  if (MaxDmiSpeed > DMI_GEN1) {
    ///
    /// Program PCH TLS
    ///
    PchDmiSetTargetLinkSpeed (MaxDmiSpeed);
    ///
    /// Set the requested speed in Target Link Speed in LCTL2[3:0].
    ///
    MmioAndThenOr16 ((UINTN) (DmiBar + R_SA_DMIBAR_LCTL2_OFFSET), (UINT16) ~0xF, MaxDmiSpeed);
  }

  ///
  /// Set Phase2 Bypass for first link train to Gen3
  ///
  if (MaxDmiSpeed >= DMI_GEN3) {
    MmioOr32 ((UINTN) (DmiBar + R_SA_PEG_EQCFG_OFFSET), BIT15);
  }

  ///
  /// Retrain link
  ///
  DmiLinkTrain (DmiBar);

  ///
  /// Retrain link if link speed is less than max speed
  ///
  if ((((MmioRead16 ((UINTN) (DmiBar + R_SA_DMIBAR_LSTS_OFFSET))) & 0x0F) < MaxDmiSpeed) ||
      ((PchPcrRead16 (PID_DMI, R_PCH_DMI15_PCR_LSTS) & 0x0F) < MaxDmiSpeed)) {
    DEBUG ((DEBUG_INFO, "Retraining DMI Link to reach Gen%d\n", MaxDmiSpeed));
    DmiLinkTrain (DmiBar);
  }

  ///
  /// Post initial Gen3 link training steps
  ///
  if (MaxDmiSpeed >= DMI_GEN3) {
    DmiPostDetectionEqProgramming ((UINTN) DmiBar, PciePeiPreMemConfig, MaxDmiSpeed);
    DmiEqPh2Ph3BypassProgramming ((UINTN) DmiBar, PciePeiPreMemConfig);
  }

  ///
  /// Get the current link status
  ///
  LinkStatus = MmioRead16 ((UINTN) (DmiBar + R_SA_DMIBAR_LSTS_OFFSET));
  ActualDmiSpeed = (UINT8) (LinkStatus & 0x0F);
  DEBUG ((DEBUG_INFO, "DMI trained to x%d at Gen%d\n", ((LinkStatus >> 4) & 0x3F), ActualDmiSpeed));

  ///
  /// Set DMI Offset 0xC28 [31:26]
  ///
  Data32And = (UINT32) ~(BIT31 | BIT30 | BIT29 | BIT28 | BIT27 | BIT26);
  if (ActualDmiSpeed >= DMI_GEN3) {
    Data32Or  = 0x10 << 26;
  } else if (ActualDmiSpeed == DMI_GEN2) {
    Data32Or  = 0x8 << 26;
  } else {
    Data32Or  = 0x4 << 26;
  }
  MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_DMIBAR_AFE_PM_TMR_OFFSET), Data32And, Data32Or);
}

/**
  Configure DMI De-Emphasis

  @param[in]  DmiBar                      - Current value for DMIBAR
  @param[in]  DeEmphasisValue             - De-emphasis policy value
**/
VOID
DmiConfigureDeEmphasis (
  IN  UINT64                            DmiBar,
  IN  UINT8                             DeEmphasisValue
  )
{
  UINT32                  Data32And;
  UINT32                  Data32Or;

  ASSERT (DeEmphasisValue <= 0x1);

  ///
  /// Set LCTL2[6] SELECTABLEDEEMPHASIS according to the policy
  ///
  Data32And = (UINT32) ~(BIT6);
  Data32Or  = (UINT32) (DeEmphasisValue) << 6;
  MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_DMIBAR_LCTL2_OFFSET), Data32And, Data32Or);
  DEBUG ((DEBUG_INFO, "DmiDeEmphasis set to x%d\n", DeEmphasisValue));
}

/**
  Initialize DMI.

  @param[in] PCIE_PEI_PREMEM_CONFIG          PCIe config block from SA Policy PPI

  @retval EFI_SUCCESS
**/
VOID
CpuDmiInit (
VOID
  )
{
  UINT64                       McD0BaseAddress;
  UINT64_STRUCT                MchBar;
  UINT64_STRUCT                DmiBar;
  UINT32                       Data32And;
  UINT32                       Data32Or;
  UINT32                       Bundle;

  SI_PREMEM_POLICY_PPI          *SiPreMemPolicyPpi;
  PCIE_PEI_PREMEM_CONFIG        *PciePeiPreMemConfig;
  EFI_STATUS                    Status;

  McD0BaseAddress    = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, 0);
  MchBar.Data32.High = PciSegmentRead32 (McD0BaseAddress + R_SA_MCHBAR + 4);
  MchBar.Data32.Low  = PciSegmentRead32 (McD0BaseAddress + R_SA_MCHBAR);
  MchBar.Data       &= (UINT64) ~BIT0;
  DmiBar.Data32.High = PciSegmentRead32 (McD0BaseAddress + R_SA_DMIBAR + 4);
  DmiBar.Data32.Low  = PciSegmentRead32 (McD0BaseAddress + R_SA_DMIBAR);
  DmiBar.Data       &= (UINT64) ~BIT0;

  Status = PeiServicesLocatePpi(
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **)&SiPreMemPolicyPpi
             );
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuPciePeiPreMemConfigGuid, (VOID *) &PciePeiPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  ///
  ///
  ///
  /// DCBLNC = 0
  ///
  Data32And = (UINT32) ~(BIT3 | BIT2);
  Data32Or  = 0;
  for (Bundle = 0; Bundle < GetMaxDmiBundles(); Bundle++) {
    MmioAndThenOr32 ((UINTN)(DmiBar.Data + R_SA_PEG_G3CTL0_OFFSET + (Bundle * BUNDLE_STEP)), Data32And, Data32Or);
  }

  ///
  /// Perform DMI Recipe steps
  ///
  DEBUG ((DEBUG_INFO, "DMI Recipe...\n"));

  ///
  /// Configure DMI nFTS
  ///
  DmiConfigureNFts (
    DmiBar.Data
    );
  ///
  /// Program DMI de-emphasis value
  ///
  DmiConfigureDeEmphasis (
    DmiBar.Data,
    (UINT8) PciePeiPreMemConfig->DmiDeEmphasis
    );
  ///
  /// Program DMI Link Speed
  ///
  DmiConfigureMaxSpeed (
    PciePeiPreMemConfig,
    DmiBar.Data
    );
  return;
}

/**
  DMI link training

  @param[in] DmiBar - DMIBAR address
**/
VOID
DmiLinkTrain (
  IN  UINT64 DmiBar
  )
{
  ///
  /// Retrain link
  ///
  MmioOr8 ((UINTN) (DmiBar + R_SA_DMIBAR_LCTL_OFFSET), BIT5);
}

/**
  DMI Max Payload Size Init
**/
VOID
MaxPayloadSizeInit (
  VOID
  )
{
  UINT64_STRUCT DmiBar;
  UINT32        Data32;
  UINT32        Data32And;
  UINT32        Data32Or;
  UINT32        DmiMaxPayloadSizeCap;
  UINT64        McBaseAddress;


  //
  // Read R_SA_DMIBAR_DCAP_OFFSET register
  //
  McBaseAddress      = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, 0);
  DmiBar.Data32.High = PciSegmentRead32 (McBaseAddress + R_SA_DMIBAR + 4);
  DmiBar.Data32.Low  = PciSegmentRead32 (McBaseAddress + R_SA_DMIBAR);
  DmiBar.Data       &= (UINT64) ~BIT0;
  ASSERT (DmiBar.Data != 0);

  Data32 = MmioRead32 ((UINTN)(DmiBar.Data + R_SA_DMIBAR_DCAP_OFFSET));
  DmiMaxPayloadSizeCap = (UINT32) (Data32 & B_SA_DMIBAR_DCAP_OFFSET_LOCK);  // Apply this to extract Bit [2:0] only

  //
  //
  //
  // Write back after reading contents to ensure bit 0 of DMI DCAP register (RW_O) does not get over ridden
  //
  MmioWrite32 ((UINTN)(DmiBar.Data + R_SA_DMIBAR_DCAP_OFFSET), Data32);

  //
  // Checking if PCIE_TL_CR_DCAP_0_0_0_DMIBAR[2:0] is set to 1.
  // If yes, then DMI can be configured for MPS of 256B or 128B
  //
  if (DmiMaxPayloadSizeCap == 0x1) {
    ///
    /// Verified that DMI MPS is 256/128B configurable.
    /// For SKL setting,
    /// PCIE_TL_CR_DCTL_0_0_0_DMIBAR[7:5] = 1 ; 256B of MPS
    /// PCIE_TL_CR_DCTL_0_0_0_DMIBAR[7:5] = 0 ; 128B of MPS
    ///
    Data32And = (UINT32) B_SA_DMIBAR_DCTL_OFFSET_MASK;
    Data32Or  = (UINT32) (DmiMaxPayloadSizeCap  << 5);
    MmioAndThenOr32 ((UINTN)(DmiBar.Data + R_SA_DMIBAR_DCTL_OFFSET), Data32And, Data32Or);
  }
  return;
}

/**
  Configure DMI nFTS

  @param[in] DmiBar                 - DMIBAR address
**/
VOID DmiConfigureNFts (
  IN UINT64                    DmiBar
  )
{
  UINT32                       Data32And;
  UINT32                       Data32Or;
  ///
  /// Set L0SLAT[15:0] to 0x2020
  ///
  Data32And = (UINT32) ~(0xFFFF);
  Data32Or  = 0x2020;
  MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_L0SLAT_OFFSET), Data32And, Data32Or);
  DEBUG ((DEBUG_INFO, "DMI nFTS 0x%x\n", Data32Or));
}


/**
  Programs the PCIe/DMI recipe.

  @param[in]  DmiBar                      - DMIBAR
  @param[in]  MchBar                      - MCHBAR
  @param[in]  Gen3CtlePeaking             - Array of CTLE Peaking values to program per bundle
  @param[in]  Gen3CtlePeakingLength       - Length of the Gen3CtlePeaking array
  @param[in]  Gen3RxCtleOverride          - RxCTLE override configuration
  @param[in]  ProgramDmiRecipe            - Set to TRUE to program DMI, FALSE to program PCIe
**/
VOID
EFIAPI
CnlPegDmiRecipe (
  IN  UINT64                            DmiBar,
  IN  UINT64                            MchBar,
  IN  UINT8                             *Gen3CtlePeaking,
  IN  UINT8                             Gen3CtlePeakingLength,
  IN  UINT8                             Gen3RxCtleOverride,
  IN  BOOLEAN                           ProgramDmiRecipe
  )
{
  UINT64            BaseAddress;
  UINT64            Peg0BaseAddress;
  UINT64            Peg3BaseAddress;
  UINTN             Index;
  UINTN             Count;
  UINTN             BundlesCount;
  UINTN             LanesCount;
  UINT32            Mask32And;
  UINT32            Data32Or;
  CPU_SKU           CpuSku;
  UINT32            CrOffset;
  UINT8             PegMaxFuncNum;
  UINT8             SaPeg3MaxBundle;
  UINT8             SaPeg3CntMaxLane;
  CPU_GENERATION    CpuGeneration;

  CpuSku            = GetCpuSku ();
  CpuGeneration     = GetCpuGeneration();
  BaseAddress     = 0;
  Peg0BaseAddress = 0;
  Peg3BaseAddress = 0;

  if (CpuGeneration == EnumCmlCpu) {
    PegMaxFuncNum    = SA_PEG_MAX_LANE_GEN3;
    SaPeg3MaxBundle     = SA_PEG3_CNT_MAX_BUNDLE_GEN3;
    SaPeg3CntMaxLane = SA_PEG3_CNT_MAX_LANE_GEN3;
  } else {
    PegMaxFuncNum    = SA_PEG_MAX_LANE;
    SaPeg3MaxBundle     = SA_PEG3_CNT_MAX_BUNDLE;
    SaPeg3CntMaxLane = SA_PEG3_CNT_MAX_LANE;
  }

  if (ProgramDmiRecipe) {
    ///
    /// DMI
    ///
    LanesCount  = GetMaxDmiLanes();

    //
    // BaseAddress will not be used in DMI recipe.
    //
    BaseAddress = (UINT64) 0xFFFFFFFF;
  } else {
    ///
    /// PEG
    ///
    LanesCount  = PegMaxFuncNum;
    Peg0BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG_DEV_NUM, SA_PEG0_FUN_NUM, 0);
    Peg3BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG3_DEV_NUM, SA_PEG3_FUN_NUM, 0);
  }

  BundlesCount = LanesCount >> 1;

  ///
  ///
  if (!ProgramDmiRecipe) {
    ///
    /// AFEBND0CFG1[21:16] PGACQ = 0xA (PEG)
    ///
    Mask32And = (UINT32) ~(BIT21 | BIT20 | BIT19 | BIT18 | BIT17 | BIT16 );
    Data32Or  = 0xA << 16;
    for (Index = 0; Index < SA_PEG0_CNT_MAX_BUNDLE; Index++) {  // PEG0
      PciSegmentAndThenOr32 (Peg0BaseAddress + R_SA_PEG_AFEBND0CFG1_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
    for (Index = 0; Index < SaPeg3MaxBundle; Index++) {  // PEG3
      PciSegmentAndThenOr32 (Peg3BaseAddress + R_SA_PEG_AFEBND0CFG1_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
  }
  ///
  ///
  if (ProgramDmiRecipe) {
    ///
    /// AFEBND0CFG2[23:23] G3BYPCOEFF = 0x0 (DMI)
    ///
    Mask32And = (UINT32) ~(BIT23);
    Data32Or  = 0x0 << 23;
  ///
    /// AFEBND0CFG2[21:15] TXEQPOSTCUR = 0xC (DMI)
    ///
    Mask32And &= (UINT32) ~(BIT21 | BIT20 | BIT19 | BIT18 | BIT17 | BIT16 | BIT15);
    Data32Or  |= 0xC << 15;
  ///
    /// AFEBND0CFG2[14:8] TXEQPRECUR = 0x8 (DMI)
    ///
    Mask32And &= (UINT32) ~(BIT14 | BIT13 | BIT12 | BIT11 | BIT10 | BIT9 | BIT8);
    Data32Or  |= 0x8 << 8;
    ///
    /// AFEBND0CFG2[7:1] TXEQCUR = 0x2A (DMI)
    ///
    Mask32And &= (UINT32) ~(BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1);
    Data32Or  |= 0x2A << 1;


    for (Index = 0; Index < BundlesCount; Index++) {
      MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_AFEBND0CFG2_OFFSET + (Index * BUNDLE_STEP)), Mask32And, Data32Or);
    }
  } else {
    ///
    /// AFEBND0CFG2[7:1] TXEQCUR = 0x2A (PEG)
    ///
    Mask32And = (UINT32) ~(BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1);
    Data32Or  = 0x2A << 1;
    ///
    /// AFEBND0CFG2[21:15] TXEQPOSTCUR = 0xC (PEG)
    ///
    Mask32And &= (UINT32) ~(BIT21 | BIT20 | BIT19 | BIT18 | BIT17 | BIT16 | BIT15);
    Data32Or  |= 0xC << 15;
    ///
    /// AFEBND0CFG2[14:8] TXEQPRECUR = 0x8 (PEG)
    ///
    Mask32And &= (UINT32) ~(BIT14 | BIT13 | BIT12 | BIT11 | BIT10 | BIT9 | BIT8);
    Data32Or  |= 0x8 << 8;
    for (Index = 0; Index < SA_PEG0_CNT_MAX_BUNDLE; Index++) {  // PEG0
      PciSegmentAndThenOr32 (Peg0BaseAddress + R_SA_PEG_AFEBND0CFG2_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
    for (Index = 0; Index < SaPeg3MaxBundle; Index++) {  // PEG3
      PciSegmentAndThenOr32 (Peg3BaseAddress + R_SA_PEG_AFEBND0CFG2_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
  }

  ///
  ///
  /// AFEBND0CFG3[24:21] RXRTBIN = 0x3 (PEG & DMI)
  ///
  Mask32And = (UINT32) ~(BIT24 | BIT23 | BIT22 | BIT21);
  Data32Or  = 0x3 << 21;
  ///
  /// AFEBND0CFG3[29:26] TXRTBIN = 0x4 (PEG & DMI)
  ///
  Mask32And &= (UINT32) ~(BIT29 | BIT28 | BIT27 | BIT26);
  Data32Or  |= 0x4 << 26;
  if (ProgramDmiRecipe) {
    for (Index = 0; Index < BundlesCount; Index++) {
      MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_AFEBND0CFG3_OFFSET + (Index * BUNDLE_STEP)), Mask32And, Data32Or);
    }
  } else {
    ///
    /// AFEBND0CFG3[10:5] PGTRK = 0xA (PEG)
    ///
    Mask32And &= (UINT32) ~(BIT10 | BIT9 | BIT8 | BIT7 | BIT6 | BIT5);
    Data32Or  |= (0xA << 5);
    for (Index = 0; Index < SA_PEG0_CNT_MAX_BUNDLE; Index++) {  // PEG0
      PciSegmentAndThenOr32 (Peg0BaseAddress + R_SA_PEG_AFEBND0CFG3_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
    for (Index = 0; Index < SaPeg3CntMaxLane; Index++) {  // PEG3
      PciSegmentAndThenOr32 (Peg3BaseAddress + R_SA_PEG_AFEBND0CFG3_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
  }

  ///
  ///
  /// AFEBND0CFG4[13:10] G2RXCTLEPEAK = 0x2 (PEG & DMI)
  ///
  Mask32And = (UINT32) ~(BIT13 | BIT12 | BIT11 | BIT10);
  Data32Or  = (0x2 << 10);
  if (ProgramDmiRecipe) {
    ///
    /// AFEBND0CFG4[9:6] G3RXCTLEPEAK = 0x0(DMI)
    ///
    Mask32And &= (UINT32) ~(BIT9 | BIT8 | BIT7 | BIT6);
    Data32Or  |= (0x0 << 6);
    ///
    /// AFEBND0CFG4[31:25] AFEBNDSPARE = 0x30(DMI)
    ///
    Mask32And &= (UINT32) ~(BIT31 | BIT30 | BIT29 | BIT28 | BIT27 | BIT26 | BIT25);
    Data32Or  |= (0x30 << 25);
    for (Index = 0; Index < BundlesCount; Index++) {
      MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_AFEBND0CFG4_OFFSET + (Index * BUNDLE_STEP)), Mask32And, Data32Or);
    }
  } else {
    ///
    /// AFEBND0CFG4[31:25] AFEBNDSPARE = 48 or 0x30  (PEG)
    ///
    Mask32And &= (UINT32) ~(BIT31 | BIT30 | BIT29 | BIT28 | BIT27 | BIT26 |BIT25);
    Data32Or  |= (0x30 << 25);
    ///
    /// AFEBND0CFG4[9:6] G3RXCTLEPEAK = 0x0 [48dec] (PEG)
    ///
    Mask32And &= (UINT32) ~(BIT9 | BIT8 | BIT7 | BIT6);
    Data32Or  |= (0x0 << 6);
    for (Index = 0; Index < SA_PEG0_CNT_MAX_BUNDLE; Index++) {  // PEG0
      PciSegmentAndThenOr32 (Peg0BaseAddress + R_SA_PEG_AFEBND0CFG4_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
    for (Index = 0; Index < SaPeg3MaxBundle; Index++) {  // PEG3
      PciSegmentAndThenOr32 (Peg3BaseAddress + R_SA_PEG_AFEBND0CFG4_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
  }

  ///
  ///
  /// AFELN0CFG0[4:0] G23RXVREF = 0xA (Peg & DMI)
  ///
  Mask32And = (UINT32) ~(BIT4 | BIT3 | BIT2 | BIT1 | BIT0);
  Data32Or  = 0xA;
  if (ProgramDmiRecipe) {
    for (Index = 0; Index < LanesCount; Index++) {
      MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_AFELN0CFG0_OFFSET + (Index * LANE_STEP)), Mask32And, Data32Or);
    }
  } else {
    for (Index = 0; Index < SA_PEG0_CNT_MAX_LANE; Index++) {  // PEG0
      PciSegmentAndThenOr32 (Peg0BaseAddress + R_SA_PEG_AFELN0CFG0_OFFSET + (Index * LANE_STEP), Mask32And, Data32Or);
    }
    for (Index = 0; Index < SA_PEG3_CNT_MAX_LANE; Index++) {  // PEG3
      PciSegmentAndThenOr32 (Peg3BaseAddress + R_SA_PEG_AFELN0CFG0_OFFSET + (Index * LANE_STEP), Mask32And, Data32Or);
    }
  }

  ///
  ///
  if (ProgramDmiRecipe) {
    ///
    /// AFELN0CFG1[8:4] G1RXVREFSEL = 0xC (DMI)
    ///
    Mask32And = (UINT32) ~(BIT8 | BIT7 | BIT6 | BIT5 | BIT4);
    Data32Or  = 0xC << 4;
    for (Index = 0; Index < LanesCount; Index++) {
      MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_AFELN0CFG1_OFFSET + (Index * LANE_STEP)), Mask32And, Data32Or);
    }
  }

  ///
  ///
  /// CRI0_CR_DWORD22[22]    RESSEL_OVRD_EN = 0x1 (DMI, PEG)
  ///
  Mask32And = (UINT32) ~(BIT22);
  Data32Or  = 0x1 << 22;
  ///
  /// CRI0_CR_DWORD22[21:16] PI_RESSEL_OVRD = 0x10 (DMI, PEG)
  ///
  Mask32And &= (UINT32) ~(BIT21 | BIT20 | BIT19 | BIT18 | BIT17 | BIT16);
  Data32Or  |= 0x10 << 16;
  if (ProgramDmiRecipe) {
    for (Index = 0; Index < BundlesCount; Index++) {
      CrOffset  = ((R_SA_PEG_BND10_CRI0_CR_DWORD22_OFFSET << 8)|((Index*2) << 16));
      PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, DMI_PORT);
    }
  } else {
    for (Index = 0; Index < BundlesCount; Index++) {
      CrOffset  = ((R_SA_PEG_BND0_CRI0_CR_DWORD22_OFFSET << 8)|((Index*2) << 16));
      PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, PEG_PORT);
    }
  }

  ///
  ///
  if (ProgramDmiRecipe) {
    ///
    /// CRI2_FUSE_DWORD15 [12:8] RXD_SUM_GAIN_GEN1_L1 = 0x4 (DMI)
    ///
    Mask32And  = (UINT32) ~(BIT12 | BIT11 | BIT10 | BIT9 | BIT8);
    Data32Or   = 0x4 << 8;
    ///
    /// CRI2_FUSE_DWORD15[20:16] RXD_SUM_GAIN_GEN2_L1 = 0x3 (DMI)
    ///
    Mask32And &= (UINT32) ~(BIT20 | BIT19 | BIT18 | BIT17 | BIT16);
    Data32Or  |= 0x3 << 16;
  ///
    /// CRI2_FUSE_DWORD15[28:24] RXD_SUM_GAIN_GEN3_L0 = 0x3 (DMI)
    ///
    Mask32And &= (UINT32) ~(BIT28 | BIT27 | BIT26 | BIT25 | BIT24);
    Data32Or  |= 0x3 << 16;
  ///
  /// CRI2_FUSE_DWORD15[4:0] RXD_SUM_GAIN_GEN3_L1 = 0x3 (DMI)
    ///
    Mask32And &= (UINT32) ~(BIT4 | BIT3 | BIT2 | BIT1 | BIT0);
    Data32Or  |= 0x3 << 16;

    for (Index = 0; Index < BundlesCount; Index++) {
      CrOffset  = ((R_SA_PEG_BND10_CRI2_FUSE_DWORD15_OFFSET << 8)|((Index*2) << 16));
      PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, DMI_PORT);
    }
  } else {
    ///
    /// CRI2_FUSE_DWORD15 [28:24] RXD_SUM_GAIN_GEN3_L1 = 0x3 (PEG)
    ///
    Mask32And  = (UINT32) ~(BIT28 | BIT27 | BIT26 | BIT25 | BIT24);
    Data32Or   = 0x3 << 24;
    ///
    /// CRI2_FUSE_DWORD15[4:0] RXD_SUM_GAIN_GEN3_L0 = 0x3 (PEG)
    ///
    Mask32And &= (UINT32) ~(BIT4 | BIT3 | BIT2 | BIT1 | BIT0);
    Data32Or  |= 0x3 << 0;
    for (Index = 0; Index < BundlesCount; Index++) {
      CrOffset  = ((R_SA_PEG_BND0_CRI2_FUSE_DWORD15_OFFSET << 8)|((Index*2) << 16));
      PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, PEG_PORT);
    }
  }

  ///
  ///
  if (ProgramDmiRecipe) {
    ///
    /// CRI2_FUSE_DWORD14 [28:24] RXD_SUM_GAIN_GEN2_L0 = 0x3 (DMI)
    ///
    Mask32And  = (UINT32) ~(BIT28 | BIT27 | BIT26 | BIT25 | BIT24);
    Data32Or   = 0x3 << 24;
    ///
    /// CRI2_FUSE_DWORD14 [20:16] RXD_SUM_GAIN_GEN1_L0 = 0x4 (DMI)
    ///
    Mask32And &= (UINT32) ~(BIT20 | BIT19 | BIT18 | BIT17 | BIT16);
    Data32Or  |= 0x4 << 16;
    for (Index = 0; Index < BundlesCount; Index++) {
      CrOffset  = ((R_SA_PEG_BND10_CRI2_FUSE_DWORD14_OFFSET << 8)|((Index*2) << 16));
      PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, DMI_PORT);
    }
  }

  ///
  ///
  if (ProgramDmiRecipe) {
    ///
    /// CRI2_FUSE_DWORD16 [7:0] SWING_CONTROL = 136 or 0x88 (DMI)
    ///
    Mask32And  = (UINT32) ~(BIT7 | BIT6 | BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0);
    Data32Or   = 0x88 << 0;
    for (Index = 0; Index < BundlesCount; Index++) {
      CrOffset  = ((R_SA_PEG_BND10_CRI2_FUSE_DWORD16_OFFSET << 8)|((Index*2) << 16));
      PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, DMI_PORT);
    }
  }
  ///
  ///
  /// BND0SPARE[30:30] GEN3SUMBIASSEL = 0x0 (DMI & PEG)
  ///
  Mask32And = (UINT32) ~(BIT30);
  Data32Or  = 0x0 << 30;
  ///
  /// BND0SPARE[29:27] GEN3DFEIDACPD = 0x3 (PEG & DMI)
  ///
  Mask32And &= (UINT32) ~(BIT29 | BIT28 | BIT27);
  Data32Or  |= 0x3 << 27;
  ///
  /// BND0SPARE[18:17] GEN3DFELSBSEL = 0x0 (PEG & DMI)
  ///
  Mask32And &= (UINT32) ~(BIT18 | BIT17);
  Data32Or  |= 0x0 << 17;
  ///
  /// BND0SPARE[14:13] GEN3DFESUM_MFC_10GEN = 0x0 (PEG & DMI)
  ///
  Mask32And &= (UINT32) ~(BIT14 | BIT13);
  Data32Or  |= 0x0 << 13;
  ///
  /// BND0SPARE[10:9] GEN3DFESUMADDMFC = 0x0 (PEG & DMI)
  ///
  Mask32And &= (UINT32) ~(BIT10 | BIT9);
  Data32Or  |= 0x0 << 9;
  ///
  /// BND0SPARE[7:6] GEN2DFESUMADDMFC = 0x0 (DMI & DMI)
  ///
  Mask32And &= (UINT32) ~(BIT7 | BIT6);
  Data32Or  |= 0x0 << 6;
  if (ProgramDmiRecipe) {
    ///
    /// BND0SPARE[25:23] GEN2DFEIDACPD = 0x2 (DMI)
    ///
    Mask32And &= (UINT32) ~(BIT25 | BIT24 | BIT23);
    Data32Or  |= 0x2 << 23;
    for (Index = 0; Index < BundlesCount; Index++) {
      MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_BND0SPARE_OFFSET + (Index * BUNDLE_STEP)), Mask32And, Data32Or);
    }
  } else {
    ///
    /// BND0SPARE[25:23] GEN2DFEIDACPD = 0x2 (PEG)
    ///
    Mask32And &= (UINT32) ~(BIT25 | BIT24 | BIT23);
    Data32Or  |= 0x2 << 23;
    for (Index = 0; Index < SA_PEG0_CNT_MAX_BUNDLE; Index++) {  // PEG0
      PciSegmentAndThenOr32 (Peg0BaseAddress + R_SA_PEG_BND0SPARE_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
    for (Index = 0; Index < SaPeg3MaxBundle; Index++) {  // PEG3
      PciSegmentAndThenOr32 (Peg3BaseAddress + R_SA_PEG_BND0SPARE_OFFSET + (Index * BUNDLE_STEP), Mask32And, Data32Or);
    }
  }

  ///
  ///
  //
  // @todo: Is it required , not finding in KBL/CNL
  //
  if (ProgramDmiRecipe) {
    ///
    /// Set CFG5[14:11] UPCFGSLVTMR = 4
    ///
    Mask32And = (UINT32) ~(BIT14 | BIT13 | BIT12 | BIT11);
    Data32Or  = (UINT32)  (4 << 11);
    MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_DMIBAR_CFG5_OFFSET), Mask32And, Data32Or);
  }

  ///
  ///
  /// EQCFG[0] EXTEIEOS = 0x1 (Peg & DMI)
  ///
  Mask32And = (UINT32) ~(BIT0);
  Data32Or  = 0x1 << 0;
  if (ProgramDmiRecipe) {
    MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_EQCFG_OFFSET), Mask32And, Data32Or);
  } else {
    ///
    /// EQCFG[25:20] LF = 0x14
    ///
    //
    // @todo: Is it required for CNL , not finding in KBL/CNL.
    //
    Mask32And &= (UINT32) ~(BIT25 | BIT24 | BIT23 | BIT22 | BIT21 | BIT20);
    Data32Or  |= 0x14 << 20;
    ///
    /// EQCFG[31:26] FS = 0x3E
    ///
    //
    // @todo: Is it required , not finding in KBL/CNL
    //
    Mask32And &= (UINT32) ~(BIT31 | BIT30 | BIT29 | BIT28 | BIT27 | BIT26);
    Data32Or  |= 0x3E << 26;
    PciSegmentAndThenOr32 (Peg0BaseAddress + R_SA_PEG_EQCFG_OFFSET, Mask32And, Data32Or);
  }
  if (!ProgramDmiRecipe) {
    PciSegmentAndThenOr32 (
      PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG_DEV_NUM, SA_PEG1_FUN_NUM, R_SA_PEG_EQCFG_OFFSET),
      Mask32And,
      Data32Or
      );
    PciSegmentAndThenOr32 (
      PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG_DEV_NUM, SA_PEG2_FUN_NUM, R_SA_PEG_EQCFG_OFFSET),
      Mask32And,
      Data32Or
      );
    PciSegmentAndThenOr32 (
      PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_PEG_BUS_NUM, SA_PEG3_DEV_NUM, SA_PEG3_FUN_NUM, R_SA_PEG_EQCFG_OFFSET),
      Mask32And,
      Data32Or
      );
  }
  ///
  /// EQCFG[1] BYPADFSM = 0x1 (DMI)
  ///
  if (ProgramDmiRecipe) {
    Mask32And = (UINT32) ~(BIT1);
    Data32Or  = 0x1 << 1;
    MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_EQCFG_OFFSET), Mask32And, Data32Or);
  }
  ///
  /// EQPH3[17:0] PH3_COEFF = 0x160C7 (DMI) This requires to set and clear PH3_COEFF_CTRL for each lane
  ///
  // @todo: Is it required for CNL ? not part of recipe , currently keeping same as KBL
  //
  if (ProgramDmiRecipe) {
    Data32Or  = 0x160C7;
    for (Index = 0; Index < LanesCount; Index++) {
      MmioWrite32 ((UINTN) (DmiBar + R_SA_PEG_EQPH3_OFFSET), Data32Or | (Index << 19) | BIT18);
      MmioAnd32 ((UINTN) (DmiBar + R_SA_PEG_EQPH3_OFFSET), (UINT32) ~(BIT21 | BIT20 | BIT19 | BIT18)); // bits[21:19] for lane 0-7
    }
  }

  ///
  ///
  /// Preset Definitions
  ///
  //
  // @todo: check if required for CNL also , get Peg0/Peg3 presets. Currently using same as KBL for both PEG0, PEG3
  //
  if (!ProgramDmiRecipe) {
    for (Index = 0; Index < 2; Index++) { // Loop over PEG0 & PEG3 Controller
      if (Index == 0) { // PEG0
        BaseAddress = Peg0BaseAddress;
      } else if (Index == 1) { // PEG3
        BaseAddress = Peg3BaseAddress;
      }
      PciSegmentWrite32 (BaseAddress + R_SA_PEG_EQPRESET1_2_OFFSET,   0x32012B00);
      PciSegmentWrite32 (BaseAddress + R_SA_PEG_EQPRESET2_3_4_OFFSET, 0x003B000C);
      PciSegmentWrite32 (BaseAddress + R_SA_PEG_EQPRESET4_5_OFFSET,   0x00F802B4);
      PciSegmentWrite32 (BaseAddress + R_SA_PEG_EQPRESET6_7_OFFSET,   0x34280D88);
      PciSegmentWrite32 (BaseAddress + R_SA_PEG_EQPRESET7_8_9_OFFSET, 0x09369200);
      PciSegmentWrite32 (BaseAddress + R_SA_PEG_EQPRESET9_10_OFFSET,  0x00C8C26C);
      PciSegmentWrite32 (BaseAddress + R_SA_PEG_EQPRESET11_OFFSET,    0x00015A40);
    }
  }

  if ((CpuSku  == EnumCpuHalo) || (CpuSku  == EnumCpuTrad)) {
    ///
    ///
    if (ProgramDmiRecipe) {
      ///
      /// CRI0_CR_DWORD24[31:30] FFEWIN_CTRL = 0x0 (DMI)
      ///
      Mask32And = (UINT32) ~(BIT31 | BIT30);
      Data32Or = 0x0 << 30;
      ///
      /// CRI0_CR_DWORD24[29:28] CYCLES_CTRL = 0x0 (DMI)
      ///
      Mask32And &= (UINT32) ~(BIT29 | BIT28);
      Data32Or |= 0x0 << 28;
      ///
      /// CRI0_CR_DWORD24[27:25] PH3CDR_CTL = 0x1 (DMI)
      ///
      Mask32And &= (UINT32) ~(BIT27 | BIT26 | BIT25);
      Data32Or |= 0x1 << 25;
      ///
      /// CRI0_CR_DWORD24[24:21] PTRNSEL = 0x0 (DMI)
      ///
      Mask32And &= (UINT32) ~(BIT24 | BIT23 | BIT22 | BIT21);
      Data32Or |= 0x0 << 21;
      ///
      /// CRI0_CR_DWORD24[20:18] INITPS = 0x1 (DMI)
      ///
      Mask32And &= (UINT32) ~(BIT20 | BIT19 | BIT18);
      Data32Or |= 0x1 << 18;
      ///
      /// CRI0_CR_DWORD24[17:14] INITCTLEP = 0x2 (DMI)
      ///
      Mask32And &= (UINT32) ~(BIT17 | BIT16 | BIT15 | BIT14);
      Data32Or |= 0x2 << 14;
      ///
      /// CRI0_CR_DWORD24[13:12] DFX_CTRL = 0x0 (DMI)
      ///
      Mask32And &= (UINT32) ~(BIT13 | BIT12);
      Data32Or |= 0x0 << 12;
    } else {
      ///
      /// CRI0_CR_DWORD24[31:30] FFEWIN_CTRL = 0x0 (PEG)
      ///
      Mask32And = (UINT32) ~(BIT31 | BIT30);
      Data32Or  = 0x0 << 30;
      ///
      /// CRI0_CR_DWORD24[29:28] CYCLES_CTRL = 0x0 (PEG)
      ///
      Mask32And &= (UINT32) ~(BIT29 | BIT28);
      Data32Or |= 0x0 << 28;
      ///
      /// CRI0_CR_DWORD24[27:25] PH3CDR_CTL = 0x1 (PEG)
      ///
      ///
      Mask32And &= (UINT32) ~(BIT27 | BIT26 | BIT25);
      Data32Or |= 0x1 << 25;
      ///
      /// CRI0_CR_DWORD24[24:21] PTRNSEL = 0x0 (PEG)
      ///
      Mask32And &= (UINT32) ~(BIT24 | BIT23 | BIT22 | BIT21);
      Data32Or |= 0x0 << 21;
      ///
      /// CRI0_CR_DWORD24[20:18] INITPS = 0x1 (PEG)
      ///
      Mask32And &= (UINT32) ~(BIT20 | BIT19 | BIT18);
      Data32Or |= 0x1 << 18;
      ///
      /// CRI0_CR_DWORD24[17:14] INITCTLEP = 0x2 (PEG)
      ///
      Mask32And &= (UINT32) ~(BIT17 | BIT16 | BIT15 | BIT14);
      Data32Or |= 0x2 << 14;
      ///
      /// CRI0_CR_DWORD24[13:12] DFX_CTRL = 0x0 (PEG)
      ///
      Mask32And &= (UINT32) ~(BIT13 | BIT12);
      Data32Or |= 0x0 << 12;
    }
    for (Index = 0; Index < BundlesCount; Index++) {
      if (!ProgramDmiRecipe) {
        CrOffset  = ((R_SA_PEG_BND0_CRI0_CR_DWORD24_OFFSET << 8)|((Index*2) << 16));
        PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, PEG_PORT);
      } else {
        CrOffset  = ((R_SA_PEG_BND10_CRI0_CR_DWORD24_OFFSET << 8)|((Index*2) << 16));
        PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, DMI_PORT);
      }
    }
    ///
    ///
    Mask32And = 0x0;
    if (ProgramDmiRecipe) {
      ///
      /// CRI0_CR_DWORD26[31:16] PH3SUP_CTRL = 384 or 0x180 (DMI)
      ///
      Data32Or = 0x180 << 16;
      ///
      /// CRI0_CR_DWORD26[15:0] CTLEADJUST_CTRL = 21878 or 0x5576 (DMI)
      ///
      Data32Or |= 0x5576 << 0;
    } else {
      ///
      /// CRI0_CR_DWORD26[31:16] PH3SUP_CTRL = 384 or 0x180 (PEG)
      ///
      Data32Or = 0x180 << 16;
      ///
      /// CRI0_CR_DWORD26[15:0] CTLEADJUST_CTRL = 21878 or 0x5576 (PEG)
      ///
      Data32Or |= 0x5576 << 0;
    }
    for (Index = 0; Index < BundlesCount; Index++) {
      if (!ProgramDmiRecipe) {
        CrOffset  = ((R_SA_PEG_BND0_CRI0_CR_DWORD26_OFFSET << 8)|((Index*2) << 16));
        PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, PEG_PORT);
      } else {
        CrOffset  = ((R_SA_PEG_BND10_CRI0_CR_DWORD26_OFFSET << 8)|((Index*2) << 16));
        PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, DMI_PORT);
      }
    }

    ///
    ///
    if (ProgramDmiRecipe) {
      ///
      /// CRI0_CR_DWORD28[2:0] AVG_CTRL = 0x2 (DMI)
      ///
      Mask32And = (UINT32) ~(BIT2 | BIT1 | BIT0);
      Data32Or = 0x2 << 0;
      ///
      /// CRI0_CR_DWORD28[3:3] CR_EN_PH3_NEW_CTLE_PK = 0x1 (DMI)
      ///
      Mask32And &= (UINT32) ~(BIT3);
      Data32Or |= 0x1 << 3;
      ///
      /// Only applicable to DT/Halo
      ///
      if ((CpuSku  == EnumCpuHalo) || (CpuSku  == EnumCpuTrad)) {
        ///
        /// CRI0_CR_DWORD28[4:4] CR_ENABLE_ROUND_FIX_PH3 = 0x1 (DMI)
        ///
        Mask32And &= (UINT32) ~(BIT4);
        Data32Or |= 0x1 << 4;
      }
    } else {
      ///
      /// CRI0_CR_DWORD28[2:0] AVG_CTRL = 0x2 (PEG)
      ///
      Mask32And = (UINT32) ~(BIT2 | BIT1 | BIT0);
      Data32Or = 0x2 << 0;
      ///
      /// CRI0_CR_DWORD28[3:3] CR_EN_PH3_NEW_CTLE_PK = 0x1 only if RxCTLE Override is disabled (PEG)
      ///
      if (Gen3RxCtleOverride == 0) {
        Mask32And &= (UINT32) ~(BIT3);
        Data32Or |= 0x1 << 3;
      }
      ///
      /// Only applicable to DT/Halo
      ///
      if ((CpuSku  == EnumCpuHalo) || (CpuSku  == EnumCpuTrad)) {
        ///
        /// CRI0_CR_DWORD28[4:4] CR_ENABLE_ROUND_FIX_PH3 = 0x1 (PEG)
        ///
        Mask32And &= (UINT32) ~(BIT4);
        Data32Or |= 0x1 << 4;
      }
    }
    for (Index = 0; Index < BundlesCount; Index++) {
      if (!ProgramDmiRecipe) {
        CrOffset  = ((R_SA_PEG_BND0_CRI0_CR_DWORD28_OFFSET << 8)|((Index*2) << 16));
        PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, PEG_PORT);
      } else {
        CrOffset  = ((R_SA_PEG_BND10_CRI0_CR_DWORD28_OFFSET << 8)|((Index*2) << 16));
        PcodeMailboxReadThenWrite(CrOffset, Mask32And, Data32Or, DMI_PORT);
      }
    }
  }

  ///
  /// AFEBND0CFG4[9:6] G3RXCTLEPEAK = Read from configuration. Default = 0x0 PEG, 0x0 DMI. (PEG & DMI)
  ///
  Mask32And = (UINT32) ~(BIT9 | BIT8 | BIT7 | BIT6);
  for (Index = 0; Index < BundlesCount; Index++) {
    if ((Index < Gen3CtlePeakingLength) && (Gen3CtlePeaking[Index] < 16)) {
      Data32Or = Gen3CtlePeaking[Index] << 6;
    } else {
      if (ProgramDmiRecipe) {
        Data32Or = 0 << 6;
      } else {
        Data32Or = 0 << 6;
      }
      DEBUG ((DEBUG_ERROR, "CTLE Peaking array incorrect length or invalid value!\n"));
      return;
    }
    if (ProgramDmiRecipe) {
      MmioAndThenOr32 ((UINTN) (DmiBar + R_SA_PEG_AFEBND0CFG4_OFFSET + (Index * BUNDLE_STEP)), Mask32And, Data32Or);
    } else {
      if (Index < SaPeg3MaxBundle) { // Bundle 0-1 belongs to PEG3
        BaseAddress = Peg3BaseAddress;
        Count = Index;
      } else { // Bundle 2-9 belongs to PEG0, but for BIOS Bundle Index start from 0
        BaseAddress = Peg0BaseAddress;
        Count = Index-SaPeg3MaxBundle;
      }
      PciSegmentAndThenOr32 (BaseAddress + R_SA_PEG_AFEBND0CFG4_OFFSET + (Count * BUNDLE_STEP), Mask32And, Data32Or);
    }
  }
  return;
}

