/** @file
Header file for CpuPcieInitCommonLib.

@copyright
INTEL CONFIDENTIAL
Copyright 2018 - 2019 Intel Corporation.

The source code contained or described herein and all documents related to the
source code ("Material") are owned by Intel Corporation or its suppliers or
licensors. Title to the Material remains with Intel Corporation or its suppliers
and licensors. The Material may contain trade secrets and proprietary and
confidential information of Intel Corporation and its suppliers and licensors,
and is protected by worldwide copyright and trade secret laws and treaty
provisions. No part of the Material may be used, copied, reproduced, modified,
published, uploaded, posted, transmitted, distributed, or disclosed in any way
without Intel's prior express written permission.

No license under any patent, copyright, trade secret or other intellectual
property right is granted to or conferred upon you by disclosure or delivery
of the Materials, either expressly, by implication, inducement, estoppel or
otherwise. Any license under such intellectual property rights must be
express and approved by Intel in writing.

Unless otherwise agreed by Intel in writing, you may not remove or alter
this notice or any other notice embedded in Materials by Intel or
Intel's suppliers or licensors in any way.

This file contains an 'Intel Peripheral Driver' and is uniquely identified as
"Intel Reference Module" and is licensed for Intel CPUs and chipsets under
the terms of your license agreement with Intel or your vendor. This file may
be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _CPU_PCIE_INIT_COMMON_H_
#define _CPU_PCIE_INIT_COMMON_H_

#include <Library/CpuRegbarAccessLib.h>

/**
  This function reads PCI Config register via SBI Access or through IOM mailbox

  @param[in]  RpIndex             Root Port Index (0-based)
  @param[in]  Offset              Offset of Config register
  @param[out] *Data32             Value of Config register

  @retval EFI_SUCCESS             SBI Read successful.
**/
EFI_STATUS
CpuPcieRpSbiRead32 (
  IN    UINT32  RpIndex,
  IN    UINT32  Offset,
  OUT   UINT32  *Data32
  );

/**
  This function And then Or Pci Config register via SBI Access

  @param[in]  RpIndex             Root Port Index (0-based)
  @param[in]  Offset              Offset of Config register
  @param[in]  Data32And           Value of Config register to be And-ed
  @param[in]  Data32AOr           Value of Config register to be Or-ed

  @retval EFI_SUCCESS             SBI Read and Write successful.
**/
EFI_STATUS
CpuPcieRpSbiAndThenOr32 (
  IN  UINT32  RpIndex,
  IN  UINT32  Offset,
  IN  UINT32  Data32And,
  IN  UINT32  Data32Or
  );

/**
  Print registers value

  @param[in] PrintMmioBase       Mmio base address
  @param[in] PrintSize           Number of registers
  @param[in] OffsetFromBase      Offset from mmio base address

  @retval None
**/
VOID
SaPrintRegisters (
  IN  UINTN        PrintMmioBase,
  IN  UINT32       PrintSize,
  IN  UINT32       OffsetFromBase
  );

/**
  Print registers value

  @param[in] PrintPciSegmentBase Pci segment base address
  @param[in] PrintSize           Number of registers
  @param[in] OffsetFromBase      Offset from mmio base address

  @retval None
**/
VOID
SaPrintPciRegisters (
  IN  UINT64       PrintPciSegmentBase,
  IN  UINT32       PrintSize,
  IN  UINT32       OffsetFromBase
  );

//
// 2LM: PegPcie APIs for Sideband Access Mechanism in 2LM mode
//
/**
Reads an 8-bit PCI configuration register.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentRead8 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.

@return The 8-bit PCI configuration register specified by Address.

**/
UINT8
EFIAPI
PegPciSegmentRead8 (
  IN UINT64                    Address
  );

/**
Writes an 8-bit PCI configuration register.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentWrite8 function.

@param  Address     Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  Value       The value to write.

@return The value written to the PCI configuration register.

**/
UINT8
EFIAPI
PegPciSegmentWrite8 (
  IN UINT64                    Address,
  IN UINT8                     Value
  );

/**
Reads a 16-bit PCI configuration register.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentRead16 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.

@return The 16-bit PCI configuration register specified by Address.

**/
UINT16
EFIAPI
PegPciSegmentRead16 (
  IN UINT64                    Address
  );

/**
Writes a 16-bit PCI configuration register.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentWrite16 function.

@param  Address     Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  Value       The value to write.

@return The parameter of Value.

**/
UINT16
EFIAPI
PegPciSegmentWrite16  (
  IN UINT64                    Address,
  IN UINT16                    Value
  );

/**
Reads a 32-bit PCI configuration register.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentRead32 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.

@return The 32-bit PCI configuration register specified by Address.

**/
UINT32
EFIAPI
PegPciSegmentRead32 (
  IN UINT64                    Address
  );

/**
Writes a 32-bit PCI configuration register.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentWrite32 function.

@param  Address     Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  Value       The value to write.

@return The parameter of Value.

**/
UINT32
EFIAPI
PegPciSegmentWrite32 (
  IN UINT64                    Address,
  IN UINT32                    Value
  );

/**
Performs a bitwise OR of a 16-bit PCI configuration register with a 16-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentOr16 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  OrData    The value to OR with the PCI configuration register.

@return The value written to the PCI configuration register.
**/
UINT16
EFIAPI
PegPciSegmentOr16 (
  IN UINT64                    Address,
  IN UINT16                    OrData
  );

/**
Performs a bitwise AND of a 16-bit PCI configuration register with a 16-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentAnd32 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  AndData   The value to AND with the PCI configuration register.

@return The value written to the PCI configuration register.

**/
UINT16
EFIAPI
PegPciSegmentAnd16 (
  IN UINT64                    Address,
  IN UINT16                    AndData
  );

/**
Performs a bitwise AND of a 8-bit PCI configuration register with a 8-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentAnd8 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  AndData   The value to AND with the PCI configuration register.

@return The value written to the PCI configuration register.

**/
UINT8
EFIAPI
PegPciSegmentAnd8 (
  IN UINT64                    Address,
  IN UINT8                     AndData
  );

/**
Performs a bitwise OR of a 32-bit PCI configuration register with a 32-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentOr32 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  OrData    The value to OR with the PCI configuration register.

@return The value written to the PCI configuration register.

**/
UINT32
EFIAPI
PegPciSegmentOr32 (
  IN UINT64                    Address,
  IN UINT32                    OrData
  );

/**
Performs a bitwise OR of a 8-bit PCI configuration register with a 8-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentOr8 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  OrData    The value to OR with the PCI configuration register.

@return The value written to the PCI configuration register.

**/
UINT8
EFIAPI
PegPciSegmentOr8 (
  IN UINT64                    Address,
  IN UINT8                     OrData
  );

/**
Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentAnd32 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  AndData   The value to AND with the PCI configuration register.

@return The value written to the PCI configuration register.

**/
UINT32
EFIAPI
PegPciSegmentAnd32 (
  IN UINT64                    Address,
  IN UINT32                    AndData
  );

/**
Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value,
followed a  bitwise OR with another 32-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentAndThenOr32 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  AndData   The value to AND with the PCI configuration register.
@param  OrData    The value to OR with the PCI configuration register.

@return The value written to the PCI configuration register.

**/
UINT32
EFIAPI
PegPciSegmentAndThenOr32 (
  IN UINT64                    Address,
  IN UINT32                    AndData,
  IN UINT32                    OrData
  );

/**
Performs a bitwise AND of a 16-bit PCI configuration register with a 16-bit value,
followed a  bitwise OR with another 16-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentAndThenOr16 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  AndData   The value to AND with the PCI configuration register.
@param  OrData    The value to OR with the PCI configuration register.

@return The value written to the PCI configuration register.

**/
UINT16
EFIAPI
PegPciSegmentAndThenOr16 (
  IN UINT64                    Address,
  IN UINT16                    AndData,
  IN UINT16                    OrData
  );

/**
Performs a bitwise AND of a 8-bit PCI configuration register with a 8-bit value,
followed a  bitwise OR with another 8-bit value.

Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
Other calls to this function will be routed to core PciSegmentAndThenOr8 function.

@param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
@param  AndData   The value to AND with the PCI configuration register.
@param  OrData    The value to OR with the PCI configuration register.

@return The value written to the PCI configuration register.

**/
UINT8
EFIAPI
PegPciSegmentAndThenOr8 (
  IN UINT64                    Address,
  IN UINT8                     AndData,
  IN UINT8                     OrData
  );

/**
Find the Offset to a Capabilities ID
@param[in] Segment              Pci Segment Number
@param[in] Bus                  Pci Bus Number
@param[in] Device               Pci Device Number
@param[in] Function             Pci Function Number
@param[in] CapId                CAPID to search for

@retval 0                       CAPID not found
@retval Other                   CAPID found, Offset of desired CAPID
**/
UINT8
PegPcieFindCapId (
  IN UINT8   Segment,
  IN UINT8   Bus,
  IN UINT8   Device,
  IN UINT8   Function,
  IN UINT8   CapId
  );
#endif // _CPU_PCIE_INIT_COMMON_H_
