/** @file
 common library for CPU PCIe INIT PEI/DXE/SMM modules

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Uefi/UefiBaseType.h>
#include <Library/PcieHelperLib.h>
#include <Library/DebugLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/IoLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/CpuPcieRpLib.h>
#include <Library/CpuSbiAccessLib.h>
#include <Register/CpuPcieRegs.h>
#include <Library/CpuPcieInitCommon.h>
#include <CpuSbInfo.h>
#include <Library/PeiDxeSmmTwoLmLib.h>
#include <Library/CpuRegbarAccessLib.h>
#include <Library/CpuPlatformLib.h>
#include <Library/PcdLib.h>
#include <Library/CpuPcieInfoFruLib.h>

/**
  This function reads PCI Config register via SBI Access or through IOM mailbox

  @param[in]  RpIndex             Root Port Index (0-based)
  @param[in]  Offset              Offset of Config register
  @param[out] *Data32             Value of Config register

  @retval EFI_SUCCESS             SBI Read successful.
**/
EFI_STATUS
CpuPcieRpSbiRead32 (
  IN    UINT32  RpIndex,
  IN    UINT32  Offset,
  OUT   UINT32  *Data32
  )
{
  EFI_STATUS    Status;
  UINT8         Response;
  UINT16        Fid;

//
// ICL-R does not support FID manipulation
// @TODO : Modify this function to use IOM mailbox command in the case of ICL-R
//

  //
  // Fid = 1 only for PEG12 which according to our design will always be RpIndex 3
  //
  if(RpIndex == 3){
    Fid = 1;
  } else {
    Fid = 0;
  }

  Status = CpuSbiExecutionEx (
             GetCpuPcieRpSbiPid (RpIndex),
             Offset,
             PciConfigRead,
             FALSE,
             0xF,
             0,
             Fid,
             Data32,
             &Response
             );
  if (Status != EFI_SUCCESS) {
    DEBUG((DEBUG_ERROR,"Sideband Read Failed of RpIndex %d Offset 0x%x. Fid = 0x%x\n",RpIndex, Offset, Fid));
    ASSERT (FALSE);
  }
  return Status;
}

/**
  This function And then Or Pci Config register via SBI Access

  @param[in]  RpIndex             Root Port Index (0-based)
  @param[in]  Offset              Offset of Config register
  @param[in]  Data32And           Value of Config register to be And-ed
  @param[in]  Data32AOr           Value of Config register to be Or-ed

  @retval EFI_SUCCESS             SBI Read and Write successful.
**/
EFI_STATUS
CpuPcieRpSbiAndThenOr32 (
  IN  UINT32  RpIndex,
  IN  UINT32  Offset,
  IN  UINT32  Data32And,
  IN  UINT32  Data32Or
  )
{
  EFI_STATUS  Status;
  UINT32      Data32;
  UINT8       Response;
  UINT16      Fid;

//
// ICL-R does not support FID manipulation
// @TODO : Modify this function to use IOM mailbox command in the case of ICL-R
//

  // Fid = 1 only for PEG12 which according to our design will always be RpIndex 3
  if(RpIndex == 3){
    Fid = 1;
  } else {
    Fid = 0;
  }

  Status = CpuPcieRpSbiRead32 (RpIndex, Offset, &Data32);
  if (Status == EFI_SUCCESS) {
    Data32 &= Data32And;
    Data32 |= Data32Or;
    Status = CpuSbiExecutionEx (
               GetCpuPcieRpSbiPid (RpIndex),
               Offset,
               PciConfigWrite,
               FALSE,
               0xF,
               0,
               Fid,
               &Data32,
               &Response
               );
    if (Status != EFI_SUCCESS) {
      DEBUG((DEBUG_ERROR,"Sideband Write Failed of RpIndex %d Offset 0x%x. Fid = 0x%x\n",RpIndex, Offset, Fid));
      ASSERT (FALSE);
    }
  } else {
    ASSERT (FALSE);
  }
  return Status;
}

/**
  Print registers value

  @param[in] PrintMmioBase       Mmio base address
  @param[in] PrintSize           Number of registers
  @param[in] OffsetFromBase      Offset from mmio base address

  @retval None
**/
VOID
SaPrintRegisters (
  IN  UINTN        PrintMmioBase,
  IN  UINT32       PrintSize,
  IN  UINT32       OffsetFromBase
  )
{
  UINT32  Offset;
  DEBUG ((DEBUG_VERBOSE, "       00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F"));
  for (Offset = 0; Offset < PrintSize; Offset++) {
    if ((Offset % 16) == 0) {
      DEBUG ((DEBUG_VERBOSE, "\n %04X: ", (Offset + OffsetFromBase) & 0xFFF0));
    }
    DEBUG ((DEBUG_VERBOSE, "%02X ", MmioRead8 (PrintMmioBase + Offset)));
  }
  DEBUG ((DEBUG_VERBOSE, "\n"));
}

/**
  Print registers value

  @param[in] PrintPciSegmentBase Pci segment base address
  @param[in] PrintSize           Number of registers
  @param[in] OffsetFromBase      Offset from mmio base address

  @retval None
**/
VOID
SaPrintPciRegisters (
  IN  UINT64       PrintPciSegmentBase,
  IN  UINT32       PrintSize,
  IN  UINT32       OffsetFromBase
  )
{
  UINT32  Offset;
  DEBUG ((DEBUG_VERBOSE, "       00 01 02 03 04 05 06 07 08 09 0A 0B 0C 0D 0E 0F"));
  for (Offset = 0; Offset < PrintSize; Offset++) {
    if ((Offset % 16) == 0) {
      DEBUG ((DEBUG_VERBOSE, "\n %04X: ", (Offset + OffsetFromBase) & 0xFFF0));
    }
    DEBUG ((DEBUG_VERBOSE, "%02X ", PciSegmentRead8 (PrintPciSegmentBase + Offset)));
  }
  DEBUG ((DEBUG_VERBOSE, "\n"));
}

//
// 2LM: PegPcie APIs using the Sideband Access Mechanism
//
/**
  Reads an 8-bit PCI configuration register.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentRead8 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.

  @return The 8-bit PCI configuration register specified by Address.
**/
UINT8
EFIAPI
PegPciSegmentRead8 (
  IN UINT64                    Address
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarRead8 (CPU_SB_PID_PEG60, (Address & 0xFFF));
    }
  }

  return PciSegmentRead8 (Address);
}

/**
  Writes an 8-bit PCI configuration register.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentWrite8 function.

  @param  Address     Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  Value       The value to write.

  @return The value written to the PCI configuration register.
**/
UINT8
EFIAPI
PegPciSegmentWrite8 (
  IN UINT64                    Address,
  IN UINT8                     Value
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarWrite8 (CPU_SB_PID_PEG60, (Address & 0xFFF), Value);
    }
  }

  return PciSegmentWrite8 (Address, Value);
}

/**
  Reads a 16-bit PCI configuration register.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentRead16 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.

  @return The 16-bit PCI configuration register specified by Address.
**/
UINT16
EFIAPI
PegPciSegmentRead16 (
  IN UINT64                    Address
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarRead16 (CPU_SB_PID_PEG60, (Address & 0xFFF));
    }
  }

  return PciSegmentRead16 (Address);
}

/**
  Writes a 16-bit PCI configuration register.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentWrite16 function.

  @param  Address     Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  Value       The value to write.

  @return The parameter of Value.
**/
UINT16
EFIAPI
PegPciSegmentWrite16 (
  IN UINT64                    Address,
  IN UINT16                     Value
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarWrite16 (CPU_SB_PID_PEG60, (Address & 0xFFF), Value);
    }
  }

  return PciSegmentWrite16 (Address, Value);
}

/**
  Reads a 32-bit PCI configuration register.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentRead32 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.

  @return The 32-bit PCI configuration register specified by Address.
**/
UINT32
EFIAPI
PegPciSegmentRead32 (
  IN UINT64                    Address
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarRead32 (CPU_SB_PID_PEG60, (Address & 0xFFF));
    }
  }

  return PciSegmentRead32 (Address);
}

/**
  Writes a 32-bit PCI configuration register.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentWrite32 function.

  @param  Address     Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  Value       The value to write.

  @return The parameter of Value.
**/
UINT32
EFIAPI
PegPciSegmentWrite32 (
  IN UINT64                    Address,
  IN UINT32                     Value
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarWrite32 (CPU_SB_PID_PEG60, (Address & 0xFFF), Value);
    }
  }

  return PciSegmentWrite32 (Address, Value);
}

/**
  Performs a bitwise OR of a 16-bit PCI configuration register with a 16-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentOr16 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  OrData    The value to OR with the PCI configuration register.

  @return The value written to the PCI configuration register.
**/
UINT16
EFIAPI
PegPciSegmentOr16 (
  IN UINT64                    Address,
  IN UINT16                    OrData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarOr16 (CPU_SB_PID_PEG60, (Address & 0xFFF), OrData);
    }
  }

  return PciSegmentOr16 (Address, OrData);
}

/**
  Performs a bitwise AND of a 16-bit PCI configuration register with a 16-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentAnd32 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  AndData   The value to AND with the PCI configuration register.

  @return The value written to the PCI configuration register.

**/
UINT16
EFIAPI
PegPciSegmentAnd16 (
  IN UINT64                    Address,
  IN UINT16                    AndData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarAnd16 (CPU_SB_PID_PEG60, (Address & 0xFFF), AndData);
    }
  }

  return PciSegmentAnd16 (Address, AndData);
}

/**
  Performs a bitwise AND of a 8-bit PCI configuration register with a 8-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentAnd8 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  AndData   The value to AND with the PCI configuration register.

  @return The value written to the PCI configuration register.

**/
UINT8
EFIAPI
PegPciSegmentAnd8 (
  IN UINT64                    Address,
  IN UINT8                     AndData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarAnd8 (CPU_SB_PID_PEG60, (Address & 0xFFF), AndData);
    }
  }

  return PciSegmentAnd8 (Address, AndData);
}

/**
  Performs a bitwise OR of a 32-bit PCI configuration register with a 32-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentOr32 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  OrData    The value to OR with the PCI configuration register.

  @return The value written to the PCI configuration register.
**/
UINT32
EFIAPI
PegPciSegmentOr32 (
  IN UINT64                    Address,
  IN UINT32                    OrData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarOr32 (CPU_SB_PID_PEG60, (Address & 0xFFF), OrData);
    }
  }

  return PciSegmentOr32 (Address, OrData);
}

/**
  Performs a bitwise OR of a 8-bit PCI configuration register with a 8-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentOr8 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  OrData    The value to OR with the PCI configuration register.

  @return The value written to the PCI configuration register.
**/
UINT8
EFIAPI
PegPciSegmentOr8 (
  IN UINT64                    Address,
  IN UINT8                     OrData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarOr8 (CPU_SB_PID_PEG60, (Address & 0xFFF), OrData);
    }
  }

  return PciSegmentOr8 (Address, OrData);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentAnd32 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  AndData   The value to AND with the PCI configuration register.

  @return The value written to the PCI configuration register.

**/
UINT32
EFIAPI
PegPciSegmentAnd32 (
  IN UINT64                    Address,
  IN UINT32                    AndData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarAnd32 (CPU_SB_PID_PEG60, (Address & 0xFFF), AndData);
    }
  }

  return PciSegmentAnd32 (Address, AndData);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value,
  followed a  bitwise OR with another 32-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentAndThenOr32 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  AndData   The value to AND with the PCI configuration register.
  @param  OrData    The value to OR with the PCI configuration register.

  @return The value written to the PCI configuration register.
**/
UINT32
EFIAPI
PegPciSegmentAndThenOr32 (
  IN UINT64                    Address,
  IN UINT32                    AndData,
  IN UINT32                    OrData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarAndThenOr32 (CPU_SB_PID_PEG60, (Address & 0xFFF), AndData, OrData);
    }
  }

  return PciSegmentAndThenOr32 (Address, AndData, OrData);
}


/**
  Performs a bitwise AND of a 16-bit PCI configuration register with a 16-bit value,
  followed a  bitwise OR with another 16-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentAndThenOr16 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  AndData   The value to AND with the PCI configuration register.
  @param  OrData    The value to OR with the PCI configuration register.

  @return The value written to the PCI configuration register.
**/
UINT16
EFIAPI
PegPciSegmentAndThenOr16 (
  IN UINT64                    Address,
  IN UINT16                    AndData,
  IN UINT16                    OrData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarAndThenOr16 (CPU_SB_PID_PEG60, (Address & 0xFFF), AndData, OrData);
    }
  }

  return PciSegmentAndThenOr16 (Address, AndData, OrData);
}


/**
  Performs a bitwise AND of a 8-bit PCI configuration register with a 8-bit value,
  followed a  bitwise OR with another 8-bit value.

  Its a wrapper library function. This function uses side band access for PEG60 when 2LM mode is enabled.
  Other calls to this function will be routed to core PciSegmentAndThenOr8 function.

  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register.
  @param  AndData   The value to AND with the PCI configuration register.
  @param  OrData    The value to OR with the PCI configuration register.

  @return The value written to the PCI configuration register.
**/
UINT8
EFIAPI
PegPciSegmentAndThenOr8 (
  IN UINT64                    Address,
  IN UINT8                     AndData,
  IN UINT8                     OrData
  )
{
  if (IsTwoLmEnabled ()) {
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)Address & 0xFFFF000) >> 12) == 0x30) {
      return CpuRegbarAndThenOr8 (CPU_SB_PID_PEG60, (Address & 0xFFF), AndData, OrData);
    }
  }

  return PciSegmentAndThenOr8 (Address, AndData, OrData);
}


/**
  Find the Offset to a Capabilities ID
  @param[in] Segment              Pci Segment Number
  @param[in] Bus                  Pci Bus Number
  @param[in] Device               Pci Device Number
  @param[in] Function             Pci Function Number
  @param[in] CapId                CAPID to search for

  @retval 0                       CAPID not found
  @retval Other                   CAPID found, Offset of desired CAPID
**/
UINT8
PegPcieFindCapId (
  IN UINT8   Segment,
  IN UINT8   Bus,
  IN UINT8   Device,
  IN UINT8   Function,
  IN UINT8   CapId
  )
{
  UINT8   CapHeaderOffset;
  UINT8   CapHeaderId;
  UINT64  DeviceBase;


  if (IsTwoLmEnabled ()) {

    DeviceBase = PCI_SEGMENT_LIB_ADDRESS (Segment, Bus, Device, Function, 0);
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)DeviceBase & 0xFFFF000) >> 12) == 0x30) {
      if ((CpuRegbarRead8 (CPU_SB_PID_PEG60, ((DeviceBase + PCI_PRIMARY_STATUS_OFFSET) & 0xFFF)) & EFI_PCI_STATUS_CAPABILITY) == 0x00) {
        ///
        /// Function has no capability pointer
        ///
        return 0;
      }
      ///
      /// Check the header layout to determine the Offset of Capabilities Pointer Register
      ///
      if ((CpuRegbarRead8 (CPU_SB_PID_PEG60, ((DeviceBase + PCI_HEADER_TYPE_OFFSET) & 0xFFF)) & HEADER_LAYOUT_CODE) == (HEADER_TYPE_CARDBUS_BRIDGE)) {
        ///
        /// If CardBus bridge, start at Offset 0x14
        ///
        CapHeaderOffset = 0x14;
      } else {
        ///
        /// Otherwise, start at Offset 0x34
        ///
        CapHeaderOffset = 0x34;
      }
      ///
      /// Get Capability Header, A pointer value of 00h is used to indicate the last capability in the list.
      ///
      CapHeaderId     = 0;
      CapHeaderOffset = CpuRegbarRead8 (CPU_SB_PID_PEG60, ((DeviceBase + CapHeaderOffset) & 0xFFF)) & ((UINT8) ~(BIT0 | BIT1));
      while (CapHeaderOffset != 0 && CapHeaderId != 0xFF) {
        CapHeaderId = CpuRegbarRead8 (CPU_SB_PID_PEG60, ((DeviceBase + CapHeaderOffset) & 0xFFF));
        if (CapHeaderId == CapId) {
          return CapHeaderOffset;
        }
        ///
        /// Each capability must be DWORD aligned.
        /// The bottom two bits of all pointers (including the initial pointer at 34h) are reserved
        /// and must be implemented as 00b although software must mask them to allow for future uses of these bits.
        ///
        CapHeaderOffset = CpuRegbarRead8 (CPU_SB_PID_PEG60, ((DeviceBase + CapHeaderOffset + 1) & 0xFFF)) & ((UINT8) ~(BIT0 | BIT1));
      }
      return 0;
    }
  }
  return PcieFindCapId (Segment, Bus, Device, Function, CapId);
}


/**
  Search and return the offset of desired Pci Express extended Capability ID
  @param[in] Segment              Pci Segment Number
  @param[in] Bus                  Pci Bus Number
  @param[in] Device               Pci Device Number
  @param[in] Function             Pci Function Number
  @param[in] CapId                Extended CAPID to search for

  @retval 0                       CAPID not found
  @retval Other                   CAPID found, Offset of desired CAPID
**/
UINT16
PegPcieFindExtendedCapId (
  IN UINT8   Segment,
  IN UINT8   Bus,
  IN UINT8   Device,
  IN UINT8   Function,
  IN UINT16  CapId
  )
{
  UINT16  CapHeaderOffset;
  UINT16  CapHeaderId;
  UINT64  DeviceBase;

  if (IsTwoLmEnabled ()) {

    DeviceBase = PCI_SEGMENT_LIB_ADDRESS (Segment, Bus, Device, Function, 0);
    //
    // Use side band access for PEG60 (Bus 0 Dev 6 Fun 0) when 2LM mode is enabled
    //
    if ((((UINT32)DeviceBase & 0xFFFF000) >> 12) == 0x30) {
      ///
      /// Start to search at Offset 0x100
      /// Get Capability Header, A pointer value of 00h is used to indicate the last capability in the list.
      ///
      CapHeaderId     = 0;
      CapHeaderOffset = 0x100;
      while (CapHeaderOffset != 0 && CapHeaderId != 0xFFFF) {
        CapHeaderId = CpuRegbarRead16 (CPU_SB_PID_PEG60, ((DeviceBase + CapHeaderOffset) & 0xFFF));
        if (CapHeaderId == CapId) {
          return CapHeaderOffset;
        }
        ///
        /// Each capability must be DWORD aligned.
        /// The bottom two bits of all pointers are reserved and must be implemented as 00b
        /// although software must mask them to allow for future uses of these bits.
        ///
        CapHeaderOffset = CpuRegbarRead16 (CPU_SB_PID_PEG60, ((DeviceBase + CapHeaderOffset+2) & 0xFFF)) >> 4 & ((UINT16) ~(BIT0 | BIT1));
      }
      return 0;
    }
  }
  return PcieFindExtendedCapId (Segment, Bus, Device, Function, CapId);
}
