/** @file
  This file contains functions needed to create PCIE_ROOT_PORT_DEV structure.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Register/PchRegs.h>
#include <Library/DebugLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/PeiDxeSmmTwoLmLib.h>
#include <Library/CpuPcieRpLib.h>
#include <Library/CpuPlatformLib.h>
#include <Library/CpuPcieInfoFruLib.h>
#include <Library/CpuPcieInitCommon.h>

#include "CpuPcieRpInitLibInternal.h"

/**
  Reads a 16-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT8
CpuPcieRpCfgRead8 (
  IN PCIE_ROOT_PORT_DEV  *RpDev,
  IN UINT64              Offset
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentRead8 (Private->PciSegmentBase + Offset);
}

/**
  Writes a 8-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT8
CpuPcieRpCfgWrite8 (
  IN PCIE_ROOT_PORT_DEV  *RpDev,
  IN UINT64              Offset,
  IN UINT8               Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentWrite8 (Private->PciSegmentBase + Offset, Value);
}

/**
  Performs a bitwise OR of a 8-bit PCI configuration register with a 8-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT8
CpuPcieRpCfgOr8 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT8                     OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentOr8 (Private->PciSegmentBase + Offset, OrData);
}

/**
  Performs a bitwise AND of a 8-bit PCI configuration register with a 8-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT8
CpuPcieRpCfgAnd8 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT8                     AndData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentAnd8 (Private->PciSegmentBase + Offset, AndData);
}

/**
  Performs a bitwise AND of a 8-bit PCI configuration register with a 8-bit value,
  followed a  bitwise OR with another 8-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT8
CpuPcieRpCfgAndThenOr8 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT8                     AndData,
  IN UINT8                     OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentAndThenOr8 (Private->PciSegmentBase + Offset, AndData, OrData);
}

/**
  Reads a 16-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT16
CpuPcieRpCfgRead16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentRead16 (Private->PciSegmentBase + Offset);
}

/**
  Writes a 16-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT16
CpuPcieRpCfgWrite16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT16                    Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentWrite16 (Private->PciSegmentBase + Offset, Value);
}

/**
  Performs a bitwise OR of a 16-bit PCI configuration register with a 16-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT16
CpuPcieRpCfgOr16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT16                    OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentOr16 (Private->PciSegmentBase + Offset, OrData);
}

/**
  Performs a bitwise AND of a 16-bit PCI configuration register with a 16-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT16
CpuPcieRpCfgAnd16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT16                    AndData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentAnd16 (Private->PciSegmentBase + Offset, AndData);
}

/**
  Performs a bitwise AND of a 16-bit PCI configuration register with a 16-bit value,
  followed a  bitwise OR with another 16-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT16
CpuPcieRpCfgAndThenOr16 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT16                    AndData,
  IN UINT16                    OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentAndThenOr16 (Private->PciSegmentBase + Offset, AndData, OrData);
}

/**
  Reads a 32-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT32
CpuPcieRpCfgRead32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentRead32 (Private->PciSegmentBase + Offset);
}

/**
  Writes a 32-bit PCI configuration register

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT32
CpuPcieRpCfgWrite32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentWrite32 (Private->PciSegmentBase + Offset, Value);
}

/**
  Performs a bitwise OR of a 32-bit PCI configuration register with a 32-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpCfgOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentOr32 (Private->PciSegmentBase + Offset, OrData);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpCfgAnd32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentAnd32 (Private->PciSegmentBase + Offset, AndData);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value,
  followed a  bitwise OR with another 32-bit value

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpCfgAndThenOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData,
  IN UINT32                    OrData
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  return PegPciSegmentAndThenOr32 (Private->PciSegmentBase + Offset, AndData, OrData);
}

/**
  Reads a 32-bit PCI configuration register via the SBI message

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT32
CpuPcieRpSbiMsgCfgRead32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset
  )
{
  EFI_STATUS                  Status;
  UINT8                       Response;
  UINT32                      Data32;
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  Status = CpuSbiExecutionEx (
             Private->Pid,
             Offset,
             PciConfigRead,
             FALSE,
             0xF,
             0,
             Private->Fid,
             &Data32,
             &Response
             );
  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR,"Sideband Read Failed of Offset 0x%x, Pid = 0x%x, Fid = 0x%x\n", (UINT32)Offset, (UINT32)Private->Pid, (UINT32)Private->Fid));
    Data32 = MAX_UINT32;
  }

  return Data32;
}

/**
  Writes a 32-bit PCI configuration register via the SBI message

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT32
CpuPcieRpSbiMsgCfgWrite32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;
  UINT8                       Response;
  EFI_STATUS                  Status;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  Status = CpuSbiExecutionEx (
             Private->Pid,
             Offset,
             PciConfigWrite,
             FALSE,
             0xF,
             0,
             Private->Fid,
             &Value,
             &Response
             );
  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR,"Sideband Write Failed of Offset 0x%x, Pid = 0x%x, Fid = 0x%x\n", (UINT32)Offset, (UINT32)Private->Pid, (UINT32)Private->Fid));
    Value = MAX_UINT32;
  }

  return Value;
}

/**
  Performs a bitwise OR of a 32-bit PCI configuration register with a 32-bit value
  via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpSbiMsgCfgOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    OrData
  )
{
  UINT32                      Data32;

  Data32 = CpuPcieRpSbiMsgCfgRead32 (RpDev, Offset);
  Data32 |= OrData;
  return CpuPcieRpSbiMsgCfgWrite32 (RpDev, Offset, Data32);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value
  via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpSbiMsgCfgAnd32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData
  )
{
  UINT32                      Data32;

  Data32 = CpuPcieRpSbiMsgCfgRead32 (RpDev, Offset);
  Data32 &= AndData;
  return CpuPcieRpSbiMsgCfgWrite32 (RpDev, Offset, Data32);
}

/**
  Performs a bitwise AND of a 32-bit PCI configuration register with a 32-bit value,
  followed a  bitwise OR with another 32-bit value via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpSbiMsgCfgAndThenOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData,
  IN UINT32                    OrData
  )
{
  UINT32                      Data32;

  Data32 = CpuPcieRpSbiMsgCfgRead32 (RpDev, Offset);
  Data32 &= AndData;
  Data32 |= OrData;
  return CpuPcieRpSbiMsgCfgWrite32 (RpDev, Offset, Data32);
}

/**
  Reads a 32-bit MEM register via the SBI message

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config

  @return The 32-bit PCI configuration register specified by Address
**/
UINT32
CpuPcieRpSbiMsgMemRead32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset
  )
{
  EFI_STATUS                  Status;
  UINT8                       Response;
  UINT32                      Data32;
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  Status = CpuSbiExecutionEx (
             Private->Pid,
             Offset,
             MemoryRead,
             FALSE,
             0xF,
             0,
             Private->Fid,
             &Data32,
             &Response
             );
  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR,"Sideband Read Failed of Offset 0x%x, Pid = 0x%x, Fid = 0x%x\n", (UINT32)Offset, (UINT32)Private->Pid, (UINT32)Private->Fid));
    Data32 = MAX_UINT32;
  }

  return Data32;
}

/**
  Writes a 32-bit MEM register via the SBI message

  @param  RpDev       Pointer to the PCIe root port device
  @param  Offset      Offset of the register within the PCI config
  @param  Value       The value to write

  @return The parameter of Value
**/
UINT32
CpuPcieRpSbiMsgMemWrite32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    Value
  )
{
  PCIE_ROOT_PORT_DEV_PRIVATE  *Private;
  UINT8                       Response;
  EFI_STATUS                  Status;

  Private = (PCIE_ROOT_PORT_DEV_PRIVATE*) RpDev;

  Status = CpuSbiExecutionEx (
             Private->Pid,
             Offset,
             MemoryWrite,
             FALSE,
             0xF,
             0,
             Private->Fid,
             &Value,
             &Response
             );
  if (Status != EFI_SUCCESS) {
    DEBUG ((DEBUG_ERROR,"Sideband Write Failed of Offset 0x%x, Pid = 0x%x, Fid = 0x%x\n", (UINT32)Offset, (UINT32)Private->Pid, (UINT32)Private->Fid));
    Value = MAX_UINT32;
  }

  return Value;
}

/**
  Performs a bitwise OR of a 32-bit MEM register with a 32-bit value
  via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpSbiMsgMemOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    OrData
  )
{
  UINT32                      Data32;

  Data32 = CpuPcieRpSbiMsgMemRead32 (RpDev, Offset);
  Data32 |= OrData;
  return CpuPcieRpSbiMsgMemWrite32 (RpDev, Offset, Data32);
}

/**
  Performs a bitwise AND of a 32-bit MEM register with a 32-bit value
  via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  Address   Address that encodes the PCI Segment, Bus, Device, Function, and Register
  @param  AndData   The value to AND with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpSbiMsgMemAnd32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData
  )
{
  UINT32                      Data32;

  Data32 = CpuPcieRpSbiMsgMemRead32 (RpDev, Offset);
  Data32 &= AndData;
  return CpuPcieRpSbiMsgMemWrite32 (RpDev, Offset, Data32);
}

/**
  Performs a bitwise AND of a 32-bit MEM register with a 32-bit value,
  followed a  bitwise OR with another 32-bit value via the SBI message

  @param  RpDev     Pointer to PCIe root port device
  @param  Offset    Offset of the register within the PCI config
  @param  AndData   The value to AND with the PCI configuration register
  @param  OrData    The value to OR with the PCI configuration register

  @return The value written to the PCI configuration register
**/
UINT32
CpuPcieRpSbiMsgMemAndThenOr32 (
  IN PCIE_ROOT_PORT_DEV        *RpDev,
  IN UINT64                    Offset,
  IN UINT32                    AndData,
  IN UINT32                    OrData
  )
{
  UINT32                      Data32;

  Data32 = CpuPcieRpSbiMsgMemRead32 (RpDev, Offset);
  Data32 &= AndData;
  Data32 |= OrData;
  return CpuPcieRpSbiMsgMemWrite32 (RpDev, Offset, Data32);
}

/**
  Create PCIE_ROOT_PORT_DEV_PRIVATE object

  @param[in]  RpIndex  Index of the root port on the CPU
  @param[out] RpDev    On output pointer to the valid PCIE_ROOT_PORT_DEV_PRIVATE
**/
VOID
CpuPcieGetRpDev (
  IN UINT32                       RpIndex,
  OUT PCIE_ROOT_PORT_DEV_PRIVATE  *RpDevPrivate
  )
{
  UINTN               RpDevice;
  UINTN               RpFunction;
  PCIE_ROOT_PORT_DEV  *RpDev;

  if (RpDevPrivate == NULL) {
    DEBUG ((DEBUG_ERROR, "RpDevPrivate can't be NULL\n"));
    return;
  }

  RpDev = &RpDevPrivate->RpDev;

  if (GetCpuPcieSipInfo (RpIndex) == PCIE_SIP17) {
    RpDev->SipVersion = PcieSip17;
  } else {
    RpDev->SipVersion = PcieSip16;
  }

  RpDev->Integration = CpuPcie;

  GetCpuPcieRpDevFun (RpIndex, &RpDevice, &RpFunction);
  RpDev->Sbdf.Segment = SA_SEG_NUM;
  RpDev->Sbdf.Bus = SA_MC_BUS;
  RpDev->Sbdf.Device = (UINT16)RpDevice;
  RpDev->Sbdf.Function = (UINT16)RpFunction;

  RpDev->PciCfgAccess.Read8 = CpuPcieRpCfgRead8;
  RpDev->PciCfgAccess.Write8 = CpuPcieRpCfgWrite8;
  RpDev->PciCfgAccess.Or8 = CpuPcieRpCfgOr8;
  RpDev->PciCfgAccess.And8 = CpuPcieRpCfgAnd8;
  RpDev->PciCfgAccess.AndThenOr8 = CpuPcieRpCfgAndThenOr8;

  RpDev->PciCfgAccess.Read16 = CpuPcieRpCfgRead16;
  RpDev->PciCfgAccess.Write16 = CpuPcieRpCfgWrite16;
  RpDev->PciCfgAccess.Or16 = CpuPcieRpCfgOr16;
  RpDev->PciCfgAccess.And16 = CpuPcieRpCfgAnd16;
  RpDev->PciCfgAccess.AndThenOr16 = CpuPcieRpCfgAndThenOr16;

  RpDev->PciCfgAccess.Read32 = CpuPcieRpCfgRead32;
  RpDev->PciCfgAccess.Write32 = CpuPcieRpCfgWrite32;
  RpDev->PciCfgAccess.Or32 = CpuPcieRpCfgOr32;
  RpDev->PciCfgAccess.And32 = CpuPcieRpCfgAnd32;
  RpDev->PciCfgAccess.AndThenOr32 = CpuPcieRpCfgAndThenOr32;

  RpDev->PciSbiMsgCfgAccess.Read32 = CpuPcieRpSbiMsgCfgRead32;
  RpDev->PciSbiMsgCfgAccess.Write32 = CpuPcieRpSbiMsgCfgWrite32;
  RpDev->PciSbiMsgCfgAccess.Or32 = CpuPcieRpSbiMsgCfgOr32;
  RpDev->PciSbiMsgCfgAccess.And32 = CpuPcieRpSbiMsgCfgAnd32;
  RpDev->PciSbiMsgCfgAccess.AndThenOr32 = CpuPcieRpSbiMsgCfgAndThenOr32;

  RpDev->PciSbiMsgMemAccess.Read32 = CpuPcieRpSbiMsgMemRead32;
  RpDev->PciSbiMsgMemAccess.Write32 = CpuPcieRpSbiMsgMemWrite32;
  RpDev->PciSbiMsgMemAccess.Or32 = CpuPcieRpSbiMsgMemOr32;
  RpDev->PciSbiMsgMemAccess.And32 = CpuPcieRpSbiMsgMemAnd32;
  RpDev->PciSbiMsgMemAccess.AndThenOr32 = CpuPcieRpSbiMsgMemAndThenOr32;

  RpDevPrivate->PciSegmentBase = PCI_SEGMENT_LIB_ADDRESS (
                                   SA_SEG_NUM,
                                   SA_MC_BUS,
                                   RpDevice,
                                   RpFunction,
                                   0
                                   );
  RpDevPrivate->Pid = GetCpuPcieRpSbiPid (RpIndex);
  // Fid = 1 only for PEG12 which according to our design will always be RpIndex 3
  if(RpIndex == 3){
    RpDevPrivate->Fid = 1;
  } else {
    RpDevPrivate->Fid = 0;
  }
}

