/** @file
  This file provides Fusa Display Isoch Port Parity E2E CTC
  implementation

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/TimerLib.h>
#include <Register/SaRegsHostBridge.h>
#include <Register/IgdRegs.h>

#include "PeiFusaPrivateLibInternal.h"
#include "PeiFusaResultReportingLib.h"

STATIC UINTN             mMchBase;
STATIC UINTN             mGttMmAdr;

/**
  "Poll Status" for GT Readiness

  @note 3s polling timeout behaviour for the debug build is
        assert while the release build is nothing. If there is a
        failure in polling, the CTC test result would reflects
        that.

  @param[in] Base            - Base address of MMIO
  @param[in] Offset          - MMIO Offset
  @param[in] Mask            - Mask
  @param[in] Result          - Value to wait for
**/
VOID
PollStatus (
  IN UINT32 Base,
  IN UINT32 Offset,
  IN UINT32 Mask,
  IN UINT32 Result
  )
{
  UINT32  GtStatus;
  UINT32  StallCount;

  StallCount = 0;

  ///
  /// Register read
  ///
  GtStatus = MmioRead32 (Base + Offset);

  while (((GtStatus & Mask) != Result)
         && (StallCount < 3000000U)) {
    ///
    /// 10 microSec wait
    ///
    MicroSecondDelay (10);
    StallCount += 10;

    GtStatus = MmioRead32 (Base + Offset);
  }

  ASSERT ((StallCount < 3000000U));
}

#define P24C_PCODE_MAILBOX_INTERFACE_0_2_0_GTTMMADR 0x138130U
#define PCODE_P24C_MAILBOX_INTERFACE_0_2_0_GTTMMADR 0x138138U
#define DPLL0_ENABLE                                0x46010U
#define PWR_WELL_CTL_DDI1                           0x45450U
#define TRANS_CLK_SEL_A                             0x46140U
#define WM_LINETIME_A                               0x45270U
#define PLANE_SIZE_1_A                              0x70190U
#define PIPE_SRCSZ_A                                0x6001CU
#define PLANE_WM_1_1_A                              0x70244U
#define PLANE_STRIDE_1_A                            0x70188U
#define PLANE_SURF_1_A                              0x7019CU
#define PLANE_COLOR_CTL_1_A                         0x701CCU
#define PLANE_CTL_1_A                               0x70180U
#define PS_CTRL_1_A                                 0x68180U
#define PS_WIN_SZ_1_A                               0x68174U
#define TRANS_HTOTAL_A                              0x60000U
#define TRANS_HBLANK_A                              0x60004U
#define TRANS_HSYNC_A                               0x60008U
#define TRANS_VTOTAL_A                              0x6000CU
#define TRANS_VBLANK_A                              0x60010U
#define TRANS_VSYNC_A                               0x60014U
#define TRANS_DDI_FUNC_CTL_A                        0x60400U
#define TRANS_CONF_A                                0x70008U


UINT32
InjectionAndTrafficDisplayEngine (
  IN UINT32 IpIndex,
  IN UINT32 InjectionPayload
  )
{
  DEBUG ((DEBUG_INFO, "TrafficDisplayEngineVC1Read at 0x%x\n", InjectionAndTrafficDisplayEngine));

  FuSaMsrWrite (IpIndex, InjectionPayload);

  //traffic
  //Enable PCH Reset Handshake
  MmioOr32 ((mGttMmAdr + R_SA_GTTMMADR_NDE_RSTWRN_OPT_OFFSET), BIT4);
  // Enable Display Power Well #1
  MmioOr32 (mGttMmAdr + R_SA_GTTMMADR_PWR_WELL_CTL_OFFSET, B_SA_GTTMMADR_PWR_WELL_CTL_PG_1_ENABLE);
  // Poll for PG1 state
  PollStatus (mGttMmAdr, R_SA_GTTMMADR_PWR_WELL_CTL_OFFSET, B_SA_GTTMMADR_PWR_WELL_CTL_PG_1_STATE, B_SA_GTTMMADR_PWR_WELL_CTL_PG_1_STATE);
  DEBUG ((DEBUG_INFO, "Display Power Well #1 is ON \n"));
  // Inform Power control of upcoming frequency change
  PollStatus (mGttMmAdr, P24C_PCODE_MAILBOX_INTERFACE_0_2_0_GTTMMADR, BIT31, 0); // Poll run-busy before start
  PollStatus (mGttMmAdr, PCODE_P24C_MAILBOX_INTERFACE_0_2_0_GTTMMADR, BIT31, 0); // Poll run-busy before start
  PollStatus (mGttMmAdr, R_SA_GTTMMADR_MAILBOX_INTERFACE_OFFSET, BIT31, 0); // Poll run-busy before start
  MmioWrite32 (mGttMmAdr + R_SA_GTTMMADR_MAILBOX_DATA_LOW_OFFSET, 0x00000003);  // mailbox_low       = 0x00000003
  MmioWrite32 (mGttMmAdr + R_SA_GTTMMADR_MAILBOX_DATA_HIGH_OFFSET, 0x00000000); // mailbox_high      = 0x00000000
  MmioWrite32 (mGttMmAdr + R_SA_GTTMMADR_MAILBOX_INTERFACE_OFFSET, 0x80000007); // mailbox Interface = 0x80000007
  PollStatus (mGttMmAdr, R_SA_GTTMMADR_MAILBOX_INTERFACE_OFFSET, BIT31, 0);;   // Poll Run Busy cleared
  // Check for MailBox Data read status successful
  if ((MmioRead32 (mGttMmAdr + R_SA_GTTMMADR_MAILBOX_DATA_LOW_OFFSET) & BIT0) == 1) {
    DEBUG ((DEBUG_INFO, "Mailbox Data low read Successfull \n"));
  }

  //Enabling CDCLK
  MmioWrite32 (mGttMmAdr + R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, V_SA_CDCLK_PLL_RATIO_652_8MHZ_REF_38_4MHZ);
  MmioOr32 (mGttMmAdr + R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, B_SA_CDCLK_PLL_ENABLE_BIT);
  PollStatus (mGttMmAdr, R_SA_GTTMMADR_CDCLK_PLL_ENABLE_OFFSET, B_SA_CDCLK_PLL_LOCK_BIT, B_SA_CDCLK_PLL_LOCK_BIT);
  MmioAndThenOr32 (mGttMmAdr + R_SA_GTTMMADR_CDCLK_CTL_OFFSET, 0xFF3FF800U, V_SA_CDCLK_CTL_CD_FREQ_DECIMAL_652_8);
  MmioWrite32 (mGttMmAdr + 0x164280U, 0x1E17C00U); //DISPLAY_CCU_DPCLKA_CFGCR0 ==0x164280U
  MmioWrite32 (mGttMmAdr + DPLL0_ENABLE, 0);
  PollStatus (mGttMmAdr, DPLL0_ENABLE, 0xFFFFFFFFU, 0U);
  MmioOr32(mGttMmAdr + DPLL0_ENABLE, BIT27); //enable power
  PollStatus (mGttMmAdr, DPLL0_ENABLE, (BIT27 | BIT26), (BIT27 | BIT26)); //power state being true
  MmioAnd32 (mGttMmAdr + 0x164284U, 0xFDFFFFFFU);
  MmioWrite32 (mGttMmAdr + 0x164284U, 0x8001E1U);
  MmioOr32 (mGttMmAdr + DPLL0_ENABLE, BIT31);
  PollStatus (mGttMmAdr, DPLL0_ENABLE, (BIT31 | BIT30), (BIT31 | BIT30)); //enable state
  MmioAnd32 (mGttMmAdr + 0x164280U, 0xFFFFFFFCU);
  MmioAnd32 (mGttMmAdr + 0x164280U, 0xFFFFFBFFU);
  MmioWrite32 (mGttMmAdr + PWR_WELL_CTL_DDI1, BIT1);
  PollStatus (mGttMmAdr, PWR_WELL_CTL_DDI1, (BIT1 | BIT0), (BIT1 | BIT0)); //power state being true
  DEBUG ((DEBUG_INFO, "CDCLK Enabling Successfull \n"));

  MmioOr32 (mGttMmAdr + TRANS_CLK_SEL_A, BIT28);
  MmioWrite32 (mGttMmAdr + WM_LINETIME_A, 0x6CU);
  MmioWrite32 (mGttMmAdr + PLANE_SIZE_1_A, 0x257031FU);
  MmioWrite32 (mGttMmAdr + PIPE_SRCSZ_A, 0x31F0257U);
  MmioAndThenOr32 (mGttMmAdr + PLANE_WM_1_1_A, 0xFFFFF800U, 0x9U);
  MmioAndThenOr32 (mGttMmAdr + PLANE_WM_1_1_A, 0xFFF83FFFU, 0x00004000U);
  MmioOr32 (mGttMmAdr + PLANE_WM_1_1_A, BIT31);
  MmioWrite32 (mGttMmAdr + PLANE_STRIDE_1_A, 0x32U);
  MmioWrite32 (mGttMmAdr + PLANE_SURF_1_A, 0U);
  MmioOr32 (mGttMmAdr + PLANE_COLOR_CTL_1_A, BIT13);
  MmioAndThenOr32 (mGttMmAdr + PLANE_CTL_1_A, 0xF07FFFFFU, 0x04000000U);
  MmioOr32 (mGttMmAdr + PLANE_CTL_1_A, BIT31);
  MmioOr32 (mGttMmAdr + PLANE_SURF_1_A, 0); //read and writeback
  MmioOr32 (mGttMmAdr + PS_CTRL_1_A, BIT31);
  MmioWrite32 (mGttMmAdr + PS_WIN_SZ_1_A, 0x78004B0U);
  MmioWrite32 (mGttMmAdr + TRANS_HTOTAL_A, 0x81F077FU);
  MmioWrite32 (mGttMmAdr + TRANS_HBLANK_A, 0x81F077FU);
  MmioWrite32 (mGttMmAdr + TRANS_HSYNC_A, 0x7CF07AFU);
  MmioWrite32 (mGttMmAdr + TRANS_VTOTAL_A, 0x4D204AFU);
  MmioWrite32 (mGttMmAdr + TRANS_VBLANK_A, 0x4D204AFU);
  MmioWrite32 (mGttMmAdr + TRANS_VSYNC_A, 0x4B804B2U);
  MmioAnd32 (mGttMmAdr + TRANS_DDI_FUNC_CTL_A, 0xFFFCFFFFU);
  MmioAnd32 (mGttMmAdr + TRANS_DDI_FUNC_CTL_A, 0xFF8FFFFFU);
  MmioAndThenOr32 (mGttMmAdr + TRANS_DDI_FUNC_CTL_A, 0x0fffffffU, 0x08000000U);
  MmioOr32 (mGttMmAdr + TRANS_DDI_FUNC_CTL_A, 0x80000000U);
  MmioOr32 (mGttMmAdr + TRANS_CONF_A, BIT31);
  MmioOr32 (mGttMmAdr + 0x64000U, BIT31);
  MicroSecondDelay (100U);
  MmioAnd32 (mGttMmAdr + TRANS_CONF_A, ~BIT31);
  MmioAnd32 (mGttMmAdr + 0x64000U, ~BIT31);

  return FuSaMsrRead (IpIndex);
}

///
/// Fusa Error Injection Configuration Registers
///

//common FuSa error injection bit field
#define B_FUSA_PERRINJ_CTRL_ADDRESS_PARITY_FLIP                BIT8     ///<FUSA_PERRINJ_CTRL.ADDRESS_PARITY_FLIP
#define B_FUSA_PERRINJ_CTRL_DATA_PARITY_FLIP                   0xFFU    ///<FUSA_PERRINJ_CTRL.DATA_PARITY_FLIP

#define R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH                       0x1016CU

//readback to make sure it is 0x7ff (default enable all parity mechanism), also use this to selectively enable reporting
#define R_SA_MCHBAR_FUSA_ERROR_REPORTING_EN_0_0_0_MCHBAR_IMPH                                     0x10170U
#define B_SA_MCHBAR_FUSA_ERROR_REPORTING_EN_0_0_0_MCHBAR_IMPH_CMI_CPL_DATA_PARITY_REPORTING_EN    0xFF0U
#define B_SA_MCHBAR_FUSA_ERROR_REPORTING_EN_0_0_0_MCHBAR_IMPH_UP_CMD_ADDRESS_PARITY_REPORTING_EN  BIT1
#define B_SA_MCHBAR_FUSA_ERROR_REPORTING_EN_0_0_0_MCHBAR_IMPH_IOSF_MCMD_PARITY_REPORTING_EN       BIT0


/**
  DIP Parity Error E2E CTC internal main routine

  @param[in,out] pFusaTestResult - test result buffer
       where the test result to be updated
**/
STATIC
FUSA_LIB_STATUS
DisplayIsochPortCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8 McaBankNum = MCA_IOP;
  UINT64 ExpectedMcaStatus = 0x11010E0BULL;
  UINT64 McaStatusMask = MCACOD_MASK | MSCOD_MASK;
  UINT32 Data32;

  DEBUG ((DEBUG_INFO, "DisplayIsochPortCtc\n"));

//Verfy that no parity checker over-rides are set (all bits in bitmap clear):
//Read and verify that FUSA_PARITY_EN_OVRD_0_0_0_MCHBAR_IMPH = 0x0;
//Verify that error reporting is enabled for all parity checker sites (all bits in bitmap set):
//Read and verify that FUSA_ERROR_REPORTING_EN_0_0_0_MCHBAR_IMPH = 0xFF7


  LibStatus = FusaMsrCtcTestGeneric (
                DIP_IOSF_MCMD,
                B_FUSA_PERRINJ_CTRL_ADDRESS_PARITY_FLIP,
                InjectionAndTrafficDisplayEngine,
                McaBankNum,
                McaStatusMask,
                ExpectedMcaStatus,
                0U, // check number
                pFusaTestResult,
                TRUE
                );
  ASSERT (LibStatus == FusaNoError);

  Data32 = MmioRead32 (mMchBase + R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value is 0x%x\n", Data32));
  MmioWrite32 (mMchBase + R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH, Data32); //clear the status

  LibStatus |= FusaMsrCtcTestGeneric (
                DIP_UP_CMD_ADDRESS,
                B_FUSA_PERRINJ_CTRL_ADDRESS_PARITY_FLIP,
                InjectionAndTrafficDisplayEngine,
                McaBankNum,
                McaStatusMask,
                ExpectedMcaStatus,
                1U, // check number
                pFusaTestResult,
                TRUE
                );
  ASSERT (LibStatus == FusaNoError);

  Data32 = MmioRead32 (mMchBase + R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value is 0x%x\n", Data32));
  MmioWrite32 (mMchBase + R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH, Data32); //clear the status

  LibStatus |= FusaMsrCtcTestGeneric (
                DIP_CMI_CPL_DATA,
                B_FUSA_PERRINJ_CTRL_DATA_PARITY_FLIP,
                InjectionAndTrafficDisplayEngine,
                McaBankNum,
                McaStatusMask,
                ExpectedMcaStatus,
                2U, // check number
                pFusaTestResult,
                TRUE
                );
  ASSERT (LibStatus == FusaNoError);

  Data32 = MmioRead32 (mMchBase + R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value is 0x%x\n", Data32));
  MmioWrite32 (mMchBase + R_SA_MCHBAR_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH, Data32); //clear the status

  return LibStatus;
}

/**
  Perform Display Isoch Port (DIP) Parity Error E2E CTC. The
  test targets DIP multiple interfaces.

  @note The test assumes MchBar and GttMmBar have been set up
        with 32 bit addressing and being enabled.

  @param[out] pFusaTestResult - pointer to test result
                         buffer for test FusaTestNumDip
**/
VOID
FspDxCheckDip (
  OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{

  FUSA_LIB_STATUS   LibStatus;
  UINT32            Data32;

  DEBUG ((DEBUG_INFO, "FspDxCheckDip\n"));
  LibStatus = MchBarAddressGet (&mMchBase);
  LibStatus |= GttMmBarAddressGet (&mGttMmAdr);
  if (LibStatus == FusaNoError) {
    //enable the error reporting to MCA
    MmioOr32 (
      mMchBase + R_SA_MCHBAR_FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH,
      B_SA_MCHBAR_FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH_MCA_REPORTING_EN
      );

    //Read and verify that FUSA_ERROR_REPORTING_EN_0_0_0_MCHBAR_IMPH bit[x] is 1 for the feature;
    //i.e. parity error reporting is enabled
    //The verifying step is simply a development debug flow, no special code step need to be taken because if it fails,
    //the test result will be shown as fail. It can even be used by the validation team to test this CTC feature.
    Data32 = MmioRead32 (mMchBase + R_SA_MCHBAR_FUSA_ERROR_REPORTING_EN_0_0_0_MCHBAR_IMPH);
    DEBUG ((DEBUG_INFO, "FUSA_ERROR_REPORTING_EN_0_0_0_MCHBAR_IMPH value is 0x%x\n", Data32));

    LibStatus = FusaTestAndReporting (
                  DisplayIsochPortCtc,
                  FusaTestNumDip,
                  3U,
                  pFusaTestResult
                  );
    ASSERT (LibStatus == FusaNoError);
    DumpResults (pFusaTestResult);
  } else {
    LibStatus = FusaTestAndReporting (
                  NULL,
                  FusaTestNumDip,
                  3U,
                  pFusaTestResult
                  );
    ASSERT (LibStatus == FusaNoError);
    ASSERT (FALSE);
  }
}

