/** @file
  This file provides Fusa IOP Parity E2E CTC
  implementation

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/PciSegmentLib.h>
#include "PeiFusaPrivateLibInternal.h"
#include "PeiFusaResultReportingLib.h"

#define R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH                  0x6F2C   ///<FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH
//common FuSa error injection bit field
#define B_FUSA_PERRINJ_CTRL_BYTE_ENBALE_PARITY_FLIP_ENABLE              BIT9     ///<FUSA_PERRINJ_CTRL.BYTE_ENBALE_PARITY_FLIP_ENABLE
#define B_FUSA_PERRINJ_CTRL_ADDRESS_PARITY_FLIP_ENABLE                  BIT8     ///<FUSA_PERRINJ_CTRL.ADDRESS_PARITY_FLIP_ENABLE
//data flip enable is BIT7-BIT0
#define R_SA_IOP_FUSA_PARITY_EN_OVRD_0_0_0_MCHBAR_IMPH                  0x6F20  ///<FUSA_PARITY_EN_OVRD_0_0_0_MCHBAR_IMPH

#define R_SA_WRC_VC_ENABLE                                              0x6F10U   ///<WRC_VC_ENABLE
#define R_SA_HCTL0_0_0_0_MCHBAR_IMPH                                    0x7100U   ///<HCTL0_0_0_0_MCHBAR_IMPH


STATIC UINTN mMchBase;
STATIC UINTN mGmAdr;
STATIC UINTN mHpetBase;

/**
  Error injection and corresponding traffic routine for IO Port
  Cs0 (IosfUpData) and Cs4 (Rch)

  @param[in] IpIndex - IP index vale for MSR MSR_PERRINJ_AT_IP
  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL

  @return injection control value after traffic generation
**/
STATIC
UINT32
InjectionAndTrafficIopCs0_Cs4(
  IN UINT32 IpIndex,
  IN UINT32 InjectionPayload
  )
{
  UINT32 Data32;
  UINT8 Data8;
  UINTN PchDeviceBaseAddress;

  DEBUG ((DEBUG_INFO, "InjectionAndTrafficIopCs0_Cs4\n"));
  //debug
  Data32 = MmioRead32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value is 0x%x\n", Data32));
  FuSaMsrWrite (IpIndex, InjectionPayload);
  PchD31F0BaseAddressGet (&PchDeviceBaseAddress);
  PciSegmentWrite32 (PchDeviceBaseAddress, 0U);
  Data32 = PciSegmentRead32 (PchDeviceBaseAddress);
  Data8 = IoRead8 (0x80);
  Data32 = MmioRead32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value after injection is 0x%x\n", Data32));
  //clear the status if any
  MmioWrite32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH, Data32);

  Data32 = FuSaMsrRead (IpIndex);
  DEBUG ((DEBUG_INFO, "First read back of control is 0x%x\n", Data32));
  return Data32;
}

/**
  Error injection and corresponding traffic routine for IO Port
  CTC other than Cs0 (IosfUpData) and Cs4 (Rch)

  @param[in] IpIndex - IP index vale for MSR MSR_PERRINJ_AT_IP
  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL

  @return injection control value after traffic generation
**/
STATIC
UINT32
InjectionAndTrafficIopGraphic(
  IN UINT32 IpIndex,
  IN UINT32 InjectionPayload
  )
{
  UINT32  Data32;
  UINTN   Address;

  DEBUG ((DEBUG_INFO, "InjectionAndTrafficIopGtt\n"));
  //debug
  Data32 = MmioRead32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value is 0x%x\n", Data32));
  FuSaMsrWrite (IpIndex, InjectionPayload);

  for (UINT32 i = 0; i < 0x80; i=i+4) {
    Address = mGmAdr + 0xFFF000U + i;
    Data32 = MmioRead32 (Address);
    MmioWrite32 (Address, Data32 ); //for CMI_UP traffic
  }

  Data32 = MmioRead32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value after injection is 0x%x\n", Data32));
  //clear the status if any
  MmioWrite32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH, Data32);

  Data32 = FuSaMsrRead (IpIndex);
  DEBUG ((DEBUG_INFO, "First read back of control is 0x%x\n", Data32));
  return Data32;
}

/**
  Error injection and corresponding traffic routine for IO Port
  CTC for MSG FIFO (using MSI thru HPET)

  @param[in] IpIndex - IP index vale for MSR MSR_PERRINJ_AT_IP
  @param[in] InjectionPayload - Payload value for MSR MSR_PERRINJ_CTRL

  @return injection control value after traffic generation
**/
STATIC
UINT32
InjectionAndTrafficIopHpet(
  IN UINT32 IpIndex,
  IN UINT32 InjectionPayload
  )
{
  UINT32  Data32;
  UINT64  Data64;

  DEBUG ((DEBUG_INFO, "InjectionAndTrafficIopHpet\n"));
  //debug
  Data32 = MmioRead32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value is 0x%x\n", Data32));
  FuSaMsrWrite (IpIndex, InjectionPayload);

  Data32 = MmioRead32(mHpetBase + 0xF0); //read main counter
  Data32 += 24000; //cycle count ~ 1ms
  MmioWrite32(mHpetBase + 0x128, Data32);
  MmioWrite32(mHpetBase + 0x134, 0xFEE10000); //time1 interrupt setting, point to non-exisint CPU APIC to avoid interrupt
  MmioWrite32(mHpetBase + 0x130, 0x140); //dummy vector not going to be triggered
  MmioWrite32(mHpetBase + 0x120, 0x0000C034U); //enable timer 1
  MmioWrite32(mHpetBase + 0x124, 0x00F00000U);
  Data64 = MmioRead64(mHpetBase + 0xF0); //read main counter
  while (MmioRead64(mHpetBase + 0xF0) < Data64 + 48000U); //poll 2x24000 for the MSI to happen

  Data32 = MmioRead32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH);
  DEBUG ((DEBUG_INFO, "R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH value after injection is 0x%x\n", Data32));
  //clear the status if any
  MmioWrite32 (mMchBase + R_SA_IOP_FUSA_PARITY_ERR_LOG_0_0_0_MCHBAR_IMPH, Data32);

  Data32 = FuSaMsrRead (IpIndex);
  DEBUG ((DEBUG_INFO, "First read back of control is 0x%x\n", Data32));
  return Data32;
}

///A lookup array for getting the injection permutation supported by different IOP check sites
STATIC UINT8 CONST mIopFusaPerrInjCapabilities[12][3] = {
    // 9 8 7:0  <- bit order [9] byte enable. [8] address. [7:0] data.
    //| | |   ||site | purpose
    // v v v   ||----------------------------------------------------------
    {0, 1, 3},    // 0    | data 1:0 - Upstream traffic from a PSF IP
    {0, 0, 0xF},  // 1    | data 3:0 - when wrc is enabled. do not use site.
    {0, 0, 0xF},  // 2    | WRC data 7:0 Downstram completion from Memory. HW only support 3:0
    {0, 0, 0},    // 3    | Not used.  see IOP HAS for details.
    {1, 1, 0},    // 4    | Normal IO traffic.
    {0, 1, 0},    // 5    | Display Traffic - Current code is not supporting this due to unknown traffic recipe
    {0, 1, 0},    // 6    | non VC1 transactions
    {0, 0, 1},    // 7    | data 0 - Upstream Write data from a PSF IP
    {0, 0, 1},    // 8    | data 0 - P2P - Current code drops the support due to external test card requirement
    {0, 0, 1},    // 9    | data 0 - MSI
    {0, 1, 0},    // 10   | data 0 - P2P - Current code drops the support due to external test card requirement
    {0, 0, 1}     // 11   | data 0 - Upstream Completion traffic from a PSF IP
};

/**
  Generic CTC routine for Core IDI Data

  @param[in] IpIndex - Ip index for MSR MSR_PERRINJ_AT_IP (check site)
  @param[in] InjectionAndTraffic - the matching injection and
        traffic routine for the check site
  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortGenericCtc (
  IN UINT8                         IpIndex,
  IN FUSA_INJECTION_TRAFFIC_FUNC   InjectionAndTraffic,
  IN OUT FUSA_TEST_RESULT          *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus = FusaNoError;
  STATIC UINT8    CheckNum = 0;
  CONST UINT8     *InjectionVectors;
  UINT32          InjectionBits;
  UINT8           BitIndex = 0;

  DEBUG ((DEBUG_INFO, "IoPortGenericCtc\n"));
  ASSERT (IpIndex >= IOP_CS0_IOSF_UP && IpIndex <=IOP_CS11_UP_COMPLETION);

  InjectionVectors = mIopFusaPerrInjCapabilities[IpIndex-IOP_CS0_IOSF_UP];
  InjectionBits = (InjectionVectors[0]<<9) | (InjectionVectors[1]<<8) | InjectionVectors[2];

  while (InjectionBits != 0)
  {
    if (InjectionBits & BIT0)
    {
      LibStatus |= FusaMsrCtcTestGeneric (
                     IpIndex,
                     1U << BitIndex,
                     InjectionAndTraffic,
                     MCA_IOP,
                     MCACOD_MASK | MSCOD_MASK,
                     0x10010E0BULL,
                     CheckNum,
                     pFusaTestResult,
                     TRUE
                     );

      CheckNum++;
    }

    BitIndex++;
    InjectionBits = InjectionBits>>1;
  }

  return LibStatus;
}

/**
  CTC routine for IO Port Iosf Upstream

  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortIosfUpstreamCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  DEBUG ((DEBUG_INFO, "%s\n", __FUNCTION__));

  MmioOr32 (mMchBase + R_SA_HCTL0_0_0_0_MCHBAR_IMPH, BIT20); //Disable WRC
  MmioWrite32(mMchBase + R_SA_WRC_VC_ENABLE, 0U); //disable all VC to WRC

  LibStatus = IoPortGenericCtc (
                IOP_CS0_IOSF_UP,
                InjectionAndTrafficIopCs0_Cs4,
                pFusaTestResult
                );

  return LibStatus;
}

/**
  CTC routine for IO Port Cmi

  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortCmiCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  DEBUG ((DEBUG_INFO, "%s\n", __FUNCTION__));

  MmioOr32 (mMchBase + R_SA_HCTL0_0_0_0_MCHBAR_IMPH, BIT20); //Disable WRC
  MmioWrite32(mMchBase + R_SA_WRC_VC_ENABLE, 0U); //disable all VC to WRC

  LibStatus = IoPortGenericCtc (
                IOP_CS1_CMI,
                InjectionAndTrafficIopGraphic,
                pFusaTestResult
                );

  return LibStatus;
}

/**
  CTC routine for IO Port Cii

  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortCiiCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  DEBUG ((DEBUG_INFO, "%s\n", __FUNCTION__));

  MmioAnd32 (mMchBase + R_SA_HCTL0_0_0_0_MCHBAR_IMPH, (UINT32) ~BIT20); //Eanble WRC
  MmioWrite32(mMchBase + R_SA_WRC_VC_ENABLE, 0xFFFFFFFFU); //enable all VC to WRC

  LibStatus = IoPortGenericCtc (
                IOP_CS2_CII,
                InjectionAndTrafficIopGraphic,
                pFusaTestResult
                );

  return LibStatus;
}

/**
  CTC routine for IO Port Rch

  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortRchCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  DEBUG ((DEBUG_INFO, "%s\n", __FUNCTION__));

  MmioOr32 (mMchBase + R_SA_HCTL0_0_0_0_MCHBAR_IMPH, BIT20); //Disable WRC
  MmioWrite32(mMchBase + R_SA_WRC_VC_ENABLE, 0U); //disable all VC to WRC

  LibStatus = IoPortGenericCtc (
                IOP_CS4_RCH,
                InjectionAndTrafficIopCs0_Cs4,
                pFusaTestResult
                );

  return LibStatus;
}

/**
  CTC routine for IO Port Iotrk

  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortIotrkCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  DEBUG ((DEBUG_INFO, "%s\n", __FUNCTION__));

  MmioOr32 (mMchBase + R_SA_HCTL0_0_0_0_MCHBAR_IMPH, BIT20); //Disable WRC
  MmioWrite32(mMchBase + R_SA_WRC_VC_ENABLE, 0U); //disable all VC to WRC

  LibStatus = IoPortGenericCtc (
                IOP_CS6_IOTRK,
                InjectionAndTrafficIopGraphic,
                pFusaTestResult
                );

  return LibStatus;
}

/**
  CTC routine for IO Port Cmi Up

  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortCmiUpCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  DEBUG ((DEBUG_INFO, "%s\n", __FUNCTION__));

  MmioOr32 (mMchBase + R_SA_HCTL0_0_0_0_MCHBAR_IMPH, BIT20); //Disable WRC
  MmioWrite32(mMchBase + R_SA_WRC_VC_ENABLE, 0U); //disable all VC to WRC

  LibStatus = IoPortGenericCtc (
                IOP_CS7_CMI_UP,
                InjectionAndTrafficIopGraphic,
                pFusaTestResult
                );

  return LibStatus;
}

/**
  CTC routine for IO Port Msg Fifo

  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortMsgFifoCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  DEBUG ((DEBUG_INFO, "%s\n", __FUNCTION__));

  MmioOr32 (mMchBase + R_SA_HCTL0_0_0_0_MCHBAR_IMPH, BIT20); //Disable WRC
  MmioWrite32(mMchBase + R_SA_WRC_VC_ENABLE, 0U); //disable all VC to WRC

  LibStatus = IoPortGenericCtc (
                IOP_CS9_MSG_FIFO,
                InjectionAndTrafficIopHpet,
                pFusaTestResult
                );

  return LibStatus;
}

/**
  CTC routine for IO Port Upstream Data Completion

  @param[in,out] pFusaTestResult - pointer to the test result
        buffer for result to be updated

  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
IoPortUpstreamCompletionCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  DEBUG ((DEBUG_INFO, "%s\n", __FUNCTION__));

  MmioOr32 (mMchBase + R_SA_HCTL0_0_0_0_MCHBAR_IMPH, BIT20); //Disable WRC
  MmioWrite32(mMchBase + R_SA_WRC_VC_ENABLE, 0U); //disable all VC to WRC

  LibStatus = IoPortGenericCtc (
                IOP_CS11_UP_COMPLETION,
                InjectionAndTrafficIopHpet,
                pFusaTestResult
                );

  return LibStatus;
}

/**
  IO Port Parity Error E2E CTC internal main routine

  @param[in,out] pFusaTestResult - test result buffer
       where the test result to be updated
**/
STATIC
FUSA_LIB_STATUS
IoPortCtc (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;

  LibStatus = IoPortIosfUpstreamCtc (pFusaTestResult);
  ASSERT (LibStatus == FusaNoError);
  LibStatus |= IoPortCmiCtc (pFusaTestResult);
  ASSERT (LibStatus == FusaNoError);
  LibStatus |= IoPortCiiCtc (pFusaTestResult);
  ASSERT (LibStatus == FusaNoError);
  LibStatus |= IoPortRchCtc (pFusaTestResult);
  ASSERT (LibStatus == FusaNoError);
  LibStatus |= IoPortIotrkCtc (pFusaTestResult);
  ASSERT (LibStatus == FusaNoError);
  LibStatus |= IoPortCmiUpCtc (pFusaTestResult);
  ASSERT (LibStatus == FusaNoError);
  LibStatus |= IoPortMsgFifoCtc (pFusaTestResult);
  ASSERT (LibStatus == FusaNoError);
  LibStatus |= IoPortUpstreamCompletionCtc (pFusaTestResult);
  ASSERT (LibStatus == FusaNoError);

  return LibStatus;
}

/**
  Perform IO Port Parity Error E2E CTC.
  The test targets IO Port multiple interfaces.

  @note The test assumes MchBar, GmBar and HpetBase have been
        set up with 32 bit addressing and being enabled.

  @param[out] pFusaTestResult - pointer to test result
                         buffer for test FusaTestNumIop

**/
VOID
FspDxCheckIoPort (
  OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT32 Data32;

  DEBUG ((DEBUG_INFO, "FspDxCheckIoPort\n"));

  LibStatus = MchBarAddressGet (&mMchBase);
  LibStatus |= GmBarAddressGet (&mGmAdr);
  mHpetBase = PcdGet32 (PcdSiHpetBaseAddress);
  if (LibStatus == FusaNoError
      && 0x8086A701U == MmioRead32(mHpetBase)) //sanity check on mHpetBase
    {
    //Read and verify that FUSA_PARITY_EN_OVRD_0_0_0_MCHBAR_IMPH = 0x0;
    //i.e. no parity checker over-rides are set (all bits in bitmap clear)
    //The verifying step is simply a development debug flow, no special code step need to be taken because if it fails,
    //the test result will be shown as fail. It can even be used by the validation team to test this CTC feature.
    Data32 = MmioRead32 (mMchBase + R_SA_IOP_FUSA_PARITY_EN_OVRD_0_0_0_MCHBAR_IMPH);
    DEBUG ((DEBUG_INFO, "R_SA_IOP_FUSA_PARITY_EN_OVRD_0_0_0_MCHBAR_IMPH value is 0x%x\n", Data32));

    //Enable FuSa error handling flows in IOP (vs. legacy)
    //Set FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH[MCA_REPORTING_EN] =1
    MmioOr32(
      mMchBase + R_SA_MCHBAR_FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH,
      B_SA_MCHBAR_FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH_MCA_REPORTING_EN
      );

    LibStatus = FusaTestAndReporting (
                  IoPortCtc,
                  FusaTestNumIop,
                  17U,
                  pFusaTestResult
                  );
    ASSERT (LibStatus == FusaNoError);
    DumpResults (pFusaTestResult);
  } else {
    LibStatus = FusaTestAndReporting (
                  NULL,
                  FusaTestNumIop,
                  17U,
                  pFusaTestResult
                  );
    ASSERT (LibStatus == FusaNoError);
    ASSERT (FALSE);
  }
}




