/** @file
  This file provides Fusa OPI link Parity E2E CTC
  implementation

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/PciSegmentLib.h>

#include "PeiFusaPrivateLibInternal.h"
#include "PeiFusaResultReportingLib.h"

///
/// Fusa Error Injection Configuration Registers
///FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR
#define R_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR                             0x500U
//default is 1, good to check if it is 1
#define B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_PARITY_GENERATE_ENABLE      BIT31
//When set to 1, enables IOSF parity checking on transactions received from PSF.
#define B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_PARITY_CHECK_ENABLE         BIT30
//When set to 1, enables the parity injection logic to force parity error.
#define B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_DATA_PARITY_INJECT          BIT28
//configuration bit to control data parity error fatality. This controls data parity error fatality when transactions
//target OPI's local registers. When set to 1, data parity errors are reported as fatal. Default is NON_FATAL.
#define B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_DATA_PARITY_FATAL_OPI_CFG   BIT27
//configuration bit to control data parity error fatality. This controls data parity error fatality when transactions
//target PCH. When set to 1, data parity errors are reported as fatal. Default is NON_FATAL.

#define B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_DATA_PARITY_FATAL           BIT26
#define R_SA_DMIBAR_FUSA_E2EPARITY_STS_0_0_0_DMIBAR                              0x514U
#define B_SA_DMIBAR_FUSA_E2EPARITY_STS_0_0_0_DMIBAR_CMD_PARITY_ERROR             BIT1
#define B_SA_DMIBAR_FUSA_E2EPARITY_STS_0_0_0_DMIBAR_DATA_PARITY_ERROR            BIT0

STATIC  UINTN           mDmiBase;
STATIC  UINTN           mMchBase;

/**
  Traffic for triggering OPI Link Data path
**/
STATIC
VOID
TrafficOpiLinkData (
   VOID
   )
{
  UINTN            EspiPciBase;
  PchD31F0BaseAddressGet (&EspiPciBase);
  PciSegmentWrite32 (EspiPciBase + 0U, 0xdeadbeef);//writing to readonly region is ok. PCH always have 0:31:0 device
}

/**
  OPI Link Data Parity Error E2E CTC internal main routine

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
OpiLinkDataParityCtc (
   IN OUT FUSA_TEST_RESULT *pFusaTestResult
   )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           McaBankNum = MCA_IOP;
  UINT64          McaStatusMask = MCACOD_MASK | MSCOD_MASK;
  UINT64          ExpectedMcaStatus = 0x03010E0BULL;
  UINT32          Data32;
  UINT32          Count;
  UINT8           CheckResult;


  DEBUG ((DEBUG_INFO, "OpiLinkDataParityCtc\n"));

  //effectively disable all MCA reporting because target 0xFF does not exist,
  //it does not prevent the MCA status from reflecting the error routed to MCA
  McaReportingEnable (0xFF);
  McaBankInfoClear (McaBankNum);

  //enable OPI Link parity injection
  MmioOr32 (
    mDmiBase+ R_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR,
    B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_DATA_PARITY_INJECT
    );
  TrafficOpiLinkData ();
  //print the status first, just incase the DEBUG print itself triggr the injection
  Data32 = MmioRead32 (mDmiBase+ R_SA_DMIBAR_FUSA_E2EPARITY_STS_0_0_0_DMIBAR);
  DEBUG ((DEBUG_INFO, "FUSA_E2EPARITY_STS_0_0_0_DMIBAR value after injection is 0x%x\n", Data32));
  MmioWrite32 (mDmiBase+ R_SA_DMIBAR_FUSA_E2EPARITY_STS_0_0_0_DMIBAR, 0U); //write 0 to clear

  Data32 = MmioRead32 (mDmiBase+ R_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR);
  Count = 1;
  while ((Data32 & B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_DATA_PARITY_INJECT) != 0
         && Count <= 10U)
  {
    Data32 = MmioRead32 (mDmiBase+ R_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR);
    Count++;
  }
  DEBUG ((DEBUG_INFO, "Repeat %d times reading the contol, its value is 0x%x\n", Count, Data32));

  CheckResult = McaBankStatusCheck (McaBankNum, McaStatusMask, ExpectedMcaStatus);
  if (FUSA_TEST_FAIL == CheckResult) {
    //remove injection here
    MmioAnd32 (
      mDmiBase+ R_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR,
      (UINT32) ~B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_DATA_PARITY_INJECT
      );
  }

  LibStatus = UpdateResults (CheckResult, 0U, pFusaTestResult); //checkNum = 0

  McaBankInfoDump (McaBankNum);
  McaBankInfoClear (McaBankNum);

  return LibStatus;
}

/**
  Perform OPI Link Data Parity Error E2E CTC.
  The test targets OPI Link Data path.

  @note The test assumes MchBar and DmiBar have been set up
        with 32 bit addressing and being enabled.

  @param[out] pFusaTestResult - pointer to test result
                         buffer for test
                         FusaTestNumOpiLinkIosfData

**/
VOID
FspDxCheckOpiLink (
  OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT32          Data32;

  DEBUG ((DEBUG_INFO, "FspDxCheckOpilink\n"));
  LibStatus = DmiBarAddressGet (&mDmiBase);
  LibStatus |= MchBarAddressGet (&mMchBase);

  if (FusaNoError == LibStatus) {
    //enable the IOP error reporting to MCA
    MmioOr32(
      mMchBase + R_SA_MCHBAR_FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH,
      B_SA_MCHBAR_FUSA_MCA_REPORTING_EN_0_0_0_MCHBAR_IMPH_MCA_REPORTING_EN
      );

    //enable OPI Link parity check
    Data32 = B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_PARITY_CHECK_ENABLE
      | B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_DATA_PARITY_FATAL_OPI_CFG
      | B_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR_DATA_PARITY_FATAL;
    MmioOr32 (
      mDmiBase+ R_SA_DMIBAR_FUSA_E2EPARITY_CTRL_0_0_0_DMIBAR,
      Data32
      );

    LibStatus = FusaTestAndReporting (
                  OpiLinkDataParityCtc,
                  FusaTestNumOpiLinkIosfData,
                  1U,
                  pFusaTestResult
                  );
    ASSERT (LibStatus == FusaNoError);
    DumpResults (pFusaTestResult);
  } else {
    LibStatus = FusaTestAndReporting (
                  NULL,
                  FusaTestNumOpiLinkIosfData,
                  1U,
                  pFusaTestResult
                  );
    ASSERT (LibStatus == FusaNoError);
    ASSERT(FALSE);
  }
}


