/** @file
  This file provides Fusa Core Array Mbist implementation

@copyright
  INTEL CONFIDENTIAL Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <PiPei.h>
#include <Ppi/MpServices.h>
#include <Library/PeiServicesLib.h>
#include <Library/LocalApicLib.h>

#include "PeiFusaPrivateLibInternal.h"
#include "PeiFusaResultReportingLib.h"
#include "PeiFusaE2eCtcLibInternal.h"

extern EFI_GUID                 gEfiPeiMpServicesPpiGuid;

/// Structure of the parameter passed into the Core Array test routine
typedef struct
{
  FUSA_TEST_RESULT *pFusaTestResult;
  UINT32           CoreNumber;
  UINT8            TestResult[2];
} CORE_ARRAY_TEST_PARAM;

/**
  Core Array MBIST internal main routine

  @param[in,out] pCoreArrayTestParam - contain result buffer
       where the test result to be updated to beside core number
       information
**/
VOID
FspDxCheckCoreMbistInternal (
  IN CORE_ARRAY_TEST_PARAM *pCoreArrayTestParam
  )
{
  UINT32 CoreNumber;
  UINT32 CurrentThreadApicId = GetInitialApicId();

  CoreNumber = pCoreArrayTestParam->CoreNumber;

  if ((CurrentThreadApicId >> 1) == CoreNumber)
  {
    UINT8 LocalTestResult;
    _asm
    {
      //mask of all '1 is also good since there is no reserved bit checking
      mov eax, 0xffffffff
      xor edx, edx
      mov ecx, 0x105
      MBIST_LOOP:
      //activate BIST
      wrmsr
      //read results
      rdmsr
      //check bit 31 for fail indication
      btr edx, 31
      jb BIST_FAILED
      //check that EAX is all zero, meaning we are done
      or eax, eax
      je BIST_PASSED
      jmp MBIST_LOOP

      BIST_FAILED:
      mov LocalTestResult, FUSA_TEST_FAIL
      jmp TEST_END

      BIST_PASSED:
      mov LocalTestResult, FUSA_TEST_PASS

      TEST_END:
    }

    pCoreArrayTestParam->TestResult[CurrentThreadApicId%2] = LocalTestResult;
  }
}

/**
  Perform MBIST Test on CPU array

  @param[out] pFusaTestResult - pointer to test result
                         buffer for test from
                         FusaTestNumCpu0Mbist to
                         FusaTestNumCpu7Mbist
  @param[in] PeiServices    Pointer to PEI Services Table
**/
VOID
FspDxCheckCoreArrayMbist (
  OUT FUSA_TEST_RESULT *pFusaTestResult,
  IN  CONST EFI_PEI_SERVICES  **PeiServices
  )
{
  UINT32 CoreIndex;
  CORE_ARRAY_TEST_PARAM CoreArrayTestParam;
  CONST UINT32          *UniqueCoreList = UniqueCoreListGet ();
  UINT32                BspApicId;
  FUSA_LIB_STATUS       LibStatus;
  UINT8                 CheckResult;
  EFI_STATUS            Status;
  CONST EFI_PEI_SERVICES   **mPeiServices;
  EFI_PEI_MP_SERVICES_PPI  *mMpServicesPpi = NULL;

  DEBUG ((DEBUG_INFO, "FspDxCheckCoreArrayMbist\n"));

  BspApicId = GetInitialApicId ();
  mPeiServices = PeiServices;
  // Locate CpuMpCpu MpService Ppi
  Status = PeiServicesLocatePpi (
             &gEfiPeiMpServicesPpiGuid,
             0,
             NULL,
             (VOID **) &mMpServicesPpi
             );
  ASSERT_EFI_ERROR (Status);

  //intialize the test result buffer as some test may not run because of the device not being available
  //do this differently comparing to the CoreIdi test as we do not use the FusaTestAndReporting for
  //running the test, but simply set up the TestNum and TotalNumberOfChecks
  for (CoreIndex = 0; CoreIndex < MAX_TGL_CORE_COUNT; CoreIndex++) {
    LibStatus = FusaTestAndReporting (
                  NULL,
                  FusaTestNumCpu0Mbist + CoreIndex,
                  1U,
                  &(pFusaTestResult[CoreIndex])
                  );
    ASSERT(LibStatus == FusaNoError);
  }

  for (CoreIndex = 0; CoreIndex < MAX_TGL_CORE_COUNT; CoreIndex++)
  {
    if (UniqueCoreList[CoreIndex] > 0)
    {
      DEBUG ((DEBUG_INFO, "UniqueCoreList[%d] = %d\n", CoreIndex, UniqueCoreList[CoreIndex] ));
      CoreArrayTestParam.CoreNumber = CoreIndex;
      CoreArrayTestParam.pFusaTestResult = &(pFusaTestResult[CoreArrayTestParam.CoreNumber]);
      CoreArrayTestParam.TestResult[0] = FUSA_TEST_NOTRUN;
      CoreArrayTestParam.TestResult[1] = FUSA_TEST_NOTRUN;
      mMpServicesPpi->StartupAllAPs (
                           mPeiServices,
                           mMpServicesPpi,
                           (EFI_AP_PROCEDURE) FspDxCheckCoreMbistInternal,
                           FALSE, //all AP thread execute simultaneously
                           0,
                           &CoreArrayTestParam
                           );

      DEBUG ((DEBUG_INFO, "Core %d Thread 0 Mbist result = %d\n", CoreIndex, CoreArrayTestParam.TestResult[0]));
      DEBUG ((DEBUG_INFO, "Core %d Thread 1 Mbist result = %d\n", CoreIndex, CoreArrayTestParam.TestResult[1]));

      //it is possible HT is not avaiable, thus either TestResults is considered as the final CheckResult
      if (FUSA_TEST_FAIL == CoreArrayTestParam.TestResult[0]
          || FUSA_TEST_FAIL == CoreArrayTestParam.TestResult[1])
      {
        CheckResult = FUSA_TEST_FAIL;
      }
      else if (FUSA_TEST_PASS == CoreArrayTestParam.TestResult[0]
          || FUSA_TEST_PASS == CoreArrayTestParam.TestResult[1])
      {
        CheckResult = FUSA_TEST_PASS;
      }
      else
      {
        CheckResult = FUSA_TEST_NOTRUN;
      }
      LibStatus = UpdateResults(CheckResult, 0U, CoreArrayTestParam.pFusaTestResult);
      LibStatus |= GenerateCrc (CoreArrayTestParam.pFusaTestResult);
      ASSERT(LibStatus == FusaNoError);
      DumpResults (CoreArrayTestParam.pFusaTestResult);
    }
  }

  CoreArrayTestParam.CoreNumber = BspApicId >> 1;
  CoreArrayTestParam.pFusaTestResult = &(pFusaTestResult[CoreArrayTestParam.CoreNumber]);
  CoreArrayTestParam.TestResult[0] = FUSA_TEST_NOTRUN;
  CoreArrayTestParam.TestResult[1] = FUSA_TEST_NOTRUN;
  FspDxCheckCoreMbistInternal(&CoreArrayTestParam);

  DEBUG ((DEBUG_INFO, "Core %d Thread 0 Mbist result = %d\n", BspApicId>>1, CoreArrayTestParam.TestResult[0]));
  DEBUG ((DEBUG_INFO, "Core %d Thread 1 Mbist result = %d\n", BspApicId>>1, CoreArrayTestParam.TestResult[1]));
  //Not making assumption on the BSP being HT0 or HT1 here.
  if (FUSA_TEST_FAIL == CoreArrayTestParam.TestResult[0]
      || FUSA_TEST_FAIL == CoreArrayTestParam.TestResult[1])
  {
    CheckResult = FUSA_TEST_FAIL;
  }
  else if (FUSA_TEST_PASS == CoreArrayTestParam.TestResult[0]
      || FUSA_TEST_PASS == CoreArrayTestParam.TestResult[1])
  {
    CheckResult = FUSA_TEST_PASS;
  }
  else
  {
    CheckResult = FUSA_TEST_NOTRUN;
  }
  LibStatus = UpdateResults(CheckResult, 0U, CoreArrayTestParam.pFusaTestResult);
  LibStatus |= GenerateCrc (CoreArrayTestParam.pFusaTestResult);
  ASSERT(LibStatus == FusaNoError);
  DumpResults (CoreArrayTestParam.pFusaTestResult);
}

