/** @file
  This file provides Fusa Memory Mbist reporting implementation.
  The real test happen within MRC

@copyright
  INTEL CONFIDENTIAL Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/TimerLib.h>
#include <Register/SaRegsHostBridge.h>
#include <Register/IgdRegs.h>

#include "PeiFusaPrivateLibInternal.h"
#include "PeiFusaResultReportingLib.h"

STATIC UINTN             mMchBase;

#define MC0_IBECC_MBIST_POST_CTRL_WDB_REG 0x0D490U
#define MC0_IBECC_MBIST_POST_CTRL_RDB_REG 0x0D494U
#define MC1_IBECC_MBIST_POST_CTRL_WDB_REG 0x1D490U
#define MC1_IBECC_MBIST_POST_CTRL_RDB_REG 0x1D494U
#define MC0_CH0_CR_WDB_MBIST_0_REG        0x42E8U
#define MC0_CH0_CR_WDB_MBIST_1_REG        0x42ECU
#define MC0_CH1_CR_WDB_MBIST_0_REG        0x46E8U
#define MC0_CH1_CR_WDB_MBIST_1_REG        0x46ECU
#define MC1_CH0_CR_WDB_MBIST_0_REG        0x142E8U
#define MC1_CH0_CR_WDB_MBIST_1_REG        0x142ECU
#define MC1_CH1_CR_WDB_MBIST_0_REG        0x146E8U
#define MC1_CH1_CR_WDB_MBIST_1_REG        0x146ECU
#define MC0_CH0_CR_RDB_MBIST_REG          0x042F8U
#define MC0_CH1_CR_RDB_MBIST_REG          0x046F8U
#define MC1_CH0_CR_RDB_MBIST_REG          0x142F8U
#define MC1_CH1_CR_RDB_MBIST_REG          0x146F8U

typedef union {
  struct {
    UINT32  Pass      : 1;
    UINT32  Completed : 1;
    UINT32  Reserved  : 29;
    UINT32  RunBusy   : 1;
  } Field;
  UINT32 Data32;
} MBIST_REG_STRUCT;

UINT8
MbistResultExtract (
  IN UINT32 RegisterOffset
  )
{
  MBIST_REG_STRUCT Reg;
  UINT8 CheckResult;
  Reg.Data32 = MmioRead32 (mMchBase + RegisterOffset);
  DEBUG ((DEBUG_INFO, "Mbist register offset 0x%x, value = 0x%x\n", RegisterOffset, Reg.Data32));

  //imply the register is not accessible
  if (0xFFFFFFFFU == Reg.Data32)
  {
    CheckResult = FUSA_TEST_NOTRUN;
  }
  else if (1 == Reg.Field.Completed)
  {
    if (1 == Reg.Field.Pass)
    {
      CheckResult = FUSA_TEST_PASS;
    }
    else
    {
      CheckResult = FUSA_TEST_FAIL;
    }
  }
  else
  {
    CheckResult = FUSA_TEST_NOTRUN;
  }

  return CheckResult;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc0MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC0_CH0_CR_RDB_MBIST_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  CheckResult = MbistResultExtract(MC0_CH1_CR_RDB_MBIST_REG);
  LibStatus |= UpdateResults(CheckResult, 1U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc1MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC1_CH0_CR_RDB_MBIST_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  CheckResult = MbistResultExtract(MC1_CH1_CR_RDB_MBIST_REG);
  LibStatus |= UpdateResults(CheckResult, 1U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc0Ch0MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC0_CH0_CR_WDB_MBIST_0_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc0Ch1MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC0_CH0_CR_WDB_MBIST_1_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc0Ch2MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC0_CH1_CR_WDB_MBIST_0_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc0Ch3MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC0_CH1_CR_WDB_MBIST_1_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc1Ch0MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC1_CH0_CR_WDB_MBIST_0_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc1Ch1MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC1_CH0_CR_WDB_MBIST_1_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc1Ch2MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC1_CH1_CR_WDB_MBIST_0_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Mc1Ch3MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC1_CH1_CR_WDB_MBIST_1_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Ibecc0MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC0_IBECC_MBIST_POST_CTRL_WDB_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  CheckResult = MbistResultExtract(MC0_IBECC_MBIST_POST_CTRL_RDB_REG);
  LibStatus |= UpdateResults(CheckResult, 1U, pFusaTestResult);
  return LibStatus;
}

/**
  Perform the related MBIST result extraction

  @param[in,out] pFusaTestResult - result buffer where the test
       result to be updated to
  @return FUSA_LIB_STATUS
**/
STATIC
FUSA_LIB_STATUS
Ibecc1MbistResultExtract (
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS LibStatus;
  UINT8           CheckResult;
  CheckResult = MbistResultExtract(MC1_IBECC_MBIST_POST_CTRL_WDB_REG);
  LibStatus = UpdateResults(CheckResult, 0U, pFusaTestResult);
  CheckResult = MbistResultExtract(MC1_IBECC_MBIST_POST_CTRL_RDB_REG);
  LibStatus |= UpdateResults(CheckResult, 1U, pFusaTestResult);
  return LibStatus;
}

/**
  Extract Memory SubSystem MBIST result which resided within
  registers, where the MBIST has been performed within the MRC
  stage and yet for some reason not able to populate the result
  HOB at the point of time.

  @note The test assumes MchBar has been set up with 32 bit
        addressing and being enabled.

  @param[out] pFusaTestResult - pointer to test result
                         buffer for test from
                         FusaTestNumMc0Mbist to
                         FusaTestNumIbecc1Mbist
**/
VOID
FspDxCheckMemoryMbist (
  OUT FUSA_TEST_RESULT *pFusaTestResult
  )
{
  FUSA_LIB_STATUS   LibStatus;

  DEBUG ((DEBUG_INFO, "FspDxCheckMemoryMbist\n"));
  LibStatus = MchBarAddressGet (&mMchBase);
  if (LibStatus == FusaNoError) {
    LibStatus = FusaTestAndReporting (
                  Mc0MbistResultExtract,
                  FusaTestNumMc0Mbist,
                  2U,
                  &(pFusaTestResult[FusaTestNumMc0Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc1MbistResultExtract,
                  FusaTestNumMc1Mbist,
                  2U,
                  &(pFusaTestResult[FusaTestNumMc1Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc0Ch0MbistResultExtract,
                  FusaTestNumMc0Ch0Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc0Ch0Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc0Ch1MbistResultExtract,
                  FusaTestNumMc0Ch1Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc0Ch1Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc0Ch2MbistResultExtract,
                  FusaTestNumMc0Ch2Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc0Ch2Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc0Ch3MbistResultExtract,
                  FusaTestNumMc0Ch3Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc0Ch3Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc1Ch0MbistResultExtract,
                  FusaTestNumMc1Ch0Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc1Ch0Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc1Ch1MbistResultExtract,
                  FusaTestNumMc1Ch1Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc1Ch1Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc1Ch2MbistResultExtract,
                  FusaTestNumMc1Ch2Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc1Ch2Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Mc1Ch3MbistResultExtract,
                  FusaTestNumMc1Ch3Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc1Ch3Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Ibecc0MbistResultExtract,
                  FusaTestNumIbecc0Mbist,
                  2U,
                  &(pFusaTestResult[FusaTestNumIbecc0Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  Ibecc1MbistResultExtract,
                  FusaTestNumIbecc1Mbist,
                  2U,
                  &(pFusaTestResult[FusaTestNumIbecc1Mbist - FusaTestNumMc0Mbist])
                  );
    ASSERT (LibStatus == FusaNoError);
    for (UINT32 i = 0; i <=  FusaTestNumIbecc1Mbist - FusaTestNumMc0Mbist; i++)
    {
      DumpResults (&(pFusaTestResult[i]));
    }
  } else {
    LibStatus = FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc0Mbist,
                  2U,
                  &(pFusaTestResult[FusaTestNumMc0Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc1Mbist,
                  2U,
                  &(pFusaTestResult[FusaTestNumMc1Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc0Ch0Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc0Ch0Mbist - FusaTestNumMc0Mbist])
                  );

    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc0Ch1Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc0Ch1Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc0Ch2Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc0Ch2Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc0Ch3Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc0Ch3Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc1Ch0Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc1Ch0Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc1Ch1Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc1Ch1Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc1Ch2Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc1Ch2Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumMc1Ch3Mbist,
                  1U,
                  &(pFusaTestResult[FusaTestNumMc1Ch3Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumIbecc0Mbist,
                  2U,
                  &(pFusaTestResult[FusaTestNumIbecc0Mbist - FusaTestNumMc0Mbist])
                  );
    LibStatus |= FusaTestAndReporting (
                  NULL,
                  FusaTestNumIbecc1Mbist,
                  2U,
                  &(pFusaTestResult[FusaTestNumIbecc1Mbist - FusaTestNumMc0Mbist])
                  );
    ASSERT (LibStatus == FusaNoError);
    ASSERT (FALSE);
  }
}


