/** @file
  Header file for Fusa Result Reporting Lib functions.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _FUSA_RESULT_REPORTING_LIB_H_
#define _FUSA_RESULT_REPORTING_LIB_H_

#include <Uefi/UefiBaseType.h>
#include <Library/BaseLib.h>
#include <Library/PeiFusaE2eCtcLib.h>

/**
  Function pointer for the generic FUSA test
  @note Expect the pointed function to use the UpdateResults()
  to update the test result.
**/
typedef FUSA_LIB_STATUS (* const FUSA_TEST_FUNC)( FUSA_TEST_RESULT *pFusaTestResult );

/**
  Updates the reporting structure with check result.
  Will need to be called after every check is done.

  @note It is expected that for every test, InitializeResults()
        routine has called once before the first call to this
        routine.

  @param[in] TestCheckVal - result of executing a check
  @param[in] CheckNum - check number for subtest
  @param[in,out] pFusaTestResult - summarized reporting
  structure for the whole test

  @retval FusaNoError - if succeed
  @retval FusaInvalidParam - if pFusaTestResult is NULL
  @retval FusalInternalError - if CheckNum >=
          pFusaTestResult->TotalChecks or
          pFusaTestResult->CheckResults[CheckNum] has been used
          before or not been initialized
**/
FUSA_LIB_STATUS
UpdateResults(
  IN UINT8 TestCheckVal,
  IN UINT8 CheckNum,
  IN OUT FUSA_TEST_RESULT *pFusaTestResult
  );

/**
  Generates crc32 for reporting structure except for the last 4
  bytes. It uses IA instruction crc32 for the calculation which
  implies its polynomial is 0x11EDC6F41. It uses 0xffffffff as
  initial value. The final CRC32 value is not negated.

  @note it expects FUSA_TEST_RESULT structure size to be
        multiple of 4

  @param[in, out] pFusaTestResult - summarized reporting
  structure for the whole test

  @retval FusaNoError - if succeed
  @retval FusaInvalidParam - if pStlResult is NULL
**/
FUSA_LIB_STATUS
GenerateCrc(
  IN FUSA_TEST_RESULT * const pFusaTestResult
  );

/**
  Dump Fusa test result for debug purpose. Does nothing in
  release build.

  @param[in] pFusaTestResult - pointer to the result buffer to
        be dumped
**/
VOID
DumpResults(
  IN FUSA_TEST_RESULT *pFusaTestResult
  );

/**
  The FusaTestGeneric wraps the calling to stlTest with the
  generic Fusa test result reporting flow.

  @param[in] FusaTest - function pointer to the test
  @param[in] TestNum - test number assigned to the test
  @param[in] TotalNumberOfChecks - total number of checks for
        the test
  @param[in,out] pFusaTestResult - summarized reporting
    structure for the whole test

  @retval FusaNoError - if succeed
  @retval FusaInvalidParam - if pFusaTestResult is NULL
  @retval FusaInternalError - if error caused by internal
          implementation or defensive programming path taken

**/
FUSA_LIB_STATUS
FusaTestAndReporting (
  IN FUSA_TEST_FUNC fusaTest,
  IN FUSA_TEST_NUMBER testNum,
  IN UINT32 totalNumberOfChecks,
  IN FUSA_TEST_RESULT *pFusaTestResult
  );


#endif // _FUSA_RESULT_REPORTING_LIB_H_
