/** @file
  PEIM to initialize Early Display.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2015 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/PeiDisplayInitLib.h>
#include <Library/PeiGraphicsInitLib.h>
#include <Ppi/Graphics.h>
#include <Ppi/SiPolicy.h>
#include <Library/PostCodeLib.h>
#include <Register/IgdRegs.h>
#include <Library/FspErrorInfoLib.h>
#include <Guid/GraphicsInfoHob.h>

GRAPHICS_PEI_PREMEM_CONFIG             *mGtPreMemConfig;
GRAPHICS_PEI_CONFIG                    *mGtConfig;
EFI_GRAPHICS_OUTPUT_PROTOCOL_MODE      *mMode = NULL;

STATIC
EFI_PEI_NOTIFY_DESCRIPTOR  mPeiGraphicsNotifyList = {
  (EFI_PEI_PPI_DESCRIPTOR_NOTIFY_CALLBACK | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gEfiPeiGraphicsPpiGuid,
  PeiGraphicsNotifyCallback
};

EFI_PEI_PPI_DESCRIPTOR  mPeiGraphicsFramebufferReadyPpi = {
  (EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gPeiGraphicsFramebufferReadyPpiGuid,
  NULL
};


/**
  PeiDisplayInit: Initialize the IGD using GraphicsInitPeim and display logo in PEI

  @param[in] GtPreMemConfig  - GRAPHICS_PEI_PREMEM_PREMEM_CONFIG to access the GtPreMemConfig related information
  @param[in] GtConfig        - GRAPHICS_PEI_CONFIG to access the GtConfig related information

  @retval     EFI_STATUS
**/
EFI_STATUS
PeiDisplayInit (
  IN   GRAPHICS_PEI_PREMEM_CONFIG      *GtPreMemConfig,
  IN   GRAPHICS_PEI_CONFIG             *GtConfig
  )
{
  EFI_STATUS         Status;
  UINT64             PciCfgBase;

  Status = EFI_SUCCESS;

  DEBUG ((DEBUG_INFO, "PeiDisplayInit: Begin \n"));

  if (GtConfig->PeiGraphicsPeimInit == 0) {
    return EFI_SUCCESS;
  }
  if (GtConfig->GraphicsConfigPtr == NULL) {
    DEBUG ((DEBUG_INFO, "Bad VBT Pointer passed, Exiting PeiDisplayInit \n"));
    return EFI_SUCCESS;
  }

  mGtPreMemConfig = GtPreMemConfig;
  mGtConfig = GtConfig;
  PciCfgBase = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, 0);

  if ((PciSegmentRead16 (PciCfgBase + PCI_VENDOR_ID_OFFSET) != 0xFFFF)) {
    ///
    /// Register notify to on PeiGraphicsPpi
    ///
    Status = PeiServicesNotifyPpi (&mPeiGraphicsNotifyList);
  }

  return Status;
}

/**
  Run this function after SaPolicy PPI to be installed.

  @param[in]  PeiServices       General purpose services available to every PEIM.
  @param[in]  NotifyDescriptor  Notify that this module published.
  @param[in]  Ppi               PPI that was installed.

  @retval     EFI_SUCCESS       The function completed successfully.
**/
EFI_STATUS
EFIAPI
PeiGraphicsNotifyCallback (
  IN EFI_PEI_SERVICES           **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR  *NotifyDescriptor,
  IN VOID                       *Ppi
  )
{
  EFI_STATUS    Status;
  UINT64        PciCfgBase;
  LARGE_INTEGER GmAdrValue;

  DEBUG ((DEBUG_INFO, "PeiGraphicsNotifyCallback Entry\n"));

  PostCode (0xA04);
  PciCfgBase = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, 0);
  ///
  /// Initialize Gttmmadr and Gmadr
  ///
  PciSegmentWrite32 (PciCfgBase + R_SA_IGD_GTTMMADR, mGtPreMemConfig->GttMmAdr);

  GmAdrValue.Data = mGtPreMemConfig->GmAdr64;
  PciSegmentWrite32 (PciCfgBase + R_SA_IGD_GMADR, GmAdrValue.Data32.Low);
  PciSegmentWrite32 (PciCfgBase + R_SA_IGD_GMADR + 4, GmAdrValue.Data32.High);

  PciSegmentWrite32 (PciCfgBase + PCI_COMMAND_OFFSET, (UINT32) (EFI_PCI_COMMAND_BUS_MASTER | EFI_PCI_COMMAND_MEMORY_SPACE | EFI_PCI_COMMAND_IO_SPACE));

  DEBUG ((DEBUG_INFO, "CallPpiAndFillFrameBuffer\n"));
  PostCode (0xA05);

  Status = CallPpiAndFillFrameBuffer (mGtConfig);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_WARN, "Failed to Call Ppi and fill frame buffer:%r\n", Status));
  }
  DEBUG ((DEBUG_INFO, "PeiGraphicsNotifyCallback Exit\n"));
  PostCode (0xA0F);

  return Status;
}

/**
  CallPpiAndFillFrameBuffer: Call GraphicsInitPeim PeiGraphicsPpi to initalize display and get Mode info.
  Publish GraphicsInfoHob and call FillFrameBufferAndShowLogo

  @param[in] GtConfig  - GRAPHICS_PEI_CONFIG to access the GtConfig related information

  @retval    EFI_OUT_OF_RESOURCES - no enough reosurce for completing this task.
  @retval    EFI_STATUS - status code from calling several sub functions.
**/
EFI_STATUS
EFIAPI
CallPpiAndFillFrameBuffer (
  IN   GRAPHICS_PEI_CONFIG             *GtConfig
  )
{
  EFI_STATUS                       Status;
  EFI_PEI_GRAPHICS_PPI             *GfxInitPpi;
  EFI_PEI_GRAPHICS_INFO_HOB        *PlatformGraphicsOutput;
  EFI_PEI_GRAPHICS_DEVICE_INFO_HOB *GfxDeviceInfoHob;
  UINT64                           McD2BaseAddress;

  DEBUG ((DEBUG_INFO, "CallPpiAndFillFrameBuffer: Begin \n"));

  Status = EFI_SUCCESS;
  ///
  /// Locate GfxInit Ppi
  ///
  Status = PeiServicesLocatePpi (&gEfiPeiGraphicsPpiGuid, 0, NULL, (VOID *)&GfxInitPpi);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_WARN, "Failed to locate Ppi GraphicsPpiInit and GraphicsPpiGetMode. \n"));
    return Status;
  }

  ///
  /// Call PeiGraphicsPpi.GraphicsPpiInit to initilize the display
  ///
  DEBUG ((DEBUG_INFO, "GraphicsPpiInit Start\n"));
  PostCode (0xA06);
  Status = GfxInitPpi->GraphicsPpiInit (GtConfig->GraphicsConfigPtr);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_WARN, "GraphicsPpiInit failed. \n"));
    SendFspErrorInfo (
      gSaFspErrorTypeCallerId,
      gSaFspErrorTypePeiGopInit,
      Status
      );
    return Status;
  }

  mMode       = AllocateZeroPool (sizeof (EFI_GRAPHICS_OUTPUT_PROTOCOL_MODE));
  if (mMode == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  mMode->Info = AllocateZeroPool (sizeof (EFI_GRAPHICS_OUTPUT_MODE_INFORMATION));
  if (mMode->Info == NULL) {
    return EFI_OUT_OF_RESOURCES;
  }
  ///
  /// Call PeiGraphicsPpi.GraphicsPpiGetMode to get display resolution
  ///
  DEBUG ((DEBUG_INFO, "GraphicsPpiGetMode Start\n"));
  PostCode (0xA07);
  Status = GfxInitPpi->GraphicsPpiGetMode (mMode);
  if (EFI_ERROR (Status)) {
    DEBUG ((DEBUG_WARN, "GraphicsPpiGetMode failed. \n"));
    SendFspErrorInfo (
      gSaFspErrorTypeCallerId,
      gSaFspErrorTypePeiGopGetMode,
      Status
      );
    return Status;
  }

  ///
  /// Print mMode information recived from GraphicsPeim
  ///
  DEBUG ((DEBUG_INFO, "MaxMode:0x%x \n", mMode->MaxMode));
  DEBUG ((DEBUG_INFO, "Mode:0x%x \n", mMode->Mode));
  DEBUG ((DEBUG_INFO, "SizeOfInfo:0x%x \n", mMode->SizeOfInfo));
  DEBUG ((DEBUG_INFO, "FrameBufferBase:0x%x \n", mMode->FrameBufferBase));
  DEBUG ((DEBUG_INFO, "FrameBufferSize:0x%x \n", mMode->FrameBufferSize));
  DEBUG ((DEBUG_INFO, "Version:0x%x \n", mMode->Info->Version));
  DEBUG ((DEBUG_INFO, "HorizontalResolution:0x%x \n", mMode->Info->HorizontalResolution));
  DEBUG ((DEBUG_INFO, "VerticalResolution:0x%x \n", mMode->Info->VerticalResolution));
  DEBUG ((DEBUG_INFO, "PixelFormat:0x%x \n", mMode->Info->PixelFormat));
  DEBUG ((DEBUG_INFO, "PixelsPerScanLine:0x%x \n", mMode->Info->PixelsPerScanLine));

  ///
  /// Publish GraphicsInfoHob and GraphicsDeviceInfoHob to be used by platform code
  ///
  PlatformGraphicsOutput = BuildGuidHob (&gEfiGraphicsInfoHobGuid, sizeof (EFI_PEI_GRAPHICS_INFO_HOB));

  if (PlatformGraphicsOutput == NULL) {
    DEBUG ((DEBUG_WARN, "Failed to build GraphicsInfoHob. \n"));
    return EFI_OUT_OF_RESOURCES;
  }

  PlatformGraphicsOutput->GraphicsMode.Version              = mMode->Info->Version;
  PlatformGraphicsOutput->GraphicsMode.HorizontalResolution = mMode->Info->HorizontalResolution;
  PlatformGraphicsOutput->GraphicsMode.VerticalResolution   = mMode->Info->VerticalResolution;
  PlatformGraphicsOutput->GraphicsMode.PixelFormat          = mMode->Info->PixelFormat;
  PlatformGraphicsOutput->GraphicsMode.PixelInformation     = mMode->Info->PixelInformation;
  PlatformGraphicsOutput->GraphicsMode.PixelsPerScanLine    = mMode->Info->PixelsPerScanLine;
  PlatformGraphicsOutput->FrameBufferBase                   = mMode->FrameBufferBase;
  PlatformGraphicsOutput->FrameBufferSize                   = mMode->FrameBufferSize;

  GfxDeviceInfoHob = BuildGuidHob (&gEfiGraphicsDeviceInfoHobGuid, sizeof (EFI_PEI_GRAPHICS_DEVICE_INFO_HOB));
  if (GfxDeviceInfoHob == NULL) {
    DEBUG ((DEBUG_WARN, "Failed to build GfxDeviceInfoHob. \n"));
    return EFI_OUT_OF_RESOURCES;
  }

  McD2BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, 0);
  GfxDeviceInfoHob->VendorId = PciSegmentRead16 (McD2BaseAddress + PCI_VENDOR_ID_OFFSET);
  GfxDeviceInfoHob->DeviceId = PciSegmentRead16 (McD2BaseAddress + PCI_DEVICE_ID_OFFSET);
  GfxDeviceInfoHob->SubsystemVendorId = PciSegmentRead16 (McD2BaseAddress + PCI_SUBSYSTEM_VENDOR_ID_OFFSET);
  GfxDeviceInfoHob->SubsystemId = PciSegmentRead16 (McD2BaseAddress + PCI_SUBSYSTEM_ID_OFFSET);
  GfxDeviceInfoHob->RevisionId = PciSegmentRead8 (McD2BaseAddress + PCI_REVISION_ID_OFFSET);
  GfxDeviceInfoHob->BarIndex = 1; // Second BAR is for framebuffer in iGFX case.

  ///
  /// Install PeiGraphicsFramebufferReadyPpi
  ///
  Status = PeiServicesInstallPpi (&mPeiGraphicsFramebufferReadyPpi);
  ASSERT_EFI_ERROR (Status);
  ///
  /// Display Logo if user provides valid Bmp image
  ///
  if (GtConfig->LogoPtr != 0) {
    DEBUG ((DEBUG_INFO, "FillFrameBufferAndShowLogo Start\n"));
    PostCode (0xA08);
    Status = FillFrameBufferAndShowLogo (GtConfig);
  }

  DEBUG ((DEBUG_INFO, "CallPpiAndFillFrameBuffer: End \n"));

  return Status;
}

/**
  FillFrameBufferAndShowLogo: Fill frame buffer with the image

  @param[in] GtConfig  - GRAPHICS_PEI_CONFIG to access the GtConfig related information

  @retval    EFI_STATUS
**/
EFI_STATUS
FillFrameBufferAndShowLogo (
  IN   GRAPHICS_PEI_CONFIG             *GtConfig
  )
{
  EFI_GRAPHICS_OUTPUT_BLT_PIXEL *Blt;
  UINTN                         Height;
  UINTN                         Width;
  EFI_STATUS                    Status;
  UINTN                         LogoDestX;
  UINTN                         LogoDestY;
  UINTN                         SrcY,DstY;
  UINT8                         *SrcAddress;
  UINT8                         *DstAddress;
  UINT32                        BytesPerScanLine;

  DEBUG ((DEBUG_INFO, "FillFrameBufferAndShowLogo: Start \n"));

  Blt = NULL;
  Status = EFI_SUCCESS;
  if (mMode == NULL) {
    DEBUG ((DEBUG_INFO, "Returning from FillFrameBufferAndShowLogo() due to invalid mode\n"));
    return EFI_UNSUPPORTED;
  }

  //
  // frame buffer with the image
  //
  Width      = GtConfig->LogoPixelWidth;
  Height     = GtConfig->LogoPixelHeight;
  Blt        = GtConfig->BltBufferAddress;

  //
  // if Convert Bmp to blt successful Center the logo and fill frame buffer.
  //
  LogoDestX = (mMode->Info->HorizontalResolution - Width) / 2;
  LogoDestY = (mMode->Info->VerticalResolution - Height) / 2;
  BytesPerScanLine = mMode->Info->PixelsPerScanLine * sizeof (EFI_GRAPHICS_OUTPUT_BLT_PIXEL);
  //
  // Fill framebuffer with the logo line by line
  //
  for (SrcY = 0, DstY = LogoDestY; DstY < (LogoDestY + Height); SrcY++, DstY++) {
    DstAddress = (UINT8 *) (UINTN) (mMode->FrameBufferBase + DstY * BytesPerScanLine + LogoDestX * sizeof (EFI_GRAPHICS_OUTPUT_BLT_PIXEL));
    SrcAddress = (UINT8 *) ((UINT8 *) Blt + (SrcY * Width * sizeof (EFI_GRAPHICS_OUTPUT_BLT_PIXEL)));
    CopyMem (DstAddress, SrcAddress, Width * sizeof (EFI_GRAPHICS_OUTPUT_BLT_PIXEL));
  }

  DEBUG ((DEBUG_INFO, "FillFrameBufferAndShowLogo: END \n"));

  return Status;
}

