/** @file
  Initializes IEH Device in PEI

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/PchPcrLib.h>
#include <Library/PciSegmentLib.h>
#include <Register/IehRegs.h>
#include <Register/ItssRegs.h>
#include <Register/PchRegs.h>
#include <Register/PchPcrRegs.h>
#include <IndustryStandard/Pci30.h>
#include <Library/PchInfoLib.h>

/**
  Programs severity register bits based on ITSS UEV register

  @param[in]  IehPciBase    IEH Controller PCI Base address
**/
VOID
IehProgramSeverity (
  IN  UINT64         IehPciBase
  )
{
  UINT64         Data64;
  UINT32         DataMask;
  UINT32         ItssUev;

  //
  // Programming serverity values (LERRUNCSEV) based on ITSS UEV register
  //
  ItssUev = PchPcrRead32 (PID_ITSS, R_ITSS_PCR_UEV);
  Data64  = BIT5 | BIT4 | BIT3 | BIT2 | BIT1 | BIT0;
  if (ItssUev & BIT20) {
    Data64 |= BIT40 | BIT36 | BIT32 | BIT28 | BIT24 | BIT20 | BIT16 | BIT12 | BIT8;
  }
  //
  // UC severity bits
  //
  if (ItssUev & BIT16) {
    Data64 |= BIT39 | BIT35 | BIT31 | BIT27 | BIT23 | BIT19 | BIT15 | BIT11 | BIT7;
  }
  //
  // Cmd and Data parity bits
  //
  if (ItssUev & BIT22) {
    Data64 |= BIT38 | BIT34 | BIT30 | BIT29 | BIT26 | BIT25 | BIT22 | BIT21 | BIT18 | BIT17 | BIT14 | BIT13 | BIT10 | BIT9 | BIT6;
  }

  if (IsPchH ()) {
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRUNCSEV, (UINT32) Data64);
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRUNCSEV + 4, (UINT32) RShiftU64 (Data64, 32));
  } else {
    //
    // For LP only register is valid only for 29 bits
    //
    DataMask = (1 << 29) - 1;
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRUNCSEV, (UINT32) (Data64 & DataMask));
  }
}

/**
  Programs IEH to Enable Mode

  @param[in]  IehPciBase    IEH Controller PCI Base address
**/
VOID
IehEnableMode (
  IN  UINT64         IehPciBase
  )
{
  PciSegmentWrite32 (IehPciBase + R_IEH_CFG_GCOERRMSK, 0xFFE00000);
  PciSegmentWrite32 (IehPciBase + R_IEH_CFG_GNFERRMSK, 0xFFE00000);
  PciSegmentWrite32 (IehPciBase + R_IEH_CFG_GFAERRMSK, 0xFFE00000);
  PciSegmentWrite32 (IehPciBase + R_IEH_CFG_GSYSEVTMSK, 0x0);
  PciSegmentAndThenOr32 (IehPciBase + R_IEH_CFG_GSYSEVTMAP, 0x7FFFFFC0, 0x3F);
  PciSegmentAnd32 (IehPciBase + R_IEH_CFG_IEHTYPEVER, (UINT32)~(B_IEH_CFG_IEHTYPEVER_IEH_BUSNUM));
  PciSegmentAndThenOr32 (IehPciBase + R_IEH_CFG_ERRPINCTRL, 0xFFFFFFC0, 0x2A);
  PciSegmentAnd32 (IehPciBase + R_IEH_CFG_ERRPINDATA, 0xFFFFFFF8);
  PciSegmentAnd32 (IehPciBase + R_IEH_CFG_BITMAP, 0xFFFFFFE0);
  if (IsPchH ()) {
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRUNCMSK, 0x0);
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRUNCMSK + 4, 0xFFFFFC00);
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRCORMSK, 0x0);
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRCORMSK + 4, 0xFFFFFC00);
  } else {
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRUNCMSK, 0xC0000000);
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRUNCMSK + 4, 0xFFFFFFFF);
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRCORMSK, 0xC0000000);
    PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LERRCORMSK + 4, 0xFFFFFFFF);
  }
  PciSegmentWrite32 (IehPciBase + R_IEH_CFG_LCOERRCNTSEL, 0x0);
//
// IOTG Workaround to to prevent un-handled error events from hanging the system due to UFS
//
#if FixedPcdGet8(PcdEmbeddedEnable) == 0x1
  DEBUG ((DEBUG_INFO, "IEH: Workaround to mask global registers and syseventmask\n"));
  PciSegmentWrite32 (IehPciBase + R_IEH_CFG_GCOERRMSK, 0xFFFFFFFF);
  PciSegmentWrite32 (IehPciBase + R_IEH_CFG_GSYSEVTMAP, 0x0);
  //
  // Clear Error Status
  //
  if ((PciSegmentRead32(IehPciBase + R_IEH_CFG_GCOERRSTS) & BIT0) == 1) {
    PciSegmentOr32 (IehPciBase + R_IEH_CFG_LERRCORSTS, 0xFFFFFFFF);
  }
  if ((PciSegmentRead32(IehPciBase + R_IEH_CFG_GNFERRSTS) & BIT0) == 1) {
    PciSegmentOr32 (IehPciBase + R_IEH_CFG_LERRUNCSTS, 0xFFFFFFFF);
  }
  if ((PciSegmentRead32(IehPciBase + R_IEH_CFG_GFAERRSTS) & BIT0) == 1) {
    PciSegmentOr32 (IehPciBase + R_IEH_CFG_LERRUNCSTS, 0xFFFFFFFF);
  }
  PciSegmentOr32 (IehPciBase + R_IEH_CFG_GCOERRSTS, 0xFFFFFFFF);
  PciSegmentOr32 (IehPciBase + R_IEH_CFG_GNFERRSTS, 0xFFFFFFFF);
  PciSegmentOr32 (IehPciBase + R_IEH_CFG_GFAERRSTS, 0xFFFFFFFF);
#endif
}
