/** @file
  The PEI Library Implements ME Init.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include <Library/DebugLib.h>
#include <Register/MeRegs.h>
#include <Library/PeiServicesLib.h>
#include <Library/HobLib.h>
#include <Library/PeiMeLib.h>
#include <Library/PciSegmentLib.h>
#include <Library/MeShowBufferLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/MeFwStsLib.h>
#include <Library/PchSbiAccessLib.h>
#include <Library/MeUefiFwHealthStatusLib.h>
#include <Library/PmcLib.h>
#include <Ppi/SiPolicy.h>
#include <Library/PchInfoLib.h>
#include <Register/HeciRegs.h>
#include <Library/TimerLib.h>
#include "Library/MeTypeLib.h"
#include "MeInitPostMem.h"
#include <ConfigBlock/MeGetConfigBlock.h>
#include <MeDefines.h>

/**
  Disable ME Devices when needed

  @param[in] IsHeci3Supported    Determines if HECI3 is supported and should be enabled

  @retval EFI_STATUS             Operation status
  @retval EFI_INVALID_PARAMETER  Passed invalid input parameters
**/
EFI_STATUS
MeDeviceConfigure (
  IN BOOLEAN                      IsHeci3Supported
  )
{
  SPS_PEI_CONFIG       *SpsPeiConfig = NULL;
  ME_PEI_PREMEM_CONFIG *MePreMemConfig;
  ME_DEVICE_FUNC_CTRL  HeciCommunication1;
  ME_DEVICE_FUNC_CTRL  HeciCommunication3;

  SpsPeiConfig = GetSpsPreMemPeiConfigBlock (NULL);
  if (SpsPeiConfig == NULL) {
    ASSERT (SpsPeiConfig != NULL);
    return EFI_INVALID_PARAMETER;
  }

  MePreMemConfig = GetMePreMemConfigBlock (NULL);
  if (MePreMemConfig == NULL) {
    ASSERT (MePreMemConfig != NULL);
    return EFI_INVALID_PARAMETER;
  }

  DEBUG ((DEBUG_INFO, "[ME] Disabling ME functions:"));

  HeciCommunication1 = Enabled;
  HeciCommunication3 = Enabled;

  if (MePreMemConfig->HeciCommunication2 == ME_DEVICE_DISABLED) {
    DEBUG ((DEBUG_INFO, " %d (HECI-2)", HECI2));
    SetMeUefiFwHealthStatus (Heci2State, StatusOther);
  } else {
    SetMeUefiFwHealthStatus (Heci2State, StatusOk);
  }
  if (SpsPeiConfig->HeciCommunication4 == ME_DEVICE_DISABLED) {
    DEBUG ((DEBUG_INFO, " %d (HECI-4)", HECI4));
    SetMeUefiFwHealthStatus (Heci4State, StatusOther);
  } else {
    SetMeUefiFwHealthStatus (Heci4State, StatusOk);
  }
  if (SpsPeiConfig->IderDeviceEnable == ME_DEVICE_DISABLED) {
    DEBUG ((DEBUG_INFO, " %d (IDE-R)", IDER));
    SetMeUefiFwHealthStatus (IdeRState, StatusOther);
  } else {
    SetMeUefiFwHealthStatus (IdeRState, StatusOk);
  }
  if (MePreMemConfig->KtDeviceEnable == ME_DEVICE_DISABLED) {
    DEBUG ((DEBUG_INFO, " %d (KT)", SOL));
    SetMeUefiFwHealthStatus (KTState, StatusOther);
  } else {
    SetMeUefiFwHealthStatus (KTState, StatusOk);
  }
  DEBUG ((DEBUG_INFO, "\n"));

  //
  // Before potential disabling HECI1
  // We have to store Firmware Statuses in to HOB
  //
  SaveFwStsToHob();

  MeDeviceControl (HECI1, HeciCommunication1);
  MeDeviceControl (HECI2, MePreMemConfig->HeciCommunication2);
  MeDeviceControl (HECI3, HeciCommunication3);
  MeDeviceControl (HECI4, SpsPeiConfig->HeciCommunication4);
  MeDeviceControl (IDER, SpsPeiConfig->IderDeviceEnable);
  MeDeviceControl (SOL, MePreMemConfig->KtDeviceEnable);

  return EFI_SUCCESS;
} // MeDeviceConfigure

/**
  Configure HECI devices on End Of Pei
**/
VOID
MeOnEndOfPei (
  VOID
  )
{

  DEBUG ((DEBUG_INFO, "MeOnEndOfPei\n"));

  if (GetBootModeHob () == BOOT_ON_S3_RESUME) {
      if (MeHeciD0I3Enabled () == FALSE) {
        return;
      }
  } else {
    if (MeTypeIsClient ()) {
      //
      // Send optional HECI command
      //
      SendOptionalHeciCommand ();
    }

    //
    // Save ME/AMT policies in PEI phase to HOB for using in DXE.
    //
    SavePolicyToHob ();
  }

  DEBUG ((DEBUG_INFO, "MeOnEndOfPei done.\n"));
}

/**
  Function forces a reinit of the heci interface by following the reset heci interface via host algorithm
  in HPS section 4.1.1.1

  @param[in] HeciDev              The HECI device to be accessed.

  @retval EFI_SUCCESS             Interface reset successful
  @retval EFI_TIMEOUT             ME is not ready
**/
EFI_STATUS
EFIAPI
ResetHeciInterface (
  IN  HECI_DEVICE                 HeciDev
  )
{
  return EFI_UNSUPPORTED;
}

/**
  Build Sps Guid Data Hob

  @param[in] SiPreMemPolicyPpi     Pointer to PreMem policy PPI
**/
VOID
BuildSpsGuidDataHob (
  IN SI_PREMEM_POLICY_PPI *SiPreMemPolicyPpi
  )
{
  SPS_PEI_CONFIG            *SpsPeiConfig;
  UINTN                     HobSize;
  VOID                      *HobPtr;

  ASSERT (SiPreMemPolicyPpi != 0);
  if (SiPreMemPolicyPpi != NULL) {
    SpsPeiConfig = GetSpsPreMemPeiConfigBlock (SiPreMemPolicyPpi);
    ASSERT (SpsPeiConfig != 0);
    if (SpsPeiConfig != NULL) {
      HobSize = SpsPeiConfig->Header.GuidHob.Header.HobLength;
      DEBUG ((DEBUG_INFO, "HobSize for SpsPeiConfig: %x\n", HobSize));
      HobPtr = BuildGuidDataHob (&gSpsPolicyHobGuid, SpsPeiConfig, HobSize);
      ASSERT (HobPtr != 0);
    }
  }
}
