/** @file
  .

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _MrcChipApi_h_
#define _MrcChipApi_h_

#include "MrcDdrIoApi.h"

///
/// Defines and Macros
///
#define MAX_MR_GEN_FSM  (28)  ///< Maximum number of MR Addresses that can be sent.

// Separation required by hardware.
#define TX_FIFO_SEPARATION         (-2)

#define MAX_ADD_RANK_DELAY  (7)
#define MAX_ADD_DELAY       (7)
#define MAX_DEC_DELAY       (15)

///
/// Struct and Types
///

// Generic MRS FSM CommandTypes
// Based on MC0_CH0_CR_GENERIC_MRS_FSM_CONTROL_0_STRUCT.COMMAND_TYPE
// defintion:
// MRW  :  2'b00
// MPC  :  2'b01
// VREF :  2'b10
typedef enum {
  GmfCmdMrw  = 0,
  GmfCmdMpc  = 1,
  GmfCmdVref = 2
} GmfCmdType;

// Delay timing types for Generic MRS FSM
typedef enum {
  GmfDelay_tMOD,
  GmfDelay_tZQCAL,
  GmfDelay_tZQLAT,
  GmfDelay_tDFE,
  GmfDelay_tVREFCA
} GmfDelayType;

typedef struct {
  UINT16      Delay;            ///< Specified in tCK
  UINT8       MrAddr;
  UINT8       MrData;
  BOOLEAN     Valid;
  BOOLEAN     FspWrToggle;      ///< Defines that this MR write should toggle the FSP state for WR
  BOOLEAN     FspOpToggle;      ///< Defines that this MR write should toggle the FSP state for OP
  BOOLEAN     FreqSwitchPoint;  ///< Defines that after this MR, we should execute the Frequency switch.
  GmfCmdType  CmdType;          ///< Defines the type of command to output. The default (0) is MRW
} MRC_GEN_MRS_FSM_MR_TYPE;

/**
  This function configures the Generic MRS FSM shadow registers based on the MrData inputs.
  It will determine if it needs to use the per-rank feature if the MR value differs across ranks.

  @param[in] MrcData - Pointer to MRC global data.
  @param[in] MrData  - Pointer to an array of MR data to configure the MRS FSM with.

  @retval mrcSuccess if successful.
  @retval mrcFail if MrData pointer is null, the timing or per-rank registers are out of free entries.
**/
MrcStatus
MrcGenMrsFsmConfig (
  IN  MrcParameters *MrcData,
  IN  MRC_GEN_MRS_FSM_MR_TYPE MrData[MAX_CONTROLLER][MAX_CHANNEL][MAX_RANK_IN_CHANNEL][MAX_MR_GEN_FSM]
  );

/**
  This function executes the MRS FSM and waits for the FSM to complete.
  If the FSM does not complete after 10 seconds, it will return an error message.

  @param[in] MrcData - Pointer to MRC global data.

  @retval mrcFail if the FSM is not idle.
  @retval mrcSuccess otherwise.
**/
MrcStatus
MrcGenMrsFsmRun (
  IN  MrcParameters *MrcData
  );

UINT32
MrcQclkToTck (
  IN  MrcParameters *const MrcData,
  IN  UINT32               Qclk
  );

UINT32
MrcTckToQclk (
  IN  MrcParameters *const MrcData,
  IN  UINT32               Tck
  );

/**
  Program MC/DDRIO registers to Gear1 or Gear2 mode.
  This only includes Gear2 mode enable/disable, not other registers that are impacted by gear mode.

  @param[in] MrcData - The MRC general data.
  @param[in] Gear2   - TRUE for Gear2, FALSE for Gear1

  @retval None
**/
void
MrcSetGear2 (
  IN MrcParameters *const MrcData,
  IN BOOLEAN              Gear2
  );

/**
  Programming of CCC_CR_DDRCRCCCCLKCONTROLS_BlockTrainRst

  @param[in] MrcData - The MRC global data.
  @param[in] BlockTrainReset - TRUE to BlockTrainReset for most training algos.  FALSE for specific training algos that need PiDivider sync.

**/
VOID
MrcBlockTrainResetToggle (
  IN MrcParameters *const MrcData,
  IN BOOLEAN              BlockTrainReset
  );

/**
  This function configures the DDRCRCMDBUSTRAIN register to values for normal mode.

  @param[in]  MrcData - Pointer to global MRC data.

  @retval none.
**/
VOID
MrcSetWritePreamble (
  IN  MrcParameters *const  MrcData
  );

/**
  Returns the currently configured DRAM Command Intput Rate NMode
  Note: In DDR4 3N is used during training (in Gear1), but this
  function won't report this.

  @param[in] MrcData    - Include all MRC global data.

  @retval 1 = 1N Mode
  @retval 2 = 2N Mode
**/
UINT32
MrcGetNMode (
  IN MrcParameters *const MrcData
  );

/**
  Configure the MC to issue multicycle CS_n MPC commands.
  The DRAM must be configured separately by either setting
  DDR5 MR2.OP[4] = 0 or by resetting the DRAM.

  @param[in] MrcData    - Include all MRC global data.

  @retval None
**/
void
EnableMcMulticycleCs (
  IN MrcParameters *const MrcData
  );

/**
  Configure the MC and DRAM for single cycle CS_n MPC
  commands. An MRW is issued to the DRAM to configure
  DDR5 MR2[4] = 1.

  @param[in] MrcData    - Include all MRC global data.

  @retval None
**/
void
DisableMcMulticycleCs (
  IN MrcParameters *const MrcData
  );

UINT32
ReadFirstRcompReg (
  IN OUT MrcParameters *const MrcData
  );

/**
  Find the maximum possible cycle increment and decrement values for write leveling trainings

  @param[in] MrcData - Pointer to MRC global data.
  @param[out] MaxAdd  - Max possible increment value for write leveling trainings
  @param[out] MaxDec  - Max possible decrement value for write leveling trainings

  @retval None
**/
MrcStatus
MrcMcTxCycleLimits (
  IN MrcParameters *const MrcData,
  OUT UINT32              *MaxAdd,
  OUT UINT32              *MaxDec
  );

/**
  Find the minimum and maximum PI setting across Tx DQ/DQS on a given Rank, on all channels.
  Determine how far we can use the PI value to shift the Cycle. Min value will use DQS,
  Max value will use DQ (DQ is DQS + 32 (gear1) or DQS + 96 (gear2), and also plus tDQS2DQ for LP4/LP5).

  @param[in]  MrcData         - Pointer to MRC global data.
  @param[in]  Rank            - Current working rank
  @param[in]  MaxAdd          - Max possible increment value for write leveling trainings
  @param[in]  MaxDec          - Max possible decrement value for write leveling trainings
  @param[out] StartOffset     - Starting offset
  @param[out] EndOffset       - End of limit
  @param[out] SavedTxDqsVal   - Pointer to array where current Tx Dqs timings will be stored

  @retval MrcStatus - If it succeeded, return mrcSuccess
**/

MrcStatus
MrcIoTxLimits (
  IN  MrcParameters *const MrcData,
  IN  UINT32               Rank,
  IN  UINT32               MaxAdd,
  IN  UINT32               MaxDec,
  OUT INT32               *StartOffset,
  OUT INT32               *EndOffset,
  OUT UINT16              SavedTxDqsVal[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM]
  );

/**
  Programs new delay offsets to Tx DQ/DQS timing

  @param[in]  MrcData       - Pointer to MRC global data.
  @param[in]  Rank          - Current working rank
  @param[in]  MaxAdd        - Max possible increment value for write leveling trainings
  @param[in]  MaxDec        - Max possible decrement value for write leveling trainings
  @param[in]  SavedTxDqsVal - array of the initial TX DQS Delays
  @param[in]  TxDelayOffset - Delay to be programmed

  @retval @retval mrcSuccess or mrcTimingError if minimum TX fifo separation is not met
**/

MrcStatus
SetWriteCycleDelay (
  IN  MrcParameters *const  MrcData,
  IN  UINT32                Rank,
  IN  UINT32                MaxAdd,
  IN  UINT32                MaxDec,
  IN  UINT16                SavedTxDqsVal[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM],
  IN  INT32                 TxDelayOffset
  );
#endif // _MrcChipApi_h_
