/** @file
  The functions in this file implement the memory controller registers that
  are not training specific. After these functions are executed, the
  memory controller will be ready to execute the timing training sequences.

@copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
//
// Include files
//
#include "MrcMcConfiguration.h"
#include "MrcMaintenance.h"
#include "MrcDdrIoOffsets.h"
#include "MrcMemoryApi.h"
#include "MrcLpddr4.h"
#include "MrcDdr4.h"
#include "MrcCommon.h"
#include "Cpgc20.h"
#include "MrcCpgcOffsets.h"
#include "MrcCpgcApi.h"
#include "MrcDdrIoApi.h"
#include "MrcDdrIoApiInt.h"
#include "MrcChipApi.h"

#define AV_PHY_INIT 1

///
/// Structs and Types
///

/// CCC Enable Mappings.
/// The numbers after the Ccc union define which CCC instance this mapping should be applied.
typedef union {
  union { // DDR4
    union { // UY
      struct {
        UINT16      : 5; ///< Bits 0:4
        UINT16 Clk0 : 2; ///< Bits 5:6 Clock is a 2 bit enable for P & N
        UINT16      : 9; ///< Bits 7:15
      } Ccc04;
      struct {
        UINT16      : 1;  ///< Bit 0
        UINT16 Odt1 : 1;  ///< Bit 1
        UINT16 Cs1  : 1;  ///< Bit 2
        UINT16 Cs0  : 1;  ///< Bit 3
        UINT16 Odt0 : 1;  ///< Bit 4
        UINT16      : 11; ///< Bits 5:15
      } Ccc15;
      struct {
        UINT16      : 10; ///< Bits 0:9
        UINT16 Cke1 : 1;  ///< Bit  10
        UINT16      : 1;  ///< Bit  11
        UINT16 Cke0 : 1;  ///< Bit  12
        UINT16      : 3;  ///< Bits 13:15
      } Ccc26;
      struct {
        UINT16      : 5; ///< Bits 0:4
        UINT16 Clk1 : 2; ///< Bits 5:6 Clock is a 2 bit enable for P & N
        UINT16      : 9; ///< Bits 7:15
      } Ccc37;
    } UY;
    union { // HS
      struct {
        UINT16 Cke0 : 1; ///< Bit  0
        UINT16      : 1; ///< Bit  1
        UINT16 Cke2 : 1; ///< Bit  2
        UINT16      : 1; ///< Bit  3
        UINT16 Cke3 : 1; ///< Bit  4
        UINT16      : 4; ///< Bits 5:8
        UINT16 Cke1 : 1; ///< Bit  9
        UINT16      : 6; ///< Bits 10:15
      } Ccc02;
      struct {
        UINT16         : 5; ///< Bits 0:4
        UINT16 Clk0or2 : 2; ///< Bits 5:6
        UINT16 Clk1or3 : 2; ///< Bits 7:8
        UINT16         : 7; ///< Bits 9:15
      } Ccc1357;
      struct {
        UINT16      : 1; ///< Bit  0
        UINT16 Odt0 : 1; ///< Bit  1
        UINT16 Cs0  : 1; ///< Bit  2
        UINT16 Odt2 : 1; ///< Bit  3
        UINT16 Cs2  : 1; ///< Bit  4
        UINT16      : 6; ///< Bits 5:10
        UINT16 Odt1 : 1; ///< Bit  11
        UINT16 Cs1  : 1; ///< Bit  12
        UINT16 Odt3 : 1; ///< Bit  13
        UINT16 Cs3  : 1; ///< Bit  14
        UINT16      : 2; ///< Bits 15:16
      } Ccc46;
    } HS;
  } Ddr4;
  union {
    struct {
      UINT16 Cs0    : 1; ///< Bit  0
      UINT16        : 1; ///< Bit  1
      UINT16 Cs1    : 1; ///< Bit  2
      UINT16        : 2; ///< Bits 3:4
      UINT16 Clk3   : 2; ///< Bits 5:6
      UINT16 Clk2   : 2; ///< Bits 7:8
      UINT16 Cs3    : 1; ///< Bit  9
      UINT16        : 6; ///< Bits 10:15
    } Ccc02;
    struct {
      UINT16        : 5; ///< Bits 0:4
      UINT16 Clk0   : 2; ///< Bits 5:6
      UINT16 Clk1   : 2; ///< Bits 7:8
      UINT16        : 2; ///< Bits 9:10
      UINT16 Cs2    : 1; ///< Bit  11
      UINT16        : 3; ///< Bits 12:15
    } Ccc13;
    struct {
      UINT16        : 5; ///< Bits 0:4
      UINT16 Clk3   : 2; ///< Bits 5:6
      UINT16 Clk2   : 2; ///< Bits 7:8
      UINT16        : 2; ///< Bits 9:10
      UINT16 Cs1    : 1; ///< Bit  11
      UINT16        : 2; ///< Bits 12:13
      UINT16 Cs0    : 1; ///< Bit  14
      UINT16        : 1; ///< Bit  15
    } Ccc46;
    struct {
      UINT16        : 5; ///< Bits 0:4
      UINT16 Clk1   : 2; ///< Bits 5:6
      UINT16 Clk0   : 2; ///< Bits 7:8
      UINT16 Cs3    : 1; ///< Bit  9
      UINT16        : 1; ///< Bit  10
      UINT16 Cs2    : 1; ///< Bit  11
      UINT16        : 4; ///< Bits 12:15
    } Ccc57;
  } Ddr5;
  union {
    struct {
      UINT16      : 2; ///< Bits  0:1
      UINT16 Cs1  : 1; ///< Bit   2
      UINT16      : 1; ///< Bit   3
      UINT16 Cs0  : 1; ///< Bit   4
      UINT16 Clk  : 2; ///< Bits  5:6 Clock is a 2 bit enable for P & N
      UINT16 Cke  : 2; ///< Bits  7:8
      UINT16      : 7; ///< Bits  9:15
    } Ccc023467;
    struct {
      UINT16 Cs1  : 1; ///< Bit   0
      UINT16      : 3; ///< Bits  0:3
      UINT16 Cs0  : 1; ///< Bit   4
      UINT16 Clk  : 2; ///< Bits  5:6 Clock is a 2 bit enable for P & N
      UINT16 Cke  : 2; ///< Bits  7:8
      UINT16      : 7; ///< Bits  9:15
    } Ccc15;
  } Lp4;
  union {
    struct {
      UINT16      : 2; ///< Bits  0:1
      UINT16 Cs   : 2; ///< Bits  2:3
      UINT16      : 1; ///< Bit   4
      UINT16 Clk  : 2; ///< Bits  5:6 Clock is a 2 bit enable for P & N
      UINT16 Wck  : 2; ///< Bits  7:8 Clock is a 2 bit enable for P & N
      UINT16      : 7; ///< Bits  9:15
    } Ccc023467;
    struct {
      UINT16 Cs0  : 1; ///< Bits  0
      UINT16      : 2; ///< Bits  1:2
      UINT16 Cs1  : 1; ///< Bit   3
      UINT16      : 1; ///< Bit   4
      UINT16 Clk  : 2; ///< Bits  5:6 Clock is a 2 bit enable for P & N
      UINT16 Wck  : 2; ///< Bits  7:8 Clock is a 2 bit enable for P & N
      UINT16      : 7; ///< Bits  9:15
    } Ccc15;
  } Lp5A;
  union {
    struct {
      UINT16      : 5; ///< Bit   0:4
      UINT16 Clk  : 2; ///< Bits  5:6 Clock is a 2 bit enable for P & N
      UINT16 Wck  : 2; ///< Bits  7:8 Clock is a 2 bit enable for P & N
      UINT16      : 1; ///< Bits  9
      UINT16 Cs   : 2; ///< Bits  10:11
      UINT16      : 4; ///< Bits  12:15
    } Ccc0145;
    struct {
      UINT16      : 5; ///< Bit   0:4
      UINT16 Clk  : 2; ///< Bits  5:6 Clock is a 2 bit enable for P & N
      UINT16 Wck  : 2; ///< Bits  7:8 Clock is a 2 bit enable for P & N
      UINT16 Cs0  : 1; ///< Bits  9
      UINT16      : 1; ///< Bits  10
      UINT16 Cs1  : 1; ///< Bits  11
      UINT16      : 4; ///< Bits  12:15
    } Ccc2367;
  } Lp5D;
  union {
    struct {
      UINT16      : 5; ///< Bits 0:4
      UINT16 Clk  : 2; ///< Bits 5:6 Clock is a 2 bit enable for P & N
      UINT16 Cke  : 2; ///< Bits 7:8
      UINT16 Cs3  : 1; ///< Bit  9
      UINT16 Cs2  : 1; ///< Bit  10
      UINT16 Cs1  : 1; ///< Bit  11
      UINT16 Cs0  : 1; ///< Bit  12
      UINT16      : 3; ///< Bits 13:15
    } Ccc0246;
    struct {
      UINT16 Cs3  : 1; ///< Bit  0
      UINT16      : 4; ///< Bit  1:4
      UINT16 Clk  : 2; ///< Bits 5:6 Clock is a 2 bit enable for P & N
      UINT16 Cke  : 2; ///< Bits 7:8
      UINT16 Cs1  : 1; ///< Bit  9
      UINT16 Cs0  : 1; ///< Bit  10
      UINT16      : 1; ///< Bit  11
      UINT16 Cs2  : 1; ///< Bit  12
      UINT16      : 3; ///< Bits 13:15
    } Ccc13;
    struct {
      UINT16 Cs1  : 1; ///< Bit  0
      UINT16      : 4; ///< Bit  1:4
      UINT16 Clk  : 2; ///< Bits 5:6 Clock is a 2 bit enable for P & N
      UINT16 Cke  : 2; ///< Bits 7:8
      UINT16      : 1; ///< Bit  9
      UINT16 Cs2  : 1; ///< Bit  10
      UINT16 Cs0  : 1; ///< Bit  11
      UINT16      : 1; ///< Bit  12
      UINT16 Cs3  : 1; ///< Bit  13
      UINT16      : 2; ///< Bit  14:15
    } Ccc5;
    struct {
      UINT16      : 5; ///< Bits 0:4
      UINT16 Clk  : 2; ///< Bits 5:6 Clock is a 2 bit enable for P & N
      UINT16 Cke  : 2; ///< Bits 7:8
      UINT16      : 1; ///< Bit  9
      UINT16 Cs2  : 1; ///< Bit  10
      UINT16 Cs0  : 1; ///< Bit  11
      UINT16 Cs1  : 1; ///< Bit  12
      UINT16      : 1; ///< Bit  13
      UINT16 Cs3  : 1; ///< Bit  14
      UINT16      : 1; ///< Bit  15
    } Ccc7;
  } LpHS;
  struct {
    UINT16        : 5; ///< Bit   0:4
    UINT16 Clk    : 2; ///< Bits  5:6 Clock is a 2 bit enable for P & N
    UINT16 CkeWck : 2; ///< Bits  7:8 Clock is a 2 bit enable for P & N
    UINT16        : 7; ///< Bits  9:15
  } CccCommonLp; ///< Clk is common for all technologies.  Cke/Wck are the same across all fubs for Lp4/Lp5 on UY & HS.
  UINT16  Data16;
} CCC_TX_EN_TYPE;

/// Defines
#define MRC_NUM_BYTE_GROUPS    (8)

/// CCC Command TxEn Bits
#define MRC_UY_LP4_CCC023467_CA_MSK  (0x1E03)
#define MRC_UY_LP4_CCC15_CA_MSK      (0x1E06)
#define MRC_UY_LP5A_CCC023467_CA_MSK (0x1E13)
#define MRC_UY_LP5A_CCC15_CA_MSK     (0x1E16)
#define MRC_UY_LP5D_CCC0145_CA_MSK   (0x121F)
#define MRC_UY_LP5D_CCC2367_CA_MSK   (0x141F)
#define MRC_UY_DDR4_CCC04_CA_MSK     (0x1E1C)
#define MRC_UY_DDR4_CCC15_CA_MSK     (0x1E01)
#define MRC_UY_DDR4_CCC26_CA_MSK     (0xA0F)
#define MRC_UY_DDR4_CCC37_CA_MSK     (0x1F)
#define MRC_HS_DDR4_CCC02_CA_MSK     (0x7C0A)
#define MRC_HS_DDR4_CCC13_CA_MSK     (0x601F)
#define MRC_HS_DDR4_CCC46_CA_MSK     (0x601)
#define MRC_HS_DDR4_CCC57_CA_MSK     (0x7418)
#define MRC_HS_DDR5_CCC02_CA_MSK     (0x7C0A)
#define MRC_HS_DDR5_CCC13_CA_MSK     (0x201F)
#define MRC_HS_DDR5_CCC46_CA_MSK     (0x161F)
#define MRC_HS_DDR5_CCC57_CA_MSK     (0x7410)
#define MRC_HS_LP4_CCC012346_CA_MSK  (0x601E)
#define MRC_HS_LP4_CCC5_CA_MSK       (0x501E)
#define MRC_HS_LP4_CCC7_CA_MSK       (0x221E)
#define MRC_HS_LP5_CCC0246_CA_MSK    (0x601F)
#define MRC_HS_LP5_CCC13_CA_MSK      (0x681E)
#define MRC_HS_LP5_CCC5_CA_MSK       (0x521E)
#define MRC_HS_LP5_CCC7_CA_MSK       (0x221F)
#define PANICV0                      (25)   // mV
#define PANICV1                      (40)   // mV
#define VCCANA_EH                    (1800) // mV
#define MRC_VCCDLL_TARGET            (850)  // mV
#define MRC_VCCBG                    (1000) // mV
#define CA_VOLT_SEL                  (0)
#define CLK_VOLT_SEL                 (0)

///
/// Global Constants
///
const UINT8 CompParamList[] = { RdOdt, WrDS, WrDSCmd, WrDSCtl, WrDSClk };
const UINT8 ByteStagger[] = {1, 4, 1, 5, 2, 6, 3, 7, 8};  // Increased Byte Stagger for byte0
const UINT8 MinCycleStageDelay[] = {46, 70, 70, 46};      // Avoid corner case

/// 8 banks case (x8 or x16):
/// Logical Bank:         0  1  2  3  4  5  6  7
///                       ----------------------
/// Physical Bank:        0  0  1  1  2  2  3  3
/// Physical Bank Group:  0  1  0  1  0  1  0  1
static MRC_BG_BANK_PAIR Ddr4BankMap[MAX_DDR4_x16_BANKS] = {
  {0,0}, {1,0}, {0,1}, {1,1}, {0,2}, {1,2}, {0,3}, {1,3}
};


/**
  This function calculates the two numbers that get you closest to the slope.

  @param[in]  MrcData - Pointer to global MRC data.
  @param[in]  Slope   - Targeted slope (multiplied by 1000 for integer math)

  @retval Returns the Slope Index to be programmed for VtSlope in terms of the CR.
**/
UINT8
MrcCalcVtSlopeCode (
  IN  MrcParameters * const MrcData,
  IN  const UINT16 Slope
  )
{
  static const INT16 Coding[] = {0, -250, -125, -62, 500, 250, 125, 62}; // 0: 0, 1: -1/4, 2: -1/8, 3: -1/16, 4: +1/2, 5: +1/4, 6: +1/8, 7: +1/16
  INT16       Error;
  INT16       BestError;
  UINT8       BestI;
  UINT8       BestJ;
  UINT8       i;
  UINT8       j;

  BestError = 1000;
  BestI     = 0;
  BestJ     = 0;
  for (i = 0; i < (sizeof (Coding) / sizeof (Coding[0])); i++) {
    for (j = 0; j < (sizeof (Coding) / sizeof (Coding[0])); j++) {
      Error = Slope - (Coding[i] + Coding[j]);
      if (Error < 0) {
        Error = -Error;
      }

      if (BestError > Error) {
        BestError = Error;
        BestI     = i;
        BestJ     = j;
      }
    }
  }

  return (BestI << 3) + BestJ;
}

/**
  This function configures the BCLK RFI frequency for each SAGV point.

  @param[in, out] MrcData - MRC global data.

  @retval VOID
**/
VOID
MrcBclkRfiConfiguration (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  MrcDebug          *Debug;
  UINT32            *BclkRfiFreqInput;
  UINT32              MailboxCommand;
  UINT32              MailboxData;
  UINT32              MailboxStatus;
  UINT32              MinBclkRfiFreq;
  MrcIntOutput        *MrcIntData;
  const MRC_FUNCTION  *MrcCall;


  MrcIntData        = ((MrcIntOutput *) (MrcData->IntOutputs.Internal));
  Inputs            = &MrcData->Inputs;
  Outputs           = &MrcData->Outputs;
  Debug             = &Outputs->Debug;
  MrcCall           = Inputs->Call.Func;
  BclkRfiFreqInput  = &MrcData->Inputs.BclkRfiFreq[MrcIntData->SaGvPoint];

  // Configure BCLK RFI frequency if needed for this SAGV point
  if (*BclkRfiFreqInput != 0) {
    // Get Min BCLK RFI range
    MailboxCommand = CPU_MAILBOX_BCLK_CONFIG_CMD |
                    (CPU_MAILBOX_BCLK_CONFIG_READ_BCLK_RFI_RANGE_SUBCOMMAND << CPU_MAILBOX_CMD_PARAM_1_OFFSET);
    MrcCall->MrcCpuMailboxRead (MAILBOX_TYPE_PCODE, MailboxCommand, &MailboxData, &MailboxStatus);
    if (MailboxStatus != PCODE_MAILBOX_CC_SUCCESS) {
      MinBclkRfiFreq = BCLK_DEFAULT;
    } else {
      // Mailbox format is in 7.3 MHz, need to convert to Hz.
      // 1,000,000 / (2^3) = 125,000
      MinBclkRfiFreq = (MailboxData & CPU_MAILBOX_BCLK_CONFIG_READ_BCLK_RFI_RANGE_MIN_MASK) * BCLK_RFI_FREQ_CONVERSION;
    }
    MRC_DEBUG_MSG (
      Debug,
      MSG_LEVEL_NOTE,
      "CPU_MAILBOX_BCLK_CONFIG_CMD %s. MailboxStatus = %Xh\n",
      (MailboxStatus == PCODE_MAILBOX_CC_SUCCESS) ? "success" : "failed",
      MailboxStatus
      );

    // Verify BCLK RFI input range
    *BclkRfiFreqInput = RANGE(*BclkRfiFreqInput, MinBclkRfiFreq, BCLK_DEFAULT);

    // Update BCLK RFI frequency
    MailboxCommand = CPU_MAILBOX_BCLK_CONFIG_CMD |
                    (CPU_MAILBOX_BCLK_CONFIG_SET_BCLK_RFI_FREQ_SUBCOMMAND << CPU_MAILBOX_CMD_PARAM_1_OFFSET) |
                    (MrcIntData->SaGvPoint << CPU_MAILBOX_CMD_PARAM_2_OFFSET);
    // Mailbox format is in 7.3 Mhz, BclkRfiFreq is in Hz.
    // Mailbox format = BclkRfiFreq * (2^3) / 1,000,000.
    // (2^3)/1,000,000 = 1/125,000
    MailboxData = *BclkRfiFreqInput / BCLK_RFI_FREQ_CONVERSION;
    MrcCall->MrcCpuMailboxWrite (MAILBOX_TYPE_PCODE, MailboxCommand, MailboxData, &MailboxStatus);
    MRC_DEBUG_MSG (
      Debug,
      MSG_LEVEL_NOTE,
      "CPU_MAILBOX_BCLK_CONFIG_SET_BCLK_RFI_FREQ_SUBCOMMAND %s. MailboxStatus = %Xh\n",
      (MailboxStatus == PCODE_MAILBOX_CC_SUCCESS) ? "success" : "failed",
      MailboxStatus
      );
  }
}

/**
  This function locks the DDR frequency requested from SPD or User.
  It will update the frequency related members in the output structure.

  @param[in, out] MrcData - MRC global data.

  @retval mrcSuccess
**/
MrcStatus
MrcFrequencyLock (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput          *Inputs;
  MrcOutput         *Outputs;
  MrcDebug          *Debug;
  MrcChannelOut     *ChannelOut;
  MrcStatus         Status;
  MrcProfile        Profile;
  UINT32            Channel;
  UINT32            Controller;
  MrcIntOutput      *MrcIntData;

  MrcIntData        = ((MrcIntOutput *) (MrcData->IntOutputs.Internal));
  Inputs            = &MrcData->Inputs;
  Outputs           = &MrcData->Outputs;
  Debug             = &Outputs->Debug;
  Profile           = Inputs->MemoryProfile;
  // Make sure tCL-tCWL <= 4
  // This is needed to support ODT properly for 2DPC case
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (MrcChannelExist (MrcData, Controller, (UINT8) Channel)) {
        ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
        if (ChannelOut->DimmCount == 2) {
          if ((ChannelOut->Timing[Profile].tCL - ChannelOut->Timing[Profile].tCWL) > 4) {
            ChannelOut->Timing[Profile].tCWL = ChannelOut->Timing[Profile].tCL - 4;
            MRC_DEBUG_MSG (
              Debug,
              MSG_LEVEL_NOTE,
              "(tCL-tCWL) > 4, Mc%u.Ch%u - tCWL has been updated to %u\n",
              Controller,
              Channel,
              ChannelOut->Timing[Profile].tCWL
              );
          }
        }
      }
    }
  }

  // For LPDDR4/5, we need to do the first lock at the low frequency for ECT flow.
  // If we are disabling ECT or frequency switching, we need to lock at the intended frequency.
  if ((Outputs->Lpddr) &&
      (Inputs->LpFreqSwitch == TRUE) &&
      (Inputs->TrainingEnables.ECT == 1)) {
    Outputs->Frequency = f1067;
    MRC_DEBUG_MSG (
      Debug,
      MSG_LEVEL_NOTE,
      "Locking at low(%d) first.  Will switch to high(%d) after ECT\n",
      Outputs->Frequency,
      Outputs->HighFrequency
      );
  }

  // Configure BCLK RFI frequency
  MrcBclkRfiConfiguration(MrcData);

  // We don't always use MrcFrequencySwitch because of LPDDR4 Self-Refresh is command based.
  // On first execution of the frequency, Command has not been training and Self-Refresh may fail.
  if ((MrcData->Inputs.SaGv != MrcSaGvEnabled) || (MrcIntData->SaGvPoint == MrcSaGvPoint1)) {
    Status = McFrequencySet (MrcData, MRC_PRINTS_ON);
  } else {
    Status = MrcFrequencySwitch (MrcData, Outputs->Frequency, MRC_PRINTS_ON);
  }
  if (Status != mrcSuccess) {
    return Status;
  }
  MrcInternalCheckPoint (MrcData, OemFrequencySetDone, NULL);

  // Save MRC Version into CR
  MrcSetMrcVersion (MrcData);

  return Status;
}

/**
  This function initializes the Memory Controller: Scheduler, Timings, Address Decode,
  Odt Control, and refresh settings.

  @param[in, out] MrcData - MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status.
**/
MrcStatus
MrcMcConfiguration (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcOutput *Outputs;
  MrcDebug  *Debug;

  Outputs = &MrcData->Outputs;
  Debug   = &Outputs->Debug;

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Timing Config\n");
  MrcTimingConfiguration (MrcData);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Refresh Config\n");
  MrcRefreshConfiguration (MrcData);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Scheduler parameters\n");
  MrcSchedulerParametersConfig (MrcData);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Address Decoding Config\n");
  MrcAdConfiguration (MrcData);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Maintenance Configuration\n");
  MrcMaintenanceConfig (MrcData);

  return mrcSuccess;
}

/**
  This function initializes memory subsystem registers that are not specific to MC or DDRIO.

  @param[in, out] MrcData - Pointer to the global MRC data.

  @retval MrcStatus - mrcSuccess if successful or an error status.
**/
MrcStatus
MrcMemorySsInit (
  IN OUT  MrcParameters *const MrcData
  )
{
  ControllerZoneConfiguration (MrcData);

  return mrcSuccess;
}

/**
  This function initializes the following type of ddrio registers.
  MRC:RestrictedBegin
  1) Registers which has a default value of 0.
  2) Registers which are updated after Comp Training.
  3) Registers which are updated by hardware.
  These initialization are required to fulfill CR Restore architecture of ddrio
  Without these ddrphy_initcomplete wouldnt assert
  MRC:RestrictedEnd

  @param[in, out] MrcData - MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status.
**/
MrcStatus
MrcDdrioCrRestore (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcStatus Status;
  UINT32    i;

  // @todo_adl
  static UINT32 const AddUnder3000[] = {
//    DDRVREF_CR_DDRRCOMPDATA_REG,
//    DDRPHY_COMP_CR_DDRCOMPDATA_REG,
//    DDRVSSHIAFEA_CR_DDRRCOMPDATA_REG,
//    DDRPHY_COMP_CR_DDRCRVSXHICOMPDATA_REG,
//    DDRPHY_COMP_CR_DDRCRCOMPTEMP_REG,
//    DDRVSSHIAFEA_CR_DDRCRVSSHICOMPOFFSET_REG,
//    DDRPHY_COMP_CR_DDRCRDATACOMPVTT_REG,
//    DDRPHY_COMP_CR_DDRCRCOMPDVFSRCOMP_REG,
//    DDRPHY_COMP_CR_VCCDLLCOMPDATACCC_REG,
    MCMISCS_DDRWCKCONTROL_REG,
//    DDRPHY_COMP_CR_VSXHIFFCOMPREF0_REG,
//    DDRPHY_COMP_CR_VSXHIFFCOMPREF1_REG,
//    DDRPHY_COMP_CR_DDRCRCOMPOVR0_REG,
//    DDRPHY_COMP_CR_DDRCRCOMPOVR1_REG,
//    DDRPHY_COMP_CR_DDRCRCACOMP_REG
  };

  static UINT32 const FllRegs[] = { FLL_DYNAMIC_CFG_REG_REG, FLL_EXTIP_STAT_REG_REG, FLL_DIAG_STAT_REG_REG, FLL_DIAG_STAT_1_REG_REG, FLL_DIAG_STAT_2_REG_REG, FLL_DEBUG_CFG_REG_REG };

  static UINT32 const MulticastAddresses[] = {
    CCC_CR_DDRCRCLKCOMP_REG,
    CCC_CR_DDRCRCMDCOMP_REG,      // adl ?
    CCC_CR_DDRCRCTLCOMP_REG,      // adl ?
//    CCC_CR_DDRCRCACOMP_REG,
//    CCC_CR_DDRCRCTLCACOMPOFFSET_REG,
    DATA_CR_DDRCRDATACOMPVTT_REG, // adl ?
//    DDRVCCDLL_CR_DDRCRVCCDLLCOMPDLL_REG,
//    DLLDDR_CR_DDRCRVCCDLLCOMPDLL_REG,
//    DDRVCCDLL_CR_DDRCRVCCDLLVOLTAGES_REG,
//    CCC_CR_DDRCRCCCPERBITDESKEWPRISENFALL_REG,
//    CCC_CR_DDRCRCCCPERBITDESKEWPFALLNRISE_REG,
//    DATA_CR_DCCLANETARGET_REG,
//    DATA_CR_DCCPILUT0_REG,
//    DATA_CR_DCCPILUT1_REG,
//    DATA_CR_DCCPILUT2_REG,
//    DATA_CR_DCCPILUT3_REG,
//    CCC_CR_DDRCRPERBITTCO0_REG,
//    CCC_CR_DDRCRPERBITTCO1_REG,
//    CCC_CR_DDRCRPERBITTCO2_REG,
//    CCC_CR_DDRCRVSSHICLKCOMPOFFSET_REG,
//    DLLDDR_CR_PITUNE_REG,
//    DLLDDR_CR_DDRCRVCCDLLCOMPOFFSET_REG,
    /*DATA_CR_DDRCRDATACONTROL2_REG,*/
    DATA_CR_DDRCRWRRETRAINRANK0_REG,
    DATA_CR_DDRCRWRRETRAINRANK1_REG,
    DATA_CR_DDRCRWRRETRAINRANK2_REG,
    DATA_CR_DDRCRWRRETRAINRANK3_REG
  };

  static UINT32 const MulticastReadModifyWrite[] = {
    CCC_CR_PICODE0_REG,
    CCC_CR_PICODE1_REG,
    CCC_CR_PICODE2_REG,
    CCC_CR_PICODE3_REG,
    DATA_CR_TXCONTROL0RANK0_REG,
    DATA_CR_TXCONTROL0RANK1_REG,
    DATA_CR_TXCONTROL0RANK2_REG,
    DATA_CR_TXCONTROL0RANK3_REG
  };

  static UINT32 const ReadModifyWrite[] = {
    CH0CCC_CR_PICODE0_REG,
    CH0CCC_CR_PICODE1_REG,
    CH0CCC_CR_PICODE2_REG,
    CH0CCC_CR_PICODE3_REG,
    DATA0CH0_CR_TXCONTROL0RANK0_REG,
    DATA0CH0_CR_TXCONTROL0RANK1_REG,
    DATA0CH0_CR_TXCONTROL0RANK2_REG,
    DATA0CH0_CR_TXCONTROL0RANK3_REG
  };
  Status = mrcSuccess;

  for (i = 0; i < ARRAY_COUNT (AddUnder3000); i++) {
    MrcWriteCR (MrcData, AddUnder3000[i], 0);
  }

  for (i = 0; i < ARRAY_COUNT (MulticastAddresses); i++) {
    MrcWriteCrMulticast (MrcData, MulticastAddresses[i], 0);
  }

  for (i = 0; i < ARRAY_COUNT (MulticastReadModifyWrite); i++) {
    MrcWriteCrMulticast (MrcData, MulticastReadModifyWrite[i], MrcReadCR (MrcData, ReadModifyWrite[i]));
  }

  for (i = 0; i < ARRAY_COUNT (FllRegs); i++) {
    MrcWriteCR (MrcData, FllRegs[i], 0);
  }

  return Status;
}

/**
  This function will setup registers for DCC for PhyInitComplete

  @params[in] MrcData   - Pointer to MRC global data.

  @retval Success
**/
MrcStatus
MrcDccSetup (
  IN  MrcParameters * const MrcData
  )
{
  /* @todo_adl
  MrcChannelOut *ChannelOut;
  UINT32 Byte;
  UINT32 Gear2;
  UINT32 ByteEnd;
  UINT32 ByteStart;
  UINT32 Channel;
  UINT32 Controller;
  UINT32 Offset;
  MrcOutput *Outputs;
  MrcInput  *Inputs;
  UINT32    TempVar1;
  UINT32    tPBDMinFm;
  UINT32    DccStepSize;
  UINT32    Index;
  UINT16    Safe;
  UINT8     RankMask;
  DATA0CH0_CR_DCCFSMCONTROL_STRUCT  DccFsmCtl;
  CH0CCC_CR_DCCFSMCONTROL_STRUCT    CCCDccFsmCtl;
  DATA0CH0_CR_DCCCALCCONTROL_STRUCT DccCalCtl;
  CH0CCC_CR_DCCCALCCONTROL_STRUCT   CCCDccCalCtl;

  Outputs = &MrcData->Outputs;
  DccFsmCtl.Data = 0; //Initialize to avoid "potentially uninitialized variables
  DccCalCtl.Data = 0;
  ByteStart = 0;
  ByteEnd = Outputs->SdramCount;
  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Gear2 = (Outputs->Gear2) ? 1 : 0;
  Safe = 0;
  tPBDMinFm = 1750; // in fs
  if (Inputs->SafeMode) {
    Safe = 0xFFFF;
  }
  TempVar1 = ((1000000000 / 322581) / 2) * ((Outputs->Qclkps * 1000) / 312500);  // scale to fs
  TempVar1 = MrcLog2 (TempVar1) - 12;
  // Basic DCC register configuration needed for DDRIO Init Complete
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
        RankMask = ChannelOut->ValidRankBitMask;
        if (MrcData->Outputs.Lpddr) {
          ByteStart = (MAX_BYTE_IN_LP_CHANNEL * Channel);
          ByteEnd = ByteStart + MAX_BYTE_IN_LP_CHANNEL;
        }
        for (Byte = ByteStart; Byte < ByteEnd; Byte++) {
          Offset = DATA0CH0_CR_DCCFSMCONTROL_REG +
            ((DATA0CH1_CR_DCCFSMCONTROL_REG - DATA0CH0_CR_DCCFSMCONTROL_REG) * Controller) +
            ((DATA1CH0_CR_DCCFSMCONTROL_REG - DATA0CH0_CR_DCCFSMCONTROL_REG) * Byte);
          DccFsmCtl.Data = MrcReadCR (MrcData, Offset);
          DccFsmCtl.Bits.RankEn = 0;
          DccFsmCtl.Bits.LaneEn = 0x2FF;
          DccFsmCtl.Bits.RankMap = (RankMask > 3) ? 2 : (RankMask == 2) ? 1 : 0;
          DccFsmCtl.Bits.DccSamples = MIN (TempVar1, 4);
          DccFsmCtl.Bits.EnRankOvrd = Safe ? 0 : 1;
          DccFsmCtl.Bits.WaitForUpdate = 1;
          DccFsmCtl.Bits.rsvd = 1;
          if (!Inputs->UlxUlt) {
            // P0 Alignment here is the same except it added an additional field.
            // Checking rank mask for bits 0 and 2 set will tell us if we have Rank 0/2 populated and set Restore4Ranks accordingly.
            DccFsmCtl.P0Bits.Restore4Ranks = ((RankMask & 0x5) ? 1 : 0);
          }
          Offset = DATA0CH0_CR_DCCCALCCONTROL_REG +
            ((DATA0CH1_CR_DCCCALCCONTROL_REG - DATA0CH0_CR_DCCCALCCONTROL_REG) * Controller) +
            ((DATA1CH0_CR_DCCCALCCONTROL_REG - DATA0CH0_CR_DCCCALCCONTROL_REG) * Byte);
          DccCalCtl.Data = MrcReadCR (MrcData, Offset);
          DccStepSize = (32000 * Outputs->Qclkps) / 512; // scale to fs, therefore 32000 instead of 32
          DccStepSize = DIVIDEROUND (DccStepSize, tPBDMinFm);
          DccCalCtl.Bits.DccStepSize = DccStepSize;
          DccCalCtl.Bits.LargeChange = 3;
          //Exit for loops by setting conditions to exit
          Byte = ByteEnd;
          Channel = MAX_CHANNEL;
          Controller = MAX_CONTROLLER;
        }
      }
    }
  }
  MrcWriteCrMulticast (MrcData, DATA_CR_DCCFSMCONTROL_REG, DccFsmCtl.Data);
  MrcWriteCrMulticast (MrcData, DATA_CR_DCCCALCCONTROL_REG, DccCalCtl.Data);

  CCCDccCalCtl.Data = MrcReadCR (MrcData, CH0CCC_CR_DCCCALCCONTROL_REG);
  CCCDccCalCtl.Bits.LargeChange = 3;
  MrcWriteCrMulticast (MrcData, CCC_CR_DCCCALCCONTROL_REG, CCCDccCalCtl.Data);

  for (Index = 0; Index < MRC_NUM_CCC_INSTANCES; Index++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DCCFSMCONTROL_REG, CH1CCC_CR_DCCFSMCONTROL_REG, Index);
    CCCDccFsmCtl.Data = 0;
    CCCDccFsmCtl.Bits.DccSamples = DccFsmCtl.Bits.DccSamples;
    CCCDccFsmCtl.Bits.WaitForUpdate = DccFsmCtl.Bits.WaitForUpdate;
    CCCDccFsmCtl.Bits.UpdateTcoComp = 0;
#ifdef CTE_FLAG
    CCCDccFsmCtl.Bits.UpdateTcoComp = 1;
#endif
    CCCDccFsmCtl.Bits.LaneEn = (Gear2) ? 0x2FF : 4;
    MrcWriteCR (MrcData, Offset, CCCDccFsmCtl.Data);
  }
  */
  return mrcSuccess;
}

/**
  PHY init - sequence static_DQEARLY in DQ FUB.
  Programs the following registers:
   - DataControl0/1/2/3
   - DataDqRankXLaneY
   - TxPbdOffset0/1

  @param[in, out] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcPhyInitStaticDqEarly (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcStatus         Status;
  MrcOutput         *Outputs;
  MrcDebug          *Debug;
  MrcDdrType        DdrType;
  INT64             SenseAmpDurationMax;
  INT64             DqsOdtDurationMax;
  INT64             RxClkStgMax;
  UINT32            i;
  UINT32            Gear1;
  UINT32            ResetNumPre;
  UINT32            NumToggles;
  UINT32            RcvEnWaveShape;
  UINT32            RxTogglePreamble;
  UINT32            UiLock;
  UINT32            RdDqsOffset;
  UINT32            RxVrefVal;
  BOOLEAN           Lpddr;
  BOOLEAN           Lpddr4;
  BOOLEAN           Lpddr5;
  BOOLEAN           Ddr5;
  BOOLEAN           Ddr4;
  BOOLEAN           Gear4;
  BOOLEAN           MatchedRx;
  DATA0CH0_CR_DDRCRDATACONTROL0_STRUCT  DataControl0;
  DATA0CH0_CR_DDRCRDATACONTROL1_STRUCT  DataControl1;
  DATA0CH0_CR_DDRCRDATACONTROL2_STRUCT  DataControl2;
  DATA0CH0_CR_DDRCRDATACONTROL3_STRUCT  DataControl3;

  Outputs     = &MrcData->Outputs;
  Debug       = &Outputs->Debug;
  Status      = mrcSuccess;
  DdrType     = Outputs->DdrType;
  Lpddr4      = (DdrType == MRC_DDR_TYPE_LPDDR4);
  Ddr4        = (DdrType == MRC_DDR_TYPE_DDR4);
  Lpddr5      = (DdrType == MRC_DDR_TYPE_LPDDR5);
  Ddr5        = (DdrType == MRC_DDR_TYPE_DDR5);
  Lpddr       = Outputs->Lpddr;
  Gear1       = (Outputs->Gear2) ? 0 : 1;
  Gear4       = FALSE;                      // @todo_adl

  // @todo_adl  Need SAFE values for DataControl0..3

  DataControl0.Data = 0;
  DataControl0.Bits.rxrankmuxdelay_2ndstg = 0; // @todo_adl TGL value is 3
  DataControl0.Bits.vrefpmctrl            = 1;
  DataControl0.Bits.biaspmctrl            = 2;
  DataControl0.Bits.endqsodtparkmode      = 1;
  DataControl0.Bits.endqodtparkmode       = 0;
  DataControl0.Bits.internalclockson      = (Ddr4 || Ddr5) ? 1 : 0; // @todo_adl W/A on ADL A0; HSD link TBD
  DataControl0.Bits.Gear4                 = Gear4;
  DataControl0.Bits.local_gate_d0tx       = 0;                      // @todo_adl check with DE
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATACONTROL0_REG, DataControl0.Data);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "DataControl0: 0x%08X\n", DataControl0.Data);

  MrcGetSetLimits (MrcData, SenseAmpDuration, NULL, &SenseAmpDurationMax, NULL); // SenseAmpDuration and McOdtDuration are the same.
  MrcGetSetLimits (MrcData, DqsOdtDuration,   NULL, &DqsOdtDurationMax,   NULL);

  DataControl1.Data = 0;
  DataControl1.Bits.safemodeenable    = 1;
  DataControl1.Bits.dqodtdelay        = 0;
  DataControl1.Bits.dqodtduration     = (UINT32) SenseAmpDurationMax;
  DataControl1.Bits.dqsodtdelay       = 0;
  DataControl1.Bits.dqsodtduration    = (UINT32) DqsOdtDurationMax;
  DataControl1.Bits.senseampdelay     = 0;
  DataControl1.Bits.senseampduration  = (UINT32) SenseAmpDurationMax;
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATACONTROL1_REG, DataControl1.Data);

  MrcGetSetLimits (MrcData, GsmIocRxClkStg, NULL, &RxClkStgMax, NULL);

  DataControl2.Data = 0;
  DataControl2.Bits.datainvertnibble  = 0;        // @todo_adl shall we use this feature ?
  DataControl2.Bits.gear1             = Gear1;
  DataControl2.Bits.wrpreamble        = MrcGetWpre (MrcData) - 1;
  DataControl2.Bits.dbimode           = 1;
  DataControl2.Bits.txdqsrankmuxdelay_2nd_stage_offset  = 2;
  DataControl2.Bits.txrankmuxdelay_2nd_stage_offset     = 2;
  DataControl2.Bits.disabletxdqs      = Lpddr5;
  DataControl2.Bits.rxburstlen        = Lpddr4 ? 3 : 1;  // 0: BL4, 1: BL8, 2: BL10, 3: BL16, 4: BL18
  DataControl2.Bits.rxclkstgnum       = (UINT32) RxClkStgMax;
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATACONTROL2_REG, DataControl2.Data);

  MatchedRx = (Outputs->RxMode == MrcRxModeMatchedN);

  UiLock            = 0;  // @todo_adl
  RdDqsOffset       = 0;  // @todo_adl
  ResetNumPre       = 0;
  RcvEnWaveShape    = 0;
  RxTogglePreamble  = 0;
  if (!MatchedRx) {
    RdDqsOffset   = 0;                            // @todo_adl
    NumToggles    = 2;
    ResetNumPre   = (NumToggles / 2) + 4;                                                 // matched_rx ? 0 : (num_toggle / 2) + 4
    RxTogglePreamble = (NumToggles / 2) - ((UiLock + 1 - RdDqsOffset * 2 == 1) ? 0 : 1);  // matched_rx? 0 : (num_toggle/2 - ( (uilock+1 - rd_dqs_offset * 2 == 1) ? 0 : 1) )
    if (Lpddr5) {
      RcvEnWaveShape = Gear1 ? 3 : (Gear4 ? 1 : 2);
    } else { // DDR5
      if (Gear4) {
        RcvEnWaveShape = 1;
      } else if (NumToggles == 4) {
        RcvEnWaveShape  = Gear1 ? 3 : 2;
      }
    }
  }

  // 0.25 * Vddq for LP4/5  (Vss termination)
  // 0.75 * Vddq for DDR4/5 (Vddq termination)
  // Later in PHY init we will program more precise value according to DimmRon, number of DIMMs / ranks etc.
  RxVrefVal = Lpddr ? (512 / 4) : (512 * 3) / 4;

  DataControl3.Data = 0;
  DataControl3.Bits.usedefaultrdptrcalc       = 1;
  DataControl3.Bits.rstnumpre                 = ResetNumPre;
  DataControl3.Bits.rxtogglepreamble          = RxTogglePreamble;
  DataControl3.Bits.rcvenwaveshape            = RcvEnWaveShape;
  DataControl3.Bits.rxreadpointer             = 0;
  DataControl3.Bits.rxvref                    = RxVrefVal;
  DataControl3.Bits.wr1p5tckpostamble_toggle  = 0;
  DataControl3.Bits.wr1p5tckpostamble_enable  = 0;                        // @todo_adl Might need this for DDR5
  DataControl3.Bits.lpddr5                    = Lpddr5;
  DataControl3.Bits.lpddr                     = Lpddr;
  DataControl3.Bits.dram_rddqsoffset          = Ddr5 ? RdDqsOffset : 0;   // (rd_preamble == 0) ? 'h0 : (rd_preamble == 2)  ? 'h0 : rd_dqs_offset
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATACONTROL3_REG, DataControl3.Data);

  for (i = 0; i < MAX_RANK_IN_CHANNEL * MAX_BITS * 4; i = i + 4) {
    MrcWriteCrMulticast (MrcData, (DATA_CR_DDRDATADQRANK0LANE0_REG + i), 0);
  }

  MrcWriteCrMulticast (MrcData, DATA_CR_TXPBDOFFSET0_REG, 0);
  MrcWriteCrMulticast (MrcData, DATA_CR_TXPBDOFFSET1_REG, 0);

  return Status;
}

/**
  PHY init - sequence static_DIG1 in VTT FUB.
  Programs the following registers:
   - DDRVTT_VttGenControl
   - DDRVTT_VttCompOffset
   - DDRVTT_VttCompOffset2
   - DDRVTT_DDREARLY_RSTRDONE

  @param[in, out] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcPhyInitVttStaticDig1 (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcStatus         Status;
  MrcOutput         *Outputs;
  MrcDdrType        DdrType;
  BOOLEAN           Lpddr4;
  BOOLEAN           Lpddr5;
  BOOLEAN           Ddr4;
  DDRVTT0_CR_DDRCRVTTGENCONTROL_STRUCT  VttGenControl;
  DDRVTT0_CR_DDRCRVTTCOMPOFFSET_STRUCT  VttCompOffset;
  DDRVTT0_CR_DDRCRVTTCOMPOFFSET2_STRUCT VttCompOffset2;
  DDRVTT0_CR_DDREARLY_RSTRDONE_STRUCT   DdrEarlyRestoreDone;

  Status      = mrcSuccess;
  Outputs     = &MrcData->Outputs;
  DdrType     = Outputs->DdrType;
  Lpddr4      = (DdrType == MRC_DDR_TYPE_LPDDR4);
  Lpddr5      = (DdrType == MRC_DDR_TYPE_LPDDR5);
  Ddr4        = (DdrType == MRC_DDR_TYPE_DDR4);

  VttGenControl.Data = 0;
  /* TGL code :
  TempVar1 = (382 * VttTargetV);
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv) - 32;
  // Assumes VttTargetV in LPDDR is 150mV based on the system studies.
  VttGenControl.Bits.Target = TempVar2;

  TempVar1 = (382 * PANICV0);
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv);
  VttGenControl.Bits.Panic0 = TempVar2;

  TempVar1 = (382 * PANICV1);
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv);
  VttGenControl.Bits.Panic1 = TempVar2;

  VttGenControl.Bits.Spare1 = SAFE ? 1 : 0;
  // VttGenControl.Bits.DisSensorPwrDn = SAFE ? 1 : 0; // Field no longer exists
  VttGenControl.Bits.EnDacPM = (A0 || SAFE) ? 0 : 2;
// @todo_adl  VttGenControl.Bits.DdrCRForceODTOn = (DataControl0.Bits.OdtForceQDrvEn || DataControl0.Bits.DdrCRForceODTOn) && VttGenControl.Bits.EnVttOdt;
  //This field is programmed in SetVtt function
 // VttGenControl.Bits.EnVttOdt = (DataControl0.OdtMode == 2 * Vtt);   // (DDRControl5.OdtMode == 2 (VTT))
  TempVar1 = (Ddr4 || Ddr5) ? 10 : 16; // MaxMin( RndUp( (DDR4or5?10nS:16nS)/tQCLK/8 )-1, 0, 7)
  TempVar2 = DIVIDEROUND (TempVar1, QclkPs);
  TempVar3 = DIVIDEROUND (TempVar2, 8) - 1;
  VttGenControl.Bits.WakeUpDelay = TempVar3;

  TempVar1 = (60 / RefClkPs) - 1;
  TempVar2 = MrcLog2(TempVar1);
  VttGenControl.Bits.LockTimer = TempVar2;

  VttGenControl.Bits.AckOffset = 1;
  VttGenControl.Bits.EnVttOdt = 1; */

  VttGenControl.Bits.Target = Lpddr4 ? 0x26 : (Lpddr5 ? 0x29 : 0xA0);   // @todo_adl Use TGL formula ?
  VttGenControl.Bits.Panic0 = Ddr4 ? 8 : 9;                             // @todo_adl Use TGL formula ?
  VttGenControl.Bits.Panic1 = Ddr4 ? 13 : 14;                           // @todo_adl Use TGL formula ?
  VttGenControl.Bits.WakeUpDelay  = 4;                                  // @todo_adl Use TGL formula ?
  VttGenControl.Bits.LockTimer    = 0;                                  // @todo_adl Use TGL formula ?
  VttGenControl.Bits.AckOffset    = 0;                                  // @todo_adl TGL sets this to 1
  VttGenControl.Bits.EnVttOdt     = 1;                                  // @todo_adl Only if we use Vtt termination ?
  MrcWriteCrMulticast (MrcData, DDRVTT_CR_DDRCRVTTGENCONTROL_REG, VttGenControl.Data);

  /* TGL code:
  VttCompOffset.Data = MrcReadCR (MrcData, DDRVTT0_CR_DDRCRVTTCOMPOFFSET_REG);
  VttCompOffset2.Data = 0;
  VttCompOffset2.Bits.PanicLo0UsePmos = (VsxHiTargetMv + (-PANICV0) + VttCompOffset.Bits.PanicLo0CompOfst) < 375;
  VttCompOffset2.Bits.PanicLo1UsePmos = (VsxHiTargetMv + (-PANICV1) + VttCompOffset.Bits.PanicLo1CompOfst) < 375;
  VttCompOffset2.Bits.PanicHi0UsePmos = (VsxHiTargetMv + (PANICV0) + VttCompOffset.Bits.PanicHi0CompOfst)  < 375;
  VttCompOffset2.Bits.PanicHi1UsePmos = (VsxHiTargetMv + (PANICV1) + VttCompOffset.Bits.PanicHi1CompOfst)  < 375;
  */

  VttCompOffset.Data = 0;
  MrcWriteCrMulticast (MrcData, DDRVTT_CR_DDRCRVTTCOMPOFFSET_REG, VttCompOffset.Data);

  VttCompOffset2.Data = 0;
  MrcWriteCrMulticast (MrcData, DDRVTT_CR_DDRCRVTTCOMPOFFSET2_REG, VttCompOffset2.Data);  // @todo_adl Use TGL formula ?

  DdrEarlyRestoreDone.Data = 0;
  DdrEarlyRestoreDone.Bits.early_restoredone = 1;
  MrcWriteCrMulticast (MrcData, DDRVTT_CR_DDREARLY_RSTRDONE_REG, DdrEarlyRestoreDone.Data);

  return Status;
}

/**
  PHY init - sequence static_compinit in COMP FUB.
  Programs 46 registers in the COMP Fub.

  @param[in, out] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcPhyInitCompStaticCompInit (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcStatus         Status;
  MrcOutput         *Outputs;
  MrcDdrType        DdrType;
  BOOLEAN           Lpddr;
  BOOLEAN           Lpddr5;
  BOOLEAN           Ddr4;
  BOOLEAN           Ddr5;
  BOOLEAN           Gear2;
  BOOLEAN           Gear4;
  UINT32            CompVref;
  DDRPHY_COMP_CR_RCOMPCTRL1_STRUCT                RcompCtrl1;
  DDRPHY_COMP_CR_RCOMPCTRL_STRUCT                 RcompCtrl;
  DDRPHY_COMP_CR_RCOMP_EXTCOMP_STRUCT             RcompExtComp;
  DDRPHY_COMP_CR_RCOMP_RELCOMP_STRUCT             RcompRelComp;
  DDRPHY_COMP_CR_PNCCOMP_CTRL0_STRUCT             PanicCompCtrl0;
  DDRPHY_COMP_CR_PNCCOMP_CTRL1_STRUCT             PanicCompCtrl1;
  DDRPHY_COMP_CR_PNCCOMP_STRUCT                   PanicComp;
  DDRPHY_COMP_CR_SCOMP_STRUCT                     ScompReg;
  DDRPHY_COMP_CR_RXDQSCOMP_RXCODE_STRUCT          RxDqsCompRxCode;
  DDRPHY_COMP_CR_RXDQSCOMP_CMN_CTRL0_STRUCT       RxDqsCompCmnCtrl0;
  DDRPHY_COMP_CR_RXDQSCOMP_CMN_CTRL1_STRUCT       RxDqsCompCmnCtrl1;
  DDRPHY_COMP_CR_RXDQSCOMP_CMN_CTRL1_NEW_STRUCT   RxDqsCompCmnCtrl1New;
  DDRPHY_COMP_CR_RXDQSCOMP_ENABLE_STRUCT          RxDqsCompEnable;
  DDRPHY_COMP_CR_DLLCOMP_LDOCTRL1_STRUCT          DllCompLdoCtrl1;
  DDRPHY_COMP_CR_DLLCOMP_LDOCTRL1_NEW_STRUCT      DllCompLdoCtrl1New;
  DDRPHY_COMP_CR_DLLCOMP_PIDELAY_STRUCT           DllCompPiDelay;
  DDRPHY_COMP_CR_DLLCOMP_PICTRL_STRUCT            DllCompPiCtrl;
  DDRPHY_COMP_CR_DLLCOMP_DLLCTRL_STRUCT           DllCompDllCtrl;
  DDRPHY_COMP_CR_DLLCOMP_VCDLCTRL_STRUCT          DllCompVcdlCtrl;
  DDRPHY_COMP_CR_DLLCOMP_VDLLCTRL_STRUCT          DllCompVdllCtrl;
  DDRPHY_COMP_CR_DLLCOMP_SWCAPCTRL_STRUCT         DllCompSwCapCtrl;
  DDRPHY_COMP_CR_DDREARLYCR_STRUCT                DdrEarlyCr;
  DDRPHY_COMP_CR_VSSHICOMP_CTRL0_STRUCT           VsshiCompCtrl0;
  DDRPHY_COMP_CR_VSSHICOMP_CTRL1_STRUCT           VsshiCompCtrl1;
  DDRPHY_COMP_CR_VSSHICOMP_FFCOMP_STRUCT          VsshiCompFfComp;
  DDRPHY_COMP_CR_VSXHIOPAMPCH0_CTRL_STRUCT        VsxhiOpAmpCh0Ctrl;
  DDRPHY_COMP_CR_VSXHIOPAMPCH1_CTRL_STRUCT        VsxhiOpAmpCh1Ctrl;
  DDRPHY_COMP_CR_COMPDLLWL_STRUCT                 CompDllWl;
  DDRPHY_COMP_CR_TXCONTROL_STRUCT                 TxControl;
  DDRPHY_COMP_CR_VIEWDIGTX_CTRL_STRUCT            ViewDigTxCtrl;
  DDRPHY_COMP_CR_VIEWDIGTX_SEL_STRUCT             ViewDigTxSel;
  DDRPHY_COMP_CR_VIEWANA_CTRL_STRUCT              ViewAnaCtrl;
  DDRPHY_COMP_CR_VIEWANA_CBBSEL_STRUCT            ViewAnaCbbSel;
  DDRPHY_COMP_CR_COMP_IOLVRFFCTL_STRUCT           CompIoLvrFfCtl;
  DDRPHY_COMP_CR_MISCCOMPCODES_STRUCT             MiscCompCodes;
  DDRPHY_COMP_CR_COMPDLLSTRTUP_STRUCT             CompDllStartup;
  DDRPHY_COMP_CR_DDRCRDATACOMP0_STRUCT            DataComp0;
  DDRPHY_COMP_CR_DDRCRCMDCOMP_STRUCT              CmdComp;
  DDRPHY_COMP_CR_DDRCRCLKCOMP_STRUCT              ClkComp;
  DDRPHY_COMP_CR_DDRCRCTLCOMP_STRUCT              CtlComp;
  DDRPHY_COMP_CR_DDRCRCOMPCTL0_STRUCT             CompCtl0;
  DDRPHY_COMP_CR_DDRCRCOMPCTL1_STRUCT             CompCtl1;
  DDRPHY_COMP_CR_DDRCRCOMPCTL2_STRUCT             CompCtl2;
  DDRPHY_COMP_CR_DDRCRCOMPCTL3_STRUCT             CompCtl3;
  DDRPHY_COMP_CR_DDRCRCOMPCTL4_STRUCT             CompCtl4;
  DDRPHY_COMP_CR_SCOMPCTL_STRUCT                  ScompCtl;

  Status      = mrcSuccess;
  Outputs     = &MrcData->Outputs;
  DdrType     = Outputs->DdrType;
  Lpddr       = Outputs->Lpddr;
  Lpddr5      = (DdrType == MRC_DDR_TYPE_LPDDR5);
  Ddr4        = (DdrType == MRC_DDR_TYPE_DDR4);
  Ddr5        = (DdrType == MRC_DDR_TYPE_DDR5);
  Gear2       = Outputs->Gear2;
  Gear4       = FALSE;                      // @todo_adl

  RcompCtrl1.Data = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RCOMPCTRL1_REG, RcompCtrl1.Data);

  RcompCtrl.Data = 0;
  RcompCtrl.Bits.rcomp_cmn_rcompshrtxpupusevcciog = 0;
  RcompCtrl.Bits.rcomp_cmn_rcompshrtxpdnusevcciog = !Lpddr; // @todo_adl Also for Lpddr4 NMOS pull-up: txpdnusevcciog = (DDR4 || DDR5 || ennmospup) & ~ddr5hpmode
  RcompCtrl.Bits.rcomp_cmn_rcompvddqvsshibyp      = Lpddr;
  RcompCtrl.Bits.rcomp_cmn_rcomplocalvsshibyp     = Lpddr;  // localvsshibyp = (LP4 || LP5) || ddr5hpmode
  RcompCtrl.Bits.rcomp_cmn_rcompglobalvsshibyp    = Lpddr5;
  RcompCtrl.Bits.rcompexttxdrvusevdd2             = Lpddr;
  RcompCtrl.Bits.swcaprxsampnum_cycles            = 1;
  RcompCtrl.Bits.swcaprxeval_cycles               = 1;
  RcompCtrl.Bits.swcaprxprech_cycles              = 3;
  RcompCtrl.Bits.swcaprxrst_cycles                = 3;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RCOMPCTRL_REG, RcompCtrl.Data);

  RcompExtComp.Data = 0;
  RcompExtComp.Bits.rcompextvttodten            = !Lpddr;
  RcompExtComp.Bits.rcompextvrefen              = 1;
  RcompExtComp.Bits.rcompextvrefcode            = 0;      // [8 bit] @todo_adl Should be based on RCOMP target ?
  RcompExtComp.Bits.rcomp_cmn_rcompextstatlegen = 1;      // @todo_adl Should be based on RCOMP target ?
  RcompExtComp.Bits.rcompextrxrstb              = 0;
  RcompExtComp.Bits.rcomp_cmn_rcompextennmospup = Lpddr;  // @todo_adl Only if we use NMOS pull-up ?
  RcompExtComp.Bits.rcompextdqpdnen             = 0;
  RcompExtComp.Bits.rcompextdqen                = 0;
  RcompExtComp.Bits.rcompextcodepuplive         = 0;      // [6 bit]
  RcompExtComp.Bits.rcompextcccpdnen            = 0;
  RcompExtComp.Bits.rcompextcccen               = 0;
  RcompExtComp.Bits.rcompextampen               = 0;
  RcompExtComp.Bits.extswcaprxcmpen             = 0;
  RcompExtComp.Bits.rcompextrxrstb              = 0;
  RcompExtComp.Bits.rcompextrxrstb              = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RCOMP_EXTCOMP_REG, RcompExtComp.Data);

  RcompRelComp.Data = 0;
  RcompRelComp.Bits.rcomprelvrefen              = 1;
  RcompRelComp.Bits.rcomprelvrefcode            = 0;    // [8 bit]
  RcompRelComp.Bits.rcomprelrxrstb              = 0;
  RcompRelComp.Bits.rcomp_cmn_rcomprelennmospup = Lpddr;
  RcompRelComp.Bits.rcomprelcodepup             = 0;    // [6 bit]
  RcompRelComp.Bits.rcomprelcodepdnlive         = 0;
  RcompRelComp.Bits.rcomprelampen               = 0;
  RcompRelComp.Bits.rcomprelpupstatlegen        = 1;
  RcompRelComp.Bits.rcomprelpdnstatlegen        = 1;
  RcompRelComp.Bits.relswcaprxcmpen             = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RCOMP_RELCOMP_REG, RcompRelComp.Data);

  PanicCompCtrl0.Data = 0;
  PanicCompCtrl0.Bits.pnccomp_cmn_vrefvddqvsshibyp    = Lpddr;
  PanicCompCtrl0.Bits.vrefusevcciog                   = 1;
  PanicCompCtrl0.Bits.rloadnumsegs                    = 8;
  PanicCompCtrl0.Bits.localvsshibyp                   = Lpddr;
  PanicCompCtrl0.Bits.pnccomp_cmn_globalvsshibyp      = Lpddr5;
  PanicCompCtrl0.Bits.swcaprxeval_cycles              = 1;
  PanicCompCtrl0.Bits.swcaprxprech_cycles             = 3;
  PanicCompCtrl0.Bits.swcaprxrst_cycles               = 3;
  PanicCompCtrl0.Bits.twovtcompen                     = !Lpddr;
  PanicCompCtrl0.Bits.ddrmode                         = !Lpddr;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_PNCCOMP_CTRL0_REG, PanicCompCtrl0.Data);

  PanicCompCtrl1.Data = 0;
  PanicCompCtrl1.Bits.pnccomp_cmn_txcccstatlegen      = 1;
  PanicCompCtrl1.Bits.pnccomp_cmn_txcccpupusevcciog   = 0;            // 1 if Lpddr and NMOS pull-up ?
  PanicCompCtrl1.Bits.pnccomp_cmn_txcccpdnusevcciog   = Ddr4 || Ddr5; // Also if Lpddr and NMOS pull-up ?
  PanicCompCtrl1.Bits.txcccennmospup                  = 0;            // 1 if Lpddr and NMOS pull-up ?
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_PNCCOMP_CTRL1_REG, PanicCompCtrl1.Data);

  PanicComp.Data = 0;
  PanicComp.Bits.pnccomp_code2vt_vrefcode = 0x40;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_PNCCOMP_REG, PanicComp.Data);

  ScompReg.Data = 0;
  ScompReg.Bits.scomp_cmn_slewstatlegen   = 1;
  ScompReg.Bits.scomp_cmn_refclkphasesel  = 1;    // @todo_adl 0 - cycle lock, 1 - phase lock. Trained in SCOMP init ?
  ScompReg.Bits.cmddlytapselstage         = 0xF;  // @todo_adl Number of delay line cells. Trained in SCOMP init ?
  ScompReg.Bits.scomp_cmn_refclk2cyclesel = 1;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_SCOMP_REG, ScompReg.Data);

  RxDqsCompRxCode.Data = 0;
  RxDqsCompRxCode.Bits.dllcomp_cmn_picoderxdqsnref = 0; // @todo_adl  What is the formula ?
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RXDQSCOMP_RXCODE_REG, RxDqsCompRxCode.Data);

  RxDqsCompCmnCtrl0.Data = 0;
  RxDqsCompCmnCtrl0.Bits.rxdqscomp_cmn_dqspcomplockui = 0;    // @todo_adl  What is the formula ?
  RxDqsCompCmnCtrl0.Bits.rxdqscomp_cmn_dqsncomplockui = 0;    // @todo_adl  What is the formula ?
  RxDqsCompCmnCtrl0.Bits.rxdqscomp_cmn_accoupvrefsel  = 0x80;
  RxDqsCompCmnCtrl0.Bits.dllcomp_cmn_wlrdacholden     = 1;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RXDQSCOMP_CMN_CTRL0_REG, RxDqsCompCmnCtrl0.Data);

  RxDqsCompCmnCtrl1New.Data = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RXDQSCOMP_CMN_CTRL1_NEW_REG, RxDqsCompCmnCtrl1New.Data);

  RxDqsCompEnable.Data = 0;
  RxDqsCompEnable.Bits.rxdqscomp_cmn_rxbiasvrefsel  = 0x5;    // @todo_adl  What is the formula ?
  RxDqsCompEnable.Bits.rxdqscomp_cmn_rxbiasicomp    = 0x8;    // @todo_adl  What is the formula ?
  RxDqsCompEnable.Bits.rxdqscomp_cmn_rxrloadcomp    = 0xE;    // @todo_adl  What is the formula ?
  RxDqsCompEnable.Bits.rxdqscomp_cmn_rxampoffset    = 0;      // @todo_adl  What is the formula ?
  RxDqsCompEnable.Bits.rxdqscomp_cmn_rxlpddrmode    = !Lpddr;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RXDQSCOMP_ENABLE_REG, RxDqsCompEnable.Data);

  DllCompLdoCtrl1New.Data = 0;
  DllCompLdoCtrl1New.Bits.dllcomp_cmn_rxdqmindlypat     = 0xAA;
  DllCompLdoCtrl1New.Bits.dllcomp_cmn_ldofbdivsel       = 1;
  DllCompLdoCtrl1New.Bits.dllcomp_cmn_vctlcompoffsetcal = 0x10;
  DllCompLdoCtrl1New.Bits.dllcomp_cmn_ldoffreadsegen    = 3;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_LDOCTRL1_NEW_REG, DllCompLdoCtrl1New.Data);

  DllCompPiDelay.Data = 0;
  DllCompPiDelay.Bits.dllcomp_cmn_picoderxdqspref = 0;  // @todo_adl Formula ?
  DllCompPiDelay.Bits.dllcomp_cmn_picoderxdqsd0   = 0;
  DllCompPiDelay.Bits.dllcomp_cmn_picodemisc      = 0;
  DllCompPiDelay.Bits.dllcomp_cmn_picodedcs       = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_PIDELAY_REG, DllCompPiDelay.Data);

  DllCompPiCtrl.Data = 0;
  DllCompPiCtrl.Bits.dllcomp_cmn_picoderxldoffph1 = 0x40;
  DllCompPiCtrl.Bits.dllcomp_cmn_gear1en          = !Gear2 && !Gear4;
  DllCompPiCtrl.Bits.dllcomp_cmn_gear2en          = Gear2;
  DllCompPiCtrl.Bits.dllcomp_cmn_gear4en          = Gear4;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_PICTRL_REG, DllCompPiCtrl.Data);

  DllCompDllCtrl.Data = 0;
  DllCompDllCtrl.Bits.dllcomp_cmn_turboonstartup  = 1;
  DllCompDllCtrl.Bits.dllcomp_cmn_turbocaptrim    = 0;    // @todo_adl Formula ?
  DllCompDllCtrl.Bits.dllcomp_cmn_phsdrvprocsel   = 7;
  DllCompDllCtrl.Bits.dllcomp_cmn_mindlybwsel     = 0x20; // @todo_adl Formula ?
  DllCompDllCtrl.Bits.dllcomp_cmn_mdllengthsel    = Gear2 ? 1 : (Gear4 ? 2 : 0);  // G1: 0, G2: 1, G4: 2
  DllCompDllCtrl.Bits.mdllen                      = 0;    // @todo_adl
  DllCompDllCtrl.Bits.dllcomp_cmn_bonus           = 0;    // @todo_adl There is various functionality in these 8 bits
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_DLLCTRL_REG, DllCompDllCtrl.Data);

  DllCompVcdlCtrl.Data = 0;
  DllCompVcdlCtrl.Bits.vctldaccode                = 0x187;
  DllCompVcdlCtrl.Bits.dllcomp_cmn_vctlcaptrim    = 0;    // @todo_adl Formula ?
  DllCompVcdlCtrl.Bits.vcdlbwsel                  = 0x20; // @todo_adl Formula ? Same as DllCompDllCtrl.Bits.dllcomp_cmn_mindlybwse ?
  DllCompVcdlCtrl.Bits.dllcomp_cmn_vcdldcccode    = 8;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_VCDLCTRL_REG, DllCompVcdlCtrl.Data);

  DllCompVdllCtrl.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_VDLLCTRL_REG);    // Keep reset value ?
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_VDLLCTRL_REG, DllCompVdllCtrl.Data);

  DllCompSwCapCtrl.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_SWCAPCTRL_REG);  // Keep reset value ?
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_SWCAPCTRL_REG, DllCompSwCapCtrl.Data);

  VsshiCompCtrl0.Data = 0;
  VsshiCompCtrl0.Bits.vsshiffleakstlegen                = 1;
  VsshiCompCtrl0.Bits.txvsshiffstlegen                  = 1;
  VsshiCompCtrl0.Bits.rxvsshiffstrobestlegen            = 1;
  VsshiCompCtrl0.Bits.vsshicomp_cmn_ffleakrodivsel      = 1;
  VsshiCompCtrl0.Bits.vsshicomp_cmn_encompleaker        = 1;
  VsshiCompCtrl0.Bits.compffrxendata                    = Ddr5;
  VsshiCompCtrl0.Bits.compennmospup                     = 0;            // 1 if Lpddr and NMOS pull-up ?
  VsshiCompCtrl0.Bits.vsshicomp_cmn_complocalvsshibyp   = Lpddr;
  VsshiCompCtrl0.Bits.compleaken                        = 1;
  VsshiCompCtrl0.Bits.vsshicomp_cmn_compglobalvsshibyp  = Lpddr5;
  VsshiCompCtrl0.Bits.vsshicomp_cmn_compvddqvsshibyp    = Lpddr;
  VsshiCompCtrl0.Bits.vsshicomp_cmn_comptxpupusevcciog  = 0;            // 1 if Lpddr and NMOS pull-up ?
  VsshiCompCtrl0.Bits.vsshicomp_cmn_comptxpdnusevcciog  = Ddr4 || Ddr5; // Also if Lpddr and NMOS pull-up ?
  VsshiCompCtrl0.Bits.vsshicomp_cmn_accoupvrefsel       = 0x10;
  VsshiCompCtrl0.Bits.vsshicomp_cmn_accoupselswing      = 3;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHICOMP_CTRL0_REG, VsshiCompCtrl0.Data);

  VsshiCompCtrl1.Data = 0;
  VsshiCompCtrl1.Bits.vsshicomp_cmn_ffleakrofreqadj = 0x5;
  VsshiCompCtrl1.Bits.swcaprxrst_cycles             = 0x3;
  VsshiCompCtrl1.Bits.swcaprxprech_cycles           = 0x3;
  VsshiCompCtrl1.Bits.swcaprxeval_cycles            = 0x1;
  VsshiCompCtrl1.Bits.swcaprxsampnum_cycles         = 0x1;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHICOMP_CTRL1_REG, VsshiCompCtrl1.Data);

  VsshiCompFfComp.Data = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHICOMP_FFCOMP_REG, VsshiCompFfComp.Data);  // All zero values

  VsxhiOpAmpCh0Ctrl.Data = 0;
  VsxhiOpAmpCh0Ctrl.Bits.ch0_biasvrefen                   = 0x1;
  VsxhiOpAmpCh0Ctrl.Bits.vsxhiopamp_ch0_opampbiasctl      = 0x2;
  VsxhiOpAmpCh0Ctrl.Bits.vsxhiopamp_ch0_opampbsxhibwctl   = 0x8;
  VsxhiOpAmpCh0Ctrl.Bits.ch0_opampen                      = 0x1;
  VsxhiOpAmpCh0Ctrl.Bits.vsxhiopamp_ch0_opamptargetvref   = 0x20;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSXHIOPAMPCH0_CTRL_REG, VsxhiOpAmpCh0Ctrl.Data);

  VsxhiOpAmpCh1Ctrl.Data = 0;
  VsxhiOpAmpCh1Ctrl.Bits.vsxhiopamp_cmn_rxrloadcomp       = 0xE;
  VsxhiOpAmpCh1Ctrl.Bits.ch1_biasvrefen                   = 0x1;
  VsxhiOpAmpCh1Ctrl.Bits.vsxhiopamp_ch1_opampbiasctl      = 0x2;
  VsxhiOpAmpCh1Ctrl.Bits.vsxhiopamp_ch1_opampbsxhibwctl   = 0x8;
  VsxhiOpAmpCh1Ctrl.Bits.ch1_opampen                      = 0x1;
  VsxhiOpAmpCh1Ctrl.Bits.vsxhiopamp_ch1_opamptargetvref   = 0x20;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSXHIOPAMPCH1_CTRL_REG, VsxhiOpAmpCh1Ctrl.Data);

  CompDllWl.Data = 0;
  CompDllWl.Bits.vctl_compperiodicen  = 0x1;
  CompDllWl.Bits.num_samples          = 0x1;
  CompDllWl.Bits.ch0_pwrgood          = 0x1;
  CompDllWl.Bits.ch1_pwrgood          = 0x1;
  CompDllWl.Bits.wl_step_size         = 0x1;
  CompDllWl.Bits.wl_sample_wait       = 0x8;
  CompDllWl.Bits.vctlcompareen_dly    = 0x6;
  CompDllWl.Bits.vctldaccode          = 0x100;
  CompDllWl.Bits.wlcompen             = 0x1;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_COMPDLLWL_REG, CompDllWl.Data);

  TxControl.Data = 0;
  TxControl.Bits.dllcomp_cmn_dccrangesel          = 0x3;
  TxControl.Bits.txpbd_cmn_txdccrangesel          = 0x3;
  TxControl.Bits.bonus11                          = 0x5;
  TxControl.Bits.allcomps_cmn_txshrennbiasboost1  = Ddr5;  // ennbiasboost = (TECH == 'DDR5')
  TxControl.Bits.txpbd_cmn_dccctrlcodeph0         = 0x80;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_TXCONTROL_REG, TxControl.Data);

  ViewDigTxCtrl.Data = 0;
  ViewDigTxCtrl.Bits.viewdigtx_cmn_txrcompdrvdn   = 0x2F;
  ViewDigTxCtrl.Bits.viewdigtx_cmn_txrcompdrvup   = 0x2F;
  ViewDigTxCtrl.Bits.viewdigtx_cmn_txtcocomp      = 0x20;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VIEWDIGTX_CTRL_REG, ViewDigTxCtrl.Data);

  ViewDigTxSel.Data = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VIEWDIGTX_SEL_REG, ViewDigTxSel.Data);      // All zero values

  ViewAnaCtrl.Data = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VIEWANA_CTRL_REG, ViewAnaCtrl.Data);        // All zero values

  ViewAnaCbbSel.Data = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VIEWANA_CBBSEL_REG, ViewAnaCbbSel.Data);    // All zero values

  CompIoLvrFfCtl.Data = 0;
  CompIoLvrFfCtl.Bits.iolvrseg_cmn_enffleg    = 0x1;
  CompIoLvrFfCtl.Bits.iolvrseg_cmn_enlvrleg   = 0x1;
  CompIoLvrFfCtl.Bits.iolvrseg_cmn_codeffleg  = 0xF;
  CompIoLvrFfCtl.Bits.iolvrseg_cmn_codelvrleg = 0xF;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_COMP_IOLVRFFCTL_REG, CompIoLvrFfCtl.Data);

  RxDqsCompCmnCtrl1.Data = 0;
  RxDqsCompCmnCtrl1.Bits.rxdqscomp_cmn_rxsdlbwsel       = 0x20;   // @todo_adl Formula ?
  RxDqsCompCmnCtrl1.Bits.rxdqscomp_cmn_rxphsdrvprocsel  = 0x7;
  RxDqsCompCmnCtrl1.Bits.rxdqscomp_cmn_rxmctlecap       = 0x1;
  RxDqsCompCmnCtrl1.Bits.rxdqscomp_cmn_rxmctleeq        = 0x8;
  RxDqsCompCmnCtrl1.Bits.rxdqscomp_cmn_rxmctleres       = 0x1;
  RxDqsCompCmnCtrl1.Bits.rxdqscomp_cmn_rxsdllengthsel   = 0x3;
  RxDqsCompCmnCtrl1.Bits.rxdqscomp_cmn_rxtailctl        = 0x3;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_RXDQSCOMP_CMN_CTRL1_REG, RxDqsCompCmnCtrl1.Data);

  DllCompLdoCtrl1.Data = 0;
  DllCompLdoCtrl1.Bits.dll_pien_timer_val           = 0x1E0;
  DllCompLdoCtrl1.Bits.dllcomp_cmn_ldonbiasctrl     = 0x7;
  DllCompLdoCtrl1.Bits.dllcomp_cmn_ldonbiasvrefcode = 0x1B;
  DllCompLdoCtrl1.Bits.dllcomp_cmn_ldopbiasctrl     = 0x3;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DLLCOMP_LDOCTRL1_REG, DllCompLdoCtrl1.Data);

  MiscCompCodes.Data = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_MISCCOMPCODES_REG, MiscCompCodes.Data); // All zero values; do we even need to write this ?

  CompDllStartup.Data = 0;
  CompDllStartup.Bits.dll_lock_timer        = 0x1E0;
  CompDllStartup.Bits.dll_disable_timer_val = 0x1;
  CompDllStartup.Bits.dllen_timer_val       = 0x68;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_COMPDLLSTRTUP_REG, CompDllStartup.Data);

  // @todo_adl these are COMP codes, no need to program
  DataComp0.Data = 0;
  DataComp0.Bits.VssHiFF_dq   = 0x20;
  DataComp0.Bits.RcompOdtDown = 0x20;
  DataComp0.Bits.RcompOdtUp   = 0x20;
  DataComp0.Bits.RcompDrvDown = 0x20;
  DataComp0.Bits.RcompDrvUp   = 0x20;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, DataComp0.Data);

  CmdComp.Data = 0;
  CmdComp.Bits.VssHiFF_cmd  = 0x20;
  CmdComp.Bits.ScompCmd     = 0x80;
  CmdComp.Bits.RcompDrvDown = 0x20;
  CmdComp.Bits.RcompDrvUp   = 0x20;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCMDCOMP_REG, CmdComp.Data);

  ClkComp.Data = 0;
  ClkComp.Bits.VssHiFF_clk  = 0x20;
  ClkComp.Bits.ScompClk     = 0x20;
  ClkComp.Bits.RcompDrvDown = 0x20;
  ClkComp.Bits.RcompDrvUp   = 0x20;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCLKCOMP_REG, ClkComp.Data);

  CtlComp.Data = 0;
  CtlComp.Bits.CkeCsUp      = 0x0;          // CkeCS up code for LPDDR technology   @todo_adl Should be indeed zero ?
  CtlComp.Bits.VssHiFF_ctl  = 0x20;
  CtlComp.Bits.ScompCtl     = 0x20;
  CtlComp.Bits.RcompDrvDown = 0x20;
  CtlComp.Bits.RcompDrvUp   = 0x20;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCTLCOMP_REG, CtlComp.Data);

  // @todo_Adl program using existing routine
  CompCtl0.Data = 0;
  CompCtl0.Bits.dqvrefup    = Lpddr ? 0x86 : 0x6E;
  CompCtl0.Bits.dqvrefdn    = Lpddr ? 0x57 : 0x40;
  CompCtl0.Bits.dqodtvrefup = Lpddr ? 0x74 : 0x40;
  CompCtl0.Bits.dqodtvrefdn = Lpddr ? 0x57 : 0x40;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL0_REG, CompCtl0.Data);

  CompCtl1.Data = 0;
  CompVref = Lpddr ? 0x6D : 0x64;
  CompCtl1.Bits.cmdvrefup   = CompVref;
  CompCtl1.Bits.ctlvrefup   = CompVref;
  CompCtl1.Bits.clkvrefup   = CompVref;
  CompCtl1.Bits.ckecsvrefup = Lpddr ? 0x6D : 0;     // vref 8bit coding for CKECSUP with step size of Vccddq/191. Valid range is [0..191] @todo_adl - This is not used in DDR4 ?
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL1_REG, CompCtl1.Data);

  CompCtl2.Data = 0;
  CompVref = Lpddr ? 0x57 : 0x50;
  CompCtl2.Bits.cmdvrefdn   = CompVref;
  CompCtl2.Bits.ctlvrefdn   = CompVref;
  CompCtl2.Bits.clkvrefdn   = CompVref;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL2_REG, CompCtl2.Data);

  CompCtl3.Data = 0;
  CompCtl3.Bits.stage3delay     = 0x40;
  CompCtl3.Bits.stage2delay     = 0x40;
  CompCtl3.Bits.codeswitchdelay = 0x40;
  CompCtl3.Bits.compinitdelay   = 0x7F;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL3_REG, CompCtl3.Data);

  CompCtl4.Data = 0;
  CompCtl4.Bits.enodt       = 1;      // should be (!envttodt)
  CompCtl4.Bits.envttodt    = 0;      // Enable if we use Vtt termination
  CompCtl4.Bits.ckecsen     = Lpddr;  // @todo_adl Not needed for DDR4 ?
  CompCtl4.Bits.stage6delay = 0x40;
  CompCtl4.Bits.stage5delay = 0x40;
  CompCtl4.Bits.stage4delay = 0x40;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL4_REG, CompCtl4.Data);

  ScompCtl.Data = 0;
  ScompCtl.Bits.vsxhiopampcr_ch0_globalvsshibyp = Lpddr5;
  ScompCtl.Bits.vsxhiopampcr_ch1_globalvsshibyp = Lpddr5;
  ScompCtl.Bits.vsxhiopampcr_chsel              = 3;
  ScompCtl.Bits.vsxhiopampcr_tbias2opamp        = 0x16; // @todo_adl formula ?
  ScompCtl.Bits.vsxhiopampcr_topampen2rdy       = 0x2C; // @todo_adl formula ?
  ScompCtl.Bits.clkdlytapselstage               = 0xF;  // @todo_adl formula ?
  ScompCtl.Bits.ctldlytapselstage               = 0xF;  // @todo_adl formula ?
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_SCOMPCTL_REG, ScompCtl.Data);

  DdrEarlyCr.Data = 0;
  DdrEarlyCr.Bits.early_restoredone_comp = 1;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDREARLYCR_REG, DdrEarlyCr.Data);

  return Status;
}

/**
  PHY init - modmem_init_configflow_tagged_seq1

  @param[in, out] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcPhyInitSeq1 (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcStatus         Status;
  MrcOutput         *Outputs;
  MrcDebug          *Debug;
  MrcDdrType        DdrType;
  BOOLEAN           Lpddr;
  BOOLEAN           Lpddr4;
  BOOLEAN           Ddr4;
  BOOLEAN           Ddr5;
  BOOLEAN           Gear1;
  BOOLEAN           Gear2;
  BOOLEAN           Gear4;
  UINT32            QclkFrequency;
  INT32             SData32;
  UINT32            UiLock;
  DDRPHY_COMP_CR_COMPOVERRIDE_STRUCT              CompOverride;
  DDRPHY_COMP_CR_FSMSKIPCTRL_STRUCT               FsmSkipCtrl;
  DDRPHY_COMP_NEW_CR_PNCCOMP_CTRL3_STRUCT         PanicCompCtrl3;
  DDRPHY_COMP_NEW_CR_DDRCRDATACOMP1_STRUCT        CompDataComp1;
  DDRPHY_COMP_NEW_CR_VCCDLLCOMPDATACCC_STRUCT     VccDllCompDataCcc;
  DDRPHY_COMP_NEW_CR_DIMMVREF_VREFCONTROL_STRUCT  DimmVrefControl;
  DDRPHY_COMP_NEW_CR_DIMMVREF_VREFCH0_STRUCT      DimmVrefCh0;
  DDRPHY_COMP_NEW_CR_DIMMVREF_VREFCH1_STRUCT      DimmVrefCh1;
  DDRPHY_COMP_NEW_CR_DIMMVREF_VREFADJUST2_STRUCT  VrefAdjust2;
  DDRPHY_COMP_NEW_CR_VSSHICOMP_CTRL2_STRUCT       VsshiCompCtrl2;
  DDRPHY_COMP_NEW_CR_VSSHICOMP_CTRL3_STRUCT       VsshiCompCtrl3;
  DDRPHY_COMP_CR_COMPDLLWL_STRUCT                 CompDllWl;
  DDRSCRAM_CR_DDREARLYCR_STRUCT                   ScramDdrEarlyCr;
  CH0CCC_CR_MDLLCTL0_STRUCT                       CccMdllCtl0;
  CH0CCC_CR_MDLLCTL1_STRUCT                       CccMdllCtl1;
  CH0CCC_CR_MDLLCTL2_STRUCT                       CccMdllCtl2;
  CH0CCC_CR_MDLLCTL3_STRUCT                       CccMdllCtl3;
  CH0CCC_CR_COMP3_STRUCT                          CccComp3;
  DATA0CH0_CR_MDLLCTL0_STRUCT                     DataMdllCtl0;
  DATA0CH0_CR_MDLLCTL1_STRUCT                     DataMdllCtl1;
  DATA0CH0_CR_MDLLCTL2_STRUCT                     DataMdllCtl2;
  DATA0CH0_CR_MDLLCTL3_STRUCT                     DataMdllCtl3;
  CCC_GLOBAL_IOLVRCTL_STRUCT                      CccGlobalIoLvrCtl;
  CCC_GLOBAL_CR_DDREARLYCR_STRUCT                 CccGlobalDdrEarlyCr;
  DATA_GLOBAL_CR_IOLVRCTL0_STRUCT                 DataGlobalIoLvrCtl0;
  DATA_GLOBAL_CR_DDREARLYCR_STRUCT                DataGlobalDdrEarlyCr;
  CCC_CR_CLKALIGNCTL1_STRUCT                      CccClkAlignCtl1;
  DATA_CR_CLKALIGNCTL1_STRUCT                     DataClkAlignCtl1;
  DATA_CR_SDLCTL1_STRUCT                          DataSdlCtl1;
  DATA_CR_AFEMISCCTRL1_STRUCT                     AfeMiscCtrl1;

  Status      = mrcSuccess;
  Outputs     = &MrcData->Outputs;
  Debug       = &Outputs->Debug;
  DdrType     = Outputs->DdrType;
  Lpddr       = Outputs->Lpddr;
  Lpddr4      = (DdrType == MRC_DDR_TYPE_LPDDR4);
  Ddr4        = (DdrType == MRC_DDR_TYPE_DDR4);
  Ddr5        = (DdrType == MRC_DDR_TYPE_DDR5);
  Gear2       = Outputs->Gear2;
  Gear1       = !Gear2;
  Gear4       = FALSE;                      // @todo_adl
  QclkFrequency = UDIVIDEROUND (1000000, Outputs->Qclkps);

  //---------------------------------------------
  // static_compinit_mrc - COMP
  //---------------------------------------------
  CompOverride.Data = 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_COMPOVERRIDE_REG, CompOverride.Data);

  FsmSkipCtrl.Data = 0xFFFFFFFF; // We will enable most of the bits, except the ones below
  FsmSkipCtrl.Bits.En_cmddn200  = Lpddr ? 0 : 1;
  FsmSkipCtrl.Bits.En_ckecsup   = Lpddr ? 1 : 0;
  FsmSkipCtrl.Bits.En_code2vt   = Ddr4  ? 1 : 0;
  FsmSkipCtrl.Bits.En_vsshiffrx = Ddr5  ? 1 : 0;
  FsmSkipCtrl.Bits.En_dqsfwdclkp = (Lpddr4 || Ddr4) ? 1 : 0;
  FsmSkipCtrl.Bits.En_dqsfwdclkn = FsmSkipCtrl.Bits.En_dqsfwdclkp;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_FSMSKIPCTRL_REG, FsmSkipCtrl.Data);

  //---------------------------------------------
  // static_compinit - COMP_NEW
  //---------------------------------------------
  PanicCompCtrl3.Data = 0;
  PanicCompCtrl3.Bits.pnccomp_rloaddqs_vrefcode   = 0x40;
  PanicCompCtrl3.Bits.pnccomp_cmddn200_vrefcode   = 0x44;
  PanicCompCtrl3.Bits.pnccomp_vttpanicdn_vrefcode = 0x48;
  PanicCompCtrl3.Bits.pnccomp_vttpanicup_vrefcode = 0x42;
  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_PNCCOMP_CTRL3_REG, PanicCompCtrl3.Data); // @todo_adl - are these comp code outputs ? Do we need to program this ?

  CompDataComp1.Data = MrcReadCR (MrcData, DDRPHY_COMP_NEW_CR_DDRCRDATACOMP1_REG);
  CompDataComp1.Bits.Dqsnoffsetnui  = 0;
  CompDataComp1.Bits.Dqsntargetnui  = (Lpddr4 || Ddr4) ? 0 : 0; // Matched ? 0 : UiLock, start with UiLock = 0
  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_DDRCRDATACOMP1_REG, CompDataComp1.Data);

  VccDllCompDataCcc.Data = MrcReadCR (MrcData, DDRPHY_COMP_NEW_CR_VCCDLLCOMPDATACCC_REG);
  VccDllCompDataCcc.Bits.Dqspoffsetnui  = 0;
  VccDllCompDataCcc.Bits.Dqsptargetnui  = (Lpddr4 || Ddr4) ? 0 : 0; // Matched ? 0 : UiLock, start with UiLock = 0
  VccDllCompDataCcc.Bits.Dll_bwsel      = 0xF;
  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_VCCDLLCOMPDATACCC_REG, VccDllCompDataCcc.Data);

  DimmVrefControl.Data = 0;
  DimmVrefControl.Bits.hibwdivider    = 2;    // 16 samples
  DimmVrefControl.Bits.lobwdivider    = 3;    // 32 samples
  DimmVrefControl.Bits.sampledivider  = 2;    // QCLK/8
  DimmVrefControl.Bits.hibwenable     = 1;
  DimmVrefControl.Bits.vtoffset       = 0;    // @todo_adl Use formula from TGL ?
  DimmVrefControl.Bits.vtslope        = 0x10; // @todo_adl Use formula from TGL ?
  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_DIMMVREF_VREFCONTROL_REG, DimmVrefControl.Data);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "DIMMVREF_VREFCONTROL: 0x%X\n", DimmVrefControl.Data);

  DimmVrefCh0.Data = 0;
  DimmVrefCh1.Data = 0;
  if (Ddr4) {
    if (IsDimmPresent (MrcData, 0, 0, 0, 0) == mrcSuccess) {
      DimmVrefCh0.Bits.enca0vref = 1;
      DimmVrefCh0.Bits.ca0vref   = MID_INT_VREF;
    }
    if (IsDimmPresent (MrcData, 0, 0, 0, 1) == mrcSuccess) {
      DimmVrefCh0.Bits.enca1vref = 1;
      DimmVrefCh0.Bits.ca1vref   = MID_INT_VREF;
    }
    if (IsDimmPresent (MrcData, 0, 1, 0, 0) == mrcSuccess) {
      DimmVrefCh1.Bits.enca0vref = 1;
      DimmVrefCh1.Bits.ca0vref   = MID_INT_VREF;
    }
    if (IsDimmPresent (MrcData, 0, 1, 0, 1) == mrcSuccess) {
      DimmVrefCh1.Bits.enca1vref = 1;
      DimmVrefCh1.Bits.ca1vref   = MID_INT_VREF;
    }
  }
  DimmVrefCh0.Bits.qxcount = MrcFrequencyToRatio (MrcData, QclkFrequency, MRC_REF_CLOCK_133, 0);  // Ratio of QCLK in terms of 133.3333
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "QclkFrequency: %d, qxcount: %d\n", QclkFrequency, DimmVrefCh0.Bits.qxcount);
  DimmVrefCh0.Bits.sagvopenloopen = 1;

  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_DIMMVREF_VREFCH0_REG, DimmVrefCh0.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_DIMMVREF_VREFCH1_REG, DimmVrefCh1.Data);

  /* TGL code
  VrefAdj2.Bits.HiZTimerCtrl  = DDRVREF_CR_DDRCRVREFADJUST2_HiZTimerCtrl_MAX;
  VrefAdj2.Bits.LockOvrd = SAFE ? 0 : 1;
  TempVar1 = (1000 / RefClkPs) - 4; // TBDns
  TempVar2 = MrcLog2(TempVar1);
  VrefAdj2.Bits.LockTimer    = RANGE(TempVar2, 0, 7);
#ifdef CTE_FLAG
  VrefAdj2.Bits.LockTimer = 0;
#endif
  TempVar1 = (Vdd2Mv / 382);
  TempVar2 = VccIoMv / 128;
  TempVar3 = DIVIDEROUND(TempVar1, TempVar2);
  //MaxMin( Rnd(TBD*(VccDD2/382) / (VccIO/128)), 0, 31)
  VrefAdj2.Bits.VtSlopeSAGV = RANGE(TempVar3, 0, 31);
  SignedTempVar1 = (Vdd2Mv / 96);
  // MaxMin( Rnd(TBD mV / (VccDD2/96) ), -16, 15).
  VrefAdj2.Bits.VtOffsetSAGV = RANGE(SignedTempVar1, -16, 15);
  VrefAdj2.Bits.SagvVtCtl   = 1;
  VrefAdj2.Bits.GateICinDVFS = SAFE ? 1 : 0;
  MrcWriteCR (MrcData, DDRVREF_CR_DDRCRVREFADJUST2_REG, VrefAdj2.Data);  */

  VrefAdjust2.Data = 0;
  VrefAdjust2.Bits.hiztimerctrl = 2;    // DDRPHY_COMP_NEW_CR_DIMMVREF_VREFADJUST2_hiztimerctrl_MAX ?
  VrefAdjust2.Bits.lockovrd     = 1;
  VrefAdjust2.Bits.locktimer    = 0;    // Formula ?
  VrefAdjust2.Bits.vtslopesagv  = 0x10; // Formula ?
  VrefAdjust2.Bits.vtoffsetsagv = 0x10; // Formula ?
  VrefAdjust2.Bits.gateicindvfs = 1;
  VrefAdjust2.Bits.sagvvtctl    = 1;
  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_DIMMVREF_VREFADJUST2_REG, VrefAdjust2.Data);

  VsshiCompCtrl2.Data = 0;
  VsshiCompCtrl2.Bits.vsshicompcr_leakvrefcode  = 0x40;
  VsshiCompCtrl2.Bits.vsshicompcr_rxvrefcode    = 0x44;
  VsshiCompCtrl2.Bits.vsshicompcr_cmdvrefcode   = 0x48;
  VsshiCompCtrl2.Bits.vsshicompcr_dqvrefcode    = 0x52;
  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_VSSHICOMP_CTRL2_REG, VsshiCompCtrl2.Data);

  VsshiCompCtrl3.Data = 0;
  VsshiCompCtrl3.Bits.vsshicompcr_ctlvrefcode   = 0x42;
  VsshiCompCtrl3.Bits.vsshicompcr_clkvrefcode   = 0x50;
  MrcWriteCR (MrcData, DDRPHY_COMP_NEW_CR_VSSHICOMP_CTRL3_REG, VsshiCompCtrl3.Data);

  CompDllWl.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_COMPDLLWL_REG);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_COMPDLLWL_REG, CompDllWl.Data);

  //---------------------------------------------
  // static_DLL - CMD
  //---------------------------------------------
  CccMdllCtl0.Data = 0;
  CccMdllCtl0.Bits.mdll_cmn_mdllen          = 1;                            // @todo_adl Can set to 0 on empty CCC partitions to save power
  CccMdllCtl0.Bits.mdll_cmn_pien            = Gear4 ? 0xFF : 0x5F;          // @todo_adl We can disable some PI's depending on channel population and DRAM type, need table from DE
  CccMdllCtl0.Bits.mdll_cmn_mdllengthsel    = Gear1 ? 0 : (Gear4 ? 2 : 1);  // G1: 0, G2: 1, G4: 2
  CccMdllCtl0.Bits.mdll_cmn_dllbypassen     = 0;
  CccMdllCtl0.Bits.mdll_cmn_wlrdacholden    = 1;
  CccMdllCtl0.Bits.mdll_cmn_turboonstartup  = 1;
  SData32 = 359955 - 112 * QclkFrequency;
  SData32 = DIVIDEROUND (SData32, 100000);                                  // ROUND(QclkFrequency * (-0.00112) + 3.59955)
  CccMdllCtl0.Bits.mdll_cmn_turbocaptrim    = MAX (SData32, 0);             // Protect against negative values
  MrcWriteCrMulticast (MrcData, CCC_CR_MDLLCTL0_REG, CccMdllCtl0.Data);

  CccMdllCtl1.Data = 0;
  CccMdllCtl1.Bits.mdll_cmn_phsdrvprocsel = 7;
  CccMdllCtl1.Bits.mdll_cmn_vctlcaptrim   = CccMdllCtl0.Bits.mdll_cmn_turbocaptrim;
  MrcWriteCrMulticast (MrcData, CCC_CR_MDLLCTL1_REG, CccMdllCtl1.Data);

  CccMdllCtl2.Data = 0;
  CccMdllCtl2.Bits.mdll_cmn_dccen             = 1;
  CccMdllCtl2.Bits.mdll_cmn_dllldoen          = 1;    // @todo_adl Can set to 0 on empty CCC partitions to save power
  CccMdllCtl2.Bits.mdll_cmn_vctlcompoffsetcal = 0x10;
  CccMdllCtl2.Bits.mdll_cmn_vcdldcccode       = 8;
  CccMdllCtl2.Bits.mdll_cmn_ldofbdivsel       = 1;
  MrcWriteCrMulticast (MrcData, CCC_CR_MDLLCTL2_REG, CccMdllCtl2.Data);

  CccMdllCtl3.Data = 0;
  CccMdllCtl3.Bits.mdll_cmn_ldopbiasctrl      = 3;
  CccMdllCtl3.Bits.mdll_cmn_ldonbiasvrefcode  = 0x1B;
  CccMdllCtl3.Bits.mdll_cmn_ldonbiasctrl      = 7;
  CccMdllCtl3.Bits.mdll_cmn_rloadcomp         = 0x0E;
  MrcWriteCrMulticast (MrcData, CCC_CR_MDLLCTL3_REG, CccMdllCtl3.Data);

  CccComp3.Data = 0;
  CccComp3.Bits.imod_comp_ccctx_cmn_txtcocompp = 0x1F;
  CccComp3.Bits.imod_comp_ccctx_cmn_txtcocompn = 0x1F;
  MrcWriteCrMulticast (MrcData, CCC_CR_COMP3_REG, CccComp3.Data);

  //---------------------------------------------
  // static_DLL - DQM
  //---------------------------------------------
  DataMdllCtl2.Data = 0;
  DataMdllCtl2.Bits.mdll_cmn_dccen              = 1;
  DataMdllCtl2.Bits.mdll_cmn_dllldoen           = 1;
  DataMdllCtl2.Bits.mdll_cmn_ldoffreadsegen     = 3;
  DataMdllCtl2.Bits.mdll_cmn_vctlcompoffsetcal  = 0x10;
  DataMdllCtl2.Bits.mdll_cmn_vcdldcccode        = 8;
  DataMdllCtl2.Bits.mdll_cmn_ldoffcodelock      = 0x80;
  DataMdllCtl2.Bits.mdll_cmn_ldoffcodewl        = 0x20;
  DataMdllCtl2.Bits.mdll_cmn_ldofbdivsel        = 1;
  MrcWriteCrMulticast (MrcData, DATA_CR_MDLLCTL2_REG, DataMdllCtl2.Data);

  DataMdllCtl0.Data = 0;
  DataMdllCtl0.Bits.mdll_cmn_turbocaptrim   = CccMdllCtl0.Bits.mdll_cmn_turbocaptrim;
  DataMdllCtl0.Bits.mdll_cmn_turboonstartup = 1;
  DataMdllCtl0.Bits.mdll_cmn_ldoffcodepi    = 0x20;
  DataMdllCtl0.Bits.mdll_cmn_wlrdacholden   = 1;    // @todo_adl Should be set during weaklock enabling at the end of MRC ?
  DataMdllCtl0.Bits.mdll_cmn_dllbypassen    = 0;
  DataMdllCtl0.Bits.mdll_cmn_mdllengthsel   = CccMdllCtl0.Bits.mdll_cmn_mdllengthsel;   // G1: 0, G2: 1, G4: 2
  DataMdllCtl0.Bits.mdll_cmn_pien           = Gear4 ? 0x3F : 0x17;
  DataMdllCtl0.Bits.mdll_cmn_mdllen          = 1;
  MrcWriteCrMulticast (MrcData, DATA_CR_MDLLCTL0_REG, DataMdllCtl0.Data);

  DataMdllCtl1.Data = 0;
  DataMdllCtl1.Bits.mdll_cmn_picapsel       = 4;
  DataMdllCtl1.Bits.mdll_cmn_phsdrvprocsel  = 4;
  DataMdllCtl1.Bits.mdll_cmn_vctlcaptrim   = CccMdllCtl0.Bits.mdll_cmn_turbocaptrim;
  DataMdllCtl1.Bits.mdll_cmn_vcdlbwsel     = 0;      // @todo_adl comment says "not used in design", should we touch this ? Formula is not clear
  DataMdllCtl1.Bits.mdll_cmn_vctldaccode   = 0x187;  // @todo_adl Should we program this ?
  MrcWriteCrMulticast (MrcData, DATA_CR_MDLLCTL1_REG, DataMdllCtl1.Data);

  UiLock            = 0;  // @todo_adl

  DataMdllCtl3.Data = 0;
  DataMdllCtl3.Bits.mdll_cmn_ldopbiasctrl         = 3;
  DataMdllCtl3.Bits.mdll_cmn_ldonbiasvrefcode     = 0x1B;
  DataMdllCtl3.Bits.mdll_cmn_ldonbiasctrl         = 7;
  DataMdllCtl3.Bits.mdll_cmn_rloadcomp            = 0x1F;
  DataMdllCtl3.Bits.mdll_cmn_ldoforceen           = (UiLock & MRC_BIT2) ? 1 : 0; // static_compinit_mrc
  DataMdllCtl3.Bits.mdll_cmn_ldolocalvsshibypass  = (UiLock & MRC_BIT1) ? 1 : 0; // static_compinit_mrc
  DataMdllCtl3.Bits.mdll_cmn_ldoffcoderead        = 0x40;
  MrcWriteCrMulticast (MrcData, DATA_CR_MDLLCTL3_REG, DataMdllCtl3.Data);

  //---------------------------------------------
  // static_DQEARLY - DQM
  //---------------------------------------------
  Status |= MrcPhyInitStaticDqEarly (MrcData);

  //---------------------------------------------
  // static_DIG1 - VTT
  //---------------------------------------------
  Status |= MrcPhyInitVttStaticDig1 (MrcData);

  //---------------------------------------------
  // static_compinit - COMP
  //---------------------------------------------
  Status |= MrcPhyInitCompStaticCompInit (MrcData);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "MrcPhyInitCompStaticCompInit Done\n");

  //---------------------------------------------
  // static_DIG1 - SCRAM
  //---------------------------------------------
  ScramDdrEarlyCr.Data = 0;
  ScramDdrEarlyCr.Bits.early_restoredone = 1;
  MrcWriteCR (MrcData, DDRSCRAM_CR_DDREARLYCR_REG, ScramDdrEarlyCr.Data);

  //---------------------------------------------
  // static_DIG1 - CCC_GLOBAL
  //---------------------------------------------
  CccGlobalIoLvrCtl.Data = 0;
  CccGlobalIoLvrCtl.Bits.iolvrseg_cmn_codelvrleg  = 0x40;
  CccGlobalIoLvrCtl.Bits.iolvrseg_cmn_enlvrleg    = 1;
  CccGlobalIoLvrCtl.Bits.iolvrseg_cmn_enffleg     = 1;
  MrcWriteCrMulticast (MrcData, CCC_GLOBAL_IOLVRCTL_REG, CccGlobalIoLvrCtl.Data);

  CccGlobalDdrEarlyCr.Data = 0;
  CccGlobalDdrEarlyCr.Bits.early_restoredone = 1;
  MrcWriteCrMulticast (MrcData, CCC_GLOBAL_CR_DDREARLYCR_REG, CccGlobalDdrEarlyCr.Data);

  //---------------------------------------------
  // static_DIG1 - DATA_GLOBAL
  //---------------------------------------------
  DataGlobalIoLvrCtl0.Data = 0;
  DataGlobalIoLvrCtl0.Bits.iolvrseg_cmn_enffleg     = 1;
  DataGlobalIoLvrCtl0.Bits.iolvrseg_cmn_enlvrleg    = 1;
  DataGlobalIoLvrCtl0.Bits.iolvrseg_cmn_codeffleg   = 0x0F;
  DataGlobalIoLvrCtl0.Bits.iolvrseg_cmn_codelvrleg  = 0x0F;
  MrcWriteCrMulticast (MrcData, DATA_GLOBAL_CR_IOLVRCTL0_REG, DataGlobalIoLvrCtl0.Data);

  DataGlobalDdrEarlyCr.Data = 0;
  DataGlobalDdrEarlyCr.Bits.early_restoredone = 1;
  MrcWriteCrMulticast (MrcData, DATA_GLOBAL_CR_DDREARLYCR_REG, DataGlobalDdrEarlyCr.Data);

  //---------------------------------------------
  // Individual registers
  //---------------------------------------------
  CccClkAlignCtl1.Data = 0;
  CccClkAlignCtl1.Bits.clkalignd0piclkpicodeoffset = 4;
  MrcWriteCrMulticast (MrcData, CCC_CR_CLKALIGNCTL1_REG, CccClkAlignCtl1.Data);

  DataClkAlignCtl1.Data = 0;
  DataClkAlignCtl1.Bits.clkalignd0piclkpicodeoffset = 0x0D;
  MrcWriteCrMulticast (MrcData, DATA_CR_CLKALIGNCTL1_REG, DataClkAlignCtl1.Data);

  //---------------------------------------------
  // static_compinit_mrc - COMP
  // This sequence is repeated here for the second time, so register values are already populated
  //---------------------------------------------
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_COMPOVERRIDE_REG, CompOverride.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_FSMSKIPCTRL_REG, FsmSkipCtrl.Data);

  //---------------------------------------------
  // static_compinit_mrc - DQ
  //  @todo_adl:
  //   - DataControl3.rxtogglepreamble                                                     - can program in static_DQEARLY (Seq1, before this point) ?
  //   - DDRDATA_CR_SDLCTL1.rxsdl_cmn_passrcvenondqsfall and rxsdl_cmn_qualifysdlwithrcven - can program in static_DQ_AFE  (Seq2) ?                    <-- yes
  //   - DDRDATA_CR_MDLLCTL3.mdll_cmn_ldoforceen and mdll_cmn_ldolocalvsshibypass          - can program in static_DLL     (Seq1, before this point) ? <-- yes
  //   - DDRDATA_CR_AFEMISCCTRL1.rxsdl_cmn_enablepiwhenoff                                 - can program in static_DIG2    (Seq2) ?                    <-- yes
  //---------------------------------------------
  DataSdlCtl1.Data = 0;
  DataSdlCtl1.Bits.rxsdl_cmn_srzclkend0           = 1;
  DataSdlCtl1.Bits.rxsdl_cmn_rxsdlbwsel           = 0x20;
  DataSdlCtl1.Bits.rxsdl_cmn_burstlen             = Ddr4 ? 1 : 3;
  DataSdlCtl1.Bits.rxsdl_cmn_passrcvenondqsfall   = 0; // @todo_adl matched_rx ? 0 : (uilock + 1 - rd_dqs_offset*2 == 1) ? 0 : 1;
  DataSdlCtl1.Bits.rxsdl_cmn_qualifysdlwithrcven  = 1; // @todo_adl matched_rx ? 1 : (uilock + 1 - rd_dqs_offset*2 == 1) ? 1 : 0;
  DataSdlCtl1.Bits.rxsdl_cmn_rxphsdrvprocsel      = 4;
  DataSdlCtl1.Bits.rxsdl_cmn_rxpicapsel           = 4;
  DataSdlCtl1.Bits.rxsdl_cmn_rxpienable           = 1;
  DataSdlCtl1.Bits.rxsdl_cmn_rxsdllengthsel       = 3;
  MrcWriteCrMulticast (MrcData, DATA_CR_SDLCTL1_REG, DataSdlCtl1.Data);

  AfeMiscCtrl1.Data = 0;
  AfeMiscCtrl1.Bits.rxsdl_cmn_enablepiwhenoff     = (UiLock & MRC_BIT0) ? 1 : 0;
  AfeMiscCtrl1.Bits.dqsrx_cmn_rxdiffampdfxen      = 1;
  AfeMiscCtrl1.Bits.dxtx_cmn_tcoslewstatlegen     = 1;
  MrcWriteCrMulticast (MrcData, DATA_CR_AFEMISCCTRL1_REG, AfeMiscCtrl1.Data);

  return Status;
}

/**
  This function initializes the Memory Controller IO.
  Occupancy and Scrambler registers are initialized.

  @param[in, out] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcDdrIoPreInit (
  IN OUT MrcParameters *const MrcData
  )
{
  static const UINT8 ChNotPopMap[MRC_NUM_CCC_INSTANCES]    = { 4, 5, 0, 1, 2, 3, 6, 7 };
  static const UINT8 CccMcIdxMapDdr[MRC_NUM_CCC_INSTANCES] = { 1, 1, 0, 0, 0, 0, 1, 1 };
  MrcInput          *Inputs;
  MrcIntOutput      *MrcIntData;
  MrcStatus         Status;
  MrcOutput         *Outputs;
  MrcControllerOut  *ControllerOut;
  MrcDebug          *Debug;
  MrcDdrType        DdrType;
  INT64             GetSetVal;
  INT64             GetSetEn;
//  INT64             GetSetDis;
  UINT32            Offset;
  UINT32            Controller;
  UINT32            Index;
  UINT32            RefClkPs;
  UINT32            Delay;
  UINT32            McIndex;
  UINT32            ChIndex;
  UINT32            TimerXXClk;
  UINT32            KeepXXClkOn;
  UINT16            TxEn;
  UINT8             Gear1;
  UINT8             IoChNotPop;
  UINT8             RankPresent;
  UINT8             Rank0Present;
  UINT8             Rank1Present;
  UINT8             Rank2Present;
  UINT8             Rank3Present;
  UINT8             CccMux;
  UINT8             MaxChannel;
  BOOLEAN           Lpddr;
  BOOLEAN           Lpddr4;
  BOOLEAN           Lpddr5;
  BOOLEAN           Ddr4;
  BOOLEAN           DqPinsInterleaved;
  BOOLEAN           UlxUlt;
  BOOLEAN           DtHalo;
  GSM_GT            Group;
  CCC_TX_EN_TYPE    CccTxEn;
  CH0CCC_CR_DDRCRPINSUSED_STRUCT  CccPinsUsed;

  Inputs      = &MrcData->Inputs;
  Outputs     = &MrcData->Outputs;
  Debug       = &Outputs->Debug;
  MrcIntData  = ((MrcIntOutput *) (MrcData->IntOutputs.Internal));
  Status      = mrcSuccess;
  DdrType     = Outputs->DdrType;
  Lpddr4      = (DdrType == MRC_DDR_TYPE_LPDDR4);
  Ddr4        = (DdrType == MRC_DDR_TYPE_DDR4);
  Lpddr5      = (DdrType == MRC_DDR_TYPE_LPDDR5);
  Lpddr       = Outputs->Lpddr;
  MaxChannel  = Outputs->MaxChannels;
  UlxUlt      = Inputs->UlxUlt;
  DtHalo      = Inputs->DtHalo;
  Gear1       = (Outputs->Gear2) ? 0 : 1;
  GetSetEn    = 1;
//  GetSetDis = 0;
  DqPinsInterleaved = (Inputs->DqPinsInterleaved != 0);
  MrcIntData->ClkDccInit = FALSE;
  MrcIntData->DqsDccInit = FALSE;
  RefClkPs = (Outputs->RefClk == MRC_REF_CLOCK_133) ? 7500 : 10000; // Reference Clock Period in pS

//#ifndef AV_PHY_INIT
  MrcDdrioCrRestore (MrcData);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "MrcDdrioCrRestore done\n");
//#endif

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    ControllerOut = &Outputs->Controller[Controller];
    ControllerOut->DeviceId = (UINT16) (MrcPciRead32 (MrcData, HOST_BRIDGE_BUS, HOST_BRIDGE_DEVICE, HOST_BRIDGE_FUNCTION, HOST_BRIDGE_DEVID) >> 16);
    ControllerOut->RevisionId = (UINT8) (MrcPciRead32 (MrcData, HOST_BRIDGE_BUS, HOST_BRIDGE_DEVICE, HOST_BRIDGE_FUNCTION, HOST_BRIDGE_REVID));
  }

  // Select the interleaving mode of DQ/DQS pins
  // This must be the first DDR IO register to be programmed on ULT
  GetSetVal = (DqPinsInterleaved) ? 0 : 1;
  MrcGetSetNoScope (MrcData, GsmIocNoDqInterleave, WriteToCache | PrintValue, &GetSetVal);
  MrcWeaklockEnDis (MrcData, MRC_DISABLE);

  if (Lpddr) {
    MrcGetSetNoScope (MrcData, GsmIocScramLpMode, WriteToCache | PrintValue, &GetSetEn);
  }

  switch (DdrType) {
    case MRC_DDR_TYPE_DDR4:
      Group = GsmIocScramDdr4Mode;
      break;

    case MRC_DDR_TYPE_LPDDR4:
      Group = GsmIocScramLp4Mode;
      break;

    case MRC_DDR_TYPE_DDR5:
      Group = GsmIocScramDdr5Mode;
      break;

    case MRC_DDR_TYPE_LPDDR5:
      Group = GsmIocScramLp5Mode;
      break;

    default:
      Group = GsmGtMax;
      break;
  }
  if (Group == GsmGtMax) {
    Status = mrcUnsupportedTechnology;
  } else {
    MrcGetSetNoScope (MrcData, Group, WriteToCache | PrintValue, &GetSetEn);
  }

  IoChNotPop = 0;
  for (Index = 0; Index < MRC_NUM_CCC_INSTANCES; Index++) {
    CccTxEn.Data16 = 0;
    // Set MC Index first.
    if (DtHalo && !Lpddr) {
      McIndex = CccMcIdxMapDdr[Index];
    } else {
      // For DDR4, CCC[0:3] maps to Mc0 Ch0 and CCC[4:7] maps to Mc1 Ch0
      McIndex = Index / MAX_CHANNEL;
    }
    // Now do Channel Index
    if (Ddr4) {
      ChIndex = 0;
    } else if (Lpddr) {
      // LPDDR is 1:1
      ChIndex = Index % MaxChannel;
    } else {
      ChIndex = ((Index < 4) ? 0 : 1);
    }
    RankPresent = Outputs->Controller[McIndex].Channel[ChIndex].ValidRankBitMask;
    Rank0Present = (0x1 & RankPresent);
    Rank1Present = ((RankPresent >> 1) & 0x1);
    Rank2Present = ((RankPresent >> 2) & 0x1);
    Rank3Present = ((RankPresent >> 3) & 0x1);
    if (RankPresent == 0) {
      IoChNotPop |= (1 << ((DtHalo) ? ChNotPopMap[Index] : Index));
    }

    // Configure non-common enables:
    // LPDDR - Command, CS
    // DDR   - ODT, CKE, CS
    if (UlxUlt) {
      if (Lpddr4) {
        switch (Index) {
          case 0:
          case 2:
          case 3:
          case 4:
          case 6:
          case 7:
            CccTxEn.Data16 = MRC_UY_LP4_CCC023467_CA_MSK;
            CccTxEn.Lp4.Ccc023467.Cs0 = Rank0Present;
            CccTxEn.Lp4.Ccc023467.Cs1 = Rank1Present;
            break;

          case 1:
          case 5:
            CccTxEn.Data16 = MRC_UY_LP4_CCC15_CA_MSK;
            CccTxEn.Lp4.Ccc15.Cs0 = Rank0Present;
            CccTxEn.Lp4.Ccc15.Cs1 = Rank1Present;
            break;

          default:
            break;
        }
      } else if (Lpddr5) {
        if (Inputs->Lp5CccConfig & (1 << Index)) {
          // Descending Configuration
          switch (Index) {
            case 0:
            case 1:
            case 4:
            case 5:
              CccTxEn.Data16 = MRC_UY_LP5D_CCC0145_CA_MSK;
              CccTxEn.Lp5D.Ccc0145.Cs = RankPresent;
              break;

            case 2:
            case 3:
            case 6:
            case 7:
              CccTxEn.Data16 = MRC_UY_LP5D_CCC2367_CA_MSK;
              CccTxEn.Lp5D.Ccc2367.Cs0 = Rank0Present;
              CccTxEn.Lp5D.Ccc2367.Cs1 = Rank1Present;
              break;

            default:
              // Do nothing.  Will catch 0 in this as a failure below the DdrType Ladder.
              break;
          }
        } else {
          // Ascending Configuration
          switch (Index) {
            case 0:
            case 2:
            case 3:
            case 4:
            case 6:
            case 7:
              CccTxEn.Data16 = MRC_UY_LP5A_CCC023467_CA_MSK;
              CccTxEn.Lp5A.Ccc023467.Cs = RankPresent;
              break;

            case 1:
            case 5:
              CccTxEn.Data16 = MRC_UY_LP5A_CCC15_CA_MSK;
              CccTxEn.Lp5A.Ccc15.Cs0 = Rank0Present;
              CccTxEn.Lp5A.Ccc15.Cs1 = Rank1Present;
              break;

            default:
              // Do nothing.  Will catch 0 in this as a failure below the DdrType Ladder.
              break;
          }
        }
      } else if (Ddr4) {
        // DDR4
        switch (Index % 4) {
          case 0:
            CccTxEn.Data16 = MRC_UY_DDR4_CCC04_CA_MSK;
            if (Rank0Present) {
              CccTxEn.Ddr4.UY.Ccc04.Clk0 = 0x3;
            }
            break;

          case 1:
            CccTxEn.Data16 = MRC_UY_DDR4_CCC15_CA_MSK;
            CccTxEn.Ddr4.UY.Ccc15.Cs0  = Rank0Present;
            CccTxEn.Ddr4.UY.Ccc15.Cs1  = Rank1Present;
            CccTxEn.Ddr4.UY.Ccc15.Odt0 = Rank0Present;
            CccTxEn.Ddr4.UY.Ccc15.Odt1 = Rank1Present;
            break;

          case 2:
            CccTxEn.Data16 = MRC_UY_DDR4_CCC26_CA_MSK;
            CccTxEn.Ddr4.UY.Ccc26.Cke0 = Rank0Present;
            CccTxEn.Ddr4.UY.Ccc26.Cke1 = Rank1Present;
            break;

          case 3:
            CccTxEn.Data16 = MRC_UY_DDR4_CCC37_CA_MSK;
            if (Rank1Present) {
              CccTxEn.Ddr4.UY.Ccc37.Clk1 = 0x3;
            }
            break;
        }
      }
      // Do nothing for DDR5.  Will catch 0 in this as a failure below the DdrType Ladder.
    } else {
      // DT/Halo
      if (Lpddr) {
        if ((Index % 2) == 0) {
          CccTxEn.Data16 = ((Lpddr5) ? MRC_HS_LP5_CCC0246_CA_MSK : MRC_HS_LP4_CCC012346_CA_MSK);
          CccTxEn.LpHS.Ccc0246.Cs0 = Rank0Present;
          CccTxEn.LpHS.Ccc0246.Cs1 = Rank1Present;
          CccTxEn.LpHS.Ccc0246.Cs2 = Rank2Present;
          CccTxEn.LpHS.Ccc0246.Cs3 = Rank3Present;
        } else if (Index < 4) {
          CccTxEn.Data16 = ((Lpddr5) ? MRC_HS_LP5_CCC13_CA_MSK : MRC_HS_LP4_CCC012346_CA_MSK);
          CccTxEn.LpHS.Ccc13.Cs0 = Rank0Present;
          CccTxEn.LpHS.Ccc13.Cs1 = Rank1Present;
          CccTxEn.LpHS.Ccc13.Cs2 = Rank2Present;
          CccTxEn.LpHS.Ccc13.Cs3 = Rank3Present;
        } else if (Index == 5) {
          CccTxEn.Data16 = ((Lpddr5) ? MRC_HS_LP5_CCC5_CA_MSK : MRC_HS_LP4_CCC5_CA_MSK);
          CccTxEn.LpHS.Ccc5.Cs0 = Rank0Present;
          CccTxEn.LpHS.Ccc5.Cs1 = Rank1Present;
          CccTxEn.LpHS.Ccc5.Cs2 = Rank2Present;
          CccTxEn.LpHS.Ccc5.Cs3 = Rank3Present;
        } else {
          CccTxEn.Data16 = ((Lpddr5) ? MRC_HS_LP5_CCC7_CA_MSK : MRC_HS_LP4_CCC7_CA_MSK);
          CccTxEn.LpHS.Ccc7.Cs0 = Rank0Present;
          CccTxEn.LpHS.Ccc7.Cs1 = Rank1Present;
          CccTxEn.LpHS.Ccc7.Cs2 = Rank2Present;
          CccTxEn.LpHS.Ccc7.Cs3 = Rank3Present;
        }
      } else if (Ddr4) { // Matches ADL-S DDR4
        switch (Index) {
          case 0:
          case 2:
            CccTxEn.Data16 = MRC_HS_DDR4_CCC02_CA_MSK;
            CccTxEn.Ddr4.HS.Ccc02.Cke0 = Rank0Present;
            CccTxEn.Ddr4.HS.Ccc02.Cke1 = Rank1Present;
            CccTxEn.Ddr4.HS.Ccc02.Cke2 = Rank2Present;
            CccTxEn.Ddr4.HS.Ccc02.Cke3 = Rank3Present;
            break;

          case 1:
          case 3:
          case 5:
          case 7:
            if (Index < 4) {
              CccTxEn.Data16 = MRC_HS_DDR4_CCC13_CA_MSK;
              CccTxEn.Ddr4.HS.Ccc1357.Clk0or2 = (Rank0Present ? 0x3 : 0);
              CccTxEn.Ddr4.HS.Ccc1357.Clk1or3 = (Rank1Present ? 0x3 : 0);
            } else {
              CccTxEn.Data16 = MRC_HS_DDR4_CCC57_CA_MSK;
              CccTxEn.Ddr4.HS.Ccc1357.Clk0or2 = (Rank2Present ? 0x3 : 0);
              CccTxEn.Ddr4.HS.Ccc1357.Clk1or3 = (Rank3Present ? 0x3 : 0);
            }
            break;

          case 4:
          case 6:
            CccTxEn.Data16 = MRC_HS_DDR4_CCC46_CA_MSK;
            CccTxEn.Ddr4.HS.Ccc46.Cs0 = Rank0Present;
            CccTxEn.Ddr4.HS.Ccc46.Cs1 = Rank1Present;
            CccTxEn.Ddr4.HS.Ccc46.Cs2 = Rank2Present;
            CccTxEn.Ddr4.HS.Ccc46.Cs3 = Rank3Present;
            CccTxEn.Ddr4.HS.Ccc46.Odt0 = Rank0Present;
            CccTxEn.Ddr4.HS.Ccc46.Odt1 = Rank1Present;
            CccTxEn.Ddr4.HS.Ccc46.Odt2 = Rank2Present;
            CccTxEn.Ddr4.HS.Ccc46.Odt3 = Rank3Present;
            break;

            default:
              // Do nothing.  Will catch 0 in this as a failure below the DdrType Ladder.
              break;
        }
      } else {
        // Ddr5
        switch (Index) {
          case 0:
          case 2:
            CccTxEn.Data16 = MRC_HS_DDR5_CCC02_CA_MSK;
            CccTxEn.Ddr5.Ccc02.Cs0 = Rank0Present;
            CccTxEn.Ddr5.Ccc02.Cs1 = Rank1Present;
            CccTxEn.Ddr5.Ccc02.Cs3 = Rank3Present;
            CccTxEn.Ddr5.Ccc02.Clk2 = (Rank2Present ? 0x3 : 0);
            CccTxEn.Ddr5.Ccc02.Clk3 = (Rank3Present ? 0x3 : 0);
            break;

          case 1:
          case 3:
            CccTxEn.Data16 = MRC_HS_DDR5_CCC13_CA_MSK;
            CccTxEn.Ddr5.Ccc13.Cs2 = Rank2Present;
            CccTxEn.Ddr5.Ccc13.Clk0 = (Rank0Present ? 0x3 : 0);
            CccTxEn.Ddr5.Ccc13.Clk1 = (Rank1Present ? 0x3 : 0);
            break;

          case 4:
          case 6:
            CccTxEn.Data16 = MRC_HS_DDR5_CCC46_CA_MSK;
            CccTxEn.Ddr5.Ccc46.Cs0 = Rank0Present;
            CccTxEn.Ddr5.Ccc46.Cs1 = Rank1Present;
            CccTxEn.Ddr5.Ccc46.Clk2 = (Rank2Present ? 0x3 : 0);
            CccTxEn.Ddr5.Ccc46.Clk3 = (Rank3Present ? 0x3 : 0);
            break;

          case 5:
          case 7:
            CccTxEn.Data16 = MRC_HS_DDR5_CCC57_CA_MSK;
            CccTxEn.Ddr5.Ccc57.Cs2 = Rank2Present;
            CccTxEn.Ddr5.Ccc57.Cs3 = Rank3Present;
            CccTxEn.Ddr5.Ccc57.Clk0 = (Rank0Present ? 0x3 : 0);
            CccTxEn.Ddr5.Ccc57.Clk1 = (Rank1Present ? 0x3 : 0);
            break;

            default:
              // Do nothing.  Will catch 0 in this as a failure below the DdrType Ladder.
              break;
        }
      }
    }

    if (CccTxEn.Data16 == 0) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "%s CCC%d TxEn not initialized properly\n", gErrString, Index);
      if (Inputs->ExitOnFailure) {
        return mrcFail;
      }
    }

    // Handle LP4/5 Clock and CKE/WCK since it is common.
    if (Lpddr) {
      CccTxEn.CccCommonLp.Clk = 0x3;
      CccTxEn.CccCommonLp.CkeWck = (Lpddr4) ? RankPresent : 0x3;
    }

    TxEn = CccTxEn.Data16;
    CccMux = 0;
    switch (DdrType) {
      case MRC_DDR_TYPE_DDR4:
        CccMux = 0;
        break;
      case MRC_DDR_TYPE_LPDDR4:
        CccMux = 1;
        break;

      case MRC_DDR_TYPE_LPDDR5:
        if (UlxUlt) {
          CccMux = ((Inputs->Lp5CccConfig >> Index) & 0x1) ? 3 : 2;
        } else {
          // DtHalo
          CccMux = 2;
        }
        break;

      case MRC_DDR_TYPE_DDR5:
        if (DtHalo) {
          CccMux = 3;
        }
        break;

      default:
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s %s - %s:%d\n", gErrString, gUnsupportedTechnology, __FILE__, __LINE__);
        return mrcUnsupportedTechnology;
        break;
    }

    KeepXXClkOn = (Ddr4) ? 1 : 0;
    Delay = ((40 + Inputs->RcompTarget[RdOdt]) * 10) / 2; //@todo - Create a variable for DDR Init.
    Delay = Delay * 1000; // Scale to pS
    Delay = DIVIDEROUND (Delay, RefClkPs);
    Delay = MrcLog2 (Delay) - 2;
    TimerXXClk = RANGE (Delay, 0, 7);
    TimerXXClk = 0; // @todo_adl non-zero values cause 'x' on DdrCrTimerXXClkQnnnH

    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRPINSUSED_REG, CH1CCC_CR_DDRCRPINSUSED_REG, CccIndexToFub[Index]);
    CccPinsUsed.Data = 0;
    CccPinsUsed.Bits.TxEn         = (Inputs->SafeMode) ? CH0CCC_CR_DDRCRPINSUSED_TxEn_MAX : TxEn;
    CccPinsUsed.Bits.CCCMuxSelect = CccMux;
    CccPinsUsed.Bits.PiEn         = 0;
    CccPinsUsed.Bits.PiEnOvrd     = 0;
    CccPinsUsed.Bits.Gear1        = Gear1;        // @todo_adl Gear4 is in this CR as well
    CccPinsUsed.Bits.KeepXXClkOn  = KeepXXClkOn;
    CccPinsUsed.Bits.TimerXXClk   = TimerXXClk;
    MrcWriteCR (MrcData, Offset, CccPinsUsed.Data);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "0x%04X: CH%dCCC_DDRCRPINSUSED: 0x%08X\n", Offset, Index, CccPinsUsed.Data);
  } // CCC Instance
  GetSetVal = IoChNotPop;
  MrcGetSetNoScope (MrcData, GsmIocChNotPop, WriteCached | PrintValue, &GetSetVal);

  GetSetVal = (Outputs->EccSupport) ?  1 : 0;
  MrcGetSetNoScope (MrcData, GsmIocEccEn, WriteToCache | PrintValue, &GetSetVal);

  MrcFlushRegisterCachedData (MrcData);

  Status |= MrcDccSetup (MrcData);

  return Status;
}

#define TXEQ_CONSTATNTZ    (1 << 4)
#define TXEQ_NODEEMPHASIS   0xC

/**
  This function initializes the Memory Controller Phy.

  @param[in, out] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcDdrIoInit (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcInput          *Inputs;
  MrcDebug          *Debug;
#ifndef AV_PHY_INIT
  MrcIntOutput      *IntOutputs;
  MrcIntCmdTimingOut *IntCmdTiming;
#endif
  MrcStatus         Status;
  MrcOutput         *Outputs;
  MrcChannelOut     *ChannelOut;
  MrcControllerOut  *ControllerOut;
  MrcTiming         *TimingProfile;
  MrcProfile        Profile;
  MrcVddSelect      Vdd;
  MrcVddSelect      DramVdd;
//  MrcFrequency      DdrFrequency;
  MrcDdrType        DdrType;
  INT64             GetSetVal;
  INT64             Roundtrip;
  INT64             GetSetEn;
//  INT64             RXDeskewForceOn;
  INT64             GetSetDis;
  INT64             RcvEnPi[MAX_CONTROLLER][MAX_CHANNEL];
  INT64             tCL4RcvEn;
  INT64             tCL4RxDqFifoRdEn;
  INT64             tCWL4TxDqFifoWrEn;
  INT64             tCWL4TxDqFifoRdEn;
  INT64             RxDqFifoRdEnRankDel;
#if !defined(AV_PHY_INIT)
  INT64             MaxVal;
  INT64             MinVal;
  INT64             RxDqsPPi;
  INT64             RxDqsNPi;
  INT64             RxEqInit;
  INT64             RxCInit;
  INT64             RxRInit;
  INT64             RxTap0Init;
  INT64             RxTap1Init;
  INT64             RxDqsBitPi;
  INT64             TxEqInit;
  INT64             TxDqPi;
  INT64             TxDqsPi;
  INT64             TxDqBitPi;
  INT64             TxDqsBitPi;
#endif
#ifndef AV_PHY_INIT
  INT64             RxDataValidDclk;
  INT64             RxDataValidQclk;
  INT64             GetSetVal2;
  INT64             GetSetVal3;
  INT64             GetSetVal4;
  INT64             GetSetVal5;
  UINT32            Index;
  UINT32            OdtTarget;
#endif
  INT32             TdqsckMin;
  INT32             tCL;
  INT32             tCWL;
//  UINT32            Data;
  UINT32            Controller;
  UINT32            Offset;
  UINT32            Data32;
//  UINT32            Itarget;
//  UINT32            IcompUp;
//  UINT32            IcompDn;
  UINT32            EffPullUp;
  UINT32            Gear2;
  UINT32            Voh;
  UINT32            VccIoMv;
  UINT32            PuDeltaV;
  UINT32            IpChannel;
  UINT32            PiToQclk;
  UINT32            RefClkMhz;
//  UINT32            RefClkPs;
//  UINT32            QclktoFreq;
//  UINT32            HiBwDiv;
//  UINT32            LowBwDiv;
//  UINT32            SampleDiv;
  UINT32            TempVar1;
  UINT32            TempVar2;
//  UINT32            TempVar3;
//  UINT32            TempVar4;
  UINT32            AddTcwl;
  UINT32            DecTcwl;
  UINT32            Byte;
  UINT32            CCCBlock;
//  UINT32            PBDStepSize;
//  UINT32            RExternal;
//  UINT32            CaRonDrvDn;
//  UINT32            CaRonDrvUp;
//  UINT32            TargetV0;
//  UINT32            VttTargetV;
  UINT32            Tdqs2dqMinFs;
//  UINT32            TxDqPUpPreDrvVccddq;
//  UINT32            DisVddqPathWithVddq;
//  UINT32            VssHiBypassVddqMode;
//  UINT32            VssHiBypassVdd2Mode;
//  INT32             VsxHiTargetMv;
//  INT32             SignedTempVar;
  INT32             SignedTempVar1;

  UINT32            Vdd2Mv;
//  UINT32            CCCPredrvUseVcciog;
//  UINT32            SevenPercentOfVccIoMv;
//  UINT32            CBTune0;
//  UINT32            ByteStart;
//  UINT32            ByteEnd;
  UINT32            FirstController;
  UINT32            FirstChannel;
//  UINT32            FirstByte;
//  UINT32            VtOffset;
//  UINT32            VtSlope;
  UINT16            CtlPiDelay;
  UINT16            ClkDelay;
  UINT16            NMode;
  UINT16            VssHiSwingTarget;
  UINT16            SAFE;
  UINT16            VssHi;  // Target VssHi Voltage
  UINT16            VssHiMax;
  UINT16            VccddqMv;
  UINT16            Rodt;
  UINT16            RonCtlTarget;
  UINT16            QclkPs;
//  UINT8             delta;
  UINT8             Channel;
#ifndef AV_PHY_INIT
  UINT8             Rank;
  BOOLEAN           StaticOdtDis;
#endif
//  UINT8             ResetNumPre;
//  UINT8             NumToggles;
  UINT8             MaxChannels;
//  UINT8             RcvEnWaveShape;
//  UINT8             PassRcvEn;
//  UINT8             Rup;
//  UINT8             RightShift;
  UINT8             DataInvertNibble;
//  UINT8             TargetNUI;
//  UINT8             CtlVoltSel;
//  UINT8             EnDqsNRcvEnSet;
  UINT8             BlockSdlWithRcvEn;
//  BOOLEAN           ObeyDDQBypassB0;
//  BOOLEAN           ObeyDDQBypassA0;
  BOOLEAN           A0;
  BOOLEAN           Lpddr;
  BOOLEAN           Ddr;
  BOOLEAN           Lpddr4;
  BOOLEAN           Lpddr5;
  BOOLEAN           Ddr4;
  BOOLEAN           Ddr5;
  BOOLEAN           UlxUlt;
  MRC_RX_MODE_TYPE  RxMode;
//  BOOLEAN           Is2DPC;
  BOOLEAN           UnMatched;
//  BOOLEAN           NPath;
//  BOOLEAN           EnDqsOdtParkMode;
  MCMISCS_DDRWCKCONTROL_STRUCT                    WckControl;
//  DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_STRUCT        VssHiSampler;
//  DDRVREF_CR_DDRCRVREFCONTROL_STRUCT              VrefControl;
//  DDRVREF_CR_DDRCRVREFADJUST1_STRUCT              VrefAdj1;
//  DDRVREF_CR_DDRCRVREFCH0_P0_STRUCT               VrefCh0;
//  DDRVREF_CR_DDRCRVREFCH0_P0_STRUCT               VrefCh1;
//  DDRVREF_CR_DDRCRVREFADJUST2_STRUCT              VrefAdj2;
//  DLLDDRDATA0_CR_DDRCRVCCDLLFFCONTROL_STRUCT      VccDllFFControl;
//  DLLDDRDATA0_CR_DDRCRVCCDLLFFNBIAS_STRUCT        VccDllFFNBias;
//  DDRVCCDLL0_CR_DDRCRVCCDLLCONTROL_STRUCT         VccDllControl;
//  DDRVCCDLL0_CR_DDRCRVCCDLLSAMPLER_STRUCT         VccDllSampler;
//  DLLDDRDATA0_CR_DDRCRVCCDLLVSXHIFF_STRUCT        VccDllVsxHiFF;
//  DLLDDRDATA0_CR_DDRCRVCCDLLCOMPDATACCC_STRUCT    VccDllCompDataCCC;
//  DLLDDRDATA0_CR_DDRCRVCCDLLCOUPLINGCAP_STRUCT    VccDllCouplingCap;
//  DLLDDRDATA0_CR_DDRCRVCCDLLCOMPDLL_STRUCT        VccDllCompDll;
//  DDRPHY_COMP_CR_DDRCRCOMPTEMP_STRUCT             CompTemp;
//  DDRPHY_COMP_CR_DDRCRCOMPCTL0_STRUCT             CompCtl0;
//  DDRPHY_COMP_CR_DDRCRCOMPCTL1_STRUCT             CompCtl1;
//  DDRPHY_COMP_CR_DDRCRCOMPCTL2_STRUCT             CompCtl2;
//  DDRPHY_COMP_CR_DDRCRCOMPCTL4_STRUCT             CompCtl4;
//  DDRPHY_COMP_CR_VSSHIPANIC_STRUCT                VssHiPanic;
//  DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_STRUCT        CompDvfsRload;
//  DATA0CH0_CR_DDRCRDATACONTROL0_STRUCT            DataControl0;
//  DATA0CH0_CR_DDRCRDATACONTROL2_STRUCT            DataControl2;
//  DATA0CH0_CR_DDRCRDATACONTROL3_STRUCT            DataControl3;
//  DATA0CH0_CR_DDRCRDATACONTROL5_STRUCT            DataControl5;
//  DATA0CH0_CR_DDRCRDATACONTROL6_STRUCT            DataControl6;
//  DATA0CH0_CR_DDRDATADQSRANK0_STRUCT              DataDqsRankX;
//  DATA0CH0_CR_DDRCRDATATCOCONTROL1_STRUCT         TcoControl1;
  DATA0CH0_CR_DDRCRWRRETRAINCONTROLSTATUS_STRUCT  WrRetrainControlStatus;
//  DATA0CH0_CR_DDRCRMARGINMODECONTROL0_STRUCT      MarginModeControl;
//  DDRPHY_COMP_CR_VSSHITARGET_STRUCT               VssHiTarget;
//  DDRPHY_COMP_CR_VCCDLLTARGET_STRUCT              VccDLLTarget;
//  DDRVTT0_CR_DDRCRVTTCOMPOFFSET2_STRUCT           VttCompOffset2;
//  DDRVTT0_CR_DDRCRVTTCOMPOFFSET_STRUCT            VttCompOffset;
//  CH0CCC_CR_DDRCRCCCPERBITDESKEW0_STRUCT          CCCPerBitDeskew0; // @todo deskewcal-Program after SAComp
//  CH0CCC_CR_DDRCRCCCPERBITDESKEW1_STRUCT          CCCPerBitDeskew1; //@todo uncomment after TurnOnDelay value is provided
//  CH0CCC_CR_DDRCRCCCPERBITDESKEW2_STRUCT          CCCPerBitDeskew2;
  CH0CCC_CR_DDRCRCCCCLKCONTROLS_STRUCT            CCCClkControls;
  CH0CCC_CR_DDRCRBSCANDATA_STRUCT                 CCCBscanData;
//  CH0CCC_CR_DDRCRBSCANDATA_P0_STRUCT              CCCBscanDataP0;
//  CH0CCC_CR_DDRCRMISR_STRUCT                      CCCMisr;
//  DDRVSSHIAFEA_CR_DDRCRVSSHICONTROL_STRUCT        VssHiCtl;
  DDRSCRAM_CR_DDRMISCCONTROL1_STRUCT              MiscControl1;
  DDRSCRAM_CR_DDRMISCCONTROL2_STRUCT              MiscControl2;
  DDRSCRAM_CR_DDRMISCCONTROL7_STRUCT              MiscControl7;
  M_COMP_PCU_STRUCT                               CrMCompPcu;
  MC0_CH0_CR_SC_WR_DELAY_STRUCT                   ScWrDelay;
//  DLLDDRDATA0_CR_DLLPITESTANDADC_STRUCT           DataDllPiTestAndAdc;
//  DDRPHY_COMP_CR_VCCDLLREPLICACTRL0_STRUCT        VccDllReplicaCtrl0;
//  DDRPHY_COMP_CR_VCCDLLREPLICACTRL1_STRUCT        VccDllReplicaCtrl1;
//  DDRPHY_COMP_CR_VCCDLLDQSDELAY_STRUCT            VccDllDqsDelay;
//  CH0CCC_CR_DDRCRCCCVOLTAGEUSED_STRUCT            CCCVoltageUsed;
//  DDRVTT0_CR_DDRCRVTTGENCONTROL_STRUCT            VttGenControl;
//  DDRPHY_COMP_CR_DDRCRALERT_STRUCT                CompAlert;
  MCMISCS_SPINEGATING_STRUCT                      McMiscSpineGating;
  DDRSCRAM_CR_DDRSCRAMBLECH0_STRUCT               DdrScramCh0;
  DDRSCRAM_CR_DDRSCRAMBLECH1_STRUCT               DdrScramCh1;
  DDRSCRAM_CR_DDRSCRAMBLECH2_STRUCT               DdrScramCh2;
  FLL_CMD_CFG_REG_STRUCT                          FllCmdCfg;
  FLL_STATIC_CFG_0_REG_STRUCT                     FllStaticCfg0;
  FLL_STATIC_CFG_1_REG_STRUCT                     FllStaticCfg1;
//  DDRPHY_COMP_CR_DDRCRFLLWIRED_STRUCT             FllWired;
//  DATA0CH0_CR_DCCPILUT4_STRUCT                    PiLut4;
//  DDRPHY_COMP_CR_VCCDLLREPLICACTRL2_STRUCT        VccDllReplicaCtrl2;
//  DDRPHY_COMP_CR_DDRCRVIEWCTL_STRUCT              ViewCtl;
//  DDRPHY_COMP_CR_DDRCRDATACOMP0_STRUCT            RcompData0;
//  DDRPHY_COMP_CR_DDRCRDATACOMP1_STRUCT            RcompData1;
#ifdef MRC_DEBUG_PRINT
//  DDRPHY_COMP_CR_DDRCRDATACOMPVTT_STRUCT  DataCompVtt;
#endif //MRC_DEBUG_PRINT

  Inputs            = &MrcData->Inputs;
  Outputs           = &MrcData->Outputs;
  Debug             = &Outputs->Debug;
#ifndef AV_PHY_INIT
  IntOutputs        = (MrcIntOutput *) (MrcData->IntOutputs.Internal);
#endif
  Profile           = Inputs->MemoryProfile;
  DdrType           = Outputs->DdrType;
  A0                = Inputs->A0;
  Status            = mrcSuccess;
  Vdd               = (Outputs->Lp4x) ? VDD_1_10 : Outputs->VddVoltage[Profile];
  Lpddr4            = (DdrType == MRC_DDR_TYPE_LPDDR4);
  Lpddr5            = (DdrType == MRC_DDR_TYPE_LPDDR5);
  Ddr4              = (DdrType == MRC_DDR_TYPE_DDR4);
  Ddr5              = (DdrType == MRC_DDR_TYPE_DDR5);
  Lpddr             = (Lpddr4 || Lpddr5);
  Ddr               = (Ddr4 || Ddr5);
  RxMode            = Outputs->RxMode;
  QclkPs            = Outputs->Qclkps;
  GetSetEn          = 1;
  GetSetDis         = 0;
  SAFE              = 0;
  VssHiSwingTarget  = 950;  // VssHi target voltage in mV
//  delta             = 20;   // VssHi change voltage during panic, in mV
  VccddqMv          = Outputs->VccddqVoltage;
  VccIoMv           = Inputs->VccIomV;
  Vdd2Mv            = Outputs->Vdd2Mv;
  //Vss               = (Outputs->OdtMode == MrcOdtModeVss);
  //DqPinsInterleaved = (Inputs->DqPinsInterleaved != 0);
//  DdrFrequency      = (Lpddr && Inputs->LpFreqSwitch) ? Outputs->HighFrequency : Outputs->Frequency;
  //EnVttOdt          = (Outputs->OdtMode == MrcOdtModeVtt);
  //CbenVal           = 3;
  UlxUlt            = Inputs->UlxUlt;
//  DtHalo            = Inputs->DtHalo;
  Gear2             = Outputs->Gear2 ? 1 : 0; // Used to add extra clock(s) in Gear2 mode
  MaxChannels       = Outputs->MaxChannels;
  PiToQclk          = 64 * (1 + Gear2);
  NMode             = 0;
  tCL               = 0;
  tCWL              = 0;
//  VsxHiTargetMv     = Vdd2Mv - VccIoMv;
//  VttTargetV        = Ddr5 ? (Vdd2Mv / 2) : 150; //150mV
//  CaRonDrvDn        = Inputs->RcompTarget[WrDSCmd];
//  Itarget           = CDIEVSSHI * delta / Outputs->UIps; // Target Current.  pF * mV / pS -> mA
//  CaRonDrvUp        = Inputs->RcompTarget[WrDSCmd]; // @todo whats the difference between Up and Dn
  DataInvertNibble  = 0;
  //  Vcm = 5;
  //  TxPwrMuxSelVddq = 0; //@todo find value
  //  TurnOnDelay = 5;
  //  Ron = Inputs->RcompTarget[WrDS];
  Rodt = Inputs->RcompTarget[RdOdt];
  RonCtlTarget = Inputs->RcompTarget[WrDSCtl];
  //  Vtester = 5;
//  Rup = 47;

  // Temporary Variables init
  TempVar1 = 0;
  TempVar2 = 0;
//  TempVar3 = 0;
//  TempVar4 = 0;
//  SignedTempVar = 0;
  SignedTempVar1 = 0;
//  RXDeskewForceOn = 1;
  TimingProfile = 0;
//  GetSetVal = 0;
//  CBTune0 = 0;
//  PBDStepSize = 5;
//  RExternal = Inputs->RcompResistor; //100 ohms
  RefClkMhz = Outputs->RefClk;
//  ByteStart = 0;
//  ByteEnd = Outputs->SdramCount;
//  QclktoFreq = 1000000 / QclkPs;
  SignedTempVar1 = VccIoMv - (UINT32) VccddqMv;
//  SevenPercentOfVccIoMv = (7 * VccIoMv) / 100;
  TempVar1 = ABS (SignedTempVar1);
//  TxDqPUpPreDrvVccddq = (TempVar1 != 0) ? (TempVar1 <= SevenPercentOfVccIoMv) : 1; // Set default value of DataControl5.TxDqPUpPreDrvVccddq if VccIoMv < VccddqMv
//  DisVddqPathWithVddq = (VsxHiTargetMv < (INT32) ((7 * Vdd2Mv) / 100));
//  VssHiBypassVddqMode = (VccddqMv >= VDD_1_20) ? 0 : (VccddqMv >= VDD_1_10) ? (DdrFrequency >= f4500) : (VccddqMv >= VDD_1_05) ? (DdrFrequency >= f2800) : 1;
//  VssHiBypassVdd2Mode = (Vdd2Mv >= VDD_1_20) ? 0 : (Vdd2Mv >= VDD_1_10) ? (DdrFrequency >= f6400) : (Vdd2Mv >= VDD_1_05) ? (DdrFrequency >= f4500) : 1;
//  CtlVoltSel = Lpddr4 ? 1 : Lpddr5 ? (VccddqMv < VDD_0_75) : 0;
  // Default RefClk = 100 MHz so RefClkPs = 1/100 MHz = 10ns or 10,000ps
  // If RefClk = 100 MHz then RefClkPs = 10000ps , If RefClk = 133 MHz then RefClkPs = 7500ps
//  RefClkPs  = (RefClkMhz == MRC_REF_CLOCK_133) ? 7500 : 10000; // Reference Clock Period in pS
#ifndef AV_PHY_INIT
  OdtTarget = Inputs->RcompTarget[RdOdt];
  StaticOdtDis = (OdtTarget > 120);
#endif

//#ifdef AV_PHY_INIT
//  MrcDdrIoPreInit (MrcData);
//#endif

  MrcPhyInitSeq1 (MrcData);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "MrcPhyInitSeq1 Done\n");



  if (Inputs->SafeMode) {
    SAFE = 0xFFFF;
  }
  UnMatched = ((RxMode == MrcRxModeUnmatchedRxWRload) || (RxMode == MrcRxModeUnmatchedRxWPpath));
//  NPath     = (RxMode == MrcRxModeMatchedN);
  BlockSdlWithRcvEn = (Lpddr && UnMatched && !A0) ? 1 : 0;
  TempVar1 = BlockSdlWithRcvEn ? 5500 : 6800;
  TempVar2 = UnMatched ? TempVar1 : 11000;
//  TempVar3 = BlockSdlWithRcvEn ? 1200 : 1900;
//  TempVar4 = UnMatched ? TempVar3 : 2500;
//  EnDqsNRcvEnSet = Lpddr && ((DdrFrequency > (Lpddr4 ? TempVar4 : TempVar2)));

  Outputs->ValidMcBitMask = 0;
  Outputs->ValidChBitMask = 0;
  Outputs->ValidRankMask  = 0;
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    if (MrcControllerExist (MrcData, Controller)) {
      ControllerOut = &Outputs->Controller[Controller];
      Outputs->ValidMcBitMask |= (1 << Controller);
      ControllerOut->FirstPopCh = MAX_CHANNEL;
      for (Channel = 0; Channel < MaxChannels; Channel++) {
        if (MrcChannelExist (MrcData, Controller, Channel)) {
          ChannelOut = &ControllerOut->Channel[Channel];
          // Initialize ValidChBitMask and ValidRankMask used during all training steps
          // Program various Rank Occupancy / RanksUsed bits.
          ControllerOut->ValidChBitMask |= (1 << Channel);
          ControllerOut->FirstPopCh = MIN (Channel, ControllerOut->FirstPopCh);
          Outputs->ValidChBitMask |= (ControllerOut->ValidChBitMask << (Controller * Outputs->MaxChannels));
          Outputs->ValidRankMask  |= ChannelOut->ValidRankBitMask;
          MRC_DEBUG_MSG (
            Debug,
            MSG_LEVEL_NOTE,
            "Mc%u.C%u: ValidRankBitMask=0x%x, Overall ValidRankMask=0x%x\n",
            Controller,
            Channel,
            ChannelOut->ValidRankBitMask,
            Outputs->ValidRankMask
            );
          // Determine NMode.  Constraint is timing must be the same in MC/CH
          NMode = ChannelOut->Timing[Profile].NMode;
        } // MrcChannelExist
      } // Channel
      MRC_DEBUG_MSG (
        Debug,
        MSG_LEVEL_NOTE,
        "Mc%u: ValidChBitMask=0x%x\n",
        Controller,
        ControllerOut->ValidChBitMask
        );
    } // MrcControllerExists
  } // Controller
  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE,
    "ValidMcBitMask=0x%x, Overall ValidChBitMask=0x%x\n",
    Outputs->ValidMcBitMask,
    Outputs->ValidChBitMask
    );

  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = Outputs->Controller[FirstController].FirstPopCh;
//  FirstByte = (MAX_BYTE_IN_LP_CHANNEL * FirstChannel);

  // Read the first existing register
  CCCBlock = ((MAX_CCC_PER_CHANNEL * FirstController) + FirstChannel);
//  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCPERBITDESKEW2_REG, CH1CCC_CR_DDRCRCCCPERBITDESKEW2_REG, CCCBlock);
//  CCCPerBitDeskew2.Data = MrcReadCR (MrcData, Offset);
//  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRMISR_REG, CH1CCC_CR_DDRCRMISR_REG, CCCBlock);
//  CCCMisr.Data = MrcReadCR (MrcData, Offset);
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRBSCANDATA_REG, CH1CCC_CR_DDRCRBSCANDATA_REG, CCCBlock);
  CCCBscanData.Data = MrcReadCR (MrcData, Offset);

  // Overload Byte here to hold the Max byte in channel and then do the instance calculation at the end.
  // Save duplicating the multiplication instruction.
  if (MrcData->Outputs.Lpddr) {
    Byte = MAX_BYTE_IN_LP_CHANNEL;
  } else if (Ddr5) {
    Byte = MAX_BYTE_IN_DDR5_CHANNEL;
  } else {
    Byte = 0; // By default if Channel is present then first byte will be present
  }
  Byte = (Byte * FirstChannel);

//  Offset = DataControl3Offset (Controller, Byte);
//  DataControl3.Data = MrcReadCR (MrcData, Offset);

//  Offset = DataControl2Offset (Controller, Byte);
//  DataControl2.Data = MrcReadCR (MrcData, Offset);

// @todo_adl  Offset = CBTune0Offset (CCCBlock);
// @todo_adl  CBTune0 = MrcReadCR (MrcData, Offset);

//  PiLut4.Data = 0;
//  PiLut4.Bits.LUTHalfStep = (QclktoFreq >= 2100) ? 1 : 0;
//  MrcWriteCrMulticast (MrcData, DATA_CR_DCCPILUT4_REG, PiLut4.Data);

  // Set CCC PI values.  Will be consumed and configured below in various stages
  if (Ddr4) {
    // 3N Timing
    CtlPiDelay = 64;
//    ClkPiDelay = 128;
//    CmdPiDelay = 128;
  } else if (Lpddr5) {
//    ClkPiDelay = 256;
//    CmdPiDelay = 128;
    CtlPiDelay = 128;
  } else if (Lpddr4) {
//    ClkPiDelay = 160;
//    CmdPiDelay = 128;
    CtlPiDelay = 128;
  } else {
    if (NMode == 2) {
//      ClkPiDelay = 192;
//      CmdPiDelay = 128;
      CtlPiDelay = 149;
    } else {
      // 2N
//    ClkPiDelay = 192;
//    CmdPiDelay = 128;
    CtlPiDelay = 128;
    }
  }
  // Only programmed in LP5.  Just initializing here to avoid compile issues and make code smaller instead of if-ladder
//  WckPiDelay = 384;

  /* @todo_adl
  VssHiCtl.Data = MrcReadCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHICONTROL_REG);
  Data32 = 128 * VsxHiTargetMv;
  Data32 = DIVIDEROUND(Data32, VccIoMv);
  VssHiCtl.Bits.Target = Data32;
  VssHiCtl.Bits.BWError = 1;
  VssHiCtl.Bits.PanicEn = 1;
  VssHiCtl.Bits.EnDacPM = ((A0) || SAFE) ? 0 : 2;
  VssHiCtl.Bits.EnCkePanic = ((A0) || SAFE) ? 1 : 0;
  VssHiCtl.Bits.GainBoost = 1;
  VssHiCtl.Bits.SelCode = 1;
  VssHiCtl.Bits.UseMax = 1;
  if (A0) {
    VssHiCtl.Bits.SampleDivider = 0;
  }
  VssHiCtl.Bits.LoBWDivider = 0;
  MrcWriteCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHICONTROL_REG, VssHiCtl.Data);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "DdrCrVsshiControl: 0x%X\n", Data32);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Init Dimm Vref CRs\n");
  VssHiSampler.Data = 0;
  TempVar1 = 382 * VsxHiTargetMv;
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv) - 32;
  VssHiSampler.Bits.Target = TempVar2;
  TempVar1 = 382 * PANICV0;
  TempVar2 = UDIVIDEROUND (TempVar1 , Vdd2Mv);
  VssHiSampler.Bits.Panic = TempVar2;
  VssHiSampler.Bits.PanicLoUsePmos     = ((VsxHiTargetMv + PANICV0) < 375);
  VssHiSampler.Bits.VssHiTargetUsePmos = ((VsxHiTargetMv + PANICV0) < 375);
  VssHiSampler.Bits.PanicHiUsePmos     = ((VsxHiTargetMv + PANICV0) < 375);
  TempVar1 = VssHiCtl.Bits.SampleDivider + 3;
  TempVar2 = 1 << TempVar1;
  TempVar3 = DIVIDEROUND (20000, QclkPs); // Scale 20ns to ps
  TempVar4 = DIVIDEROUND (TempVar3, TempVar2);
  if (TempVar4 > 0) {
    TempVar4--;
  }
  TempVar4 = MAX (TempVar4, 7);
  VssHiSampler.Bits.WakeUpDelay = TempVar4;
  TempVar1 = 120000 / RefClkPs;
  TempVar2 = MrcLog2 (TempVar1);
  if (TempVar2 > 0) {
    TempVar2--;
  }
  TempVar2 = MAX (TempVar2, 3);
  VssHiSampler.Bits.LockTimer = TempVar2;
  VssHiSampler.Bits.AckOffset = 1;
  MrcWriteCR (MrcData, DDRVSSHIAFEA_CR_DDRCRVSSHISAMPLER_REG, VssHiSampler.Data);

  ViewCtl.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRVIEWCTL_REG);
  ViewCtl.Bits.TcoComp0 = 0x20;
  ViewCtl.Bits.TcoComp1 = 0x20;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRVIEWCTL_REG, ViewCtl.Data);

  // Reset defaults are non-zero.  Read first instance of the register to preserve those reset values and multicast the updated field.
  if (!A0) {
    TcoControl1.Data = MrcReadCR (MrcData, DATA0CH0_CR_DDRCRDATATCOCONTROL1_REG);
    SignedTempVar = 110 - DIVIDEROUND (VsxHiTargetMv, 4);
    TcoControl1.Bits.TxVsxHiLeakerComp = RANGE (SignedTempVar, 0, 63);
    MrcWriteCR (MrcData, DATA_CR_DDRCRDATATCOCONTROL1_REG, TcoControl1.Data);
  }
*/
//  DataControl0.Data = 0;
  /*
  if (SAFE) {
    DataControl0.Bits.local_gate_d0tx = 1;
    DataControl0.Bits.InternalClocksOn = 1;
    DataControl0.Bits.tx_clock_on_with_txanalogen = 1;
    DataControl0.Bits.RepeaterClocksOn = 1;
    DataControl0.Bits.TxDeskewForceOn = 1;
    DataControl0.Bits.EnDqOdtParkMode = 1;
    DataControl0.Bits.ForceRxOn = 1;
    DataControl0.Bits.TxPiOn = 1;
    DataControl0.Bits.EnDqsOdtParkMode = 1;
    DataControl0.Bits.BiasPMCtrl = 0;
    DataControl0.Bits.VrefPMCtrl = 0;
  } else {
    DataControl0.Bits.EnDqsOdtParkMode = (Lpddr4) ? 2 : (Lpddr5 ? 3 : 0); //DDR4 sets 0 and DDR5 sets ???
    DataControl0.Bits.BiasPMCtrl = (!Lpddr && A0) ? 2 : 3;
    DataControl0.Bits.VrefPMCtrl = 1; // HVM (IOLB w/ CKE power down) experiments try 0x2 and measure EH
    if ((!A0) && Lpddr & EnDqsNRcvEnSet && Inputs->IoOdtMode == MrcOdtModeVtt) {
      DataControl0.Bits.DqsOdtVss = 1;
    }
  }
  DataControl0.Bits.WlLongDelEn = 1;
  EnDqsOdtParkMode = ((DataControl0.Bits.EnDqsOdtParkMode == 1) || (DataControl0.Bits.EnDqsOdtParkMode == 2));
  DataControl0.Bits.DqsNParkLowVoh = (EnDqsOdtParkMode && Lpddr && (VccddqMv > VDD_0_75)) ? 1 : 0;
  // DataControl0.Bits.DisableOdtStatic = (OdtTarget > 120); // GetSet Programming down below
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATACONTROL0_REG, DataControl0.Data);
  */
  /*
  DataControl5.Bits.TxDqPUpPreDrvVccddq = TxDqPUpPreDrvVccddq;
  DataControl5.Bits.TxBypassVssHi = VssHiBypassVddqMode;
  DataControl5.Bits.DataInvertNibble = (UINT32) DataInvertNibble;
  //DataControl5.Bits.BiasBypassVsxHi = 0 ;
  DataControl5.Bits.TxDqPDnPreDrvVccddq = TxDqPUpPreDrvVccddq  & VssHiBypassVddqMode;
  TempVar1 = DataControl5.Bits.BiasPwrMuxSelVdd2 ? Vdd2Mv : VccIoMv;
  TempVar2 = 200000000 - (TempVar1 * 125000); //Scale it to nV
  TempVar3 = TempVar2/TempVar1;
  TempVar4 = DIVIDEROUND(TempVar3, 31250);
  DataControl5.Bits.BiasRloadVref = RANGE(TempVar4, 0, 7);
  // Need to program the bias for the matched receiver in unmatched when using DQS-N for ReceiveEnable.
  // It is ignore otherwise, so no harm to keep at 0xA always.
  DataControl5.Bits.BiasIrefAdj =  ((Ddr && NPath) || UnMatched) ? 0xA : 0x7;
  //DataControl5.Bits.BiasPwrMuxSelVdd2 = 0; // @todo : needs training
  DataControl5.Bits.BiasCasAdj = 2;
  DataControl5.Bits.ODTSingleSegEn = (OdtTarget > 55);
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATACONTROL5_REG, DataControl5.Data);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRXDeskewForceOn, WriteCached, &RXDeskewForceOn);
  */
  // @todo- where do we find RxDqsPiOffset ? Is this in a register ?  Whats unmatched ? Any MRC parameter for this one ?
  // FinalDqsPPICode  = RxDqsNDelayPi(Train) + RxDqsPiOffset(CR) + (Unmatched ? (Comp.RxDqsDelayN - 64*(DDRCOMP_CR_DDRCRDATACOMP0_0_0_0_MCHBAR.DqsNOffsetNUI+DDRDATA_CR_DDRCRDATAOFFSETTRAIN_0_0_0_MCHBAR.RxDqsPiUIOffset)) : 0)
//-----------------------------VccDLL CRs----------------------------------------------
  /*
  VccDllControl.Data = 0;

  TempVar1 = (MRC_VCCDLL_TARGET * 382);
  TempVar2 = DIVIDEROUND (TempVar1, MRC_VCCBG) - 256;
  VccDllControl.Bits.Target = RANGE (TempVar2, 0, 126);
  VccDllControl.Bits.HiBWDivider = 0; //TBD
  VccDllControl.Bits.LoBWDivider = 0; //TBD;
  TempVar2 = (2500 / QclkPs); // considering numerator 2.5 ns
  TempVar2 /= 2;
  TempVar3 = MrcLog2 (TempVar2);
  VccDllControl.Bits.SampleDivider = TempVar3;
  VccDllControl.Bits.EnRemoteSense = 1;
  VccDllControl.Bits.SelCode = 1;
  if (A0) {
    VccDllControl.Bits.SampleDivider = 3;
    VccDllControl.Bits.HiBWDivider = 2;
  }
  VccDllControl.Bits.Bypass = (SAFE) ? 1 : 0;
  MrcWriteCrMulticast (MrcData, DDRVCCDLL_CR_DDRCRVCCDLLCONTROL_REG, VccDllControl.Data);

  VccDllFFNBias.Data = 0;

  VccDllFFNBias.Bits.NbTarget = VccDllControl.Bits.Target;
  TempVar1 = QclktoFreq * 10;
  TempVar2 = (65 * 575) / 10;
  TempVar2 += TempVar1;
  VccDllFFNBias.Bits.NbiasLoad = DIVIDEROUND (TempVar2, 575);
  TempVar1 = (800000 / RefClkPs) - 5;
  TempVar2 = MrcLog2 (TempVar1);
  VccDllFFNBias.Bits.NbiasTimer0 = TempVar2;
  TempVar1 = (250000 / RefClkPs) - 4;
  TempVar2 = MrcLog2 (TempVar1);
  VccDllFFNBias.Bits.NbiasTimer1 = TempVar2;
  VccDllFFNBias.Bits.NbiasCPVccIOG = (VccIoMv >= 850);
  VccDllFFNBias.Bits.NbiasEnLeaker = 2;
  // @todo : This field needs to be programmed after Scomp execution
//  VccDllFFNBias.Bits.SideClkPulse = 0;
  VccDllFFNBias.Bits.NbiasFastStartup = 1;
  MrcWriteCrMulticast (MrcData, DLLDDR_CR_DDRCRVCCDLLFFNBIAS_REG, VccDllFFNBias.Data);

  VccDllSampler.Data = MrcReadCR (MrcData, DDRVCCDLL0_CR_DDRCRVCCDLLSAMPLER_REG);
  if (!A0) {
    VccDllSampler.Bits.PullDownSlope = 2;
  }
  VccDllSampler.Bits.FBOverlap = 2;
  TempVar1 = 382 * PANICV0;
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv);
  VccDllSampler.Bits.Panic = RANGE (TempVar2, 0, 31);
  VccDllSampler.Bits.GainBoost = 2;
  TempVar1 = MRC_VCCDLL_TARGET - 600;
  TempVar2 = DIVIDEFLOOR (TempVar1, 50);
  VccDllSampler.Bits.NSalCMCap = RANGE (TempVar2, 0, 7);
  TempVar1 = VssHiCtl.Bits.SampleDivider + 3;
  TempVar2 = 1 << TempVar1;
  TempVar3 = 28000 / QclkPs; // Scale 16ns to ps
  TempVar4 = DIVIDEROUND (TempVar3, TempVar2) - 1;
  VccDllSampler.Bits.WakeUpDelay = RANGE (TempVar4, 0, 7);
  TempVar1 = (1000000 / RefClkPs);
  TempVar1 -= 5;
  TempVar2 = MrcLog2 (TempVar1);
  VccDllSampler.Bits.LockTimer = TempVar2;
  VccDllSampler.Bits.CodeReset = 6;
  VccDllSampler.Bits.AckOffset = 1;
  if (A0) {
    VccDllSampler.Bits.LoBWDividerLP = VccDllControl.Bits.LoBWDivider;
  }
  MrcWriteCrMulticast (MrcData, DDRVCCDLL_CR_DDRCRVCCDLLSAMPLER_REG, VccDllSampler.Data);

  VccDllCompDll.Data = 0;
  SignedTempVar = Gear2 ? (DdrFrequency / 2) : DdrFrequency;
  VccDllCompDll.Bits.FFCodeIdle = (-SignedTempVar / 130) + 53;
  VccDllCompDll.Bits.FFCodePI   = (-SignedTempVar / 110) + 58;
  MrcWriteCrMulticast (MrcData, DDRVCCDLL_CR_DDRCRVCCDLLCOMPDLL_REG, VccDllCompDll.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLCOMPDLL_REG, VccDllCompDll.Data);

  VccDllCompDataCCC.Data = 0;
  VccDllCompDataCCC.Bits.FFCodeCCCDist = (DdrFrequency / 100) + 11;
  VccDllCompDataCCC.Bits.FFCodePBD = (DdrFrequency / 60) - 2;
  // @todo : Original condition was MatchedPath. Does that corrospond to both P and N matched or either of them matched
  VccDllCompDataCCC.Bits.FFCodeRead = ((RxMode == MrcRxModeMatchedP) || (RxMode == MrcRxModeMatchedN)) ? ((DdrFrequency / 120) + 25) : ((DdrFrequency / 210) + 21);
  VccDllCompDataCCC.Bits.FFCodeWrite = (DdrFrequency / 530) + 38;
  MrcWriteCrMulticast (MrcData, DLLDDR_CR_DDRCRVCCDLLCOMPDATACCC_REG, VccDllCompDataCCC.Data);

  VccDllVsxHiFF.Data = 0;
  TempVar1 = 36 + (2 * VccDllCompDataCCC.Bits.FFCodePBD);
  TempVar2 = DIVIDEROUND (TempVar1, 7);
  VccDllVsxHiFF.Bits.VsxHiFFCodePBD = TempVar2;
  TempVar1 = 800 + (20 * VccDllCompDataCCC.Bits.FFCodeRead);
  TempVar2 = DIVIDEROUND (TempVar1, 20);
  VccDllVsxHiFF.Bits.VsxHiFFCodeRead = TempVar2;
  TempVar1 = 700 + (15 * VccDllCompDataCCC.Bits.FFCodeWrite);
  TempVar2 = DIVIDEROUND (TempVar1, 20);
  VccDllVsxHiFF.Bits.VsxHiFFCodeWrite = TempVar2;
  TempVar1 = 35 + (1 * VccDllCompDll.Bits.FFCodePI);
  TempVar2 = DIVIDEROUND (TempVar1, 13);
  VccDllVsxHiFF.Bits.VsxHiFFCodePi = TempVar2;
  TempVar1 = 184 + (6 * VccDllCompDll.Bits.FFCodeIdle);
  TempVar2 = DIVIDEROUND (TempVar1, 18);
  VccDllVsxHiFF.Bits.VsxHiFFCodeIdle = TempVar2;
  MrcWriteCrMulticast (MrcData, DLLDDR_CR_DDRCRVCCDLLVSXHIFF_REG, VccDllVsxHiFF.Data);

  VccDllCouplingCap.Data = 0;
  TempVar1 = 36 + (2 * VccDllCompDataCCC.Bits.FFCodePBD);
  TempVar2 = DIVIDEROUND (TempVar1, 33);
  VccDllCouplingCap.Bits.CapCancelCodePBD = TempVar2;
  TempVar1 = 800 + (20 * VccDllCompDataCCC.Bits.FFCodeRead);
  TempVar2 = DIVIDEROUND (TempVar1, 33);
  VccDllCouplingCap.Bits.CapCancelCodeRead = TempVar2;
  TempVar1 = 700 + (15 * VccDllCompDataCCC.Bits.FFCodeWrite);
  TempVar2 = DIVIDEROUND (TempVar1, 33);
  VccDllCouplingCap.Bits.CapCancelCodeWrite = TempVar2;
  TempVar1 = 35 + (1 * VccDllCompDll.Bits.FFCodePI);
  TempVar2 = DIVIDEROUND (TempVar1, 33);
  VccDllCouplingCap.Bits.CapCancelCodePi = TempVar2;
  TempVar1 = 184 + (6 * VccDllCompDll.Bits.FFCodeIdle);
  TempVar2 = DIVIDEROUND (TempVar1, 33);
  VccDllCouplingCap.Bits.CapCancelCodeIdle = TempVar2;
  MrcWriteCrMulticast (MrcData, DLLDDR_CR_DDRCRVCCDLLCOUPLINGCAP_REG, VccDllCouplingCap.Data); // Multicast offset used
  */
/*
  CompTemp.Data = 0;
  CompTemp.Bits.FFCodeDqsDelay = (DdrFrequency / 360) + 35;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPTEMP_REG, CompTemp.Data);

  CompCtl0.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL0_REG);
  CompCtl0.Bits.DQPredrvUseVcciog = (TxDqPUpPreDrvVccddq == 0);
  CompCtl0.Bits.DisVddqPathWithVddq = DisVddqPathWithVddq;
  CompCtl0.Bits.CCCVssHiBypassVdd2Mode = VssHiBypassVdd2Mode;
  CompCtl0.Bits.CCCVssHiBypassVddqMode = VssHiBypassVddqMode;
  CompCtl0.Bits.CtlVoltageSelect = CtlVoltSel;
  CompCtl0.Bits.ClkVoltageSelect = CLK_VOLT_SEL;
  CompCtl0.Bits.CaVoltageSelect = CA_VOLT_SEL;
  TempVar1 = (VccIoMv > VccddqMv) ? (VccIoMv - VccddqMv) : 0;
  CCCPredrvUseVcciog = (TempVar1 != 0) ? (TempVar1 > SevenPercentOfVccIoMv) : 0; // Program default if VccIoMv < VccddqMv
  CompCtl0.Bits.CCCPredrvUseVcciog = CCCPredrvUseVcciog;
  CompCtl0.Bits.CCCStaticLegCtl = 1;
  CompCtl0.Bits.DQVssHiBypassVddqMode = VssHiBypassVddqMode;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL0_REG, CompCtl0.Data);
  */
  // Following register is needed for CompCtl4 so program this first
/*
  VssHiPanic.Data = 0;
  if (SAFE) {
    VssHiPanic.Bits.PanicCCCPU = (VsxHiTargetMv > 350) ? 0 : 2;
  } else {
    VssHiPanic.Bits.PanicCCCPU = (VsxHiTargetMv > 350) ? 1 : 3;
  }
  TempVar1 = (191 * (VsxHiTargetMv - PANICV0)) / VccIoMv;
  VssHiPanic.Bits.VsxHiPanicUpVref = MAX (TempVar1, 127);
  if (TempVar1 > 127) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "Clamping VsxHiPanicUpVref: %d\n",TempVar1);
  }
  TempVar2 = 191 * (VsxHiTargetMv + PANICV0);
  TempVar3 = CA_VOLT_SEL ? Vdd2Mv : VccddqMv;
  VssHiPanic.Bits.VsxHiPanicDnVref = DIVIDEROUND (TempVar2, TempVar3);

  TempVar1 = (1 + VssHiPanic.Bits.PanicCCCPU) >> 1;
  TempVar2 = DIVIDEROUND (CaRonDrvDn, TempVar1);
  TempVar3 = VsxHiTargetMv - PANICV0;
  IcompUp = DIVIDEROUND (TempVar3, TempVar2);
  TempVar4 = DIVIDEROUND (Itarget, IcompUp);
  VssHiPanic.Bits.VsxHiPanicCompUpMult = TempVar4;
  TempVar1 = (CA_VOLT_SEL ? Vdd2Mv : VccddqMv) - VsxHiTargetMv - PANICV0;
  IcompDn = DIVIDEROUND (TempVar1, CaRonDrvUp);

  TempVar2 = IcompDn * (1 + VssHiPanic.Bits.PanicPDn2xStep);
  TempVar3 = DIVIDEROUND (Itarget, TempVar2);
  VssHiPanic.Bits.VsxHiPanicCompDnMult = TempVar3;

  // Run initial compensation with 0 (1x).  If HVM Test or (PanicDrvDn/VsxHiPanicCompDnMult) > 40, then switch to 1 (2x)
  // Very similar to the flow for CCCStaticLegCtl.  Start with 0 and run compensation.  If (PanicDrvDn/VsxHiPanicCompDnMult) > 40, then switch to 1 (2x)
  VssHiPanic.Bits.PanicPDn2xStep = 0;
  VssHiPanic.Bits.PanicCCCPUVssHiBypass = SAFE ? 0 : 1; // (DVFS == Enabled)  (Assume this will be off for SAFE config)
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHIPANIC_REG, VssHiPanic.Data);

  CompDvfsRload.Data = 0;
  TempVar2 = RANGE (VsxHiTargetMv, 100, 450);
  TargetV0 = (Vdd2Mv + (4 * TempVar2)) / 5;
  TempVar3 = ((TargetV0 * 382) / Vdd2Mv) - 32;
  CompDvfsRload.Bits.VsxHiLVRTargetNormal = TempVar3;
  TempVar1 = Vdd2Mv - (VccddqMv - (VssHiBypassVddqMode ? 0 : VsxHiTargetMv));
  TempVar2 = RANGE (TempVar1, 100, 450);
  TargetV0 = (Vdd2Mv + (4 * TempVar2)) / 5;
  TempVar3 = ((TargetV0 * 382) / Vdd2Mv) - 32;
  CompDvfsRload.Bits.VsxHiLVRTargetDvfs = TempVar3;
  CompDvfsRload.Bits.VsxHiLVRBias = 22; //0x16
  CompDvfsRload.Bits.DvfsRcompNormalOp = (SAFE || A0) ? 0 : 1;
  CompDvfsRload.Bits.DisableVsxHiLVR = (SAFE || Ddr) ? 1 : 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPDVFSRLOAD_REG, CompDvfsRload.Data);

  CompCtl1.Data = 0;
  CompCtl1.Bits.En200ohmVsshiPncUp = (VssHiPanic.Bits.VsxHiPanicUpVref > 350);
  CompCtl1.Bits.DisableCompRotate = SAFE ? 1 : 0;
  CompCtl1.Bits.SelCompIntVrefSupply = DisVddqPathWithVddq;
  CompCtl1.Bits.DisableQuickComp = SAFE ? 1 : 0;
  CompCtl1.Bits.DisableOdtStatic = DataControl0.Bits.DisableOdtStatic;
  CompCtl1.Bits.CompClkOn = SAFE ? 1 : 0;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL1_REG, CompCtl1.Data);

  CompCtl2.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL2_REG);
  CompCtl2.Bits.Cmd200VrefDn = (191 * 100) / (100 + Inputs->RcompTarget[WrDSCmd]);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL2_REG, CompCtl2.Data);

  VssHiTarget.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG);
  TempVar1 = VsxHiTargetMv * 382;
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv);
  SignedTempVar = TempVar2 - 32;
  VssHiTarget.Bits.CompVTarget = MAX (SignedTempVar, 0); // This is to protect that CompVTarget doesnt get a -ve value
  VssHiTarget.Bits.DataPattern = 1;
  VssHiTarget.Bits.EnPeriodicComp = 1;
  VssHiTarget.Bits.vtnslope = 22;
  TempVar1 = VccIoMv/64;
  SignedTempVar = TempVar1;
  SignedTempVar1 = DIVIDEROUND (-239, SignedTempVar);
  VssHiTarget.Bits.vtnoffset = MAX (SignedTempVar1, -32);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VSSHITARGET_REG, VssHiTarget.Data);

  VccDLLTarget.Data = 0;
  VccDLLTarget.Bits.CompVTarget = VccDllControl.Bits.Target;
  VccDLLTarget.Bits.OffsetBinSearch = SAFE ? 3 : 2;
  TempVar1 = 21 * (Vdd2Mv / 382);
  TempVar2 = VccIoMv / 128;
  TempVar3 = DIVIDEROUND (TempVar1, TempVar2);
  VccDLLTarget.Bits.VtpSlope = TempVar3;
  SignedTempVar = (Vdd2Mv / 96);
  SignedTempVar1 = DIVIDEROUND (-180, SignedTempVar);
  VccDLLTarget.Bits.VtpOffset = RANGE (SignedTempVar1, -32, 31);
  // @todo need more info on VccDLLTarget.Bits.CodeLeakSearch
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLTARGET_REG, VccDLLTarget.Data);

  VccDllReplicaCtrl0.Data = 0;
  VccDllReplicaCtrl0.Bits.TxPBDCode = 63;
  VccDllReplicaCtrl0.Bits.RxPBDCode = 63;
  VccDllReplicaCtrl0.Bits.SlaveRxPiCodeP = 8;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLREPLICACTRL0_REG, VccDllReplicaCtrl0.Data);

  MrcGetSetNoScope (MrcData, CmdDrvVrefUp, ReadCached, &GetSetVal);

  VccDllReplicaCtrl1.Data = 0;
  // VccDllReplicaCtrl1.Bits.CCCPBDCode = 63; //0x3F
  VccDllReplicaCtrl1.Bits.DisableDataPBDDLGating = 1;
  if (QclktoFreq <= 3200) {
    SignedTempVar = 105 -((DdrFrequency * 100) / 7100); // 1.05 scaled to 105
  }
  SignedTempVar1 = (SignedTempVar * 32) - 16;
  SignedTempVar1 /= 100; // Scale it back down
  VccDllReplicaCtrl1.Bits.ScaleIdle2Weak = RANGE (SignedTempVar1, -8, 7);
  VccDllReplicaCtrl1.Bits.MatchedPath = (!UnMatched) ? 1 : 0;
  VccDllReplicaCtrl1.Bits.LocalVsxHiBypass = VssHiBypassVddqMode;
  VccDllReplicaCtrl1.Bits.RloadVref = (UINT32) GetSetVal;
  VccDllReplicaCtrl1.Bits.RloadInstances = DIVIDEROUND (700, RExternal);
  // TempVar1 = BiasPwrMuxSelVdd2 ? VccIoMv : Vdd2Mv; // DataControl5.Bits.BiasPwrMuxSelVdd2 needs training so its 0 in ddrioint
  TempVar2 = 250 / Vdd2Mv;
  TempVar3 = ((TempVar2 * 1000000) - 125000); // scale it to nV
  TempVar4 = DIVIDEROUND (TempVar3, 31250);
  VccDllReplicaCtrl1.Bits.RxBiasVref = RANGE (TempVar4, 0, 7);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLREPLICACTRL1_REG, VccDllReplicaCtrl1.Data);

  VccDllDqsDelay.Data = 0;
  VccDllDqsDelay.Bits.RxClkDisRep = 2;
  VccDllDqsDelay.Bits.RxDqsSlope = 3;
  VccDllDqsDelay.Bits.RxDqsVref = 7;
  VccDllDqsDelay.Bits.DriftLimitLocal = (36 * 9) / ( (QclkPs >> Gear2) * 10);

#ifdef CTE_FLAG
  VccDllDqsDelay.Bits.TrainTargetOffsetUI = 0;
#else
  VccDllDqsDelay.Bits.TrainTargetOffsetUI = 1;
#endif
  VccDllDqsDelay.Bits.RxPBDCode = DIVIDEROUND (50, PBDStepSize);
  VccDllDqsDelay.Bits.DriftLimitGlobal = 15 / (QclkPs >> Gear2);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLDQSDELAY_REG, VccDllDqsDelay.Data);

  CCCVoltageUsed.Data = 0;
  CCCVoltageUsed.Bits.PredrvUseVcciog = CCCPredrvUseVcciog;
  CCCVoltageUsed.Bits.VssHiBypassVddqMode = VssHiBypassVddqMode;
  TempVar1 = (!CCCVoltageUsed.Bits.PredrvUseVcciog) & VssHiBypassVddqMode;
  CCCVoltageUsed.Bits.CaPDPreDrvVccddq  = TempVar1 & !CA_VOLT_SEL;
  CCCVoltageUsed.Bits.CtlPDPreDrvVccddq = TempVar1 & !CtlVoltSel;
  CCCVoltageUsed.Bits.ClkPDPreDrvVccddq = TempVar1 & !CLK_VOLT_SEL;
  CCCVoltageUsed.Bits.CaVoltageSelect = CA_VOLT_SEL;
  CCCVoltageUsed.Bits.CtlVoltageSelect = CtlVoltSel;
  CCCVoltageUsed.Bits.ClkVoltageSelect = CLK_VOLT_SEL;
  CCCVoltageUsed.Bits.VssHiBypassVdd2Mode = VssHiBypassVdd2Mode;
  CCCVoltageUsed.Bits.CkUseCtlComp = Lpddr5 ? 1 : 0;
  CCCVoltageUsed.Bits.CsUseCaComp = Lpddr4 ? 1 : 0;
  CCCVoltageUsed.Bits.CccStaticLegCtrl = 1;
  CCCVoltageUsed.Bits.DisVddqPathWithVddq = DisVddqPathWithVddq;
  //CCCVoltageUsed.Bits.XoverAdjustmentRsvd = 0;
  CCCVoltageUsed.Bits.XoverMinDelayCtl = 1;
  if (!A0) {
    SignedTempVar = 103 - DIVIDEROUND (VsxHiTargetMv, 5);
    CCCVoltageUsed.Bits.TxVsxHiLeakerComp = RANGE (SignedTempVar, 0, 63);
  }
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CCCVoltageUsed.Data); // Multicast used
  */
//---------------------------------------Vtt CRs----------------------------------------------------------------------------
/*
  VttGenControl.Data = 0;
  TempVar1 = (382 * VttTargetV);
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv) - 32;
  // Assumes VttTargetV in LPDDR is 150mV based on the system studies.
  VttGenControl.Bits.Target = TempVar2;
  TempVar1 = (382 * PANICV0);
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv);
  VttGenControl.Bits.Panic0 = TempVar2;
  TempVar1 = (382 * PANICV1);
  TempVar2 = DIVIDEROUND (TempVar1, Vdd2Mv);
  VttGenControl.Bits.Panic1 = TempVar2;
  VttGenControl.Bits.Spare1 = SAFE ? 1 : 0;
  // VttGenControl.Bits.DisSensorPwrDn = SAFE ? 1 : 0; // Field no longer exists
  VttGenControl.Bits.EnDacPM = (A0 || SAFE) ? 0 : 2;
// @todo_adl  VttGenControl.Bits.DdrCRForceODTOn = (DataControl0.Bits.OdtForceQDrvEn || DataControl0.Bits.DdrCRForceODTOn) && VttGenControl.Bits.EnVttOdt;
  //This field is programmed in SetVtt function
 // VttGenControl.Bits.EnVttOdt = (DataControl0.OdtMode == 2 * Vtt);   // (DDRControl5.OdtMode == 2 (VTT))
  TempVar1 = (Ddr4 || Ddr5) ? 10 : 16; // MaxMin( RndUp( (DDR4or5?10nS:16nS)/tQCLK/8 )-1, 0, 7)
  TempVar2 = DIVIDEROUND (TempVar1, QclkPs);
  TempVar3 = DIVIDEROUND (TempVar2, 8) - 1;
  VttGenControl.Bits.WakeUpDelay = TempVar3;
  TempVar1 = (60 / RefClkPs) - 1;
  TempVar2 = MrcLog2(TempVar1);
  VttGenControl.Bits.LockTimer = TempVar2;
  VttGenControl.Bits.AckOffset = 1;
  VttGenControl.Bits.EnVttOdt = 1;
  MrcWriteCrMulticast (MrcData, DDRVTT_CR_DDRCRVTTGENCONTROL_REG, VttGenControl.Data);

  VttGenControl.Bits.EnDacPM = 0;
  MrcWriteCR (MrcData, DDRVTT2_CR_DDRCRVTTGENCONTROL_REG, VttGenControl.Data); // 0x2A00 is VTTComb2 in TGL UY

  VttCompOffset.Data = MrcReadCR (MrcData, DDRVTT0_CR_DDRCRVTTCOMPOFFSET_REG);
  VttCompOffset2.Data = 0;
  VttCompOffset2.Bits.PanicLo0UsePmos = (VsxHiTargetMv + (-PANICV0) + VttCompOffset.Bits.PanicLo0CompOfst) < 375;
  VttCompOffset2.Bits.PanicLo1UsePmos = (VsxHiTargetMv + (-PANICV1) + VttCompOffset.Bits.PanicLo1CompOfst) < 375;
  VttCompOffset2.Bits.PanicHi0UsePmos = (VsxHiTargetMv + (PANICV0) + VttCompOffset.Bits.PanicHi0CompOfst)  < 375;
  VttCompOffset2.Bits.PanicHi1UsePmos = (VsxHiTargetMv + (PANICV1) + VttCompOffset.Bits.PanicHi1CompOfst)  < 375;
  MrcWriteCrMulticast (MrcData, DDRVTT_CR_DDRCRVTTCOMPOFFSET2_REG, VttCompOffset2.Data);
  */
//---------------------End of Early Init---------------------------------------------------
//---------------------Start of Late Init--------------------------------------------------
  /* @todo_adl
  DataControl6.Data = 0;
  // Depends on Read Postamble number of toggles.  Since unmatched is only LPDDR5, we have 4 toggles.
  // RcvEnPre is allowed at assert when (0: RxDqs is low, 1: RxDqs falling edge).  PassRcvenOnDqsFall = Unmatched Path
  if (UnMatched) {
    PassRcvEn = 1;
    if (Lpddr4) {
      RcvEnWaveShape = 1;
      NumToggles = 0;
      ResetNumPre = 0;
    } else {
      // LPDDR5
      NumToggles = 1;
      ResetNumPre = 4;
      RcvEnWaveShape = (Gear2) ? 2 : 3;
    }
  } else {
    // Matched path
    RcvEnWaveShape = 0;
    NumToggles = 0;
    PassRcvEn = 0;
    ResetNumPre = 0;
  }

  if (QclktoFreq >= 3200) {
    RightShift = 28;
  } else if (QclktoFreq >= 2667) {
    RightShift = 24;
  } else if (QclktoFreq >= 2400) {
    RightShift = 20;
  } else if (QclktoFreq >= 2133) {
    RightShift = 16;
  } else if (QclktoFreq >= 1867) {
    RightShift = 12;
  } else if (QclktoFreq >= 1600) {
    RightShift = 8;
  } else if (QclktoFreq >= 1333) {
    RightShift = 4;
  } else {
    RightShift = 0;
  }
  TempVar1 = CBTune0 >> RightShift;
  TempVar1 &= 0xF;
  DataControl6.Bits.sdllbwctrl = TempVar1;
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, FirstByte, RxCben, ReadCached | PrintValue, &GetSetVal);
  DataControl6.Bits.sdll_picb = (UINT32) GetSetVal;
  DataControl6.Bits.rxd0picb = (UINT32) GetSetVal;
  DataControl6.Bits.rxd0bwctrl = TempVar1;
  DataControl6.Bits.UseDefaultRdPtrCalc = 1;
  DataControl6.Bits.RstNumPre = ResetNumPre;
  DataControl6.Bits.PassRcvenOnDqsFall = PassRcvEn;
  DataControl6.Bits.RXTogglePreAmble = NumToggles; //in tCK
  DataControl6.Bits.RcvEnWaveShape = RcvEnWaveShape;
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATACONTROL6_REG, DataControl6.Data);

  VccDllReplicaCtrl2.Data = 0;
  VccDllReplicaCtrl2.Bits.DisableTxDqs = Lpddr5;
  SignedTempVar = ((-333 * Vdd2Mv) + 42) / 10000; // PBiasCalibration is 3 bits field therefore Vdd2 must be considered in V. Multiply and divide by 10 to convert 33.3
  VccDllReplicaCtrl2.Bits.PbiasCalibration = (UINT32) SignedTempVar;
  VccDllReplicaCtrl2.Bits.SdllPiCB = DataControl6.Bits.sdll_picb;
  VccDllReplicaCtrl2.Bits.LeakScale = VccDllReplicaCtrl2.Bits.SdllPiCB; //Fixed Value (TBD based on tester silicon measurements.PreSilicon sims indicate 0x7 but believe this is too high)
  VccDllReplicaCtrl2.Bits.SdllBwCtrl = DataControl6.Bits.sdllbwctrl;
  VccDllReplicaCtrl2.Bits.RxD0BWCtrl = DataControl6.Bits.rxd0bwctrl;
  VccDllReplicaCtrl2.Bits.RxD0BWCtrl = DataControl6.Bits.rxd0picb;
  //VccDllReplicaCtrl2.Bits.RxPwrMuxSelVdd2 = DataControl3.Bits.RxPwrMuxSelVdd2; //DataControl3.Bits.RxPwrMuxSelVdd2 needs training
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_VCCDLLREPLICACTRL2_REG, VccDllReplicaCtrl2.Data);

#ifdef CTE_FLAG
  DataOffsetTrain.Data = 0;
  DataOffsetTrain.Bits.VrefOffset = 12;
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATAOFFSETTRAIN_REG, DataOffsetTrain.Data);
#endif

  // Using B0 structure as the base and skip programming the fields that are spares in A0.
  DataControl3.Bits.VrefBypassVssHi = ((VccddqMv - VsxHiTargetMv) < VDD_0_75) ? 1 : 0;
  DataControl3.Bits.RxPwrMuxSelVdd2 = DataControl5.Bits.BiasPwrMuxSelVdd2;
  DataControl3.Bits.VrefPwrMuxSelVdd2 = (VsxHiTargetMv >= (INT32) ((7 * Vdd2Mv) / 100));
  DataControl3.Bits.RxSALTailCtrl = 2; //   Adjust based on Vtp or (VccIoMv-Vtp)?
  DataControl3.Bits.wrpreamble = (MrcGetWpre (MrcData) - 1);
  DataControl3.Bits.QualifySdlWithRcvEn = UnMatched ? 0 : 1;
  DataControl3.Bits.rxfoldcs = NPath ? 3 : 0;
  DataControl3.Bits.dqspadmuxselvdd2 = DataControl5.Bits.BiasPwrMuxSelVdd2;
  DataControl3.Bits.DisableTxDqs = Lpddr5;
  DataControl3.Bits.RxPBDCalibration = VccDllReplicaCtrl2.Bits.RxDeskewCal; // RxDeskewCal needs CAScomp training
  if (Inputs->NewFeatureEnable2) {
    DataControl3.Bits.TxEq_RankMuxDelay_offset = 3;
    DataControl3.Bits.TxPerbit_RankMuxDelay_offset = 1;
  }
  if (!A0) {
    DataControl3.Bits.BlockSdlWithRcvEn = BlockSdlWithRcvEn;
    DataControl3.Bits.RcvEnExtFF = 0;
    DataControl3.Bits.RxRankMuxDelay_2ndStg = 3;
  }
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATACONTROL3_REG, DataControl3.Data);
  */
/*@todo_adl
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    ControllerOut = &Outputs->Controller[Controller];
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        ChannelOut = &ControllerOut->Channel[Channel];
        Is2DPC = (ChannelOut->DimmCount == 2);
        if (Lpddr) {
          ByteStart = (MAX_BYTE_IN_LP_CHANNEL * Channel);
          ByteEnd = ByteStart + MAX_BYTE_IN_LP_CHANNEL;
        }
        for (Byte = ByteStart; Byte < ByteEnd; Byte++) {
          DataControl2.Bits.DqSlewDlyByPass = !(Ddr && Is2DPC);
          //DataControl2.Bits.RxVrefVddqProgMFC = 0;//  @todo : field requires training
          //DataControl2.Bits.RxVrefVttProgMFC = 0;//  @todo : field requires training
          // @todo : CRC isnt currently supported
          DataControl2.Bits.EnDqsNRcvEn = EnDqsNRcvEnSet;
          //DataControl2.Bits.xover_mindelay_ctl = 0;
          //DataControl2.Bits.xover_adjustment_rsvd = 0;
          DataControl2.Bits.DisableDqsOdtStatic = (OdtTarget > 120);
          DataControl2.Bits.TxPBDCalibration = VccDllReplicaCtrl2.Bits.TxDeskewCal; //@ Program after CA Scomp training
          DataControl2.Bits.EnConstZEqTx = 1;
          //DataControl2.Bits.TxEqDis = 0;
          Offset = DataControl2Offset (Controller, Byte);
          MrcWriteCR (MrcData, Offset, DataControl2.Data);
        }
      } // ChannelExist
    }
  } // Controller
  */

/*@todo_adl
  TargetNUI = (Gear2) ? 1 : 0;
  RcompData0.Data = 0;
  RcompData0.Bits.DqsNTargetNUI = TargetNUI;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP0_REG, RcompData0.Data);
  RcompData1.Data = 0;
  RcompData1.Bits.DqsPTargetNUI = TargetNUI;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMP1_REG, RcompData1.Data);

  MarginModeControl.Data = 0;
  MarginModeControl.Bits.MinWidth = 0x4;
  MarginModeControl.Bits.CalcCenter = 1;
  MarginModeControl.Bits.IOLBCycles = 0xA;
  // MarginModeControl.Bits.IOLBCpgcMode = 1;
  // MarginModeControl.Bits.MisrRunOvrd = 1;
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRMARGINMODECONTROL0_REG, MarginModeControl.Data);

  DataDqsRankX.Data = 0;
  DataDqsRankX.Bits.DqsPfNrTcoDelay = 16;
  DataDqsRankX.Bits.DqsPrNfTcoDelay = 16;
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRDATADQSRANK0_REG,DataDqsRankX.Data);
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRDATADQSRANK1_REG, DataDqsRankX.Data);
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRDATADQSRANK2_REG, DataDqsRankX.Data);
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRDATADQSRANK3_REG, DataDqsRankX.Data);
  */
//-----------------------------------CCC Registers----------------------------------------------------------------------------------
  CCCClkControls.Data = 0;
  if (SAFE) {
    CCCClkControls.Bits.ClkGateDisable = 1;
    CCCClkControls.Bits.IntCkOn = 1;
    CCCClkControls.Bits.CkeIdlePiGateDisable = 1;
    CCCClkControls.Bits.CaValidPiGateDisable = 1;
    CCCClkControls.Bits.DefDrvEnLow = Lpddr5 ? 2 : 0;
  } else {
    CCCClkControls.Bits.DefDrvEnLow = Ddr4 ? 3 : 2; // Initial value (Ddr4 ? 0x3 : 0x2) but power training can try 0x1 for Ddr4 or weak Rodt.
  }
  CCCClkControls.Bits.RTO = Ddr5 ? 1 : 0;
  CCCClkControls.Bits.BlockTrainRst = 1;
  CCCClkControls.Bits.CtlSRDrv = (SAFE || (!Ddr4)) ? 0 : 2;
  CCCClkControls.Bits.c3segsel_b_for_cke = (SAFE || Ddr5) ? 1 : 0;
  if (Ddr4) {
    CCCClkControls.Bits.CaTxEq  = TXEQ_NODEEMPHASIS;
    CCCClkControls.Bits.CtlTxEq = TXEQ_NODEEMPHASIS;
    CCCClkControls.Bits.ClkTxEq = TXEQ_NODEEMPHASIS;
  } else {
    CCCClkControls.Bits.CaTxEq  = (TXEQ_NODEEMPHASIS | TXEQ_CONSTATNTZ);
    CCCClkControls.Bits.CtlTxEq = (TXEQ_NODEEMPHASIS | TXEQ_CONSTATNTZ);
    if (Lpddr && !A0) {
      CCCClkControls.Bits.ClkTxEq = 0x6;
    } else {
      CCCClkControls.Bits.ClkTxEq = (TXEQ_NODEEMPHASIS | TXEQ_CONSTATNTZ);
    }
  }


  //Only important for Ddr4.  All others can use `0 since termination is off or to Vss.
  CCCClkControls.Bits.CtlSRDrv = Ddr4 ? 2 : 0;
 // Only important for Ddr4.  All others can use `0 since termination is off or to Vss.
  if (Ddr) {
    //CCCClkControls.Bits.Spare             = 125 > ((Vdd2Mv * Rodt) / (2 * (3 * (RonCtlTarget + Rodt)))) ? 0 : 1;
    CCCClkControls.Bits.RxVref            = 150 > ((Vdd2Mv * Rodt) / (2 * (3 * (RonCtlTarget + Rodt)))) ? 1 : 2;
  }
  CCCClkControls.Bits.c3segsel_b_for_cke = ~(Ddr4);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCCLKCONTROLS_REG, CCCClkControls.Data);
  /* @todo_adl
 // @todo : Program after SAComp
  CCCPerBitDeskew0.Data = 0;
  CCCPerBitDeskew0.Bits.deskewcal = VccDllReplicaCtrl2.Bits.CCCDeskewCal;
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCPERBITDESKEW0_REG, CCCPerBitDeskew0.Data);

 // @todo : Uncomment when the value of TurnOnDelay is provided
 // TempVar1 = 5 * QclkPs;
 // TempVar2 = 10 * TurnOnDelay;
 // TempVar3 = DIVIDEROUND (TempVar2, TempVar1);
 //CCCPerBitDeskew1.Bits.FFPBDDelay = RANGE(TempVar3, 0, 2);
  CCCPerBitDeskew1.Data = 0;
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCPERBITDESKEW1_REG, CCCPerBitDeskew1.Data);

  CCCPerBitDeskew2.Bits.txdeskewpoweren = 1;
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCPERBITDESKEW2_REG, CCCPerBitDeskew2.Data);

  for (Channel = 0; Channel < MRC_NUM_CCC_INSTANCES; Channel++) {
    CCCMisr.Bits.EnLFSR = 1;
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRMISR_REG, CH1CCC_CR_DDRCRMISR_REG, Channel);
    MrcWriteCR (MrcData, Offset, CCCMisr.Data);
  }

    // Use SAFE for DLLPITESTANDADC settings
  DataDllPiTestAndAdc.Data = 0;
  DataDllPiTestAndAdc.Bits.PhaseDrvPwrSavOn = 1;
  // Multicast all DLLPITESTANDADC.  Each Fub has same CR layout.
  MrcWriteCrMulticast (MrcData, DLLDDR_CR_DLLPITESTANDADC_REG, DataDllPiTestAndAdc.Data);

  VccDllFFControl.Data = 0;
  VccDllFFControl.Bits.Bypass = SAFE ? 1 : 0;
  VccDllFFControl.Bits.LocalVsxHiBypass = VssHiBypassVddqMode;
  VccDllFFControl.Bits.VccdllPwrMuxSel = ((VccIoMv - MRC_VCCDLL_TARGET) < SevenPercentOfVccIoMv);
  DataDllPiTestAndAdc.Data = MrcReadCR(MrcData, DLLDDRDATA0_CR_DLLPITESTANDADC_REG);
  VccDllFFControl.Bits.VccdllMdllPwrSave   = (DataDllPiTestAndAdc.Bits.PhaseDrvPwrSavOn == 1);
  VccDllFFControl.Bits.VccdllPiPwrSave     = (DataDllPiTestAndAdc.Bits.PhaseDrvPwrSavOn == 1);
  MrcWriteCrMulticast (MrcData, DLLDDR_CR_DDRCRVCCDLLFFCONTROL_REG, VccDllFFControl.Data);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "SdramCount: 0x%X\n",Outputs->SdramCount );
  */
//--------------------------------------Comp CRs----------------------------------------------------------------------------
  /* @todo_adl
  CompAlert.Data = 0;
  CompAlert.Bits.AlertVref = (192 / 2) * ((34 + Rup + 34) / (34 + Rup));
  CompAlert.Bits.AlertEn = (Ddr4 || Ddr5);
  TempVar1 = (((CompAlert.Bits.AlertVref / 191) * VccddqMv) - 600);
  TempVar2 = DIVIDEROUND (TempVar1, 50);
  CompAlert.Bits.AlertCMCap = RANGE (TempVar2, 0, 7);

  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRALERT_REG, CompAlert.Data);
  */
// --------------------- CLOCK and MCMISCS CRs - Added in MAS_80----------------------------------------------------------------------
  McMiscSpineGating.Data = 0;
  TempVar1 = DIVIDEROUND (160000, QclkPs);
  if ((TempVar1 > 128) && (TempVar1 <= 192)) {
    TempVar2 = 1;
  } else if ((TempVar1 > 192) && (TempVar1 <= 256)) {
    TempVar2 = 2;
  } else if ((TempVar1 > 256) && (TempVar1 <= 384)) {
    TempVar2 = 3;
  } else if ((TempVar1 > 384) && (TempVar1 <= 512)) {
    TempVar2 = 4;
  } else if ((TempVar1 > 512) && (TempVar1 <= 768)) {
    TempVar2 = 5;
  } else if ((TempVar1 > 768) && (TempVar1 <= 1024)) {
    TempVar2 = 6;
  } else if ((TempVar1 > 1024) && (TempVar1 <= 1536)) {
    TempVar2 = 7;
  } else {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "Condition didnt meet any of the allowed Sleepcycles values, default SleepCycles = %d\n", TempVar2);
    TempVar2 = 0;
  }

  McMiscSpineGating.Bits.sleepcycles = TempVar2;
  McMiscSpineGating.Bits.awakecycles = 2;
  McMiscSpineGating.Bits.enhiphase = 2;
  McMiscSpineGating.Bits.gracelimitentry = SAFE ? 7 : 3;
  // @todo_adl This causes 'x' due to MCP, use zero for now. This feature is not POR on ADL-S
  McMiscSpineGating.Data = 0;
  MrcWriteCR (MrcData, MCMISCS_SPINEGATING_REG, McMiscSpineGating.Data);

  DdrScramCh0.Data = MrcReadCR (MrcData, DDRSCRAM_CR_DDRSCRAMBLECH0_REG);
  DdrScramCh0.Bits.dis_cmdanalogen = SAFE ? 0 : 1;
  MrcWriteCR (MrcData, DDRSCRAM_CR_DDRSCRAMBLECH0_REG, DdrScramCh0.Data);

  DdrScramCh1.Data = MrcReadCR (MrcData, DDRSCRAM_CR_DDRSCRAMBLECH1_REG);
  DdrScramCh1.Bits.spare = SAFE ? 0 : 1;
  MrcWriteCR (MrcData, DDRSCRAM_CR_DDRSCRAMBLECH1_REG, DdrScramCh1.Data);

  DdrScramCh2.Data = MrcReadCR (MrcData, DDRSCRAM_CR_DDRSCRAMBLECH2_REG);
  DdrScramCh2.Bits.spare = SAFE ? 0 : 1;
  MrcWriteCR (MrcData, DDRSCRAM_CR_DDRSCRAMBLECH2_REG, DdrScramCh2.Data);

  MiscControl1.Data = MrcReadCR (MrcData, DDRSCRAM_CR_DDRMISCCONTROL1_REG);
  MiscControl1.Bits.companddeltadqsupdateclkgatedisable = SAFE ? 1 : 0;
// @todo_adl  MiscControl1.Bits.BGBiasTrim = 4;
// @todo_adl  MiscControl1.Bits.BGAgshTrim = ((VCCANA_EH > 1800) ? 3 : ((VCCANA_EH > 1550) ? 0 : ((VCCANA_EH > 1300) ? 1 : 2)));
  MrcWriteCR (MrcData, DDRSCRAM_CR_DDRMISCCONTROL1_REG, MiscControl1.Data);

  MiscControl2.Data = MrcReadCR (MrcData, DDRSCRAM_CR_DDRMISCCONTROL2_REG);
  MiscControl2.Bits.rx_analogen_grace_cnt = DDRSCRAM_CR_DDRMISCCONTROL2_rx_analogen_grace_cnt_MAX;
  MrcWriteCR (MrcData, DDRSCRAM_CR_DDRMISCCONTROL2_REG, MiscControl2.Data);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "DDRMISCCONTROL2 = 0x%X\n", MiscControl2.Data);

  MiscControl7.Data = 0;
  // MRC tracks BurstLength in tCK, which is DCLK for LP4/DDR4.
  //LP5 is not Dclk due to 4:1 scaling.  WCK is the equivalent to DCLK.  Thus we multiply by 4.
  Data32 = Outputs->BurstLength;
  if (Lpddr5) {
    Data32 *= 4;
  }
  MiscControl7.Bits.rxburstlen = Data32;
  MiscControl7.Bits.txburstlen = Data32;
// @todo_adl  MiscControl7.Bits.cas2fswck = (MrcGetWckEnlFs (MrcData) * 4) + ((Gear2) ? -1 : 1);
  MiscControl7.Bits.cmdduration = Ddr4 ? 0 : (Lpddr5 ? 7 : 3);
  if (Ddr5) {
    MiscControl7.Bits.cmdduration = (NMode == 1) ? 1 : 2;  // 1 for 1N and 2 for 2N
  }
  MiscControl7.Bits.run1stcomp = (Inputs->TrainingEnables2.DCC) ? 1 : 0;
  MiscControl7.Bits.datainvertnibble = DataInvertNibble;
// @todo_adl  MiscControl7.Bits.trainwcksyncratio = 2;
  MrcWriteCR (MrcData, DDRSCRAM_CR_DDRMISCCONTROL7_REG, MiscControl7.Data);
  // Select the DDRIO ODT Mode. This also programs default RxVref.
// @todo_adl --------------------------------->  MrcSetIoOdtMode (MrcData, Outputs->OdtMode);

  /* @todo_adl
  CompCtl4.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL4_REG);
  TempVar2 = DIVIDEROUND (22000, QclkPs); //Scaled 22ns to 22000ps
  TempVar3 = DIVIDEROUND (TempVar2, 8);
  CompCtl4.Bits.CompInitDelay = SAFE ? 7 : (TempVar3 - 2);
  CompCtl4.Bits.CompCodeSwitch = CompCtl4.Bits.CompInitDelay;
  CompCtl4.Bits.SWCapUseVdd2 = SAFE ? 1 : (VccIoMv < 950);
  TempVar1 = (!CCCPredrvUseVcciog) && VssHiBypassVddqMode;
  CompCtl4.Bits.CaPDPreDrvCVccddq = CCCVoltageUsed.Bits.CaPDPreDrvVccddq;
  CompCtl4.Bits.CtlPDPreDrvCVccddq = CCCVoltageUsed.Bits.CtlPDPreDrvVccddq;
  CompCtl4.Bits.ClkPDPreDrvCVccddq = CCCVoltageUsed.Bits.ClkPDPreDrvVccddq;
  CompCtl4.Bits.DqPDPreDrvCVccddq = (TxDqPUpPreDrvVccddq && VssHiBypassVddqMode);
  if ((VsxHiTargetMv > 450) || (VsxHiTargetMv < 100)) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "VsxHiTargetMv is out of range: %d, it should be within 100mV to 450mV\n",VsxHiTargetMv);
  }
  TempVar1 = RANGE (VsxHiTargetMv, 100, 450);
  SignedTempVar  = VccddqMv - Vdd2Mv;
  SignedTempVar1 = VccddqMv - (Vdd2Mv - TempVar1);
  ObeyDDQBypassB0 = (ABS (SignedTempVar) < ABS (SignedTempVar1));
  ObeyDDQBypassA0 = (ObeyDDQBypassB0 && VssHiBypassVddqMode && VssHiBypassVddqMode);
  CompCtl4.Bits.ObeyDDQBypass = A0 ? ObeyDDQBypassA0 : ObeyDDQBypassB0;
  TempVar1 = (CA_VOLT_SEL) ? 80000 : 22000;
  TempVar2 = DIVIDEROUND (TempVar1, QclkPs);
  CompCtl4.Bits.CompStage2Switch = TempVar2;
  TempVar1 = CLK_VOLT_SEL ? 80000 : 22000;
  TempVar2 = DIVIDEROUND (TempVar1, QclkPs);
  CompCtl4.Bits.CompStage3Switch = TempVar2;
  TempVar1 = ((CLK_VOLT_SEL != CtlVoltSel) | (CA_VOLT_SEL != CLK_VOLT_SEL)) ? 80000 : 22000;
  TempVar2 = DIVIDEROUND (TempVar1, QclkPs);
  CompCtl4.Bits.CompStage4Switch = TempVar2;
  TempVar1 = ((CompCtl0.Bits.EnVttOdt | CompCtl0.Bits.EnVddqOdt) & (CtlVoltSel)) ? 80000 : 22000;
  TempVar2 = DIVIDEROUND (TempVar1, QclkPs);
  CompCtl4.Bits.CompStage5Switch = TempVar2;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL4_REG, CompCtl4.Data);
  */
  // Configure Strobe
  MrcSetWritePreamble (MrcData);

  //---------------------------------Initialize FLL CRs------------------------------------------------------------------------
  FllCmdCfg.Data = MrcReadCR (MrcData, FLL_CMD_CFG_REG_REG);
  TempVar1 = (RefClkMhz == MRC_REF_CLOCK_133) ? 66 : 100;
  FllCmdCfg.Bits.FLL_RATIO = 3133 / TempVar1;

  MrcWriteCR (MrcData, FLL_CMD_CFG_REG_REG, FllCmdCfg.Data);

  FllStaticCfg0.Data = MrcReadCR (MrcData, FLL_STATIC_CFG_0_REG_REG);
  FllStaticCfg0.Bits.LDO_VREFSEL = ((Vdd2Mv >= 1075) && (Vdd2Mv <= 1150)) ? 3 : 4;
  FllStaticCfg0.Bits.FAST_CAL_WINDOW_VAL = 0x0;
  FllStaticCfg0.Bits.SLOW_CAL_WINDOW_VAL = 0x2;
  MrcWriteCR (MrcData, FLL_STATIC_CFG_0_REG_REG,  FllStaticCfg0.Data);

  FllStaticCfg1.Data = MrcReadCR (MrcData, FLL_STATIC_CFG_1_REG_REG);
  FllStaticCfg1.Bits.REFCLK_DIVIDE_RATIO_SEL = (RefClkMhz == MRC_REF_CLOCK_133) ? 2 : 1;
  FllStaticCfg1.Bits.COARSECAL_CNTR_EN = 0xF;
  FllStaticCfg1.Bits.FINECAL_CNTR_EN = 0x2;
  FllStaticCfg1.Bits.VSUPPLY_CFG = 1;
  MrcWriteCR (MrcData, FLL_STATIC_CFG_1_REG_REG,  FllStaticCfg1.Data);

  MrcWriteCR (MrcData, FLL_STATIC_CFG_2_REG_REG, MrcReadCR (MrcData, FLL_STATIC_CFG_2_REG_REG));
  /* @todo_adl
  FllWired.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRFLLWIRED_REG);
  FllWired.Bits.FllWireRatio = (FllStaticCfg1.Bits.REFCLK_DIVIDE_RATIO_SEL == 2) ? 47 : 31;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRFLLWIRED_REG, FllWired.Data);
  */
  //-----------------------------------------------------------------------------------------------------------------------------

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Init Data CRs\n");
  // Initialize the Receive Enable Delay Fields, RX FIFO Delays, and Roundtrip Latency.
  // Even though Roundtrip Latency is in the MC domain, it depends on RX IO delays, so we will init it here.
  // Cached fields sits in cache until later flush call.
  // DdrIoInit above was done with CR multicast.  Cache may have some other values in it causing register corruption.
  // Invalidate cache here to cause GetSet to RMW.
  InitializeRegisterCache (MrcData);

  if (Lpddr4) {
    TdqsckMin = tDQSCK_MIN_LP4;
  } else {
    TdqsckMin = 0;
  }
  TdqsckMin = DIVIDECEIL (TdqsckMin, QclkPs);
//  GetSetVal2 = 0; // RxFlybyDelay(Dclk)
#ifndef AV_PHY_INIT
  RxDataValidDclk = Gear2 ? 14 : 8;  // RTL constant
  RxDataValidQclk = 0;  // RTL constant
#endif
  // initial CLK/CTL PI
  ClkDelay = CtlPiDelay / 64; // Convert PI to QCLKs

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      TimingProfile = &Outputs->Controller[Controller].Channel[Channel].Timing[Profile];
      tCL = (UINT8) TimingProfile->tCL;
      if (Lpddr5) {
        // Scale to WCK which matches QCLK in the IO.
        tCL *= 4;
      }
      RcvEnPi[Controller][Channel] = (Lpddr5 ? 0 : 128) + (Gear2 ? 0 : CtlPiDelay) + (Ddr4 ? 32 : 0);  // Start from CTL PI value
      Data32 = (tCL * (2 - Gear2)) + TdqsckMin;
      if (Gear2) {
        Data32 -= 10;
      } else {
        Data32 -= ((UlxUlt) ? 15 : 14);
      }
      //Data32 = Data32 - ((Gear2) ? 10 : 15);
      Data32 -= (Controller ? 4 : 0); // Adjust delay for MC1
      if (Lpddr4 && UnMatched) {
        Data32 -= 2;
      }
      if (!Gear2) {
        RcvEnPi[Controller][Channel] += ((Data32 % 2) * 64);   // If odd number of QCLKs, add one 1 QCLK to RcvEnPi
      } else if (Lpddr4) {
        Data32 -= 1;
      } else if (Lpddr5) {
        Data32 -= 8;
      }
      tCL4RcvEn = Data32 / (2 - Gear2); // RxIoTclDelay (tCK)

      tCL4RxDqFifoRdEn = tCL4RcvEn;
      tCL4RxDqFifoRdEn += RxFifoChDelay[DdrType][Gear2][(Controller * MaxChannels) + Channel];
      RxDqFifoRdEnRankDel = DIVIDECEIL ((UINT32) RcvEnPi[Controller][Channel], PiToQclk);
      RxDqFifoRdEnRankDel += ClkDelay;
#ifdef AV_PHY_INIT
      RxDqFifoRdEnRankDel = 0;
#endif
      Data32 = (UINT32) ((((UINT32) tCL4RxDqFifoRdEn) * (2 - Gear2)) + RxDqFifoRdEnRankDel);
      Data32 += 10 + 2;
      if (Gear2) {
        // DDR4 or LP4 or LP5
        Data32 += (Ddr4) ? 12 : (Lpddr4) ? 11 : 16;
      } else {
        // DDR4 or LP4/5
        Data32 += (Ddr4) ? 17 : 18;  // For DDR4 this assumes 3N mode
      }
      Roundtrip = Data32;
#ifndef AV_PHY_INIT
      MrcGetSetMcCh (MrcData, Controller, Channel, RxIoTclDelay,           WriteToCache | PrintValue, &tCL4RcvEn);
      MrcGetSetMcCh (MrcData, Controller, Channel, RxFifoRdEnTclDelay,     WriteToCache | PrintValue, &tCL4RxDqFifoRdEn);
      MrcGetSetMcCh (MrcData, Controller, Channel, RxDqDataValidDclkDelay, WriteToCache | PrintValue, &RxDataValidDclk);
      MrcGetSetMcCh (MrcData, Controller, Channel, RxDqDataValidQclkDelay, WriteToCache | PrintValue, &RxDataValidQclk);
      MrcGetSetMcChRnk (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, RxFlybyDelay,         WriteToCache | PrintValue, &GetSetVal2);
      MrcGetSetMcChRnk (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, RxFifoRdEnFlybyDelay, WriteToCache | PrintValue, &RxDqFifoRdEnRankDel);
#endif
      MrcGetSetMcChRnk (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, RoundTripDelay,       WriteCached | PrintValue, &Roundtrip);
    } // Channel
  } // Controller

  // Initialize TX FIFO
  // Cached fields sits in cache until later flush call.
//  GetSetVal3  = 0; // TxDqFifoRdEnFlybyDelay(Dclk), RptChRepClkOn, TxDqFifoRdEnPerRankDelDis
//  GetSetVal4  = 7; // CmdAnlgEnGraceCnt
//  GetSetVal5  = MCMISCS_WRITECFGCH0_txanlgengracecnt_MAX; // TxAnlgEnGraceCnt
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      IpChannel = LP_IP_CH (Lpddr, Channel);
      TimingProfile = &Outputs->Controller[Controller].Channel[Channel].Timing[Profile];
      tCWL = TimingProfile->tCWL;
      if (Lpddr5) {
        // Scale to WCK which aligns to QCLK in the IO.
        tCWL *= 4;
      }

      ScWrDelay.Data = 0; // @todo: GetSet
      AddTcwl = 0;
      DecTcwl = 2;
      DecTcwl = MIN (DecTcwl, (UINT8) tCWL + AddTcwl - 2);
      DecTcwl += (Controller ? 2 : 0); // Adjust delay for MC1
      ScWrDelay.Bits.Add_tCWL = AddTcwl;
      ScWrDelay.Bits.Dec_tCWL = DecTcwl;
#ifdef AV_PHY_INIT
      if (Ddr4) {
        ScWrDelay.Bits.Dec_tCWL = 6; // DDR4 2400/G1
      } else if (Ddr5) {
        ScWrDelay.Bits.Dec_tCWL = 6; // DDR5 3200/G2
      }
#endif
      Offset = OFFSET_CALC_MC_CH (MC0_CH0_CR_SC_WR_DELAY_REG, MC1_CH0_CR_SC_WR_DELAY_REG, Controller, MC0_CH1_CR_SC_WR_DELAY_REG, IpChannel);
      MrcWriteCR (MrcData, Offset, ScWrDelay.Data);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Mc%d C%d Dec_tCWL: %d\n", Controller, Channel, ScWrDelay.Bits.Dec_tCWL);

      MrcGetTxDqFifoDelays (MrcData, Controller, Channel, tCWL, AddTcwl, DecTcwl, &tCWL4TxDqFifoWrEn, &tCWL4TxDqFifoRdEn);

#ifndef AV_PHY_INIT
      MrcGetSetMcCh (MrcData, Controller, Channel, TxDqFifoWrEnTcwlDelay,           WriteToCache | PrintValue, &tCWL4TxDqFifoWrEn);
      MrcGetSetMcCh (MrcData, Controller, Channel, TxDqFifoRdEnTcwlDelay,           WriteToCache | PrintValue, &tCWL4TxDqFifoRdEn);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmIocRptChRepClkOn,             WriteToCache | PrintValue, &GetSetVal3);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmIocTxDqFifoRdEnPerRankDelDis, WriteToCache | PrintValue, &GetSetVal3);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmIocCmdAnlgEnGraceCnt,         WriteToCache | PrintValue, &GetSetVal4);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmIocTxAnlgEnGraceCnt,          WriteToCache | PrintValue, &GetSetVal5);
      MrcGetSetMcChRnk (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, TxDqFifoRdEnFlybyDelay, WriteToCache | PrintValue, &GetSetVal3);
#endif
    } // Channel
  } // Controller

#if !defined(AV_PHY_INIT)
  // Initialize Rx and Tx Data CRs
  // RxDqsN/P_Pi = 32, RcvEn = 64
  RxDqsPPi    = 32;
  RxDqsNPi    = 32;
  RxEqInit    = 0xA;
  RxCInit     = 0x1;
  RxRInit     = 0x1;
  RxTap0Init  = 0xA;
  RxTap1Init  = 0x5;

  RxDqsBitPi  = 0xC;

  // Set TxEq to full strength, TxDqs = 0 and TxDq = 32,
  TxEqInit    = 0xA; // Full Drive
  TxDqsPi     = 64;
  TxDqPi      = TxDqsPi + (Gear2 ? 96 : 32);
  // TxDqsBitPi has smaller range than TxDqBitPi.  We want to start with a matched delay in both strobe and data.
  // Thus assign DQ the same value as DQS.  Start the delay 1/2 into the range.
  MrcGetSetLimits (MrcData, TxDqsBitDelay, &MinVal, &MaxVal, NULL);
  TxDqsBitPi = ((UINT32) (MaxVal - MinVal)) / 2;
  TxDqBitPi = TxDqsBitPi;
  // Rx Amplifier voltage offset {0: Most negative offset,... 15: zero offset, ... 30: Most positive offset}
// @todo_adl  MrcGetSetLimits (MrcData, RxVoc, &MinVal, &MaxVal, NULL);
  GetSetVal = ((UINT32) (MaxVal - MinVal)) / 2;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      // RxGroup
      if (Outputs->RxMode == MrcRxModeUnmatchedRxWRload || Outputs->RxMode == MrcRxModeUnmatchedRxWPpath) {
        MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxTap0, WriteToCache | PrintValue, &RxTap0Init);
        MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxTap1, WriteToCache | PrintValue, &RxTap1Init);
      } else {
        MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxEq, WriteToCache | PrintValue, &RxEqInit);
        MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxC, WriteToCache | PrintValue, &RxCInit);
        MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxR, WriteToCache | PrintValue, &RxRInit);
      }
      MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxDqsNDelay, WriteToCache | PrintValue, &RxDqsNPi);
      MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxDqsPDelay, WriteToCache | PrintValue, &RxDqsPPi);
      MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RecEnDelay,  WriteToCache | PrintValue, &RcvEnPi[Controller][Channel]);
      MrcGetSetBit (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, MAX_BITS, RxDqsBitDelay,    WriteToCache | GSM_UPDATE_HOST | PrintValue, &RxDqsBitPi);
      MrcGetSetBit (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, MAX_BITS, RxVoc,           WriteToCache | PrintValue, &GetSetVal);
      MrcGetSetBit (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, MAX_BITS, RxVocUnmatched,  WriteToCache | PrintValue, &GetSetVal);
      MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxDqsAmpOffset,         WriteToCache | PrintValue, &GetSetVal);
      // TxGroup
      MrcGetSetBit (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, MAX_BITS, TxDqBitDelay, WriteToCache | GSM_UPDATE_HOST | PrintValue, &TxDqBitPi);
      MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, TxDqsBitDelay, WriteToCache | GSM_UPDATE_HOST | PrintValue, &TxDqsBitPi);
      MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, TxEq,        WriteToCache | PrintValue, &TxEqInit);
      MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, TxDqDelay,   WriteToCache | PrintValue, &TxDqPi);
      MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, TxDqsDelay,  WriteToCache | PrintValue, &TxDqsPi);
    } // Channel
  } // Controller
  // Set Rx/Tx per-bit deskew step size to Single Step (x1)
  MrcGetSetNoScope (MrcData, GsmIocVccDllRxDeskewCal, WriteToCache, &GetSetDis);
  MrcGetSetNoScope (MrcData, GsmIocVccDllTxDeskewCal, WriteToCache, &GetSetDis);
  MrcFlushRegisterCachedData (MrcData);
#else // AV_PHY_INIT
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxAmpOffsetEn, WriteNoCache, &GetSetDis);
#endif

  // Initial value to corresponding 0.
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATAOFFSETTRAIN_REG, 0x0);
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRDATAOFFSETCOMP_REG, 0x0);
#ifndef AV_PHY_INIT
  // Initialize DataControl0:6
  GetSetVal = SAFE ? 1 : (Lpddr) ? 1 : 0;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDqOdtParkMode, WriteToCache, &GetSetVal);
  GetSetVal = (Lpddr4) ? 2 : (Lpddr5) ? 3 : 0; // DDR4 is the last case
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataDqsOdtParkMode, WriteToCache, &GetSetVal);
  GetSetVal = RxMode;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxVocMode,  WriteToCache, &GetSetVal);
  GetSetVal = StaticOdtDis;
  MrcGetSetNoScope (MrcData, GsmIocCompOdtStaticDis, WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataOdtStaticDis, WriteToCache | PrintValue, &GetSetVal);
  GetSetVal = 1;  // 2 qclk DLL mask
  MrcGetSetChStrb (MrcData, 0, 0, MAX_SDRAM_IN_DIMM, GsmIocDllMask,  WriteToCache | PrintValue, &GetSetVal);
  //MRC:RestricteContent - See DATA0CH0_CR_DDRCRDATACONTROL1_STRUCT
  //MrcGetSetLimits (MrcData, SenseAmpDelay, &GetSetVal, NULL, NULL); // SenseAmpDelay, McOdtDelay are the same.
  GetSetVal = 0;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, McOdtDelay,        WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, SenseAmpDelay,     WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, DqsOdtDelay,       WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetLimits (MrcData, SenseAmpDuration, NULL, &GetSetVal, NULL); // SenseAmpDuration, McOdtDuration are the same.
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, SenseAmpDuration,  WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, McOdtDuration,     WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetLimits (MrcData, DqsOdtDuration, NULL, &GetSetVal, NULL);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, DqsOdtDuration,    WriteToCache | PrintValue, &GetSetVal);
#endif
  GetSetVal = (Lpddr || Ddr5) ? 3 : 1;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataRxBurstLen, WriteToCache | PrintValue, &GetSetVal);
  MrcGetSetLimits (MrcData, GsmIocRxClkStg, NULL, &GetSetVal, NULL);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxClkStg, WriteToCache | PrintValue, &GetSetVal);
#ifndef AV_PHY_INIT
  GetSetVal = 1;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocConstZTxEqEn, WriteToCache | PrintValue, &GetSetVal);
#endif
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      // Rx Vref Config for LPDDR
      if (Lpddr) {
        DramVdd = Outputs->VddVoltage[Profile];
        // @todo: This is tied to the Pull Up Calibration selected in LPDDR4 MR.
        Voh =  ((Lpddr5) ? DramVdd / 2 : ((Outputs->Lp4x) ? 360 : 440));  // mV
        PuDeltaV = DramVdd - Voh;
        EffPullUp = PuDeltaV * Inputs->RcompTarget[RdOdt];
        EffPullUp = DIVIDEROUND (EffPullUp, Voh);
        MrcSetIdealRxVref (MrcData, Controller, Channel, EffPullUp, 40, Inputs->RcompTarget[RdOdt], TRUE);
      }

      ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
      for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
        // These CRs do a lot of RMW.
        ChannelOut->DataCompOffset[Byte]  = 0;
      }
    } // Channel
  } // Controller

  if (Ddr4) {
    MrcSetDefaultRxVrefDdr4 (MrcData, TRUE, TRUE);
  }

  // Initialize VssHi CRs @todo update for TGL
  // VssHi:
  //   For Y/U: Max {Vccddq-0.95V, 0.3V}
  //   For H/S: Max {Vccddq-0.95V, 0.25V}
  VssHi = ((UINT16) Vdd - VssHiSwingTarget);
  VssHiMax = UlxUlt ? 300 : 250;
  VssHi = MAX (VssHi, VssHiMax);

// @todo_adl  CCCBscanData.Bits.BiasRloadVref = DataControl5.Bits.BiasRloadVref;
// @todo_adl  CCCBscanData.Bits.BiasIrefAdj   = DataControl5.Bits.BiasIrefAdj;
  CCCBscanData.Bits.BiasCasAdj    = 2 ;
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRBSCANDATA_REG, CCCBscanData.Data); // Broadcast write

/* @todo_adl
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Init Dimm Vref CRs\n");
  // Initialize Dimm Vref CRs
  VrefControl.Data = 0;

  // @todo: Select correct target based on CPU SKU. Target = 42 mV which has different precision for different ULTs. ICL code
  TempVar1 = DIVIDEROUND (VccIoMv, 128);
  TempVar2 = 2 * Vdd2Mv;
  TempVar3 = DIVIDEROUND (TempVar2, 382);
  TempVar4 = DIVIDEROUND (TempVar3, TempVar1);
  VtSlope = TempVar4;
  TempVar1 = DIVIDEROUND (Vdd2Mv, 96);
  SignedTempVar = DIVIDEROUND (42, TempVar1);
  VtOffset = RANGE (SignedTempVar, -16, 15);
  if (Inputs->NewFeatureEnable2) {
    HiBwDiv = 2;
    LowBwDiv = 3;
    // QclkFreq/2^(5+LoBWDivider+HiBWDivider+SampleDivider) <= 0.6 MHz
    // Scale to kHz for precision.
    TempVar1 = QclktoFreq * 1000;
    TempVar1 /= (1 << (5 + HiBwDiv + LowBwDiv));
    SampleDiv = 0;
    while (TempVar1 > 600) {
      TempVar1 /= 2;
    }
  } else {
    HiBwDiv = 0;
    LowBwDiv = 0;
    SampleDiv = 1;
  }

  if (Inputs->UlxUlt) {
    VrefControl.Bits.HiBWEnable = 1;
    VrefControl.Bits.VtSlope = VtSlope;
    VrefControl.Bits.VtOffset = VtOffset;
    VrefControl.Bits.SelCode = 1;
    VrefControl.Bits.SampleDivider = SampleDiv;

#ifdef CTE_FLAG
    // For fast convergence on CTE.SampleDivider can no longer be 0 #2204469942
    VrefControl.Bits.SampleDivider = 1;  // Bits 6:4
#endif // CTE_FLAG
  } else {
    VrefControl.P0Bits.HiBWEnable     = 1;
    VrefControl.P0Bits.VtSlope        = VtSlope;
    VrefControl.P0Bits.VtOffset       = VtOffset;
    VrefControl.P0Bits.SelCode        = 1;
    VrefControl.P0Bits.SampleDivider  = SampleDiv;
    VrefControl.P0Bits.HiBWDivider    = HiBwDiv;
    VrefControl.P0Bits.LoBWDivider    = LowBwDiv;

#ifdef CTE_FLAG
    // For fast convergence on CTE.SampleDivider can no longer be 0 #2204469942
    VrefControl.P0Bits.SampleDivider = 1;  // Bits 5:4
#endif // CTE_FLAG
  }

  MrcWriteCR (MrcData, DDRVREF_CR_DDRCRVREFCONTROL_REG, VrefControl.Data);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "DdrCrVrefControl: 0x%X\n", VrefControl.Data);
  */
/* @todo_adl
  // Enable all DimmVref and VddHi based on VddVoltage
  if (Inputs->UlxUlt) {
    VrefAdj1.Data = 0;
    // Only DDR4 uses these Vrefs.  Only enable if the DDR4 Controller exists.
    // Set Vref fields to 1/2 VDD.
    if (Ddr4) {
      if (Outputs->ValidMcBitMask & 1) {
        VrefAdj1.Bits.enCA0Vref = 1;
      }
      if (Outputs->ValidMcBitMask & 2) {
        VrefAdj1.Bits.enCA1Vref = 1;
      }
    }
    VrefAdj1.Bits.SAGVOpenLoopEn = 1;

    MrcWriteCR (MrcData, DDRVREF_CR_DDRCRVREFADJUST1_REG, VrefAdj1.Data);
  } else {
    VrefCh0.Data = 0;
    VrefCh1.Data = 0;
    // Only DDR4 uses these Vrefs.  Only enable if the DDR4 Controller exists.
    // Set Vref fields to 1/2 VDD.
    if (Ddr4) {
      if (IsDimmPresent (MrcData, 0, 0, 0, 0) == mrcSuccess) {
        VrefCh0.P0Bits.enCA0Vref = 1;
      }
      if (IsDimmPresent (MrcData, 0, 0, 0, 1) == mrcSuccess) {
        VrefCh0.P0Bits.enCA1Vref = 1;
      }
      if (IsDimmPresent (MrcData, 0, 1, 0, 0) == mrcSuccess) {
        VrefCh1.P0Bits.enCA0Vref = 1;
      }
      if (IsDimmPresent (MrcData, 0, 1, 0, 1) == mrcSuccess) {
        VrefCh1.P0Bits.enCA1Vref = 1;
      }
    }
    VrefCh0.P0Bits.SAGVOpenLoopEn = 1;
    VrefCh1.P0Bits.SAGVOpenLoopEn = 1;

    MrcWriteCR (MrcData, DDRVREF_CR_DDRCRVREFCH0_P0_REG, VrefCh0.Data);
    MrcWriteCR (MrcData, DDRVREF_CR_DDRCRVREFCH1_P0_REG, VrefCh1.Data);
  }
  */
  if (Ddr4) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "DDR4: setting initial CmdV\n");
// @todo_adl    ChangeMargin (MrcData, CmdV, MID_INT_VREF, 0, 1, 0, 0, 0, 0, 0, 1, 0);
  }
  /* @todo_adl
  VrefAdj2.Data = 0;
  VrefAdj2.Bits.HiZTimerCtrl  = DDRVREF_CR_DDRCRVREFADJUST2_HiZTimerCtrl_MAX;
  VrefAdj2.Bits.LockOvrd = SAFE ? 0 : 1;
  TempVar1 = (1000 / RefClkPs) - 4; // TBDns
  TempVar2 = MrcLog2(TempVar1);
  VrefAdj2.Bits.LockTimer    = RANGE(TempVar2, 0, 7);
#ifdef CTE_FLAG
  VrefAdj2.Bits.LockTimer = 0;
#endif
  TempVar1 = (Vdd2Mv / 382);
  TempVar2 = VccIoMv / 128;
  TempVar3 = DIVIDEROUND(TempVar1, TempVar2);
  //MaxMin( Rnd(TBD*(VccDD2/382) / (VccIO/128)), 0, 31
  VrefAdj2.Bits.VtSlopeSAGV = RANGE(TempVar3, 0, 31);
  SignedTempVar1 = (Vdd2Mv / 96);
  // MaxMin( Rnd(TBD mV / (VccDD2/96) ), -16, 15).
  VrefAdj2.Bits.VtOffsetSAGV = RANGE(SignedTempVar1, -16, 15);
  VrefAdj2.Bits.SagvVtCtl   = 1;
  VrefAdj2.Bits.GateICinDVFS = SAFE ? 1 : 0;
  MrcWriteCR (MrcData, DDRVREF_CR_DDRCRVREFADJUST2_REG, VrefAdj2.Data);
  */
  MrcGetTdqs2dq (MrcData, TRUE, &Tdqs2dqMinFs);
  TempVar1 = 150 / (QclkPs / (Gear2 ? 128 : 64));

  // This register config has been moved here because TimingProfile is assigned above.
  // Expectation : tCK should be same across all populated instances MC/CH/DIMM
  WrRetrainControlStatus.Data = 0;
  WrRetrainControlStatus.Bits.inittrain = 0;
  WrRetrainControlStatus.Bits.duration = ((4096 - 1) * 2 * Tdqs2dqMinFs) / (16 * ((UINT8)TimingProfile->tCK));
  WrRetrainControlStatus.Bits.largechangedelta = MrcLog2 (TempVar1);
  MrcWriteCrMulticast (MrcData, DATA_CR_DDRCRWRRETRAINCONTROLSTATUS_REG, WrRetrainControlStatus.Data);

  // WCK Configuration
  WckControl.Data = 0;
  if (Lpddr5) {
    if (Inputs->UlxUlt) {
      // Lp5Mode bit field does not exist on TGL P0
      WckControl.Bits.Lp5Mode = 1;
    }
    Data32 = MrcGetWckPreStatic (MrcData);
    WckControl.Bits.tWckPre = Data32 * ((Gear2) ? 4 : 2) + 1;
    WckControl.Bits.tWckHalfRate = 2 + ((Gear2) ? 0 : 1);
    Data32 = MrcGetWckPreWrTotal (MrcData);
    //WckControl.Bits.cas2wrwck = Data32 * ((Gear2) ? 4 : 2);
    // JEDEC Spec Table 26. WCK2CK Sync AC Parameters for Write operation
    // tCWL was scaled to QCLK/WCK above.
    WckControl.Bits.cas2wrwck = tCWL + 4 - (Data32 * 4) + ((Gear2) ? -1 : 1);
    Data32 = MrcGetWckPreRdTotal (MrcData);
    // tCL was scaled to QCLK/WCK above.
    WckControl.Bits.cas2rdwck = tCL + 4 - (Data32 * 4) + ((Gear2) ? -1 : 1);
    MrcGetSetNoScope (MrcData, GsmIocLp5Wck2CkRatio, WriteToCache, &GetSetEn);
  }
  MrcWriteCR (MrcData, MCMISCS_DDRWCKCONTROL_REG, WckControl.Data);

  // Initialize CCC Non-training parameters
  // @todo: Defining init dummies to be completed later.
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CH1CCC_CR_DDRCRCCCVOLTAGEUSED_REG, CCCBlock);
  Data32 = MrcReadCR (MrcData, Offset);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCVOLTAGEUSED_REG, Data32);

  // TxEq, CtlSRDrv, PiGate, LpddrMode
  Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRCCCCLKCONTROLS_REG, CH1CCC_CR_DDRCRCCCCLKCONTROLS_REG, CCCBlock);
  Data32 = MrcReadCR (MrcData, Offset);
  MrcWriteCrMulticast (MrcData, CCC_CR_DDRCRCCCCLKCONTROLS_REG, Data32);

#ifndef AV_PHY_INIT
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannels; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        IntCmdTiming = &IntOutputs->Controller[Controller].CmdTiming[Channel];

        // Initialize CRs shared between CKE/CTL/CMD/CLK
        GetSetVal2 = CmdPiDelay;
        //@todo - DDR5 support.
        Offset = (Ddr4) ? MRC_DDR4_CMD_GRP_MAX : (Ddr5) ? MRC_DDR5_CMD_GRP_MAX: 1;
        for (Index = 0; Index < Offset; Index++) {
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, Index, CmdGrpPi, WriteToCache | PrintValue, &GetSetVal2);
          IntCmdTiming->CmdPiCode[Index] = (UINT16) GetSetVal2;
        }
        GetSetVal  = ClkPiDelay;
        GetSetVal2 = CtlPiDelay;
        GetSetVal3 = WckPiDelay;

        for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, CtlGrpPi, WriteToCache | PrintValue, &GetSetVal2);
          IntCmdTiming->CtlPiCode[Rank] = (UINT16) GetSetVal2;
          if (Ddr) {
            MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, ClkGrpPi, WriteToCache | PrintValue, &GetSetVal);
            IntCmdTiming->ClkPiCode[Rank] = (UINT16) GetSetVal;
            MrcGetSetCcc (MrcData, Controller, Channel, Rank, 0, CkeGrpPi, WriteToCache | PrintValue, &GetSetVal2);
            IntCmdTiming->CkePiCode[Rank] = (UINT16) GetSetVal2;
          }
        }
        // Clk/Cke/Wck are per-channel for LPDDR
        if (Lpddr) {
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, 0, ClkGrpPi, WriteToCache | PrintValue, &GetSetVal);
          IntCmdTiming->ClkPiCode[0] = (UINT16) GetSetVal;
        }
        if (Lpddr4) {
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, 0, CkeGrpPi, WriteToCache | PrintValue, &GetSetVal2);
          IntCmdTiming->CkePiCode[0] = (UINT16) GetSetVal2;
        } else if (Lpddr5) {
          MrcGetSetCcc (MrcData, Controller, Channel, MRC_IGNORE_ARG, 0, WckGrpPi, WriteToCache | PrintValue, &GetSetVal3);
          IntCmdTiming->WckPiCode = (UINT16) GetSetVal3;
        }
      }
    } // Channel
  } // Controller
#endif
  MrcFlushRegisterCachedData (MrcData);

  // Disable Periodic RComp
  // Set periodic comp = (10uS * 2^COMP_INT)
  CrMCompPcu.Data               = 0;
  CrMCompPcu.Bits.COMP_DISABLE  = 1;
  CrMCompPcu.Bits.COMP_INTERVAL = COMP_INT;  // Set COMP_INT to happen every 10mS
  MrcWriteCR (MrcData, M_COMP_PCU_REG, CrMCompPcu.Data);

#ifndef AV_PHY_INIT
  // Initialize COMP CRs for DqOdt, DqDrv, CmdDrv, CtlDrv, ClkDrv
  GetSetVal = SAFE ? 1 : 0; // Override PwrDn in Safe Mode
  MrcGetSetNoScope (MrcData, GsmIocCompClkOn, WriteCached, &GetSetVal);

  MrcSetupVtt (MrcData, TRUE);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Init MISC CRs\n");
  // Initialize MISC CRs
  for (Index = 0; Index < MRC_NUM_BYTE_GROUPS; Index++) {
    GetSetVal = 3; // 8 QCLKs
    MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, GsmIocWlWakeCyc, WriteCached, &GetSetVal);
    TempVar1 = DIVIDEROUND (20000, QclkPs);
    GetSetVal = (MrcLog2 (TempVar1)) - 2;
    MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, GsmIocWlSleepCyclesAct, WriteCached, &GetSetVal);
    if (!UlxUlt) {
      TempVar1 = DIVIDEROUND (160000, QclkPs);
      GetSetVal = (MrcLog2 (TempVar1)) - 2;
      MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Index, GsmIocWlSleepCyclesLp, WriteCached, &GetSetVal);
    }
  }
#endif
  // Keep scrambling disabled for training
  for (Channel = 0; Channel < MRC_NUM_CCC_INSTANCES; Channel++) {
    Offset = OFFSET_CALC_CH (DDRSCRAM_CR_DDRSCRAMBLECH0_REG, DDRSCRAM_CR_DDRSCRAMBLECH1_REG, Channel);
    MrcWriteCR (MrcData, Offset, 0);
  }

  // Set the DDR voltage in PCU
  MrcSetPcuDdrVoltage (MrcData, Vdd);

  if (Inputs->BootMode == bmCold) {
    // Must be the last register written for basic init (Must be after MC Init).
    MrcWriteCR (MrcData, DDRSCRAM_CR_DDRLASTCR_REG, 1);
  }
#ifndef AV_PHY_INIT
  // First RCOMP happens in this function.
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Init COMP CRs\n");
  // Walk backwards here because RdOdt depends on WrDS.
  for (Index = (sizeof (CompParamList) / sizeof (CompParamList[0])); Index > 0; Index--) {
    UpdateCompTargetValue (MrcData, CompParamList[Index - 1], Inputs->RcompTarget, TRUE);
  }
  MrcVssHiRegulatorOffsetCorrection (MrcData, TRUE);
#endif

#ifdef MRC_DEBUG_PRINT
  /* @todo_adl
  DataCompVtt.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRDATACOMPVTT_REG);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "final panicvttup%d = 0x%X\n", 0, DataCompVtt.Bits.PanicVttUp0);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "final panicvttup%d = 0x%X\n", 1, DataCompVtt.Bits.PanicVttUp1);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "final panicvttdn%d = 0x%X\n", 0, DataCompVtt.Bits.PanicVttDn0);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "final panicvttdn%d = 0x%X\n", 1, DataCompVtt.Bits.PanicVttDn1);
  */
#endif //MRC_DEBUG_PRINT

  MrcBlockTrainResetToggle (MrcData, FALSE);
  IoReset (MrcData);
  MrcBlockTrainResetToggle (MrcData, TRUE);

  return Status;
}

/**
  This function checks for Scomp Training or Bypass mode.

  @param[in, out] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcDdrScomp (
  IN OUT MrcParameters *const MrcData
  )
{
  static const UINT8  ScompParam[4] = {SCompDq, SCompCmd, SCompCtl, SCompClk};
  MrcInput           *Inputs;
  MrcStatus          Status;
  MrcOutput          *Outputs;
  UINT8              ScompIndex;
  UINT8              ScompInitMax;
  BOOLEAN            Lpddr;
  UINT8              ScompBypassBitMask;

  Inputs             = &MrcData->Inputs;
  Outputs            = &MrcData->Outputs;
  Lpddr              = Outputs->Lpddr;
  ScompInitMax       = ARRAY_COUNT (ScompParam);
  Status             = mrcSuccess;
  ScompBypassBitMask = 0;

  for (ScompIndex = 0; ScompIndex < ScompInitMax; ScompIndex++) {
    if (Lpddr && Inputs->ScompBypass[ScompIndex]) {
      ScompBypassBitMask |= (1 << ScompIndex);
    }
  }

  // Skip MrcDdrScompInit if all ScompBypass is set
  if (ScompBypassBitMask < 0xF) {
    Status = MrcDdrScompInit(MrcData, ScompBypassBitMask);
  }
  MrcDdrScompBypass(MrcData, ScompBypassBitMask);

  return Status;
}

/**
This function bypass the Slew Rate Delay Cells of the give Scomp Type.

  @param[in, out] MrcData    - Include all MRC global data.
  @param[in]      ScompIndex - Scomp Type

**/
VOID
MrcDdrScompBypass (
  IN OUT MrcParameters *const MrcData,
  IN     UINT8         ScompBypassBitMask
  )
{
  /* @todo_adl
  static const UINT8  ScompParam[4]  = { SCompDq, SCompCmd, SCompCtl, SCompClk };
  static const GSM_GT SCompBypass[4] = { SCompBypassDq, SCompBypassCmd, SCompBypassCtl, SCompBypassClk };
  static const GSM_GT SCompCells[4]  = {TxSlewRate, CmdSlewRate, CtlSlewRate, ClkSlewRate};
  static const GSM_GT SCompCode[4]   = { SCompCodeDq, SCompCodeCmd, SCompCodeCtl, SCompCodeClk };
  const MrcInput    *Inputs;
  MrcDebug          *Debug;
  MrcOutput         *Outputs;
  UINT32            Offset;
  UINT32            Data;
  UINT32            Index;
  UINT32            DdrCaSlwDlyBypass;
  UINT8             ScompIndex;
  UINT8             ScompInitMax;
  INT64             GetSetVal;
  BOOLEAN           Lpddr;
  DDRPHY_COMP_CR_DDRCRCOMPOVR0_STRUCT  DdrCrCompOvr;
  DATA0CH0_CR_DDRCRDATACONTROL2_STRUCT DataControl2;
  CH0CCC_CR_DDRCRPINSUSED_STRUCT       CccPinsUsed;
  DATA0CH0_CR_RCOMPDATA0_STRUCT        CompData0;

  Inputs  = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Debug = &Outputs->Debug;
  ScompInitMax = ARRAY_COUNT(ScompParam);
  DataControl2.Data = MrcReadCR (MrcData, DATA_CR_DDRCRDATACONTROL2_REG);
  Lpddr = Outputs->Lpddr;
  DdrCrCompOvr.Data = MrcReadCR(MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR0_REG);

  CompData0.Data = ReadFirstRcompReg (MrcData);

  for (ScompIndex = 0; ScompIndex < ScompInitMax; ScompIndex++) {
    if ((ScompBypassBitMask >> ScompIndex) & 1) {
      // Set Bypass for the given Scomp Type
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Bypass %s\n", CompGlobalOffsetParamStr[ScompParam[ScompIndex]]);
      GetSetVal = 1;
      if (SCompBypass[ScompIndex] == SCompBypassDq) {
        MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, SCompBypass[ScompIndex], WriteToCache, &GetSetVal);
      } else {
        MrcGetSetCcc (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MRC_IGNORE_ARG, MAX_SYS_CHANNEL, SCompBypass[ScompIndex], WriteCached, &GetSetVal);
      }
      // Set SComp Override
      if (DataControl2.Bits.DqSlewDlyByPass || Lpddr) {
        DdrCrCompOvr.Bits.DqSR = 1;
      }
      GetSetVal = 0;
      MrcGetSetNoScope (MrcData, SCompCode[ScompIndex], WriteCached, &GetSetVal);
      // Set Scomp Cells to 0
      MrcGetSetNoScope (MrcData, SCompCells[ScompIndex], WriteCached, &GetSetVal);
    }
  }
  if (DataControl2.Bits.DqSlewDlyByPass) {
    CompData0.Bits.SlewRateComp = 0;
  }
  MrcWriteCrMulticast (MrcData, DATA_CR_RCOMPDATA0_REG, CompData0.Data);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR0_REG, DdrCrCompOvr.Data);

  DdrCaSlwDlyBypass = (Outputs->DdrType != MRC_DDR_TYPE_DDR4);
  for (Index = 0; Index < MRC_NUM_CCC_INSTANCES; Index++) {
    Offset = OFFSET_CALC_CH (CH0CCC_CR_DDRCRPINSUSED_REG, CH1CCC_CR_DDRCRPINSUSED_REG, Index);
    CccPinsUsed.Data = MrcReadCR (MrcData, Offset);
    CccPinsUsed.Bits.DdrClkSlwDlyBypass = 1;
    CccPinsUsed.Bits.DdrCtlSlwDlyBypass = 1;//(Outputs->DdrType == MRC_DDR_TYPE_DDR5);
    CccPinsUsed.Bits.DdrCaSlwDlyBypass = DdrCaSlwDlyBypass;
    Data = CccPinsUsed.Data;
    MrcWriteCR (MrcData, Offset, Data);
  }
  */
}

/**
  This function initializes the Slew Rate of DQ, CMD, CTL and CLK buffers in DDRIO.

  @param[in, out] MrcData    - Include all MRC global data.
  @param[in]      ScompIndex - Scomp Type

  @retval MrcStatus - mrcSuccess if successful or an error status
**/
MrcStatus
MrcDdrScompInit (
  IN OUT MrcParameters *const MrcData,
  IN     UINT8         ScompBypassBitMask
  )
{
  static const GSM_GT SCompCells[4]           = {TxSlewRate, CmdSlewRate, CtlSlewRate, ClkSlewRate};
  static const GSM_GT SCompPC[4]              = {DqScompPC, CmdScompPC, CtlScompPC, ClkScompPC};
  static const GSM_GT SCompCode[4]            = {SCompCodeDq, SCompCodeCmd, SCompCodeCtl, SCompCodeClk};
  //static const UINT8  DesiredSlewRateUDdr4[4] = {45, 32, 30, 45};  // units of [mV/ps] and multiplied by 10 for integer math precision in DesiredBufferSegmentDelay calculation below.
  static const UINT8  DesiredSlewRateUDdr4[4] = {45, 35, 35, 40};  // units of [mV/ps] and multiplied by 10 for integer math precision in DesiredBufferSegmentDelay calculation below.
  static const UINT8  DesiredSlewRateHDdr4[4] = {45, 30, 30, 50};  // units of [mV/ps] and multiplied by 10 for integer math precision in DesiredBufferSegmentDelay calculation below.
  static const UINT8  DesiredSlewRateLpdr4[4] = {55, 35, 35, 45};  // units of [mV/ps] and multiplied by 10 for integer math precision in DesiredBufferSegmentDelay calculation below.
  static const UINT8  RcompParam[4]           = {WrDS, WrDSCmd, WrDSCtl, WrDSClk};
  static const UINT8  ScompParam[4]           = {SCompDq, SCompCmd, SCompCtl, SCompClk};
#ifdef MRC_DEBUG_PRINT
  static const char   *ScompHeader[4]         = { "Dq", "Cmd", "Ctl", "Clk" };
#endif
  MrcInput          *Inputs;
  MrcDebug          *Debug;
  const MRC_FUNCTION *MrcCall;
  MrcOutput         *Outputs;
  MrcControllerOut  *ControllerOut;
  MrcChannelOut     *ChannelOut;
  UINT8 const       *DesiredSlewRate;
  MrcVddSelect      Vdd;
  INT64             GetSetVal;
  UINT32            OdtWr;
  UINT32            OdtNom;
  UINT32            DelayCells;
  UINT32            DesiredBufferSegmentDelay[4];
  UINT32            MinChainLength;
  UINT32            MaxChainLength;
  UINT32            VSwing;
  UINT32            VHigh;
  UINT32            VLow;
  UINT32            Voh;
  UINT32            Vtt;
  UINT32            VccIo;
  UINT32            Rtt;
  UINT32            Divisor;
  UINT32            Dividend;
  UINT32            OdtValue;
  UINT32            OdtPark;
  UINT32            Data32;
  UINT32            Controller;
  UINT16            CpuRon;
  UINT8             ScompIndex;
  UINT8             ScompInitMax;
  UINT8             CellIndex;
  UINT8             Max;
  UINT8             Channel;
  UINT8             Rank;
  UINT8             Dimm;
  UINT8             SCompResult[4][SCOMP_CELLS_MAX + 1];  // @todo_adl find the correct definition
  UINT8             SCompCount[4];
  UINT8             NCells[4][MAX_EDGES];
  UINT8             ChannelsPopulated;
  UINT8             DimmSerialResistor;
  UINT8             SegmentsPerBuffer;
  BOOLEAN           Ddr4;
  BOOLEAN           Lpddr4;
  BOOLEAN           Is2DPC;
  BOOLEAN           Is2RDimm;
// @todo_adl DDRPHY_COMP_CR_DDRCRCOMPOVR0_STRUCT DdrCrCompOvr;

  Inputs            = &MrcData->Inputs;
  MrcCall           = Inputs->Call.Func;
  Outputs           = &MrcData->Outputs;
  ControllerOut     = &Outputs->Controller[0];
  Debug             = &Outputs->Debug;
  Ddr4              = (Outputs->DdrType == MRC_DDR_TYPE_DDR4);
  Lpddr4            = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  Vdd               = Outputs->VddVoltage[Inputs->MemoryProfile];
  VccIo             = Inputs->VccIomV;
  ScompInitMax      = ARRAY_COUNT (ScompParam);
  Dividend          = 0;

  // Determine slew rate targets according to CPU Sku and DDR type
  if (Inputs->UlxUlt) {
    if (Lpddr4) {
      DesiredSlewRate = DesiredSlewRateLpdr4;
    } else {
      DesiredSlewRate = DesiredSlewRateUDdr4;
    }
  } else {
    // DT/Halo
    DesiredSlewRate = DesiredSlewRateHDdr4;
  }

  for (ScompIndex = 0; ScompIndex < ScompInitMax; ScompIndex++) {
    if (((ScompBypassBitMask >> ScompIndex) & 1) != 1) {
      // Setup of DQ, CMD, CTL, CLK SCompPC
      GetSetVal = 1; // Cycle lock
      MrcGetSetNoScope(MrcData, SCompPC[ScompIndex], WriteCached, &GetSetVal);
    }
  }

  MrcCall->MrcSetMem (SCompResult[0], sizeof (SCompResult), 0);
  MrcCall->MrcSetMem (SCompCount, sizeof (SCompCount), 0);
  MrcCall->MrcSetMem ((UINT8 *) NCells, sizeof (NCells), 0);
  // Gather results for 3-15 ScompCells
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t\tScompCode\n\t");
  for (ScompIndex = 0; ScompIndex < ScompInitMax; ScompIndex++) {
    if (((ScompBypassBitMask >> ScompIndex) & 1) != 1) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t%s", ScompHeader[ScompIndex]);
    }
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nScompCells\n");
  for (CellIndex = SCOMP_CELLS_MIN; CellIndex <= SCOMP_CELLS_MAX; CellIndex++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t%d\t", CellIndex);
    for (ScompIndex = 0; ScompIndex < ScompInitMax; ScompIndex++) {
      if (((ScompBypassBitMask >> ScompIndex) & 1) != 1) {
        // Setup of DQ, CMD, CTL, CLK ScompCells
        GetSetVal = CellIndex;
        MrcGetSetNoScope (MrcData, SCompCells[ScompIndex], WriteCached, &GetSetVal);
      }
    }

    // Force Comp Cycle
    ForceRcomp (MrcData);

    // Read the SCOMP results
    for (ScompIndex = 0; ScompIndex < ScompInitMax; ScompIndex++) {
      if (((ScompBypassBitMask >> ScompIndex) & 1) != 1) {
        MrcGetSetNoScope (MrcData, SCompCode[ScompIndex], ReadUncached, &GetSetVal);
        SCompResult[ScompIndex][CellIndex] = (UINT8) GetSetVal;
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", SCompResult[ScompIndex][CellIndex]);
      }
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");
  }

// @todo_adl  DdrCrCompOvr.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR0_REG);

  // Compare results
  for (ScompIndex = 0; ScompIndex < ScompInitMax; ScompIndex++) {
    if (((ScompBypassBitMask >> ScompIndex) & 1) != 1) {
      if (ScompParam[ScompIndex] != SCompDq) {
        Max = 62;
      } else {
        Max = 29;
      }
      // DQ, CMD, CTL, CLK ScompCells
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s ", CompGlobalOffsetParamStr[ScompParam[ScompIndex]]);
      if (SCompResult[ScompIndex][SCOMP_CELLS_MIN] <= 1) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Setting Slew Rate Compensation Override\n");
// @todo_adl        DdrCrCompOvr.Data |= (1 << (ScompIndex + 10));
      } else {
        for (CellIndex = SCOMP_CELLS_MIN; CellIndex <= SCOMP_CELLS_MAX; CellIndex++) {
          // Find Minimum usable chain length with unsaturated COMP code value, and at least 2 ticks of margin.
          if ((NCells[ScompIndex][0] == 0) && (SCompResult[ScompIndex][CellIndex] >= 2) && (SCompResult[ScompIndex][CellIndex] < Max)) {
            NCells[ScompIndex][0] = CellIndex;
          }
          if (SCompResult[ScompIndex][CellIndex] > Max) {
            //Exceeds upper limit.
            continue;
          }
          NCells[ScompIndex][1] = CellIndex;
          if (CellIndex == SCOMP_CELLS_MIN) {
            // Skip first index for determining Harmonic lock.
            continue;
          }
          if (SCompResult[ScompIndex][CellIndex] > SCompResult[ScompIndex][CellIndex - 1]) {
            // Harmonic lock occurred.
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Harmonic Lock Occured - ");
            NCells[ScompIndex][1] = CellIndex - 1;
            break;
          }
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "SCompCellsMin = %d, SCompCellsMax = %d\n", NCells[ScompIndex][0], NCells[ScompIndex][1]);
      }
    }
  }
// @todo_adl  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPOVR0_REG, DdrCrCompOvr.Data);

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nScompType\tVSwing\tBufferSegDelay\tSCompPC \tSCompCells\n");
  for (ScompIndex = 0; ScompIndex < ScompInitMax; ScompIndex++) {
    if (((ScompBypassBitMask >> ScompIndex) & 1) != 1) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, " %s\t", CompGlobalOffsetParamStr[ScompParam[ScompIndex]]);
      // Determine Vtt and Voh based on partition being calculated, Dram Type, and source mux
      if (ScompParam[ScompIndex] == SCompDq) {
        Voh = (Lpddr4 && Inputs->LowSupplyEnData) ? VccIo : Vdd;
        Vtt = (Lpddr4) ? 0 : Vdd;
        SegmentsPerBuffer = 2;
      } else {
        Voh = (Lpddr4 && Inputs->LowSupplyEnCcc && (ScompParam[ScompIndex] != SCompCtl)) ? VccIo : Vdd;
        Vtt = (Lpddr4) ? 0 : Vdd / 2;
        SegmentsPerBuffer = (ScompParam[ScompIndex] == SCompClk) ? 2 : 3;
      }
      if ((NCells[ScompIndex][0] == 0) && (NCells[ScompIndex][1] == 0)) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Slew Rate Compensation Override Set\n");
        GetSetVal = 0;
      } else {
        VSwing = 0;
        MinChainLength = MAX (SCOMP_CELLS_MIN, NCells[ScompIndex][0]);
        MaxChainLength = MIN (SCOMP_CELLS_MAX, NCells[ScompIndex][1]);
        ChannelsPopulated = 0;
        CpuRon = Inputs->RcompTarget[RcompParam[ScompIndex]];
        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
            if (MrcChannelExist (MrcData, Controller, Channel)) {
              ChannelOut = &ControllerOut->Channel[Channel];
              ChannelsPopulated++;
              Is2DPC =  (ChannelOut->DimmCount == 2);
              DimmSerialResistor = 15;
              OdtValue = 0;
              if (ScompParam[ScompIndex] == SCompDq) {
                if (!Outputs->DramDqOdtEn) {
                  VSwing += Voh;
                  continue;
                }
              }
              for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank += MAX_RANK_IN_DIMM) {
                if (MrcRankExist (MrcData, Controller, Channel, Rank)) {
                  Dimm = RANK_TO_DIMM_NUMBER (Rank);
                  Is2RDimm = (ChannelOut->Dimm[Dimm].RankInDimm == 1);
                  if ((DimmSerialResistor != 0) && (ChannelOut->Dimm[Dimm].ModuleType == NonDimmMemoryPackage)) {
                    DimmSerialResistor = 0;
                  }
                  switch (ScompParam[ScompIndex]) {
                    case SCompDq:
                      OdtWr = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtWr, FALSE, 0, TRUE);
                      if (Ddr4) {
                        OdtNom = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtNom, FALSE, 0, TRUE);
                        OdtPark = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtPark, FALSE, 0, TRUE);
                        OdtPark = (Is2RDimm) ? OdtPark : 0xFFFF;
                        if (Is2DPC) {
                          // OdtValue = OdtWr || (1st Dimm is 2 rank) ? OdtPark : 0xFFFF) || OdtNom || (2nd Dimm is 2 rank) ? OdtPark : 0xFFFF
                          if (Dimm == 0) {
                            Dividend = OdtWr * OdtPark;
                            Divisor = OdtWr + OdtPark;
                            OdtValue = DIVIDEROUND (Dividend, Divisor);
                          } else {
                            Dividend = OdtNom * OdtPark;
                            Divisor = OdtNom + OdtPark;
                            Data32 = DIVIDEROUND (Dividend, Divisor);
                            Dividend = Data32 * OdtValue;
                            Divisor = Data32 + OdtValue;
                            OdtValue = DIVIDEROUND (Dividend, Divisor);
                          }
                        } else {
                          // OdtValue = OdtWr || (Dimm is 2 rank) ? OdtPark : 0xFFFF
                          Dividend = OdtWr * OdtPark;
                          Divisor = OdtWr + OdtPark;
                          OdtValue = DIVIDEROUND (Dividend, Divisor);
                        }
                      } else { // Lpddr3 || Lpddr4
                        OdtValue = OdtWr;
                      }
                      break;

                    case SCompCmd:
                      if (Ddr4) {
                        if (Is2DPC) {
                          Dividend = 55 * 55;
                          Divisor = 55 + 55;
                          OdtValue = DIVIDEROUND (Dividend, Divisor);
                        } else {
                          OdtValue = 55;
                        }
                      } else { // Lpddr4
                        OdtValue = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtCA, FALSE, 0, TRUE);
                      }
                      break;

                    case SCompCtl:
                      if (Ddr4) {
                        if (Is2DPC) {
                          Dividend = 55 * 55;
                          Divisor = 55 + 55;
                          OdtValue = DIVIDEROUND (Dividend, Divisor);
                        } else {
                          OdtValue = 55;
                        }
                      } else { // Lpddr4
                        OdtValue = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtCA, FALSE, 0, TRUE);
                      }
                      break;

                    case SCompClk:
                      if (Ddr4) {
                        if (Is2DPC) {
                          Dividend = 93 * 93;
                          Divisor = 93 + 93;
                          OdtValue = DIVIDEROUND (Dividend, Divisor);
                        } else {
                          OdtValue = 93;
                        }
                      } else { // Lpddr4
                        OdtValue = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtCA, FALSE, 0, TRUE);
                      }
                      break;

                    default:
                      break;
                  } // Switch ScompType
                } // RankExist
              } // For Rank
              Rtt = DimmSerialResistor + OdtValue;
              Divisor = Rtt + CpuRon;
              if (Divisor == 0) {
                MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "\n%s - ODT, CPU Impedance and Dimm Resistor values are zero\n", gErrString, CompGlobalOffsetParamStr[ScompParam[ScompIndex]]);
                return mrcFail;
              }
              //VHigh = Vtt + ((Voh - Vtt) * Rtt / (CpuRon + Rtt));
              //VLow = Vtt * (CpuRon / (CpuRon + Rtt));
              VHigh = Vtt + (((Voh - Vtt) * Rtt) / (Divisor));
              VLow = (Vtt * CpuRon) / Divisor;
              VSwing += (VHigh - VLow);
            } // ChannelExist
          } // For Channel
        } // Controller
        VSwing /= (ChannelsPopulated == 0) ? 1 : ChannelsPopulated;
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", VSwing);

        // Derating value is 0.8, but because DesiredSlewRate is multiplied by 10 for precision, value of 8 is used here.
        DesiredBufferSegmentDelay[ScompIndex] = (VSwing * 8) / (DesiredSlewRate[ScompIndex] * SegmentsPerBuffer);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t\t", DesiredBufferSegmentDelay[ScompIndex]);
        if (DesiredBufferSegmentDelay[ScompIndex] == 0) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "\n%s BufferSegDelay value is zero\n", gErrString);
          return mrcFail;
        }
        DelayCells = DIVIDEROUND (Outputs->Qclkps, DesiredBufferSegmentDelay[ScompIndex]);
        if (DelayCells > (MaxChainLength + 1)) {
          // Calculated value is larger than maximum chain length.
          DelayCells = DIVIDEROUND (Outputs->Qclkps, 2 * DesiredBufferSegmentDelay[ScompIndex]);
          GetSetVal = 0; // Phase Lock
        } else {
          GetSetVal = 1; // Cycle Lock
        }

        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s\t\t", (GetSetVal == 0) ? "Phase" : "Cycle");
        MrcGetSetNoScope (MrcData, SCompPC[ScompIndex], WriteCached, &GetSetVal);
        DelayCells = RANGE (DelayCells - 1, MinChainLength, MaxChainLength);
        GetSetVal = DelayCells;
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\n", DelayCells);
      }
      MrcGetSetNoScope (MrcData, SCompCells[ScompIndex], WriteCached, &GetSetVal);
    }
  }
  MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n");

  return mrcSuccess;
}

/**
  This function initializes all the necessary registers for basic training by
  Activating and initializing CPGC Engine
  Enables CKE_On for CPGC usage

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess
**/
MrcStatus
MrcPreTraining (
  IN MrcParameters *const MrcData
  )
{
  MrcOutput         *Outputs;
  MrcControllerOut  *ControllerOut;
  MrcChannelOut     *ChannelOut;
  MrcDdrType        DdrType;
  UINT32            Channel;
  UINT32            IpChannel;
  UINT32            ChannelLoopIncrement;
  UINT32            Controller;
  UINT32            Offset;
  UINT8             ChannelMask;
  BOOLEAN           Ddr4;
  BOOLEAN           Lpddr;
  MC0_REQ0_CR_CPGC2_ADDRESS_SIZE_STRUCT Cpgc2AddrSize;
  MC0_REQ0_CR_CPGC_SEQ_CFG_A_STRUCT   CpgcSeqCfgA;
  MC0_CR_CPGC2_CREDIT_CFG_STRUCT      Cpgc20Credits;
  MC0_CR_CPGC2_V_CHICKEN_STRUCT       Cpgc2Chicken;

  Outputs       = &MrcData->Outputs;
  DdrType       = Outputs->DdrType;
  Ddr4          = (DdrType == MRC_DDR_TYPE_DDR4);
  Lpddr         = Outputs->Lpddr;

  ChannelLoopIncrement = (Lpddr) ? 2 : 1;
  // @todo - Update for 1 TE per controller.
  MrcSetNormalMode (MrcData, FALSE); // Go to CPGC mode

  // Assign channel to its serial number
  // Activate CPGC Engines on populated channels and subchannels.
  CpgcSeqCfgA.Data = 0;

  Cpgc2AddrSize.Data = 0;
  Cpgc2AddrSize.Bits.Block_Size_Max_Bank  = (Ddr4) ? MAX_DDR4_x16_BANKS - 1 : 0;
  Cpgc2AddrSize.Bits.Region_Size_Max_Bank = Cpgc2AddrSize.Bits.Block_Size_Max_Bank;
  Cpgc2AddrSize.Bits.Block_Size_Bits_Row  = 0;
  Cpgc2AddrSize.Bits.Block_Size_Bits_Col  = (Ddr4) ? 4 : 5;
  Cpgc2AddrSize.Bits.Region_Size_Bits_Row = Cpgc2AddrSize.Bits.Block_Size_Bits_Row;
  Cpgc2AddrSize.Bits.Region_Size_Bits_Col = Cpgc2AddrSize.Bits.Block_Size_Bits_Col;
  Cpgc2AddrSize.Bits.REQUEST_DATA_SIZE    = 1;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    if (!MrcControllerExist (MrcData, Controller)) {
      continue;
    }
    ControllerOut = &Outputs->Controller[Controller];
    Outputs->McChBitMask |= ControllerOut->ValidChBitMask << (Controller * Outputs->MaxChannels);
    CpgcSeqCfgA.Data = 0;

    Offset = OFFSET_CALC_CH (MC0_CR_CPGC2_V_CHICKEN_REG, MC1_CR_CPGC2_V_CHICKEN_REG, Controller);
    Cpgc2Chicken.Data = MrcReadCR (MrcData, Offset);
    Cpgc2Chicken.Bits.RTN_BYPASS_DISABLE = 1;   // Needed because of two CMF instances in ADL
    MrcWriteCR (MrcData, Offset, Cpgc2Chicken.Data);

    for (Channel = 0; Channel < Outputs->MaxChannels; Channel += ChannelLoopIncrement) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      ChannelMask = (Lpddr) ? ((ControllerOut->ValidChBitMask >> Channel) & 0x3) : 0x1;
      ChannelOut = &ControllerOut->Channel[Channel];
      IpChannel = LP_IP_CH (Lpddr, Channel);

      Offset = OFFSET_CALC_CH (MC0_CR_CPGC2_CREDIT_CFG_REG, MC1_CR_CPGC2_CREDIT_CFG_REG, Controller);
      Cpgc20Credits.Data = MrcReadCR (MrcData, Offset);
      Cpgc20Credits.Bits.RD_CPL_CREDITS_INIT = 28;
      MrcWriteCR (MrcData, Offset, Cpgc20Credits.Data);
      // Set credits_config_done in a separate register write
      Cpgc20Credits.Bits.CREDITS_CONFIG_DONE = 1;
      MrcWriteCR (MrcData, Offset, Cpgc20Credits.Data);

      CpgcSeqCfgA.Bits.CHANNEL_ASSIGN      = (ChannelMask << Channel);
      ChannelOut->CpgcChAssign             = (UINT8) CpgcSeqCfgA.Bits.CHANNEL_ASSIGN;
      CpgcSeqCfgA.Bits.INITIALIZATION_MODE = CPGC20_ACTIVE_MODE;
      Offset = MrcGetCpgcSeqCfgOffset (MrcData, Controller, Channel / ChannelLoopIncrement);
      MrcWriteCR (MrcData, Offset, CpgcSeqCfgA.Data);

      if (Ddr4) {
        // Setup Bank Logical to Physical mapping
        MrcGetSetBankSequence (
          MrcData,
          Controller,
          Channel,
          Ddr4BankMap,
          MAX_DDR4_x16_BANKS,
          MRC_SET
        );
      }
      Offset = OFFSET_CALC_MC_CH (
        MC0_REQ0_CR_CPGC2_ADDRESS_SIZE_REG,
        MC1_REQ0_CR_CPGC2_ADDRESS_SIZE_REG, Controller,
        MC0_REQ1_CR_CPGC2_ADDRESS_SIZE_REG, IpChannel);
      MrcWriteCR64 (MrcData, Offset, Cpgc2AddrSize.Data);
    } // Channel
  } // Controller
  MrcCkeOnProgramming (MrcData);

  return mrcSuccess;
}

/**
  Print non-LP MRs

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess
**/
MrcStatus
MrcPrintDdrMrs (
  IN MrcParameters *const MrcData
  )
{
  MrcDebug          *Debug;
  MrcOutput         *Outputs;
  MrcChannelOut     *ChannelOut;
  UINT32            Channel;
  UINT32            Controller;
  UINT32            Rank;
  UINT32            RankMod2;
  UINT32            MrIndex;

  Outputs       = &MrcData->Outputs;
  Debug         = &Outputs->Debug;

  if (Outputs->Lpddr) {
    return mrcSuccess;
  }
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      // Dump the MR registers for DDR4
      // LPDDR3/4 Jedec Init is done after Early Command Training
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (!(MrcRankExist (MrcData, Controller, Channel, Rank))) {
          continue;
        }
        ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
        RankMod2 = Rank % 2;
        for (MrIndex = mrMR0; MrIndex <= mrMR6; MrIndex++) {
          MRC_DEBUG_MSG (
            Debug,
            MSG_LEVEL_NOTE,
            "MrcSetMR%u  Mc %u Channel %u Rank %u = 0x%X\n",
            MrIndex,
            Controller,
            Channel,
            Rank,
            ChannelOut->Dimm[RANK_TO_DIMM_NUMBER (Rank)].Rank[RankMod2].MR[MrIndex]
          );
        }
      } // Rank
    } // Channel
  } // Controller
  return mrcSuccess;
}

/**
  This function initializes all the necessary registers after main training steps but before LCT.

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess

**/
MrcStatus
MrcPostTraining (
  IN MrcParameters *const MrcData
  )
{
  MrcOutput *Outputs;
  MrcTiming *Timing;
  UINT32    Channel;
  UINT32    Controller;

  Outputs = &MrcData->Outputs;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        Timing = &Outputs->Controller[Controller].Channel[Channel].Timing[MrcData->Inputs.MemoryProfile];
        // Update CmdN timing, Round Trip Latency and tXP
        // OldN = 3N (Gear1) or 2N (Gear2), NewN = 1N or 2N
        UpdateCmdNTiming (
          MrcData,
          Controller,
          Channel,
          Outputs->Gear2 ? (2 * 1) : (2 * 2),
          (Timing->NMode == 2) ? 2 : 0
          );
      }
    }
  }

  return mrcSuccess;
}

/**
  Program PCU_CR_DDR_VOLTAGE register.

  @param[in] MrcData    - Include all MRC global data.
  @param[in] VddVoltage - Current DDR voltage.

**/
void
MrcSetPcuDdrVoltage (
  IN OUT MrcParameters *MrcData,
  IN     MrcVddSelect  VddVoltage
  )
{
  MrcOutput                     *Outputs;
  MrcDebug                      *Debug;
  UINT8                         Data8;
  DDR_VOLTAGE_PCU_STRUCT        DdrVoltage;

  Outputs = &MrcData->Outputs;
  Debug   = &Outputs->Debug;

  switch (VddVoltage) {
    case VDD_1_35:
      Data8 = 1;
      break;

    case VDD_1_20:
      Data8 = 3;
      if (Outputs->DdrType == MRC_DDR_TYPE_DDR4) {
        Data8 = 7;
      }
      break;

    case VDD_1_10:
      Data8 = 6;
      break;

    default:
      Data8 = 0;
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "PCU_CR_DDR_VOLTAGE = 0x%02X\n", Data8);
  DdrVoltage.Data = 0;
  DdrVoltage.Bits.DDR_VOLTAGE = Data8;
  MrcWriteCR (MrcData, DDR_VOLTAGE_PCU_REG, DdrVoltage.Data);
}

/**
  This function sets up the Vtt termination.

  @param[in] MrcData  - Include all MRC global data.
  @param[in] DebugMsg - Whether to display debug messages

**/
void
MrcSetupVtt (
  IN MrcParameters *const MrcData,
  IN BOOLEAN              DebugMsg
  )
{
  /* @todo_adl
  const MrcInput    *Inputs;
  MrcDebug          *Debug;
  MrcOutput         *Outputs;
  INT64             GetSetVal;
  UINT32            Channel;
  UINT32            Controller;
  UINT32            Rank;
  UINT32            Itarget;
  UINT32            IcompUp;
  UINT32            IcompDn;
  UINT32            RonDimm;
  UINT32            TotalRon;
  UINT32            CmdTarget;
  UINT32            CaRonDrvDn;
  UINT32            VttTargetV;
  UINT32            Vdd2Mv;
  UINT32            VccIoMv;
  UINT32            IttcompUp;
  UINT32            TempVar1;
  UINT32            TempVar2;
  UINT32            TempVar3;
  UINT32            TempVar4;
  UINT32            PanicV;
  UINT32            PanicCCCPU;
  UINT16            RodtCpu;
  UINT8             TotalDimms;
  BOOLEAN           CaVoltageSel;
  MrcDebugMsgLevel  DebugLevel;
  DDRPHY_COMP_CR_DDRCRCOMPCTL1_STRUCT       CompCtl1;
  DDRPHY_COMP_CR_DDRCRCOMPCTL0_STRUCT       CompCtl0;
  DDRPHY_COMP_CR_VSSHIPANIC_STRUCT          VssHiPanic;
  DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC_STRUCT   CompVttPanic;
  DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC2_STRUCT  CompVttPanic2;
  DDRVTT0_CR_DDRCRVTTGENCONTROL_STRUCT      VttGenControl;

  Inputs              = &MrcData->Inputs;
  Outputs             = &MrcData->Outputs;
  Debug               = &Outputs->Debug;
  TotalRon            = 0;
  TotalDimms          = 0;
  CaRonDrvDn          = Inputs->RcompTarget[WrDSCmd]; // @todo Check the difference between DrvUp and DrvDn
  Vdd2Mv              = Outputs->Vdd2Mv;
  VccIoMv             = Inputs->VccIomV;
  Itarget             = 10; // Up0=Up1=Dn0=Dn1=10mA

  VssHiPanic.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_VSSHIPANIC_REG);
  CompCtl0.Data   = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL0_REG);
  VttGenControl.Data = MrcReadCR (MrcData, DDRVTT0_CR_DDRCRVTTGENCONTROL_REG); // All VttGenControl are programmed to same in ddrioint
  VttTargetV = (Outputs->DdrType == MRC_DDR_TYPE_DDR5) ? (Vdd2Mv / 2) : 150;
  PanicCCCPU = VssHiPanic.Bits.PanicCCCPU;
  DebugLevel = (DebugMsg) ? MSG_LEVEL_NOTE : MSG_LEVEL_NEVER;
//  GsmMode    = (DebugMsg) ? PrintValue : 0;

  // Calculate RodtCPU
  // @todo: in power training this might vary between different bytes so need to avg
  RodtCpu = Inputs->RcompTarget[RdOdt];
  // Calculate RonDimm Per Dimm (assuming)
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank += 2) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }
        TotalDimms++;
        TotalRon += CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmRon, TRUE, INIT_DIMM_RON_34, FALSE);
      } // Rank exist
    } // Channel
  } // Controller

  if (TotalDimms == 0) {
    TotalDimms++;
  }
  RonDimm = TotalRon / TotalDimms;

  MRC_DEBUG_MSG (Debug, DebugLevel, "MrcSetupVtt: RodtCpu=%u ohm, RonDimm=%u ohm\n", RodtCpu, RonDimm);
  MRC_DEBUG_MSG (Debug, DebugLevel, "Itarget=%u uA,\n", Itarget);

  CaVoltageSel    = (CompCtl0.Bits.CaVoltageSelect == 1);

  // Configure VTT Panic Comp Command Ron Vref
  CmdTarget = Inputs->RcompTarget[WrDSCmd];

  // IcompUp and Dn should equal to 10mA
  PanicV = VttGenControl.Bits.Panic0;
  IcompUp = ((VttTargetV - PanicV) / CmdTarget) / (1 + (PanicCCCPU >> 1));

  TempVar1 = CaVoltageSel ? Vdd2Mv : Outputs->VccddqVoltage;
  TempVar2 = TempVar1 - VttTargetV - PanicV;
  IcompDn = TempVar2 / CmdTarget;

  MRC_DEBUG_MSG (Debug, DebugLevel, "IcompDn=%u uA, IcompUp=%u uA\n", IcompDn, IcompUp);

  CompVttPanic.Data = 0;

 //LPDDR Vtt target may be adjusted based on panic count feedback and margin results
  TempVar1 = 191 * (VttTargetV - PanicV);
  //VccIoG is gated version of VccIo
  TempVar2 = DIVIDEROUND (TempVar1, VccIoMv);
  if (TempVar2 > 127) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "Clamping VttPanicUpVref: %d\n",TempVar2);
  }
  CompVttPanic.Bits.VttPanicUpVref = MAX (TempVar2, 127);
  TempVar1 = 191 * (VttTargetV + PanicV);
  TempVar2 = CaVoltageSel ? Vdd2Mv : Outputs->VccddqVoltage;
  CompVttPanic.Bits.VttPanicDnVref = DIVIDEROUND (TempVar1, TempVar2);
  TempVar1 = (VttTargetV - PanicV) / CaRonDrvDn;
  TempVar2 = 1 + (PanicCCCPU >> 1);
  IttcompUp = TempVar1 / TempVar2;
  TempVar3 = (2 * Itarget);
  TempVar4 = DIVIDEROUND (TempVar3, IttcompUp);
  CompVttPanic.Bits.VttPanicCompUp0Mult = TempVar4;
  CompVttPanic.Bits.VttPanicCompUp1Mult = TempVar4;
  TempVar1 = 2 * Itarget;
  TempVar2 = TempVar1 / IcompDn;
  TempVar3 = (1 + VssHiPanic.Bits.PanicPDn2xStep);
  TempVar4 = TempVar2 * TempVar3;
  if (TempVar4 > 15) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "Clamping VttPanicCompDn0Mult: %d\n",TempVar4);
  }
  TempVar4 = RANGE (TempVar4, 0, 15);
  CompVttPanic.Bits.VttPanicCompDn0Mult = TempVar4;
  CompVttPanic.Bits.VttPanicCompDn1Mult = TempVar4;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC_REG, CompVttPanic.Data);

  CompVttPanic2.Data = 0;
  CompVttPanic2.Bits.PanicCCCPU = PanicCCCPU;
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPVTTPANIC2_REG, CompVttPanic2.Data);
  GetSetVal = 0;
  MrcGetSetNoScope (MrcData, GsmIocVttPanicCompUpMult, WriteOffsetCached, &GetSetVal);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocVttPanicCompUpMult, WriteOffsetCached, &GetSetVal);

  CompCtl1.Data = MrcReadCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL1_REG);
  TempVar1 = (350 * 191) / VccIoMv; // Converting mV to ticks
  CompCtl1.Bits.En200ohmVttPncUp = (CompVttPanic.Bits.VttPanicUpVref > TempVar1);
  MrcWriteCR (MrcData, DDRPHY_COMP_CR_DDRCRCOMPCTL1_REG, CompCtl1.Data);

  ForceRcomp (MrcData);
  */
}

