/** @file
  This file contains internal PSF routines for PCH PSF CNL lib usage

@copyright
  INTEL CONFIDENTIAL
  Copyright 2016 - 2017 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include "MrcOemPlatform.h"
#include "PchLimits.h"
#include "PchRegsPsf.h"
#include "PchPcrLib.h"
#include "PchRegsPsfCnl.h"
#include "PchPsfPrivateLib.h"
#include "PchPsfPrivateLibInternal.h"
#include "PchSerialIoLib.h"


GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchLpSerialIoPsfRegs[PCH_SERIALIO_MAX_CONTROLLERS] =
{
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_I2C0_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_I2C1_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_I2C2_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_I2C3_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_I2C4_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_I2C5_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_SPI0_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_SPI1_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_SPI2_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_UART0_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_UART1_REG_BASE,
  R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_UART2_REG_BASE
};

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchHSerialIoPsfRegs[PCH_SERIALIO_MAX_CONTROLLERS] =
{
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_I2C0_REG_BASE,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_I2C1_REG_BASE,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_I2C2_REG_BASE,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_I2C3_REG_BASE,
  0,
  0,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_SPI0_REG_BASE,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_SPI1_REG_BASE,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_SPI2_REG_BASE,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_UART0_REG_BASE,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_UART1_REG_BASE,
  R_CNL_PCH_H_PSF3_PCR_T0_SHDW_UART2_REG_BASE
};

/**
  This procedure will return PSF_PORT for SerialIO device

  @param[in] SerialIoDevice  SERIAL IO device (I2C0-5, SPI0-1, UART0-2)

  @retval    PsfPort         PSF PORT structure for SerialIO device
**/
PSF_PORT
PsfSerialIoPort (
  IN PCH_SERIAL_IO_CONTROLLER  SerialIoDevice
  )
{
  PSF_PORT PsfPort;

  /*if (IsPchH () &&
      ((SerialIoDevice == PchSerialIoIndexI2C4) ||
      (SerialIoDevice == PchSerialIoIndexI2C5))) {
    ASSERT (FALSE);
  }*/

  PsfPort.PsfPid = CNL_PID_PSF3;

  if (MrcGetPchSeries () == PchSeriesLp) {
    if (SerialIoDevice < ARRAY_COUNT(mPchLpSerialIoPsfRegs)) {
      PsfPort.RegBase = mPchLpSerialIoPsfRegs[SerialIoDevice];
      return PsfPort;
    }
  } else {
    if (SerialIoDevice < ARRAY_COUNT(mPchHSerialIoPsfRegs)) {
      PsfPort.RegBase = mPchHSerialIoPsfRegs[SerialIoDevice];
      return PsfPort;
    }
  }

  //ASSERT(FALSE);
  return PsfPort;
}

/**
  This procedure will return PSF_PORT for TraceHub device

  @retval    PsfPort         PSF PORT structure for TraceHub device
**/
PSF_PORT
PsfTraceHubPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = CNL_PID_PSF2;

  if (MrcGetPchSeries () == PchSeriesLp) {
    PsfPort.RegBase = R_CNL_PCH_LP_PSF2_PCR_T0_SHDW_TRACE_HUB_REG_BASE;
  } else {
    PsfPort.RegBase = R_CNL_PCH_H_PSF2_PCR_T0_SHDW_TRACE_HUB_REG_BASE;
  }
  return PsfPort;
}

/**
  This procedure will return PSF_PORT for TraceHub ACPI device

  @retval    PsfPort         PSF PORT structure for TraceHub ACPI device
**/
PSF_PORT
PsfTraceHubAcpiDevPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = CNL_PID_PSF2;

  if (MrcGetPchSeries () == PchSeriesLp) {
    PsfPort.RegBase = R_CNL_PCH_LP_PSF2_PCR_T0_SHDW_TRACE_HUB_ACPI_REG_BASE;
  } else {
    PsfPort.RegBase = R_CNL_PCH_H_PSF2_PCR_T0_SHDW_TRACE_HUB_ACPI_REG_BASE;
  }
  return PsfPort;
}

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchLpHeciPsfRegBase[] =
{
  R_CNL_PCH_LP_PSF1_PCR_T0_SHDW_HECI1_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T0_SHDW_HECI2_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T0_SHDW_HECI3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T0_SHDW_HECI4_REG_BASE
};

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchHHeciPsfRegBase[] =
{
  R_CNL_PCH_H_PSF1_PCR_T0_SHDW_HECI1_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T0_SHDW_HECI2_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T0_SHDW_HECI3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T0_SHDW_HECI4_REG_BASE
};


/**
  This procedure will return PSF_PORT for SOL device

  @retval    PsfPort         PSF PORT structure for SOL device
**/
PSF_PORT
PsfSolPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = CNL_PID_PSF1;

  if (MrcGetPchSeries () == PchSeriesLp) {
    PsfPort.RegBase = R_CNL_PCH_LP_PSF1_PCR_T0_SHDW_KT_REG_BASE;
  } else {
    PsfPort.RegBase = R_CNL_PCH_H_PSF1_PCR_T0_SHDW_KT_REG_BASE;
  }
  return PsfPort;
}

/**
  Disable xHCI VTIO Phantom device at PSF level
**/
VOID
PsfDisableXhciVtioDevice (
  VOID
  )
{
/*
  PSF_PORT PsfPort;

  if (IsPchLp () && (PchStepping () < PCH_B0)) {
    //
    // xHCI VTIO Phantom device did not exist on CNL-PCH-LP-A0
    //
    return;
  }

  PsfPort.PsfPid = CNL_PID_PSF2;

  if (IsPchLp ()) {
    PsfPort.RegBase = R_CNL_PCH_LP_PSF2_PCR_T0_SHDW_XHCI_VTIO_REG_BASE;
  } else {
    PsfPort.RegBase = R_CNL_PCH_H_PSF2_PCR_T0_SHDW_XHCI_VTIO_REG_BASE;
  }

  PsfDisableDevice (PsfPort);
  */
}


/**
  Program PSF grant counts for SATA
  Call this before SATA ports are accessed for enumeration
**/
VOID
PsfConfigureSataGrantCounts (
  VOID
  )
{
/*
  //
  // PCH-LP:
  //   Set PSF_4_DEV_GNTCNT_RELOAD_DGCR2-5.GNTCNTRELOAD to 0x1
  //     VR    -> PSF_4_DEV_GNTCNT_RELOAD_DGCR2
  //     VS0-2 -> PSF_4_DEV_GNTCNT_RELOAD_DGCR3-5
  // PCH-H:
  //   Set PSF_4_DEV_GNTCNT_RELOAD_DGCR2-10.GNTCNTRELOAD to 0x1
  //     VR    -> PSF_4_DEV_GNTCNT_RELOAD_DGCR2
  //     VS0-7 -> PSF_4_DEV_GNTCNT_RELOAD_DGCR3-10
  //
  UINT16  Dgcr0Addr;
  UINT8   DgcrMinIndex;
  UINT8   DgcrMaxIndex;
  UINT8   DgcrIndex;

  if (IsPchLp ()) {
    Dgcr0Addr = R_CNL_PCH_LP_PSF4_PCR_DEV_GNTCNT_RELOAD_DGCR0;
    DgcrMinIndex = 2;
    DgcrMaxIndex = 5;
  } else {
    Dgcr0Addr = R_CNL_PCH_H_PSF4_PCR_DEV_GNTCNT_RELOAD_DGCR0;
    DgcrMinIndex = 2;
    DgcrMaxIndex = 10;
  }

  for (DgcrIndex = DgcrMinIndex; DgcrIndex <= DgcrMaxIndex; DgcrIndex++) {
    PchPcrAndThenOr32 (
      CNL_PID_PSF4,
      Dgcr0Addr  + (DgcrIndex * S_PCH_PSFX_PCR_DEV_GNTCNT_RELOAD_DGCR),
      (UINT32) ~B_PCH_PSFX_PCR_DEV_GNTCNT_RELOAD_DGCR_GNT_CNT_RELOAD,
      0x1
      );
  }
  */
}


/**
  Disable ISM NP Completion Tracking for GbE PSF port
**/
VOID
PsfDisableIsmNpCompletionTrackForGbe (
  VOID
  )
{
  return;
}

GLOBAL_REMOVE_IF_UNREFERENCED PSF_PORT mPchLpScsDevicePsfPort[PchPsfMaxScsDevNum] =
{
  {CNL_PID_PSF2, R_CNL_PCH_LP_PSF2_PCR_T0_SHDW_EMMC_REG_BASE},  // eMMC
  {       0,                                           0},  // SDIO (Not supported)
  {CNL_PID_PSF3, R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_SDCARD_REG_BASE},// SDCard
  {CNL_PID_PSF2, R_CNL_PCH_LP_PSF2_PCR_T0_SHDW_UFS_REG_BASE}    // UFS
};

/**
  This procedure will return PSF_PORT for SCS device

  @param[in] ScsDevNum       SCS Device

  @retval    PsfPort         PSF PORT structure for SCS device
**/
PSF_PORT
PsfScsPort (
  IN PCH_PSF_SCS_DEV_NUM  ScsDevNum
  )
{
  PSF_PORT PsfPort;
  PsfPort.PsfPid = CNL_PID_PSF1;

  return PsfPort;
}

//
// PSF1_T1_SHDW_PCIExx_REG_BASE registers for PCH-LP
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchLpPsfRootPortRs0RegBase[] =
{
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE01_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE02_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE03_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE04_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE05_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE06_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE07_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE08_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE09_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE10_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE11_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE12_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE13_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE14_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE15_RS0_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE16_RS0_REG_BASE
};

//
// PSF1_T1_SHDW_PCIExx_REG_BASE registers for PCH-LP
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchLpPsfRootPortRs3RegBase[] =
{
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE01_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE02_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE03_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE04_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE05_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE06_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE07_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE08_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE09_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE10_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE11_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE12_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE13_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE14_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE15_RS3_REG_BASE,
  R_CNL_PCH_LP_PSF1_PCR_T1_SHDW_PCIE16_RS3_REG_BASE
};

//
// PSF1_T1_SHDW_PCIExx_REG_BASE registers for PCH-H
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchHPsfRootPortRs0RegBase[] =
{
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE01_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE02_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE03_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE04_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE05_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE06_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE07_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE08_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE09_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE10_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE11_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE12_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE13_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE14_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE15_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE16_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE17_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE18_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE19_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE20_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE21_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE22_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE23_RS0_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE24_RS0_REG_BASE
};

//
// PSF1_T1_SHDW_PCIExx_REG_BASE registers for PCH-H
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchHPsfRootPortRs3RegBase[] =
{
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE01_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE02_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE03_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE04_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE05_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE06_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE07_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE08_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE09_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE10_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE11_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE12_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE13_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE14_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE15_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE16_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE17_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE18_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE19_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE20_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE21_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE22_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE23_RS3_REG_BASE,
  R_CNL_PCH_H_PSF1_PCR_T1_SHDW_PCIE24_RS3_REG_BASE
};

GLOBAL_REMOVE_IF_UNREFERENCED PSF_PORT mPchHSecLevelPsfRootPortRegBase[] =
{
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_SHDW_PCIE01_REG_BASE},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_SHDW_PCIE02_REG_BASE},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_SHDW_PCIE03_REG_BASE},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_SHDW_PCIE04_REG_BASE},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_SHDW_PCIE05_REG_BASE},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_SHDW_PCIE06_REG_BASE},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_SHDW_PCIE07_REG_BASE},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_SHDW_PCIE08_REG_BASE},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_SHDW_PCIE09_REG_BASE},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_SHDW_PCIE10_REG_BASE},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_SHDW_PCIE11_REG_BASE},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_SHDW_PCIE12_REG_BASE},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_SHDW_PCIE13_REG_BASE},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_SHDW_PCIE14_REG_BASE},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_SHDW_PCIE15_REG_BASE},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_SHDW_PCIE16_REG_BASE},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_SHDW_PCIE17_REG_BASE},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_SHDW_PCIE18_REG_BASE},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_SHDW_PCIE19_REG_BASE},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_SHDW_PCIE20_REG_BASE},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_SHDW_PCIE21_REG_BASE},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_SHDW_PCIE22_REG_BASE},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_SHDW_PCIE23_REG_BASE},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_SHDW_PCIE24_REG_BASE}
};


GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8 mDgcrLp [PCH_LP_PCIE_MAX_CONTROLLERS][PCH_PCIE_CONTROLLER_PORTS] = {
    {  8,  9, 10, 11 }, //SPA
    { 12, 13, 14, 15 }, //SPB
    { 16, 17, 18, 19 }, //SPC
    { 20, 21, 22, 23 }  //SPD
  };

GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8 mPgTgtLp [PCH_LP_PCIE_MAX_CONTROLLERS][PCH_PCIE_CONTROLLER_PORTS] = {
    {  0,  1,  2,  3 }, //SPA
    {  8,  9, 10, 11 }, //SPB
    { 16, 17, 18, 19 }, //SPC
    { 24, 25, 26, 27 }  //SPD
  };

GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8 mDgcrH [PCH_H_PCIE_MAX_CONTROLLERS][PCH_PCIE_CONTROLLER_PORTS] = {
    {  2,  3,  4,  5 }, //SPA
    {  2,  3,  4,  5 }, //SPB
    {  6,  7,  8,  9 }, //SPC
    {  2,  3,  4,  5 }, //SPD
    {  6,  7,  8,  9 }, //SPE
    {  6,  7,  8,  9 }  //SPF
  };

GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8 mPgTgtH [PCH_H_PCIE_MAX_CONTROLLERS][PCH_PCIE_CONTROLLER_PORTS] = {
    {  0,  1,  2,  3 }, //SPA
    {  0,  1,  2,  3 }, //SPB
    {  4,  5,  6,  7 }, //SPC
    {  0,  1,  2,  3 }, //SPD
    {  4,  5,  6,  7 }, //SPE
    {  4,  5,  6,  7 }  //SPF
  };

GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8 mPchHUpperLevelDgcr[] = {
    5, //PSF6
    6, //PSF7
    7  //PSF8
  };

GLOBAL_REMOVE_IF_UNREFERENCED CONST UINT8 mPchHUpperLevelPgTgt[] = {
    0, //PSF6
    1, //PSF7
    2  //PSF8
  };

//
// Structure for storing data on PCIe controller to PSF assignment and GrantCount register offsets
//
typedef struct {
  PCH_SBI_PID  PsfPid;
  UINT16       DevGntCnt0Base;
  UINT16       TargetGntCntPg1Tgt0Base;
} PSF_GRANT_COUNT_REGS;

GLOBAL_REMOVE_IF_UNREFERENCED CONST PSF_GRANT_COUNT_REGS mPchHGntCntRegs[PCH_H_PCIE_MAX_CONTROLLERS] = {
    {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_DEV_GNTCNT_RELOAD_DGCR0, R_CNL_PCH_H_PSF6_PCR_TARGET_GNTCNT_RELOAD_PG1_TGT0}, //SPA
    {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_DEV_GNTCNT_RELOAD_DGCR0, R_CNL_PCH_H_PSF8_PCR_TARGET_GNTCNT_RELOAD_PG1_TGT0}, //SPB
    {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_DEV_GNTCNT_RELOAD_DGCR0, R_CNL_PCH_H_PSF6_PCR_TARGET_GNTCNT_RELOAD_PG1_TGT0}, //SPC
    {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_DEV_GNTCNT_RELOAD_DGCR0, R_CNL_PCH_H_PSF7_PCR_TARGET_GNTCNT_RELOAD_PG1_TGT0}, //SPD
    {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_DEV_GNTCNT_RELOAD_DGCR0, R_CNL_PCH_H_PSF7_PCR_TARGET_GNTCNT_RELOAD_PG1_TGT0}, //SPE
    {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_DEV_GNTCNT_RELOAD_DGCR0, R_CNL_PCH_H_PSF8_PCR_TARGET_GNTCNT_RELOAD_PG1_TGT0}  //SPF
  };

/**
  Program PSF grant counts for PCI express depending on controllers configuration

  @param[in] PsfPcieCtrlConfigTable   Table with PCIe controllers configuration
  @param[in] NumberOfPcieControllers  Number of PCIe controllers. This is also the size of PsfPcieCtrlConfig table

  @retval Status
**/
EFI_STATUS
PsfConfigurePcieGrantCounts (
  IN PSF_PCIE_CTRL_CONFIG  *PsfPcieCtrlConfigTable,
  IN UINT32                NumberOfPcieControllers
  )
{
/*
  UINT32      Dgcr0Addr;
  UINT32      PgTgt0Addr;
  UINT32      DgcrNo0;
  UINT32      PgTgtNo;
  UINT32      Channel;
  UINT32      ChannelGrant[PCH_PCIE_CONTROLLER_PORTS];
  UINT8       Controller;
  PCH_SBI_PID PsfPid;

  DEBUG ((DEBUG_INFO, "PsfConfigurePcieGrantCounts() Start\n"));

  for (Controller = 0; Controller < NumberOfPcieControllers; ++Controller) {
    PsfPcieChannelGrantCounts (PsfPcieCtrlConfigTable[Controller], &ChannelGrant[0]);

    if (IsPchLp ()) {
      PsfPid     = CNL_PID_PSF1;
      Dgcr0Addr  = R_CNL_PCH_LP_PSF1_DEV_GNTCNT_RELOAD_DGCR0;
      PgTgt0Addr = R_CNL_PCH_LP_PSF1_TARGET_GNTCNT_RELOAD_PG1_TGT0;
    } else {
      PsfPid     = mPchHGntCntRegs[Controller].PsfPid;
      Dgcr0Addr  = mPchHGntCntRegs[Controller].DevGntCnt0Base;
      PgTgt0Addr = mPchHGntCntRegs[Controller].TargetGntCntPg1Tgt0Base;
    }

    for (Channel = 0; Channel < PCH_PCIE_CONTROLLER_PORTS; ++Channel) {
      if (ChannelGrant[Channel] == DEFAULT_PCIE_GRANT_COUNT) {
        //
        // If HW default move to next channel
        //
        continue;
      }

      if (IsPchLp ()) {
        DgcrNo0 = mDgcrLp[Controller][Channel];
        PgTgtNo = mPgTgtLp[Controller][Channel];
      } else {
        DgcrNo0 = mDgcrH[Controller][Channel];
        PgTgtNo = mPgTgtH[Controller][Channel];
      }

      DEBUG ((DEBUG_INFO, "SP%c[%d] - DGCR%d = %d\n", 65 + Controller, Channel, DgcrNo0, ChannelGrant[Channel]));
      PchPcrAndThenOr32 (
        PsfPid,
        (UINT16) (Dgcr0Addr  + (DgcrNo0 * S_PCH_PSFX_PCR_DEV_GNTCNT_RELOAD_DGCR)),
        (UINT32) ~B_PCH_PSFX_PCR_DEV_GNTCNT_RELOAD_DGCR_GNT_CNT_RELOAD,
        ChannelGrant[Channel]
        );

      DEBUG ((DEBUG_INFO, "SP%c[%d] - PG1_TGT%d = %d\n", 65 + Controller, Channel, PgTgtNo, ChannelGrant[Channel]));
      PchPcrAndThenOr32 (
        PsfPid,
        (UINT16) (PgTgt0Addr + (PgTgtNo * S_PCH_PSFX_PCR_TARGET_GNTCNT_RELOAD)),
        (UINT32) ~B_PCH_PSFX_PCR_TARGET_GNTCNT_RELOAD_GNT_CNT_RELOAD,
        ChannelGrant[Channel]
        );
    }
  }

  //
  // For PCH-H configure upper level PSF (PSF1) Grant Count
  //
  if (IsPchH ()) {
    PsfPchHConfigureUpLvlSegPcieGrantCounts ();
  }

  DEBUG ((DEBUG_INFO, "PsfConfigurePcieGrantCounts() End\n"));
  */
  return EFI_SUCCESS;
}

/**
  This function enables EOI message forwarding in PSF for PCIe ports
  for cases where IOAPIC is present behind this root port.

  @param[in] RpIndex        Root port index (0 based)

  @retval Status
**/
EFI_STATUS
PsfConfigurEoiForPciePort (
  IN  UINT32  RpIndex
  )
{
/*
  UINT32          Data32;
  UINT8           NumMcVal;
  UINT32          RpDestId;
  UINT32          SpxControllerIndex;
  PCH_SBI_PID     PsfPid;

  static UINT32   PchLpRpDestId[PCH_LP_PCIE_MAX_ROOT_PORTS] =
  {
    0x18000, 0x18001, 0x18002, 0x18003, // SPA: PSF1, PortID = 0
    0x18200, 0x18201, 0x18202, 0x18203, // SPB: PSF1, PortID = 2
    0x18400, 0x18401, 0x18402, 0x18403, // SPC: PSF1, PortID = 4
    0x18600, 0x18601, 0x18602, 0x18603  // SPD: PSF1, PortID = 6
  };

  static UINT32   PchHRpDestId[CNL_PCH_H_PCIE_MAX_ROOT_PORTS] =
  {
    0x68000, 0x68001, 0x68002, 0x68003, // SPA: PSF6, PortID = 0
    0x88000, 0x88001, 0x88002, 0x88003, // SPB: PSF8, PortID = 0
    0x68100, 0x68101, 0x68102, 0x68103, // SPC: PSF6, PortID = 1
    0x78000, 0x78001, 0x78002, 0x78003, // SPD: PSF7, PortID = 0
    0x78100, 0x78101, 0x78102, 0x78103, // SPE: PSF7, PortID = 1
    0x88100, 0x88101, 0x88102, 0x88103  // SPF: PSF8, PortID = 1
  };

  typedef struct {
    PCH_SBI_PID  PsfPid;
    UINT16       MulticastCtrlRegBase;
    UINT16       DestIdRegBase;
  } PSF_EOI_MSG_DATA;

  static PSF_EOI_MSG_DATA PchHEoiRegs[] = {
    {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_PSF_MC_CONTROL_MCAST0_EOI, R_CNL_PCH_H_PSF6_PCR_PSF_MC_AGENT_MCAST0_TGT0_EOI},  //SPA
    {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_PSF_MC_CONTROL_MCAST0_EOI, R_CNL_PCH_H_PSF8_PCR_PSF_MC_AGENT_MCAST0_TGT0_EOI},  //SPB
    {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_PSF_MC_CONTROL_MCAST0_EOI, R_CNL_PCH_H_PSF6_PCR_PSF_MC_AGENT_MCAST0_TGT0_EOI},  //SPC
    {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_PSF_MC_CONTROL_MCAST0_EOI, R_CNL_PCH_H_PSF7_PCR_PSF_MC_AGENT_MCAST0_TGT0_EOI},  //SPD
    {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_PSF_MC_CONTROL_MCAST0_EOI, R_CNL_PCH_H_PSF7_PCR_PSF_MC_AGENT_MCAST0_TGT0_EOI},  //SPE
    {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_PSF_MC_CONTROL_MCAST0_EOI, R_CNL_PCH_H_PSF8_PCR_PSF_MC_AGENT_MCAST0_TGT0_EOI},  //SPF
  };

  UINT16          PsfReg;

  ASSERT (RpIndex < GetPchMaxPciePortNum ());

  //
  // If there is an IOAPIC discovered behind root port program PSF Multicast registers
  // accordingly to CNL PCH BWG PSF EOI Multicast Configuration
  //
  // Read PSF_1_PSF_MC_CONTROL_MCAST0_RS0_EOI bit [7:1], increase it by 1, write back
  //
  if (IsPchLp ()) {
    PsfReg = R_CNL_PCH_LP_PSF1_PCR_PSF_MC_CONTROL_MCAST0_EOI;
  } else {
    PsfReg = R_CNL_PCH_H_PSF1_PCR_PSF_MC_CONTROL_MCAST0_EOI;
  }
  //
  // Since there is a device behind RootPort to which EOI needs to be forwarded
  // enable multicast (MULTCEN) and increase the number of multicast agents (NUMMC)
  // in Multicast Control Register. This configuration is done for the base PSF behind DMI - PSF1
  //
  // On CNL-LP PCIe controllers are connected directly to this PSF
  // so no additional PSF segment needs to be configured
  //
  Data32 = PchPcrRead32 (CNL_PID_PSF1, PsfReg);
  NumMcVal = (UINT8) (Data32 >> N_PCH_PSFX_PCR_MC_CONTROL_MCAST0_EOI_NUMMC);
  Data32 = ((NumMcVal + 1) << N_PCH_PSFX_PCR_MC_CONTROL_MCAST0_EOI_NUMMC) | B_PCH_PSFX_PCR_MC_CONTROL_MCAST0_EOI_MULTCEN;
  PchPcrWrite32 (CNL_PID_PSF1, PsfReg, Data32);

  if (IsPchLp ()) {
    PsfReg = R_CNL_PCH_LP_PSF1_PCR_PSF_MC_AGENT_MCAST0_TGT0_EOI;
    RpDestId = PchLpRpDestId[RpIndex];
  } else {
    PsfReg = R_CNL_PCH_H_PSF1_PCR_PSF_MC_AGENT_MCAST0_TGT0_EOI;
    RpDestId = PchHRpDestId[RpIndex];
  }

  //
  // Configure Multicast Destination ID register with target
  // device on PSF. Configuration must be done in next available
  // PSF_MC_AGENT_MCAST0_RS0_TGT<x>_EOI register so that other targets
  // are not overridden. <x> is known from the number of multicast agents
  // enabled before this programming. Value programmed is based on
  // PsfID, PortGroupID, PortID and ChannelID of the root port number
  //
  PchPcrAndThenOr32 (
    CNL_PID_PSF1, PsfReg + (NumMcVal * 4),
    (UINT32) ~0,
    RpDestId
    );

  if (IsPchH ()) {
    //
    // On CNL-PCH-H PCIe controllers are not placed on PSF1 but are connected
    // to additional PSF segments - PSF6/7/8
    // PSF6/7/8 need to be configured besides PSF1 for EOI forwarding to work
    //

    SpxControllerIndex = RpIndex / 4;
    PsfPid = PchHEoiRegs[SpxControllerIndex].PsfPid;
    PsfReg = PchHEoiRegs[SpxControllerIndex].MulticastCtrlRegBase;

    //
    // Since there is a device behind RootPort to which EOI needs to be forwarded
    // enable multicast (MULTCEN) and increase the number of multicast agents (NUMMC)
    // in Multicast Control Register. This configuration is done for PSF
    // segment to which this Pcie port belongs to.
    //
    Data32 = PchPcrRead32 (PsfPid, PsfReg);
    NumMcVal = (UINT8) (Data32 >> N_PCH_PSFX_PCR_MC_CONTROL_MCAST0_EOI_NUMMC);
    Data32 = ((NumMcVal + 1) << N_PCH_PSFX_PCR_MC_CONTROL_MCAST0_EOI_NUMMC) | B_PCH_PSFX_PCR_MC_CONTROL_MCAST0_EOI_MULTCEN;
    PchPcrWrite32 (PsfPid, PsfReg, Data32);

    PsfReg = PchHEoiRegs[SpxControllerIndex].DestIdRegBase;

    //
    // Configure Multicast Destination ID register with target
    // device on PSF. Configuration must be done in next available
    // PSF_MC_AGENT_MCAST0_RS0_TGT<x>_EOI register on this PSF so that other targets
    // are not overridden. <x> is known from the number of multicast agents
    // enabled before this programming. Value programmed is based on
    // PsfID, PortGroupID, PortID and ChannelID of the root port number
    //
    RpDestId = PchHRpDestId[RpIndex];
    PchPcrAndThenOr32 (
      PsfPid, PsfReg + (NumMcVal * 4),
      (UINT32) ~0,
      RpDestId
      );
  }
  */
  return EFI_SUCCESS;
}

//
// PSF1_T1_AGENT_FUNCTION_CONFIG_SPx_Dxx_Fx registers for PCH-LP
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchLpPsfRootPortRs0FuncCfgBase[] =
{
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F0,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F1,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F2,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F3,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F4,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F5,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F6,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F7,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F0,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F1,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F2,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F3,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F4,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F5,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F6,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F7
};
//
// PSF1_T1_AGENT_FUNCTION_CONFIG_SPx_Dxx_Fx registers for PCH-LP
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchLpPsfRootPortRs3FuncCfgBase[] =
{
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS3_D28_F0,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS3_D28_F1,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS3_D28_F2,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS3_D28_F3,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS3_D28_F4,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS3_D28_F5,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS3_D28_F6,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS3_D28_F7,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS3_D29_F0,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS3_D29_F1,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS3_D29_F2,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS3_D29_F3,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS3_D29_F4,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS3_D29_F5,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS3_D29_F6,
  R_CNL_PCH_LP_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS3_D29_F7
};

//
// PSF1_T1_AGENT_FUNCTION_CONFIG_SPx_Dxx_Fx registers for PCH-H
//
GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchHPsfRootPortRs0FuncCfgBase[] =
{
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F0,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F1,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F2,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS0_D28_F3,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F4,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F5,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F6,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS0_D28_F7,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F0,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F1,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F2,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS0_D29_F3,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F4,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F5,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F6,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS0_D29_F7,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS0_D27_F0,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS0_D27_F1,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS0_D27_F2,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS0_D27_F3,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS0_D27_F4,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS0_D27_F5,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS0_D27_F6,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS0_D27_F7
};

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchHPsfRootPortRs3FuncCfgBase[] =
{
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS3_D28_F0,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS3_D28_F1,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS3_D28_F2,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_RS3_D28_F3,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS3_D28_F4,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS3_D28_F5,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS3_D28_F6,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_RS3_D28_F7,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS3_D29_F0,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS3_D29_F1,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS3_D29_F2,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_RS3_D29_F3,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS3_D29_F4,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS3_D29_F5,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS3_D29_F6,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_RS3_D29_F7,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS3_D27_F0,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS3_D27_F1,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS3_D27_F2,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_RS3_D27_F3,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS3_D27_F4,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS3_D27_F5,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS3_D27_F6,
  R_CNL_PCH_H_PSF1_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_RS3_D27_F7
};

GLOBAL_REMOVE_IF_UNREFERENCED PSF_PORT mPchHSecLevelPsfRootPortFuncCfgBase[] =
{
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_D28_F0},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_D28_F1},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_D28_F2},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_AGENT_FUNCTION_CONFIG_SPA_D28_F3},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_D28_F4},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_D28_F5},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_D28_F6},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_AGENT_FUNCTION_CONFIG_SPB_D28_F7},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_D29_F0},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_D29_F1},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_D29_F2},
  {CNL_PID_PSF6, R_CNL_PCH_H_PSF6_PCR_T1_AGENT_FUNCTION_CONFIG_SPC_D29_F3},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_D29_F4},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_D29_F5},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_D29_F6},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPD_D29_F7},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_D27_F0},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_D27_F1},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_D27_F2},
  {CNL_PID_PSF7, R_CNL_PCH_H_PSF7_PCR_T1_AGENT_FUNCTION_CONFIG_SPE_D27_F3},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_D27_F4},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_D27_F5},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_D27_F6},
  {CNL_PID_PSF8, R_CNL_PCH_H_PSF8_PCR_T1_AGENT_FUNCTION_CONFIG_SPF_D27_F7}
};


/**
  Reload default RP PSF device number and function number.
  The PSF register doesn't got reset after system reset, which will result in mismatch between
  PSF register setting and PCIE PCR PCD register setting. Reset PSF register in early phase
  to avoid cycle decoding confusing.
**/
VOID
PsfReloadDefaultPcieRpDevFunc (
  VOID
  )
{
/*
  UINT32   PortIndex;
  UINT32   MaxPciePorts;
  UINT32   Data32;
  UINT32   RootPortDev;

  PortIndex = 0;
  MaxPciePorts = GetPchMaxPciePortNum ();

  //
  // Reload D28/D29/D27 registers
  //
  for (PortIndex = 0; PortIndex < MaxPciePorts; PortIndex++) {
    if (PortIndex < 8) {
      RootPortDev = PCI_DEVICE_NUMBER_PCH_PCIE_DEVICE_1;
    } else if (PortIndex < 16) {
      RootPortDev = PCI_DEVICE_NUMBER_PCH_PCIE_DEVICE_2;
    } else {
      RootPortDev = PCI_DEVICE_NUMBER_PCH_PCIE_DEVICE_3;
    }
    Data32 = ((RootPortDev << N_PCH_PSFX_PCR_TX_AGENT_FUNCTION_CONFIG_DEVICE) +
      ((PortIndex % 8) << N_PCH_PSFX_PCR_TX_AGENT_FUNCTION_CONFIG_FUNCTION));

    if (IsPchH ()) {
      //
      // For PCH-H PCIe Root Port have additional second level PSF segment (PSF6/7/8)
      // which requires same configuration as upper level PSF (PSF1)
      //
      PchPcrWrite32 (
        mPchHSecLevelPsfRootPortFuncCfgBase[PortIndex].PsfPid,
        mPchHSecLevelPsfRootPortFuncCfgBase[PortIndex].RegBase,
        Data32
        );
    }

    PchPcrWrite32 (
      CNL_PID_PSF1,
      PsfPcieRs0FunctionConfigRegBase (PortIndex),
      Data32
      );
    PchPcrWrite32 (
      CNL_PID_PSF1,
      PsfPcieRs3FunctionConfigRegBase (PortIndex),
      Data32
      );
  }
  */
}

/**
  This procedure will return PSF_PORT for CNVi device

  @retval    PsfPort         PSF PORT structure for CNVi device
**/
PSF_PORT
PsfCnviPort (
  VOID
  )
{

  PSF_PORT PsfPort;

  PsfPort.PsfPid = CNL_PID_PSF3;
/*
  if (IsPchLp ()) {
    PsfPort.RegBase = R_CNL_PCH_LP_PSF3_PCR_T0_SHDW_CNVI_REG_BASE;
  } else {
    PsfPort.RegBase = R_CNL_PCH_H_PSF3_PCR_T0_SHDW_CNVI_REG_BASE;
  }
  */
  return PsfPort;
}

/**
  This procedure will disable CNVi WiFi device at PSF level
**/
VOID
PsfDisableCnviWifiDevice (
  VOID
  )
{
  PsfDisableDevice (PsfCnviPort ());
}

/**
  Hide CNVi WiFi device PciCfgSpace at PSF level
**/
VOID
PsfHideCnviWifiDevice (
  VOID
  )
{
  PsfHideDevice (PsfCnviPort ());
}

/**
  This procedure will return PSF_PORT for PMC device

  @retval    PsfPort         PSF PORT structure for PMC device
**/
PSF_PORT
PsfPmcPort (
  VOID
  )
{
  PSF_PORT PsfPort;

  PsfPort.PsfPid = CNL_PID_PSF3;

  return PsfPort;
}

GLOBAL_REMOVE_IF_UNREFERENCED PCH_SBI_PID PsfLpPidTable[] = {CNL_PID_PSF1, CNL_PID_PSF2, CNL_PID_PSF3, CNL_PID_PSF4};
GLOBAL_REMOVE_IF_UNREFERENCED PCH_SBI_PID PsfHPidTable[]  = {CNL_PID_PSF1, CNL_PID_PSF2, CNL_PID_PSF3, CNL_PID_PSF4, CNL_PID_PSF6, CNL_PID_PSF7, CNL_PID_PSF8};

/**
  Enable VTd support in PSF.
**/
VOID
PchPsfEnableVtd (
  VOID
  )
{
/*
  UINTN              PsfPidIndex;
  UINTN              PsfPidTableSize;
  PCH_SBI_PID        *PsfPidTable;

  if (IsPchLp ()) {
    PsfPidTable = PsfLpPidTable;
    PsfPidTableSize = sizeof (PsfLpPidTable) / sizeof (PsfLpPidTable[0]);
  } else {
    PsfPidTable = PsfHPidTable;
    PsfPidTableSize = sizeof (PsfHPidTable) / sizeof (PsfHPidTable[0]);
  }

  for (PsfPidIndex = 0; PsfPidIndex < PsfPidTableSize; PsfPidIndex++) {
    PchPcrAndThenOr32 (
      PsfPidTable[PsfPidIndex],
      R_PCH_PSF_PCR_ROOTSPACE_CONFIG_RS0,
      ~0u,
      B_PCH_PSF_PCR_ROOTSPACE_CONFIG_RSX_VTDEN
      );
  }
  */
}

/**
  Disable PSF address-based peer-to-peer decoding.
**/
VOID
PchPsfDisableP2pDecoding (
  VOID
  )
{
/*
  UINTN              PsfPidIndex;
  UINTN              PsfPidTableSize;
  PCH_SBI_PID        *PsfPidTable;

  if (IsPchLp ()) {
    PsfPidTable = PsfLpPidTable;
    PsfPidTableSize = sizeof (PsfLpPidTable) / sizeof (PsfLpPidTable[0]);
  } else {
    PsfPidTable = PsfHPidTable;
    PsfPidTableSize = sizeof (PsfHPidTable) / sizeof (PsfHPidTable[0]);
  }

  for (PsfPidIndex = 0; PsfPidIndex < PsfPidTableSize; PsfPidIndex++) {
    PchPcrAndThenOr32 (
      PsfPidTable[PsfPidIndex],
      R_PCH_PSF_PCR_ROOTSPACE_CONFIG_RS0,
      (UINT32)~B_PCH_PSF_PCR_ROOTSPACE_CONFIG_RSX_ENADDRP2P,
      0
      );

    PchPcrAndThenOr32 (
      PsfPidTable[PsfPidIndex],
      R_PCH_PSF_PCR_ROOTSPACE_CONFIG_RS1,
      (UINT32)~B_PCH_PSF_PCR_ROOTSPACE_CONFIG_RSX_ENADDRP2P,
      0
      );
  }
  */
}

/**
  This procedure will disable D3:F0 device at PSF level for PCH-LP
**/
VOID
PsfDisableD3F0 (
  VOID
  )
{
  PSF_PORT  PsfPort;

  PsfPort.PsfPid = CNL_PID_PSF1;
  PsfPort.RegBase = R_CNL_PCH_LP_PSF1_PCR_T0_SHDW_NVM_D3_F0_BASE;

  PsfDisableDevice (PsfPort);
}

/**
  This procedure will disable PSF upstream completion tracking for HDAudio on PCH-LP
**/
VOID
PsfDisableUpstreamCompletionTrackingForHda (
  VOID
  )
{
  PchPcrAndThenOr32 (CNL_PID_PSF1, R_CNL_PCH_LP_PSF1_PCR_PSF_PORT_CONFIG_PG1_PORT7, (UINT32)~B_PCH_PSFX_PCR_PORT_CONFIG_PGX_PORTX_ISM_CPL_TRACK, 0);
  PchPcrAndThenOr32 (CNL_PID_PSF2, R_CNL_PCH_LP_PSF2_PCR_PSF_PORT_CONFIG_PG1_PORT4, (UINT32)~B_PCH_PSFX_PCR_PORT_CONFIG_PGX_PORTX_ISM_CPL_TRACK, 0);
  PchPcrAndThenOr32 (CNL_PID_PSF3, R_CNL_PCH_LP_PSF3_PCR_PSF_PORT_CONFIG_PG1_PORT0, (UINT32)~B_PCH_PSFX_PCR_PORT_CONFIG_PGX_PORTX_ISM_CPL_TRACK, 0);
}


