/** @file
  Register names for PCH LPC/eSPI device

  Conventions:

  - Prefixes:
    Definitions beginning with "R_" are registers
    Definitions beginning with "B_" are bits within registers
    Definitions beginning with "V_" are meaningful values within the bits
    Definitions beginning with "S_" are register sizes
    Definitions beginning with "N_" are the bit position
  - In general, PCH registers are denoted by "_PCH_" in register names
  - Registers / bits that are different between PCH generations are denoted by
    "_PCH_[generation_name]_" in register/bit names.
  - Registers / bits that are specific to PCH-H denoted by "_H_" in register/bit names.
    Registers / bits that are specific to PCH-LP denoted by "_LP_" in register/bit names.
    e.g., "_PCH_H_", "_PCH_LP_"
    Registers / bits names without _H_ or _LP_ apply for both H and LP.
  - Registers / bits that are different between SKUs are denoted by "_[SKU_name]"
    at the end of the register/bit names
  - Registers / bits of new devices introduced in a PCH generation will be just named
    as "_PCH_" without [generation_name] inserted.

@copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2017 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#ifndef _PCH_REGS_LPC_H_
#define _PCH_REGS_LPC_H_

#include "MrcOemPlatform.h"

//
// PCI to LPC Bridge Registers (D31:F0)
//
#define PCI_DEVICE_NUMBER_PCH_LPC       31
#define PCI_FUNCTION_NUMBER_PCH_LPC     0

typedef UINT8 PCH_STEPPING;
#define PCH_A0                0x00
#define PCH_A1                0x01
#define PCH_B0                0x10
#define PCH_B1                0x11
#define PCH_C0                0x20
#define PCH_C1                0x21
#define PCH_D0                0x30
#define PCH_D1                0x31
#define PCH_STEPPING_MAX      0xFF

#define SKL_PCH_H_MIN_SUPPORTED_STEPPING          PCH_B0
#define KBL_PCH_H_MIN_SUPPORTED_STEPPING          PCH_A0
#define CNL_PCH_H_MIN_SUPPORTED_STEPPING          PCH_A0

#define SPT_PCH_LP_MIN_SUPPORTED_STEPPING         PCH_B0
#define CNL_PCH_LP_MIN_SUPPORTED_STEPPING         PCH_A0

#define V_PCH_INTEL_VENDOR_ID   0x8086      ///< Default Intel PCH Vendor ID
#define V_LPC_CFG_VENDOR_ID                       V_PCH_INTEL_VENDOR_ID

#define B_LPC_CFG_DID         0xFFE0

//
//
// SKL PCH Server/WS LPC Device IDs
//
#define V_SKL_PCH_H_LPC_DEVICE_ID_SVR_0             0xA149          ///< Server SKU Intel C236 Chipset
#define V_SKL_PCH_H_LPC_DEVICE_ID_SVR_1             0xA14A          ///< Server SKU Intel C232 Chipset
#define V_SKL_PCH_H_LPC_DEVICE_ID_SVR_2             0xA150          ///< Server SKU Intel CM236 Chipset
#define V_SKL_PCH_H_LPC_DEVICE_ID_A14B              0xA14B          ///< Super SKU Unlocked

//
// SKL PCH-H Desktop LPC Device IDs
//
#define V_SKL_PCH_H_LPC_DEVICE_ID_DT_SUPER_SKU      0xA141          ///< PCH H Desktop Super SKU unlocked
#define V_SKL_PCH_H_LPC_DEVICE_ID_DT_0              0xA142          ///< PCH H Desktop Super SKU locked
#define V_SKL_PCH_H_LPC_DEVICE_ID_DT_1              0xA143          ///< PCH H Desktop H110
#define V_SKL_PCH_H_LPC_DEVICE_ID_DT_2              0xA144          ///< PCH H Desktop H170
#define V_SKL_PCH_H_LPC_DEVICE_ID_DT_3              0xA145          ///< PCH H Desktop Z170
#define V_SKL_PCH_H_LPC_DEVICE_ID_DT_4              0xA146          ///< PCH H Desktop Q170
#define V_SKL_PCH_H_LPC_DEVICE_ID_DT_5              0xA147          ///< PCH H Desktop Q150
#define V_SKL_PCH_H_LPC_DEVICE_ID_DT_6              0xA148          ///< PCH H Desktop B150
//
// SKL PCH-H Mobile LPC Device IDs
//
#define V_SKL_PCH_H_LPC_DEVICE_ID_MB_SUPER_SKU    0xA141          ///< PCH H Mobile Super SKU unlocked
#define V_SKL_PCH_H_LPC_DEVICE_ID_MB_0            0xA14D          ///< PCH H Mobile QM170
#define V_SKL_PCH_H_LPC_DEVICE_ID_MB_1            0xA14E          ///< PCH H Mobile HM170
#define V_SKL_PCH_H_LPC_DEVICE_ID_MB_2            0xA14F          ///< PCH H Mobile QMS170 (SFF)
#define V_SKL_PCH_H_LPC_DEVICE_ID_MB_3            0xA151          ///< PCH H Mobile QMS180 (SFF)
//
// SKL PCH-LP LPC Device IDs
//
#define V_PCH_LP_LPC_DEVICE_ID_MB_SUPER_SKU       0x9D41          ///< PCH LP Mobile Super SKU unlocked
#define V_PCH_LP_LPC_DEVICE_ID_MB_0               0x9D42          ///< PCH LP Mobile Super SKU locked
#define V_PCH_LP_LPC_DEVICE_ID_MB_1               0x9D43          ///< PCH LP Mobile (U) Base SKU
#define V_PCH_LP_LPC_DEVICE_ID_MB_2               0x9D46          ///< PCH LP Mobile (Y) Premium SKU
#define V_PCH_LP_LPC_DEVICE_ID_MB_3               0x9D48          ///< PCH LP Mobile (U) Premium SKU

//
// CNL PCH-LP Device IDs
//
#define V_CNL_PCH_LP_LPC_DEVICE_ID_MB_SUPER_SKU   0x9D80          ///< PCH LP Mobile Super SKU
#define V_CNL_PCH_LP_LPC_DEVICE_ID_MB_0           0x9D81          ///< PCH LP Mobile Super SKU unLocked
#define V_CNL_PCH_LP_LPC_DEVICE_ID_MB_1           0x9D82          ///< PCH LP Mobile Super SKU Locked
#define V_CNL_PCH_LP_LPC_DEVICE_ID_MB_2           0x9D83          ///< PCH LP Mobile (Y) Premium SKU
#define V_CNL_PCH_LP_LPC_DEVICE_ID_MB_3           0x9D84          ///< PCH LP Mobile (U) Premium SKU
#define V_CNL_PCH_LP_LPC_DEVICE_ID_MB_4           0x9D85          ///< PCH LP Mobile (U) Base SKU

//
// CNL PCH-H Desktop LPC Device IDs
//
#define V_CNL_PCH_H_LPC_DEVICE_ID_DT_SUPER_SKU    0xA280
#define V_CNL_PCH_H_LPC_DEVICE_ID_A300_SKU        0xA300
#define V_CNL_PCH_H_LPC_DEVICE_ID_A301_SKU        0xA301
#define V_CNL_PCH_H_LPC_DEVICE_ID_A302_SKU        0xA302
#define V_CNL_PCH_H_LPC_DEVICE_ID_A303_SKU        0xA303
#define V_CNL_PCH_H_LPC_DEVICE_ID_A304_SKU        0xA304
#define V_CNL_PCH_H_LPC_DEVICE_ID_A305_SKU        0xA305
#define V_CNL_PCH_H_LPC_DEVICE_ID_A306_SKU        0xA306
#define V_CNL_PCH_H_LPC_DEVICE_ID_A307_SKU        0xA307
#define V_CNL_PCH_H_LPC_DEVICE_ID_A308_SKU        0xA308
#define V_CNL_PCH_H_LPC_DEVICE_ID_A309_SKU        0xA309
#define V_CNL_PCH_H_LPC_DEVICE_ID_A30A_SKU        0xA30A
#define V_CNL_PCH_H_LPC_DEVICE_ID_A30B_SKU        0xA30B
#define V_CNL_PCH_H_LPC_DEVICE_ID_A30C_SKU        0xA30C          ///< PCH H Mobile HM370
#define V_CNL_PCH_H_LPC_DEVICE_ID_A30D_SKU        0xA30D          ///< PCH H Mobile QM370
#define V_CNL_PCH_H_LPC_DEVICE_ID_A30E_SKU        0xA30E          ///< PCH H Mobile CM246
#define V_CNL_PCH_H_LPC_DEVICE_ID_A30F_SKU        0xA30F
#define V_CNL_PCH_H_LPC_DEVICE_ID_A310_SKU        0xA310
#define V_CNL_PCH_H_LPC_DEVICE_ID_A311_SKU        0xA311
#define V_CNL_PCH_H_LPC_DEVICE_ID_A312_SKU        0xA312
#define V_CNL_PCH_H_LPC_DEVICE_ID_A313_SKU        0xA313
#define V_CNL_PCH_H_LPC_DEVICE_ID_A314_SKU        0xA314
#define V_CNL_PCH_H_LPC_DEVICE_ID_A315_SKU        0xA315
#define V_CNL_PCH_H_LPC_DEVICE_ID_A316_SKU        0xA316
#define V_CNL_PCH_H_LPC_DEVICE_ID_A317_SKU        0xA317
#define V_CNL_PCH_H_LPC_DEVICE_ID_A318_SKU        0xA318
#define V_CNL_PCH_H_LPC_DEVICE_ID_A319_SKU        0xA319
#define V_CNL_PCH_H_LPC_DEVICE_ID_A31A_SKU        0xA31A
#define V_CNL_PCH_H_LPC_DEVICE_ID_A31B_SKU        0xA31B
#define V_CNL_PCH_H_LPC_DEVICE_ID_A31C_SKU        0xA31C
#define V_CNL_PCH_H_LPC_DEVICE_ID_A31D_SKU        0xA31D
#define V_CNL_PCH_H_LPC_DEVICE_ID_A31E_SKU        0xA31E
#define V_CNL_PCH_H_LPC_DEVICE_ID_A31F_SKU        0xA31F


#define V_LPC_CFG_DID_CML_H                       0x0680
#define V_LPC_CFG_DID_CML_LP                      0x0280

//
// CML PCH-LP Device IDs
//
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_SUPER_SKU   0x0280          ///< PCH LP Mobile Super SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_0           0x0281          ///< PCH LP Mobile (U) Super SKU (SSKU)
//#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_1           0x0282          ///< Reserved
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_2           0x0283          ///< PCH LP Mobile (Y) Premium SKU (RST)
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_3           0x0284          ///< PCH LP Mobile (U) Premium SKU (RST)
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_4           0x0285          ///< PCH LP Mobile (U) Base/Mainstream SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_5           0x0286          ///< PCH LP Mobile (Y) Super SKU (SSKU)
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_6           0x0287          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_7           0x0288          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_8           0x0289          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_9           0x028A          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_10          0x028B          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_11          0x028C          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_12          0x028D          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_13          0x028E          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_14          0x028F          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_15          0x0290          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_16          0x0291          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_17          0x0292          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_18          0x0293          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_19          0x0294          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_20          0x0295          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_21          0x0296          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_22          0x0297          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_23          0x0298          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_24          0x0299          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_25          0x029A          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_26          0x029B          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_27          0x029C          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_28          0x029D          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_29          0x029E          ///< PCH LP Mobile SKU
#define V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_30          0x029F          ///< PCH LP Mobile SKU

//
// CML PCH-H LPC Device IDs
//
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0680_SKU        0x0680          ///< LPC/eSPI Controller
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0681_SKU        0x0681          ///< Client Super SKU (SSKU) unlocked
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0682_SKU        0x0682          ///< Client Super SKU (SSKU) locked
//#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0683_SKU        0x0683          ///< NA
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0684_SKU        0x0684          ///< PCH H Mobile H470
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0685_SKU        0x0685          ///< PCH H Mobile Z490
//#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0686_SKU        0x0686          ///< NA
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0687_SKU        0x0687          ///< PCH H Mobile Q490
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0688_SKU        0x0688          ///< PCH H Mobile B460
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0689_SKU        0x0689          ///< PCH H Mobile C256
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_068A_SKU        0x068A          ///< PCH H Mobile C252
//#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_068B_SKU        0x068B          ///< NA
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_068C_SKU        0x068C          ///< PCH H Mobile QM470
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_068D_SKU        0x068D          ///< PCH H Mobile HM470
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_068E_SKU        0x068E          ///< PCH H Mobile CM256
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_068F_SKU        0x068F          ///< PCH H Mobile Internal Super SKU (SSKU)
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0690_SKU        0x0690          ///< PCH H Mobile Workstation Server Super SKU (SSKU) Unlocked
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0691_SKU        0x0691          ///< PCH H Mobile Workstation Server Super SKU (SSKU) Locked
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0692_SKU        0x0692          ///< PCH H Mobile Small Form Factor (SFF) Super SKU (SSKU)
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0693_SKU        0x0693          ///< PCH H Mobile QMS480
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0694_SKU        0x0694          ///< PCH H Mobile Internal Production (PSKU)
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0695_SKU        0x0695          ///< PCH H Mobile Super SKU (SSKU) Unlocked
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0696_SKU        0x0696          ///< PCH H Mobile Super SKU (SSKU) Locked
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0697_SKU        0x0697          ///< PCH H Mobile SKU
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0698_SKU        0x0698          ///< PCH H Mobile SKU
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_0699_SKU        0x0699          ///< PCH H Mobile SKU
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_069A_SKU        0x069A          ///< PCH H Mobile SKU
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_069B_SKU        0x069B          ///< PCH H Mobile SKU
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_069C_SKU        0x069C          ///< PCH H Mobile SKU
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_069D_SKU        0x069D          ///< PCH H Mobile SKU
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_069E_SKU        0x069E          ///< PCH H Mobile SKU
#define V_CML_PCH_H_LPC_CFG_DEVICE_ID_069F_SKU        0x069F          ///< PCH H Mobile SKU


//
// CedarFork LPC Device ID's
//
#define V_CDF_PCH_LPC_DEVICE_ID_18DC                0x18DC          ///< CDF LPC Device ID

//
// KBL PCH-H Desktop LPC Device IDs
//
#define V_KBL_PCH_H_LPC_DEVICE_ID_DT_ES_SUPER_SKU   0xA2C0        ///< PCH H Desktop ES Super SKU unlocked. This is SKL-PCH-H in KBL-PCH-H package
#define V_KBL_PCH_H_LPC_DEVICE_ID_DT_SUPER_SKU      0xA2C1        ///< PCH H Desktop Super SKU unlocked
#define V_KBL_PCH_H_LPC_DEVICE_ID_DT_H270           0xA2C4        ///< PCH H Desktop H270
#define V_KBL_PCH_H_LPC_DEVICE_ID_DT_Z270           0xA2C5        ///< PCH H Desktop Z270
#define V_KBL_PCH_H_LPC_DEVICE_ID_DT_Q270           0xA2C6        ///< PCH H Desktop Q270
#define V_KBL_PCH_H_LPC_DEVICE_ID_DT_Q250           0xA2C7        ///< PCH H Desktop Q250
#define V_KBL_PCH_H_LPC_DEVICE_ID_DT_B250           0xA2C8        ///< PCH H Desktop B250

#define V_PCH_LPC_RID_0                           0x00
#define V_PCH_LPC_RID_1                           0x01
#define V_PCH_LPC_RID_9                           0x09
#define V_PCH_LPC_RID_10                          0x10
#define V_PCH_LPC_RID_11                          0x11
#define V_PCH_LPC_RID_20                          0x20
#define V_PCH_LPC_RID_21                          0x21
#define V_PCH_LPC_RID_30                          0x30
#define V_PCH_LPC_RID_31                          0x31
#define R_PCH_LPC_SERIRQ_CNT                      0x64
#define B_PCH_LPC_SERIRQ_CNT_SIRQEN               0x80
#define B_PCH_LPC_SERIRQ_CNT_SIRQMD               0x40
#define B_PCH_LPC_SERIRQ_CNT_SIRQSZ               0x3C
#define N_PCH_LPC_SERIRQ_CNT_SIRQSZ               2
#define B_PCH_LPC_SERIRQ_CNT_SFPW                 0x03
#define N_PCH_LPC_SERIRQ_CNT_SFPW                 0
#define V_PCH_LPC_SERIRQ_CNT_SFPW_4CLK            0x00
#define V_PCH_LPC_SERIRQ_CNT_SFPW_6CLK            0x01
#define V_PCH_LPC_SERIRQ_CNT_SFPW_8CLK            0x02
#define R_PCH_LPC_IOD                             0x80
#define B_PCH_LPC_IOD_FDD                         0x1000
#define N_PCH_LPC_IOD_FDD                         12
#define V_PCH_LPC_IOD_FDD_3F0                     0
#define V_PCH_LPC_IOD_FDD_370                     1
#define B_PCH_LPC_IOD_LPT                         0x0300
#define N_PCH_LPC_IOD_LPT                         8
#define V_PCH_LPC_IOD_LPT_378                     0
#define V_PCH_LPC_IOD_LPT_278                     1
#define V_PCH_LPC_IOD_LPT_3BC                     2
#define B_PCH_LPC_IOD_COMB                        0x0070
#define N_PCH_LPC_IOD_COMB                        4
#define V_PCH_LPC_IOD_COMB_3F8                    0
#define V_PCH_LPC_IOD_COMB_2F8                    1
#define V_PCH_LPC_IOD_COMB_220                    2
#define V_PCH_LPC_IOD_COMB_228                    3
#define V_PCH_LPC_IOD_COMB_238                    4
#define V_PCH_LPC_IOD_COMB_2E8                    5
#define V_PCH_LPC_IOD_COMB_338                    6
#define V_PCH_LPC_IOD_COMB_3E8                    7
#define B_PCH_LPC_IOD_COMA                        0x0007
#define N_PCH_LPC_IOD_COMA                        0
#define V_PCH_LPC_IOD_COMA_3F8                    0
#define V_PCH_LPC_IOD_COMA_2F8                    1
#define V_PCH_LPC_IOD_COMA_220                    2
#define V_PCH_LPC_IOD_COMA_228                    3
#define V_PCH_LPC_IOD_COMA_238                    4
#define V_PCH_LPC_IOD_COMA_2E8                    5
#define V_PCH_LPC_IOD_COMA_338                    6
#define V_PCH_LPC_IOD_COMA_3E8                    7
#define R_PCH_LPC_IOE                             0x82
#define B_PCH_LPC_IOE_ME2                         BIT13           ///< Microcontroller Enable #2, Enables decoding of I/O locations 4Eh and 4Fh to LPC
#define B_PCH_LPC_IOE_SE                          BIT12           ///< Super I/O Enable, Enables decoding of I/O locations 2Eh and 2Fh to LPC.
#define B_PCH_LPC_IOE_ME1                         BIT11           ///< Microcontroller Enable #1, Enables decoding of I/O locations 62h and 66h to LPC.
#define B_PCH_LPC_IOE_KE                          BIT10           ///< Keyboard Enable, Enables decoding of the keyboard I/O locations 60h and 64h to LPC.
#define B_PCH_LPC_IOE_HGE                         BIT9            ///< High Gameport Enable, Enables decoding of the I/O locations 208h to 20Fh to LPC.
#define B_PCH_LPC_IOE_LGE                         BIT8            ///< Low Gameport Enable, Enables decoding of the I/O locations 200h to 207h to LPC.
#define B_PCH_LPC_IOE_FDE                         BIT3            ///< Floppy Drive Enable, Enables decoding of the FDD range to LPC. Range is selected by LIOD.FDE
#define B_PCH_LPC_IOE_PPE                         BIT2            ///< Parallel Port Enable, Enables decoding of the LPT range to LPC. Range is selected by LIOD.LPT.
#define B_PCH_LPC_IOE_CBE                         BIT1            ///< Com Port B Enable, Enables decoding of the COMB range to LPC. Range is selected LIOD.CB.
#define B_PCH_LPC_IOE_CAE                         BIT0            ///< Com Port A Enable, Enables decoding of the COMA range to LPC. Range is selected LIOD.CA.
#define R_PCH_LPC_GEN1_DEC                        0x84
#define R_PCH_LPC_GEN2_DEC                        0x88
#define R_PCH_LPC_GEN3_DEC                        0x8C
#define R_PCH_LPC_GEN4_DEC                        0x90
#define B_PCH_LPC_GENX_DEC_IODRA                  0x00FC0000
#define B_PCH_LPC_GENX_DEC_IOBAR                  0x0000FFFC
#define B_PCH_LPC_GENX_DEC_EN                     0x00000001
#define R_PCH_LPC_ULKMC                           0x94
#define B_PCH_LPC_ULKMC_SMIBYENDPS                BIT15
#define B_PCH_LPC_ULKMC_TRAPBY64W                 BIT11
#define B_PCH_LPC_ULKMC_TRAPBY64R                 BIT10
#define B_PCH_LPC_ULKMC_TRAPBY60W                 BIT9
#define B_PCH_LPC_ULKMC_TRAPBY60R                 BIT8
#define B_PCH_LPC_ULKMC_SMIATENDPS                BIT7
#define B_PCH_LPC_ULKMC_PSTATE                    BIT6
#define B_PCH_LPC_ULKMC_A20PASSEN                 BIT5
#define B_PCH_LPC_ULKMC_USBSMIEN                  BIT4
#define B_PCH_LPC_ULKMC_64WEN                     BIT3
#define B_PCH_LPC_ULKMC_64REN                     BIT2
#define B_PCH_LPC_ULKMC_60WEN                     BIT1
#define B_PCH_LPC_ULKMC_60REN                     BIT0
#define R_PCH_LPC_LGMR                            0x98
#define R_PCH_ESPI_CS1GIR1                        0xA4
#define R_PCH_ESPI_CS1GMR1                        0xA8
#define B_PCH_LPC_LGMR_MA                         0xFFFF0000
#define B_PCH_LPC_LGMR_LMRD_EN                    BIT0

#define R_PCH_LPC_FWH_BIOS_SEL                    0xD0
#define B_PCH_LPC_FWH_BIOS_SEL_F8                 0xF0000000
#define B_PCH_LPC_FWH_BIOS_SEL_F0                 0x0F000000
#define B_PCH_LPC_FWH_BIOS_SEL_E8                 0x00F00000
#define B_PCH_LPC_FWH_BIOS_SEL_E0                 0x000F0000
#define B_PCH_LPC_FWH_BIOS_SEL_D8                 0x0000F000
#define B_PCH_LPC_FWH_BIOS_SEL_D0                 0x00000F00
#define B_PCH_LPC_FWH_BIOS_SEL_C8                 0x000000F0
#define B_PCH_LPC_FWH_BIOS_SEL_C0                 0x0000000F
#define R_PCH_LPC_FWH_BIOS_SEL2                   0xD4
#define B_PCH_LPC_FWH_BIOS_SEL2_70                0xF000
#define B_PCH_LPC_FWH_BIOS_SEL2_60                0x0F00
#define B_PCH_LPC_FWH_BIOS_SEL2_50                0x00F0
#define B_PCH_LPC_FWH_BIOS_SEL2_40                0x000F
#define R_PCH_LPC_BDE                             0xD8                          ///< BIOS decode enable
#define B_PCH_LPC_BDE_F8                          0x8000
#define B_PCH_LPC_BDE_F0                          0x4000
#define B_PCH_LPC_BDE_E8                          0x2000
#define B_PCH_LPC_BDE_E0                          0x1000
#define B_PCH_LPC_BDE_D8                          0x0800
#define B_PCH_LPC_BDE_D0                          0x0400
#define B_PCH_LPC_BDE_C8                          0x0200
#define B_PCH_LPC_BDE_C0                          0x0100
#define B_PCH_LPC_BDE_LEG_F                       0x0080
#define B_PCH_LPC_BDE_LEG_E                       0x0040
#define B_PCH_LPC_BDE_70                          0x0008
#define B_PCH_LPC_BDE_60                          0x0004
#define B_PCH_LPC_BDE_50                          0x0002
#define B_PCH_LPC_BDE_40                          0x0001
#define R_PCH_LPC_PCC                             0xE0
#define B_PCH_LPC_PCC_CLKRUN_EN                   0x0001
#define B_PCH_LPC_FVEC0_USB_PORT_CAP              0x00000C00
#define V_PCH_LPC_FVEC0_USB_14_PORT               0x00000000
#define V_PCH_LPC_FVEC0_USB_12_PORT               0x00000400
#define V_PCH_LPC_FVEC0_USB_10_PORT               0x00000800
#define B_PCH_LPC_FVEC0_SATA_RAID_CAP             0x00000080
#define B_PCH_LPC_FVEC0_SATA_PORT23_CAP           0x00000040
#define B_PCH_LPC_FVEC0_SATA_PORT1_6GB_CAP        0x00000008
#define B_PCH_LPC_FVEC0_SATA_PORT0_6GB_CAP        0x00000004
#define B_PCH_LPC_FVEC0_PCI_CAP                   0x00000002
#define R_PCH_LPC_FVEC1                           0x01
#define B_PCH_LPC_FVEC1_USB_R_CAP                 0x00400000
#define R_PCH_LPC_FVEC2                           0x02
#define V_PCH_LPC_FVEC2_PCIE_PORT78_CAP           0x00200000
#define V_PCH_LPC_FVEC2_PCH_IG_SUPPORT_CAP        0x00020000      ///< PCH Integrated Graphics Support Capability
#define R_PCH_LPC_FVEC3                           0x03
#define B_PCH_LPC_FVEC3_DCMI_CAP                  0x00002000      ///< Data Center Manageability Interface (DCMI) Capability
#define B_PCH_LPC_FVEC3_NM_CAP                    0x00001000      ///< Node Manager Capability

#define R_PCH_LPC_MDAP                            0xC0
#define B_PCH_LPC_MDAP_POLICY_EN                  BIT31
#define B_PCH_LPC_MDAP_PDMA_EN                    BIT30
#define B_PCH_LPC_MDAP_VALUE                      0x0001FFFF

//
// APM Registers
//
#define R_PCH_APM_CNT                             0xB2
#define R_PCH_APM_STS                             0xB3

#define R_PCH_LPC_BC                              0xDC            ///< Bios Control
#define S_PCH_LPC_BC                              1
#define B_PCH_LPC_BC_BILD                         BIT7            ///< BIOS Interface Lock-Down
#define B_PCH_LPC_BC_BBS                          BIT6            ///< Boot BIOS strap
#define N_PCH_LPC_BC_BBS                          6
#define V_PCH_LPC_BC_BBS_SPI                      0               ///< Boot BIOS strapped to SPI
#define V_PCH_LPC_BC_BBS_LPC                      1               ///< Boot BIOS strapped to LPC
#define B_PCH_LPC_BC_EISS                         BIT5            ///< Enable InSMM.STS
#define B_PCH_LPC_BC_TS                           BIT4            ///< Top Swap
#define B_PCH_LPC_BC_LE                           BIT1            ///< Lock Enable
#define N_PCH_LPC_BC_LE                           1
#define B_PCH_LPC_BC_WPD                          BIT0            ///< Write Protect Disable

#define R_PCH_ESPI_PCBC                           0xDC            ///< Peripheral Channel BIOS Control
#define S_PCH_ESPI_PCBC                           4               ///< Peripheral Channel BIOS Control register size
#define B_PCH_ESPI_PCBC_BWRE                      BIT11           ///< BIOS Write Report Enable
#define N_PCH_ESPI_PCBC_BWRE                      11              ///< BIOS Write Report Enable bit position
#define B_PCH_ESPI_PCBC_BWRS                      BIT10           ///< BIOS Write Report Status
#define N_PCH_ESPI_PCBC_BWRS                      10              ///< BIOS Write Report Status bit position
#define B_PCH_ESPI_PCBC_BWPDS                     BIT8            ///< BIOS Write Protect Disable Status
#define N_PCH_ESPI_PCBC_BWPDS                     8               ///< BIOS Write Protect Disable Status bit position
#define B_PCH_ESPI_PCBC_ESPI_EN                   BIT2            ///< eSPI Enable Pin Strap
#define B_PCH_ESPI_PCBC_LE                        BIT1            ///< Lock Enable

//
// eSPI slave registers
//
#define R_ESPI_SLAVE_CHA_0_CAP_AND_CONF           0x10            ///< Channel 0 Capabilities and Configurations
#define B_ESPI_SLAVE_BME                          BIT2            ///< Bus Master Enable

//
// Processor interface registers
//
#define R_PCH_NMI_SC                              0x61
#define B_PCH_NMI_SC_SERR_NMI_STS                 BIT7
#define B_PCH_NMI_SC_IOCHK_NMI_STS                BIT6
#define B_PCH_NMI_SC_TMR2_OUT_STS                 BIT5
#define B_PCH_NMI_SC_REF_TOGGLE                   BIT4
#define B_PCH_NMI_SC_IOCHK_NMI_EN                 BIT3
#define B_PCH_NMI_SC_PCI_SERR_EN                  BIT2
#define B_PCH_NMI_SC_SPKR_DAT_EN                  BIT1
#define B_PCH_NMI_SC_TIM_CNT2_EN                  BIT0
#define R_PCH_NMI_EN                              0x70
#define B_PCH_NMI_EN_NMI_EN                       BIT7

//
// Reset Generator I/O Port
//
#define R_PCH_RST_CNT                             0xCF9
#define B_PCH_RST_CNT_FULL_RST                    BIT3
#define B_PCH_RST_CNT_RST_CPU                     BIT2
#define B_PCH_RST_CNT_SYS_RST                     BIT1
#define V_PCH_RST_CNT_FULLRESET                   0x0E
#define V_PCH_RST_CNT_HARDRESET                   0x06
#define V_PCH_RST_CNT_SOFTRESET                   0x04
#define V_PCH_RST_CNT_HARDSTARTSTATE              0x02
#define V_PCH_RST_CNT_SOFTSTARTSTATE              0x00

//
// RTC register
//
#define R_PCH_RTC_INDEX                           0x70
#define R_PCH_RTC_TARGET                          0x71
#define R_PCH_RTC_EXT_INDEX                       0x72
#define R_PCH_RTC_EXT_TARGET                      0x73
#define R_PCH_RTC_INDEX_ALT                       0x74
#define R_PCH_RTC_TARGET_ALT                      0x75
#define R_PCH_RTC_EXT_INDEX_ALT                   0x76
#define R_PCH_RTC_EXT_TARGET_ALT                  0x77
#define R_PCH_RTC_REGA                            0x0A
#define B_PCH_RTC_REGA_UIP                        0x80
#define R_PCH_RTC_REGB                            0x0B
#define B_PCH_RTC_REGB_SET                        0x80
#define B_PCH_RTC_REGB_PIE                        0x40
#define B_PCH_RTC_REGB_AIE                        0x20
#define B_PCH_RTC_REGB_UIE                        0x10
#define B_PCH_RTC_REGB_DM                         0x04
#define B_PCH_RTC_REGB_HOURFORM                   0x02
#define R_PCH_RTC_REGC                            0x0C
#define R_PCH_RTC_REGD                            0x0D

//
// Private Configuration Register
// RTC PCRs (PID:RTC)
//
#define R_PCH_PCR_RTC_CONF                        0x3400               ///< RTC Configuration register
#define B_PCH_PCR_RTC_CONF_BILD                   BIT31                ///< BIOS Interface Lock-Down
#define B_PCH_PCR_RTC_CONF_HPM_HW_DIS             BIT6                 ///< RTC High Power Mode HW Disable
#define B_PCH_PCR_RTC_CONF_UCMOS_LOCK             BIT4                 ///< Upper 128 Byte Lock
#define B_PCH_PCR_RTC_CONF_LCMOS_LOCK             BIT3                 ///< Lower 128 Byte Lock
#define B_PCH_PCR_RTC_CONF_UCMOS_EN               BIT2                 ///< Upper CMOS bank enable
#define R_PCH_PCR_RTC_BUC                         0x3414               ///< Backed Up Control
#define B_PCH_PCR_RTC_BUC_DSO                     BIT4                 ///< Daylight Savings Override
#define B_PCH_PCR_RTC_BUC_TS                      BIT0                 ///< Top Swap
#define R_PCH_PCR_RTC_RTCDCG                      0x3418               ///< RTC Dynamic Clock Gating Control
#define R_PCH_PCR_RTC_RTCDCG_RTCPGCBDCGEN         BIT2                 ///< pgcb_clk (12Mhz) Dynamic Clock Gate Enable
#define R_PCH_PCR_RTC_RTCDCG_RTCPCICLKDCGEN       BIT1                 ///< ipciclk_clk (24 MHz) Dynamic Clock Gate Enable
#define R_PCH_PCR_RTC_RTCDCG_RTCROSIDEDCGEN       BIT0                 ///< rosc_side_clk (120 MHz) Dynamic Clock Gate Enable
#define R_PCH_PCR_RTC_3F00                        0x3F00
#define R_PCH_PCR_RTC_UIPSMI                      0x3F04               ///< RTC Update In Progress SMI Control

//
// LPC PCR Registers
//
#define R_PCH_PCR_LPC_HVMTCTL                     0x3410
#define R_PCH_PCR_LPC_GCFD                        0x3418
#define R_PCH_PCR_LPC_PRC                         0x341C
#define R_PCH_PCR_LPC_PCT                         0x3420
#define R_PCH_PCR_LPC_SCT                         0x3424
#define R_PCH_PCR_LPC_LPCCT                       0x3428
#define R_PCH_PCR_LPC_ULTOR                       0x3500

//
// eSPI PCR Registers
//
#define R_PCH_PCR_ESPI_SLV_CFG_REG_CTL            0x4000                  ///< Slave Configuration Register and Link Control
#define B_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRE       BIT31                   ///< Slave Configuration Register Access Enable
#define B_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRS       (BIT30 | BIT29 | BIT28) ///< Slave Configuration Register Access Status
#define N_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRS       28                      ///< Slave Configuration Register Access Status bit position
#define B_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SBLCL      BIT27                   ///< IOSF-SB eSPI Link Configuration Lock
#define V_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRS_NOERR 7                       ///< No errors (transaction completed successfully)
#define B_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SID        (BIT20 | BIT19)         ///< Slave ID
#define N_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SID        19                      ///< Slave ID bit position
#define B_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRT       (BIT17 | BIT16)         ///< Slave Configuration Register Access Type
#define N_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRT       16                      ///< Slave Configuration Register Access Type bit position
#define V_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRT_RD    0                       ///< Slave Configuration register read from address SCRA[11:0]
#define V_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRT_WR    1                       ///< Slave Configuration register write to address SCRA[11:0]
#define V_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRT_STS   2                       ///< Slave Status register read
#define V_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRT_RS    3                       ///< In-Band reset
#define B_PCH_PCR_ESPI_SLV_CFG_REG_CTL_SCRA       0x00000FFF              ///< Slave Configuration Register Address
#define R_PCH_PCR_ESPI_SLV_CFG_REG_DATA           0x4004                  ///< Slave Configuration Register Data

#define R_PCH_PCR_ESPI_PCERR_SLV0                 0x4020          ///< Peripheral Channel Error for Slave 0
#define R_PCH_PCR_ESPI_PCERR_SLV1                 0x4024          ///< Peripheral Channel Error for Slave 1
#define R_PCH_PCR_ESPI_VWERR_SLV0                 0x4030          ///< Virtual Wire Channel Error for Slave 0
#define R_PCH_PCR_ESPI_VWERR_SLV1                 0x4034          ///< Virtual Wire Channel Error for Slave 1
#define R_PCH_PCR_ESPI_FCERR_SLV0                 0x4040          ///< Flash Access Channel Error for Slave 0
#define B_PCH_PCR_ESPI_XERR_XNFEE                 (BIT14 | BIT13) ///< Non-Fatal Error Reporting Enable bits
#define N_PCH_PCR_ESPI_XERR_XNFEE                 13              ///< Non-Fatal Error Reporting Enable bit position
#define V_PCH_PCR_ESPI_XERR_XNFEE_SMI             3               ///< Enable Non-Fatal Error Reporting as SMI
#define B_PCH_PCR_ESPI_XERR_XNFES                 BIT12           ///< Fatal Error Status
#define B_PCH_PCR_ESPI_XERR_XFEE                  (BIT6 | BIT5)   ///< Fatal Error Reporting Enable bits
#define N_PCH_PCR_ESPI_XERR_XFEE                  5               ///< Fatal Error Reporting Enable bit position
#define V_PCH_PCR_ESPI_XERR_XFEE_SMI              3               ///< Enable Fatal Error Reporting as SMI
#define B_PCH_PCR_ESPI_XERR_XFES                  BIT4            ///< Fatal Error Status
#define B_PCH_PCR_ESPI_PCERR_SLV0_PCURD           BIT24           ///< Peripheral Channel Unsupported Request Detected
#define R_PCH_PCR_ESPI_LNKERR_SLV0                0x4050          ///< Link Error for Slave 0
#define S_PCH_PCR_ESPI_LNKERR_SLV0                4               ///< Link Error for Slave 0 register size
#define B_PCH_PCR_ESPI_LNKERR_SLV0_SLCRR          BIT31           ///< eSPI Link and Slave Channel Recovery Required
#define B_PCH_PCR_ESPI_LNKERR_SLV0_LFET1E         (BIT22 | BIT21) ///< Fatal Error Type 1 Reporting Enable
#define N_PCH_PCR_ESPI_LNKERR_SLV0_LFET1E         21              ///< Fatal Error Type 1 Reporting Enable bit position
#define V_PCH_PCR_ESPI_LNKERR_SLV0_LFET1E_SMI     3               ///< Enable Fatal Error Type 1 Reporting as SMI
#define B_PCH_PCR_ESPI_LNKERR_SLV0_LFET1S         BIT20           ///< Link Fatal Error Type 1 Status


//
//  LPC Device ID macros
//
//
// Device IDs that are SKL PCH-H Desktop specific
//
#define IS_SKL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) \
    (  \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_DT_0) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_DT_1) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_DT_2) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_DT_3) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_DT_4) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_DT_5) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_DT_6) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_DT_SUPER_SKU) || \
      (DeviceId == V_KBL_PCH_H_LPC_DEVICE_ID_DT_ES_SUPER_SKU /* SKL-PCH-H in KBL-PCH-H package */) \
    )

//
// Device IDs that are KBL PCH-H Desktop specific
//
#define IS_KBL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) \
    (  \
      (DeviceId == V_KBL_PCH_H_LPC_DEVICE_ID_DT_SUPER_SKU) || \
      (DeviceId == V_KBL_PCH_H_LPC_DEVICE_ID_DT_H270) || \
      (DeviceId == V_KBL_PCH_H_LPC_DEVICE_ID_DT_Z270) || \
      (DeviceId == V_KBL_PCH_H_LPC_DEVICE_ID_DT_Q270) || \
      (DeviceId == V_KBL_PCH_H_LPC_DEVICE_ID_DT_Q250) || \
      (DeviceId == V_KBL_PCH_H_LPC_DEVICE_ID_DT_B250) \
    )

//
// Device IDs that are CNL PCH-H Desktop specific
//
#define IS_CNL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) \
    (  \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_DT_SUPER_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A300_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A301_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A302_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A303_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A304_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A305_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A306_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A307_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A308_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A309_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30A_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30B_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30C_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30D_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30E_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30F_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A310_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A311_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A312_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A313_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A314_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A315_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A316_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A317_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A318_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A319_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31A_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31B_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31C_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31D_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31E_SKU) || \
      (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31F_SKU)    \
    )

//
// Device IDs that are PCH-H Desktop specific
//
#define IS_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) \
    (  \
      IS_SKL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) || \
      IS_KBL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) || \
      IS_CNL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) || \
      IS_CML_PCH_H_LPC_DEVICE_ID(DeviceId) \
    )

#define IS_PCH_LPC_DEVICE_ID_DESKTOP(DeviceId) \
    ( \
      IS_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) \
    )

//
// Device IDs that are PCH-H Mobile specific
//

#define IS_PCH_H_LPC_DEVICE_ID_MOBILE(DeviceId) \
    ( \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_MB_0) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_MB_1) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_MB_2) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_MB_3) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_MB_SUPER_SKU) \
    )

//
// Device IDs that are PCH-LP Mobile specific
//
#define IS_SKL_PCH_LP_LPC_DEVICE_ID_MOBILE(DeviceId) \
    ( \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_SUPER_SKU) || \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_0) || \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_1) || \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_2) || \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_3) \
    )

#define IS_CNL_PCH_LP_LPC_DEVICE_ID_MOBILE(DeviceId) \
    ( \
      (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_SUPER_SKU) || \
      (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_0)         || \
      (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_1)         || \
      (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_2)         || \
      (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_3)         || \
      (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_4)            \
    )

#define IS_PCH_LPC_DEVICE_ID_MOBILE(DeviceId) \
    ( \
      IS_PCH_H_LPC_DEVICE_ID_MOBILE(DeviceId) || \
      IS_PCH_LP_LPC_DEVICE_ID_MOBILE(DeviceId) \
    )

//
// Device IDS that are SKL PCH Server\Workstation specific
//
#define IS_SKL_PCH_H_LPC_DEVICE_ID_SERVER(DeviceId) \
    ( \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_SVR_0) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_SVR_1) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_SVR_2) || \
      (DeviceId == V_SKL_PCH_H_LPC_DEVICE_ID_A14B) \
    )


#define IS_PCH_H_LPC_DEVICE_ID_SERVER(DeviceId) \
    ( \
      IS_SKL_PCH_H_LPC_DEVICE_ID_SERVER(DeviceId) \
    )

#define IS_PCH_LPC_DEVICE_ID_SERVER(DeviceId) \
    ( \
      IS_PCH_H_LPC_DEVICE_ID_SERVER(DeviceId) \
    )

#define IS_PCH_H_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) || \
      IS_PCH_H_LPC_DEVICE_ID_MOBILE(DeviceId) || \
      IS_PCH_H_LPC_DEVICE_ID_SERVER(DeviceId) \
    )

#define IS_SKL_PCH_H_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_SKL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) || \
      IS_PCH_H_LPC_DEVICE_ID_MOBILE(DeviceId) || \
      IS_SKL_PCH_H_LPC_DEVICE_ID_SERVER(DeviceId) \
    )

#define IS_KBL_PCH_H_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_KBL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) \
    )

#define IS_PCH_LP_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_PCH_LP_LPC_DEVICE_ID_MOBILE(DeviceId) \
    )

#define IS_PCH_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_PCH_H_LPC_DEVICE_ID(DeviceId) || \
      IS_PCH_LP_LPC_DEVICE_ID(DeviceId) \
    )

#define IS_SKL_PCH_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_SKL_PCH_H_LPC_DEVICE_ID_DESKTOP(DeviceId) || \
      IS_PCH_LPC_DEVICE_ID_MOBILE(DeviceId) || \
      IS_SKL_PCH_H_LPC_DEVICE_ID_SERVER(DeviceId) \
    )

#define IS_KBL_PCH_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_KBL_PCH_H_LPC_DEVICE_ID(DeviceId) \
    )

#define IS_SKL_PCH_LP_LPC_DEVICE_ID(DeviceId) \
    ( \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_SUPER_SKU) || \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_0) || \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_1) || \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_2) || \
      (DeviceId == V_PCH_LP_LPC_DEVICE_ID_MB_3) \
    )

#define IS_CNL_PCH_LP_LPC_DEVICE_ID(DeviceId) \
      ( \
        (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_SUPER_SKU) || \
        (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_0)         || \
        (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_1)         || \
        (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_2)         || \
        (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_3)         || \
        (DeviceId == V_CNL_PCH_LP_LPC_DEVICE_ID_MB_4)            \
      )

#define IS_CNL_PCH_H_LPC_DEVICE_ID(DeviceId) \
      ( \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_DT_SUPER_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A300_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A301_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A302_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A303_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A304_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A305_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A306_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A307_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A308_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A309_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30A_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30B_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30C_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30D_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30E_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A30F_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A310_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A311_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A312_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A313_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A314_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A315_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A316_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A317_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A318_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A319_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31A_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31B_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31C_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31D_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31E_SKU) || \
        (DeviceId == V_CNL_PCH_H_LPC_DEVICE_ID_A31F_SKU)    \
      )

#define IS_CNL_PCH_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_CNL_PCH_LP_LPC_DEVICE_ID(DeviceId) || \
      IS_CNL_PCH_H_LPC_DEVICE_ID(DeviceId) \
    )

#define IS_CDF_PCH_LPC_DEVICE_ID(DeviceId) \
    ( \
      (DeviceId == V_CDF_PCH_LPC_DEVICE_ID_18DC) \
    )

#define IS_PCH_SERVER_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_CDF_PCH_LPC_DEVICE_ID(DeviceId) \
    )

#define IS_CML_PCH_LP_LPC_DEVICE_ID(DeviceId) \
      ( \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_SUPER_SKU) || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_0)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_2)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_3)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_4)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_5)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_6)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_7)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_8)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_9)         || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_10)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_11)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_12)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_13)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_14)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_15)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_16)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_17)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_18)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_19)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_20)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_21)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_22)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_23)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_24)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_25)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_26)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_27)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_28)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_29)        || \
        (DeviceId == V_CML_PCH_LP_LPC_CFG_DEVICE_ID_MB_30)         \
      )

#define IS_CML_PCH_H_LPC_DEVICE_ID(DeviceId) \
      ( \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0680_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0681_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0682_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0684_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0685_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0687_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0688_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0689_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_068A_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_068C_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_068D_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_068E_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_068F_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0690_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0691_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0692_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0693_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0694_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0695_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0696_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0697_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0698_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_0699_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_069A_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_069B_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_069C_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_069D_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_069E_SKU) || \
        (DeviceId == V_CML_PCH_H_LPC_CFG_DEVICE_ID_069F_SKU) \
      )

#define IS_CML_PCH_LPC_DEVICE_ID(DeviceId) \
    ( \
      IS_CML_PCH_LP_LPC_DEVICE_ID(DeviceId) || \
      IS_CML_PCH_H_LPC_DEVICE_ID(DeviceId) \
    )


#endif
