/** @file
  PCH Serial IO Lib implementation.
  All function in this library is available for PEI, DXE, and SMM,
  But do not support UEFI RUNTIME environment call.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2014 - 2017 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/

#include "MrcOemPlatform.h"
#include "PchSerialIoLib.h"
#include "PchPcrLib.h"
#include "PchRegsSerialIo.h"
#include "PchLimits.h"
#include "PchRegsLpc.h"
#include "PchPsfPrivateLib.h"

typedef struct {
  UINT32 Bar0;
  UINT32 Bar1;
} SERIAL_IO_CONTROLLER_DESCRIPTOR;

typedef struct {
  UINT8  DevNum;
  UINT8  FuncNum;
} SERIAL_IO_BDF_NUMBERS;

typedef struct {
  UINT16 PciDevIdPchLp;
  UINT16 PciDevIdPchH;
} SERIAL_IO_ID;

GLOBAL_REMOVE_IF_UNREFERENCED SERIAL_IO_ID mSerialIoId [PCH_SERIALIO_MAX_CONTROLLERS] =
{
  {V_PCH_LP_SERIAL_IO_CFG_I2C0_DEVICE_ID,  V_PCH_H_SERIAL_IO_CFG_I2C0_DEVICE_ID},
  {V_PCH_LP_SERIAL_IO_CFG_I2C1_DEVICE_ID,  V_PCH_H_SERIAL_IO_CFG_I2C1_DEVICE_ID},
  {V_PCH_LP_SERIAL_IO_CFG_I2C2_DEVICE_ID,  V_PCH_H_SERIAL_IO_CFG_I2C2_DEVICE_ID},
  {V_PCH_LP_SERIAL_IO_CFG_I2C3_DEVICE_ID,  V_PCH_H_SERIAL_IO_CFG_I2C3_DEVICE_ID},
  {V_PCH_LP_SERIAL_IO_CFG_I2C4_DEVICE_ID,  0                                   },
  {V_PCH_LP_SERIAL_IO_CFG_I2C5_DEVICE_ID,  0                                   },
  {V_PCH_LP_SERIAL_IO_CFG_SPI0_DEVICE_ID,  V_PCH_H_SERIAL_IO_CFG_SPI0_DEVICE_ID},
  {V_PCH_LP_SERIAL_IO_CFG_SPI1_DEVICE_ID,  V_PCH_H_SERIAL_IO_CFG_SPI1_DEVICE_ID},
  {V_PCH_LP_SERIAL_IO_CFG_SPI2_DEVICE_ID,  V_PCH_H_SERIAL_IO_CFG_SPI2_DEVICE_ID},
  {V_CML_PCH_LP_SERIAL_IO_CFG_UART0_DEVICE_ID, V_CML_PCH_H_SERIAL_IO_CFG_UART0_DEVICE_ID},
  {V_CML_PCH_LP_SERIAL_IO_CFG_UART1_DEVICE_ID, V_CML_PCH_H_SERIAL_IO_CFG_UART1_DEVICE_ID},
  {V_CML_PCH_LP_SERIAL_IO_CFG_UART2_DEVICE_ID, V_CML_PCH_H_SERIAL_IO_CFG_UART2_DEVICE_ID}
};

GLOBAL_REMOVE_IF_UNREFERENCED SERIAL_IO_BDF_NUMBERS mSerialIoBdf [PCH_SERIALIO_MAX_CONTROLLERS] =
{
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C0,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C0},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C1,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C1},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C2,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C2},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C3,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C3},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C4,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C4},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_I2C5,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_I2C5},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI0,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI0},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI1,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI1},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_SPI2,  PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_SPI2},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART0, PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART0},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART1, PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART1},
  {PCI_DEVICE_NUMBER_PCH_SERIAL_IO_UART2, PCI_FUNCTION_NUMBER_PCH_SERIAL_IO_UART2}
};


GLOBAL_REMOVE_IF_UNREFERENCED SERIAL_IO_CONTROLLER_DESCRIPTOR mSerialIoAcpiAddress [PCH_SERIALIO_MAX_CONTROLLERS] =
{
  {PCH_SERIAL_IO_BASE_ADDRESS + 0x0000,  PCH_SERIAL_IO_BASE_ADDRESS + 0x1000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0x2000,  PCH_SERIAL_IO_BASE_ADDRESS + 0x3000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0x4000,  PCH_SERIAL_IO_BASE_ADDRESS + 0x5000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0x6000,  PCH_SERIAL_IO_BASE_ADDRESS + 0x7000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0x8000,  PCH_SERIAL_IO_BASE_ADDRESS + 0x9000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0xA000,  PCH_SERIAL_IO_BASE_ADDRESS + 0xB000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0xC000,  PCH_SERIAL_IO_BASE_ADDRESS + 0xD000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0xE000,  PCH_SERIAL_IO_BASE_ADDRESS + 0xF000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0x10000, PCH_SERIAL_IO_BASE_ADDRESS + 0x11000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0x12000, PCH_SERIAL_IO_BASE_ADDRESS + 0x13000},
  {PCH_SERIAL_IO_BASE_ADDRESS + 0x14000, PCH_SERIAL_IO_BASE_ADDRESS + 0x15000}
  ,{PCH_SERIAL_IO_BASE_ADDRESS + 0x16000, PCH_SERIAL_IO_BASE_ADDRESS + 0x17000}
};


GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchLpSerialIoPciCfgCtrAddr [PCH_SERIALIO_MAX_CONTROLLERS] =
{
  R_SERIAL_IO_PCR_PCICFGCTRL1,
  R_SERIAL_IO_PCR_PCICFGCTRL2,
  R_SERIAL_IO_PCR_PCICFGCTRL3,
  R_SERIAL_IO_PCR_PCICFGCTRL4,
  R_SERIAL_IO_PCR_PCICFGCTRL5,
  R_SERIAL_IO_PCR_PCICFGCTRL6,
  R_SERIAL_IO_PCR_PCICFGCTRL13,
  R_SERIAL_IO_PCR_PCICFGCTRL14,
  R_SERIAL_IO_PCR_PCICFGCTRL15,
  R_SERIAL_IO_PCR_PCICFGCTRL9,
  R_SERIAL_IO_PCR_PCICFGCTRL10,
  R_SERIAL_IO_PCR_PCICFGCTRL11
};

GLOBAL_REMOVE_IF_UNREFERENCED UINT16 mPchHSerialIoPciCfgCtrAddr[PCH_SERIALIO_MAX_CONTROLLERS] =
{
  R_SERIAL_IO_PCR_PCICFGCTRL1,
  R_SERIAL_IO_PCR_PCICFGCTRL2,
  R_SERIAL_IO_PCR_PCICFGCTRL3,
  R_SERIAL_IO_PCR_PCICFGCTRL4,
  0,
  0,
  R_SERIAL_IO_PCR_PCICFGCTRL13,
  R_SERIAL_IO_PCR_PCICFGCTRL14,
  R_SERIAL_IO_PCR_PCICFGCTRL15,
  R_SERIAL_IO_PCR_PCICFGCTRL9,
  R_SERIAL_IO_PCR_PCICFGCTRL10,
  R_SERIAL_IO_PCR_PCICFGCTRL11
};


/**
  Returns index of the last i2c controller

  @param[in] Number  Number of SerialIo controller

  @retval            Index of I2C controller
**/
PCH_SERIAL_IO_CONTROLLER
GetMaxI2cNumber (
  VOID
  )
{
  if (MrcGetPchSeries () == PchSeriesH) {
    return PchSerialIoIndexI2C3;
  } else {
    return PchSerialIoIndexI2C5;
  }
}

/**
  Finds PCI Device Number of SerialIo devices.

  @param[in] SerialIoNumber             Serial IO device

  @retval                               SerialIo device number
**/
UINT8
GetSerialIoDeviceNumber (
  IN PCH_SERIAL_IO_CONTROLLER  SerialIoNumber
  )
{
  return mSerialIoBdf[SerialIoNumber].DevNum;
}

/**
  Finds PCI Function Number of SerialIo devices.

  @param[in] SerialIoNumber             Serial IO device

  @retval                               SerialIo funciton number
**/
UINT8
GetSerialIoFunctionNumber (
  IN PCH_SERIAL_IO_CONTROLLER  SerialIoNumber
  )
{
  return mSerialIoBdf[SerialIoNumber].FuncNum;
}

/**
  Finds BAR values of SerialIo devices.
  SerialIo devices can be configured to not appear on PCI so traditional method of reading BAR might not work.
  If the SerialIo device is in PCI mode, a request for BAR1 will return its PCI CFG space instead

  @param[in] SerialIoDevice             Serial IO device
  @param[in] BarNumber                  0=BAR0, 1=BAR1

  @retval                               SerialIo Bar value
**/
UINTN
FindSerialIoBar (
  IN PCH_SERIAL_IO_CONTROLLER           SerialIoDevice,
  IN UINT8                              BarNumber
  )
{
  UINT32 Bar;
  UINTN  PcieBase;
  UINT32 VenId;

  PcieBase = MmPciBase (DEFAULT_PCI_BUS_NUMBER_PCH, GetSerialIoDeviceNumber (SerialIoDevice), GetSerialIoFunctionNumber (SerialIoDevice));
  VenId = MrcOemMmioRead32 (PcieBase + PCI_VENDOR_ID_OFFSET) & 0xFFFF;
  if (VenId == V_PCH_INTEL_VENDOR_ID) {
    if (BarNumber == 1) {
      return PcieBase;
    }
    Bar = (MrcOemMmioRead32 (PcieBase + PCI_BASE_ADDRESSREG_OFFSET + 8*BarNumber) & 0xFFFFF000);
    return Bar;
  }
  //PCI mode failed? Try hardcoded addresses from ACPI
  if (BarNumber == 0) {
    Bar = mSerialIoAcpiAddress[SerialIoDevice].Bar0;
  } else {
    Bar = mSerialIoAcpiAddress[SerialIoDevice].Bar1;
  }
  return Bar;
}

/**
  Configures Serial IO Controller

  @param[in] Controller
  @param[in] DeviceMode

  @retval None
**/
VOID
ConfigureSerialIoController (
  IN PCH_SERIAL_IO_CONTROLLER Controller,
  IN PCH_SERIAL_IO_MODE       DeviceMode
  )
{
  UINTN           PciCfgBase;
  UINT32          Data32And;
  UINT32          Data32Or;
  UINT16         *SerialIoPciCfgCtrAddr;

/*
  Please do not add DEBUG message here because this routine is configuring SerialIoUart.
  Printing DEBUG message before SerialIoUart initialization may cause system hang (in Debug build).

  if (Controller >= PchSerialIoIndexMax) {
    return;
  }*/

  PciCfgBase = MmPciBase (0, GetSerialIoDeviceNumber (Controller), GetSerialIoFunctionNumber (Controller));

  /*
  // Do not modify a device that has already been disabled/hidden
  //
  if (MmioRead16 (PciCfgBase + PCI_VENDOR_ID_OFFSET) != V_PCH_INTEL_VENDOR_ID) {
    return;
  }*/

  ///
  /// Step 0. set Bit 16,17,18.
  ///
  MmioOr32 (PciCfgBase + R_SERIAL_IO_CFG_D0I3MAXDEVPG, MRC_BIT18 | MRC_BIT17 | MRC_BIT16);



  if ((MrcGetPchSeries () == PchSeriesH)) {
    SerialIoPciCfgCtrAddr = mPchHSerialIoPciCfgCtrAddr;
  } else {
    SerialIoPciCfgCtrAddr = mPchLpSerialIoPciCfgCtrAddr;
  }

  switch (DeviceMode) {
    case PchSerialIoEnabled:
      //
      //  Check If device is already initialized
      //
      /*
      if (MrcOemMmioRead32 (PciCfgBase + PCI_BASE_ADDRESSREG_OFFSET) & 0xFFFFF000) {
        return;
      }
      */
      ///
      /// reenable PciCfg in case it was disabled earlier
      ///
      PchPcrAndThenOr32 (CNL_PID_SERIALIO, SerialIoPciCfgCtrAddr[Controller], (UINT32) ~(B_SERIAL_IO_PCR_PCICFGCTRL_PCI_CFG_DIS | B_SERIAL_IO_PCR_PCICFGCTRL_ACPI_INTR_EN),0x0);
      PsfRevealSerialIoDevice (Controller);
      ///
      /// Disable Bar1
      /// Disable Bar1 in PSF
      ///
      PchPcrAndThenOr32 (CNL_PID_SERIALIO, SerialIoPciCfgCtrAddr[Controller], 0xFFFFFFFF, B_SERIAL_IO_PCR_PCICFGCTRL_BAR1_DIS);
      PsfDisableSerialIoDeviceBar1 (Controller);
      //
      // Assign BAR0 and Set Memory space Enable
      //
      MrcOemMmioWrite32 (PciCfgBase + R_SERIAL_IO_CFG_BAR0_LOW,  mSerialIoAcpiAddress[Controller].Bar0);
      MrcOemMmioWrite32 (PciCfgBase + R_SERIAL_IO_CFG_BAR0_HIGH, 0x0);
      MmioOr32    (PciCfgBase + PCI_COMMAND_OFFSET, EFI_PCI_COMMAND_MEMORY_SPACE);
      ///
      /// get controller out of reset
      ///
      MmioOr32 (mSerialIoAcpiAddress[Controller].Bar0 + R_SERIAL_IO_MEM_PPR_RESETS,
        B_SERIAL_IO_MEM_PPR_RESETS_FUNC | B_SERIAL_IO_MEM_PPR_RESETS_APB | B_SERIAL_IO_MEM_PPR_RESETS_IDMA);
      break;
    default:
      return;
  }

  ///
  /// Step X. Program clock dividers for UARTs
  /// Step Y. Enable Byte addressing for UARTs in legacy mode
  ///
  if ((Controller >= PchSerialIoIndexUart0 && Controller <= PchSerialIoIndexUart2) && (DeviceMode != PchSerialIoDisabled)) {
    MrcOemMmioWrite32 (mSerialIoAcpiAddress[Controller].Bar0 + R_SERIAL_IO_MEM_PPR_CLK,
      (B_SERIAL_IO_MEM_PPR_CLK_UPDATE | (V_SERIAL_IO_MEM_PPR_CLK_N_DIV << 16) |
       (V_SERIAL_IO_MEM_PPR_CLK_M_DIV << 1) | B_SERIAL_IO_MEM_PPR_CLK_EN )
      );

    Data32And = (UINT32) (~(B_SERIAL_IO_PCR_GPPRVRW7_UART0_BYTE_ADDR_EN << (Controller - PchSerialIoIndexUart0)));
    Data32Or = 0x0;
    if (DeviceMode == PchSerialIoHidden) {
      Data32Or = (B_SERIAL_IO_PCR_GPPRVRW7_UART0_BYTE_ADDR_EN << (Controller - PchSerialIoIndexUart0));
    }
    PchPcrAndThenOr32 (CNL_PID_SERIALIO, R_SERIAL_IO_PCR_GPPRVRW7,Data32And,Data32Or);
    //
    // Dummy read after setting any of GPPRVRW7.
    // Required for UART 16550 8-bit Legacy mode to become active
    //
    MrcOemMmioRead32 (mSerialIoAcpiAddress[Controller].Bar0 + R_SERIAL_IO_MEM_PPR_CLK);
  }
/*
  ///
  /// Step Z. Program I2C SDA hold registers
  ///
  if (Controller >= PchSerialIoIndexI2C0 && Controller <= GetMaxI2cNumber ()) {
    if (DeviceMode != PchSerialIoDisabled) {
      MmioOr32 (mSerialIoAcpiAddress[Controller].Bar0 + R_SERIAL_IO_MEM_I2C_SDA_HOLD, V_SERIAL_IO_MEM_I2C_SDA_HOLD_VALUE);
    }
  }
*/
}
