/** @file
  These functions implement the crosser training algorithm.

@copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
// Include files
#include "MrcCrosser.h"
#include "MrcWriteDqDqs.h"
#include "MrcReadDqDqs.h"
#include "MrcCommon.h"
#include "MrcMcConfiguration.h"
#include "MrcCpgcApi.h"
#include "MrcChipApi.h"
#include "MrcMemoryApi.h"
#include "MrcLpddr4.h"
#include "MrcDdr4.h"
#include "MrcMemoryApi.h"
#include "MrcCpgcOffsets.h"
#include "Cpgc20TestCtl.h"
#include "MrcDdrIoOffsets.h"

#if SUPPORT_SODIMM == SUPPORT
#include "MrcSpdProcessing.h"
#endif //SUPPORT_SODIMM == SUPPORT


/// Module Definitions
#define DIMM_ODT_DIMM_MASK_SHIFT  (4)

/// Power optimization loop count
#define OPT_PARAM_LOOP_COUNT (12)

// Re-centering loop count
#define RE_CENTER_LOOP_COUNT (10)

// Re-centering loop count
#define CCC_RE_CENTER_LOOP_COUNT (10)

/// Power optimization re-centering step size
#define T_RECENTERING_STEP_SIZE (2)
#define V_RECENTERING_STEP_SIZE (3)

/// Power optimization final centering step size
#define V_FINAL_STEP_SIZE (2)

/// Power optimization cmd timing margining step size
#define CMD_T_MARGIN_STEP_SIZE (3)

/// UPM/PWR increment value if margins are at or below the retrain limit.
#define MRC_UPM_PWR_INC_VAL (80)

/// Vswing value = Vindiff_Dqs/2 * 1.1 (Guardband) = 216mV
#define VSWING (216)

/// Zo
#define Z_O (40)

/// Vref step size for the following: DqOdt, DqDrv, CmdDrv, CtlDrv, ClkDrv
#define MRC_COMP_VREF_STEP_SIZE   (191)

/// Number of VTT regs
#define MAX_VTT_REGS (4)

/// SenseAmplifier Wake-Up Time in pS
#define MRC_SENSE_AMP_WAKEUP_TIME (2000)

/// SOT Spacing Macro
///   GroupIdx | Spaces | Trained
///   ---------|--------|-------------
///      0     |   9    | 8 DQs, 1 DQS
///      1     |   8    | 8 DQs
///      2     |   1    | 1 DQS
#define MRC_SOT_BIT_SPACE(x) (((x) == 0) ? ("         ") : (((x) == 1) ? ("        ") : (" ")))

/// We need to make one of the bits in the combined Scomp value the PC bit value. It can be any bit as long as we always use the same one everywhere, and don't overwrite any bits used to store the SComp values.
/// The combined Scomp value is split back into individual PC and Scomp values before being written to registers using the get/set methods.
#define SCOMP_PC_STORAGE_BIT_OFFSET (4)

// Param step sizes. IF YOU CHANGE THESE, YOU MUST ADJUST THE SWEEP RANGE TABLE TO MATCH!!
#define DQ_TCO_COMP_STEP (7)
#define DQS_TCO_COMP_STEP (3)
#define CLK_TCO_COMP_STEP (2)
#define VDDQ_STEP (4)
#define WR_DS_STEP (3)
#define WR_TXEQ_STEP (2)
#define WR_DS_COARSE_STEP (4)
#define CCC_DS_STEP (2)
#define CCC_TXEQ_STEP (2)
#define CCC_DS_COARSE_STEP (3)
#define RXDQ_ODT_STEP (3)
#define RXDQS_ODT_STEP (3)
#define RXEQ_STEP (2)
#define DFE_STEP_PER_CHANNEL (3)
#define DFE_STEP (1)
#define RX_LOAD_STEP (2)
#define RX_BIAS_STEP (2)
#define RX_CB_DATA_STEP (2)
#define RX_CB_COMP_STEP (2)

// Adjustments to line up 0 ticks with 0 volts
#define VDDQ_BASE_TICKS (0) // Supposedly zero, documentation does not say for certain.
#define RXVREF_BASE_TICKS (0) // Supposedly zero, documentation does not say for certain. 0-381. Step size is Vddxx/382
#define DDR4_CMDVREF_BASE_TICKS (0) // Supposedly zero, documentation does not say for certain. 0-381. Step size is Vddxx/382
#define DDR4_TXVREF_BASE_TICKS (69) // 45% at 0.65% per tick
#define LP4_CMDVREF_BASE_TICKS (25) // 15% at 0.6% per tick
#define LP4_TXVREF_BASE_TICKS (25) // 15% at 0.6% per tick
// TGL_POWER_TRAINING_VDDQ - Update for DDR5. Also check if DDR5 ODT types, value ranges, and RZQ value has changed at all.
//#define DDR5_CMDVREF_BASE_TICKS (?)
//#define DDR5_TXVREF_BASE_TICKS (?)
// TGL_POWER_TRAINING_VDDQ - Update for LPDDR5 depending on manufacturer spec
#define LP5_CMDVREF_BASE_TICKS (20) // Samsung has 15% at 0.5% per tick, JEDEC spec has 10%
#define LP5_TXVREF_BASE_TICKS (20) // Samsung has 15% at 0.5% per tick, JEDEC spec has 10%

// Slew rate constants
#define SLEW_RATE_ENABLED (0)
#define CYCLE_LOCK (1)

// Number of read-write command margin failures.
#define NUM_RW_CMD_MARGIN_FAILURES (3)

// CCC margins are twice as large.
#define MAX_CCC_POSSIBLE_TIME (MAX_POSSIBLE_TIME * 2)

// 8 values can be set in each decap register
#define MAX_DECAP_VALUES (8)

// Max CCC TxEq limit
#define CCC_TXEQ_MAX (12)

// Max TxEq Limit
#define TXEQ_MAX (10)

// Min LPDDR CCC TxEq limit
#define LP_CCC_TXEQ_LOW_LIMIT (8)

// Up and Down
#define MAX_COMP_DIRECTIONS (2)
#define COMP_UP (0)
#define COMP_DN (1)

// The max number of comps/comp vrefs we ever optimize at once
#define MAX_COMPS_OPTIMIZED (3)

// The max number of comp offsets we ever optimize at once
#define MAX_COMP_OFFSETS_OPTIMIZED (4)

// The array index in which we store the name of the optimized comp vref during comp optimization
#define OPTIMIZED_COMP_VREF_INDEX (0)

// The max number of comps we ever check for saturation at once
#define MAX_COMPS_CHECKED (3)

// The number of comp codes we reserve to prevent saturation during training
#define RESERVED_COMP_CODES (3)

// Segment numbers for comp vrefs
#define ONE_SEGMENT  (1)
#define TWO_SEGMENTS (2)
#define THREE_SEGMENTS (3)

// Enable FIVR power Measuremnt
#define FIVR_POWER_MEASUREMENT_ENABLED (0)

// Right now, we can only run 1D and 2D optimizations
#define MAX_OPT_PARAM_LENGTH (2)

// 2 places of decimal accuracy
#define TWO_DECIMAL_PLACES (100)

// Scale for linear normalized power
#define LINEAR_NORMALIZED_POWER_SCALE (1000)

// Margins are multiplied by this when returned from the margining functions
#define MARGIN_MULTIPLIER (10)

// Less than this amount of margin is considered a failure
#define MARGIN_FAIL (2 * (MARGIN_MULTIPLIER))

/// Module Globals - TGL_POWER_TRAINING - Fill this in with actual numbers for everything.
/// Retraining limits are per eye-side. UPM/PWR limits are total eye width/height. Both are in offset register ticks, not raw register ticks!
GLOBAL_REMOVE_IF_UNREFERENCED const MrcUpmPwrRetrainLimits  InitialLimits[MRC_NUMBER_UPM_PWR_RETRAIN_MARGINS] = {
  //           UPM,          PWR       Retrain
  {RdT,       {300,          1280,     90 }},
  {WrT,       {300,          1280,     90 }},
  {RdV,       {400,          1280,     160}},
  // For ULT DDR3L rcF the values are increased by 20 ticks, see MrcGetUpmPwrLimit()
  {WrV,       {600,          1280,     160}},
  {RdFan2,    {240,          1280,       0}},
  {WrFan2,    {240,          1280,       0}},
  {RdFan3,    {(240*4)/5,    1280,       0}},
  {WrFan3,    {(240*4)/5,    1280,       0}},
  // {650ps,750ps} * 64 pi ticks * 2 (for width) = 134 PI Ticks ->  ~1.3nsec for UPM, 154 PI Ticks -> ~1.5nsec for PWR
  // Margin function works in steps of 4, so we divide the margin by 4.
  // Margin numbers are scaled by 10.
  {RcvEnaX,   {((134*10)/4), 1280,       0}},
  {CmdT,       {600,         1280,     180}}, // CCC retrain limit is around twice what data is, as margins are +-64 instead of +-32 (Data is DDR, CLK is not)
  {CmdV,       {800,         1280,     160}}
};

GLOBAL_REMOVE_IF_UNREFERENCED const MrcOptParamsLimits  MrcOptParamLimit[MRC_NUMBER_OPT_PARAMS_TRAIN] = {
  //            Normal/SAGV(high), SAGV (low), DT/Halo (max perf.)
  // Use full range for now, optimize postsilicon
  {OptRdDqOdt,        {-4,    4}, {-4,    4}, {-4,    4}}, // This is a comp offset on a side effect comp vref, so needs at least 3 codes of guardband to account for comp offset drift during comp offset optimization
  {OptRdDqsOdt,       {-5,    5}, {-5,    5}, {-5,    5}},
  {OptRxBias,         {0,     7}, {0,     7}, {0,     7}},
  {OptRxLoad,         {-8,    7}, {-8,    7}, {-8,    7}},
  {OptRxCbData,       {0,     7}, {0,     7}, {0,     7}},
  {OptRxCbComp,       {0,     7}, {0,     7}, {0,     7}},
  {OptRxEq,           {0,     7}, {0,     7}, {0,     7}},
  {OptTxEq,           {0,     5}, {0,     5}, {0,     5}},
  {OptWrDS,           {-10,  10}, {-10,  10}, {-10,  10}},
  {OptWrDSDnCoarse,   {-7,    7}, {-7,    7}, {-7,    7}}, // This is a comp offset on a side effect comp vref, so needs at least 3 codes of guardband to account for comp offset drift during comp offset optimization
  {OptWrDSUpCoarse,   {-8,    7}, {-8,    7}, {-8,    7}},
  {OptSComp,          {-16,  15}, {-16,  15}, {-16,  15}},
  {OptCCCTco,         {0,    31}, {0,    31}, {0,    31}},
  {OptTxDqTco,        {0,     9}, {0,     9}, {0,     9}},
  {OptTxDqsTco,       {0,    10}, {0,    10}, {0,    10}},
  {OptVddq,           {0,    30}, {0,    30}, {0,    30}},
  {OptCCCTxEq,        {0,     6}, {0,     6}, {0,     6}},
  {OptCCCDS,          {-4,    3}, {-4,    3}, {-4,    3}},
  {OptCCCDSDnCoarse,  {-1,    1}, {-1,    1}, {-1,    1}}, // This is a comp offset on a side effect comp vref, so needs at least 3 codes of guardband to account for comp offset drift during comp offset optimization
  {OptCCCDSUpCoarse,  {-2,    2}, {-2,    2}, {-2,    2}},
  {OptCCCSComp,       {-8,    7}, {-8,    7}, {-8,    7}},
  {OptVccDLLBypass,   {0,     1}, {0,     1}, {0,     1}},
  {OptRxVrefVttDecap, {0,     7}, {0,     7}, {0,     7}},
  {OptRxVrefVddqDecap,{0,     7}, {0,     7}, {0,     7}},
  {OptDFETap0,        {0,     5}, {0,     5}, {0,     5}},
  {OptDFETap1,        {0,     5}, {0,     5}, {0,     5}},
  {OptRxC,            {0,     3}, {0,     3}, {0,     3}},
  {OptRxR,            {0,     3}, {0,     3}, {0,     3}},
  {OptPanicVttDnLp,   {0,    15}, {0,    15}, {0,    15}}
};

// DDR Params encoding
#ifdef MRC_DEBUG_PRINT
GLOBAL_REMOVE_IF_UNREFERENCED const char  *TOptParamOffsetString[] = {
  "WrDS",
  "Vddq",
  "SComp",
  "TxDqTCoComp",
  "TxDqsTCoComp",
  "CCCTCoComp",
  "TxTCoCompoff",
  "TxEq",
  "RdOdt",
  "RdDqOdt",
  "RdDqsOdt",
  "RxEq",
  "RxBias",
  "RxLoad",
  "RxCbData",
  "RxCbComp",
  "DimmOdt",
  "DimmOdtWr",
  "DimmOdtNom",
  "DimmOdtNomNT",
  "OptOdtPark",
  "OptOdtParkNT",
  "OptDimmOdtComb",
  "DimmOdtCa",
  "DimmRon",
  "WrDSUp",
  "WrDSDn",
  "WrDSUpCoarse",
  "WrDSDnCoarse",
  "RdOdtUp",
  "RdOdtDn",
  "CCCDS",
  "CCCDSUp",
  "CCCDSDn",
  "CmdDS",
  "CtlDS",
  "ClkDS",
  "RxBiasCb",
  "TxEqWrDS",
  "DimmSocOdt",
  "CCCTxEq",
  "CCCDSDnCoarse",
  "CCCDSUpCoarse",
  "CmdDSDnCoarse",
  "CmdDSUpCoarse",
  "CtlDSDnCoarse",
  "CtlDSUpCoarse",
  "ClkDSDnCoarse",
  "ClkDSUpCoarse",
  "CCCTxEqCCCDS",
  "CCCSComp",
  "CmdSComp",
  "CtlSComp",
  "ClkSComp",
  "VccDLLBypass",
  "RxVrefVttDecap",
  "RxVrefVddqDecap",
  "DFETap0",
  "DFETap1",
  "RxC",
  "RxR",
  "PanicVttDnLp",
  "OptDefault",
  "Default"
};

/// These strings match the OptResultPerByteDbgStr enum for indexing
/// the switch PrintCalcResultTableCh and PrintODTResultTable.
const char *OptResultDbgStrings[] = {
  "Best",
  "GrdBnd",
  "OffSel",
  "Scale",
  "MaxPost",
  "MinPost"
};

// Strings for TGlobalCompOffset decoding
const char *GlobalCompOffsetStr[] = {
  "RdOdt",
  "WrDS",
  "WrDSCmd",
  "WrDSCtl",
  "WrDSClk"
};

// Strings for CompGlobalOffsetParam decoding
const char *CompGlobalOffsetParamStr[] = {
  "RdOdtUp",
  "RdOdtDn",
  "WrDSUp",
  "WrDSDn",
  "WrDSCmdUp",
  "WrDSCmdDn",
  "WrDSCtlUp",
  "WrDSCtlDn",
  "WrDSClkUp",
  "WrDSClkDn",
  "SCompDq",
  "SCompCmd",
  "SCompCtl",
  "SCompClk",
  "RxLoad",
  "DisOdtStatic"
};

const char *OdtTypeString[] = {
  "RttWr",
  "RttNom",
  "RttPark",
  "RttMaxType"
};

const char  *CmdIterTypesString[] = {
  "MrcIterationClock",
  "MrcIterationCmd",
  "MrcIterationCtl",
  "MrcIterationCmdCtl"
};
#endif // MRC_DEBUG_PRINT

/**
  This function implements Sense Amp Offset training.
  The algorithm sweeps Per-Bit Vref Offset and parks the SenseAmpOffset
  at the "last 1" setting before transitions to 0.

  Algorithm will account for speckling, and consider the "last 1" after specling.

  @param[in,out] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeded return mrcSuccess
**/
MrcStatus
MrcSenseAmpOffsetTraining (
  IN OUT MrcParameters *const MrcData
  )
{
  const MrcInput      *Inputs;
  MrcDebug            *Debug;
  const MRC_FUNCTION  *MrcCall;
  MrcOutput           *Outputs;
  MrcChannelOut       *ChannelOut;
  MrcStatus           Status;
  MRC_RX_MODE_TYPE    RxModeSave;
  GSM_GT              DqSaGrp;
  GSM_GT              DqsSaGrp;
  INT64               GetSetVal;
  INT64               GetSetMax;
  INT64               GetSetMin;
  INT64               GetSetEn;
  INT64               GetSetDis;
  INT64               DataTrainFeedbackReg;
  UINT32              DataTrainFeedback;
  UINT32              SenseAmpTestDelay;
  UINT32              SdramCount;
  UINT32              FirstController;
  UINT32              FirstChannel;
  INT8                FirstZero[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_BITS_FOR_OFFSET_TRAINING];  // Additional bit for DQS per each byte
  INT8                LastOne[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_BITS_FOR_OFFSET_TRAINING];  // Additional bit for DQS per each byte
  INT8                SampOffset;
  INT8                Vref;
  INT8                MinValue;
  INT8                MaxValue;
  UINT8               Controller;
  UINT8               Channel;
  UINT8               Rank;
  UINT8               Byte;
  UINT8               Bit;
  UINT8               GroupIdx;
  UINT8               MaxChannel;
  UINT8               MaxGrp;
  UINT8               MaxRank;
  UINT8               BitStart;
  UINT8               BitEnd;
  UINT8               BiasPMCtrl;
  UINT8               OrigRxTap0;
  UINT8               OrigRxTap1;
  UINT8               OrigForceRxOn;
  UINT8               RankPresent[MAX_CONTROLLER][MAX_CHANNEL];
  BOOLEAN             Ddr4;
  BOOLEAN             Ddr5;
  BOOLEAN             Lpddr5;
  BOOLEAN             EnDqsN;
  BOOLEAN             CheckDqs;
  BOOLEAN             CheckDq;
  BOOLEAN             Unmatched;

  Inputs          = &MrcData->Inputs;
  MrcCall         = Inputs->Call.Func;
  Outputs         = &MrcData->Outputs;
  Debug           = &Outputs->Debug;
  Status          = mrcSuccess;
  Ddr4            = (Outputs->DdrType == MRC_DDR_TYPE_DDR4);
  Ddr5            = (Outputs->DdrType == MRC_DDR_TYPE_DDR5);
  Lpddr5          = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);
  MaxRank         = MAX_RANK_IN_CHANNEL;
  MaxChannel      = Outputs->MaxChannels;
  SdramCount      = Outputs->SdramCount;
  RxModeSave      = Outputs->RxMode;
  GetSetEn        = 1;
  GetSetDis       = 0;
  EnDqsN          = 0;
  OrigRxTap0      = 0;
  OrigRxTap1      = 0;
  SenseAmpTestDelay = MRC_TIMER_1US;
  MrcCall->MrcSetMem ((UINT8 *) RankPresent,  sizeof (RankPresent), 0xFF);


  // Read the first populated channel
  FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = Outputs->Controller[FirstController].FirstPopCh;
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocBiasPMCtrl, ReadFromCache, &GetSetVal);
  BiasPMCtrl = (UINT8) GetSetVal;
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocEnDqsNRcvEn, ReadFromCache, &GetSetVal);
  EnDqsN = (GetSetVal == 1) ? 1 : 0;
  MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocForceRxAmpOn, ReadFromCache, &GetSetVal);
  OrigForceRxOn = (UINT8) GetSetVal;
  for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
    if (MrcRankExist (MrcData, FirstController, FirstChannel, Rank)) {
      MrcGetSetStrobe (MrcData, FirstController, FirstChannel, Rank, 0, RxTap0, ReadFromCache, &GetSetVal);
      OrigRxTap0 = (UINT8) GetSetVal;
      MrcGetSetStrobe (MrcData, FirstController, FirstChannel, Rank, 0, RxTap1, ReadFromCache, &GetSetVal);
      OrigRxTap1 = (UINT8) GetSetVal;
      break;
    }
  }

  Unmatched = (RxModeSave == MrcRxModeUnmatchedRxWRload) || (RxModeSave == MrcRxModeUnmatchedRxWPpath);
  if (Unmatched) {
    if (Lpddr5 && EnDqsN) {
      MaxGrp = 3;
    } else {
      MaxGrp = 2;
    }
  } else {
    MaxGrp = 1;
  }

  if (Unmatched) {
    //Disable DFE for SOT
    MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxTap0, WriteToCache, &GetSetDis);
    MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxTap1, WriteToCache, &GetSetDis);
  }

  GetSetVal = 0;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocBiasPMCtrl, WriteToCache, &GetSetVal);
  GetSetVal = 1;
  // Need to ungate SOT
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocInternalClocksOn, WriteToCache, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  // Find first rank present.  Need to set Rank Mux to this rank for training.
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (MrcRankExist (MrcData, Controller, Channel, Rank)) {
          RankPresent[Controller][Channel] = Rank;
          break;
        }
      }
    }
  }

  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocForceRxAmpOn, WriteCached, &GetSetEn);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRankOverrideEn, WriteNoCache, &GetSetEn);
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    if (!MrcControllerExist (MrcData, Controller)) {
      continue;
    }
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        GetSetVal = RankPresent[Controller][Channel];
        MrcGetSetChStrb (MrcData, Controller, Channel, MAX_SDRAM_IN_DIMM, GsmIocRankOverrideVal, WriteNoCache, &GetSetVal);
      }
    }
  }

  for (GroupIdx = 0; GroupIdx < MaxGrp; GroupIdx++) {
    switch (GroupIdx) {
      case 0:
        // Matched or Unmatched.  Sweep DQ and DQS.
        CheckDqs = 1;
        CheckDq = 1;
        BitStart = 0;
        BitEnd = MAX_BITS_FOR_OFFSET_TRAINING;
        break;

      case 1:
        // Unmatched case.  Sweep only DQ.
        CheckDqs = 0;
        CheckDq = 1;
        BitStart = 0;
        BitEnd = MAX_BITS;
        break;

      case 2:
        // Unmatched with EnDqsN.  Sweep only DQS.
        // Must set RxMode to matched p to train.
        GetSetVal = MrcRxModeMatchedP;
        // This is needed because MrcGetSetLimits limits depend on if we are in matched or unmatched.
        Outputs->RxMode = MrcRxModeMatchedP;
        MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxVocMode,  WriteCached, &GetSetVal);
        CheckDqs = 1;
        CheckDq = 0;
        BitStart = MAX_BITS_FOR_OFFSET_TRAINING - 1;
        BitEnd = MAX_BITS_FOR_OFFSET_TRAINING;
        break;

      default:
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Invalid iteration index of SOT: %d\n", GroupIdx);
        return mrcFail;
    }

    // Only train the strobe SenseAmp Offset on the first pass.
    if (Ddr4 || Ddr5) {
      GetSetVal = 1;
      DqsSaGrp = RxDqsAmpOffset;
      DqSaGrp = RxVoc;
    } else if (!Unmatched) {
      GetSetVal = 0;
      DqsSaGrp = RxDqsAmpOffset;
      DqSaGrp = RxVoc;
    } else {
      // LPDDR Unmatched
      if (GroupIdx == 0) {
        DqsSaGrp = RxDqsUnmatchedAmpOffset;
        DqSaGrp = RxVoc;
        GetSetVal = 1;
      } else {
        DqsSaGrp = RxDqsAmpOffset;
        DqSaGrp = RxVocUnmatched;
        GetSetVal = 0;
      }
    }

    MrcGetSetLimits (MrcData, DqSaGrp, &GetSetMin, &GetSetMax, NULL);
    MinValue = (INT8) GetSetMin;
    MaxValue = (INT8) GetSetMax;

    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxAmpOffsetEn, WriteNoCache, &GetSetVal);

    // Init FirstZero and LastOne to 0
    MrcCall->MrcSetMem ((UINT8 *) FirstZero, sizeof (FirstZero), 0);
    MrcCall->MrcSetMem ((UINT8 *) LastOne, sizeof (LastOne), 0);

    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
          if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
            continue;
          }
          for (Byte = 0; Byte < SdramCount; Byte++) {
            //Force the GetSet cache to be updated to current TxDqsDccOffset and TxDqDccOffset
            MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, TxDqsDccOffset, ReadCached, &GetSetVal);
            for (Bit = 0; Bit < MAX_BITS; Bit++) {
              MrcGetSetBit (MrcData, Controller, Channel, Rank, Byte, Bit, TxDqDccOffset, ReadCached, &GetSetVal);
            }
          }
        }
      }
    }
    // Print Controller header.
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "SampOffset Training:");
    if (CheckDq) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s,", GsmGtDebugStrings[DqSaGrp]);
    }
    if (CheckDqs) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s", GsmGtDebugStrings[DqsSaGrp]);
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nMC\t");
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      if (!MrcControllerExist (MrcData, Controller)) {
        continue;
      }
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d", Controller);
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        if (!MrcChannelExist (MrcData, Controller, Channel)) {
          continue;
        }
        for (Byte = 0; Byte < SdramCount; Byte++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s%s", MRC_SOT_BIT_SPACE (GroupIdx), ((Byte != (SdramCount -1) ? " " : "")));
        }
      }
    }
    // Print Channel Header.
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nChannel\t");
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        if (!MrcChannelExist (MrcData, Controller, Channel)) {
          continue;
        }
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d", Channel);
        for (Byte = 0; Byte < SdramCount; Byte++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s%s", MRC_SOT_BIT_SPACE (GroupIdx), ((Byte != (SdramCount -1) ? " " : "")));
        }
      }
    }
    // Print Byte Header
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nByte\t");
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        if (!MrcChannelExist (MrcData, Controller, Channel)) {
          continue;
        }
        for (Byte = 0; Byte < SdramCount; Byte++) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d%s", Byte, MRC_SOT_BIT_SPACE (GroupIdx));
        }
      }
    }
    //Print Bit Header
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nBits\t");
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        if (!MrcChannelExist (MrcData, Controller, Channel)) {
          continue;
        }
        for (Byte = 0; Byte < SdramCount; Byte++) {
          for (Bit = BitStart; Bit < BitEnd; Bit++) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, ((Bit == 8) ? "S" : "%d"), Bit);
          }
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, " ");
        }
      }
    }

    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n SAmp");

    for (SampOffset = MinValue; SampOffset <= MaxValue; SampOffset++) {
      // Display per Byte Feedback from REUT Registers
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n% 5d\t", SampOffset);
      // Clear out DataTrainFeedback field
      GetSetVal = 0;
      MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataTrainFeedback, WriteNoCache, &GetSetVal);

      // Program Offset
      GetSetVal = SampOffset;
      if (CheckDq) {
        MrcGetSetBit (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, MAX_BITS, DqSaGrp, WriteToCache, &GetSetVal);
      }
      if (CheckDqs) {
        MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, DqsSaGrp, WriteToCache, &GetSetVal);
      }
      MrcFlushRegisterCachedData (MrcData);

      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        if (!MrcControllerExist (MrcData, Controller)) {
          continue;
        }

        for (Channel = 0; Channel < MaxChannel; Channel++) {
          if (!MrcChannelExist (MrcData, Controller, Channel)) {
            continue;
          }

          MrcGetSetChStrb (MrcData, Controller, Channel, MAX_SDRAM_IN_DIMM, GsmIocForceOdtOn, WriteCached, &GetSetEn);
          MrcGetSetChStrb (MrcData, Controller, Channel, MAX_SDRAM_IN_DIMM, GsmIocSenseAmpMode, WriteNoCache, &GetSetEn);
          MrcWait (MrcData, SenseAmpTestDelay);

          for (Byte = 0; Byte < SdramCount; Byte++) {
            if (!MrcByteExist (MrcData, Controller, Channel, Byte)) {
              continue;
            }
            MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocSenseAmpMode, WriteNoCache, &GetSetDis);
            MrcGetSetChStrb (MrcData, Controller, Channel, Byte, GsmIocDataTrainFeedback, ReadUncached, &DataTrainFeedbackReg);
            DataTrainFeedback = (UINT32) DataTrainFeedbackReg;
            //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "DataTrainFeedback.Bits.DataTrainFeedback = 0x%x, SampOffset = %d\n", DataTrainFeedback, SampOffset);

            for (Bit = BitStart; Bit < BitEnd; Bit++) {
              if (DataTrainFeedback & (MRC_BIT0 << Bit)) {
                LastOne[Controller][Channel][Byte][Bit] = SampOffset;
              } else {
                if (FirstZero[Controller][Channel][Byte][Bit] == 0) {
                  FirstZero[Controller][Channel][Byte][Bit] = SampOffset;
                }
              }
              // Display in bits
              MRC_DEBUG_MSG (
                Debug,
                MSG_LEVEL_NOTE,
                ((MRC_BIT0 << Bit) & DataTrainFeedback) ? "1" : "0"
              );
            }

            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, " ");
          } // for Byte
        } // for Channel
      } // for Controller
    } // for SampOffset
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nResults\n");

    // Calculate and Program Offsets and display per bit SenseAmp Offset
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      if (!MrcControllerExist (MrcData, Controller)) {
        continue;
      }
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        if (!MrcChannelExist (MrcData, Controller, Channel)) {
          continue;
        }
        ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
        for (Byte = 0; Byte < SdramCount; Byte++) {
          for (Bit = BitStart; Bit < BitEnd; Bit++) {
            // Find Vref center, add 1 for Round Up
            Vref = (FirstZero[Controller][Channel][Byte][Bit] + LastOne[Controller][Channel][Byte][Bit]) / 2;

            // Check for saturation conditions
            // to make sure we are as close as possible to vih/2
            if (FirstZero[Controller][Channel][Byte][Bit] == 0) {
              Vref = MaxValue;
            }

            if (LastOne[Controller][Channel][Byte][Bit] == 0) {
              Vref = MinValue;
            }

            GetSetVal = Vref;
            if (Bit == 8) {
              MrcGetSetStrobe (MrcData, Controller, Channel, MAX_RANK_IN_CHANNEL, Byte, DqsSaGrp, WriteCached | PrintValue, &GetSetVal);
              break;
            }

            for (Rank = 0; Rank < MaxRank; Rank++) {
              if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
                continue;
              }
              ChannelOut->RxDqVrefPb[Rank][Byte][Bit].Center = Vref;
              MrcGetSetBit (MrcData, Controller, Channel, Rank, Byte, Bit, DqSaGrp, WriteCached | PrintValue, &GetSetVal);
            }
          } // for Bit
        } // for Byte
      } // for Channel
    } // for Controller
  } // SenseAmp Knob

  // Clean up after test.
  // Disable Senseamp Mode and ForceOdtOn mode.
  // Disable ForceOdtOn before ODT mode switch.
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocSenseAmpMode, WriteNoCache, &GetSetDis);
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocForceOdtOn, WriteCached, &GetSetDis);
  // Restore training mode if we were LP5 unmatched with EnDqsN set
  if (Unmatched && Lpddr5 && EnDqsN) {
    GetSetVal = RxModeSave;
    Outputs->RxMode = RxModeSave;
    MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRxVocMode,  WriteToCache, &GetSetVal);
  }

  if (Unmatched) {
    //Re-enable DFE after SOT
    GetSetVal = OrigRxTap0;
    MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxTap0, WriteToCache, &GetSetVal);
    GetSetVal = OrigRxTap1;
    MrcGetSetStrobe (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_RANK_IN_CHANNEL, MAX_SDRAM_IN_DIMM, RxTap1, WriteToCache, &GetSetVal);
  }
  // Disable Rank Mux Override
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocRankOverrideEn, WriteNoCache, &GetSetDis);

  GetSetVal = BiasPMCtrl;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocBiasPMCtrl, WriteToCache, &GetSetVal);
  GetSetVal = OrigForceRxOn;
  MrcGetSetChStrb (MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocForceRxAmpOn, WriteToCache, &GetSetVal);
  MrcFlushRegisterCachedData (MrcData);

  Status = IoReset (MrcData);

  return Status;
}

/**
  Force Rcomp to update if need be

  @param[in, out] MrcData - MRC global data.
  @param[in] OptParam - Param type
  @param[in] ForceComp - Force the comp to run regardless of the param type

  @retval BOOLEAN - Whether a comp was performed or not
**/
BOOLEAN
ForceSystemRComp(
  IN OUT MrcParameters *const MrcData,
  IN     UINT8                OptParam,
  IN     BOOLEAN              ForceComp
  )
{
  MrcOutput               *Outputs;
  UINT8                   Controller;
  UINT8                   Channel;
  UINT8                   Byte;
  UINT8                   Comp = 0;
  INT64                   GetSetVal;

  Outputs = &MrcData->Outputs;

  if (ForceComp == FALSE) {
    switch (OptParam) { // Re-run comps after any comp or voltage change
      case OptTxDqTco:
      case OptTxDqsTco:
      case OptCCCTco:
      case OptWrDS:
      case OptWrDSUpCoarse:
      case OptWrDSDnCoarse:
      case DqDrvUpCompOffset:
      case DqDrvDnCompOffset:
      case OptRdDqOdt:
      case OptRdDqsOdt:
      case DqsOdtCompOffset:
      case DqOdtCompOffset:
      case OptRxLoad:
      case RloadCompOffset:
      case OptSComp:
      case DqSCompOffset:
      case OptCCCDS:
      case OptCCCDSUp:
      case OptCCCDSDn:
      case OptCCCDSUpCoarse:
      case OptCCCDSDnCoarse:
      case CmdRCompDrvUpOffset:
      case CmdRCompDrvDownOffset:
      case CtlRCompDrvUpOffset:
      case CtlRCompDrvDownOffset:
      case OptCCCSComp:
      case CmdSCompOffset:
      case CtlSCompOffset:
      case OptVddq:
      case OptVccDLLBypass:
      case OptRxVrefVttDecap:
      case OptRxVrefVddqDecap:
        Comp = 1;
        break;
      default:
        Comp = 0;
    }
  }

  if (Comp || ForceComp == TRUE) {

    GetSetVal = 1;
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        if (!(MC_CH_MASK_CHECK(Outputs->ValidChBitMask, Controller, Channel, Outputs->MaxChannels))) {
          continue;
        }
        for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
          if (!MrcByteExist(MrcData, Controller, Channel, Byte)) {
            continue;
          }
          MrcGetSetChStrb(MrcData, Controller, Channel, Byte, GsmIocInternalClocksOn, ForceWriteCached, &GetSetVal);
        }
      }
    }

    MrcGetSetNoScope(MrcData, GsmIocForceCmpUpdt, ForceWriteUncached, &GetSetVal);

    GetSetVal = 0;
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        if (!(MC_CH_MASK_CHECK(Outputs->ValidChBitMask, Controller, Channel, Outputs->MaxChannels))) {
          continue;
        }
        for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
          if (!MrcByteExist(MrcData, Controller, Channel, Byte)) {
            continue;
          }
          MrcGetSetChStrb(MrcData, Controller, Channel, Byte, GsmIocInternalClocksOn, ForceWriteCached, &GetSetVal);
        }
      }
    }

    ForceRcomp(MrcData);
    return TRUE;
  }

  return FALSE;
}

/**
  This function looks at the margin values stored in the global data structure and checks
  WrT, WrV, RdT, and RdV to see if they are above the minimum margin required.

  @param[in, out] MrcData - MRC global data.

  @retval mrcSuccess - If margins are acceptable.
  @retval mrcRetrain - If margins are not acceptable.
**/
MrcStatus
MrcRetrainMarginCheck (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcDebug                *Debug;
  MrcOutput               *Outputs;
  MRC_FUNCTION            *MrcCall;
  MRC_MarginTypes         MarginParam;
  MrcMarginResult         LastResultParam;
  MrcStatus               Status;
  MRC_MARGIN_LIMIT_TYPE   MarginLimitType;
  UINT16                  (*LastMargins)[MAX_RANK_IN_CHANNEL][MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_EDGES];
  UINT32                  BERStats[4];
  UINT16                  MinEdgeMargin[MAX_EDGES];
  UINT16                  RetrainMarginLimit;
  UINT16                  CurrentMargin;
  UINT8                   Controller;
  UINT8                   Channel;
  UINT8                   McChMask;
  UINT8                   Rank;
  UINT8                   RankMask;
  UINT8                   Edge;
  UINT8                   Loopcount;
  UINT8                   MaxMargin;
  BOOLEAN                 RdWrCmdMarginFail[NUM_RW_CMD_MARGIN_FAILURES];

  MrcCall             = MrcData->Inputs.Call.Func;
  Outputs             = &MrcData->Outputs;
  Debug               = &Outputs->Debug;
  LastMargins         = Outputs->MarginResult;
  Status              = mrcSuccess;
  Loopcount           = 17;
  MrcCall->MrcSetMem ((UINT8 *) BERStats, sizeof (BERStats), 0);
  MrcCall->MrcSetMem ((UINT8 *) RdWrCmdMarginFail, sizeof (RdWrCmdMarginFail), FALSE);
  MaxMargin         = 0;
  // Loop is dependent on the order of MRC_MarginTypes.  If this changes, please ensure functionality
  // stays the same.
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Loopcount: %d\n", Loopcount);
  // @todo Update with McChBitMask
  SetupIOTestBasicVA (MrcData, Outputs->ValidChBitMask, Loopcount, NSOE, 0, 0, 8, PatWrRd, 0, 0);
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
      RankMask    = MRC_BIT0 << Rank;
      McChMask = 0;
      for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
        McChMask |= SelectReutRanks (MrcData, Controller, Channel, RankMask, FALSE, 0);
      }

      if (McChMask == 0) {
        continue;
      }

      for (MarginParam = RdT; MarginParam <= CmdV; MarginParam++) {
        switch (MarginParam) {
          case RdT:
            MaxMargin = MAX_POSSIBLE_TIME;
            break;

          case CmdT:
          case WrT:
            Outputs->DQPat = BasicVA;
            MaxMargin = (CmdT == MarginParam) ? MAX_CCC_POSSIBLE_TIME : MAX_POSSIBLE_TIME;
            break;

          case CmdV:
          case RdV:
          case WrV:
            Outputs->DQPat = BasicVA;
            MaxMargin = GetVrefOffsetLimits (MrcData, MarginParam);
            break;

          default:
            // Skip margin parameter.
            continue;
        }

        Status = MrcGetBERMarginCh (
          MrcData,
          LastMargins,
          McChMask,
          RankMask,
          RankMask,
          MarginParam,
          0,
          1,
          MaxMargin,
          0,
          BERStats
          );
      }
    }
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Margins\nParams: RdT\tWrT\tRdV\tWrV\tCmdT\tCmdV\n\tLft Rgt Lft Rgt Low Hi  Low Hi  Lft Rgt Low Hi");
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }

      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (!MrcRankExist (MrcData, Controller, Channel, Rank)) {
          continue;
        }

        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nCo%uCh%uRa%u\t", Controller, Channel, Rank);
        for (MarginParam = RdT; MarginParam <= CmdV; MarginParam++) {
          if (MarginParam != RdT && MarginParam != RdV && MarginParam != WrT && MarginParam != WrV && MarginParam != CmdT && MarginParam != CmdV) {
            continue;
          }

          LastResultParam     = GetMarginResultType (MarginParam);
          RetrainMarginLimit  = MrcGetUpmPwrLimit (MrcData, MarginParam, RetrainLimit) / 10;
          MrcCall->MrcSetMemWord (MinEdgeMargin, MAX_EDGES, (UINT16) (~0));

          for (Edge = 0; Edge < MAX_EDGES; Edge++) {
            CurrentMargin       = LastMargins[LastResultParam][Rank][Controller][Channel][0][Edge] / 10;
            MinEdgeMargin[Edge] = MIN (MinEdgeMargin[Edge], CurrentMargin);
            if ((CurrentMargin <= RetrainMarginLimit)) {
              Status =  mrcRetrain;
              if ((MarginParam == RdT) || (MarginParam == RdV)) {
                RdWrCmdMarginFail[0] = TRUE;
              } else if ((MarginParam == WrT) || (MarginParam == WrV)) {
                RdWrCmdMarginFail[1] = TRUE;
              } else if ((MarginParam == CmdT) || (MarginParam == CmdV)) {
                RdWrCmdMarginFail[2] = TRUE;
              }
            }
          }
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%2d  %2d\t", MinEdgeMargin[0], MinEdgeMargin[1]);
          if ((RdWrCmdMarginFail[0] == TRUE) && (RdWrCmdMarginFail[1] == TRUE) && (RdWrCmdMarginFail[2] == TRUE)) {
            Rank    = MAX_RANK_IN_CHANNEL;
            Channel = MAX_CHANNEL;
            break;
          }
        }
      }
    }
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");  // End of table

  if (Status == mrcRetrain) {
    // Loop is dependent on the order of MRC_MarginTypes.  If this changes, please ensure functionality
    // stays the same.
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "*** Margin Limit Check Failed! ***\nNew Limits:\nParam\tUPM\tPWR");
    for (MarginParam = RdT; MarginParam <= CmdV; MarginParam++) {
      if (((RdWrCmdMarginFail[0] == FALSE) && ((MarginParam == RdT) || (MarginParam == RdV))) ||
          ((RdWrCmdMarginFail[1] == FALSE) && ((MarginParam == WrT) || (MarginParam == WrV))) ||
          ((RdWrCmdMarginFail[2] == FALSE) && ((MarginParam == CmdT) || (MarginParam == CmdV))) ||
          (MarginParam != RdT && MarginParam != RdV && MarginParam != WrT && MarginParam != WrV && MarginParam != CmdT && MarginParam != CmdV)) {
        continue;
      }
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n%s", gMarginTypesStr[MarginParam]);
      for (MarginLimitType = UpmLimit; MarginLimitType < RetrainLimit; MarginLimitType++) {
        RetrainMarginLimit = MrcUpdateUpmPwrLimits (MrcData, MarginParam, MarginLimitType, MRC_UPM_PWR_INC_VAL);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t%d", RetrainMarginLimit);
      }
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");  // End of table.
  }

  return Status;
}

/**
  Calculate the VDDQ limits allowed by the LPDDR overshoot equations by sweep index

  @param[in] MrcData  - Include all MRC global data.

  @retval MrcStatus - if it succeed return mrcSuccess
**/
extern
MrcStatus
CalcVDDQLimits(
  IN MrcParameters *const MrcData,
  OUT UINT16       *const Limits
)
{
  Limits[0] = 0;
  Limits[1] = 0xFFFF;

  // TGL_POWER_TRAINING_VDDQ - Need to fill this out if VDDQ training is implemented. No guardband is required here.
  return mrcSuccess;
}

/**
  Calculate the RTT write limits allowed by the LPDDR overshoot equations in ohms.
  We can ignore PHY ODT digital offsets here if DRAM ODT is trained before PHY ODT.

  @param[in] MrcData  - Include all MRC global data.

  @retval MrcStatus - if it succeed return mrcSuccess
**/
extern
MrcStatus
CalcRttWriteLimits(
  IN MrcParameters *const MrcData,
  OUT UINT16       *const Limits
)
{
  MrcInput            *Inputs;
  MrcOutput           *Outputs;
  INT64               GetSetVal;
  UINT32              FirstController;
  UINT32              FirstChannel;
  UINT16              RTT_WR;
  UINT16              Temp;
  UINT16              VLimit;

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  FirstController = (MrcControllerExist(MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = MrcData->Outputs.Controller[FirstController].FirstPopCh;
  Limits[0] = 0;
  Limits[1] = 0xFFFF;

  if (!Outputs->Lpddr) {
    return mrcSuccess;
  }

  MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsOdtParkMode, ReadFromCache | (MRC_POWER_TRAINING_DEBUG ? PrintValue : 0), &GetSetVal);

  if (GetSetVal == 1 || GetSetVal == 2) {
    MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsNParkLow, ReadFromCache | (MRC_POWER_TRAINING_DEBUG ? PrintValue : 0), &GetSetVal);

    if (GetSetVal == 1) {
      Temp = (7 * ((UINT16)Outputs->VccddqVoltage * TWO_DECIMAL_PLACES / VSWING - (UINT16)((3 * TWO_DECIMAL_PLACES) / 2))) / 10;
      RTT_WR = (Inputs->RcompTarget[RdOdt] * TWO_DECIMAL_PLACES) / Temp;
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRttWriteLimits lower limit Temp = %d, RTT_WR = %d\n", Temp, RTT_WR);
      }
      if ((Temp * RTT_WR) / TWO_DECIMAL_PLACES < 250) {
        Limits[0] = RTT_WR; // No guardband is required here
      }
    } else {
      VLimit = MrcGetVLimit(MrcData);
      Limits[1] = (Inputs->RcompTarget[RdOdt] * VLimit - (UINT16)Outputs->VccddqVoltage * Z_O) / ((UINT16)Outputs->VccddqVoltage - VLimit); // No guardband is required here
      Temp = (7 * ((UINT16)Outputs->VccddqVoltage * TWO_DECIMAL_PLACES / VSWING - (UINT16)(1 * TWO_DECIMAL_PLACES))) / 10;
      RTT_WR = (Inputs->RcompTarget[RdOdt] * TWO_DECIMAL_PLACES) / Temp;
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRttWriteLimits lower limit Temp = %d, RTT_WR = %d\n", Temp, RTT_WR);
      }
      if ((Temp * RTT_WR) / TWO_DECIMAL_PLACES < 250) {
        Limits[0] = RTT_WR; // No guardband is required here
      }
    }
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRttWriteLimits before flipped limit check Limits[0] = %d, Limits[1] = %d\n", Limits[0], Limits[1]);
    }
    if (Limits[1] < Limits[0]) {
      Limits[0] = (Limits[0] + Limits[1]) / 2;
      Limits[1] = Limits[0];
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRttWriteLimits calculated a lower limit higher than the upper limit! LPDDR Overshoot violated!");
    }
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRttWriteLimits final Limits[0] = %d, Limits[1] = %d\n", Limits[0], Limits[1]);
    }
  }

  return mrcSuccess;
}

/**
  Calculate the Rx comp vref limits allowed by the LPDDR overshoot equations in ohms

  @param[in] MrcData  - Include all MRC global data.

  @retval MrcStatus - if it succeed return mrcSuccess
**/
extern
MrcStatus
CalcRxCompVrefLimits(
  IN MrcParameters *const MrcData,
  IN BOOLEAN       UseTable,
  OUT UINT16       *const Limits
)
{
  MrcOutput           *Outputs;
  MrcChannelOut       *ChannelOut;
  INT64               GetSetVal;
  UINT32              FirstController;
  UINT32              FirstChannel;
  UINT16              RTT_WR_Min = 0xFFFF;
  UINT16              RTT_WR_Max = 0;
  INT16               RTT_WR_Temp;
  UINT16              RxOdt;
  UINT16              RxOdt2;
  UINT16              VLimit;
  UINT8               Controller;
  UINT8               Channel;

  Outputs = &MrcData->Outputs;
  FirstController = (MrcControllerExist(MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = MrcData->Outputs.Controller[FirstController].FirstPopCh;
  Limits[0] = 0;
  Limits[1] = 0xFFFF;

  if (!Outputs->Lpddr) {
    return mrcSuccess;
  }

  MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsOdtParkMode, ReadFromCache | (MRC_POWER_TRAINING_DEBUG ? PrintValue : 0), &GetSetVal);

  if (GetSetVal == 1 || GetSetVal == 2) {
    if (UseTable) {
      RTT_WR_Min = MrcGetLpddrDqOdtTableValue(MrcData);
      RTT_WR_Max = RTT_WR_Min;
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits UseTable = 1, RTT_WR_Max & RTT_WR_Min = %d\n", RTT_WR_Max);
      }
    } else {
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
          if (!(MrcChannelExist(MrcData, Controller, Channel))) {
            continue;
          }
          ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
          RTT_WR_Temp = LpddrOdtDecode(0x7 & (ChannelOut->Dimm[dDIMM0].Rank[0].MR[MrcGetDqOdtMrIndex(MrcData)]));
          if (RTT_WR_Temp == -2) {
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_ERROR, "Invalid DIMM 0 RANK 0 LPDDR Dq ODT MR value! %d\n");
            return mrcFail;
          }
          if (MRC_POWER_TRAINING_DEBUG) {
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits MR11 = %d\n", ChannelOut->Dimm[dDIMM0].Rank[0].MR[MrcGetDqOdtMrIndex(MrcData)]);
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits RTT_WR_Temp = %d\n", RTT_WR_Temp);
          }
          if (RTT_WR_Temp > RTT_WR_Max) {
            RTT_WR_Max = RTT_WR_Temp;
          }
          if (RTT_WR_Temp < RTT_WR_Min) {
            RTT_WR_Min = RTT_WR_Temp;
          }
        }
      }
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits RTT_WR_Max = %d, RTT_WR_Min: %d\n", RTT_WR_Max, RTT_WR_Min);
      }
    }

    MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsNParkLow, ReadFromCache | (MRC_POWER_TRAINING_DEBUG ? PrintValue : 0), &GetSetVal);

    if (GetSetVal == 1) {
      RxOdt = (7 * RTT_WR_Min * ((UINT16)Outputs->VccddqVoltage * TWO_DECIMAL_PLACES / VSWING - (UINT16)((3 * TWO_DECIMAL_PLACES) / 2))) / (TWO_DECIMAL_PLACES * 10);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits upper limit RxOdt = %d\n", RxOdt);
      }
      Limits[1] = MIN(250, RxOdt); // No guardband is required here
    } else {
      VLimit = MrcGetVLimit(MrcData);
      RxOdt = ((((UINT16)Outputs->VccddqVoltage * TWO_DECIMAL_PLACES) / VLimit * (RTT_WR_Max + Z_O)) - RTT_WR_Max * TWO_DECIMAL_PLACES) / TWO_DECIMAL_PLACES;
      RxOdt2 = (Z_O * ((2 * (UINT16)Outputs->VccddqVoltage * TWO_DECIMAL_PLACES) / VLimit - 1 * TWO_DECIMAL_PLACES)) / TWO_DECIMAL_PLACES;
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits lower limit RxOdt = %d, RxOdt2 = %d\n", RxOdt, RxOdt2);
      }
      Limits[0] = MAX(RxOdt, RxOdt2);

      RxOdt = (7 * RTT_WR_Min * ((UINT16)Outputs->VccddqVoltage * TWO_DECIMAL_PLACES / VSWING - (UINT16)(1 * TWO_DECIMAL_PLACES))) / (TWO_DECIMAL_PLACES * 10);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits upper limit RxOdt = %d\n", RxOdt);
      }
      Limits[1] = MIN(250, RxOdt); // No guardband is required here
    }

    if (Limits[1] < Limits[0]) {
      Limits[0] = (Limits[0] + Limits[1]) / 2;
      Limits[1] = Limits[0];
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits calculated a lower limit higher than the upper limit! LPDDR Overshoot violated!");
    }
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxCompVrefLimits Limits[0] = %d, Limits[1] = %d\n", Limits[0], Limits[1]);
    }
  }

  return mrcSuccess;
}

/**
  Calculate the RxDqs comp offset limits allowed by the LPDDR overshoot equations by sweep index

  @param[in] MrcData  - Include all MRC global data.

  @retval MrcStatus - if it succeed return mrcSuccess
**/
extern
MrcStatus
CalcRxDqsCompOffsetLimits(
  IN MrcParameters *const MrcData,
  OUT INT8         *const Limits
  )
{
  MrcInput            *Inputs;
  MrcOutput           *Outputs;
  UINT32              FirstController;
  UINT32              FirstChannel;
  UINT32              Rleg;
  INT64               StrobeOdtStaticDis;
  INT64               RcompOdt;
  UINT16              RxCompVrefLimits[2];

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  FirstController = (MrcControllerExist(MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = MrcData->Outputs.Controller[FirstController].FirstPopCh;

  if (!Outputs->Lpddr) {
    return mrcSuccess;
  }

  CalcRxCompVrefLimits(MrcData, FALSE, RxCompVrefLimits); // Find limits in Ohms

  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxDqsCompOffsetLimits Limits[0] = %d, Limits[1] = %d\n", RxCompVrefLimits[0], RxCompVrefLimits[1]);
    MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "\nCalcRxDqsCompOffsetLimits RcompTarget[RdOdt] = %d\n", Inputs->RcompTarget[RdOdt]);
  }

  MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocStrobeOdtStaticDis, ReadFromCache | (MRC_POWER_TRAINING_DEBUG ? PrintValue : 0), &StrobeOdtStaticDis);
  MrcGetSetNoScope(MrcData, (Outputs->OdtMode == MrcOdtModeVss) ? CompRcompOdtDn : CompRcompOdtUp, ReadUncached | (MRC_POWER_TRAINING_DEBUG ? PrintValue : 0), &RcompOdt);

  Rleg = Inputs->RcompTarget[RdOdt] * (23 * (!StrobeOdtStaticDis) + (UINT32)RcompOdt);
  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "\nCalcRxDqsCompOffsetLimits Rleg = %d\n", Rleg);
  }

  // High PHY impedance limit dictates low digital offset limit and vice versa
  if (RxCompVrefLimits[1] == 0) {
    Limits[0] = 0x7F;
  } else {
    Limits[0] = (INT8)(((INT32)Rleg - (23 + (INT32)RcompOdt) * (INT32)RxCompVrefLimits[1]) / (INT32)RxCompVrefLimits[1]); // No guardband is required here
  }
  if (RxCompVrefLimits[0] == 0) {
    Limits[1] = 0x7F;
  } else {
    Limits[1] = (INT8)(((INT32)Rleg - (23 + (INT32)RcompOdt) * (INT32)RxCompVrefLimits[0]) / (INT32)RxCompVrefLimits[0]); // No guardband is required here
  }

  if (Limits[1] < Limits[0]) {
    Limits[0] = 0;
    Limits[1] = 0;
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxDqsCompOffsetLimits calculated a lower limit higher than the upper limit! LPDDR Overshoot violated!");
  }

  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxDqsCompOffsetLimits Limits[0] = %d, Limits[1] = %d\n", Limits[0], Limits[1]);
  }

  return mrcSuccess;
}

/**
  This function calculates the average Rx ODT impedance for the given channel (in ohms).

  @param[in]  MrcData       - Pointer to global MRC data.
  @param[in]  Controller    - 0-based index to controller
  @param[in]  Channel       - 0-based index to legacy channel (x64 bit)
  @param[out] ImpedenceOhms - Pointer to impedance return value

  @retval mrcSuccess Successfully calculated impedance
**/
MrcStatus
CalcRxDqOdtAverageByteImpedance(
  IN  MrcParameters *const MrcData,
  IN  UINT8                Controller,
  IN  UINT8                Channel,
  OUT UINT16        *const ImpedenceOhms
)
{
  MrcOutput          *Outputs;
  MrcInput           *Inputs;
  MrcChannelOut      *ChannelOut;
  const MRC_FUNCTION *MrcCall;
  INT64          GetSetVal;
  INT16          AverageRxDqOdtCompOffset = 0;
  UINT8          CompVref;
  INT64          MinCompVref;
  INT64          MaxCompVref;
  UINT8          AdjustedCompVref;
  UINT32         AdjustedRxDqOdtTarget;
  UINT32         RxDqDrvUpTarget;
  UINT8          FirstChannel;
  UINT8          Byte;
  UINT8          Bytes = 0;
  UINT8          Rank;
  INT8           NewCompUp;
  INT8           NewCompDn;
  INT8           NewComp = 0;
  INT8           CurrentComp;
  INT8           MinDelta;
  INT8           CurrentDelta;
  INT64          MaxComp;
  INT64          MinComp;
  UINT8          NewCompVref;
  INT8           Sign;
  UINT8          Offset = 1;
  UINT32         FirstController;
  UINT8          ODTSingleSegEn;
  UINT64         Timeout;
  UINT32         Delay;
  DATA0CH0_CR_DDRCRDATAOFFSETCOMP_STRUCT DdrCrDataOffsetComp;
  MrcStatus      Status;
  UINT32         NewCompValue;

  Outputs      = &MrcData->Outputs;
  Inputs       = &MrcData->Inputs;
  MrcCall      = Inputs->Call.Func;
  ChannelOut   = &Outputs->Controller[Controller].Channel[Channel];

  if (NULL == ImpedenceOhms) {
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "%s: ImpedenceOhms argument\n", gNullPtrErrStr);
    return mrcWrongInputParameter;
  }

  MrcGetSetLimits(MrcData, DqOdtVrefUp, &MinCompVref, &MaxCompVref, &Delay);
  MrcGetSetLimits(MrcData, CompRcompOdtUp, &MinComp, &MaxComp, &Delay);

  FirstController = (MrcControllerExist(MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = MrcData->Outputs.Controller[FirstController].FirstPopCh;
  MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataODTSingleSegEn, ReadFromCache, &GetSetVal);
  ODTSingleSegEn = (GetSetVal == 1) ? 1 : 0;;

  MrcGetSetNoScope(MrcData, (Outputs->OdtMode == MrcOdtModeVss) ? DqOdtVrefDn : DqOdtVrefUp, ReadFromCache, &GetSetVal);
  CompVref = (UINT8)GetSetVal;
  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\nCalcRxDqOdtAverageByteImpedance initial CompVref: %d\n", CompVref);
  }

  for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
    if (!(MrcRankExist(MrcData, Controller, Channel, Rank))) {
      continue;
    }
    for (Byte = 0; Byte < MAX_SDRAM_IN_DIMM; Byte++) {
      if (!MrcByteExist(MrcData, Controller, Channel, Byte)) {
        continue;
      }
      DdrCrDataOffsetComp.Data = ChannelOut->DataCompOffset[Byte];
      AverageRxDqOdtCompOffset += (INT16)MrcSE((UINT16)DdrCrDataOffsetComp.Bits.DqOdtCompOffset, DATA0CH0_CR_DDRCRDATAOFFSETCOMP_DqOdtCompOffset_WID, 16);
      Bytes++;
    }
  }
  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance total AverageRxDqOdtCompOffset: %d\n", AverageRxDqOdtCompOffset);
  }
  if (Bytes) {
    AverageRxDqOdtCompOffset /= Bytes;
  }
  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance average AverageRxDqOdtCompOffset: %d\n", AverageRxDqOdtCompOffset);
  }

  // Now, we need to know how much the base impedance value should be adjusted due to the average RxDq ODT Comp Offset
  Sign = (AverageRxDqOdtCompOffset < 0) ? -1 : 1;
  Sign *= (Outputs->OdtMode == MrcOdtModeVss) ? -1 : 1; // Using ODT Comp Vref down flips the sign

  AdjustedCompVref = CompVref;
  MinDelta = (INT8)ABS(AverageRxDqOdtCompOffset);
  CurrentComp = GetCompCode(MrcData, OptRdDqOdt, COMP_UP);
  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance current comp up: %d\n", CurrentComp);
  }
  CurrentComp = (CurrentComp == -1) ? GetCompCode(MrcData, OptRdDqOdt, COMP_DN) : CurrentComp;
  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance current comp down: %d\n", CurrentComp);
  }

  if(AverageRxDqOdtCompOffset != 0) { // Loop until we find a break condition as long as the average comp offset is not 0
    Timeout = MrcCall->MrcGetCpuTime() + 10000; // 10 seconds timeout
    while (1) {
      if ((MrcCall->MrcGetCpuTime() >= Timeout)) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_ERROR, "\nTimeout in CalcRxDqOdtAverageByteImpedance! \n");
        return mrcFail;
      }
      NewCompVref = CompVref + (Sign * Offset);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance new CompVref: %d\n", NewCompVref);
      }
      if ((MinCompVref > NewCompVref) || (NewCompVref > MaxCompVref)) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_WARNING, "Warning! We saturated the RxDq ODT comp vref before we could shift it enough to make the comp shift match the average comp offset. Impedance will only be partially accurate! \n");
        break;
      }

      Status = UpdateCompGlobalOffset(MrcData, (Outputs->OdtMode == MrcOdtModeVss) ? RdOdtDn : RdOdtUp, NewCompVref, FALSE, FALSE, &NewCompValue);
      if (mrcSuccess != Status) {
        MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "UpdateCompGlobalOffset() error %d\n", Status);
        return mrcFail;
      }
      NewCompUp = GetCompCode(MrcData, OptRdDqOdt, COMP_UP);
      NewCompDn = GetCompCode(MrcData, OptRdDqOdt, COMP_DN);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance new comp up: %d\n", NewCompUp);
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance new comp down: %d\n", NewCompDn);
      }
      if (NewCompUp != -1) {
        if ((RESERVED_COMP_CODES > NewCompUp) || (NewCompUp > (MaxComp - RESERVED_COMP_CODES))) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_WARNING, "Warning! We saturated the RxDq ODT comp up before we could shift it as far as the average comp offset. Impedance will only be partially accurate! \n");
          break;
        }
        NewComp = NewCompUp;
      }
      if (NewCompDn != -1) {
        if ((RESERVED_COMP_CODES > NewCompDn) || (NewCompDn > (MaxComp - RESERVED_COMP_CODES))) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_WARNING, "Warning! We saturated the RxDq ODT comp down before we could shift it as far as the average comp offset. Impedance will only be partially accurate! \n");
          break;
        }
        NewComp = NewCompDn;
      }

      // Move the Comp Vref until the comp output changes an amount equivalent to the average comp offset
      CurrentDelta = ABS(CurrentComp + (INT8)AverageRxDqOdtCompOffset - NewComp);
      if (CurrentDelta <= MinDelta) {
        MinDelta = CurrentDelta;
        AdjustedCompVref = NewCompVref;
        if (MinDelta == 0) {
          if (MRC_POWER_TRAINING_DEBUG) {
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance (MinDelta == 0) \n");
          }
          break; // We're done
        }
      } else {
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance (CurrentDelta > MinDelta) CurrentDelta = %d, MinDelta = %d \n", CurrentDelta, MinDelta);
        }
        break; // We overshot, so we're done
      }

      Offset++;
    }

    // Restore original comp vref value
    Status = UpdateCompGlobalOffset(MrcData, (Outputs->OdtMode == MrcOdtModeVss) ? RdOdtDn : RdOdtUp, CompVref, FALSE, TRUE, &NewCompValue);
    if (mrcSuccess != Status) {
      MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "UpdateCompGlobalOffset() error %d\n", Status);
      return mrcFail;
    }
  }

  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance AdjustedCompVref: %d\n", AdjustedCompVref);
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance ODTSingleSegEn: %d\n", ODTSingleSegEn);
  }

  // DqOdtVrefUp = 191 * (100 Ohms / (100 Ohms + RxDqOdtUpTarget));  If !ODTSingleSegEn, compensation programs DqOdtVref using (2 * RxDqOdtUpTarget).
  // DqOdtVrefDn = 191 * (RxDqOdtDnTarget / (RxDqOdtDnTarget + RxDqDrvUpTarget)); If !ODTSingleSegEn, compensation programs DqOdtVref using (2 * RxDqOdtUpTarget) and (2 * RxDqDrvUpTarget), which cancels out.
  // DqDrvVrefUp = 191 * (100 Ohms / (100 Ohms + RxDqDrvUptarget));
  if (Outputs->OdtMode == MrcOdtModeVss) {
    MrcGetSetNoScope(MrcData, DqDrvVrefUp, ReadFromCache, &GetSetVal);
    CompVref = (UINT8)GetSetVal;
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance RxDqDrvUpCompVref: %d\n", CompVref);
    }
    RxDqDrvUpTarget = ((MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor * TWO_DECIMAL_PLACES) / CompVref - Inputs->RcompResistor * TWO_DECIMAL_PLACES) / (TWO_DECIMAL_PLACES * TWO_SEGMENTS);
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance RxDqDrvUpTarget: %d\n", RxDqDrvUpTarget);
    }
    AdjustedRxDqOdtTarget = (RxDqDrvUpTarget * AdjustedCompVref) / (MRC_COMP_VREF_STEP_SIZE - AdjustedCompVref);
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance AdjustedRxDqOdtTarget: %d\n", AdjustedRxDqOdtTarget);
    }
  } else {
    AdjustedRxDqOdtTarget = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor * TWO_DECIMAL_PLACES) / ((Outputs->OdtMode == MrcOdtModeVtt ? 2 : 1) * AdjustedCompVref) - Inputs->RcompResistor * TWO_DECIMAL_PLACES; // * 100 to preserve 2 decimal places of accuracy
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance AdjustedRxDqOdtTarget Step 1: %d\n", AdjustedRxDqOdtTarget);
    }
    AdjustedRxDqOdtTarget = AdjustedRxDqOdtTarget / (TWO_DECIMAL_PLACES * (!ODTSingleSegEn + 1));
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "CalcRxDqOdtAverageByteImpedance AdjustedRxDqOdtTarget Final: %d\n", AdjustedRxDqOdtTarget);
    }
  }

  *ImpedenceOhms = (UINT16) AdjustedRxDqOdtTarget; // Actual RxDqOdt impedance value in ohms
  return mrcSuccess;
}

/**
  This function is responsible for updating the SOC ODT value in LPDDR4/5 memory based on the CPU ODT settings.
  This reads the CPU ODT using the current comp settings, find the closest SOC ODT settings and programs it to the corresponding MR.
  Note - calculating CPU impedance assumes the cached value holds the current settings.

  @param[in]  MrcData - Pointer to global MRC data.

  @retval -Nothing.
**/
void
LpddrUpdateSocOdt(
  IN MrcParameters *const MrcData
  )
{
  UINT8           Channel;
  UINT8           Controller;
  UINT16          CpuImpedance;
  UINT16          SocOdt;
  MrcStatus       Status;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (!(MrcChannelExist(MrcData, Controller, Channel))) {
        continue;
      }

      Status = CalcRxDqOdtAverageByteImpedance(MrcData, Controller, Channel, &CpuImpedance);
      if (mrcSuccess != Status) {
        MRC_DEBUG_MSG (&MrcData->Outputs.Debug, MSG_LEVEL_ERROR, "CalcRxDqOdtAverageByteImpedance() error %d\n", Status);
      }
      SocOdt = MrcCheckForSocOdtEnc(MrcData, CpuImpedance);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "LpddrUpdateSocOdt: CpuImpedance = %d, SocOdt = %d \n", CpuImpedance, SocOdt);
      }
      SetDimmParamValue(MrcData, Controller, Channel, 0xF, OptDimmSocOdt, SocOdt, TRUE);
    }
  }
}

/**
  Print DIMM ODT / Ron values per DIMM.
  Also print CPU ODT / Ron.

  @param[in] MrcData  - Include all MRC global data.

  @retval none
**/
extern
void
MrcPrintDimmOdtValues (
  IN MrcParameters *const MrcData
  )
{
#ifdef MRC_DEBUG_PRINT
  MrcOutput     *Outputs;
  MrcChannelOut *ChannelOut;
  MrcDebug      *Debug;
  UINT32        Controller;
  UINT32        Channel;
  UINT8         RankMask;
  UINT8         Rank;
  UINT8         Dimm;
  BOOLEAN       Ddr4;
  UINT16        OdtWrite;
  UINT16        OdtNom;
  UINT16        OdtPark;
  UINT16        OdtCA;
  UINT16        DimmRon;
  UINT16        CpuOdt[MAX_CHANNEL];
  //UINT16        CpuRon[MAX_CHANNEL];
  MrcStatus     Status;

  Outputs = &MrcData->Outputs;
  Debug   = &Outputs->Debug;
  Ddr4    = (Outputs->DdrType == MRC_DDR_TYPE_DDR4);

//   DIMM ODT summary:
//
//              Ron     OdtWr   OdtNom  OdtPark (DDR4)
//              Ron     OdtDQ   OdtCA           (LPDDR4)
//   ---------------------------------------
//   Mc0C0D0:   48      Hi-Z    48      80
//   Mc0C0D1:   48      240     60      120
//   ---------------------------------------
//   Mc1C0D0:   48      48      34      34
//   Mc1C0D1:   48      48      34      34
//   ---------------------------------------
//
//   CPU Summary: Ron = 56, Read ODT = 109
//
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nDIMM ODT summary:\n\t\tRon\t%s\n", Ddr4 ? "OdtWr\tOdtNom\tOdtPark" : "OdtDQ\tOdtCA");
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "---------------------------------------\n");
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      CpuOdt[Channel] = 0;
      if (!(MC_CH_MASK_CHECK(Outputs->ValidChBitMask, Controller, Channel, Outputs->MaxChannels))) {
        continue;
      }
      ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
      for (Dimm = 0; Dimm < MAX_DIMMS_IN_CHANNEL; Dimm++) {
        RankMask = DIMM_TO_RANK_MASK (Dimm);
        Rank = Dimm * 2;
        if ((RankMask & ChannelOut->ValidRankBitMask) == 0) {
          continue;
        }
        OdtNom   = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtNom,  FALSE, 0, FALSE);
        OdtPark  = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtPark, FALSE, 0, FALSE);
        OdtCA    = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtCA,   FALSE, 0, FALSE) * ((Outputs->Lpddr) ? 2 : 1);
        OdtWrite = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmOdtWr,   FALSE, 0, FALSE);
        DimmRon  = CalcDimmImpedance (MrcData, Controller, Channel, Rank, OptDimmRon,     FALSE, 0, FALSE);
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Mc%dC%dD%d:\t%d\t", Controller, Channel, Dimm, DimmRon);
        if ((OdtWrite == 0xFFFF) || (OdtWrite == 0)) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, ((OdtWrite == 0xFFFF) && Ddr4) ? "Hi-Z\t" : "Off\t");
        } else {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", OdtWrite);
        }
        if (Ddr4) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, (OdtNom == 0xFFFF) ? "Hi-Z\t" : "%d\t", OdtNom);
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, (OdtPark == 0xFFFF) ? "Hi-Z\n" : "%d\n", OdtPark);
        } else {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, (OdtCA == 0xFFFF) ? "Off\t" : "%d\t", OdtCA);
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");
        }
      } // for Dimm
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "---------------------------------------\n");
      Status = CalcRxDqOdtAverageByteImpedance (MrcData, (UINT8)Controller, (UINT8)Channel, &CpuOdt[Channel]);
      if (mrcSuccess != Status) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "CalcRxDqOdtAverageByteImpedance() error %d\n", Status);
      }
    } // for Channel

    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!(MC_CH_MASK_CHECK(Outputs->ValidChBitMask, Controller, Channel, Outputs->MaxChannels))) {
        continue;
      }
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "CPU Summary: MC = %d Channel = %d Read ODT = %d\n", Controller, Channel, CpuOdt[Channel]);
    }
  } // for Controller

#endif // MRC_DEBUG_PRINT
}

/**
  Wrapper for DQTimeCentering1D taking duty cycle into account.

  @param[in,out] MrcData        - Include all MRC global data.
  @param[in]     ResetPerBit    - Option to Reset PerBit Deskew to middle value before byte training
  @param[in]     StepSize       - Step size
  @param[in]     loopcount      - loop count
  @param[in]     MsgPrint       - Show debug prints
  @param[in]     EarlyCentering - Execute as early centering routine

  @retval MrcStatus -  If succeeded, return mrcSuccess
**/
MrcStatus
DQTimeCentering1D_RxDC (
  IN OUT MrcParameters *const MrcData,
  IN     const UINT8          ResetPerBit,
  IN     const UINT8          StepSize,
  IN     const UINT8          loopcount,
  IN     BOOLEAN              MsgPrint,
  IN     BOOLEAN              EarlyCentering
  )
{
  DQTimeCentering1D (MrcData, RdTP, ResetPerBit, StepSize, loopcount, MsgPrint, EarlyCentering);
  DQTimeCentering1D (MrcData, RdTN, ResetPerBit, StepSize, loopcount, MsgPrint, EarlyCentering);

  return mrcSuccess;
}

/**
  This function wrap DimmODTCATraining routine.
  This step is for LPDDR only.
  We don't train 2D with CA DS because the training time would be too long, and we don't need the margin that badly.

  @param[in] MrcData  - Include all MRC global data.

  @retval MrcStatus - if it succeed return mrcSuccess
**/
MrcStatus
MrcDimmOdtCaTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  static const UINT8  TestList[] = { CmdV, CmdT };
  static const UINT8  OptParam[] = { OptDimmOdtCA };
  UINT8               Scale[] = { 1, 1, 0, 0, 0 };
  MrcOutput           *Outputs;
  UINT16              *DimmCaOdtVals;
  OptOffsetChByte     BestOff;
  INT8                Start;
  INT8                Stop;
  INT8                Off;

  Outputs = &MrcData->Outputs;

  if (!Outputs->Lpddr) {
    return mrcSuccess;
  }

  GetDimmOptParamValues (MrcData, OptParam[0], &DimmCaOdtVals, (UINT8 *) &Stop);
  Start = 0;
  Stop  = Stop - 1;

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\nDIMM CaOdt Offset - Value mapping \nOffset\t Value\n");
  for (Off = Start; Off < Stop + 1; Off++) {
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "%d:\t %u \n", Off, DimmCaOdtVals[(UINT8)Off]);
  }

  // TGL_POWER_TRAINING_DDR5 - if DDR5 is run in 2DPC, you may want to call this per DIMM and set CaVref per DIMM
  // Train CA ODT only
  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT(OptParam),
    FullGrid,
    TestList,
    ARRAY_COUNT(TestList),
    Scale,
    NULL,
    &Start,  // Start
    &Stop,  // Stop
    OPT_PARAM_LOOP_COUNT - 2,
    1,      // Repeats
    0,      // NoPrint
    0,      // SkipOdtUpdate
    0,      // GuardBand
    0,       // PatternType
    SaveMarginsArray
    );

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Cmd Vref\n");
  CmdVoltageCentering(MrcData, CCC_RE_CENTER_LOOP_COUNT, V_FINAL_STEP_SIZE, MRC_POWER_TRAINING_DEBUG);
  MrcResetSequence(MrcData);

  return mrcSuccess;
}

/**
  This function is the DimmODTTraining routine.

  @param[in] MrcData  - Include all MRC global data.

  @retval MrcStatus - if it succeed return mrcSuccess
**/
MrcStatus
MrcDimmOdtTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  static const UINT8  TestList[] = { WrV, WrT };
  static const UINT8  OptParam[] = { OptDimmOdtWr, OptWrDS };
  UINT8               Scale[] = { 1, 1, 0, 0, 0 };
  MrcOutput           *Outputs;
  UINT16              *DimmWrOdtVals;
  UINT16              LpddrOvershootLimits[] = { 0, 0 };
  OptOffsetChByte     BestOff;
  INT8                Start[ARRAY_COUNT (OptParam)];
  INT8                Stop[ARRAY_COUNT (OptParam)];
  INT8                Off;

  Outputs = &MrcData->Outputs;

  MrcDimmOdtParkNomTraining (MrcData); // Train RTT_PARK and RTT_NOM

  GetDimmOptParamValues (MrcData, OptParam[0], &DimmWrOdtVals, (UINT8 *)&Stop[0]);

 if (DimmWrOdtVals == NULL) {
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_ERROR, "Invalid pointer values\n");
    return mrcFail;
  }

  Start[0] = 0;
  Stop[0] = Stop[0] - 1;

  if (Outputs->Lpddr) {
    CalcRttWriteLimits (MrcData, LpddrOvershootLimits);

    while (LpddrOvershootLimits[0] > DimmWrOdtVals[(UINT8)(Stop[0])] && Stop[0] > 0) { // Stop if 'Stop' == 0, because we need at least one value to run
      Stop[0]--;
      MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Changing RTT_WR sweep limits due to LPDDR overshoot limits. 1 value cut from end of RTT_WR list.\n");
    }

    while (LpddrOvershootLimits[1] < DimmWrOdtVals[(UINT8)(Start[0])] && Start[0] < Stop[0]) { // Stop if 'Start' == 'Stop', because we need at least one value to run
      Start[0]++;
      MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Changing RTT_WR sweep limits due to LPDDR overshoot limits. 1 value cut from start of RTT_WR list.\n");
    }
  }

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\nDIMM WrOdt Offset - Value mapping \nOffset\t Value\n");
  for (Off = Start[0]; Off < Stop[0] + 1; Off++) {
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "%d:\t %u \n", Off, DimmWrOdtVals[(UINT8)Off]);
  }

  Start[1] = OptParamLimitValue (MrcData, OptParam[1], 0);
  Stop[1] = OptParamLimitValue (MrcData, OptParam[1], 1);

  // Train Wr ODT only
  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    FullGrid,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    Start,  // Start
    Stop,  // Stop
    OPT_PARAM_LOOP_COUNT,
    1,      // Repeats
    0,      // NoPrint
    0,      // SkipOdtUpdate
    0,      // GuardBand
    0,       // PatternType
    SaveMarginsArray
    );

  // Re-Center Write voltage and update Host Struct with new center
  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Vref\n");
  DQTimeCentering1D (MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  return mrcSuccess;
}

/**
  This function wrap DimmODTParkNomTraining routine.

  @param[in] MrcData  - Include all MRC global data.

  @retval MrcStatus - if it succeed return mrcSuccess
**/
MrcStatus
MrcDimmOdtParkNomTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[4][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  static const UINT8  TestList[] = { WrV, RdV, WrT, RdT };
  static const UINT8  OptParam1[] = { OptDimmOdtPark };
  static const UINT8  OptParam2[] = { OptDimmOdtNom };
  UINT8               Scale1[] = { 1, 1, 1, 1, 0 };
  UINT8               Scale2[] = { 1, 1, 1, 1, 0 };
  MrcOutput          *Outputs;
  MrcControllerOut   *ControllerOut;
  MrcChannelOut      *ChannelOut;
  OptOffsetChByte     BestOff;
  UINT16             *DimmOdtParkVals;
  UINT16             *DimmOdtNomVals;
  UINT8               Channel;
  UINT8               Controller;
  INT8                Start1;
  INT8                Stop1;
  INT8                Start2;
  INT8                Stop2;
  INT8                Off;
  UINT8               Channel_2DPC_Mask = 0;

  Outputs = &MrcData->Outputs;

  if ((Outputs->DdrType != MRC_DDR_TYPE_DDR4) || ((Outputs->DdrType == MRC_DDR_TYPE_DDR4) && (Outputs->OdtMode == MrcOdtModeVtt))) {
    return mrcSuccess;
  }

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    ControllerOut = &Outputs->Controller[Controller];
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!(MC_CH_MASK_CHECK (Outputs->ValidChBitMask, Controller, Channel, Outputs->MaxChannels))) {
        continue;
      }
      ChannelOut = &ControllerOut->Channel[Channel];

      if (ChannelOut->DimmCount == 2) {
        Channel_2DPC_Mask |= 1 << MC_CH_IDX ((Controller), (Channel), (Outputs->MaxChannels));
      }
    }
  }

  GetDimmOptParamValues (MrcData, OptParam1[0], &DimmOdtParkVals, (UINT8 *)&Stop1);
  Start1 = 0;
  Stop1 = Stop1 - 1;

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\nDIMM Park Odt Offset - Value mapping \nOffset\t Value\n");
  for (Off = Start1; Off < Stop1 + 1; Off++) {
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "%d:\t %u \n", Off, DimmOdtParkVals[(UINT8)Off]);
  }

  // Train RTT PARK
  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam1,
    ARRAY_COUNT (OptParam1),
    FullGrid,
    TestList,
    ARRAY_COUNT (TestList),
    Scale1,
    NULL,
    &Start1,  // Start
    &Stop1,  // Stop
    OPT_PARAM_LOOP_COUNT,
    FALSE,   // Repeats
    0,      // NoPrint
    0,      // SkipOdtUpdate
    0,      // GuardBand
    0,       // PatternType
    SaveMarginsArray
    );

  // Train RTT NOM
  if (Channel_2DPC_Mask > 0) {
    GetDimmOptParamValues (MrcData, OptParam2[0], &DimmOdtNomVals, (UINT8 *)&Stop2);
    Start2 = 0;
    Stop2 = Stop2 - 1;

    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\nDIMM Nom Odt Offset - Value mapping \nOffset\t Value\n");
    for (Off = Start2; Off < Stop2 + 1; Off++) {
      MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "%d:\t %u \n", Off, DimmOdtNomVals[(UINT8)Off]);
    }

    TrainDDROptParam (
      MrcData,
      &BestOff,
      Channel_2DPC_Mask,
      MrcData->Outputs.ValidRankMask,
      OptParam2,
      ARRAY_COUNT (OptParam2),
      FullGrid,
      TestList,
      ARRAY_COUNT (TestList),
      Scale2,
      NULL,
      &Start2,  // Start
      &Stop2,  // Stop
      OPT_PARAM_LOOP_COUNT,
      FALSE,      // Repeats
      0,      // NoPrint
      0,      // SkipOdtUpdate
      0,      // GuardBand
      0,       // PatternType
      SaveMarginsArray
      );
  }

  // Re-Center Write voltage and update Host Struct with new center
  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Vref\n");
  DQTimeCentering1D (MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  // Re-Center Read voltage and update Host Struct with new center
  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Vref\n");
  DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  return mrcSuccess;
}

/**
  This function implements Read Equalization training.

  @param[in] MrcData - Include all MRC global data.
  @param[in] NoPrint - Switch to disable printing.
  @param[in] FinalRecenter - Switch to disable final re-centering.
  @param[in] PerChannel - Switch to per-channel training.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
ReadEQTraining (
  IN MrcParameters *const MrcData,
  IN  BOOLEAN             NoPrint,
  IN  BOOLEAN             FinalRecenter,
  IN  BOOLEAN             PerChannel
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  UINT8               RankMask;
  static const UINT8  TestList[]  = { RdV, RdT };
  UINT8               *OptParam1 = 0x0;
  UINT8               *OptParam2 = 0x0;
  UINT8               OptParamCTLE1[] = { OptRxEq }; // May set RxEq constant and sweep R/C 2D
  UINT8               OptParamCTLE2[] = { OptRxC, OptRxR };
  UINT8               OptParamDFE1[] = { OptDFETap0 };
  UINT8               OptParamDFE2[] = { OptDFETap1 };
  UINT8               Scale[] = { 1, 1, 255, 0, 0 }; // Ignore power optimization, need a seperate scale array for each param
  OptOffsetChByte     BestOff;
  UINT8               GridMode1;
  UINT8               GridMode2;
  UINT8               OptParamLength1;
  UINT8               OptParamLength2;
  INT8                StartCTLE1[1];
  INT8                StopCTLE1[1];
  INT8                StartCTLE2[2];
  INT8                StopCTLE2[2];
  INT8                StartDFE1[1];
  INT8                StopDFE1[1];
  INT8                StartDFE2[1];
  INT8                StopDFE2[1];
  INT8                *Start1;
  INT8                *Stop1;
  INT8                *Start2;
  INT8                *Stop2;
  MrcOutput           *Outputs;

  Outputs = &MrcData->Outputs;
  if (Outputs->RxMode == MrcRxModeMatchedN || Outputs->RxMode == MrcRxModeMatchedP) {
      // CTLE
      OptParamLength1 = 1;
      OptParamLength2 = 2;
      GridMode1 = FullGrid;
      GridMode2 = FullGrid;
      OptParam1 = OptParamCTLE1;
      OptParam2 = OptParamCTLE2;

      StartCTLE1[0] = OptParamLimitValue (MrcData, OptParamCTLE1[0], 0);
      StopCTLE1[0] = OptParamLimitValue (MrcData, OptParamCTLE1[0], 1);
      StartCTLE2[0] = OptParamLimitValue (MrcData, OptParamCTLE2[0], 0);
      StopCTLE2[0] = OptParamLimitValue (MrcData, OptParamCTLE2[0], 1);
      StartCTLE2[1] = OptParamLimitValue(MrcData, OptParamCTLE2[1], 0);
      StopCTLE2[1] = OptParamLimitValue(MrcData, OptParamCTLE2[1], 1);
      Start1 = StartCTLE1;
      Stop1 = StopCTLE1;
      Start2 = StartCTLE2;
      Stop2 = StopCTLE2;
  } else {
      // DFE
      OptParamLength1 = 1;
      OptParamLength2 = 1;
      GridMode1 = FullGrid;
      GridMode2 = FullGrid;
      OptParam1 = OptParamDFE1;
      OptParam2 = OptParamDFE2;

      StartDFE1[0] = OptParamLimitValue (MrcData, OptParamDFE1[0], 0) * ((PerChannel == TRUE) ? 1 : DFE_STEP_PER_CHANNEL);
      StopDFE1[0] = OptParamLimitValue (MrcData, OptParamDFE1[0], 1) * ((PerChannel == TRUE) ? 1 : DFE_STEP_PER_CHANNEL);
      StartDFE2[0] = OptParamLimitValue (MrcData, OptParamDFE2[0], 0) * ((PerChannel == TRUE) ? 1 : DFE_STEP_PER_CHANNEL);
      StopDFE2[0] = OptParamLimitValue (MrcData, OptParamDFE2[0], 1) * ((PerChannel == TRUE) ? 1 : DFE_STEP_PER_CHANNEL);
      Start1 = StartDFE1;
      Stop1 = StopDFE1;
      Start2 = StartDFE2;
      Stop2 = StopDFE2;
  }

  // Function Call for RxEQ Training
  Outputs->PowerTrainingPerChannel = (FinalRecenter == FALSE) ? TRUE : FALSE;
  for (RankMask = 1; RankMask < (0x1 << MAX_RANK_IN_CHANNEL); RankMask <<= 1) {
    if (RankMask & MrcData->Outputs.ValidRankMask) {
      if (NoPrint == FALSE && PerChannel == FALSE) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\n *** Optimizing RankMask %x *** \n", RankMask);
      }
      Scale[ARRAY_COUNT(TestList)] = 255;
      TrainDDROptParam (
        MrcData,
        &BestOff,
        MrcData->Outputs.ValidChBitMask,
        (PerChannel == TRUE) ? MrcData->Outputs.ValidRankMask : RankMask,
        OptParam1,
        OptParamLength1,
        GridMode1,
        TestList,
        ARRAY_COUNT (TestList),
        Scale,
        NULL,
        Start1, // Start
        Stop1,  // Stop
        (FinalRecenter == TRUE) ? OPT_PARAM_LOOP_COUNT + 4 : OPT_PARAM_LOOP_COUNT + 2,
        1,      // Repeats
        NoPrint,// NoPrint
        0,      // SkipOptUpdate
        0,      // GuardBand
        0,      // PatType
        SaveMarginsArray
        );
      if (PerChannel == TRUE) {
        break;
      }
    }
  }
  if (Outputs->RxMode == MrcRxModeMatchedN || Outputs->RxMode == MrcRxModeMatchedP) {
    if (NoPrint == FALSE) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
    }
    DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    if (NoPrint == FALSE) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Rd Vref\n");
    }
    DQTimeCentering1D(MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);
  }
  for (RankMask = 1; RankMask < (0x1 << MAX_RANK_IN_CHANNEL); RankMask <<= 1) {
    if (RankMask & MrcData->Outputs.ValidRankMask) {
      if (NoPrint == FALSE  && PerChannel == FALSE) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "\n *** Optimizing RankMask %x *** \n", RankMask);
      }
      Scale[ARRAY_COUNT(TestList)] = 255;
      TrainDDROptParam (
        MrcData,
        &BestOff,
        MrcData->Outputs.ValidChBitMask,
        (PerChannel == TRUE) ? MrcData->Outputs.ValidRankMask : RankMask,
        OptParam2,
        OptParamLength2,
        GridMode2,
        TestList,
        ARRAY_COUNT (TestList),
        Scale,
        NULL,
        Start2, // Start
        Stop2,  // Stop
        (FinalRecenter == TRUE) ? OPT_PARAM_LOOP_COUNT + 4 : OPT_PARAM_LOOP_COUNT + 2,
        1,      // Repeats
        NoPrint,// NoPrint
        0,      // SkipOptUpdate
        0,      // GuardBand
        0,       // PatType
        SaveMarginsArray
        );
      if (PerChannel == TRUE) {
        break;
      }
    }
  }

  if (FinalRecenter) {
    if (NoPrint == FALSE) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
    }
    DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    if (NoPrint == FALSE) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Rd Vref\n");
    }
    DQTimeCentering1D(MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);
  }

  return mrcSuccess;
}

/**
  This function implements Read Equalization training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcReadEQTraining (
  IN MrcParameters *const MrcData
  )
{
  return ReadEQTraining (MrcData, FALSE, TRUE, TRUE);
}

/**
  Training the Cmd/CTL TxEq and Ron for best margins.
  Steps:
  1. Find the minimal Vref (Ron) for which Comp is not saturated (Start).
  2. Find the maximal Vref (Ron) for which Comp is not saturated (Stop).
  3. Train CCC Ron (Vref) & TxEq in the region [Start, Stop].
  4. Center Cmd Timing.
  5. Center Cmd Voltage.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcCmdEqDsTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  MrcOutput           *Outputs;
  static const UINT8  TestList[] = { CmdV, CmdT };
  UINT8               Scale[] = { 1, 1, 0, 0, 0 }; // Must specify scale = 0 to unpopulate slots!!
  static const UINT8  OptParam[] = { OptCCCTxEq, OptCCCDS }; // Param affecting vref must be second
  //MrcOutput           *Outputs;
  MrcStatus           Status;
  OptOffsetChByte     BestOff;
  INT8                Start[2];
  INT8                Stop[2];

  Outputs = &MrcData->Outputs;

  Start[0] = OptParamLimitValue (MrcData, OptParam[0], 0);
  Stop[0] = OptParamLimitValue (MrcData, OptParam[0], 1);
  Start[1] = OptParamLimitValue (MrcData, OptParam[1], 0);
  Stop[1] = OptParamLimitValue (MrcData, OptParam[1], 1);

  if (Outputs->Lpddr) {
    if ((CCC_TXEQ_MAX - Stop[0] * CCC_TXEQ_STEP) < LP_CCC_TXEQ_LOW_LIMIT) {
      Stop[0] = (CCC_TXEQ_MAX - LP_CCC_TXEQ_LOW_LIMIT) / CCC_TXEQ_STEP;
    }
  }

  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    ChessEven,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    Start,
    Stop,
    OPT_PARAM_LOOP_COUNT - 2, // Loopcount
    1,                    // Repeats
    0,                    // NoPrint
    0,                    // SkipOptUpdate
    0,                    // GuardBand
    0,                     // PatType
    SaveMarginsArray
    );

  // Re-Center CMD Timing and voltage and update Host Struct with new center (may be required, but probably vref centering will be sufficient as timing effects should be pretty symmetrical).
  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Cmd Vref\n");
  Status = CmdVoltageCentering(MrcData, CCC_RE_CENTER_LOOP_COUNT, V_FINAL_STEP_SIZE, MRC_POWER_TRAINING_DEBUG);

  return Status;
}

/**
  This function implements CMD/CTL Drive Strength Up/Dn 2D.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcCmdDsUpDnTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  MrcInput            *Inputs;
  MrcOutput           *Outputs;
  static const UINT8  TestList[] = { CmdV, CmdT };
  UINT8               Scale[] = { 1, 1, 0, 0, 0 }; // Must specify scale = 0 to unpopulated slots!!
  static const UINT8  OptParam[] = { OptCCCDSUpCoarse, OptCCCDSDnCoarse };
  OptOffsetChByte     BestOff;
  //UINT8               UPMOptimize[MAX_TRADEOFF_TYPES] = { 0, 1 };
  INT8                Start[2];
  INT8                Stop[2];
  MrcStatus           Status;

  Inputs  = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;

  Start[0] = OptParamLimitValue (MrcData, OptParam[0], 0);
  Stop[0] = OptParamLimitValue (MrcData, OptParam[0], 1);
  Start[1] = OptParamLimitValue (MrcData, OptParam[1], 0);
  Stop[1] = OptParamLimitValue (MrcData, OptParam[1], 1);

  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    UpDnCompOffsetSweepRange1,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    Start,
    Stop,
    OPT_PARAM_LOOP_COUNT - 2,
    1,                    // Repeats
    0,                    // NoPrint
    0,                    // SkipOptUpdate
    0,                    // GuardBand
    0,                     // PatType
    SaveMarginsArray
    );

  // Re-Center CMD Timing and voltage and update Host Struct with new center (may be required, but probably vref centering will be sufficient as timing effects should be pretty symmetrical).
  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Cmd Vref\n");
  Status = CmdVoltageCentering (MrcData, CCC_RE_CENTER_LOOP_COUNT, V_FINAL_STEP_SIZE, MRC_POWER_TRAINING_DEBUG);

  if (!(Inputs->TrainingEnables2.CMDSR) || Outputs->Lpddr) { // We need to Re-center Cmd/Ctl timings if CMD SR and CAODT training were not run
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Cmd/Ctl Timings\n");
    Status = MrcCmdTimingCentering (MrcData, Outputs->ValidChBitMask, CCC_RE_CENTER_LOOP_COUNT, TRUE, MRC_POWER_TRAINING_DEBUG, 1);
    if (Status != mrcSuccess) {
      return Status;
    }

    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Timing \n");
    DQTimeCentering1D (MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
    DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);
  }

  return Status;
}

/**
  Training the Cmd/CTL/CLK slew rate for best margins.
  Steps:
  1. Get Min/Max Stage Number from DDR Scomp config step (Start).
  2. Train CCC Slew rate stages in the region [Start, Stop].
  3. Re-center Cmd Timing.
  4. Re-center Cmd Voltage.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcCmdSlewRate (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  static const UINT8  TestList[] = { CmdV, CmdT };
  static const UINT8  OptParam[] = { OptCCCSComp };
  UINT8               Scale[] = { 1, 1, 0, 0, 0 };
  MrcDebug            *Debug;
  MrcOutput           *Outputs;
  MrcIntOutput        *IntOutputs;
  MrcStatus           Status = mrcSuccess;
  OptOffsetChByte     BestOff;
  INT8                MaxNumStages;
  INT8                MinNumStages;

  Outputs = &MrcData->Outputs;
  IntOutputs = (MrcIntOutput *) (MrcData->IntOutputs.Internal);
  Debug = &Outputs->Debug;

  // Can run this on LPDDR4 if need be, but probably we should skip it
  if (Outputs->Lpddr) {
    return mrcSuccess;
  }

  MinNumStages = IntOutputs->CmdSRData.MinChainLengthCmd + OptParamLimitValue(MrcData, OptParam[0], 0); // We have to convert the 0->15 index into a -8->7 index.
  MaxNumStages = IntOutputs->CmdSRData.MaxChainLengthCmd + OptParamLimitValue(MrcData, OptParam[0], 0); // We have to convert the 0->15 index into a -8->7 index.
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nCmd SComp Stages: Min %d, Max %d\n", MinNumStages, MaxNumStages);

  // Train Slew Rate for best margins
  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    CustomSR2,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    &MinNumStages,  // Start
    &MaxNumStages,  // Stop
    OPT_PARAM_LOOP_COUNT,
    1,      // Repeats
    0,      // NoPrint
    0,      // SkipOdtUpdate
    0,      // GuardBand
    0,       // PatType
    SaveMarginsArray
    );

  // Re-Center CMD Timing and voltage and update Host Struct with new center
  MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Re-center Cmd/Ctl Timings\n");
  Status = MrcCmdTimingCentering (MrcData, Outputs->ValidChBitMask, CCC_RE_CENTER_LOOP_COUNT, TRUE, MRC_POWER_TRAINING_DEBUG, 1);
  if (Status != mrcSuccess) {
    return Status;
  }

  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Timing \n");
  DQTimeCentering1D(MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training
  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
  DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Re-center Cmd Vref\n");
  Status = CmdVoltageCentering (MrcData, CCC_RE_CENTER_LOOP_COUNT, V_FINAL_STEP_SIZE, MRC_POWER_TRAINING_DEBUG);
  return Status;
}

/**
  This function implements VccDLL Bypass training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcVccDLLBypassTraining (
  IN MrcParameters *const MrcData
  )
{
    UINT16              SaveMarginsArray[3][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
    static const UINT8  TestList[] = { CmdT, RdT, WrT};
    static const UINT8  OptParam[] = { OptVccDLLBypass };
    UINT8               Scale[] = { 2, 1, 1, 0, 0 };
    OptOffsetChByte     BestOff;
    INT8                Start;
    INT8                Stop;
    MrcOutput           *Outputs;

    Outputs = &MrcData->Outputs;

    Start = OptParamLimitValue (MrcData, OptParam[0], 0);
    Stop = OptParamLimitValue (MrcData, OptParam[0], 1);

    TrainDDROptParam (
        MrcData,
        &BestOff,
        MrcData->Outputs.ValidChBitMask,
        MrcData->Outputs.ValidRankMask,
        OptParam,
        ARRAY_COUNT (OptParam),
        FullGrid,
        TestList,
        ARRAY_COUNT (TestList),
        Scale,
        NULL,
        &Start,           // Start
        &Stop,            // Stop
        OPT_PARAM_LOOP_COUNT,
        1,                // Repeats
        0,                // NoPrint
        0,                // SkipOdtUpdate
        0,                // GuardBand
        0,                 // PatType
        SaveMarginsArray
        );

    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Timing \n");
    DQTimeCentering1D (MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
    DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    return mrcSuccess;
}

/**
  This function implements Write (Transmitter) Equalization and Drive Strength training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcWriteEqDsTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  MrcOutput           *Outputs;
  static const UINT8  TestList[]  = { WrV, WrT };
  UINT8               Scale[]     = { 1, 1, 0, 0, 0 }; // Must specify scale = 0 to unpopulate slots!!
  static const UINT8  OptParam[] = { OptTxEq, OptWrDS }; // Param affecting vref must be second
  OptOffsetChByte     BestOff;
  INT8                Start[ARRAY_COUNT (OptParam)];
  INT8                Stop[ARRAY_COUNT (OptParam)];

  Outputs = &MrcData->Outputs;

  Start[0] = OptParamLimitValue (MrcData, OptParam[0], 0);
  Stop[0]  = OptParamLimitValue (MrcData, OptParam[0], 1);
  Start[1] = OptParamLimitValue (MrcData, OptParam[1], 0);
  Stop[1]  = OptParamLimitValue (MrcData, OptParam[1], 1);

  // Run TxEq/DS 2D
  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    ChessEven,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    Start,
    Stop,
    OPT_PARAM_LOOP_COUNT + 2,
    1,            // Repeats
    0,
    0,
    0,           // GuardBand
    0,            // PatType
    SaveMarginsArray
    );

  // Re-Center Write Timing and voltage and update Host Struct with new center (may be required, but probably vref centering will be sufficient as timing effects should be pretty symmetrical).
  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Vref\n");
  DQTimeCentering1D(MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  /*MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_ERROR, "Re-center Write Timing \n");
  DQTimeCentering1D(MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training
  */

  return mrcSuccess;
}

/**
  This function implements Write (Transmitter) Drive Strength Up / Down training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcWriteDsUpDnTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  MrcDebug            *Debug;
  MrcInput            *Inputs;
  MrcOutput           *Outputs;
  static const UINT8  TestList[]  = { WrV, WrT };
  UINT8               Scale[]     = { 1, 1, 0, 0, 0 }; // Must specify scale = 0 to unpopulated slots!!
  static const UINT8  OptParam[]  = { OptWrDSUpCoarse, OptWrDSDnCoarse };
  static const UINT8  OptParam2[] = { OptTxEq };
  OptOffsetChByte     BestOff;
  UINT8               RankMask;
  INT8                Start[2];
  INT8                Stop[2];
  INT8                StartTxEq;
  INT8                StopTxEq;

  Inputs  = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Debug   = &Outputs->Debug;

  Start[0] = OptParamLimitValue (MrcData, OptParam[0], 0);
  Stop[0]  = OptParamLimitValue (MrcData, OptParam[0], 1);
  Start[1] = OptParamLimitValue (MrcData, OptParam[1], 0);
  Stop[1]  = OptParamLimitValue (MrcData, OptParam[1], 1);

  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    UpDnCompOffsetSweepRange2,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    Start,
    Stop,
    OPT_PARAM_LOOP_COUNT + 2,
    1,                    // Repeats
    0,                    // NoPrint
    0,                    // SkipOptUpdate
    0,                    // GuardBand
    0,                     // PatType
    SaveMarginsArray
    );

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Vref\n");
  DQTimeCentering1D (MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  StartTxEq = OptParamLimitValue (MrcData, OptParam2[0], 0);
  StopTxEq = OptParamLimitValue (MrcData, OptParam2[0], 1);

  // Run TxEq 1D
  for (RankMask = 1; RankMask < (0x1 << MAX_RANK_IN_CHANNEL); RankMask <<= 1) {
    if (RankMask & MrcData->Outputs.ValidRankMask) {
      MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\n *** Optimizing RankMask %x *** \n", RankMask);
      Scale[ARRAY_COUNT (TestList)] = 0;
      TrainDDROptParam (
        MrcData,
        &BestOff,
        MrcData->Outputs.ValidChBitMask,
        RankMask,
        OptParam2,
        ARRAY_COUNT (OptParam2),
        FullGrid,
        TestList,
        ARRAY_COUNT (TestList),
        Scale,
        NULL,
        &StartTxEq,
        &StopTxEq,
        OPT_PARAM_LOOP_COUNT + 2,
        1,                    // Repeats
        0,                    // NoPrint
        0,                    // SkipOptUpdate
        0,                    // GuardBand
        0,                     // PatType
        SaveMarginsArray
        );
    }
  }

  // Re-Center Write Timing and voltage and update Host Struct with new center (may be required, but probably vref centering will be sufficient as timing effects should be pretty symmetrical).
  MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Re-center Write Vref\n");
  DQTimeCentering1D(MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  if (!(Inputs->TrainingEnables.WRSRT) || Outputs->Lpddr) { // We need to Re-center Wr timings if Wr SR training was not run
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Timing \n");
    DQTimeCentering1D (MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training
  }

  return mrcSuccess;
}

/**
  This function implements RxVref Decap training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcRxVrefDecapTraining (
  IN MrcParameters *const MrcData
  )
{
    UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
    MrcDebug            *Debug;
    MrcOutput           *Outputs;
    static const UINT8  TestList[] = { RdT, RdV };
    UINT8               *OptParam = 0x0;
    UINT8               OptParam1[] = { OptRxVrefVddqDecap };
    UINT8               OptParam2[] = { OptRxVrefVttDecap, OptRxVrefVddqDecap };
    UINT8               OptParam3[] = { OptRxVrefVttDecap };
    UINT8               Scale[] = { 2, 1, 255, 0, 0 }; // Ignore Power Optimization
    OptOffsetChByte     BestOff;
    UINT8               OptParamLength;
    UINT8               GridMode;
    INT8                *Start;
    INT8                *Stop;
    INT8                Start1;
    INT8                Stop1;
    INT8                Start2[2];
    INT8                Stop2[2];

    Outputs = &MrcData->Outputs;
    Debug = &Outputs->Debug;

    switch (Outputs->OdtMode) {
        case MrcOdtModeVss:
        case MrcOdtModeVddq:
          if (Outputs->Lpddr) {
            return mrcSuccess;
          }
          Start1 = OptParamLimitValue (MrcData, OptParam1[0], 0);
          Stop1 = OptParamLimitValue (MrcData, OptParam1[0], 1);
          Start = &Start1;
          Stop = &Stop1;
          OptParam = OptParam1;
          OptParamLength = 1;
          GridMode = FullGrid;
          break;

        case MrcOdtModeVtt:
          if (Outputs->Lpddr) {
            Start1 = OptParamLimitValue(MrcData, OptParam3[0], 0);
            Stop1 = OptParamLimitValue(MrcData, OptParam3[0], 1);
            Start = &Start1;
            Stop = &Stop1;
            OptParam = OptParam3;
            OptParamLength = 1;
            GridMode = FullGrid;
          } else {
            Start2[0] = OptParamLimitValue(MrcData, OptParam2[0], 0);
            Stop2[0] = OptParamLimitValue(MrcData, OptParam2[0], 1);
            Start2[1] = OptParamLimitValue(MrcData, OptParam2[1], 0);
            Stop2[1] = OptParamLimitValue(MrcData, OptParam2[1], 1);
            Start = Start2;
            Stop = Stop2;
            OptParam = OptParam2;
            OptParamLength = 2;
            GridMode = DecapSweep;
          }
          break;

        case MrcOdtModeDefault:
        default:
          return mrcSuccess;
    }

    TrainDDROptParam (
        MrcData,
        &BestOff,
        MrcData->Outputs.ValidChBitMask,
        MrcData->Outputs.ValidRankMask,
        OptParam,
        OptParamLength,
        GridMode,
        TestList,
        ARRAY_COUNT (TestList),
        Scale,
        NULL,
        Start,
        Stop,
        OPT_PARAM_LOOP_COUNT + 1,
        1,                    // Repeats
        0,                    // NoPrint
        0,                    // SkipOptUpdate
        0,                    // GuardBand
        0,                     // PatType
        SaveMarginsArray
        );

    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Re-center Rd Vref\n");
    DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
    DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    return mrcSuccess;
}

/**
  This function implements PanicVttDnLp Training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - if it succeeds return mrcSuccess
**/
MrcStatus
MrcPanicVttDnLpTraining (
  IN MrcParameters *const MrcData
  )
{
    UINT16              SaveMarginsArray[1][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
    MrcDebug            *Debug;
    MrcOutput           *Outputs;
    static const UINT8  TestList[] = { RdV };
    static const UINT8  OptParam[] = { OptPanicVttDnLp };
    UINT8               Scale[] = { 1, 0, 0, 0, 0 }; // Must specify scale = 0 to unpopulated slots!!
    OptOffsetChByte     BestOff;
    UINT8               VTTIndex;
    INT64               GetSetVal;
    INT8                Start;
    INT8                Stop;

    Outputs = &MrcData->Outputs;
    Debug = &Outputs->Debug;

    if ((!Outputs->Lpddr) || (Outputs->OdtMode != MrcOdtModeVtt)) {
        return mrcSuccess;
    }

    Start = OptParamLimitValue (MrcData, OptParam[0], 0);
    Stop = OptParamLimitValue (MrcData, OptParam[0], 1);

    // Enable the panic event counter
    for (VTTIndex = 0; VTTIndex < MAX_VTT_REGS; VTTIndex++) {
      GetSetVal = 1;
      MrcGetSetChStrb(MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VTTIndex, VttGenStatusEnCount, WriteCached, &GetSetVal);
    }

    TrainDDROptParam (
        MrcData,
        &BestOff,
        MrcData->Outputs.ValidChBitMask,
        MrcData->Outputs.ValidRankMask,
        OptParam,
        ARRAY_COUNT (OptParam),
        FullGrid,
        TestList,
        ARRAY_COUNT (TestList),
        Scale,
        NULL,
        &Start,
        &Stop,
        OPT_PARAM_LOOP_COUNT,
        1,                    // Repeats
        0,                    // NoPrint
        0,                    // SkipOptUpdate
        0,                    // GuardBand
        0,                     // PatType
        SaveMarginsArray
        );

    // Disable the panic event counter
    for (VTTIndex = 0; VTTIndex < MAX_VTT_REGS; VTTIndex++) {
      GetSetVal = 0;
      MrcGetSetChStrb(MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VTTIndex, VttGenStatusEnCount, WriteCached, &GetSetVal);
    }

    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Re-center Rd Vref\n");
    DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    return mrcSuccess;
}

/**
  This function implements Vddq power training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - If it succeeds return mrcSuccess
**/
MrcStatus
MrcVddqTraining (
  IN MrcParameters *const MrcData
  )
{
    UINT16              SaveMarginsArray[4][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
    MrcStatus           Status;
    MrcOutput           *Outputs;
    MrcDebug            *Debug;
    static const UINT8  TestList[] = { CmdV, CmdT, WrV, WrT };
    static const UINT8  OptParam[] = { OptVddq };
    UINT8               Scale[] = { 2, 2, 1, 1 };
    OptOffsetChByte     BestOff;
    UINT16              LpddrOvershootLimits[] = { 0, 0 };
    INT8                Start;
    INT8                Stop;
    BOOLEAN             Lpddr;

    // TGL_POWER_TRAINING_VDDQ: This algorithm can potentially (but shouldn't under operational conditions) overflow vref values during VDDQ adjustments. Make sure postsilicon that the VDDQ sweep range doesn't cause Vref overflows.
    Status = mrcSuccess;
    Outputs = &MrcData->Outputs;
    Debug = &Outputs->Debug;
    Lpddr = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4) || (Outputs->DdrType == MRC_DDR_TYPE_LPDDR5);

    if (!Lpddr) {
        return mrcSuccess;
    }

    Start = OptParamLimitValue (MrcData, OptParam[0], 0);
    Stop = OptParamLimitValue (MrcData, OptParam[0], 1);

    if (Outputs->Lpddr) {
      CalcVDDQLimits(MrcData, LpddrOvershootLimits);

      if (LpddrOvershootLimits[0] > Start) {
        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Changing VDDQ sweep limits due to LPDDR overshoot limits. Original %d, Final %d \n", Start, LpddrOvershootLimits[0]);
        Start = (INT8)LpddrOvershootLimits[0];
      }
      if (LpddrOvershootLimits[1] < Stop) {
        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Changing VDDQ sweep limits due to LPDDR overshoot limits. Original %d, Final %d \n", Stop, LpddrOvershootLimits[1]);
        Stop = (INT8)LpddrOvershootLimits[1];
      }
    }

    if (Stop > Start) { // Don't run if VDDQ can't be changed
      TrainDDROptParam (
        MrcData,
        &BestOff,
        MrcData->Outputs.ValidChBitMask,
        MrcData->Outputs.ValidRankMask,
        OptParam,
        ARRAY_COUNT (OptParam),
        FullGrid,
        TestList,
        ARRAY_COUNT (TestList),
        Scale,
        NULL,
        &Start,           // Start
        &Stop,            // Stop
        OPT_PARAM_LOOP_COUNT,
        1,                // Repeats
        0,                // NoPrint
        0,                // SkipOdtUpdate
        0,
        0,                 // PatType
        SaveMarginsArray
      );

      if (Outputs->Lpddr) {
        MrcLpddrOvershoot(MrcData);
      }

      // JEDEC init in case the last Vddq setting messed up the DRAMs (only needed if running data margins blindly after command margins).
      // MrcResetSequence (MrcData);

      // May not be neccesary if we can adjust in the get/set Vddq method
      // Timings need not be re-centered since we are using widths for power training after this, we just need to make sure the Vrefs aren't really far off.
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Re-center Cmd Vref\n");
      Status = CmdVoltageCentering (MrcData, CCC_RE_CENTER_LOOP_COUNT, V_FINAL_STEP_SIZE, MRC_POWER_TRAINING_DEBUG);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Re-center Write Vref\n");
      DQTimeCentering1D (MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Re-center Rd Vref\n");
      DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);
      // May need to re-center CMD/CTL-CLK if this disturbs the CMD/CTL to CLK alignment by more than a few ticks.
      // Re-Center CMD Timing and voltage and update Host Struct with new center
      //MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Re-center Cmd/Ctl Timings\n");
//#ifndef LOCAL_STUB_FLAG
      //Status = MrcCmdTimingCentering(MrcData, Outputs->ValidChBitMask, CCC_RE_CENTER_LOOP_COUNT, TRUE, MRC_POWER_TRAINING_DEBUG, 1);
//#endif
      //if (Status != mrcSuccess) {
      //  return Status;
      //}
      // May need to re-center Write Leveling if this disturbs the TxDqs to CLK alignment by more than a few ticks. DQTimeCentering1D does not currently support 'WrDqsT'.
      // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Re-center TxDqs (Write Leveling Fine)\n");
      // DQTimeCentering1D (MrcData, WrDqsT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);
    }

    return Status;
}

/**
  This function implements Read Amplifier Power training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - If it succeeds return mrcSuccess
**/
MrcStatus
MrcReadAmplifierPower (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[3][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  MrcStatus           Status;
  MrcOutput           *Outputs;
  static const UINT8  TestList[]  = { RdV, RdT, RcvEnaX };
  UINT8               Scale1[]    = { 1, 1, 1, 0, 0 };   // Need seperate scale arrays for each param in case power is optimized differently for each one
  UINT8               Scale2[]    = { 1, 1, 1, 255, 0 }; // Need seperate scale arrays for each param in case power is optimized differently for each one
  UINT8               Scale3[]    = { 1, 1, 1, 255, 0 }; // Need seperate scale arrays for each param in case power is optimized differently for each one
  UINT8               OptParam1[] = { OptRxLoad };
  UINT8               OptParam2[] = { OptRxCbData };
  UINT8               OptParam3[] = { OptRxCbComp };
  OptOffsetChByte     BestOff;
  INT8                Start;
  INT8                Stop;

  Status      = mrcSuccess;
  Outputs     = &MrcData->Outputs;

  if ((Outputs->RxMode == MrcRxModeMatchedN) || (Outputs->RxMode == MrcRxModeMatchedP)) {
    OptParam1[0] = OptRxBias;
  }

  Start = OptParamLimitValue(MrcData, OptParam1[0], 0);
  Stop = OptParamLimitValue(MrcData, OptParam1[0], 1);

  // Function Call for RxBias/RxLoad
  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam1,
    ARRAY_COUNT (OptParam1),
    FullGrid,
    TestList,
    ARRAY_COUNT (TestList),
    Scale1,
    NULL,
    &Start,           // Start
    &Stop,            // Stop
    OPT_PARAM_LOOP_COUNT + 1,
    1,                // Repeats
    0,                // NoPrint
    0,                // SkipOdtUpdate
    0,
    0,                 // PatType
    SaveMarginsArray
    );

  if ((Outputs->RxMode == MrcRxModeMatchedN) || (Outputs->RxMode == MrcRxModeMatchedP)) {
    Start = OptParamLimitValue(MrcData, OptParam2[0], 0);
    Stop = OptParamLimitValue(MrcData, OptParam2[0], 1);

    // Function Call for RxCb
    TrainDDROptParam(
      MrcData,
      &BestOff,
      MrcData->Outputs.ValidChBitMask,
      MrcData->Outputs.ValidRankMask,
      OptParam2,
      ARRAY_COUNT(OptParam2),
      FullGrid,
      TestList,
      ARRAY_COUNT(TestList),
      Scale2,
      NULL,
      &Start,           // Start
      &Stop,            // Stop
      OPT_PARAM_LOOP_COUNT + 1,
      1,                // Repeats
      0,                // NoPrint
      0,                // SkipOdtUpdate
      0,
      0,                 // PatType
      SaveMarginsArray
    );

    Start = OptParamLimitValue(MrcData, OptParam3[0], 0);
    Stop = OptParamLimitValue(MrcData, OptParam3[0], 1);

    // Function Call for RxCb
    TrainDDROptParam(
      MrcData,
      &BestOff,
      MrcData->Outputs.ValidChBitMask,
      MrcData->Outputs.ValidRankMask,
      OptParam3,
      ARRAY_COUNT(OptParam3),
      FullGrid,
      TestList,
      ARRAY_COUNT(TestList),
      Scale3,
      NULL,
      &Start,           // Start
      &Stop,            // Stop
      OPT_PARAM_LOOP_COUNT + 1,
      1,                // Repeats
      0,                // NoPrint
      0,                // SkipOdtUpdate
      0,
      0,                 // PatType
      SaveMarginsArray
    );
  }

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Vref \n");
  DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
  DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center RcvEnable \n");
  DQTimeCentering1D (MrcData, RcvEnaX, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  return Status;
}

/**
  This function implements Dimm Ron training.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus -  if it succeeds return mrcSuccess
**/
MrcStatus
MrcDimmRonTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  OptOffsetChByte     BestOff;
  MrcOutput           *Outputs;
  static const UINT8  TestList[]  = { RdV, RdT };
  UINT8               Scale[]     = { 1, 1, 0, 0, 0 }; // Must specify scale = 0 to unpopulate slots!!
  static const UINT8  OptParam[]  = { OptDimmRon, OptRdDqOdt };
  UINT16              *DimmRonVals;
  INT8                Start[ARRAY_COUNT (OptParam)];
  INT8                Stop[ARRAY_COUNT (OptParam)];
  INT8                Off;

  Outputs = &MrcData->Outputs;
  GetDimmOptParamValues (MrcData, OptParam[0], &DimmRonVals, (UINT8 *) &Stop[0]);
  Start[0] = 0;
  Stop[0] = Stop[0] - 1;

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "\nDIMM Ron Offset - Value mapping \nOffset\t Value\n");
  for (Off = Start[0]; Off < Stop[0] + 1; Off++) {
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "%d:\t %u \n", Off, DimmRonVals[(UINT8) Off]);
  }

  Start[1] = OptParamLimitValue (MrcData, OptParam[1], 0);
  Stop[1] = OptParamLimitValue (MrcData, OptParam[1], 1);

  // Need to run per channel so the optimal RxVref is the same for all ranks
  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask, // Channels
    MrcData->Outputs.ValidRankMask,         // Ranks
    OptParam,
    ARRAY_COUNT (OptParam),
    FullGrid,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    Start,           // Start
    Stop,            // Stop
    OPT_PARAM_LOOP_COUNT + 3, // Loopcount
    1,               // Repeats
    0,               // NoPrint
    0,               // SkipOdtUpdate
    0,               // GuardBand
    0,                // PatType
    SaveMarginsArray
    );

  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Vref \n");
  DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  return mrcSuccess;
}

/**
  This function implements the general training algorithm for DDR and IO parameters
  that impact margin and power.

  This function can train for power or for margin, and the function determines the mode as follows:
    PowerTraining: (NumTests <= MAX_TRADEOFF_TYPES) && (Scale[NumTests] != 0)
    else MarginTraining.

  The Parameters that are supported:
    [0: WrDS, 1: RdODT, 2: SComp, 3: TComp, 3: TxEq, 4: RxEq, 5: RxBias, 6: DimmOdt, 7: DimmOdtWr]

  @param[in,out] MrcData           - Include all MRC global data.
  @param[in,out] BestOff           - Structure containg the best offest and margins (values) for th Opt param.
  @param[in]     McChannelMask     - MCs and Channels to train
  @param[in]     RankMask          - Condenses down the results from multiple ranks
  @param[in]     OptParam          - Defines the OptParam Offsets (e.g. OptRdDqOdt, OptSComp..etc)
  @param[in]     OptParamLen       - Defines the size of OptParam[].
  @param[in]     GridMode          - Selects the way to sweep the params
  @param[in]     TestList          - List of margin params that will be tested (up to 4)
  @param[in]     NumTests          - The length of TestList[].
  @param[in]     Scale             - List of the relative importance between the 4 tests
  @param[in]     UPMOptimize       - Optimize in FindOptimalTradeOff only for UPM limit for selected params, so if they pass UPM they do not affect the score.
  @param[in]     Start             - Start point of sweeping the Comp values
  @param[in]     Stop              - Stop point of sweeping the Comp values
  @param[in]     LoopCount         - The number of loops to run in IO tests.
  @param[in]     CPGCAllRanks      - Run CPGC on all the specified ranks at once. Else, one rank at a time.
  @param[in]     NoPrint           - Switch to disable printing.
  @param[in]     SkipOptUpdate     - Switch to train but not update Opt settings. Not really supported any longer
  @param[in]     GuardBand         - Signed offset to apply to the Opt param best value.
  @param[in]     PatType           - Type of pattern the will be applied for optimization, trying to keep MrcDqPat definitions. If not specified, LFSR VA is used. Allowed values: [StaticPattern (Simple Pattern)]
  @param[in]     SaveMargin        - The array for saving margins in. Can be declared to be only as large as needed to save memory.

  @retval Nothing
**/
void
TrainDDROptParam (
  IN OUT MrcParameters *const MrcData,
  IN OUT OptOffsetChByte      *BestOff,
  IN     UINT8                McChannelMask,
  IN     UINT8                RankMask,
  IN     const UINT8          OptParam[],
  IN     UINT8                OptParamLen,
  IN     UINT8                GridMode,
  IN     const UINT8          *TestList,
  IN     UINT8                NumTests,
  IN     UINT8                *Scale,
  IN     UINT8                UPMOptimize[MAX_TRADEOFF_TYPES],
  IN     INT8                 Start[],
  IN     INT8                 Stop[],
  IN     UINT8                LoopCount,
  IN     BOOLEAN              CPGCAllRanks,
  IN     BOOLEAN              NoPrint,
  IN     BOOLEAN              SkipOptUpdate, // Not really supported any longer
  IN     INT8                 GuardBand,
  IN     UINT8                PatType,
  IN     UINT16               SaveMargin[][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]
  )
{
  const MrcInput      *Inputs;
  MrcDebug            *Debug;
  const MRC_FUNCTION  *MrcCall;
  MrcOutput           *Outputs;
  MrcChannelOut       *ChannelOut;
  MrcControllerOut    *ControllerOut;
  UINT16              (*MarginByte)[MAX_RESULT_TYPE][MAX_RANK_IN_CHANNEL][MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_EDGES];
  UINT32              BERStats[4];
  INT32               NewVddq;
  INT32               DefaultVddq;
  INT32               TxVrefBaseTicks;
  INT32               CmdVrefBaseTicks;
  INT32               DefaultCmdVref[MAX_CONTROLLER][MAX_CHANNEL][MAX_DIMMS_IN_CHANNEL];
  INT32               DefaultTxVref[MAX_CONTROLLER][MAX_CHANNEL][MAX_RANK_IN_CHANNEL];
  INT32               DefaultRxVref[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];
  UINT16              PostMargin[MAX_MARGINS_TRADEOFF][MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];
  UINT8               Test;
  UINT16              MinEye[MAX_CONTROLLER][MAX_CHANNEL];
  UINT16              Margins[MAX_TRADEOFF_TYPES][MAX_OPT_POINTS]; // TestParam X 24 Comp Points
  UINT16              UpmLimits[MAX_TRADEOFF_TYPES];
  UINT16              PwrLimits[MAX_TRADEOFF_TYPES];
  UINT16              MarginResult;
  INT16               Best;
  INT64               GetSetValue = 0;
  UINT8               UPMOptimizeDefaults[MAX_TRADEOFF_TYPES];
  UINT8               OptimizationMode;
  UINT8               ResultType;
  UINT8               AveN;
  UINT8               McChBitMask;
  UINT8               Controller;
  UINT8               Channel;
  UINT8               ChDimm;
  UINT8               Byte;
  UINT8               Rank;
  UINT8               Edge;
  UINT8               FirstRank;
  UINT8               NumBytes;
  UINT8               BMap[MAX_SDRAM_IN_DIMM]; // Need by GetBERMarginByte
  UINT8               Param;
  UINT8               MaxMargin;
  UINT8               localR[MAX_CONTROLLER][MAX_CHANNEL];
  UINT8               FirstRankPerCh[MAX_CONTROLLER][MAX_CHANNEL];
  UINT8               RankRepIndex;
  UINT8               RankRep;
  UINT8               AdjustedLoopCount;
  UINT8               LastAdjustedLoopCount;
  INT8                CurrentComp;
  UINT8               StepSize = 1;
  UINT8               NumChannels = 0;
  UINT8               CompChecks = 0;
  UINT8               OptCompParams[MAX_COMPS_CHECKED];
  UINT8               RdRd2Test;
  INT8                MaxComp;
  UINT16              OptPower[MAX_CONTROLLER][MAX_CHANNEL][MAX_OPT_POINTS];
  UINT16              Power;
  INT8                Delta;
  UINT8               Index;
  UINT8               OffLen[MAX_GRID_DIM];
  INT8                ParamOff[MAX_GRID_DIM];
  UINT8               LenMargin;
  UINT8               MaxOptPoints;
  UINT8               MinRanksPerChannel;
  UINT8               MaxRanksPerChannel;
  UINT8               NumRanksInChannel;
  UINT8               BitShift;
  BOOLEAN             IncEnds;
  BOOLEAN             IncEndsForPrint;
  BOOLEAN             CPUComp;
  BOOLEAN             printPerCh;
  BOOLEAN             BusFailure = FALSE;
  BOOLEAN             GlobalSetDone = FALSE;
  BOOLEAN             TxVrefJedecReset = FALSE;
  UINT8               OptIdx;
  BOOLEAN             EnBer;
  UINT16              Margin;
  BOOLEAN             IgnoreUpmPwrLimit;
  UINT16              MinChByte;
  UINT8               ChangeRxVrefIndex = 0;
  UINT8               ChangeTxVrefIndex = 0;
  UINT8               ChangeCmdVrefIndex = 0;
  UINT8               MaxRankReps;
  UINT8               BytesToSet[MAX_OPT_PARAM_LENGTH];
  UINT8               LastTestWasCADB;
  BOOLEAN             GlobalParams[MAX_OPT_PARAM_LENGTH];
  BOOLEAN             PerRankParam;
  BOOLEAN             CCCParam;
  BOOLEAN             AllGlobalParam;
  BOOLEAN             AnyPerRankParam;
  BOOLEAN             AnyGlobalParam;
  BOOLEAN             AnyCCCParam;
  BOOLEAN             UlxUlt;
  BOOLEAN             ChangeRxVref;
  BOOLEAN             ChangeTxVref;
  BOOLEAN             ChangeCmdVref;
  BOOLEAN             ChangeRxVrefNow;
  BOOLEAN             ChangeTxVrefNow;
  BOOLEAN             ChangeCmdVrefNow;
  MC0_CH0_CR_CADB_CFG_STRUCT  CadbConfig;
#ifdef LB_STUB_FLAG
  UINT16                ParamVector[3];
  UINT16                SimMargin;
  MRC_POWER_SYS_CONFIG  SysConfig;
#endif
  OptResultsPerByte calcResultSummary; // Result print summary: 5 columns per byte
  static UINT32 Pattern[8][2] = {
    { 0xF0F0F0F0, 0xF0F0F0F0 },
    { 0xCCCCCCCC, 0xCCCCCCCC },
    { 0x0F0F0F0F, 0x0F0F0F0F },
    { 0x55555555, 0x55555555 },
    { 0xCCCCCCCC, 0xCCCCCCCC },
    { 0xF0F0F0F0, 0xF0F0F0F0 },
    { 0x55555555, 0x55555555 },
    { 0x0F0F0F0F, 0x0F0F0F0F }
  };

  static UINT32 Pattern2[8][2] = {
    { 0xAAAAAAAA, 0x55555555 },
    { 0xAAAAAAAA, 0x55555555 },
    { 0xAAAAAAAA, 0x55555555 },
    { 0xAAAAAAAA, 0x55555555 },
    { 0xAAAAAAAA, 0x55555555 },
    { 0xAAAAAAAA, 0x55555555 },
    { 0xAAAAAAAA, 0x55555555 },
    { 0xAAAAAAAA, 0x55555555 }
  };

  LastAdjustedLoopCount = 0;
  LastTestWasCADB     = 2;
  MinRanksPerChannel  = 0xFF;
  MaxRanksPerChannel  = 1;
  NumBytes            = 1;
  DefaultVddq         = 0;
  AllGlobalParam      = TRUE;
  AnyPerRankParam     = FALSE;
  AnyGlobalParam      = FALSE;
  AnyCCCParam         = FALSE;
  PerRankParam        = FALSE;
  CCCParam            = FALSE;
  ResultType          = 0;
  CurrentComp         = 0;
  IncEnds             = 1;
  IncEndsForPrint     = 1;
  printPerCh          = 0;
  ChangeRxVref        = FALSE;
  ChangeTxVref        = FALSE;
  ChangeCmdVref       = FALSE;
  ChangeRxVrefNow     = FALSE;
  ChangeTxVrefNow     = FALSE;
  ChangeCmdVrefNow    = FALSE;
  Inputs              = &MrcData->Inputs;
  MrcCall             = Inputs->Call.Func;
  Outputs             = &MrcData->Outputs;
  Debug               = &Outputs->Debug;
  UlxUlt              = Inputs->UlxUlt;
  MarginByte          = &Outputs->MarginResult;
  McChannelMask      &= Outputs->ValidChBitMask;
  RankMask           &= Outputs->ValidRankMask;
  if (Outputs->Lpddr) {
    TxVrefBaseTicks  = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4) ? LP4_TXVREF_BASE_TICKS : LP5_TXVREF_BASE_TICKS;
    CmdVrefBaseTicks = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4) ? LP4_CMDVREF_BASE_TICKS : LP5_CMDVREF_BASE_TICKS;
  } else {
    TxVrefBaseTicks  = DDR4_TXVREF_BASE_TICKS;
    CmdVrefBaseTicks = DDR4_CMDVREF_BASE_TICKS;
  }
  IgnoreUpmPwrLimit = FALSE;
  MrcCall->MrcSetMem ((UINT8 *) &calcResultSummary, sizeof (calcResultSummary), 0);
  MrcCall->MrcSetMem ((UINT8 *) BestOff, sizeof (OptOffsetChByte), 0xFF); // @todo: cleanup multiple clears.
  MrcCall->MrcSetMem ((UINT8 *) Margins, sizeof (Margins), 0);
  MrcCall->MrcSetMem ((UINT8 *) PostMargin, sizeof (PostMargin), 0);
  MrcCall->MrcSetMem ((UINT8 *) OptPower, sizeof (OptPower), 0);
  MrcCall->MrcSetMem ((UINT8 *) localR, sizeof (localR), 0);
  MrcCall->MrcSetMem ((UINT8 *) FirstRankPerCh, sizeof (FirstRankPerCh), 0);
  MrcCall->MrcSetMem ((UINT8 *) BERStats, sizeof (BERStats), 0);
  MrcCall->MrcSetMem ((UINT8 *) OffLen, sizeof (OffLen), 0);
  MrcCall->MrcSetMem ((UINT8 *) GlobalParams, sizeof (GlobalParams), 0);
  MrcCall->MrcSetMem ((UINT8 *) BytesToSet, sizeof (BytesToSet), 1);
  MrcCall->MrcSetMemWord (UpmLimits, MAX_TRADEOFF_TYPES, 0);
  MrcCall->MrcSetMemWord (PwrLimits, MAX_TRADEOFF_TYPES, 0);

  for (Test = 0; Test < NumTests; Test++) {
    for (Index = 0; Index < MAX_OPT_POINTS; Index++) {
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Byte = 0; Byte < MAX_SDRAM_IN_DIMM; Byte++) {
          SaveMargin[Test][Index][Controller][Byte] = 0;
        }
      }
    }
  }

  for (Byte = 0; Byte < ARRAY_COUNT (BMap); Byte++) {
    BMap[Byte] = Byte;
  }

  if (Inputs->PowerTrainingMode == MrcTmMargin) {
    if (OptParam[0] == OptPanicVttDnLp) {
      Scale[NumTests] = 1; // Pay attention to power in tradeoff evaluation
      if (!NoPrint) {
        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Optimizing for maximum margins (ignoring UPMs), considering power in optimization\n");
      }
    } else {
      Scale[NumTests] = 0; // If we have this param set, we are optimizing for margins only, so set 'Scale[NumTests]' to 0 to scale power to zero and thus ignore it
      if (!NoPrint) {
        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Optimizing for maximum margins (ignoring UPMs), ignoring power in optimization\n");
      }
    }
    UPMOptimize = NULL; // We want max margins, so don't optimize for UPMs
  } else if (Inputs->PowerTrainingMode == MrcTmPower) {
    if (Scale[NumTests] == 0 && !(OptParam[0] == OptRdDqOdt && Outputs->OdtMode == MrcOdtModeVtt)) { // no power considuration in Vtt when training RxDq ODT
      Scale[NumTests] = 1; // Pay attention to power in tradeoff evaluation
    }
    if (Scale[NumTests] != 0 && Scale[NumTests] != 255) { // 255 specifically ignores power
      if (UPMOptimize == NULL) {
        UPMOptimize = UPMOptimizeDefaults;
        MrcCall->MrcSetMem((UINT8 *) UPMOptimize, sizeof (UPMOptimizeDefaults), 1); // UPM optimize everything unless a specific setup was given
      }
      if (!NoPrint) {
        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Optimizing for minimum power (driving to UPMs), considering power in optimization\n");
      }
    } else {
      Scale[NumTests] = 0;
      UPMOptimize = NULL; // We want max margins, so don't optimize for UPMs
      if (!NoPrint) {
        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Optimizing for maximum margins (ignoring UPMs), ignoring power in optimization\n");
      }
    }
  }

  // Select All Ranks for REUT test
  McChBitMask = 0;
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    ControllerOut = &Outputs->Controller[Controller];
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!(MC_CH_MASK_CHECK(McChannelMask, Controller, Channel, Outputs->MaxChannels))) {
        continue;
      }

      ChannelOut      = &ControllerOut->Channel[Channel];
      localR[Controller][Channel] = ChannelOut->ValidRankBitMask & RankMask;

      NumRanksInChannel = 0;
      for (BitShift = 0; BitShift < MAX_BITS * sizeof(localR[Controller][Channel]); BitShift++) {
        if (localR[Controller][Channel] & 0x1 << BitShift) {
          NumRanksInChannel++;
        }
      }
      if (NumRanksInChannel < MinRanksPerChannel) {
        MinRanksPerChannel = NumRanksInChannel;
      }
      if ((NumRanksInChannel > MaxRanksPerChannel) && CPGCAllRanks == FALSE) {
        MaxRanksPerChannel = NumRanksInChannel;
      }

      if (localR[Controller][Channel] != 0x0) {
        NumChannels++;
      }

      // Use McChBitMask from here down - if ch is set that mean at least 1 rank for testing, also remove ch w/o active ranks
      McChBitMask |= SelectReutRanks (MrcData, Controller, Channel, localR[Controller][Channel], FALSE, 0);
      // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "reut ranks McChBitMask %x Local ranks=%x\n", McChBitMask,localR[Channel]);
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if ((0x1 << Rank) & localR[Controller][Channel]) {
          FirstRankPerCh[Controller][Channel] = Rank;
          break;
        }
      }
    }
  }

  if (McChBitMask == 0) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Error! TrainDDROptParam() McChBitMask == 0 \n");
    return;
  }

  // Find the first selected rank
  FirstRank = GetRankToStoreResults(MrcData, RankMask);

  // Store margin results (per byte or ch)
  for (OptIdx = 0; OptIdx < OptParamLen; OptIdx++) {
    // A param may only be one of these three types, never more.
    // GlobalParams = Train all channels to the same result based on the worst case results across all channels
    GlobalParams[OptIdx] = (OptParam[OptIdx] == OptCCCTco) || (OptParam[OptIdx] == OptVccDLLBypass) || (OptParam[OptIdx] == OptPanicVttDnLp) || (OptParam[OptIdx] == OptVddq) || (OptParam[OptIdx] == OptRxCbComp);
    // PerRankParam - Train all bytes to the same result based on the worst case across all bytes on all ranks in the rank mask
    PerRankParam = (OptParam[OptIdx] == OptDimmOdt) || (OptParam[OptIdx] == OptDimmOdtWr) || (OptParam[OptIdx] == OptDimmRon) || (OptParam[OptIdx] == OptDimmOdtCA) || (OptParam[OptIdx] == OptDimmOdtPark) ||
      (OptParam[OptIdx] == OptDimmOdtNom) || (OptParam[OptIdx] == OptDimmOdtParkNT) || (OptParam[OptIdx] == OptDimmOdtNomNT) || ((OptParam[OptIdx] == OptRdDqOdt) && Outputs->Lpddr);
    // CCCParam - CCC params take a different code path. These params only margin CmdT and CmdV.
    CCCParam = (OptParam[OptIdx] == OptCCCDSDnCoarse) || (OptParam[OptIdx] == OptCCCDSUpCoarse) || (OptParam[OptIdx] == OptCCCDS) || (OptParam[OptIdx] == OptCCCTxEq) || (OptParam[OptIdx] == OptCCCSComp);

    AllGlobalParam &= GlobalParams[OptIdx];
    AnyGlobalParam |= GlobalParams[OptIdx];
    AnyPerRankParam |= PerRankParam;
    AnyCCCParam |= CCCParam;
    if ((GlobalParams[OptIdx] == TRUE) || (AnyCCCParam == TRUE)) {
      CPGCAllRanks = TRUE;
    }

    // If there is a 2D sweep with 1 byte param and 1 non-byte param, make them both non-byte
    if (!AnyCCCParam && !AnyPerRankParam && !AnyGlobalParam) {
      NumBytes = (UINT8)Outputs->SdramCount;
    } else {
      NumBytes = 1;
    }

    // Calculate Start/Stop Point for Comp Optimization
    // For TcoComp avoid Reserved comps as it isn't really compensated
    CPUComp = ((OptParam[OptIdx] == OptWrDS) || (OptParam[OptIdx] == OptCCCDS) || (OptParam[OptIdx] == OptWrDSUpCoarse) || (OptParam[OptIdx] == OptWrDSDnCoarse) ||
               (OptParam[OptIdx] == OptCCCDSUpCoarse) || (OptParam[OptIdx] == OptCCCDSDnCoarse) || (OptParam[OptIdx] == OptRdDqOdt) || (OptParam[OptIdx] == OptRdDqsOdt) ||
               (OptParam[OptIdx] == OptSComp) || (OptParam[OptIdx] == OptCCCSComp) || (OptParam[OptIdx] == OptRxLoad));
    if (CPUComp) {
      if (OptParam[OptIdx] == OptSComp) {
        MaxComp = 31;
      } else {
        MaxComp = 63;
      }

      switch(OptParam[OptIdx]) {
        case OptWrDSUpCoarse:
        case OptWrDSDnCoarse:
          StepSize = WR_DS_COARSE_STEP;
          break;
        case OptCCCDSUpCoarse:
        case OptCCCDSDnCoarse:
          StepSize = CCC_DS_COARSE_STEP;
          break;
        case OptWrDS:
          StepSize = WR_DS_STEP;
          break;
        case OptCCCDS:
          StepSize = CCC_DS_STEP;
          break;
        case OptTxEq:
          StepSize = WR_TXEQ_STEP;
          break;
        case OptCCCTxEq:
          StepSize = CCC_TXEQ_STEP;
          break;
        case OptRdDqOdt:
          StepSize = RXDQ_ODT_STEP;
          break;
        case OptRdDqsOdt:
          StepSize = RXDQS_ODT_STEP;
          break;
        case OptRxEq:
          StepSize = RXEQ_STEP;
          break;
        case OptDFETap0:
        case OptDFETap1:
          if (Outputs->PowerTrainingPerChannel == TRUE) { // Per channel
            StepSize = DFE_STEP_PER_CHANNEL;
          } else {
            StepSize = DFE_STEP;
          }
          break;
        case OptRxLoad:
          StepSize = RX_LOAD_STEP;
          break;
        case OptRxBias:
          StepSize = RX_BIAS_STEP;
          break;
        case OptRxCbData:
          StepSize = RX_CB_DATA_STEP;
          break;
        case OptRxCbComp:
          StepSize = RX_CB_COMP_STEP;
          break;
      }

      // First param is CMD, second is CTL, third is CLK
      switch (OptParam[OptIdx]) {
        case OptCCCDS:
          OptCompParams[0] = OptCmdDS;
          OptCompParams[1] = OptCtlDS;
          CompChecks = 2;
          break;
        case OptCCCDSDnCoarse:
          OptCompParams[0] = OptCmdDSDnCoarse;
          OptCompParams[1] = OptCtlDSDnCoarse;
          CompChecks = 2;
          break;
        case OptCCCDSUpCoarse:
          OptCompParams[0] = OptCmdDSUpCoarse;
          OptCompParams[1] = OptCtlDSUpCoarse;
          CompChecks = 2;
          break;
        case OptCCCSComp:
          OptCompParams[0] = OptCmdSComp;
          OptCompParams[1] = OptCtlSComp;
          CompChecks = 2;
          break;
        default:
          OptCompParams[0] = OptParam[OptIdx];
          CompChecks = 1;
      }
      for (Index = 0; Index < CompChecks * MAX_COMP_DIRECTIONS; Index++) {
        CurrentComp = GetCompCode(MrcData, OptCompParams[Index / MAX_COMP_DIRECTIONS], Index % MAX_COMP_DIRECTIONS);

        if (CurrentComp >= 0) { // Comp is valid
          Delta = CurrentComp - RESERVED_COMP_CODES + Start[OptIdx] * StepSize;
          if (Delta < 0) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Comp eval %s: Start of sweep range for comp %s is clipped by %d to prevent saturation\n", (Index % MAX_COMP_DIRECTIONS == COMP_UP) ? "Up" : "Down", TOptParamOffsetString[OptCompParams[Index / MAX_COMP_DIRECTIONS]], Delta);
            Start[OptIdx] -= (Delta / (INT8)StepSize + (Delta % (INT8)StepSize == 0 ? 0 : -1)); // Value delta, not total steps, so compute steps here
          }

          Delta = MaxComp - CurrentComp - RESERVED_COMP_CODES - Stop[OptIdx] * StepSize;
          if (Delta < 0) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Comp eval %s: End of sweep range for comp %s is clipped by %d to prevent saturation\n", (Index % MAX_COMP_DIRECTIONS == COMP_UP) ? "Up" : "Down", TOptParamOffsetString[OptCompParams[Index / MAX_COMP_DIRECTIONS]], Delta);
            Stop[OptIdx] += (Delta / (INT8)StepSize + (Delta % (INT8)StepSize == 0 ? 0 : -1)); // Value delta, not total steps, so compute steps here
          }

          if (OptParam[OptIdx] == OptSComp || OptParam[OptIdx] == OptCCCSComp) {
            Start[OptIdx] -= 1; // Start one index earlier on slew rate training to account for needing a point to test slew rate being disabled. This doesn't violate comp ranges, as the slew rate is set to 0 when disabled.
          }

          if (!SkipOptUpdate && !NoPrint) {
            // print is irrelevant in this case
            MRC_DEBUG_MSG(
              Debug,
              MSG_LEVEL_NOTE,
              "Comp %s, comp eval %s: CurrentComp = %d, CurrentStart = %d, CurrentStop = %d\n",
              TOptParamOffsetString[OptCompParams[Index / MAX_COMP_DIRECTIONS]],
              (Index % MAX_COMP_DIRECTIONS == COMP_UP) ? "Up" : "Down",
              CurrentComp,
              Start[OptIdx],
              Stop[OptIdx]
            );
          }
        }
      }

      if (Stop[OptIdx] < Start[OptIdx]) {
        Stop[OptIdx] = Start[OptIdx];
      }
    }

    if ((OptParam[OptIdx] == OptDimmOdt) || (OptParam[OptIdx] == OptRdDqOdt) || (OptParam[OptIdx] == OptDimmRon) || (OptParam[OptIdx] == OptVccDLLBypass) ||
      (OptParam[OptIdx] == OptDimmOdtPark) || (OptParam[OptIdx] == OptDimmOdtNom) || (OptParam[OptIdx] == OptDimmOdtParkNT) || (OptParam[OptIdx] == OptDimmOdtNomNT)) {
      ChangeRxVref = TRUE;
      if (OptIdx == 1 && ChangeRxVrefIndex != 1) {
        ChangeRxVrefIndex = Stop[0] - Start[0] + 1; // If only one param affects vref, it should always be the second one due to the way indexing works in this function. The range is dictated by the first param.
      } else {
        ChangeRxVrefIndex = 1; // Both params affect vref
      }
    }
    if ((OptParam[OptIdx] == OptDimmOdt) || (OptParam[OptIdx] == OptDimmOdtWr) || (OptParam[OptIdx] == OptWrDS) || (OptParam[OptIdx] == OptWrDSDnCoarse) || (OptParam[OptIdx] == OptWrDSUpCoarse) ||
      (OptParam[OptIdx] == OptVccDLLBypass) || (OptParam[OptIdx] == OptDimmOdtPark) || (OptParam[OptIdx] == OptDimmOdtNom) || (OptParam[OptIdx] == OptDimmOdtParkNT) || (OptParam[OptIdx] == OptDimmOdtNomNT)) {
      ChangeTxVref = TRUE;
      if (OptIdx == 1 && ChangeTxVrefIndex != 1) {
        ChangeTxVrefIndex = Stop[0] - Start[0] + 1; // If only one param affects vref, it should always be the second one due to the way indexing works in this function. The range is dictated by the first param.
      } else {
        ChangeTxVrefIndex = 1; // Both params affect vref
      }
    }
    if ((OptParam[OptIdx] == OptDimmOdtCA) || (OptParam[OptIdx] == OptCCCDS) || (OptParam[OptIdx] == OptCCCDSDnCoarse) || (OptParam[OptIdx] == OptCCCDSUpCoarse) || (OptParam[OptIdx] == OptVccDLLBypass)) {
      ChangeCmdVref = TRUE;
      if (OptIdx == 1 && ChangeCmdVrefIndex != 1) {
        ChangeCmdVrefIndex = Stop[0] - Start[0] + 1; // If only one param affects vref, it should always be the second one due to the way indexing works in this function. The range is dictated by the first param.
      } else {
        ChangeCmdVrefIndex = 1; // Both params affect vref
      }
    }
  } // End OptIdx loop to store margin results

  // Determine where the power column is, or if we're training for best margin.
  OptimizationMode = ((NumTests < MAX_TRADEOFF_TYPES) && (Scale[NumTests] != 0)) ? NumTests : 100;

  // Loop through all test params and measure margin.
  // Calc total num of points for full grid.
  LenMargin = 1;
  for (OptIdx = 0; OptIdx < OptParamLen; OptIdx++) {
    OffLen[OptIdx] = (Stop[OptIdx] - Start[OptIdx] + 1);
    LenMargin *= OffLen[OptIdx];
    BestOff->GridDataSet.Start[OptIdx] = Start[OptIdx];
    BestOff->GridDataSet.OffLen[OptIdx] = OffLen[OptIdx];
  }
  if (SkipOptUpdate) {
    // Just run Margins test
    LenMargin = 1;
    GridMode = FullGrid;
  }

  BestOff->GridDataSet.GridMode = GridMode;
  BestOff->GridDataSet.OptParamLen = OptParamLen;
  MaxOptPoints = MAX_OPT_POINTS;
  if (LenMargin > MaxOptPoints) { // This case will never be hit, as LenMargin is 8 bits. Should make LenMargin 16 bits eventually.
    LenMargin = MaxOptPoints;
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_WARNING, "------------> warning : LenMargin exceed max: %d length is clipped\n", MAX_OPT_POINTS);
  }
  if (!NoPrint) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "LenMargin = %d, Test LoopCount = %d\n", LenMargin, LoopCount);
  }

  // Loop Through all Comp Codes
  for (Index = 0; Index < LenMargin; Index++) {
    if (OptParam[0] == OptVddq && BusFailure) { // If the CAC and/or data bus failed for a previous Vddq setting, don't run any more Vddq settings as they should all fail.
      continue;
    } else if (OptParam[0] == OptVddq && Index == 0) {
      // TGL_POWER_TRAINING_VDDQ Replace this when Vddq get/set is complete.
      DefaultVddq = 440 + VDDQ_BASE_TICKS; // MrcGetSetDdrIoGroupController0 (MrcData, Vddq, ReadFromCache, &DefaultVddq);
    }

    if (ChangeRxVref == TRUE && Index % ChangeRxVrefIndex == 0) {
      ChangeRxVrefNow = TRUE;
    }
    if (ChangeTxVref == TRUE && Index % ChangeTxVrefIndex == 0) {
      ChangeTxVrefNow = TRUE;
    }
    if (ChangeCmdVref == TRUE && Index % ChangeCmdVrefIndex == 0) {
      ChangeCmdVrefNow = TRUE;
    }

    // Continue accordingly to GridMode
    if (GetParamsXYZ (MrcData, ParamOff, OptParamLen, GridMode, Index, Start, OffLen)) { // return ParamOff[param]
      continue;
    }
    for (OptIdx = 0; OptIdx < OptParamLen; OptIdx++) {
      // These params are trained per rank(s) at times, but all bytes must still be set anyways.
      if (((OptParam[OptIdx] == OptRdDqOdt) || (OptParam[OptIdx] == OptWrDS)) && AnyPerRankParam) {
        BytesToSet[OptIdx] = Outputs->SdramCount;
      } else {
        BytesToSet[OptIdx] = NumBytes;
      }

      GlobalSetDone = 0;
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        ControllerOut = &Outputs->Controller[Controller];
        for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
          if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
            continue;
          }
          ChannelOut = &ControllerOut->Channel[Channel];

          for (Byte = 0; Byte < BytesToSet[OptIdx]; Byte++) {
            if (!SkipOptUpdate) {
              // Change OpParam offset for all ch/byte/LocalR
              // Note: When using multi OptParams need to take care that one is not overwritten
              //       by the other in UpdateOptParamOffset routine (because UpdateHost=0).
              //       Changed function to always read CSRs to avoid this.
              // Note: Some are limited in range inside e.g: RdOdt +15:-16
              if (GlobalSetDone != 1) {
                UpdateOptParamOffset (MrcData, Controller, Channel, localR[Controller][Channel], Byte, OptParam[OptIdx], ParamOff[OptIdx], TRUE);
              }
              if (GlobalParams[OptIdx]) {
                GlobalSetDone = 1;
              }
            }
            // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n--Channel=%d, localR[Controller][Channel]=%x Byte=%d OffsetComp=%d Off=%d\n",Channel,localR[Controller][Channel],Byte,OffsetComp,Off);
          }
        }
      }

      // If OptParamLen != 1, then the ODT is being run per-rank with another param and we don't want to waste time doing this
      // Setting RxDqODT invalidates the current CTLE/DFE settings
      if (Outputs->Lpddr && (OptParam[OptIdx] == OptRdDqOdt)) {
        LpddrUpdateSocOdt (MrcData);
        if (OptParamLen == 1) {
          ReadEQTraining (MrcData, (MRC_POWER_TRAINING_DEBUG) ? FALSE : TRUE, FALSE, TRUE);
        }
      } else if ((Outputs->DdrType == MRC_DDR_TYPE_DDR4) && (OptParam[OptIdx] == OptRdDqOdt) && (OptParamLen == 1)) {
        ReadEQTraining (MrcData, (MRC_POWER_TRAINING_DEBUG) ? FALSE : TRUE, FALSE, TRUE);
      }

      GlobalSetDone = 0;

      // Re-center any margins required as a result of updating the optimization param
      if (OptParam[OptIdx] == OptVddq) {
        // Run a JEDEC init in case the last Vddq setting messed up the DRAMs (only needed if running data margins blindly after command margins).
        //MrcResetSequence (MrcData);

        // Let's just re-center the vrefs. The timings will be by width, and should be accurate if the Vrefs are good. This will save time and limit JEDEC inits.
        if (!NoPrint) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Re-center Cmd Vref, Tx Vref, and Rx Vref by calculation\n");
        }

        NewVddq = DefaultVddq - VDDQ_STEP * ParamOff[OptIdx];

        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
            if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
              continue;
            }

            if (Index == 0) { // Cache
              for (ChDimm = 0; ChDimm < MAX_DIMMS_IN_CHANNEL; ChDimm++) {
                MrcGetSetMcChRnk (MrcData, Controller, Channel, CmdVref, (UlxUlt ? MRC_IGNORE_ARG : MAX_RANK_IN_DIMM * ChDimm), ReadFromCache, &GetSetValue);
                DefaultCmdVref[Controller][Channel][ChDimm] = (INT32) GetSetValue;
                if (UlxUlt) {
                  // Rank parameter is not used
                  break;
                }
              }
            }
            for (ChDimm = 0; ChDimm < MAX_DIMMS_IN_CHANNEL; ChDimm++) {
              GetSetValue = (INT64) ((NewVddq * (DefaultCmdVref[Controller][Channel][ChDimm] + CmdVrefBaseTicks)) / DefaultVddq - CmdVrefBaseTicks);
              MrcGetSetMcChRnk (MrcData, Controller, Channel, CmdVref, MAX_RANK_IN_DIMM * ChDimm, WriteCached, &GetSetValue);
              if (UlxUlt) {
                // Rank parameter is not used
                break;
              }
            }

            for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
              if ((0x1 << Rank) & RankMask) {
                continue;
              }

              if (Index == 0) { // Cache
                // TGL_POWER_TRAINING_VDDQ - Add TxVref to Get/Set
                // MrcGetSetDdrIoGroupStrobe (MrcData, Channel, Rank, MRC_IGNORE_ARG, TxVref, ReadFromCache, &DefaultTxVref[Controller][Channel][Rank]);
              }
              GetSetValue = (INT64) ((NewVddq * (DefaultTxVref[Controller][Channel][Rank] + TxVrefBaseTicks)) / DefaultVddq - TxVrefBaseTicks);
              // TGL_POWER_TRAINING_VDDQ - Add TxVref to Get/Set
              // MrcGetSetDdrIoGroupStrobe (MrcData, Channel, Rank, MRC_IGNORE_ARG, TxVref, WriteCached, &GetSetValue);
            }

            for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
              if (!MrcByteExist (MrcData, Controller, Channel, Byte)) {
                continue;
              }

              if (Index == 0) { // Cache
                MrcGetSetChStrb (MrcData, Controller, Channel, Byte, RxVref, ReadFromCache, &GetSetValue);
                DefaultRxVref[Controller][Channel][Byte] = (INT32) GetSetValue;
              }
              GetSetValue = (INT64) ((DefaultVddq * (DefaultRxVref[Controller][Channel][Byte] + RXVREF_BASE_TICKS)) / NewVddq - RXVREF_BASE_TICKS);
              MrcGetSetChStrb (MrcData, Controller, Channel, Byte, RxVref, WriteCached, &GetSetValue);
            }
          }
        }
      }
    }

    for (OptIdx = 0; OptIdx < OptParamLen; OptIdx++) {
      if (ForceSystemRComp(MrcData, OptParam[OptIdx], FALSE) == TRUE) {
        break;
      }
    }

    if (ChangeRxVrefNow == TRUE) {
      // Set vref accordingly to odt's / Ron's
      MrcSetDefaultRxVref(MrcData, FALSE, MRC_POWER_TRAINING_DEBUG);
    }
    if (ChangeTxVrefNow == TRUE) {
      DQTimeCentering1D(MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);
      ChangeTxVrefNow = FALSE;
      LastTestWasCADB = 0;
      LastAdjustedLoopCount = RE_CENTER_LOOP_COUNT;
    }
    if (ChangeRxVrefNow == TRUE) {
      DQTimeCentering1D(MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);
      ChangeRxVrefNow = FALSE;
      LastTestWasCADB = 0;
      LastAdjustedLoopCount = RE_CENTER_LOOP_COUNT;
    }
    if (ChangeCmdVrefNow == TRUE) {
      CmdVoltageCentering(MrcData, LoopCount, V_RECENTERING_STEP_SIZE, MRC_POWER_TRAINING_DEBUG);
      ChangeCmdVrefNow = FALSE;
      LastTestWasCADB = 1;
      LastAdjustedLoopCount = LoopCount;
    }

    for (Test = 0; Test < NumTests; Test++) {
      if (OptParam[0] == OptVddq && BusFailure) { // If the CAC and/or data bus failed for this Vddq setting, don't run any more tests.
        continue;
      }

      Param = TestList[Test]; // tl[0]=4 tl[1]=1
      if (PatType == StaticPattern) {
        if (OptParam[0] == OptTxDqTco) {
          SetupIOTestStatic(MrcData, McChBitMask, LoopCount, NSOE, 0, 0, 1, Pattern2);
        } else {
          SetupIOTestStatic(MrcData, McChBitMask, LoopCount, NSOE, 0, 0, 4, Pattern);
        }
        IgnoreUpmPwrLimit = TRUE;
      } else { // LFSR
        if (CPGCAllRanks == FALSE) {
          AdjustedLoopCount = 0;
          while (MinRanksPerChannel > 1) {
            MinRanksPerChannel /= 2;
            AdjustedLoopCount++;
          }
          if (LoopCount > AdjustedLoopCount) {
            AdjustedLoopCount = LoopCount - AdjustedLoopCount; // Adjust loopcount per min number of ranks per channel
          } else {
            AdjustedLoopCount = 1;
          }
        } else {
          AdjustedLoopCount = LoopCount;
        }

        if (Param == RcvEnaX) {
          RdRd2Test = (Outputs->Lpddr) ? RdRdTA_All : RdRdTA;
          if (AdjustedLoopCount > ((Outputs->Lpddr) ? 3 : 1)) {
            AdjustedLoopCount = (Outputs->Lpddr) ? (AdjustedLoopCount - 3) : (AdjustedLoopCount - 1);
          } else {
            AdjustedLoopCount = 1;
          }
        } else {
          RdRd2Test = FALSE;
        }

        if ((Param == CmdT) || (Param == CmdV)) {
          if ((OptParam[0] == OptTxDqTco) || (OptParam[0] == OptTxDqsTco)) {
            // Need to run CCC margining in these steps for now, but with a different loopcount than the data bus
            AdjustedLoopCount = (AdjustedLoopCount > 4) ? AdjustedLoopCount - 4 : 1;
          }
          if (LastTestWasCADB != 1 || LastAdjustedLoopCount != AdjustedLoopCount) {
            SetupIOTestCADB(MrcData, McChBitMask, AdjustedLoopCount, NSOE, 1, 0);
            LastTestWasCADB = 1;
          }
        } else {
          if (LastTestWasCADB != 0 || LastAdjustedLoopCount != AdjustedLoopCount) {
            SetupIOTestBasicVA(MrcData, McChBitMask, AdjustedLoopCount, NSOE, 0, 0, 8, PatWrRd, 0, 0); // Set test to all channels.
            LastTestWasCADB = 0;
          }
        }
        LastAdjustedLoopCount = AdjustedLoopCount;

        if (RdRd2Test != FALSE) {
          Outputs->DQPat = RdRd2Test;
        }
      }
      ResultType = GetMarginResultType (Param); // rxv=0 rxt=1
                                                // Assign to last pass margin results by reference
                                                // get lowest margin from all ch/rankS/byte save in FirstRank
      GetMarginByte (MrcData, Outputs->MarginResult, Param, FirstRank, RankMask);
      // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n--- FirstRank = %d ResultType=%d Param=%d ranks=0x%x\n", FirstRank,ResultType,Param,RankMask);
      // Calculate the MaxMargin for this test
      if ((Param == WrV) || (Param == WrFan3) || (Param == WrFan2) ||
        (Param == RdV) || (Param == RdFan3) || (Param == RdFan2) || (Param == CmdV)) {
        MaxMargin = GetVrefOffsetLimits (MrcData, Param);
      } else {
        MaxMargin = MAX_POSSIBLE_TIME;
      }
      if (IgnoreUpmPwrLimit) {
        // Set Upm / Power limits to maximal search region
        UpmLimits[Test] = 20 * MaxMargin;
        PwrLimits[Test] = 20 * MaxMargin;
      } else {
        UpmLimits[Test] = MrcGetUpmPwrLimit (MrcData, Param, UpmLimit);
        PwrLimits[Test] = MrcGetUpmPwrLimit (MrcData, Param, PowerLimit);
      }
      MaxMargin = MIN (MaxMargin, (UINT8)(PwrLimits[Test] / 20));

      MaxRankReps = ((CPGCAllRanks == TRUE) ? 1 : MAX_RANK_IN_CHANNEL);
      RankRep = 0;

      for (RankRepIndex = 0; RankRepIndex < MaxRankReps; RankRepIndex++) {
        if (!(Outputs->ValidRankMask & ((CPGCAllRanks == TRUE) ? RankMask : (1 << RankRepIndex)))) { // If the rank(s) don't exist, move on
          continue;
        }
        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          ControllerOut = &Outputs->Controller[Controller];
          for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
            if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
              continue;
            }
            ChannelOut = &ControllerOut->Channel[Channel];
            if (CPGCAllRanks == TRUE) {
              // We have to call this after SetupIOTestBasicVA, so call it again
              SelectReutRanks(MrcData, Controller, Channel, localR[Controller][Channel], FALSE, 0);
            } else {
              SelectReutRanks(MrcData, Controller, Channel, ChannelOut->ValidRankBitMask & (1 << RankRepIndex), FALSE, 0);
            }
          }
        }

        // Run Margin Test - margin_1d with chosen param
        // run on all ranks but change param only for firstRank??
        EnBer = 1;

#ifndef LB_STUB_FLAG
        if (Param == CmdT) {
          MrcBlockTrainResetToggle(MrcData, FALSE);
          // When margining CMD/CTL, works for both DDR and LPDDR
          CmdLinearFindEdgesLpddr (MrcData, MrcIterationCmdCtl, McChBitMask, RankMask, 0xFF, CMD_T_MARGIN_STEP_SIZE, MRC_PRINTS_OFF); // 1F means all CMD groups
          // Restore centered value
          for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
            for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
              if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
                continue;
              }
              for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
                if (MrcRankExist (MrcData, Controller, Channel, Rank)) {
                  ShiftPIforCmdTraining (MrcData, Controller, Channel, MrcIterationCmdCtl, 1 << Rank, 0xFF, 0, 0);
                }
              }
            }
          }
          MrcResetSequence(MrcData);
          MrcBlockTrainResetToggle(MrcData, TRUE);
        } else if (Param == CmdV) {
          if (OptParam[0] != OptVddq) {
            MrcResetSequence(MrcData); // Run this in case the CCC bus was dysfunctional for the last param value, it will reset it for the new param value
          }
          MrcGetBERMarginCh (
            MrcData,
            Outputs->MarginResult,
            McChBitMask,
            0xFF,
            RankMask,
            Param,
            0,  // Mode
            0,
            MaxMargin,
            0,
            BERStats
            );
        } else {
          if (OptParam[0] == OptVccDLLBypass) {
            MrcResetSequence(MrcData); // Run this in case the CCC bus was dysfunctional for the last param value, it will reset it for the new param value
          }
          //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "MrcGetBERMarginByte McChBitMask=%x FirstRank=%d Param=%d\n",McChBitMask, FirstRank, Param);
          MrcGetBERMarginByte (
            MrcData,
            Outputs->MarginResult,
            McChBitMask,
            (Param == RdV) ? 0xFF : RankMask,
            (Param == RdV) ? 0xFF : RankMask,
            Param,
            0,  // Mode
            BMap,
            EnBer,
            MaxMargin,
            0,
            BERStats
            );
          if (OptParam[0] == OptVccDLLBypass) {
            MrcResetSequence(MrcData); // Run this in case the CCC bus was dysfunctional for the last param value, it will reset it for the new param value
          }
        }

#endif
        // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, " finish MrcGetBERMarginByte \n");
        // Record Test Results
        MinChByte = MRC_UINT16_MAX;
        for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
          ControllerOut = &Outputs->Controller[Controller];
          for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
            if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
              continue;
            }
            ChannelOut = &ControllerOut->Channel[Channel];

            if (RankRep == 0) {
              for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
                SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte] = 0;

                if ((Param == CmdT) || (Param == CmdV)) {
                  break;  // Exit per-byte loop
                }
              }

              MinEye[Controller][Channel] = 0xFFFF;
            }

            if (CPGCAllRanks == TRUE) {
              if (!(ChannelOut->ValidRankBitMask & RankMask)) { // If the rank(s) don't exist on this controller/channel, move on
                continue;
              }
            } else {
              if (!(ChannelOut->ValidRankBitMask & (1 << RankRepIndex))) { // If the rank(s) don't exist on this controller/channel, move on
                continue;
              }
            }

            for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
              if (Param != CmdT) {
                MarginResult = 0;
                for (Edge = 0; Edge < MAX_EDGES; Edge++) {
                  // CmdT margins are stored according to a slightly different convention - Save in first populated rank per channel
                  // All other margins are saved to first populated rank among all channels
                  Margin = (*MarginByte)[ResultType][FirstRank][Controller][Channel][Byte][Edge];
                  MarginResult += Margin;

                  if (Param == WrV && Outputs->DdrType == MRC_DDR_TYPE_DDR4) {
                    // We margin TxVref in PDA mode. If there was no margin at all, we may not have been able to exit PDA mode correctly, so need a JEDEC reset.
                    if (Margin < MARGIN_FAIL) {
                      TxVrefJedecReset = TRUE;
                    }
                  }
                }
              } else {
                MarginResult = (*MarginByte)[ResultType][FirstRankPerCh[Controller][Channel]][Controller][Channel][Byte][0] +
                               (*MarginByte)[ResultType][FirstRankPerCh[Controller][Channel]][Controller][Channel][Byte][1];
              }

              if (OptParam[0] == OptVddq) {
                if (SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte] < UpmLimits[Test]) {
                  // This indicates a failure of the CAC or data bus, so set a flag that skips any further tests (and leaves all margins at 0).
                  // This will save time and cause the algorithm to decide this Vddq setting doesn't work.
                  BusFailure = 1;
                }
              }
              if (RankRep == 0 || SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte] > MarginResult) {
                SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte] = MarginResult;
              }
              if (MinEye[Controller][Channel] > SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte]) {
                MinEye[Controller][Channel] = SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte];
              }
              if ((Param == CmdT) || (Param == CmdV)) {
                break;  // Exit per-byte loop
              }
            }
            if (AllGlobalParam || (AnyPerRankParam && (RankRep == MaxRanksPerChannel - 1))) {
              SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + 0] = MinEye[Controller][Channel];
            }
            if ((Param == CmdT) || (Param == CmdV) || (AnyPerRankParam && (RankRep == MaxRanksPerChannel - 1))) {
              for (Byte = 0; Byte < Outputs->SdramCount; Byte++) { // Copy the first byte results to all the other bytes
                SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte] = SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + 0];
              }
            }
          }
        }
        if (TxVrefJedecReset == TRUE) {
          MrcResetSequence(MrcData);
          TxVrefJedecReset = FALSE;
        }
        if (AllGlobalParam) {
          // All bytes already collapsed into the first byte for any global param (per channel), so just look at first byte
          for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
            for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
              if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
                continue;
              }
              // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Ch%u Byte%u MinChByte = %u\n", Channel, Byte, MinChByte);
              MinChByte = MIN (MinChByte, SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + 0]);
            }
          }
          // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "MinChByte = %u\n", MinChByte);
          for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
            // All bytes already collapsed into the first byte for any global param (per channel), so just look at first byte
            for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
              if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
                continue;
              }
              for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
                SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte] = MinChByte;
              }
            }
          }
        }
        RankRep++;
      }

      if ((Param == CmdT) || (Param == CmdV)) {
        // Disable CADB Deselects
        CadbConfig.Data = 0;
        CadbConfig.Bits.CADB_MODE = 0;
        CadbConfig.Bits.CADB_TO_CPGC_BIND = 1;
        Cadb20ConfigRegWrite (MrcData, CadbConfig);
      }
    } // end of test list

    // Use one of the Margin Arrays for fine grain power tradeoffs. This is only used if Scale[NumTests] is not 0, so skip it to save time if possible.
    if (Scale[NumTests] != 0) {
      CalcSysPower(MrcData, (OptParam[0] == OptPanicVttDnLp), 10, OptParam, ParamOff, OptParamLen); // Get the power for the entire system (for however many ranks are being run)
      Power = Outputs->OdtPowerSavingData.MrcSavingTotal;
      Margins[NumTests][Index] = Power / (NumBytes * NumChannels); // Average power per byte per channel in the system (for however many ranks are being run)
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
          if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
            continue;
          }
          OptPower[Controller][Channel][Index] = Power / (NumChannels); // Average power per channel in the system (for however many ranks are being run)
        }
      }
    } else {
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
          OptPower[Controller][Channel][Index] = 0;
        }
      }
    }
  } // end of offset

  if (AllGlobalParam && !NoPrint) {
    MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n Optimization params are all global. \n Margins shown are the composite worst case for the entire system. \n");
  }
  if (AnyPerRankParam && !NoPrint) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n Optimization params are per rank mask. \n Margins shown are the composite worst case for the entire rank mask. \n");
  }
  if (AnyCCCParam && !NoPrint) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n Optimization params are per channel. \n Margins shown are the composite worst case for the entire channel. \n");
  }

  for (Test = 0; Test < NumTests; Test++) {
#ifdef MRC_DEBUG_PRINT
    PrintResultTableByte4by24 (
      MrcData,
      McChBitMask,
      SaveMargin,
      Test,
      LenMargin,
      0,
      OptParam[0],
      TestList[Test],
      PwrLimits,
      NoPrint
      );
#endif // MRC_DEBUG_PRINT
  } // end of test list

  // Calculate the best value for every byte
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    ControllerOut = &Outputs->Controller[Controller];
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels)) || GlobalSetDone) {
        continue;
      }
       ChannelOut = &ControllerOut->Channel[Channel];
      // no need to run on channel with no selected ranks
      if (!(ChannelOut->ValidRankBitMask & localR[Controller][Channel]) || GlobalSetDone) {
        continue;
      }
      for (Byte = 0; Byte < NumBytes && !GlobalSetDone; Byte++) {
        // Populate Margins array and asymmetric penalty
        for (Test = 0; Test < NumTests; Test++) {
          for (Index = 0; Index < LenMargin; Index++) {
            Margins[Test][Index] = SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte];
          }
        }

        // Special Cases for Running Average Filter. Curve smoothing can help us run less points and extrapolate the right settings (assuming non-piecewise sweeps)
        if (OptParamLen == 1) { // 1D case
          if ((OptParam[0] == OptDimmOdt) || (OptParam[0] == OptDimmOdtWr) || (OptParam[0] == OptDimmRon) || (OptParam[0] == OptDimmOdtCA) || (OptParam[0] == OptVccDLLBypass) ||
              (OptParam[0] == OptDimmOdtNom) || (OptParam[0] == OptDimmOdtPark) || (OptParam[0] == OptDimmOdtParkNT) || (OptParam[0] == OptDimmOdtNomNT) ||
              (OptParam[0] == OptPanicVttDnLp) || (OptParam[0] == OptVddq) || (OptParam[0] == OptRxVrefVttDecap)|| (OptParam[0] == OptRxVrefVddqDecap) || (OptParam[0] == OptCCCSComp) ||
              (OptParam[0] == OptSComp) || (OptParam[0] == OptDFETap0) || (OptParam[0] == OptDFETap1)) {
            AveN = 1;
          } else {
            AveN = 3;
          }

          if (LenMargin < AveN) {
            AveN = LenMargin - 1;
          }
        } else {
          // 2D case (for now)
          if ((OptParam[0] == OptTxEq) || (OptParam[0] == OptWrDS) || (OptParam[0] == OptCCCTxEq) || (OptParam[0] == OptCCCDS) || (OptParam[0] == OptRxC) || (OptParam[0] == OptRxR)) {
            AveN = 3;
          } else {
            AveN = 1;
          }

          for (Test = 0; Test < NumTests; Test++) {
            if (GridMode < FullGrid) {
              Fill2DAverage (MrcData, Margins, Test, LenMargin, OffLen[0], 0, 1);
              for (Index = 0; Index < LenMargin; Index++) { // refill the SaveMargin array after filling the gaps
                SaveMargin[Test][Index][Controller][Channel * Outputs->SdramCount + Byte] = Margins[Test][Index];
              }
            }
            if (AveN != 1) {
              RunningAverage2D(&Margins[Test][0], LenMargin, OffLen[0], 0, 1);
            }
          }
        }

        // need to provide set of power numbers depending on the OffsetComp codes (per byte)for trend line.
        NormalizePowerToMargins (MrcData, Margins, MAX_OPT_POINTS, LenMargin, NumTests);

        // Use that value to create Margin Results based on power.
        // Creates a smooth, linear function that goes from MaxSum to N/(N-1)*MaxSum
        // RatioNum = FinePwrRatio[OptParam] * LenMargin; //e.g FinePwrRatio[RdOdt]=5
        // Find the Best Overall Setting
        // senSq=0,caleM=1,powerOpHigh=0
        FindOptimalTradeOff (
          MrcData,
          &calcResultSummary,
          &Margins[0][0],
          MAX_OPT_POINTS,
          LenMargin,
          Scale,
          UPMOptimize,
          0,
          AveN,
          IncEnds,
          1,
          UpmLimits,
          OptimizationMode,
          GuardBand
          );

        // Get the best index considering the GuardBand
        Best = calcResultSummary.Best + calcResultSummary.GuardBand;
        for (Test = 0; Test < NumTests; Test++) {
          // PostMargin will be reported back to DimmOdt
          // in old method you can get non coherency between result here and what reported out
          PostMargin[Test][Controller][Channel][Byte] = calcResultSummary.Margins[Test][Best].EW;
        }

        // Best is Index, need to convert to OptParam Offset (ParamOff)
        //Best -= Shift  // update take offset look like bug

        GetParamsXYZ (MrcData, ParamOff, OptParamLen, GridMode, (UINT8) Best, Start, OffLen);
        for (OptIdx = 0; OptIdx < OptParamLen; OptIdx++) {
          if (NumBytes == 1 && BytesToSet[OptIdx] == Outputs->SdramCount) {
            for (Byte = 0; Byte < BytesToSet[OptIdx]; Byte++) {
              UpdateOptParamOffset(MrcData, Controller, Channel, localR[Controller][Channel], Byte, OptParam[OptIdx], ParamOff[OptIdx], TRUE);
            }
            Byte = 0; // Reset for rest of loop
          } else {
            UpdateOptParamOffset(MrcData, Controller, Channel, localR[Controller][Channel], Byte, OptParam[OptIdx], ParamOff[OptIdx], TRUE);
          }
        }

        // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, " localR[Channel]=%x Best =%d ch=%d byte=%d \n",localR[Controller][Channel],(UINT8) Best,Channel,Byte);
        BestOff->Offset[Controller][Channel * Outputs->SdramCount + Byte] = Best;
#ifdef MRC_DEBUG_PRINT
        for (Test = 0; Test < NumTests; Test++) {
          Print2DResultTableChByte (
            MrcData,
            Controller,
            Channel,
            Byte,
            &calcResultSummary,
            BestOff,
            TestList[Test],
            LenMargin,
            Start,
            Stop,
            OptParam,
            OffLen,
            PwrLimits,
            OptParamLen,
            Test,
            NumTests,
            NoPrint
            );
        }
#endif // MRC_DEBUG_PRINT

        if (AllGlobalParam) {
          GlobalSetDone = 1;
        }
      } // end byte

      // Print After Calc (Stack Size Reduction)
      // Printing the results
#ifdef MRC_DEBUG_PRINT
      if (0) {
        // printing the results
        // @todo: make some loop here
        IncEndsForPrint =
          (
          OptParam[0] == OptDimmOdt ||
          OptParam[0] == OptDimmOdtWr ||
          OptParam[0] == OptDimmRon ||
          OptParam[0] == OptRxEq ||
          IncEnds
          );
        printPerCh = (OptParam[0] == OptDimmOdt || OptParam[0] == OptDimmOdtWr || OptParam[0] == OptDimmRon);

        // lower bytes
        PrintCalcResultTableCh (
          MrcData,
          &calcResultSummary,
          TestList,
          NumTests,
          LenMargin,
          0,
          IncEndsForPrint,
          OptParam[0],
          OptPower[Controller][Channel],
          Controller,
          Channel,
          localR[Controller][Channel],
          Scale[NumTests],
          0,
          printPerCh,
          NoPrint
          );

        // higher bytes
        if (!printPerCh) {
          PrintCalcResultTableCh (
            MrcData,
            &calcResultSummary,
            TestList,
            NumTests,
            LenMargin,
            0,
            IncEndsForPrint,
            OptParam[0],
            OptPower[Controller][Channel],
            Controller,
            Channel,
            localR[Controller][Channel],
            Scale[NumTests],
            1,
            printPerCh,
            NoPrint
            );
          }
        } // if (0)
#endif // MRC_DEBUG_PRINT
    } // End of Calculating best value (ch)
  }

  for (OptIdx = 0; OptIdx < OptParamLen; OptIdx++) {
    if (ForceSystemRComp(MrcData, OptParam[OptIdx], FALSE) == TRUE) {
      break;
    }
  }

  for (OptIdx = 0; OptIdx < OptParamLen; OptIdx++) {
    if (Outputs->Lpddr && OptParam[OptIdx] == OptRdDqOdt) {
      LpddrUpdateSocOdt(MrcData);
      break;
    }
  }

  // Update the LastPass points in host
  for (Test = 0; Test < NumTests; Test++) {
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
          continue;
        }

        ResultType = GetMarginResultType(TestList[Test]);
        for (Byte = 0; Byte < NumBytes; Byte++) {
          // save the margins in best offset point for each byte/ch in  rank 0/1
          Best = (UINT8)(BestOff->Offset[Controller][Channel * Outputs->SdramCount + Byte]);

          for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
            if (((0x1 << Rank) & localR[Controller][Channel]) == 0x0 || !(MrcRankExist(MrcData, Controller, Channel, Rank))) {
              continue;
            }
            (*MarginByte)[ResultType][Rank][Controller][Channel][Byte][0] = SaveMargin[Test][Best][Controller][Channel * Outputs->SdramCount + Byte] / 2;
            (*MarginByte)[ResultType][Rank][Controller][Channel][Byte][1] = SaveMargin[Test][Best][Controller][Channel * Outputs->SdramCount + Byte] / 2;
          }
          // MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "best offset= %d ;byte=%d ;(*MarginByte)[ResultType][0][Controller][Channel][Byte][0] -%d (*MarginByte)[ResultType][0][Controller][Channel][Byte][1] -%d add=%d\n",(UINT8) (BestOff->Offset[Channel][Byte] - Start),Byte,(UINT16) (*MarginByte)[ResultType][0][/**Controller**/ CONTROLLER_0][Channel][Byte][0] , (*MarginByte)[ResultType][0][/**Controller**/ CONTROLLER_0][Channel][Byte][1],((UINT16) (*MarginByte)[ResultType][0][/**Controller**/ CONTROLLER_0][Channel][Byte][0] + (UINT16)(*MarginByte)[ResultType][0][/**Controller**/ CONTROLLER_0][Channel][Byte][1]));
        }
      }
    }
    ScaleMarginByte (MrcData, Outputs->MarginResult, TestList[Test], 0);
  }

  // Save the BestOff Values
  BestOff->NumTests = NumTests;
  for (Test = 0; Test < NumTests; Test++) {
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, Outputs->MaxChannels))) {
          continue;
        }
        // Track minimum eye width per ch
        for (Byte = 0; Byte < NumBytes; Byte++) {
          if (Byte == 0) {
            BestOff->Margins[Test][Controller][Channel] = PostMargin[Test][Controller][Channel][0];
          } else if (BestOff->Margins[Test][Controller][Channel] > PostMargin[Test][Controller][Channel][Byte]) {
            BestOff->Margins[Test][Controller][Channel] = PostMargin[Test][Controller][Channel][Byte];
          }
        }
        BestOff->TestList[Test][Controller][Channel] = TestList[Test];
        // @todo : add OptParam list
      }
    }
  }

  return;
}

/**
  This function implements Read ODT training.
  Optimize Read ODT strength for performance & power

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - If it succeeded return mrcSuccess
**/
MrcStatus
MrcReadODTTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16          SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  MrcStatus       Status;
  MrcOutput       *Outputs;
  INT8            StartDqs;
  INT8            StopDqs;
  INT8            LpddrOvershootLimits[] = { 0, 0 };
  INT8            StartDq;
  INT8            StopDq;
  OptOffsetChByte BestOff;
  UINT8           OptParamDqs[]   = { OptRdDqsOdt };
  UINT8           OptParamDq[]    = { OptRdDqOdt };
  UINT8           TestListDqs[]   = { RdT, RcvEnaX };
  UINT8           ScaleDqs[]    = { 1, 1, 0, 0, 0 };
  UINT8           TestListDq[]    = { RdV, RdT };
  UINT8           ScaleDq[]    = { 1, 1, 0, 0, 0 };

  Outputs     = &MrcData->Outputs;
  Status      = mrcSuccess;

  StartDqs = OptParamLimitValue (MrcData, OptParamDqs[0], 0);
  StopDqs  = OptParamLimitValue (MrcData, OptParamDqs[0], 1);
  StartDq  = OptParamLimitValue (MrcData, OptParamDq[0], 0);
  StopDq   = OptParamLimitValue (MrcData, OptParamDq[0], 1);

  if (Outputs->Lpddr) {
    CalcRxDqsCompOffsetLimits(MrcData, LpddrOvershootLimits);

    if (LpddrOvershootLimits[0] > StartDqs) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Changing RxDqs comp offset sweep limits due to LPDDR overshoot limits. Original %d, Final %d \n", StartDqs, LpddrOvershootLimits[0]);
      StartDqs = LpddrOvershootLimits[0];
    }
    if (LpddrOvershootLimits[1] < StopDqs) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Changing RxDqs comp offset sweep limits due to LPDDR overshoot limits. Original %d, Final %d \n", StopDqs, LpddrOvershootLimits[1]);
      StopDqs = LpddrOvershootLimits[1];
    }
  }

  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParamDq,
    ARRAY_COUNT (OptParamDq),
    FullGrid,
    TestListDq,
    ARRAY_COUNT(TestListDq),
    ScaleDq,
    NULL,
    &StartDq,
    &StopDq,        // Stop
    OPT_PARAM_LOOP_COUNT + 3,
    1,              // Repeats
    0,              // NoPrint
    0,              // SkipOptUpdate
    0,              // GuardBand
    0,               // PatType
    SaveMarginsArray
    );

  if (StopDqs > StartDqs) { // Don't run if RxDqs comp offset can't be changed
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Vref \n");
    DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
    DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center RcvEnable \n");
    DQTimeCentering1D (MrcData, RcvEnaX, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    TrainDDROptParam (
      MrcData,
      &BestOff,
      MrcData->Outputs.ValidChBitMask,
      MrcData->Outputs.ValidRankMask,
      OptParamDqs,
      ARRAY_COUNT (OptParamDqs),
      FullGrid,
      TestListDqs,
      ARRAY_COUNT (TestListDqs),
      ScaleDqs,
      NULL,
      &StartDqs,
      &StopDqs,           // Stop
      OPT_PARAM_LOOP_COUNT + 3, // Loopcount
      1,              // Repeats
      0,              // NoPrint
      0,              // SkipOptUpdate
      0,              // GuardBand
      0,               // PatType
      SaveMarginsArray
      );
  }

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Vref \n");
  DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
  DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center RcvEnable \n");
  DQTimeCentering1D (MrcData, RcvEnaX, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  return Status;
}

/**
  This function implements Write Slew Rate training.
  Optimize Write Slew Rate for performance & power

  @param[in] MrcData - Include all MRC global data.

  @retval mrcSuccess
**/
MrcStatus
MrcWriteSlewRate (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  static const UINT8  TestList[] = { WrV, WrT };
  static const UINT8  OptParam[] = { OptSComp };
  UINT8               Scale[] = { 1, 1, 0, 0, 0 };
  OptOffsetChByte     BestOff;
  INT8                Start;
  INT8                Stop;
  MrcOutput           *Outputs;

  Outputs = &MrcData->Outputs;

  if (Outputs->Lpddr) {
      return mrcSuccess;
  }

  Start = OptParamLimitValue (MrcData, OptParam[0], 0);
  Stop  = OptParamLimitValue (MrcData, OptParam[0], 1);

  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    CustomSR1,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    &Start,            // Start
    &Stop,             // Stop
    OPT_PARAM_LOOP_COUNT + 2,
    1,                // Repeats
    0,                // NoPrint
    0,                // SkipOdtUpdate
    0,                // GuardBand
    0,                 // PatType
    SaveMarginsArray
    );

  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Vref\n");
  DQTimeCentering1D(MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Timing \n");
  DQTimeCentering1D(MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training

  return mrcSuccess;
}

/**
  Updates a given ch/Rank/byte combination with a new value for OptParam
  OptParam can be: WrDS, RdOdt, TComp, SComp, RxEq, TxEq, RxBias or DimmOdt
  OptParam == OptDefault restore values from Host except Dimms Odt's
  @param[in,out] MrcData         - Include all MRC global data.
  @param[in]     Controller      - Controller index to work on.
  @param[in]     Channel         - Channel index to work on.
  @param[in]     Ranks           - Condenses down the results from multiple ranks
  @param[in]     Byte            - Byte index to work on.
  @param[in]     OptParam        - Defines the OptParam Offsets.
                                   Supported OptParam = [0: WrDS, 1: RdODT, 2: SComp, 3: TComp, 4: TxEq,
                                                         5: RxEq, 6: RxBias, 7: DimmOdt, 8: DimmOdtWr]
  @param[in]     Off             - Offset
  @param[in]     UpdateHost      - Desides if MrcData has to be updated. Should always use TRUE for now,
                                   since many multi-D sweeps are setting params within the same registers.

  @retval Nothing
**/
void
UpdateOptParamOffset (
  IN OUT MrcParameters *const MrcData,
  IN     const UINT8          Controller,
  IN     const UINT8          Channel,
  IN           UINT8          Ranks,
  IN     const UINT8          Byte,
  IN           UINT8          OptParam,
  IN     INT32                Off,
  IN     const UINT8          UpdateHost
  )
{

  UINT16         *DimmOptParamVals;
  UINT8           NumDimmOptParamVals;

  const MrcInput  *Inputs;
  MrcOutput       *Outputs;
  MrcChannelOut   *ChannelOut;
  INT64           GetSetVal;
  UINT8           GsmMode;
  BOOLEAN         Lpddr;
  UINT8           RankMask;
  UINT8           NumRanksInDimm = 0;
  UINT8           NumRanksInCurrentDimm;
  UINT8           Dimm;
  UINT8           Rank;
  UINT8           Index;
  UINT8           TotalIndex;
  UINT32          Offset;
  INT16           OffMin = 0;
  INT16           OffMax = 0;
  INT64           Min;
  INT64           Max;
  UINT32          Delay;
  MrcStatus       Status;
  DATA0CH0_CR_DDRCRDATAOFFSETCOMP_STRUCT DdrCrDataOffsetComp;
  CH0CCC_CR_DDRCRCTLCACOMPOFFSET_STRUCT DdrCrCmdOffsetComp;

  Inputs        = &MrcData->Inputs;
  Outputs       = &MrcData->Outputs;
  Lpddr         = Outputs->Lpddr;
  ChannelOut    = &Outputs->Controller[Controller].Channel[Channel];
  if (OptParam == OptVddq) {
    GsmMode = (UpdateHost) ? ForceWriteOffsetCached : ForceWriteOffsetUncached;
  } else {
    GsmMode = (UpdateHost) ? ForceWriteCached : ForceWriteUncached;
  }
  GsmMode |= GSM_READ_CSR; // We are updating multiple params in the same register without updating the cache, so we are forced to read the CSRs now.

  if (MRC_POWER_TRAINING_DEBUG) {
    GsmMode |= GSM_PRINT_VAL;
  }

  switch (OptParam) {
    case OptTxDqTco:
      GetSetVal = DQ_TCO_COMP_STEP * Off;
      MrcGetSet(MrcData, MRC_IGNORE_ARG, Controller, Channel, MRC_IGNORE_ARG, MRC_IGNORE_ARG, Byte, MAX_BITS, MRC_IGNORE_ARG, DdrLevel, TxTco, GsmMode, &GetSetVal);
      break;
    case OptTxDqsTco:
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if ((MrcRankExist(MrcData, Controller, Channel, Rank)) && ((Ranks & (MRC_BIT0 << Rank)))) {
          GetSetVal = DQS_TCO_COMP_STEP * Off;
          MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, TxDqsTcoPRiseNFall, GsmMode, &GetSetVal);
          MrcGetSetLimits(MrcData, TxDqsTcoPRiseNFall, &Min, &Max, &Delay);
          GetSetVal = Max - GetSetVal;

          MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, TxDqsTcoPFallNRise, GsmMode, &GetSetVal);
        }
      }
      break;
    case OptCCCTco:
      GetSetVal = CLK_TCO_COMP_STEP * Off;
      MrcGetSetNoScope(MrcData, TcoCompCodeCmd, GsmMode, &GetSetVal);
      MrcGetSetNoScope(MrcData, TcoCompCodeCtl, GsmMode, &GetSetVal);
      break;
    case OptWrDS:
    case OptWrDSUpCoarse:
    case OptWrDSDnCoarse:
    case DqDrvUpCompOffset:
    case DqDrvDnCompOffset:
    case OptRdDqOdt:
    case OptRdDqsOdt:
    case DqsOdtCompOffset:
    case DqOdtCompOffset:
    case OptRxLoad:
    case RloadCompOffset:
    case OptSComp:
    case DqSCompOffset:
      DdrCrDataOffsetComp.Data = ChannelOut->DataCompOffset[Byte];
      switch(OptParam) {
        case OptWrDS:
          OffMin = -32 / WR_DS_STEP;
          OffMax = 31 / WR_DS_STEP;
          break;
        case OptWrDSUpCoarse:
        case OptWrDSDnCoarse:
          OffMin = -32 / WR_DS_COARSE_STEP;
          OffMax = 31 / WR_DS_COARSE_STEP;
          break;
        case DqDrvUpCompOffset:
        case DqDrvDnCompOffset:
          OffMin = -32;
          OffMax = 31;
          break;
        case OptRdDqOdt:
          OffMin = -16 / RXDQ_ODT_STEP;
          OffMax = 15 / RXDQ_ODT_STEP;
          break;
        case OptRdDqsOdt:
          OffMin = -16 / RXDQS_ODT_STEP;
          OffMax = 15 / RXDQS_ODT_STEP;
          break;
        case OptRxLoad:
          OffMin = -16 / RX_LOAD_STEP;
          OffMax = 15 / RX_LOAD_STEP;
          break;
        case DqsOdtCompOffset:
        case DqOdtCompOffset:
         case RloadCompOffset:
          OffMin = -16;
          OffMax = 15;
          break;
        case OptSComp:
        case DqSCompOffset:
          OffMin = -17; // -17 for SR dis.
          OffMax = 15;
          break;
      }

      if (Off > OffMax) {
        Off = OffMax;
      } else if (Off < OffMin) {
        Off = OffMin;
      }

      switch (OptParam) {
        case OptWrDS:
          DdrCrDataOffsetComp.Bits.DqDrvUpCompOffset = Off * WR_DS_STEP;
          DdrCrDataOffsetComp.Bits.DqDrvDownCompOffset = Off * WR_DS_STEP;
          break;
        case OptWrDSUpCoarse:
          DdrCrDataOffsetComp.Bits.DqDrvUpCompOffset = Off * WR_DS_COARSE_STEP;
        case OptWrDSDnCoarse:
          DdrCrDataOffsetComp.Bits.DqDrvDownCompOffset = Off * WR_DS_COARSE_STEP;
          break;
        case DqDrvUpCompOffset:
          DdrCrDataOffsetComp.Bits.DqDrvUpCompOffset = Off;
        case DqDrvDnCompOffset:
          DdrCrDataOffsetComp.Bits.DqDrvDownCompOffset = Off;
          break;
        case OptRdDqOdt:
          DdrCrDataOffsetComp.Bits.DqOdtCompOffset = Off * RXDQ_ODT_STEP;
          break;
        case DqOdtCompOffset:
          DdrCrDataOffsetComp.Bits.DqOdtCompOffset = Off;
          break;
        case OptRdDqsOdt:
          GetSetVal = (Off < 0) ? Off * RXDQS_ODT_STEP + MRC_BIT5 : Off * RXDQS_ODT_STEP; // Convert to 5-bit 2's complement
          MrcGetSetChStrb(MrcData, Controller, Channel, Byte, DqsOdtCompOffset, GsmMode, &GetSetVal);
          break;
        case DqsOdtCompOffset:
          GetSetVal = (Off < 0) ? Off + MRC_BIT5 : Off; // Convert to 5-bit 2's complement
          MrcGetSetChStrb(MrcData, Controller, Channel, Byte, DqsOdtCompOffset, GsmMode, &GetSetVal);
          break;
        case OptRxLoad:
          DdrCrDataOffsetComp.Bits.RloadOffset = Off * RX_LOAD_STEP;
          break;
        case RloadCompOffset:
          DdrCrDataOffsetComp.Bits.RloadOffset = Off;
          break;
        case OptSComp:
        case DqSCompOffset:
          if (Off == -17) { // Just picked value outside SR range.
            DdrCrDataOffsetComp.Bits.DqSlewRateCompOffset = 0;
            GetSetVal = 1;
          } else {
            DdrCrDataOffsetComp.Bits.DqSlewRateCompOffset = Off;
            GetSetVal = 0;
          }
          MrcGetSetChStrb(MrcData, Controller, Channel, Byte, GsmIocDqSlewDlyByPass, GsmMode, &GetSetVal);
          break;
      }

      Offset = MrcGetOffsetDataOffsetComp(MrcData, Controller, (Channel * Outputs->SdramCount) + Byte);
      MrcWriteCR(MrcData, Offset, DdrCrDataOffsetComp.Data);
      if (MRC_POWER_TRAINING_DEBUG) {
        if (OptParam == DqDrvUpCompOffset || OptParam == DqDrvDnCompOffset || OptParam == DqSCompOffset || OptParam == RloadCompOffset || OptParam == DqsOdtCompOffset || OptParam == DqOdtCompOffset) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DdrCrDataOffsetComp raw: Controller = %d, Channel = %d, Byte = %d, OptParam = %d, Off = %d \n", Controller, Channel, Byte, OptParam, Off);
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DdrCrDataOffsetComp translated: Channel = %d, Byte = %d, OptParam = %d, Off = %d \n", Controller, (Channel * Outputs->SdramCount) + Byte, OptParam, Off);
        } else {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DdrCrDataOffsetComp raw: Controller = %d, Channel = %d, Byte = %d, OptParam = %s, Off = %d \n", Controller, Channel, Byte, TOptParamOffsetString[OptParam], Off);
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DdrCrDataOffsetComp translated: Channel = %d, Byte = %d, OptParam = %s, Off = %d \n", Controller, (Channel * Outputs->SdramCount) + Byte, TOptParamOffsetString[OptParam], Off);
        }
      }
      if (UpdateHost) {
        ChannelOut->DataCompOffset[Byte] = DdrCrDataOffsetComp.Data;
      }
      break;
    case OptCCCDS:
    case OptCCCDSUp:
    case OptCCCDSDn:
    case OptCCCDSUpCoarse:
    case OptCCCDSDnCoarse:
    case CmdRCompDrvUpOffset:
    case CmdRCompDrvDownOffset:
    case CtlRCompDrvUpOffset:
    case CtlRCompDrvDownOffset:
    case OptCCCSComp:
    case CmdSCompOffset:
    case CtlSCompOffset:
      DdrCrCmdOffsetComp.Data = ChannelOut->CmdCompOffset;
      switch (OptParam) {
        case OptCCCDS:
        case OptCCCDSUp:
        case OptCCCDSDn:
          OffMin = -8 / CCC_DS_STEP;
          OffMax = 7 / CCC_DS_STEP;
          break;
        case CmdRCompDrvUpOffset:
        case CmdRCompDrvDownOffset:
        case CtlRCompDrvUpOffset:
        case CtlRCompDrvDownOffset:
          OffMin = -8;
          OffMax = 7;
          break;
        case OptCCCDSUpCoarse:
        case OptCCCDSDnCoarse:
          OffMin = -8 / CCC_DS_COARSE_STEP;
          OffMax = 7 / CCC_DS_COARSE_STEP;
          break;
        case OptCCCSComp:
        case CmdSCompOffset:
        case CtlSCompOffset:
          OffMin = -9; // -9 for SR dis
          OffMax = 7;
          break;
      }

      if (Off > OffMax) {
        Off = OffMax;
      } else if (Off < OffMin) {
        Off = OffMin;
      }

      switch (OptParam) {
        case OptCCCDS:
          DdrCrCmdOffsetComp.Bits.CaRcompDrvUpOffset = Off * CCC_DS_STEP;
          DdrCrCmdOffsetComp.Bits.CtlRcompDrvUpOffset = Off * CCC_DS_STEP;
          DdrCrCmdOffsetComp.Bits.CaRcompDrvDownOffset = Off * CCC_DS_STEP;
          DdrCrCmdOffsetComp.Bits.CtlRcompDrvDownOffset = Off * CCC_DS_STEP;
          break;
        case OptCCCDSUp:
          DdrCrCmdOffsetComp.Bits.CaRcompDrvUpOffset = Off * CCC_DS_STEP;
          DdrCrCmdOffsetComp.Bits.CtlRcompDrvUpOffset = Off * CCC_DS_STEP;
          break;
        case OptCCCDSDn:
          DdrCrCmdOffsetComp.Bits.CaRcompDrvDownOffset = Off * CCC_DS_STEP;
          DdrCrCmdOffsetComp.Bits.CtlRcompDrvDownOffset = Off * CCC_DS_STEP;
          break;
        case OptCCCDSUpCoarse:
          DdrCrCmdOffsetComp.Bits.CaRcompDrvUpOffset = Off * CCC_DS_COARSE_STEP;
          DdrCrCmdOffsetComp.Bits.CtlRcompDrvUpOffset = Off * CCC_DS_COARSE_STEP;
          break;
        case OptCCCDSDnCoarse:
          DdrCrCmdOffsetComp.Bits.CaRcompDrvDownOffset = Off * CCC_DS_COARSE_STEP;
          DdrCrCmdOffsetComp.Bits.CtlRcompDrvDownOffset = Off * CCC_DS_COARSE_STEP;
          break;
        case CmdRCompDrvUpOffset:
          DdrCrCmdOffsetComp.Bits.CaRcompDrvUpOffset = Off;
          break;
        case CmdRCompDrvDownOffset:
          DdrCrCmdOffsetComp.Bits.CaRcompDrvDownOffset = Off;
          break;
        case CtlRCompDrvUpOffset:
          DdrCrCmdOffsetComp.Bits.CtlRcompDrvUpOffset = Off;
          break;
        case CtlRCompDrvDownOffset:
          DdrCrCmdOffsetComp.Bits.CtlRcompDrvDownOffset = Off;
          break;
        case OptCCCSComp:
          if (Off == -9) { // Just picked value outside SR range.
            DdrCrCmdOffsetComp.Bits.CaScompOffset = 0;
            DdrCrCmdOffsetComp.Bits.CtlScompOffset = 0;
            GetSetVal = 1;
          } else {
            DdrCrCmdOffsetComp.Bits.CaScompOffset = Off;
            DdrCrCmdOffsetComp.Bits.CtlScompOffset = Off;
            GetSetVal = 0;
          }
          MrcGetSetMcCh(MrcData, Controller, Channel, SCompBypassCtl, GsmMode, &GetSetVal);
          MrcGetSetMcCh(MrcData, Controller, Channel, SCompBypassCmd, GsmMode, &GetSetVal);
          break;
        case CmdSCompOffset:
          DdrCrCmdOffsetComp.Bits.CaScompOffset = Off;
          DdrCrCmdOffsetComp.Bits.CtlScompOffset = Off;
          break;
        case CtlSCompOffset:
          DdrCrCmdOffsetComp.Bits.CaScompOffset = Off;
          DdrCrCmdOffsetComp.Bits.CtlScompOffset = Off;
          break;
      }

      Offset = GetDdrIoCommandOffsets(MrcData, CmdSCompOffset, 0, (Controller * MAX_CHANNEL) + (Channel * (MAX_CHANNEL / Outputs->MaxChannels)), 0, 0, 0, 0);
      MrcWriteCR(MrcData, Offset, DdrCrCmdOffsetComp.Data);
      if (MRC_POWER_TRAINING_DEBUG) {
        if (OptParam == CmdSCompOffset || OptParam == CtlSCompOffset || OptParam == CtlRCompDrvUpOffset || OptParam == CtlRCompDrvDownOffset || OptParam == CmdRCompDrvUpOffset || OptParam == CmdRCompDrvDownOffset) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DdrCrCmdOffsetComp raw: Controller = %d, Channel = %d, OptParam = %d, Off = %d \n", Controller, Channel, OptParam, Off);
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DdrCrCmdOffsetComp translated: Channel = %d, OptParam = %d, Off = %d \n", (Controller * MAX_CHANNEL) + (Channel * (MAX_CHANNEL / Outputs->MaxChannels)), OptParam, Off);
        } else {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DdrCrCmdOffsetComp raw: Controller = %d, Channel = %d, OptParam = %s, Off = %d \n", Controller, Channel, TOptParamOffsetString[OptParam], Off);
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DdrCrCmdOffsetComp translated: Channel = %d, OptParam = %s, Off = %d \n", (Controller * MAX_CHANNEL) + (Channel * (MAX_CHANNEL / Outputs->MaxChannels)), TOptParamOffsetString[OptParam], Off);
        }
      }
      if (UpdateHost) {
        ChannelOut->CmdCompOffset = DdrCrCmdOffsetComp.Data;
      }
      break;
    case OptTxEq:
    case OptRxEq:
    case OptRxC:
    case OptRxR:
    case OptDFETap0:
    case OptDFETap1:
      switch (OptParam) {
        case OptTxEq:
          GetSetVal = TXEQ_MAX - Off * WR_TXEQ_STEP; // Sweep from high to low
          break;
        case OptRxEq:
          GetSetVal = Off * RXEQ_STEP;
          break;
        case OptRxC:
        case OptRxR:
          GetSetVal = Off;
          break;
        case OptDFETap0:
        case OptDFETap1:
          if (Outputs->PowerTrainingPerChannel == TRUE) { // Per channel
            GetSetVal = Off * DFE_STEP_PER_CHANNEL;
          } else {
            GetSetVal = Off * DFE_STEP;
          }
          break;
      }
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if ((MrcRankExist(MrcData, Controller, Channel, Rank)) && ((Ranks & (MRC_BIT0 << Rank)))) {
          switch (OptParam) {
            case OptTxEq:
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, TxEq, GsmMode, &GetSetVal);
              break;
            case OptRxEq:
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, RxEq, GsmMode, &GetSetVal);
              break;
            case OptRxC:
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, RxC, GsmMode, &GetSetVal);
              break;
            case OptRxR:
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, RxR, GsmMode, &GetSetVal);
              break;
            case OptDFETap0:
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, RxTap0, GsmMode, &GetSetVal);
              break;
            case OptDFETap1:
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, RxTap1, GsmMode, &GetSetVal);
              break;
          }
        }
      }
      break;
    case OptCCCTxEq:
      GetSetVal = CCC_TXEQ_MAX - Off * CCC_TXEQ_STEP; // Sweep from high (12) to low
      if (!Inputs->A0) {
        if (Lpddr) {
          if (GetSetVal < LP_CCC_TXEQ_LOW_LIMIT) {
            GetSetVal = LP_CCC_TXEQ_LOW_LIMIT; // Sweep from high (12) to 0x8
          }
        }
      } else {
        GetSetVal = CCC_TXEQ_MAX; // This is only for A step.
      }
      // For CCC, the 5th bit TxEq[4] = !(DDR4). If DDR4, set to 0, else 1. High equalizations save power for DDR4 but cost power for other modes.
      if (Lpddr) {
        GetSetVal |= 0x10;
      }
      MrcGetSetMcCh(MrcData, Controller, Channel, CmdTxEq, GsmMode, &GetSetVal);
      MrcGetSetMcCh(MrcData, Controller, Channel, CtlTxEq, GsmMode, &GetSetVal);
      break;
    case OptRxCbData:
      GetSetVal = Off * RX_CB_DATA_STEP;
      MrcGetSetChStrb(MrcData, Controller, Channel, Byte, DataRxD0PiCb, GsmMode, &GetSetVal);
      break;
    case OptRxCbComp:
      GetSetVal = Off * RX_CB_COMP_STEP;
      MrcGetSetNoScope(MrcData, VccDllRxD0PiCb, GsmMode, &GetSetVal);
      break;
    case OptRxBias:
      GetSetVal = Off * RX_BIAS_STEP;
      MrcGetSetChStrb(MrcData, Controller, Channel, Byte, RxBiasCtl, GsmMode, &GetSetVal);
      break;
    case OptRxVrefVttDecap:
      GetSetVal = Off;
      MrcGetSetChStrb(MrcData, Controller, Channel, Byte, RxVrefVttDecap, GsmMode, &GetSetVal);
      break;
    case OptRxVrefVddqDecap:
      GetSetVal = Off;
      MrcGetSetChStrb(MrcData, Controller, Channel, Byte, RxVrefVddqDecap, GsmMode, &GetSetVal);
      break;
    case OptVccDLLBypass:
      GetSetVal = Off;
      MrcGetSetChStrb(MrcData, Controller, Channel, Byte, GsmIocVccDllFFControlBypass_V, GsmMode, &GetSetVal);
      MrcGetSetChStrb(MrcData, Controller, Channel, Byte, GsmIocVccDllControlBypass_V, GsmMode, &GetSetVal);
      break;
    case OptPanicVttDnLp:
      GetSetVal = Off;
      MrcGetSetNoScope(MrcData, PanicVttDnLp, GsmMode, &GetSetVal);
      break;
    case OptVddq:
      // Sweep Vddq from high to low
      GetSetVal = -VDDQ_STEP * Off;
      // TGL_POWER_TRAINING_VDDQ Need to set Vddq via FIVR register:
      // MC_BIOS_REQ_PCU.REQ_VOLTAGE_DATA -> Set by MRC, read by Pcode (11 bits, 2.5 mV per tick)
      // MC_BIOS_DATA.VDDQ_TX_VOLTAGE -> Set by Pcode to what was actually requested from FIVR (may not be the same as what MRC requested)(11 bits, 2.5 mV per tick)
      // MAILBOX_BIOS_CMD_READ_BIOS_MC_REQ_ERROR -> Set by Pcode, read by MRC to see if there are any error codes indicating problems during setting Vddq. BIOS should retry or throw an error if this occurs.
      // Details for how to set Vddq in BIOS are here: https://sharepoint.amr.ith.intel.com/sites/tgldoc/tglhas/Shared%20Documents/Power%20Management%20and%20Delivery/Features/Vddq_tx%20FIVR/Vddq_tx%20FIVR.html#active-power-management
      // TGL_POWER_TRAINING_VDDQ There are some PHY init settings that depend on the Vddq setting, and those need to be copied from the PHY init and added to a function that is called as a side effects of the Vddq get/set.
      // MrcGetSetDdrIoGroupController0 (MrcData, Vddq, GsmMode, &GetSetVal);
      break;
    case OptDimmRon:
    case OptDimmOdtWr:
    case OptDimmOdtNom:
    case OptDimmOdtPark:
    case OptDimmOdtParkNT:
    case OptDimmOdtNomNT:
    case OptDimmOdtCA:
      if (OptParam == OptDimmOdtNomNT) {
        OptParam = OptDimmOdtNom;
      } else if (OptParam == OptDimmOdtParkNT) {
        OptParam = OptDimmOdtPark;
      }
      Status = GetDimmOptParamValues (MrcData, OptParam, &DimmOptParamVals, &NumDimmOptParamVals);
      if (Status == mrcSuccess) {
        Index = (UINT8) Off;
        if (DimmOptParamVals != NULL) {
          if (MRC_POWER_TRAINING_DEBUG) {
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "UpdateOptParamOffset DRAM Param: Controller = %d, Channel = %d, RankMask = %d, OptParam = %s, DimmOptParamVal = %d \n", Controller, Channel, Ranks & RankMask, TOptParamOffsetString[OptParam], DimmOptParamVals[Index]);
          }

          if (ChannelOut->DimmCount == 2) {
            for (Dimm = 0; Dimm < MAX_DIMMS_IN_CHANNEL; Dimm++) {
              NumRanksInCurrentDimm = MrcGetRankInDimm (MrcData, Controller, Channel, Dimm);
              if (NumRanksInCurrentDimm != 0) {
                if (NumRanksInDimm != 0 && NumRanksInCurrentDimm != NumRanksInDimm) {
                  break;
                }
                NumRanksInDimm = NumRanksInCurrentDimm;
              }
            }
          }

          for (Dimm = 0; Dimm < MAX_DIMMS_IN_CHANNEL; Dimm++) {
            NumRanksInDimm = MrcGetRankInDimm (MrcData, Controller, Channel, Dimm);
            RankMask = Ranks & ((NumRanksInDimm == 1) ? DIMM_TO_VARIABLE_RANK_MASK (Dimm, 0x1) : DIMM_TO_VARIABLE_RANK_MASK (Dimm, 0x3));

            if (RankMask) {
              TotalIndex = Index;
              if (TotalIndex < 0) {
                TotalIndex = 0;
              } else if (TotalIndex > NumDimmOptParamVals - 1) {
                TotalIndex = NumDimmOptParamVals - 1;
              }
              SetDimmParamValue (MrcData, Controller, Channel, RankMask, OptParam, DimmOptParamVals[TotalIndex], UpdateHost);
            }
          }
        }
      }
      break;
  }
}

/**
  Slightly penalize any Asymmetry in margin

  @param[in] NegEdge - Negative edge of the margin
  @param[in] PosEdge - Positive edge of the margin

  @retval p2p - Width/Height reduced by the asymmetric difference in margin.
**/
UINT16
EffectiveMargin (
  IN const UINT16 NegEdge,
  IN const UINT16 PosEdge
  )
{
  INT16 p2p;
  UINT16 p2pDiff;

  p2p     = 2 * (PosEdge + NegEdge);
  p2pDiff = PosEdge - NegEdge;

  if (PosEdge > NegEdge) {
    p2p -= p2pDiff;
  } else {
    p2p += p2pDiff;
  }

  return p2p / 2;
}

/**
  This function does a running average on Margins in two dimentional fashion.

  @param[in,out] Margins - Margins to average in a 1D array.
  @param[in]     MLen    - Determines the Y-Dimension lengths
  @param[in]     XDim    - Determines the X-Dimension lengths
  @param[in]     XMin    - Used to skip the first elements in the Margin when averaging.
  @param[in]     CScale  - Used to place more weight on the center point.

  @retval Nothing
**/
void
RunningAverage2D (
  IN OUT UINT16       Margins[MAX_OPT_POINTS],
  IN     const UINT8  MLen,
  IN     const UINT8  XDim,
  IN     const UINT8  XMin,
  IN     const UINT8  CScale
)

{
  UINT8  XMax;
  UINT8  YMax;
  UINT16 TMargins[MAX_OPT_POINTS];
  UINT8  i;
  UINT8  x;
  UINT8  y;
  UINT8  xo;
  UINT8  yo;
  UINT8  XOff;
  INT8   YOff;

  if (MLen == 1) {
    return;
  }
  XMax  = XDim - 1;
  YMax  = ((MLen + XDim - 1) / XDim) - 1; // Ceiling to int in case the matrix is not fully populated

  for (i = 0; i < MLen; i++) {
    x = (i % XDim);
    y = (i / XDim);

    // Center Point
    TMargins[i] = Margins[i] * (CScale - 1); // Also add margin at the centerpoint below
    // Sum up surrounding results
    for (xo = 0; xo < 3; xo++) {
      XOff = x + xo - 1;
      // Avoid negative numbers on XOff
      if ((x == 0) && (xo == 0)) {
        XOff = 0;
      }
      // (x < XMin) allows averaging across points (1;0) and (2;0)
      if ((XOff < XMin) && (x < XMin)) {
        XOff = x; // RxEq special case.  Skip averaging on Col0/Col1
      }

      if (XOff > XMax) {
        XOff = XMax;
      }

      for (yo = 0; yo < 3; yo++) {
        YOff = y + yo - 1;
        if (YOff < 0) {
          YOff = 0;
        }

        if (YOff > YMax) {
          YOff = YMax;
        }
        // Avoid averaging with unpopulated matrix elements when dealing with partially populated matrices
        if ((XDim * YOff + XOff) > (MLen - 1)) {
          YOff = YOff - 1;
        }

        TMargins[i] += Margins[XDim * YOff + XOff];
      }
    }
  }
  // Copy TempMargins back over to real margins
  for (i = 0; i < MLen; i++) {
    Margins[i] = TMargins[i] / (8 + CScale); // Added div to maintain margin scaling
  }

  return;
}

/**
  This function does a running average on Margins in two dimentional fashion.

  @param[in,out] Margins - Margins to average
  @param[in]     Test    - Selects the Margins to average
  @param[in]     MLen    - Determines the Y-Dimension lengths
  @param[in]     XDim    - Determines the X-Dimension lengths
  @param[in]     XMin    - Used to skip the first elements in the Margin when averaging.
  @param[in]     CScale  - Used to place more weight on the center point.

  @retval Nothing
**/
void
Fill2DAverage (
  IN     MrcParameters *const MrcData,
  IN OUT UINT16               Margins[2][MAX_OPT_POINTS],
  IN     const UINT8          Test,
  IN     const UINT8          MLen,
  IN     const UINT8          XDim,
  IN     const UINT8          XMin,
  IN     const UINT8          CScale
)
{
  UINT8  XMax;
  UINT8  YMax;
  UINT16 TMargins[MAX_OPT_POINTS];
  UINT8  i;
  UINT8  x;
  UINT8  y;
  UINT8  xo;
  UINT8  yo;
  UINT8  XOff;
  INT8   YOff;
  UINT8  Edge;
  INT16  Gradient;
  INT16  MaxGradient;

  XMax  = XDim - 1;
  YMax  = ((MLen + XDim - 1) / XDim) - 1; // Ceiling to int in case the matrix is not fully populated

  for (i = 0; i < MLen; i++) {
    if (Margins[Test][i]) { // skip already populated entries
    continue;
    }

    x = (i % XDim);
    y = (i / XDim);
    Edge = 0;

//     if (i == 14)
//       MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "x=%d y=%d Margins[%d][%d]=%d\n", x, y, Test, i, Margins[Test][i]);

    // Center Point
    TMargins[i] = Margins[Test][i] * (CScale - 1); // Also add margin at the centerpoint below
    // Sum up surrounding results
    for (xo = 0; xo < 3; xo += 2) {
      XOff = x + xo - 1;
      // Avoid negative numbers on XOff
      if (((INT8) XOff) < 0) {
        //XOff = 0;
        Edge++;
        continue;
      }
      // (x < XMin) allows averaging across points (1;0) and (2;0)
      if ((XOff < XMin) && (x < XMin)) {
        XOff = x; // RxEq special case.  Skip averaging on Col0/Col1
      }

      if (XOff > XMax) {
        //XOff = XMax;
        Edge++;
        continue;
      }
      // Avoid averaging with unpopulated matrix elements when dealing with partially populated matrices
      if ((XDim * y + XOff) > (MLen - 1)) {
        Edge++;
        continue;
      }
      TMargins[i] += Margins[Test][XDim * y + XOff];
//       if (i == 14)
//         MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "x=%d y=%d TMargins[%d]=%d Edge=%d\n", x, y, i, TMargins[i],Edge);
    }
    for (yo = 0; yo < 3; yo += 2) {
      YOff = y + yo - 1;
      if (YOff < 0) {
        // YOff = 0;
        Edge++;
        continue;
      }

      if (YOff > YMax) {
        // YOff = YMax;
        Edge++;
        continue;
      }

      TMargins[i] += Margins[Test][XDim * YOff + x];
//       if (i == 14)
//         MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "x=%d y=%d TMargins[%d]=%d Edge=%d\n", x, y, i, TMargins[i],Edge);
    }
    // Copy TempMargins back over to real margins
//     if (Edge > 0) {
//       Margins[Test][i] = Margins[Test][i] * 8 / 10; // Penalize the edges by decreaseing margin by 20%
//     }
    Margins[Test][i] = TMargins[i] / (4 - Edge + CScale - 1); // Added div to maintain margin scaling
      //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "x=%d y=%d TMargins[%d]=%d Margins[%d][i]=%d Edge=%d\n", x, y, i, TMargins[i], Test, Margins[Test][i], Edge);
  }

  // penalize for high margin gradient
  for (i = 0; i < MLen; i++) {
    x = (i % XDim);
    y = (i / XDim);
    MaxGradient = 0;
    // Sum up surrounding results
    for (xo = 0; xo < 3; xo += 2) {
      XOff = x + xo - 1;
      // Avoid negative numbers on XOff
      if ((x == 0) && (xo == 0)) {
        XOff = 0;
      }
      if (XOff > XMax) {
        XOff = XMax;
      }
      // Avoid averaging with unpopulated matrix elements when dealing with partially populated matrices
      if ((XDim * y + XOff) > (MLen - 1)) {
        XOff = XOff - 1;
      }
      Gradient = Margins[Test][XDim * y + x] - Margins[Test][XDim * y + XOff];
      //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "x=%d y=%d Gradient=%d Margins[Test][XDim * y + x] = %d Margins[Test][XDim * y + XOff] = %d XOff = %d\n",x, y, Gradient, Margins[Test][XDim * y + x], Margins[Test][XDim * y + XOff], XOff);
      if (Gradient > MaxGradient) {
        // if we loose margin update MaxGradient
        MaxGradient = Gradient;
      }
    }
    for (yo = 0; yo < 3; yo += 2) {
      YOff = y + yo - 1;
      if (YOff < 0) {
        YOff = 0;
      }

      if (YOff > YMax) {
        YOff = YMax;
      }

      // Avoid averaging with unpopulated matrix elements when dealing with partially populated matrices
      if ((XDim * YOff + x) > (MLen - 1)) {
        YOff = YOff - 1;
      }

      Gradient = (Margins[Test][XDim * y + x] - Margins[Test][XDim * YOff + x]);
      //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "x=%d y=%d Gradient=%d Margins[Test][XDim * y + x] = %d Margins[Test][XDim * y + XOff] = %d YOff = %d\n",x, y, Gradient, Margins[Test][XDim * y + x], Margins[Test][XDim * y + YOff], YOff);
      if (Gradient > MaxGradient) {
        MaxGradient = Gradient;
      }
      // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "x=%d y=%d TMargins[%d]=%d Edge=%d\n", x, y, i, TMargins[i],Edge);
    }
    // save MaxGradient in Temp array and clip for max margin.
    if (MaxGradient > Margins[Test][i]) {
      MaxGradient = Margins[Test][i];
    }
    TMargins[i] = MaxGradient;
    //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "x=%d y=%d  Margins[%d][%d]=%d MaxGradient=%d \n", x, y, i, Test, Margins[Test][i], TMargins[i]);
  }
  // apply the MaxGradient to Original array
  for (i = 0; i < MLen; i++) {
    Margins[Test][i] -= TMargins[i];
  }
  return;
}

/**
  This function takes in 2D array of Margins: MarginType / Parameter Index.
  This index to the array represents some arbitrary parameter value that we are optimizing.
  The function will look for up to MAX_TRADEOFF_TYPES entries to optimize on.
  OptResultByte will store the results of the optimization, and various other data.

  In addition to optimizing for margin, this function can also optimize for power.
  GoodPowerLimit is an array that sets level where power is more important than margin.
    i.e. Any points where ((Margin[0]>GoodPowerLimit[0]) & (Margin[1]>GoodPowerLimit[1]) & ... )
  To avoid overflow, this function will automatic scale margins to fit in UINT64

  @param[in]     MrcData          - The global MRC data structure.
  @param[in,out] OptResultByte    - Structure containing the optimized results.
  @param[in]     InputMargins     - Margins we are optimizing
  @param[in]     MarginsLength    - The length of InputMargins
  @param[in]     LenMargin        - The length of InputMargins we are optimizing (0 - LenMargin -1).
  @param[in]     Scale            - Controls the relative importance on Margins[0] vs. [1] ...
                                      ex: To make Margins[0] twice as important, set Scale = [1, 2, 2, 2, 2].
                                      Since the search optimizes the lowest margin, increasing [1:4] makes 0 more important.
                                      This function can be used to optimize only Margin[0] by setting Scale = [1, 0, 0, 0, 0].
  @param[in]     UPMOptimize      - Optimize only for UPM limit for selected params, so if they pass UPM they do not affect the score.
  @param[in]     EnSq             - Enables the square root term in the optimization functions to make the tradeoff steeper.
  @param[in]     AveN             - The number of points used for the averaging filter.
  @param[in]     IncEnds          - Controls if the endpoints are to be included.
  @param[in]     ScaleM           - Controls the scaling of the middle point in 1-D average filter.
  @param[in]     GoodPowerLimit   - The power limit above which we only trade-off for power and not margin.
  @param[in]     OptimizationMode - 0:    Returns first good margin limit point.
                                    1-4:  Return the first index that meets GoodPowerLimit and lowest power.
                                            OptimizationMode is power column index.
                                    5-99: Return the index that meets GoodPowerLimit and >= % of the Max Optimization result.
                                    >100: Returns the highest Optimization Result.
  @param[in]     GuardBand        - Signed offest to check if margin drop is acceptable.  Save good guardband
                                    in OptResultByte.

  @retval Nothing.
**/
void
FindOptimalTradeOff (
  IN     MrcParameters      *const  MrcData,
  IN OUT OptResultsPerByte          *OptResultByte,
  IN     UINT16                     *InputMargins,
  IN     UINT8                      MarginsLength,
  IN     UINT8                      LenMargin,
  IN     const UINT8                Scale[MAX_TRADEOFF_TYPES],
  IN     const UINT8                UPMOptimize[MAX_TRADEOFF_TYPES],
  IN     UINT8                      EnSq,
  IN     UINT8                      AveN,
  IN     UINT8                      IncEnds,
  IN     UINT8                      ScaleM,
  IN     UINT16                     GoodPowerLimit[],
  IN     UINT8                      OptimizationMode,
  IN     INT8                       GuardBand
  )
{
  const MrcInput      *Inputs;
  MrcDebug            *Debug;
  const MRC_FUNCTION  *MrcCall;
  UINT32               PostMar[MAX_TRADEOFF_TYPES][MAX_OPT_POINTS];
  UINT32               SMaxPost[MAX_TRADEOFF_TYPES];
  UINT32               MaxPost[MAX_TRADEOFF_TYPES];
  UINT32               MinPost[MAX_TRADEOFF_TYPES];
  UINT16               GoodPwrLimitPost[MAX_TRADEOFF_TYPES];
  UINT32               ScaleMin;
  UINT8                Nby2;
  UINT8                EqOrder;
  UINT8                MarginArray;
  UINT8                yArr;
  UINT8                x;
  UINT8                i;
  UINT8                Off;
  INT16                xEff;
  UINT8                NumBits;
  UINT32               localY;
  UINT8                Shift;
  UINT8                Adder;
  UINT8                Start;
  UINT8                Stop;
  UINT64               Result;
  UINT64               LocalResult;
  UINT64               MaxR;
  UINT64               MarginLimit;
  UINT8                BestX;
  UINT8                PowerX;
  UINT8                FoundPwrOpt;
  UINT8                NumCalcArr;
  INT8                 StepSize;
  UINT8                MarginDropPercent;
  UINT32               MinPost1;
  BOOLEAN              GoodPower;

  Inputs            = &MrcData->Inputs;
  MrcCall           = Inputs->Call.Func;
  Debug             = &MrcData->Outputs.Debug;
  MarginDropPercent = 10;  // 10% loss of margin is a bad guardband offset.
  EqOrder           = 0; // Is the optimization equation: X^1, X^2, X^5
  Result            = 0;
  MaxR              = 0;
  BestX             = 0;
  PowerX            = 0;
  FoundPwrOpt       = 0;
  MrcCall->MrcSetMemDword (MaxPost, sizeof (MaxPost) / sizeof (UINT32), 1);
  MrcCall->MrcSetMemDword (SMaxPost, sizeof (SMaxPost) / sizeof (UINT32), 1);
  MrcCall->MrcSetMemDword (MinPost, sizeof (MinPost) / sizeof (UINT32), 0xFFFFFFFF);
  MrcCall->MrcSetMemWord (GoodPwrLimitPost, sizeof (GoodPwrLimitPost) / sizeof (UINT16), 0);
  MrcCall->MrcSetMem ((UINT8 *) PostMar, sizeof (PostMar), 0);
  MrcCall->MrcSetMem ((UINT8 *) OptResultByte, sizeof (OptResultsPerByte), 0);

  if (AveN <= 0) {
    AveN = 1;
  }
  Nby2    = (AveN >> 1);

  // Process Raw Margins Results with a running average filter of AveN
  for (MarginArray = 0; MarginArray < MAX_TRADEOFF_TYPES; MarginArray++) {
    // Scale GoodPowerLimit to match PostMar results
    GoodPwrLimitPost[MarginArray] = GoodPowerLimit[MarginArray] * (AveN + ScaleM - 1) * Scale[MarginArray];

    for (x = 0; x < LenMargin; x++) {
      if (Scale[MarginArray] == 0) {
        // Not in the game
        MinPost[MarginArray] = PostMar[MarginArray][x] = 1;
      } else {
        if (x == 0) {
          // Update EqOrder once for each MarginArray value with a non-zero scale factor.
          //   e.g.:so for {RdT,RdV,0,0} it will be = 2
          EqOrder += 1;
        }

        for (Off = 0; Off < AveN; Off++) {
          xEff = x + Off - Nby2;
          if (xEff < 0) {
            PostMar[MarginArray][x] += *(InputMargins + MarginArray * MarginsLength + 0);  // InputMargins[MarginArray][0];
          } else if (xEff >= LenMargin) {
            PostMar[MarginArray][x] += *(InputMargins + MarginArray * MarginsLength + LenMargin - 1);
          } else if (x == xEff) {
            PostMar[MarginArray][x] += ScaleM * *(InputMargins + MarginArray * MarginsLength + xEff);
          } else {
            PostMar[MarginArray][x] += *(InputMargins + MarginArray * MarginsLength + xEff);
          }
        }
        // save none scaled margins after avg filtering
        OptResultByte->Margins[MarginArray][x].EW = (UINT16) PostMar[MarginArray][x] / (AveN + ScaleM - 1);
        PostMar[MarginArray][x] *= Scale[MarginArray];
        if (MaxPost[MarginArray] < PostMar[MarginArray][x]) {
          MaxPost[MarginArray] = PostMar[MarginArray][x];
        }

        if (MinPost[MarginArray] > PostMar[MarginArray][x]) {
          MinPost[MarginArray] = PostMar[MarginArray][x];
        }
      }
    }

    if (Scale[MarginArray] == 0) {
      continue;
    }

    SMaxPost[MarginArray]               = MaxPost[MarginArray];
    OptResultByte->Scale[MarginArray]   = Scale[MarginArray];
    OptResultByte->MaxPost[MarginArray] = MaxPost[MarginArray] / (AveN + ScaleM - 1);
    OptResultByte->MinPost[MarginArray] = MinPost[MarginArray] / (AveN + ScaleM - 1);
  }
  // Sort Array
  MrcBsort (SMaxPost, MAX_TRADEOFF_TYPES);
  // Calculate Number of Bits Required to represent this number. Make sure to handle care of EnSq
  NumBits = 0;

  for (MarginArray = 0; MarginArray < MAX_TRADEOFF_TYPES; MarginArray++) {
    if (MarginArray < (MAX_TRADEOFF_TYPES - 1)) {
      // if EnSq we do Max^2 so the num get twice the bits...
      localY = SMaxPost[MarginArray];
      if (EnSq) {
        localY = (localY * localY);
      }

      NumBits += MrcLog2 (localY);
    } else {
      NumBits += MrcLog2 (SMaxPost[MarginArray]);
    }
  }

  NumBits += 11; // reserved another 10 bits for division in order to format for printing ; 3 for adding - up to 8
  // EqOrder for square terms
  if (EnSq) {
    EqOrder = (EqOrder + (EqOrder - 1));
  }
  // Handle Potential Saturation
  if (NumBits > 64) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Warning number of bits exceeds 64 bit : %d \n", NumBits);
    // Shift all numbers to reduce final result to be less than 32 bits.  Round Up
    Shift = EqOrder ? (NumBits - 64 + EqOrder - 1) / EqOrder : 0;
    // RoundUp Adder
    Adder = (1 << (Shift - 1));
    // Divide by (1<<Shift) and Round Up
    for (MarginArray = 0; MarginArray < MAX_TRADEOFF_TYPES; MarginArray++) {
      MaxPost[MarginArray]       = (MaxPost[MarginArray] + Adder) >> Shift;
      GoodPwrLimitPost[MarginArray]  = (GoodPwrLimitPost[MarginArray] + Adder) >> Shift;
      for (x = 0; x < LenMargin; x++) {
        PostMar[MarginArray][x] = (PostMar[MarginArray][x] + Adder) >> Shift;
      }
    }
  }
  // Calculate Square terms:
  if (EnSq) {
    for (MarginArray = 0; MarginArray < MAX_TRADEOFF_TYPES; MarginArray++) {
      MaxPost[MarginArray] = MaxPost[MarginArray] * MaxPost[MarginArray];
    }
  }
  // Set Limits for Search
  Start = 0;
  Stop  = LenMargin;
  if ((IncEnds == 0) && (LenMargin > AveN)) {
    if (Nby2 > 0) {
      Start++;
      Stop--;
    }
  }
  if (UPMOptimize != NULL) {
    for (x = Start; x < Stop; x++) {
      for (MarginArray = 0; MarginArray < MAX_TRADEOFF_TYPES; MarginArray++) {
        if (UPMOptimize[MarginArray] == 0 || Scale[MarginArray] == 0) {
          continue; // not need to calculate those
        }
        if (PostMar[MarginArray][x] > GoodPwrLimitPost[MarginArray]) {
          PostMar[MarginArray][x] = GoodPwrLimitPost[MarginArray];
          OptResultByte->Margins[MarginArray][x].EW = (UINT16) PostMar[MarginArray][x] / (AveN + ScaleM - 1) / Scale[MarginArray];
        }
      }
    }
  }
  // Find Optimal Point from Margin Point of View
  // Combine the points using the formula:
  //   Max0*Max1*Max2*Post3 + Max1*Max2*Max3*Post0 + Max2*Max3*Max0*Post1 +
  //   Max3*Max0*Max1*Post2 + Scale*min(Post0,Post1,Post2,Post3)^EqOrder
  //   Scale = 1 + (10*(SMaxPost[0]-SMaxPost[1]))/SMaxPost[MAX_TRADEOFF_TYPES-1]
  for (x = Start; x < Stop; x++) {
    Result      = 0;
    MinPost1    = 0xFFFFFFFF;
    GoodPower   = TRUE;
    NumCalcArr  = 0;
    for (MarginArray = 0; MarginArray < MAX_TRADEOFF_TYPES; MarginArray++) {
      if (Scale[MarginArray] == 0) {
        continue; // not need to calculate those
      }
      NumCalcArr++; // Count the number of MarginTypes in the optimization function.
      // Find Min of all PostMar at offset x
      // Does this point meet the min power Margin requirements?
      if (Scale[MarginArray] > 0) {
        if (MinPost1 > PostMar[MarginArray][x]) {
          MinPost1 = PostMar[MarginArray][x];
        }

        if (PostMar[MarginArray][x] < GoodPwrLimitPost[MarginArray]) {
          GoodPower = FALSE;
        }
      }
      // Calculate this portion of result
      LocalResult = 1;
      for (yArr = 0; yArr < MAX_TRADEOFF_TYPES; yArr++) {
        if (Scale[yArr] == 0) {
          continue; // not need to calculate those
        }

        if (MarginArray == yArr) {
          continue;
        } else {
          LocalResult = MrcCall->MrcMultU64x32 (LocalResult, MaxPost[yArr]);
        }
      }

      Result += MrcCall->MrcMultU64x32 (LocalResult, PostMar[MarginArray][x]);
    }
    // Add in (MinPost ^ EqOrder)
    // If NumCalcArr is 0, set it to 1 so that it still in the range of array size.
    //  This will cause PowerCalcIndex to underflow.  Set to 1 in this case.
    if (NumCalcArr == 0) {
      MRC_DEBUG_MSG (
        Debug,
        MSG_LEVEL_NOTE,
        "Error: wrong input parameter caused NumCalcArr = 0 when calling FindOptimalTradeOff()\n"
        );
      NumCalcArr = 1;
    }

    ScaleMin = 1 + (10 * (SMaxPost[0] - SMaxPost[1])) / SMaxPost[NumCalcArr - 1];
    if (ScaleMin > 5) {
      ScaleMin = 5;
    }

    ScaleMin  = 1;
    LocalResult    = ScaleMin;
    for (i = 0; i < EqOrder; i++) {
      LocalResult = MrcCall->MrcMultU64x32 (LocalResult, MinPost1);
    }

    Result += LocalResult;

    if (Result > MaxR) {
      MaxR  = Result;
      BestX = x; // save first highest function result offset
    }

    OptResultByte->Result[x] = Result;
    // Find Optimal Point from Power Point of View.
    // All the margin types must be >= GoodPowerLimit
    if (GoodPower) {
      if (FoundPwrOpt == 0) {
        FoundPwrOpt = 1;
        PowerX      = x;
      } else {
        // Find the first lowest power point.
        // PostMar[Power] is inverted: higher number is lower power.
        if ((OptimizationMode > 0) && (OptimizationMode < MAX_TRADEOFF_TYPES)) {
          if (PostMar[OptimizationMode][x] > PostMar[OptimizationMode][PowerX]) {
            PowerX = x;
          }
        }
      }
    }
  } // end shmoo offsets
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, (MaxR == 0) ? "Warning : MaxR is Zero !!!\n" : "");
  // At this point, BestX will have the highest Margin Result index.  PowerX will have
  // the highest or lowest index that is marked good based on MarginLimits.
  if (FoundPwrOpt) {
    if (OptimizationMode < MAX_TRADEOFF_TYPES) {
      BestX = PowerX;
    } else if (OptimizationMode < 100) {
      for (x = PowerX; x < Stop; x++) {
        Result = (UINT32) (MrcCall->MrcDivU64x64 (MrcCall->MrcMultU64x32 (OptResultByte->Result[x], 100), MaxR, NULL));
        if (Result >= OptimizationMode) {
          BestX = x;
          break;
        }
      }
    }
  }

  OptResultByte->Best     = BestX;
  OptResultByte->MaxR     = MaxR;
  // Apply a guard band to the best setting, clamped at edges of the search.
  if (GuardBand != 0) {
    // Determine step direction and limit to the search edge.
    if (GuardBand < 0) {
      StepSize = 1;
      Off = ((BestX + GuardBand) < Start) ? Start : (BestX + GuardBand);
    } else {
      StepSize = -1;
      Off = ((BestX + GuardBand) >= Stop) ? (Stop - 1) : (BestX + GuardBand);
    }
    // Check each test for margin drop of MarginDropPercent.
    // If any test fails, we step towards the original selection.
    MarginLimit = MrcCall->MrcMultU64x32 (OptResultByte->Result[BestX], (100 - MarginDropPercent));
    MarginLimit = MrcCall->MrcDivU64x64 (MarginLimit, 100, NULL);
    for(; (Off != BestX); Off += StepSize) {
      if (OptResultByte->Result[Off] > MarginLimit) {
        break;
      }
    }

    OptResultByte->GuardBand = Off - (INT8) BestX;
  }

  return;
}

/**
  This function implements Turn Around Timing training.
  Optimize TA ODT Delay and Duration

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - If it succeeds return mrcSuccess.
**/
MrcStatus
MrcTurnAroundTiming (
  IN MrcParameters *const MrcData
  )
{
  MrcDebug            *Debug;
  MrcOutput           *Outputs;
  MrcChannelOut       *ChannelOut;
  MrcStatus           Status;
  INT64               GetSetVal;
  INT64               MaxVal;
  UINT32              Controller;
  UINT32              Channel;
  UINT32              IpChannel;
  UINT32              ChannelIncrement;
  UINT8               MaxChannel;
  UINT8               RankMaskCh;
  UINT8               RankMask;
  BOOLEAN             RunDD;
  BOOLEAN             RunDR;
  BOOLEAN             Lpddr;
  static const UINT8  ParamListX[6]  = {RdV, RdT, WrV, WrT, RcvEnaX, WrLevel};
  static const UINT8  TestListRdX[3] = {RdV, RdT, RcvEnaX};
  static const UINT8  TestListWr[2]  = {WrV, WrT};
  static const INT8   ClkShifts[2]   = {-7, 7};
  INT8                Range;
  UINT8               GuardBand;
  UINT8               Update;
  UINT8               LoopCount;
  UINT32              RelaxBy;
  UINT32              MinNomWR2WR_dr;
  UINT32              NomWR2RD_dr;
  UINT32              NomWR2RD_dd;
  UINT32              NomRD2WR_dd;
  UINT32              NomRD2WR_dr;
  UINT32              NomRD2RD_dr;
  UINT32              NomRD2RD_dd;
  UINT32              NomWR2WR_dr;
  UINT32              NomWR2WR_dd;
  UINT32              Offset;
  MC0_CH0_CR_SC_PCIT_STRUCT  ScPcit;
  MC0_CH0_CR_SC_PCIT_STRUCT  ScPcitSave[MAX_CONTROLLER][MAX_CHANNEL];

#ifdef MRC_DEBUG_PRINT
  UINT8 GroupIdx;
  UINT8 TaTypeIdx;
  static const GSM_GT PrintGroups [4][2] = {{GsmMctRDRDdr, GsmMctRDRDdd},
                                            {GsmMctRDWRdr, GsmMctRDWRdd},
                                            {GsmMctWRRDdr, GsmMctWRRDdd},
                                            {GsmMctWRWRdr, GsmMctWRWRdd}};
  static const char *TaString[] = {"RdRd", "RdWr", "WrRd", "WrWr"};
  static const char *TypeStr[] = {"Dr", "Dd"};
#endif

  Status        = mrcSuccess;
  RankMaskCh    = 0;
  Update        = 1;
  LoopCount     = 10;  //1024 per Rank
  RelaxBy       = 4;
  RunDD         = FALSE;
  RunDR         = FALSE;
  NomWR2RD_dr   = 0;
  NomWR2RD_dd   = 0;
  NomRD2WR_dd   = 0;
  NomRD2WR_dr   = 0;
  NomRD2RD_dr   = 0;
  NomRD2RD_dd   = 0;
  NomWR2WR_dr   = 0;
  NomWR2WR_dd   = 0;
  MaxVal        = 0;
  RankMask      = MrcData->Outputs.ValidRankMask;
  Range         = 1;

  Outputs       = &MrcData->Outputs;
  Debug         = &Outputs->Debug;
  MaxChannel    = Outputs->MaxChannels;
  Lpddr         = Outputs->Lpddr;

  ChannelIncrement = Lpddr ? 2 : 1;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel += ChannelIncrement) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      ChannelOut = &Outputs->Controller[Controller].Channel[Channel];

      RankMaskCh  = ChannelOut->ValidRankBitMask;
      RunDD       = RunDD || (ChannelOut->DimmCount == 2);
      RunDR       = RunDR || ((RankMaskCh & 0xC) == 0xC) || ((RankMaskCh & 0x3) == 0x3);
      RunDD       = 0;
      MRC_DEBUG_MSG (
        Debug,
        MSG_LEVEL_NOTE,
        "Mc %d, Channel %d: RunDR = 0x%x, RunDD = 0x%x, RankMaskCh = 0x%x\n",
        Controller,
        Channel,
        RunDR,
        RunDD,
        RankMaskCh
      );

      // Use nominal values (previously programmed) +1 and -1 to test for Gear1.  Gear2 will need to do +2 and -2 with steps of 2.
      // Taking worst case of both channels.
      // Ideally the Cliff routine should support offset per channel or better make the param a real offset (not the abs value)
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRRDdr, ReadFromCache, &GetSetVal);
      NomWR2RD_dr = MAX (NomWR2RD_dr, (UINT8) GetSetVal);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRRDdd, ReadFromCache, &GetSetVal);
      NomWR2RD_dd = MAX (NomWR2RD_dd, (UINT8) GetSetVal);

      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRdr, ReadFromCache, &GetSetVal);
      NomRD2WR_dr = MAX (NomRD2WR_dr, (UINT8) GetSetVal);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRdd, ReadFromCache, &GetSetVal);
      NomRD2WR_dd = MAX (NomRD2WR_dd, (UINT8) GetSetVal);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDRDdr, ReadFromCache, &GetSetVal);
      NomRD2RD_dr = MAX (NomRD2RD_dr, (UINT8) GetSetVal);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDRDdd, ReadFromCache, &GetSetVal);
      NomRD2RD_dd = MAX (NomRD2RD_dd, (UINT8) GetSetVal);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRWRdr, ReadFromCache, &GetSetVal);
      NomWR2WR_dr = MAX (NomWR2WR_dr, (UINT8) GetSetVal);
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRWRdd, ReadFromCache, &GetSetVal);
      NomWR2WR_dd = MAX (NomWR2WR_dd, (UINT8) GetSetVal);

    // Change PCIT to 0xFF
    // This allows proper tRDWR_dg stress without extra PRE/ACT cycles.
      IpChannel   = LP_IP_CH (Lpddr, Channel);
      Offset = OFFSET_CALC_MC_CH (MC0_CH0_CR_SC_PCIT_REG, MC1_CH0_CR_SC_PCIT_REG, Controller, MC0_CH1_CR_SC_PCIT_REG, IpChannel);
      ScPcit.Data = MrcReadCR (MrcData, Offset);
      ScPcitSave[Controller][IpChannel] = ScPcit;
      ScPcit.Bits.PCIT = 0xFF;
      MrcWriteCR (MrcData, Offset, ScPcit.Data);
    } // for Channel
  } // for Controller

  if (Outputs->Gear2) {
    Range = 2;
    MinNomWR2WR_dr = MAX (NomWR2WR_dr - Range, 7); // 7 QClks
    MinNomWR2WR_dr = 2 * (DIVIDECEIL (MinNomWR2WR_dr, 2));
  } else {
    MinNomWR2WR_dr = MAX (NomWR2WR_dr - Range, DIVIDECEIL (7, 2)); // 7 QClks in DClks
  }

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel += ChannelIncrement) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      // Adjust initial values to be more relaxed so any detected failures are due to the parameter being tested.
      if (RunDR) {
        // Different Rank, Same DIMM
        MrcGetSetLimits (MrcData, GsmMctRDRDdr, NULL, &MaxVal, NULL);
        GetSetVal = MIN ((UINT32) MaxVal, NomRD2RD_dr + RelaxBy);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDRDdr, WriteToCache, &GetSetVal);
        MrcGetSetLimits (MrcData, GsmMctWRWRdr, NULL, &MaxVal, NULL);
        GetSetVal = MIN ((UINT32) MaxVal, NomWR2WR_dr + RelaxBy);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRWRdr, WriteToCache, &GetSetVal);

        MrcGetSetLimits (MrcData, GsmMctWRRDdr, NULL, &MaxVal, NULL);
        GetSetVal = MIN ((UINT32) MaxVal, NomWR2RD_dr + RelaxBy);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRRDdr, WriteToCache, &GetSetVal);
        MrcGetSetLimits (MrcData, GsmMctRDWRdr, NULL, &MaxVal, NULL);
        GetSetVal = MIN ((UINT32) MaxVal, NomRD2WR_dr + RelaxBy);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRdr, WriteToCache, &GetSetVal);
      }
      if (RunDD) {
        // Different DIMMs
        MrcGetSetLimits (MrcData, GsmMctWRRDdd, NULL, &MaxVal, NULL);
        GetSetVal = MIN ((UINT32) MaxVal, NomWR2RD_dd + RelaxBy);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRRDdd, WriteToCache, &GetSetVal);
        MrcGetSetLimits (MrcData, GsmMctRDWRdd, NULL, &MaxVal, NULL);
        GetSetVal = MIN ((UINT32) MaxVal, NomRD2WR_dd + RelaxBy);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRdd, WriteToCache, &GetSetVal);
        MrcGetSetLimits (MrcData, GsmMctRDRDdd, NULL, &MaxVal, NULL);
        GetSetVal = MIN ((UINT32) MaxVal, NomRD2RD_dd + RelaxBy);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDRDdd, WriteToCache, &GetSetVal);
        MrcGetSetLimits (MrcData, GsmMctWRWRdd, NULL, &MaxVal, NULL);
        GetSetVal = MIN ((UINT32) MaxVal, NomWR2WR_dd + RelaxBy);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRWRdd, WriteToCache, &GetSetVal);
      }
      MrcFlushRegisterCachedData (MrcData);
      // Must update the XARB bubble injector when TAT values change
      SetTcBubbleInjector (MrcData, Controller, Channel);
    } // for Channel
  } // for Controller

  // Different DIMM turnarounds
  if (RunDD) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s DDRD2RD\n", "\n ##### Running");
    Status = TrainDDROptParamCliff (
              MrcData,
              ddrd2rd,
              TestListRdX,
              ARRAY_COUNT (TestListRdX),
              (INT8) NomRD2RD_dd,
              (INT8) NomRD2RD_dd + Range,
              LoopCount,
              Update,
              Outputs->MarginResult,
              ClkShifts,
              ARRAY_COUNT (ClkShifts),
              0,
              RankMask,
              0
              );
    if (Status != mrcSuccess) {
      return mrcFail;
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s DDWR2WR\n", "\n ##### Running");
    Status = TrainDDROptParamCliff (
              MrcData,
              ddwr2wr,
              TestListWr,
              ARRAY_COUNT (TestListWr),
              (INT8) NomWR2WR_dd - Range,
              (INT8) NomWR2WR_dd + Range,
              LoopCount,
              Update,
              Outputs->MarginResult,
              ClkShifts,
              ARRAY_COUNT (ClkShifts),
              0,
              RankMask,
              0
              );
    if (Status != mrcSuccess) {
      return mrcFail;
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s DDWR2RD\n", "\n ##### Running");
    Status = TrainDDROptParamCliff (
              MrcData,
              ddwr2rd,
              ParamListX,
              ARRAY_COUNT (ParamListX),
              MAX ((INT8) NomWR2RD_dd - Range, 4),
              (INT8) NomWR2RD_dd + Range,
              LoopCount,
              Update,
              Outputs->MarginResult,
              ClkShifts,
              ARRAY_COUNT (ClkShifts),
              0,
              RankMask,
              0
              );
    if (Status != mrcSuccess) {
      return mrcFail;
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s DDRD2WR\n", "\n ##### Running");
    Status = TrainDDROptParamCliff (
              MrcData,
              ddrd2wr,
              ParamListX,
              ARRAY_COUNT (ParamListX),
              (INT8) NomRD2WR_dd - Range,
              (INT8) NomRD2WR_dd + Range,
              LoopCount,
              Update,
              Outputs->MarginResult,
              ClkShifts,
              ARRAY_COUNT (ClkShifts),
              0,
              RankMask,
              0
              );
    if (Status != mrcSuccess) {
      return mrcFail;
    }
  }
  // Different Rank turnarounds
  if (RunDR) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s DRRD2RD\n", "\n ##### Running");
    Status = TrainDDROptParamCliff (
              MrcData,
              drrd2rd,
              TestListRdX,
              ARRAY_COUNT (TestListRdX),
              (INT8) NomRD2RD_dr,
              (INT8) NomRD2RD_dr + Range,
              LoopCount,
              Update,
              Outputs->MarginResult,
              ClkShifts,
              ARRAY_COUNT (ClkShifts),
              0,
              RankMask,
              0
              );
    if (Status != mrcSuccess) {
      return mrcFail;
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s DRWR2WR\n", "\n ##### Running");
    Status = TrainDDROptParamCliff (
              MrcData,
              drwr2wr,
              TestListWr,
              ARRAY_COUNT (TestListWr),
              (INT8) MinNomWR2WR_dr,
              (INT8) NomWR2WR_dr + Range,
              LoopCount,
              Update,
              Outputs->MarginResult,
              ClkShifts,
              ARRAY_COUNT (ClkShifts),
              0,
              RankMask,
              0
              );
    if (Status != mrcSuccess) {
      return mrcFail;
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s DRWR2RD\n", "\n ##### Running");
    Status = TrainDDROptParamCliff (
              MrcData,
              drwr2rd,
              ParamListX,
              ARRAY_COUNT (ParamListX),
              MAX ((INT8) NomWR2RD_dr - Range, 4),
              (INT8) NomWR2RD_dr + Range,
              LoopCount,
              Update,
              Outputs->MarginResult,
              ClkShifts,
              ARRAY_COUNT (ClkShifts),
              0,
              RankMask,
              0
              );
    if (Status != mrcSuccess) {
      return mrcFail;
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s DRRD2WR\n", "\n ##### Running");
    Status = TrainDDROptParamCliff (
              MrcData,
              drrd2wr,
              ParamListX,
              ARRAY_COUNT (ParamListX),
              (INT8) NomRD2WR_dr - Range,
              (INT8) NomRD2WR_dr + Range,
              LoopCount,
              Update,
              Outputs->MarginResult,
              ClkShifts,
              ARRAY_COUNT (ClkShifts),
              0,
              RankMask,
              0
              );
    if (Status != mrcSuccess) {
      return mrcFail;
    }
  }

  // Program SAFE values for ODT and SAmp
  GuardBand = 1;
  if (Outputs->Gear2) {
    GuardBand *= 2;
  }
  UpdateSampOdtTiming (MrcData, GuardBand);
  // ODT Delay (start) / Duration
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s mcodts\n", "\n ##### Running");
  Status = TrainDDROptParamCliff (
            MrcData,
            mcodts,
            TestListRdX,          // Including RcvEnaX to the test list
            ARRAY_COUNT (TestListRdX),
            0,
            2 + GuardBand,
            LoopCount,
            Update,
            Outputs->MarginResult,
            ClkShifts,
            ARRAY_COUNT (ClkShifts),
            0,
            RankMask,
            GuardBand
            );
  if (Status != mrcSuccess) {
    return mrcFail;
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s mcodtd\n", "\n ##### Running");
  Status = TrainDDROptParamCliff (
            MrcData,
            mcodtd,
            TestListRdX,
            ARRAY_COUNT (TestListRdX),
            (-1 - GuardBand),
            0,
            LoopCount,
            Update,
            Outputs->MarginResult,
            ClkShifts,
            ARRAY_COUNT (ClkShifts),
            0,
            RankMask,
            GuardBand
            );
  if (Status != mrcSuccess) {
    return mrcFail;
  }

  // Restore PCIT value
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel += ChannelIncrement) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      IpChannel = LP_IP_CH (Lpddr, Channel);
      Offset = OFFSET_CALC_MC_CH (MC0_CH0_CR_SC_PCIT_REG, MC1_CH0_CR_SC_PCIT_REG, Controller, MC0_CH1_CR_SC_PCIT_REG, IpChannel);
      MrcWriteCR (MrcData, Offset, ScPcitSave[Controller][IpChannel].Data);
    }
  }

  // Print out the end results of the training step in Table Format
#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n--- Final TA values ---\n");
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel += ChannelIncrement) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\tMc%d.C%d", Controller, Channel);
    }
  }
  for (TaTypeIdx = 0; TaTypeIdx < (sizeof (PrintGroups) / (sizeof (PrintGroups[0]))); TaTypeIdx++) {
    for (GroupIdx = 0; GroupIdx < (sizeof (PrintGroups[0]) / (sizeof (PrintGroups[0][0]))); GroupIdx++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n%s%s", TaString[TaTypeIdx], TypeStr[GroupIdx]);
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannel; Channel += ChannelIncrement) {
          if (!MrcChannelExist (MrcData, Controller, Channel)) {
            continue;
          }
          MrcGetSetMcCh (MrcData, Controller, Channel, PrintGroups[TaTypeIdx][GroupIdx], ReadFromCache, &GetSetVal);
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t%d", (UINT8) GetSetVal);
        }
      }
    } // GroupIdx
  } // TaTypeIdx
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");
#endif
  return Status;
}

/**
  General purpose function to optimize an arbitrary value, OptParam (see list above)
    OptParam is generally some timing number that impacts performance or power
    Expects that as OptParam gets smaller*, margins are flat until we hit a cliff
    This procedure defines a cliff as a reduction of 4 ticks in eye height/width
    * In the case of mcodts, higher values are actually worst
    To stress out the timing, xxDDR_CLK is shifted by +/- 15 PI ticks

  @param[in] MrcData         - Include all MRC global data.
  @param[in] OptParam        - Supports Turnaround Timings and ODT Start / Duration
  @param[in] TestList        - List of margin param to check to make sure timing are okay.
  @param[in] NumTests        - The size of TestList
  @param[in] Start           - Start point for this turn around time setting.
  @param[in] Stop            - Stop point for this turnaround time setting.
                                 Note that the Start/Stop values are the real values, not the encoded value
  @param[in] LoopCount       - Length of a given test (per rank)
  @param[in] Update          - Update the CRs and host structure with ideal values
  @param[in] MarginByte      - Byte level margins
  @param[in] ClkShifts       - Array of PI clocks to be shifted
  @param[in] NumR2RPhases    - Number of PI clock phases
  @param[in] rank            - rank to work on
  @param[in] RankMask        - RankMask to be optimized
  @param[in] GuardBand       - GuardBand to be added to last pass value (to be a bit conservative).

  @retval MrcStatus      - If it succeeds return mrcSuccess
**/
MrcStatus
TrainDDROptParamCliff (
  IN MrcParameters *const MrcData,
  IN UINT8                OptParam,
  IN const UINT8          TestList[],
  IN UINT8                NumTests,
  IN INT8                 Start,
  IN INT8                 Stop,
  IN UINT8                LoopCount,
  IN UINT8                Update,
  IN UINT16               MarginByte[MAX_RESULT_TYPE][MAX_RANK_IN_CHANNEL][MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_EDGES],
  IN const INT8           *ClkShifts,
  IN UINT8                NumR2RPhases,
  IN UINT8                rank,
  IN UINT8                RankMask,
  IN UINT8                GuardBand
  )
{
  static const UINT8 OptParamDDType[] = { // Does this test run dr, dd or both
  //drrd2rd ddrd2rd drwr2wr ddwr2wr drrd2wr ddrd2wr drwr2rd ddwr2rd rdodtd wrodtd mcodts mcodtd rtl
    1,      2,      1,      2,      1,      2,      1,      2,      3,     3,     3,     3,     0};
  static const UINT8 RankMapping[16] = {15, 15, 15, 4, 15, 3, 15, 1, 15, 15, 15, 15, 5, 2, 15, 0};
    // Order of rank turnarounds for dr & dd.
  static const UINT32  RankOrder[2][6] = {
    { 0x32320101, 0x10101010, 0x32323232, 0x20, 0x10, 0x23 }, // RankOrder[0]: drsd - same DIMM
    { 0x21303120, 0x00002120, 0x00003020, 0x20, 0x00, 0x00 }  // RankOrder[1]: drdd - diff DIMM
  };
  static const Cpgc20Address CPGCAddressConst = {
    CPGC20_RANK_2_ROW_COL_2_BANK,
    CPGC20_FAST_Y,
    0,
    0,
    0,
    0,
    8,
    8
  };
  Cpgc20Address     CpgcAddress;
  const MrcInput    *Inputs;
  MrcDebug          *Debug;
  const MRC_FUNCTION *MrcCall;
  MrcOutput         *Outputs;
  MrcChannelOut     *ChannelOut;
  MrcStatus         Status;
  MRC_PATTERN_CTL   PatternCtl; // For 8 bit VA, this walks through each WDB pointer ~ 2X
  INT64             WrRdSg[MAX_CONTROLLER][MAX_CHANNEL];
  INT64             RdWrSg[MAX_CONTROLLER][MAX_CHANNEL];
  UINT32            BERStats[4];  // Track BER results
  UINT32            RankList;
  UINT32            Offset;
  UINT32            Controller;
  UINT32            Channel;
  UINT32            IpChannel;
  UINT16            Margins[MAX_TESTS_OPT_PARAM_CLIFF][2][2][MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];  // Tests X DR/DD x ClkPhases x Ch X Byte
  UINT16            NumCL;  // Number of cachelines per SubSeq
  UINT16            m;
  UINT16            ByteMask;
  UINT16            ByteFailMask[MAX_CONTROLLER][MAX_CHANNEL];  // Per ch mask indicating which bytes have failed
  UINT16            ByteDone;
  INT8              Inc;
  INT8              Off;
  INT8              Index;
  INT8              LastPass[MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];     // Lass Pass Value for off
  INT8              Begin;
  INT8              End;
  INT8              ChLastPass;
  INT8              ActualGuardBand;
  UINT8             MaxChannel;
  UINT8             McChannelMask;
  UINT8             Byte;
  UINT8             Rank;
  UINT8             McChBitMask;
  UINT8             RankCount;
  UINT8             OrigRankCount;
  UINT8             McChBitMaskdd;
  UINT8             RankMaskCh;
  UINT8             GlobalRankMask;
  UINT8             drddPresent[2]; // [0]: ChBitMask for dr, [1]: ChBitMask for dd
  UINT8             CmdPat;
  UINT8             BMap[9];  // Needed for GetBERMarginByte function
  UINT8             MarginLimit;  // Need to change it to 20%of eye heigth
  UINT8             ResetDDR;
  UINT8             SelfRefresh;
  INT16             offs[MAX_CONTROLLER][MAX_CHANNEL];
  UINT8             Param;
  UINT8             iparam;
  UINT8             dd;
  UINT8             test0;
  INT16             v0;
  UINT8             RankOrderIndex;
  UINT8             UpdateHostMargin;
  UINT8             Done;
  UINT8             MaxMargin;
  UINT8             ResultType;
  UINT8             WDBIncRate; // Number of cachelines between incrementing WDB.
  UINT8             LoopEnd;
  UINT8             ResultRank;
  UINT8             ShiftValue;
  BOOLEAN           WriteVrefParam;
  BOOLEAN           DramVref;
  BOOLEAN           Ddr4;
  BOOLEAN           Lpddr;
  BOOLEAN           IsDual;
  BOOLEAN           ODT;
  BOOLEAN           PerByte;
  BOOLEAN           NotRankTraining;
  BOOLEAN           FindFirstPass;
#ifdef MRC_DEBUG_PRINT
  INT64             GetSetVal;
  INT8              ChLastPass1[MAX_CONTROLLER][MAX_CHANNEL];
#endif // MRC_DEBUG_PRINT
  UINT8             RepeatInitialTest;
  MC0_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_STRUCT  CpgcChSeqRankL2PMapping;
  MC0_REQ0_CR_CPGC2_ADDRESS_SIZE_STRUCT           Cpgc2AddrSize;
//  REUT_CH_SEQ_CFG_0_STRUCT                         ReutChSeqCfg; // @todo <ICL> Update with CPGC 2.0 implementation
//  REUT_CH_SEQ_BASE_ADDR_WRAP_0_STRUCT ReutChSeqBaseAddrWrap;

  Inputs          = &MrcData->Inputs;
  MrcCall         = Inputs->Call.Func;
  Outputs         = &MrcData->Outputs;
  Debug           = &Outputs->Debug;
  Status          = mrcSuccess;
  Ddr4            = (Outputs->DdrType == MRC_DDR_TYPE_DDR4);
  Lpddr           = Outputs->Lpddr;
  Done            = 0;
  drddPresent[0]  = 0;
  drddPresent[1]  = 0;
  MarginLimit     = 10;  // Drop of X% in margin means failure
  ResetDDR        = 1;
  SelfRefresh     = 0;
  WDBIncRate      = 13;
  NumCL           = (Lpddr) ? 64 : 128;
  MaxChannel      = Outputs->MaxChannels;
  RepeatInitialTest = 4;  // Run initial RTL point 5 times

  MRC_DEBUG_ASSERT (NumTests <= MAX_TESTS_OPT_PARAM_CLIFF, Debug, "ERROR: too many tests passed into TrainDDROptParamCliff()\n");

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }
      RankMask = RankMask & Outputs->Controller[Controller].Channel[Channel].ValidRankBitMask;
      ShiftPIforCmdTraining(MrcData, Controller, Channel, MrcIterationClock, RankMask, MRC_IGNORE_ARG_8, 0, 2); // Reload the local DQ PI cache before offsetting CLKs
    } // channel
  } // Controller

  PatternCtl.IncRate  = 0;
  PatternCtl.DQPat    = 0;
  PatternCtl.EnableXor = FALSE;
  PatternCtl.PatSource = MrcPatSrcDynamic;
  MrcCall->MrcSetMem ((UINT8 *) ByteFailMask, sizeof (ByteFailMask), 0);
  MrcCall->MrcSetMem ((UINT8 *) offs, sizeof (offs), 0);
  MrcCall->MrcSetMem ((UINT8 *) BERStats, sizeof (BERStats), 0);
  MrcCall->MrcCopyMem ((UINT8 *) &CpgcAddress, (UINT8 *) &CPGCAddressConst, sizeof (CpgcAddress));
  switch (OptParam) {
  case mcodts:
  case mcodtd:
    CpgcAddress.AddrIncOrder = CPGC20_BANK_2_ROW_COL_2_RANK;
    CpgcAddress.ColSizeBits = 3;
    NumCL           = 32;
    break;

  case rtl:
    CpgcAddress.BankSize = 1;
    break;

  default:
  case drwr2rd:
  case drwr2wr:
  case drrd2rd:
  case drrd2wr:
    //Use currently defined CpgcAddress
    break;
  }
  for (Byte = 0; Byte < ARRAY_COUNT (BMap); Byte++) {
    BMap[Byte] = Byte;
  }

  GlobalRankMask = Outputs->ValidRankMask & RankMask;

  NotRankTraining = (OptParam == rtl);
  FindFirstPass   = (OptParam == rtl);  // FindFirstPass logic only works for RTL!
  ODT             = (OptParam == rdodtd) || (OptParam == wrodtd) || (OptParam == mcodtd) || (OptParam == mcodts);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nNotRankTraining = %u, ODT = %d\n", NotRankTraining, ODT);

  // Decide which channels need to be run and program NumCachelines and LC
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
      if (ChannelOut->ValidRankBitMask) {
        McChannelMask = 1 << MC_CH_IDX(Controller, Channel, MaxChannel);
        RankMaskCh  = ChannelOut->ValidRankBitMask;
        IsDual      = ((RankMaskCh & 0xC) == 0xC) || ((RankMaskCh & 0x3) == 0x3);

        // Continue if no ranks in this channel
        if ((RankMaskCh & RankMask) == 0) {
          continue;
        }

        if ((OptParamDDType[OptParam] & 0x2) && (ChannelOut->DimmCount == 2)) {
          drddPresent[1] |= McChannelMask; // dd parameter and channel has 2 DIMMs
        }

        if (((OptParamDDType[OptParam] & 0x1) && IsDual) || NotRankTraining) {
          drddPresent[0] |= McChannelMask; // dr parameter and channel has a dual rank
        }

        if (ODT && ((drddPresent[0] & McChannelMask) == 0)) {
          // ODT matters when Single rank
          // dr parameter and channel has a dual rank
          drddPresent[0] |= McChannelMask;
        }
      }
    }
  }

  McChBitMask = drddPresent[1] | drddPresent[0]; // Channel is present if it has either a dr or dd
  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE,
    "drddPresent[0] = 0x%x, drddPresent[1] = 0x%x, McChBitMask = 0x%x\n",
    drddPresent[0],
    drddPresent[1],
    McChBitMask
    );

  // There is nothing to optimize for this parameter
  if ((McChBitMask == 0) || (Stop <= Start)) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "McChBitMask = %d, Start = 0x%x, Stop = 0x%x\n", McChBitMask, Start, Stop);
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Error! No need to optimize TA, OptParam = %d\n", OptParam);
    return (Inputs->ExitOnFailure) ? mrcFail : mrcSuccess;
  }
  // Setup the REUT Test
  if ((OptParam == ddwr2rd) || (OptParam == drwr2rd)) {
    CmdPat = PatWrRdTA;
    Outputs->DQPat  = TurnAroundWR;
    // WrRdSg is a long delay.  Extend RdWrSg to cover the remaining WrRdSg delay so we get WrRdDr properly.
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        if ((!MrcChannelExist (MrcData, Controller, Channel)) || IS_MC_SUB_CH (Lpddr, Channel)) {
            continue;
        }
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRRDsg, ReadFromCache, &WrRdSg[Controller][Channel]);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRsg, ReadFromCache, &RdWrSg[Controller][Channel]);
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRsg, WriteCached, &WrRdSg[Controller][Channel]);
        // Must update the XARB bubble injector when TAT values change
        SetTcBubbleInjector (MrcData, Controller, Channel);
      }
    }
  } else if ((OptParam == ddrd2wr) || (OptParam == drrd2wr)) {
    CmdPat          = PatRdWrTA;
    Outputs->DQPat  = TurnAroundRW;
//    RankInc         = 1;
  } else if (ODT) {
    CmdPat          = PatODTTA;
    Outputs->DQPat  = TurnAroundODT;
//    RankInc         = 1;
  } else if (OptParam == rtl) {
    CmdPat = PatWrRd;
    Outputs->DQPat  = Outputs->Lpddr ? RdRdTA_All : RdRdTA;
    // Less optimistic values since we are updating values and RMT fails
    WDBIncRate  = 16;
    NumCL       = 4;
    LoopCount = Outputs->Lpddr ? LoopCount - 3 : LoopCount - 1;
  } else {
    CmdPat = PatWrRd;
    Outputs->DQPat  = TurnAround;
  }

  PatternCtl.DQPat        = Outputs->DQPat;
  PatternCtl.IncRate      = WDBIncRate;
//  REUTAddress.IncRate[0]  = RankInc; // Rank
//  REUTAddress.IncRate[3]  = RankInc; // Col

  // SOE=0, EnCADB=0, EnCKE=0, SubSeqWait=0
  // @todo Update with McChBitMask
  SetupIOTest (MrcData, McChBitMask, CmdPat, NumCL, LoopCount, &CpgcAddress, NSOE, &PatternCtl, 0, 0, 0);

  Outputs->DQPatLC = MRC_BIT0 << (LoopCount - MrcLog2 ((UINT32) (NumCL - 1)));
  if (Outputs->DQPatLC < 1) {
    Outputs->DQPatLC = 1;
  }
  // Optimize parameter per byte.  Everything else is per channel
  PerByte = (OptParam == mcodts) || (OptParam == mcodtd);

  // Keep track of which bytes have failed and are we done yet
  ByteDone = (1 << Outputs->SdramCount) - 1;

  // ###########################################################
  // ####  Loop through OptParam X DD X ClkPhases X Params and measure margin #####
  // ###########################################################
  if (OptParam == mcodts) {
    // In the case of mcodts, higher values are actually worst.
    Begin = Start;
    End   = Stop;
    Inc   = 1;
  } else {
    Begin = Stop;
    End   = Start;
    Inc   = -1;
  }

  if ((Outputs->Gear2) && (OptParam != mcodts) && (OptParam != mcodtd)) {
    Inc *= 2;
  }

  ActualGuardBand = (Inc * GuardBand);

  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE,
    "Start = %d, Stop = %d, Begin = %d, End = %d, Inc = %d\n",
    Start,
    Stop,
    Begin,
    End,
    Inc
    );
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, (OptParam == rtl) ? "Rank = %d\n" : "", rank);
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Channel 0\t\t\t\t\t\t\t\t1\nByte\t");
  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE, (
    Outputs->SdramCount == MAX_SDRAM_IN_DIMM
    ) ? "0\t1\t2\t3\t4\t5\t6\t7\t8\t0\t1\t2\t3\t4\t5\t6\t7\t8\n" :
    "0\t1\t2\t3\t4\t5\t6\t7\t0\t1\t2\t3\t4\t5\t6\t7\n"
    );

  // Init Variables
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        MrcCall->MrcSetMem ((UINT8 *) &LastPass[Controller][Channel][0], Outputs->SdramCount, (UINT8) (Begin - ActualGuardBand));
        for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
          for (iparam = 0; iparam < NumTests; iparam++) {
            for (dd = 0; dd < 2; dd++) {
              for (test0 = 0; test0 < NumR2RPhases; test0++) {
                Margins[iparam][dd][test0][Controller][Channel][Byte] = 1280;
              }
            }
          }
        }
      }
    }
  }
  // Walk through different OptParam values
  for (Off = (INT8) Begin; Off != (INT8) (End + Inc); Off += Inc) {
    if (Done) {
      break;
    }
    Index = (Off - Begin) * Inc; // Index = 0, 1, 2..
    if (Index == 1) {
      if ((RepeatInitialTest != 0) && FindFirstPass) {  // Repeat initial measurement of RTL to filter out repetition noise
        Off -= Inc;
        Index = 0;
        RepeatInitialTest--;
      }
    }
    // Inc can only take a value of +/- 1.
    if ((Index == 1) && FindFirstPass) {
      Inc  *= -1;
      Off   = End;
      End   = Begin - Inc;  // One Inc less since we have already done Index 0.
      Begin = Off - Inc;    // One Inc less to get us starting at Index 1
      MRC_DEBUG_MSG (
        Debug,
        MSG_LEVEL_NOTE,
        "Find First Pass - Walking backwards.\n Off = %d, Begin = %d, End = %d, Inc = %d\n",
        Off,
        Begin,
        End,
        Inc
        );
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Param^ Offset-> %d\n   Actl\t", Off);

    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
        RankMaskCh = ChannelOut->ValidRankBitMask & RankMask;
        if (IS_MC_SUB_CH (Lpddr, Channel)) {
          //Dont print
          continue;
        }
        // if nothing for this channel OR No Ranks in this channel
        if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, MaxChannel)) || (RankMaskCh == 0)) {
#ifdef MRC_DEBUG_PRINT
          if (Channel == 0) { //@todo need to adjust for MaxChannel printing
            if (Outputs->SdramCount == (MAX_SDRAM_IN_DIMM - 1)) {
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t\t\t\t\t\t\t\t");
            } else {
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t\t\t\t\t\t\t\t\t");
            }
          }
#endif // MRC_DEBUG_PRINT
          continue;
        }

        if (FindFirstPass && (Index == 0)) {
          // We use the current RTL value for the initial test
#ifdef MRC_DEBUG_PRINT
          MrcGetSetMcChRnk (MrcData, Controller, Channel, rank, RxFlybyDelay, ReadFromCache, &GetSetVal);
#endif
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n Mc%u Ch%u initial RTL: %u", Controller, Channel, (UINT32) GetSetVal);
        } else {
          // No need to update MrcData host during this step even if not collecting data
          LoopEnd = (UINT8) ((PerByte) ? Outputs->SdramCount : 1);
          for (Byte = 0; Byte < LoopEnd; Byte++) {
            UpdateTAParamOffset (MrcData, Controller, Channel, Byte, OptParam, Off, 0, 0, RankMaskCh);
          }
          if ((OptParam == wrodtd) || (OptParam == rdodtd)) {
            // Set UpdateHost to TRUE because TA values share the same register
            MrcTatStretch (MrcData, Controller, Channel, OptParam, Off - Start, TRUE);
          }
        }
      }
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");

    // Test both: different dr and dd as required
    for (dd = 0; dd < 2; dd++) {
      if (Done) {
        break;
      }
      // Check if this test type should be run
      McChBitMaskdd = drddPresent[dd];
      if (McChBitMaskdd == 0) {
        continue;
      }

      if (OptParam != rtl) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, (dd == 0) ? "Dual Rank\n" : "Dual Dimm\n");
      }
      // Select Ranks in the correct order based on the test type
      // Need to re-order the ranks based on the value of ddw2r
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannel; Channel++) {
          ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
          RankMaskCh = ChannelOut->ValidRankBitMask & RankMask;
          if ((RankMaskCh == 0) || IS_MC_SUB_CH (Lpddr, Channel)) {
            // No Ranks in this channel
            // For LPDDR4/5, only program register on even channels.
            continue;
          }
          // Initialize variables and read out ordered rank list
          CpgcChSeqRankL2PMapping.Data = 0;
          RankCount = 0;
          IpChannel = LP_IP_CH (Lpddr, Channel);
          if (NotRankTraining) {
            RankList = 0x00003210;
          } else {
            RankOrderIndex = RankMapping[RankMaskCh];
            if (RankOrderIndex == 15) {
              RankList = 0x00003210;
            } else {
              RankList = RankOrder[dd][RankOrderIndex];
            }
          }

          while (RankList > 0) {
            if ((RankCount % 6 == 0) && (RankCount)) {
              // Program the RankMapping register if we exceed 6 ranks that fits within the register width

              Offset = OFFSET_CALC_MC_CH (MC0_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_REG, MC1_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_REG, Controller, MC0_REQ1_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_REG, IpChannel);
              Offset += (RankCount / 6) * (MC0_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_B_REG - MC0_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_REG);
              MrcWriteCR (MrcData, Offset, CpgcChSeqRankL2PMapping.Data);

              // Reset RankMapping register
              CpgcChSeqRankL2PMapping.Data = 0;
            }
            Rank = (RankList & 0xF); // Nibble by Nibble
            RankList = (RankList >> 4);
            if (!(RankMaskCh & (1 << Rank))) {
              continue;
            }

            ShiftValue = (RankCount % 6) * MC0_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_L2P_RANK0_MAPPING_WID;
            CpgcChSeqRankL2PMapping.Data |= (Rank << ShiftValue);
            RankCount++;
          }

          if (CpgcChSeqRankL2PMapping.Data != 0) {
            // Program the RankMapping register that did not get programmed in the while loop

            Offset = OFFSET_CALC_MC_CH (MC0_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_REG, MC1_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_REG, Controller, MC0_REQ1_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_REG, IpChannel);
            Offset += (RankCount / 6) * (MC0_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_B_REG - MC0_REQ0_CR_CPGC_SEQ_RANK_L2P_MAPPING_A_REG);
            MrcWriteCR (MrcData, Offset, CpgcChSeqRankL2PMapping.Data);
          }

          Offset = OFFSET_CALC_MC_CH (MC0_REQ0_CR_CPGC2_ADDRESS_SIZE_REG, MC1_REQ0_CR_CPGC2_ADDRESS_SIZE_REG, Controller, MC0_REQ1_CR_CPGC2_ADDRESS_SIZE_REG, IpChannel);
          Cpgc2AddrSize.Data = MrcReadCR64 (MrcData, Offset);
          OrigRankCount = (UINT8) (Cpgc2AddrSize.Bits.Block_Size_Max_Rank + 1);
          if (OrigRankCount != RankCount) {
            Cpgc2AddrSize.Bits.Block_Size_Max_Rank = RankCount - 1;
            Cpgc2AddrSize.Bits.Region_Size_Max_Rank = RankCount - 1;
            MrcWriteCR64 (MrcData, Offset, Cpgc2AddrSize.Data);
            Cpgc20AdjustNumOfRanks (MrcData, Controller, IpChannel, OrigRankCount, RankCount);
          }
        } // for Channel
      } // for Controller
      // ###################################################
      // ### Walk through different sets of rank2rank timings  ###
      // ###################################################
      for (test0 = 0; test0 < NumR2RPhases; test0++) {
        if (Done) {
          break;
        }

        v0 = ClkShifts[test0];

        // Program rank offsets differently for dd vs. dr
        if (NotRankTraining) {
          for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
            for (Channel = 0; Channel < MaxChannel; Channel++) {
              if (!(MC_CH_MASK_CHECK(McChBitMaskdd, Controller, Channel, MaxChannel))) {
                offs[Controller][Channel] = 0;
              } else {
                // Shift all signals in the channel(Clk/Ctl/Cmd/Dq) by v0
                offs[Controller][Channel] = v0;
              }
            }
          }
          if (v0 != 0) {
            ShiftCh2Ch (MrcData, RankMask, offs, ResetDDR, SelfRefresh, 0);
          }
        } else if (dd == 1) {
          // For DD
          // Shift Clk/DQ on one DIMM by v0 and Clk/DQ on other DIMM by -v0
          // @todo: CTL picode should be optionally shifted to improve margins
          SetCmdMargin (MrcData, McChBitMaskdd, 0x3, WrT, v0, 0, ResetDDR, SelfRefresh);
          SetCmdMargin (MrcData, McChBitMaskdd, 0xC, WrT, -v0, 0, ResetDDR, SelfRefresh);
        } else {
          // For DR
          // Shift Clk/DQ on front side by v0 and Clk/DQ on backside by -v0
          // @todo: CTL picode should be optionally shifted to improve margins
          SetCmdMargin (MrcData, McChBitMaskdd, 0x5, WrT, v0, 0, ResetDDR, SelfRefresh);
          SetCmdMargin (MrcData, McChBitMaskdd, 0xA, WrT, -v0, 0, ResetDDR, SelfRefresh);
        }
        // Test different margin param
        for (iparam = 0; iparam < NumTests; iparam++) {
          Param = TestList[iparam];
          WriteVrefParam = ((Param == WrV) || (Param == WrFan2) || (Param == WrFan3));
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s ", gMarginTypesStr[Param]);

          ResultType = GetMarginResultType (Param);

          // MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d ", MinMarginLimit);
          // Calculate MaxMargin and Starting Point for margin search
          ResultRank = rank;
          if ((Param == WrV) || (Param == WrFan3) || (Param == WrFan2) ||
              (Param == RdV) || (Param == RdFan3) || (Param == RdFan2) || (Param == CmdV)) {
            MaxMargin = GetVrefOffsetLimits (MrcData, Param);
          } else {
            MaxMargin = MAX_POSSIBLE_TIME;
          }
          DramVref = (Ddr4 || Lpddr) && WriteVrefParam;
          // Are we done yet or should we keep testing?
          Done = 1;
          for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
            for (Channel = 0; Channel < MaxChannel; Channel++) {
              if (!(MC_CH_MASK_CHECK(McChBitMaskdd, Controller, Channel, MaxChannel))) {
                continue;
              }

              ChannelOut  = &Outputs->Controller[Controller].Channel[Channel];
              RankMaskCh  = ChannelOut->ValidRankBitMask & RankMask;
              if (RankMaskCh == 0) {
                continue; // No Ranks in this channel
              }

              // When FindFirstPass is used, all Bytes have to have passed before we stop.
              // We uses ByteFailMask[] to track the passing bytes in this case.
              if (PerByte || FindFirstPass) {
                if (ByteFailMask[Controller][Channel] != ByteDone) {
                  Done = 0;
                }
              } else {
                if (ByteFailMask[Controller][Channel] == 0) {
                  Done = 0;
                }
              }
            }
          }

          if (Done) {
            break;
          }

          Status = GetMarginByte (MrcData, Outputs->MarginResult, Param, 0, 0xF);
          if (Status != mrcSuccess) {
            return Status;
          }

          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "% 3d\t", (INT8) v0);
          Status = MrcGetBERMarginByte (
                    MrcData,
                    Outputs->MarginResult,
                    McChBitMaskdd,
                    GlobalRankMask,
                    GlobalRankMask,
                    Param,
                    0,  // Mode
                    BMap,
                    1,
                    MaxMargin,
                    0,
                    BERStats
                    );
          if (Status != mrcSuccess) {
            return Status;
          }

          if (DramVref) {
            // We return results on first available rank.
            for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
              if ((1 << Rank) & GlobalRankMask) {
                ResultRank = Rank;
                break;
              }
            }
          }
          // Record Results
          UpdateHostMargin = 1;

          for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
            for (Channel = 0; Channel < MaxChannel; Channel++) {
              ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
              RankMaskCh = ChannelOut->ValidRankBitMask & RankMask;
              // if nothing for this channel OR No Ranks in this channel
              if (!(MC_CH_MASK_CHECK(McChBitMaskdd, Controller, Channel, MaxChannel)) || (RankMaskCh == 0)) {
#ifdef MRC_DEBUG_PRINT
                if (Channel == 0) {
                  if (Outputs->SdramCount == (MAX_SDRAM_IN_DIMM - 1)) {
                    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t\t\t\t\t\t\t\t");
                  } else {
                    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t\t\t\t\t\t\t\t\t");
                  }
                }
#endif // MRC_DEBUG_PRINT
                continue;
              }

              for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
                // For this optimization, it makes more sense to look at the full sum
                ByteMask = 1 << Byte;
                m = EffectiveMargin (
                    MarginByte[ResultType][ResultRank][Controller][Channel][Byte][0],
                    MarginByte[ResultType][ResultRank][Controller][Channel][Byte][1]
                      );

                if (m < 20) {
                  m = 20;
                }
                MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d", m);

                // If previously failed, this is also a failure unless we are looking for
                // the first passing offset.
                if ((ByteFailMask[Controller][Channel] & ByteMask) && !FindFirstPass) {
                  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "#\t");
                  continue;
                }

                // Check if the first RoundTripLatency test is failing with zero margin.
                // Stop the test in this case - initial point should be passing.
                if (FindFirstPass && (Index == 0)) {
                  if (m == 20) {
                    ByteFailMask[Controller][Channel] = ByteDone;
                    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "#\t");
                    continue;
                  }
                }

                /* @todo: disable this as it make the results very conservative
                // Byte fails if margin is below MinMarginLimit at any time
                if (m < MinMarginLimit) {
                  // If we are looking for pass, continue and do not update LastPass
                  if (TRUE == FindFirstPass) {
                    if (Index == 0) {
                      // When training from the most aggressive setting to the conservative setting,
                      // if we fail the first setting we stop.
                      ByteFailMask[Controller][Channel] = ByteDone;
                    }
                    UpdateHostMargin = 0;
                  } else {
                    ByteFailMask[Controller][Channel] |= ByteMask;
                    LastPass[Controller][Channel][Byte] = Off - Inc - ActualGuardBand;
                  }
                  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "#\t");
                  continue;
                }
                */
                if (Index == 0) {
                  // Get the smallest marging at Index 0
                  if (Margins[iparam][dd][test0][Controller][Channel][Byte] > m) {
                    Margins[iparam][dd][test0][Controller][Channel][Byte] = m;
                  }
                } else {
                  // Check if we dropped more than the percent allowed
                  if (m < ((Margins[iparam][dd][test0][Controller][Channel][Byte] * (100 - MarginLimit)) / 100)) {
                    if (!FindFirstPass) {
                      ByteFailMask[Controller][Channel] |= ByteMask;
                      LastPass[Controller][Channel][Byte] = Off - Inc - ActualGuardBand;
                    }
                    UpdateHostMargin = 0;
                    MRC_DEBUG_MSG (
                      Debug,
                      MSG_LEVEL_NOTE,
                      "#-%d\t",
                      (ABS (m - Margins[iparam][dd][test0][Controller][Channel][Byte]) * 100) / Margins[iparam][dd][test0][Controller][Channel][Byte]
                      );
                    continue;
                  } else {
                    if (FindFirstPass) {
                      if ((ByteFailMask[Controller][Channel] & ByteMask) != ByteMask) {
                        LastPass[Controller][Channel][Byte] = Off - ActualGuardBand;
                        ByteFailMask[Controller][Channel] |= ByteMask;
                      }
                    } else {
                      LastPass[Controller][Channel][Byte] = Off - ActualGuardBand;
                    }
                  }
                }

                MRC_DEBUG_MSG (
                  Debug,
                  MSG_LEVEL_NOTE,
                  ".%c%d\t",
                  (m > Margins[iparam][dd][test0][Controller][Channel][Byte]) ? '+' : '-',
                  (ABS(m - Margins[iparam][dd][test0][Controller][Channel][Byte]) * 100) / Margins[iparam][dd][test0][Controller][Channel][Byte]
                  );
              } // for Byte
            } // for Channel
          } // for Controller
          // Stop the test if we fail on the initial RTL setting

          if (FindFirstPass && (Index == 0)) {
            for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
              for (Channel = 0; Channel < MaxChannel; Channel++) {
                ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
                RankMaskCh = ChannelOut->ValidRankBitMask & RankMask;
                // if nothing for this channel OR No Ranks in this channel
                if (!(MC_CH_MASK_CHECK(McChBitMaskdd, Controller, Channel, MaxChannel)) || (RankMaskCh == 0)) {
                  continue;
                }
                if (ByteFailMask[Controller][Channel] != 0) {
                  MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "\nError! Mc%u.Ch%u failed initial roundtrip latency value!\n", Controller, Channel);
                  return mrcFail;
                }
              }
            }
          }

          if (UpdateHostMargin) {
            Status = ScaleMarginByte (MrcData, Outputs->MarginResult, Param, ResultRank);
          }

          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");
        } // for iparam

        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");

        // Clean up
        if (NotRankTraining) {
          MrcCall->MrcSetMem ((UINT8 *) offs, sizeof (offs), 0);
          // UpdateHost=0, SkipTx=0
          if (v0 != 0) {
            ShiftCh2Ch (MrcData, RankMask, offs, ResetDDR, SelfRefresh, 0);
          }
        } else {
          SetCmdMargin (MrcData, McChBitMaskdd, RankMask, WrT, 0, 0, ResetDDR, SelfRefresh);
        }
      } // for test0 in ClkShifts[]

      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");
    } // for dd
    if ((OptParam == wrodtd) || (OptParam == rdodtd)) {
      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < MaxChannel; Channel++) {
          ChannelOut  = &Outputs->Controller[Controller].Channel[Channel];
          RankMaskCh  = ChannelOut->ValidRankBitMask & RankMask;
          // If nothing for this channel OR No Ranks in this channel
          if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, MaxChannel)) || (RankMaskCh == 0) || (IS_MC_SUB_CH (Lpddr, Channel))) {
            continue;
          }
          // MrcTatStretch was called with UpdateHost so need to clean back to original values
          MrcTatStretch (MrcData, Controller, Channel, OptParam, - (Off - Start), TRUE);
        }
      }
    }
  } // for Off

  // If we are sweeping aggressive settings to conservative settings, we
  // need to restore original Inc, Begin, and End values to select the
  // proper offset if bytes have varying offsets values for a parameter
  // that is NOT specified per Byte.
  if (FindFirstPass) {
    Off   = End;         // Temp storage for swap
    End   = Begin + Inc;
    Begin = Off + Inc;
    Inc  *= -1;
    MRC_DEBUG_MSG (
      Debug,
      MSG_LEVEL_NOTE,
      "Find First Pass - Reverting Inc, Begin, and End\n Begin = %d, End = %d, Inc = %d,\n",
      Begin,
      End,
      Inc
      );
  }

#ifdef MRC_DEBUG_PRINT
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Optimal offset per Byte\n\t");
  // Print optimal value
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      ChannelOut            = &Outputs->Controller[Controller].Channel[Channel];
      RankMaskCh            = ChannelOut->ValidRankBitMask & RankMask;
      ChLastPass1[Controller][Channel]  = End;
      // if nothing for this channel OR No Ranks in this channel
      if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, MaxChannel)) || (RankMaskCh == 0)) {
        if (Channel == 0) {
          if (Outputs->SdramCount == (MAX_SDRAM_IN_DIMM - 1)) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t\t\t\t\t\t\t\t");
          } else {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t\t\t\t\t\t\t\t\t");
          }
        }
        continue;
      }

      for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", LastPass[Controller][Channel][Byte]);
        if ((Inc > 0) && (ChLastPass1[Controller][Channel] > LastPass[Controller][Channel][Byte])) {
          ChLastPass1[Controller][Channel] = LastPass[Controller][Channel][Byte];
        }

        if ((Inc < 0) && (ChLastPass1[Controller][Channel] < LastPass[Controller][Channel][Byte])) {
          ChLastPass1[Controller][Channel] = LastPass[Controller][Channel][Byte];
        }
      }
    }
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      ChannelOut  = &Outputs->Controller[Controller].Channel[Channel];
      RankMaskCh  = ChannelOut->ValidRankBitMask & RankMask;
      // if nothing for this channel OR No Ranks in this channel
      if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, MaxChannel)) || (RankMaskCh == 0)) {
        continue;
      }

      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Optimal offset Mc %d Channel %d = %d\n", Controller, Channel, ChLastPass1[Controller][Channel]);
    }
  }
#endif // MRC_DEBUG_PRINT
  // Program new value
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      ChannelOut  = &Outputs->Controller[Controller].Channel[Channel];
      RankMaskCh  = ChannelOut->ValidRankBitMask & RankMask;
      // if nothing for this channel OR No Ranks in this channel
      if (!(MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, MaxChannel)) || (RankMaskCh == 0) || (IS_MC_SUB_CH (Lpddr, Channel))) {
        continue;
      }

      // Start with the most aggressive setting
      ChLastPass = End;
      for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
        if (Update == 0) {
          LastPass[Controller][Channel][Byte] = Begin;
        }

        if ((Inc > 0) && (ChLastPass > LastPass[Controller][Channel][Byte])) {
          ChLastPass = LastPass[Controller][Channel][Byte];
        }

        if ((Inc < 0) && (ChLastPass < LastPass[Controller][Channel][Byte])) {
          ChLastPass = LastPass[Controller][Channel][Byte];
        }

        if (PerByte) {
          UpdateTAParamOffset (MrcData, Controller, Channel, Byte, OptParam, LastPass[Controller][Channel][Byte], Update, 1, RankMaskCh);
        }
      }

      if (PerByte == FALSE) {
        UpdateTAParamOffset (MrcData, Controller, Channel, 0, OptParam, ChLastPass, Update, 1, RankMaskCh);
        if ((OptParam == wrodtd) || (OptParam == rdodtd)) {
          MrcTatStretch (MrcData, Controller, Channel, OptParam, ChLastPass - Start, TRUE);
        }
      }

      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Selected Offset for Mc %d Channel %d is = %d\n", Controller, Channel, ChLastPass);
    }

    if ((OptParam == ddwr2rd) || (OptParam == drwr2rd)) {
      for (Channel = 0; Channel < MaxChannel; Channel++) {
        if ((!MrcChannelExist (MrcData, Controller, Channel)) || IS_MC_SUB_CH (Lpddr, Channel)) {
          continue;
        }
        MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRsg, WriteCached,   &RdWrSg[Controller][Channel]);
      }
    }
  }
  return Status;
}

/**
  Sets command margins when moving WrT, WrTBox, or WrV
  NOTE: ONLY one, ResetDDR or SelfRefresh can be set inside this function

  @param[in] MrcData         - Include all MRC global data.
  @param[in] McChBitMask     - Bit mask of populated controllers/channels
  @param[in] Ranks           - Bit Mask of populated ranks
  @param[in] Param           - Input parameter to update
  @param[in] Value0          - value to be added
  @param[in] Value1          - value to be added
  @param[in] ResetDDR        - Do we reset DDR?
  @param[in] SelfRefresh     - Do we perform Self refresh?

  @retval MrcStatus      - If it succeeds return mrcSuccess
**/
void
SetCmdMargin (
  IN MrcParameters *const MrcData,
  IN const UINT8          McChBitMask,
  IN const UINT8          Ranks,
  IN const UINT8          Param,
  IN const INT16          Value0,
  IN const UINT8          Value1,
  IN UINT8                ResetDDR,
  IN const UINT8          SelfRefresh
  )
{
  MrcOutput         *Outputs;
  INT64             GetSetVal;
  UINT32            Controller;
  UINT32            Channel;
  UINT8             MaxChannel;
  UINT8             RankMaskCh;
  INT16             Offset;

  Outputs       = &MrcData->Outputs;
  MaxChannel    = Outputs->MaxChannels;
  Offset        = 0;

  if (SelfRefresh && ResetDDR) {
    MRC_DEBUG_MSG (
      &Outputs->Debug,
      MSG_LEVEL_ERROR,
      "Error! SelfRefresh OR ResetDDR can be set at once...performing SelfRefresh\n"
      );
    ResetDDR = 0;
  }

  if (SelfRefresh) {
    GetSetVal = 1;
    MrcGetSetMc (MrcData, MAX_CONTROLLER, GsmMccEnableRefresh, WriteNoCache, &GetSetVal);
  }

  // Walk though all mcs, chs and ranks
  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      if (MC_CH_MASK_CHECK(McChBitMask, Controller, Channel, MaxChannel)) {
        // determine which ranks from parameter "Ranks" exist in this channel
        RankMaskCh = Ranks & Outputs->Controller[Controller].Channel[Channel].ValidRankBitMask;
        if (RankMaskCh == 0) {
          continue; // No Ranks in this channel
        }
        switch (Param) {
          case WrTBox:
          case WrT:
            ShiftPIforCmdTraining (MrcData, Controller, Channel, MrcIterationClock, RankMaskCh, MRC_IGNORE_ARG_8, Value0, 0);
            if (Param != WrTBox) {
              break;
            }
            /*FALLTHROUGH*/
          // Fall through to WrV if Param is WrTBox.
          case WrV:
            Offset = (Param == WrTBox) ? ((2 * Value1) - 1) * 8 : Value0;
            ChangeMargin (MrcData, WrV, Offset, 0, 0, (UINT8) Controller, (UINT8) Channel, RankMaskCh, 0, 0, 0, 0);
            break;

          default:
            return;
        }
      }
    }
  }

  if (ResetDDR) {
    MrcResetSequence (MrcData);
  } else if (SelfRefresh) {
    MrcSelfRefreshState (MrcData, MRC_SR_EXIT);
  }

  return;
}

/**
  Updates the value for following OptParamCliff variables:
  drrd2rd=0, ddrd2rd=1, drwr2wr=2, ddwr2wr=3, drrd2wr=4, ddrd2wr=5, drwr2rd=6, ddwr2rd=7,
  rdodtd=8, wrodtd=9, mcodts=10, mcodtd=11, rtl=12}

  @param[in,out] MrcData    - Include all MRC global data.
  @param[in]     Controller - Controller to update the specified parameter.
  @param[in]     Channel    - Channel to update the specified parameter.
  @param[in]     Byte       - Byte to update the specified parameter.
  @param[in]     OptParam   - Parameter to update.
  @param[in]     Off        - Value to offset the current setting.
  @param[in]     UpdateHost - Switch to update the host structure with the new value.
  @param[in]     SkipPrint  - Switch to skip debug prints.
  @param[in]     RankMask   - Bit mask of Ranks to update.

  @retval Nothing
**/
void
UpdateTAParamOffset (
  IN OUT MrcParameters *const MrcData,
  IN     const UINT32         Controller,
  IN     const UINT32         Channel,
  IN     const UINT8          Byte,
  IN     const UINT8          OptParam,
  IN     const INT8           Off,
  IN     const UINT8          UpdateHost,
  IN     const UINT8          SkipPrint,
  IN     const UINT8          RankMask
  )
{
  MrcDebug  *Debug;
  MrcOutput *Outputs;
  INT64     GetSetVal;
  INT64     GetSetValSaved1[MAX_RANK_IN_CHANNEL];
  INT64     GetSetValSaved2[MAX_RANK_IN_CHANNEL];
  UINT8     Rank;
  INT8      New;
  UINT8     GsmModeSingle;  //For parameters that only updates one setting
  UINT8     GsmModeMulti;   //For parameters that updates more than one setting
  BOOLEAN   TatUpdate;

  Outputs       = &MrcData->Outputs;
  Debug         = &Outputs->Debug;
  GsmModeMulti  = ForceWriteCached;
  GsmModeSingle = (UpdateHost) ? ForceWriteCached : ForceWriteUncached;
  GetSetVal     = Off;
  TatUpdate     = FALSE;
  if (SkipPrint) {
    GsmModeSingle |= GSM_PRINT_VAL;
    GsmModeMulti  |= GSM_PRINT_VAL;
  }

  //@todo: Update
  switch (OptParam) {
    case drrd2rd:
      // Different Rank RD 2 RD Turn Around offsets
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDRDdr, GsmModeSingle, &GetSetVal);
      TatUpdate = TRUE;
      break;

    case ddrd2rd:
      // Different DIMM RD 2 RD Turn Around offsets
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDRDdd, GsmModeSingle, &GetSetVal);
      TatUpdate = TRUE;
      break;

    case drwr2wr:
      // Different Rank WR 2 WR Turn Around offsets
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRWRdr, GsmModeSingle, &GetSetVal);
      TatUpdate = TRUE;
      break;

    case ddwr2wr:
      // Different DIMM WR 2 WR Turn Around offsets
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRWRdd, GsmModeSingle, &GetSetVal);
      TatUpdate = TRUE;
      break;

    case drrd2wr:
      // Different Rank RD 2 WR Turn Around offsets
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRdr, GsmModeSingle, &GetSetVal);
      TatUpdate = TRUE;
      break;

    case ddrd2wr:
      // Different DIMM RD 2 WR Turn Around offsets
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctRDWRdd, GsmModeSingle, &GetSetVal);
      TatUpdate = TRUE;
      break;

    case drwr2rd:
      // Different Rank WR 2 RD Turn Around offsets
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRRDdr, GsmModeSingle, &GetSetVal);
      TatUpdate = TRUE;
      break;

    case ddwr2rd:
      // Different DIMM WR 2 RD Turn Around offsets
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctWRRDdd, GsmModeSingle, &GetSetVal);
      TatUpdate = TRUE;
      break;

    case rdodtd:
      // Convert into Register values. 2'b00 = BL/2 + 2 (For Enhanced Channels 10 DCLKs, otherwise 6 DCLKs)
      // @todo: <CNL> Enhance Channel switch.
      GetSetVal  = Off - 6;
      MrcGetSetMcCh (MrcData, Controller, Channel, GsmMctOdtRdDuration, GsmModeSingle, &GetSetVal);
      break;

/* // Do not seem like this is being used
    case wrodtd:
      // Convert into Register values. 2'b00 = BL/2 + 2 (For Enhanced Channels 10 DCLKs, otherwise 6 DCLKs)
      // @todo: <CNL> Enhance Channel switch.
      GetSetVal  = Off - 6;
      MrcGetSetDdrIoGroupChannel (MrcData, Channel, GsmMctOdtWrDuration, GsmModeSingle, &GetSetVal);
      break;
*/
    case mcodts:
      // MC ODT delay
      if (!UpdateHost) {
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, SenseAmpDelay, ReadFromCache, &GetSetValSaved2[0]);
      }
      MrcGetSetChStrb (MrcData, Controller, Channel, Byte, McOdtDelay, ReadFromCache, &GetSetValSaved1[0]);
      New                   = (INT8) GetSetValSaved1[0] + Off;
      if (New < -4) {
        New = -4; // RcvEnPi[8:6] - 5 qclk Min
      } else if (New > 6) {
        New = 6; // RcvEnPi[8:6] + 5 qclk Max
      }

      GetSetVal             = New;
      MrcGetSetChStrb (MrcData, Controller, Channel, Byte, McOdtDelay, GsmModeMulti, &GetSetVal);
      MrcGetSetChStrb (MrcData, Controller, Channel, Byte, SenseAmpDelay, GsmModeMulti, &GetSetVal);
      if (!UpdateHost) {
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, McOdtDelay, WriteToCache, &GetSetValSaved1[0]);
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, SenseAmpDelay, WriteToCache, &GetSetValSaved2[0]);
      }

      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, (SkipPrint) ? "" : "%d\t", New);
      break;

    case mcodtd:
      // Duration
      if (!UpdateHost) {
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, SenseAmpDuration, ReadFromCache, &GetSetValSaved2[0]);
      }
      MrcGetSetChStrb (MrcData, Controller, Channel, Byte, McOdtDuration, ReadFromCache, &GetSetValSaved1[0]);
      GetSetVal = GetSetValSaved1[0] + Off;

      MrcGetSetChStrb (MrcData, Controller, Channel, Byte, McOdtDuration, GsmModeMulti, &GetSetVal);
      MrcGetSetChStrb (MrcData, Controller, Channel, Byte, SenseAmpDuration, GsmModeMulti, &GetSetVal);
      if (!UpdateHost) {
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, McOdtDuration, WriteToCache, &GetSetValSaved1[0]);
        MrcGetSetChStrb (MrcData, Controller, Channel, Byte, SenseAmpDuration, WriteToCache, &GetSetValSaved2[0]);
      }
      break;

    case rtl:
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (RankMask & (1 << Rank)) {
          if (!UpdateHost) {
            MrcGetSetMcChRnk (MrcData, Controller, Channel, Rank, RxFlybyDelay, ReadFromCache, &GetSetValSaved2[Rank]);
          }
          // Keep RecvEn in place - adjust IoLatency by RTL delta
          MrcGetSetMcChRnk (MrcData, Controller, Channel, Rank, RoundTripDelay, ReadFromCache, &GetSetValSaved1[Rank]);
          GetSetVal = 0 - (GetSetValSaved1[Rank] - Off);
          MrcGetSetMcChRnk (MrcData, Controller, Channel, Rank, RxFlybyDelay, ForceWriteOffsetCached, &GetSetVal);

          GetSetVal = Off;
          MrcGetSetMcChRnk (MrcData, Controller, Channel, Rank, RoundTripDelay, ForceWriteCached, &GetSetVal);
        }
      }
      if (!UpdateHost) {
        for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
          if (RankMask & (1 << Rank)) {
            MrcGetSetMcChRnk (MrcData, Controller, Channel, Rank, RoundTripDelay, WriteToCache, &GetSetValSaved1[Rank]);
            MrcGetSetMcChRnk (MrcData, Controller, Channel, Rank, RxFlybyDelay, WriteToCache, &GetSetValSaved2[Rank]);
          }
        }
      }
      break;

    default:
      break;
  }

  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE,
    ((OptParam != mcodtd) && (OptParam != mcodts) && (!SkipPrint)) ? "%d\t" : "",
    Off
    );

  if (TatUpdate) {
    // Must update the XARB bubble injector when TAT values change
    SetTcBubbleInjector (MrcData, Controller, Channel);
  }
  return;
}

/**
  This function applies an offset to the global compensation logic.
  Reruns Compensation and returns the new comp value

  @param[in,out] MrcData         - Include all MRC global data.
  @param[in]     param           - Parameter defining the desired global compensation logic
  @param[in]     offset          - Value to apply
  @param[in]     AdjustOdtStatic  - Decides if Static ODT will be adjusted for ReadODT param
  @param[in]     UpdateHost      - Decides if MrcData has to be updated
  @param[out]    NewComp         - Pointer to new comp return value

  @retval mrcSuccess Successfully computed new compensation value
**/
extern
MrcStatus
UpdateCompGlobalOffset (
  IN OUT MrcParameters *const MrcData,
  IN     const UINT8          param,
  IN     const INT32          offset,
  IN     const BOOLEAN        AdjustOdtStatic,
  IN     const BOOLEAN        UpdateHost,
  OUT          UINT32  *const NewComp
  )
{
  const MrcInput *Inputs;
  const MrcOutput *Outputs;
  INT64     RcompOdt;
  INT64     GetSetVal = 0;
  INT64     Min;
  INT64     Max;
  BOOLEAN   EnDqsOdtParkMode;
  BOOLEAN   ODTSingleSegEn;
  UINT32    GsModeFlags;
  UINT32    FirstController;
  UINT32    FirstChannel;
  UINT32    RxDqOdtUpTarget;
  GSM_GT    GetSetWrite[MaxCompGlobalOffsetParam] = {DqOdtVrefUp, DqOdtVrefDn, DqDrvVrefUp, DqDrvVrefDn, CmdDrvVrefUp, CmdDrvVrefDn, CtlDrvVrefUp, CtlDrvVrefDn, ClkDrvVrefUp, ClkDrvVrefDn, GsmGtDelim, GsmGtDelim, GsmGtDelim, GsmGtDelim, GsmGtDelim, GsmGtDelim};
  GSM_GT    GetSetRead[MaxCompGlobalOffsetParam] = {CompRcompOdtUp, CompRcompOdtDn, TxRonUp, TxRonDn, WrDSCodeUpCmd, WrDSCodeDnCmd, WrDSCodeUpCtl, WrDSCodeDnCtl, WrDSCodeUpClk, WrDSCodeDnClk, SCompCodeDq, SCompCodeCmd, SCompCodeCtl, SCompCodeClk, RloadDqsUp, CompRcompOdtUp};

  Inputs = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;

  MrcGetSetLimits(MrcData, DqOdtVrefUp, &Min, &Max, NULL);

  if (NULL == NewComp) {
    MRC_DEBUG_MSG (&MrcData->Outputs.Debug, MSG_LEVEL_ERROR, "%s: NewComp argument\n", gNullPtrErrStr);
    return mrcWrongInputParameter;
  }

  if (param > (UINT8) (MaxCompGlobalOffsetParam - 1)) {
    // param is out of bounds.
    MRC_DEBUG_MSG (&MrcData->Outputs.Debug, MSG_LEVEL_ERROR, "%s: param %u > %d\n", gWrongInputParam, param, (MaxCompGlobalOffsetParam - 1));
    return mrcWrongInputParameter;
  }

  GsModeFlags = (UpdateHost) ? ForceWriteCached : ForceWriteUncached;
  if (MRC_POWER_TRAINING_DEBUG) {
    GsModeFlags |= PrintValue;
  }
  // Update offset in local CR variable
  if (GetSetWrite[param] != GsmGtDelim) {
    GetSetVal = offset;
    MrcGetSetNoScope (MrcData, GetSetWrite[param], GsModeFlags, &GetSetVal);
  }
  switch (param) {
    case SCompDq:
      // Apply Comp Offset to Scomp-DQ
      GetSetVal = offset & 0xF;
      MrcGetSetNoScope (MrcData, TxSlewRate, GsModeFlags, &GetSetVal);
      GetSetVal = (offset >> SCOMP_PC_STORAGE_BIT_OFFSET) & 0x1; // The value is stored in bit 4 of the value passed in, because bits 0-3 are now used to store the Scomp value
      MrcGetSetNoScope (MrcData, DqScompPC, GsModeFlags, &GetSetVal);
      break;

    case SCompCmd:
      // Apply Comp Offset to Scomp-CMD/CTL/CLK
      GetSetVal = offset & 0xF;
      MrcGetSetNoScope (MrcData, CmdSlewRate, GsModeFlags, &GetSetVal);
      GetSetVal = (offset >> SCOMP_PC_STORAGE_BIT_OFFSET) & 0x1; // The value is stored in bit 4 of the value passed in, because bits 0-3 are now used to store the Scomp value
      MrcGetSetNoScope (MrcData, CmdScompPC, GsModeFlags, &GetSetVal);
      break;

    case SCompCtl:
      // Apply Comp Offset to Scomp-CMD/CTL/CLK
      GetSetVal = offset & 0xF;
      MrcGetSetNoScope(MrcData, CtlSlewRate, GsModeFlags, &GetSetVal);
      GetSetVal = (offset >> SCOMP_PC_STORAGE_BIT_OFFSET) & 0x1; // The value is stored in bit 4 of the value passed in, because bits 0-3 are now used to store the Scomp value
      MrcGetSetNoScope(MrcData, CtlScompPC, GsModeFlags, &GetSetVal);
      break;

    case SCompClk:
      // Apply Comp Offset to Scomp-CMD/CTL/CLK
      GetSetVal = offset & 0xF;
      MrcGetSetNoScope(MrcData, ClkSlewRate, GsModeFlags, &GetSetVal);
      GetSetVal = (offset >> SCOMP_PC_STORAGE_BIT_OFFSET) & 0x1; // The value is stored in bit 4 of the value passed in, because bits 0-3 are now used to store the Scomp value
      MrcGetSetNoScope(MrcData, ClkScompPC, GsModeFlags, &GetSetVal);
      break;

    case RxLoad:
      // Apply Comp Offset to RxLoad
      GetSetVal = offset;
      MrcGetSetNoScope (MrcData, RxLoadCompVref, GsModeFlags, &GetSetVal);
      break;

    default:
      break;
  }
  // Run Compensation
  // Start Comp Engine
  ForceRcomp (MrcData);
  FirstController = (MrcControllerExist(MrcData, cCONTROLLER0)) ? 0 : 1;
  FirstChannel = MrcData->Outputs.Controller[FirstController].FirstPopCh;
  MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataDqsOdtParkMode, ReadFromCache, &GetSetVal);
  EnDqsOdtParkMode = (GetSetVal > 0);
  if ((param == RdOdtDn) && (AdjustOdtStatic) && (!EnDqsOdtParkMode) && (!(Inputs->A0))) { // We set the RdOdtDn after the RdOdtUp, and we want to adjust this once everything has been set.
    // Check if we are close to saturation in either direction
    if (MrcData->Outputs.OdtMode == MrcOdtModeVss) {
      MrcGetSetNoScope(MrcData, CompRcompOdtDn, ReadUncached, &RcompOdt);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "AjustOdtStatic First CompRcompOdtDn: %d\n", (INT32)RcompOdt);
      }
    } else {
      MrcGetSetNoScope(MrcData, CompRcompOdtUp, ReadUncached, &RcompOdt);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "AjustOdtStatic First CompRcompOdtUp: %d\n", (INT32)RcompOdt);
      }
    }
    if (RcompOdt > 48) { // First, try changing ODTSingleSegEn. If that isn't enough, then also change static ODT legs.
      // Change ODTSingleSegEn
      // Adjust comp vref up
      MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataODTSingleSegEn, ReadFromCache, &GetSetVal);
      ODTSingleSegEn = (GetSetVal == 1) ? 1 : 0;

      MrcGetSetNoScope(MrcData, DqOdtVrefUp, ReadCached, &GetSetVal);
      RxDqOdtUpTarget = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor * TWO_DECIMAL_PLACES) / ((Outputs->OdtMode == MrcOdtModeVtt ? 2 : 1) * (UINT32)GetSetVal) - Inputs->RcompResistor * TWO_DECIMAL_PLACES;
      RxDqOdtUpTarget = RxDqOdtUpTarget / (TWO_DECIMAL_PLACES * (!ODTSingleSegEn + 1));

      GetSetVal = 0;
      MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteCached, &GetSetVal);

      GetSetVal = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor) / ((Inputs->RcompResistor + (!((UINT32)GetSetVal) + 1) * RxDqOdtUpTarget) * (Outputs->OdtMode == MrcOdtModeVtt ? 2 : 1));
      if ((GetSetVal >= Min) && (GetSetVal <= Max)) {
        MrcGetSetNoScope(MrcData, DqOdtVrefUp, ForceWriteCached, &GetSetVal);
      } else {
        // Make sure ODTSingleSegEn is set back if changing it caused Rx ODT comp vref to go out of bounds
        GetSetVal = 1;
        MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteCached, &GetSetVal);
      }

      // Make sure the static read Odt legs are disabled
      GetSetVal = 1;
      MrcGetSetNoScope(MrcData, GsmIocCompOdtStaticDis, WriteCached, &GetSetVal);
      MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataOdtStaticDis, WriteCached, &GetSetVal);
      MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocStrobeOdtStaticDis, WriteCached, &GetSetVal);
      ForceRcomp (MrcData);
      if (MrcData->Outputs.OdtMode == MrcOdtModeVss) {
        MrcGetSetNoScope(MrcData, CompRcompOdtDn, ReadUncached, &RcompOdt);
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "AjustOdtStatic RcompOdt > 48 CompRcompOdtDn: %d\n", (INT32)RcompOdt);
        }
      } else {
        MrcGetSetNoScope(MrcData, CompRcompOdtUp, ReadUncached, &RcompOdt);
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "AjustOdtStatic RcompOdt > 48 CompRcompOdtUp: %d\n", (INT32)RcompOdt);
        }
      }

      if (RcompOdt > 48) {
        // Enable static read Odt legs
        GetSetVal = 0;
        MrcGetSetNoScope(MrcData, GsmIocCompOdtStaticDis, WriteCached, &GetSetVal);
        MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataOdtStaticDis, WriteCached, &GetSetVal);
        MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocStrobeOdtStaticDis, WriteCached, &GetSetVal);
        ForceRcomp (MrcData);
      }
    } else if (RcompOdt < 16) { // First, try changing static ODT legs. If that isn't enough, then also change ODTSingleSegEn
      // Disable static read Odt legs
      GetSetVal = 1;
      MrcGetSetNoScope(MrcData, GsmIocCompOdtStaticDis, WriteCached, &GetSetVal);
      MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataOdtStaticDis, WriteCached, &GetSetVal);
      MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocStrobeOdtStaticDis, WriteCached, &GetSetVal);

      // Adjust comp vref up
      MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataODTSingleSegEn, ReadFromCache, &GetSetVal);
      ODTSingleSegEn = (GetSetVal == 1) ? 1 : 0;

      MrcGetSetNoScope(MrcData, DqOdtVrefUp, ReadCached, &GetSetVal);
      RxDqOdtUpTarget = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor * TWO_DECIMAL_PLACES) / ((Outputs->OdtMode == MrcOdtModeVtt ? 2 : 1) * (UINT32)GetSetVal) - Inputs->RcompResistor * TWO_DECIMAL_PLACES;
      RxDqOdtUpTarget = RxDqOdtUpTarget / (TWO_DECIMAL_PLACES * (!ODTSingleSegEn + 1));

      // Make sure ODTSingleSegEn is set to 0
      GetSetVal = 0;
      MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteCached, &GetSetVal);

      GetSetVal = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor) / ((Inputs->RcompResistor + (!((UINT32)GetSetVal) + 1) * RxDqOdtUpTarget) * (Outputs->OdtMode == MrcOdtModeVtt ? 2 : 1));
      if ((GetSetVal >= Min) && (GetSetVal <= Max)) {
        MrcGetSetNoScope(MrcData, DqOdtVrefUp, ForceWriteCached, &GetSetVal);
      } else {
        // Make sure ODTSingleSegEn is set back if changing it caused Rx ODT comp vref to go out of bounds
        GetSetVal = 1;
        MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteCached, &GetSetVal);
      }

      ForceRcomp (MrcData);
      if (MrcData->Outputs.OdtMode == MrcOdtModeVss) {
        MrcGetSetNoScope(MrcData, CompRcompOdtDn, ReadUncached, &RcompOdt);
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "AjustOdtStatic RcompOdt < 16 CompRcompOdtDn: %d\n", (INT32)RcompOdt);
        }
      } else {
        MrcGetSetNoScope(MrcData, CompRcompOdtUp, ReadUncached, &RcompOdt);
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "AjustOdtStatic RcompOdt < 16 CompRcompOdtUp: %d\n", (INT32)RcompOdt);
        }
      }

      if (RcompOdt < 16) {
        // Change ODTSingleSegEn
        // Adjust comp vref up
        MrcGetSetChStrb(MrcData, FirstController, FirstChannel, 0, GsmIocDataODTSingleSegEn, ReadFromCache, &GetSetVal);
        ODTSingleSegEn = (GetSetVal == 1) ? 1 : 0;

        MrcGetSetNoScope(MrcData, DqOdtVrefUp, ReadCached, &GetSetVal);
        RxDqOdtUpTarget = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor * TWO_DECIMAL_PLACES) / ((Outputs->OdtMode == MrcOdtModeVtt ? 2 : 1) * (UINT32)GetSetVal) - Inputs->RcompResistor * TWO_DECIMAL_PLACES;
        RxDqOdtUpTarget = RxDqOdtUpTarget / (TWO_DECIMAL_PLACES * (!ODTSingleSegEn + 1));

        GetSetVal = 1;
        MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteCached, &GetSetVal);

        GetSetVal = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor) / ((Inputs->RcompResistor + (!((UINT32)GetSetVal) + 1) * RxDqOdtUpTarget) * (Outputs->OdtMode == MrcOdtModeVtt ? 2 : 1));
        if ((GetSetVal >= Min) && (GetSetVal <= Max)) {
          MrcGetSetNoScope(MrcData, DqOdtVrefUp, ForceWriteCached, &GetSetVal);
        } else {
          // Make sure ODTSingleSegEn is set back if changing it caused Rx ODT comp vref to go out of bounds
          GetSetVal = 0;
          MrcGetSetChStrb(MrcData, MAX_CONTROLLER, MAX_CHANNEL, MAX_SDRAM_IN_DIMM, GsmIocDataODTSingleSegEn, WriteCached, &GetSetVal);
        }

        ForceRcomp (MrcData);
      }
    }
  }

  // Return the new comp code (hardware output)
  if(param == RdOdtUp || param == RdOdtDn) {
    if (MrcData->Outputs.OdtMode == MrcOdtModeVss) {
      // In Vss mode only odt down is valid
      MrcGetSetNoScope(MrcData, CompRcompOdtDn, ReadUncached, &RcompOdt);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "UpdateCompGlobalOffset final CompRcompOdtDn: %d\n", (INT32)RcompOdt);
      }
    } else {
      MrcGetSetNoScope(MrcData, CompRcompOdtUp, ReadUncached, &RcompOdt);
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&(MrcData->Outputs.Debug), MSG_LEVEL_NOTE, "UpdateCompGlobalOffset final CompRcompOdtUp: %d\n", (INT32)RcompOdt);
      }
    }
    GetSetVal = RcompOdt;
  } else if (GetSetRead[param] != GsmGtDelim) {
    MrcGetSetNoScope (MrcData, GetSetRead[param], ReadUncached, &GetSetVal);
  }

  *NewComp = (UINT32) GetSetVal;
  return mrcSuccess;
}

/**
  Programs Delay/Duration for the SenseAmp and MCODT based on RcvEn timing
  Provide GuardBand > 0 if needed to be more conservative in timing
  Main goal is to optimize power

  @param[in,out] MrcData   - Include all MRC global data.
  @param[in]     GuardBand - QCLK guardband on Duration.

  @retval Nothing
**/
void
UpdateSampOdtTiming (
  IN OUT MrcParameters *const MrcData,
  IN     const UINT8          GuardBand
  )
{
  MrcOutput *Outputs;
  MrcDebug  *Debug;
  MrcDdrType  DdrType;
  INT64       Min;
  INT64       Max;
  INT64       GetSetOn;   // SenseAmp / ODT delay
  INT64       GetSetOff;  // SenseAmp / ODT duration
  INT64       DqsOff;
  INT64       GetSetVal;
  INT32       On;         // SenseAmp / ODT delay
  INT32       Off;        // SenseAmp / ODT duration
  UINT32      Channel;
  UINT32      Controller;
  UINT32      Byte;
  UINT32      Rank;
  UINT32      Strobe2ClkDriftPs;
  UINT32      Strobe2ClkDriftQclk;
  UINT32      WakeUpQclk;
  UINT32      QclkPs;
  INT16       MaxRcvEn;
  INT16       MinRcvEn;
  INT16       CurRcvEn;
  INT16       QclkPi;
  BOOLEAN     Gear2;
  BOOLEAN     Lpddr5;
  BOOLEAN     Lpddr4;
  BOOLEAN     Ddr5;

  Outputs   = &MrcData->Outputs;
  Debug     = &Outputs->Debug;
  DdrType   = Outputs->DdrType;
  Gear2     = Outputs->Gear2;
  Lpddr5    = (DdrType == MRC_DDR_TYPE_LPDDR5);
  Lpddr4    = (DdrType == MRC_DDR_TYPE_LPDDR4);
  Ddr5      = (DdrType == MRC_DDR_TYPE_DDR5);

  if (Lpddr5) {
    // tWCK2DQO is drift of DRAM read data to our WCK.
    Strobe2ClkDriftPs = 1600; //ps
    QclkPs = Outputs->Wckps / ((Gear2) ? 1 : 2);
  } else {
    QclkPs = Outputs->Qclkps;
    if (Lpddr4) {
      Strobe2ClkDriftPs = 1200; //ps
    } else if (Ddr5) {
      Strobe2ClkDriftPs = 3750; //ps
    } else {
      Strobe2ClkDriftPs = 0;
    }
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "UpdateSampOdtTiming: GuardBand = %d\n", GuardBand);
  MrcGetSetLimits (MrcData, RecEnDelay, &Min, &Max, NULL);

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
          MaxRcvEn  = 0;
          MinRcvEn  = (UINT16) Max;

          for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
            if (MrcRankExist (MrcData, Controller, Channel, Rank)) {
              MrcGetSetStrobe (MrcData, Controller, Channel, Rank, Byte, RecEnDelay, ReadFromCache, &GetSetVal);
              CurRcvEn = (INT16) GetSetVal;
              if (MaxRcvEn < CurRcvEn) {
                MaxRcvEn = CurRcvEn;
              }

              if (MinRcvEn > CurRcvEn) {
                MinRcvEn = CurRcvEn;
              }
            }
          }
          // Round Max to nearest cycle
          QclkPi = (Gear2) ? 128 : 64;
          MaxRcvEn = DIVIDECEIL (MaxRcvEn, QclkPi);
          MinRcvEn = MinRcvEn >> (6 + ((Gear2) ? 1 : 0));
          Strobe2ClkDriftQclk = DIVIDECEIL (Strobe2ClkDriftPs, QclkPs);

          // SENSE AMP CAN ONLY BE ON WHEN ODT IS ON FOR EOS REASONS.
          // Ideal: On = 4 - MAX (WakeUpNs, 2Qclk) - TdqsckQclk + MinRcvEnQclk
          //WakeUpQclk = (((QclkPi * MRC_SENSE_AMP_WAKEUP_TIME) + TdqsckPs) / Outputs->Qclkps); // Convert to PI codes
          WakeUpQclk = MRC_SENSE_AMP_WAKEUP_TIME + Strobe2ClkDriftPs;
          WakeUpQclk = DIVIDECEIL (WakeUpQclk, QclkPs); // Convert to PI codes
          WakeUpQclk = MAX (WakeUpQclk, 2);
          On = 4 - WakeUpQclk + MinRcvEn;
          if (MrcData->Inputs.SafeMode) {
            On = 0;
          }

          // Turn Off Samp/ODT 1 qclk after postamble
          Off = (MaxRcvEn - MinRcvEn) + Strobe2ClkDriftQclk + 1 + GuardBand + ((Gear2) ? 2 : 5);

          GetSetOn = On;
          GetSetOff = Off;
          DqsOff = Off * 2;
          MrcGetSetChStrb (MrcData, Controller, Channel, Byte, McOdtDelay,       WriteToCache | PrintValue, &GetSetOn);
          MrcGetSetChStrb (MrcData, Controller, Channel, Byte, McOdtDuration,    WriteToCache | PrintValue, &GetSetOff);
          MrcGetSetChStrb (MrcData, Controller, Channel, Byte, DqsOdtDelay,      WriteToCache | PrintValue, &GetSetOn);
          MrcGetSetChStrb (MrcData, Controller, Channel, Byte, DqsOdtDuration,   WriteToCache | PrintValue, &DqsOff);
          MrcGetSetChStrb (MrcData, Controller, Channel, Byte, SenseAmpDelay,    WriteToCache | PrintValue, &GetSetOn);
          if (Outputs->Lpddr && Gear2) {
            GetSetOff += 2;
          }
          MrcGetSetChStrb (MrcData, Controller, Channel, Byte, SenseAmpDuration, WriteToCache | PrintValue, &GetSetOff);
        }
      }
    }
  }
  MrcFlushRegisterCachedData (MrcData);

  return;
}

/**
  Turns off unused portions of the slave DLL to save power

  @param[in,out] MrcData - Include all MRC global data.

  @retval Nothing
**/
void
UpdateSlaveDLLLength (
  IN OUT MrcParameters *const MrcData
  )
{
  MrcOutput           *Outputs;
  UINT8             Controller;
  UINT8             Channel;
  UINT8             MaxChannel;
  UINT8             byte;
  UINT8             rank;
  UINT8             MaxPi;
  INT64             GetSetVal;
  UINT32            Ratio;
  UINT32            TempVar1;
  UINT32            AvgSdllSegmentDisable;
  UINT32            SdllSegDisable;
  UINT32            Average;
  UINT32            Count;
  UINT32            Offset;
  UINT32            VccDllBlock;
  BOOLEAN           BreakOut;
  DLLDDRDATA0_CR_DDRCRVCCDLLFFCONTROL_STRUCT         VccDllFFControl;

  Outputs  = &MrcData->Outputs;
  MaxChannel = Outputs->MaxChannels;
  Ratio    = 0;
  Average  = 0;
  Count    = 0;
  SdllSegDisable = 0;
  BreakOut = FALSE;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < MaxChannel; Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        for (byte = 0; byte < Outputs->SdramCount; byte++) {
          MaxPi = 0;
          for (rank = 0; rank < MAX_RANK_IN_CHANNEL; rank++) {
            if (MrcRankExist (MrcData, Controller, Channel, rank)) {
              MrcGetSetStrobe (MrcData, Controller, Channel, rank, byte, RxDqsPDelay, ReadFromCache, &GetSetVal);
              if (MaxPi < GetSetVal) {
                MaxPi = (UINT8) GetSetVal;
              }

              MrcGetSetStrobe (MrcData, Controller, Channel, rank, byte, RxDqsNDelay, ReadFromCache, &GetSetVal);
              if (MaxPi < GetSetVal) {
                MaxPi = (UINT8) GetSetVal;
              }
            }
          }
          // Update SlaveDLL Length for power Savings
          // Calculate which segments to turn off:
          // NEW (OFF: 0, PI<48: 0x2, PI<32: 0x4, PI<16: 0x6)
          // results are:   0, 2 , 4 or 6
          GetSetVal = ((7 - (MaxPi >> 3)) & ~MRC_BIT0);
          MrcGetSetChStrb (MrcData, Controller, Channel, byte, GsmIocSdllSegmentDisable, WriteCached, &GetSetVal);
          SdllSegDisable = (UINT32) GetSetVal;
          Average = Average + (RANGE (SdllSegDisable, 0, 10));
          Count++;
        }
      }
    }
  }//controller

  // Read the first populated VccDllFFControl register
  // @todo : Review this for DDR4 2DPC and DDR5
  for (Controller = 0; (Controller < MAX_CONTROLLER) && (BreakOut == FALSE); Controller++) {
    for (Channel = 0; (Channel < MaxChannel) && (BreakOut == FALSE); Channel++) {
      if (MrcChannelExist (MrcData, Controller, Channel)) {
        VccDllBlock = ((MAX_CCC_PER_CONTROLLER * Controller) + Channel);
        Offset = OFFSET_CALC_CH (DLLDDRDATA0_CR_DDRCRVCCDLLFFCONTROL_REG, DLLDDRDATA1_CR_DDRCRVCCDLLFFCONTROL_REG, VccDllBlock) ;
        VccDllFFControl.Data = MrcReadCR (MrcData, Offset);
        BreakOut = TRUE;
      }
    }
  } //controller
  if (Count == 0) {
    Count = 1;
    MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_WARNING, "SdllSegDisable for each byte is 0 and is forced to be 1 to avoid divide by zero error. Possibly unpopulated channel\n", Count);
  }
  AvgSdllSegmentDisable = Average / Count;

  TempVar1 = 183 / 8;
  Ratio = (((10 - AvgSdllSegmentDisable) * 170) +  TempVar1* 12) + 670;
  //Ratio = DIVIDEROUND (TempVar2, 1000); // Dont divide, instead multiply the RHS by 1000
  TempVar1 = ((Ratio > 900) && (Ratio <= 1000)) ? 0 : ((Ratio > 800) && (Ratio <= 900)) ? 1 : ((Ratio > 700) && (Ratio <= 800)) ? 2 : 3;
  VccDllFFControl.Bits.VccdllReadPwrSave = (4 * (!(Outputs->Gear2))) + TempVar1;
  MrcWriteCrMulticast (MrcData, DLLDDR_CR_DDRCRVCCDLLFFCONTROL_REG, VccDllFFControl.Data);
  return;
}

/**
  This function Shifts the CMD timing.
  NOTE: ONLY one, ResetDDR or SelfRefresh can be set inside this function

  @param[in,out] MrcData     - Include all MRC global data.
  @param[in]     Ranks       - Parameter defining the desired global compensation logic
  @param[in]     offset      - per channel Value to shift picode for
  @param[in]     ResetDDR    - Do we reset DDR?
  @param[in]     SelfRefresh - Do we perform Self refresh?
  @param[in]     UpdateHost  - Determines if MrcData has to be updated

  @retval MrcStatus       - If it succeeds return mrcSuccess
**/
MrcStatus
ShiftCh2Ch (
  IN OUT MrcParameters *const MrcData,
  IN     const UINT8          Ranks,
  IN     INT16                offset[MAX_CONTROLLER][MAX_CHANNEL],
  IN     UINT8                ResetDDR,
  IN     const UINT8          SelfRefresh,
  IN     const UINT8          UpdateHost
  )
{
  MrcDebug          *Debug;
  MrcOutput         *Outputs;
  MrcIntOutput      *IntOutputs;
  MrcIntCmdTimingOut  *IntCmdTiming;
  MrcStatus         Status;
  INT64             GetSetVal;
  UINT32            Controller;
  UINT32            Channel;
  UINT8             Rank;
  UINT8             RankMaskCh;
  UINT8             CmdGroup;
  INT32             NewValue;
  INT32             Offset;

  Status        = mrcSuccess;
  Outputs       = &MrcData->Outputs;
  Debug         = &Outputs->Debug;
  IntOutputs    = (MrcIntOutput *) (MrcData->IntOutputs.Internal);

  if (SelfRefresh && ResetDDR) {
    MRC_DEBUG_MSG (
      Debug,
      MSG_LEVEL_ERROR,
      "Error! SelfRefresh OR ResetDDR can be set at once...performing SelfRefresh\n"
      );
    ResetDDR = 0;
  }

  if (SelfRefresh) {
    GetSetVal = 1;
    MrcGetSetMc (MrcData, MAX_CONTROLLER, GsmMccEnableRefresh, WriteNoCache, &GetSetVal);
  }

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!MrcChannelExist (MrcData, Controller, Channel)) {
        continue;
      }

      IntCmdTiming = &IntOutputs->Controller[Controller].CmdTiming[Channel];
      RankMaskCh = Ranks & Outputs->Controller[Controller].Channel[Channel].ValidRankBitMask;

      if (RankMaskCh == 0) {
        continue;
      }

      Offset = offset[Controller][Channel];

      // Shift CLK (this will shift DQ PIs as well)
      ShiftPIforCmdTraining (MrcData, Controller, Channel, MrcIterationClock, RankMaskCh, MRC_IGNORE_ARG_8, Offset, UpdateHost);

      // Shift CTL
      NewValue = 0;
      for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
        if (RankMaskCh & (1 << Rank)) {
          NewValue = (INT32) (IntCmdTiming->CtlPiCode[Rank]) + Offset;
          break;
        }
      }

      ShiftPIforCmdTraining (MrcData, Controller, Channel, MrcIterationCtl, RankMaskCh, MRC_IGNORE_ARG_8, NewValue, UpdateHost);

      // Shift Cmd
      for (CmdGroup = 0; CmdGroup < MAX_COMMAND_GROUPS; CmdGroup++) {
        ShiftPIforCmdTraining (
          MrcData,
          Controller,
          Channel,
          MrcIterationCmd,
          MRC_IGNORE_ARG_8,
          1 << CmdGroup,
          (INT32) IntCmdTiming->CmdPiCode[CmdGroup] + Offset,
          UpdateHost
        );
      }
    } // for Channel
  } // for Controller

  // Reset DDR is required
  if (ResetDDR) {
    Status = MrcResetSequence (MrcData);
  } else if (SelfRefresh) {
    Status = MrcSelfRefreshState (MrcData, MRC_SR_EXIT);
  }

  return Status;
}

/**
  Returns the index into the array OptResult in the MrcOutput structure.

  @param[in] OptParam - Margin parameter

  @retval One of the following values: RdSAmpOfft(0), WrDSOfft (1), RxEqOfft(2), TxEqOfft (3), RdOdtOfft(4)
**/
UINT8
GetOptResultType (
  IN UINT8 OptParam
  )
{
  switch (OptParam) {
  case OptRxBias:
  case OptRxBiasCb:
    return RdSAmpOfft;

  case OptWrDS:
  case OptTxEqWrDS:
    return WrDSOfft;

  case OptRxEq:
    return RxEqOfft;

  case OptTxEq:
    return TxEqOfft;

  case OptRdDqOdt:
    return RdOdtOfft;

  default:
    return 0; // Return RdSAmpOfft to point to the beginning of the array
  }
}

/**
  Normalizes the Power values to the Margins passed in Points2Calc.
  Assumes that power numbers are represented as lowest number is the lowest power,
  and inverts the scale such that highest number is the lowest power.  This is done
  before normalizing to margins.

  @param[in]     MrcData       - Include all MRC global data.
  @param[in]     Points2calc   - Data normalize power.
  @param[in]     ArrayLength   - Array length of Points2calc.
  @param[in]     LenMargin     - The length of inputMargins we are optimizing (0 - LenMargin -1).
  @param[in]     TestListSize  - Size of TestList/Scale

  @retval Nothing
**/
void
NormalizePowerToMargins (
  IN     MrcParameters   *MrcData,
  IN     void            *Points2calc,
  IN     UINT8           ArrayLength,
  IN     UINT8           LenMargin,
  IN     UINT8           TestListSize
  )
{
  MrcDebug        *Debug;
  const MRC_FUNCTION *MrcCall;
  UINT16          MaxPoints[MAX_MARGINS_TRADEOFF];
  UINT16          MinPoints[MAX_MARGINS_TRADEOFF];
  UINT16          MaxPwr;
  UINT16          MinPwr;
  UINT8           off;
  UINT8           test;
  UINT16          AveOfMax;
  UINT16          X;
  UINT16          *Points;
  UINT16          *PointsElement;

  MrcCall     = MrcData->Inputs.Call.Func;

  Debug       = &MrcData->Outputs.Debug;
  Points      = (UINT16 *) Points2calc;
  MaxPwr      = 0;
  MinPwr      = 0xffff;

  MrcCall->MrcSetMemWord (MaxPoints, sizeof (MaxPoints) / sizeof (UINT16), 0);
  MrcCall->MrcSetMemWord (MinPoints, sizeof (MinPoints) / sizeof (UINT16), 0xFFFF);

  // Sorting the min max margin points for each test
  for (off = 0; off < LenMargin; off++) {
    for (test = 0; test < TestListSize; test++) {
      PointsElement = (Points + ArrayLength * test + off);
      if (MaxPoints[test] < *PointsElement) {
        MaxPoints[test] = *PointsElement;
      }

      if (MinPoints[test] > *PointsElement) {
        MinPoints[test] = *PointsElement;
      }
    }
    PointsElement = (Points + ArrayLength * TestListSize + off);

    if (MaxPwr < *PointsElement) {
      MaxPwr = *PointsElement;
    }

    if (MinPwr > *PointsElement) {
      MinPwr = *PointsElement;
    }

    if (LenMargin == 1) {
      MaxPwr  = *PointsElement;
      MinPwr  = *PointsElement;
    }
  }
  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "MaxPwr=%d MinPwr=%d\n", MaxPwr, MinPwr);
  }
  AveOfMax  = 0;
  TestListSize = MIN (TestListSize, ARRAY_COUNT (MaxPoints));
  for (test = 0; test < TestListSize; test++) {
    AveOfMax += MaxPoints[test];
  }
  AveOfMax = AveOfMax / TestListSize;

  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Power Values\nBefore\tAfter\n");
  }
  for (off = 0; off < LenMargin; off++) {
    PointsElement = (Points + ArrayLength * TestListSize + off);
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", *PointsElement);
    }
    if (MaxPwr == MinPwr) {
      X = 0;
    } else {
      X = 100 - 100 * (*PointsElement - MinPwr) / (MaxPwr);
    }
    *PointsElement = AveOfMax * X / 100;
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\n", *PointsElement);
    }
  }
}

#ifdef MRC_DEBUG_PRINT

/**
  This function implements switch to print the correct format and data for the
  OptResultsPerByte struct members.

  @param[in] Debug     - Debug pointer for printing.
  @param[in] Data      - Pointer to OptResultsPerByte struct.
  @param[in] TypeIndex - Member of OptResultsPerByte to print.
  @param[in] TestIndex - Some parameters store multiple test results to be printed.
  @param[in] MidPoint  - Used to convert from zero-based indexing to the selected value

  @retval Nothing.
**/
void
MrcOptResultsPerBytePrint (
  IN MrcDebug *const    Debug,
  IN OptResultsPerByte  *Data,
  IN UINT8              TypeIndex,
  IN UINT8              TestIndex,
  IN INT8               MidPoint
  )
{
  switch (TypeIndex) {
    case (MrcOptResultBest):
      (TestIndex == 0) ? MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "=== %d ===", Data->Best - MidPoint) :
                         MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t");
      break;

    case (MrcOptResultGrdBnd):
      (TestIndex == 0) ? MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "*** %d ***", Data->GuardBand) :
                         MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t");
      break;

    case(MrcOptResultOffSel):
      (TestIndex == 0) ? MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "--> %d <--", Data->Best - MidPoint + Data->GuardBand) :
                         MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t");
      break;

    case (MrcOptResultScale):
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", Data->Scale[TestIndex]);
      break;

    case (MrcOptResultMaxPost):
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", Data->MaxPost[TestIndex]);
      break;

    case (MrcOptResultMinPost):
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", Data->MinPost[TestIndex]);
      break;

    default:
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Error! OptResultPerByteDbgStr Switch exceeded number of cases defined\n");
  }
}

/**
  This function prints the Optimize margin result table
  e.g: calcResultSummary[MAX_CHANNEL][MAX_SDRAM_IN_DIMM]

  @param[in] MrcData           - MRC data structure
  @param[in] calcResultSummary - The data array [MAX_CHANNEL][MAX_SDRAM_IN_DIMM]
  @param[in] TestList          - Test list
  @param[in] NumTest           - Number of test
  @param[in] NumOffsets        - Number of offsets
  @param[in] MidPoint          - Middle point
  @param[in] IncEnds           - Print ends points
  @param[in] OptParam          - Used to convert to the Opt param string for printing
  @param[in] OptPower          - Opt Power values to be printed
  @param[in] Controller        - Controller to print
  @param[in] Channel           - Channel to print
  @param[in] Ranks             - Ranks to print
  @param[in] TrendLine         - Switch to print the trend line
  @param[in] Nibble            - take low/high bytes
  @param[in] perCh             - Switch to only print 1 Byte of data
  @param[in] noPrint           - Boolean used to disable printing of results

  @retval Nothing
**/
void
PrintCalcResultTableCh (
  IN MrcParameters *const MrcData,
  IN OptResultsPerByte    calcResultSummary[MAX_SDRAM_IN_DIMM],
  IN const UINT8          *TestList,
  IN UINT8                NumTest,
  IN UINT8                NumOffsets,
  IN INT8                 MidPoint,
  IN BOOLEAN              IncEnds,
  IN UINT8                OptParam,
  IN UINT16               *OptPower,
  IN UINT8                Controller,
  IN UINT8                Channel,
  IN UINT8                Ranks,
  IN BOOLEAN              TrendLine,
  IN UINT8                Nibble,
  IN BOOLEAN              perCh,
  IN BOOLEAN              noPrint
  )
{
  const MRC_FUNCTION *MrcCall;
  MrcDebug          *Debug;
  OptResultsPerByte *data;
  UINT8             Off; // @todo: change for the new alg.
  UINT8             Start;
  UINT8             Stop;
  UINT8             i;
  UINT8             j;
  UINT8             b;
  UINT8             FirstByte;
  UINT8             NumBytes;
  UINT8             NumTestPlus;
  UINT32            Result;
  BOOLEAN           Format64Results;
  UINT8             Param;

  MrcCall         = MrcData->Inputs.Call.Func;
  Format64Results = 1;
  // Display result in %/Delta , 0-displat raw 64bit result in HEX
  Debug = &MrcData->Outputs.Debug;
  Start = (!IncEnds);
  Stop  = NumOffsets - (!IncEnds);
  if (noPrint) {
    return ;

  }

  FirstByte = (Nibble) ? 4 : 0;
  NumBytes  = FirstByte + 4 + Nibble * MrcData->Outputs.SdramCount % 8;
  if (perCh) {
    NumBytes = 1;
  }

  NumTestPlus = (TrendLine) ? NumTest + 1 : NumTest;

  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE,
    "\n<======== optimize %s ========>Plot results ",
    TOptParamOffsetString[OptParam]
    );
  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE,
    "<Controller=%d><Channel=%d><rank/s=0x%x><Nibble=%s> across settings :(Start=%d,Stop=%d)\n",
    Controller,
    Channel,
    Ranks,
    (Nibble) ? "High" : "Low",
    Start - MidPoint,
    Stop - MidPoint - 1
    );
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Bytes\t");
  for (b = FirstByte; b < NumBytes; b++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", b);
    for (i = 0; i < NumTestPlus + 1; i++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t"); // tab insertion
    }
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n"); // row end here !
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Offset\t"); // row starts here !
  if (OptPower[Stop - 1] != 0) {//WA: need to add param to enable this print
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s\t","TotPwr");
  }

  for (b = FirstByte; b < NumBytes; b++) {
    for (i = 0; i < NumTest; i++) {
      // Test types header
      Param = TestList[i];
      if (Param > CmdV) {
        Param = (Param % 16) + 4;
      }

      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s\t", gMarginTypesStr[Param]);
    }

    if (TrendLine) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s\t", "PwrVal");
    }

    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Opt.func\t"); // more header..
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n\n"); // row end here !
  for (Off = Start; Off < Stop; Off++) {
    // row starts here !
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", Off - MidPoint);
    if (OptPower[Stop - 1] != 0) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d.%d\t", OptPower[Off]/10, OptPower[Off]%10);
    }

    for (b = FirstByte; b < NumBytes; b++) {
      if (b < MAX_SDRAM_IN_DIMM) {
        data = &calcResultSummary[b];
      } else {
        MRC_DEBUG_MSG (
          Debug,
          MSG_LEVEL_NOTE,
          "Error: calcResultSummary array out of bounds! %d > %d \n",
          b,
          MAX_SDRAM_IN_DIMM - 1
          );
        return;
      }

      for (i = 0; i < NumTestPlus; i++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", data->Margins[i][Off].EW);
      }

      if (Format64Results) {
        Result = (UINT32) (MrcCall->MrcDivU64x64 (MrcCall->MrcMultU64x32 (data->Result[Off], 200), data->MaxR, NULL));
        Result /= 2;
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t\t", Result);
      }

      if (!Format64Results) {
        MRC_DEBUG_MSG (
          Debug,
          MSG_LEVEL_NOTE,
          "%08x-%08x\t\t",
          (UINT32) MrcCall->MrcRightShift64 (data->Result[Off],
                32),
          (UINT32) (data->Result[Off])
          );
      }
    }

    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n"); // row end here !
  }

  for (i = 0; i < ARRAY_COUNT (OptResultDbgStrings); i++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s\t", OptResultDbgStrings[i]);
    for (b = FirstByte; b < NumBytes; b++) {
      if (b < MAX_SDRAM_IN_DIMM) {
        data = &calcResultSummary[b];
      } else {
        MRC_DEBUG_MSG (
          Debug,
          MSG_LEVEL_NOTE,
          "Error: calcResultSummary array out of bounds! %d > %d \n",
          b,
          MAX_SDRAM_IN_DIMM - 1
          );
        return;
      }

      for (j = 0; j < NumTestPlus; j++) {
        MrcOptResultsPerBytePrint (Debug, data, i, j, MidPoint);
      }

      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t\t"); // tab insertion
    }

    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");
  } // row end here !
  return;
}

/**
  This function prints the Optimize margin result table
  e.g: MarginResult[Test][Offset][Channel][Byte][sign]

  @param[in] MrcData     - MRC data structure
  @param[in] ChMask      - Channels to print
  @param[in] ResultArray - Array with saved margin results
  @param[in] TestNum     - Test index
  @param[in] OffsetsNum  - number of offsets
  @param[in] MidPoint    - Zero point
  @param[in] OptParam    - Used to convert to the Opt param string for printing
  @param[in] Param       - Margin type to be printed.
  @param[in] PowerLimits - Power limits to print.
  @param[in] noPrint     - Used to skip printing.

  @retval Nothing
**/
void
PrintResultTableByte4by24 (
  IN MrcParameters   *MrcData,
  IN UINT8           McChMask,
  IN UINT16          ResultArray[][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM],
  IN UINT16          TestNum,
  IN UINT8           OffsetsNum,
  IN UINT8           MidPoint,
  IN UINT8           OptParam,
  IN UINT8           Param,
  IN UINT16          *PowerLimits,
  IN BOOLEAN         noPrint
  )
{
  MrcDebug        *Debug;
  MrcOutput       *Outputs;
  UINT8           Channel;
  UINT8           Byte;
  UINT8           Controller;
  UINT8           Off;
  UINT8           Start;
  UINT8           Stop;

  // @todo new alg. change pass start stop...
  Outputs = &MrcData->Outputs;
  Debug   = &Outputs->Debug;
  Start   = -MidPoint;
  Stop    = OffsetsNum - MidPoint - 1;
  if (Param > CmdV) {
    Param = (Param % 16) + 4;
  }

  if (noPrint) {
    return;
  }

  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE,
    "\nTest number: %d - %s, Plot results across OptParam=%s (Start=%d,Stop=%d)",
    TestNum,
    gMarginTypesStr[Param],
    TOptParamOffsetString[OptParam],
    Start,
    Stop
    );

  MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "w/power limits(width): %d\nMc.Ch\t", PowerLimits[TestNum]);

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d.%d", Controller, Channel);
      for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t");
      }
    }
  }

  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\nByte\t");

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%u\t", Byte);
      }
    }
  }
  // Sweep through OpParam settings
  for (Off = Start; Off < Stop + 1; Off++) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n  %d:\t", Off);
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        // spaces for non populated channel
        if (!(MC_CH_MASK_CHECK(McChMask, Controller, Channel, Outputs->MaxChannels))) {
          if (Controller != MAX_CONTROLLER && Channel != Outputs->MaxChannels) {
            for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t");
            }
          }
          continue;
        }

        for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
          if (Byte < MAX_SDRAM_IN_DIMM) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", ResultArray[TestNum][Off - Start][Controller][Channel * Outputs->SdramCount + Byte]);
          }
        }
      }
    }
  }
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");  // New line after the end of the table

  return;
}

/**
  This function prints the Optimize margin result table

  @param[in] MrcData            - MRC data structure
  @param[in] Controller         - Controller to print
  @param[in] Channel            - Channel to print
  @param[in] Byte               - Byte to print
  @param[in] calcResultSummary  - Array with saved margin results
  @param[in] BestOff            - Pointer to the selected offsets
  @param[in] Param              - Margin type to print.
  @param[in] OffsetsNum         - number of offsets
  @param[in] Start              - Start offsets
  @param[in] Stop               - End offsets
  @param[in] OptParam           - List of optimization parameters
  @param[in] OptParamLen        - Number of optimization parameters
  @param[in] PowerLimits        - Power limits to print.
  @param[in] Dim                - Dimension
  @param[in] TestNum            - Test index
  @param[in] NumTests           - Number of tests
  @param[in] noPrint            - Used to skip printing.

  @retval Nothing
**/
void
Print2DResultTableChByte (
  IN MrcParameters      *MrcData,
  IN UINT8              Controller,
  IN UINT8              Channel,
  IN UINT8              Byte,
  IN OptResultsPerByte  *calcResultSummary,
  IN OptOffsetChByte    *BestOff,
  IN UINT8              Param,
  IN UINT8              OffsetsNum,
  IN INT8               *Start,
  IN INT8               *Stop,
  IN const UINT8        *OptParam,
  IN UINT8              *OptParamLen,
  IN UINT16             *PowerLimits,
  IN UINT8              Dim,
  IN UINT16             TestNum,
  IN UINT8              NumTests,
  IN BOOLEAN            noPrint
  )
{
  MrcDebug          *Debug;
  MrcOutput         *Outputs;
  UINT8             Off;
  UINT8             XOff;
  UINT8             YOff;
  OptResultsPerByte *data;

  Outputs = &MrcData->Outputs;
  Debug   = &Outputs->Debug;
  if (Param > CmdV) {
    Param = (Param % 16) + 4;
  }

  if (noPrint) {
    return;
  }

  if ((Byte == 0) && ((Dim > 1) || (TestNum == 0))) {
    MRC_DEBUG_MSG (
      Debug,
      MSG_LEVEL_NOTE,
      "\nPlot Margin %s results across OptParam=%s Start=%d Stop=%d ",
      (Dim > 1)? gMarginTypesStr[Param] : "",
      TOptParamOffsetString[OptParam[0]],
      Start[0],
      Stop[0]
    );

    if (Dim > 1) {
      MRC_DEBUG_MSG (
        Debug,
        MSG_LEVEL_NOTE,
        "& OptParam=%s Start=%d Stop=%d, power limits=%d\n",
        TOptParamOffsetString[OptParam[1]],
        Start[1],
        Stop[1],
        PowerLimits[TestNum]
      );
    } else {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");
    }
  }
  // Sweep through OpParam settings
  data = calcResultSummary;
  // print param0 header and offsets
  if ((Dim > 1) || ((TestNum == 0) && (Byte == 0))) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s\n\t", TOptParamOffsetString[OptParam[0]]);
  //     MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "MC%dC%dB%d\t", Controller, Channel, Byte);
    for (Off = 0; Off < OptParamLen[0]; Off++) {// print param0 offsets
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d\t", Off + Start[0]);
    }
  }
  // print param1 header
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n%s\t", (Dim > 1) ? TOptParamOffsetString[OptParam[1]] : "");

  if ((Dim > 1 ) || (TestNum == 0)) {
  // print Ch Byte and separators, the -1 is result of the 2 tab width of below line
    for (Off = 0; Off < (OptParamLen[0] - 1); Off++) {
      if (Off == (OptParamLen[0] / 2 - 1)) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, " MC%dC%dB%d %s", Controller, Channel, Byte, (Dim > 1)? gMarginTypesStr[Param] : "---");
      } else {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "--------");
      }
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "-");
  }

  for (Off = 0; Off <  OffsetsNum; Off++) {
    XOff = Off % OptParamLen[0];
    YOff = Off / OptParamLen[0];
    if (XOff == 0) {
      if (Dim > 1) { // print param1 offset
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n%d\t", YOff + Start[1]);
      } else {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n%s\t", gMarginTypesStr[Param]);
      }
    }
    //(UINT32) (MrcCall->MrcDivU64x64 (MrcCall->MrcMultU64x32 (data->Result[Off], 200), data->MaxR, NULL)/2)
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%d", data->Margins[TestNum][Off].EW);
    if (BestOff->Offset[Controller][Channel * Outputs->SdramCount + Byte] == Off) { // mark the chosen one
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "**");
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t");
  }

  if ((Dim > 1) || (TestNum == (NumTests - 1))) {
    // print separators between ch or at last test in list
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n\t");
    for (Off = 0; Off < (OptParamLen[0] - 1); Off++) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "--------");
    }
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "-----\n");
  }

  return;
}
#endif // MRC_DEBUG_PRINT

/**
  This function returns the UPM or PWR limit value for the specified parameter

  @param[in] MrcData   - Pointer to MRC global data.
  @param[in] Param     - Margin type
  @param[in] LimitType - Type of limit: UpmLimit or PowerLimit

  @retval Returns the UPM or PWR limit
**/
UINT16
MrcGetUpmPwrLimit (
  IN MrcParameters *const MrcData,
  IN UINT8                Param,
  IN UINT8                LimitType
  )
{
  MrcIntOutput           *IntOutputs;
  MrcOutput              *Outputs;
  MrcDdrType             DdrType;
  MrcUpmPwrRetrainLimits *MrcLimits;
  UINT32                 Index;
  UINT16                 Limit;

  Limit      = 0;
  Outputs    = &MrcData->Outputs;
  IntOutputs = (MrcIntOutput *) (MrcData->IntOutputs.Internal);
  MrcLimits  = IntOutputs->UpmPwrRetrainLimits.Pointer;
  DdrType    = Outputs->DdrType;

  for (Index = 0; Index < MRC_NUMBER_UPM_PWR_RETRAIN_MARGINS; Index++) {
    if (Param == MrcLimits[Index].Param) {
      Limit = MrcLimits[Index].ParamLimit[LimitType];
      break;
    }
  }

  if (LimitType == UpmLimit) {
    // TGL_POWER_TRAINING - Fill in any specific UPM alterations here
  }

  // TGL_POWER_TRAINING_DDR5 - Need to look into how this limit changes with DDR5.
  // Table has ticks of margin relative to LP4 step sizes. Convert for other types of memory.
  if (Param == WrV || Param == CmdV) {
    // Divide by (<memory> vref step size / LP4 vref step size * 10)
    if (DdrType == MRC_DDR_TYPE_DDR4) {
      if (Param == WrV) {
        Limit = ((Limit * 10) / 16); // DDR4 TxVref 0.65% step size
      } else {
        Limit = ((Limit * 10) / 7); // DDR4 CmdVref 0.26% step size
      }
    } else if (DdrType == MRC_DDR_TYPE_LPDDR4 && Outputs->Lp4x) {
      Limit = ((Limit * 10) / 15); // LP4x 0.6% step size
    } else if (DdrType == MRC_DDR_TYPE_LPDDR5) {
      Limit = ((Limit * 10) / 8); // LP5 0.5% step size
    }
  }

  // Table has ticks of margin relative to VDDQ = 1.1 V. Convert for other voltage levels.
  if (Param == RdV) {
    Limit = (Limit * 1100) / (UINT16)Outputs->VccddqVoltage;
  }

  return Limit;
}


/**
  This function returns the Start/Stop limits value for the specified parameter

  @param[in] MrcData   - Pointer to MRC global data.
  @param[in] Param     - Opt Param type
  @param[in] LimitType - Type of limit: 0 - Start 1 - Stop

  @retval Returns the Start or Stop limit
**/
INT8
OptParamLimitValue (
  IN MrcParameters *const MrcData,
  IN UINT8                Param,
  IN UINT8                LimitType
  )
{
  MrcDebug                 *Debug;
  MrcOutput                *Outputs;
  const MrcInput           *Inputs;
  const MrcOptParamsLimits *MrcLimits;
  UINT32                   Index;
  INT8                     Limit;
  BOOLEAN                  NotFound;

  Limit      = 0;
  NotFound   = TRUE;
  Outputs    = &MrcData->Outputs;
  Debug      = &Outputs->Debug;
  Inputs     = &MrcData->Inputs;
  MrcLimits  = MrcOptParamLimit;

  for (Index = 0; Index < MRC_NUMBER_OPT_PARAMS_TRAIN; Index++) {
    if (Param == MrcLimits[Index].Param) {
      if (Outputs->Frequency < f1600) {
        Limit = MrcLimits[Index].SaGvLimits[LimitType];
      } else if (Inputs->PowerTrainingMode == MrcTmMargin || Inputs->DtHalo) {
        Limit = MrcLimits[Index].MaxPerfLimits[LimitType];
      } else {
        Limit = MrcLimits[Index].Limits[LimitType];
      }
      NotFound =FALSE;
      break;
    }
  }

  if (NotFound) {
    MRC_DEBUG_ASSERT (
      FALSE,
      Debug,
      "Can't find OptParamLimit for param %s",
      TOptParamOffsetString[Param]
      );
  }
  return Limit;
}

/**
  This function will adjust the requested Limit Type of the margin parameter by the signed offset passed in.

  @param[in]  MrcData   - MRC global data.
  @param[in]  Param     - Margin parameter type to adjust.
  @param[in]  LimitType - MRC_MARGIN_LIMIT_TYPE to adjust.
  @param[in]  Offset    - The adjustment value.

  @retval UINT16 - The new value of Param[MRC_MARGIN_LIMIT_TYPE]
**/
UINT16
MrcUpdateUpmPwrLimits (
  IN OUT MrcParameters * const  MrcData,
  IN UINT8                      Param,
  IN UINT8                      LimitType,
  IN INT8                       Offset
  )
{
  MrcIntOutput            *IntOutputs;
  MrcUpmPwrRetrainLimits  *MrcLimits;
  UINT32                  Index;
  INT32                   UpdatedValue;

  IntOutputs    = (MrcIntOutput *) (MrcData->IntOutputs.Internal);
  MrcLimits     = IntOutputs->UpmPwrRetrainLimits.Pointer;
  UpdatedValue  = 0;

  for (Index = 0; Index < MRC_NUMBER_UPM_PWR_RETRAIN_MARGINS; Index++) {
    if (Param == MrcLimits[Index].Param) {
      UpdatedValue = MrcLimits[Index].ParamLimit[LimitType];
      break;
    }
  }

  UpdatedValue += Offset;
  UpdatedValue = MAX (UpdatedValue, 0);
  UpdatedValue = MIN (UpdatedValue, 0xFFFF);

  MrcLimits[Index].ParamLimit[LimitType] = (UINT16) UpdatedValue;

  return (UINT16) UpdatedValue;
}

/**
  Returns the Actual DIMM Driver/Odt Impedance in Ohms.
  Note: host structure calculation based.

  @param[in] MrcData       - Pointer to MRC global data.
  @param[in] Controller    - Zero based controller number.
  @param[in] Channel       - Zero based channel number.
  @param[in] Rank          - Zero based rank number.
  @param[in] OptParam      - Param to read.
  @param[in] Override      - Override host read value.
  @param[in] OverrideValue - Value to override.
  @param[in] GetFromTable  - Get the Values from Odt tables

  @retval Returns the DIMM driver impedance value in Ohms
**/
UINT16
CalcDimmImpedance (
  IN MrcParameters *const MrcData,
  IN UINT32               Controller,
  IN UINT32               Channel,
  IN UINT32               Rank,
  IN UINT8                OptParam,
  IN BOOLEAN              Override,
  IN UINT16               OverrideValue,
  IN BOOLEAN              GetFromTable
  )
{
  MrcOutput                   *Outputs;
  MrcChannelOut               *ChannelOut;
  MrcDebug                    *Debug;
  MrcDdrType                  DdrType;
  // DDR4 Parameters
  static const UINT16         Ddr4RttNomDic[8] = { 0xFFFF, 60, 120, 40, 240, 48, 80, 34 }; // Same for Rtt_Park
  static const UINT16         Ddr4RttWrDic[5]  = { 0, 120, 240, 0xFFFF, 80 };              // in DDR4 we dont assert odt to target so 0 = off also
  static const UINT16         Ddr4RonDic[2]    = { 34, 48};
  // LPDDR4/LPDDR5 Parameters
  static const UINT16         Lpddr4OdtDic[7]  = { 0xFFFF, 240, 120, 80, 60, 48, 40 };
  static const UINT16         Lpddr4RonDic[6]  = { 240, 120, 80, 60, 48, 40 };             // MR3 PD-DS ; Note: Index starts from 1 (0 field values is RFU)
  DDR4_MODE_REGISTER_1_STRUCT Ddr4ModeRegister1;
  DDR4_MODE_REGISTER_2_STRUCT Ddr4ModeRegister2;
  DDR4_MODE_REGISTER_5_STRUCT Ddr4ModeRegister5;
  UINT8                       MaxLpddrIndex;
  UINT32                      Dimm;
  BOOLEAN                     Ddr4;
  BOOLEAN                     Lpddr;
  BOOLEAN                     Lpddr5;
  UINT16                      LpddrMr;
  UINT16                      *MR;
  TOdtValueDdr4               *Ddr4OdtTableIndex;
  TOdtValueLpddr              *Lpddr4OdtTableIndex;
  UINT16                      Impedance = 0xFFFF; // Odt off
  UINT8                       MrIndex;
  UINT8                       MrNum;
  UINT16                      MrValue;

  Outputs     = &MrcData->Outputs;
  ChannelOut  = &Outputs->Controller[Controller].Channel[Channel];
  Debug       = &Outputs->Debug;
  DdrType     = Outputs->DdrType;
  Ddr4        = (DdrType == MRC_DDR_TYPE_DDR4);
  Lpddr5      = (DdrType == MRC_DDR_TYPE_LPDDR5);
  Lpddr       = Outputs->Lpddr;
  Dimm        = RANK_TO_DIMM_NUMBER (Rank);

  if (GetFromTable) {
    if (Ddr4) {
      Ddr4OdtTableIndex = (TOdtValueDdr4 *) GetOdtTableIndex (MrcData, Controller, Channel, Dimm);
      if (Ddr4OdtTableIndex == NULL) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s: OdtTableIndex is NULL!\n", gErrString);
        return 0;
      }
      if (OptParam == OptDimmOdtWr) {
        Impedance = Ddr4OdtTableIndex->RttWr;
      } else if (OptParam == OptDimmOdtPark) {
        Impedance = (Ddr4OdtTableIndex->RttPark == 0) ? 0xFFFF : Ddr4OdtTableIndex->RttPark;
      } else if (OptParam == OptDimmOdtNom) {
        // Our convention is that ODT Write/Park is symmetric, ODT Nom is not (e.g 1R-2R vs. 2R-1R)
        Impedance = (Ddr4OdtTableIndex->RttNom == 0) ? 0xFFFF : Ddr4OdtTableIndex->RttNom;
      }
    } else if (Lpddr) {
      Lpddr4OdtTableIndex = (TOdtValueLpddr *) GetOdtTableIndex (MrcData, Controller, Channel, Dimm);
      if (Lpddr4OdtTableIndex == NULL) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s: OdtTableIndex is NULL!\n", gErrString);
        return 0;
      }
      if (OptParam == OptDimmOdtCA) {
        Impedance = Lpddr4OdtTableIndex->RttCa;
      } else if (OptParam == OptDimmOdtWr) {
        Impedance = Lpddr4OdtTableIndex->RttWr;
      }
    }
  } else { // else GetFromTable == FALSE
    MR = ChannelOut->Dimm[(Rank / 2) % MAX_DIMMS_IN_CHANNEL].Rank[Rank % 2].MR;

    if (OptParam == OptDimmOdtWr) {
      GetOptDimmParamMrIndex (MrcData, OptParam, &MrIndex, &MrNum);
      MrValue = MR[MrIndex];
      if (Ddr4) {
        Ddr4ModeRegister2.Data = MrValue;
        Impedance = Override ? OverrideValue : Ddr4RttWrDic[Ddr4ModeRegister2.Bits.DynamicOdt];
      } else if (Lpddr) {
        LpddrMr = MrValue & 0x7;
        MaxLpddrIndex = ARRAY_COUNT (Lpddr4OdtDic) - 1;
        Impedance = Override ? OverrideValue : Lpddr4OdtDic[RANGE (LpddrMr, 0, MaxLpddrIndex)];
      }
    }
    if (OptParam == OptDimmOdtNom) {
      if (Ddr4) {
        GetOptDimmParamMrIndex (MrcData, OptParam, &MrIndex, &MrNum);
        Ddr4ModeRegister1.Data = MR[MrIndex];
        Impedance = Override ? OverrideValue : Ddr4RttNomDic[Ddr4ModeRegister1.Bits.OdtRttValue];
      }
    }

    if (OptParam == OptDimmOdtPark) {
      if (Ddr4) {
        GetOptDimmParamMrIndex (MrcData, OptParam, &MrIndex, &MrNum);
        Ddr4ModeRegister5.Data = MR[MrIndex];
        Impedance = Override ? OverrideValue : Ddr4RttNomDic[Ddr4ModeRegister5.Bits.RttPark];
      }
    }

    if (OptParam == OptDimmRon) {
      GetOptDimmParamMrIndex (MrcData, OptParam, &MrIndex, &MrNum);
      MrValue = MR[MrIndex];
      if (Ddr4) {
        Ddr4ModeRegister1.Data = MrValue;
        Impedance = Override ? OverrideValue : Ddr4RonDic[Ddr4ModeRegister1.Bits.ODImpedance];
      } else if (Lpddr) {
        if (Lpddr5) {
          LpddrMr = MrValue & 0x7;
        } else {
          LpddrMr = (MrValue >> 3) & 0x7;
        }
        MaxLpddrIndex = ARRAY_COUNT(Lpddr4RonDic) - 1;
        Impedance = Override ? OverrideValue : Lpddr4RonDic[RANGE(LpddrMr - 1, 0, MaxLpddrIndex)];
      }
    }

    if (OptParam == OptDimmOdtCA) {
      if (Lpddr) {
        GetOptDimmParamMrIndex (MrcData, OptParam, &MrIndex, &MrNum);
        LpddrMr = (MR[MrIndex] >> 4) & 0x7;
        MaxLpddrIndex = ARRAY_COUNT (Lpddr4OdtDic) - 1;
        Impedance = Override ? OverrideValue : Lpddr4OdtDic[RANGE(LpddrMr, 0, MaxLpddrIndex)];
      }
    }
  } //end else GetFromTable

  if ((OptParam == OptDimmOdtCA) && Lpddr) {
    Impedance /= 2;
  }

  return Impedance;
}

/**
  Check the selected param to see if it supports FIVR power measurement

  @param[in] MrcData     - Include all MRC global data.
  @param[in] Param       - Param to check for FIVR power measurement support

  @retval BOOLEAN If the param supports FIVR power measurement
**/
extern
BOOLEAN
SupportsFivrPower (
  IN MrcParameters   *MrcData,
  IN const UINT8     Param
  )
{
  switch (Param) {
    case OptDimmRon:
    case OptDimmOdtWr:
    case OptDimmOdtNom:
    case OptDimmOdtPark:
    case OptDimmOdtParkNT:
    case OptDimmOdtNomNT:
    case OptDimmOdtCA:
      return FALSE;
    default:
      return TRUE;
  }
}

/**
Calculate linear normalized power

@param[in] MrcData        - Include all MRC global data.
@param[in] Param          - Param that linear normalized power should be estimated for
@param[in] ParamValue     - Param value that linear normalized power should be estimated for
@param[in] PowerDirection - If TRUE, increasing param values means increasing power. FALSE means increasing param values means decreasing power.

@retval Linear normalized power
**/
UINT16
CalcLinearNormalizedPower(
  IN MrcParameters   *MrcData,
  IN const UINT8     Param,
  IN INT8            ParamValue,
  IN BOOLEAN         PowerDirection
  )
{
  UINT16         *DimmOptParamVals;
  UINT8           NumDimmOptParamVals;
  UINT16          LinearNormalizedPower = 0;
  INT8            MinValue;
  INT8            MaxValue;

  switch (Param) {
    case OptDimmRon:
    case OptDimmOdtWr:
    case OptDimmOdtNom:
    case OptDimmOdtPark:
    case OptDimmOdtParkNT:
    case OptDimmOdtNomNT:
    case OptDimmOdtCA:
      GetDimmOptParamValues (MrcData, Param, &DimmOptParamVals, &NumDimmOptParamVals); // TGL_POWER_TRAINING_DDR5 - Check that the values are in order from most power to least power
      MinValue = 0;
      MaxValue = NumDimmOptParamVals - 1;
      break;
    default:
      MinValue = OptParamLimitValue (MrcData, Param, 0);
      MaxValue = OptParamLimitValue (MrcData, Param, 1);
  }


  if (Param == OptSComp || Param == OptCCCSComp) {
    MinValue -= 1; // Start one index earlier on slew rate training to account for needing a point to test slew rate being disabled
  }

  LinearNormalizedPower = 1 + (LINEAR_NORMALIZED_POWER_SCALE * ((PowerDirection == TRUE) ? (ParamValue - MinValue) : (MaxValue - ParamValue))) / (MaxValue - MinValue);

  return LinearNormalizedPower;
}

/**
  Calculate Power for the selected Opt param based on:

  @param[in] MrcData     - Include all MRC global data.
  @param[in] PanicVttDnLp- Measures panic comp event rates, not power
  @param[in] Scale       - Multiplication factor for margins and power to preserve decimal accuracy (usually 10 to preserve 1 decimal place)
  @param[in] Params      - Params that power should be estimated for
  @param[in] ParamValues - Param values that power should be estimated for
  @param[in] ParamsCount - Number of params that power should be estimated for

  @retval Calc power in mW if using FIVR, else use linear approximation
**/
MrcStatus
CalcSysPower(
  IN MrcParameters   *MrcData,
  IN BOOLEAN         PanicVttDnLp,
  IN UINT8           Scale,
  IN const UINT8     Params[],
  IN INT8            ParamValues[],
  IN UINT8           ParamsCount
  )
{
  MrcOutput             *Outputs;
  MrcInput              *Inputs;
  MrcChannelOut         *ChannelOut;
  const MRC_FUNCTION    *MrcCall;
  MrcOdtPowerSaving     *OdtPowerSaving;
  MrcPower              SystemPower;
  UINT8                 Channel;
  UINT8                 Controller;
  UINT8                 McChBitMask = 0;
  UINT8                 Iterator;
  UINT32                FivrPInValueStart;
  UINT32                FivrPInCountStart;
  UINT32                FivrPInValueEnd;
  UINT32                FivrPInCountEnd;
  UINT32                MaxPOutValue;
  UINT32                MaxPOutUnits;
  UINT32                CountAccumulator;
  INT64                 GetSetVal = 0x0;
  UINT64                Timeout;
  UINT8                 VTTIndex;

  Outputs             = &MrcData->Outputs;
  Inputs              = &MrcData->Inputs;
  MrcCall             = Inputs->Call.Func;
  OdtPowerSaving      = &Outputs->OdtPowerSavingData;

  SystemPower.RdPower = 0;
  SystemPower.WrPower = 0;
  SystemPower.TotalPwr = 0;

  for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
    for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
      if (!(MC_CH_MASK_CHECK(MrcData->Outputs.ValidChBitMask, Controller, Channel, Outputs->MaxChannels))) {
        continue;
      }
      ChannelOut = &Outputs->Controller[Controller].Channel[Channel];
      // no need to run on channel with no selected ranks
      if (!(ChannelOut->ValidRankBitMask)) {
        continue;
      }
      // TGL_POWER_TRAINING_FUTURE - Cannot run SelectReutRanks before SetupIOTestBasicVA. Push to afterwards.
      McChBitMask |= SelectReutRanks(MrcData, Controller, Channel, 0xFF, FALSE, 0); // Run all ranks
    }
  }

  if (McChBitMask == 0x0) {
    return mrcSuccess; // Nothing to do here
  }

  // For PanicVttDnLp, don't calculate power; instead, find the number of [panic high + (panic high - panic lo)] panic events that occur (and minimize it while making sure [panic high - panic lo > 0])
  if (PanicVttDnLp != FALSE) { // In order to run this, DDRVtt_CR_DDRCrVttGenStatus.EnCount must be set to 1 or it won't work.
    SetupIOTestBasicVA (MrcData, McChBitMask, OPT_PARAM_LOOP_COUNT, NSOE, 0, 0, 8, PatWrRd, 0, 0); // TGL_POWER_TRAINING_FUTURE - LC may need to be optimized based on number of ranks and test type (for Rx Turnarounds)

    for (VTTIndex = 0; VTTIndex < MAX_VTT_REGS; VTTIndex++) {
      GetSetVal = 0;
      MrcGetSetChStrb (MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VTTIndex, VttGenStatusSelCount, WriteCached, &GetSetVal);
      GetSetVal = 1;
      MrcGetSetChStrb(MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VTTIndex, VttGenStatusResetCount, WriteCached, &GetSetVal);
    }

    // Run CPGC
    RunIOTest (MrcData, McChBitMask, Outputs->DQPat, 1, 0);

    CountAccumulator = 0;
    for (VTTIndex = 0; VTTIndex < MAX_VTT_REGS; VTTIndex++) {
      MrcGetSetNoScope(MrcData, VttGenStatusCount, ReadUncached, &GetSetVal);
      CountAccumulator += (UINT32)(GetSetVal & 0x7FFF); // Use the absolute value of the 16 bit signed error count, so cut off the MSB sign bit
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "VttGenStatusSelCount = 1: CountAccumulator = %d \n", CountAccumulator);
      }
    }
    SystemPower.RdPower += (UINT16)(CountAccumulator / MAX_VTT_REGS);

    for (VTTIndex = 0; VTTIndex < MAX_VTT_REGS; VTTIndex++) {
      GetSetVal = 4;
      MrcGetSetChStrb(MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VTTIndex, VttGenStatusSelCount, WriteCached, &GetSetVal);
      GetSetVal = 1;
      MrcGetSetChStrb(MrcData, MRC_IGNORE_ARG, MRC_IGNORE_ARG, VTTIndex, VttGenStatusResetCount, WriteCached, &GetSetVal);
    }

    // Run CPGC
    RunIOTest (MrcData, McChBitMask, Outputs->DQPat, 1, 0);

    CountAccumulator = 0;
    for (VTTIndex = 0; VTTIndex < MAX_VTT_REGS; VTTIndex++) {
      MrcGetSetNoScope(MrcData, VttGenStatusCount, ReadUncached, &GetSetVal);
      if (GetSetVal > 0x7FFF) { // Value is actually negative (sign/magnitude encoding)
        CountAccumulator = 0xFFFF; // Set panic count to maximum if SelCount=4 gives a negative result, so that this PanicVttDnLp value is not chosen.
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "VttGenStatusSelCount = 4: CountAccumulator = 0xFFFF \n");
        }
        break;
      }
      CountAccumulator += (UINT32)GetSetVal;
      if (MRC_POWER_TRAINING_DEBUG) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "VttGenStatusSelCount = 4: CountAccumulator = %d \n", CountAccumulator);
      }
    }
    if (CountAccumulator == 0xFFFF) { // Set panic count to maximum if SelCount=4 gives a negative result, so that this PanicVttDnLp value is not chosen.
      SystemPower.RdPower = 0xFFFF;
    } else {
      SystemPower.RdPower += (UINT16)(CountAccumulator / MAX_VTT_REGS);
    }


    SystemPower.WrPower = 0;
    SystemPower.TotalPwr = 0;
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "VttGenStatus: Final RdPower = %d \n", SystemPower.RdPower);
    }
    return mrcSuccess;
  }

  // We can use FIVR for DRAM ODT training, as we don't care about power changes related to DRAM ODT changes due to sweeping from low to high power consumption and stopping once UPMs are met.
  // TGL_POWER_TRAINING_FUTURE - As we don't need to set any registers in the PCU with read or write specific power, we may be able to use total power for everything and eliminate the read-only and write-only CPGC tests
  if (FIVR_POWER_MEASUREMENT_ENABLED && SupportsFivrPower(MrcData, Params[0])) {
    //DdrCrDataOffsetComp.Data = ChannelOut->DataCompOffset[Byte];

    SetupIOTestBasicVA(MrcData, McChBitMask, OPT_PARAM_LOOP_COUNT, NSOE, 0x4, 0, 8, PatWrRd, 0, 0); // Read-only test all channels (0x4 used as a flag). TGL_POWER_TRAINING_FUTURE - LC may need to be optimized based on number of ranks and test type (Rx Turnarounds), write-only test may not be setup correctly.
    Timeout = MrcCall->MrcGetCpuTime() + 10000; // 10 seconds timeout
    do {
      // TGL_POWER_TRAINING_FUTURE - Read initial FIVR input power accumulator from P_IN_ACCUMULATOR and sample count from P_IN_NUM_SAMPLES_SNAPSHOT
      // MrcGetSetDdrIoGroupController0 (MrcData, FivrPInValue, ReadUncached, &GetSetVal);
      FivrPInValueStart = (UINT16)GetSetVal;
      // MrcGetSetDdrIoGroupController0 (MrcData, FivrPInCount, ReadUncached, &GetSetVal);
      FivrPInCountStart = (UINT16)GetSetVal;

      RunIOTest(MrcData, McChBitMask, Outputs->DQPat, 1, 0); // This is a standard point test

      // TGL_POWER_TRAINING_FUTURE - Read final FIVR input power accumulator from P_IN_ACCUMULATOR and sample count from P_IN_NUM_SAMPLES_SNAPSHOT
      // MrcGetSetDdrIoGroupController0 (MrcData, FivrPInValue, ReadUncached, &GetSetVal);
      FivrPInValueEnd = (UINT16)GetSetVal;
      // MrcGetSetDdrIoGroupController0 (MrcData, FivrPInCount, ReadUncached, &GetSetVal);
      FivrPInCountEnd = (UINT16)GetSetVal;
    } while (FivrPInCountEnd < FivrPInCountStart); // If this is not true, then we overflowed
    if ((MrcCall->MrcGetCpuTime() >= Timeout) && (MrcCall->MrcGetCpuTime() < Timeout)) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_ERROR, "\nTimeout in FIVR read power measurement! \n");
    }

    FivrPInValueEnd = FivrPInValueEnd - FivrPInValueStart;
    FivrPInCountEnd = FivrPInCountEnd - FivrPInCountStart;
    // TGL_POWER_TRAINING_FUTURE - Result must be in mW, so we have to adjust for this by finding out what the units are in FIVR. Units are (VR_MAX.MAX_P_OUT * 2^VR_MAX.P_OUT_EXP / 255) mW
    // MrcGetSetDdrIoGroupController0 (MrcData, MaxPOutValue, ReadUncached, &GetSetVal);
    MaxPOutValue = (UINT32)GetSetVal;
    // MrcGetSetDdrIoGroupController0 (MrcData, MaxPOutExp, ReadUncached, &GetSetVal);
    MaxPOutUnits = 1;
    for (Iterator = 0; Iterator < GetSetVal; Iterator++) {
      MaxPOutUnits *= 2;
    }
    MaxPOutValue *= MaxPOutUnits;
    //#ifdef LOCAL_STUB_FLAG
    FivrPInValueEnd = 12750;
    MaxPOutValue = 0; // 1000; // TGL_POWER_TRAINING_FUTURE Hardcode to 0 for now, enable later
    FivrPInCountEnd = 100;
    //#endif
    //SystemPower.RdPower = (UINT16)((FivrPInValueEnd * MaxPOutValue * Scale) / (FivrPInCountEnd * 255));

    SetupIOTestBasicVA (MrcData, McChBitMask, OPT_PARAM_LOOP_COUNT, NSOE, 0x2, 0, 8, PatWrRd, 0, 0); // Write-only test all channels (0x2 used as a flag). TGL_POWER_TRAINING - LC may need to be optimized based on number of ranks and test type (Rx Turnarounds), write-only test may not be setup correctly.
    Timeout = MrcCall->MrcGetCpuTime() + 10000; // 10 seconds timeout
    do {
      // TGL_POWER_TRAINING_FUTURE - Read initial FIVR input power accumulator from P_IN_ACCUMULATOR and sample count from P_IN_NUM_SAMPLES_SNAPSHOT
      // MrcGetSetDdrIoGroupController0 (MrcData, FivrPInValue, ReadUncached, &GetSetVal);
      FivrPInValueStart = (UINT16)GetSetVal;
      // MrcGetSetDdrIoGroupController0 (MrcData, FivrPInCount, ReadUncached, &GetSetVal);
      FivrPInCountStart = (UINT16)GetSetVal;

      RunIOTest(MrcData, McChBitMask, Outputs->DQPat, 1, 0); // This is a standard point test

      // TGL_POWER_TRAINING_FUTURE - Read final FIVR input power accumulator from P_IN_ACCUMULATOR and sample count from P_IN_NUM_SAMPLES_SNAPSHOT
      // MrcGetSetDdrIoGroupController0 (MrcData, FivrPInValue, ReadUncached, &GetSetVal);
      FivrPInValueEnd = (UINT16)GetSetVal;
      // MrcGetSetDdrIoGroupController0 (MrcData, FivrPInCount, ReadUncached, &GetSetVal);
      FivrPInCountEnd = (UINT16)GetSetVal;
    } while ((FivrPInCountEnd < FivrPInCountStart) && (MrcCall->MrcGetCpuTime() < Timeout)); // If this is not true, then we overflowed
    if (MrcCall->MrcGetCpuTime() >= Timeout) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_ERROR, "\nTimeout in FIVR write power measurement! \n");
    }

    FivrPInValueEnd = FivrPInValueEnd - FivrPInValueStart;
    FivrPInCountEnd = FivrPInCountEnd - FivrPInCountStart;
    //#ifdef LOCAL_STUB_FLAG
    FivrPInValueEnd = 12750;
    MaxPOutValue = 0; // 1000; // TGL_POWER_TRAINING_FUTURE Hardcode to 0 for now, enable later
    FivrPInCountEnd = 100;
    //#endif
    //SystemPower.WrPower = (UINT16)((FivrPInValueEnd * MaxPOutValue * Scale) / (FivrPInCountEnd * 255));

    //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Power->RdPower : %d Power->WrPower: %d ParamVector: %d ParamVector: %d ParamVector: %d ParamVector: %d \n", Power->RdPower, Power->WrPower, ParamVector[0], ParamVector[1], ParamVector[2], ParamVector[3]);
    //SystemPower.TotalPwr = ((SystemPower.RdPower) + (SystemPower.WrPower)) / 2; // [mW] x Scale
    //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Power->TotalPwr %d\n", Power->TotalPwr);
  } else {
    SystemPower.TotalPwr = 0;

    // The only reason you use power is if every combination is above UPM, otherwise you just drive to the UPM limit.
    // If every single combination is above UPMs, then the only thing we need to know is which of these combinations has the lowest power, nothing else.
    // So two linear approximations added together will do the job, as it will tell us which value set has the smallest power burn.
    // None of the power values for the two params will be accurate relative to each other, but we don't care, we just need the smallest combination.
    // There's only 1 global minimum that satisfies that criteria regardless of whether the power values for the linear approximations are acccurate relative to each other.
    for (Iterator = 0; Iterator < ParamsCount; Iterator++) {
      switch (Params[Iterator]) {
        case OptRxLoad:
        case OptRxBias:
        case OptRdDqOdt:
        case OptRdDqsOdt:
        case OptCCCSComp:
        case OptCCCDS:
        case OptCCCDSDnCoarse:
        case OptCCCDSUpCoarse:
        case OptDimmRon:
        case OptDimmOdtCA:
        case OptDimmOdtNom:
        case OptDimmOdtPark:
        case OptDimmOdtParkNT:
        case OptDimmOdtNomNT:
        case OptDimmOdtWr:
        case OptSComp:
        case OptWrDS:
        case OptWrDSDnCoarse:
        case OptWrDSUpCoarse:
          SystemPower.TotalPwr += CalcLinearNormalizedPower (MrcData, Params[Iterator], ParamValues[Iterator], TRUE);
          break;
        case OptCCCTxEq:
        case OptTxEq:
        case OptVddq:
          SystemPower.TotalPwr += CalcLinearNormalizedPower (MrcData, Params[Iterator], ParamValues[Iterator], FALSE);
          break;
        default:
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_ERROR, "CalcSysPower called with invalid param! %d\n", Params[Iterator]);
          break;
      }
    }

    SystemPower.TotalPwr *= Scale;
  }

  // update Mrc struct with Base line numbers
  if (OdtPowerSaving->BaseFlag == FALSE) {
    OdtPowerSaving->BaseSavingTotal = (UINT16) SystemPower.TotalPwr;
  } else {
    OdtPowerSaving->MrcSavingTotal = (UINT16) SystemPower.TotalPwr;
  }

  return mrcSuccess;
}

#if EYE_MARGIN_SUPPORT
/**
  This function prints out the Margin eye diagram for ParamT/ParamV.

  @param[in] MrcData - Include all MRC global data.
  @param[in] Channel - Channel to margin.
  @param[in] Ranks   - Bit mask of Ranks to margin.
  @param[in] ParamT  - Time parameter to margin.
  @param[in] ParamV  - Voltage parameter to margin.
  @param[in] CmdIteration - Whether doing CLK/CMD/CTL (only used with ParamT == CmdT and ParamV == CmdV)
  @param[in] CmdGroup     - Determines which CmdGrp to use (0 through 4, only used with CmdIteration == MrcIterationCmd)
  @param[in] Start   - Starting point for margining.
  @param[in] Stop    - Stopping point for margining.
  @param[in] Repeats - Number of times to repeat the test to average out any noise.
  @param[in] NoPrint - Switch to skip printing.

  @retval Nothing
**/
#define MRC_EM_MAX_H  (192)
#define MRC_EM_MAX_W  (128)
void
EyeMargin (
  IN MrcParameters *const MrcData,
  IN UINT8                Ranks,
  IN UINT8                ParamT,
  IN UINT8                ParamV,
  IN UINT8                CmdIteration,
  IN UINT8                CmdGroup,
  IN INT8                 Start,
  IN INT8                 Stop,
  IN UINT16               SearchLimits,
  IN UINT8                LoopCount,
  IN UINT8                Repeats
  )
{
  const MrcInput    *Inputs;
  MrcDebug          *Debug;
  const MRC_FUNCTION *MrcCall;
  MrcOutput         *Outputs;
  MrcChannelOut     *ChannelOut;
  MrcControllerOut  *ControllerOut;
  MrcIntOutput      *IntOutputs;
  MrcIntCmdTimingOut *IntCmdTiming;
  MrcStatus         Status;
  UINT32            (*MarginByte)[MAX_RESULT_TYPE][MAX_RANK_IN_CHANNEL][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_EDGES];
  UINT32            BERStats[4];
  UINT16            SaveMargin[MRC_EM_MAX_W][MAX_CHANNEL][MAX_SDRAM_IN_DIMM][MAX_EDGES];
  BOOLEAN           Eye[MAX_CHANNEL][MRC_EM_MAX_W][MRC_EM_MAX_H];
  BOOLEAN           Lines[MAX_CHANNEL][MRC_EM_MAX_H];
  UINT32            Channel;
  UINT8             i,j;
  UINT16            MinEdge;
  UINT8             ResultTypeV = 0;
  UINT8             ChBitMask;
  UINT8             Byte;
  UINT8             Rank;
  UINT8             Edge;
  UINT8             FirstRank;
  UINT8             NumBytes;
  UINT8             BMap[9];  // Need by GetBERMarginByte
  UINT8             MaxMarginV;
  UINT8             localR[MAX_CHANNEL];
  UINT8             Rep;
  INT8              Index;
  UINT8             IndexOff;
  INT8              Off;
  UINT8             OffStep;
  UINT8             byteMax[MAX_CHANNEL];
  UINT8             sign;
  UINT8             SafeOff[MAX_CHANNEL]={0,0};
  UINT8             CurrValue;
  INT64             GetSetVal;
  BOOLEAN           Ddr4;
  BOOLEAN           Lpddr4;
  BOOLEAN           IsMrVref;
  Inputs        = &MrcData->Inputs;
  MrcCall       = Inputs->Call.Func;
  Outputs       = &MrcData->Outputs;
  Debug         = &Outputs->Debug;
  MarginByte    = &Outputs->MarginResult;
  ControllerOut = &Outputs->Controller[0];
  IndexOff      = 0;
  Ddr4          = (Outputs->DdrType == MRC_DDR_TYPE_DDR4);
  Lpddr4        = (Outputs->DdrType == MRC_DDR_TYPE_LPDDR4);
  IntOutputs    = (MrcIntOutput *) (MrcData->IntOutputs.Internal);

  MrcCall->MrcSetMem ((UINT8 *) localR, sizeof (localR), 0);
  MrcCall->MrcSetMem ((UINT8 *) Eye, sizeof (Eye), 0);
  MrcCall->MrcSetMem ((UINT8 *) Lines, sizeof (Lines), 0);
  MrcCall->MrcSetMem ((UINT8 *) SaveMargin, sizeof (SaveMargin), 0);
  MrcCall->MrcSetMemDword (BERStats, sizeof (BERStats) / sizeof (UINT32), 0);
  for (Byte = 0; Byte < ARRAY_COUNT (BMap); Byte++) {
    BMap[Byte] = Byte;
  }
  ResultTypeV = GetMarginResultType (ParamV);
  IsMrVref = (ParamV == WrV) || (Lpddr4 && (ParamV == CmdV));

  if ((ParamT == CmdT) | (ParamT == CmdV)) {
    // @todo Update with McChBitMask
    SetupIOTestCADB (MrcData, Outputs->ValidChBitMask, LoopCount, NSOE, 1, 0);
  } else {
    // @todo Update with McChBitMask
    SetupIOTestBasicVA (MrcData, Outputs->ValidChBitMask, LoopCount, 0, 0, 0, 8, PatWrRd, 0, 0);  //set test to all channels
  }

  ChBitMask = 0;
  for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
    if (MrcChannelExist (MrcData, 0, Channel)) {
      ChannelOut      = &ControllerOut->Channel[Channel];
      localR[Channel] = ChannelOut->ValidRankBitMask & Ranks;
      ChBitMask |= SelectReutRanks (MrcData, (UINT8) Channel, localR[Channel], FALSE, 0);
    } // Channel Exists
  } // Channel
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "EM: ChBitMask 0x%X, Ranks 0x%X, LocalR[C0] 0x%X, LocalR[C1] 0x%X\n", ChBitMask, Ranks, localR[0], localR[1]);
  if (ChBitMask == 0) {
    return;
  }

  // Find the first selected rank
  FirstRank = 0;
  for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
    for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
      if ((1 << Rank) & localR[Channel]) {
        FirstRank = Rank;  // could be in any channel
        break;
      }
    } // Rank
  } // Channel
  MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "EM: FirstRank 0x%X\n", FirstRank);

  // Store margin results for
  NumBytes = (UINT8) Outputs->SdramCount;
  //if (ParamV == CmdV) {
  //  NumBytes = 1;
  //}

  // Ensure all bit are checking for errors.
  MrcSetDataAndEccErrMsk (MrcData, 0xFFFFFFFFFFFFFFFFULL, 0xFF);

  // Find MaxMargin for this channel
  MaxMarginV = GetVrefOffsetLimits (MrcData, ParamV);

  //if (MAX_POSSIBLE_TIME < Stop) {
  //  Stop = MAX_POSSIBLE_TIME;
  //}
  //if (-MAX_POSSIBLE_TIME > Start) {
  //  Start = -MAX_POSSIBLE_TIME;
  //}

  if (ParamT == RdT) {
    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (!MrcChannelExist (MrcData, 0, Channel)) {
        continue;
      }
      for (sign = 0; sign < 2; sign++) {
        byteMax[Channel] = (sign) ? ABS(Stop) : ABS(Start);
        for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
          byteMax[Channel] = MrcCalcMaxRxMargin (MrcData, ParamT, /**Controller**/ CONTROLLER_0, (UINT8) Channel, FirstRank, Byte, 0, byteMax[Channel]);
        }
      }
    }
    Start = - byteMax[0];
    Stop  = byteMax[1];
  }

  IndexOff = MRC_EM_MAX_W / 2 + Start;
  // No need to search too far
  if (MaxMarginV > SearchLimits) {
    MaxMarginV = (UINT8) (SearchLimits);
  }

  OffStep = 1;

  // Loop through all Test Params and Measure Margin
  for (Off = Start; Off < Stop + 1; Off += OffStep) {
    //MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Timing: %d\n", Off);
    Index = Off - Start;

    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (!MrcChannelExist (MrcData, 0, Channel)) {
        continue;
      }
      if (ParamT == CmdT) {
        IntCmdTiming  = &IntOutputs->Controller[0].CmdTiming[Channel];
        if (CmdIteration == MrcIterationClock) {
          // the 3 is to select two SubChannels
          //@todo - 2MC
          ShiftPIforCmdTraining (MrcData, CONTROLLER_0, (UINT8) Channel, CmdIteration, localR[Channel], 3, Off, 0);
        }
        if (CmdIteration == MrcIterationCmd) {
          if (CmdGroup >= MAX_COMMAND_GROUPS) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Error: CmdGroup out of bounds! %d", CmdGroup);
            return;
          }
          //@todo - 2MC
          ShiftPIforCmdTraining (MrcData, CONTROLLER_0, (UINT8) Channel, CmdIteration, localR[Channel], 1 << CmdGroup, (INT32) IntCmdTiming->CmdPiCode[CmdGroup] + Off, 0);
        }
        if (CmdIteration == MrcIterationCtl) {
          CurrValue = 0;
          for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
            if (localR[Channel] & (1 << Rank)) {
              MrcGetSetDdrIoGroupChannelStrobe (MrcData, Channel, Rank, CtlGrpPi, ReadFromCache, &GetSetVal);
              CurrValue = (UINT8) IntCmdTiming->CtlPiCode[Rank];
              break;
            }
          }
          //@todo - 2MC
          ShiftPIforCmdTraining (MrcData, CONTROLLER_0, (UINT8) Channel, CmdIteration, localR[Channel], localR[Channel], CurrValue + Off, 0);
        }
        MrcResetSequence (MrcData);
      } else {
        for (Byte = 0; Byte < NumBytes; Byte++) {
          Status = ChangeMargin (MrcData, ParamT, Off, 0, 0, /**Controller**/ 0, (UINT8) Channel, FirstRank, Byte, 0, 1, 0);
        }
      }
      // Assign to last pass margin results by reference
      // get lowest margin from all ch/rankS/byte save in FirstRank
      //if (ParamT != CmdT) {
      //  Status = GetMarginByte (
      //            MrcData,
      //            Outputs->MarginResult,
      //            ParamV,
      //            FirstRank,
      //            localR[Channel]
      //            );
      //}
    } // Channel

    for (Rep = 0; Rep < Repeats; Rep++) {
      // Run Margin Test - margin_1d with chosen param
      Status = MrcGetBERMarginByte (
        MrcData,
        Outputs->MarginResult,
        ChBitMask,
        (IsMrVref) ? Ranks : FirstRank,
        (IsMrVref) ? Ranks : FirstRank,
        ParamV,
        0, // Mode
        BMap,
        0,
        MaxMarginV,
        0,
        BERStats
        );
      //MrcDisplayMarginResultArray (MrcData, ResultTypeV);
      // Record Results
      for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
        if (!MrcChannelExist (MrcData, 0, Channel)) {
          continue;
        }
        for (Edge = 0; Edge < MAX_EDGES; Edge++) {
          for (Byte = 0; Byte < NumBytes; Byte++) {
            if ((Index > MRC_EM_MAX_W) || (Index < 0)) {
              MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Error: SaveMargin array out of bounds! %d", Index);
              return;
            }

            if (Rep == 0) {
              SaveMargin[Index][Channel][Byte][Edge] = 0;
            }
            SaveMargin[Index][Channel][Byte][Edge] += (UINT16) (*MarginByte)[ResultTypeV][FirstRank][Channel][Byte][Edge];
          } // Byte
        } // Edge
      } // Channel
    } // Repeats

    for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
      if (!MrcChannelExist (MrcData, 0, Channel)) {
        continue;
      }
      for (Edge = 0; Edge < MAX_EDGES; Edge++) {
        MinEdge = 0xFFFF;
        for (Byte = 0; Byte < NumBytes; Byte++) {
          if ((Index > (MRC_EM_MAX_W - 1)) || (Index < 0)) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Error: SaveMargin array out of bounds! %d", Index);
            return;
          }

          SaveMargin[Index][Channel][Byte][Edge] /= Repeats;
          if (MinEdge > SaveMargin[Index][Channel][Byte][Edge]) {
            MinEdge = SaveMargin[Index][Channel][Byte][Edge];
          }
        } // Byte

        if ((Index > (MRC_EM_MAX_W - 1)) ||
            (Index < 0) ||
            ((MRC_EM_MAX_H / 2 - (MinEdge - 1) / 10) > (MRC_EM_MAX_H - 1)) ||
            ((MRC_EM_MAX_H / 2 - (MinEdge - 1) / 10) < 0) ||
            ((MRC_EM_MAX_H / 2 + (MinEdge - 1) / 10) > (MRC_EM_MAX_H - 1)) ||
            ((MRC_EM_MAX_H / 2 + (MinEdge - 1) / 10) < 0)
            ) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Error: Eye or Lines array out of bounds!\n");
          return;
        }

        if (Edge) {
          Eye[Channel][Index + IndexOff][MRC_EM_MAX_H / 2 - (MinEdge) / 10]  = 1;
          Lines[Channel][MRC_EM_MAX_H / 2 - (MinEdge) / 10] = 1;
        } else {
          Eye[Channel][Index + IndexOff][MRC_EM_MAX_H / 2 + (MinEdge) / 10]  = 1;
          Lines[Channel][MRC_EM_MAX_H / 2 + (MinEdge) / 10] = 1;
        }
      } // Edge
    } // Channel
  } // Offset

  // Print the box
  for (Channel = 0; Channel < MAX_CHANNEL; Channel++) {
    if (!MrcChannelExist (MrcData, 0, Channel)) {
      continue;
    }
    MRC_DEBUG_MSG (
      Debug,
      MSG_LEVEL_NOTE,
      "Plot Eye across ParamT = %s ParamV = %s CmdIter = %s CmdGrp = 0x%x settings:(Start=%d,Stop=%d) LC = %d  Channel = %d Ranks = 0x%x\n",
      gMarginTypesStr[ParamT],
      gMarginTypesStr[ParamV],
      CmdIterTypesString[CmdIteration],
      CmdGroup,
      Start,
      Stop,
      LoopCount,
      Channel,
      localR[Channel]
      );
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t---------------------------------------------------------- +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++\n");
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\t66666555555555544444444443333333333222222222211111111110000000000000000000111111111122222222223333333333444444444455555555556666\n");
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "Vref\t43210987654321098765432109876543210987654321098765432109876543210123456789012345678901234567890123456789012345678901234567890123\n");
    for (i = 0; i < MRC_EM_MAX_H; i++) {
      if (Lines[Channel][i]) {
        // print only fail lines
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%3d:\t", MRC_EM_MAX_H / 2 - i); // per ch
        for (j = 0; j < MRC_EM_MAX_W; j++) {
          if (Eye[Channel][j][i]) {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s", "#"); // per ch
          } else {
            MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "%s", ((j == (MRC_EM_MAX_W) / 2) || (i == (MRC_EM_MAX_H) / 2)) ? "+" : " "); // per ch
          }
        } // j
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "\n");//per ch
      }
    } // i
  } // Channel
  if (ParamT != CmdT) {
    Status = ChangeMargin (MrcData, ParamT, 0, 0, 1, /**Controller**/ 0, 0, 0, 0, 0, 1, 0);
  } else {
    //restore settings
    ShiftCh2Ch (MrcData, 0xff, SafeOff, 1, 0, 0, 0);
  }
  Status = MrcResetSequence (MrcData);
}
#endif // EYE_MARGIN_SUPPORT

/**
This function returns if slew rate is enabled

@param[in] MrcData    - Include all MRC global data.
@param[in] Param      - Parameter defining the desired digital compensation offset.
@param[in] Controller - Controller index to work on.
@param[in] Channel    - Channel index to work on.
@param[in] Byte       - Byte index to work on.

@retval BOOLEAN - TRUE if slew rate is enabled, else FALSE
**/
BOOLEAN
IsSlewRateEnabled(
  IN MrcParameters *const MrcData,
  IN const UINT8          Param,
  IN const UINT8          Controller,
  IN const UINT8          Channel,
  IN const UINT8          Byte
  )
{
  BOOLEAN Enabled;
  INT64 GetSetVal;

  switch (Param) {
    case SCompDq:
      MrcGetSetChStrb(MrcData, Controller, Channel, Byte, GsmIocDqSlewDlyByPass, ReadFromCache, &GetSetVal);
      Enabled = (GetSetVal == SLEW_RATE_ENABLED);
      MrcGetSetNoScope(MrcData, DqScompPC, ReadFromCache, &GetSetVal);
      Enabled &= (GetSetVal == CYCLE_LOCK);
      break;
    case SCompCmd:
      MrcGetSetMcCh(MrcData, Controller, Channel, SCompBypassCmd, ReadFromCache, &GetSetVal);
      Enabled = (GetSetVal == SLEW_RATE_ENABLED);
      MrcGetSetNoScope(MrcData, CmdScompPC, ReadFromCache, &GetSetVal);
      Enabled &= (GetSetVal == CYCLE_LOCK);
      break;
    case SCompCtl:
      MrcGetSetMcCh(MrcData, Controller, Channel, SCompBypassCtl, ReadFromCache, &GetSetVal);
      Enabled = (GetSetVal == SLEW_RATE_ENABLED);
      MrcGetSetNoScope(MrcData, CtlScompPC, ReadFromCache, &GetSetVal);
      Enabled &= (GetSetVal == CYCLE_LOCK);
      break;
    default:
      Enabled = TRUE;
  }
  return Enabled;
}

/**
  This function optimize the digital offsets by reducing the digital
  offset and apply the difference to the global one.

  @param[in] MrcData    - Include all MRC global data.
  @param[in] Param      - Parameter defining the desired digital compensation offset.

  @retval MrcStatus - mrcSuccess
**/
MrcStatus
OptimizeCompOffset(
  IN MrcParameters *const MrcData,
  IN const UINT8          Param
  )
{
  MrcDebug           *Debug;
  MrcInput           *Inputs;
  MrcOutput          *Outputs;
  MrcChannelOut      *ChannelOut;
  const MRC_FUNCTION *MrcCall;
  UINT8             GlobalParam[MAX_COMPS_OPTIMIZED];
  UINT8             CurrCompVref[MAX_COMPS_OPTIMIZED];
  UINT8             NewCompVref[MAX_COMPS_OPTIMIZED];
  UINT8             NewCompCode[MAX_COMPS_OPTIMIZED];
  INT8              Sign;
  INT8              RoundSign;
  INT16             AverageCompOffsetPerCompOffset[MAX_COMP_OFFSETS_OPTIMIZED];
  INT16             AverageCompOffsetPerComp[MAX_COMPS_OPTIMIZED];
  UINT8             NumCompOffsetsPerComp[MAX_COMPS_OPTIMIZED];
  UINT16            Offset = 0;
  UINT16            ExtendedCompOffsetValue = 0;
  INT16             MinDelta = 0;
  INT16             TotalCompOffset = 0;
  INT16             CurrentDelta = 0;
  UINT8             CompVrefOffset = 1;
  UINT8             Comp;
  UINT8             CompOffset;
  UINT8             CompOffsetBytes[MAX_COMP_OFFSETS_OPTIMIZED];
  UINT8             CompOffsetCompIndex[MAX_COMP_OFFSETS_OPTIMIZED];
  UINT8             NumTotalCompCodes = 0;
  UINT8             NumTotalSideEffectCompCodes = 0;
  UINT8             NumCompOffsets = 0;
  UINT8             BestCompVref[MAX_COMPS_OPTIMIZED];
  UINT8             MaxCompOffsetBits[MAX_COMP_OFFSETS_OPTIMIZED];
  UINT8             Byte;
  UINT8             Bytes[MAX_COMP_OFFSETS_OPTIMIZED];
  UINT8             Controller = 0;
  UINT8             Channel;
  UINT8             MaxComp[MAX_COMPS_OPTIMIZED];
  UINT8             MaxCompVref[MAX_COMPS_OPTIMIZED];
  UINT8             MinCompVref[MAX_COMPS_OPTIMIZED];
  UINT8             CurrSCompPC[MAX_COMPS_OPTIMIZED];
  INT16             TotalCompCodes[MAX_COMPS_OPTIMIZED][MAX_CONTROLLER][MAX_CHANNEL][MAX_SDRAM_IN_DIMM];
  UINT8             CurrentCompCode[MAX_COMPS_OPTIMIZED];
  UINT32            NewMainCompVref_Target;
  UINT32            OldMainCompVref_Target;
  UINT32            SideEffectCompVref_Target;
  INT64             GetSetVal;
  UINT64            Timeout;
  BOOLEAN           ContinueOptimizing = TRUE;
  TOptParamOffset   CompCodeGroup[MAX_COMPS_OPTIMIZED];
  UINT8             CompCodeDirection[MAX_COMPS_OPTIMIZED];
  BOOLEAN           CompCodeSideEffect[MAX_COMPS_OPTIMIZED];
  GSM_GT            CompOffsetGroup[MAX_COMP_OFFSETS_OPTIMIZED];
  DATA0CH0_CR_DDRCRDATAOFFSETCOMP_STRUCT DdrCrDataOffsetComp;
  CH0CCC_CR_DDRCRCTLCACOMPOFFSET_STRUCT DdrCrCmdOffsetComp;
  MrcStatus         Status;
  UINT32            NewCompValue;

  Inputs  = &MrcData->Inputs;
  Outputs = &MrcData->Outputs;
  Inputs = &MrcData->Inputs;
  Debug = &Outputs->Debug;
  MrcCall = Inputs->Call.Func;

  DdrCrDataOffsetComp.Data = 0;
  DdrCrCmdOffsetComp.Data = 0;

  MrcCall->MrcSetMem((UINT8 *)&CurrSCompPC, sizeof(CurrSCompPC), 0);
  MrcCall->MrcSetMem((UINT8 *)&AverageCompOffsetPerComp, sizeof(AverageCompOffsetPerComp), 0);
  MrcCall->MrcSetMem((UINT8 *)&NumCompOffsetsPerComp, sizeof(NumCompOffsetsPerComp), 0);
  MrcCall->MrcSetMem((UINT8 *)&CompCodeSideEffect, sizeof(CompCodeSideEffect), FALSE);
  MrcCall->MrcSetMem((UINT8 *)&AverageCompOffsetPerCompOffset, sizeof(AverageCompOffsetPerCompOffset), 0);
  MrcCall->MrcSetMem((UINT8 *)&Bytes, sizeof(Bytes), 0);
  MrcCall->MrcSetMem((UINT8 *)&NewCompVref, sizeof(NewCompVref), 0xFF);

  // When defining arrays here, the first comp vref must always be an optimized comp vref, not a side effect comp vref
  switch (Param) {
    case WrDSUp:
      // Comps
      CompCodeGroup[0] = OptWrDS; // TxRonUp
      CompCodeDirection[0] = COMP_UP;

      // Comp Offsets
      CompOffsetGroup[0] = DqDrvUpCompOffset;
      CompOffsetBytes[0] = (UINT8)Outputs->SdramCount;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 6; // Comp offset

      // Comp Vref
      MrcGetSetNoScope(MrcData, DqDrvVrefUp, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = MRC_COMP_VREF_STEP_SIZE;
      MinCompVref[0] = 0;
      GlobalParam[0] = WrDSUp;
      MaxComp[0] = 63; // Comp

      // Side effects

        // Comps
        CompCodeGroup[1] = OptWrDS; // TxRonDn
        CompCodeDirection[1] = COMP_DN;
        CompCodeSideEffect[1] = TRUE;

        // Comp Offsets
        CompOffsetGroup[1] = DqDrvDnCompOffset;
        CompOffsetBytes[1] = (UINT8)Outputs->SdramCount;
        CompOffsetCompIndex[1] = 1;
        MaxCompOffsetBits[1] = 6; // Comp offset

        // Comp Vref
        MrcGetSetNoScope(MrcData, DqDrvVrefDn, ReadFromCache, &GetSetVal);
        CurrCompVref[1] = (UINT8)GetSetVal;
        MaxCompVref[1] = MRC_COMP_VREF_STEP_SIZE;
        MinCompVref[1] = 0;
        GlobalParam[1] = WrDSDn;
        MaxComp[1] = 63; // Comp

        if (MrcData->Outputs.OdtMode == MrcOdtModeVss) {
          // In Vss mode only odt down is valid

          // Comps
          CompCodeGroup[2] = OptRdDqOdt; // CompRcompOdtDn
          CompCodeDirection[2] = COMP_DN;
          CompCodeSideEffect[2] = TRUE;
          NumTotalCompCodes = 3;
          NumTotalSideEffectCompCodes = 2;

          // Comp Offsets
          CompOffsetGroup[2] = DqsOdtCompOffset;
          CompOffsetGroup[3] = DqOdtCompOffset;
          CompOffsetBytes[2] = (UINT8)Outputs->SdramCount;
          CompOffsetBytes[3] = (UINT8)Outputs->SdramCount;
          CompOffsetCompIndex[2] = 2;
          CompOffsetCompIndex[3] = 2;
          MaxCompOffsetBits[2] = 5;
          MaxCompOffsetBits[3] = 5;
          NumCompOffsets = 3;

          // Comp Vref
          MrcGetSetNoScope(MrcData, DqOdtVrefDn, ReadFromCache, &GetSetVal);
          CurrCompVref[2] = (UINT8)GetSetVal;
          MaxCompVref[2] = MRC_COMP_VREF_STEP_SIZE;
          MinCompVref[2] = 0;
          GlobalParam[2] = RdOdtDn;
          MaxComp[2] = 63;
        } else {
          // Comps
          NumTotalCompCodes = 2;
          NumCompOffsets = 2;
          NumTotalSideEffectCompCodes = 1;
        }

      break;

    case WrDSDn:
      // Comps
      CompCodeGroup[0] = OptWrDS; // TxRonDn
      CompCodeDirection[0] = COMP_DN;
      NumTotalCompCodes = 1;

      // Comp Offsets
      CompOffsetGroup[0] = DqDrvDnCompOffset;
      CompOffsetBytes[0] = (UINT8)Outputs->SdramCount;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 6; // Comp offset
      NumCompOffsets = 1;

      // Comp Vref
      MrcGetSetNoScope(MrcData, DqDrvVrefDn, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = MRC_COMP_VREF_STEP_SIZE;
      MinCompVref[0] = 0;
      GlobalParam[0] = WrDSDn;
      MaxComp[0] = 63; // Comp

      break;

    case WrDSCmdUp:
      // Both CmdRCompDrvUpOffset and RloadCompOffset use comp vref registers that need to be at the same level

      // Comps
      CompCodeGroup[0] = OptCmdDS; // WrDSCodeUpCmd;
      CompCodeDirection[0] = COMP_UP;

      // Comp Offsets
      CompOffsetGroup[0] = CmdRCompDrvUpOffset;
      CompOffsetBytes[0] = 1;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 4;

      // Comp Vref
      MrcGetSetNoScope(MrcData, CmdDrvVrefUp, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = MRC_COMP_VREF_STEP_SIZE;
      MinCompVref[0] = 0;
      GlobalParam[0] = WrDSCmdUp;
      MaxComp[0] = 63;

      // Side effects

        // Comps
        CompCodeGroup[1] = OptCmdDS; // WrDSCodeDnCmd;
        CompCodeDirection[1] = COMP_DN;
        CompCodeSideEffect[1] = TRUE;
        NumTotalSideEffectCompCodes = 1;

        // Comp Offsets
        CompOffsetGroup[1] = CmdRCompDrvDownOffset;
        CompOffsetBytes[1] = 1;
        CompOffsetCompIndex[1] = 1;
        MaxCompOffsetBits[1] = 4;

        // Comp Vref
        MrcGetSetNoScope(MrcData, CmdDrvVrefDn, ReadFromCache, &GetSetVal);
        CurrCompVref[1] = (UINT8)GetSetVal;
        MaxCompVref[1] = MRC_COMP_VREF_STEP_SIZE;
        MinCompVref[1] = 0;
        GlobalParam[1] = WrDSCmdDn;
        MaxComp[1] = 63;

      if (Outputs->RxMode != MrcRxModeMatchedN && Outputs->RxMode != MrcRxModeMatchedP) {
        // Comps
        CompCodeGroup[2] = OptRxLoad; // RloadDqsUp
        CompCodeDirection[2] = COMP_UP;
        NumTotalCompCodes = 3;

        // Comp Offsets
        CompOffsetGroup[2] = RloadCompOffset;
        CompOffsetBytes[2] = (UINT8)Outputs->SdramCount;
        CompOffsetCompIndex[2] = 2;
        MaxCompOffsetBits[2] = 5;
        NumCompOffsets = 3;

        // Comp Vref
        MrcGetSetDdrIoGroupController0(MrcData, RxLoadCompVref, ReadFromCache, &GetSetVal);
        CurrCompVref[2] = (UINT8)GetSetVal;
        MaxCompVref[2] = MRC_COMP_VREF_STEP_SIZE;
        MinCompVref[2] = 0;
        GlobalParam[2] = RxLoad;
        MaxComp[2] = 63; // Comp
      } else {
        // Comps
        NumTotalCompCodes = 2;

        // Comp Offsets
        NumCompOffsets = 2;
      }
      break;

    case WrDSCmdDn:
      // Comps
      CompCodeGroup[0] = OptCmdDS; // WrDSCodeDnCmd;
      CompCodeDirection[0] = COMP_DN;
      NumTotalCompCodes = 1;

      // Comp Offsets
      CompOffsetGroup[0] = CmdRCompDrvDownOffset;
      CompOffsetBytes[0] = 1;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 4;
      NumCompOffsets = 1;

      // Comp Vref
      MrcGetSetNoScope(MrcData, CmdDrvVrefDn, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = MRC_COMP_VREF_STEP_SIZE;
      MinCompVref[0] = 0;
      GlobalParam[0] = WrDSCmdDn;
      MaxComp[0] = 63;

      break;

    case WrDSCtlUp:
      // Comps
      CompCodeGroup[0] = OptCtlDS; // WrDSCodeUpCtl;
      CompCodeDirection[0] = COMP_UP;

      // Comp Offsets
      CompOffsetGroup[0] = CtlRCompDrvUpOffset;
      CompOffsetBytes[0] = 1;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 4;

      // Comp Vref
      MrcGetSetNoScope(MrcData, CtlDrvVrefUp, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = MRC_COMP_VREF_STEP_SIZE;
      MinCompVref[0] = 0;
      GlobalParam[0] = WrDSCtlUp;
      MaxComp[0] = 63;

      // Side effects

        // Comps
        CompCodeGroup[1] = OptCtlDS; // WrDSCodeDnCtl;
        CompCodeDirection[1] = COMP_DN;
        CompCodeSideEffect[1] = TRUE;
        NumTotalCompCodes = 2;
        NumTotalSideEffectCompCodes = 1;

        // Comp Offsets
        CompOffsetGroup[1] = CtlRCompDrvDownOffset;
        CompOffsetBytes[1] = 1;
        CompOffsetCompIndex[1] = 1;
        MaxCompOffsetBits[1] = 4;
        NumCompOffsets = 2;

        // Comp Vref
        MrcGetSetNoScope(MrcData, CtlDrvVrefDn, ReadFromCache, &GetSetVal);
        CurrCompVref[1] = (UINT8)GetSetVal;
        MaxCompVref[1] = MRC_COMP_VREF_STEP_SIZE;
        MinCompVref[1] = 0;
        GlobalParam[1] = WrDSCtlDn;
        MaxComp[1] = 63;

      break;

    case WrDSCtlDn:
      // Comps
      CompCodeGroup[0] = OptCtlDS; // WrDSCodeDnCtl;
      CompCodeDirection[0] = COMP_DN;
      NumTotalCompCodes = 1;

      // Comp Offsets
      CompOffsetGroup[0] = CtlRCompDrvDownOffset;
      CompOffsetBytes[0] = 1;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 4;
      NumCompOffsets = 1;

      // Comp Vref
      MrcGetSetNoScope(MrcData, CtlDrvVrefDn, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = MRC_COMP_VREF_STEP_SIZE;
      MinCompVref[0] = 0;
      GlobalParam[0] = WrDSCtlDn;
      MaxComp[0] = 63;

      break;

    case RdOdtUp:
      switch (MrcData->Outputs.OdtMode) {
        case MrcOdtModeVss:
          // In Vss mode only odt down is valid

          // Comps
          CompCodeGroup[0] = OptRdDqOdt; // CompRcompOdtDn
          CompCodeDirection[0] = COMP_DN;

          // Comp Offsets
          CompOffsetGroup[0] = DqsOdtCompOffset;
          CompOffsetGroup[1] = DqOdtCompOffset;
          break;
        case MrcOdtModeVddq:
        case MrcOdtModeVtt:
          // In Vtt/Vddq mode only odt up is valid

          // Comps
          CompCodeGroup[0] = OptRdDqOdt; // CompRcompOdtUp
          CompCodeDirection[0] = COMP_UP;

          // Comp Offsets
          CompOffsetGroup[0] = DqsOdtCompOffset;
          CompOffsetGroup[1] = DqOdtCompOffset;
          break;
        default:
          MRC_DEBUG_MSG(Debug, MSG_LEVEL_ERROR, "Invalid ODT Mode!");
          return mrcFail;
      }

      // Comps
      NumTotalCompCodes = 1;

      // Comp Offsets
      CompOffsetBytes[0] = (UINT8)Outputs->SdramCount;
      CompOffsetBytes[1] = (UINT8)Outputs->SdramCount;
      CompOffsetCompIndex[0] = 0;
      CompOffsetCompIndex[1] = 0;
      MaxCompOffsetBits[0] = 5;
      MaxCompOffsetBits[1] = 5;
      NumCompOffsets = 2;

      // Comp Vref
      MrcGetSetNoScope(MrcData, (Outputs->OdtMode == MrcOdtModeVss) ? DqOdtVrefDn : DqOdtVrefUp, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = MRC_COMP_VREF_STEP_SIZE;
      MinCompVref[0] = 0;
      GlobalParam[0] = (Outputs->OdtMode == MrcOdtModeVss) ? RdOdtDn : RdOdtUp;
      MaxComp[0] = 63;

      break;

    case SCompDq:
      // Comp
      CompCodeGroup[0] = OptSComp; // SCompCodeDq;
      CompCodeDirection[0] = COMP_DN;
      NumTotalCompCodes = 1;

      // Comp Offset
      CompOffsetGroup[0] = DqSCompOffset;
      CompOffsetBytes[0] = (UINT8)Outputs->SdramCount;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 5;
      NumCompOffsets = 1;

      // Comp Vref
      MrcGetSetNoScope(MrcData, DqScompPC, ReadFromCache, &GetSetVal);
      CurrSCompPC[0] = (UINT8)GetSetVal;
      MrcGetSetNoScope(MrcData, TxSlewRate, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = DDRPHY_COMP_CR_DDRCRCOMPCTL1_DqScompCells_MAX;
      MinCompVref[0] = 4;
      GlobalParam[0] = SCompDq;
      MaxComp[0] = 31;

      break;

    case SCompCmd:
      // Comp
      CompCodeGroup[0] = OptCmdSComp; // SCompCodeCmd
      CompCodeDirection[0] = COMP_DN;
      NumTotalCompCodes = 1;

      // Comp Offset
      CompOffsetGroup[0] = CmdSCompOffset;
      CompOffsetBytes[0] = 1;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 4;
      NumCompOffsets = 1;

      // Comp Vref
      MrcGetSetNoScope(MrcData, CmdScompPC, ReadFromCache, &GetSetVal);
      CurrSCompPC[0] = (UINT8)GetSetVal;
      MrcGetSetNoScope(MrcData, CmdSlewRate, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = DDRPHY_COMP_CR_DDRCRCOMPCTL1_CmdScompCells_MAX;
      MinCompVref[0] = 4;
      GlobalParam[0] = SCompCmd;
      MaxComp[0] = 63;

      break;

    case SCompCtl:
      // Comp
      CompCodeGroup[0] = OptCtlSComp; // SCompCodeCtl
      CompCodeDirection[0] = COMP_DN;
      NumTotalCompCodes = 1;

      // Comp Offset
      CompOffsetGroup[0] = CtlSCompOffset;
      CompOffsetBytes[0] = 1;
      CompOffsetCompIndex[0] = 0;
      MaxCompOffsetBits[0] = 4;
      NumCompOffsets = 1;

      // Comp Vref
      MrcGetSetNoScope(MrcData, CtlScompPC, ReadFromCache, &GetSetVal);
      CurrSCompPC[0] = (UINT8)GetSetVal;
      MrcGetSetNoScope(MrcData, CtlSlewRate, ReadFromCache, &GetSetVal);
      CurrCompVref[0] = (UINT8)GetSetVal;
      MaxCompVref[0] = DDRPHY_COMP_CR_DDRCRCOMPCTL1_CtlScompCells_MAX;
      MinCompVref[0] = 4;
      GlobalParam[0] = SCompCtl;
      MaxComp[0] = 63;

      break;

    default:
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_ERROR, "Invalid Comp Optimization Param : %d", Param);
      return mrcFail;
  }

  // Find initial comp values
  for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
    CurrentCompCode[Comp] = (UINT8)GetCompCode(MrcData, CompCodeGroup[Comp], CompCodeDirection[Comp]);
    NewCompCode[Comp] = CurrentCompCode[Comp];
  }

  // Get current comp offsets
  MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n***** Opt Param:%s *****", CompGlobalOffsetParamStr[Param]);
  MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\nInitial Values");
  for (CompOffset = 0; CompOffset < NumCompOffsets; CompOffset++) {
    MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n Comp Offset Param: %d, side effect = %s", CompOffset, (CompCodeSideEffect[CompOffsetCompIndex[CompOffset]] == TRUE) ? "TRUE" : "FALSE");
    MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n  Comp: %d", CurrentCompCode[CompOffsetCompIndex[CompOffset]]);
    MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n  Comp Vref: %d", CurrCompVref[CompOffsetCompIndex[CompOffset]] + (CurrSCompPC[CompOffsetCompIndex[CompOffset]] << SCOMP_PC_STORAGE_BIT_OFFSET));

    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        if ((MrcChannelExist(MrcData, Controller, Channel))) {
          MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n    MC%d.Ch%d\n", Controller, Channel);
          ChannelOut = &MrcData->Outputs.Controller[Controller].Channel[Channel];
          DdrCrCmdOffsetComp.Data = ChannelOut->CmdCompOffset;

          for (Byte = 0; Byte < CompOffsetBytes[CompOffset]; Byte++) {
            ExtendedCompOffsetValue = 0;
            if (MrcByteExist(MrcData, Controller, Channel, Byte) && IsSlewRateEnabled(MrcData, Param, Controller, Channel, Byte)) {
              DdrCrDataOffsetComp.Data = ChannelOut->DataCompOffset[Byte];

              switch (CompOffsetGroup[CompOffset]) {
                case DqDrvUpCompOffset:
                  Offset = (UINT16)DdrCrDataOffsetComp.Bits.DqDrvUpCompOffset;
                  break;
                case CmdRCompDrvUpOffset:
                  Offset = (UINT16)DdrCrCmdOffsetComp.Bits.CaRcompDrvUpOffset;
                  break;
                case CtlRCompDrvUpOffset:
                  Offset = (UINT16)DdrCrCmdOffsetComp.Bits.CtlRcompDrvUpOffset;
                  break;
                case DqsOdtCompOffset:
                  MrcGetSetChStrb(MrcData, Controller, Channel, Byte, DqsOdtCompOffset, ReadFromCache, &GetSetVal);
                  Offset = (UINT16)GetSetVal;
                  break;
                case DqOdtCompOffset:
                  Offset = (UINT16)DdrCrDataOffsetComp.Bits.DqOdtCompOffset;
                  break;
                case DqDrvDnCompOffset:
                  Offset = (UINT16)DdrCrDataOffsetComp.Bits.DqDrvDownCompOffset;
                  break;
                case CmdRCompDrvDownOffset:
                  Offset = (UINT16)DdrCrCmdOffsetComp.Bits.CaRcompDrvDownOffset;
                  break;
                case CtlRCompDrvDownOffset:
                  Offset = (UINT16)DdrCrCmdOffsetComp.Bits.CtlRcompDrvDownOffset;
                  break;
                case DqSCompOffset:
                  Offset = (UINT16)DdrCrDataOffsetComp.Bits.DqSlewRateCompOffset;
                  break;
                case CmdSCompOffset:
                  Offset = (UINT16)DdrCrCmdOffsetComp.Bits.CaScompOffset;
                  break;
                case CtlSCompOffset:
                  Offset = (UINT16)DdrCrCmdOffsetComp.Bits.CtlScompOffset;
                  break;
                case RloadCompOffset:
                  Offset = (UINT16)DdrCrDataOffsetComp.Bits.RloadOffset;
                  break;
                default:
                  Offset = 0;
              }
              Bytes[CompOffset]++;

              // We store the comp and comp offset values as unsigned integers even though the comp offsets are 2's complement.
              // We just have to make sure all the values are 8 bits long (longer than the longest comp or comp offset field) and the math works out.
              ExtendedCompOffsetValue = MrcSE(Offset, MaxCompOffsetBits[CompOffset], 16);
              MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "      Byte %d , Comp Offset: %d\n", Byte, (INT16)ExtendedCompOffsetValue);
              TotalCompCodes[CompOffset][Controller][Channel][Byte] = CurrentCompCode[CompOffsetCompIndex[CompOffset]] + ExtendedCompOffsetValue;
            }
            if (CompCodeSideEffect[CompOffsetCompIndex[CompOffset]] == FALSE) {
              AverageCompOffsetPerCompOffset[CompOffset] += (INT16)ExtendedCompOffsetValue;
            }
          }
        }
      }
    }
  }

  // Compute average comp offset per comp offset
  for (CompOffset = 0; CompOffset < NumCompOffsets; CompOffset++) {
    if (CompCodeSideEffect[CompOffsetCompIndex[CompOffset]] == TRUE) {
      continue;
    }
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Comp offset param: %d \n", CompOffset);
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Total comp offset for this comp offset param: %d\n", AverageCompOffsetPerCompOffset[CompOffset]);
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Total bytes: %d\n", Bytes[CompOffset]);
    }
    RoundSign = AverageCompOffsetPerCompOffset[CompOffset] > 0 ? 1 : -1;
    if (Bytes[CompOffset] != 0) {
      AverageCompOffsetPerCompOffset[CompOffset] = (AverageCompOffsetPerCompOffset[CompOffset] / Bytes[CompOffset]) + (RoundSign * ((((AverageCompOffsetPerCompOffset[CompOffset] % Bytes[CompOffset]) * RoundSign) > (Bytes[CompOffset] / 2)) ? 1 : 0));
    } else {
      AverageCompOffsetPerCompOffset[CompOffset] = 0;
    }
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "AverageCompOffsetPerCompOffset: %d\n", AverageCompOffsetPerCompOffset[CompOffset]);
    }
  }

  // Compute average comp offset per comp
  for (CompOffset = 0; CompOffset < NumCompOffsets; CompOffset++) { // Comp offset per comp
    if (CompCodeSideEffect[CompOffsetCompIndex[CompOffset]] == TRUE) {
      continue;
    }
    AverageCompOffsetPerComp[CompOffsetCompIndex[CompOffset]] += AverageCompOffsetPerCompOffset[CompOffset];
    NumCompOffsetsPerComp[CompOffsetCompIndex[CompOffset]]++;
  }
  for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
    if (CompCodeSideEffect[Comp] == TRUE) {
      continue;
    }
    RoundSign = AverageCompOffsetPerComp[Comp] > 0 ? 1 : -1;
    AverageCompOffsetPerComp[Comp] = (AverageCompOffsetPerComp[Comp] / NumCompOffsetsPerComp[Comp]) + (RoundSign * ((((AverageCompOffsetPerComp[Comp] % NumCompOffsetsPerComp[Comp]) * RoundSign) > (NumCompOffsetsPerComp[Comp] / 2)) ? 1 : 0));
    if (MRC_POWER_TRAINING_DEBUG) {
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Comp Param: %d, AverageCompOffsetPerComp: %d \n", Comp, AverageCompOffsetPerComp[Comp]);
    }
  }

  // Compute the minimum delta, which is the average of the AverageCompOffsetPerComp values for all non side effect comps
  for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
    if (CompCodeSideEffect[Comp] == TRUE) {
      continue;
    }
    MinDelta += ABS(AverageCompOffsetPerComp[Comp]);
    // Add together all comp offsets averages for all comps, adjusting for sign differences. This will show which way to sweep the comp vref.
    TotalCompOffset += AverageCompOffsetPerComp[Comp] * ((CompCodeDirection[Comp] == COMP_DN) ? -1 : 1); // If we decrease the offset value for Dn comps, it decreases the comp vref as they are only pulldown comps.
  }
  MinDelta = (MinDelta / (NumTotalCompCodes - NumTotalSideEffectCompCodes)) + (((MinDelta % (NumTotalCompCodes - NumTotalSideEffectCompCodes)) > ((NumTotalCompCodes - NumTotalSideEffectCompCodes) / 2)) ? 1 : 0);

  // Now, we need to know how much the base impedance value should be adjusted due to the average comp Offset
  Sign = (TotalCompOffset < 0) ? -1 : 1; // All Comp Vrefs are set to the same value
  if (MRC_POWER_TRAINING_DEBUG) {
    MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "MinDelta: %d, Sign: %d, TotalCompOffset: %d \n", MinDelta, Sign, TotalCompOffset);
  }
  for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
    // In this code, we assume all associated comp vrefs are set to the same point in the PHY init prior to this optimization.
    if (CurrSCompPC[Comp] < MRC_BIT1) { // Any valid code will be 0 or 1
      BestCompVref[Comp] = (CurrSCompPC[Comp] << SCOMP_PC_STORAGE_BIT_OFFSET) + CurrCompVref[Comp];
    }
    BestCompVref[Comp] = CurrCompVref[Comp];
  }

  // We optimize all comp vrefs together. This assumes all comp vrefs in this optimization have the same value per step size, and start at the same point.
  if (MinDelta != 0) { // Loop until we find a break condition as long as the average comp offset is not 0 (if it's 0, we're already at the optimal comp vref)
    Timeout = MrcCall->MrcGetCpuTime() + 10000; // 10 seconds timeout
    while (1) {
      if ((MrcCall->MrcGetCpuTime() >= Timeout)) {
        MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_ERROR, "\nTimeout in comp optimization! \n");
        return mrcFail;
      }
      for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Comp Param: %d \n", Comp);
        }
        if (CompCodeSideEffect[Comp] == FALSE) {
          NewCompVref[Comp] = CurrCompVref[Comp] + (Sign * CompVrefOffset);
        } else {
          // All side effect comp vrefs use the same equations:
          // No need to account for segments here, as they cancel out due to the way the equations are used.
          //   #1: NewMainCompVref_Target = (191 * 100 Ohms) / NewMainCompVref - 100 Ohms
          //   #2: OldMainCompVref_Target = (191 * 100 Ohms) / OldMainCompVref - 100 Ohms
          //   #2: SideEffectCompVref_Target = OldMainCompVref_Target * OldSideEffectCompVref / (191 - OldSideEffectCompVref)
          //   #3: NewSideEffectCompVref = 191 * (SideEffectCompVref_Target / (SideEffectCompVref_Target + NewMainCompVref_Target))
          //   We don't take into account Vtt mode here, as only the Rx ODT down comp is a side effect, and it's not used in Vtt mode.
          NewMainCompVref_Target = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor * TWO_DECIMAL_PLACES) / NewCompVref[OPTIMIZED_COMP_VREF_INDEX] - Inputs->RcompResistor * TWO_DECIMAL_PLACES; // * 100 to preserve 2 decimal places of accuracy
          OldMainCompVref_Target = (MRC_COMP_VREF_STEP_SIZE * Inputs->RcompResistor * TWO_DECIMAL_PLACES) / CurrCompVref[OPTIMIZED_COMP_VREF_INDEX] - Inputs->RcompResistor * TWO_DECIMAL_PLACES; // * 100 to preserve 2 decimal places of accuracy
          SideEffectCompVref_Target = (OldMainCompVref_Target * CurrCompVref[Comp]) / (MRC_COMP_VREF_STEP_SIZE - CurrCompVref[Comp]);
          if (MRC_POWER_TRAINING_DEBUG) {
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "NewMainCompVref_Target: %d\n", NewMainCompVref_Target);
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "OldMainCompVref_Target: %d\n", OldMainCompVref_Target);
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "SideEffectCompVref_Target: %d\n", SideEffectCompVref_Target);
          }
          NewCompVref[Comp] = (UINT8)((MRC_COMP_VREF_STEP_SIZE * SideEffectCompVref_Target) / (SideEffectCompVref_Target + NewMainCompVref_Target));
        }
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "new CompVref: %d\n", NewCompVref[Comp]);
        }
        if ((MinCompVref[Comp] > NewCompVref[Comp]) || (NewCompVref[Comp] > MaxCompVref[Comp])) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_WARNING, "Warning! We saturated the comp vref before we could shift it enough to make the comp shift match the average comp offset. Impedance will only be partially accurate! \n");
          ContinueOptimizing = FALSE;
          break;
        }
        if (CurrSCompPC[Comp] < MRC_BIT1) { // Any valid code will be 0 or 1
          NewCompVref[Comp] = (CurrSCompPC[Comp] << SCOMP_PC_STORAGE_BIT_OFFSET) + NewCompVref[Comp];
        }
        Status = UpdateCompGlobalOffset(MrcData, GlobalParam[Comp], NewCompVref[Comp], FALSE, TRUE, &NewCompValue);
        if (mrcSuccess != Status) {
          MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "UpdateCompGlobalOffset() error %d\n", Status);
          return mrcFail;
        }
      }
      if (!ContinueOptimizing) {
        break;
      }

      for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Comp Param: %d \n", Comp);
        }
        NewCompCode[Comp] = (UINT8)GetCompCode(MrcData, CompCodeGroup[Comp], CompCodeDirection[Comp]);

        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "New comp: %d\n", NewCompCode[Comp]);
        }
        if ((RESERVED_COMP_CODES > NewCompCode[Comp]) || (NewCompCode[Comp] > (MaxComp[Comp] - RESERVED_COMP_CODES))) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_WARNING, "Warning! We saturated the comp before we could shift it as far as the average comp offset. Impedance will only be partially accurate! \n");
          ContinueOptimizing = FALSE;
        }
      }
      if (!ContinueOptimizing) {
        break;
      }

      // Compute average delta across all comps
      CurrentDelta = 0;
      for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
        if (CompCodeSideEffect[Comp] == TRUE) {
          continue;
        }
        CurrentDelta += ABS(CurrentCompCode[Comp] + AverageCompOffsetPerComp[Comp] - NewCompCode[Comp]);
      }
      if (!ContinueOptimizing) {
        break;
      }
      CurrentDelta = (CurrentDelta / (NumTotalCompCodes - NumTotalSideEffectCompCodes)) + (((CurrentDelta % (NumTotalCompCodes - NumTotalSideEffectCompCodes)) > ((NumTotalCompCodes - NumTotalSideEffectCompCodes) / 2)) ? 1 : 0);

      // Move the Comp Vref until the comp outputs change an amount equivalent to the average comp offset
      if (CurrentDelta <= MinDelta) {
        MinDelta = CurrentDelta;
        MrcCall->MrcCopyMem(BestCompVref, NewCompVref, NumTotalCompCodes);
        if (MRC_POWER_TRAINING_DEBUG) {
          for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Comp = %d, BestCompVref = %d \n", Comp, BestCompVref[Comp]);
          }
        }
        if (MinDelta == 0) {
          if (MRC_POWER_TRAINING_DEBUG) {
            MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "(MinDelta == 0) \n");
          }
          break; // We're done
        }
      } else {
        if (MRC_POWER_TRAINING_DEBUG) {
          MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "(CurrentDelta > MinDelta) CurrentDelta = %d, MinDelta = %d \n", CurrentDelta, MinDelta);
        }
        break; // We overshot, so we're done
      }

      CompVrefOffset++;
    }

    // Update new comp vrefs or revert to the initial values. Run this in a seperate loop to ensure all comp vrefs are updated before comps are read
    for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
      Status = UpdateCompGlobalOffset(MrcData, GlobalParam[Comp], BestCompVref[Comp], FALSE, TRUE, &NewCompValue);
      if (mrcSuccess != Status) {
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "UpdateCompGlobalOffset() error %d\n", Status);
        return mrcFail;
      }
    }

    // Read the new comp values
    for (Comp = 0; Comp < NumTotalCompCodes; Comp++) {
      NewCompCode[Comp] = (UINT8)GetCompCode(MrcData, CompCodeGroup[Comp], CompCodeDirection[Comp]);
    }

    // Set the new comp offset values and print them out
    MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\nFinal Values");
    for (CompOffset = 0; CompOffset < NumCompOffsets; CompOffset++) {
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n Comp Offset Param: %d, side effect = %s", CompOffset, (CompCodeSideEffect[CompOffsetCompIndex[CompOffset]] == TRUE) ? "TRUE" : "FALSE");
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n  Comp: %d", NewCompCode[CompOffsetCompIndex[CompOffset]]);
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n  Comp Vref: %d", BestCompVref[CompOffsetCompIndex[CompOffset]]);

      for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
        for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
          if ((MrcChannelExist(MrcData, Controller, Channel))) {
            ChannelOut = &MrcData->Outputs.Controller[Controller].Channel[Channel];
            MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n    MC%d.Ch%d\n", Controller, Channel);
            for (Byte = 0; Byte < CompOffsetBytes[CompOffset]; Byte++) {
              if (MrcByteExist(MrcData, Controller, Channel, Byte) && IsSlewRateEnabled(MrcData, Param, Controller, Channel, Byte)) {
                MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "      Byte %d , Comp Offset: %d\n", Byte, TotalCompCodes[CompOffset][Controller][Channel][Byte] - NewCompCode[CompOffsetCompIndex[CompOffset]]);
                UpdateOptParamOffset(MrcData, Controller, Channel, 0, Byte, CompOffsetGroup[CompOffset], TotalCompCodes[CompOffset][Controller][Channel][Byte] - NewCompCode[CompOffsetCompIndex[CompOffset]], TRUE);
              }
            }
          }
        }
      }
    }

    ForceSystemRComp(MrcData, 0, TRUE);
  } else {
    MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n MinDelta = 0, so comp offsets are already at optimal values. Final values equal initial values.");
  }

  MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n");

  return mrcSuccess;
}

/**
  This step performs Comp Offset optimization on the param list defined in this function.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess
**/
MrcStatus
MrcOptimizeComp (
  IN MrcParameters *const MrcData
  )
{
  // Comp vrefs that depend on the values of other comp vrefs must be optimized after their dependencies
  // RloadCompOffset is run with CmdRCompDrvUpOffset, as it depends on the same comp vref value
  // DqsOdtCompOffset is run with DqOdtCompOffset, as it depends on the same comp vref
  static UINT8 ParamList[] = { SCompCmd, SCompCtl, SCompDq, WrDSCmdUp, WrDSCmdDn, WrDSCtlUp, WrDSCtlDn, WrDSUp, WrDSDn, RdOdtUp };

  MrcDebug            *Debug;
  MrcOutput           *Outputs;
  MrcIntOutput        *IntOutputs;
  MrcIntControllerOut *IntControllerOut;
  MrcIntCmdTimingOut  *IntCmdTiming;
  MrcStatus           Status;
  INT64               GetSetVal;
  UINT32              Controller;
  UINT32              Channel;
  UINT32              MinCode;
  UINT8               Byte;
  UINT8               Rank;
  UINT8               CmdGroup;
  UINT8               CmdGroupMax;
  UINT8               Param;
  BOOLEAN             Ddr4;

  Outputs = &MrcData->Outputs;
  Debug = &Outputs->Debug;
  IntOutputs = (MrcIntOutput *)(MrcData->IntOutputs.Internal);
  Ddr4 = (Outputs->DdrType == MRC_DDR_TYPE_DDR4);
  CmdGroupMax = (Ddr4) ? MAX_COMMAND_GROUPS : 1;
  Status = mrcSuccess;


  Outputs = &MrcData->Outputs;

  for (Param = 0; Param < ARRAY_COUNT (ParamList); Param++) {
    if(OptimizeCompOffset (MrcData, ParamList[Param]) == mrcFail) {
      return mrcFail;
    }
  }

  // We don't need to change SOC ODT, because while the Dq ODT Comp Vref reference changed during comp optimization, the effective comp vref (accounting for comp/comp offset) did not.

  // Normalize timing back to 0 to improve performance
  if (Ddr4) {
    MrcBlockTrainResetToggle(MrcData, FALSE);

    MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "\n*** Normalize timing back to 0\n");
    for (Controller = 0; Controller < MAX_CONTROLLER; Controller++) {
      IntControllerOut = &IntOutputs->Controller[Controller];
      for (Channel = 0; Channel < Outputs->MaxChannels; Channel++) {
        if (!MrcChannelExist(MrcData, Controller, Channel)) {
          continue;
        }
        IntCmdTiming = &IntControllerOut->CmdTiming[Channel];

        // Find the minimum PI Code across all relevant CMD and CTL fubs
        // CLK shift will also change RcvEn / TxDq / TxDqs, so check them as well.
        MinCode = MRC_UINT32_MAX;
        for (CmdGroup = 0; CmdGroup < CmdGroupMax; CmdGroup++) {
          MinCode = MIN(MinCode, IntCmdTiming->CmdPiCode[CmdGroup]);
        }

        for (Rank = 0; Rank < MAX_RANK_IN_CHANNEL; Rank++) {
          if (MrcRankExist(MrcData, Controller, Channel, Rank)) {
            // Cke PI
            MinCode = MIN(MinCode, IntCmdTiming->CkePiCode[Rank]);
            // Ctl PI
            MinCode = MIN(MinCode, IntCmdTiming->CtlPiCode[Rank]);
            for (Byte = 0; Byte < Outputs->SdramCount; Byte++) {
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, RecEnDelay, ReadFromCache, &GetSetVal);
              MinCode = MIN(MinCode, (UINT32)GetSetVal);
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, TxDqsDelay, ReadFromCache, &GetSetVal);
              MinCode = MIN(MinCode, (UINT32)GetSetVal);
              MrcGetSetStrobe(MrcData, Controller, Channel, Rank, Byte, TxDqDelay, ReadFromCache, &GetSetVal);
              MinCode = MIN(MinCode, (UINT32)GetSetVal);
            }
          }
        }

        if (MinCode >= 32) {
          MinCode = MIN(MinCode, MinCode - 32);  // Keep at least 32 PI ticks for margining.
        }
        else {
          MinCode = 0;
        }

        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Mc%d C%d: shifting all PI settings by Min PI Code = %d\n", Controller, Channel, MinCode);
        // Don't do final shifting as CLK cannnot be shifted in CTE
        ShiftChannelTiming(MrcData, Controller, Channel, (-1) * MinCode);
      } // for Channel
    } // for Controller

    Status = MrcResetSequence(MrcData);
    MrcBlockTrainResetToggle(MrcData, TRUE);
  } // Ddr4

  // We need to run this because we changed the base Rx ODT comp vref and/or comp outputs. When we trained the Rx comp offsets neither changed, so there was no need to run it then.
  //if (Outputs->Lpddr) {
  //  MrcLpddrOvershoot (MrcData);
  //}

#ifdef LB_STUB_FLAG
  return mrcFail;
#endif

  // Comp optimization can skew around the margins quite a bit (it will recover with advanced training), so re-center here to ensure everything is stable

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Timing \n");
  DQTimeCentering1D (MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
  DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Vref\n");
  DQTimeCentering1D (MrcData, WrV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Vref \n");
  DQTimeCentering1D (MrcData, RdV, 0, V_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG (&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center RcvEnable \n");
  DQTimeCentering1D (MrcData, RcvEnaX, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  return Status;
}

/**
  This function calculates the percent of power saving from the power optimization steps and
  updates the proper registers in the PCU.  To get the correct base line for this calculation,
  this routing needs to run first time early in the training in order to update the MrcStruct
  with the base line.  After the power training steps, it will run again to get the actual
  percent of power saving.

  @param[in] MrcData - Include all MRC global data.

  @retval MrcStatus - mrcSuccess

**/
MrcStatus
MrcPowerSavingMeter (
  IN MrcParameters *const MrcData
  )
{
  MrcDebug                                *Debug;
  MrcOutput                               *Outputs;
  MrcOdtPowerSaving                       *OdtPowerSaving;
  UINT8                                   PercentTotal = 0;

  Outputs = &MrcData->Outputs;
  OdtPowerSaving = &Outputs->OdtPowerSavingData;

  if (FIVR_POWER_MEASUREMENT_ENABLED) {
    Debug = &Outputs->Debug;
    CalcSysPower(MrcData, FALSE, 10, NULL, NULL, 0); // mW x10

    if (OdtPowerSaving->BaseFlag) {
      if (OdtPowerSaving->BaseSavingTotal > OdtPowerSaving->MrcSavingTotal) {
        PercentTotal = (UINT8)(((UINT32)(OdtPowerSaving->BaseSavingTotal - OdtPowerSaving->MrcSavingTotal) * 256) / OdtPowerSaving->BaseSavingTotal);
      } else {
        PercentTotal = 0;
      }
    } else {
      OdtPowerSaving->BaseFlag = TRUE;
    }

    MRC_DEBUG_MSG(
      Debug,
      MSG_LEVEL_NOTE,
      "\tBaseLine\tMrcSaving\tSaving\nAvgtotal\t%d\t\t%d\t\t%d%%\n",
      OdtPowerSaving->BaseSavingTotal / 10,
      OdtPowerSaving->MrcSavingTotal / 10,
      ((UINT16)PercentTotal) * 100 / 256
    );
  } else {
    if (!OdtPowerSaving->BaseFlag) {
      OdtPowerSaving->BaseFlag = TRUE;
    }
  }

  MrcPrintDimmOdtValues (MrcData);  // Print DIMM ODT table

  return mrcSuccess;
}

/**
  This function reads the selected comp code.

  @param[in] MrcData  - Include all MRC global data.
  @param[in] OptParam - Parameter to read the relevant comp code.
  @param[in] UpDown   - 0 : up only. 1 : down only.

  @retval The selected comp code (or -1 if the comp is unused and thus invalid in this configuration)
**/
INT8
GetCompCode (
  IN OUT MrcParameters *const MrcData,
  IN     UINT8                OptParam,
  IN     UINT8                UpDown
  )
{
  MrcDebug                            *Debug;
  INT64                               GetSetVal;
  GSM_GT                              GroupUp = GsmGtMax;
  GSM_GT                              GroupDn = GsmGtMax;

  Debug  = &MrcData->Outputs.Debug;

  switch (OptParam) {
    case OptWrDS:
    case OptWrDSUpCoarse:
    case OptWrDSDnCoarse:
      GroupUp = TxRonUp;
      GroupDn = TxRonDn;
      break;
    case OptCmdDS:
    case OptCmdDSUpCoarse:
    case OptCmdDSDnCoarse:
      GroupUp = WrDSCodeUpCmd;
      GroupDn = WrDSCodeDnCmd;
      break;
    case OptCtlDS:
    case OptCtlDSUpCoarse:
    case OptCtlDSDnCoarse:
      GroupUp = WrDSCodeUpCtl;
      GroupDn = WrDSCodeDnCtl;
      break;
    case OptClkDS:
    case OptClkDSUpCoarse:
    case OptClkDSDnCoarse:
      GroupUp = WrDSCodeUpClk;
      GroupDn = WrDSCodeDnClk;
      break;
    case OptRdDqsOdt:
    case OptRdDqOdt:
      switch (MrcData->Outputs.OdtMode) {
        case MrcOdtModeVss:
          // in Vss mode only odt down is valid
          GroupDn = CompRcompOdtDn;
          break;
        case MrcOdtModeVddq:
        case MrcOdtModeVtt:
          // in Vtt/Vddq mode only odt up is valid
          GroupUp = CompRcompOdtUp;
          break;
        default:
          GroupDn = GsmGtMax;
          GroupUp = GsmGtMax;
          MRC_DEBUG_MSG(Debug, MSG_LEVEL_ERROR, "Error! %s(): Invalid OdtMode %d\n", __FUNCTION__, MrcData->Outputs.OdtMode);
      }
      break;
    case OptRxLoad:
      GroupUp = RloadDqsUp;
      break;
    case OptSComp:
      GroupDn = SCompCodeDq;
      break;
    case OptCmdSComp:
      GroupDn = SCompCodeCmd;
      break;
    case OptCtlSComp:
      GroupDn = SCompCodeCtl;
      break;
    case OptClkSComp:
      GroupDn = SCompCodeClk;
      break;
    case OptTxDqsTco:
      GroupUp = TxDqsTcoPFallNRise;
      GroupDn = TxDqsTcoPRiseNFall;
      break;
    default:
      GroupUp = GsmGtMax;
      GroupDn = GsmGtMax;
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Error! GetCompCode(): Invalid OptParam %d\n", OptParam);
  }

  if (UpDown == COMP_UP && GroupUp != GsmGtMax) {
    MrcGetSetNoScope(MrcData, GroupUp, ReadUncached, &GetSetVal);
    return (INT8)GetSetVal;
  } else if (UpDown == COMP_DN && GroupDn != GsmGtMax) {
    MrcGetSetNoScope(MrcData, GroupDn, ReadUncached, &GetSetVal);
    return (INT8)GetSetVal;
  }

  return -1;
}

/**
  Program COMP Vref offset according to the passed parameter and target values.
  DqOdt segments:
   Based on setting of GsmIocDataODTSingleSegEn enumration, 1 segment if enum is set otherwise 2.

  @param[in] MrcData      - Include all MRC global data.
  @param[in] Param        - COMP Vref parameter (see TGlobalCompOffset).
  @param[in] RcompTarget  - Array of the Rcomp Targets COMP for { DqOdt, DqDrv, CmdDrv, CtlDrv, ClkDrv }
  @param[in] UpdateHost   - Update host struct with the new value or not.

  @retval mrcSuccess  - if Param is a valid COMP Vref parameter
**/
MrcStatus
UpdateCompTargetValue (
  MrcParameters *const MrcData,
  UINT8                Param,
  UINT16               RcompTarget[MAX_RCOMP_TARGETS],
  BOOLEAN              UpdateHost
  )
{
  MrcDebug   *Debug;
  MrcInput   *Inputs;
  MrcOutput  *Outputs;
  INT64      GetSetVal;
  UINT32     Numerator;
  UINT32     Denominator;
  UINT32     FirstController;
  UINT32     FirstChannel;
  INT16      CompVrefUp;
  INT16      CompVrefDn;
  UINT16     ReferenceRUp;
  UINT16     ReferenceRDn;
  UINT16     TargetUpValue;
  UINT16     TargetDnValue;
  UINT8      NumOfSegments;
  UINT8      CompCodeUp;
  UINT8      CompCodeDn;
  BOOLEAN    VttOdt;
  CompGlobalOffsetParam GlobalCompUp;
  CompGlobalOffsetParam GlobalCompDn;
  MrcStatus  Status;
  UINT32     NewCompValue;
  UINT16     LpddrOvershootLimits[] = { 0, 0 };

  Inputs   = &MrcData->Inputs;
  Outputs  = &MrcData->Outputs;
  Debug    = &Outputs->Debug;
  VttOdt   = (Outputs->OdtMode == MrcOdtModeVtt);
  if (Param >= MAX_RCOMP_TARGETS) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "Error! Bad Param : %d\n", Param);
    return mrcFail;
  }

  if (Outputs->Lpddr && Param == RdOdt) {
    CalcRxCompVrefLimits(MrcData, TRUE, LpddrOvershootLimits);

    if (LpddrOvershootLimits[0] > RcompTarget[RdOdt]) {
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Changing PHY ODT due to LPDDR overshoot limits. Original %d, Final %d \n", RcompTarget[RdOdt], LpddrOvershootLimits[0]);
      RcompTarget[RdOdt] = LpddrOvershootLimits[0];
      MrcSetDefaultRxVref(MrcData, TRUE, TRUE); // Recalculate RxVref
    }
    if (LpddrOvershootLimits[1] < RcompTarget[RdOdt]) {
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "Changing PHY ODT due to LPDDR overshoot limits. Original %d, Final %d \n", RcompTarget[RdOdt], LpddrOvershootLimits[1]);
      RcompTarget[RdOdt] = LpddrOvershootLimits[1];
      MrcSetDefaultRxVref(MrcData, TRUE, TRUE); // Recalculate RxVref
    }
  }

  TargetDnValue = TargetUpValue = RcompTarget[Param];
  ReferenceRDn = RcompTarget[Param];

  // The number of segments for CmdDrv, CtlDrv, and ClkDrv is 3
  // DqDrv is 2 segments
  // DqOdt is based on ODT mode: 1 for VTT and 2 for the rest.
  // RCOMP0: DQ ODT (read)
  // RCOMP1: DQ  / CLK Ron (drive strength)
  // RCOMP2: CMD / CTL Ron (drive strength)
  NumOfSegments = THREE_SEGMENTS;
  switch (Param) {
    case RdOdt:
      FirstController = (MrcControllerExist (MrcData, cCONTROLLER0)) ? 0 : 1;
      FirstChannel = MrcData->Outputs.Controller[FirstController].FirstPopCh;
      MrcGetSetChStrb (MrcData, FirstController, FirstChannel, 0, GsmIocDataODTSingleSegEn, ReadFromCache, &GetSetVal);
      NumOfSegments = (GetSetVal) ? ONE_SEGMENT : TWO_SEGMENTS;
      GlobalCompUp = RdOdtUp;
      GlobalCompDn = RdOdtDn;
      TargetDnValue = RcompTarget[WrDS]; //Override the default
      break;

    case WrDS:
      NumOfSegments = TWO_SEGMENTS;
      GlobalCompUp = WrDSUp;
      GlobalCompDn = WrDSDn;
      break;

    case WrDSClk:
      GlobalCompUp = WrDSClkUp;
      GlobalCompDn = WrDSClkDn;
      break;

    case WrDSCmd:
      GlobalCompUp = WrDSCmdUp;
      GlobalCompDn = WrDSCmdDn;
      break;

    case WrDSCtl:
      GlobalCompUp = WrDSCtlUp;
      GlobalCompDn = WrDSCtlDn;
      break;

    default:
      GlobalCompUp = 0;
      GlobalCompDn = 0;
      break;
  }

  ReferenceRUp = DIVIDEROUND (Inputs->RcompResistor, NumOfSegments);
  Numerator    = MRC_COMP_VREF_STEP_SIZE * ReferenceRUp;
  Denominator  = TargetUpValue + ReferenceRUp;
  if (VttOdt && (Param == RdOdt)) {
    // Do multiply by 2 here to optimize to 1 divide which handles rounding.
    Denominator *= 2;
  }
  Denominator = MAX (Denominator, 1);
  // Used UINT32 to prevent overflow of multiply with large UINT16 numbers.
  // Result should be less than UINT8 Max as register field is smaller than UINT8.
  CompVrefUp = (UINT16) DIVIDEROUND (Numerator, Denominator);

  Numerator    = MRC_COMP_VREF_STEP_SIZE * ReferenceRDn;
  Denominator  = TargetDnValue + ReferenceRDn;
  if (VttOdt && (Param == RdOdt)) {
    // Do multiply by 2 here to optimize to 1 divide which handles rounding.
    Denominator *= 2;
  }
  Denominator = MAX (Denominator, 1);
  // Used UINT32 to prevent overflow of multiply with large UINT16 numbers.
  // Result should be less than UINT8 Max as register field is smaller than UINT8.
  CompVrefDn = (UINT16) DIVIDEROUND (Numerator, Denominator);

  // Callee handles saturation at Min/Max values.
  Status = UpdateCompGlobalOffset (MrcData, GlobalCompUp, CompVrefUp, TRUE, UpdateHost, &NewCompValue);
  if (mrcSuccess != Status) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "UpdateCompGlobalOffset() error %d\n", Status);
    return mrcFail;
  }
  CompCodeUp = (UINT8) NewCompValue;

  Status = UpdateCompGlobalOffset (MrcData, GlobalCompDn, CompVrefDn, TRUE, UpdateHost, &NewCompValue);
  if (mrcSuccess != Status) {
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "UpdateCompGlobalOffset() error %d\n", Status);
    return mrcFail;
  }
  CompCodeDn = (UINT8) NewCompValue;

  // For VSS mode, comp dn is always returned, and for VDDQ/VTT mode, comp up is always returned.
  // CompCodeDn is the final adjusted comp up or comp dn
  if (Param == RdOdt) {
    CompCodeUp = CompCodeDn;
  }

  MRC_DEBUG_MSG (
    Debug,
    MSG_LEVEL_NOTE,
    "%8s: Target (Up/Down) value: %3d/%3d ReferenceR (Up/Down): %3d/%3d VrefOffset (Up/Down): %3d/%3d Current Comp code (Up/Down): %3d/%3d\n",
    GlobalCompOffsetStr[Param],
    TargetUpValue,
    TargetDnValue,
    ReferenceRUp,
    ReferenceRDn,
    CompVrefUp,
    CompVrefDn,
    CompCodeUp,
    CompCodeDn
    );

  if ((CompCodeUp <= 0) || (CompCodeUp >= 63) || (CompCodeDn <= 0) || (CompCodeDn >= 63)) {
    MRC_DEBUG_MSG(Debug, MSG_LEVEL_WARNING, "WARNING: COMP code is saturated !\n");
  }

  if (GlobalCompUp == WrDSCmdUp) {
    Status = UpdateCompGlobalOffset (MrcData, RxLoad, CompVrefUp + 10, TRUE, UpdateHost, &NewCompValue); // RLoad needs to be higher than CMD DS Up
    if (mrcSuccess != Status) {
      MRC_DEBUG_MSG (Debug, MSG_LEVEL_ERROR, "UpdateCompGlobalOffset() error %d\n", Status);
      return mrcFail;
    }
    CompCodeUp = (UINT8) NewCompValue;
    MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "RloadVref: %3d set equal to CmdDrvVrefUp (WrDSCmd)\n", CompCodeUp);

    if ((CompCodeUp <= 0) || (CompCodeUp >= 63)) {
      MRC_DEBUG_MSG(Debug, MSG_LEVEL_WARNING, "WARNING: RLoad COMP code is saturated !\n");
    }
  }

  if (Outputs->Lpddr) {
    MrcLpddrOvershoot(MrcData);
  }

  return mrcSuccess;
}
/**
  This function returns the params setting accordingly to input.

  @param[in out] ParamOff- Parameters result Offsets
  @param[in]     GridMode- Selects the way we sweep the params
  @param[in]     Index   - Linear Index for all params
  @param[in]     YLen    - Determines the Y-Dimension lengths
  @param[in]     XLen    - Determines the X-Dimension lengths


  @retval if point is valid (tested)
**/
BOOLEAN
GetParamsXYZ (
  IN     MrcParameters      *const MrcData,
  IN OUT INT8              *ParamOff,
  IN     const UINT8        OptParamLen,
  IN     const UINT8        GridMode,
  IN     const UINT8        Index,
  IN     const INT8         *Start,
  IN     const UINT8        *ParamOffLen
)

{
  MrcDebug          *Debug;
  INT8              XOff;
  INT8              YOff;
  BOOLEAN           SkipPoint = FALSE;

  Debug  = &MrcData->Outputs.Debug;

  if (OptParamLen == 1) {
    switch (GridMode) {
      case FullGrid:
        break;
      case CustomSR1:
        if (Index == 0) {
          ParamOff[0] = -17; // disable SR for fastest SR
        }
        break;
      case CustomSR2:
        if (Index == 0) {
          ParamOff[0] = -9; // disable SR for fastest SR
        }
        break;
      default:
        MRC_DEBUG_MSG(Debug, MSG_LEVEL_NOTE, "unknow GridMode %d\t", GridMode);
    }
    if (!((GridMode == CustomSR1 || GridMode == CustomSR2) && Index == 0)) {
      ParamOff[0] = Index + Start[0];
    }
  } else if (OptParamLen == 2) {
    XOff =  Index % ParamOffLen[0];
    YOff =  Index / ParamOffLen[0];
    switch (GridMode) {
      case FullGrid:
        break;
      case ChessOdd:
        if (XOff % 2) {
          if (YOff % 2) {
            SkipPoint = TRUE;
          }
        } else {
          if (YOff % 2 == 0) {
            SkipPoint = TRUE;
          }
        }
        break;
      case ChessEven:
        if (XOff % 2) {
          if (YOff % 2 == 0) {
            SkipPoint = TRUE;
          }
        } else {
          if (YOff % 2) {
            SkipPoint = TRUE;
          }
        }
        break;
      case DecapSweep:
        if (((Index % MAX_DECAP_VALUES) - (MAX_DECAP_VALUES - Index / MAX_DECAP_VALUES)) > 0) {
          SkipPoint = TRUE;
        }
        break;
      case UpDnCompOffsetSweepRange1:
        if ((XOff + Start[0]) - (YOff + Start[1]) > 1 || (XOff + Start[0]) - (YOff + Start[1]) < -1) {
          SkipPoint = TRUE;
        }
        break;
      case UpDnCompOffsetSweepRange2:
        if ((XOff + Start[0]) - (YOff + Start[1]) > 2 || (XOff + Start[0]) - (YOff + Start[1]) < -2) {
          SkipPoint = TRUE;
        }
        break;
      default:
        MRC_DEBUG_MSG (Debug, MSG_LEVEL_NOTE, "unknow GridMode %d\t", GridMode);
    }
    ParamOff[0] = XOff + Start[0];
    ParamOff[1] = YOff + Start[1];
  }

  if (GridMode == Reversed1D) {
    ParamOff[0] =  Start[0] + ParamOffLen[0] - 1 - Index; // reversed ordering for param
  }

    return SkipPoint;
}

/**
  Returns the index into RttType array

  @param[in] OptDimmOdt - DimmOdt type

  @retval One of the following values: RttWr RttNom RttPark

**/
DimmOdtType
GetRttType (
  IN const UINT8 OptDimmOdt
  )
{
  switch (OptDimmOdt) {
    case OptDimmOdtWr:
      return RttWr;

    case OptDimmOdtNom:
    case OptDimmOdtNomNT:
      return RttNom;

    case OptDimmOdtPark:
    case OptDimmOdtParkNT:
      return RttPark;

    default:
      break;
  }

  return RttMaxType;
}

/**
This function is used to train the Tx TCO Comp offset for Dq.
TCO Comp performs training using fixed pattern, in order to avoid optimization of TCO Comp based on ISI / Crosstalk
In order to leverage for very high margins resulting of fixed pattern, we ignore the power / UPM limits.

@param[in] MrcData - Pointer to global MRC data.

@retval - mrcSuccess
**/
MrcStatus
MrcTxDqTcoCompTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  static const UINT8  TestList[] = { WrT, CmdT }; // This affects RxDqs sometimes, need to take that into consideration
  static const UINT8  OptParam[] = { OptTxDqTco };
  UINT8               Scale[] = { 1, 1, 255, 0, 0 }; // Ignore power optimization
  OptOffsetChByte     BestOff;
  INT8                Start;
  INT8                Stop;
  MrcOutput           *Outputs;

  Outputs = &MrcData->Outputs;

  // Coarse search for Global comp
  Start = OptParamLimitValue (MrcData, OptParam[0], 0);
  Stop  = OptParamLimitValue (MrcData, OptParam[0], 1);

  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    FullGrid,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    &Start,
    &Stop,
    OPT_PARAM_LOOP_COUNT + 2,
    1,                    // Repeats
    0,                    // NoPrint
    0,                    // SkipOptUpdate
    0,                    // GuardBand
    0,                    // PatType
    SaveMarginsArray
    );

  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
  DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

  MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Timing \n");
  DQTimeCentering1D(MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training

  return mrcSuccess;
}

/**
  This function is used to train the Tx TCO Comp offset for Dqs.
  TcoDqs trained per byte.

  @param[in] MrcData - Pointer to global MRC data.

  @retval - mrcSuccess
**/
MrcStatus
MrcTxDqsTcoCompTraining (
  IN MrcParameters *const MrcData
  )
{
  UINT16              SaveMarginsArray[2][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  static const UINT8  TestList[] = { WrT, CmdT };
  static const UINT8  OptParam[] = { OptTxDqsTco };
  UINT8               Scale[] = { 1, 1, 255, 0, 0 }; // Ignore power optimization
  OptOffsetChByte     BestOff;
  INT8                Start;
  INT8                Stop;
  MrcOutput           *Outputs;

  Outputs = &MrcData->Outputs;

  // Coarse search for Global comp
  Start = OptParamLimitValue (MrcData, OptParam[0], 0);
  Stop = OptParamLimitValue (MrcData, OptParam[0], 1);

  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    FullGrid,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    &Start,
    &Stop,
    OPT_PARAM_LOOP_COUNT + 2,
    1,                    // Repeats
    0,                    // NoPrint
    0,                    // SkipOptUpdate
    0,                    // GuardBand
    0,                     // PatType
    SaveMarginsArray
    );

  // Turn on if VCCDLL Bypass training is disabled
  if ((!(MrcData->Inputs.TrainingEnables2.VCCDLLBP)) || (MrcData->Inputs.PowerTrainingMode != MrcTmPower)) {
    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Read Timing \n");
    DQTimeCentering1D_RxDC (MrcData, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE);

    MRC_DEBUG_MSG(&Outputs->Debug, MSG_LEVEL_NOTE, "Re-center Write Timing \n");
    DQTimeCentering1D(MrcData, WrT, 0, T_RECENTERING_STEP_SIZE, RE_CENTER_LOOP_COUNT, MRC_POWER_TRAINING_DEBUG, FALSE); // No need to use WrTLp4 after basic training
  }

  return mrcSuccess;
}

/**
  This function is used to train the Clk TCO Comp code.

  @param[in] MrcData - Pointer to global MRC data.

  @retval - mrcSuccess
**/
MrcStatus
MrcClkTcoCompTraining (
  IN MrcParameters *const MrcData
)
{
  UINT16              SaveMarginsArray[1][MAX_OPT_POINTS][MAX_CONTROLLER][MAX_SDRAM_IN_DIMM]; // Must match test list size
  static const UINT8  TestList[] = { CmdT };
  static const UINT8  OptParam[] = { OptCCCTco };
  UINT8               Scale[] = { 1, 0, 0, 0, 0 };
  OptOffsetChByte     BestOff;
  INT8                Start;
  INT8                Stop;
  MrcStatus           Status = mrcSuccess;

  Start = OptParamLimitValue (MrcData, OptParam[0], 0);
  Stop  = OptParamLimitValue (MrcData, OptParam[0], 1);

  TrainDDROptParam (
    MrcData,
    &BestOff,
    MrcData->Outputs.ValidChBitMask,
    MrcData->Outputs.ValidRankMask,
    OptParam,
    ARRAY_COUNT (OptParam),
    FullGrid,
    TestList,
    ARRAY_COUNT (TestList),
    Scale,
    NULL,
    &Start,
    &Stop,
    OPT_PARAM_LOOP_COUNT,
    1,                    // Repeats
    0,                    // NoPrint
    0,                    // SkipOptUpdate
    0,                    // GuardBand
    StaticPattern,         // PatType
    SaveMarginsArray
    );

  // Re-Center CMD Timing and voltage and update Host Struct with new center
  Status = MrcCmdTimingCentering(MrcData, MrcData->Outputs.ValidChBitMask, CCC_RE_CENTER_LOOP_COUNT, TRUE, MRC_POWER_TRAINING_DEBUG, 1);
  if (Status != mrcSuccess) {
    return Status;
  }

  return mrcSuccess;
}
